/*
 * Electric(tm) VLSI Design System
 *
 * File: global.h
 * Definitions of interest to all modules
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "config.h"

/* system includes */
#include <stdio.h>
#include <ctype.h>
#if STDC_HEADERS
#  include <stdlib.h>
#  include <string.h>
#  include <stdarg.h>
#  define var_start(x, y) va_start(x, y)
#else
#  ifndef HAVE_STRCHR
#    define strchr index
#  endif
#  include <varargs.h>
#  define var_start(x, y) va_start(x)
#endif
#ifdef HAVE_STRINGS_H
#  include <strings.h>
#endif
#ifdef HAVE_STDINT_H
#  include <stdint.h>
#else
#  ifdef HAVE_INTTYPES_H
#    include <inttypes.h>
#  endif
#endif

/* #define DEBUGMEMORY 1 */					/* uncomment to debug memory */

typedef          long   INTBIG;				/* at least 32 bits, can hold address */
typedef unsigned long  UINTBIG;
typedef          short  INTSML;				/* at least 16 bits */
typedef unsigned short UINTSML;
typedef        int64_t  INTHUGE;			/* at least 64 bits */
typedef      u_int64_t UINTHUGE;

#define SIZEOFINTSML  (sizeof (INTSML))		/* bytes per short integer */
#define SIZEOFINTBIG  (sizeof (INTBIG))		/* bytes per long integer */
#define SIZEOFINTHUGE (sizeof (INTHUGE))	/* bytes per huge integer */
#define MAXINTBIG     0x7FFFFFFF			/* largest possible integer */
#ifdef WIN32
#  define INTHUGECONST(a) (a ## i64)
#else
#  define INTHUGECONST(a) (a ## LL)
#endif
typedef INTBIG        XARRAY[3][3];			/* 3x3 transformation matrix */
#define REGISTER      register

typedef struct
{
	INTSML top, left, bottom, right;
} RECTAREA;

/* forward declarations for structures */
struct Icomcomp;
struct Inodeinst;
struct Iportarcinst;
struct Iportexpinst;
struct Inodeproto;
struct Iportproto;
struct Iarcinst;
struct Iarcproto;
struct Inetwork;
struct Igeom;
struct Irtnode;
struct Ilibrary;
struct Itechnology;

#if defined(__cplusplus) && !defined(ALLCPLUSPLUS)
extern "C"
{
#endif

/* for internationalization */
#define _(String) (String)
#define N_(String) (String)
#define M_(String) (String)
#define textdomain(Domain)
#define bindtextdomain(Package, Directory)

/*************************** VARIABLES ****************************/

#define NOVARIABLE  ((VARIABLE *)-1)

/*
 * when adding objects to this list, also update:
 *   "dbchange.c:db_namevariable()"
 *   "dblang*"
 *   "dbvars.c"
 *   "dbtext.c:db_makestringvar()"
 *   "usrcom.c:us_nextvars()"
 *   "usrstatus.c:us_vartypename[]"
 */
/* the meaning of VARIABLE->type */
#define VUNKNOWN                   0		/* undefined variable */
#define VINTEGER                  01		/* 32-bit integer variable */
#define VADDRESS                  02		/* unsigned address */
#define VCHAR                     03		/* character variable */
#define VSTRING                   04		/* string variable */
#define VFLOAT                    05		/* floating point variable */
#define VDOUBLE                   06		/* double-precision floating point */
#define VNODEINST                 07		/* nodeinst pointer */
#define VNODEPROTO               010		/* nodeproto pointer */
#define VPORTARCINST             011		/* portarcinst pointer */
#define VPORTEXPINST             012		/* portexpinst pointer */
#define VPORTPROTO               013		/* portproto pointer */
#define VARCINST                 014		/* arcinst pointer */
#define VARCPROTO                015		/* arcproto pointer */
#define VGEOM                    016		/* geometry pointer */
#define VLIBRARY                 017		/* library pointer */
#define VTECHNOLOGY              020		/* technology pointer */
#define VAID                     021		/* aid pointer */
#define VRTNODE                  022		/* R-tree pointer */
#define VFRACT                   023		/* fractional integer (scaled by WHOLE) */
#define VNETWORK                 024		/* network pointer */
#define VCELL                    025		/* cell pointer */
#define VVIEW                    026		/* view pointer */
#define VWINDOWPART              027		/* window partition pointer */
#define VGRAPHICS                030		/* graphics object pointer */
#define VSHORT                   031		/* 16-bit integer */
#define VCONSTRAINT              032		/* constraint solver */
#define VGENERAL                 033		/* general address/type pairs (used only in fixed-length arrays) */
#define VWINDOWFRAME             034		/* window frame pointer */
#define VPOLYGON                 035		/* polygon pointer */
#define VTYPE                    037		/* all above type fields */
#define VCODE1                   040		/* variable is interpreted code (with VCODE2) */
#define VDISPLAY                0100		/* display variable (uses textdescript field) */
#define VISARRAY                0200		/* set if variable is array of above objects */
#define VCREF                   0400		/* variable points into C structure */
#define VLENGTH          03777777000		/* array length (0: array is -1 terminated) */
#define VLENGTHSH                  9		/* right shift for VLENGTH */
#define VCODE2           04000000000		/* variable is interpreted code (with VCODE1) */
#define VLISP                 VCODE1		/* variable is LISP */
#define VTCL                  VCODE2		/* variable is TCL */
#define VJAVA          VCODE1|VCODE2		/* variable is Java */
#define VDONTSAVE       010000000000		/* set to prevent saving on disk */
#define VCANTSET        020000000000		/* set to prevent changing value */

/* the meaning of VARIABLE->textdescript */
#define VTPOSITION               017		/* position of text relative to point */
#define VTPOSCENT                  0		/* text centered about point */
#define VTPOSUP                    1		/* text centered above point */
#define VTPOSDOWN                  2		/* text centered below point */
#define VTPOSLEFT                  3		/* text centered to left of point */
#define VTPOSRIGHT                 4		/* text centered to right of point */
#define VTPOSUPLEFT                5		/* text centered to upper-left of point */
#define VTPOSUPRIGHT               6		/* text centered to upper-right of point */
#define VTPOSDOWNLEFT              7		/* text centered to lower-left of point */
#define VTPOSDOWNRIGHT             8		/* text centered to lower-right of point */
#define VTPOSBOXED                 9		/* text centered and limited to object size */
#define VTDISPLAYPART            060		/* bits telling what to display */
#define VTDISPLAYVALUE             0		/* just display the value */
#define VTDISPLAYNAME            020		/* just display the name */
#define VTDISPLAYNAMEVALUE       040		/* display the name and value */
#define VTSIZE                 01700		/* size of text (in points) */
#define VTSIZESH                   6		/* right shift of VTSIZE */
#define VTINTERIOR             02000		/* text only appears inside facet */
#define VTINHERIT              04000		/* set to inherit value from proto to inst */
#define VTXOFF              07770000		/* X offset of text */
#define VTXOFFSH                  12		/* right shift of VTXOFF */
#define VTXOFFNEG          010000000		/* set if X offset is negative */
#define VTYOFF          017760000000		/* Y offset of text */
#define VTYOFFSH                  22		/* right shift of VTYOFF */
#define VTYOFFNEG       020000000000		/* set if Y offset is negative */

typedef struct
{
	INTBIG    key;							/* library-specific key to this name */
	UINTBIG   type;							/* the type of variables (see above) */
	UINTBIG   textdescript;					/* nature of text that displays variable */
	INTBIG    addr;							/* contents of variable */
} VARIABLE;

extern char  **el_namespace;				/* names in name space */
extern INTBIG  el_numnames;					/* number of names in name space */

/* some keys to commonly used variable names */
extern INTBIG  el_node_name;				/* variable "NODE_name" */
extern INTBIG  el_arc_name;					/* variable "ARC_name" */
extern INTBIG  el_arc_radius;				/* variable "ARC_radius" */
extern INTBIG  el_trace;					/* variable "trace" */
extern INTBIG  el_facet_message;			/* variable "FACET_message" */
extern INTBIG  el_schematic_page_size;		/* variable "FACET_schematic_page_size" */
extern INTBIG  el_transistor_width;			/* variable "transistor_width" */
extern INTBIG  el_prototype_center;			/* variable "prototype_center" */
extern INTBIG  el_node_size_default;		/* variable "NODE_size_default" */
extern INTBIG  el_arc_width_default;		/* variable "ARC_width_default" */

/*************************** MEMORY ALLOCATION ****************************/

#define NOCLUSTER	((CLUSTER *)-1)

#define CLUSTERFILLING   1					/* set if no need to search entire cluster */

typedef struct Icluster
{
	INTBIG           address;				/* base address of this cluster */
	INTBIG           flags;					/* information bits about this cluster */
	char             clustername[30];		/* for debugging only */
	INTBIG           clustersize;			/* number of pages to allocate at a time */
	struct Icluster *nextcluster;			/* next in linked list */
} CLUSTER;

extern CLUSTER *el_tempcluster;				/* cluster for temporary allocation */
extern CLUSTER *db_cluster;					/* database general allocation */

/************************** COMMAND COMPLETION ***************************/

#define TEMPLATEPARS               5		/* maximum parameters in a template */
#define MAXPARS                   20		/* maximum parameters in a command */
#define NOCOMCOMP    ((COMCOMP *)-1)
#define NOKEYWORD    ((KEYWORD *)-1)
#define NOKEY              NOCOMCOMP		/* short form for null keyword */
#define TERMKEY      {NULL, 0, {NULL, NULL, NULL, NULL, NULL}}
#define NOTOPLIST     (us_patoplist)		/* no routine for the keyword table */
#define NONEXTLIST (us_panextinlist)		/* no routine for table slot */
#define NOPARAMS       (us_paparams)		/* no routine for table slot */
#define NOBACKUP     (us_pabackupto)		/* no routine for entry backing up */

/* bits in COMCOMP->interpret */
#define NOFILL                    01		/* if set, don't fill out keyword */
#define NOSHOALL                  02		/* if set, suppress options list when null */
#define INPUTOPT                  04		/* if set, this can be input option on popup */
#define INCLUDENOISE             010		/* if set, include "noise" option in list */
#define MULTIOPT                 020		/* if set, allow multiple menu picks */

/*
 * tables of command options use this structure
 */
typedef struct Ikeyword
{
	char            *name;					/* name of this command */
	INTSML           params;				/* number of parameters to command */
	struct Icomcomp *par[TEMPLATEPARS];		/* parameter types */
} KEYWORD;

/*
 * this structure defines the basic command parameter
 */
typedef struct Icomcomp
{
	KEYWORD *ifmatch;						/* list of keywords to search if it matches */
	INTSML (*toplist)(char**);				/* reset to top of list of keywords */
	char  *(*nextcomcomp)(void);			/* give next keyword in list */
	INTSML (*params)(char*, struct Icomcomp*[], char);	/* set parameters to keyword */
	void   (*backupto)(INTSML, char*[]);	/* backup to a keyword position */
	INTSML   interpret;						/* bits for interpretation */
	char    *breakchrs;						/* keyword separation characters */
	char    *noise;							/* describe list */
	char    *def;							/* default value */
} COMCOMP;

/*************************** AIDS ****************************/

#define NOAID	((AIDENTRY *)-1)

/* aid descriptors */
typedef struct Iaidentry
{
	char    *aidname;						/* name of aid */
	INTBIG   aidstate;						/* state of aid */
	INTSML   aidindex;						/* aid index */
	COMCOMP *parse;							/* parsing structure for aid direction */
	CLUSTER *cluster;						/* virtual memory cluster for this aid */

	void   (*init)(INTBIG*, char*[], struct Iaidentry*);	/* initialization */
	void   (*done)(void);									/* completion */
	INTSML (*setmode)(INTSML, char*[]);						/* user-command options */
	INTBIG (*request)(char*, va_list);						/* direct call options */
	void   (*examinenodeproto)(struct Inodeproto*);			/* to examine an entire facet at once */
	void   (*slice)(void);									/* time slice */

	void   (*startbatch)(struct Iaidentry*,INTSML);			/* start change broadcast */
	void   (*endbatch)(void);								/* end change broadcast */

	void   (*startobjectchange)(INTBIG, INTBIG);			/* broadcast that object about to be changed */
	void   (*endobjectchange)(INTBIG, INTBIG);				/* broadcast that object done being changed */

	void   (*modifynodeinst)(struct Inodeinst*,INTBIG,INTBIG,INTBIG,INTBIG,INTSML,INTSML);	/* broadcast modified nodeinst */
	void   (*modifyarcinst)(struct Iarcinst*,INTBIG,INTBIG,INTBIG,INTBIG,INTBIG,INTBIG);	/* broadcast modified arcinst */
	void   (*modifyportproto)(struct Iportproto*, struct Inodeinst*, struct Iportproto*);	/* broadcast modified portproto */
	void   (*modifynodeproto)(struct Inodeproto*);			/* broadcast modified nodeproto */
	void   (*modifydescript)(INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast modified descriptor */

	void   (*newobject)(INTBIG, INTBIG);					/* broadcast new object */
	void   (*killobject)(INTBIG, INTBIG);					/* broadcast deleted object */
	void   (*newvariable)(INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast new variable */
	void   (*killvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast deleted variable */
	void   (*modifyvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast modified array variable entry */
	void   (*insertvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);			/* broadcast inserted array variable entry */
	void   (*deletevariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* broadcast deleted array variable entry */

	void   (*readlibrary)(struct Ilibrary*);				/* broadcast newly read library */
	void   (*eraselibrary)(struct Ilibrary*);				/* broadcast library to be erased */
	void   (*writelibrary)(struct Ilibrary*, INTSML);		/* broadcast library writing */

	/* variables */
	VARIABLE *firstvar;						/* first variable in list */
	INTSML    numvar;						/* number of variables in list */
} AIDENTRY;

extern AIDENTRY  el_aids[];					/* defined in "aidtable.c" */
extern INTSML    el_maxaid;					/* current number of aids */
extern AIDENTRY *us_aid;					/* the User aid object */
extern AIDENTRY *io_aid;					/* the I/O aid object */
extern AIDENTRY *net_aid;					/* the Network aid object */

/* the meaning of AIDENTRY->aidstate */
#define AIDON                     01		/* set if aid is on */
#define AIDBG                     02		/* set if aid is running in background */
#define AIDFIX                    04		/* set if aid will fix errors */
#define AIDLANG                  010		/* set if aid is coded in interpretive language */
#define AIDINCREMENTAL           020		/* set if aid functions incrementally */
#define AIDANALYSIS              040		/* set if aid does analysis */
#define AIDSYNTHESIS            0100		/* set if aid does synthesis */

/*************************** VIEWS ****************************/

#define NOVIEW          ((VIEW *)-1)

/* the meaning of VIEW->viewstate */
#define TEXTVIEW                  01		/* view contains only text */
#define MULTIPAGEVIEW             02		/* view is one of multiple pages */
#define PERMANENTVIEW             04		/* view is statically defined and cannot be deleted */

typedef struct Iview
{
	char         *viewname;					/* name of this view */
	char         *sviewname;				/* short name of this view */
	struct Iview *nextview;					/* next in linked list */
	INTBIG        temp1, temp2;				/* working variables */
	INTBIG        viewstate;				/* information about the view */

	/* variables */
	VARIABLE     *firstvar;					/* first variable in list */
	INTSML        numvar;					/* number of variables in list */
} VIEW;

extern VIEW *el_views;						/* list of existing view */
extern VIEW *el_unknownview;				/* the unknown view */
extern VIEW *el_layoutview;					/* the layout view */
extern VIEW *el_schematicview;				/* the schematic view */
extern VIEW *el_iconview;					/* the icon view */
extern VIEW *el_simsnapview;				/* the simulation-snapshot view */
extern VIEW *el_skeletonview;				/* the skeleton view */
extern VIEW *el_compview;					/* the compensated view */
extern VIEW *el_vhdlview;					/* the VHDL view (text) */
extern VIEW *el_verilogview;				/* the Verilog view (text) */
extern VIEW *el_netlistview;				/* the netlist view, generic (text) */
extern VIEW *el_netlistnetlispview;			/* the netlist view, netlisp (text) */
extern VIEW *el_netlistalsview;				/* the netlist view, als (text) */
extern VIEW *el_netlistquiscview;			/* the netlist view, quisc (text) */
extern VIEW *el_netlistrsimview;			/* the netlist view, rsim (text) */
extern VIEW *el_netlistsilosview;			/* the netlist view, silos (text) */
extern VIEW *el_docview;					/* the documentation view (text) */

/*************************** CELLS ****************************/

#define NOCELL          ((CELL *)-1)

typedef struct Icell
{
	char              *cellname;			/* name of this cell */
	struct Icell      *nextcell;			/* next in list of cells */
	struct Inodeproto *firstincell;			/* first cell in list */
	struct Ilibrary   *lib;					/* library containing this cell */
	CLUSTER           *cluster;				/* virtual memory cluster for this cell */
	struct Inetwork   *freenetwork;			/* free nets in this cell */
	INTBIG             temp1, temp2;		/* working variables */

	/* variables */
	VARIABLE          *firstvar;			/* first variable in list */
	INTSML             numvar;				/* number of variables in list */
} CELL;

/*************************** NODE INSTANCES ****************************/

#define NONODEINST  ((NODEINST *)-1)

/* the meaning of NODEINST->userbits */
#define DEADN                     01		/* node is not in use */
#define NEXPAND                   04		/* if on, draw node expanded */
#define WIPED                    010		/* set if node not drawn due to wiping arcs */
#define NSHORT                   020		/* set if node is to be drawn shortened */
/*  used by database:           0140 */
/*  used by user:            0377600 */
#define NTECHBITS          037400000		/* technology-specific bits for primitives */
#define NTECHBITSSH               17		/* right-shift of NTECHBITS */

/* node instances */
typedef struct Inodeinst
{
	INTBIG               lowx, highx;		/* bounding X box of nodeinst */
	INTBIG               lowy, highy;		/* bounding Y box of nodeinst */
	INTSML               transpose;			/* nonzero to transpose before rotation */
	INTSML               rotation;			/* angle from normal (0 to 359) */
	struct Inodeproto   *proto;				/* current nodeproto */
	struct Inodeproto   *parent;			/* facet that contains this nodeinst */
	struct Inodeinst    *lastnodeinst;		/* list of instances in parent facet */
	struct Inodeinst    *nextnodeinst;
	struct Igeom        *geom;				/* relative geometry list pointer */
	struct Inodeinst    *lastinst;			/* list of others of this type */
	struct Inodeinst    *nextinst;
	struct Iportarcinst *firstportarcinst;	/* first portarcinst on node */
	struct Iportexpinst *firstportexpinst;	/* first portexpinst on node */
	UINTBIG              textdescript;		/* nature of text that displays facet name */

	/* change information */
	char                *changeaddr;		/* change associated with this nodeinst */
	INTBIG               changed;			/* clock entry for changes to this nodeinst */

	/* aid information */
	UINTBIG              userbits;			/* state flags */
	INTBIG               temp1, temp2;		/* temporaries */

	/* variables */
	VARIABLE            *firstvar;			/* first variable in list */
	INTSML               numvar;			/* number of variables in list */
} NODEINST;

/*************************** PORT ARC INSTANCES ****************************/

#define NOPORTARCINST ((PORTARCINST *)-1)

typedef struct Iportarcinst
{
	struct Iportproto   *proto;				/* portproto of this portarcinst */
	struct Iarcinst     *conarcinst;		/* arcinst connecting to this portarcinst */
	struct Iportarcinst *nextportarcinst;	/* next portarcinst in list */

	/* variables */
	VARIABLE            *firstvar;			/* first variable in list */
	INTSML               numvar;			/* number of variables in list */
} PORTARCINST;

/************************** PORT EXPORT INSTANCES ***************************/

#define NOPORTEXPINST ((PORTEXPINST *)-1)

typedef struct Iportexpinst
{
	struct Iportproto   *proto;				/* portproto of this portexpinst */
	struct Iportexpinst *nextportexpinst;	/* next portexpinst in list */
	struct Iportproto   *exportproto;		/* portproto on parent facet */

	/* variables */
	VARIABLE            *firstvar;			/* first variable in list */
	INTSML               numvar;			/* number of variables in list */
} PORTEXPINST;

/*************************** NODE PROTOTYPES ****************************/

#define NONODEPROTO ((NODEPROTO *)-1)

/* the meaning of NODEPROTO->userbits */
#define NODESHRINK                01		/* set if nonmanhattan instances shrink */
#define WANTNEXPAND               02		/* set if instances should be expanded */
#define NFUNCTION               0374		/* node function (from efunction.h) */
#define NFUNCTIONSH                2		/* right shift for NFUNCTION */
#define ARCSHRINK               0400		/* set if nonmanhattan arcs on this shrink */
#define ARCSWIPE               01000		/* set if instances can be wiped */
#define NSQUARE                02000		/* set if node is to be kept square in size */
#define HOLDSTRACE             04000		/* primitive can hold trace information */
#define REDOFACETNET          010000		/* set to reevaluate this facet's network */
#define WIPEON1OR2            020000		/* set to erase if connected to 1 or 2 arcs */
#define LOCKEDPRIM            040000		/* set if primitive is lockable (cannot move) */
#define NEDGESELECT          0100000		/* set if primitive is selectable by edge, not area */
/*  used by database:       01400000 */
#define NNOTUSED            02000000		/* set if not used (don't put in menu) */
#define NLOCKED             04000000		/* set if everything in facet is locked */
#define NILOCKED           010000000		/* set if instances in facet are locked */
#define INCELLLIBRARY      020000000		/* set if facet is part of a "cell library" */
/*  used by user:       017700000000 */

typedef struct Inodeproto
{
	/* used by all aids */
	char               *primname;			/* primitive nodeproto name */
	INTSML              primindex;			/* nonzero if primitive */
	INTBIG              lowx, highx;		/* bounds in X */
	INTBIG              lowy, highy;		/* bounds in Y */
	NODEINST           *firstinst;			/* first in list of instances of this kind */
	NODEINST           *firstnodeinst;		/* head of list of nodeinsts in this facet */
	struct Iarcinst    *firstarcinst;		/* head of list of arcinsts in this facet */
	struct Itechnology *tech;				/* technology with this nodeproto (prim only) */
	struct Inodeproto  *lastnodeproto;		/* next in library/technology list */
	struct Inodeproto  *nextnodeproto;		/* last in library/technology list */
	struct Iportproto  *firstportproto;		/* list of ports */
	struct Inodeproto  *cachedequivfacet;	/* cache of equivalent facet for quick port matching */

	CELL               *cell;				/* cell in which this facet belongs */
	VIEW               *cellview;			/* view of this facet */
	INTSML              version;			/* version number of this facet */
	struct Inodeproto  *lastversion;		/* earlier version of this facet */
	struct Inodeproto  *newestversion;		/* most recent version of this facet */
	struct Inodeproto  *nextincell;			/* next in list of facets in this cell */
	UINTBIG             creationdate;		/* date facet was created */
	UINTBIG             revisiondate;		/* date facet was last changed */

	struct Inetwork    *firstnetwork;		/* nets in this facet */
	struct Irtnode     *rtree;				/* top of geometric tree of objects in facet */

	/* change information */
	char               *changeaddr;			/* change associated with this facet */

	/* aid specific */
	unsigned            adirty;				/* "dirty" bit for each aid */
	UINTBIG             userbits;			/* state flags */
	INTBIG              temp1, temp2;		/* temporaries */

	/* variables */
	VARIABLE           *firstvar;			/* first variable in list */
	INTSML              numvar;				/* number of variables in list */
} NODEPROTO;

/*************************** PORT PROTOTYPES ****************************/

#define NOPORTPROTO ((PORTPROTO *)-1)

/* the meaning of PORTPROTO->userbits */
#define PORTANGLE               0777		/* angle of this port from node center */
#define PORTANGLESH                0		/* right shift of PORTANGLE field */
#define PORTARANGE           0377000		/* range of valid angles about port angle */
#define PORTARANGESH               9		/* right shift of PORTARANGE field */
#define PORTNET           0177400000		/* electrical net of primitive port (0-31) */
#define PORTNETSH                 17		/* right shift of PORTNET field */
#define PORTISOLATED      0200000000		/* set if arcs to this port do not connect */
#define PORTDRAWN         0400000000		/* set if this port should always be drawn */
#define BODYONLY         01000000000		/* set to exclude this port from the icon */
#define STATEBITS       036000000000		/* input/output/power/ground/clock state: */
#define CLKPORT          02000000000		/* un-phased clock port */
#define C1PORT           04000000000		/* clock phase 1 */
#define C2PORT           06000000000		/* clock phase 2 */
#define C3PORT          010000000000		/* clock phase 3 */
#define C4PORT          012000000000		/* clock phase 4 */
#define C5PORT          014000000000		/* clock phase 5 */
#define C6PORT          016000000000		/* clock phase 6 */
#define INPORT          020000000000		/* input port */
#define OUTPORT         022000000000		/* output port */
#define BIDIRPORT       024000000000		/* bidirectional port */
#define PWRPORT         026000000000		/* power port */
#define GNDPORT         030000000000		/* ground port */
#define REFOUTPORT      032000000000		/* bias-level reference output port */
#define REFINPORT       034000000000		/* bias-level reference input port */

typedef struct Iportproto
{
	struct Iarcproto **connects;			/* arc prototypes that can touch this port */
	NODEPROTO         *parent;				/* nodeproto that this portproto resides in */
	NODEINST          *subnodeinst;			/* subnodeinst that portproto comes from */
	PORTEXPINST       *subportexpinst;		/* portexpinst in subnodeinst */
	struct Iportproto *subportproto;		/* portproto in above subnodeinst */
	struct Iportproto *nextportproto;		/* next in list of port prototypes */
	char              *protoname;			/* name of this port prototype */
	UINTBIG            textdescript;		/* nature of text that displays proto name */
	struct Iportproto *cachedequivport;		/* cache of equivalent port in other facet */

	/* change information */
	char              *changeaddr;			/* change associated with this port */

	/* for the aids */
	UINTBIG            userbits;			/* state flags */
	struct Inetwork   *network;				/* network object within facet */
	INTBIG             temp1, temp2;		/* temporaries */

	/* variables */
	VARIABLE          *firstvar;			/* first variable in list */
	INTSML             numvar;				/* number of variables in list */
} PORTPROTO;

/*************************** ARC INSTANCES ****************************/

#define NOARCINST    ((ARCINST *)-1)

/* the meaning of ARCINST->userbits */
#define FIXED                     01		/* fixed-length arc */
#define FIXANG                    02		/* fixed-angle arc */
#define DEADA                    020		/* arc is not in use */
#define AANGLE                037740		/* angle of arc from end 0 to end 1 */
#define AANGLESH                   5		/* bits of right shift for AANGLE field */
#define ASHORT                040000		/* set if arc is to be drawn shortened */
#define NOEXTEND             0400000		/* set if ends do not extend by half width */
#define ISNEGATED           01000000		/* set if ends are negated */
#define ISDIRECTIONAL       02000000		/* set if arc aims from end 0 to end 1 */
#define NOTEND0             04000000		/* no extension/negation/arrows on end 0 */
#define NOTEND1            010000000		/* no extension/negation/arrows on end 1 */
#define REVERSEEND         020000000		/* reverse extension/negation/arrow ends */
#define CANTSLIDE          040000000		/* set if arc can't slide around in ports */
/*  used by database:     0100000000 */
/*  used by user:       037600000004 */

typedef struct Iarcinst
{
	/* physical description of arcinst */
	struct Iarcproto *proto;				/* arc prototype of this arcinst */
	INTBIG            length;				/* length of arcinst */
	INTBIG            width;				/* width of arcinst */
	INTBIG            endshrink;			/* shrinkage factor on ends */
	struct
	{
		INTBIG        xpos, ypos;			/* position of arcinst end */
		NODEINST     *nodeinst;				/* connecting nodeinst */
		PORTARCINST  *portarcinst;			/* portarcinst in the connecting nodeinst */
	} end[2];								/* for each end of the arcinst */
	struct Iarcinst  *lastarcinst;			/* list of arcs in the parent facet */
	struct Iarcinst  *nextarcinst;
	struct Igeom     *geom;					/* geometry entry */
	NODEPROTO        *parent;				/* parent facet */

	/* change information */
	char             *changeaddr;			/* change associated with this nodeinst */
	INTBIG            changed;				/* clock entry for changes to this nodeinst */

	/* aid specific */
	UINTBIG           userbits;				/* state flags */
	struct Inetwork  *network;				/* network object within facet */
	INTBIG            temp1, temp2;			/* temporaries */

	/* variables */
	VARIABLE         *firstvar;				/* first variable in list */
	INTSML            numvar;				/* number of variables in list */
} ARCINST;

/*************************** ARC PROTOTYPES ****************************/

#define NOARCPROTO  ((ARCPROTO *)-1)

/* the meaning of ARCPROTO->userbits */
#define WANTFIX                   01		/* these arcs are fixed-length */
#define WANTFIXANG                02		/* these arcs are fixed-angle */
#define WANTCANTSLIDE             04		/* set if arcs should not slide in ports */
#define WANTNOEXTEND             010		/* set if ends do not extend by half width */
#define WANTNEGATED              020		/* set if arcs should be negated */
#define WANTDIRECTIONAL          040		/* set if arcs should be directional */
#define CANWIPE                 0100		/* set if arcs can wipe wipable nodes */
#define CANCURVE                0200		/* set if arcs can curve */
#define AFUNCTION             017400		/* arc function (from efunction.h) */
#define AFUNCTIONSH                8		/* right shift for AFUNCTION */
#define AANGLEINC          017760000		/* angle increment for this type of arc */
#define AANGLEINCSH               13		/* right shift for AANGLEINC */
#define AEDGESELECT        020000000		/* set if arc is selectable by edge, not area */
/*  used by user:       017740000000 */

typedef struct Iarcproto
{
	/* information for all aids */
	char               *protoname;			/* full arcproto name */
	INTBIG              nominalwidth;		/* default width of arcs */
	INTSML              arcindex;			/* index number of arcproto in technology */
	struct Itechnology *tech;				/* technology this arcproto is in */
	struct Iarcproto   *nextarcproto;		/* next in list for this technology */

	/* information for specific aid */
	UINTBIG             userbits;			/* state flags */
	INTBIG              temp1, temp2;		/* temporaries */

	/* variables */
	VARIABLE           *firstvar;			/* first variable in list */
	INTSML              numvar;				/* number of variables in list */
} ARCPROTO;

/**************************** NETWORKS ****************************/

#define NONETWORK    ((NETWORK *)-1)

typedef struct Inetwork
{
	char             *netname;				/* name of this network */
	INTSML            namecount;			/* number of names */
	INTSML            tempname;				/* nonzero if name is temporary (for back annotation) */
	INTBIG            arcaddr;				/* address of arc(s) on this network */
	INTSML            arccount;				/* number of arcs on this network */
	INTSML            refcount;				/* number of arcs on network */
	INTSML            portcount;			/* number of ports on this network */
	INTSML            buslinkcount;			/* number of busses referencing this network */
	NODEPROTO        *parent;				/* facet that has this network */
	INTSML            signals;				/* width of bus or index into bus */
	struct Inetwork **networklist;			/* list of single-wire networks on bus */
	struct Inetwork  *nextnetwork;			/* next in linked list */
	struct Inetwork  *lastnetwork;			/* previous in linked list */
	INTBIG            temp1, temp2;			/* temporaries */

	/* variables */
	VARIABLE         *firstvar;				/* first variable in list */
	INTSML            numvar;				/* number of variables in list */
} NETWORK;

/**************************** GEOMETRY POINTERS ****************************/

#define NOGEOM          ((GEOM *)-1)

/* the meaning of GEOM->entrytype */
#define OBJNODEINST                1		/* entrytype value for nodes */
#define OBJARCINST                 2		/* entrytype value for arcs */

/*
 * Each nodeinst and arcinst points to a geometry module.  The module points
 * back to the nodeinst or arcinst and describes the position relative to
 * other nodes and arcs.  Geometry modules are also at the leaf positions
 * in R-trees, which organize them spatially.
 */
typedef struct Igeom
{
	INTSML         entrytype;				/* OBJNODEINST, OBJARCINST, etc. */
	union          u_entry
	{
		NODEINST  *ni;
		ARCINST   *ai;
		void      *blind;
	} entryaddr;							/* pointer to the element */
	INTBIG         lowx, highx;				/* horizontal bounds */
	INTBIG         lowy, highy;				/* vertical bounds */

	/* variables */
	VARIABLE      *firstvar;				/* first variable in list */
	INTSML         numvar;					/* number of variables in list */
} GEOM;

#define MINRTNODESIZE              4		/* lower bound on R-tree node size */
#define MAXRTNODESIZE (MINRTNODESIZE*2)

#define NORTNODE      ((RTNODE *)-1)

typedef struct Irtnode
{
	INTBIG          lowx, highx;			/* X bounds of this node */
	INTBIG          lowy, highy;			/* Y bounds of this node */
	INTSML          total;					/* number of pointers */
	INTSML          flag;					/* nonzero if children are terminal */
	UINTBIG         pointers[MAXRTNODESIZE];/* pointers */
	struct Irtnode *parent;					/* parent node */

	/* variables */
	VARIABLE       *firstvar;				/* first variable in list */
	INTSML          numvar;					/* number of variables in list */
} RTNODE;

/*************************** LIBRARIES ****************************/

#define NOLIBRARY    ((LIBRARY *)-1)

/* the meaning of LIBRARY->userbits */
#define LIBCHANGEDMAJOR           01		/* library has changed significantly */
#define REDOFACETLIB              02		/* recheck networks in library */
#define READFROMDISK              04		/* set if library came from disk */
#define LIBUNITS                 070		/* internal units in library (see INTERNALUNITS) */
#define LIBUNITSSH                 3		/* right shift for LIBUNITS */
#define LIBCHANGEDMINOR         0100		/* library has changed insignificantly */
#define HIDDENLIBRARY           0200		/* library is "hidden" (clipboard library) */

typedef struct Ilibrary
{
	NODEPROTO       *firstnodeproto;		/* list of nodeprotos in this library */
	NODEPROTO       *tailnodeproto;			/* end of list of nodeprotos in this library */
	CELL            *firstcell;				/* list of cells in this library */
	CELL            *tailcell;				/* end of list of cells in this library */
	INTBIG           numcells;				/* number of cells in this library */
	CELL           **cellhashtable;			/* hash table of cells in this library */
	INTBIG           cellhashtablesize;		/* size of cell hash table */
	char            *libname;				/* name of this library */
	char            *libfile;				/* file this library comes from */
	INTBIG          *lambda;				/* half-millimicrons per unit in this library */
	NODEPROTO       *curnodeproto;			/* top facet of this library (if any) */
	struct Ilibrary *nextlibrary;			/* next library in list */
	UINTBIG          userbits;				/* state flags */
	INTBIG           temp1, temp2;			/* temporaries */
	CLUSTER         *cluster;				/* memory cluster for whole library */

	/* variables */
	VARIABLE        *firstvar;				/* first variable in list */
	INTSML           numvar;				/* number of variables in list */
} LIBRARY;

extern LIBRARY  *el_curlib;					/* pointer to current library (list head) */

/*************************** GRAPHICS and POLYGONS ****************************/

#define NOGRAPHICS  ((GRAPHICS *)-1)

typedef struct
{
	INTSML    bits;							/* bit planes to use (color displays) */
	INTSML    col;							/* color to draw */
	INTSML    colstyle;						/* drawing style for color displays */
	INTSML    bwstyle;						/* drawing style for B&W displays */
	UINTSML   raster[8];					/* 16x8 raster pattern (for stippling) */

	/* variables */
	VARIABLE *firstvar;						/* first variable in list */
	INTSML    numvar;						/* number of variables in list */
} GRAPHICS;


#define NOPOLYGON    ((POLYGON *)-1)

typedef struct Ipolygon
{
	INTBIG             *xv, *yv;			/* the polygon coordinates */
	INTSML              limit;				/* maximum number of points in polygon */
	INTSML              count;				/* current number of points in polygon */
	CLUSTER            *cluster;			/* virtual memory cluster with this polygon */
	INTSML              style;				/* the polygon style */
	char               *string;				/* string (if text graphics) */
	INTSML              font;				/* font type (if text graphics) */
	struct Itechnology *tech;				/* technology (if text graphics) */
	GRAPHICS           *desc;				/* graphical description of polygon */
	INTSML              layer;				/* layer number of this polygon */
	PORTPROTO          *portproto;			/* port prototype associated with this polygon */
	struct Ipolygon    *nextpolygon;		/* for linking them into a list */

	/* variables */
	VARIABLE *firstvar;						/* first variable in list */
	INTSML    numvar;						/* number of variables in list */
} POLYGON;

/******************** EDITOR MODULES ********************/

#define NOEDITOR      ((EDITOR *)-1)

/* the meaning of EDITOR->state */
#define EDITORTYPE                07		/* type of editor */
#define PACEDITOR                 01		/* editor is point-and-click */
#define EMACSEDITOR               02		/* editor is EMACS-like */
#define EDITORINITED             010		/* set if editor has been initialized */
#define EGRAPHICSOFF             020		/* set if editor graphics is disabled */
#define LINESFIXED               040		/* set if editor disallows line insertion/deletion */
#define TEXTTYPING              0100		/* set if typing into the editor (text undo) */
#define TEXTTYPED               0200		/* set if typed into the editor (text undo) */

/* an overloaded entry */
#define charposition highlightedline

typedef struct Ieditor
{
	INTSML    state;						/* status bits (see above) */
	INTSML    swid, shei;					/* screen width and height */
	INTSML    revy;							/* for reversing Y coordinates */
	INTSML    offx;							/* for adjusting X coordinates */
	INTBIG    maxlines;						/* maximum lines in buffer */
	INTBIG    linecount;					/* actual number of lines in buffer */
	INTSML    screenlines;					/* number of lines on screen */
	INTBIG    working;						/* line number being changed */
	INTBIG    firstline;					/* top line on screen */
	INTSML    screenchars;					/* maximum characters across screen */
	INTSML    mostchars;					/* most characters on any line */
	INTBIG    curline, curchar;				/* current start position */
	INTBIG    endline, endchar;				/* current end position */
	INTBIG    highlightedline;				/* line being highlighted (EMACS only) */
	INTSML    horizfactor;					/* position of 1st character on left side */
	INTSML    vthumbpos;					/* Y position of vertical thumb slider */
	INTSML    hthumbpos;					/* X position of horizontal thumb slider */
	INTSML    dirty;						/* nonzero if buffer has changed */
	INTSML   *maxchars;						/* number of characters in each line */
	char     *formerline;					/* line being changed: former state */
	char    **textarray;					/* image of screen contents */
	INTBIG    savedbox;						/* where the bits are kept for popups */
	char     *header;						/* header text */
	char     *editobjaddr;					/* variable editing: parent object address */
	INTBIG    editobjtype;					/* variable editing: parent object type */
	char     *editobjqual;					/* variable editing: attribute name on parent */
	VARIABLE *editobjvar;					/* variable editing: actual variable */
	struct Ieditor *nexteditor;				/* next in linked list */
} EDITOR;

/******************** WINDOW FRAME MODULES ********************/

#define NOWINDOWFRAME ((WINDOWFRAME *)-1)

#ifdef ONUNIX
typedef struct
{
	XImage              *image;				/* the offscreen buffer that matches the visual */
	int                  wid, hei;			/* size of the offscreen buffer */
	INTBIG               depth;				/* the depth of the screen */
	INTBIG               truedepth;			/* the depth of the buffer */
	INTBIG               colorvalue[256];	/* the color map for 8-bit depth */
	char                *addr;				/* the data in "world" */
	char               **rowstart;			/* pointers to "addr" */
} WINDOWDISPLAY;
#endif

typedef struct IWindowFrame
{
#ifdef MACOS
	GWorldPtr            window;			/* the offscreen buffer for the editing window */
	CGrafPtr             realwindow;		/* the real window */
	void /*Tk_Window*/  *tkwin;				/* the TK window placeholder */
#endif
#ifdef WIN32
	void /*CChildFrame*/*wndframe;			/* address of Windows subframe */
	HWND                 realwindow;		/* the real window */
	char                *data;				/* raster data for window */
	void /*CDC*/        *hDC;				/* device context for real window */
	HBITMAP              hBitmap;			/* 8-bit offscreen bitmap */
	BITMAPINFO          *bminfo;			/* bit map info structure */
	void /*CFont*/      *hFont;				/* current font in bitmap */
	void /*CDC*/        *hDCOff;			/* device context for the offscreeen bitmap */
	void /*CPalette*/   *hPalette;			/* color information */
	LOGPALETTE          *pColorPalette;		/* color information */
#endif
#ifdef ONUNIX
 	Widget               toplevelwidget;	/* widget for top-level frame */
	Widget               intermediategraphics;	/* intermediate graphics widget */
	Widget               graphicswidget;	/* drawing widget */
	Widget               menubar;			/* menu in frame */
	Widget               firstmenu;			/* first menu (dummy placeholder) */
	Display             *topdpy;			/* display of frame */
	WINDOWDISPLAY       *wd;				/* structure with offscreen data */
	Window               topwin;			/* window of top-level frame */
	Window               win;				/* window of graphics widget */
	INTSML               pulldownmenucount;	/* number of pulldown menus */
	Widget              *pulldownmenus;		/* the current pulldown menu handles */
	INTSML              *pulldownmenusize;	/* the number of entries in a menu */
	Widget             **pulldownmenulist;	/* the entries in a menu */
	struct Ipopupmenu  **pulldowns;			/* the current pulldown menus */
	GC                   gc;				/* graphics context for drawing */
	GC                   gcinv;				/* graphics context for inverting */
	GC                   gcstip;			/* graphics context for stippling */
	Colormap             colormap;			/* color map */
	char                *dataaddr8;			/* offscreen 8-bit data */
	INTSML               trueheight;		/* full screen dimensions including status */
#endif
	INTSML               windindex;			/* identifying index of this frame */
	INTSML               floating;			/* nonzero for floating palette window */
	INTSML               swid, shei;		/* screen dimensions for drawing */
	INTSML               revy;				/* for reversing Y coordinates */
	UINTBIG              starttime;			/* time when this window was refreshed from offscreen */
	INTSML               offscreendirty;	/* nonzero if offscreen area is "dirty" */
	RECTAREA             copyrect;			/* rectangle to copy to screen */
	INTSML               screenleft,		/* left bound of editing window screen */
                         screenright,		/* right bound of editing window screen */
                         screentop,			/* top bound of editing window screen */
                         screenbottom;		/* bottom bound of editing window screen */
	char               **rowstart;			/* start of each row */
	struct IWindowFrame *nextwindowframe;	/* next in list */

	/* variables */
	VARIABLE       *firstvar;				/* first variable in list */
	INTSML          numvar;					/* number of variables in list */
} WINDOWFRAME;

extern WINDOWFRAME  *el_firstwindowframe;
extern WINDOWFRAME  *el_curwindowframe;		/* current window frame */

/******************** WINDOW PARTITION MODULES ********************/

#define NOWINDOWPART      ((WINDOWPART *)-1)

/* the meaning of WINDOWPART->state */
#define GRIDON                     1		/* grid is to be displayed */
#define GRIDTOOSMALL               2		/* grid is too small to display */
#define WINDOWTYPE               070		/* type of this window */
#define DISPWINDOW               010		/* this is a normal display window */
#define TEXTWINDOW               020		/* this is a text editor window */
#define POPTEXTWINDOW            030		/* this is a popup text editor window */
#define WAVEFORMWINDOW           040		/* this is a signal waveform window */
#define DISP3DWINDOW             050		/* this is a 3-D display window */
#define OVERVIEWWINDOW           060		/* this is an overview window */
#define EXPLORERWINDOW           070		/* this is an "explorer" window */
#define HASOWNFRAME             0100		/* set if window was in its own frame */
#define WINDOWSIMULATING        0200		/* set if window is simulating */

#define SIMULATINGBORDERSIZE       3		/* size of red border when simulating */
#define DISPLAYSLIDERSIZE         10		/* size of sliders on the right and bottom */

/* the meaning of the second argument to "WINDOWPART->changehandler" */
#define REPLACETEXTLINE            1
#define DELETETEXTLINE             2
#define INSERTTEXTLINE             3
#define REPLACEALLTEXT             4

typedef struct
{
	float     eye[3];				/* location of the viewer */
	float     view[3];				/* what the viewer is looking at */
	float     up[3];				/* which way is up */
	float     fieldofview;			/* field of view (in degrees) */
	float     nearplane, farplane;	/* near and far clipping planes */
	float     screenx, screeny;		/* screen width and height */
	float     aspect;				/* aspect ratio */
	float     xform[4][4];			/* overall transformation matrix */
} XFORM3D;

typedef struct Iwindowpart
{
	INTSML              uselx, usehx;		/* X: bounds of window drawing area */
	INTSML              usely, usehy;		/* Y: bounds of window drawing area */
	INTBIG              screenlx,screenhx;	/* X: low and high of window */
	INTBIG              screenly,screenhy;	/* Y: low and high of window */
	INTSML              framelx, framehx;	/* X: bounds of window frame (global coords) */
	INTSML              framely, framehy;	/* Y: bounds of window frame (global coords) */
	INTSML              thumblx, thumbhx;	/* X: bounds of slider thumb (DISPWINDOW) */
	INTSML              thumbly, thumbhy;	/* Y: bounds of slider thumb (DISPWINDOW) */
	INTSML              hratio, vratio;		/* percentage of overall when split */
	float               scalex, scaley;		/* X and Y scale from window to drawing area */
	XFORM3D             xf3;				/* 3D information (if DISP3DWINDOW) */
	NODEPROTO          *curnodeproto;		/* facet in window */
	INTBIG              gridx,gridy;		/* size of grid in window */
	INTBIG              state;				/* miscellaneous state bits (see above) */
	char               *location;			/* string describing location */
	WINDOWFRAME        *frame;				/* window frame that contains this */
	EDITOR             *editor;				/* editor running in this window */
	struct Iwindowpart *linkedwindowpart;	/* window associated with this */
	INTSML            (*charhandler)(struct Iwindowpart*, INTSML);							/* routine for characters in window */
	void              (*buttonhandler)(struct Iwindowpart*, INTSML, INTSML, INTSML);		/* routine for buttons in window */
	void              (*changehandler)(struct Iwindowpart*, INTSML, char*, char*, INTBIG);	/* routine for changes to window */
	void              (*termhandler)(struct Iwindowpart*);									/* routine for termination of window */
	void              (*redisphandler)(struct Iwindowpart*);								/* routine for redisplay of window */
	struct Iwindowpart *nextwindowpart;		/* next window in list */
	struct Iwindowpart *lastwindowpart;		/* previous window in list */

	/* variables */
	VARIABLE           *firstvar;			/* first variable in list */
	INTSML              numvar;				/* number of variables in list */
} WINDOWPART;

extern WINDOWPART *el_topwindowpart;		/* top window in list */
extern WINDOWPART *el_curwindowpart;		/* current window */

/*************************** TECHNOLOGIES ****************************/

/* ===== LAYER DESCRIPTIONS ===== */

/* definition of a color value */
typedef struct Itech_colormap
{
	INTSML red, green, blue;				/* color values from 0 to 255 */
} TECH_COLORMAP;

/* the meaning of TECH_POLYGON->representation */
#define POINTS                     0		/* list of scalable points */
#define BOX                        1		/* a rectangle */
#define ABSPOINTS                  2		/* list of absolute points */
#define MINBOX                     3		/* minimum sized rectangle */

/* the structure for layers of a node prototype */
typedef struct Itech_polygon
{
	INTSML  layernum;						/* drawing layer in technology */
	INTSML  portnum;						/* the port number in this technology */
	INTSML  count;							/* number of points in polygon */
	INTSML  style;							/* polygon style */
	INTSML  representation;					/* see above list */
	INTBIG *points;							/* data list */
} TECH_POLYGON;

/* ===== ARC DESCRIPTIONS ===== */

/* the structure for layers of an arc prototype */
typedef struct Itech_arclay
{
	INTSML lay;								/* index of this layer */
	INTBIG off;								/* width offset for this layer */
	INTSML style;							/* polygon style */
} TECH_ARCLAY;

/* the structure for an arc prototype */
typedef struct Itech_arcs
{
	char        *arcname;					/* layer name */
	INTBIG       arcwidth;					/* default layer width */
	INTSML       arcindex;					/* index of this arcinst */
	INTSML       laycount;					/* number of layers */
	TECH_ARCLAY *list;						/* list of layers that compose arc */
	UINTBIG      initialbits;				/* initial userbits for this arcproto */
} TECH_ARCS;

/* ===== PORT DESCRIPTIONS ===== */

/* the structure for ports of a node prototype */
typedef struct Itech_ports
{
	INTBIG     *portarcs;					/* allowable arcs (list ends with -1) */
	char      *protoname;					/* name of this port */
	PORTPROTO *addr;						/* address used by later routines */
	UINTBIG    initialbits;					/* initial userbits for this port */
	INTSML     lowxmul,  lowxsum;			/* defines low X of portinst area */
	INTSML     lowymul,  lowysum;			/* defines low Y of portinst area */
	INTSML     highxmul, highxsum;			/* defines high X of portinst area */
	INTSML     highymul, highysum;			/* defines high Y of portinst area */
} TECH_PORTS;

/* ===== NODE DESCRIPTIONS ===== */

/* the structure for serpentine MOS transistor description */
typedef struct Itech_serpent
{
	TECH_POLYGON basics;					/* the basic information */
	INTSML       lwidth;					/* the extension of width on the left */
	INTSML       rwidth;					/* the extension of width on the right */
	INTSML       extend;					/* the extension of length on the ends */
} TECH_SERPENT;

/* the meaning of TECH_NODES->special */
#define SERPTRANS                  1		/* serpentine transistor */
#define POLYGONAL                  2		/* polygonally defined transistor */
#define MULTICUT                   3		/* multi-cut contact */
#define MOSTRANS                   4		/* MOS transistor (nonserpentine) */

/* the structure for a node prototype */
typedef struct Itech_nodes
{
	char         *nodename;
	INTSML        nodeindex;
	NODEPROTO    *creation;
	INTBIG        xsize, ysize;
	INTSML        portcount;
	TECH_PORTS   *portlist;
	INTSML        layercount;
	TECH_POLYGON *layerlist;
	INTBIG        initialbits;
	INTSML        special;
	INTSML        f1, f2, f3, f4, f5, f6;
	TECH_SERPENT *gra, *ele;
} TECH_NODES;

/* ===== VARIABLE DESCRIPTIONS ===== */

/* the structure for a variable */
typedef struct Itech_variables
{
	char *name;								/* attribute name to set */
	char  *value;							/* initial (address/integer) value for attribute */
	float  fvalue;							/* initial (floating) value for attribute */
	INTBIG type;							/* initial type of attribute */
} TECH_VARIABLES;

/* ===== MAIN STRUCTURE ===== */

#define NOTECHNOLOGY ((TECHNOLOGY *)-1)

/* the meaning of TECHNOLOGY->userbits */
#define NONELECTRICAL             01		/* technology is not electrical  */
#define NODIRECTIONALARCS         02		/* has no directional arcs */
#define NONEGATEDARCS             04		/* has no negated arcs */
#define NONSTANDARD              010		/* nonstandard technology (cannot be edited) */
#define STATICTECHNOLOGY         020		/* statically allocated (don't deallocate memory) */

/* scaling factor for fixed-point numbers (used in technologies) */
#define WHOLE                    120

typedef struct Itechnology
{
	char               *techname;			/* name of this technology */
	INTSML              techindex;			/* 0-based index of this technology */
	INTBIG              deflambda;			/* the default size of a unit */
	NODEPROTO          *firstnodeproto;		/* list of primitive nodeprotos */
	ARCPROTO           *firstarcproto;		/* pointer to type description */
	VARIABLE           *firstvar;			/* first variable in list */
	INTSML              numvar;				/* number of variables in list */
	COMCOMP            *parse;				/* parsing structure for tech direction */
	CLUSTER            *cluster;			/* virtual memory cluster for technology */
	char               *techdescript;		/* description of this technology */

	INTSML              layercount;			/* number of layers */
	GRAPHICS          **layers;				/* layer descriptions */
	INTSML              arcprotocount;		/* number of arcs */
	TECH_ARCS         **arcprotos;			/* raw data for arcs */
	INTSML              nodeprotocount;		/* number of nodes */
	TECH_NODES        **nodeprotos;			/* raw data for nodes */
	TECH_VARIABLES     *variables;			/* variable descriptions */

	INTSML            (*init)(struct Itechnology*, INTSML);			/* process initialization */
	void              (*term)(void);								/* completion */
	void              (*setmode)(INTSML, char *[]);					/* set operation mode */
	INTBIG            (*request)(char*, va_list);					/* direct call options */

	/* node description */
	INTBIG            (*nodepolys)(NODEINST*, INTBIG*, WINDOWPART*);	/* returns total real polygons in node */
	INTBIG            (*nodeEpolys)(NODEINST*, INTBIG*, WINDOWPART*);	/* returns total electrical polys in node */
	void              (*shapenodepoly)(NODEINST*, INTBIG, POLYGON*);	/* returns real polygon shape in node */
	void              (*shapeEnodepoly)(NODEINST*, INTBIG, POLYGON*);	/* returns electrical polygon shape in node */
	void              (*nodesizeoffset)(NODEINST*, INTBIG*, INTBIG*, INTBIG*, INTBIG*);	/* returns offset to true size */

	/* node port description */
	void              (*shapeportpoly)(NODEINST*, PORTPROTO*, POLYGON*, XARRAY, INTSML);/* returns port shape on a node */

	/* arc description */
	INTBIG            (*arcpolys)(ARCINST*, WINDOWPART*);			/* returns total polygons in arc */
	void              (*shapearcpoly)(ARCINST*, INTBIG, POLYGON*);	/* returns polygon shape in arc */
	INTBIG            (*arcwidthoffset)(ARCINST*);					/* returns offset to true width */

	struct Itechnology *nexttechnology;		/* next in linked list */
	UINTBIG             userbits;			/* state flags */
	INTBIG              temp1, temp2;		/* temporaries */
} TECHNOLOGY;

extern TECHNOLOGY *el_technologies;			/* defined in "tectable.c" */
extern TECHNOLOGY *el_curtech;				/* pointer to current technology */
extern INTSML      el_maxtech;				/* current number of technologies */

/*************************** CONSTRAINT SYSTEMS ****************************/

#define NOCONSTRAINT ((CONSTRAINT *)-1)

typedef struct Iconstraint
{
	char     *conname;						/* constraint system name */
	char     *condesc;						/* constraint system description */
	COMCOMP  *parse;						/* parsing structure for constraint direction */
	CLUSTER  *cluster;						/* virtual memory cluster for constraint */
	VARIABLE *firstvar;						/* first variable in list */
	INTSML    numvar;						/* number of variables in list */
	void    (*init)(struct Iconstraint*);	/* initialize a constraint system */
	void    (*term)(void);					/* terminate a constraint system */
	void    (*setmode)(INTSML, char*[]);	/* user-command options */
	INTBIG  (*request)(char*, INTBIG);		/* direct call options */
	void    (*solve)(NODEPROTO*);			/* solve a batch of changes */

	void    (*newobject)(INTBIG,INTBIG);				/* constraint newly created object */
	void    (*killobject)(INTBIG,INTBIG);				/* constraint deleted object */
	INTSML  (*setobject)(INTBIG,INTBIG,INTBIG,INTBIG);	/* set constraint properties on object */

	void    (*modifynodeinst)(NODEINST*,INTBIG,INTBIG,INTBIG,INTBIG,INTSML,INTSML);		/* constrain a modified node instance */
	void    (*modifyarcinst)(ARCINST*, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* constrain a modified arc instance */
	void    (*modifyportproto)(PORTPROTO*, NODEINST*, PORTPROTO*);				/* constrain a modified port prototype */
	void    (*modifynodeproto)(NODEPROTO*);										/* constrain a modified node prototype */
	void    (*modifydescript)(INTBIG, INTBIG, INTBIG, INTBIG);					/* constrain a modified descriptor */

	void    (*newlib)(LIBRARY*);			/* constrain a newly created library */
	void    (*killlib)(LIBRARY*);			/* constrain a deleted library */

	void    (*newvariable)(INTBIG, INTBIG, INTBIG, INTBIG);						/* constrain a newly created variable */
	void    (*killvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* constrain a deleted variable */
	void    (*modifyvariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);	/* constrain a modified array variable entry */
	void    (*insertvariable)(INTBIG, INTBIG, INTBIG, INTBIG);					/* constrain an inserted array variable entry */
	void    (*deletevariable)(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);			/* constrain a deleted array variable entry */
	void    (*setvariable)(void);												/* set constraint properties on variable */
} CONSTRAINT;

extern CONSTRAINT *el_curconstraint;		/* current constraint solver */
extern CONSTRAINT  el_constraints[];		/* list of constraint solvers */

/*************************** MISCELLANEOUS ****************************/

#define EPI               3.14159265		/* Pi */
#define NOSTRING        ((char *)-1)

/* the meaning of "el_units" */
#define DISPLAYUNITS             017		/**** the display units ****/
#define DISPUNITLAMBDA             0		/* lambda display (12.3) */
#define DISPUNITINCH              01		/* inch display (12.3") */
#define DISPUNITCM                02		/* centimeter display (12.3cm) */
#define DISPUNITMM                04		/* millimeter display (12.3mm) */
#define DISPUNITMIL              010		/* mil display (12.3mil) */
#define DISPUNITMIC              011		/* micron display (12.3u) */
#define DISPUNITCMIC             012		/* centimicron display (12.3cu) */
#define DISPUNITMMIC             013		/* millimicron display (12.3mu) */
#define INTERNALUNITS           0160		/**** the internal units ****/
#define INTERNALUNITSSH            4		/* right shift for INTERNALUNITS */
#define INTUNITHMMIC               0		/* half-millimicron units */
#define INTUNITHDMIC             020		/* half-decimicron units */

/*
 * reasons for aborting (the argument to "stopping()")
 * (update table "dbtext.c:db_stoppingreason[]" when changing this)
 */
#define STOPREASONCONTOUR   0		/* "Contour gather" */
#define STOPREASONDRC       1		/* "DRC" */
#define STOPREASONPLAYBACK  2		/* "Playback" */
#define STOPREASONBINARY    3		/* "Binary" */
#define STOPREASONCIF       4		/* "CIF" */
#define STOPREASONDXF       5		/* "DXF" */
#define STOPREASONEDIF      6		/* "EDIF" */
#define STOPREASONVHDL      7		/* "VHDL" */
#define STOPREASONCOMPACT   8		/* "Compaction" */
#define STOPREASONERC       9		/* "ERC" */
#define STOPREASONCHECKIN  10		/* "Check-in" */
#define STOPREASONLOCK     11		/* "Lock wait" */
#define STOPREASONNCC      12		/* "Network comparison" */
#define STOPREASONPORT     13		/* "Port exploration" */
#define STOPREASONROUTING  14		/* "Routing" */
#define STOPREASONSILCOMP  15		/* "Silicon compiler" */
#define STOPREASONDISPLAY  16		/* "Display" */
#define STOPREASONSIMULATE 17		/* "Simulation" */
#define STOPREASONDECK     18		/* "Deck generation" */
#define STOPREASONSPICE    19		/* "SPICE" */
#define STOPREASONCHECK    20		/* "Check" */
#define STOPREASONARRAY    21		/* "Array" */
#define STOPREASONITERATE  22		/* "Iteration" */
#define STOPREASONREPLACE  23		/* "Replace" */
#define STOPREASONSPREAD   24		/* "Spread" */
#define STOPREASONEXECUTE  25		/* "Execution" */
#define STOPREASONCOMFILE  26		/* "Command file" */
#define STOPREASONSELECT   27		/* "Selection" */
#define STOPREASONTRACK    28		/* "Tracking" */

/* a macro for converting Macintosh file types into integers */
#ifdef MACOS
#  define MACFSTAG(x) x
#else
#  define MACFSTAG(x) 0
#endif

/*
 * the "filetype" parameter in "multifileselectin", "describefiletype",
 * "setupfiletype", "getfiletype", "xopen", "xcreate"
 */
#define FILETYPE               01777		/* type of file */
#define FILETYPEWRITE          02000		/* bit set to write file (instead of read) */
#define FILETYPEAPPEND         04000		/* bit set to append to file (instead of read) */

/* the "purpose" parameter (cursor shape) for "trackcursor()" */
#define TRACKNORMAL                0		/* standard cursor */
#define TRACKDRAWING               1		/* pen cursor */
#define TRACKDRAGGING              2		/* hand cursor */
#define TRACKSELECTING             3		/* horizontal arrow cursor */
#define TRACKHSELECTING            4		/* arrow cursor */

extern XARRAY el_matid;						/* identity matrix */
extern INTSML el_pleasestop;				/* nonzero if abort is requested */
extern INTBIG el_units;						/* display and internal units */
extern char  *el_libdir;					/* pointer to library directory */
extern char  *el_version;					/* current version string */
extern INTBIG el_filetypetext;				/* plain text disk file descriptor */

/*************************** DECLARATIONS ****************************/

/* node routines */
NODEINST    *newnodeinst(NODEPROTO *proto, INTBIG lowx, INTBIG highx, INTBIG lowy, INTBIG highy,
				INTSML transposition, INTSML rotation, NODEPROTO *parent);
INTSML       killnodeinst(NODEINST *node);
void         defaultnodesize(NODEPROTO *np, INTBIG *xs, INTBIG *ys);
void         modifynodeinst(NODEINST *node, INTBIG Dlx, INTBIG Dly, INTBIG Dhx,
				INTBIG Dhy, INTSML Drotation, INTSML Dtransposition);
NODEINST    *replacenodeinst(NODEINST *oldnode, NODEPROTO *newproto, INTSML ignoreportnames,
				INTSML allowdeletedports);
INTBIG       nodepolys(NODEINST *node, INTBIG *reasonable, WINDOWPART *win);
void         shapenodepoly(NODEINST *node, INTBIG box, POLYGON *poly);
INTBIG       nodeEpolys(NODEINST *node, INTBIG *reasonable, WINDOWPART *win);
void         shapeEnodepoly(NODEINST *node, INTBIG box, POLYGON *poly);
void         nodesizeoffset(NODEINST *node, INTBIG *lowx, INTBIG *lowy, INTBIG *highx, INTBIG *highy);
void         nodeprotosizeoffset(NODEPROTO *np, INTBIG *lowx, INTBIG *lowy, INTBIG *highx, INTBIG *highy);
INTBIG       nodefunction(NODEINST *node, char **extra);
char        *nodefunctionname(INTBIG function, NODEINST *ni);
char        *describenodeinst(NODEINST *node);
void         transistorsize(NODEINST *node, INTBIG *length, INTBIG *width);
void         getarcdegrees(NODEINST *ni, double *startoffset, double *endangle);
void         setarcdegrees(NODEINST *ni, double startoffset, double endangle);
void         getarcendpoints(NODEINST *ni, double startoffset, double endangle, INTBIG *fx, INTBIG *fy,
				INTBIG *tx, INTBIG *ty);
void         transistorports(NODEINST *node, PORTPROTO **gateleft, PORTPROTO **gateright,
				PORTPROTO **activetop, PORTPROTO **activebottom);
INTBIG       lambdaofnode(NODEINST *ni);
VARIABLE    *gettrace(NODEINST *node);
void         corneroffset(NODEINST *node, NODEPROTO *proto, INTSML rotation, INTSML transpose,
				INTBIG *xoff, INTBIG *yoff, INTSML centerbased);
NODEINST    *dummynode(void);
INTSML       isfet(GEOM *pos);
NODEPROTO   *getpurelayernode(TECHNOLOGY *tech, INTBIG layer, INTBIG function);
void         drcminnodesize(NODEPROTO *np, LIBRARY *lib, INTBIG *sizex, INTBIG *sizey, char **rule);

/* arc routines */
ARCINST     *newarcinst(ARCPROTO *proto, INTBIG width, INTBIG initialuserbits, NODEINST *nodeA,
				PORTPROTO *portA, INTBIG xA, INTBIG yA, NODEINST *nodeB, PORTPROTO *portB,
				INTBIG xB, INTBIG yB, NODEPROTO *parent);
INTSML       killarcinst(ARCINST *arc);
INTBIG       defaultarcwidth(ARCPROTO *ap);
INTSML       modifyarcinst(ARCINST *arc, INTBIG Dwidth, INTBIG Dx1, INTBIG Dy1, INTBIG Dx2, INTBIG Dy2);
ARCINST     *aconnect(GEOM *fromgeom, PORTPROTO *fromport, GEOM *togeom, PORTPROTO *toport,
				ARCPROTO *arc, INTBIG x, INTBIG y, ARCINST **arc2, ARCINST **arc3, INTSML ang,
				INTSML nozigzag);
ARCINST     *replacearcinst(ARCINST *oldarc, ARCPROTO *newproto);
INTBIG       arcpolys(ARCINST *arc, WINDOWPART *win);
void         shapearcpoly(ARCINST *arc, INTBIG box, POLYGON *poly);
char        *arcfunctionname(INTBIG function);
char        *describearcinst(ARCINST *arc);
char        *describearcproto(ARCPROTO *arcproto);
ARCPROTO    *getarcproto(char *specification);
INTBIG       arcwidthoffset(ARCINST *arc);
INTBIG       arcprotowidthoffset(ARCPROTO *arc);
NODEPROTO   *getpinproto(ARCPROTO *arc);
void         makearcpoly(INTBIG length, INTBIG width, ARCINST *arc, POLYGON *poly, INTSML style);
INTSML       curvedarcoutline(ARCINST *arc, POLYGON *poly, INTSML style, INTBIG width);
INTBIG       setshrinkvalue(ARCINST *arc, INTSML extend);
void         determineangle(ARCINST *arc);
INTBIG       lambdaofarc(ARCINST *ai);
INTSML       arcconnects(INTSML ang, INTBIG lx1, INTBIG hx1, INTBIG ly1, INTBIG hy1, INTBIG lx2, INTBIG hx2,
				INTBIG ly2, INTBIG hy2, INTBIG *x1, INTBIG *y1, INTBIG *x2, INTBIG *y2);
ARCINST     *dummyarc(void);

/* facet routines */
NODEPROTO   *newnodeproto(char *name, LIBRARY *library);
INTSML       killnodeproto(NODEPROTO *facet);
NODEPROTO   *copynodeproto(NODEPROTO *originalfacet, LIBRARY *destlibrary, char *name);
NODEPROTO   *getnodeproto(char *specification);
char        *describenodeproto(NODEPROTO *nodeproto);
char        *nldescribenodeproto(NODEPROTO *nodeproto);
void         grabpoint(NODEPROTO *nodeproto, INTBIG *grabx, INTBIG *graby);
INTSML       isachildof(NODEPROTO *parent, NODEPROTO *child);
INTSML       framesize(INTBIG *x, INTBIG *y, NODEPROTO *np);
INTSML       framepolys(NODEPROTO *np);
void         framepoly(INTSML box, POLYGON *poly, NODEPROTO *np);
void         begintraversehierarchy(void);
void         downhierarchy(NODEINST *ni);
void         uphierarchy(void);
void         gettraversalpath(NODEINST ***nilist, INTSML *depth);
NODEPROTO   *getcurfacet(void);
INTBIG       lambdaoffacet(NODEPROTO *np);
NODEINST    *descentparent(NODEPROTO *np);

/* cell routines */
CELL        *getcell(char *specification);

/* export routines */
PORTPROTO   *newportproto(NODEPROTO *facet, NODEINST *nodeinfacet, PORTPROTO *portinnode, char *name);
INTSML       killportproto(NODEPROTO *facet, PORTPROTO *port);
INTSML       moveportproto(NODEPROTO *facet, PORTPROTO *port, NODEINST *newnode, PORTPROTO *newnodeport);
PORTPROTO   *getportproto(NODEPROTO *facet, char *portname);
void         shapeportpoly(NODEINST *node, PORTPROTO *port, POLYGON *poly, INTSML purpose);
void         shapetransportpoly(NODEINST *node, PORTPROTO *port, POLYGON *poly, XARRAY trans);
void         portposition(NODEINST *node, PORTPROTO *port, INTBIG *x, INTBIG *y);
void         reduceportpoly(POLYGON *poly, NODEINST *node, PORTPROTO *port, INTBIG width);
PORTPROTO   *equivalentport(NODEPROTO *facet, PORTPROTO *port, NODEPROTO *equivalentfacet);
INTSML       portispower(PORTPROTO *port);
INTSML       portisground(PORTPROTO *port);
INTSML       portisnamedpower(PORTPROTO *port);
INTSML       portisnamedground(PORTPROTO *port);
void         changeallports(PORTPROTO *port);
char        *describeportbits(PORTPROTO *port);

/* view routines */
VIEW        *newview(char *viewname, char *abbreviation);
INTSML       killview(VIEW *view);
INTSML       changefacetview(NODEPROTO *facet, VIEW *view);
VIEW        *getview(char *specification);
NODEPROTO   *contentsview(NODEPROTO *facet);
NODEPROTO   *iconview(NODEPROTO *facet);
NODEPROTO   *layoutview(NODEPROTO *facet);
NODEPROTO   *anyview(NODEPROTO *np, VIEW *v);

/* window routines */
WINDOWPART  *newwindowpart(char *location, WINDOWPART *protowindow);
void         killwindowpart(WINDOWPART *window);
void         copywindowpart(WINDOWPART *dwindow, WINDOWPART *swindow);
INTBIG       truefontsize(INTBIG font, WINDOWPART *window, TECHNOLOGY *tech);
void         computewindowscale(WINDOWPART *window);
INTBIG       applyxscale(WINDOWPART *window, INTBIG value);
INTBIG       applyyscale(WINDOWPART *window, INTBIG value);

/* variable routines */
VARIABLE    *getval(INTBIG addr, INTBIG  type, INTBIG want, char *name);
VARIABLE    *getvalkey(INTBIG addr, INTBIG type, INTBIG want, INTBIG key);
VARIABLE    *getvalnoeval(INTBIG addr, INTBIG  type, INTBIG want, char *name);
VARIABLE    *getvalkeynoeval(INTBIG addr, INTBIG type, INTBIG want, INTBIG key);
VARIABLE    *evalvar(VARIABLE *var);
VARIABLE    *getparentval(char *name);
VARIABLE    *getparentvalkey(INTBIG key);
#ifdef DEBUGMEMORY
#  define    setval(a, t, n, na, nt) _setval((a), (t), (n), (na), (nt), (char *)__FILE__, (INTBIG)__LINE__)
  VARIABLE *_setval(INTBIG addr, INTBIG type, char *name, INTBIG newaddr, INTBIG newtype, char*, INTBIG);
#  define    setvalkey(a, t, k, na, nt) _setvalkey((a), (t), (k), (na), (nt), (char *)__FILE__, (INTBIG)__LINE__)
  VARIABLE *_setvalkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG newaddr, INTBIG newtype, char*, INTBIG);
#else
  VARIABLE  *setval(INTBIG addr, INTBIG type, char *name, INTBIG newaddr, INTBIG newtype);
  VARIABLE  *setvalkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG newaddr, INTBIG newtype);
#endif
INTSML       delval(INTBIG addr, INTBIG type, char *name);
INTSML       delvalkey(INTBIG addr, INTBIG type, INTBIG key);
INTSML       getind(INTBIG addr, INTBIG type, char *name, INTBIG aindex, INTBIG *value);
INTSML       getindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG aindex, INTBIG *value);
INTSML       setind(INTBIG addr, INTBIG type, char *name, INTBIG aindex, INTBIG newaddr);
INTSML       setindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG aindex, INTBIG newaddr);
INTSML       insind(INTBIG addr, INTBIG type, char *name, INTBIG aindex, INTBIG newaddr);
INTSML       insindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG aindex, INTBIG newaddr);
INTSML       delind(INTBIG addr, INTBIG type, char *name, INTBIG aindex);
INTSML       delindkey(INTBIG addr, INTBIG type, INTBIG key, INTBIG aindex);
INTSML       renameval(char *oldname, char *newname);
INTBIG       makekey(char *name);
char        *makename(INTBIG key);
INTBIG       initobjlist(INTBIG addr, INTBIG type, INTSML restrictdirect);
char        *nextobjectlist(VARIABLE **var, INTBIG search);
INTSML       copyvars(INTBIG fromaddr, INTBIG fromtype, INTBIG toaddr, INTBIG totype);
void         modifydescript(INTBIG addr, INTBIG type, VARIABLE *var, INTBIG newdescript);
INTBIG       getlength(VARIABLE *var);
char        *describevariable(VARIABLE *var, INTSML aindex, INTSML purpose);
char        *describedisplayedvariable(VARIABLE *var, INTSML aindex, INTSML purpose);
char        *describesimplevariable(VARIABLE *var);
void         makeoptionstemporary(LIBRARY *lib);
void         restoreoptionstate(void);
INTSML       isoptionvariable(INTBIG addr, INTBIG type, char *name);
INTSML       describeoptions(INTBIG aindex, char **name, INTBIG *bits);
void         explainoptionchanges(void);
char        *truevariablename(VARIABLE *var);

/* change control routines */
void         startobjectchange(INTBIG address, INTBIG type);
void         endobjectchange(INTBIG address, INTBIG type);
INTSML       undonature(INTSML undo);
INTSML       undoabatch(AIDENTRY **aid);
INTSML       redoabatch(AIDENTRY **aid);
void         noundoallowed(void);
void         noredoallowed(void);
INTBIG       historylistsize(INTBIG newsize);
void         setactivity(char *message);
void         showhistorylist(INTSML batch);
char        *changedvariablename(INTBIG type, INTBIG key, INTBIG subtype);
INTBIG       getcurrentbatchnumber(void);
void         nextchangequiet(void);
void         changesquiet(INTSML quiet);

/* layer routines */
char        *layername(TECHNOLOGY *tech, INTBIG layer);
INTBIG       layerfunction(TECHNOLOGY *tech, INTBIG layer);
INTSML       layerismetal(INTBIG fun);
INTSML       layerispoly(INTBIG fun);
INTSML       layeriscontact(INTBIG fun);
INTBIG       layerfunctionheight(INTBIG layerfunct);
INTBIG       maxdrcsurround(TECHNOLOGY *tech, LIBRARY *lib, INTBIG layer);
INTBIG       drcmindistance(TECHNOLOGY *tech, LIBRARY *lib, INTBIG layer1, INTBIG size1,
				INTBIG layer2, INTBIG size2, INTBIG connected, INTBIG multicut, char **rule);
INTBIG       drcminwidth(TECHNOLOGY *tech, LIBRARY *lib, INTBIG layer, char **rule);
INTSML       getecolor(char *colorname);
INTSML       ecolorname(INTSML color, char **colorname, char **colorsymbol);
INTSML       get3dfactors(TECHNOLOGY *tech, INTBIG layer, INTBIG *height, INTBIG *thickness);
void         set3dheight(TECHNOLOGY *tech, INTBIG *depth);
void         set3dthickness(TECHNOLOGY *tech, INTBIG *thickness);

/* geometric routines */
INTBIG       initsearch(INTBIG lowx, INTBIG highx, INTBIG lowy, INTBIG highy, NODEPROTO *facet);
GEOM        *nextobject(INTBIG initsearchvalue);
void         termsearch(INTBIG initsearchvalue);
void         boundobj(GEOM *object, INTBIG *lowx, INTBIG *highx, INTBIG *lowy, INTBIG *highy);
char        *geomname(GEOM *geom);
NODEPROTO   *geomparent(GEOM *geom);
void         linkgeom(GEOM *geom, NODEPROTO *facet);
void         undogeom(GEOM *geom, NODEPROTO *facet);
void         updategeom(GEOM *geom, NODEPROTO *facet);
INTSML       geomstructure(NODEPROTO *facet);

/* library routines */
LIBRARY     *getlibrary(char *libname);
LIBRARY     *newlibrary(char *name, char *diskfile);
void         selectlibrary(LIBRARY *library);
void         killlibrary(LIBRARY *library);
void         eraselibrary(LIBRARY *library);

/* technology routines */
TECHNOLOGY  *gettechnology(char *techname);
TECHNOLOGY  *whattech(NODEPROTO *facet);
INTSML       addtechnology(TECHNOLOGY *technology);
INTSML       killtechnology(TECHNOLOGY *technology);
void         registertechnologycache(void (*routine)(void), INTBIG count, INTBIG *variablekeys);
void         changedtechnologyvariable(INTBIG key);
void         telltech(TECHNOLOGY *technology, INTSML count, char *par[]);
INTBIG       asktech(TECHNOLOGY *technology, char *command, ...);

/* aid routines */
void         aidturnon(AIDENTRY *aid, INTSML nocatchup);
void         aidturnoff(AIDENTRY *aid, INTSML permanently);
INTSML       tellaid(AIDENTRY *aid, INTSML count, char *args[]);
INTBIG       askaid(AIDENTRY *aid, char *command, ...);
AIDENTRY    *getaid(char *aidname);

/* transformations and mathematics */
void         makeangle(INTSML rotation, INTSML transposition, XARRAY matrix);
INTBIG       sine(INTSML angle);
INTBIG       cosine(INTSML angle);
INTBIG       pickprime(INTBIG start);
void         makerot(NODEINST *node, XARRAY matrix);
void         makerotI(NODEINST *node, XARRAY matrix);
void         maketrans(NODEINST *node, XARRAY matrix);
void         maketransI(NODEINST *node, XARRAY matrix);
void         transid(XARRAY matrix);
void         transcpy(XARRAY matsource, XARRAY matdest);
void         transmult(XARRAY mata, XARRAY matb, XARRAY matdest);
INTSML       ismanhattan(XARRAY trans);
void         xform(INTBIG x, INTBIG y, INTBIG *newx, INTBIG *newy, XARRAY matrix);
void         xformbox(INTBIG*, INTBIG*, INTBIG*, INTBIG*, XARRAY);
INTBIG       muldiv(INTBIG a, INTBIG b, INTBIG c);
INTBIG       mult(INTBIG a, INTBIG b);
INTBIG       mini(INTBIG a, INTBIG b);
INTBIG       maxi(INTBIG a, INTBIG b);
INTBIG       roundfloat(float v);
INTBIG       rounddouble(double v);
INTBIG       castint(float f);
float        castfloat(INTBIG i);
INTBIG       intsqrt(INTBIG v);
INTBIG       scalefromdispunit(float value, INTBIG dispunit);
float        scaletodispunit(INTBIG value, INTBIG dispunit);
float        scaletodispunitsq(INTBIG value, INTBIG dispunit);
INTBIG       computedistance(INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2);
INTSML       figureangle(INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2);
INTSML       intersect(INTBIG x1, INTBIG y1, INTBIG angle1, INTBIG x2, INTBIG y2, INTBIG angle2,
				INTBIG *x, INTBIG *y);
INTSML       isonline(INTBIG, INTBIG, INTBIG, INTBIG, INTBIG, INTBIG);
INTBIG       closestpointtosegment(INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2, INTBIG *x, INTBIG *y);
INTSML       findcenters(INTBIG r, INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2, INTBIG d,
				INTBIG *ix1, INTBIG *iy1, INTBIG *ix2, INTBIG *iy2);
INTSML       circlelineintersection(INTBIG icx, INTBIG icy, INTBIG isx, INTBIG isy, INTBIG lx1,
				INTBIG ly1, INTBIG lx2, INTBIG ly2, INTBIG *ix1, INTBIG *iy1, INTBIG *ix2, INTBIG *iy2,
				INTBIG tolerance);
INTSML       circletangents(INTBIG x, INTBIG y, INTBIG cx, INTBIG cy, INTBIG sx, INTBIG sy,
				INTBIG *ix1, INTBIG *iy1, INTBIG *ix2, INTBIG *iy2);
void         arcbbox(INTBIG xs, INTBIG ys, INTBIG xe, INTBIG ye, INTBIG xc, INTBIG yc,
				INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy);
INTSML       clipline(INTBIG *fx, INTBIG *fy, INTBIG *tx, INTBIG *ty, INTBIG lx, INTBIG hx,
				INTBIG ly, INTBIG hy);
void         cliparc(POLYGON *in, INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy);
void         subrange(INTBIG low, INTBIG high, INTBIG lowmul, INTBIG lowsum, INTBIG highmul, INTBIG highsum,
				INTBIG *newlow, INTBIG *newhigh, INTBIG lambda);
INTSML       polyinrect(POLYGON *poly, INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy);
void         clippoly(POLYGON *poly, INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy);
void         closestpoint(POLYGON *poly, INTBIG *x, INTBIG *y);
INTBIG       getrange(INTBIG low, INTBIG high, INTBIG mul,INTBIG sum, INTBIG lambda);
INTSML       rotatelabel(INTSML oldstyle, XARRAY trans);
INTSML       cropbox(INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy, INTBIG bx, INTBIG ux, INTBIG by,
				INTBIG uy);
void         arctopoly(INTBIG cx, INTBIG cy, INTBIG sx, INTBIG sy, INTBIG ex, INTBIG ey, POLYGON *poly,
				INTBIG arcres, INTBIG arcsag);
void         circletopoly(INTBIG cx, INTBIG cy, INTBIG radius, POLYGON *poly, INTBIG arcres, INTBIG arcsag);
void         adjustdisoffset(INTBIG addr, INTBIG type, TECHNOLOGY *tech, POLYGON *poly, INTBIG textdescription);
void         vectoradd3d(float *a, float *b, float *sum);
void         vectorsubtract3d(float *a, float *b, float *diff);
void         vectormultiply3d(float *a, float b, float *res);
void         vectordivide3d(float *a, float b, float *res);
float        vectormagnitude3d(float *a);
float        vectordot3d(float *a, float *b);
void         vectornormalize3d(float *a);
void         vectorcross3d(float *a, float *b, float *res);
void         matrixid3d(float xform[4][4]);
void         matrixmult3d(float a[4][4], float b[4][4], float res[4][4]);
void         matrixinvert3d(float mats[4][4], float matd[4][4]);
void         matrixxform3d(float *vec, float mat[4][4], float *res);

/* box merging routines */
void         mrginit(void);
void         mrgterm(void);
void         mrgstorebox(INTSML layer, TECHNOLOGY *tech, INTBIG length, INTBIG width, INTBIG xc, INTBIG yc);
void         mrgdonefacet(void (*writepolygon)(INTSML layernum, TECHNOLOGY *tech,
				INTBIG *x, INTBIG *y, INTSML count));
void         mergeinit(void);
void         mergestorepolygon(INTBIG layer, TECHNOLOGY *tech, POLYGON *poly);
void         mergedone(void (*writepolygon)(INTSML, TECHNOLOGY*, INTBIG*, INTBIG*, INTSML));

/* polygon routines */
#ifdef DEBUGMEMORY
  POLYGON  *_allocpolygon(INTSML points, CLUSTER *cluster, char*, INTBIG);
#  define    allocpolygon(p, c) _allocpolygon((p), (c), (char *)__FILE__, (INTBIG)__LINE__)
#else
  POLYGON   *allocpolygon(INTSML points, CLUSTER *cluster);
#endif
POLYGON     *allocstaticpolygon(INTSML points, CLUSTER *cluster);
void         freepolygon(POLYGON *poly);
INTSML       extendpolygon(POLYGON *poly, INTSML newcount);
INTSML       polysame(POLYGON *poly1, POLYGON *poly2);
void         polycopy(POLYGON *dpoly, POLYGON *spoly);
INTSML       isinside(INTBIG x, INTBIG y, POLYGON *poly);
float        areapoly(POLYGON *poly);
float        areapoints(INTBIG count, INTBIG *x, INTBIG *y);
void         xformpoly(POLYGON *poly, XARRAY trans);
INTSML       isbox(POLYGON *poly, INTBIG *xl, INTBIG *xh, INTBIG *yl, INTBIG *yh);
void         maketruerect(POLYGON *poly);
INTBIG       polydistance(POLYGON *poly, INTBIG x, INTBIG y);
INTBIG       polyminsize(POLYGON *poly);
INTBIG       polyseparation(POLYGON*, POLYGON*);
INTSML       polyintersect(POLYGON*, POLYGON*);
void         makerectpoly(INTBIG lx, INTBIG ux, INTBIG ly, INTBIG uy, POLYGON *poly);
void         maketruerectpoly(INTBIG lx, INTBIG ux, INTBIG ly, INTBIG uy, POLYGON *poly);
void         makedisparrayvarpoly(GEOM *geom, WINDOWPART *win, VARIABLE *var, POLYGON *poly);
void         getdisparrayvarlinepos(INTBIG addr, INTBIG type, TECHNOLOGY *tech,
				WINDOWPART *win, VARIABLE *var, INTBIG line, INTBIG *x, INTBIG *y, INTSML absolute);
void         getcenter(POLYGON *poly, INTBIG *x, INTBIG *y);
void         getbbox(POLYGON *poly, INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy);
INTSML       polysplithoriz(INTBIG yl, POLYGON *which, POLYGON **abovep, POLYGON **belowp);
INTSML       polysplitvert(INTBIG xl, POLYGON *which, POLYGON **leftp, POLYGON **rightp);

/* string manipulation routines */
char        *latoa(INTBIG value);
INTBIG       atola(char *str);
char        *frtoa(INTBIG value);
INTBIG       atofr(char *str);
INTBIG       myatoi(char *str);
INTSML       isanumber(char *str);
char        *explainduration(float duration);
void         parseelectricversion(char *version, INTBIG *major, INTBIG *minor, INTBIG *detail);
char        *makeplural(char *word, INTBIG amount);
char        *makeabbrev(char *str, INTSML upper);
INTSML       namesame(char *str1, char *str2);
INTSML       namesamen(char *str1, char *str2, INTBIG length);
INTSML       stringmatch(char *str1, char *str2);
INTSML       parse(char *keyword, COMCOMP *list, INTSML noise);
INTSML       initinfstr(void);
INTSML       addtoinfstr(char c);
INTSML       addstringtoinfstr(char *str);
INTSML       formatinfstr(char *msg, ...);
char        *returninfstr(void);
void        *newstringarray(CLUSTER *cluster);
void         killstringarray(void *sa);
void         clearstrings(void *sa);
void         addtostringarray(void *sa, char *string);
void         stringarraytotextfacet(void *sa, NODEPROTO *np, INTSML permanent);
char       **getstringarray(void *sa, INTBIG *count);
char        *getkeyword(char **ptin, char *brk);
char         tonextchar(char **ptin);
char         seenextchar(char **ptin);
INTBIG       defaulttextdescript(GEOM *geom);
INTBIG       defaulttextsize(INTBIG texttype);
char        *unitsname(INTBIG units);
void         myencrypt(char *text, char *key);

/* language interface routines */
char        *languagename(void);
INTSML       languageconverse(INTBIG language);
INTSML       doquerry(char *code, INTBIG language, INTBIG type, INTBIG *retval);
INTSML       loadcode(char *program, INTBIG language);

/* error reporting routines */
void         initerrorlogging(char *system);
void        *logerror(char *message, NODEPROTO *facet, INTBIG sortkey);
INTSML       addgeomtoerror(void *errorlist, GEOM *geom, INTSML showit);
INTSML       addfacettoerror(void *errorlist, NODEPROTO *facet);
INTSML       addlinetoerror(void *errorlist, INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2);
INTSML       addpolytoerror(void *errorlist, POLYGON *poly);
INTSML       addpointtoerror(void *errorlist, INTBIG x, INTBIG y);
void         termerrorlogging(void);
void         sorterrors(void);
INTBIG       numerrors(void);
char        *reportnexterror(INTBIG showhigh, GEOM **g1, GEOM **g2);
char        *reportpreverror(void);
char        *reportcurrenterror(INTBIG showhigh, GEOM **g1, GEOM **g2);

/* memory allocation routines */
#ifdef DEBUGMEMORY
  INTBIG   *_emalloc(INTBIG size, CLUSTER *cluster, char*, INTBIG);
# define     emalloc(x, c) _emalloc((x), (c), (char *)__FILE__, (INTBIG)__LINE__)
  INTSML    _allocstring(char **addr, char *str, CLUSTER *cluster, char*, INTBIG);
# define     allocstring(a, s, c) _allocstring((a), (s), (c), (char *)__FILE__, (INTBIG)__LINE__)
  INTSML    _reallocstring(char **addr, char *str, CLUSTER *cluster, char*, INTBIG);
# define     reallocstring(a, s, c) _reallocstring((a), (s), (c), (char *)__FILE__, (INTBIG)__LINE__)
#else
  INTBIG    *emalloc(INTBIG size, CLUSTER *cluster);
  INTSML     allocstring(char **addr, char *str, CLUSTER *cluster);
  INTSML     reallocstring(char **addr, char *str, CLUSTER *cluster);
#endif
void         efree(char *addr);
CLUSTER     *alloccluster(char *name);
void         freecluster(CLUSTER *cluster);
NODEINST    *allocnodeinst(CLUSTER *cluster);
void         freenodeinst(NODEINST *ni);
ARCINST     *allocarcinst(CLUSTER *cluster);
void         freearcinst(ARCINST *ai);
PORTPROTO   *allocportproto(CLUSTER *cluster);
void         freeportproto(PORTPROTO *pp);
PORTARCINST *allocportarcinst(CLUSTER *cluster);
void         freeportarcinst(PORTARCINST *pi);
PORTEXPINST *allocportexpinst(CLUSTER *cluster);
void         freeportexpinst(PORTEXPINST *pe);
NODEPROTO   *allocnodeproto(CLUSTER *cluster);
void         freenodeproto(NODEPROTO *np);
CELL        *alloccell(CLUSTER *cluster);
void         freecell(CELL *cell);
VIEW        *allocview(void);
void         freeview(VIEW *view);
GEOM        *allocgeom(CLUSTER *cluster);
void         freegeom(GEOM *geom);
RTNODE      *allocrtnode(CLUSTER *cluster);
void         freertnode(RTNODE *rtnode);
LIBRARY     *alloclibrary(void);
void         freelibrary(LIBRARY *lib);
TECHNOLOGY  *alloctechnology(CLUSTER *cluster);
void         freetechnology(TECHNOLOGY *tech);
void         freewindowpart(WINDOWPART *win);

/* terminal output routines */
void         ttynewcommand(void);
void         ttyputerr(char *format, ...);
void         ttyputmsg(char *format, ...);
void         ttyputverbose(char *format, ...);
void         ttyputinstruction(char *keystroke, INTBIG length, char *meaning);
void         ttyputusage(char *usage);
void         ttyputbadusage(char *command);
void         ttyputnomemory(void);
void         error(char *format, ...);
INTBIG       ttyquiet(INTBIG flag);
void         ttybeep(void);
void         telldatabaseerror(void);

/* terminal input routines */
INTSML       ttygetchar(void);
char        *ttygetline(char *prompt);
char        *ttygetlinemessages(char *prompt);
INTSML       ttygetparam(char *prompt, COMCOMP *parameter, INTSML keycount, char *paramstart[]);
INTSML       ttygetfullparam(char *prompt, COMCOMP *parameter, INTSML keycount, char *paramstart[]);
INTSML       ttydataready(void);
void         checkforinterrupt(void);

/* mouse/tablet input routines */
INTSML       getxy(INTBIG *x, INTBIG *y);
void         gridalign(INTBIG *xcur, INTBIG *ycur, INTBIG alignment);
INTSML       buttoncount(void);
char        *buttonname(INTSML but, INTSML *important);
INTSML       doublebutton(INTSML but);
INTSML       shiftbutton(INTSML but);
INTSML       wheelbutton(INTSML but);
void         waitforbutton(INTSML *x, INTSML *y, INTSML *but);
void         trackcursor(INTSML waitforpush, INTSML (*whileup)(INTBIG x, INTBIG y), void (*whendown)(void),
				INTSML (*eachdown)(INTBIG x, INTBIG y), INTSML (*eachchar)(INTBIG x, INTBIG y, INTSML chr),
				void (*done)(void), INTSML purpose);
void         readtablet(INTSML *x, INTSML *y);
void         stoptablet(void);

/* time routines */
UINTBIG      ticktime(void);
INTBIG       doubleclicktime(void);
void         gotosleep(INTBIG ticks);
void         starttimer(void);
float        endtimer(void);
UINTBIG      getcurrenttime(void);
char        *timetostring(UINTBIG time);
void         parsetime(UINTBIG time, INTSML *year, INTSML *month, INTSML *mday,
				INTSML *hour, INTSML *minute, INTSML *second);
INTBIG       parsemonth(char *monthname);

/* disk file routines */
char        *truepath(char *path);
char        *skippath(char *file);
void         emaketemp(char *file);
INTBIG       erename(char *file, char *newfile);
INTBIG       eunlink(char *file);
char        *fileselect(char*, INTBIG, char*);
char        *multifileselectin(char *msg, INTBIG filetype);
INTSML       fileexistence(char *file);
char        *currentdirectory(void);
char        *hashomedir(void);
INTSML       createdirectory(char *directory);
INTSML       filesindirectory(char *directory, char ***filelist);
INTSML       lockfile(char *file);
void         unlockfile(char *file);
UINTBIG      filedate(char *file);
INTBIG       filesize(FILE *stream);
INTSML       browsefile(char *filename);
void         describefiletype(INTBIG filetype, char **extension, char **winfilter,
				INTBIG *mactype, INTBIG *binary, char **shortname, char **longname);
INTBIG       setupfiletype(char *extension, char *winfilter, INTBIG mactype, INTBIG binary,
				char *shortname, char *longname);
INTBIG       getfiletype(char *shortname);
FILE        *xopen(char *file, INTBIG filetype, char *otherdir, char **truename);
FILE        *xcreate(char *name, INTBIG filetype, char *prompt, char **truename);
FILE        *xappend(char *file);
void         xclose(FILE *stream);
void         xflushbuf(FILE *stream);
INTSML       xeof(FILE *stream);
void         xseek(FILE *stream, INTBIG pos, INTSML nature);
INTBIG       xtell(FILE *f);
void         xprintf(FILE *stream, char *format, ...);
INTSML       xgetc(FILE *stream);
void         xungetc(char chr, FILE *stream);
void         xputc(char chr, FILE *stream);
INTSML       xputs(char *str, FILE *stream);
INTBIG       xfread(char *str, INTBIG size, INTBIG count, FILE *stream);
INTBIG       xfwrite(char *str, INTBIG size, INTBIG count, FILE *stream);
INTSML       xfgets(char *str, INTSML limit, FILE *stream);

/* channel routines */
INTBIG       eread(int channel, char *addr, INTBIG count);
INTBIG       ewrite(int channel, char *addr, INTBIG count);
INTBIG       eclose(int channels);
INTBIG       epipe(int channels[2]);
INTBIG       channelreplacewithfile(int channel, char *file);
INTBIG       channelreplacewithchannel(int channel, int newchannel);
void         channelrestore(int channel, INTBIG saved);
void         setinteractivemode(int channel);

/* miscellaneous routines */
void         changelambda(INTBIG count, TECHNOLOGY **techarray, INTBIG *newlam,
				LIBRARY *whichlib, INTBIG how);
void         changetechnologylambda(TECHNOLOGY *tech, INTBIG newlambda);
INTBIG       figurelambda(GEOM *geom);
void         changeinternalunits(LIBRARY *whichlib, INTBIG oldunits, INTBIG newunits);
NETWORK     *getnetwork(char *name, NODEPROTO *facet);
NETWORK     *getcomplexnetwork(char *name, NODEPROTO *facet);
NETWORK     *getnetonport(NODEINST *node, PORTPROTO *port);
char        *describenetwork(NETWORK *net);
void         bringdown(void);
INTSML       stopping(INTBIG reason);
char        *egetenv(char *name);
INTBIG       efork(void);
INTBIG       esystem(char *command);
void         eexec(char *program, char *args[]);
INTBIG       ekill(INTBIG process);
void         ewait(INTBIG process);
char       **eprinterlist(void);
void         flushscreen(void);
void         exitprogram(void);

/* graphics and control (not documented in Internals Manual) */
void         osprimaryosinit(void);
void         ossecondaryinit(INTBIG, char*[]);
void         tooltimeslice(void);
void         graphicsoptions(char *name, INTBIG *argc, char **argv);
void         setupenvironment(void);
void         setlibdir(char *libdir);
INTSML       initgraphics(INTSML messages);
void         termgraphics(void);
WINDOWFRAME *newwindowframe(INTSML floating, RECTAREA *r);
void         killwindowframe(WINDOWFRAME*);
void         getpaletteparameters(INTSML *wid, INTSML *hei, INTSML *palettewidth);
void         resetpaletteparameters(void);
void         sizewindowframe(WINDOWFRAME *frame, INTSML wid, INTSML hei);
void         movewindowframe(WINDOWFRAME *frame, INTSML left, INTSML top);
void         bringwindowtofront(WINDOWFRAME *frame);
void         adjustwindowframe(INTSML how);
WINDOWFRAME *getwindowframe(INTSML canfloat);
void         getwindowframesize(WINDOWFRAME *frame, INTSML *wid, INTSML *hei);
INTSML       graphicshas(INTSML want);
void         setdefaultcursortype(INTSML state);
void         setnormalcursor(INTSML curs);
void         colormapload(INTBIG *red, INTBIG *green, INTBIG *blue, INTSML low, INTSML high);
void         screendrawline(WINDOWPART *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2, GRAPHICS *desc,
				INTSML texture);
void         screeninvertline(WINDOWPART *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2);
void         screendrawpolygon(WINDOWPART *win, INTBIG *x, INTBIG *y, INTSML count, GRAPHICS *desc);
void         screendrawbox(WINDOWPART *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy, GRAPHICS *desc);
void         screeninvertbox(WINDOWPART *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy);
void         screenmovebox(WINDOWPART *win, INTSML sx, INTSML sy, INTSML wid, INTSML hei, INTSML dx, INTSML dy);
INTBIG       screensavebox(WINDOWPART *win, INTSML lx, INTSML hx, INTSML ly, INTSML hy);
void         screenmovesavedbox(INTBIG code, INTSML dx, INTSML dy);
INTSML       screenrestorebox(INTBIG code, INTSML destroy);
void         screensettextsize(WINDOWPART *win, INTBIG fnt);
void         screengettextsize(WINDOWPART *win, char *str, INTSML *x, INTSML *y);
void         screendrawtext(WINDOWPART *win, INTSML atx, INTSML aty, char *s, GRAPHICS *desc);
INTSML       gettextbits(WINDOWPART *win, char *msg, INTSML *wid, INTSML *hei, char ***rowstart);
void         screendrawcircle(WINDOWPART *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc);
void         screendrawdisc(WINDOWPART *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc);
void         screendrawcirclearc(WINDOWPART *win, INTBIG centerx, INTBIG centery, INTBIG p1x, INTBIG p1y,
				INTBIG p2x, INTBIG p2y, GRAPHICS *desc);
void         screendrawgrid(WINDOWPART *win, POLYGON *obj);
INTSML       getnxtchar(void);
UINTBIG      machinetimeoffset(void);
INTSML       logplayback(char *file);
void         logstartrecord(void);
void         logfinishrecord(void);
void         getacceleratorstrings(char **acceleratorstring, char **acceleratorprefix);
char        *getinterruptkey(void);
char        *getmessagesstring(char *prompt);
void         putmessagesstring(char *str, INTSML important);
char        *getmessageseofkey(void);
void         setmessagesfont(void);
INTSML       closefrontmostmessages(void);
INTSML       cutfrommessages(void);
INTSML       copyfrommessages(void);
INTSML       pastetomessages(void);
char        *getcutbuffer(void);
void         setcutbuffer(char *msg);
char        *optionsfilepath(void);

/* command completion coroutines (not documented in Internals Manual) */
void         requiredextension(char *extension);
char        *nextaids(void);
char        *nextarcs(void);
char        *nextcells(void);
char        *nextfacets(void);
char        *nextfile(void);
char        *nextlibs(void);
char        *nextnets(void);
char        *nexttechs(void);
char        *nextviews(void);
INTSML       topofaids(char **chrpos);
INTSML       topofcells(char **chrpos);
INTSML       topoffile(char **chrpos);
INTSML       topoflibfile(char **chrpos);
INTSML       topoflibs(char **chrpos);
INTSML       topofnets(char **chrpos);
INTSML       topoftechs(char **chrpos);
INTSML       topofviews(char **chrpos);
INTSML       topoffacets(char **chrpos);
INTSML       topofarcs(char **chrpos);
INTSML       us_patoplist(char **chrpos);
INTSML       us_paparams(char *word, COMCOMP *arr[], char breakc);
void         us_pabackupto(INTSML a, char *b[]);
char        *us_panextinlist(void);

#if defined(__cplusplus) && !defined(ALLCPLUSPLUS)
}
#endif
