// Copyright (C) 1999-2005 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however    
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.    
//
// This exception applies only to the code released under the name GNU
// ccAudio.  If you copy code from other releases into a copy of GNU
// ccAudio, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own for GNU ccAudio, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.
//


#ifndef	CCXX_AUDIO_H_
#define	CCXX_AUDIO_H_

#ifndef	W32
#if defined(_WIN32) && defined(_MSC_VER)
#define	W32
#endif
#if defined(__BORLANDC__) && defined(__Windows)
#define	W32
#endif
#endif

#if !defined(__EXPORT) && defined(W32)
#define	__EXPORT __declspec(dllimport)
#endif

#ifndef	__EXPORT
#define	__EXPORT
#endif

#ifdef	W32
#include <windows.h>
#else
#include <cstddef>
#include <cstdlib>
#include <sys/types.h>
#include <netinet/in.h>
#endif

namespace ost {

#define	AUDIO_SIGNED_LINEAR_RAW	1
#define	AUDIO_LINEAR_CONVERSION 1
#define	AUDIO_CODEC_MODULES	1



class __EXPORT AudioCodec;
class __EXPORT AudioDevice;

/**
 * Generic audio class to hold master data types and various useful
 * class encapsulated friend functions as per GNU Common C++ 2 coding
 * standard.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Master audio class.
 */
class __EXPORT Audio
{
public:
#ifdef	W32
	typedef	short	Sample;
	typedef	short	*Linear;
	typedef	short	Level;
	typedef	DWORD	timeout_t;
	typedef	WORD	snd16_t;
	typedef	DWORD	snd32_t;
#else
	typedef	int16_t snd16_t;
	typedef	int32_t	snd32_t;
	typedef	int16_t	Level;
	typedef	int16_t	Sample;
	typedef	int16_t	*Linear;
	typedef	unsigned long	timeout_t;
#endif

	typedef struct
	{
	float v2;
		float v3;
		float fac;
	} goertzel_state_t;

	typedef struct
	{
		int hit1;
		int hit2;
		int hit3;
		int hit4;
		int mhit;

		goertzel_state_t row_out[4];
		goertzel_state_t col_out[4];
		goertzel_state_t row_out2nd[4];
		goertzel_state_t col_out2nd[4];
		goertzel_state_t fax_tone;
		goertzel_state_t fax_tone2nd;
		float energy;

		int current_sample;
		char digits[129];
		int current_digits;
		int detected_digits;
		int lost_digits;
		int digit_hits[16];
		int fax_hits;
	} dtmf_detect_state_t;

	typedef struct
	{
		float fac;
	} tone_detection_descriptor_t;

	typedef	unsigned char *Encoded;

	enum	Rate
	{
		rateUnknown,
		rate6khz = 6000,
		rate8khz = 8000,
		rate44khz = 44100
	};

	typedef	enum Rate Rate;

	enum	Mode
	{
		modeRead,
		modeReadAny,
		modeReadOne,
		modeWrite,
		modeCache,
		modeInfo,
		modeFeed
	};

	typedef enum Mode Mode;

	enum	Encoding
	{
		unknownEncoding = 0,
		g721ADPCM,
		g722Audio,
		g722_7bit,
		g722_6bit,
		g723_2bit,
		g723_3bit,
		g723_5bit,
		gsmVoice,
		msgsmVoice,
		mulawAudio,
		alawAudio,
		mp1Audio,
		mp2Audio,
		mp3Audio,
		okiADPCM,
		voxADPCM,
		sx73Voice,
		sx96Voice,

		// Please keep the PCM types at the end of the list -
		// see the "is this PCM or not?" code in
		// AudioFile::close for why.
		cdaStereo,
		cdaMono,
		pcm8Stereo,
		pcm8Mono,
		pcm16Stereo,
		pcm16Mono,
		pcm32Stereo,
		pcm32Mono
	};
	typedef enum Encoding Encoding;

	enum Format
	{
		raw,
		snd,
		riff,
		mpeg,
		wave
	};
	typedef enum Format Format;

	enum DeviceMode
	{
		PLAY,
		RECORD,
		PLAYREC
	};
	typedef enum DeviceMode DeviceMode;

	enum Error
	{
		errSuccess = 0,
		errReadLast,
		errNotOpened,
		errEndOfFile,
		errStartOfFile,
		errRateInvalid,
		errEncodingInvalid,
		errReadInterrupt,
		errWriteInterrupt,
		errReadFailure,
		errWriteFailure,
		errReadIncomplete,
		errWriteIncomplete,
		errRequestInvalid,
		errTOCFailed,
		errStatFailed,
		errInvalidTrack,
		errPlaybackFailed,
		errNotPlaying,
		errNoCodec
	};
	typedef enum Error Error;

	class __EXPORT Info
	{
	public:
		Format format;
		Encoding encoding;
		unsigned long rate;
		unsigned long bitrate;
		unsigned order;
		unsigned framesize, framecount;
		timeout_t framing;
		char *annotation;

		Info();
		void clear(void);
		void set(void);
		void setFraming(timeout_t frame);
	};

	static bool hasDevice(unsigned index = 0);
	static AudioDevice *getDevice(unsigned index = 0, DeviceMode = PLAY);
	static const char *getCodecPath(void);
	static const char *getMIME(Info &info);
	static const char *getName(Encoding encoding);
	static Encoding getEncoding(const char *name);
	static Encoding getStereo(Encoding);
	static Encoding getMono(Encoding);
	static bool isLinear(Encoding encoding);
	static bool isMono(Encoding encoding);
	static bool isStereo(Encoding encoding);
	static Rate getRate(Encoding encoding);
	static timeout_t getFraming(Encoding encoding, timeout_t timeout = 0);
	static timeout_t getFraming(Info &info, timeout_t timeout = 0);
	static bool isEndian(Encoding encoding);
	static bool isEndian(Info &info);
	static bool swapEndian(Encoding encoding, void *buffer, unsigned samples);
	static bool swapEndian(Info &info, void *buffer, unsigned samples);
	static Linear getSingleTone(Linear buffer, timeout_t timeout, unsigned f1, Level m1 = 32767, Rate rate = rate8khz);
	static Linear getDualTone(Linear buffer, timeout_t timeout, unsigned f1, unsigned f2, Level m1 = 32767, Level m2 = 32767, Rate rate = rate8khz);
	static Level getImpulse(Encoding encoding, void *buffer, unsigned samples);
	static Level getImpulse(Info &info, void *buffer, unsigned samples = 0);
        static Level getPeak(Encoding encoding, void *buffer, unsigned samples);
        static Level getPeak(Info &info, void *buffer, unsigned samples = 0);

	/**
	 * Returns the number of bytes in a sample frame for the given
	 * encoding type, rounded up to the nearest integer.  A frame
	 * is defined as the minimum number of bytes necessary to
	 * create a point or points in the output waveform for all
	 * output channels.  For example, 16-bit mono PCM has a frame
	 * size of two (because those two bytes constitute a point in
	 * the output waveform).  GSM has it's own definition of a
	 * frame which involves decompressing a sequence of bytes to
	 * determine the final points on the output waveform.  The
	 * minimum number of bytes you can feed to the decompression
	 * engine is 32.5 (260 bits), so this function will return 33
	 * (because we round up) given an encoding type of GSM.  Other
	 * compressed encodings will return similar results.  Be
	 * prepared to deal with nonintuitive return values for
	 * rare encodings.
	 *
	 * @param encoding The encoding type to get the frame size for.
	 * @param samples Reserved.  Use zero.
	 *
	 * @return The number of bytes in a frame for the given encoding.
	 */
	static int getFrame(Encoding encoding, int samples = 0);

	/**
	 * Returns the number of samples in all channels for a frame
	 * in the given encoding.  For example, pcm32Stereo has a
	 * frame size of 8 bytes: Note that different codecs have
	 * different definitions of a frame - for example, compressed
	 * encodings have a rather large frame size relative to the
	 * sample size due to the way bytes are fed to the
	 * decompression engine.
	 *
	 * @param encoding The encoding to calculate the frame sample count for.
	 * @return samples The number of samples in a frame of the given encoding.
	 */
	static int getCount(Encoding encoding);
	static unsigned long toSamples(Encoding encoding, size_t bytes);
	static unsigned long toSamples(Info &info, size_t bytes);
	static size_t toBytes(Info &info, unsigned long samples);
	static size_t toBytes(Encoding encoding, unsigned long samples);
	static void fill(unsigned char *addr, int samples, Encoding encoding);
	static bool loadPlugin(const char *path);
};

/**
 * The AudioTone class is used to create a frame of audio encoded single or
 * dualtones.  The frame will be iterated for each request, so a
 * continual tone can be extracted by frame.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short audio tone generator class.
 */
class __EXPORT AudioTone : public Audio
{
private:
	Rate rate;
	unsigned samples;
	Linear frame;
	double df1, df2, p1, p2;
	Level m1, m2;

public:
	inline Rate getRate(void)
		{return rate;};

	inline size_t getSamples(void)
		{return samples;};

	/**
	 * Iterate the tone frame, and extract linear samples in
	 * native frame.  If endian flag passed, then convert for
	 * standard endian representation (byte swap) if needed.
	 *
	 * @return pointer to samples.
	 */
	Linear getFrame(bool endian = false);	

	/**
	 * Construct a dual tone frame generator.
	 *
	 * @param freq of tone 1.
	 * @param freq of tone 2.
	 * @param level of tone 1.
	 * @param level of tone 2.
	 * @param duration of frame in milliseconds.
	 * @param sample rate being generated.
	 */
	AudioTone(unsigned freq1, unsigned freq2, Level l1, Level l2, timeout_t duration = 20, Rate r = rate8khz);

	/**
	 * Construct a single tone frame generator.
	 *
	 * @param freq of tone.
	 * @param level of tone.
	 * @param duration of frame in milliseconds.
	 * @param sample rate being generated.
	 */
	AudioTone(unsigned freq, Level l, timeout_t duration = 20, Rate r = rate8khz); 

	~AudioTone();
};

/**
 * The AudioBuffer class is for mixing one-to-one
 * soft joins.
 *
 * @author Mark Lipscombe <markl@gasupnow.com>
 * @short audio buffer mixer class
 */
class __EXPORT AudioBuffer
{
public:
        AudioBuffer(unsigned size = 4096);
        ~AudioBuffer();

        void getBuffer(char *data, unsigned amount);
        void putBuffer(const char *data, unsigned amount);
private:
        char *buf;
        unsigned len;
        unsigned start;
        unsigned size;
	void *mutexObject;

	void enter(void);
	void leave(void);
};

/**
 * A class used to manipulate audio data.  This class provides file
 * level access to audio data stored in different formats.  This class
 * also provides the ability to write audio data into a disk file.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short audio file access.
 */
class __EXPORT AudioFile: public Audio
{
protected:
	char *pathname;
	Error error;
	Info info;
	unsigned long header;		// offset to start of audio
	unsigned long minimum;		// minimum sample size required
	unsigned long length;           // current size of file, including header

	void initialize(void);
	void getWaveFormat(int size);

	union
	{
		int fd;
		void *handle;
	} file;

	Mode mode;
	unsigned long iolimit;

	virtual bool afCreate(const char *path, bool exclusive = false);
	virtual bool afOpen(const char *path, Mode m = modeWrite);
	virtual bool afPeek(unsigned char *data, unsigned size);

	AudioCodec *getCodec(void);

	/**
	 * Read a given number of bytes from the file, starting from
	 * the current file pointer.  May be overridden by derived
	 * classes.
	 *
	 * @param data A pointer to the buffer to copy the bytes to.
	 * @param size The number of bytes to read.
	 * @return The number of bytes read, or -1 if an error occurs.
	 * On UNIX platforms, use strerror(errno) to get the
	 * human-readable error string or
	 * FormatMessage(GetLastError()) on Windows platforms.
	 */
	virtual int afRead(unsigned char *data, unsigned size);

	/** 
	 * Write a number of bytes into the file at the current file
	 * pointer.  May be overridden by derived classes.
	 * 
	 * @param data A pointer to the buffer with the bytes to write.
	 * @param size The number of bytes to write from the buffer.
	 * @return The number of bytes written, or -1 if an error
	 * occurs.  On UNIX platforms, use strerror(errno) to get the
	 * human-readable error string or
	 * FormatMessage(GetLastError()) on Windows platforms.
	 */
	virtual int afWrite(unsigned char *data, unsigned size);

	/**
	 * Seek to the given position relative to the start of the
	 * file and set the file pointer.  This does not use 64-bit
	 * clean seek functions, so seeking to positions greater than
	 * (2^32)-1 will result in undefined behavior.
	 * 
	 * @param pos The position to seek to.
	 * @return true if successful, false otherwise.
	 */
	virtual bool afSeek(unsigned long pos);
	virtual void afClose(void);

	virtual char *getContinuation(void)
		{return NULL;};

	/**
	 * Return a human-readable error message given a numeric error
	 * code of type Audio::Error.
	 *
	 * @param err The numeric error code to translate.
	 * @return A pointer to a character string containing the
	 * human-readable error message.
	 */
	const char * getErrorStr(Error err);
	Error setError(Error err);

	inline unsigned long getHeader(void)
		{return header;};

	unsigned short getShort(unsigned char *data);
	void setShort(unsigned char *data, unsigned short value);
	unsigned long getLong(unsigned char *data);
	void setLong(unsigned char *data, unsigned long value);

public:
	AudioFile(const char *fname, unsigned long samples = 0);
	AudioFile(const char *fname, Info *info, unsigned long min = 0);

	inline AudioFile()
		{initialize();};

	virtual ~AudioFile();

	/**
	 * Open an audio file and associate it with this object.
	 * Called implicitly by the two-argument version of the
	 * constructor.
	 *
	 * @param fname The name of the file to open.  Don't forget to
	 * double your backslashes for DOS-style pathnames.
	 */
	void open(const char *fname, Mode m = modeWrite, timeout_t framing = 0);

	/**
	 * Create a new audio file and associate it with this object.
	 * Called implicitly by the three-argument version of the
	 * constructor.
	 *
	 * @param fname The name of the file to open.
	 * @param info The type of the audio file to be created.
	 * @param exclusive create option.
	 */
	void create(const char *fname, Info *info, bool exclusive = false, timeout_t framing = 0);

	/**
	 * Returns age since last prior access.  Used for cache
	 * computations.
	 * /
	time_t getAge(void);

	/**
	 * Close an object associated with an open file.  This
	 * updates the header metadata with the file length if the
	 * file length has changed.
	 */
	void close(void);

	/**
	 * Clear the AudioFile structure.  Called by
	 * AudioFile::close().  Sets all fields to zero and deletes
	 * the dynamically allocated memory pointed to by the pathname
	 * and info.annotation members.  See AudioFile::initialize()
	 * for the dynamic allocation code.
	 */
	void clear(void);

	/**
	 * Retrieve bytes from the file into a memory buffer.  This
	 * increments the file pointer so subsequent calls read further
	 * bytes.  If you want to read a number of samples rather than
	 * bytes, use getSamples().
	 *
	 * @param addr A pointer to the memory area to copy the samples to.
	 * @param len The number of bytes (not samples) to copy.
	 * @return The number of bytes (not samples) read.  Returns -1
	 * if no bytes are read and an error occurs.
	 */
	int getBuffer(void *addr, unsigned len);

	/**
	 * Retrieve and convert content to linear encoded audio data
	 * from it's original form.
	 *
	 * @param addr A pointer to copy linear data into.
	 * @param len Number of linear samples to extract.
	 * @return number of samples retrieved, 0 if no codec or eof.
	 */
	unsigned getLinear(Linear buffer, unsigned request);

	/**
	 * Insert bytes into the file from a memory buffer.  This
	 * increments the file pointer so subsequent calls append
	 * further samples.  If you want to write a number of samples
	 * rather than bytes, use putSamples().
	 *
	 * @param attr A pointer to the memory area to append the samples
	 * from.
	 * @param len The number of bytes (not samples) to append.
	 * @return The number of bytes (not samples) read.  Returns -1
	 * if an error occurs and no bytes are written.
	 */
	int putBuffer(void *attr, unsigned len);

        /**
         * Convert and store content from linear encoded audio data
         * to the format of the audio file.
         *
         * @param addr A pointer to copy linear data from.
         * @param len Number of linear samples to save.
         * @return number of samples saved, 0 if no codec or eof.
         */
        unsigned putLinear(Linear buffer, unsigned request);

	/**
	 * Retrieve samples from the file into a memory buffer.  This
	 * increments the file pointer so subsequent calls read
	 * further samples.  If a limit has been set using setLimit(),
	 * the number of samples read will be truncated to the limit
	 * position.  If you want to read a certain number of bytes
	 * rather than a certain number of samples, use getBuffer().
	 *
	 * @param addr A pointer to the memory area to copy the samples to.
	 * @param samples The number of samples to read.
	 * @return errSuccess if successful, !errSuccess if
	 * error.  Use getErrorStr() to retrieve the human-readable
	 * error string.
	 */
	Error getSamples(void *addr, unsigned samples);

	/**
	 * Insert samples into the file from a memory buffer.  This
	 * increments the file pointer so subsequent calls append
	 * further samples.  If you want to write a certain number of
	 * bytes rather than a certain number of samples, use
	 * putBuffer().
	 *
	 * @param addr A pointer to the memory area to append the samples
	 * from.
	 * @param samples The number of samples (not bytes) to append.
	 * @return errSuccess if successful, !errSuccess if
	 * error.  Use getErrorStr() to retrieve the human-readable
	 * error string.
	 */
	Error putSamples(void *addr, unsigned samples);
	Error skip(long samples);
	Error setPosition(unsigned long samples = ~0l);
	Error setLimit(unsigned long samples = 0l);
	Error getInfo(Info *info);
	Error setMinimum(unsigned long samples);

	/**
	 * Get the current file pointer in bytes relative to the start
	 * of the file.  See getPosition() to determine the position
	 * relative to the start of the sample buffer.
	 *
	 * @return The current file pointer in bytes relative to the
	 * start of the file.  Returns 0 if the file is not open, is
	 * empty, or an error has occured.
	 */
	unsigned long getAbsolutePosition(void);

	/**
	 * Get the current file pointer in samples relative to the
	 * start of the sample buffer.  Note that you must multiply
	 * this result by the result of a call to
	 * toBytes(info.encoding, 1) in order to determine the offset
	 * in bytes.
	 *
	 * @return the current file pointer in samples relative to the
	 * start of the sample buffer.  Returns 0 if the file is not
	 * open, is empty, or an error has occured.
	 */
	unsigned long getPosition(void);
	virtual bool isOpen(void);
	virtual bool hasPositioning(void)
		{return true;};

	inline Encoding getEncoding(void)
		{return info.encoding;};

	inline Format getFormat(void)
		{return info.format;};

	inline unsigned getSampleRate(void)
		{return info.rate;};

	inline char *getAnnotation(void)
		{return info.annotation;};

	inline Error getError(void)
		{return error;};

	inline bool operator!(void)
		{return (bool)!isOpen();};

	/**
	 * Return if the current content is signed or unsigned samples.
	 *
	 * @return true if signed.
	 */
	bool isSigned(void);
};

/**
 * AudioStream accesses AudioFile base class content as fixed frames
 * of streaming linear samples.  If a codec must be assigned to perform
 * conversion to/from linear data, AudioStream will handle conversion
 * automatically.  AudioStream will also convert between mono and stereo
 * audio content.  AudioStream uses linear samples in the native
 * machine endian format and perform endian byte swapping as needed.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Audio Streaming with Linear Conversion
 */
class __EXPORT AudioStream : public AudioFile
{
protected:
	AudioCodec *codec;	// if needed
	Encoded	framebuf;
	bool streamable;
	Linear bufferFrame;
	unsigned bufferPosition;
	unsigned bufferChannels;
	
	unsigned bufAudio(Linear samples, unsigned count, unsigned size);

public:
	AudioStream();
	AudioStream(const char *filename, Mode m = modeRead, timeout_t framing = 0);
	AudioStream(const char *filename, Info *info, bool excl = false, timeout_t framing = 0);

	virtual ~AudioStream()
		{close();};

	void open(const char *filename, Mode m = modeRead, timeout_t framing = 0);
	void create(const char *fname, Info *info, bool exclusive = false, timeout_t framing = 0);
	void close(void);
	void flush(void);

	bool isStreamable(void);

	unsigned getCount(void);	// frame count
	unsigned getMono(Linear buffer, unsigned frames = 1);
	unsigned getStereo(Linear buffer, unsigned frames = 1);
	unsigned putMono(Linear buffer, unsigned frames = 1);
	unsigned putStereo(Linear buffer, unsigned frames = 1);
	unsigned bufMono(Linear Buffer, unsigned samples);
	unsigned bufStereo(Linear Buffer, unsigned samples);
};

/**
 * The codec class is a virtual used for transcoding audio samples between
 * linear frames (or other known format) and an encoded "sample" buffer.
 * This class is only abstract and describes the core interface for
 * loadable codec modules.  This class is normally merged with AudioSample.
 * A derived AudioCodecXXX will typically include a AudioRegisterXXX static
 * class to automatically initialize and register the codec with the codec
 * registry.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short process codec interface.
 */
class __EXPORT AudioCodec : public Audio
{
protected:
	static AudioCodec *first;
	AudioCodec *next;
	Encoding encoding;
	const char *name;
	Info info;

	AudioCodec();

	// often used to create a "new" codec of a subtype based on
	// encoding format, default returns the current codec entity
	virtual AudioCodec *getByFormat(const char *format)
		{return this;};

	// get a codec by it's info descriptor.
	virtual AudioCodec *getByInfo(Info &info)
		{return this;};

public:
	AudioCodec(const char *n, Encoding e);
	virtual ~AudioCodec() {};

	/**
	 * End use of a requested codec.  If constructed then will be
	 * deleted.
	 *
	 * @param pointer to getCodec returned coded pointer.
	 */
	static void endCodec(AudioCodec *codec);

	static AudioCodec *getCodec(Encoding encoding, const char *format = NULL, bool loaded = false);
	static AudioCodec *getCodec(Info &info, bool loaded = false);

	/**
	 * Load a named codec set into process memory.
	 *
	 * @return true if successful.
	 * @param name of codec set to load.
	 */
	static bool load(const char *name);

	/**
	 * Find and load a codec file by it's encoding type.  Converts
	 * the type into a codec name and invokes the other loader...
	 *
	 * @return true if successful.
	 * @param encoding type for file.
	 */
	static bool load(Encoding e);

	/**
	 * Get the impulse energy level of a frame of X samples in
	 * the specified codec format.
	 *
 	 * @return average impulse energy of frame (sumnation).
	 * @param buffer of encoded samples.
	 * @param number of encoded samples.
	 */
	virtual Level getImpulse(void *data, unsigned samples = 0);

	/**
	 * Get the peak energy level within the frame of X samples.
	 *
	 * @return peak energy impulse in frame (largest).
	 * @param buffer of encoded samples.
	 * @param number of encoded samples.
	 */
	virtual Level getPeak(void *data, unsigned samples = 0);

	/**
	 * Signal if the current audio frame is silent.  This can be
	 * deterimed either by an impulse computation, or, in some
	 * cases, some codecs may signal and flag silent packets.
	 *
	 * @return true if silent
	 * @param threashold to use if not signaled.
	 * @param buffer of encoded samples.
	 * @param number of encoded samples.
	 */
	virtual bool isSilent(Level threashold, void *data, unsigned samples = 0);

	/**
	 * Encode a linear sample frame into the codec sample buffer.
	 *
	 * @param number of bytes written.
	 * @param buffer linear sample buffer to use.
	 * @param dest buffer to store encoded results.
	 * @param lsamples The number of linear samples.
	 */
	virtual unsigned encode(Linear buffer, void *dest, unsigned lsamples) = 0;

	/**
	 * Decode the sample frame into linear samples.
	 *
	 * @return number of bytes scanned.
	 * @param buffer sample buffer to save linear samples into.
	 * @param source for encoded data.
	 * @param number of samples to extract.
	 */
	virtual unsigned decode(Linear buffer, void *source, unsigned lsamples) = 0;

	/**
	 * Get an info description for this codec.
	 *
	 * @return info.
	 */
	inline Info getInfo(void)
		{return info;};
};

class __EXPORT AudioDevice : public Audio
{
protected:
	Info info;
	bool enabled;

public:
	virtual ~AudioDevice() {};

	virtual unsigned putSamples(Linear samples, unsigned count) = 0;
	virtual unsigned getSamples(Linear samples, unsigned count) = 0;

	virtual unsigned putEncoded(Encoded data, unsigned len)
		{return 0;};

	virtual unsigned getEncoded(Encoded data, unsigned len)
		{return 0;};

	virtual bool setEncoded(Info &info)
		{return false;};

	virtual bool setAudio(Rate rate = rate8khz, bool stereo = false, timeout_t framing = 20) = 0;

	virtual void sync(void)
		{return;};

	virtual void flush(void) = 0;

	unsigned bufMono(Linear samples, unsigned count);
	unsigned bufStereo(Linear samples, unsigned count);

	inline Info *getInfo(void)
		{return &info;};

	inline bool isEnabled(void)
		{return enabled;};
};

/**
 * DTMFDetect is used for detecting DTMF tones in a stream of audio.
 * It currently only supports 8000Hz input.
 */

class __EXPORT DTMFDetect : public Audio
{
public:
	DTMFDetect();
	~DTMFDetect();

	int putSamples(Linear buffer, int count);
	int getResult(char *buf, int max);
protected:
	void goertzelInit(goertzel_state_t *s, tone_detection_descriptor_t *t);
	void goertzelUpdate(goertzel_state_t *s, Sample x[], int samples);
	float goertzelResult(goertzel_state_t *s);
private:
	dtmf_detect_state_t *state;
	tone_detection_descriptor_t dtmf_detect_row[4];
	tone_detection_descriptor_t dtmf_detect_col[4];
	tone_detection_descriptor_t dtmf_detect_row_2nd[4];
	tone_detection_descriptor_t dtmf_detect_col_2nd[4];
	tone_detection_descriptor_t fax_detect;
	tone_detection_descriptor_t fax_detect_2nd;
};

}

#endif

