// Copyright 2023 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_PUBSUB_INTERNAL_PULL_LEASE_MANAGER_H
#define GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_PUBSUB_INTERNAL_PULL_LEASE_MANAGER_H

#include "google/cloud/pubsub/internal/subscriber_stub.h"
#include "google/cloud/pubsub/subscription.h"
#include "google/cloud/completion_queue.h"
#include "google/cloud/future.h"
#include "google/cloud/options.h"
#include "google/cloud/status_or.h"
#include <chrono>
#include <functional>
#include <memory>
#include <mutex>

namespace google {
namespace cloud {
namespace pubsub_internal {
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_BEGIN

/**
 * Interface to maintain the lease for a single message.
 */
class PullLeaseManager {
 public:
  virtual ~PullLeaseManager() = default;

  virtual void StartLeaseLoop() = 0;

  virtual std::chrono::milliseconds LeaseRefreshPeriod() const = 0;

  virtual future<Status> ExtendLease(std::shared_ptr<SubscriberStub> stub,
                                     std::chrono::system_clock::time_point now,
                                     std::chrono::seconds extension) = 0;

  virtual std::string ack_id() const { return ""; }

  virtual pubsub::Subscription subscription() const {
    return pubsub::Subscription{};
  }
};

/**
 * Interface to make a modify ack deadline rpc request.
 */
class PullLeaseManagerImpl {
 public:
  virtual ~PullLeaseManagerImpl() = default;
  virtual future<Status> AsyncModifyAckDeadline(
      std::shared_ptr<SubscriberStub> stub, google::cloud::CompletionQueue& cq,
      std::shared_ptr<grpc::ClientContext> context,
      google::cloud::internal::ImmutableOptions options,
      google::pubsub::v1::ModifyAckDeadlineRequest const& request) = 0;
};

GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_END
}  // namespace pubsub_internal
}  // namespace cloud
}  // namespace google

#endif  // GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_PUBSUB_INTERNAL_PULL_LEASE_MANAGER_H
