package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Task}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableTask.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableTask implements Task {
  private final String id;
  private final Version version;
  private final Date createdAt;
  private final Date updatedAt;
  private final @Nullable String name;
  private final @Nullable Map<String, String> labels;
  private final TaskSpec spec;
  private final String serviceId;
  private final @Nullable Integer slot;
  private final @Nullable String nodeId;
  private final TaskStatus status;
  private final String desiredState;
  private final @Nullable List<NetworkAttachment> networkAttachments;
  private final @Nullable Version jobIteration;

  private ImmutableTask(
      String id,
      Version version,
      Date createdAt,
      Date updatedAt,
      @Nullable String name,
      @Nullable Map<String, String> labels,
      TaskSpec spec,
      String serviceId,
      @Nullable Integer slot,
      @Nullable String nodeId,
      TaskStatus status,
      String desiredState,
      @Nullable List<NetworkAttachment> networkAttachments,
      @Nullable Version jobIteration) {
    this.id = id;
    this.version = version;
    this.createdAt = createdAt;
    this.updatedAt = updatedAt;
    this.name = name;
    this.labels = labels;
    this.spec = spec;
    this.serviceId = serviceId;
    this.slot = slot;
    this.nodeId = nodeId;
    this.status = status;
    this.desiredState = desiredState;
    this.networkAttachments = networkAttachments;
    this.jobIteration = jobIteration;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("ID")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code version} attribute
   */
  @JsonProperty("Version")
  @Override
  public Version version() {
    return version;
  }

  /**
   * @return The value of the {@code createdAt} attribute
   */
  @JsonProperty("CreatedAt")
  @Override
  public Date createdAt() {
    return createdAt;
  }

  /**
   * @return The value of the {@code updatedAt} attribute
   */
  @JsonProperty("UpdatedAt")
  @Override
  public Date updatedAt() {
    return updatedAt;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public @Nullable String name() {
    return name;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code spec} attribute
   */
  @JsonProperty("Spec")
  @Override
  public TaskSpec spec() {
    return spec;
  }

  /**
   * @return The value of the {@code serviceId} attribute
   */
  @JsonProperty("ServiceID")
  @Override
  public String serviceId() {
    return serviceId;
  }

  /**
   * @return The value of the {@code slot} attribute
   */
  @JsonProperty("Slot")
  @Override
  public @Nullable Integer slot() {
    return slot;
  }

  /**
   * @return The value of the {@code nodeId} attribute
   */
  @JsonProperty("NodeID")
  @Override
  public @Nullable String nodeId() {
    return nodeId;
  }

  /**
   * @return The value of the {@code status} attribute
   */
  @JsonProperty("Status")
  @Override
  public TaskStatus status() {
    return status;
  }

  /**
   * @return The value of the {@code desiredState} attribute
   */
  @JsonProperty("DesiredState")
  @Override
  public String desiredState() {
    return desiredState;
  }

  /**
   * @return The value of the {@code networkAttachments} attribute
   */
  @JsonProperty("NetworksAttachments")
  @Override
  public @Nullable List<NetworkAttachment> networkAttachments() {
    return networkAttachments;
  }

  /**
   * @return The value of the {@code jobIteration} attribute
   */
  @JsonProperty("JobIteration")
  @Override
  public @Nullable Version jobIteration() {
    return jobIteration;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableTask(
        newValue,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#version() version} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for version
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withVersion(Version value) {
    if (this.version == value) return this;
    Version newValue = Objects.requireNonNull(value, "version");
    return new ImmutableTask(
        this.id,
        newValue,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#createdAt() createdAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for createdAt
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withCreatedAt(Date value) {
    if (this.createdAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "createdAt");
    return new ImmutableTask(
        this.id,
        this.version,
        newValue,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#updatedAt() updatedAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for updatedAt
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withUpdatedAt(Date value) {
    if (this.updatedAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "updatedAt");
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        newValue,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withName(@Nullable String value) {
    if (Objects.equals(this.name, value)) return this;
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        value,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by replacing the {@link Task#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the labels map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableTask withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        newValue,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#spec() spec} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for spec
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withSpec(TaskSpec value) {
    if (this.spec == value) return this;
    TaskSpec newValue = Objects.requireNonNull(value, "spec");
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        newValue,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#serviceId() serviceId} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for serviceId
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withServiceId(String value) {
    String newValue = Objects.requireNonNull(value, "serviceId");
    if (this.serviceId.equals(newValue)) return this;
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        newValue,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#slot() slot} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for slot (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withSlot(@Nullable Integer value) {
    if (Objects.equals(this.slot, value)) return this;
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        value,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#nodeId() nodeId} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for nodeId (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withNodeId(@Nullable String value) {
    if (Objects.equals(this.nodeId, value)) return this;
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        value,
        this.status,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#status() status} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for status
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withStatus(TaskStatus value) {
    if (this.status == value) return this;
    TaskStatus newValue = Objects.requireNonNull(value, "status");
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        newValue,
        this.desiredState,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#desiredState() desiredState} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for desiredState
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withDesiredState(String value) {
    String newValue = Objects.requireNonNull(value, "desiredState");
    if (this.desiredState.equals(newValue)) return this;
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        newValue,
        this.networkAttachments,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Task#networkAttachments() networkAttachments}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableTask withNetworkAttachments(@Nullable NetworkAttachment... elements) {
    if (elements == null) {
      return new ImmutableTask(
          this.id,
          this.version,
          this.createdAt,
          this.updatedAt,
          this.name,
          this.labels,
          this.spec,
          this.serviceId,
          this.slot,
          this.nodeId,
          this.status,
          this.desiredState,
          null,
          this.jobIteration);
    }
    @Nullable List<NetworkAttachment> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        newValue,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Task#networkAttachments() networkAttachments}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of networkAttachments elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableTask withNetworkAttachments(@Nullable Iterable<? extends NetworkAttachment> elements) {
    if (this.networkAttachments == elements) return this;
    @Nullable List<NetworkAttachment> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        newValue,
        this.jobIteration);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Task#jobIteration() jobIteration} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for jobIteration (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTask withJobIteration(@Nullable Version value) {
    if (this.jobIteration == value) return this;
    return new ImmutableTask(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.name,
        this.labels,
        this.spec,
        this.serviceId,
        this.slot,
        this.nodeId,
        this.status,
        this.desiredState,
        this.networkAttachments,
        value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableTask} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableTask
        && equalTo(0, (ImmutableTask) another);
  }

  private boolean equalTo(int synthetic, ImmutableTask another) {
    return id.equals(another.id)
        && version.equals(another.version)
        && createdAt.equals(another.createdAt)
        && updatedAt.equals(another.updatedAt)
        && Objects.equals(name, another.name)
        && Objects.equals(labels, another.labels)
        && spec.equals(another.spec)
        && serviceId.equals(another.serviceId)
        && Objects.equals(slot, another.slot)
        && Objects.equals(nodeId, another.nodeId)
        && status.equals(another.status)
        && desiredState.equals(another.desiredState)
        && Objects.equals(networkAttachments, another.networkAttachments)
        && Objects.equals(jobIteration, another.jobIteration);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code version}, {@code createdAt}, {@code updatedAt}, {@code name}, {@code labels}, {@code spec}, {@code serviceId}, {@code slot}, {@code nodeId}, {@code status}, {@code desiredState}, {@code networkAttachments}, {@code jobIteration}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + version.hashCode();
    h += (h << 5) + createdAt.hashCode();
    h += (h << 5) + updatedAt.hashCode();
    h += (h << 5) + Objects.hashCode(name);
    h += (h << 5) + Objects.hashCode(labels);
    h += (h << 5) + spec.hashCode();
    h += (h << 5) + serviceId.hashCode();
    h += (h << 5) + Objects.hashCode(slot);
    h += (h << 5) + Objects.hashCode(nodeId);
    h += (h << 5) + status.hashCode();
    h += (h << 5) + desiredState.hashCode();
    h += (h << 5) + Objects.hashCode(networkAttachments);
    h += (h << 5) + Objects.hashCode(jobIteration);
    return h;
  }

  /**
   * Prints the immutable value {@code Task} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Task{"
        + "id=" + id
        + ", version=" + version
        + ", createdAt=" + createdAt
        + ", updatedAt=" + updatedAt
        + ", name=" + name
        + ", labels=" + labels
        + ", spec=" + spec
        + ", serviceId=" + serviceId
        + ", slot=" + slot
        + ", nodeId=" + nodeId
        + ", status=" + status
        + ", desiredState=" + desiredState
        + ", networkAttachments=" + networkAttachments
        + ", jobIteration=" + jobIteration
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Task} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Task instance
   */
  public static ImmutableTask copyOf(Task instance) {
    if (instance instanceof ImmutableTask) {
      return (ImmutableTask) instance;
    }
    return ImmutableTask.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableTask ImmutableTask}.
   * <pre>
   * ImmutableTask.builder()
   *    .id(String) // required {@link Task#id() id}
   *    .version(org.mandas.docker.client.messages.swarm.Version) // required {@link Task#version() version}
   *    .createdAt(Date) // required {@link Task#createdAt() createdAt}
   *    .updatedAt(Date) // required {@link Task#updatedAt() updatedAt}
   *    .name(String | null) // nullable {@link Task#name() name}
   *    .labels(Map&amp;lt;String, String&amp;gt; | null) // nullable {@link Task#labels() labels}
   *    .spec(org.mandas.docker.client.messages.swarm.TaskSpec) // required {@link Task#spec() spec}
   *    .serviceId(String) // required {@link Task#serviceId() serviceId}
   *    .slot(Integer | null) // nullable {@link Task#slot() slot}
   *    .nodeId(String | null) // nullable {@link Task#nodeId() nodeId}
   *    .status(org.mandas.docker.client.messages.swarm.TaskStatus) // required {@link Task#status() status}
   *    .desiredState(String) // required {@link Task#desiredState() desiredState}
   *    .networkAttachments(List&amp;lt;org.mandas.docker.client.messages.swarm.NetworkAttachment&amp;gt; | null) // nullable {@link Task#networkAttachments() networkAttachments}
   *    .jobIteration(org.mandas.docker.client.messages.swarm.Version | null) // nullable {@link Task#jobIteration() jobIteration}
   *    .build();
   * </pre>
   * @return A new ImmutableTask builder
   */
  public static ImmutableTask.Builder builder() {
    return new ImmutableTask.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableTask ImmutableTask}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private static final long INIT_BIT_VERSION = 0x2L;
    private static final long INIT_BIT_CREATED_AT = 0x4L;
    private static final long INIT_BIT_UPDATED_AT = 0x8L;
    private static final long INIT_BIT_SPEC = 0x10L;
    private static final long INIT_BIT_SERVICE_ID = 0x20L;
    private static final long INIT_BIT_STATUS = 0x40L;
    private static final long INIT_BIT_DESIRED_STATE = 0x80L;
    private long initBits = 0xffL;

    private String id;
    private Version version;
    private Date createdAt;
    private Date updatedAt;
    private String name;
    private Map<String, String> labels = null;
    private TaskSpec spec;
    private String serviceId;
    private Integer slot;
    private String nodeId;
    private TaskStatus status;
    private String desiredState;
    private List<NetworkAttachment> networkAttachments = null;
    private Version jobIteration;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Task} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Task instance) {
      Objects.requireNonNull(instance, "instance");
      this.id(instance.id());
      this.version(instance.version());
      this.createdAt(instance.createdAt());
      this.updatedAt(instance.updatedAt());
      @Nullable String nameValue = instance.name();
      if (nameValue != null) {
        name(nameValue);
      }
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      this.spec(instance.spec());
      this.serviceId(instance.serviceId());
      @Nullable Integer slotValue = instance.slot();
      if (slotValue != null) {
        slot(slotValue);
      }
      @Nullable String nodeIdValue = instance.nodeId();
      if (nodeIdValue != null) {
        nodeId(nodeIdValue);
      }
      this.status(instance.status());
      this.desiredState(instance.desiredState());
      @Nullable List<NetworkAttachment> networkAttachmentsValue = instance.networkAttachments();
      if (networkAttachmentsValue != null) {
        addAllNetworkAttachments(networkAttachmentsValue);
      }
      @Nullable Version jobIterationValue = instance.jobIteration();
      if (jobIterationValue != null) {
        jobIteration(jobIterationValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Task#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ID")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#version() version} attribute.
     * @param version The value for version 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Version")
    public final Builder version(Version version) {
      this.version = Objects.requireNonNull(version, "version");
      initBits &= ~INIT_BIT_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#createdAt() createdAt} attribute.
     * @param createdAt The value for createdAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("CreatedAt")
    public final Builder createdAt(Date createdAt) {
      this.createdAt = Objects.requireNonNull(createdAt, "createdAt");
      initBits &= ~INIT_BIT_CREATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#updatedAt() updatedAt} attribute.
     * @param updatedAt The value for updatedAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("UpdatedAt")
    public final Builder updatedAt(Date updatedAt) {
      this.updatedAt = Objects.requireNonNull(updatedAt, "updatedAt");
      initBits &= ~INIT_BIT_UPDATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#name() name} attribute.
     * @param name The value for name (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(@Nullable String name) {
      this.name = name;
      return this;
    }

    /**
     * Put one entry to the {@link Task#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link Task#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link Task#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link Task#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Task#spec() spec} attribute.
     * @param spec The value for spec 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Spec")
    public final Builder spec(TaskSpec spec) {
      this.spec = Objects.requireNonNull(spec, "spec");
      initBits &= ~INIT_BIT_SPEC;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#serviceId() serviceId} attribute.
     * @param serviceId The value for serviceId 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ServiceID")
    public final Builder serviceId(String serviceId) {
      this.serviceId = Objects.requireNonNull(serviceId, "serviceId");
      initBits &= ~INIT_BIT_SERVICE_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#slot() slot} attribute.
     * @param slot The value for slot (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Slot")
    public final Builder slot(@Nullable Integer slot) {
      this.slot = slot;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#nodeId() nodeId} attribute.
     * @param nodeId The value for nodeId (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("NodeID")
    public final Builder nodeId(@Nullable String nodeId) {
      this.nodeId = nodeId;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#status() status} attribute.
     * @param status The value for status 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Status")
    public final Builder status(TaskStatus status) {
      this.status = Objects.requireNonNull(status, "status");
      initBits &= ~INIT_BIT_STATUS;
      return this;
    }

    /**
     * Initializes the value for the {@link Task#desiredState() desiredState} attribute.
     * @param desiredState The value for desiredState 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("DesiredState")
    public final Builder desiredState(String desiredState) {
      this.desiredState = Objects.requireNonNull(desiredState, "desiredState");
      initBits &= ~INIT_BIT_DESIRED_STATE;
      return this;
    }

    /**
     * Adds one element to {@link Task#networkAttachments() networkAttachments} list.
     * @param element A networkAttachments element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder networkAttachment(NetworkAttachment element) {
      if (this.networkAttachments == null) {
        this.networkAttachments = new ArrayList<NetworkAttachment>();
      }
      this.networkAttachments.add(Objects.requireNonNull(element, "networkAttachments element"));
      return this;
    }

    /**
     * Adds elements to {@link Task#networkAttachments() networkAttachments} list.
     * @param elements An array of networkAttachments elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder networkAttachments(NetworkAttachment... elements) {
      if (this.networkAttachments == null) {
        this.networkAttachments = new ArrayList<NetworkAttachment>();
      }
      for (NetworkAttachment element : elements) {
        this.networkAttachments.add(Objects.requireNonNull(element, "networkAttachments element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Task#networkAttachments() networkAttachments} list.
     * @param elements An iterable of networkAttachments elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("NetworksAttachments")
    public final Builder networkAttachments(@Nullable Iterable<? extends NetworkAttachment> elements) {
      if (elements == null) {
        this.networkAttachments = null;
        return this;
      }
      this.networkAttachments = new ArrayList<NetworkAttachment>();
      return addAllNetworkAttachments(elements);
    }

    /**
     * Adds elements to {@link Task#networkAttachments() networkAttachments} list.
     * @param elements An iterable of networkAttachments elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllNetworkAttachments(Iterable<? extends NetworkAttachment> elements) {
      Objects.requireNonNull(elements, "networkAttachments element");
      if (this.networkAttachments == null) {
        this.networkAttachments = new ArrayList<NetworkAttachment>();
      }
      for (NetworkAttachment element : elements) {
        this.networkAttachments.add(Objects.requireNonNull(element, "networkAttachments element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Task#jobIteration() jobIteration} attribute.
     * @param jobIteration The value for jobIteration (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("JobIteration")
    public final Builder jobIteration(@Nullable Version jobIteration) {
      this.jobIteration = jobIteration;
      return this;
    }

    /**
     * Builds a new {@link ImmutableTask ImmutableTask}.
     * @return An immutable instance of Task
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableTask build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableTask(
          id,
          version,
          createdAt,
          updatedAt,
          name,
          labels == null ? null : createUnmodifiableMap(false, false, labels),
          spec,
          serviceId,
          slot,
          nodeId,
          status,
          desiredState,
          networkAttachments == null ? null : createUnmodifiableList(true, networkAttachments),
          jobIteration);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_VERSION) != 0) attributes.add("version");
      if ((initBits & INIT_BIT_CREATED_AT) != 0) attributes.add("createdAt");
      if ((initBits & INIT_BIT_UPDATED_AT) != 0) attributes.add("updatedAt");
      if ((initBits & INIT_BIT_SPEC) != 0) attributes.add("spec");
      if ((initBits & INIT_BIT_SERVICE_ID) != 0) attributes.add("serviceId");
      if ((initBits & INIT_BIT_STATUS) != 0) attributes.add("status");
      if ((initBits & INIT_BIT_DESIRED_STATE) != 0) attributes.add("desiredState");
      return "Cannot build Task, some of required attributes are not set " + attributes;
    }
  }

  /**
   * Immutable implementation of {@link Task.Criteria}.
   * <p>
   * Use the builder to create immutable instances:
   * {@code ImmutableTask.Criteria.builder()}.
   */
  static final class Criteria implements Task.Criteria {
    private final @Nullable String taskId;
    private final @Nullable String taskName;
    private final @Nullable String serviceName;
    private final @Nullable String nodeId;
    private final @Nullable String label;
    private final @Nullable String desiredState;

    private Criteria(
        @Nullable String taskId,
        @Nullable String taskName,
        @Nullable String serviceName,
        @Nullable String nodeId,
        @Nullable String label,
        @Nullable String desiredState) {
      this.taskId = taskId;
      this.taskName = taskName;
      this.serviceName = serviceName;
      this.nodeId = nodeId;
      this.label = label;
      this.desiredState = desiredState;
    }

    /**
     * @return The value of the {@code taskId} attribute
     */
    @JsonProperty("taskId")
    @Override
    public @Nullable String taskId() {
      return taskId;
    }

    /**
     * @return The value of the {@code taskName} attribute
     */
    @JsonProperty("taskName")
    @Override
    public @Nullable String taskName() {
      return taskName;
    }

    /**
     * @return The value of the {@code serviceName} attribute
     */
    @JsonProperty("serviceName")
    @Override
    public @Nullable String serviceName() {
      return serviceName;
    }

    /**
     * @return The value of the {@code nodeId} attribute
     */
    @JsonProperty("nodeId")
    @Override
    public @Nullable String nodeId() {
      return nodeId;
    }

    /**
     * @return The value of the {@code label} attribute
     */
    @JsonProperty("label")
    @Override
    public @Nullable String label() {
      return label;
    }

    /**
     * @return The value of the {@code desiredState} attribute
     */
    @JsonProperty("desiredState")
    @Override
    public @Nullable String desiredState() {
      return desiredState;
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Task.Criteria#taskId() taskId} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for taskId (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableTask.Criteria withTaskId(@Nullable String value) {
      if (Objects.equals(this.taskId, value)) return this;
      return new ImmutableTask.Criteria(value, this.taskName, this.serviceName, this.nodeId, this.label, this.desiredState);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Task.Criteria#taskName() taskName} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for taskName (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableTask.Criteria withTaskName(@Nullable String value) {
      if (Objects.equals(this.taskName, value)) return this;
      return new ImmutableTask.Criteria(this.taskId, value, this.serviceName, this.nodeId, this.label, this.desiredState);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Task.Criteria#serviceName() serviceName} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for serviceName (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableTask.Criteria withServiceName(@Nullable String value) {
      if (Objects.equals(this.serviceName, value)) return this;
      return new ImmutableTask.Criteria(this.taskId, this.taskName, value, this.nodeId, this.label, this.desiredState);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Task.Criteria#nodeId() nodeId} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for nodeId (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableTask.Criteria withNodeId(@Nullable String value) {
      if (Objects.equals(this.nodeId, value)) return this;
      return new ImmutableTask.Criteria(this.taskId, this.taskName, this.serviceName, value, this.label, this.desiredState);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Task.Criteria#label() label} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for label (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableTask.Criteria withLabel(@Nullable String value) {
      if (Objects.equals(this.label, value)) return this;
      return new ImmutableTask.Criteria(this.taskId, this.taskName, this.serviceName, this.nodeId, value, this.desiredState);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Task.Criteria#desiredState() desiredState} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for desiredState (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableTask.Criteria withDesiredState(@Nullable String value) {
      if (Objects.equals(this.desiredState, value)) return this;
      return new ImmutableTask.Criteria(this.taskId, this.taskName, this.serviceName, this.nodeId, this.label, value);
    }

    /**
     * This instance is equal to all instances of {@code Criteria} that have equal attribute values.
     * @return {@code true} if {@code this} is equal to {@code another} instance
     */
    @Override
    public boolean equals(Object another) {
      if (this == another) return true;
      return another instanceof ImmutableTask.Criteria
          && equalTo(0, (ImmutableTask.Criteria) another);
    }

    private boolean equalTo(int synthetic, ImmutableTask.Criteria another) {
      return Objects.equals(taskId, another.taskId)
          && Objects.equals(taskName, another.taskName)
          && Objects.equals(serviceName, another.serviceName)
          && Objects.equals(nodeId, another.nodeId)
          && Objects.equals(label, another.label)
          && Objects.equals(desiredState, another.desiredState);
    }

    /**
     * Computes a hash code from attributes: {@code taskId}, {@code taskName}, {@code serviceName}, {@code nodeId}, {@code label}, {@code desiredState}.
     * @return hashCode value
     */
    @Override
    public int hashCode() {
      int h = 5381;
      h += (h << 5) + Objects.hashCode(taskId);
      h += (h << 5) + Objects.hashCode(taskName);
      h += (h << 5) + Objects.hashCode(serviceName);
      h += (h << 5) + Objects.hashCode(nodeId);
      h += (h << 5) + Objects.hashCode(label);
      h += (h << 5) + Objects.hashCode(desiredState);
      return h;
    }

    /**
     * Prints the immutable value {@code Criteria} with attribute values.
     * @return A string representation of the value
     */
    @Override
    public String toString() {
      return "Criteria{"
          + "taskId=" + taskId
          + ", taskName=" + taskName
          + ", serviceName=" + serviceName
          + ", nodeId=" + nodeId
          + ", label=" + label
          + ", desiredState=" + desiredState
          + "}";
    }

    /**
     * Creates an immutable copy of a {@link Task.Criteria} value.
     * Uses accessors to get values to initialize the new immutable instance.
     * If an instance is already immutable, it is returned as is.
     * @param instance The instance to copy
     * @return A copied immutable Criteria instance
     */
    public static ImmutableTask.Criteria copyOf(Task.Criteria instance) {
      if (instance instanceof ImmutableTask.Criteria) {
        return (ImmutableTask.Criteria) instance;
      }
      return ImmutableTask.Criteria.builder()
          .from(instance)
          .build();
    }

    /**
     * Creates a builder for {@link ImmutableTask.Criteria Criteria}.
     * <pre>
     * ImmutableTask.Criteria.builder()
     *    .taskId(String | null) // nullable {@link Task.Criteria#taskId() taskId}
     *    .taskName(String | null) // nullable {@link Task.Criteria#taskName() taskName}
     *    .serviceName(String | null) // nullable {@link Task.Criteria#serviceName() serviceName}
     *    .nodeId(String | null) // nullable {@link Task.Criteria#nodeId() nodeId}
     *    .label(String | null) // nullable {@link Task.Criteria#label() label}
     *    .desiredState(String | null) // nullable {@link Task.Criteria#desiredState() desiredState}
     *    .build();
     * </pre>
     * @return A new Criteria builder
     */
    public static ImmutableTask.Criteria.Builder builder() {
      return new ImmutableTask.Criteria.Builder();
    }

    /**
     * Builds instances of type {@link ImmutableTask.Criteria Criteria}.
     * Initialize attributes and then invoke the {@link #build()} method to create an
     * immutable instance.
     * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
     * but instead used immediately to create instances.</em>
     */
    static final class Builder implements Task.Criteria.Builder {
      private String taskId;
      private String taskName;
      private String serviceName;
      private String nodeId;
      private String label;
      private String desiredState;

      private Builder() {
      }

      /**
       * Fill a builder with attribute values from the provided {@code Criteria} instance.
       * Regular attribute values will be replaced with those from the given instance.
       * Absent optional values will not replace present values.
       * @param instance The instance from which to copy values
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder from(Task.Criteria instance) {
        Objects.requireNonNull(instance, "instance");
        @Nullable String taskIdValue = instance.taskId();
        if (taskIdValue != null) {
          taskId(taskIdValue);
        }
        @Nullable String taskNameValue = instance.taskName();
        if (taskNameValue != null) {
          taskName(taskNameValue);
        }
        @Nullable String serviceNameValue = instance.serviceName();
        if (serviceNameValue != null) {
          serviceName(serviceNameValue);
        }
        @Nullable String nodeIdValue = instance.nodeId();
        if (nodeIdValue != null) {
          nodeId(nodeIdValue);
        }
        @Nullable String labelValue = instance.label();
        if (labelValue != null) {
          label(labelValue);
        }
        @Nullable String desiredStateValue = instance.desiredState();
        if (desiredStateValue != null) {
          desiredState(desiredStateValue);
        }
        return this;
      }

      /**
       * Initializes the value for the {@link Task.Criteria#taskId() taskId} attribute.
       * @param taskId The value for taskId (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("taskId")
      public final Builder taskId(@Nullable String taskId) {
        this.taskId = taskId;
        return this;
      }

      /**
       * Initializes the value for the {@link Task.Criteria#taskName() taskName} attribute.
       * @param taskName The value for taskName (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("taskName")
      public final Builder taskName(@Nullable String taskName) {
        this.taskName = taskName;
        return this;
      }

      /**
       * Initializes the value for the {@link Task.Criteria#serviceName() serviceName} attribute.
       * @param serviceName The value for serviceName (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("serviceName")
      public final Builder serviceName(@Nullable String serviceName) {
        this.serviceName = serviceName;
        return this;
      }

      /**
       * Initializes the value for the {@link Task.Criteria#nodeId() nodeId} attribute.
       * @param nodeId The value for nodeId (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("nodeId")
      public final Builder nodeId(@Nullable String nodeId) {
        this.nodeId = nodeId;
        return this;
      }

      /**
       * Initializes the value for the {@link Task.Criteria#label() label} attribute.
       * @param label The value for label (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("label")
      public final Builder label(@Nullable String label) {
        this.label = label;
        return this;
      }

      /**
       * Initializes the value for the {@link Task.Criteria#desiredState() desiredState} attribute.
       * @param desiredState The value for desiredState (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("desiredState")
      public final Builder desiredState(@Nullable String desiredState) {
        this.desiredState = desiredState;
        return this;
      }

      /**
       * Builds a new {@link ImmutableTask.Criteria Criteria}.
       * @return An immutable instance of Criteria
       * @throws java.lang.IllegalStateException if any required attributes are missing
       */
      public ImmutableTask.Criteria build() {
        return new ImmutableTask.Criteria(taskId, taskName, serviceName, nodeId, label, desiredState);
      }
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
