package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link NodeStatus}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNodeStatus.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNodeStatus implements NodeStatus {
  private final String state;
  private final @Nullable String addr;

  private ImmutableNodeStatus(String state, @Nullable String addr) {
    this.state = state;
    this.addr = addr;
  }

  /**
   * @return The value of the {@code state} attribute
   */
  @JsonProperty("State")
  @Override
  public String state() {
    return state;
  }

  /**
   * @return The value of the {@code addr} attribute
   */
  @JsonProperty("Addr")
  @Override
  public @Nullable String addr() {
    return addr;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeStatus#state() state} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for state
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNodeStatus withState(String value) {
    String newValue = Objects.requireNonNull(value, "state");
    if (this.state.equals(newValue)) return this;
    return new ImmutableNodeStatus(newValue, this.addr);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeStatus#addr() addr} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for addr (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNodeStatus withAddr(@Nullable String value) {
    if (Objects.equals(this.addr, value)) return this;
    return new ImmutableNodeStatus(this.state, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNodeStatus} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNodeStatus
        && equalTo(0, (ImmutableNodeStatus) another);
  }

  private boolean equalTo(int synthetic, ImmutableNodeStatus another) {
    return state.equals(another.state)
        && Objects.equals(addr, another.addr);
  }

  /**
   * Computes a hash code from attributes: {@code state}, {@code addr}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + state.hashCode();
    h += (h << 5) + Objects.hashCode(addr);
    return h;
  }

  /**
   * Prints the immutable value {@code NodeStatus} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NodeStatus{"
        + "state=" + state
        + ", addr=" + addr
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NodeStatus} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NodeStatus instance
   */
  public static ImmutableNodeStatus copyOf(NodeStatus instance) {
    if (instance instanceof ImmutableNodeStatus) {
      return (ImmutableNodeStatus) instance;
    }
    return ImmutableNodeStatus.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNodeStatus ImmutableNodeStatus}.
   * <pre>
   * ImmutableNodeStatus.builder()
   *    .state(String) // required {@link NodeStatus#state() state}
   *    .addr(String | null) // nullable {@link NodeStatus#addr() addr}
   *    .build();
   * </pre>
   * @return A new ImmutableNodeStatus builder
   */
  public static ImmutableNodeStatus.Builder builder() {
    return new ImmutableNodeStatus.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNodeStatus ImmutableNodeStatus}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_STATE = 0x1L;
    private long initBits = 0x1L;

    private String state;
    private String addr;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NodeStatus} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NodeStatus instance) {
      Objects.requireNonNull(instance, "instance");
      this.state(instance.state());
      @Nullable String addrValue = instance.addr();
      if (addrValue != null) {
        addr(addrValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NodeStatus#state() state} attribute.
     * @param state The value for state 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("State")
    public final Builder state(String state) {
      this.state = Objects.requireNonNull(state, "state");
      initBits &= ~INIT_BIT_STATE;
      return this;
    }

    /**
     * Initializes the value for the {@link NodeStatus#addr() addr} attribute.
     * @param addr The value for addr (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Addr")
    public final Builder addr(@Nullable String addr) {
      this.addr = addr;
      return this;
    }

    /**
     * Builds a new {@link ImmutableNodeStatus ImmutableNodeStatus}.
     * @return An immutable instance of NodeStatus
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNodeStatus build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNodeStatus(state, addr);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_STATE) != 0) attributes.add("state");
      return "Cannot build NodeStatus, some of required attributes are not set " + attributes;
    }
  }
}
