package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Distribution}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableDistribution.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableDistribution implements Distribution {
  private final Descriptor descriptor;
  private final @Nullable List<Platform> platforms;

  private ImmutableDistribution(
      Descriptor descriptor,
      @Nullable List<Platform> platforms) {
    this.descriptor = descriptor;
    this.platforms = platforms;
  }

  /**
   * @return The value of the {@code descriptor} attribute
   */
  @JsonProperty("Descriptor")
  @Override
  public Descriptor descriptor() {
    return descriptor;
  }

  /**
   * @return The value of the {@code platforms} attribute
   */
  @JsonProperty("Platforms")
  @Override
  public @Nullable List<Platform> platforms() {
    return platforms;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Distribution#descriptor() descriptor} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for descriptor
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDistribution withDescriptor(Descriptor value) {
    if (this.descriptor == value) return this;
    Descriptor newValue = Objects.requireNonNull(value, "descriptor");
    return new ImmutableDistribution(newValue, this.platforms);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Distribution#platforms() platforms}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableDistribution withPlatforms(@Nullable Platform... elements) {
    if (elements == null) {
      return new ImmutableDistribution(this.descriptor, null);
    }
    @Nullable List<Platform> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableDistribution(this.descriptor, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Distribution#platforms() platforms}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of platforms elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableDistribution withPlatforms(@Nullable Iterable<? extends Platform> elements) {
    if (this.platforms == elements) return this;
    @Nullable List<Platform> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableDistribution(this.descriptor, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableDistribution} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableDistribution
        && equalTo(0, (ImmutableDistribution) another);
  }

  private boolean equalTo(int synthetic, ImmutableDistribution another) {
    return descriptor.equals(another.descriptor)
        && Objects.equals(platforms, another.platforms);
  }

  /**
   * Computes a hash code from attributes: {@code descriptor}, {@code platforms}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + descriptor.hashCode();
    h += (h << 5) + Objects.hashCode(platforms);
    return h;
  }

  /**
   * Prints the immutable value {@code Distribution} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Distribution{"
        + "descriptor=" + descriptor
        + ", platforms=" + platforms
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Distribution} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Distribution instance
   */
  public static ImmutableDistribution copyOf(Distribution instance) {
    if (instance instanceof ImmutableDistribution) {
      return (ImmutableDistribution) instance;
    }
    return ImmutableDistribution.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableDistribution ImmutableDistribution}.
   * <pre>
   * ImmutableDistribution.builder()
   *    .descriptor(org.mandas.docker.client.messages.Descriptor) // required {@link Distribution#descriptor() descriptor}
   *    .platforms(List&amp;lt;org.mandas.docker.client.messages.Platform&amp;gt; | null) // nullable {@link Distribution#platforms() platforms}
   *    .build();
   * </pre>
   * @return A new ImmutableDistribution builder
   */
  public static ImmutableDistribution.Builder builder() {
    return new ImmutableDistribution.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableDistribution ImmutableDistribution}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements Distribution.Builder {
    private static final long INIT_BIT_DESCRIPTOR = 0x1L;
    private long initBits = 0x1L;

    private Descriptor descriptor;
    private List<Platform> platforms = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Distribution} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Distribution instance) {
      Objects.requireNonNull(instance, "instance");
      this.descriptor(instance.descriptor());
      @Nullable List<Platform> platformsValue = instance.platforms();
      if (platformsValue != null) {
        addAllPlatforms(platformsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Distribution#descriptor() descriptor} attribute.
     * @param descriptor The value for descriptor 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Descriptor")
    public final Builder descriptor(Descriptor descriptor) {
      this.descriptor = Objects.requireNonNull(descriptor, "descriptor");
      initBits &= ~INIT_BIT_DESCRIPTOR;
      return this;
    }

    /**
     * Adds one element to {@link Distribution#platforms() platforms} list.
     * @param element A platforms element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder platform(Platform element) {
      if (this.platforms == null) {
        this.platforms = new ArrayList<Platform>();
      }
      this.platforms.add(Objects.requireNonNull(element, "platforms element"));
      return this;
    }

    /**
     * Adds elements to {@link Distribution#platforms() platforms} list.
     * @param elements An array of platforms elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder platforms(Platform... elements) {
      if (this.platforms == null) {
        this.platforms = new ArrayList<Platform>();
      }
      for (Platform element : elements) {
        this.platforms.add(Objects.requireNonNull(element, "platforms element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Distribution#platforms() platforms} list.
     * @param elements An iterable of platforms elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Platforms")
    public final Builder platforms(@Nullable Iterable<? extends Platform> elements) {
      if (elements == null) {
        this.platforms = null;
        return this;
      }
      this.platforms = new ArrayList<Platform>();
      return addAllPlatforms(elements);
    }

    /**
     * Adds elements to {@link Distribution#platforms() platforms} list.
     * @param elements An iterable of platforms elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllPlatforms(Iterable<? extends Platform> elements) {
      Objects.requireNonNull(elements, "platforms element");
      if (this.platforms == null) {
        this.platforms = new ArrayList<Platform>();
      }
      for (Platform element : elements) {
        this.platforms.add(Objects.requireNonNull(element, "platforms element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableDistribution ImmutableDistribution}.
     * @return An immutable instance of Distribution
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableDistribution build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableDistribution(descriptor, platforms == null ? null : createUnmodifiableList(true, platforms));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DESCRIPTOR) != 0) attributes.add("descriptor");
      return "Cannot build Distribution, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
