/**
 * File:    ExportJWTTransformationDesc.java
 * Created: 26.11.2008
 *
 *
/*******************************************************************************
 * Copyright (c) 2008  Open Wide (www.openwide.fr)
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Mickael Istria, Open Wide, Lyon, France
 *      - Creation and implementation
 *      - Refactoring to fit with jwt-we extensions
 *		- Refactoring to share this composite in wizards and dialogs
 *    Guillaume Decarnin, Open Wide, Lyon, France
 *      - Small improves on UI
 *******************************************************************************/
package org.eclipse.jwt.transformations.exportTransformations;

import java.io.File;
import java.util.Set;

import org.eclipse.emf.common.CommonPlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jwt.transformations.base.JWTTransformationRequestDesc;
import org.eclipse.jwt.transformations.base.TransformationsMessages;
import org.eclipse.jwt.transformations.base.init.ProcessServiceMembers;
import org.eclipse.jwt.transformations.base.init.TransformationRegistry;
import org.eclipse.jwt.we.misc.util.GeneralHelper;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;

/**
 * Description of an export transformation, with the definition of a common
 * UI composite that can be used in any circumstance
 * @author mistria
 *
 */
public class ExportJWTTransformationRequestDesc extends JWTTransformationRequestDesc {

	private boolean canUseCurrentModel;
	private String currentModelInFilePath;
	
	
	public ExportJWTTransformationRequestDesc(Composite parent, Listener listener) {
		super(parent, listener);
	}

	public ExportJWTTransformationRequestDesc(Composite parent) {
		super(parent);
	}

	public String getCurrentModelInFilePath() {
		if (currentModelInFilePath == null && this.canUseCurrentModel)
			useCurrentModel();
		return currentModelInFilePath;
	}

	public void setCurrentModelInFilePath(String currentModelInFilePath) {
		this.currentModelInFilePath = currentModelInFilePath;
	}
	
	
	public Composite createControl() {
		Composite composite = new Composite(parentComposite, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// ////////////////
		// In File Group
		Group inFileGroup = new Group(composite, SWT.None);
		inFileGroup.setText(TransformationsMessages.inputWorkflow);
		inFileGroup.setLayout(new GridLayout(2, false));
		inFileGroup.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Button checkbox = new Button(inFileGroup, SWT.CHECK);
		checkbox.setSelection(true);
		checkbox.setText(TransformationsMessages.useCurrentJWTModel);
		checkbox.setLayoutData(new GridData(SWT.NULL, SWT.NULL, false, false,
				2, 1));

		final Text inFileText = new Text(inFileGroup, SWT.SINGLE | SWT.BORDER);
		GridData filegridData = new GridData(GridData.FILL_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL);
		inFileText.setLayoutData(filegridData);

		inFileText.setText(""); //$NON-NLS-1$

		final Button inFileBrowse = new Button(inFileGroup, SWT.NULL);
		inFileBrowse.setText(TransformationsMessages.browse);
		// End of In File Group
		// ///////////////////////
		
		// //////////////////////
		// Transformation Group
		final Group transformationGroup = new Group(composite, SWT.NULL);
		transformationGroup.setText(TransformationsMessages.transformationGroup);
		transformationGroup.setLayout(new GridLayout(1, false));
		transformationGroup.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Combo transformationsCombo = new Combo(transformationGroup, SWT.READ_ONLY);
		transformationsCombo.setText(TransformationsMessages.chooseExportTransformation);
		transformationsCombo.setLayoutData(new GridData(GridData.FILL_BOTH));
		Set<String> transformationsName = TransformationRegistry.getInstance().getTransformationsName();
		for (String trans : transformationsName) {
			if (ProcessServiceMembers.Type.EXPORT.equals(TransformationRegistry.getInstance().getTransformation(trans).getTye()))
				transformationsCombo.add(trans);
		}
		final Label descriptionLabel = new Label(transformationGroup, SWT.LEFT | SWT.WRAP | SWT.H_SCROLL);
		// End of Transformation Group
		///////////////////////////////
		
		// ////////////////////
		// Out File Group
		Group outFileGroup = new Group(composite, SWT.NULL);
		outFileGroup.setText(TransformationsMessages.outputFileGroup);
		outFileGroup.setLayout(new GridLayout(2, false));
		outFileGroup.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Text outFileText = new Text(outFileGroup, SWT.SINGLE | SWT.BORDER);
		outFileText.setLayoutData(filegridData);

		Button outFileBrowse = new Button(outFileGroup, SWT.NULL);
		outFileBrowse.setText(TransformationsMessages.browse);
		
		final Button openOutputCheckbox = new Button(outFileGroup, SWT.CHECK);
		openOutputCheckbox.setSelection(this.openCreatedFile);
		openOutputCheckbox.setText(TransformationsMessages.openCreatedFile);
		openOutputCheckbox.setLayoutData(new GridData(SWT.NULL, SWT.NULL, false, false,
				2, 1));
		// End of Out File Group
		// ///////////////////////
		
		//////////////////////////
		// Event handlers
		outFileBrowse.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				// opens a dialog to select the file
				FileDialog dialog = new FileDialog(ExportJWTTransformationRequestDesc.this.getShell(), SWT.SAVE);
				dialog.setText(TransformationsMessages.selectExportOutputFile);
				dialog.setFilterExtensions(new String[] { "*.*" });  //$NON-NLS-1$
				String path = dialog.open();
				outFileText.setText(path);
				if ((ExportJWTTransformationRequestDesc.this.getOutFilePath() != null) && (ExportJWTTransformationRequestDesc.this.getOutFilePath().length() != 0)) {
					outFileText.setText(ExportJWTTransformationRequestDesc.this.getOutFilePath());
				}
			}
		});
		inFileBrowse.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				// opens a dialog to select the file
				FileDialog dialog = new FileDialog(ExportJWTTransformationRequestDesc.this.getShell(), SWT.OPEN);
				dialog.setText(TransformationsMessages.selectExportInputFile);
				dialog.setFilterExtensions(new String[] { "*.workflow", "*.agilpro", "*.jwt" }); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ 
				String path = dialog.open();
				ExportJWTTransformationRequestDesc.this.setInFilePath(path);
				inFileText.setText(ExportJWTTransformationRequestDesc.this.getInFilePath());
			}
		});
		inFileText.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event event) {
				ExportJWTTransformationRequestDesc.this.setInFilePath(inFileText.getText());
				listener.handleEvent(event);
			}
			
		});
		outFileText.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				ExportJWTTransformationRequestDesc.this.setOutFilePath(outFileText.getText());
				listener.handleEvent(e);
			}
		});
		checkbox.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				if (checkbox.getSelection()) {
					inFileText.setText(ExportJWTTransformationRequestDesc.this.getCurrentModelInFilePath());
				}
				inFileText.setEnabled(!checkbox.getSelection());
				inFileBrowse.setEnabled(!checkbox.getSelection());
			}
		});
		transformationsCombo.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				ExportJWTTransformationRequestDesc.this.setTransformationName(transformationsCombo.getText());
				listener.handleEvent(e);
				descriptionLabel.setText(TransformationRegistry.getInstance()
								.getTransformation(
										transformationsCombo.getText())
								.getDescription());
				transformationGroup.layout(true);
				getShell().update();
			}
		});
		openOutputCheckbox.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				openCreatedFile = openOutputCheckbox.getSelection();
			}
		});
		//
		///////////////////
		
		//////////////////
		// Initialisation
		checkbox.setEnabled(this.canUseCurrentModel);
		checkbox.setSelection(this.canUseCurrentModel);
		if (this.canUseCurrentModel)
			inFileText.setText(this.getCurrentModelInFilePath());
		else if (this.getInFilePath() != null)
			inFileText.setText(this.getInFilePath());
		else
			inFileText.setText(""); //$NON-NLS-1$
		inFileText.setEnabled(! this.canUseCurrentModel);
		inFileBrowse.setEnabled(! this.canUseCurrentModel);
		//
		////////////////////
		
		return composite;
	}
	

	public boolean hasAllRequiredInfo() {
		return getInFilePath() != null && getInFilePath().length() != 0 &&
			isJwtModelFile(getInFilePath()) &&
			getOutFilePath() != null && getOutFilePath().length() != 0 &&
			getTransformationName() != null && TransformationRegistry.getInstance().getTransformationsName().contains(getTransformationName());
	}
	
	private boolean isJwtModelFile(String inFilePath2) {
		return new File(inFilePath2).exists() && 
			(inFilePath2.endsWith(".workflow") || inFilePath2.endsWith(".agilpro") || inFilePath2.endsWith(".jwt")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}
	

	public void setToExportItem(String inFilePath) {
		this.setInFilePath(inFilePath);
	}

	public void useCurrentModel() {
		if (!this.canUseCurrentModel) {
			Resource resource = GeneralHelper.getActiveInstance()
					.getCurrentActivitySheet().getActivityModel().eResource();
			this.setCurrentModelInFilePath(CommonPlugin.resolve(resource.getURI()).toFileString());
			this.canUseCurrentModel = true;
		}
	}
	
}
