/**
 * File:    JWTTransformationRequestDesc.java
 * Created: 12.12.2008
 *
 *
/*******************************************************************************
 * Copyright (c) 2008  Open Wide (www.openwide.fr)
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Mickael Istria, Open Wide, Lyon, France
 *      - creation and implementation
 *******************************************************************************/
package org.eclipse.jwt.transformations.base;

import java.io.File;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jwt.transformations.api.TransformationService;
import org.eclipse.jwt.transformations.base.init.TransformationRegistry;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

/**
 * This abstract class contains the description of a transformation
 * request and a set of utility class signatures and definitions
 * in order to be used each time a UI asks for a transformation
 * @author mistria
 *
 */
public abstract class JWTTransformationRequestDesc {

	private static final String JWT_TRANSFORMATION_BASE_PLUGIN_ID = "org.eclipse.jwt.transformations.baseTransformation"; //$NON-NLS-1$
	protected String outFilePath;
	protected String inFilePath;
	protected String transformationName;
	protected boolean openCreatedFile = false;
	
	protected Composite parentComposite;
	protected Listener listener;
	
	public JWTTransformationRequestDesc(Composite parent, Listener listener) {
		this.parentComposite = parent;
		this.listener = listener;
	}

	public JWTTransformationRequestDesc(Composite parent) {
		// Stub Listener
		this(parent, new Listener() {
			public void handleEvent(Event event) {
			}
		});
	}
	
	public abstract Composite createControl();
	
	public abstract boolean hasAllRequiredInfo();

	public boolean apply() {
		TransformationService transfo = TransformationRegistry.getInstance().getTransformation(this.getTransformationName());
		
		boolean cancelled = PlatformUI.getWorkbench().saveAllEditors(true);
		if (!cancelled)
			return false;
		
		try {
			File destFile = new File(this.getOutFilePath());
			if (!destFile.exists()) {
				destFile.createNewFile();
			}
			transfo.transform(this.getInFilePath(), this.getOutFilePath());
			
			if (openCreatedFile) {
			    IFileStore fileStore = EFS.getLocalFileSystem().getStore(new File(this.getOutFilePath()).toURI());
			    IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
			    IDE.openEditorOnFileStore( page, fileStore );
			}
		
		    return true;
		} catch (Exception e1) {
			Status status = new Status(IStatus.ERROR, JWT_TRANSFORMATION_BASE_PLUGIN_ID, e1.getMessage(), e1);
			ErrorDialog.openError(getShell(), TransformationsMessages.transformationError_title,
					TransformationsMessages.transformationError_details, status);
			return false;
		}
	}
	

	public Shell getShell() {
		return parentComposite.getShell();
	}
	
	public String getOutFilePath() {
		return outFilePath;
	}

	public void setOutFilePath(String outFilePath) {
		this.outFilePath = outFilePath;
	}

	public String getInFilePath() {
		return inFilePath;
	}

	public void setInFilePath(String inFilePath) {
		this.inFilePath = inFilePath;
	}

	public String getTransformationName() {
		return transformationName;
	}

	public void setTransformationName(String transformationName) {
		this.transformationName = transformationName;
	}

	public void setListener(Listener listener) {
		this.listener = listener;
	}
	
}
