/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.util;

import java.math.BigInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.math3.exception.MathArithmeticException;
import org.apache.commons.math3.exception.NotPositiveException;
import org.apache.commons.math3.exception.NumberIsTooLargeException;
import org.apache.commons.math3.exception.util.Localizable;
import org.apache.commons.math3.exception.util.LocalizedFormats;
import org.apache.commons.math3.util.FastMath;

public final class ArithmeticUtils {
    static final long[] FACTORIALS = new long[]{1L, 1L, 2L, 6L, 24L, 120L, 720L, 5040L, 40320L, 362880L, 3628800L, 39916800L, 479001600L, 6227020800L, 87178291200L, 1307674368000L, 20922789888000L, 355687428096000L, 6402373705728000L, 121645100408832000L, 2432902008176640000L};
    static final AtomicReference<long[][]> STIRLING_S2 = new AtomicReference<Object>(null);

    private ArithmeticUtils() {
    }

    public static int addAndCheck(int x, int y) throws MathArithmeticException {
        long s = (long)x + (long)y;
        if (s < Integer.MIN_VALUE || s > Integer.MAX_VALUE) {
            throw new MathArithmeticException(LocalizedFormats.OVERFLOW_IN_ADDITION, x, y);
        }
        return (int)s;
    }

    public static long addAndCheck(long a, long b) throws MathArithmeticException {
        return ArithmeticUtils.addAndCheck(a, b, LocalizedFormats.OVERFLOW_IN_ADDITION);
    }

    public static long binomialCoefficient(int n, int k) throws NotPositiveException, NumberIsTooLargeException, MathArithmeticException {
        ArithmeticUtils.checkBinomial(n, k);
        if (n == k || k == 0) {
            return 1L;
        }
        if (k == 1 || k == n - 1) {
            return n;
        }
        if (k > n / 2) {
            return ArithmeticUtils.binomialCoefficient(n, n - k);
        }
        long result = 1L;
        if (n <= 61) {
            int i = n - k + 1;
            int j = 1;
            while (j <= k) {
                result = result * (long)i / (long)j;
                ++i;
                ++j;
            }
        } else if (n <= 66) {
            int i = n - k + 1;
            int j = 1;
            while (j <= k) {
                long d = ArithmeticUtils.gcd(i, j);
                result = result / ((long)j / d) * ((long)i / d);
                ++i;
                ++j;
            }
        } else {
            int i = n - k + 1;
            int j = 1;
            while (j <= k) {
                long d = ArithmeticUtils.gcd(i, j);
                result = ArithmeticUtils.mulAndCheck(result / ((long)j / d), (long)i / d);
                ++i;
                ++j;
            }
        }
        return result;
    }

    public static double binomialCoefficientDouble(int n, int k) throws NotPositiveException, NumberIsTooLargeException, MathArithmeticException {
        ArithmeticUtils.checkBinomial(n, k);
        if (n == k || k == 0) {
            return 1.0;
        }
        if (k == 1 || k == n - 1) {
            return n;
        }
        if (k > n / 2) {
            return ArithmeticUtils.binomialCoefficientDouble(n, n - k);
        }
        if (n < 67) {
            return ArithmeticUtils.binomialCoefficient(n, k);
        }
        double result = 1.0;
        int i = 1;
        while (i <= k) {
            result *= (double)(n - k + i) / (double)i;
            ++i;
        }
        return FastMath.floor(result + 0.5);
    }

    public static double binomialCoefficientLog(int n, int k) throws NotPositiveException, NumberIsTooLargeException, MathArithmeticException {
        ArithmeticUtils.checkBinomial(n, k);
        if (n == k || k == 0) {
            return 0.0;
        }
        if (k == 1 || k == n - 1) {
            return FastMath.log(n);
        }
        if (n < 67) {
            return FastMath.log(ArithmeticUtils.binomialCoefficient(n, k));
        }
        if (n < 1030) {
            return FastMath.log(ArithmeticUtils.binomialCoefficientDouble(n, k));
        }
        if (k > n / 2) {
            return ArithmeticUtils.binomialCoefficientLog(n, n - k);
        }
        double logSum = 0.0;
        int i = n - k + 1;
        while (i <= n) {
            logSum += FastMath.log(i);
            ++i;
        }
        i = 2;
        while (i <= k) {
            logSum -= FastMath.log(i);
            ++i;
        }
        return logSum;
    }

    public static long factorial(int n) throws NotPositiveException, MathArithmeticException {
        if (n < 0) {
            throw new NotPositiveException((Localizable)LocalizedFormats.FACTORIAL_NEGATIVE_PARAMETER, n);
        }
        if (n > 20) {
            throw new MathArithmeticException();
        }
        return FACTORIALS[n];
    }

    public static double factorialDouble(int n) throws NotPositiveException {
        if (n < 0) {
            throw new NotPositiveException((Localizable)LocalizedFormats.FACTORIAL_NEGATIVE_PARAMETER, n);
        }
        if (n < 21) {
            return FACTORIALS[n];
        }
        return FastMath.floor(FastMath.exp(ArithmeticUtils.factorialLog(n)) + 0.5);
    }

    public static double factorialLog(int n) throws NotPositiveException {
        if (n < 0) {
            throw new NotPositiveException((Localizable)LocalizedFormats.FACTORIAL_NEGATIVE_PARAMETER, n);
        }
        if (n < 21) {
            return FastMath.log(FACTORIALS[n]);
        }
        double logSum = 0.0;
        int i = 2;
        while (i <= n) {
            logSum += FastMath.log(i);
            ++i;
        }
        return logSum;
    }

    public static int gcd(int p, int q) throws MathArithmeticException {
        int a = p;
        int b = q;
        if (a == 0 || b == 0) {
            if (a == Integer.MIN_VALUE || b == Integer.MIN_VALUE) {
                throw new MathArithmeticException(LocalizedFormats.GCD_OVERFLOW_32_BITS, p, q);
            }
            return FastMath.abs(a + b);
        }
        long al = a;
        long bl = b;
        boolean useLong = false;
        if (a < 0) {
            if (Integer.MIN_VALUE == a) {
                useLong = true;
            } else {
                a = -a;
            }
            al = -al;
        }
        if (b < 0) {
            if (Integer.MIN_VALUE == b) {
                useLong = true;
            } else {
                b = -b;
            }
            bl = -bl;
        }
        if (useLong) {
            if (al == bl) {
                throw new MathArithmeticException(LocalizedFormats.GCD_OVERFLOW_32_BITS, p, q);
            }
            long blbu = bl;
            bl = al;
            if ((al = blbu % al) == 0L) {
                if (bl > Integer.MAX_VALUE) {
                    throw new MathArithmeticException(LocalizedFormats.GCD_OVERFLOW_32_BITS, p, q);
                }
                return (int)bl;
            }
            blbu = bl;
            b = (int)al;
            a = (int)(blbu % al);
        }
        return ArithmeticUtils.gcdPositive(a, b);
    }

    private static int gcdPositive(int a, int b) {
        if (a == 0) {
            return b;
        }
        if (b == 0) {
            return a;
        }
        int aTwos = Integer.numberOfTrailingZeros(a);
        a >>= aTwos;
        int bTwos = Integer.numberOfTrailingZeros(b);
        b >>= bTwos;
        int shift = Math.min(aTwos, bTwos);
        while (a != b) {
            int delta = a - b;
            b = Math.min(a, b);
            a = Math.abs(delta);
            a >>= Integer.numberOfTrailingZeros(a);
        }
        return a << shift;
    }

    public static long gcd(long p, long q) throws MathArithmeticException {
        long u = p;
        long v = q;
        if (u == 0L || v == 0L) {
            if (u == Long.MIN_VALUE || v == Long.MIN_VALUE) {
                throw new MathArithmeticException(LocalizedFormats.GCD_OVERFLOW_64_BITS, p, q);
            }
            return FastMath.abs(u) + FastMath.abs(v);
        }
        if (u > 0L) {
            u = -u;
        }
        if (v > 0L) {
            v = -v;
        }
        int k = 0;
        while ((u & 1L) == 0L && (v & 1L) == 0L && k < 63) {
            u /= 2L;
            v /= 2L;
            ++k;
        }
        if (k == 63) {
            throw new MathArithmeticException(LocalizedFormats.GCD_OVERFLOW_64_BITS, p, q);
        }
        long t = (u & 1L) == 1L ? v : -(u / 2L);
        while (true) {
            if ((t & 1L) == 0L) {
                t /= 2L;
                continue;
            }
            if (t > 0L) {
                u = -t;
            } else {
                v = t;
            }
            if ((t = (v - u) / 2L) == 0L) break;
        }
        return -u * (1L << k);
    }

    public static int lcm(int a, int b) throws MathArithmeticException {
        if (a == 0 || b == 0) {
            return 0;
        }
        int lcm = FastMath.abs(ArithmeticUtils.mulAndCheck(a / ArithmeticUtils.gcd(a, b), b));
        if (lcm == Integer.MIN_VALUE) {
            throw new MathArithmeticException(LocalizedFormats.LCM_OVERFLOW_32_BITS, a, b);
        }
        return lcm;
    }

    public static long lcm(long a, long b) throws MathArithmeticException {
        if (a == 0L || b == 0L) {
            return 0L;
        }
        long lcm = FastMath.abs(ArithmeticUtils.mulAndCheck(a / ArithmeticUtils.gcd(a, b), b));
        if (lcm == Long.MIN_VALUE) {
            throw new MathArithmeticException(LocalizedFormats.LCM_OVERFLOW_64_BITS, a, b);
        }
        return lcm;
    }

    public static int mulAndCheck(int x, int y) throws MathArithmeticException {
        long m = (long)x * (long)y;
        if (m < Integer.MIN_VALUE || m > Integer.MAX_VALUE) {
            throw new MathArithmeticException();
        }
        return (int)m;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static long mulAndCheck(long a, long b) throws MathArithmeticException {
        if (a > b) {
            return ArithmeticUtils.mulAndCheck(b, a);
        }
        if (a < 0L) {
            if (b < 0L) {
                if (a < Long.MAX_VALUE / b) throw new MathArithmeticException();
                return a * b;
            }
            if (b <= 0L) return 0L;
            if (Long.MIN_VALUE / b > a) throw new MathArithmeticException();
            return a * b;
        }
        if (a <= 0L) return 0L;
        if (a > Long.MAX_VALUE / b) throw new MathArithmeticException();
        return a * b;
    }

    public static int subAndCheck(int x, int y) throws MathArithmeticException {
        long s = (long)x - (long)y;
        if (s < Integer.MIN_VALUE || s > Integer.MAX_VALUE) {
            throw new MathArithmeticException(LocalizedFormats.OVERFLOW_IN_SUBTRACTION, x, y);
        }
        return (int)s;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static long subAndCheck(long a, long b) throws MathArithmeticException {
        if (b != Long.MIN_VALUE) return ArithmeticUtils.addAndCheck(a, -b, LocalizedFormats.OVERFLOW_IN_ADDITION);
        if (a >= 0L) throw new MathArithmeticException(LocalizedFormats.OVERFLOW_IN_ADDITION, a, -b);
        return a - b;
    }

    public static int pow(int k, int e) throws NotPositiveException {
        if (e < 0) {
            throw new NotPositiveException((Localizable)LocalizedFormats.EXPONENT, e);
        }
        int result = 1;
        int k2p = k;
        while (e != 0) {
            if ((e & 1) != 0) {
                result *= k2p;
            }
            k2p *= k2p;
            e >>= 1;
        }
        return result;
    }

    public static int pow(int k, long e) throws NotPositiveException {
        if (e < 0L) {
            throw new NotPositiveException((Localizable)LocalizedFormats.EXPONENT, e);
        }
        int result = 1;
        int k2p = k;
        while (e != 0L) {
            if ((e & 1L) != 0L) {
                result *= k2p;
            }
            k2p *= k2p;
            e >>= 1;
        }
        return result;
    }

    public static long pow(long k, int e) throws NotPositiveException {
        if (e < 0) {
            throw new NotPositiveException((Localizable)LocalizedFormats.EXPONENT, e);
        }
        long result = 1L;
        long k2p = k;
        while (e != 0) {
            if ((e & 1) != 0) {
                result *= k2p;
            }
            k2p *= k2p;
            e >>= 1;
        }
        return result;
    }

    public static long pow(long k, long e) throws NotPositiveException {
        if (e < 0L) {
            throw new NotPositiveException((Localizable)LocalizedFormats.EXPONENT, e);
        }
        long result = 1L;
        long k2p = k;
        while (e != 0L) {
            if ((e & 1L) != 0L) {
                result *= k2p;
            }
            k2p *= k2p;
            e >>= 1;
        }
        return result;
    }

    public static BigInteger pow(BigInteger k, int e) throws NotPositiveException {
        if (e < 0) {
            throw new NotPositiveException((Localizable)LocalizedFormats.EXPONENT, e);
        }
        return k.pow(e);
    }

    public static BigInteger pow(BigInteger k, long e) throws NotPositiveException {
        if (e < 0L) {
            throw new NotPositiveException((Localizable)LocalizedFormats.EXPONENT, e);
        }
        BigInteger result = BigInteger.ONE;
        BigInteger k2p = k;
        while (e != 0L) {
            if ((e & 1L) != 0L) {
                result = result.multiply(k2p);
            }
            k2p = k2p.multiply(k2p);
            e >>= 1;
        }
        return result;
    }

    public static BigInteger pow(BigInteger k, BigInteger e) throws NotPositiveException {
        if (e.compareTo(BigInteger.ZERO) < 0) {
            throw new NotPositiveException((Localizable)LocalizedFormats.EXPONENT, e);
        }
        BigInteger result = BigInteger.ONE;
        BigInteger k2p = k;
        while (!BigInteger.ZERO.equals(e)) {
            if (e.testBit(0)) {
                result = result.multiply(k2p);
            }
            k2p = k2p.multiply(k2p);
            e = e.shiftRight(1);
        }
        return result;
    }

    public static long stirlingS2(int n, int k) throws NotPositiveException, NumberIsTooLargeException, MathArithmeticException {
        if (k < 0) {
            throw new NotPositiveException(k);
        }
        if (k > n) {
            throw new NumberIsTooLargeException(k, (Number)n, true);
        }
        Object stirlingS2 = STIRLING_S2.get();
        if (stirlingS2 == null) {
            int maxIndex = 26;
            stirlingS2 = new long[26][];
            stirlingS2[0] = new long[]{1L};
            int i = 1;
            while (i < ((long[][])stirlingS2).length) {
                stirlingS2[i] = new long[i + 1];
                stirlingS2[i][0] = 0L;
                stirlingS2[i][1] = 1L;
                stirlingS2[i][i] = 1L;
                int j = 2;
                while (j < i) {
                    stirlingS2[i][j] = (long)j * stirlingS2[i - 1][j] + stirlingS2[i - 1][j - 1];
                    ++j;
                }
                ++i;
            }
            STIRLING_S2.compareAndSet((long[][])null, (long[][])stirlingS2);
        }
        if (n < ((long[][])stirlingS2).length) {
            return stirlingS2[n][k];
        }
        if (k == 0) {
            return 0L;
        }
        if (k == 1 || k == n) {
            return 1L;
        }
        if (k == 2) {
            return (1L << n - 1) - 1L;
        }
        if (k == n - 1) {
            return ArithmeticUtils.binomialCoefficient(n, 2);
        }
        long sum = 0L;
        long sign = (k & 1) == 0 ? 1 : -1;
        int j = 1;
        while (j <= k) {
            if ((sum += (sign = -sign) * ArithmeticUtils.binomialCoefficient(k, j) * (long)ArithmeticUtils.pow(j, n)) < 0L) {
                throw new MathArithmeticException(LocalizedFormats.ARGUMENT_OUTSIDE_DOMAIN, n, 0, ((long[][])stirlingS2).length - 1);
            }
            ++j;
        }
        return sum / ArithmeticUtils.factorial(k);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static long addAndCheck(long a, long b, Localizable pattern) throws MathArithmeticException {
        if (a > b) {
            return ArithmeticUtils.addAndCheck(b, a, pattern);
        }
        if (a < 0L) {
            if (b >= 0L) return a + b;
            if (Long.MIN_VALUE - b > a) throw new MathArithmeticException(pattern, a, b);
            return a + b;
        }
        if (a > Long.MAX_VALUE - b) throw new MathArithmeticException(pattern, a, b);
        return a + b;
    }

    private static void checkBinomial(int n, int k) throws NumberIsTooLargeException, NotPositiveException {
        if (n < k) {
            throw new NumberIsTooLargeException((Localizable)LocalizedFormats.BINOMIAL_INVALID_PARAMETERS_ORDER, (Number)k, n, true);
        }
        if (n < 0) {
            throw new NotPositiveException((Localizable)LocalizedFormats.BINOMIAL_NEGATIVE_PARAMETER, n);
        }
    }

    public static boolean isPowerOfTwo(long n) {
        return n > 0L && (n & n - 1L) == 0L;
    }
}

