// SPDX-License-Identifier: BSD-3-Clause
/* Copyright (c) 2022, Linaro Limited */

#include <sys/mman.h>
#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "debugcc.h"

static struct gcc_mux gcc = {
	.mux = {
		.phys = 0x1400000,
		.size = 0x1f0000,

		.measure = measure_gcc,

		.enable_reg = 0x30004,
		.enable_mask = BIT(0),

		.mux_reg = 0x62000,
		.mux_mask = 0x3ff,

		.div_reg = 0x30000,
		.div_mask = 0xf,
		.div_val = 1,
	},

	.xo_div4_reg = 0x28008,
	.debug_ctl_reg = 0x62038,
	.debug_status_reg = 0x6203c,
};

static struct debug_mux cpu_cc = {
	.phys = 0xf111000,
	.size = 0x1000,
	.block_name = "cpu",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0xab,

	.enable_reg = 0x1c,
	.enable_mask = BIT(0),

	.mux_reg = 0x1c,
	.mux_mask = 0x3ff << 8,
	.mux_shift = 8,

	.div_reg = 0x1c,
	.div_mask = 0xf << 28,
	.div_shift = 28,
	.div_val = 8,
};

static struct debug_mux disp_cc = {
	.phys = 0x5f00000,
	.size = 0x20000,
	.block_name = "disp",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x41,

	.enable_reg = 0x500c,
	.enable_mask = BIT(0),

	.mux_reg = 0x7000,
	.mux_mask = 0xff,

	.div_reg = 0x5008,
	.div_mask = 0x3,
	.div_val = 4,
};

static struct debug_mux gpu_cc = {
	.phys = 0x5990000,
	.size = 0x9000,
	.block_name = "gpu",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0xdd,

	.enable_reg = 0x1100,
	.enable_mask = BIT(0),

	.mux_reg = 0x1568,
	.mux_mask = 0xff,

	.div_reg = 0x10fc,
	.div_mask = 0x3,
	.div_val = 2,
};

static struct debug_mux video_cc = {
	.phys = 0x5b00000,
	.size = 0x10000,
	.block_name = "video",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x42,

	.enable_reg = 0x1100,
	.enable_mask = BIT(0),

	.mux_reg = 0xa4c,
	.mux_mask = 0x3f,

	.div_reg = 0xa30,
	.div_mask = 0x7,
	.div_val = 5,
};

static struct measure_clk sm6125_clocks[] = {
	{ "perfcl_clk", &cpu_cc, 0x1 },
	{ "pwrcl_clk", &cpu_cc, 0x0 },
	{ "snoc_clk", &gcc.mux, 0x7 },
	{ "bimc_clk", &gcc.mux, 0x97 },
	{ "pnoc_clk", &gcc.mux, 0x11 },
	{ "gcc_ahb2phy_csi_clk", &gcc.mux, 0x5f },
	{ "gcc_ahb2phy_usb_clk", &gcc.mux, 0x60 },
	{ "gcc_apc_vs_clk", &gcc.mux, 0xbf },
	{ "gcc_boot_rom_ahb_clk", &gcc.mux, 0x72 },
	{ "gcc_camera_ahb_clk", &gcc.mux, 0x36 },
	{ "gcc_camera_xo_clk", &gcc.mux, 0x3e },
	{ "gcc_camss_cci_ahb_clk", &gcc.mux, 0x11f },
	{ "gcc_camss_cci_clk", &gcc.mux, 0x11e },
	{ "gcc_camss_cphy_csid0_clk", &gcc.mux, 0x13a },
	{ "gcc_camss_cphy_csid1_clk", &gcc.mux, 0x140 },
	{ "gcc_camss_cphy_csid2_clk", &gcc.mux, 0x145 },
	{ "gcc_camss_cphy_csid3_clk", &gcc.mux, 0x14b },
	{ "gcc_camss_cpp_ahb_clk", &gcc.mux, 0x154 },
	{ "gcc_camss_cpp_axi_clk", &gcc.mux, 0x156 },
	{ "gcc_camss_cpp_clk", &gcc.mux, 0x153 },
	{ "gcc_camss_cpp_tsctr_clk", &gcc.mux, 0x158 },
	{ "gcc_camss_cpp_vbif_ahb_clk", &gcc.mux, 0x155 },
	{ "gcc_camss_csi0_ahb_clk", &gcc.mux, 0x13b },
	{ "gcc_camss_csi0_clk", &gcc.mux, 0x139 },
	{ "gcc_camss_csi0phytimer_clk", &gcc.mux, 0x120 },
	{ "gcc_camss_csi0pix_clk", &gcc.mux, 0x13d },
	{ "gcc_camss_csi0rdi_clk", &gcc.mux, 0x13c },
	{ "gcc_camss_csi1_ahb_clk", &gcc.mux, 0x141 },
	{ "gcc_camss_csi1_clk", &gcc.mux, 0x13f },
	{ "gcc_camss_csi1phytimer_clk", &gcc.mux, 0x121 },
	{ "gcc_camss_csi1pix_clk", &gcc.mux, 0x143 },
	{ "gcc_camss_csi1rdi_clk", &gcc.mux, 0x142 },
	{ "gcc_camss_csi2_ahb_clk", &gcc.mux, 0x146 },
	{ "gcc_camss_csi2_clk", &gcc.mux, 0x144 },
	{ "gcc_camss_csi2phytimer_clk", &gcc.mux, 0x122 },
	{ "gcc_camss_csi2pix_clk", &gcc.mux, 0x148 },
	{ "gcc_camss_csi2rdi_clk", &gcc.mux, 0x147 },
	{ "gcc_camss_csi3_ahb_clk", &gcc.mux, 0x14c },
	{ "gcc_camss_csi3_clk", &gcc.mux, 0x14a },
	{ "gcc_camss_csi3pix_clk", &gcc.mux, 0x14e },
	{ "gcc_camss_csi3rdi_clk", &gcc.mux, 0x14d },
	{ "gcc_camss_csi_vfe0_clk", &gcc.mux, 0x12e },
	{ "gcc_camss_csi_vfe1_clk", &gcc.mux, 0x12f },
	{ "gcc_camss_csiphy0_clk", &gcc.mux, 0x135 },
	{ "gcc_camss_csiphy1_clk", &gcc.mux, 0x136 },
	{ "gcc_camss_csiphy2_clk", &gcc.mux, 0x137 },
	{ "gcc_camss_csiphy3_clk", &gcc.mux, 0x138 },
	{ "gcc_camss_gp0_clk", &gcc.mux, 0x118 },
	{ "gcc_camss_gp1_clk", &gcc.mux, 0x119 },
	{ "gcc_camss_ispif_ahb_clk", &gcc.mux, 0x134 },
	{ "gcc_camss_jpeg_ahb_clk", &gcc.mux, 0x124 },
	{ "gcc_camss_jpeg_axi_clk", &gcc.mux, 0x125 },
	{ "gcc_camss_jpeg_clk", &gcc.mux, 0x123 },
	{ "gcc_camss_jpeg_tsctr_clk", &gcc.mux, 0x127 },
	{ "gcc_camss_mclk0_clk", &gcc.mux, 0x11a },
	{ "gcc_camss_mclk1_clk", &gcc.mux, 0x11b },
	{ "gcc_camss_mclk2_clk", &gcc.mux, 0x11c },
	{ "gcc_camss_mclk3_clk", &gcc.mux, 0x11d },
	{ "gcc_camss_micro_ahb_clk", &gcc.mux, 0x152 },
	{ "gcc_camss_throttle_nrt_axi_clk", &gcc.mux, 0x150 },
	{ "gcc_camss_throttle_rt_axi_clk", &gcc.mux, 0x151 },
	{ "gcc_camss_top_ahb_clk", &gcc.mux, 0x14f },
	{ "gcc_camss_vfe0_ahb_clk", &gcc.mux, 0x12a },
	{ "gcc_camss_vfe0_clk", &gcc.mux, 0x128 },
	{ "gcc_camss_vfe0_stream_clk", &gcc.mux, 0x129 },
	{ "gcc_camss_vfe1_ahb_clk", &gcc.mux, 0x12d },
	{ "gcc_camss_vfe1_clk", &gcc.mux, 0x12b },
	{ "gcc_camss_vfe1_stream_clk", &gcc.mux, 0x12c },
	{ "gcc_camss_vfe_tsctr_clk", &gcc.mux, 0x133 },
	{ "gcc_camss_vfe_vbif_ahb_clk", &gcc.mux, 0x130 },
	{ "gcc_camss_vfe_vbif_axi_clk", &gcc.mux, 0x131 },
	{ "gcc_ce1_ahb_clk", &gcc.mux, 0x82 },
	{ "gcc_ce1_axi_clk", &gcc.mux, 0x81 },
	{ "gcc_ce1_clk", &gcc.mux, 0x80 },
	{ "gcc_cfg_noc_usb3_prim_axi_clk", &gcc.mux, 0x1d },
	{ "gcc_cpuss_ahb_clk", &gcc.mux, 0xa5 },
	{ "gcc_cpuss_gnoc_clk", &gcc.mux, 0xa6 },
	{ "gcc_cpuss_throttle_core_clk", &gcc.mux, 0xae },
	{ "gcc_cpuss_throttle_xo_clk", &gcc.mux, 0xad },
	{ "gcc_disp_ahb_clk", &gcc.mux, 0x37 },
	{ "gcc_disp_gpll0_div_clk_src", &gcc.mux, 0x46 },
	{ "gcc_disp_hf_axi_clk", &gcc.mux, 0x3c },
	{ "gcc_disp_throttle_core_clk", &gcc.mux, 0x48 },
	{ "gcc_disp_xo_clk", &gcc.mux, 0x3f },
	{ "gcc_gp1_clk", &gcc.mux, 0xb6 },
	{ "gcc_gp2_clk", &gcc.mux, 0xb7 },
	{ "gcc_gp3_clk", &gcc.mux, 0xb8 },
	{ "gcc_gpu_cfg_ahb_clk", &gcc.mux, 0xdb },
	{ "gcc_gpu_gpll0_clk_src", &gcc.mux, 0xe1 },
	{ "gcc_gpu_gpll0_div_clk_src", &gcc.mux, 0xe2 },
	{ "gcc_gpu_memnoc_gfx_clk", &gcc.mux, 0xde },
	{ "gcc_gpu_snoc_dvm_gfx_clk", &gcc.mux, 0xe0 },
	{ "gcc_gpu_throttle_core_clk", &gcc.mux, 0xe5 },
	{ "gcc_gpu_throttle_xo_clk", &gcc.mux, 0xe4 },
	{ "gcc_mss_vs_clk", &gcc.mux, 0xbe },
	{ "gcc_pdm2_clk", &gcc.mux, 0x6f },
	{ "gcc_pdm_ahb_clk", &gcc.mux, 0x6d },
	{ "gcc_pdm_xo4_clk", &gcc.mux, 0x6e },
	{ "gcc_prng_ahb_clk", &gcc.mux, 0x70 },
	{ "gcc_qmip_camera_nrt_ahb_clk", &gcc.mux, 0x39 },
	{ "gcc_qmip_camera_rt_ahb_clk", &gcc.mux, 0x47 },
	{ "gcc_qmip_cpuss_cfg_ahb_clk", &gcc.mux, 0xac },
	{ "gcc_qmip_disp_ahb_clk", &gcc.mux, 0x3a },
	{ "gcc_qmip_gpu_cfg_ahb_clk", &gcc.mux, 0xe3 },
	{ "gcc_qmip_video_vcodec_ahb_clk", &gcc.mux, 0x38 },
	{ "gcc_qupv3_wrap0_core_2x_clk", &gcc.mux, 0x66 },
	{ "gcc_qupv3_wrap0_core_clk", &gcc.mux, 0x65 },
	{ "gcc_qupv3_wrap0_s0_clk", &gcc.mux, 0x67 },
	{ "gcc_qupv3_wrap0_s1_clk", &gcc.mux, 0x68 },
	{ "gcc_qupv3_wrap0_s2_clk", &gcc.mux, 0x69 },
	{ "gcc_qupv3_wrap0_s3_clk", &gcc.mux, 0x6a },
	{ "gcc_qupv3_wrap0_s4_clk", &gcc.mux, 0x6b },
	{ "gcc_qupv3_wrap0_s5_clk", &gcc.mux, 0x6c },
	{ "gcc_qupv3_wrap1_core_2x_clk", &gcc.mux, 0xed },
	{ "gcc_qupv3_wrap1_core_clk", &gcc.mux, 0xec },
	{ "gcc_qupv3_wrap1_s0_clk", &gcc.mux, 0xee },
	{ "gcc_qupv3_wrap1_s1_clk", &gcc.mux, 0xef },
	{ "gcc_qupv3_wrap1_s2_clk", &gcc.mux, 0xf0 },
	{ "gcc_qupv3_wrap1_s3_clk", &gcc.mux, 0xf1 },
	{ "gcc_qupv3_wrap1_s4_clk", &gcc.mux, 0xf2 },
	{ "gcc_qupv3_wrap1_s5_clk", &gcc.mux, 0xf3 },
	{ "gcc_qupv3_wrap_0_m_ahb_clk", &gcc.mux, 0x63 },
	{ "gcc_qupv3_wrap_0_s_ahb_clk", &gcc.mux, 0x64 },
	{ "gcc_qupv3_wrap_1_m_ahb_clk", &gcc.mux, 0xea },
	{ "gcc_qupv3_wrap_1_s_ahb_clk", &gcc.mux, 0xeb },
	{ "gcc_sdcc1_ahb_clk", &gcc.mux, 0xe8 },
	{ "gcc_sdcc1_apps_clk", &gcc.mux, 0xe7 },
	{ "gcc_sdcc1_ice_core_clk", &gcc.mux, 0xe9 },
	{ "gcc_sdcc2_ahb_clk", &gcc.mux, 0x62 },
	{ "gcc_sdcc2_apps_clk", &gcc.mux, 0x61 },
	{ "gcc_sys_noc_cpuss_ahb_clk", &gcc.mux, 0x9,  },
	{ "gcc_sys_noc_ufs_phy_axi_clk", &gcc.mux, 0x19 },
	{ "gcc_sys_noc_usb3_prim_axi_clk", &gcc.mux, 0x18 },
	{ "gcc_ufs_phy_ahb_clk", &gcc.mux, 0x10f },
	{ "gcc_ufs_phy_axi_clk", &gcc.mux, 0x10e },
	{ "gcc_ufs_phy_ice_core_clk", &gcc.mux, 0x115 },
	{ "gcc_ufs_phy_phy_aux_clk", &gcc.mux, 0x116 },
	{ "gcc_ufs_phy_rx_symbol_0_clk", &gcc.mux, 0x111 },
	{ "gcc_ufs_phy_tx_symbol_0_clk", &gcc.mux, 0x110 },
	{ "gcc_ufs_phy_unipro_core_clk", &gcc.mux, 0x114 },
	{ "gcc_usb30_prim_master_clk", &gcc.mux, 0x58 },
	{ "gcc_usb30_prim_mock_utmi_clk", &gcc.mux, 0x5a },
	{ "gcc_usb30_prim_sleep_clk", &gcc.mux, 0x59 },
	{ "gcc_usb3_prim_phy_com_aux_clk", &gcc.mux, 0x5b },
	{ "gcc_usb3_prim_phy_pipe_clk", &gcc.mux, 0x5c },
	{ "gcc_vdda_vs_clk", &gcc.mux, 0xbb },
	{ "gcc_vddcx_vs_clk", &gcc.mux, 0xb9 },
	{ "gcc_vddmx_vs_clk", &gcc.mux, 0xba },
	{ "gcc_video_ahb_clk", &gcc.mux, 0x35 },
	{ "gcc_video_axi0_clk", &gcc.mux, 0x3b },
	{ "gcc_video_throttle_core_clk", &gcc.mux, 0x49 },
	{ "gcc_video_xo_clk", &gcc.mux, 0x3d },
	{ "gcc_vs_ctrl_ahb_clk", &gcc.mux, 0xbd },
	{ "gcc_vs_ctrl_clk", &gcc.mux, 0xbc },
	{ "gcc_wcss_vs_clk", &gcc.mux, 0xc0 },
	{ "measure_only_ipa_2x_clk", &gcc.mux, 0xc2 },

	{ "disp_cc_mdss_ahb_clk", &disp_cc, 0x1a },
	{ "disp_cc_mdss_byte0_clk", &disp_cc, 0x12 },
	{ "disp_cc_mdss_byte0_intf_clk", &disp_cc, 0x13 },
	{ "disp_cc_mdss_dp_aux_clk", &disp_cc, 0x19 },
	{ "disp_cc_mdss_dp_crypto_clk", &disp_cc, 0x17 },
	{ "disp_cc_mdss_dp_link_clk", &disp_cc, 0x15 },
	{ "disp_cc_mdss_dp_link_intf_clk", &disp_cc, 0x16 },
	{ "disp_cc_mdss_dp_pixel_clk", &disp_cc, 0x18 },
	{ "disp_cc_mdss_esc0_clk", &disp_cc, 0x14 },
	{ "disp_cc_mdss_mdp_clk", &disp_cc, 0xe, },
	{ "disp_cc_mdss_mdp_lut_clk", &disp_cc, 0x10 },
	{ "disp_cc_mdss_non_gdsc_ahb_clk", &disp_cc, 0x1b },
	{ "disp_cc_mdss_pclk0_clk", &disp_cc, 0xd, },
	{ "disp_cc_mdss_rot_clk", &disp_cc, 0xf, },
	{ "disp_cc_mdss_vsync_clk", &disp_cc, 0x11 },
	{ "disp_cc_sleep_clk", &disp_cc, 0x24 },
	{ "disp_cc_xo_clk", &disp_cc, 0x23 },

	{ "gpu_cc_ahb_clk", &gpu_cc, 0x10 },
	{ "gpu_cc_crc_ahb_clk", &gpu_cc, 0x11 },
	{ "gpu_cc_cx_apb_clk", &gpu_cc, 0x14 },
	{ "gpu_cc_cx_gfx3d_clk", &gpu_cc, 0x1a },
	{ "gpu_cc_cx_gfx3d_slv_clk", &gpu_cc, 0x1b },
	{ "gpu_cc_cx_gmu_clk", &gpu_cc, 0x18 },
	{ "gpu_cc_cx_snoc_dvm_clk", &gpu_cc, 0x15 },
	{ "gpu_cc_cxo_aon_clk", &gpu_cc, 0xa },
	{ "gpu_cc_cxo_clk", &gpu_cc, 0x19 },
	{ "gpu_cc_gx_cxo_clk", &gpu_cc, 0xe },
	{ "gpu_cc_gx_gfx3d_clk", &gpu_cc, 0xb },
	{ "gpu_cc_sleep_clk", &gpu_cc, 0x16 },

	{ "video_cc_apb_clk", &video_cc, 0x8 },
	{ "video_cc_at_clk", &video_cc, 0xb },
	{ "video_cc_sleep_clk", &video_cc, 0xd },
	{ "video_cc_vcodec0_axi_clk", &video_cc, 0x6 },
	{ "video_cc_vcodec0_core_clk", &video_cc, 0x3 },
	{ "video_cc_venus_ahb_clk", &video_cc, 0x9 },
	{ "video_cc_venus_ctl_axi_clk", &video_cc, 0x5 },
	{ "video_cc_venus_ctl_core_clk", &video_cc, 0x1 },
	{ "video_cc_xo_clk", &video_cc, 0xc },
	{}
};

struct debugcc_platform sm6125_debugcc = {
	"sm6125",
	sm6125_clocks,
};
