/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.blob.cassandra;

import com.github.fge.lambdas.Throwing;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.io.ByteSource;
import jakarta.inject.Inject;
import jakarta.inject.Named;
import java.io.InputStream;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.Consumer;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.james.backends.cassandra.init.configuration.CassandraConfiguration;
import org.apache.james.blob.api.BlobId;
import org.apache.james.blob.api.BlobStoreDAO;
import org.apache.james.blob.api.BucketName;
import org.apache.james.blob.api.ObjectNotFoundException;
import org.apache.james.blob.api.ObjectStoreIOException;
import org.apache.james.blob.cassandra.CassandraBucketDAO;
import org.apache.james.blob.cassandra.CassandraDefaultBucketDAO;
import org.apache.james.metrics.api.Metric;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.util.DataChunker;
import org.apache.james.util.ReactorUtils;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

public class CassandraBlobStoreDAO
implements BlobStoreDAO {
    public static final boolean LAZY = false;
    public static final String CASSANDRA_BLOBSTORE_CL_ONE_MISS_COUNT_METRIC_NAME = "cassandraBlobStoreClOneMisses";
    public static final String CASSANDRA_BLOBSTORE_CL_ONE_HIT_COUNT_METRIC_NAME = "cassandraBlobStoreClOneHits";
    private static final Logger LOGGER = LoggerFactory.getLogger(CassandraBlobStoreDAO.class);
    private final CassandraDefaultBucketDAO defaultBucketDAO;
    private final CassandraBucketDAO bucketDAO;
    private final CassandraConfiguration configuration;
    private final BucketName defaultBucket;
    private final Metric metricClOneHitCount;
    private final Metric metricClOneMissCount;

    @Inject
    @VisibleForTesting
    public CassandraBlobStoreDAO(CassandraDefaultBucketDAO defaultBucketDAO, CassandraBucketDAO bucketDAO, CassandraConfiguration cassandraConfiguration, @Named(value="defaultBucket") BucketName defaultBucket, MetricFactory metricFactory) {
        this.defaultBucketDAO = defaultBucketDAO;
        this.bucketDAO = bucketDAO;
        this.configuration = cassandraConfiguration;
        this.defaultBucket = defaultBucket;
        this.metricClOneMissCount = metricFactory.generate(CASSANDRA_BLOBSTORE_CL_ONE_MISS_COUNT_METRIC_NAME);
        this.metricClOneHitCount = metricFactory.generate(CASSANDRA_BLOBSTORE_CL_ONE_HIT_COUNT_METRIC_NAME);
        if (Objects.equals(System.getenv("cassandra.blob.store.disable.startup.warning"), "false")) {
            LOGGER.warn("WARNING: JAMES-3591 Cassandra is not made to store large binary content, its use will be suboptimal compared to  alternatives (namely S3 compatible BlobStores backed by for instance S3, MinIO or Ozone)");
        }
    }

    public InputStream read(BucketName bucketName, BlobId blobId) throws ObjectStoreIOException, ObjectNotFoundException {
        return ReactorUtils.toInputStream(this.readBlobParts(bucketName, blobId));
    }

    public Publisher<InputStream> readReactive(BucketName bucketName, BlobId blobId) {
        return Mono.just((Object)this.read(bucketName, blobId));
    }

    public Mono<byte[]> readBytes(BucketName bucketName, BlobId blobId) {
        return this.readBlobParts(bucketName, blobId).collectList().map(this::byteBuffersToBytesArray);
    }

    public Mono<Void> save(BucketName bucketName, BlobId blobId, byte[] data) {
        Preconditions.checkNotNull((Object)data);
        return Mono.fromCallable(() -> DataChunker.chunk((byte[])data, (int)this.configuration.getBlobPartSize())).flatMap(chunks -> this.save(bucketName, blobId, (Flux<ByteBuffer>)chunks));
    }

    public Mono<Void> save(BucketName bucketName, BlobId blobId, InputStream inputStream) {
        Preconditions.checkNotNull((Object)bucketName);
        Preconditions.checkNotNull((Object)inputStream);
        return Mono.fromCallable(() -> ReactorUtils.toChunks((InputStream)inputStream, (int)this.configuration.getBlobPartSize()).subscribeOn(Schedulers.boundedElastic())).flatMap(chunks -> this.save(bucketName, blobId, (Flux<ByteBuffer>)chunks)).onErrorMap(e -> new ObjectStoreIOException("Exception occurred while saving input stream", e));
    }

    public Mono<Void> save(BucketName bucketName, BlobId blobId, ByteSource content) {
        return Mono.using(() -> ((ByteSource)content).openBufferedStream(), stream -> this.save(bucketName, blobId, (InputStream)stream), (Consumer)Throwing.consumer(InputStream::close).sneakyThrow(), (boolean)false);
    }

    private Mono<Void> save(BucketName bucketName, BlobId blobId, Flux<ByteBuffer> chunksAsFlux) {
        return this.saveBlobParts(bucketName, blobId, chunksAsFlux).flatMap(numberOfChunk -> this.saveBlobPartReference(bucketName, blobId, (Integer)numberOfChunk));
    }

    private Mono<Integer> saveBlobParts(BucketName bucketName, BlobId blobId, Flux<ByteBuffer> chunksAsFlux) {
        return chunksAsFlux.index().concatMap(pair -> this.writePart(bucketName, blobId, ((Long)pair.getT1()).intValue(), (ByteBuffer)pair.getT2())).count().map(Long::intValue);
    }

    private Mono<?> writePart(BucketName bucketName, BlobId blobId, int position, ByteBuffer data) {
        Mono<Void> write = this.isDefaultBucket(bucketName) ? this.defaultBucketDAO.writePart(data, blobId, position) : this.bucketDAO.writePart(data, bucketName, blobId, position);
        int anyNonEmptyValue = 1;
        return write.thenReturn((Object)anyNonEmptyValue);
    }

    private Mono<Void> saveBlobPartReference(BucketName bucketName, BlobId blobId, Integer numberOfChunk) {
        if (this.isDefaultBucket(bucketName)) {
            return this.defaultBucketDAO.saveBlobPartsReferences(blobId, numberOfChunk);
        }
        return this.bucketDAO.saveBlobPartsReferences(bucketName, blobId, numberOfChunk);
    }

    private boolean isDefaultBucket(BucketName bucketName) {
        return bucketName.equals((Object)this.defaultBucket);
    }

    public Mono<Void> delete(BucketName bucketName, BlobId blobId) {
        if (this.isDefaultBucket(bucketName)) {
            return this.defaultBucketDAO.deletePosition(blobId).then(this.defaultBucketDAO.deleteParts(blobId));
        }
        return this.bucketDAO.deletePosition(bucketName, blobId).then(this.bucketDAO.deleteParts(bucketName, blobId));
    }

    public Publisher<Void> delete(BucketName bucketName, Collection<BlobId> blobIds) {
        return Flux.fromIterable(blobIds).flatMap(id -> this.delete(bucketName, (BlobId)id), 16).then();
    }

    public Mono<Void> deleteBucket(BucketName bucketName) {
        Preconditions.checkNotNull((Object)bucketName);
        Preconditions.checkArgument((!this.isDefaultBucket(bucketName) ? 1 : 0) != 0, (Object)"Deleting the default bucket is forbidden");
        return this.bucketDAO.listAll().filter(bucketNameBlobIdPair -> ((BucketName)bucketNameBlobIdPair.getKey()).equals((Object)bucketName)).map(Pair::getValue).flatMap(blobId -> this.delete(bucketName, (BlobId)blobId), 16).then();
    }

    private Mono<ByteBuffer> readPart(BucketName bucketName, BlobId blobId, Integer partIndex) {
        if (this.configuration.isOptimisticConsistencyLevel()) {
            return this.readPartClOne(bucketName, blobId, partIndex).doOnNext(any -> this.metricClOneHitCount.increment()).switchIfEmpty(Mono.fromRunnable(() -> ((Metric)this.metricClOneMissCount).increment()).then(this.readPartClDefault(bucketName, blobId, partIndex)));
        }
        return this.readPartClDefault(bucketName, blobId, partIndex);
    }

    private Mono<ByteBuffer> readPartClOne(BucketName bucketName, BlobId blobId, Integer partIndex) {
        if (this.isDefaultBucket(bucketName)) {
            return this.defaultBucketDAO.readPartClOne(blobId, partIndex);
        }
        return this.bucketDAO.readPartClOne(bucketName, blobId, partIndex);
    }

    private Mono<ByteBuffer> readPartClDefault(BucketName bucketName, BlobId blobId, Integer partIndex) {
        if (this.isDefaultBucket(bucketName)) {
            return this.defaultBucketDAO.readPart(blobId, partIndex);
        }
        return this.bucketDAO.readPart(bucketName, blobId, partIndex);
    }

    private Mono<Integer> selectRowCount(BucketName bucketName, BlobId blobId) {
        if (this.configuration.isOptimisticConsistencyLevel()) {
            return this.selectRowCountClOne(bucketName, blobId).doOnNext(any -> this.metricClOneHitCount.increment()).switchIfEmpty(Mono.fromRunnable(() -> ((Metric)this.metricClOneMissCount).increment()).then(this.selectRowCountClDefault(bucketName, blobId)));
        }
        return this.selectRowCountClDefault(bucketName, blobId);
    }

    private Mono<Integer> selectRowCountClOne(BucketName bucketName, BlobId blobId) {
        if (this.isDefaultBucket(bucketName)) {
            return this.defaultBucketDAO.selectRowCountClOne(blobId);
        }
        return this.bucketDAO.selectRowCountClOne(bucketName, blobId);
    }

    private Mono<Integer> selectRowCountClDefault(BucketName bucketName, BlobId blobId) {
        if (this.isDefaultBucket(bucketName)) {
            return this.defaultBucketDAO.selectRowCount(blobId);
        }
        return this.bucketDAO.selectRowCount(bucketName, blobId);
    }

    private Flux<ByteBuffer> readBlobParts(BucketName bucketName, BlobId blobId) {
        return this.selectRowCount(bucketName, blobId).single().onErrorMap(NoSuchElementException.class, e -> new ObjectNotFoundException(String.format("Could not retrieve blob metadata for %s", blobId))).flatMapMany(rowCount -> Flux.range((int)0, (int)rowCount).concatMap(partIndex -> this.readPart(bucketName, blobId, (Integer)partIndex).single().onErrorMap(NoSuchElementException.class, e -> new ObjectNotFoundException(String.format("Missing blob part for blobId %s and position %d", blobId.asString(), partIndex)))));
    }

    private byte[] byteBuffersToBytesArray(List<ByteBuffer> byteBuffers) {
        int targetSize = byteBuffers.stream().mapToInt(Buffer::remaining).sum();
        return byteBuffers.stream().reduce(ByteBuffer.allocate(targetSize), ByteBuffer::put).array();
    }

    public Publisher<BucketName> listBuckets() {
        return this.bucketDAO.listAll().map(Pair::getLeft).distinct();
    }

    public Publisher<BlobId> listBlobs(BucketName bucketName) {
        if (this.isDefaultBucket(bucketName)) {
            return this.defaultBucketDAO.listBlobs();
        }
        return this.bucketDAO.listAll(bucketName);
    }
}

