/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.replication.regionserver;

import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.OptionalLong;
import java.util.concurrent.PriorityBlockingQueue;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.regionserver.wal.AbstractProtobufWALReader;
import org.apache.hadoop.hbase.regionserver.wal.WALHeaderEOFException;
import org.apache.hadoop.hbase.replication.regionserver.MetricsSource;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceLogQueue;
import org.apache.hadoop.hbase.replication.regionserver.WALFileLengthProvider;
import org.apache.hadoop.hbase.util.LeaseNotRecoveredException;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.wal.AbstractFSWALProvider;
import org.apache.hadoop.hbase.wal.WAL;
import org.apache.hadoop.hbase.wal.WALFactory;
import org.apache.hadoop.hbase.wal.WALTailingReader;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
class WALEntryStream
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(WALEntryStream.class);
    private WALTailingReader reader;
    private WALTailingReader.State state;
    private Path currentPath;
    private WAL.Entry currentEntry;
    private long currentPositionOfEntry = 0L;
    private long currentPositionOfReader = 0L;
    private final ReplicationSourceLogQueue logQueue;
    private final String walGroupId;
    private final FileSystem fs;
    private final Configuration conf;
    private final WALFileLengthProvider walFileLengthProvider;
    private final MetricsSource metrics;
    private boolean eofAutoRecovery;

    public WALEntryStream(ReplicationSourceLogQueue logQueue, FileSystem fs, Configuration conf, long startPosition, WALFileLengthProvider walFileLengthProvider, MetricsSource metrics, String walGroupId) {
        this.logQueue = logQueue;
        this.fs = fs;
        this.conf = conf;
        this.currentPositionOfEntry = startPosition;
        this.walFileLengthProvider = walFileLengthProvider;
        this.metrics = metrics;
        this.walGroupId = walGroupId;
        this.eofAutoRecovery = conf.getBoolean("replication.source.eof.autorecovery", false);
    }

    public HasNext hasNext() {
        if (this.currentEntry == null) {
            return this.tryAdvanceEntry();
        }
        return HasNext.YES;
    }

    public WAL.Entry peek() {
        return this.currentEntry;
    }

    public WAL.Entry next() {
        if (this.currentEntry == null) {
            throw new IllegalStateException("Call hasNext first");
        }
        WAL.Entry save = this.peek();
        this.currentPositionOfEntry = this.currentPositionOfReader;
        this.currentEntry = null;
        this.state = null;
        return save;
    }

    @Override
    public void close() {
        this.closeReader();
    }

    public long getPosition() {
        return this.currentPositionOfEntry;
    }

    public Path getCurrentPath() {
        return this.currentPath;
    }

    private String getCurrentPathStat() {
        StringBuilder sb = new StringBuilder();
        if (this.currentPath != null) {
            sb.append("currently replicating from: ").append(this.currentPath).append(" at position: ").append(this.currentPositionOfEntry).append("\n");
        } else {
            sb.append("no replication ongoing, waiting for new log");
        }
        return sb.toString();
    }

    private void setCurrentPath(Path path) {
        this.currentPath = path;
    }

    private void resetReader() throws IOException {
        if (this.currentPositionOfEntry > 0L) {
            this.reader.resetTo(this.currentPositionOfEntry, this.state.resetCompression());
        } else {
            this.reader.resetTo(-1L, true);
        }
    }

    @SuppressWarnings(value={"DCN_NULLPOINTER_EXCEPTION"}, justification="HDFS-4380")
    private HasNext prepareReader() {
        if (this.reader != null) {
            if (this.state != null && this.state != WALTailingReader.State.NORMAL) {
                LOG.debug("Reset reader {} to pos {}, reset compression={}", new Object[]{this.currentPath, this.currentPositionOfEntry, this.state.resetCompression()});
                try {
                    this.resetReader();
                    return HasNext.YES;
                }
                catch (FileNotFoundException e) {
                    throw new UncheckedIOException(e);
                }
                catch (IOException e) {
                    LOG.warn("Failed to reset reader {} to pos {}, reset compression={}", new Object[]{this.currentPath, this.currentPositionOfEntry, this.state.resetCompression(), e});
                    return HasNext.RETRY;
                }
            }
            return HasNext.YES;
        }
        PriorityBlockingQueue<Path> queue = this.logQueue.getQueue(this.walGroupId);
        Path nextPath = queue.peek();
        if (nextPath == null) {
            LOG.debug("No more WAL files in queue");
            this.setCurrentPath(null);
            return HasNext.NO;
        }
        this.setCurrentPath(nextPath);
        boolean beingWritten = this.walFileLengthProvider.getLogFileSizeIfBeingWritten(nextPath).isPresent();
        LOG.debug("Creating new reader {}, startPosition={}, beingWritten={}", new Object[]{nextPath, this.currentPositionOfEntry, beingWritten});
        try {
            this.reader = WALFactory.createTailingReader(this.fs, nextPath, this.conf, this.currentPositionOfEntry > 0L ? this.currentPositionOfEntry : -1L);
            return HasNext.YES;
        }
        catch (WALHeaderEOFException e) {
            if (!this.eofAutoRecovery) {
                return HasNext.RETRY;
            }
            LOG.warn("EOF while trying to open WAL reader for path: {}, startPosition={}", new Object[]{nextPath, this.currentPositionOfEntry, e});
            if (beingWritten) {
                return HasNext.RETRY;
            }
            this.dequeueCurrentLog();
            return HasNext.RETRY_IMMEDIATELY;
        }
        catch (LeaseNotRecoveredException e) {
            LOG.warn("Try to recover the WAL lease " + nextPath, (Throwable)e);
            AbstractFSWALProvider.recoverLease(this.conf, nextPath);
            return HasNext.RETRY;
        }
        catch (IOException | NullPointerException e) {
            LOG.warn("Failed to open WAL reader for path: {}", (Object)nextPath, (Object)e);
            return HasNext.RETRY;
        }
    }

    private HasNext lastAttempt() {
        LOG.debug("Reset reader {} for the last time to pos {}, reset compression={}", new Object[]{this.currentPath, this.currentPositionOfEntry, this.state.resetCompression()});
        try {
            this.resetReader();
        }
        catch (IOException e) {
            LOG.warn("Failed to reset reader {} to pos {}, reset compression={}", new Object[]{this.currentPath, this.currentPositionOfEntry, this.state.resetCompression(), e});
            return HasNext.RETRY;
        }
        Pair<WALTailingReader.State, Boolean> pair = this.readNextEntryAndRecordReaderPosition();
        this.state = pair.getFirst();
        assert (!pair.getSecond().booleanValue());
        if (!this.state.eof()) {
            return this.state == WALTailingReader.State.NORMAL ? HasNext.YES : HasNext.RETRY;
        }
        if (this.checkAllBytesParsed()) {
            this.dequeueCurrentLog();
            return HasNext.RETRY_IMMEDIATELY;
        }
        this.currentPositionOfEntry = 0L;
        this.currentPositionOfReader = 0L;
        this.state = WALTailingReader.State.ERROR_AND_RESET_COMPRESSION;
        return HasNext.RETRY;
    }

    private HasNext tryAdvanceEntry() {
        HasNext prepared = this.prepareReader();
        if (prepared != HasNext.YES) {
            return prepared;
        }
        Pair<WALTailingReader.State, Boolean> pair = this.readNextEntryAndRecordReaderPosition();
        this.state = pair.getFirst();
        boolean beingWritten = pair.getSecond();
        LOG.trace("Reading WAL {}; result={}, currently open for write={}", new Object[]{this.currentPath, this.state, beingWritten});
        switch (this.state) {
            case NORMAL: {
                return HasNext.YES;
            }
            case EOF_WITH_TRAILER: {
                if (beingWritten && this.logQueue.getQueue(this.walGroupId).size() <= 1) {
                    LOG.warn("We have reached the trailer while reading the file '{}' which is currently beingWritten, but it is the last file in log queue {}. This should not happen typically, try to read again so we will not miss anything", (Object)this.currentPath, (Object)this.walGroupId);
                    return HasNext.RETRY;
                }
                assert (!beingWritten || this.logQueue.getQueue(this.walGroupId).size() > 1);
                this.dequeueCurrentLog();
                return HasNext.RETRY_IMMEDIATELY;
            }
            case EOF_AND_RESET: 
            case EOF_AND_RESET_COMPRESSION: {
                if (beingWritten) {
                    return HasNext.RETRY;
                }
                return this.lastAttempt();
            }
            case ERROR_AND_RESET: 
            case ERROR_AND_RESET_COMPRESSION: {
                return HasNext.RETRY;
            }
        }
        throw new IllegalArgumentException("Unknown read next result: " + (Object)((Object)this.state));
    }

    private FileStatus getCurrentPathFileStatus() throws IOException {
        try {
            return this.fs.getFileStatus(this.currentPath);
        }
        catch (FileNotFoundException e) {
            Path archivedWAL = AbstractFSWALProvider.findArchivedLog(this.currentPath, this.conf);
            if (archivedWAL != null) {
                return this.fs.getFileStatus(archivedWAL);
            }
            throw e;
        }
    }

    private boolean checkAllBytesParsed() {
        long trailerSize = this.currentTrailerSize();
        FileStatus stat = null;
        try {
            stat = this.getCurrentPathFileStatus();
        }
        catch (IOException e) {
            LOG.warn("Couldn't get file length information about log {}, it {} closed cleanly {}", new Object[]{this.currentPath, trailerSize < 0L ? "was not" : "was", this.getCurrentPathStat(), e});
            this.metrics.incrUnknownFileLengthForClosedWAL();
        }
        if (stat != null) {
            if (trailerSize < 0L) {
                if (this.currentPositionOfReader < stat.getLen()) {
                    long skippedBytes = stat.getLen() - this.currentPositionOfReader;
                    LOG.warn("Reached the end of WAL {}. It was not closed cleanly, so we did not parse {} bytes of data.", (Object)this.currentPath, (Object)skippedBytes);
                    this.metrics.incrUncleanlyClosedWALs();
                    this.metrics.incrBytesSkippedInUncleanlyClosedWALs(skippedBytes);
                }
            } else if (this.currentPositionOfReader + trailerSize < stat.getLen()) {
                LOG.warn("Processing end of WAL {} at position {}, which is too far away from reported file length {}. Restarting WAL reading (see HBASE-15983 for details). {}", new Object[]{this.currentPath, this.currentPositionOfReader, stat.getLen(), this.getCurrentPathStat()});
                this.metrics.incrRestartedWALReading();
                this.metrics.incrRepeatedFileBytes(this.currentPositionOfReader);
                return false;
            }
        }
        LOG.debug("Reached the end of {} and length of the file is {}", (Object)this.currentPath, stat == null ? "N/A" : Long.valueOf(stat.getLen()));
        this.metrics.incrCompletedWAL();
        return true;
    }

    private void dequeueCurrentLog() {
        LOG.debug("EOF, closing {}", (Object)this.currentPath);
        this.closeReader();
        this.logQueue.remove(this.walGroupId);
        this.setCurrentPath(null);
        this.currentPositionOfEntry = 0L;
        this.state = null;
    }

    private Pair<WALTailingReader.State, Boolean> readNextEntryAndRecordReaderPosition() {
        WALTailingReader.Result readResult;
        OptionalLong fileLength;
        block4: {
            fileLength = this.logQueue.getQueueSize(this.walGroupId) > 1 ? OptionalLong.empty() : this.walFileLengthProvider.getLogFileSizeIfBeingWritten(this.currentPath);
            readResult = this.reader.next(fileLength.orElse(-1L));
            long readerPos = readResult.getEntryEndPos();
            WAL.Entry readEntry = readResult.getEntry();
            if (readResult.getState() == WALTailingReader.State.NORMAL) {
                LOG.trace("reading entry: {} ", (Object)readEntry);
                this.metrics.incrLogEditsRead();
                this.metrics.incrLogReadInBytes(readerPos - this.currentPositionOfEntry);
                this.currentEntry = readResult.getEntry();
                this.currentPositionOfReader = readerPos;
            } else {
                LOG.trace("reading entry failed with: {}", (Object)readResult.getState());
                this.currentEntry = null;
                try {
                    this.currentPositionOfReader = this.reader.getPosition();
                }
                catch (IOException e) {
                    LOG.warn("failed to get current position of reader", (Throwable)e);
                    if (!readResult.getState().resetCompression()) break block4;
                    return Pair.newPair(WALTailingReader.State.ERROR_AND_RESET_COMPRESSION, fileLength.isPresent());
                }
            }
        }
        return Pair.newPair(readResult.getState(), fileLength.isPresent());
    }

    private void closeReader() {
        if (this.reader != null) {
            this.reader.close();
            this.reader = null;
        }
    }

    private long currentTrailerSize() {
        long size = -1L;
        if (this.reader instanceof AbstractProtobufWALReader) {
            AbstractProtobufWALReader pbwr = (AbstractProtobufWALReader)((Object)this.reader);
            size = pbwr.trailerSize();
        }
        return size;
    }

    public static enum HasNext {
        YES,
        RETRY,
        RETRY_IMMEDIATELY,
        NO;

    }
}

