/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hadoop.hbase.nio.SingleByteBuff;
import org.apache.hadoop.hbase.util.ReflectionUtils;
import org.apache.hadoop.hbase.util.UnsafeAccess;
import org.apache.hbase.thirdparty.com.google.common.collect.Sets;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class ByteBuffAllocator {
    private static final Logger LOG = LoggerFactory.getLogger(ByteBuffAllocator.class);
    public static final ByteBuffAllocator HEAP = ByteBuffAllocator.createOnHeap();
    public static final String ALLOCATOR_POOL_ENABLED_KEY = "hbase.server.allocator.pool.enabled";
    public static final String MAX_BUFFER_COUNT_KEY = "hbase.server.allocator.max.buffer.count";
    public static final String BUFFER_SIZE_KEY = "hbase.server.allocator.buffer.size";
    public static final String MIN_ALLOCATE_SIZE_KEY = "hbase.server.allocator.minimal.allocate.size";
    public static final String BYTEBUFF_ALLOCATOR_CLASS = "hbase.bytebuff.allocator.class";
    @Deprecated
    public static final String DEPRECATED_ALLOCATOR_POOL_ENABLED_KEY = "hbase.ipc.server.reservoir.enabled";
    @Deprecated
    static final String DEPRECATED_MAX_BUFFER_COUNT_KEY = "hbase.ipc.server.reservoir.initial.max";
    @Deprecated
    static final String DEPRECATED_BUFFER_SIZE_KEY = "hbase.ipc.server.reservoir.initial.buffer.size";
    public static final int DEFAULT_BUFFER_SIZE = 66560;
    public static final Recycler NONE = () -> {};
    protected final boolean reservoirEnabled;
    protected final int bufSize;
    private final int maxBufCount;
    private final AtomicInteger usedBufCount = new AtomicInteger(0);
    private boolean maxPoolSizeInfoLevelLogged = false;
    private final int minSizeForReservoirUse;
    private final Queue<ByteBuffer> buffers = new ConcurrentLinkedQueue<ByteBuffer>();
    private final LongAdder poolAllocationBytes = new LongAdder();
    private final LongAdder heapAllocationBytes = new LongAdder();
    private long lastPoolAllocationBytes = 0L;
    private long lastHeapAllocationBytes = 0L;

    public static ByteBuffAllocator create(Configuration conf, boolean reservoirEnabled) {
        int poolBufSize = conf.getInt(BUFFER_SIZE_KEY, 66560);
        if (reservoirEnabled) {
            if (poolBufSize <= 0) {
                throw new IllegalArgumentException("hbase.server.allocator.buffer.size must be positive. Please disable the reservoir rather than setting the size of the buffer to zero or negative.");
            }
            int bufsForTwoMB = 0x200000 / poolBufSize;
            int maxBuffCount = conf.getInt(MAX_BUFFER_COUNT_KEY, conf.getInt("hbase.regionserver.handler.count", 30) * bufsForTwoMB * 2);
            int minSizeForReservoirUse = conf.getInt(MIN_ALLOCATE_SIZE_KEY, poolBufSize / 6);
            if (minSizeForReservoirUse <= 0) {
                LOG.warn("The minimal size for reservoir use is less or equal to zero, all allocations will be from the pool. Set a higher hbase.server.allocator.minimal.allocate.size to avoid this.");
            }
            Class<?> clazz = conf.getClass(BYTEBUFF_ALLOCATOR_CLASS, ByteBuffAllocator.class);
            return (ByteBuffAllocator)ReflectionUtils.newInstance(clazz, true, maxBuffCount, poolBufSize, minSizeForReservoirUse);
        }
        return HEAP;
    }

    private static ByteBuffAllocator createOnHeap() {
        return new ByteBuffAllocator(false, 0, 66560, Integer.MAX_VALUE);
    }

    protected ByteBuffAllocator(boolean reservoirEnabled, int maxBufCount, int bufSize, int minSizeForReservoirUse) {
        this.reservoirEnabled = reservoirEnabled;
        this.maxBufCount = maxBufCount;
        this.bufSize = bufSize;
        this.minSizeForReservoirUse = minSizeForReservoirUse;
    }

    public boolean isReservoirEnabled() {
        return this.reservoirEnabled;
    }

    public long getHeapAllocationBytes() {
        return this.heapAllocationBytes.sum();
    }

    public long getPoolAllocationBytes() {
        return this.poolAllocationBytes.sum();
    }

    public int getBufferSize() {
        return this.bufSize;
    }

    public int getUsedBufferCount() {
        return this.usedBufCount.intValue();
    }

    public int getFreeBufferCount() {
        return this.buffers.size();
    }

    public int getTotalBufferCount() {
        return this.maxBufCount;
    }

    public static long getHeapAllocationBytes(ByteBuffAllocator ... allocators) {
        long heapAllocBytes = 0L;
        for (ByteBuffAllocator alloc : Sets.newHashSet(allocators)) {
            heapAllocBytes += alloc.getHeapAllocationBytes();
        }
        return heapAllocBytes;
    }

    public static double getHeapAllocationRatio(ByteBuffAllocator ... allocators) {
        double heapDelta = 0.0;
        double poolDelta = 0.0;
        for (ByteBuffAllocator alloc : Sets.newHashSet(allocators)) {
            long heapAllocBytes = alloc.heapAllocationBytes.sum();
            long poolAllocBytes = alloc.poolAllocationBytes.sum();
            heapDelta += (double)(heapAllocBytes - alloc.lastHeapAllocationBytes);
            poolDelta += (double)(poolAllocBytes - alloc.lastPoolAllocationBytes);
            alloc.lastHeapAllocationBytes = heapAllocBytes;
            alloc.lastPoolAllocationBytes = poolAllocBytes;
        }
        if (Math.abs(heapDelta + poolDelta) < 0.001) {
            return 0.0;
        }
        return heapDelta / (heapDelta + poolDelta);
    }

    public SingleByteBuff allocateOneBuffer() {
        ByteBuffer bb;
        if (this.isReservoirEnabled() && (bb = this.getBuffer()) != null) {
            return new SingleByteBuff(() -> this.putbackBuffer(bb), bb);
        }
        return (SingleByteBuff)ByteBuff.wrap(this.allocateOnHeap(this.bufSize));
    }

    private ByteBuffer allocateOnHeap(int size) {
        this.heapAllocationBytes.add(size);
        return ByteBuffer.allocate(size);
    }

    public ByteBuff allocate(int size) {
        ByteBuffer bb;
        int remain;
        if (size < 0) {
            throw new IllegalArgumentException("size to allocate should >=0");
        }
        if (!this.isReservoirEnabled() || size == 0) {
            return ByteBuff.wrap(this.allocateOnHeap(size));
        }
        int reminder = size % this.bufSize;
        int len = size / this.bufSize + (reminder > 0 ? 1 : 0);
        ArrayList<ByteBuffer> bbs = new ArrayList<ByteBuffer>(len);
        for (remain = size; remain >= this.minSizeForReservoirUse && (bb = this.getBuffer()) != null; remain -= this.bufSize) {
            bbs.add(bb);
        }
        int lenFromReservoir = bbs.size();
        if (remain > 0) {
            bbs.add(this.allocateOnHeap(remain));
        }
        ByteBuff bb2 = lenFromReservoir == 0 ? ByteBuff.wrap(bbs) : ByteBuff.wrap(bbs, () -> {
            for (int i = 0; i < lenFromReservoir; ++i) {
                this.putbackBuffer((ByteBuffer)bbs.get(i));
            }
        });
        bb2.limit(size);
        return bb2;
    }

    public void clean() {
        while (!this.buffers.isEmpty()) {
            ByteBuffer b = this.buffers.poll();
            if (!b.isDirect()) continue;
            UnsafeAccess.freeDirectBuffer(b);
        }
        this.usedBufCount.set(0);
        this.maxPoolSizeInfoLevelLogged = false;
        this.poolAllocationBytes.reset();
        this.heapAllocationBytes.reset();
        this.lastPoolAllocationBytes = 0L;
        this.lastHeapAllocationBytes = 0L;
    }

    private ByteBuffer getBuffer() {
        int c;
        ByteBuffer bb = this.buffers.poll();
        if (bb != null) {
            bb.clear();
            this.poolAllocationBytes.add(this.bufSize);
            return bb;
        }
        do {
            if ((c = this.usedBufCount.intValue()) < this.maxBufCount) continue;
            if (!this.maxPoolSizeInfoLevelLogged) {
                LOG.info("Pool already reached its max capacity : {} and no free buffers now. Consider increasing the value for '{}' ?", (Object)this.maxBufCount, (Object)MAX_BUFFER_COUNT_KEY);
                this.maxPoolSizeInfoLevelLogged = true;
            }
            return null;
        } while (!this.usedBufCount.compareAndSet(c, c + 1));
        this.poolAllocationBytes.add(this.bufSize);
        return ByteBuffer.allocateDirect(this.bufSize);
    }

    protected void putbackBuffer(ByteBuffer buf) {
        if (buf.capacity() != this.bufSize || this.reservoirEnabled ^ buf.isDirect()) {
            LOG.warn("Trying to put a buffer, not created by this pool! Will be just ignored");
            return;
        }
        this.buffers.offer(buf);
    }

    public static interface Recycler {
        public void free();
    }
}

