/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* HEADER.C --- Window headers that allow child movement.
   Approach to moving windows is to create a transparent
   cursor window and another transparent window just to
   catch button events that blanket everything. */
#include <stdlib.h>
#include <assert.h>
#include <graphics.h>
#include "window.h"

enum {

header_y_margin = 3,
header_x_margin = 4,
header_bar_color = cLIGHTGRAY,
header_text_color = cBLACK,
/* Following is work-around because xor color of
   cursor is computed wrt the root window color
   rather than the canvas color. */
move_cursor_color = cLIGHTGRAY ^ (/* fudge: */ cDARKGRAY ^ cBLUE),
move_button = bLEFT,

};

/* Handle the map of a header bar by drawing it. */
static void handle_map(EVENT e)
{
  int center_x, half_title_width, x0, x1, y1, bars_p;
  HEADER h = (HEADER)e->map.window;
  char *title = *h->title;

  push_graphics_state(&h->window, 1);
  settextjustify(CENTER_TEXT, TOP_TEXT);
  setfillstyle(SOLID_FILL, header_bar_color);
  setcolor(header_text_color);
  center_x = h->window.width/2;
  half_title_width = textwidth(title) / 2 + header_x_margin;
  x1 = center_x - half_title_width;
  y1 = h->window.height - header_y_margin - 1;

  /* Only draw the accent bars if there's room. */
  bars_p = (2*header_x_margin < x1);

  protect_cursor(&h->window);

  if (bars_p) {
    bar(header_x_margin+1, header_y_margin+1, x1-1, y1-1);
    rectangle(header_x_margin, header_y_margin, x1, y1);
  }

  outtextxy(center_x, header_y_margin, title);
  x0 = center_x + half_title_width;
  x1 = h->window.width - header_x_margin;

  if (bars_p) {
    bar(x0+1, header_y_margin+1, x1-1, y1-1);
    rectangle(x0, header_y_margin, x1, y1);
  }

  unprotect_cursor();
  pop_graphics_state();
}
#undef title

static CURSOR cursor = NULL;
static WINDOW moved_window;

/* Handle a button release in the overlaid button window.
   A certain order is needed to keep the xor cursor in
   synch and prevent spurious enter/leave events in the
   underlying window. */
static void handle_button_release(EVENT e)
{
  if (cursor == NULL || e->mouse.button != move_button)
    return;

  /* Turn off cursor. */
  show_cursor(cursor, 0);

  /* Move window.
     Coords of event are wrt cursor window, which are the same
     as root window coords.  So we must calculate new location coords
     wrt moved window parent for locate to function correctly.
     We retrieve the offset of the mouse cursor in the window
     rect cursor from the rect cursor description.  */
  locate_window(moved_window,
		e->mouse.x - moved_window->parent->x - cursor->x0,
		e->mouse.y - moved_window->parent->y - cursor->y0, 1);
  map_window(moved_window);
  unprotect_focus();

  /* Kill cursor window. */
  destroy_window((WINDOW)cursor);
  cursor = NULL;
}

/* Handle a button press in a header.  Create the
   transparent cursor and button-catching windows
   and map them.  Erase the parent of the header
   and turn on the xor window cursor. */
static void handle_button_press(EVENT e)
{
  if (e->mouse.button != move_button)
    return;

  moved_window = e->mouse.window->parent;
  walloc(&cursor, sizeof(CURSOR_REC));
  open_cursor(cursor, 0, 0,
	      root_window->width, root_window->height,
	      move_cursor_color, root_window);
  set_cursor_handlers(cursor, 
		      null_handler_code, 
		      handle_button_release);
  set_cursor(cursor, bit(cRECT)|bit(cMOUSE), 
	     e->mouse.x, e->mouse.y,
	     moved_window->width, moved_window->height);
  map_window(&cursor->window);
  cursor->window.event_mask |= bit(eKEYSTROKE);
  protect_focus();
  unmap_window(moved_window);
  show_cursor(cursor, 1);
  ungrab_mouse();
}

/* Return the overall height of a header bar. */
int header_bar_height (void)
{
  return header_y_margin + textheight("Hy") + header_y_margin;
}

/* Event dispatcher. */
BeginDefDispatch(header_bar)
  Dispatch(eMAP, handle_map)
  Dispatch(eBUTTON_PRESS, handle_button_press)
  Dispatch(eBUTTON_RELEASE, handle_button_release)
EndDefDispatch(header_bar)

/* Build a new header bar with the given title,
   which is allowed to change between mappings. */
HEADER make_header_bar(char **title, WINDOW parent)
{
  HEADER h = NULL;

  walloc(&h, sizeof(HEADER_REC));
  open_window(&h->window, parent, 0, 0, parent->width, header_bar_height(),
	      parent->border_width, parent->border_color, parent->bg_color,
	      bit(eMAP)|bit(eBUTTON_PRESS)|bit(eBUTTON_RELEASE));
  h->title = title;
  SetDispatch(&h->window, header_bar);
  map_window(&h->window);
  return h;
}

