/*
 * LDAPTableQuery.java    1.1
 *
 * Copyright (c) 1997 Netscape Communications Corporation
 *
 * Netscape grants you a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Netscape.
 *
 * This software is provided "AS IS," without a warranty of any kind.
 * See the CDK License Agreement for additional terms and conditions.
 */
package netscape.ldap.beans;

import java.util.*;
import netscape.peas.*;
import netscape.ldap.*;

/**
 *<pre>
 * History:
 *  v1.0: Rob Weltman
 *  v1.1: cdk_team
 *</pre>
 */
public class LDAPTableQuery  extends TableSupport {
    public void rowChange( RowChangeEvent oEvent ) {
        super.rowChange(oEvent);
    }

	public void tableChange( TableChangeEvent oEvent ) {
        super.tableChange(oEvent);
    }

    /**
	 * "true" is passed to supertype TableSupport to
     * indicate that this object must own its data context (column
	 * names & row numbers).
     */
    public LDAPTableQuery() {
        super( true );
        setFetchBufferSize( 10 );
		/* Set some initial useful attributes to fetch */
		int n = 2;
		_attrs = new String[n];
		_attrs[0] = "cn";
		_attrs[1] = "telephonenumber";
		_colNames = new String[n];
		_colNames[0] = "Name";
		_colNames[1] = "PhoneNumber";
		_colWidths = new int[n];
		_colWidths[0] = 25*8;
		_colWidths[1] = 15*8;
    }

	/**
	 * Subclasses of TableSupport need to do the following in their
	 * version of executeQuery:
	 *    - setup the column names, if they are out of data
	 *    - be prepared for calls to fetChMoreRows.
	 *    - call TableSupport's implementation of execute query.
	 * This resets buffer and causes fetching
	 */
	public void executeQuery() {
		disconnect();
		if ( _attrs != null ) {
			// give to our RowSupport object the information it needs...
			if ( !updateColumnInfo() ) {
				return;
			}

			printDebug( "executeQuery: calling getEntries()" );
			_results = getEntries();

			// let default implementation handle the rest
			printDebug( "executeQuery: calling super.executeQuery()" );
			super.executeQuery();
		}
    } // executeQuery


    /**
     * Override of TableSupport's "do-nothing" implementation.
     * Get more rows, if there are more search results.
	 * For LDAP, just get everything.
     */
    public void fetchMoreRows() {

		printDebug( "fetchMoreRows: _results = " + _results );
		if ( _results == null )
			return;

		int iStartRow = getNumRowsFetched();
		int iFinishRow = iStartRow+getFetchBufferSize() - 1;

		int iNumCols = getNumColumns();
		Object[] aRowValues = new Object[iNumCols];

		for ( int iRow = iStartRow; ; iRow++ ) {
			//printDebug( "calling _results.hasMoreElements()" );
			/* If we're at the end of the search results, terminate */
			if ( !_results.hasMoreElements() ) {
				printDebug( "No more search results" );
				_results = null;
				disconnect();
				break;
			}
			//printDebug( "getting the next entry" );
			LDAPEntry entry = (LDAPEntry)_results.nextElement();
			printDebug( "Next row = " + entry.getDN() );
			for ( int iCol = 0; iCol < iNumCols; iCol ++ ) {
				LDAPAttribute attr = null;
				String s = "";
				try {
					attr = entry.getAttribute( _attrs[iCol] );
				} catch ( Exception e ) {
				}
				if ( attr != null ) {
					//printDebug( "Got attribute " + attr.getName() );
					Enumeration en = attr.getStringValues();
					boolean first = true;
					while ( en.hasMoreElements() ) {
						if ( !first )
							s = s + "\n";
						else
							first = false;
						s = s + (String)en.nextElement();
					}
				} else {
					//printDebug( "No attribute " + _attrs[iCol] + " returned" );
				}
				//printDebug( "Value = " + s );
				aRowValues[iCol] = s;
			}
			// Adds values to our model.  We notify listeners after
			// all values are added using a "ROWS_INSERTED" event
			//printDebug( "calling addRowValues" );
			addRowValues( aRowValues, false );
			//printDebug( "firing off a ROWS_INSERTED TableChangeEvent" );
			if ( !getBuffered() )
				fireTableChange( TableChangeEvent.ROWS_INSERTED, iRow,
								 0, null, this, null );
		}
		if ( getBuffered() )
			fireTableChange( TableChangeEvent.ROWS_INSERTED, 0,
							 0, null, this, null );
	} // fetchMoreRows


    private void disconnect() {
		try {
			if ( (_ldc != null) && _ldc.isConnected() )
				_ldc.disconnect();
		} catch ( Exception e ) {
		}
		_ldc = null;
	}

	/**
	 * Searches and returns search results.
	 * @return Search results.
	 */
    private LDAPSearchResults getEntries() {
		boolean invalid = false;
		if ( getFilter().length() < 1 ) {
		    printDebug( "Invalid filter" );
			invalid = true;
		} else if ( (getHost().length() < 1) || (getBase().length() < 1) ) {
		    printDebug( "Invalid filter" );
			invalid = true;
		} else if ( (_attrs == null) || (_attrs.length < 1) ) {
		    printDebug( "No attributes to return" );
			invalid = true;
		}
		if ( invalid ) {
			setErrorCode( INVALID_PARAMETER );
			return null;
		}

		LDAPSearchResults res = null;
		_ldc = new LDAPConnection();
		try {
			try {
				printDebug("Connecting to " + getHost() +
						   " " + getPort());
				connect( _ldc, getHost(), getPort());
			} catch (Exception e) {
				printDebug( "Failed to connect to " + getHost() + ": " +
							e.toString() );
				setErrorCode( CONNECT_ERROR );
				_ldc = null;
				throw( new Exception() );
			}

			// Authenticate?
			if ( (!getAuthDN().equals("")) &&
				 (!getAuthPassword().equals("")) ) {

				printDebug( "Authenticating " + getAuthDN() );
				try {
					_ldc.authenticate( getAuthDN(), getAuthPassword() );
				} catch (Exception e) {
					printDebug( "Failed to authenticate: " + e.toString() );
					setErrorCode( AUTHENTICATION_ERROR );
					throw( new Exception() );
				}
			}

			// Search
			try {
				LDAPSearchConstraints cons = _ldc.getSearchConstraints();
				cons.setBatchSize( 1 );
				printDebug("Searching " + getBase() +
						   " for " + getFilter() + ", scope = " + getScope());
				res = _ldc.search( getBase(),
								  getScope(),
								  getFilter(),
								  _attrs,
								  false,
								  cons );

			} catch (Exception e) {
				printDebug( "Failed to search for " + getFilter() + ": " +
							e.toString() );
				setErrorCode( PROPERTY_NOT_FOUND );
			}
		} catch (Exception e) {
		}

		return res;
	}


	// Make sure there are default widths for columns without
	// defined widths
	private int[] adjustColWidths() {
		int[] w = _colWidths;
		if ( _colNames.length != _colWidths.length ) {
			w = new int[_colNames.length];
			if ( _colNames.length > _colWidths.length ) {
				for( int i = 0; i < _colWidths.length; i++ )
					w[i] = _colWidths[i];
				for( int i = _colWidths.length; i < _colNames.length; i++ )
					w[i] = DEFAULT_COL_WIDTH;
			} else if ( _colNames.length < _colWidths.length ) {
				for( int i = 0; i < _colNames.length; i++ )
					w[i] = _colWidths[i];
			}
		}
		return w;
	}

	private String[] adjustColNames() {
		String[] n = _colNames;
		return n;
	}

	/**
	 * Set up column names and widths in RowSupport
	 *
	 */
    private boolean updateColumnInfo() {
		// Make sure there are default widths for columns without
		// defined widths
		int[] w = adjustColWidths();
		String[] n = adjustColNames();

		// give to our RowSupport object the information it needs...
		setNumColumns( n.length );

		// setting our column names.
		// this causes TableChangeEvent.COLUMN_INFO_CHANGED and
		// RowChangeEvent.COLUMN_INFO_CHANGED to be fired
		try {
			printDebug( "updateColumnInfo: " + n.length + " columns" );
			setColumnNames( n, w, null );
		} catch ( Exception e ) {
			System.err.println( "setColumnNames: " + e.toString() );
			System.err.println( "One or more column names has embedded " +
								"space" );
			System.err.println( n.length + " attributes" );
			for( int i = 0; i < n.length; i++ ) {
				System.err.println( "  Column " + n[i] + ", " +
									w[i] );
			}
			return false;
		}
		return true;
	}


	/**
	 * Returns the column headings.
	 * @return Comma-delimited list of headings.
	 */
    public String getColNames() {
		return compactNames( _colNames );
	}

	/**
	 * Sets the column headings
	 * @param attrs Comma-delimited list of column headings.
	 */
    public void setColNames( String names ) {
		_colNames = expandNames( names );
		updateColumnInfo();
	}

	/**
	 * Returns the column widths.
	 * @return Comma-delimited list of widths.
	 */
    public String getColWidths() {
		if ( _colWidths == null )
			return null;
		String s = "";
		for ( int i = 0; i < _colWidths.length; i++ ) {
			s = s + _colWidths[i];
			if ( i < (_colWidths.length - 1) )
				s = s + ",";
		}
		s.trim();
		return s; 
	}

	/**
	 * Sets the column widths
	 * @param attrs Comma-delimited list of column widths.
	 */
    public void setColWidths( String widths ) {
		_colWidths = null;
		String[] s = expandNames( widths );
		if ( (s != null) && (s.length > 0) ) {
			_colWidths = new int[s.length];
			for( int i = 0; i < s.length; i++ )
				_colWidths[i] = Integer.parseInt( s[i] );
		}
		updateColumnInfo();
	}

	/**
	 * Returns true if all rows are output before updating table.
	 * @return true if all rows are output before updating table.
	 */
    public boolean getBuffered() { 
		return _buffered; 
	}

	/**
	 * Turns buffering of output rows on or off.
	 * @param on true to not update table until all rows are in.
	 */
    public void setBuffered( boolean on ) { 
	    _buffered = on;
	}


	/***************************************/

    /* Since we can only inherit from one class, we have to include here
	   the contents of LDAPBasePropertySupport.java :-(
	*/

	/**
	 * Returns the host to search at.
	 * @return DNS name or dotted IP name of host to search at
	 */
    public String getHost() { 
		return _host; 
	}

	/**
	* Sets host string.
	* @param theHost host name
	*/
	public void setHost( String theHost ) {
	    _host = theHost;
	}

	/**
	 * Returns the port to search at.
	 * @return Port to search at
	 */
    public int getPort() { 
		return _port; 
	}

	/**
	* Sets port number.
	* @param thePort port
	*/
	public void setPort( int thePort ) {
		_port = thePort;
	}

	/**
	 * Returns the directory base to search at.
	 * @return directory base to search
	 */
    public String getBase() { 
		return _base; 
	}

	/**
	* Sets the starting base
	* @param theBase starting base
	*/
	public void setBase( String theBase ) {
		_base = theBase;
		if ( !_authRDN.equals( "" ) )
		    _authDN = _authRDN + "," + _base;
	}

	/**
	 * Returns the DN to authenticate as; null or empty for anonymous.
	 * @return DN to authenticate as
	 */
    public String getAuthDN() { 
		return _authDN; 
	}

	/**
	 * Sets the DN to authenticate as; null or empty for anonymous.
	 * @param authDN the DN to authenticate as
	 */
    public void setAuthDN( String authDN ) { 
		this._authDN =  authDN;
	}

	/**
	 * Returns the password for the DN to authenticate as
	 * @return Password of DN to authenticate as
	 */
    public String getAuthPassword() { 
		return _authPassword; 
	}

	/**
	 * Sets the password for the DN to authenticate as
	 * @param authPassword the password to use in authentication
	 */
    public void setAuthPassword( String authPassword ) { 
		this._authPassword = authPassword; 
	}

	/**
	 * Returns the RDN part of the DN to authenticate as
	 * @return RDN of DN to authenticate as
	 */
    public String getAuthRDN() { 
		return _authRDN; 
	}

	/**
	 * Sets the RDN of the DN to authenticate as. This assumes that
	 * the directory base has been set either in the constructor or
	 * with setBase. The RDN should be of the form "cn=Polly Plum".
	 * If the base was "ou=Accounting,o=Ace Industry,c=us", the DN
	 * will be "cn=Polly Plum, ou=Accounting,o=Ace Industry,c=us".
	 * @param rdn The RDN to use to compose a DN
	 */
    public void setAuthRDN( String rdn ) { 
	    _authRDN = rdn;
		if ( !_base.equals( "" ) )
		    _authDN = _authRDN + "," + _base;
	}

	/**
	 * Returns the cn of the DN to authenticate as
	 * @return cn of DN to authenticate as
	 */
    public String getUserName() { 
	    if ( !_authRDN.equals( "" ) ) {
		    StringTokenizer st = new StringTokenizer( _authRDN, "=" );
			if ( st.hasMoreTokens() )
			    return st.nextToken();
		}
		return new String("");
	}

	/**
	 * Sets the RDN of the DN to authenticate as by prepending
	 * "cn=" to the supplied user name. This assumes that
	 * the directory base has been set either in the constructor or
	 * with setBase. The name should be of the form "Polly Plum".
	 * @param name the value of the cn to be used as an RDN
	 */
    public void setUserName( String name ) { 
	    setAuthRDN( "cn=" + name );
	}

	/**
	 * Set the search scope using a string
	 * @param scope either "SUB", "ONE", or "BASE"
	 */
    public void setScope( String scope ) {
		if ( scope.equalsIgnoreCase("SUB") )
			setScope( LDAPConnection.SCOPE_SUB );
		else if ( scope.equalsIgnoreCase("ONE") )
			setScope( LDAPConnection.SCOPE_ONE );
		else if ( scope.equalsIgnoreCase("BASE") )
			setScope( LDAPConnection.SCOPE_BASE );
	}

	/**
	 * Get the current search scope
	 * @return the current search scope as integer
	 */
    public int getScope() {
		return _scope;
	}

	/**
	 * Set the search scope using an integer
	 * @param scope one of LDAPConnection.SCOPE_BASE,
	 * LDAPConnection.SCOPE_SUB, LDAPConnection.SCOPE_ONE
	 */
    public void setScope( int scope ) {
		_scope = scope;
	}

	/**
	 * Returns the search filter
	 * @return search filter
	 */
    public String getFilter() { 
		return _filter; 
	}

	/**
	 * Sets the search filter
	 * @param filter search filter
	 */
    public void setFilter( String filter ) {
		_filter = filter;
	}

    private String[] expandNames( String names ) {
		String[] s = null;
		if ( names == null ) {
			printDebug( "Null list" );
			return s;
		}
		names.trim();
		StringTokenizer st = new StringTokenizer( names, "," );
		int count = st.countTokens();
		if ( count < 1 ) {
			printDebug( "Invalid list" );
			return s;
		}
		s = new String[count];
		int i = 0;
		while( st.hasMoreTokens() ) {
			s[i] = st.nextToken();
			i++;
		}
		return s;
	}

    protected String compactNames( String[] list ) {
		if ( list == null )
			return null;
		String s = "";
		for ( int i = 0; i < list.length; i++ ) {
			s = s + list[i];
			if ( i < (list.length - 1) )
				s = s + ",";
		}
		s.trim();
		return s; 
	}

	/**
	 * Returns the attributes to be fetched on search.
	 * @return Comma-delimited list of attributes.
	 */
    public String getAttributes() {
		return compactNames( _attrs );
	}

	/**
	 * Sets the list of attributes to return on search.
	 * @param attrs Comma-delimited list of attributes.
	 */
    public void setAttributes( String attrs ) {
		_attrs = expandNames( attrs );
	}

	/**
	 * Returns true if debug output is on
	 * @return true if debug output is on
	 */
    public boolean getDebug() { 
		return _debug; 
	}

	/**
	 * Turns debug output on or off
	 * @param on true for debug output
	 */
    public void setDebug( boolean on ) { 
	    _debug = on;
	}

	/**
	 * Returns the latest error code
	 * @return The latest error code
	 */
    public int getErrorCode() { 
		return _errCode; 
	}

	/**
	 * Sets an error code for retrieval by a client
	 * @param code An error code
	 */
    protected void setErrorCode( int code ) { 
	    _errCode = code;
	}

    protected void printDebug( String s ) {
		if ( _debug )
			System.out.println( s );
	}

	/**
	 * Sets up basic connection privileges for Communicator if necessary,
	 * and connects
	 * @param host Host to connect to.
	 * @param port Port number.
	 * @exception LDAPException from connect()
	 */
    protected void connect( LDAPConnection conn, String host, int port )
		throws LDAPException {
		boolean needsPrivileges = true;
		/* Running standalone? */
		SecurityManager sec = System.getSecurityManager();
		printDebug( "Security manager = " + sec );
		if ( sec == null ) {
			printDebug( "No security manager" );
			/* Not an applet, we can do what we want to */
			needsPrivileges = false;
		/* Can't do instanceof on an abstract class */
		} else if ( sec.toString().startsWith("java.lang.NullSecurityManager") ) {
			printDebug( "No security manager" );
			/* Not an applet, we can do what we want to */
			needsPrivileges = false;
		} else if ( sec.toString().startsWith(
			"netscape.security.AppletSecurity" ) ) {

			/* Connecting to the local host? */
			try {
				if ( host.equalsIgnoreCase(
					java.net.InetAddress.getLocalHost().getHostName() ) ) {
					needsPrivileges = false;
					printDebug( "Connecting to local host: " + host );
				}
			} catch ( java.net.UnknownHostException e ) {
				printDebug( "Cannot get local host name" );
			}
		}

		if ( needsPrivileges ) {
			/* Running as applet. Is PrivilegeManager around? */
			String mgr = "netscape.security.PrivilegeManager";
			try {
				Class c = Class.forName( mgr );
				java.lang.reflect.Method[] m = c.getMethods();
				if ( m != null ) {
					for( int i = 0; i < m.length; i++ ) {
						if ( m[i].getName().equals( "enablePrivilege" ) ) {
							try {
								Object[] args = new Object[1];
								args[0] = new String( "UniversalConnect" );
								m[i].invoke( null, args );
								printDebug( "UniversalConnect enabled" );
							} catch ( Exception e ) {
								printDebug( "Exception on invoking " +
											"enablePrivilege: " +
											e.toString() );
								break;
							}
							break;
						}
					}
				}
			} catch ( ClassNotFoundException e ) {
				printDebug( "no " + mgr );
			}
		}

		conn.connect( host, port );
		printDebug( "Connected to " + host );
	}


    public static void main( String[] args ) {
    }

    // member variables
	private LDAPSearchResults _results = null;
	private boolean _debug = false;
	private boolean _buffered = true;
	private int _errCode = 0;
    private String _host = new String("localhost");
    private int _port = 389;
	private int _scope = LDAPConnection.SCOPE_SUB;
    private String _base = new String("");
	private String _filter = new String("");
	private String _authDN = new String("");
    private String _authRDN = new String("");
    private String _authPassword = new String("");
	private String[] _attrs = null;
	private String[] _colNames = null;
    private int[] _colWidths = null;
	private LDAPConnection _ldc = null;
    private static final int DEFAULT_COL_WIDTH = 30*8;
	public static final int OK = 0;
	public static final int INVALID_PARAMETER = 1;
	public static final int CONNECT_ERROR = 2;
	public static final int AUTHENTICATION_ERROR = 3;
	public static final int PROPERTY_NOT_FOUND = 4;
	public static final int AMBIGUOUS_RESULTS = 5;
    //----------------------------------------------------------------------

}
