/*
 * LDAPBasePropertySupport.java    1.1
 *
 * Copyright (c) 1997 Netscape Communications Corporation
 *
 * Netscape grants you a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Netscape.
 *
 * This software is provided "AS IS," without a warranty of any kind.
 * See the CDK License Agreement for additional terms and conditions.
 */
package netscape.ldap.beans;

import netscape.ldap.*;
import java.util.StringTokenizer;

// This class has a bound property
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeSupport;

/**
 * This is a base class that is extended by various specialized LDAP
 * Beans. It provides the common properties and accessors used by
 * them.
 *
 *<pre>
 * History:
 *  v1.0: Rob Weltman
 *  v1.1: cdk_team
 *</pre>
 */
class LDAPBasePropertySupport {

	/**
	 * Constructor with no parameters
	 */
    public LDAPBasePropertySupport() {}

	/**
	 * Returns the host to search at.
	 * @return DNS name or dotted IP name of host to search at
	 */
    public String getHost() { 
		return _host; 
	}

	/**
	* Sets host string.
	* @param theHost host name
	*/
	public void setHost( String theHost ) {
	    _host = theHost;
	}

	/**
	 * Returns the port to search at.
	 * @return Port to search at
	 */
    public int getPort() { 
		return _port; 
	}

	/**
	* Sets port number.
	* @param thePort port
	*/
	public void setPort( int thePort ) {
		_port = thePort;
	}

	/**
	 * Returns the directory base to search at.
	 * @return directory base to search
	 */
    public String getBase() { 
		return _base; 
	}

	/**
	* Sets the starting base
	* @param theBase starting base
	*/
	public void setBase( String theBase ) {
		_base = theBase;
		if ( !_authRDN.equals( "" ) )
		    _authDN = _authRDN + "," + _base;
	}

	/**
	 * Returns the DN to authenticate as; null or empty for anonymous.
	 * @return DN to authenticate as
	 */
    public String getAuthDN() { 
		return _authDN; 
	}

	/**
	 * Sets the DN to authenticate as; null or empty for anonymous.
	 * @param authDN the DN to authenticate as
	 */
    public void setAuthDN( String authDN ) { 
		this._authDN =  authDN;
	}

	/**
	 * Returns the password for the DN to authenticate as
	 * @return Password of DN to authenticate as
	 */
    public String getAuthPassword() { 
		return _authPassword; 
	}

	/**
	 * Sets the password for the DN to authenticate as
	 * @param authPassword the password to use in authentication
	 */
    public void setAuthPassword( String authPassword ) { 
		this._authPassword = authPassword; 
	}

	/**
	 * Returns the RDN part of the DN to authenticate as
	 * @return RDN of DN to authenticate as
	 */
    public String getAuthRDN() { 
		return _authRDN; 
	}

	/**
	 * Sets the RDN of the DN to authenticate as. This assumes that
	 * the directory base has been set either in the constructor or
	 * with setBase. The RDN should be of the form "cn=Polly Plum".
	 * If the base was "ou=Accounting,o=Ace Industry,c=us", the DN
	 * will be "cn=PollyPlum, ou=Accounting,o=Ace Industry,c=us".
	 * @param rdn The RDN to use to compose a DN
	 */
    public void setAuthRDN( String rdn ) { 
	    _authRDN = rdn;
		if ( !_base.equals( "" ) )
		    _authDN = _authRDN + "," + _base;
	}

	/**
	 * Returns the cn of the DN to authenticate as
	 * @return cn of DN to authenticate as
	 */
    public String getUserName() { 
	    if ( !_authRDN.equals( "" ) ) {
		    StringTokenizer st = new StringTokenizer( _authRDN, "=" );
			if ( st.hasMoreTokens() )
			    return st.nextToken();
		}
		return new String("");
	}

	/**
	 * Sets the RDN of the DN to authenticate as by prepending
	 * "cn=" to the supplied user name. This assumes that
	 * the directory base has been set either in the constructor or
	 * with setBase. The name should be of the form "Polly Plum".
	 * @param name the value of the cn to be used as an RDN
	 */
    public void setUserName( String name ) { 
	    setAuthRDN( "cn=" + name );
	}

	/**
	 * Sets the RDN of the DN to authenticate as by prepending
	 * "uid=" to the supplied user name. This assumes that
	 * the directory base has been set either in the constructor or
	 * with setBase. The name should be of the form "prose".
	 * @param name the value of the id to be used as an RDN
	 */
    public void setUserID( String name ) { 
	    setAuthRDN( "uid=" + name );
	}

	/**
	 * Set the search scope using a string
	 * @param scope either "SUB", "ONE", or "BASE"
	 */
    public void setScope( String scope ) {
		if ( scope.equalsIgnoreCase("SUB") )
			setScope( LDAPConnection.SCOPE_SUB );
		else if ( scope.equalsIgnoreCase("ONE") )
			setScope( LDAPConnection.SCOPE_ONE );
		else if ( scope.equalsIgnoreCase("BASE") )
			setScope( LDAPConnection.SCOPE_BASE );
	}

	/**
	 * Get the current search scope
	 * @return the current search scope as integer
	 */
    public int getScope() {
		return _scope;
	}

	/**
	 * Set the search scope using an integer
	 * @param scope one of LDAPConnection.SCOPE_BASE,
	 * LDAPConnection.SCOPE_SUB, LDAPConnection.SCOPE_ONE
	 */
    public void setScope( int scope ) {
		_scope = scope;
	}

	/**
	 * Returns the search filter
	 * @return search filter
	 */
    public String getFilter() { 
		return _filter; 
	}

	/**
	 * Sets the search filter
	 * @param filter search filter
	 */
    public void setFilter( String filter ) {
		_filter = filter;
	}

	/**
	 * Returns true if debug output is on
	 * @return true if debug output is on
	 */
    public boolean getDebug() { 
		return _debug; 
	}

	/**
	 * Turns debug output on or off
	 * @param on true for debug output
	 */
    public void setDebug( boolean on ) { 
	    _debug = on;
	}

	/**
	 * Returns the latest error code
	 * @return The latest error code
	 */
    public int getErrorCode() { 
		return _errCode; 
	}

	/**
	 * Sets an error code for retrieval by a client
	 * @param code An error code
	 */
    protected void setErrorCode( int code ) { 
	    _errCode = code;
	}

    /**
	 * Add a client to be notified when an authentication result is in
	 * @param listener a client to be notified of changes
	 */
    public void addPropertyChangeListener( PropertyChangeListener listener ) {
	    System.out.println( "Adding listener " + listener );
	    m_propSupport.addPropertyChangeListener( listener );
	}

    /**
	 * Remove a client which had requested notification on authentication
	 * @param listener a client to not be notified of changes
	 */
    public void removePropertyChangeListener(
							  PropertyChangeListener listener ) {
	    m_propSupport.removePropertyChangeListener( listener );
	}

    /**
	 * Support for bound property notification
	 * @param propName Name of changed property
	 * @param oldValue Previous value of property
	 * @param newValue New value of property
	 */
    public void firePropertyChange(
							  String propName,
							  Object oldValue,
							  Object newValue ) {
	    m_propSupport.firePropertyChange( propName, oldValue, newValue );
	}

    protected void printDebug( String s ) {
		if ( _debug )
			System.out.println( s );
	}

	/**
	 * Sets up basic connection privileges for Communicator if necessary,
	 * and connects
	 * @param host Host to connect to.
	 * @param port Port number.
	 * @exception LDAPException from connect()
	 */
    protected void connect( LDAPConnection conn, String host, int port )
		throws LDAPException {
		boolean needsPrivileges = true;
		/* Running standalone? */
		SecurityManager sec = System.getSecurityManager();
		printDebug( "Security manager = " + sec );
		if ( sec == null ) {
			printDebug( "No security manager" );
			/* Not an applet, we can do what we want to */
			needsPrivileges = false;
		/* Can't do instanceof on an abstract class */
		} else if ( sec.toString().startsWith("java.lang.NullSecurityManager") ) {
			printDebug( "No security manager" );
			/* Not an applet, we can do what we want to */
			needsPrivileges = false;
		} else if ( sec.toString().startsWith(
			"netscape.security.AppletSecurity" ) ) {

			/* Connecting to the local host? */
			try {
				if ( host.equalsIgnoreCase(
					java.net.InetAddress.getLocalHost().getHostName() ) )
					needsPrivileges = false;
			} catch ( java.net.UnknownHostException e ) {
			}
		}

		if ( needsPrivileges ) {
			/* Running as applet. Is PrivilegeManager around? */
			String mgr = "netscape.security.PrivilegeManager";
			try {
				Class c = Class.forName( mgr );
				java.lang.reflect.Method[] m = c.getMethods();
				if ( m != null ) {
					for( int i = 0; i < m.length; i++ ) {
						if ( m[i].getName().equals( "enablePrivilege" ) ) {
							try {
								Object[] args = new Object[1];
								args[0] = new String( "UniversalConnect" );
								m[i].invoke( null, args );
								printDebug( "UniversalConnect enabled" );
							} catch ( Exception e ) {
								printDebug( "Exception on invoking " +
											"enablePrivilege: " +
											e.toString() );
								break;
							}
							break;
						}
					}
				}
			} catch ( ClassNotFoundException e ) {
				printDebug( "no " + mgr );
			}
		}

		conn.connect( host, port );
	}

    /*
	 * Variables
	 */
	private boolean _debug = false;
	private int _errCode = 0;
    private String _host = new String("localhost");
    private int _port = 389;
	private int _scope = LDAPConnection.SCOPE_SUB;
    private String _base = new String("");
	private String _filter = new String("");
	private String _authDN = new String("");
    private String _authRDN = new String("");
    private String _authPassword = new String("");
    private PropertyChangeSupport m_propSupport =
	            new PropertyChangeSupport( this );
}
