;;; hycontrol.el --- Interactive sizing, moving, replicating and deleting of windows and frames  -*- lexical-binding: t; -*-
;;
;; Author:       Bob Weiner
;;
;; Orig-Date:     1-Jun-16 at 15:35:36
;; Last-Mod:     18-Feb-24 at 12:42:03 by Mats Lidell
;;
;; SPDX-License-Identifier: GPL-3.0-or-later
;;
;; Copyright (C) 2016-2024  Free Software Foundation, Inc.
;; See the "HY-COPY" file for license information.
;;
;; This file is part of GNU Hyperbole.

;;; Commentary:
;;
;;   This library provides full interactive control of window and
;;   frame sizes and locations utilizing quick single key commands.
;;   It has the ability to change from increasing a window height by
;;   5 lines, {.5 h}, to moving a frame 82 pixels, {.82 right-arrow},
;;   with just a few keystrokes (the leading . just resets the numeric
;;   argument to 0 prior to typing the new number).
;;
;;   See the Info manual entry "(hyperbole)HyControl" for reference
;;   documentation.  See below for key usage information.
;;
;;   ----
;;
;;   HyControl is invoked via either of two global minor modes under
;;   the Hyperbole screen menu, both of which can toggle to the other
;;   by pressing {t}.  `hycontrol-enable-frames-mode' bound to {C-h h s
;;   f} manages visible frame creation, deletion, sizing, position and
;;   face zooming (enlarging and shrinking); if called interactively,
;;   it stores the current frame configuration for restoration via a
;;   press of the `)' key.  `hycontrol-enable-windows-mode' manages
;;   per frame window creation, deletion, sizing, reframing and face
;;   zooming; if called interactively, it stores the current window
;;   configuration for restoration via a press of the `)' key.
;;   `hycontrol-enable-windows-mode' is typically bound by Hyperbole
;;   to {C-c \ } or just use {C-h h s w}.  Then press {t} if you want
;;   to switch to frame control.
;;
;;   With a HyControl minor mode active, a multi-line help summary of
;;   most available key bindings is shown in the minibuffer.  Simply
;;   read this and try each command out to get a feel for it.  Below
;;   we highlight some of the most unique commands.
;;
;;   ----
;;
;;   In either HyControl mode, you can instantly create a grid of
;;   windows to display many buffers by choosing a number of rows as
;;   your first digit, then a number of columns of windows as the
;;   second digit and then pressing {@}, e.g. {.26 @} produces 2 rows,
;;   each with 6 columns of windows in the selected frame.  Grids can
;;   be from 1x1 to 9x9 windows.  This command also works outside of a
;;   HyControl mode when in Dired, Buffer Menu or IBuffer modes with
;;   a prefix argument (no preceding period).
;;
;;   The buffers displayed by the {@} command are chosen smartly.
;;   With a current buffer in Dired, Buffer Menu or IBuffer mode with
;;   marked items, the buffers associated with those items are
;;   displayed first.  Then the most recently used buffers are
;;   displayed in each window, first selecting from buffers which
;;   match any of the predicate expressions in
;;   `hycontrol-display-buffer-predicate-list'.  Then, if there are
;;   not enough buffers for all windows, the buffers that failed to
;;   match to any predicate are used.  The default predicate list
;;   chooses buffers with attached files.  In all cases, buffers whose
;;   names start with a space are filtered out.  If a prefix argument
;;   of 0 is given, a major mode symbol is prompted for and buffers
;;   with that major mode are preferred for display instead of those
;;   matching the predicate list.
;;
;;   ----
;;
;;   HyControl allows placement of frames at screen edges and corners
;;   using the keys of the numeric keypad, matching their physical
;;   layout, e.g. {3} moves to the lower right corner.  Press {p} for
;;   a prompt with a virtual numeric keypad if you lack a physical one.
;;   You can also cycle through all of these placement positions with
;;   the {c} key.
;;
;;   HyControl can rapidly resize frames to common percentages of
;;   screen sizes via a number of commands.  Each press of {a} or {A}
;;   cycles through resizing the selected frame's width and height
;;   respectively to a percentage of the screen given by the lists,
;;   `hycontrol-frame-widths' and `hycontrol-frame-heights', e.g. 25%,
;;   50%, etc.  The keys: {i} top, {j} left, {k} right, and {m}
;;   bottom, first maximize a frame to the respective screen edge and
;;   then with successive presses, shrink the frame dimension
;;   perpendicular to that edge by 50% while keeping the original edge
;;   fixed in place.  Try them and you will quickly see how they can
;;   help.
;;   
;;   ----
;;
;;   When HyControl creates a new frame, it automatically sizes it to the
;;   same size as the previously selected frame and offsets it from that
;;   frame by the (X . Y) number of pixels given in the variable,
;;   `hycontrol-frame-offset'.
;;
;;   A display screen may span multiple physical monitors.  To prevent
;;   widgets and toolbars at the corners of the screen from being
;;   obscured, HyControl can offset each frame from each screen edge
;;   by a fixed number of pixels.  These offsets are specified by the
;;   variable, `hycontrol-screen-offset-alist' and can differ for each
;;   type of screen; see its documentation for details.  If you change
;;   its value, then call `hycontrol-set-screen-offsets' to set any
;;   new offset values.  `hycontrol-get-screen-offsets' returns the
;;   list of offsets in clockwise order starting from the top edge.
;;
;;   ----
;;
;;   Please note that the frame zoom in/out commands on Z and z will
;;   not work unless you have the separately available "zoom-frm.el"
;;   library (which itself requires another library).  If not available,
;;   this command will just beep at you.  The window-based zoom commands
;;   utilize a built-in Emacs library, so they will always work under
;;   any window system.  These commands enlarge and shrink the default
;;   text face.

;;; Code:
;;; ************************************************************************
;;; Other required Elisp libraries
;;; ************************************************************************

(require 'cl-lib)
(require 'hhist)     ; To store frame-config when hycontrol-windows-grid is used
(require 'hypb)
(require 'set)
(eval-and-compile
  (require 'framemove nil t) ;; Elpa package
  (require 'windmove))
;; Frame face enlarging/shrinking (zooming) requires this separately available library.
;; Everything else works fine without it, so don't make it a required dependency.
(require 'zoom-frm nil t)

;;; ************************************************************************
;;; Public declarations
;;; ************************************************************************

(declare-function fm-next-frame "ext:framemove")

(defvar frame-zoom-font-difference)

(defvar hyperbole-mode-map)             ; "hyperbole.el"
(defvar org-mode-map)                   ; "org.el"
(defvar outline-mode-map)               ; "outline.el"
(defvar outline-minor-mode-map)         ; "outline.el"

(declare-function kbd-key:key-series-to-events "hib-kbd")

;;; ************************************************************************
;;; Public variables
;;; ************************************************************************

(defvar hycontrol-debug nil
  "Whether some HyControl functions log debugging messages to *Messages*.")

(defvar hycontrol-display-buffer-predicate-list
  ;; Display only buffers attached to files.
  (list #'buffer-file-name)
  "List of single buffer/name predicates.
If any predicate returns non-nil for a buffer, include that buffer in
the list to display in the windows created by `hycontrol-windows-grid'.

A predicate may be either a function that takes a single buffer
argument or a boolean expression, in which case the expression is
evaluated with the buffer argument as the current buffer, e.g. (eq
major-mode \\='c-mode).")

(defcustom hycontrol-help-flag t
  "*Non-nil means show key binding help in the minibuffer when in a HyControl mode."
  :type 'boolean
  :group 'hyperbole-screen)

(defcustom hycontrol-invert-mode-line-flag t
  "*Non-nil means invert mode-line to emphasize the special key bindings in effect."
  :type 'boolean
  :group 'hyperbole-screen)

(defcustom hycontrol-keep-window-flag nil
  "*Non-nil means leave original window when tear off window to another frame."
  :type 'boolean
  :group 'hyperbole-screen)

(defcustom hycontrol-maximum-units 1000
  "*Maximum units setting allowed for hycontrol commands.
The unit counter resets to the last digit entered whenever this
value is exceeded."
  :type '(integer :match (lambda (_widget value)
			   (and (integerp value) (> value 0)
				(<= value (max 1000 (display-pixel-width))))))
  :group 'hyperbole-screen)

(defcustom hycontrol-frame-offset '(13 . 23)
  "*Increase in offset for new hycontrol frames relative to the selected frame.
Its value is an (x-offset . y-offset) pair in pixels."
  :type '(cons integer integer)
  :group 'hyperbole-screen)

(defvar hycontrol-screen-offset-alist
  '(((1920 . 1080) . (0 10 0 68)) ; 24" iMac HD display
    ((2560 . 1440) . (0 15 0 93)) ; 27" iMac HD display
    (t . (0 0 0 0)))
  "*Alist of screen predicate offset pairs.
The format is (screen-predicate . (top-offset right-offset
bottom-offset left-offset).
Offsets are integers given in pixels.  The offsets associated with the first
matching screen-predicate are used in HyControl screen edge frame placement
commands; this is set when HyControl is first loaded/used.

Screen-predicate must be one of: a boolean function of no arguments, an
integer dotted pair of (width . height) in pixels to match to, or an Emacs
Lisp boolean form to evaluate.

The final predicate should always be t, for default values, typically of zero.")

(defcustom hycontrol-screen-top-offset    0
  "*Pixel offset from top used when placing a frame at a top corner."
  :type '(integer :match (lambda (_widget value)
			   (and (integerp value) (>= value 0)
				(< value (display-pixel-height)))))
  :group 'hyperbole-screen)
(defcustom hycontrol-screen-right-offset  0
  "*Pixel offset from right used when placing a frame at a right corner."
  :type '(integer :match (lambda (_widget value)
			   (and (integerp value) (>= value 0)
				(< value (display-pixel-width)))))
  :group 'hyperbole-screen)
(defcustom hycontrol-screen-bottom-offset 0
  "*Pixel offset from bottom used when placing a frame at a bottom corner."
  :type '(integer :match (lambda (_widget value)
			   (and (integerp value) (>= value 0)
				(< value (display-pixel-height)))))
  :group 'hyperbole-screen)
(defcustom hycontrol-screen-left-offset   0
  "*Pixel offset from left used when placing a frame at a left corner."
  :type '(integer :match (lambda (_widget value)
			   (and (integerp value) (>= value 0)
				(< value (display-pixel-width)))))
  :group 'hyperbole-screen)

(defvar hycontrol-frame-widths
  '(1.0 0.75 0.666 0.5 0.333 0.25)
    "List of width percentages to cycle through when adjusting a frame's width.
0.75 and 75 are treated as the same percentage.")

(defvar hycontrol-frame-heights
  '(1.0 0.75 0.666 0.5 0.333 0.25)
  "List of height percentages to cycle through when adjusting a frame's height.
0.75 and 75 are treated as the same percentage.")

(defvar hycontrol-arg nil
  "HyControl copy of `prefix-arg' that it changes within key bindings.
`pre-command-hook' synchronizes this value to `prefix-arg'.")

;;; ************************************************************************
;;; Public declarations
;;; ************************************************************************
(defvar ibuffer-mode-map)

;;; Frame Keys
(defvar hycontrol-frames-mode-map
  (let ((map (make-sparse-keymap)))
    (suppress-keymap map t) ;; Disable self-inserting keys and prefix keys

    (define-key map [up]    (lambda () (interactive) (hycontrol-move-frame 'up hycontrol-arg)))
    (define-key map [down]  (lambda () (interactive) (hycontrol-move-frame 'down hycontrol-arg)))
    (define-key map [left]  (lambda () (interactive) (hycontrol-move-frame 'left hycontrol-arg)))
    (define-key map [right] (lambda () (interactive) (hycontrol-move-frame 'right hycontrol-arg)))

    (define-key map [kp-0]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-0 hycontrol-arg)))
    (define-key map [kp-1]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-1 hycontrol-arg)))
    (define-key map [kp-2]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-2 hycontrol-arg)))
    (define-key map [kp-3]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-3 hycontrol-arg)))
    (define-key map [kp-4]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-4 hycontrol-arg)))
    (define-key map [kp-5]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-5 hycontrol-arg)))
    (define-key map [kp-6]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-6 hycontrol-arg)))
    (define-key map [kp-7]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-7 hycontrol-arg)))
    (define-key map [kp-8]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-8 hycontrol-arg)))
    (define-key map [kp-9]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-9 hycontrol-arg)))

    ;; Clear hycontrol-arg
    (define-key map "."     'hycontrol-reset-prefix-arg)
    (define-key map "@"     'hycontrol-windows-grid)
    (define-key map "?"     'hycontrol-toggle-help)
    (define-key map "a"     'hycontrol-frame-adjust-widths)
    (define-key map "A"     'hycontrol-frame-adjust-heights)
    (define-key map "b"     'bury-buffer)
    (define-key map "c"     'hycontrol-frame-to-screen-edges)
    (define-key map "d"     'delete-frame)
    (define-key map "D"     'hycontrol-delete-other-frames)
    (define-key map "f"     'hycontrol-clone-window-to-new-frame)
    (define-key map "F"     'hycontrol-window-to-new-frame)
    (define-key map "\C-g"  'hycontrol-abort)
    (define-key map "%"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-percentage-of-screen hycontrol-arg))))
    (define-key map "H"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-height-percentage-of-screen hycontrol-arg))))
    (define-key map "W"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-width-percentage-of-screen hycontrol-arg))))
    (define-key map "h"     (lambda () (interactive) (hycontrol-set-frame-height nil (+ (frame-height) hycontrol-arg))))

    ;; Move directionally among frames
    (define-key map "I"     'hycontrol-framemove-up)
    (define-key map "J"     'hycontrol-framemove-left)
    (define-key map "K"     'hycontrol-framemove-right)
    (define-key map "M"     'hycontrol-framemove-down)

    (define-key map "i"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-resize-to-top hycontrol-arg))))
    (define-key map "j"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-resize-to-left hycontrol-arg))))
    (define-key map "k"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-resize-to-right hycontrol-arg))))
    (define-key map "l"     'lower-frame)
    (define-key map "m"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-resize-to-bottom hycontrol-arg))))
    (define-key map "n"     (lambda () (interactive) (hycontrol-set-frame-width nil (- (frame-width) hycontrol-arg))))
    (define-key map "o"     (lambda () (interactive) (let ((w (selected-window))) (other-window hycontrol-arg) (if (eq w (selected-window)) (other-window 1)))))
    (define-key map "O"     (lambda () (interactive) (let ((w (selected-window))) (other-frame hycontrol-arg) (if (eq w (selected-window)) (other-frame 1)))))
    ;; Numeric keypad emulation for keyboards that lack one.
    (define-key map "p"     (lambda () (interactive) (hycontrol-virtual-numeric-keypad hycontrol-arg)))
    (define-key map "q"     'hycontrol-quit)
    (define-key map "Q"     'hycontrol-quit)
    (define-key map "r"     'raise-frame)

    (define-key map "s"     (lambda () (interactive) (hycontrol-set-frame-height nil (- (frame-height) hycontrol-arg))))
    (define-key map "t"     'hycontrol-enable-windows-mode)
    (define-key map "u"     'unbury-buffer)
    (define-key map "w"     (lambda () (interactive) (hycontrol-set-frame-width nil (+ (frame-width) hycontrol-arg))))
    (define-key map "Z"     (lambda () (interactive) (if (> hycontrol-arg 9) (setq hycontrol-arg 1)) (hycontrol-frame-zoom 'zoom-frm-in hycontrol-arg hycontrol-debug)))
    (define-key map "z"     (lambda () (interactive) (if (> hycontrol-arg 9) (setq hycontrol-arg 1)) (hycontrol-frame-zoom 'zoom-frm-out hycontrol-arg hycontrol-debug)))
    (define-key map "\["    'hycontrol-make-frame)
    (define-key map "\]"    'hycontrol-make-frame)
    (define-key map "\("    'hycontrol-save-frame-configuration)
    (define-key map "\)"    'hycontrol-restore-frame-configuration)
    ;; Something in this command's event handling when used within HyControl's event loop slows down
    ;; frame iconification under macOS 100-fold, so don't enable it until this issue is resolved.
    ;; (define-key map "^"    'iconify-frame)
    (define-key map "~"     (lambda () (interactive)
			      (unless (hycontrol-frame-swap-buffers)
				(hycontrol-user-error hycontrol-debug "(HyControl): There must be at least two frames and the frame to swap to must have only a single window."))))
    (define-key map "-"     'hycontrol-minus-key)
    (define-key map "+"     'toggle-frame-maximized)
    (define-key map "="     (lambda () (interactive)
			      (and (> (length (visible-frame-list)) 1)
				   (y-or-n-p "Resize all other frames to the size of the selected frame?")
				   (mapc (lambda (f)
					   (hycontrol-set-frame-size
					    f (frame-pixel-width) (frame-pixel-height) t))
					 (visible-frame-list)))))

    (define-key map "\C-u"  'hycontrol-multiply-universal-arg)
    (define-key map "0"     (lambda () (interactive) (hycontrol-universal-arg-digit 0)))
    (define-key map "1"     (lambda () (interactive) (hycontrol-universal-arg-digit 1)))
    (define-key map "2"     (lambda () (interactive) (hycontrol-universal-arg-digit 2)))
    (define-key map "3"     (lambda () (interactive) (hycontrol-universal-arg-digit 3)))
    (define-key map "4"     (lambda () (interactive) (hycontrol-universal-arg-digit 4)))
    (define-key map "5"     (lambda () (interactive) (hycontrol-universal-arg-digit 5)))
    (define-key map "6"     (lambda () (interactive) (hycontrol-universal-arg-digit 6)))
    (define-key map "7"     (lambda () (interactive) (hycontrol-universal-arg-digit 7)))
    (define-key map "8"     (lambda () (interactive) (hycontrol-universal-arg-digit 8)))
    (define-key map "9"     (lambda () (interactive) (hycontrol-universal-arg-digit 9)))
    map)
  "Keymap to use when in Hyperbole HyControl frames mode.")

;;; Window Keys

;;;###autoload
(eval-after-load "buff-menu" '(define-key Buffer-menu-mode-map "@" 'hycontrol-windows-grid))
;;;###autoload
(eval-after-load "ibuffer"   '(define-key ibuffer-mode-map     "@" 'hycontrol-windows-grid))
;;;###autoload
(eval-after-load "dired"     '(define-key dired-mode-map       "@" 'hycontrol-windows-grid))


(defvar hycontrol-windows-mode-map
  (let ((map (make-sparse-keymap)))
    (suppress-keymap map t) ;; Disable self-inserting keys and prefix keys

    (define-key map [up]    (lambda () (interactive) (hycontrol-move-frame 'up hycontrol-arg)))
    (define-key map [down]  (lambda () (interactive) (hycontrol-move-frame 'down hycontrol-arg)))
    (define-key map [left]  (lambda () (interactive) (hycontrol-move-frame 'left hycontrol-arg)))
    (define-key map [right] (lambda () (interactive) (hycontrol-move-frame 'right hycontrol-arg)))

    (define-key map [kp-0]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-0 hycontrol-arg)))
    (define-key map [kp-1]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-1 hycontrol-arg)))
    (define-key map [kp-2]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-2 hycontrol-arg)))
    (define-key map [kp-3]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-3 hycontrol-arg)))
    (define-key map [kp-4]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-4 hycontrol-arg)))
    (define-key map [kp-5]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-5 hycontrol-arg)))
    (define-key map [kp-6]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-6 hycontrol-arg)))
    (define-key map [kp-7]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-7 hycontrol-arg)))
    (define-key map [kp-8]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-8 hycontrol-arg)))
    (define-key map [kp-9]  (lambda () (interactive) (hycontrol-numeric-keypad 'kp-9 hycontrol-arg)))

    ;; Clear hycontrol-arg
    (define-key map "."     'hycontrol-reset-prefix-arg)
    (define-key map "@"     'hycontrol-windows-grid)
    (define-key map "?"     'hycontrol-toggle-help)
    (define-key map "a"     'hycontrol-frame-adjust-widths)
    (define-key map "A"     'hycontrol-frame-adjust-heights)
    (define-key map "b"     'bury-buffer)
    (define-key map "c"     'hycontrol-frame-to-screen-edges)
    (define-key map "d"     'delete-window)
    (define-key map "D"     'hycontrol-delete-other-windows)
    (define-key map "f"     'hycontrol-clone-window-to-new-frame)
    (define-key map "F"     'hycontrol-window-to-new-frame)
    (define-key map "\C-g"  'hycontrol-abort)
    (define-key map "h"     (lambda () (interactive) (enlarge-window hycontrol-arg)))

    ;; Move directionally among windows within current frame
    (define-key map "I"     'windmove-up)
    (define-key map "J"     'windmove-left)
    (define-key map "K"     'windmove-right)
    (define-key map "M"     'windmove-down)

    ;; Allow frame resizing even when in window control mode because
    ;; it may be used often.
    (define-key map "i"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-resize-to-top hycontrol-arg))))
    (define-key map "j"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-resize-to-left hycontrol-arg))))
    (define-key map "k"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-resize-to-right hycontrol-arg))))
    (define-key map "m"     (lambda () (interactive) (setq hycontrol-arg (hycontrol-frame-resize-to-bottom hycontrol-arg))))
    (define-key map "n"     (lambda () (interactive) (shrink-window-horizontally hycontrol-arg)))

    (define-key map "o"     (lambda () (interactive) (let ((w (selected-window))) (other-window hycontrol-arg) (if (eq w (selected-window)) (other-window 1)))))
    (define-key map "O"     (lambda () (interactive) (let ((w (selected-window))) (other-frame hycontrol-arg) (if (eq w (selected-window)) (other-frame 1)))))
    ;; Numeric keypad emulation for keyboards that lack one.
    (define-key map "p"     (lambda () (interactive) (hycontrol-virtual-numeric-keypad hycontrol-arg)))
    (define-key map "q"     'hycontrol-quit)
    (define-key map "Q"     'hycontrol-quit)
    (define-key map "s"     (lambda () (interactive) (shrink-window hycontrol-arg)))
    (define-key map "t"     'hycontrol-enable-frames-mode)
    (define-key map "u"     'unbury-buffer)
    (define-key map "w"     (lambda () (interactive) (enlarge-window-horizontally hycontrol-arg)))
    (define-key map "Z"     (lambda () (interactive) (if (fboundp 'text-scale-increase)
							 ;; Emacs autoloaded function
							 (text-scale-increase (if (< hycontrol-arg 10) hycontrol-arg (setq hycontrol-arg 1))))))
    (define-key map "z"     (lambda () (interactive) (if (fboundp 'text-scale-decrease)
							 ;; Emacs autoloaded function
							 (text-scale-decrease (if (< hycontrol-arg 10) hycontrol-arg (setq hycontrol-arg 1))))))

    ;; Don't call these interactively because a prefix arg of 1 tries
    ;; to make one window 1 line tall.
    (define-key map "["    (lambda () (interactive) (split-window-vertically)))
    (define-key map "]"    (lambda () (interactive) (split-window-horizontally)))

    (define-key map "("    'hycontrol-save-frame-configuration)
    (define-key map ")"    'hycontrol-restore-frame-configuration)

    (define-key map "~"     (lambda () (interactive)
			      (unless (hycontrol-window-swap-buffers)
				(hycontrol-user-error hycontrol-debug "(HyControl): There must be precisely two windows within the selected frame to swap buffers."))))
    (define-key map "-"     'hycontrol-minus-key)
    (define-key map "+"     'hycontrol-window-maximize-lines)
    (define-key map "="     (lambda () (interactive) (and (> (length (window-list)) 1)
							  (y-or-n-p "Resize windows evenly across this frame?")
							  (balance-windows))))

    (define-key map "\C-u"  'hycontrol-multiply-universal-arg)
    (define-key map "0"     (lambda () (interactive) (hycontrol-universal-arg-digit 0)))
    (define-key map "1"     (lambda () (interactive) (hycontrol-universal-arg-digit 1)))
    (define-key map "2"     (lambda () (interactive) (hycontrol-universal-arg-digit 2)))
    (define-key map "3"     (lambda () (interactive) (hycontrol-universal-arg-digit 3)))
    (define-key map "4"     (lambda () (interactive) (hycontrol-universal-arg-digit 4)))
    (define-key map "5"     (lambda () (interactive) (hycontrol-universal-arg-digit 5)))
    (define-key map "6"     (lambda () (interactive) (hycontrol-universal-arg-digit 6)))
    (define-key map "7"     (lambda () (interactive) (hycontrol-universal-arg-digit 7)))
    (define-key map "8"     (lambda () (interactive) (hycontrol-universal-arg-digit 8)))
    (define-key map "9"     (lambda () (interactive) (hycontrol-universal-arg-digit 9)))
    map)
  "Keymap to use when in Hyperbole HyControl window mode.")

;;; ************************************************************************
;;; Public declarations
;;; ************************************************************************
(declare-function ibuffer-get-marked-buffers "ibuffer")
(declare-function which-key--hide-popup-ignore-command "ext:which-key")

;;; ************************************************************************
;;; Private variables
;;; ************************************************************************

(defvar hycontrol--frames-prompt-format
 (concat "FRAMES: (h=heighten, s=shorten, w=widen, n=narrow, %%/H/W=screen %%age, arrow=move frame) by %d unit%s, .=clear units\n"
	 ;; d/^/D=delete/iconify frame/others - iconify left out due to some bug on macOS (see comment near ^ below)
	 "a/A=cycle adjust width/height, d/D=delete frame/others, o/O=other win/frame, I/J/K/M=to frame, [/]=create frame, (/)=save/restore fconfig\n"
	 "@=grid of wins, f/F=clone/move win to new frame, -/+=minimize/maximize frame, ==frames same size, u/b/~=un/bury/swap bufs\n"
	 "Frame to edges: c=cycle, i/j/k/m=expand/contract, p/num-keypad=move; z/Z=zoom out/in, t=to WINDOWS mode, Q=quit")
 "HyControl frames-mode minibuffer prompt string to pass to format.
Format it with 2 arguments: `prefix-arg' and a plural string indicating if
`prefix-arg' is not equal to 1.")

(defvar hycontrol--windows-prompt-format
  (concat
   "WINDOWS: (h=heighten, s=shorten, w=widen, n=narrow, arrow=move frame) by %d unit%s, .=clear units\n"
   "a/A=cycle adjust frame width/height, d/D=delete win/others, o/O=other win/frame, I/J/K/M=to window, [/]=split win atop/sideways, (/)=save/restore wconfig\n"
   "@=grid of wins, f/F=clone/move win to new frame, -/+=minimize/maximize win, ==wins same size, u/b/~=un/bury/swap bufs\n"
   "Frame to edges: c=cycle, i/j/k/m=expand/contract, p/num-keypad=move; z/Z=zoom out/in, t=to FRAMES mode, Q=quit")
  "HyControl windows-mode minibuffer prompt string to pass to format.
Format it with 2 arguments: `prefix-arg' and a plural string indicating if
`prefix-arg' is not equal to 1.")

(defvar hycontrol--prompt-format nil
  "The current HyControl mode help format string or nil if not active.")

(defvar hycontrol--exit-status nil
  "Internal HyControl status indicator of how it was exited.
After exit, it should be one of the following symbols triggered by the
associated key: quit {q}, abort {\\`C-g'}, or toggle {t}.")


(defvar hycontrol--fconfig nil
  "Used to store a frame configuration while in hycontrol.")

(defvar hycontrol--wconfig nil
  "Used to store a window configuration while in hycontrol.")


(defvar hycontrol--invert-display-buffer-predicates nil)


(defvar hycontrol--quit-function nil
  "Store function to remove the transient-map later.
The function is auto-generated by a call to `set-transient-map'")


(defvar hycontrol--screen-edge-position 0
  "Cycles between 0-7 representing corner and center edge positions.
The cycle is in clockwise order from the upper left corner.")

(defvar hycontrol--frame-widths-pointer nil)
(defvar hycontrol--frame-heights-pointer nil)
(defvar hycontrol--buffer-list nil)
(defvar hycontrol--buffer-list-pointer nil)

(defvar hycontrol--initial-which-key-inhibit nil
  "Store value of `which-key-inhibit' flag from \"which-key\" package, if any.
Used on entry to HyControl.")

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(defsubst hycontrol-windows-grid-valid-size-p (grid-size)
  (when grid-size
    (and (setq grid-size (prefix-numeric-value grid-size))
	 (integerp grid-size) (>= grid-size 11) (<= grid-size 99)
	 (not (zerop (% grid-size 10))))))

;;; HyControl private per keyboard key functions

(defun hycontrol-pre-command-hook ()
  "Added to `pre-command-hook' while in any HyControl mode."
  (when (null hycontrol-arg) (setq hycontrol-arg 1))
  (setq prefix-arg hycontrol-arg))

(defun hycontrol-post-command-hook ()
  "Added to `post-command-hook' while in HyControl frames mode."
  (when (null hycontrol-arg) (setq hycontrol-arg 1))
  (if (zerop (minibuffer-depth))
      (if hycontrol-help-flag
	  (let ((message-log-max nil) ; prevent logging of HyControl help messages
		(resize-mini-windows t)	 ; automatically shrink
		(use-dialog-box))	 ; prevent y-or-n dialog boxes
	    (message hycontrol--prompt-format hycontrol-arg (if (= hycontrol-arg 1) "" "s"))))
    ;; Quit from HyControl at any minibuffer prompt so that
    ;; self-insert-keys work for typing at the prompt.
    (hycontrol-disable-modes)))

(defun hycontrol-end-mode ()
  "Prepare to abort or quit from HyControl."
  (interactive)
  (remove-hook 'pre-command-hook  'hycontrol-pre-command-hook)
  (remove-hook 'post-command-hook 'hycontrol-post-command-hook)
  (if (boundp 'which-key-inhibit)
      (setq which-key-inhibit hycontrol--initial-which-key-inhibit))
  ;; May be called when turning on HyControl before this next function
  ;; is defined.
  (if (functionp hycontrol--quit-function)
      (funcall hycontrol--quit-function))
  (setq inhibit-quit nil
 	hycontrol--exit-status t
	hycontrol-arg 1
	prefix-arg nil
	hycontrol-debug nil
	hycontrol-frames-mode nil
	hycontrol-windows-mode nil
	hycontrol--prompt-format nil))

(defun hycontrol-stay-in-mode ()
  "Return non-nil if HyControl mode should remain active."
  (null hycontrol--exit-status))


(defun hycontrol-invert-prefix-arg ()
  "Invert the sign of the current Hycontrol prefix argument.
If the argument is 0, set it to -1."
  (interactive)
  (setq this-command 'hycontrol-invert-prefix-arg
	hycontrol-arg (if (zerop hycontrol-arg) -1 (- hycontrol-arg))))

(defun hycontrol-minus-key ()
  "Conditional command to execute when the minus key is pressed."
  (interactive)
  (cond ((and (symbolp last-command)
	      (string-match-p "\\`\\(hui:menu-enter\\|hycontrol.*-\\(frames\\|windows\\|prefix\\|universal\\)-\\(mode\\|arg\\)\\)" (symbol-name last-command)))
	 (hycontrol-invert-prefix-arg))
	(hycontrol-frames-mode
	 (hycontrol-frame-minimize-lines))
	(t
	 (hycontrol-window-minimize-lines))))

(defun hycontrol-multiply-universal-arg ()
  "Return prefix arg based on `hycontrol-arg' and one press of the universal arg."
  (interactive)
  (setq this-command 'hycontrol-multiply-universal-arg
	hycontrol-arg (* hycontrol-arg 4))
  (when (> (abs hycontrol-arg) hycontrol-maximum-units)
    (setq hycontrol-arg (if (< hycontrol-arg 0) -4 4)))
  hycontrol-arg)

(defun hycontrol-reset-prefix-arg ()
  "Reset HyControl prefix arg to 0."
  (interactive)
  (setq hycontrol-arg 0)
  (hycontrol-frame-to-screen-edges 0))

(defun hycontrol-universal-arg-digit (digit)
  "Return the new prefix argument based on existing `hycontrol-arg' and new DIGIT."
  (setq this-command 'hycontrol-universal-arg-digit
	hycontrol-arg
	(if (< hycontrol-arg 0)
	    (if (and (= hycontrol-arg -1)
		     (not (zerop digit)))
		(- digit)
	       (- (* hycontrol-arg 10) digit))
	  (+ (* hycontrol-arg 10) digit)))
  (when (> (abs hycontrol-arg) hycontrol-maximum-units)
    (setq hycontrol-arg
	  (if (< hycontrol-arg 0)
	      (- digit)
	    digit)))
  hycontrol-arg)


;;; HyControl private initialization functions

(defun hycontrol-setup (arg setup-function)
  "HyControl initialization; passes through ARG and SETUP-FUNCTION.
SETUP-FUNCTION is HyControl mode-specific."
  ;; Save status value of which-key help package and quit any
  ;; in-progress which-key help without any user alert.
  (when (boundp 'which-key-inhibit)
    (setq hycontrol--initial-which-key-inhibit which-key-inhibit
	  which-key-inhibit t)
    (which-key--hide-popup-ignore-command))

  (setq arg (prefix-numeric-value arg)
	inhibit-quit t
	hycontrol--exit-status nil)
  (and hycontrol-debug (not (integerp hycontrol-debug)) (setq hycontrol-debug message-log-max))
  (if (called-interactively-p 'interactive) (hycontrol-save-configurations))
  (cond ((or (not (integerp arg)) (< arg 1))
	 (setq arg 1))
	((> arg hycontrol-maximum-units)
	 (setq arg hycontrol-maximum-units)))
  (setq hycontrol-arg arg
	prefix-arg arg)
  (hycontrol-invert-mode-line)
  (add-hook 'pre-command-hook  'hycontrol-pre-command-hook)
  (add-hook 'post-command-hook 'hycontrol-post-command-hook)
  (funcall setup-function))

(defun hycontrol-frames-setup ()
  "HyControl frames-specific initializations."
  (setq hycontrol--prompt-format hycontrol--frames-prompt-format)
  (hycontrol-post-command-hook)
  ;; Use normal event loop with transient-map until {C-g} or {q} is
  ;; pressed, then exit.
  (setq hycontrol--quit-function
	(set-transient-map hycontrol-frames-mode-map #'hycontrol-stay-in-mode)))

(defun hycontrol-frames (&optional arg)
  "Interactively delete, jump to, move, replicate, and resize frames.
With optional numeric prefix ARG, move and resize by ARG (an
integer) units.  If ARG is < 1, it is set to 1.  If it is >
`hycontrol-maximum-units', it is set to `hycontrol-maximum-units'."
  (interactive "p")
  (hycontrol-setup arg #'hycontrol-frames-setup)
  (unless hycontrol-help-flag
    (message "(HyControl) Frames global minor mode enabled; use {%s} for help"
	     (hycontrol-help-key-description))))

(defun hycontrol-windows-setup ()
  "HyControl windows-specific initializations."
  (setq hycontrol--prompt-format hycontrol--windows-prompt-format)
  (hycontrol-post-command-hook)
  ;; Use normal event loop with transient-map until {C-g} or {q} is
  ;; pressed, then exit.
  (setq hycontrol--quit-function
	(set-transient-map hycontrol-windows-mode-map #'hycontrol-stay-in-mode)))

(defun hycontrol-windows (&optional arg)
  "Interactively delete, jump to, rebalance, resize, and split windows.
With optional numeric prefix ARG, move and resize by ARG (an
integer) units.  If ARG is < 1, it is set to 1.  If it is >
`hycontrol-maximum-units', it is set to `hycontrol-maximum-units'."
  (interactive "p")
  (hycontrol-setup arg #'hycontrol-windows-setup)
  (unless hycontrol-help-flag
    (message "(HyControl) Windows global minor mode enabled; use {%s} for help"
	     (hycontrol-help-key-description))))


;;; HyControl general private functions

(defsubst hycontrol-frame-edges (&optional frame)
  "Return the outermost edge coordinates of optional or selected FRAME.
FRAME must be a live frame and defaults to the selected one.  The
list returned has the form (Left Top Right Bottom) where all
values are in pixels relative to the origin - the position (0, 0)
- of FRAME’s display.  For terminal frames all values are
relative to Left and Top which are both zero."
  (frame-edges frame 'outer-edges))

(defsubst hycontrol-frame-x-origin (&optional frame)
  "Return the X origin coordinate of optional FRAME or the selected frame.
This includes all graphical window manager decorations.  The X
origin coordinate is the upper left point.  Under a graphical
window system, this is in pixels; otherwise, it is in characters."
  (nth 0 (hycontrol-frame-edges frame)))

(defsubst hycontrol-frame-y-origin (&optional frame)
  "Return the Y origin coordinate of optional FRAME or the selected frame.
This includes all graphical window manager decorations.  The Y
origin coordinate is the upper left point.  Under a graphical
window system, this is in pixels; otherwise, it is in characters."
  (nth 1 (hycontrol-frame-edges frame)))

(defun hycontrol-frame-height (&optional frame)
  "Return the height of optional FRAME or the selected frame.
This includes all graphical window manager decorations.  Under a
graphical window system, this is in pixels; otherwise, it is in
characters."
  (frame-pixel-height frame))

(defun hycontrol-frame-width (&optional frame)
  "Return the width of optional FRAME or the selected frame.
This includes all graphical window manager decorations.  Under a
graphical window system, this is in pixels; otherwise, it is in
characters."
  (frame-pixel-width frame))

;; Frame Resizing Support
(defconst hycontrol-screen-offset-sensitivity 12
  "Screen edge sensitivity in pixels.
Number of pixels a frame dimension can be off from its
screen-offset and still be considered at the screen edge.")

(defun hycontrol-frame-at-left-p ()
  "Non-nil if selected frame's left edge is at the left edge of the screen.
Screen left edge is adjusted based on `hycontrol-screen-left-offset'."
  (<= (- (nth 0 (hycontrol-frame-edges)) hycontrol-screen-left-offset)
      hycontrol-screen-offset-sensitivity))

(defun hycontrol-frame-at-top-p ()
  "Non-nil if selected frame's bottom is at the top of the screen.
Screen top is adjusted based on `hycontrol-screen-top-offset'."
  (<= (- (nth 1 (hycontrol-frame-edges)) hycontrol-screen-top-offset
	 ;; Under macOS, frames are automatically offset vertically by
	 ;; the height of the global menubar, so account for that.
	 (if (eq system-type 'darwin) 23 0))
      hycontrol-screen-offset-sensitivity))

(defun hycontrol-frame-at-right-p ()
  "Non-nil if selected frame's right edge is at the right edge of the screen.
Screen right edge is adjusted based on `hycontrol-screen-right-offset'."
  (<= (- (display-pixel-width) (nth 2 (hycontrol-frame-edges)) hycontrol-screen-right-offset)
      hycontrol-screen-offset-sensitivity))

(defun hycontrol-frame-at-bottom-p ()
  "Non-nil if selected frame's bottom is at the bottom of the screen.
Screen bottom edge is adjusted based on `hycontrol-screen-bottom-offset'."
  (<= (- (display-pixel-height) (nth 3 (hycontrol-frame-edges)) hycontrol-screen-bottom-offset
	 ;; Under macOS, frames are automatically offset vertically by
	 ;; the height of the global menubar, so account for that.
	 (if (eq system-type 'darwin) -23 0))
      hycontrol-screen-offset-sensitivity))

;; Frame Zoom Support
(defun hycontrol-frame-zoom (zoom-func arg max-msgs)
  "Zoom default frame face using ZOOM-FUNC and amount ARG (must be 1-9).
MAX-MSGS is a number used only if ZOOM-FUNC is undefined and an
error message is logged."
  (if (fboundp zoom-func)
      (let ((frame-zoom-font-difference arg))
	(funcall zoom-func))
    (hycontrol-user-error max-msgs "(HyControl): Zooming requires separate \"zoom-frm.el\" Emacs Lisp library installation")))


(defun hycontrol-make-frame ()
  "Create a new frame with the same size and selected buffer as the selected frame.
The new frame is selected.  It is offset from the selected frame
by `hycontrol-frame-offset' (x . y) pixels."
  (interactive)
  (select-frame (make-frame (list (cons 'width (frame-width)) (cons 'height (frame-height))
				  (cons 'left (+ (car hycontrol-frame-offset) (car (frame-position))))
				  (cons 'top  (+ (cdr hycontrol-frame-offset) (cdr (frame-position))))))))

(defun hycontrol-move-frame (arrow pixels)
  (let ((x (car (frame-position)))
	(y (cdr (frame-position))))
    (pcase arrow
      ('up    (set-frame-position nil x (- y pixels)))
      ('down  (set-frame-position nil x (+ y pixels)))
      ('left  (set-frame-position nil (- x pixels) y))
      ('right (set-frame-position nil (+ x pixels) y)))))

(defun hycontrol-numeric-keypad (e _arg)
  "Move frame to screen location based on the last pressed numeric keypad key E."
  (let ((num (if (integerp e)
		 e
	       ;; kp-<num> symbol
	       (- (aref (symbol-name e) 3) ?0))))
    (funcall
     (nth num '(nil hycontrol-frame-to-bottom-left hycontrol-frame-to-bottom-center hycontrol-frame-to-bottom-right
		hycontrol-frame-to-left-center hycontrol-frame-to-center hycontrol-frame-to-right-center
		hycontrol-frame-to-top-left hycontrol-frame-to-top-center hycontrol-frame-to-top-right)))))

(defun hycontrol-set-frame-height (frame height &optional pretend pixelwise)
  "Set text height of frame FRAME to HEIGHT lines and fit it to the screen.
Optional third arg PRETEND non-nil means that redisplay should use
HEIGHT lines but that the idea of the actual height of the frame should
not be changed.

Optional fourth argument PIXELWISE non-nil means that FRAME should be
HEIGHT pixels high.  Note: When ‘frame-resize-pixelwise’ is nil, some
window managers may refuse to honor a HEIGHT that is not an integer
multiple of the default frame font height."
  (let ((frame-resize-pixelwise t))
    (set-frame-height frame height pretend pixelwise)
    (hycontrol-frame-fit-to-screen frame)))

(defun hycontrol-set-frame-position (frame x y)
  "Set position of FRAME to (X, Y) and ensure it fits on screen.
FRAME must be a live frame and defaults to the selected one.  X and Y,
if positive, specify the coordinate of the left and top edge of FRAME’s
outer frame in pixels relative to an origin (0, 0) of FRAME’s display.
If any of X or Y is negative, it specifies the coordinates of the right
or bottom edge of the outer frame of FRAME relative to the right or
bottom edge of FRAME’s display."
  (let ((frame-resize-pixelwise t))
    (hycontrol-frame-fit-to-screen frame)
    (set-frame-position frame x y)))

(defun hycontrol-set-frame-size (frame width height &optional pixelwise)
  "Set text size of FRAME to WIDTH by HEIGHT, measured in characters.
Ensure frame fits within the screen size.

Optional argument PIXELWISE non-nil means to measure in pixels.  Note:
When ‘frame-resize-pixelwise’ is nil, some window managers may refuse to
honor a WIDTH that is not an integer multiple of the default frame font
width or a HEIGHT that is not an integer multiple of the default frame
font height."
  (let ((x-origin (hycontrol-frame-x-origin))
	(y-origin (hycontrol-frame-y-origin))
	(frame-resize-pixelwise t))
    (set-frame-size frame width height pixelwise)
    (hycontrol-frame-fit-to-screen frame x-origin y-origin)))

(defun hycontrol-set-frame-width (frame width &optional pretend pixelwise)
  "Set text width of frame FRAME to WIDTH columns and fit it to the screen.
Optional third arg PRETEND non-nil means that redisplay should use WIDTH
columns but that the idea of the actual width of the frame should not
be changed.

Optional fourth argument PIXELWISE non-nil means that FRAME should be
WIDTH pixels wide.  Note: When ‘frame-resize-pixelwise’ is nil, some
window managers may refuse to honor a WIDTH that is not an integer
multiple of the default frame font width."
  (let ((x-origin (hycontrol-frame-x-origin))
	(y-origin (hycontrol-frame-y-origin))
	(frame-resize-pixelwise t))
    (set-frame-width frame width pretend pixelwise)
    (hycontrol-frame-fit-to-screen frame x-origin y-origin)))


(defun hycontrol-display-buffer-predicate-results (buffer)
  (condition-case err
      (mapcar (lambda (expr)
		(if (functionp expr)
		    (funcall expr buffer)
		  (with-current-buffer buffer
		    (eval expr))))
	      hycontrol-display-buffer-predicate-list)
    (error "(HyDebug): Invalid expression in `hycontrol-display-buffer-predicate-list' - %s" err)))

(defvar hycontrol--blank-buffer (get-buffer-create " BLANK")
  "Blank buffer to display in extra window grid windows.
Used after selected buffer list is exhausted.")

(defun hycontrol-window-display-buffer (window)
  "Given a WINDOW, choose the next appropriate buffer to display.
Uses `hycontrol--buffer-list'.

When `hycontrol--invert-display-buffer-predicates' is non-nil and not
\\='ignore, the list of buffers used is further filtered using the
functions and sexpressions in `hycontrol-display-buffer-predicate-list',
which by default filters a frame's buffers to just those with attached
files.

Filtering is disabled if a specific list of buffers is sent to the
`hycontrol-make-windows-grid' function that calls this."
  (let ((buf (car hycontrol--buffer-list-pointer)))
    (setq hycontrol--buffer-list-pointer (cdr hycontrol--buffer-list-pointer))
    (while (and buf (or (= (aref (buffer-name buf) 0) ?\ )
			(and (not hycontrol--invert-display-buffer-predicates)
			     (not (eval (cons 'or (hycontrol-display-buffer-predicate-results buf)))))
			(and hycontrol--invert-display-buffer-predicates
			     (not (eq hycontrol--invert-display-buffer-predicates 'ignore))
			     (eval (cons 'or (hycontrol-display-buffer-predicate-results buf))))))
      ;; Buffer is not one to display, get the next one and test again.
      (setq buf (car hycontrol--buffer-list-pointer)
	    hycontrol--buffer-list-pointer (cdr hycontrol--buffer-list-pointer)))
    (set-window-buffer window
		       (or buf
			   ;; Out of buffers to display, display a blank one
			   hycontrol--blank-buffer))))

(defun hycontrol-window-display-buffer-with-repeats (window)
  "This is no longer used since Hyperbole V8.  Left here for reference.

Given a WINDOW, choose the next appropriate buffer to display
therein using `hycontrol-display-buffer-predicate-list'.  Also
uses the value of`hycontrol--buffer-list' as the list of buffers
to distribute among the windows."
  (let ((buf (car hycontrol--buffer-list-pointer)))
    (setq hycontrol--buffer-list-pointer (cdr hycontrol--buffer-list-pointer))
    (unless buf
      ;; Now on each new pass through buffer-list, the buffer predicate tests will
      ;; be inverted to expand the list of allowed buffers because the
      ;; 1st pass did not produce a buffer for this window.
      (setq hycontrol--buffer-list-pointer hycontrol--buffer-list
	    buf (car hycontrol--buffer-list-pointer)
	    hycontrol--buffer-list-pointer (cdr hycontrol--buffer-list-pointer))
      (unless (eq hycontrol--invert-display-buffer-predicates 'ignore)
	(setq hycontrol--invert-display-buffer-predicates (not hycontrol--invert-display-buffer-predicates))))
    (while (and buf (or (= (aref (buffer-name buf) 0) ?\ )
			(and (not hycontrol--invert-display-buffer-predicates)
			     (not (eval (cons 'or (hycontrol-display-buffer-predicate-results buf)))))
			(and hycontrol--invert-display-buffer-predicates
			     (not (eq hycontrol--invert-display-buffer-predicates 'ignore))
			     (eval (cons 'or (hycontrol-display-buffer-predicate-results buf))))))
      ;; Buffer is not one to display, get the next one and test again.
      (setq buf (car hycontrol--buffer-list-pointer)
	    hycontrol--buffer-list-pointer (cdr hycontrol--buffer-list-pointer)))
    (cond (buf
	   (set-window-buffer window buf))
	  (t
	   ;; Start 2nd or greater pass through buffer list with predicates inverted.
	   (hycontrol-window-display-buffer window)))))


(defun hycontrol-message (max-msgs &rest msg-args)
  "Log MAX-MSGS, adding MSG-ARGS to the *Messages* buffer log."
  (let ((message-log-max max-msgs))
    (apply #'message msg-args)))

(defun hycontrol-user-error (max-msgs &rest err)
  "Log MAX-MSGS, adding ERR to *Messages* buffer; display ERR for 2 seconds."
  (let ((message-log-max max-msgs))
    (beep)
    (apply #'message err)
    (sit-for 2)))

;;; ************************************************************************
;;; Public functions
;;; ************************************************************************

;;; HyControl Global Minor Modes

;;;###autoload
(defun hycontrol-enable-frames-mode (&optional arg)
  "Globally enable HyControl Frames mode for rapid Emacs frame control.

  Interactively delete, jump to, move, replicate, and resize frames.
With optional numeric prefix ARG, move and resize by ARG (an
integer) units.  If ARG is < 1, it is set to 1.  If it is >
`hycontrol-maximum-units', it is set to `hycontrol-maximum-units'."
  (interactive "p")
  (hycontrol-disable-modes)
  (hycontrol-frames-mode (if (and (integerp arg) (>= arg 1)) arg 1)))

;;;###autoload
(defun hycontrol-enable-windows-mode (&optional arg)
  "Globally enable HyControl Windows mode for rapid Emacs window control.

Interactively delete, jump to, rebalance, resize, and split windows.
Optional non-negative numeric prefix ARG is used as the number of
units for commands issued while the mode is active.  If ARG is < 1, it
is set to 1.  If it is > `hycontrol-maximum-units', it is set to
`hycontrol-maximum-units'."
  (interactive "p")
  (hycontrol-disable-modes)
  (hycontrol-windows-mode (if (and (integerp arg) (>= arg 1)) arg 1)))

(defun hycontrol-disable-modes ()
  "Disable HyControl Frames and Windows modes when active."
  (interactive)
  (when (or hycontrol-frames-mode hycontrol-windows-mode)
    (hycontrol-invert-mode-line))
  (hycontrol-frames-mode -1)
  (hycontrol-windows-mode -1))

(defun hycontrol-abort ()
  "Abort HyControl, typically on a press of {\\`C-g'}."
  (interactive)
  (hycontrol-disable-modes)
  (keyboard-quit))

(defun hycontrol-quit ()
  "Globally quit HyControl Frames mode.
Typically on a press of {q}, always on a press of {Q}.
If in a help buffer where {q} is bound to `quit-window', run that
instead of quitting HyControl."
  (interactive)
  ;; Allow for quitting from help windows displayed when HyControl is active.
  (if (and (eq last-command-event ?q)
	   (eq (local-key-binding "q") #'quit-window))
      (call-interactively #'quit-window)
    (when hycontrol-windows-mode
      (message "Hyperbole finished controlling windows"))
    (when hycontrol-frames-mode
      (message "Hyperbole finished controlling frames"))
    (hycontrol-disable-modes)))

;; This just sets the keymap locally and shows the minor mode
;; indicator in the buffer's mode-line; the separate global minor mode
;; turns things on and off.
;;;###autoload
(define-minor-mode hycontrol-local-frames-mode
  "Toggle Hyperbole Frames control minor mode in the current buffer."
  :lighter " HyFrm"
  :group 'hyperbole-screen)

;;;###autoload
(define-global-minor-mode hycontrol-frames-mode hycontrol-local-frames-mode
  (lambda () (hycontrol-local-frames-mode 1))
  :group 'hyperbole-screen)

;; These hooks run by the generated `hycontrol-frames-mode' function
;; do the global work of turning on and off the mode.
(add-hook 'hycontrol-frames-mode-on-hook
	  (lambda () (hycontrol-frames current-prefix-arg)))

(add-hook 'hycontrol-frames-mode-off-hook 'hycontrol-end-mode)

;; This just sets the keymap locally and shows the minor mode
;; indicator in the buffer's mode-line; the separate global minor mode
;; turns things on and off.
;;;###autoload
(define-minor-mode hycontrol-local-windows-mode
  "Toggle Hyperbole Windows control minor mode in the current buffer."
  :lighter " HyWin"
  :group 'hyperbole-screen)

;;;###autoload
(define-global-minor-mode hycontrol-windows-mode hycontrol-local-windows-mode
  (lambda () (hycontrol-local-windows-mode 1))
  :group 'hyperbole-screen)

;; These hooks run by the generated `hycontrol-windows-mode' function
;; do the global work of turning on and off the mode.
(add-hook 'hycontrol-windows-mode-on-hook
	  (lambda () (hycontrol-windows current-prefix-arg)))

(add-hook 'hycontrol-windows-mode-off-hook 'hycontrol-end-mode)

;;; Frame Display Commands
(defun hycontrol-delete-other-frames ()
  "Confirm and then delete all other frames."
  (interactive)
  (if (y-or-n-p "Delete all frames on this screen other than the selected one?")
      (delete-other-frames)))

(defun hycontrol-frame-swap-buffers ()
  "Swap the buffers displayed by each of two visible, most recently used frames.
Return t on success.  The selected window in each frame is used.
If there are not at least two visible frames, do nothing and
return nil."
  (interactive)
  (let ((frames (filtered-frame-list #'frame-visible-p))
	frame2
	windows2
	buf1 buf2)
    (when (>= (length frames) 2)
      (setq frame2 (if (eq (car frames) (selected-frame))
		       (cadr frames)
		     (car frames))
	    windows2 (window-list frame2 'no-mini))
      (setq buf1 (window-buffer (selected-window))
	    buf2 (window-buffer (car windows2)))
      (set-window-buffer (selected-window) buf2)
      (set-window-buffer (car windows2) buf1)
      (other-frame 1)
      t)))

;;; Frame Relocation Commands

(defconst hycontrol--vnk-string
  "(Virtual   7 8 9   Enter a digit to
 Numeric   4 5 6   move the frame
 Keypad)   1 2 3   to that quadrant"
  "Prompt string for virtual numeric keypad (emulate keypad when not available).")

(defun hycontrol-virtual-numeric-keypad (arg)
  (catch 'quit
    (let (e)
      (while (and (setq e (read-char hycontrol--vnk-string))
		  (not (when (memq e '(?q ?\C-g)) (throw 'quit nil)))
		  (or (not (numberp e)) (< e ?0) (> e ?9)))
	(beep))
      (hycontrol-numeric-keypad (- e ?0) arg))))

(defun hycontrol-frame-to-screen-edges (&optional arg)
  "Cycle the selected frame's position clockwise; once per call.
Cycle through the middle of edges and corners of the screen.
With an optional ARG of 0, just reset the cycle position to 0."
  (interactive)
  (if (and arg (eq arg 0))
      (setq hycontrol--screen-edge-position 0)
    (funcall
     (nth hycontrol--screen-edge-position
	  '(hycontrol-frame-to-top-left hycontrol-frame-to-top-center
            hycontrol-frame-to-top-right hycontrol-frame-to-right-center
	    hycontrol-frame-to-bottom-right hycontrol-frame-to-bottom-center
	    hycontrol-frame-to-bottom-left hycontrol-frame-to-left-center)))
    (setq hycontrol--screen-edge-position (1+ hycontrol--screen-edge-position))
    (if (> hycontrol--screen-edge-position 7)
	(setq hycontrol--screen-edge-position 0))))

(defun hycontrol-frame-to-bottom ()
  "Move the selected frame to the bottom of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position
   nil (car (frame-position))
   (- (display-pixel-height) (hycontrol-frame-height)
      hycontrol-screen-bottom-offset)))

(defun hycontrol-frame-to-left ()
  "Move the selected frame to the left of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position nil hycontrol-screen-left-offset (cdr (frame-position))))

(defun hycontrol-frame-to-right ()
  "Move the selected frame to the right of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position
   nil (- (display-pixel-width) (hycontrol-frame-width)
	  hycontrol-screen-right-offset)
   (cdr (frame-position))))

(defun hycontrol-frame-fit-to-screen (&optional frame x-origin y-origin)
  "Ensure the selected frame fits within the screen.
Allow for hycontrol-screen-*-offsets.  Accepts optional arguments
FRAME, X-ORIGIN, and Y-ORIGIN (in pixels) to use when resizing
FRAME (defaults to selected frame)."
  (let ((max-width (- (display-pixel-width) hycontrol-screen-left-offset hycontrol-screen-right-offset 2))
	(max-height (- (display-pixel-height) hycontrol-screen-top-offset hycontrol-screen-bottom-offset 2))
	(frame-resize-pixelwise t))
    (setq x-origin (or x-origin (hycontrol-frame-x-origin frame))
	  y-origin (or y-origin (hycontrol-frame-y-origin frame)))
    (when (> (hycontrol-frame-width frame) max-width)
      ;; Adjust frame size to fit within screen
      (set-frame-width frame (min (hycontrol-frame-width frame) max-width) nil t)
      (if hycontrol-debug (hycontrol-message hycontrol-debug "(HyDebug): Screen (X,Y): %d, %d; Frame Edges (L,T,R,B): %s"
					      (display-pixel-width) (display-pixel-height) (hycontrol-frame-edges frame))))
    (when (> (hycontrol-frame-height frame) max-height)
      ;; Adjust frame size to fit within screen
      (set-frame-height frame (min (hycontrol-frame-height frame) max-height) nil t)
      (if hycontrol-debug (hycontrol-message hycontrol-debug "(HyDebug): Screen (X,Y): %d, %d; Frame Edges (L,T,R,B): %s"
				   (display-pixel-width) (display-pixel-height) (hycontrol-frame-edges frame))))
    ;; Ensure entire frame is positioned onscreen, keeping the
    ;; original frame origin coordinates if possible.
    (set-frame-position frame
			(min (max 0 x-origin)
			     (- (display-pixel-width) (hycontrol-frame-width frame) hycontrol-screen-right-offset))
			(min (max 0 y-origin)
			     (- (display-pixel-height) (hycontrol-frame-height frame) hycontrol-screen-bottom-offset)))
    (if hycontrol-debug (hycontrol-message hycontrol-debug "(HyDebug): Screen (X,Y): %d, %d; Frame Edges (L,T,R,B): %s"
					    (display-pixel-width) (display-pixel-height) (hycontrol-frame-edges frame)))))


(defun hycontrol-frame-to-top ()
  "Move the selected frame to the top of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position nil (car (frame-position)) hycontrol-screen-top-offset))

(defun hycontrol-frame-to-bottom-center ()
  "Move the selected frame to the center of the bottom of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position
   nil (round (/ (- (display-pixel-width) (hycontrol-frame-width)) 2))
   (- (display-pixel-height) (hycontrol-frame-height)
      hycontrol-screen-bottom-offset)))

(defun hycontrol-frame-to-center ()
  "Move the selected frame to the center of the screen."
  (interactive)
  (hycontrol-set-frame-position
   nil
   (round (/ (- (display-pixel-width) (hycontrol-frame-width)) 2))
   (round (/ (- (display-pixel-height) (hycontrol-frame-height)) 2))))


(defun hycontrol-frame-to-left-center ()
  "Move the selected frame to the center of the left of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position
   nil hycontrol-screen-left-offset (round (/ (- (display-pixel-height) (hycontrol-frame-height)) 2))))

(defun hycontrol-frame-to-right-center ()
  "Move the selected frame to the center of the right of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position
   nil (- (display-pixel-width) (hycontrol-frame-width)
	  hycontrol-screen-right-offset)
   (round (/ (- (display-pixel-height) (hycontrol-frame-height)) 2))))

(defun hycontrol-frame-to-top-center ()
  "Move the selected frame to the center of the top of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position nil (round (/ (- (display-pixel-width) (hycontrol-frame-width)) 2)) hycontrol-screen-top-offset))

(defun hycontrol-frame-to-bottom-left ()
  "Move the selected frame to the bottom left of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position nil
				hycontrol-screen-left-offset
				(- (display-pixel-height) (hycontrol-frame-height)
				   hycontrol-screen-bottom-offset)))

(defun hycontrol-frame-to-bottom-right ()
  "Move the selected frame to the bottom right of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position
   nil
   (- (display-pixel-width) (hycontrol-frame-width) hycontrol-screen-right-offset)
   (- (display-pixel-height) (hycontrol-frame-height) hycontrol-screen-bottom-offset)))

;; Frame Resizing

(defun hycontrol-frame-to-top-left ()
  "Move the selected frame to the top left of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position nil hycontrol-screen-left-offset hycontrol-screen-top-offset))

(defun hycontrol-frame-to-top-right ()
  "Move the selected frame to the top right of the screen.
Allow for hycontrol-screen-*-offsets."
  (interactive)
  (hycontrol-set-frame-position
   nil (- (display-pixel-width) (hycontrol-frame-width)
	  hycontrol-screen-right-offset)
   hycontrol-screen-top-offset))

;;; Frame Resizing Commands
(defun hycontrol-frame-resize-percentage (arg)
  "ARG should be between 0 and 100.  0 means don't resize (return 1).
1 is the default value which means cut the frame along the given dimension
in half (return 0.5).  2-100 is converted to a percentage to multiply by.
Over 100 is set to 100.  Under 0 is set to 0.  Floats between 0 and 1
are taken as percentages and used.  Other floats are rounded.
non-integer arguments are ignored and the default value is used."
  (cond ((numberp arg)
	 (cond
	  ((= arg 0) 1)
	  ((= arg 1) 0.5)
	  ((and (> arg 1) (<= arg 100)) (/ arg 100.0))
	  ((< arg 0) 0)
	  ((> arg 100) 1)))
	(t (hycontrol-frame-resize-percentage 1))))

(defun hycontrol-frame-resize-arg (arg)
  "Inverse result of `hycontrol-frame-resize-percentage'.
This is to provide feedback on any argument value adjustment."
  (pcase arg
    (0 0)
    (1 1)
    (50 1)
    ;; Arg must be a percentage, scale it so not fractional.
    ((pred numberp) (round (* arg 100)))
    (_ 1)))

(defun hycontrol-frame-resize-to-bottom (&optional arg)
  "Expand the selected frame to the bottom of the screen.
Allow for hycontrol-screen-*-offsets.  If already at the bottom,
adjust its height to ARG percent of the screen (50% by default if
ARG is 1 or nil) but keep it at the bottom of the screen."
  (interactive "p")
  (setq arg (hycontrol-frame-resize-percentage arg))
  (let ((frame-resize-pixelwise t))
    (if (hycontrol-frame-at-bottom-p)
	;; Reduce frame height to ARG percent, keeping bottom side fixed.
	(set-frame-height nil (min (floor (* (frame-pixel-height) arg))
				   (hycontrol-frame-height))
			  nil t)
      ;; Expand frame height all the way to the bottom, keeping top side fixed.
      (set-frame-height nil (- (display-pixel-height) (cdr (frame-position)) hycontrol-screen-bottom-offset)
			nil t))
    (hycontrol-frame-to-bottom))
  (hycontrol-frame-resize-arg arg))

(defun hycontrol-frame-resize-to-left (&optional arg)
  "Expand the selected frame to the left of the screen.
Allow for hycontrol-screen-*-offsets.  If already at the left,
adjust its width to ARG percent of the screen (50% by default if
ARG is 1 or nil) but keep it at the left of the screen."
  (interactive "p")
  (setq arg (hycontrol-frame-resize-percentage arg))
  (let ((frame-resize-pixelwise t))
    (if (hycontrol-frame-at-left-p)
	;; Reduce frame width to ARG percent, keeping left side fixed.
	(set-frame-width nil (floor (* (frame-pixel-width) arg)) nil t)
      ;; Expand frame width all the way to the left, keeping right side fixed.
      (set-frame-width nil (floor (- (+ (hycontrol-frame-width) (car (frame-position)))
				     (* 2.5 (frame-scroll-bar-width))
				     hycontrol-screen-left-offset))
		       nil t))
    (hycontrol-frame-to-left))
  (hycontrol-frame-resize-arg arg))

(defun hycontrol-frame-resize-to-right (&optional arg)
  "Expand the selected frame to the right of the screen.
Allowing for hycontrol-screen-*-offsets.  If already at the
right, adjust its width to ARG percent of the screen (50% by
default if ARG is 1 or nil) but keep it at the right of the
screen."
  (interactive "p")
  (setq arg (hycontrol-frame-resize-percentage arg))
  (let ((frame-resize-pixelwise t))
    (if (hycontrol-frame-at-right-p)
	;; Reduce frame width to ARG percent, keeping right side fixed.
	(set-frame-width nil (floor (* (frame-pixel-width) arg)) nil t)
      ;; Expand frame width all the way to the right, keeping left side fixed.
      (set-frame-width nil (floor (- (display-pixel-width) (car (frame-position))
				     (* 2.5 (frame-scroll-bar-width))
				     hycontrol-screen-right-offset))
		       nil t))
    (hycontrol-frame-to-right))
  (hycontrol-frame-resize-arg arg))

(defun hycontrol-frame-resize-to-top (&optional arg)
  "Expand the selected frame to the top of the screen.
Allow for hycontrol-screen-*-offsets.  If already at the top,
adjust its height to ARG percent of the screen (50% by default if
ARG is 1 or nil) but keep it at the top of the screen."
  (interactive "p")
  (setq arg (hycontrol-frame-resize-percentage arg))
  (let ((frame-resize-pixelwise t))
    (if (hycontrol-frame-at-top-p)
	;; Reduce frame height to ARG percent, keeping top side fixed.
	(set-frame-height nil (floor (* (frame-pixel-height) arg)) nil t)
      ;; Expand frame height all the way to the top, keeping bottom side fixed.
      (set-frame-height nil (- (+ (cdr (frame-position)) (hycontrol-frame-height)) hycontrol-screen-top-offset)
			nil t))
    (hycontrol-frame-to-top))
  (hycontrol-frame-resize-arg arg))


(defun hycontrol-frame-minimize-lines ()
  "Shrink frame to its smallest number of lines to display all existing windows.
The smallest noumber is an approximation."
  (interactive)
  (let ((l 0))
    (save-window-excursion
      (mapc (lambda (w)
	      (select-window w t)
	      (setq l (+ 2 l (min (window-height) (count-lines (point-min) (point-max))))))
	    (window-list nil 'nomini)))
    (set-frame-height nil l)))

(defun hycontrol-frame-percentage-of-screen (percent &optional dimension)
  "Resize the selected frame to be approximately PERCENT of the screen.
PERCENT may be given as a decimal percentage or a number between 0 and 100.
Optional DIMENSION if given must be either of the symbols, height or
width to affect only that dimension."
  (interactive "nResize frame to be this percent of the screen (1-100): ")
  (if (and (numberp percent)
	   (progn
	     ;; Normalize to a fractional percentage
	     (when (> percent 1)
	       (setq percent (/ percent 100.0)))
	     (setq percent (max (min (float percent) 0.998) 0.0))
	     (> percent 0.0)))
      (let ((frame-resize-pixelwise t)
	    max-height
	    max-width)
	(cond ((eq dimension 'height)
	       (set-frame-height
		nil (min (floor (* (setq max-height (- (display-pixel-height) hycontrol-screen-top-offset hycontrol-screen-bottom-offset))
				   percent))
			 max-height)
		nil t))
	      ((eq dimension 'width)
	       (set-frame-width
		nil (min (floor (* (setq max-width (- (display-pixel-width)
						      (* 2.5 (frame-scroll-bar-width))
						      hycontrol-screen-left-offset hycontrol-screen-right-offset))
				   percent))
			 max-width)
		nil t))
	      (t (set-frame-size
		  nil (min (floor (* (setq max-width (- (display-pixel-width)
							(* 2.5 (frame-scroll-bar-width))
							hycontrol-screen-left-offset hycontrol-screen-right-offset))
				     percent))
			   max-width)
		  (min (floor (* (setq max-height (- (display-pixel-height) hycontrol-screen-top-offset hycontrol-screen-bottom-offset))
				 percent))
		       max-height)
		  t)))
	;; If resize has caused right or bottom edge to move
	;; offscreen, align these edges to the edge of the screen
	;; (moving the frame).
	(when (> (+ (hycontrol-frame-x-origin) (hycontrol-frame-width))
		 (- (display-pixel-width) hycontrol-screen-right-offset))
	  (hycontrol-frame-to-right))
	(when (> (+ (hycontrol-frame-y-origin) (hycontrol-frame-height))
		 (- (display-pixel-height) hycontrol-screen-bottom-offset))
	  (hycontrol-frame-to-bottom))
	;; Return the scaled percentage for setting as numeric argument.
	(floor (* percent 100)))
    (error "(hycontrol-frame-fraction-of-screen): `%s', must be a percent value above 0 and less than or equal to 100" percent)))

(defun hycontrol-frame-height-percentage-of-screen (percent)
  "Resize the selected frame's height to be approximately PERCENT of the screen."
  (interactive "nResize frame height to be this percent of the screen (1-100): ")
  (hycontrol-frame-percentage-of-screen percent 'height))

(defun hycontrol-frame-width-percentage-of-screen (percent)
  "Resize the selected frame's width to be approximately PERCENT of the screen."
  (interactive "nResize frame width to be this percent of the screen (1-100): ")
  (hycontrol-frame-percentage-of-screen percent 'width))

;;; Frame Cycle Common Sizes

(defun hycontrol-set-width-percentage-full-height (width-percentage)
  (hycontrol-frame-width-percentage-of-screen width-percentage)
  (hycontrol-frame-height-percentage-of-screen 1))

(defun hycontrol-set-height-percentage-full-width (height-percentage)
  (hycontrol-frame-width-percentage-of-screen 1)
  (hycontrol-frame-height-percentage-of-screen height-percentage))

;;;###autoload
(defun hycontrol-frame-adjust-widths ()
  "Cycle through different common width adjustments of a frame.
Widths are given in screen percentages by the list
`hycontrol-frame-widths' and typically go from widest to narrowest."
  (interactive)
  (when (null hycontrol--frame-widths-pointer)
    (setq hycontrol--frame-widths-pointer hycontrol-frame-widths))
  (hycontrol-frame-width-percentage-of-screen
   (car hycontrol--frame-widths-pointer))
  (message "Screen Percentage: Width %.1f%%; Fixed Height %.1f%%"
	   (* 100.0 (car hycontrol--frame-widths-pointer))
	   (* 100.0 (/ (float (hycontrol-frame-height))
		       (- (display-pixel-height) hycontrol-screen-top-offset hycontrol-screen-bottom-offset))))
  (setq hycontrol--frame-widths-pointer
	(cdr hycontrol--frame-widths-pointer)))

;;;###autoload
(defun hycontrol-frame-adjust-widths-full-height ()
  "Cycle through frame width adjustments after fixing its height full-screen.
Widths are given in screen percentages by the list
`hycontrol-frame-widths' and typically go from widest to narrowest."
  (interactive)
  (when (null hycontrol--frame-widths-pointer)
    (setq hycontrol--frame-widths-pointer hycontrol-frame-widths))
  (hycontrol-set-width-percentage-full-height
   (car hycontrol--frame-widths-pointer))
  (message "Screen Percentage: Width %.1f%%; Fixed Height %d%%"
	   (* (car hycontrol--frame-widths-pointer) 100.0) 100)
  (setq hycontrol--frame-widths-pointer
	(cdr hycontrol--frame-widths-pointer)))

;;;###autoload
(defun hycontrol-frame-adjust-heights ()
  "Cycle through different common height adjustments of a frame.
Heights are given in screen percentages by the list
`hycontrol-frame-heights' and typically go from tallest to shortest."
  (interactive)
  (when (null hycontrol--frame-heights-pointer)
    (setq hycontrol--frame-heights-pointer hycontrol-frame-heights))
  (hycontrol-frame-height-percentage-of-screen
   (car hycontrol--frame-heights-pointer))
  (message "Screen Percentage: Fixed Width %.1f%%; Height %.1f%%"
	   (* 100.0 (/ (float (hycontrol-frame-width))
		       (- (display-pixel-width)
			  hycontrol-screen-left-offset hycontrol-screen-right-offset)))
	   (* 100.0 (car hycontrol--frame-heights-pointer)))
  (setq hycontrol--frame-heights-pointer
	(cdr hycontrol--frame-heights-pointer)))

;;;###autoload
(defun hycontrol-frame-adjust-heights-full-width ()
  "Cycle through height adjustments of a frame after fixing its width full-screen.
Heights are given in screen percentages by the list
`hycontrol-frame-heights' and typically go from tallest to shortest."
  (interactive)
  (when (null hycontrol--frame-heights-pointer)
    (setq hycontrol--frame-heights-pointer hycontrol-frame-heights))
  (hycontrol-set-height-percentage-full-width
   (car hycontrol--frame-heights-pointer))
  (message "Screen Percentage: Fixed Width %d%%; Height %.1f%%"
	   100 (* (car hycontrol--frame-heights-pointer) 100.0))
  (setq hycontrol--frame-heights-pointer
	(cdr hycontrol--frame-heights-pointer)))

;;; Move among frames
(defun hycontrol-framemove-direction (direction)
  (hypb:require-package 'framemove)
  (fm-next-frame direction))

(defun hycontrol-framemove-up ()
  "Move to any next frame above the selected frame."
  (interactive)
  (hycontrol-framemove-direction 'up))

(defun hycontrol-framemove-left ()
  "Move to any next frame to the left of the selected frame."
  (interactive)
  (hycontrol-framemove-direction 'left))

(defun hycontrol-framemove-right ()
  "Move to any next frame to the right of the selected frame."
  (interactive)
  (hycontrol-framemove-direction 'right))

(defun hycontrol-framemove-down ()
  "Move to any next frame below the selected frame."
  (interactive)
  (hycontrol-framemove-direction 'down))

;;; Frame Configuratons

(defun hycontrol-restore-frame-configuration ()
  (interactive)
  (when (and (y-or-n-p "Restore previously saved configuration of all frames?")
	     (frame-configuration-p hycontrol--fconfig))
    (set-frame-configuration hycontrol--fconfig)))

(defun hycontrol-save-frame-configuration ()
  (interactive)
  (setq hycontrol--fconfig (current-frame-configuration))
  (if (called-interactively-p 'interactive)
      (minibuffer-message "(Hyperbole): Saved configuration of all frames")))

(defun hycontrol-save-configurations ()
  (interactive)
  (hycontrol-save-frame-configuration)
  (hycontrol-save-window-configuration))


;;; Window Commands

(defun hycontrol-invert-mode-line ()
  "Invert the background and foreground faces of the selected window mode-line.
Enabled if `hycontrol-invert-mode-line-flag' is non-nil."
  (when hycontrol-invert-mode-line-flag
    (let* ((bg (face-background 'mode-line))
	   (fg (face-foreground 'mode-line)))
      (set-face-foreground 'mode-line bg)
      (set-face-background 'mode-line fg))
    (force-mode-line-update t)))

(defun hycontrol-windows-grid-marked-items ()
  "Return any marked items in Dired, Buffer Menu or IBuffer mode."
  ;; Get the list of marked items if in an item list buffer and
  ;; convert items to buffers.
  (let ((items (cond ((and (derived-mode-p 'dired-mode)
			   (mapcar #'find-file-noselect (dired-get-marked-files))))
		     ((and (eq major-mode 'Buffer-menu-mode)
			   (Buffer-menu-marked-buffers)))
		     ((and (eq major-mode 'ibuffer-mode)
			   (ibuffer-get-marked-buffers))))))
    ;; Ignore buffer list predicate filters when items exist so the
    ;; items are not filtered out.
    (setq hycontrol--invert-display-buffer-predicates
	  (when items 'ignore))
    ;; Return either non-nil items or frame's full buffer list.
    items))

(defun hycontrol-windows-grid-buffer-list ()
  "Return buffer list for grid.
Buffers are either any marked items in Dired, Buffer Menu or
IBuffer mode or the existing frame's buffer list."
  ;; Get the list of marked items if in an item list buffer and
  ;; convert items to buffers.
  ;; Return either non-nil items or frame's full buffer list.
  (or (hycontrol-windows-grid-marked-items)
      (buffer-list (selected-frame))))

;;; Split selected frame into a grid of windows given by row and
;;; column count, displaying different buffers in each window.
;;;###autoload
(defun hycontrol-windows-grid (arg)
  "Display a grid of windows in the selected frame, sized to prefix ARG.
Left digit of ARG is the number of grid rows and the right digit
is the number of grid columns.  Use {\\`C-h' h h} to restore the
prior frame configuration after a grid is displayed.

If ARG is 0, prompt for a major mode whose buffers should be
displayed in the grid windows, then prompt for the grid size.

If ARG is < 0, prompt for a glob-type file pattern and display
files that match the pattern in an abs(ARG) sized windows grid
or an autosized one if the ARG value is invalid.

Otherwise, prompt for the grid size if ARG is an invalid size
\(positive and more than two digits).

With a current buffer in Dired, Buffer Menu or IBuffer mode that
contains marked items, the buffers associated with those items
are displayed in the grid (unless ARG is 0).

By default, the most recently used buffers are displayed in each
window, first selecting only those buffers which match any of the
predicate expressions in `hycontrol-display-buffer-predicate-list'.
\(The default predicate list chooses buffers with attached files).
Then, if there are not enough buffers for all windows, the buffers
that failed to match to any predicate are used.  In all cases,
buffers whose names start with a space are ignored.

When done, this resets the persistent HyControl prefix argument to
1 to prevent following commands from using the often large grid size
argument.

If the key that invokes this command in `hyperbole-minor-mode' is also
bound in the current major mode map, then interactively invoke that
command instead.  Typically prevents clashes over {\\`C-c' @}."
  (interactive "P")
  (let ((numeric-arg (prefix-numeric-value current-prefix-arg)))
    (if (or (<= numeric-arg 0) (> numeric-arg 11))
	;; Very unlikely other mode commands could use prefix arg in
	;; this range, so ignore other key bindings.
	(hycontrol--windows-grid-internal arg)
      (let* ((key (hypb:cmd-key-vector #'hycontrol-windows-grid hyperbole-mode-map))
	     (mode-binding (lookup-key (current-local-map) key))
	     (this-key-flag (and (called-interactively-p 'interactive)
				 (equal (this-single-command-keys) key))))
	(cond ((and mode-binding (not (integerp mode-binding))
		    this-key-flag (if (eq major-mode #'outline-mode) (not arg) t))
	       ;; If the key that invokes this command in `hyperbole-minor-mode'
	       ;; is also bound in the current major mode map, then
	       ;; interactively invoke that command instead.  Typically
	       ;; prevents clashes over {C-c @}.
	       (call-interactively mode-binding))
	      ((and (not arg) this-key-flag
		    (boundp 'outline-minor-mode) outline-minor-mode
		    (setq mode-binding (lookup-key outline-minor-mode-map key))
		    (not (integerp mode-binding))
		    (keymapp mode-binding))
	       ;; Prevent a conflict with keymap binding in Outline minor mode
	       (kbd-key:key-series-to-events key))
	      ;;
	      ;; No key conflicts, display window grid
	      (t (hycontrol--windows-grid-internal arg)))))))

(defun hycontrol--windows-grid-internal (arg)
  (setq arg (prefix-numeric-value (or arg current-prefix-arg)))
  (cond ((> arg 0)
	 (hycontrol-make-windows-grid arg))
	((< arg 0)
	 (setq current-prefix-arg (abs arg))
	 (call-interactively #'hycontrol-windows-grid-by-file-pattern))
	(t
	 (setq current-prefix-arg 0)
	 (call-interactively #'hycontrol-windows-grid-by-major-mode))))

(defun hycontrol-windows-grid-validate (arg)
  "Return the closest valid windows grid size from the two digit numeric abs(ARG).
The digits of ARG specify the number of rows (leftmost) by
columns (rightmost) for a grid."
  (if (integerp arg)
      (let* ((grid-size (abs arg))
	     (rows (floor (/ grid-size 10)))
	     (columns (- grid-size (* rows 10))))
	(cond ((<= grid-size 10)
	       (hycontrol-windows-grid-minimum-size grid-size))
	      ((> grid-size 99)
	       99)
	      ((zerop columns)
	       (hycontrol-windows-grid-minimum-size (* rows columns)))
	      (t grid-size)))
    (error "(hycontrol-windows-grid-validate): 'arg' must be an integer, not '%s'" arg)))

(defun hycontrol-windows-grid-number-of-windows (arg)
  "Return the product of the absolute value of the two digit numeric ARG.
The digits of ARG specify the number of rows (leftmost) by
columns (rightmost) for a grid."
  (if (integerp arg)
      (let* ((grid-size (abs arg))
	     (rows (floor (/ grid-size 10)))
	     (columns (- grid-size (* rows 10))))
	(cond ((<= grid-size 10)
	       (hycontrol-windows-grid-number-of-windows
		(hycontrol-windows-grid-minimum-size grid-size)))
	      ((> grid-size 99)
	       81)
	      (t (* rows columns))))
    (error "(hycontrol-windows-grid-number-of-windows): 'arg' must be an integer, not '%s'" arg)))

(defun hycontrol-windows-grid-minimum-size (num-buffers)
  "Return the minimum integer window grid size to display abs(NUM-BUFFERS).
Minimize number of rows rather than columns.  Size is a 2 digit
whole number with the first digit number of rows and the second,
number of columns of windows.  Return 0 if NUM-BUFFERS is 0."
  (if (integerp num-buffers)
      (let* ((num-cols (ceiling (sqrt num-buffers)))
	     (num-rows (max (1- num-cols) 0))
	     (grid-size (+ (* num-rows 10) num-cols)))
	(when (< (* num-rows num-cols) num-buffers)
	  (setq grid-size (+ 10 grid-size)))
	grid-size)
    (error "(hycontrol-windows-grid-minimum-size): 'num-buffers' must be an integer, not '%s'" num-buffers)))

(defun hycontrol-windows-grid-by-buffer-list (grid-size buffers)
  "Display an abs(GRID-SIZE) window grid showing a subset of a list of BUFFERS.
If GRID-SIZE is nil, prompt for grid size.  The digits of
GRID-SIZE specify the number of rows (leftmost) by
columns (rightmost) of the grid."
  (if (null buffers)
      (error "(hycontrol-windows-grid-by-buffer-list): No matching buffers")
    (hycontrol-make-windows-grid
     (when grid-size (hycontrol-windows-grid-validate grid-size))
     (mapc #'switch-to-buffer (reverse buffers)))))

(defun hycontrol-windows-grid-by-file-list (grid-size files)
  "Display an abs(GRID-SIZE) window grid showing a subset of a list of FILES.
If GRID-SIZE is nil, prompt for grid size.  The digits of
GRID-SIZE specify the number of rows (leftmost) by
columns (rightmost) of the grid."
  (if (null files)
      (error "(hycontrol-windows-grid-by-file-list): No matching files")
    (hycontrol-make-windows-grid
     (when grid-size (hycontrol-windows-grid-validate grid-size))
     (nreverse (mapcar #'find-file (reverse files))))))

;;;###autoload
(defun hycontrol-windows-grid-by-file-pattern (arg pattern &optional full-flag)
  "Display up to an abs(prefix ARG)-sized window grid of files matching PATTERN.
Use absolute file paths if called interactively or optional
FULL-FLAG is non-nil.  PATTERN is a shell file glob pattern.

Left digit of ARG is the number of grid rows and the right digit
is the number of grid columns.  If ARG is nil, 0, 1, less than
11, greater than 99, then autosize the grid to fit the number of
files matched by PATTERN.  Otherwise, if ARG ends in a 0, adjust the
grid size to the closest valid size."
  (interactive
   (list current-prefix-arg
	 (read-string "Pattern of files to display in windows grid: ")
	 t))
  (when (memq arg '(0 1))
    (setq arg nil))
  (let* ((find-file-wildcards t)
	 (files (file-expand-wildcards pattern full-flag))
	 (num-files (length files))
	 grid-size
	 num-windows
	 max-files)
    (when (zerop num-files)
      (error "(hycontrol-windows-grid-by-file-pattern): '%s' pattern did not match any files" pattern))
    (setq grid-size (if arg
			(hycontrol-windows-grid-validate arg)
		      (hycontrol-windows-grid-minimum-size num-files)))
    (when (zerop grid-size)
      (error "(hycontrol-windows-grid-by-file-pattern): '%s' pattern produced invalid zero window grid size" pattern))
    (setq num-windows (hycontrol-windows-grid-number-of-windows (or arg grid-size))
	  max-files (min num-files num-windows))
    (when (> num-files max-files)
      ;; Cut off list at max-files items
      (setq files (cl-loop repeat max-files for files in files collect files)))
    (hycontrol-windows-grid-by-file-list grid-size files)))

;;;###autoload
(defun hycontrol-windows-grid-by-major-mode (arg mode)
  "Display a prefix ARG-sized grid of windows with buffers of major MODE.
Left digit of ARG is the number of grid rows and the right digit
is the number of grid columns.

See documentation of `hycontrol-windows-grid' for further details."
  (interactive
   (list (prefix-numeric-value current-prefix-arg)
	 (let* ((set:equal-op 'eq)
		(mode-strings (mapcar #'symbol-name (apply #'set:create (mapcar (lambda (buf) (buffer-local-value 'major-mode buf))
										(hycontrol-windows-grid-buffer-list))))))
	   (intern-soft (completing-read "(HyControl Grid Windows): Major mode of buffers to display: "
					 mode-strings nil t nil nil (symbol-name major-mode))))))
  (let ((hycontrol-display-buffer-predicate-list `((eq major-mode ',mode))))
    (hycontrol-make-windows-grid arg)))

;;;###autoload
(defun hycontrol-windows-grid-repeatedly (&optional arg)
  "Repeatedly display different window grid layouts according to prefix ARG.
ARG is prompted for each time.  Left digit of ARG is the number
of grid rows and the right digit is the number of grid columns.

See documentation of `hycontrol-windows-grid' for further details."
  (interactive "p")
  (catch 'done
    (let (hycontrol-help-flag)
      (while t
	(while (not (or (eq arg 0) (hycontrol-windows-grid-valid-size-p arg)))
	  (setq arg (read-string "Display grid of ROW digit by COLUMN digit windows, e.g. 23 for 2R by 3C (RET to quit): "))
	  (setq arg (if (string-equal arg "")
			(throw 'done t)
		      (string-to-number arg)))
	  (unless (or (eq arg 0) (hycontrol-windows-grid-valid-size-p arg))
	    (beep)))
	(hycontrol-windows-grid arg)
	(setq arg nil)))))

(defun hycontrol-make-windows-grid (arg &optional buffer-list)
  "Display a grid of windows in the selected frame, sized according to prefix ARG.
Left digit of ARG is the number of grid rows and the right digit is
the number of grid columns.  If ARG is invalid, prompt for grid size.

If optional BUFFER-LIST is provided, use its length to determine
minimum grid size, ignoring ARG.

See documentation of `hycontrol-windows-grid' for further details."
  (interactive "p")

  (unless buffer-list
    (setq buffer-list (hycontrol-windows-grid-marked-items)))

  (unless (or arg buffer-list)
    (error "(hycontrol-make-windows-grid): Both 'arg' and 'buffer-list' cannot both be null"))

  (cond (buffer-list
	 ;; Override `arg' when `buffer-list' is given
	 (setq arg (hycontrol-windows-grid-minimum-size (length buffer-list))))
	;; Check `arg': must be 2 digits of [1-9], else read a new `arg' or
	;; signal an error when in a HyControl mode and help is displayed.
	((and (and hycontrol-help-flag (or hycontrol-frames-mode hycontrol-windows-mode))
	      (not (hycontrol-windows-grid-valid-size-p arg)))
	 (let ((hyc-mode (if hycontrol-frames-mode #'hycontrol-frames-mode #'hycontrol-windows-mode)))
	   (hycontrol-disable-modes)
	   (setq arg 1)
	   (while (not (hycontrol-windows-grid-valid-size-p arg))
	     (unless (memq arg '(0 1))
	       (beep))
	     (setq arg (read-number "Display grid of ROW digit by COLUMN digit windows, e.g. 23 for 2R by 3C: ")))
	   (funcall hyc-mode arg)))
	(t (while (not (hycontrol-windows-grid-valid-size-p arg))
	     (unless (memq arg '(0 1))
	       (beep))
	     (setq arg (read-number "Display grid of ROW digit by COLUMN digit windows, e.g. 23 for 2R by 3C: ")))))

  (let ((wconfig (current-window-configuration))
	(hist-elt (hhist:element)))
    ;; If an error occurs during a window split because the window is
    ;; too small, then restore prior window configuration.
    (condition-case err
	;; Make 1 window in selected frame
	(progn (delete-other-windows)

	       (let* ((rows (floor (/ arg 10)))
		      (columns (- arg (* rows 10)))
		      (row-index (1- rows))
		      (row-window-list (list (selected-window)))
		      col-index)

		 ;; Create ARG left-digit rows via split-windows,
		 ;; balancing each time.
		 (while (> row-index 0)
		   (setq row-window-list (cons (split-window-vertically) row-window-list))
		   (balance-windows)
		   (setq row-index (1- row-index)))

		 ;; Create ARG right-digit columns in each row via
		 ;; split-windows, balancing each time.
		 (setq row-index rows)
		 (while (> row-index 0)
		   (with-selected-window (car row-window-list)
		     (setq col-index (1- columns))
		     (while (> col-index 0)
		       (split-window-horizontally)
		       (balance-windows)
		       (setq col-index (1- col-index)))
		     (setq row-index (1- row-index)
			   row-window-list (cdr row-window-list)))))

	       ;; Walk windows in this frame and display different
	       ;; buffers.  In the first pass, select only buffers
	       ;; that pass at least one predicate test in
	       ;; `hycontrol-display-buffer-predicate-list'.  If run
	       ;; out of buffers before windows, then start a 2nd
	       ;; pass at the start of the buffer list and use the
	       ;; inverse, choosing only those buffers that fail all
	       ;; the predicate tests.  Always ignore buffers that
	       ;; start with a space.  With each succeeding pass, the
	       ;; predicate list is inverted again.
	       (setq hycontrol--buffer-list (or buffer-list
						(hycontrol-windows-grid-buffer-list))
		     hycontrol--buffer-list-pointer hycontrol--buffer-list)
  	       (walk-windows #'hycontrol-window-display-buffer 'no-minibuf)

	       ;; Prevent user from mistakenly using the typically
	       ;; large argument that invoked this function; reset it
	       ;; to 1 if there was no error.
	       (setq hycontrol-arg 1))
      (error (set-window-configuration wconfig)
	     (and hycontrol-help-flag (or hycontrol-frames-mode hycontrol-windows-mode)
		  (pop-to-buffer (messages-buffer)))
	     (error "(HyDebug): Grid Size: %d; %s" arg err)))
    ;; No error, save prior frame configuration for easy return
    (hhist:add hist-elt)
    ;; Return nil so frame config structure is never displayed
    nil))

(defun hycontrol-delete-other-windows ()
  "Confirm and then delete all other windows in the selected frame."
  (interactive)
  (when (y-or-n-p "Delete all windows in this frame other than the selected one?")
    (delete-other-windows)))

(defun hycontrol-window-maximize-lines ()
  "Grow window to its maximum possible number of lines without removing windows."
  (interactive)
  (maximize-window))

(defun hycontrol-window-minimize-lines ()
  "Shrink window to its smallest number of lines to show whole buffer, if possible.
Otherwise or if the window is already displaying all of its
lines, shrink it to about one line, if possible."
  (interactive)
  (let ((neg-shrink-amount (- (1+ (count-lines (point-min) (point-max)))))
	(window-min-height 1))
    ;; Don't use minimize-window here since it shrinks regardless of
    ;; buffer size.
    (if (window-resizable-p (selected-window) neg-shrink-amount)
	(progn (goto-char (point-min))
	       (shrink-window (+ (window-height) neg-shrink-amount)))
      (shrink-window (1- (window-height))))))

(defun hycontrol-window-swap-buffers ()
  "Swap the buffers displayed in two windows in the selected frame and return t.
Swap which window is selected so that the current buffer remains
the same.  Do nothing and return nil if there are not precisely
two windows."
  (interactive)
  (let ((windows (window-list nil 'no-mini))
	buf1 buf2)
    (when (= 2 (length windows))
      (setq buf1 (window-buffer (car windows))
	    buf2 (window-buffer (cadr windows)))
      (set-window-buffer (car windows) buf2)
      (set-window-buffer (cadr windows) buf1)
      (other-window 1)
      t)))

;; Derived from Emacs mouse.el.
;;;###autoload
(defun hycontrol-window-to-new-frame ()
  "Create a new frame sized to match the selected window with the same buffer.
If there is only one window in the source frame or if
`hycontrol-keep-window-flag' is non-nil, leave the original
window and just clone it into the new frame; otherwise, delete
the original window."
  (interactive)
  (let ((w (selected-window))
	(frame-resize-pixelwise t)
	(only-one-window (one-window-p)))
    (cond ((window-minibuffer-p w)
	   (beep)
	   (minibuffer-message "(Hyperbole): Select a non-minibuffer window"))
	  (t
	   ;; Give temporary modes such as isearch a chance to turn off.
	   (run-hooks 'mouse-leave-buffer-hook)
	   (select-frame (make-frame (frame-parameters)))
	   (unless only-one-window
	     (hycontrol-set-frame-size nil (window-size w t t) (window-size w nil t) t))
	   (set-frame-position nil (+ (car hycontrol-frame-offset)
				      (car (frame-position (window-frame w))))
			       (+ (cdr hycontrol-frame-offset)
				  (cdr (frame-position (window-frame w)))))
	   (with-selected-frame (window-frame w)
	     (unless (or hycontrol-keep-window-flag (one-window-p t))
	       (delete-window w)))))))

;;;###autoload
(defun hycontrol-clone-window-to-new-frame ()
  "Create a new frame sized to match the selected window with the same buffer."
  (interactive)
  (let ((hycontrol-keep-window-flag t))
    (hycontrol-window-to-new-frame)))

(defun hycontrol-restore-window-configuration ()
  (interactive)
  (when (and (y-or-n-p "Restore saved window configuration in this frame?")
	     (window-configuration-p hycontrol--wconfig))
    (set-window-configuration hycontrol--wconfig)))

(defun hycontrol-save-window-configuration ()
  (interactive)
  (setq hycontrol--wconfig (current-window-configuration))
  (if (called-interactively-p 'interactive)
      (minibuffer-message "(Hyperbole): Saved window configuration for this frame")))

;;; Screen Offsets - Set once when this file is loaded; `hycontrol-set-screen-offsets' resets them.
(defun hycontrol-display-screen-offsets ()
  "Display a minibuffer message listing HyControl's screen edge offsets in pixels."
  (interactive)
  (message "Screen pixel offsets are: Left: %d; Top: %d; Right: %d; Bot: %d"
	   hycontrol-screen-left-offset
	   hycontrol-screen-top-offset
	   hycontrol-screen-right-offset
	   hycontrol-screen-bottom-offset))

(defun hycontrol-get-screen-offsets ()
  "Return the first matching list of screen edge .50%%%offsets.
The offsets are from `hycontrol-screen-offset-alist'.  See its
documentation for more information."
  (interactive)
  (prog1 (catch 'result
	   (let (predicate offsets width height)
	     (mapc (lambda (pred-offsets)
		     (setq predicate (car pred-offsets)
			   offsets (cdr pred-offsets))
		     (cond ((functionp predicate)
			    (if (funcall predicate) (throw 'result offsets)))
			   ;; (width . height)
			   ((and (consp predicate)
				 (integerp (car predicate))
				 (setq width (car predicate))
				 (or (and (integerp (cdr predicate))
					  (setq height (cdr predicate)))
				     ;; In case, user forgets the . in the cons.
				     (and (listp (cdr predicate))
					  (integerp (cadr predicate))
					  (setq height (cadr predicate)))))
			    (and (= width (display-pixel-width))
				 (= height (display-pixel-height))
				 (throw 'result offsets)))
			   ;; Emacs Lisp form
			   ((eval predicate)
			    (throw 'result offsets))))
		   hycontrol-screen-offset-alist))
	   (error "(HyDebug): No matching predicate in `hycontrol-screen-offset-alist' - %s"
		  hycontrol-screen-offset-alist))
    (if (called-interactively-p 'interactive) (hycontrol-display-screen-offsets))))

(defun hycontrol-set-screen-offsets ()
  "Set screen edge offsets to the first matching list of offsets.
Offsets are from `hycontrol-screen-offset-alist'.  See its
documentation for more information."
  (interactive)
  (let ((offsets (hycontrol-get-screen-offsets)))
    (setq hycontrol-screen-left-offset   (nth 0 offsets)
          hycontrol-screen-top-offset    (nth 1 offsets)
	  hycontrol-screen-right-offset  (nth 2 offsets)
	  hycontrol-screen-bottom-offset (nth 3 offsets))
    (if (called-interactively-p 'interactive) (hycontrol-display-screen-offsets))
    offsets))

(hycontrol-set-screen-offsets)

(defun hycontrol-help-key-description ()
  "Return the key description for the HyControl help key."
  (key-description (where-is-internal 'hycontrol-toggle-help hycontrol-frames-mode-map t)))

(defun hycontrol-toggle-help ()
  "Toggle whether HyControl displays key binding help in the minibuffer."
  (interactive)
  (setq hycontrol-help-flag (not hycontrol-help-flag))
  (unless (and hycontrol-help-flag (called-interactively-p 'interactive))
    (message "(HyControl): Minibuffer help is off; use {%s} to turn it on"
	     (hycontrol-help-key-description))))

(provide 'hycontrol)

;;; hycontrol.el ends here
