/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * 			Andrea Zoppello (Engineering Group) - initial API and implementation and/or initial documentation
 * 			Gianfranco Boccalon (Engineering Group) - initial API and implementation and/or initial documentation
 *          Luca Rossato ( Engineering Group ) - initial API and implementation and/or initial documentation
 *          Luca Barozzi ( Engineering Group ) - initial API and implementation and/or initial documentation
 *  		Antonietta Miele ( Engineering Group ) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebpm.connectors.http;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.wsdl.Definition;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.WSDLException;
import javax.wsdl.extensions.soap.SOAPAddress;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLReader;
import javax.xml.namespace.QName;

import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.eclipse.ebpm.connectors.http.adapters.IHTTPOutputProtocolAdapter;
import org.eclipse.ebpm.connectors.http.adapters.SpagicJettyHTTPExchange;
import org.eclipse.ebpm.connectors.http.ssl.SpagicHttpWithSSLClient;
import org.eclipse.ebpm.connectors.http.ssl.SslBean;
import org.eclipse.ebpm.connectors.http.util.WSSecurityConstants;
import org.eclipse.ebpm.core.AbstractSpagicConnector;
import org.eclipse.ebpm.core.ExchangeUtils;
import org.eclipse.ebpm.messaging.api.Exchange;
import org.eclipse.ebpm.messaging.api.Message;
import org.eclipse.ebpm.messaging.api.Pattern;
import org.eclipse.ebpm.messaging.api.Status;
import org.eclipse.ebpm.soap.bindings.soap.Soap11;
import org.eclipse.ebpm.soap.bindings.soap.impl.Wsdl1SoapBindingImpl;
import org.eclipse.ebpm.soap.wsdl.BindingFactory;
import org.eclipse.ebpm.soap.wsdl.WSDLUtils;
import org.eclipse.ebpm.util.resources.IResource;
import org.eclipse.jetty.client.Address;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.security.ProxyAuthorization;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HTTPClient extends AbstractSpagicConnector{
	
	protected Logger logger = LoggerFactory.getLogger(HTTPClient.class);
	
	private String locationURI = null;
	private boolean ssl = false; 
	private SslBean sslBean = null;
	private IResource wsdl;
	private QName service = null;
	private String port = null;
	private org.eclipse.ebpm.soap.api.model.Binding<?> binding = null;
	
	private long clientTimeout = 60000;
    private HttpClient jettyClient;
   
   // private String mep = null;
    
    private int jettyClientThreadPoolSize = 16;
    
    // PROXY
    boolean isProxy = false;
    boolean isProxyAuth = false;
    
    private String proxyHost=null;
    private int proxyPort = 80;
    private String proxyUsername=null;
    private String proxyPassword=null;
    
    private ConcurrentHashMap<String, IHTTPOutputProtocolAdapter> marshalers = new ConcurrentHashMap<String, IHTTPOutputProtocolAdapter>();
    private String marshalerId;
    private IHTTPOutputProtocolAdapter protocolAdapter;
    
    private boolean isSoap = false;
    private boolean isPipeline = false;
  
   
    public void init(){
    	
    	logger.info("-- HTTP Client Component Init --");
		this.locationURI = propertyConfigurator.getString("locationURI");
		this.clientTimeout = propertyConfigurator.getLong("timeout", (long)60000);
	
		this.isSoap = propertyConfigurator.getBoolean("isSoap", false);
		this.ssl = propertyConfigurator.getBoolean("isSSL", false);
		this.isProxy = propertyConfigurator.getBoolean("isProxyEnabled", false);
		
		if (ssl)
			this.sslBean = getSslParameters();
		
		
		if (isSoap){
			this.marshalerId = propertyConfigurator.getString("marshaller","SOAP");
			this.wsdl = propertyConfigurator.getResource("wsdl", null);
		}else{
			this.marshalerId = propertyConfigurator.getString("marshaller","PlainHTTP");
		}

		// get proxy parameters
		if (isProxy){
			this.proxyPort = propertyConfigurator.getInteger("ProxyPort", 80);
			this.proxyHost = propertyConfigurator.getString("ProxyHost", null);
			this.isProxyAuth = propertyConfigurator.getBoolean("isProxyAuthenticationRequired", false);
			if (isProxyAuth){
				this.proxyUsername = propertyConfigurator.getString("ProxyUsername", "");
				this.proxyPassword = propertyConfigurator.getString("ProxyPassword", "");
			}
		}	
		
		this.protocolAdapter = marshalers.get(marshalerId);
		
		Map<String, String> map = new HashMap<String, String>();
	 	String senderAction = propertyConfigurator.getString("senderAction", null);
	 	String receiverAction = propertyConfigurator.getString("receiverAction", null);
	 	
	 	isPipeline = propertyConfigurator.getBoolean("isPipeline",false);
	 	
	 	if(senderAction != null){
		 	if(senderAction.equals(WSSecurityConstants.SIGNATURE)){
		 		map.put("senderAction", senderAction);
		 		map.put(WSSecurityConstants.USER, propertyConfigurator.getString("user"));
		 		map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("senderPWCallbackClass"));
		 		map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("senderSignaturePropFile").pathAsString());
		 		map.put(WSSecurityConstants.SIG_KEY_ID, propertyConfigurator.getString("signatureKeyIdentifier"));
		 	}else if(senderAction.equals("Encryption")){
		 		map.put("senderAction", WSSecurityConstants.ENCRYPT);
		 		map.put(WSSecurityConstants.ENCRYPTION_USER, propertyConfigurator.getString("encryptionUser"));
				map.put(WSSecurityConstants.ENC_PROP_FILE, propertyConfigurator.getResource("encryptionPropFile").pathAsString());
				map.put(WSSecurityConstants.ENC_KEY_ID, propertyConfigurator.getString("encryptionKeyIdentifier"));
		 	}else if(senderAction.equals("Encryption + Signature")){ 
		 		map.put("senderAction", WSSecurityConstants.ENCRYPT+" "+WSSecurityConstants.SIGNATURE);
		 		map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("senderSignaturePropFile").pathAsString());
		 		map.put(WSSecurityConstants.SIG_KEY_ID, propertyConfigurator.getString("signatureKeyIdentifier"));
				map.put(WSSecurityConstants.USER, propertyConfigurator.getString("user"));
				map.put(WSSecurityConstants.ENCRYPTION_USER, propertyConfigurator.getString("encryptionUser"));
				map.put(WSSecurityConstants.ENC_PROP_FILE, propertyConfigurator.getResource("encryptionPropFile").pathAsString());
				map.put(WSSecurityConstants.ENC_KEY_ID, propertyConfigurator.getString("encryptionKeyIdentifier"));
				map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("senderPWCallbackClass"));
		 	}
	 	}
	 	
	 	if(receiverAction != null){
			if(receiverAction.equals(WSSecurityConstants.SIGNATURE)){
				map.put("receiverAction", receiverAction);
				map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("receiverSignaturePropFile").pathAsString());
			}else if(receiverAction.equals("Encryption")){
				map.put("receiverAction", WSSecurityConstants.ENCRYPT);
				map.put(WSSecurityConstants.DEC_PROP_FILE, propertyConfigurator.getResource("decryptionPropFile").pathAsString());
				map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("receiverPWCallbackClass"));
			}else if(receiverAction.equals("Encryption + Signature")){
				map.put("receiverAction", WSSecurityConstants.ENCRYPT+" "+WSSecurityConstants.SIGNATURE);
				map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("receiverSignaturePropFile").pathAsString());
				map.put(WSSecurityConstants.DEC_PROP_FILE, propertyConfigurator.getResource("decryptionPropFile").pathAsString());
				map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("receiverPWCallbackClass"));
			}
		}
		
	 	protocolAdapter.setProperty(map);
	
		validate();
		
	}

	private void validate() {
		
		if (isSoap) {
			if (wsdl != null) {

				try {
					SAXReader reader = new SAXReader();
					Document wsdlDocument = reader.read(wsdl.openStream());
					Element rootElement = wsdlDocument.getRootElement();
					if (WSDLUtils.WSDL1_NAMESPACE.equals(rootElement
							.getNamespaceURI())) {
						//
						// It's a WSDL 1 Namespace
						//
						checkWsdl11();
					} else if (WSDLUtils.WSDL2_NAMESPACE.equals(rootElement
							.getNamespaceURI())) {
						//
						// It's a WSDL2 Namespace
						//
						checkWsdl2();
					} else {
						throw new RuntimeException(
								"Unrecognized wsdl namespace: "
										+ rootElement.getNamespaceURI());
					}
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
			} else {
				this.binding = new Wsdl1SoapBindingImpl(Soap11
						.getInstance());
			}
		}
	}
    
    
    public void checkWsdl2(){
		// USE WOODEN
	}
	public void checkWsdl11(){
        try{
        	WSDLFactory wsdlFactory = WSDLFactory.newInstance();
    		WSDLReader reader = wsdlFactory.newWSDLReader();
    		Definition def = reader.readWSDL(wsdl.asURL().toString());
    		
    		
            Service svc = null;
            if (getService() != null) {
                svc = def.getService(getService());
                if (svc == null) {
                    throw new RuntimeException("Could not find service '" + getService() + "' in wsdl");
                }
            } else if (def.getServices().size() == 1) {
                svc = (Service)def.getServices().values().iterator().next();
                setService(svc.getQName());
            } else {
                throw new RuntimeException("If service is not set, the WSDL must contain a single service definition");
            }
            Port port;
            if (getPort() != null) {
                port = svc.getPort(getPort());
                if (port == null) {
                    throw new RuntimeException("Cound not find port '" + getPort()
                                                  + "' in wsdl for service '" + getService() + "'");
                }
            } else if (svc.getPorts().size() == 1) {
                port = (Port)svc.getPorts().values().iterator().next();
                setPort(port.getName());
            } else {
                throw new RuntimeException("If endpoint is not set, the WSDL service '" + getService()
                                              + "' must contain a single port definition");
            }
            SOAPAddress soapAddress = WSDLUtils.getExtension(port, SOAPAddress.class);
            /*
            if (soapAddress != null) {
            	((SOAPProtocolOutputAdapter)this.protocolAdapter).setLocationURI(this.locationURI);
            } else {
                SOAP12Address soap12Address = WSDLUtils.getExtension(port, SOAP12Address.class);
                if (soap12Address != null) {
                	((SOAPProtocolOutputAdapter) this.protocolAdapter).setLocationURI(this.locationURI);
                }
            }
            */
            this.binding = BindingFactory.createBinding(port);
           
        }catch (WSDLException e) {
			throw new RuntimeException(e);
		}
	}
	
	@Override
	public void start() throws Exception {

		super.start();
		
		jettyClient = new HttpClient();

		if (ssl){
			jettyClient = new SpagicHttpWithSSLClient(sslBean);
			
			jettyClient.setTrustStoreLocation(sslBean.getTrustStore());
			jettyClient.setTrustStorePassword(sslBean.getTrustStorePassword());
			if(sslBean.isNeedClientAuth()){
				jettyClient.setKeyManagerPassword(sslBean.getKeyPassword());
				jettyClient.setKeyStoreLocation(sslBean.getKeyStore());
				jettyClient.setKeyStorePassword(sslBean.getKeyStorePassword());	
			}
		}
		
        jettyClient.setThreadPool(new QueuedThreadPool(jettyClientThreadPoolSize));
        jettyClient.setConnectorType(HttpClient.CONNECTOR_SELECT_CHANNEL);
        
        if (proxyHost != null) {
            jettyClient.setProxy(new Address(proxyHost, proxyPort));
            if (proxyUsername != null) {
                jettyClient.setProxyAuthentication(new ProxyAuthorization(proxyUsername, proxyPassword));
            }
        }
        jettyClient.setSoTimeout((int)this.clientTimeout);
        
		
		if (this.jettyClient != null)
			this.jettyClient.start();
		
	}
	
	
	@Override
	public void stop() throws Exception {
		super.stop();
		
		if (this.jettyClient != null)
			jettyClient.stop();
		
	}
	@Override
	public void process(Exchange exchange) {
		 Message nm = null;
		if (exchange.getStatus() == Status.Active) { 
			if (isPipeline && exchange.getPattern() != Pattern.InOnly){
				throw new RuntimeException("If isPipeline flag is true exchange must be an InOnly Exchange");
			}
		 
			nm = exchange.getIn(false);
	        
	        if (nm == null) {
	                throw new IllegalStateException("Exchange has no input message");
	        }  
	            
	        SpagicJettyHTTPExchange spagicJettyHTTPExchange = new NMREchangeAwareSpagicJettyExchange(exchange);
	        protocolAdapter.fillJettyExchange(exchange, spagicJettyHTTPExchange,this.propertyConfigurator, this.binding);

	        try{
	        	this.jettyClient.send(spagicJettyHTTPExchange);
	        	if (ExchangeUtils.isSync(exchange) && ExchangeUtils.isInAndOut(exchange)){
	        		logger.debug("Exchange is Synchronous Waiting Response");
	        		spagicJettyHTTPExchange.waitForDone();
	        		logger.debug("Exchange Response Arrived");
	        		handleResponse(spagicJettyHTTPExchange, exchange);
	        	}
	        }catch (Throwable e) {
	        	System.out.println(e);
				throw new IllegalStateException(e.getMessage(), e);
			}finally{
				if (nm != null)
					nm.finalizeAttachments();
			}
	    }

		
	}
	
	private  SslBean getSslParameters(){
		IResource keyStoreResource = propertyConfigurator.getResource("keystore", null);
		String keyStorePassword = propertyConfigurator.getString("keystorePassword",null);
		String keyStoreType = propertyConfigurator.getString("keystoreType", "JKS");
		
		IResource trustStoreResource = propertyConfigurator.getResource("truststore", null);
		String trustStorePassword = propertyConfigurator.getString("truststorePassword");
		String trustStoreType = propertyConfigurator.getString("truststoreType", "JKS");
		
		String clientAuthentication = propertyConfigurator.getString("clientAuthentication", "NoClientAuthentication");
		SslBean sslBean = new SslBean();
		
		if (clientAuthentication.equalsIgnoreCase("NoClientAuthentication")){
			sslBean.setNeedClientAuth(false);
			sslBean.setWantClientAuth(false);
		}else if (clientAuthentication.equalsIgnoreCase("OptionalClientAuthentication")){
			sslBean.setNeedClientAuth(false);
			sslBean.setWantClientAuth(true);
		}else if (clientAuthentication.equalsIgnoreCase("MandatoryClientAuthentication")){
			sslBean.setNeedClientAuth(true);
			sslBean.setWantClientAuth(true);
			
			//Keystore needed only if NeedClientAuth=TRUE
			sslBean.setKeyStore(keyStoreResource.asURL().toString());
			sslBean.setKeyStorePassword(keyStorePassword);
			sslBean.setKeyStoreType(keyStoreType);
			sslBean.setKeyPassword(keyStorePassword);
		}

		sslBean.setTrustStore(trustStoreResource.asURL().toString());
		sslBean.setTrustStorePassword(trustStorePassword);
		sslBean.setTrustStoreType(trustStoreType);
		
		
		return sslBean;
	}
	
	protected void handleResponse(SpagicJettyHTTPExchange spagicJettyContentExchange, Exchange nmrExchange) {
        try {
            protocolAdapter.handleResponse(nmrExchange,spagicJettyContentExchange,this.propertyConfigurator, this.binding);
        } catch (Exception e) {
        	nmrExchange.setStatus(Status.Error);
            nmrExchange.setError(e);
        }
        configureForResponse(nmrExchange);
        if (nmrExchange.getPattern() == Pattern.InOut){
        	send(nmrExchange);
        }else{
        	// It was an InOnly Exchange send back the done
        	nmrExchange.setStatus(Status.Done);
        	send(nmrExchange);
        }
        
       
    }

    protected void handleException(SpagicJettyHTTPExchange spagicJettyContentExchange, Exchange nmrExchange, Throwable ex)  {
    	try {
    		protocolAdapter.handleException(nmrExchange,spagicJettyContentExchange,ex, this.propertyConfigurator, binding);
        } catch (Exception e) {
        	nmrExchange.setStatus(Status.Error);
            nmrExchange.setError(e);
        }
        if (!ExchangeUtils.isSync(nmrExchange)){
        	configureForResponse(nmrExchange);
        	if (nmrExchange.getPattern() == Pattern.InOut){
        		send(nmrExchange);
        	}else{
        		// It was an InOnly Exchange send back the done
        		nmrExchange.setStatus(Status.Done);
        		send(nmrExchange);
        	}
    	}
    }

    
    protected class NMREchangeAwareSpagicJettyExchange extends SpagicJettyHTTPExchange {
        private Exchange nmrExchange;

        public NMREchangeAwareSpagicJettyExchange(Exchange nmrExchange) {
            this.nmrExchange = nmrExchange;
        }

        protected void onResponseComplete() throws IOException {
        	if (!ExchangeUtils.isSync(nmrExchange))
        		handleResponse(this, nmrExchange);
        }

        protected void onConnectionFailed(Throwable throwable) {
        	if (!ExchangeUtils.isSync(nmrExchange))
        		handleException(this, nmrExchange, throwable);
        }

        protected void onException(Throwable throwable) {
        	if (!ExchangeUtils.isSync(nmrExchange))
        		handleException(this, nmrExchange, throwable);
        }
    }
    
    public QName getService() {
		return service;
	}
	public void setService(QName service) {
		this.service = service;
	}
	public String getPort() {
		return port;
	}
	public void setPort(String port) {
		this.port = port;
	}

	
	public void setMarshaler(IHTTPOutputProtocolAdapter marshaler) {
		this.marshalers.put(marshaler.getAdapterId(), marshaler);
	}

	public void unsetMarshaler(IHTTPOutputProtocolAdapter marshaler) {
		this.marshalers.remove(marshaler.getAdapterId());  }
    

}
