/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024-2025 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

import org.w3c.dom.Document
import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.testing.test
import io.ktor.client.engine.mock.*
import io.ktor.http.*
import io.ktor.http.content.*
import org.junit.Test
import tech.libeufin.nexus.cli.LibeufinNexus
import tech.libeufin.nexus.ebics.*
import tech.libeufin.nexus.*
import tech.libeufin.common.*
import tech.libeufin.common.crypto.CryptoUtil
import kotlin.io.path.*
import kotlin.test.*
import java.security.interfaces.RSAPrivateCrtKey
import java.security.interfaces.RSAPublicKey
import java.time.LocalDate

private object BadRequest: Exception()

class EbicsState {
    private val bankSignKey: RSAPrivateCrtKey = CryptoUtil.genRSAPrivate(2048)
    private val bankEncKey: RSAPrivateCrtKey = CryptoUtil.genRSAPrivate(2048)
    private val bankAuthKey: RSAPrivateCrtKey = CryptoUtil.genRSAPrivate(2048)

    private var clientSignPub: RSAPublicKey? = null
    private var clientEncrPub: RSAPublicKey? = null
    private var clientAuthPub: RSAPublicKey? = null

    private var transactionId: String? = null

    companion object {
        private val HEV_OK = XmlBuilder.toBytes("ebicsHEVResponse") {
            attr("xmlns", "http://www.ebics.org/H000")
            el("SystemReturnCode") {
                el("ReturnCode", "000000")
                el("ReportText", "[EBICS_OK] OK")
            }
            el("VersionNumber") {
                attr("ProtocolVersion", "H005")
                text("03.00")
            }
        }
        private val KEY_OK = XmlBuilder.toBytes("ebicsKeyManagementResponse") {
            attr("xmlns", "http://www.ebics.org/H005")
            el("header") {
                attr("authenticate", "true")
                el("mutable") {
                    el("ReturnCode", "000000")
                    el("ReportText", "[EBICS_OK] OK")
                }
            }
            el("body") {
                el("ReturnCode") {
                    attr("authenticate", "true")
                    text("000000")
                }
            }
        }

        private fun parseUnsecureRequest(body: String, order: String, root: String, parse: XmlDestructor.() -> Unit) {
            XmlDestructor.parse(body, "ebicsUnsecuredRequest") {
                val adminOrder = one("header").one("static").one("OrderDetails").one("AdminOrderType").text()
                assertEquals(adminOrder, order)
                val chunk = one("body").one("DataTransfer").one("OrderData").base64()
                val deflated = chunk.inputStream().inflate()
                XmlDestructor.parse(deflated, root) { parse() }
            }
        }
    }

    private fun signedResponse(doc: Document): ByteArray {
        XMLUtil.signEbicsDocument(doc, bankAuthKey)
        return XMLUtil.convertDomToBytes(doc)
    }

    private fun ebicsResponsePayload(payload: ByteArray, last: Boolean = true): ByteArray {
        transactionId = randEbicsId()
        val deflated = payload.inputStream().deflate()
        val (transactionKey, encryptedTransactionKey) = CryptoUtil.genEbicsE002Key(clientEncrPub!!)
        val encrypted = CryptoUtil.encryptEbicsE002(transactionKey, deflated)
        val doc = XmlBuilder.toDom("ebicsResponse", "http://www.ebics.org/H005") {
            attr("http://www.w3.org/2000/xmlns/", "xmlns", "http://www.ebics.org/H005")
            el("header") {
                attr("authenticate", "true")
                el("static") {
                    el("TransactionID", transactionId!!)
                    el("NumSegments", "1")
                }
                el("mutable") {
                    el("TransactionPhase", "Initialisation")
                    el("SegmentNumber") {
                        attr("lastSegment", last.toString())
                        text("1")
                    }
                    el("ReturnCode", "000000")
                    el("ReportText", "[EBICS_OK] OK")
                }
            }
            el("AuthSignature")
            el("body") {
                el("DataTransfer") {
                    el("DataEncryptionInfo") {
                        attr("authenticate", "true")
                        el("EncryptionPubKeyDigest") {
                            attr("Version", "E002")
                            attr("Algorithm", "http://www.w3.org/2001/04/xmlenc#sha256")
                            text(CryptoUtil.getEbicsPublicKeyHash(clientEncrPub!!).encodeBase64())
                        }
                        el("TransactionKey", encryptedTransactionKey.encodeBase64())
                    }
                    el("OrderData", encrypted.encodeBase64())
                }
                el("ReturnCode") {
                    attr("authenticate", "true")
                    text("000000")
                }
            }
        }
        return signedResponse(doc)
    }

    private fun ebicsResponseNoData(): ByteArray {
        val doc = XmlBuilder.toDom("ebicsResponse", "http://www.ebics.org/H005") {
            attr("http://www.w3.org/2000/xmlns/", "xmlns", "http://www.ebics.org/H005")
            el("header") {
                attr("authenticate", "true")
                el("static")
                el("mutable") {
                    el("TransactionPhase", "Initialisation")
                    el("ReturnCode", "000000")
                    el("ReportText", "[EBICS_OK] OK")
                }
            }
            el("AuthSignature")
            el("body") {
                el("ReturnCode") {
                    attr("authenticate", "true")
                    text("090005")
                }
            }
        }
        return signedResponse(doc)
    }

    fun hev(body: String): ByteArray {
        // Parse HEV request
        val hostId = XmlDestructor.parse(body, "ebicsHEVRequest") {
            one("HostID").text()
        }
        return HEV_OK
    }

    fun ini(body: String): ByteArray {
        parseUnsecureRequest(body, "INI", "SignaturePubKeyOrderData") {
            clientSignPub = one("SignaturePubKeyInfo") {
                val version = one("SignatureVersion").text()
                assertEquals(version, "A006")
                rsaPubKey()
            }
        }
        return KEY_OK
    }

    fun hia(body: String): ByteArray {
        parseUnsecureRequest(body, "HIA", "HIARequestOrderData") {
            clientAuthPub = one("AuthenticationPubKeyInfo") {
                val version = one("AuthenticationVersion").text()
                assertEquals(version, "X002")
                rsaPubKey()
            }
            clientEncrPub = one("EncryptionPubKeyInfo") {
                val version = one("EncryptionVersion").text()
                assertEquals(version, "E002")
                rsaPubKey()
            }
        }
        return KEY_OK
    }

    fun hpb(body: String): ByteArray {
        // Parse HPB request
        XmlDestructor.parse(body, "ebicsNoPubKeyDigestsRequest") {
            val order = one("header").one("static").one("OrderDetails").one("AdminOrderType").text()
            assertEquals(order, "HPB")
        }

        val payload = XmlBuilder.toBytes("HPBResponseOrderData") {
            el("AuthenticationPubKeyInfo") {
                el("PubKeyValue") {
                    el("RSAKeyValue") {
                        el("Modulus", bankAuthKey.modulus.encodeBase64())
                        el("Exponent", bankAuthKey.publicExponent.encodeBase64())
                    }
                }
                el("AuthenticationVersion", "X002")
            }
            el("EncryptionPubKeyInfo") {
                el("PubKeyValue") {
                    el("RSAKeyValue") {
                        el("Modulus", bankEncKey.modulus.encodeBase64())
                        el("Exponent", bankEncKey.publicExponent.encodeBase64())
                    }
                }
                el("EncryptionVersion", "E002")
            }
        }.inputStream().deflate()

        val (transactionKey, encryptedTransactionKey) = CryptoUtil.genEbicsE002Key(clientEncrPub!!)
        val encrypted = CryptoUtil.encryptEbicsE002(transactionKey, payload)
        
        return XmlBuilder.toBytes("ebicsKeyManagementResponse") {
            attr("xmlns:ds", "http://www.w3.org/2000/09/xmldsig#")
            attr("xmlns", "http://www.ebics.org/H005")
            el("header") {
                attr("authenticate", "true")
                el("mutable") {
                    el("ReturnCode", "000000")
                    el("ReportText", "[EBICS_OK] OK")
                }
            }
            el("body") {
                el("DataTransfer") {
                    el("DataEncryptionInfo") {
                        attr("authenticate", "true")
                        el("EncryptionPubKeyDigest") {
                            attr("Version", "E002")
                            attr("Algorithm", "http://www.w3.org/2001/04/xmlenc#sha256")
                            text(CryptoUtil.getEbicsPublicKeyHash(clientEncrPub!!).encodeBase64())
                        }
                        el("TransactionKey", encryptedTransactionKey.encodeBase64())
                    }
                    el("OrderData", encrypted.encodeBase64())
                }
                el("ReturnCode") {
                    attr("authenticate", "true")
                    text("000000")
                }
            }
        }
    }

    private fun receipt(body: String, ok: Boolean): ByteArray {
        XmlDestructor.parse(body, "ebicsRequest") {
            one("header") {
                val id = one("static").one("TransactionID").text()
                assertEquals(id, transactionId)
                val phase = one("mutable").one("TransactionPhase").text()
                assertEquals(phase, "Receipt")
            }
            val code = one("body").one("TransferReceipt").one("ReceiptCode").text()
            assertEquals(code, if (ok) { "0" } else { "1" })
        }
        val response = signedResponse(XmlBuilder.toDom("ebicsResponse", "http://www.ebics.org/H005") {
            attr("http://www.w3.org/2000/xmlns/", "xmlns", "http://www.ebics.org/H005")
            el("header") {
                attr("authenticate", "true")
                el("static") {
                    el("TransactionID", transactionId!!)
                }
                el("mutable") {
                    el("TransactionPhase", "Receipt")
                    el("ReturnCode", "000000")
                    el("ReportText", "[EBICS_OK] OK")
                }
            }
            el("AuthSignature")
            el("body") {
                el("ReturnCode") {
                    attr("authenticate", "true")
                    text("000000")
                }
            }
        })
        transactionId = null
        return response
    }

    fun receiptOk(body: String): ByteArray = receipt(body, true)
    fun receiptErr(body: String): ByteArray = receipt(body, false)

    fun hkd(body: String): ByteArray {
        XmlDestructor.parse(body, "ebicsRequest") {
            one("header") {
                val adminOrder = one("static").one("OrderDetails").one("AdminOrderType").text()
                assertEquals(adminOrder, "HKD")
                val phase = one("mutable").one("TransactionPhase").text()
                assertEquals(phase, "Initialisation")
            }
        }
        return ebicsResponsePayload(
            XmlBuilder.toBytes("HKDResponseOrderData") {
                el("PartnerInfo") {
                    el("AddressInfo")
                    el("OrderInfo") {
                        el("AdminOrderType", "BTD")
                        el("Service") {
                            el("ServiceName", "STM")
                            el("Scope", "CH")
                            el("Container") {
                                attr("containerType", "ZIP")
                            }
                            el("MsgName") {
                                attr("version", "08")
                                text("camt.052")
                            }
                        }
                        el("Description")
                    }
                }
            }
        )
    }

    fun haa(body: String): ByteArray {
        XmlDestructor.parse(body, "ebicsRequest") {
            one("header") {
                val adminOrder = one("static").one("OrderDetails").one("AdminOrderType").text()
                assertEquals(adminOrder, "HAA")
                val phase = one("mutable").one("TransactionPhase").text()
                assertEquals(phase, "Initialisation")
            }
        }
        return ebicsResponsePayload(
            XmlBuilder.toBytes("HAAResponseOrderData") {
                el("Service") {
                    el("ServiceName", "STM")
                    el("Scope", "CH")
                    el("Container") {
                        attr("containerType", "ZIP")
                    }
                    el("MsgName") {
                        attr("version", "08")
                        text("camt.052")
                    }
                }
            }
        )
    }

    private fun btdDateCheck(body: String, pinned: LocalDate?): ByteArray {
        XmlDestructor.parse(body, "ebicsRequest") {
            one("header") {
                one("static").one("OrderDetails") {
                    val adminOrder = one("AdminOrderType").text()
                    assertEquals(adminOrder, "BTD")
                    val start = one("BTDOrderParams").opt("DateRange")?.opt("Start")?.date()
                    assertEquals(start, pinned)
                }
                val phase = one("mutable").one("TransactionPhase").text()
                assertEquals(phase, "Initialisation")
            }
        }
        return ebicsResponseNoData()
    }

    fun btdNoData(body: String): ByteArray = btdDateCheck(body, null)
    fun btdNoDataNow(body: String): ByteArray = btdDateCheck(body, LocalDate.now())
    fun btdNoDataPinned(body: String): ByteArray = btdDateCheck(body, LocalDate.parse("2024-06-05"))

    fun badRequest(body: String): ByteArray {
        throw BadRequest
    }

    fun initializeTx(body: String): ByteArray = ebicsResponsePayload(ByteArray(0), false)

    fun failure(body: String): ByteArray {
        throw Exception("Not reachable")
    }
}

private fun CliktCommand.fail(cmd: String) {
    val result = test(cmd)
    require(result.statusCode != 0) { result }
}

private fun CliktCommand.succeed(cmd: String) {
    val result = test(cmd)
    require(result.statusCode == 0) { result }
}

@OptIn(kotlin.io.path.ExperimentalPathApi::class)
class EbicsTest {
    private val nexusCmd = LibeufinNexus()
    private val bank = EbicsState()
    private val args = "-L TRACE -c conf/fetch.conf"
    private val ebicsLogger = EbicsLogger(null).tx("test").step("step")

    private fun setMock(sequences: Sequence<(String) -> ByteArray>) {
        val steps = sequences.iterator()
        val cfg: MockEngineConfig = MockEngineConfig()
        cfg.addHandler { req ->
            val body = String((req.body as OutgoingContent.ByteArrayContent).bytes())
            val handler = steps.next()
            try {
                val res = handler(body)
                respond(res)
            } catch (e: BadRequest) {
                respondBadRequest()
            }
        }
        MOCK_ENGINE = MockEngine(cfg)
    }

    private fun ebicsSetup() {
        // Reset current keys
        val dir = Path("test/fetch")
        dir.deleteRecursively()
        dir.createDirectories()

        // Set setup mock
        setMock(sequence {
            yield(bank::hev)
            yield(bank::ini)
            yield(bank::hia)
            yield(bank::hpb)
            yield(bank::hkd)
            yield(bank::receiptOk)
        })

        // Run setup
        nexusCmd.succeed("ebics-setup $args --auto-accept-keys")
    }

    // POSTs an EBICS message to the mock bank.  Tests
    // the main branches: unreachable bank, non-200 status
    // code, and 200.
    @Test
    fun postMessage() = conf { config ->
        assertFailsWith<EbicsError.HTTP> {
            getMockedClient {
                respondError(HttpStatusCode.NotFound)
            }.postToBank("http://ignored.example.com/", ByteArray(0), "Test", ebicsLogger)
        }.run {
            assertEquals(HttpStatusCode.NotFound, status)
            assertEquals("Test: bank HTTP error: 404 Not Found", message)
        }
        assertFailsWith<EbicsError.Network> {
            getMockedClient {
                throw Exception("Simulate failure")
            }.postToBank("http://ignored.example.com/", ByteArray(0), "Test", ebicsLogger)
        }.run {
            assertEquals("Test: failed to contact bank", message)
            assertEquals("Simulate failure", cause!!.message)
        }
        assertFailsWith<EbicsError.Protocol> {
            getMockedClient {
                respondOk("<ebics broken></ebics>")
            }.postToBank("http://ignored.example.com/", ByteArray(0), "Test", ebicsLogger)
        }.run {
            assertEquals("Test: invalid XML bank response", message)
            assertEquals("Attribute name \"broken\" associated with an element type \"ebics\" must be followed by the ' = ' character.", cause!!.message)
        }
        getMockedClient {
            respondOk("<ebics></ebics>")
        }.postToBank("http://ignored.example.com/", ByteArray(0), "Test", ebicsLogger)
    }

    // Tests that internal repr. of keys lead to valid PDF.
    // Mainly tests that the function does not throw any error.
    @Test
    fun keysPdf() = conf { config -> 
        val pdf = generateKeysPdf(clientKeys, config.ebics)
        Path("/tmp/libeufin-nexus-test-keys.pdf").writeBytes(pdf)
    }

    @Test
    fun setup() = setup { _, _ ->
        ebicsSetup()
    }

    @Test
    fun fetchPinnedDate() = setup { db, _ ->
        ebicsSetup()

        suspend fun resetCheckpoint() {
            db.serializable("DELETE FROM kv WHERE key=?") {
                bind(CHECKPOINT_KEY)
                executeUpdate()
            }
        }

        // Default transient
        setMock(sequence {
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.succeed("ebics-fetch $args --transient")

        // Pinned transient
        setMock(sequence {
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoDataPinned)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --pinned-start 2024-06-05")

        // Init checkpoint
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint")

        // Default checkpoint
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoDataNow)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint")

        // Pinned checkpoint
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoDataPinned)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint --pinned-start 2024-06-05")

        // Reset checkpoint
        resetCheckpoint()
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint")

        // Reset checkpoint pinned
        resetCheckpoint()
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoDataPinned)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint --pinned-start 2024-06-05")
    }

    @Test
    fun closePendingTransaction() = setup { db, _ ->
        ebicsSetup()

        // Failure before first segment
        setMock(sequence {
            // Failure to perform download
            yield(bank::failure)
            // Then continue
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.succeed("ebics-fetch $args --transient")

        // Compliant server
        setMock(sequence {
            yield(bank::haa)
            yield(bank::receiptOk)
            // Failure to perform download
            yield(bank::initializeTx)
            yield(bank::failure)
            // Retry fail once
            yield(bank::failure)
            // Retry fail twice
            yield(bank::failure)
            // Retry succeed
            yield(bank::receiptErr)
            // Then continue
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.succeed("ebics-fetch $args --transient")

        // Non compliant server
        setMock(sequence {
            yield(bank::haa)
            yield(bank::receiptOk)
            // Failure to perform download
            yield(bank::initializeTx)
            yield(bank::badRequest)
            // Retry fail
            yield(bank::failure)

            // Retry succeed
            yield(bank::badRequest)
            // Then continue
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.succeed("ebics-fetch $args --transient")
    }
}