/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
import org.gnu.glib.Type;

/**
 * The FontSelectionDialog widget is a dialog for selecting a font.
 * Filters can be used to limit the fonts shown.  There are 2 filters
 * in the FileSelectionDialog - a base filter and a user filter.  The
 * base filter cannot be changed by the user, so this can be used when
 * the user must choose from the restricted set of fonts.  The user filter
 * can be changed or reset by the user, using the 'Reset Filter' button
 * or changing the options on the 'Filter' page of the dialog.
 */
public class FontSelectionDialog extends Dialog {

	/**
	 * Construct a new FontSelectionDialog object.
	 * 
	 * @param title The title for the dialog.
	 */
	public FontSelectionDialog(String title) {
		super(gtk_font_selection_dialog_new(title));
	}
	
	/**
	 * Construct a FontSelectionDialog using a handle to a native resource.
	 */
	public FontSelectionDialog(Handle handle) {
		super(handle);
	}

	/**
	 * Gets the currently selected font name.
	 * 
	 * @return The name of the currently selected font.
	 */
	public String getFontName() {
		return FontSelectionDialog.gtk_font_selection_dialog_get_font_name(getHandle());
	}
	
	/**
	 * Sets the currently selected font.
	 * 
	 * @param 	fontName The name of the font to set as the currently selected font.
	 */
	public void setFontname(String fontName) {
		FontSelectionDialog.gtk_font_selection_dialog_set_font_name(getHandle(), fontName);
	}
	
	/**
	 * Sets the text displayed in the preview area.
	 * 
	 * @param text The text to display in the preview area.
	 */
	public void setPreviewText(String text) {
		FontSelectionDialog.gtk_font_selection_dialog_set_preview_text(getHandle(), text);
	}
	
	/**
	 * Return the OK Button widget for this Dialog.
	 * 
	 * @return The OK Button.
	 */
	public Button getOKButton() {
	    Handle hndl = FontSelectionDialog.getOkButton(getHandle());
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	
	/**
	 * Return the Apply Button widget for this Dialog.
	 * 
	 * @return The Apply Button.
	 */
	public Button getApplyButton() {
	    Handle hndl = FontSelectionDialog.getApplyButton(getHandle());
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	
	/**
	 * Return the Cancel Button widget for this dialog.
	 * 
	 * @return The Cancel Button.
	 */
	public Button getCancelButton() {
	    Handle hndl = FontSelectionDialog.getCancelButton(getHandle());
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_font_selection_dialog_get_type());
	}


	native static final protected Handle getFontsel(Handle cptr);
	native static final protected Handle getMainVbox(Handle cptr);
	native static final protected Handle getOkButton(Handle cptr);
	native static final protected Handle getApplyButton(Handle cptr);
	native static final protected Handle getCancelButton(Handle cptr);
	native static final protected int getDialogWidth(Handle cptr);
	native static final protected boolean getAutoResize(Handle cptr);
	native static final protected int gtk_font_selection_dialog_get_type();
	native static final protected Handle gtk_font_selection_dialog_new(String title);
	native static final protected String gtk_font_selection_dialog_get_font_name(Handle fontsel);
	native static final protected boolean gtk_font_selection_dialog_set_font_name(Handle fontsel, String fontname);
	native static final protected String gtk_font_selection_dialog_get_preview_text(Handle fontsel);
	native static final protected void gtk_font_selection_dialog_set_preview_text(Handle fontsel, String text);

}
