/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gdk;

import org.gnu.glib.GObject;
import org.gnu.glib.Handle;

/**
 */
public class Display extends GObject {

	public Display() {
		super(gdk_display_get_default());
	}
	
	public Display(Handle hndl) {
		super(hndl);
	}
	
	public static Display open(String name) {
	    Handle hndl = gdk_display_open(name);
		return new Display(hndl);
	}
	
	public String getName() {
		return gdk_display_get_name(getHandle());
	}
	
	public int getNumScreens() {
		return gdk_display_get_n_screens(getHandle());
	}
	
	public Screen getScreen(int screenNumber) {
	    Handle hndl = gdk_display_get_screen(getHandle(), screenNumber);
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Screen)obj;
		return new Screen(hndl);
	}
	
	public Screen getDefaultScreen() {
	    Handle hndl = gdk_display_get_default_screen(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Screen)obj;
		return new Screen(hndl);
	}
	
	public void ungrabPointer() {
		gdk_display_pointer_ungrab(getHandle(), 0);
	}
	
	public void ungrabKeyboard() {
		gdk_display_keyboard_ungrab(getHandle(), 0);
	}
	
	public boolean pointerIsGrabbed() {
		return gdk_display_pointer_is_grabbed(getHandle());
	}
	
	public void beep() {
		gdk_display_beep(getHandle());
	}
	
	public void sync() {
		gdk_display_sync(getHandle());
	}
	
	public void flush() {
		gdk_display_flush(getHandle());
	}
	
	public void close() {
		gdk_display_close(getHandle());
	}
	
	public Device[] listDevice() {
	    Handle[] hndls = gdk_display_list_devices(getHandle());
		if (null == hndls)
			return null;
		Device[] devices = new Device[hndls.length];
		for (int i = 0; i < hndls.length; i++) {
			GObject obj = retrieveGObject(hndls[i]);
			if (null != obj)
				devices[i] = (Device)obj;
			else
				devices[i] = new Device(hndls[i]);
		}
		return devices;
	}

	public Device getCorePointer() {
	    Handle hndl = gdk_display_get_core_pointer(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Device)obj;
		return new Device(hndl);
	}
	
	public Window getWindowAtPointer() {
	    Handle hndl = gdk_display_get_window_at_pointer(getHandle(), 0, 0);
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Window)obj;
		return new Window(hndl);
	}

	public boolean supportsCursorAlpha() {
		return gdk_display_supports_cursor_alpha(getHandle());
	}
	
	public boolean supportsCursorColor() {
		return gdk_display_supports_cursor_color(getHandle());
	}

	public int getDefaultCursorSize() {
		return gdk_display_get_default_cursor_size(getHandle());
	}

	public boolean supportsSelectionNotification() {
		return gdk_display_supports_selection_notification(getHandle());
	}
	
	public boolean requestSelectionNotification(Atom atom) {
		return gdk_display_request_selection_notification(getHandle(), atom.getHandle());
	}
	 
	public boolean supportsClipboardPersistence() {
		return gdk_display_supports_clipboard_persistence(getHandle());
	}
	
	public void storeClipboard(Window clipboardWindow, Atom[] targets) {
		Handle[] tar = new Handle[targets.length];
		for (int i = 0; i < targets.length; i++) 
			tar[i] = targets[i].getHandle();
		gdk_display_store_clipboard(getHandle(), clipboardWindow.getHandle(), 0, tar);
	}
	

	native static final protected int gdk_display_get_type();
	native static final protected Handle gdk_display_open(String displayName);
	native static final protected String gdk_display_get_name(Handle display);
	native static final protected int gdk_display_get_n_screens(Handle display);
	native static final protected Handle gdk_display_get_screen(Handle display, int screenNum);
	native static final protected Handle gdk_display_get_default_screen(Handle display);
	native static final protected void gdk_display_pointer_ungrab(Handle display, int time);
	native static final protected void gdk_display_keyboard_ungrab(Handle display, int time);
	native static final protected boolean gdk_display_pointer_is_grabbed(Handle display);
	native static final protected void gdk_display_beep(Handle display);
	native static final protected void gdk_display_sync(Handle display);
	native static final protected void gdk_display_flush(Handle display);
	native static final protected void gdk_display_close(Handle display);
	native static final protected Handle[] gdk_display_list_devices(Handle display);
	native static final protected Handle gdk_display_get_event(Handle display);
	native static final protected Handle gdk_display_peek_event(Handle display);
	native static final protected void gdk_display_put_event(Handle display, Handle event);
	native static final protected Handle gdk_display_get_default();
	native static final protected Handle gdk_display_get_core_pointer(Handle display);
	native static final protected void gdk_display_get_pointer(Handle display, Handle screen, int[] x, int[] y, int[] modType);
	native static final protected Handle gdk_display_get_window_at_pointer(Handle display, int x, int y);
	native static final protected boolean gdk_display_supports_cursor_alpha(Handle display);
	native static final protected boolean gdk_display_supports_cursor_color(Handle display);
	native static final protected int gdk_display_get_default_cursor_size(Handle display);
	native static final protected void gdk_display_get_maximum_cursor_size(Handle display, int[] width, int[] height);
	// from gtk 2.5
	native static final protected boolean gdk_display_supports_selection_notification(Handle display); 
	native static final protected boolean gdk_display_request_selection_notification(Handle display, Handle selection);
	native static final protected boolean gdk_display_supports_clipboard_persistence(Handle display);
	native static final protected void gdk_display_store_clipboard(Handle display, Handle clipboardWindow, int time, Handle[] targets);
}
