/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gtk;

import java.util.ArrayList;
import java.util.List;

import org.gnu.glib.GObject;
import org.gnu.gtk.event.RadioActionEntryListener;

/**
 */
public class ActionGroup extends GObject {
	
	/**
	 * Construct a new ActionGroup object. 
	 * @param name Used when associating keybindings with the actions.
	 */
	public ActionGroup(String name) {
		handle = gtk_action_group_new(name);
	}
	
	ActionGroup(int hndl) {
		this.handle = hndl;
	}
	
	/**
	 * Get the name of the action group.
	 * @return the name of the action group
	 */
	public String getName() {
		return gtk_action_group_get_name(handle);
	}
	
	/**
	 * Look up an Action in the ActionGroup by name.
	 * @param actionName
	 * @return the Action or null if no Action by that name exists.
	 */
	public Action getAction(String actionName) {
		int hndl = gtk_action_group_get_action(handle, actionName);
		if (-1 == hndl)
			return null;
		return new Action(hndl);
	}
	
	/**
	 * Add an Action object to the ActionGroup
	 * @param anAction
	 */
	public void addAction(Action anAction) {
		gtk_action_group_add_action(handle, anAction.getHandle());
	}
	
	/**
	 * Add an array of ActonEntry objects to the ActionGroup
	 * @param entries
	 */
	public void addActions(ActionEntry[] entries) {
		int[] hndls = new int[entries.length];
		for (int i = 0; i < entries.length; i++) {
			hndls[i] = entries[i].getHandle();
		}
		addActions(handle, hndls, entries);
	}
	
	/**
	 * Add an array of ToggleActonEntry objects to the ActionGroup
	 * @param entries
	 */
	public void addToggleActions(ToggleActionEntry[] entries) {
		int[] hndls = new int[entries.length];
		for (int i = 0; i < entries.length; i++) {
			hndls[i] = entries[i].getHandle();
		}
		addToggleActions(handle, hndls, entries);
	}
	
	/**
	 * Add an array of RadioActionEntry objects to the ActionGroup and setup the event
	 * handling.  
	 * @param entries  
	 * @param listener
	 * @param initialValue
	 */
	public void addRadioActions(RadioActionEntry[] entries, int initialValue, RadioActionEntryListener listener) {
		int[] hndls = new int[entries.length];
		for (int i = 0; i < entries.length; i++) {
			hndls[i] = entries[i].getHandle();
		}
		addRadioActions(handle, hndls, initialValue, this);
	}
	
	/**
	 * Remove an Action object from the ActionGroup
	 * @param anAction
	 */
	public void removeAction(Action anAction) {
		gtk_action_group_remove_action(handle, anAction.getHandle());
	}
	
	/**
	 * List the Actions in the ActionGroup
	 * @return A list of Action objects.
	 */
	public List listActions() {
		int[] actions = gtk_action_group_list_actions(handle);
		List objs = new ArrayList();
		for (int i = 0; i < actions.length; i++) {
			objs.add(new Action(actions[i]));
		}
		return objs;
	}
	
	
	private void handleRadioAction(int action, int current) {
		
	}

	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_action_group_get_type ();
	native static final protected int gtk_action_group_new(String name);
	native static final protected String gtk_action_group_get_name(int group);
	native static final protected int gtk_action_group_get_action(int group, String actionName);
	native static final protected int[] gtk_action_group_list_actions(int group);
	native static final protected void gtk_action_group_add_action(int group, int action);
	native static final protected void gtk_action_group_remove_action(int group, int action);
	native static final protected void gtk_action_group_add_actions(int group, int[] entries, int numEntries);
	native static final protected void gtk_action_group_add_toggle_actions(int group, int[] entires, int numEntries);
	native static final protected void gtk_action_group_set_translation_domain(int group, String domain);
	native static final protected void gtk_action_group_add_action_with_accel(int group, int action, String name, String accel, String stockId);
	native static final protected void addActions(int group, int[] entries, Object[] cbObjs);
	native static final protected void addToggleActions(int group, int[] entries, Object[] cbObjs);
	native static final protected void addRadioActions(int group, int[] entries, int value, Object listener);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
