/* -*- mode: C; c-basic-offset: 4 -*- */
#include <pygobject.h>
#include "pygnomevfs-private.h"
#include <libgnomevfs/gnome-vfs-utils.h>

static GHashTable *monitor_hash;
static gint monitor_id_counter = 0;

static PyObject *pygnomevfs_exc;
static PyObject *pygnomevfs_not_found_exc;
static PyObject *pygnomevfs_generic_exc;
static PyObject *pygnomevfs_internal_exc;
static PyObject *pygnomevfs_bad_parameters_exc;
static PyObject *pygnomevfs_not_supported_exc;
static PyObject *pygnomevfs_io_exc;
static PyObject *pygnomevfs_corrupted_data_exc;
static PyObject *pygnomevfs_wrong_format_exc;
static PyObject *pygnomevfs_bad_file_exc;
static PyObject *pygnomevfs_too_big_exc;
static PyObject *pygnomevfs_no_space_exc;
static PyObject *pygnomevfs_read_only_exc;
static PyObject *pygnomevfs_invalid_uri_exc;
static PyObject *pygnomevfs_not_open_exc;
static PyObject *pygnomevfs_invalid_open_mode_exc;
static PyObject *pygnomevfs_access_denied_exc;
static PyObject *pygnomevfs_too_many_open_files_exc;
static PyObject *pygnomevfs_eof_exc;
static PyObject *pygnomevfs_not_a_directory_exc;
static PyObject *pygnomevfs_in_progress_exc;
static PyObject *pygnomevfs_interrupted_exc;
static PyObject *pygnomevfs_file_exists_exc;
static PyObject *pygnomevfs_loop_exc;
static PyObject *pygnomevfs_not_permitted_exc;
static PyObject *pygnomevfs_is_directory_exc;
static PyObject *pygnomevfs_no_memory_exc;
static PyObject *pygnomevfs_host_not_found_exc;
static PyObject *pygnomevfs_invalid_host_name_exc;
static PyObject *pygnomevfs_host_has_no_address_exc;
static PyObject *pygnomevfs_login_failed_exc;
static PyObject *pygnomevfs_cancelled_exc;
static PyObject *pygnomevfs_directory_busy_exc;
static PyObject *pygnomevfs_directory_not_empty_exc;
static PyObject *pygnomevfs_too_many_links_exc;
static PyObject *pygnomevfs_read_only_file_system_exc;
static PyObject *pygnomevfs_not_same_file_system_exc;
static PyObject *pygnomevfs_name_too_long_exc;
static PyObject *pygnomevfs_service_not_available_exc;
static PyObject *pygnomevfs_service_obsolete_exc;
static PyObject *pygnomevfs_protocol_error_exc;
static PyObject *pygnomevfs_no_master_browser_exc;
#if 0
static PyObject *pygnomevfs_no_default_exc;
static PyObject *pygnomevfs_no_handler_exc;
static PyObject *pygnomevfs_parse_exc;
static PyObject *pygnomevfs_launch_exc;
#endif

gboolean
pygnome_vfs_result_check(GnomeVFSResult result)
{
    switch(result)
    {
    case GNOME_VFS_OK:
	return FALSE;
    case GNOME_VFS_ERROR_NOT_FOUND:
	PyErr_SetObject(pygnomevfs_not_found_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_GENERIC:
	PyErr_SetObject(pygnomevfs_generic_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INTERNAL:
	PyErr_SetObject(pygnomevfs_internal_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_BAD_PARAMETERS:
	PyErr_SetObject(pygnomevfs_bad_parameters_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_SUPPORTED:
	PyErr_SetObject(pygnomevfs_not_supported_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_IO:
	PyErr_SetObject(pygnomevfs_io_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_CORRUPTED_DATA:
	PyErr_SetObject(pygnomevfs_corrupted_data_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_WRONG_FORMAT:
	PyErr_SetObject(pygnomevfs_wrong_format_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_BAD_FILE:
	PyErr_SetObject(pygnomevfs_bad_file_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_TOO_BIG:
	PyErr_SetObject(pygnomevfs_too_big_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NO_SPACE:
	PyErr_SetObject(pygnomevfs_no_space_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_READ_ONLY:
	PyErr_SetObject(pygnomevfs_read_only_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INVALID_URI:
	PyErr_SetObject(pygnomevfs_invalid_uri_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_OPEN:
	PyErr_SetObject(pygnomevfs_not_open_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INVALID_OPEN_MODE:
	PyErr_SetObject(pygnomevfs_invalid_open_mode_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_ACCESS_DENIED:
	PyErr_SetObject(pygnomevfs_access_denied_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES:
	PyErr_SetObject(pygnomevfs_too_many_open_files_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_EOF:
	PyErr_SetObject(pygnomevfs_eof_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_A_DIRECTORY:
	PyErr_SetObject(pygnomevfs_not_a_directory_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_IN_PROGRESS:
	PyErr_SetObject(pygnomevfs_in_progress_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INTERRUPTED:
	PyErr_SetObject(pygnomevfs_interrupted_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_FILE_EXISTS:
	PyErr_SetObject(pygnomevfs_file_exists_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_LOOP:
	PyErr_SetObject(pygnomevfs_loop_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_PERMITTED:
	PyErr_SetObject(pygnomevfs_not_permitted_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_IS_DIRECTORY:
	PyErr_SetObject(pygnomevfs_is_directory_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NO_MEMORY:
	PyErr_SetObject(pygnomevfs_no_memory_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_HOST_NOT_FOUND:
	PyErr_SetObject(pygnomevfs_host_not_found_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_INVALID_HOST_NAME:
	PyErr_SetObject(pygnomevfs_invalid_host_name_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS:
	PyErr_SetObject(pygnomevfs_host_has_no_address_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_LOGIN_FAILED:
	PyErr_SetObject(pygnomevfs_login_failed_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_CANCELLED:
	PyErr_SetObject(pygnomevfs_cancelled_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_DIRECTORY_BUSY:
	PyErr_SetObject(pygnomevfs_directory_busy_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY:
	PyErr_SetObject(pygnomevfs_directory_not_empty_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_TOO_MANY_LINKS:
	PyErr_SetObject(pygnomevfs_too_many_links_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
	PyErr_SetObject(pygnomevfs_read_only_file_system_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM:
	PyErr_SetObject(pygnomevfs_not_same_file_system_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NAME_TOO_LONG:
	PyErr_SetObject(pygnomevfs_name_too_long_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE:
	PyErr_SetObject(pygnomevfs_service_not_available_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_SERVICE_OBSOLETE:
	PyErr_SetObject(pygnomevfs_service_obsolete_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_PROTOCOL_ERROR:
	PyErr_SetObject(pygnomevfs_protocol_error_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NO_MASTER_BROWSER:
	PyErr_SetObject(pygnomevfs_no_master_browser_exc, NULL);
	return TRUE;
#if 0	
    case GNOME_VFS_ERROR_NO_DEFAULT:
	PyErr_SetObject(pygnomevfs_no_default_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_NO_HANDLER:
	PyErr_SetObject(pygnomevfs_no_handler_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_PARSE:
	PyErr_SetObject(pygnomevfs_parse_exc, NULL);
	return TRUE;
    case GNOME_VFS_ERROR_LAUNCH:
	PyErr_SetObject(pygnomevfs_launch_exc, NULL);
	return TRUE;
#endif	
    default:
	return FALSE;
    }
}

GnomeVFSResult pygnome_vfs_exception_check(void)
{
    if (!PyErr_Occurred()) {
	return -1;
    }
    
    if (PyErr_ExceptionMatches(pygnomevfs_not_found_exc)) {
	return GNOME_VFS_ERROR_NOT_FOUND;
    } else if (PyErr_ExceptionMatches(pygnomevfs_generic_exc)) {
	return GNOME_VFS_ERROR_GENERIC;
    } else if (PyErr_ExceptionMatches(pygnomevfs_internal_exc)) {
	return GNOME_VFS_ERROR_INTERNAL;
    } else if (PyErr_ExceptionMatches(pygnomevfs_bad_parameters_exc)) {
	return GNOME_VFS_ERROR_BAD_FILE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_supported_exc)) {
	return GNOME_VFS_ERROR_NOT_SUPPORTED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_io_exc)) {
	return GNOME_VFS_ERROR_IO;
    } else if (PyErr_ExceptionMatches(pygnomevfs_corrupted_data_exc)) {
	return GNOME_VFS_ERROR_CORRUPTED_DATA;
    } else if (PyErr_ExceptionMatches(pygnomevfs_wrong_format_exc)) {
	return GNOME_VFS_ERROR_WRONG_FORMAT;
    } else if (PyErr_ExceptionMatches(pygnomevfs_bad_file_exc)) {
	return GNOME_VFS_ERROR_BAD_FILE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_too_big_exc)) {
	return GNOME_VFS_ERROR_TOO_BIG;
    } else if (PyErr_ExceptionMatches(pygnomevfs_no_space_exc)) {
	return GNOME_VFS_ERROR_NO_SPACE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_read_only_exc)) {
	return GNOME_VFS_ERROR_READ_ONLY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_invalid_uri_exc)) {
	return GNOME_VFS_ERROR_INVALID_URI;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_open_exc)) {
	return GNOME_VFS_ERROR_NOT_OPEN;
    } else if (PyErr_ExceptionMatches(pygnomevfs_invalid_open_mode_exc)) {
	return GNOME_VFS_ERROR_INVALID_OPEN_MODE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_access_denied_exc)) {
	return GNOME_VFS_ERROR_ACCESS_DENIED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_too_many_open_files_exc)) {
	return GNOME_VFS_ERROR_TOO_MANY_OPEN_FILES;
    } else if (PyErr_ExceptionMatches(pygnomevfs_eof_exc)) {
	return GNOME_VFS_ERROR_EOF;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_a_directory_exc)) {
	return GNOME_VFS_ERROR_NOT_A_DIRECTORY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_in_progress_exc)) {
	return GNOME_VFS_ERROR_IN_PROGRESS;
    } else if (PyErr_ExceptionMatches(pygnomevfs_interrupted_exc)) {
	return GNOME_VFS_ERROR_INTERRUPTED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_file_exists_exc)) {
	return GNOME_VFS_ERROR_FILE_EXISTS;
    } else if (PyErr_ExceptionMatches(pygnomevfs_loop_exc)) {
	return GNOME_VFS_ERROR_LOOP;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_permitted_exc)) {
	return GNOME_VFS_ERROR_NOT_PERMITTED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_is_directory_exc)) {
	return GNOME_VFS_ERROR_IS_DIRECTORY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_no_memory_exc)) {
	return GNOME_VFS_ERROR_NO_MEMORY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_host_not_found_exc)) {
	return GNOME_VFS_ERROR_HOST_NOT_FOUND;
    } else if (PyErr_ExceptionMatches(pygnomevfs_invalid_host_name_exc)) {
	return GNOME_VFS_ERROR_INVALID_HOST_NAME;
    } else if (PyErr_ExceptionMatches(pygnomevfs_host_has_no_address_exc)) {
	return GNOME_VFS_ERROR_HOST_HAS_NO_ADDRESS;
    } else if (PyErr_ExceptionMatches(pygnomevfs_login_failed_exc)) {
	return GNOME_VFS_ERROR_LOGIN_FAILED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_cancelled_exc)) {
	return GNOME_VFS_ERROR_CANCELLED;
    } else if (PyErr_ExceptionMatches(pygnomevfs_directory_busy_exc)) {
	return GNOME_VFS_ERROR_DIRECTORY_BUSY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_directory_not_empty_exc)) {
	return GNOME_VFS_ERROR_DIRECTORY_NOT_EMPTY;
    } else if (PyErr_ExceptionMatches(pygnomevfs_too_many_links_exc)) {
	return GNOME_VFS_ERROR_TOO_MANY_LINKS;
    } else if (PyErr_ExceptionMatches(pygnomevfs_read_only_file_system_exc)) {
	return GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM;
    } else if (PyErr_ExceptionMatches(pygnomevfs_not_same_file_system_exc)) {
	return GNOME_VFS_ERROR_NOT_SAME_FILE_SYSTEM;
    } else if (PyErr_ExceptionMatches(pygnomevfs_name_too_long_exc)) {
	return GNOME_VFS_ERROR_NAME_TOO_LONG;
    } else if (PyErr_ExceptionMatches(pygnomevfs_service_not_available_exc)) {
	return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_service_obsolete_exc)) {
	return GNOME_VFS_ERROR_SERVICE_OBSOLETE;
    } else if (PyErr_ExceptionMatches(pygnomevfs_protocol_error_exc)) {
	return GNOME_VFS_ERROR_PROTOCOL_ERROR;
    } else if (PyErr_ExceptionMatches(pygnomevfs_no_master_browser_exc)) {
	return GNOME_VFS_ERROR_NO_MASTER_BROWSER;
    }
    
    return -2;
}

static PyObject *
pygvfs_create(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "open_mode", "exclusive", "perm", NULL };
    PyObject *uri;
    GnomeVFSOpenMode open_mode = GNOME_VFS_OPEN_NONE;
    gboolean exclusive = FALSE;
    guint perm = 0666;
    GnomeVFSHandle *handle;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "O|iii:gnome.vfs.create",
				     kwlist, &uri, &open_mode, &exclusive,
				     &perm))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_create_uri(&handle, pygnome_vfs_uri_get(uri),
				      open_mode, exclusive, perm);
    else if (PyString_Check(uri))
	result = gnome_vfs_create(&handle, PyString_AsString(uri),
				  open_mode, exclusive, perm);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    return pygnome_vfs_handle_new(handle);
}

static PyObject *
pygvfs_get_file_info(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "options", NULL };
    PyObject *uri;
    GnomeVFSFileInfo *finfo;
    GnomeVFSFileInfoOptions options = GNOME_VFS_FILE_INFO_DEFAULT;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O|i:gnome.vfs.get_file_info",
				     kwlist, &uri, &options))
	return NULL;

    finfo = gnome_vfs_file_info_new();
    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_get_file_info_uri(pygnome_vfs_uri_get(uri), finfo,
					     options);
    else if (PyString_Check(uri))
	result = gnome_vfs_get_file_info(PyString_AsString(uri), finfo,
					 options);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	gnome_vfs_file_info_unref(finfo);
	return NULL;
    }

    if (pygnome_vfs_result_check(result)) {
	gnome_vfs_file_info_unref(finfo);
	return NULL;
    }
    return pygnome_vfs_file_info_new(finfo);
}

static PyObject *
pygvfs_truncate(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "length", NULL };
    PyObject *uri, *py_length;
    GnomeVFSFileSize length;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "OO:gnome.vfs.truncate",
				     kwlist, &uri, &py_length))
	return NULL;

    length = PyLong_Check(py_length) ? PyLong_AsUnsignedLongLong(py_length)
	: PyInt_AsLong(py_length);
    if (PyErr_Occurred()) return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_truncate_uri(pygnome_vfs_uri_get(uri), length);
    else if (PyString_Check(uri))
	result = gnome_vfs_truncate(PyString_AsString(uri), length);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_make_directory(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "perm", NULL };
    PyObject *uri;
    gint perm;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "Oi:gnome.vfs.make_directory", kwlist,
				     &uri, &perm))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_make_directory_for_uri(pygnome_vfs_uri_get(uri),
						  perm);
    else if (PyString_Check(uri))
	result = gnome_vfs_make_directory(PyString_AsString(uri), perm);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_remove_directory(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *uri;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O:gnome.vfs.remove_directory", kwlist,
				     &uri))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_remove_directory_from_uri(pygnome_vfs_uri_get(uri));
    else if (PyString_Check(uri))
	result = gnome_vfs_remove_directory(PyString_AsString(uri));
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_unlink(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *uri;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O:gnome.vfs.unlink", kwlist, &uri))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_unlink_from_uri(pygnome_vfs_uri_get(uri));
    else if (PyString_Check(uri))
	result = gnome_vfs_unlink(PyString_AsString(uri));
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_exists(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *py_uri;
    GnomeVFSURI *uri = NULL;
    gboolean exists;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "O:gnome.vfs.exists",
				     kwlist, &py_uri))
	return NULL;

    if (PyObject_TypeCheck(py_uri, &PyGnomeVFSURI_Type))
	uri = gnome_vfs_uri_ref(pygnome_vfs_uri_get(py_uri));
    else if (PyString_Check(py_uri))
	uri = gnome_vfs_uri_new(PyString_AsString(py_uri));

    if (!uri) {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }
    exists = gnome_vfs_uri_exists(uri);
    gnome_vfs_uri_unref(uri);

    return PyInt_FromLong(exists);
}

static PyObject *
pygvfs_get_mime_type(PyObject *self, PyObject *args)
{
    char *text_uri;
    
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.get_mime_type",
			 &text_uri))
	return NULL;

    return PyString_FromString(gnome_vfs_get_mime_type(text_uri));
}

static PyObject *
pygvfs_get_mime_type_for_data(PyObject *self, PyObject *args)
{
    char *data;
    int data_size;
    
    if(!PyArg_ParseTuple(args, "si:gnome.vfs.get_mime_type_for_data",
			 &data, &data_size))
	return NULL;

    return PyString_FromString(gnome_vfs_get_mime_type_for_data(data,
								data_size));
}

static PyObject *
pygvfs_mime_get_icon(PyObject *self, PyObject *args)
{
    char *mime_type;
    const char *retval;
        
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.mime_get_icon",
			 &mime_type))
	return NULL;
        
    retval = gnome_vfs_mime_get_icon(mime_type);
    if (retval == NULL) {
            Py_INCREF(Py_None);
            return Py_None;
    }
    return PyString_FromString(retval);
}

static PyObject *
pygvfs_mime_set_icon(PyObject *self, PyObject *args)
{
    char *mime_type, *filename;
    GnomeVFSResult result;
        
    if(!PyArg_ParseTuple(args, "ss:gnome.vfs.mime_set_icon",
			 &mime_type, &filename))
	return NULL;

    result = gnome_vfs_mime_set_icon(mime_type, filename);
    if (pygnome_vfs_result_check(result)) {
	return NULL;
    }
    
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_get_description(PyObject *self, PyObject *args)
{
    char *mime_type;
    
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.mime_get_description",
			 &mime_type))
	return NULL;

    return PyString_FromString(gnome_vfs_mime_get_description(mime_type));
}

static PyObject *
pygvfs_mime_set_description(PyObject *self, PyObject *args)
{
    char *mime_type, *description;
    GnomeVFSResult result;
        
    if(!PyArg_ParseTuple(args, "ss:gnome.vfs.mime_set_description",
			 &mime_type, &description))
	return NULL;

    result = gnome_vfs_mime_set_description(mime_type, description);
    if (pygnome_vfs_result_check(result)) {
	return NULL;
    }
    
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_mime_can_be_executable(PyObject *self, PyObject *args)
{
    char *mime_type;
    
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.mime_can_be_executable",
			 &mime_type))
	return NULL;

    return PyInt_FromLong(gnome_vfs_mime_can_be_executable(mime_type));
}

static PyObject *
pygvfs_mime_set_can_be_executable(PyObject *self, PyObject *args)
{
    char *mime_type;
    gboolean new_value;
    GnomeVFSResult result;
        
    if(!PyArg_ParseTuple(args, "si:gnome.vfs.mime_set_description",
			 &mime_type, &new_value))
	return NULL;

    result = gnome_vfs_mime_set_can_be_executable(mime_type, new_value);
    if (pygnome_vfs_result_check(result)) {
	return NULL;
    }
    
    Py_INCREF(Py_None);
    return Py_None;
}

static void
pygvfs_monitor_marshal(GnomeVFSMonitorHandle *handle,
		       const gchar *monitor_uri,
		       const gchar *info_uri,
		       GnomeVFSMonitorEventType event_type,
		       PyGVFSCustomNotify *cunote)
{
    PyObject *retobj;
    
    pyg_block_threads();

    if (cunote->data)
	retobj = PyEval_CallFunction(cunote->func, "(ssiO)", monitor_uri,
				     info_uri, event_type, cunote->data);
    else
	retobj = PyObject_CallFunction(cunote->func, "(ssi)", monitor_uri,
				       info_uri, event_type);

    if (retobj == NULL) {
	PyErr_Print();
	PyErr_Clear();
    }
    
    Py_XDECREF(retobj);
    
    pyg_unblock_threads();    
}

static PyObject*
pygvfs_monitor_add(PyObject *self, PyObject *args)
{
    char *text_uri;
    int monitor_type;
    PyObject *callback;
    PyObject *extra = NULL;
    PyGVFSCustomNotify *cunote;
    GnomeVFSMonitorHandle *handle;
    GnomeVFSResult result;
    guint32 *monitor_id;
    
    if (!PyArg_ParseTuple(args, "siO|O:gnome.vfs.monitor_add",
			  &text_uri, &monitor_type,
			  &callback, &extra)) {
	return NULL;
    }
    
    if (!PyCallable_Check(callback)) {
        PyErr_SetString(PyExc_TypeError, "third argument not callable");
        return NULL;
    }
    
    cunote = g_new0(PyGVFSCustomNotify, 1);
    cunote->func = callback;
    cunote->data = extra;
    Py_INCREF(cunote->func);
    Py_XINCREF(cunote->data);
    result = gnome_vfs_monitor_add(&handle, text_uri, monitor_type,
		   (GnomeVFSMonitorCallback)pygvfs_monitor_marshal,
		   cunote);

    if (pygnome_vfs_result_check(result)) {
	return NULL;
    }

    monitor_id = g_new(guint32, 1);
    *monitor_id = ++monitor_id_counter;
    
    g_hash_table_insert(monitor_hash,
			monitor_id,
			handle);
    
    return PyInt_FromLong(*monitor_id);
}

static PyObject*
pygvfs_monitor_cancel(PyObject *self, PyObject *args)
{
    guint32 monitor_id;
    GnomeVFSMonitorHandle *handle;
    
    if (!PyArg_ParseTuple(args, "i:gnome.vfs.monitor_cancel",
			  &monitor_id)) {
	return NULL;
    }
    
    handle = g_hash_table_lookup(monitor_hash, &monitor_id);
    if (handle == NULL) {
	PyErr_SetString(PyExc_TypeError, "Invalid monitor id");
	return NULL;
    }

    gnome_vfs_monitor_cancel(handle);
    
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_read_entire_file(PyObject *self, PyObject *args)
{
    GnomeVFSResult  result;
    char           *uri;
    char           *file_contents;
    int             file_size;
    PyObject       *rv;

    if (!PyArg_ParseTuple(args, "s:gnome.vfs.read_entire_file", &uri))
	return NULL;
    result = gnome_vfs_read_entire_file (uri, &file_size, &file_contents);
    if (pygnome_vfs_result_check(result))
	return NULL;
    rv = PyString_FromStringAndSize(file_contents, file_size);
    /* I wish there was a string constructor that would take ownership
     * of the C string :( */
    g_free(file_contents);
    return rv;
}

static PyMethodDef pygnomevfs_functions[] = {
    { "create", (PyCFunction)pygvfs_create, METH_VARARGS|METH_KEYWORDS },
    { "get_file_info", (PyCFunction)pygvfs_get_file_info,
      METH_VARARGS|METH_KEYWORDS },
    { "truncate", (PyCFunction)pygvfs_truncate, METH_VARARGS|METH_KEYWORDS },
    { "make_directory", (PyCFunction)pygvfs_make_directory,
      METH_VARARGS|METH_KEYWORDS },
    { "remove_directory", (PyCFunction)pygvfs_remove_directory,
      METH_VARARGS|METH_KEYWORDS },
    { "unlink", (PyCFunction)pygvfs_unlink, METH_VARARGS|METH_KEYWORDS },
    { "exists", (PyCFunction)pygvfs_exists, METH_VARARGS|METH_KEYWORDS },
    { "get_mime_type", (PyCFunction)pygvfs_get_mime_type, METH_VARARGS },
    { "get_mime_type_for_data", (PyCFunction)pygvfs_get_mime_type_for_data,
      METH_VARARGS },
    { "mime_get_icon", (PyCFunction)pygvfs_mime_get_icon, METH_VARARGS },
    { "mime_set_icon", (PyCFunction)pygvfs_mime_set_icon, METH_VARARGS },
    { "mime_get_description", (PyCFunction)pygvfs_mime_get_description, 
      METH_VARARGS },
    { "mime_set_description", (PyCFunction)pygvfs_mime_set_description, 
      METH_VARARGS },
    { "mime_can_be_executable", (PyCFunction)pygvfs_mime_can_be_executable, 
      METH_VARARGS },
    { "mime_set_can_be_executable", (PyCFunction)pygvfs_mime_set_can_be_executable,
      METH_VARARGS },
    { "monitor_add", pygvfs_monitor_add, METH_VARARGS},
    { "monitor_cancel", pygvfs_monitor_cancel, METH_VARARGS},
    { "read_entire_file", pygvfs_read_entire_file, METH_VARARGS},
    { NULL, NULL, 0 }
};

static void
register_constants(PyObject *module)
{
#define regconst(x) PyModule_AddIntConstant(module, #x, GNOME_VFS_ ## x)
    regconst(FILE_FLAGS_NONE);
    regconst(FILE_FLAGS_SYMLINK);
    regconst(FILE_FLAGS_LOCAL);
    regconst(FILE_TYPE_UNKNOWN);
    regconst(FILE_TYPE_REGULAR);
    regconst(FILE_TYPE_DIRECTORY);
    regconst(FILE_TYPE_FIFO);
    regconst(FILE_TYPE_SOCKET);
    regconst(FILE_TYPE_CHARACTER_DEVICE);
    regconst(FILE_TYPE_BLOCK_DEVICE);
    regconst(FILE_TYPE_SYMBOLIC_LINK);
    regconst(FILE_INFO_FIELDS_NONE);
    regconst(FILE_INFO_FIELDS_TYPE);
    regconst(FILE_INFO_FIELDS_PERMISSIONS);
    regconst(FILE_INFO_FIELDS_FLAGS);
    regconst(FILE_INFO_FIELDS_DEVICE);
    regconst(FILE_INFO_FIELDS_INODE);
    regconst(FILE_INFO_FIELDS_LINK_COUNT);
    regconst(FILE_INFO_FIELDS_SIZE);
    regconst(FILE_INFO_FIELDS_BLOCK_COUNT);
    regconst(FILE_INFO_FIELDS_IO_BLOCK_SIZE);
    regconst(FILE_INFO_FIELDS_ATIME);
    regconst(FILE_INFO_FIELDS_MTIME);
    regconst(FILE_INFO_FIELDS_CTIME);
    regconst(FILE_INFO_FIELDS_SYMLINK_NAME);
    regconst(FILE_INFO_FIELDS_MIME_TYPE);
    regconst(PERM_SUID);
    regconst(PERM_SGID);
    regconst(PERM_STICKY);
    regconst(PERM_USER_READ);
    regconst(PERM_USER_WRITE);
    regconst(PERM_USER_EXEC);
    regconst(PERM_USER_ALL);
    regconst(PERM_GROUP_READ);
    regconst(PERM_GROUP_WRITE);
    regconst(PERM_GROUP_EXEC);
    regconst(PERM_GROUP_ALL);
    regconst(PERM_OTHER_READ);
    regconst(PERM_OTHER_WRITE);
    regconst(PERM_OTHER_EXEC);
    regconst(PERM_OTHER_ALL);
    regconst(FILE_INFO_DEFAULT);
    regconst(FILE_INFO_GET_MIME_TYPE);
    regconst(FILE_INFO_FORCE_FAST_MIME_TYPE);
    regconst(FILE_INFO_FORCE_SLOW_MIME_TYPE);
    regconst(FILE_INFO_FOLLOW_LINKS);
    regconst(SET_FILE_INFO_NONE);
    regconst(SET_FILE_INFO_NAME);
    regconst(SET_FILE_INFO_PERMISSIONS);
    regconst(SET_FILE_INFO_OWNER);
    regconst(SET_FILE_INFO_TIME);
    regconst(DIRECTORY_VISIT_DEFAULT);
    regconst(DIRECTORY_VISIT_SAMEFS);
    regconst(DIRECTORY_VISIT_LOOPCHECK);
    regconst(OPEN_NONE);
    regconst(OPEN_READ);
    regconst(OPEN_WRITE);
    regconst(OPEN_RANDOM);
    regconst(SEEK_START);
    regconst(SEEK_CURRENT);
    regconst(SEEK_END);
    regconst(MONITOR_FILE);
    regconst(MONITOR_DIRECTORY);
    regconst(MONITOR_EVENT_CHANGED);
    regconst(MONITOR_EVENT_DELETED);
    regconst(MONITOR_EVENT_STARTEXECUTING);
    regconst(MONITOR_EVENT_STOPEXECUTING);
    regconst(MONITOR_EVENT_CREATED);
    regconst(MONITOR_EVENT_METADATA_CHANGED);
#undef regconst
}

static void initialize_exceptions (PyObject *d)
{
    pygnomevfs_exc = PyErr_NewException ("gnome.vfs.Error",
                                                 PyExc_RuntimeError, NULL);
    PyDict_SetItemString(d, "Error", pygnomevfs_exc);
 
#define register_exception(c_name, py_name)                             \
    pygnomevfs_##c_name##_exc =                                   \
        PyErr_NewException ("gnome.vfs." py_name "Error",               \
                            pygnomevfs_exc, NULL);                \
    PyDict_SetItemString(d, py_name "Error", pygnomevfs_##c_name##_exc);
 
    register_exception(not_found,             "NotFound");
    register_exception(generic,               "Generic");
    register_exception(internal,              "Internal");
    register_exception(bad_parameters,        "BadParameters");
    register_exception(not_supported,         "NotSupported");
    register_exception(io,                    "IO");
    register_exception(corrupted_data,        "CorruptedData");
    register_exception(wrong_format,          "WrongFormat");
    register_exception(bad_file,              "BadFile");
    register_exception(too_big,               "TooBig");
    register_exception(no_space,              "NoSpace");
    register_exception(read_only,             "ReadOnly");    
    register_exception(invalid_uri,           "InvalidURI");
    register_exception(not_open,              "NotOpen");    
    register_exception(invalid_open_mode,     "InvalidOpenMode");
    register_exception(access_denied,         "AccessDenied");    
    register_exception(too_many_open_files,   "TooManyOpenFiles");
    register_exception(eof,                   "EOF");    
    register_exception(not_a_directory,       "NotADirectory");
    register_exception(in_progress,           "InProgress");
    register_exception(interrupted,           "Interrupted");
    register_exception(file_exists,           "FileExists");
    register_exception(loop,                  "Loop");
    register_exception(not_permitted,         "NotPermitted");
    register_exception(is_directory,          "IsDirectory");
    register_exception(no_memory,             "NoMemory");
    register_exception(host_not_found,        "HostNotFound");
    register_exception(invalid_host_name,     "InvalidHostName");
    register_exception(host_has_no_address,   "HostHasNoAddress");
    register_exception(login_failed,          "LoginFailed");
    register_exception(cancelled,             "Cancelled");
    register_exception(directory_busy,        "DirectoryBusy");
    register_exception(directory_not_empty,   "DirectoryNotEmpty");
    register_exception(too_many_links,        "TooManyLinks");
    register_exception(read_only_file_system, "ReadOnlyFileSystem");
    register_exception(not_same_file_system,  "NotSameFileSystem");
    register_exception(name_too_long,         "NameTooLong");
    register_exception(service_not_available, "ServiceNotAvailable");
    register_exception(service_obsolete,      "ServiceObsolete");
    register_exception(protocol_error,        "ProtocolError");
    register_exception(no_master_browser,     "NoMasterBrowser");
#if 0    
    register_exception(no_default,            "NoDefault");
    register_exception(no_handler,            "NoHandler");
    register_exception(parse,                 "Parse");
    register_exception(launch,                "Launch");
#endif    
#undef register_exception
}

struct _PyGnomeVFS_Functions pygnomevfs_api_functions = {
    pygnome_vfs_exception_check,
    pygnome_vfs_uri_new,
    &PyGnomeVFSURI_Type,
    pygnome_vfs_file_info_new,
    &PyGnomeVFSFileInfo_Type,
};

DL_EXPORT(void)
initvfs(void)
{
    PyObject *m, *d, *o;

    PyGnomeVFSURI_Type.ob_type = &PyType_Type;
    PyGnomeVFSFileInfo_Type.ob_type = &PyType_Type;
    PyGnomeVFSDirectoryHandle_Type.ob_type = &PyType_Type;
    PyGnomeVFSHandle_Type.ob_type = &PyType_Type;

    init_pygobject();
    if (!gnome_vfs_init()) {
	PyErr_SetString(PyExc_RuntimeError, "could not initialise gnome.vfs");
	return;
    }

    if (PyType_Ready(&PyGnomeVFSURI_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSFileInfo_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSDirectoryHandle_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSHandle_Type) < 0)
	return;

    m = Py_InitModule("gnome.vfs", pygnomevfs_functions);
    d = PyModule_GetDict(m);

    register_constants(m);
    initialize_exceptions(d);
    PyDict_SetItemString(d, "Error", pygnomevfs_exc);

    PyDict_SetItemString(d, "URI", (PyObject *)&PyGnomeVFSURI_Type);
    PyDict_SetItemString(d, "FileInfo", (PyObject *)&PyGnomeVFSFileInfo_Type);
    PyDict_SetItemString(d, "DirectoryHandle",
			 (PyObject *)&PyGnomeVFSDirectoryHandle_Type);
    PyDict_SetItemString(d, "Handle", (PyObject *)&PyGnomeVFSHandle_Type);

    PyDict_SetItemString(d, "open_directory",
			 (PyObject *)&PyGnomeVFSDirectoryHandle_Type);
    PyDict_SetItemString(d, "open", (PyObject *)&PyGnomeVFSHandle_Type);

    PyDict_SetItemString(d, "_PyGnomeVFS_API",
			 o=PyCObject_FromVoidPtr(&pygnomevfs_api_functions,NULL));
    Py_DECREF(o);

    monitor_hash = g_hash_table_new(g_int_hash, g_int_equal);
}
