/*****************************************************/
/* square.c                                          */
/* -- DLL imposes square window resizing.            */
/* -- To compile: cc -wd -DSTRICT square.c           */
/* -- Written by Paul R. Bonneau 07/19/94.           */
/* -- Modified by PRB, 11/25/94 to utilize           */
/*    WM_WINDOWPOSCHANGING message.                  */
/*****************************************************/
#include <windows.h>
#include "square.h"

typedef enum
    {
    antTopLeft  = 0x0000, /* Low byte encodes x. */
    antTopRight = 0x0001,
    antBotLeft  = 0x0100, /* High byte encodes y. */
    antBotRight = 0x0101
    } ANT; /* ANchor Type. */

typedef struct
    {
    POINT rgpt[2]; /* Anchor, last. */
    HWND  hwnd;
    int   dzMin;   /* Min tracking size. */
    BOOL  fInDrag; /* If frame is being sized. */
    BOOL  fNested; /* If in nested WM_GETMINMAXINFO. */
    ANT   ant;     /* Anchor type. */
    } DRD; /* DRag Data. */

void DrawGhost(DRD *pdrd);
void GetRcDrd(LPRECT lprc, DRD *pdrd);
void Restrict(DRD *pdrd, LPPOINT lppt);

int CALLBACK LibMain(HINSTANCE hinsThis, WORD wDS,
  WORD cbHeap, LPSTR lpsz);
int CALLBACK __export WEP(int wExitCode);

#if MSC_VER > 800
#define INLINE __inline
#else /*MSC_VER <= 800*/
#define INLINE
#endif /*MSC_VER <= 800*/

INLINE int WAbs(int w)
    { return 0 < w ? w : -w; }
INLINE int WMax(int w1, int w2)
    { return w1 < w2 ? w2 : w1; }
INLINE int WMin(int w1, int w2)
    { return w1 > w2 ? w2 : w1; }
INLINE void Swap(LPINT lpw1, LPINT lpw2)
  { *lpw1 ^= *lpw2; *lpw2 ^= *lpw1; *lpw1 ^= *lpw2; }

HBRUSH     hbrsDither; /* Dithered gray brush. */
const int  rgwBits[] = /* Dithered gray brush bits. */
    { 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55 };
const char szProp[]    = "DragDataProperty";
ATOM  atmProp;         /* Property atom for DRD. */

#define ropXNor 0x00a50065 /* ROP for ghost frame. */

int CALLBACK LibMain(HINSTANCE hinsThis, WORD wDS,
  WORD cbHeap, LPSTR lpsz)
/*****************************************************/
/* -- Entry point, create the patterned brush.       */
/*****************************************************/
    {
    HBITMAP  hbmp;

    /* Window property atom for drag data. */
    if (NULL == (atmProp = GlobalAddAtom(szProp)))
        return FALSE;

    if (hbmp = CreateBitmap(8, 8, 1, 1, rgwBits))
        {
        hbrsDither = CreatePatternBrush(hbmp);
        DeleteObject(hbmp);
        }
    return NULL != hbrsDither;
    }

int CALLBACK __export WEP(int wExitCode)
/*****************************************************/
/* -- Exit point, clean up.                          */
/*****************************************************/
    {
    if (NULL != hbrsDither)
        DeleteObject(hbrsDither);
    if (NULL != atmProp)
        GlobalDeleteAtom(atmProp);
    return TRUE;
    }

BOOL WINAPI __export FSquareFilter(HWND hwnd, UINT wm,
  WPARAM wParam, LPARAM lParam)
/*****************************************************/
/* -- App window message filter.                     */
/*****************************************************/
    {
    DRD        *pdrd;
    RECT        rc;
    POINT       rgpt[2];
    LPWINDOWPOS pwps;

    pdrd = (DRD *)GetProp(hwnd, (LPSTR)atmProp);
    if (NULL == pdrd && WM_CREATE != wm)
        return FALSE; /* Too early or out of memory. */

    switch (wm)
        {
    default:
        break;

    case WM_CREATE:  /* Allocate and initialize DRD. */
        pdrd = (DRD *)LocalAlloc(LPTR, sizeof(DRD));
        pdrd->hwnd = hwnd;
        pdrd->fInDrag = pdrd->fNested = FALSE;
        SetProp(hwnd, (LPSTR)atmProp, (HANDLE)pdrd);
        break;

    case WM_DESTROY:                   /* Pitch DRD. */
        LocalFree((HLOCAL)pdrd);
        RemoveProp(hwnd, (LPSTR)atmProp);
        break;

    case WM_GETMINMAXINFO:    /* Discover min. size. */
        if (!pdrd->fNested)
            {
            pdrd->fNested = TRUE;
            SendMessage(hwnd, wm, wParam, lParam);
            pdrd->fNested = FALSE;
            rgpt[0] = ((MINMAXINFO far *)lParam)->
              ptMinTrackSize;
            pdrd->dzMin = WMax(rgpt[0].x, rgpt[0].y);
            return TRUE;
            }
        break;

    case WM_NCLBUTTONDOWN:      /* Start frame drag. */
        switch (wParam)   /* Determine anchor point. */
            {
        default:
            return FALSE;
        case HTTOPLEFT:
        case HTTOP:
            pdrd->ant = antBotRight;
            break;
        case HTTOPRIGHT:
        case HTRIGHT:
            pdrd->ant = antBotLeft;
            break;
        case HTBOTTOMLEFT:
        case HTLEFT:
            pdrd->ant = antTopRight;
            break;
        case HTBOTTOMRIGHT:
        case HTBOTTOM:
            pdrd->ant = antTopLeft;
            break;
            }

        /* Restrict frame to square. */
        GetWindowRect(hwnd, (LPRECT)rgpt);
        pdrd->rgpt[0].x = rgpt[LOBYTE(pdrd->ant)].x;
        pdrd->rgpt[0].y = rgpt[HIBYTE(pdrd->ant)].y;
        Restrict(pdrd, (LPPOINT)&lParam);

        /* Enter drag mode. */
        pdrd->fInDrag = TRUE;
        pdrd->rgpt[1] = *(LPPOINT)&lParam;
        SetCapture(hwnd);
        DrawGhost(pdrd);
        return TRUE;

    case WM_MOUSEMOVE:    /* Track mouse with frame. */
        if (!pdrd->fInDrag)
            break;
        ClientToScreen(hwnd, (LPPOINT)&lParam);
        rgpt[0] = *(LPPOINT)&lParam;
        Restrict(pdrd, &rgpt[0]);
        if (rgpt[0].x != pdrd->rgpt[1].x ||
          rgpt[0].y != pdrd->rgpt[1].y)
            {
            DrawGhost(pdrd); /* Draw new ghost if it */
            pdrd->rgpt[1] = rgpt[0];   /* moved from */
            DrawGhost(pdrd);           /* last time. */
            return TRUE;
            }
        break;

    case WM_LBUTTONUP:            /* Exit drag mode. */
    case WM_CANCELMODE:
        if (pdrd->fInDrag)
            {
            pdrd->fInDrag = FALSE;
            ReleaseCapture();
            DrawGhost(pdrd);  /* Remove ghost frame. */
            GetRcDrd(&rc, pdrd); /* Position window. */
            MoveWindow(hwnd, rc.left, rc.top,
              rc.right - rc.left, rc.bottom - rc.top,
              TRUE);
            return TRUE;
            }
        break;

    case WM_SIZE: /* Prevent Windows from setting us */
        GetWindowRect(hwnd, &rc);  /* to non-square. */
        if ((rgpt[0].x = rc.right - rc.left) !=
         (rgpt[0].y = rc.bottom - rc.top))
            {
            rgpt[0].x = WMin(rgpt[0].x, rgpt[0].y);
            SetWindowPos(hwnd, NULL, 0, 0, rgpt[0].x,
              rgpt[0].x, SWP_NOMOVE | SWP_NOACTIVATE |
                SWP_NOZORDER);
            return TRUE;
            }
        break;

    case WM_WINDOWPOSCHANGING:
        /* Prevent Windows from setting us to */
        /* non-square. */
        pwps = (LPWINDOWPOS)lParam;
        pwps->cx = pwps->cy =
          WMin(pwps->cx, pwps->cy);
        return TRUE;
        }

    return FALSE;
    }

void Restrict(DRD *pdrd, LPPOINT lppt)
/*****************************************************/
/* -- Move the endpoint to the nearest location that */
/*    makes the outline of the window square.        */
/*****************************************************/
    {
    int dx    = lppt->x - pdrd->rgpt[0].x;
    int dy    = lppt->y - pdrd->rgpt[0].y;
    int dzAbs = WMax(WAbs(dx), WAbs(dy));

    dzAbs = WMax(pdrd->dzMin, dzAbs);
    lppt->x = pdrd->rgpt[0].x +
      (0 < dx ? dzAbs : -dzAbs);
    lppt->y = pdrd->rgpt[0].y +
      (0 < dy ? dzAbs : -dzAbs);
    SetCursorPos(lppt->x, lppt->y);
    }

void DrawGhost(DRD *pdrd)
/*****************************************************/
/* -- Paint a ghost rectangle on the screen.         */
/* -- hdc  : Display context to paint onto.          */
/*****************************************************/
    {
    RECT   rc;
    int    dx, dy; /* Size of outline. */
    int    dxBdr   = GetSystemMetrics(SM_CXFRAME);
    int    dyBdr   = GetSystemMetrics(SM_CYFRAME);
    HDC    hdc     = GetDC(NULL);

    if (NULL == hdc)
        return;

    GetRcDrd(&rc, pdrd);
    dx = rc.right - rc.left;
    dy = rc.bottom - rc.top;
    UnrealizeObject(hbrsDither);     /* Reset origin */
    SelectObject(hdc, hbrsDither);

    PatBlt(hdc, rc.left, rc.top, dx, dyBdr, ropXNor);
    PatBlt(hdc, rc.left, rc.bottom - dyBdr, dx, dyBdr,
      ropXNor);
    PatBlt(hdc, rc.left, rc.top + dyBdr, dxBdr,
      dy - 2 * dxBdr, ropXNor);
    PatBlt(hdc, rc.right - dxBdr, rc.top + dyBdr,
      dxBdr, dy - 2 * dxBdr, ropXNor);

    ReleaseDC(NULL, hdc);
    }

void GetRcDrd(LPRECT lprc, DRD *pdrd)
/*****************************************************/
/* -- Return the outline rectangle for the given     */
/*    drag.                                          */
/*****************************************************/
    {
    lprc->left = pdrd->rgpt[LOBYTE(pdrd->ant)].x;
    lprc->top = pdrd->rgpt[HIBYTE(pdrd->ant)].y;
    lprc->right = pdrd->rgpt[1 - LOBYTE(pdrd->ant)].x;
    lprc->bottom = pdrd->rgpt[1 - HIBYTE(pdrd->ant)].y;
    if (lprc->left > lprc->right)
        Swap(&lprc->left, &lprc->right);
    if (lprc->top > lprc->bottom)
        Swap(&lprc->top, &lprc->bottom);
    }
