/*
 * NGLE - hardware acceleration.
 *
 * Copyright (C) 2024 Michael Lorenz
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * MICHAEL LORENZ BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/* $NetBSD: ngle_accel.c,v 1.7 2025/03/05 07:22:24 macallan Exp $ */

#include <sys/types.h>
#include <dev/ic/stireg.h>


#include "ngle.h"

//#define DEBUG

#ifdef DEBUG
#define ENTER xf86Msg(X_ERROR, "%s\n", __func__)
#define LEAVE xf86Msg(X_ERROR, "%s done\n", __func__)
#define DBGMSG xf86Msg
#else
#define ENTER
#define DBGMSG if (0) xf86Msg
#define LEAVE
#endif

static void
NGLEWaitMarker(ScreenPtr pScreen, int Marker)
{
	ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);
	uint8_t stat;

	ENTER;
	do {
		stat = NGLERead1(fPtr, NGLE_REG_15b0);
		if (stat == 0)
			stat = NGLERead1(fPtr, NGLE_REG_15b0);
	} while (stat != 0);
	LEAVE;
}

static void
NGLEWaitFifo(NGLEPtr fPtr, int slots)
{
	uint32_t reg;

	ENTER;
	do {
		reg = NGLERead4(fPtr, NGLE_REG_34);
	} while (reg < slots);
	LEAVE;
}

static Bool
NGLEPrepareCopy_EG
(
    PixmapPtr pSrcPixmap,
    PixmapPtr pDstPixmap,
    int       xdir,
    int       ydir,
    int       alu,
    Pixel     planemask
)
{
	ScrnInfoPtr pScrn = xf86Screens[pDstPixmap->drawable.pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);
	int srcpitch = exaGetPixmapPitch(pSrcPixmap);
	int srcoff = exaGetPixmapOffset(pSrcPixmap);

	ENTER;

	DBGMSG(X_ERROR, "%s %d %d\n", __func__, srcoff, srcpitch);
	fPtr->offset = srcoff >> 11;
	NGLEWaitMarker(pDstPixmap->drawable.pScreen, 0);
	NGLEWrite4(fPtr, NGLE_REG_10,
	    BA(IndexedDcd, Otc04, Ots08, AddrLong, 0, BINapp0I, 0));
	NGLEWrite4(fPtr, NGLE_REG_14,
	    IBOvals(RopSrc, 0, BitmapExtent08, 1, DataDynamic, MaskOtc, 0, 0));
	NGLEWrite4(fPtr, NGLE_REG_13, planemask);

	fPtr->hwmode = HW_BLIT;

	LEAVE;
	return TRUE;
}

static Bool
NGLEPrepareCopy_HCRX
(
    PixmapPtr pSrcPixmap,
    PixmapPtr pDstPixmap,
    int       xdir,
    int       ydir,
    int       alu,
    Pixel     planemask
)
{
	ScrnInfoPtr pScrn = xf86Screens[pDstPixmap->drawable.pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);
	int srcpitch = exaGetPixmapPitch(pSrcPixmap);
	int srcoff = exaGetPixmapOffset(pSrcPixmap);

	ENTER;

	DBGMSG(X_ERROR, "%s %d %d\n", __func__, srcoff, srcpitch);
	fPtr->offset = srcoff / srcpitch;
	NGLEWaitMarker(pDstPixmap->drawable.pScreen, 0);
	NGLEWrite4(fPtr, NGLE_REG_10,
	    BA(FractDcd, Otc24, Ots08, AddrLong, 0, BINapp0F8, 0));
	NGLEWrite4(fPtr, NGLE_REG_14,
	    IBOvals(RopSrc, 0, BitmapExtent32, 0, DataDynamic, MaskOtc, 0, 0));
	NGLEWrite4(fPtr, NGLE_REG_13, planemask);

	fPtr->hwmode = HW_BLIT;

	LEAVE;
	return TRUE;
}

static void
NGLECopy
(
    PixmapPtr pDstPixmap,
    int       xs,
    int       ys,
    int       xd,
    int       yd,
    int       wi,
    int       he
)
{
	ScrnInfoPtr pScrn = xf86Screens[pDstPixmap->drawable.pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);
	int dstpitch = exaGetPixmapPitch(pDstPixmap);
	int dstoff = exaGetPixmapOffset(pDstPixmap);

	ENTER;
	NGLEWaitFifo(fPtr, 3);
	NGLEWrite4(fPtr, NGLE_REG_24, (xs << 16) | (ys + fPtr->offset));
	NGLEWrite4(fPtr, NGLE_REG_7, (wi << 16) | he);
	NGLEWrite4(fPtr, NGLE_REG_25, (xd << 16) | (yd + (dstoff / dstpitch)));

	exaMarkSync(pDstPixmap->drawable.pScreen);
	LEAVE;
}

static void
NGLEDoneCopy(PixmapPtr pDstPixmap)
{
    ENTER;
    LEAVE;
}

static Bool
NGLEPrepareSolid_EG(
    PixmapPtr pPixmap,
    int alu,
    Pixel planemask,
    Pixel fg)
{
	ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);

	ENTER;
	NGLEWaitFifo(fPtr, 4);
	/* plane mask */
	NGLEWrite4(fPtr, NGLE_REG_13, planemask);
	/* bitmap op */
	NGLEWrite4(fPtr, NGLE_REG_14, 
	    IBOvals(alu, 0, BitmapExtent08, 1, DataDynamic, MaskOtc, 1, 0));
	/* dst bitmap access */
	NGLEWrite4(fPtr, NGLE_REG_11,
	    BA(IndexedDcd, Otc32, OtsIndirect, AddrLong, 0, BINapp0I, 0));
    	NGLEWrite4(fPtr, NGLE_REG_35, fg);
	fPtr->hwmode = HW_FILL;

	LEAVE;
	return TRUE;
}

static Bool
NGLEPrepareSolid_HCRX(
    PixmapPtr pPixmap,
    int alu,
    Pixel planemask,
    Pixel fg)
{
	ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);

	ENTER;
	NGLEWaitFifo(fPtr, 4);
	/* plane mask */
	NGLEWrite4(fPtr, NGLE_REG_13, planemask);
	/* bitmap op */
	NGLEWrite4(fPtr, NGLE_REG_14, 
	    IBOvals(alu, 0, BitmapExtent32, 1, DataDynamic, MaskOtc, 1, 0));
	/* dst bitmap access */
	NGLEWrite4(fPtr, NGLE_REG_11,
	    BA(FractDcd, Otc32, OtsIndirect, AddrLong, 0, BINapp0F8, 0));
    	NGLEWrite4(fPtr, NGLE_REG_35, fg);
	fPtr->hwmode = HW_FILL;

	LEAVE;
	return TRUE;
}

static void
NGLESolid(
    PixmapPtr pPixmap,
    int x1,
    int y1,
    int x2,
    int y2)
{
	ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);
	int w = x2 - x1, h = y2 - y1;
	int pitch = exaGetPixmapPitch(pPixmap);
	int offset = exaGetPixmapOffset(pPixmap);
	uint32_t mask;
	int wi, rest;

	ENTER;
	
	y1 += offset / pitch;

	NGLEWaitFifo(fPtr, 3);
	/* transfer data */
	NGLEWrite4(fPtr, NGLE_REG_8, 0xffffffff);
	/* dst XY */
	NGLEWrite4(fPtr, NGLE_REG_6, (x1 << 16) | y1);
	/* len XY start */
	NGLEWrite4(fPtr, NGLE_REG_9, (w << 16) | h);

	exaMarkSync(pPixmap->drawable.pScreen);
	LEAVE;
}

Bool
NGLEPrepareAccess_EG(PixmapPtr pPixmap, int index)
{
	ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);

	if (fPtr->hwmode == HW_FB) return TRUE;

	NGLEWaitMarker(pPixmap->drawable.pScreen, 0);
	NGLEWrite4(fPtr, NGLE_REG_10,
	    BA(IndexedDcd, Otc04, Ots08, AddrByte, 0, BINapp0I, 0));
	NGLEWrite4(fPtr, NGLE_REG_14, 0x83000300);
	NGLEWrite4(fPtr, NGLE_REG_13, 0xff);
	NGLEWaitMarker(pPixmap->drawable.pScreen, 0);
	NGLEWrite1(fPtr, NGLE_REG_16b1, 1);
	fPtr->hwmode = HW_FB;
	return TRUE;
}

Bool
NGLEPrepareAccess_HCRX(PixmapPtr pPixmap, int index)
{
	ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);

	if (fPtr->hwmode == HW_FB) return TRUE;

	NGLEWaitMarker(pPixmap->drawable.pScreen, 0);
	NGLEWrite4(fPtr, NGLE_REG_10,
	    BA(FractDcd, Otc24, Ots08, AddrLong, 0, BINapp0F8, 0));
	NGLEWrite4(fPtr, NGLE_REG_14, 0x83000300);
	NGLEWrite4(fPtr, NGLE_REG_13, 0xffffffff);
	NGLEWaitMarker(pPixmap->drawable.pScreen, 0);
	NGLEWrite1(fPtr, NGLE_REG_16b1, 1);
	fPtr->hwmode = HW_FB;
	return TRUE;
}
Bool
NGLEInitAccel(ScreenPtr pScreen)
{
	ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
	NGLEPtr fPtr = NGLEPTR(pScrn);
	ExaDriverPtr pExa;
	int lines, bpp = pScrn->bitsPerPixel >> 3;

	pExa = exaDriverAlloc();
	if (!pExa)
		return FALSE;

	fPtr->pExa = pExa;

	pExa->exa_major = EXA_VERSION_MAJOR;
	pExa->exa_minor = EXA_VERSION_MINOR;

	pExa->memoryBase = fPtr->fbmem;
	lines = fPtr->fbmem_len / fPtr->fbi.fbi_stride;
	DBGMSG(X_ERROR, "lines %d\n", lines);	
	pExa->memorySize = fPtr->fbmem_len;
	pExa->offScreenBase = fPtr->fbi.fbi_stride * fPtr->fbi.fbi_height;
	pExa->pixmapOffsetAlign = fPtr->fbi.fbi_stride;
	pExa->pixmapPitchAlign = fPtr->fbi.fbi_stride;

	pExa->flags = EXA_OFFSCREEN_PIXMAPS | EXA_MIXED_PIXMAPS;

	pExa->maxX = 2048;
	pExa->maxY = 2048;	

	fPtr->hwmode = -1;

	pExa->WaitMarker = NGLEWaitMarker;
	pExa->Solid = NGLESolid;
	pExa->DoneSolid = NGLEDoneCopy;
	pExa->Copy = NGLECopy;
	pExa->DoneCopy = NGLEDoneCopy;
	switch (fPtr->gid) {
		case STI_DD_EG:
			pExa->PrepareCopy = NGLEPrepareCopy_EG;
			pExa->PrepareSolid = NGLEPrepareSolid_EG;
			pExa->PrepareAccess = NGLEPrepareAccess_EG;
			break;
		case STI_DD_HCRX:
			pExa->PrepareCopy = NGLEPrepareCopy_HCRX;
			pExa->PrepareSolid = NGLEPrepareSolid_HCRX;
			pExa->PrepareAccess = NGLEPrepareAccess_HCRX;
			break;
		default:
			xf86Msg(X_ERROR,
			    "unsupported device GID %08x\n", fPtr->gid);
			return FALSE;
	}
	NGLEWaitMarker(pScreen, 0);

	return exaDriverInit(pScreen, pExa);
}
