/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ClusterBuilder;
import org.ovirt.engine.sdk4.types.Cluster;

/**
 * A service to manage clusters.
 */
public interface ClustersService extends Service {
    /**
     * Creates a new cluster.
     * 
     * This requires the `name`, `cpu.type` and `data_center` attributes. Identify the data center with either the `id`
     * or `name` attributes.
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <cluster>
     *   <name>mycluster</name>
     *   <cpu>
     *     <type>Intel Penryn Family</type>
     *   </cpu>
     *   <data_center id="123"/>
     * </cluster>
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest cluster(Cluster cluster);
        AddRequest cluster(ClusterBuilder cluster);
    }
    
    /**
     * Creates a new cluster.
     * 
     * This requires the `name`, `cpu.type` and `data_center` attributes. Identify the data center with either the `id`
     * or `name` attributes.
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <cluster>
     *   <name>mycluster</name>
     *   <cpu>
     *     <type>Intel Penryn Family</type>
     *   </cpu>
     *   <data_center id="123"/>
     * </cluster>
     * ----
     */
    public interface AddResponse extends Response {
        Cluster cluster();
    }
    
    /**
     * Creates a new cluster.
     * 
     * This requires the `name`, `cpu.type` and `data_center` attributes. Identify the data center with either the `id`
     * or `name` attributes.
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/clusters
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <cluster>
     *   <name>mycluster</name>
     *   <cpu>
     *     <type>Intel Penryn Family</type>
     *   </cpu>
     *   <data_center id="123"/>
     * </cluster>
     * ----
     */
    AddRequest add();
    
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Indicates if the search performed using the `search` parameter should be performed taking case into
         * account. The default value is `true`, which means that case is taken into account. If you want to search
         * ignoring case set it to `false`.
         */
        ListRequest caseSensitive(Boolean caseSensitive);
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        ListRequest filter(Boolean filter);
        /**
         * Sets the maximum number of clusters to return. If not specified all the clusters are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of clusters to return. If not specified all the clusters are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of clusters to return. If not specified all the clusters are returned.
         */
        ListRequest max(BigInteger max);
        /**
         * A query string used to restrict the returned clusters.
         */
        ListRequest search(String search);
    }
    
    public interface ListResponse extends Response {
        List<Cluster> clusters();
    }
    
    ListRequest list();
    
    /**
     * Reference to the service that manages a specific cluster.
     */
    ClusterService clusterService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

