/*
 * Quick & dirty crypto testing module.
 *
 * This will only exist until we have a better testing mechanism
 * (e.g. a char device).
 *
 * Copyright (c) 2002 James Morris <jmorris@intercode.com.au>
 * Copyright (c) 2002 Jean-Francois Dive <jef@linuxbe.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * 14 - 09 - 2003 Changes by Kartikey Mahendra Bhatt
 *
 */
#ifndef _CRYPTO_TCRYPT_H
#define _CRYPTO_TCRYPT_H

#define MAX_DIGEST_SIZE		64
#define MAX_TAP			8

#define MAX_KEYLEN		56
#define MAX_IVLEN		32

struct hash_testvec {
	char plaintext[128];
	unsigned char psize;
	char digest[MAX_DIGEST_SIZE];
	unsigned char np;
	unsigned char tap[MAX_TAP];		
	char key[128]; /* only used with keyed hash algorithms */
	unsigned char ksize;
};

struct hmac_testvec {	
	char key[128];
	unsigned char ksize;
	char plaintext[128];
	unsigned char psize;
	char digest[MAX_DIGEST_SIZE];
	unsigned char np;
	unsigned char tap[MAX_TAP];		
};

struct cipher_testvec {
	unsigned char fail;
	unsigned char wk; /* weak key flag */
	char key[MAX_KEYLEN];
	unsigned char klen;
	char iv[MAX_IVLEN];
	char input[48];
	unsigned char ilen;
	char result[48];
	unsigned char rlen;
	int np;
	unsigned char tap[MAX_TAP];	
};

/*
 * MD4 test vectors from RFC1320
 */
#define MD4_TEST_VECTORS	7

struct hash_testvec md4_tv_template [] = {
	{
		.plaintext = "",
		.digest	= { 0x31, 0xd6, 0xcf, 0xe0, 0xd1, 0x6a, 0xe9, 0x31,
			    0xb7, 0x3c, 0x59, 0xd7, 0xe0, 0xc0, 0x89, 0xc0 },
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0xbd, 0xe5, 0x2c, 0xb3, 0x1d, 0xe3, 0x3e, 0x46,
			    0x24, 0x5e, 0x05, 0xfb, 0xdb, 0xd6, 0xfb, 0x24 },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xa4, 0x48, 0x01, 0x7a, 0xaf, 0x21, 0xd8, 0x52,
			    0x5f, 0xc1, 0x0a, 0xe8, 0x7a, 0xa6, 0x72, 0x9d },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0xd9, 0x13, 0x0a, 0x81, 0x64, 0x54, 0x9f, 0xe8,
			    0x18, 0x87, 0x48, 0x06, 0xe1, 0xc7, 0x01, 0x4b },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xd7, 0x9e, 0x1c, 0x30, 0x8a, 0xa5, 0xbb, 0xcd,
			    0xee, 0xa8, 0xed, 0x63, 0xdf, 0x41, 0x2d, 0xa9 },
		.np	= 2,
		.tap	= { 13, 13 },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0x04, 0x3f, 0x85, 0x82, 0xf2, 0x41, 0xdb, 0x35,
			    0x1c, 0xe6, 0x27, 0xe1, 0x53, 0xe7, 0xf0, 0xe4 },
	}, {
		.plaintext = "123456789012345678901234567890123456789012345678901234567890123"
			     "45678901234567890",
		.psize	= 80,
		.digest	= { 0xe3, 0x3b, 0x4d, 0xdc, 0x9c, 0x38, 0xf2, 0x19,
			    0x9c, 0x3e, 0x7b, 0x16, 0x4f, 0xcc, 0x05, 0x36 },
	},
};

/*
 * MD5 test vectors from RFC1321
 */
#define MD5_TEST_VECTORS	7

struct hash_testvec md5_tv_template[] = {
	{
		.digest	= { 0xd4, 0x1d, 0x8c, 0xd9, 0x8f, 0x00, 0xb2, 0x04,
			    0xe9, 0x80, 0x09, 0x98, 0xec, 0xf8, 0x42, 0x7e },
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0x0c, 0xc1, 0x75, 0xb9, 0xc0, 0xf1, 0xb6, 0xa8,
			    0x31, 0xc3, 0x99, 0xe2, 0x69, 0x77, 0x26, 0x61 },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0x90, 0x01, 0x50, 0x98, 0x3c, 0xd2, 0x4f, 0xb0,
			    0xd6, 0x96, 0x3f, 0x7d, 0x28, 0xe1, 0x7f, 0x72 },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0xf9, 0x6b, 0x69, 0x7d, 0x7c, 0xb7, 0x93, 0x8d,
			    0x52, 0x5a, 0x2f, 0x31, 0xaa, 0xf1, 0x61, 0xd0 },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xc3, 0xfc, 0xd3, 0xd7, 0x61, 0x92, 0xe4, 0x00,
			    0x7d, 0xfb, 0x49, 0x6c, 0xca, 0x67, 0xe1, 0x3b },
		.np	= 2,
		.tap	= {13, 13}
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0xd1, 0x74, 0xab, 0x98, 0xd2, 0x77, 0xd9, 0xf5,
			    0xa5, 0x61, 0x1c, 0x2c, 0x9f, 0x41, 0x9d, 0x9f },
	}, {
		.plaintext = "12345678901234567890123456789012345678901234567890123456789012"
			     "345678901234567890",
		.psize	= 80,
		.digest	= { 0x57, 0xed, 0xf4, 0xa2, 0x2b, 0xe3, 0xc9, 0x55,
			    0xac, 0x49, 0xda, 0x2e, 0x21, 0x07, 0xb6, 0x7a },
	}
};

/*
 * SHA1 test vectors  from from FIPS PUB 180-1
 */
#define SHA1_TEST_VECTORS	2

struct hash_testvec sha1_tv_template[] = {
	{ 
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xa9, 0x99, 0x3e, 0x36, 0x47, 0x06, 0x81, 0x6a, 0xba, 0x3e,
			    0x25, 0x71, 0x78, 0x50, 0xc2, 0x6c, 0x9c, 0xd0, 0xd8, 0x9d },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x84, 0x98, 0x3e, 0x44, 0x1c, 0x3b, 0xd2, 0x6e, 0xba, 0xae,
			    0x4a, 0xa1, 0xf9, 0x51, 0x29, 0xe5, 0xe5, 0x46, 0x70, 0xf1 },
		.np	= 2,
		.tap	= { 28, 28 }
	}
};

/*
 * SHA256 test vectors from from NIST
 */
#define SHA256_TEST_VECTORS	2

struct hash_testvec sha256_tv_template[] = {	
	{ 
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
			    0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
			    0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
			    0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
			    0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
			    0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
			    0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1 },
		.np	= 2,
		.tap	= { 28, 28 }
	},
};

/*
 * SHA384 test vectors from from NIST and kerneli
 */
#define SHA384_TEST_VECTORS	4

struct hash_testvec sha384_tv_template[] = {
	{ 
		.plaintext= "abc",
		.psize	= 3,
		.digest	= { 0xcb, 0x00, 0x75, 0x3f, 0x45, 0xa3, 0x5e, 0x8b,
			    0xb5, 0xa0, 0x3d, 0x69, 0x9a, 0xc6, 0x50, 0x07,
			    0x27, 0x2c, 0x32, 0xab, 0x0e, 0xde, 0xd1, 0x63,
			    0x1a, 0x8b, 0x60, 0x5a, 0x43, 0xff, 0x5b, 0xed,
			    0x80, 0x86, 0x07, 0x2b, 0xa1, 0xe7, 0xcc, 0x23,
			    0x58, 0xba, 0xec, 0xa1, 0x34, 0xc8, 0x25, 0xa7 },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x33, 0x91, 0xfd, 0xdd, 0xfc, 0x8d, 0xc7, 0x39,
			    0x37, 0x07, 0xa6, 0x5b, 0x1b, 0x47, 0x09, 0x39,
			    0x7c, 0xf8, 0xb1, 0xd1, 0x62, 0xaf, 0x05, 0xab,
			    0xfe, 0x8f, 0x45, 0x0d, 0xe5, 0xf3, 0x6b, 0xc6,
			    0xb0, 0x45, 0x5a, 0x85, 0x20, 0xbc, 0x4e, 0x6f,
			    0x5f, 0xe9, 0x5b, 0x1f, 0xe3, 0xc8, 0x45, 0x2b},
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
 			     "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= { 0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
			    0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
			    0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
			    0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
			    0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
			    0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39  },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcd"
			     "efghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz",
		.psize	= 104,
		.digest	= { 0x3d, 0x20, 0x89, 0x73, 0xab, 0x35, 0x08, 0xdb,
			    0xbd, 0x7e, 0x2c, 0x28, 0x62, 0xba, 0x29, 0x0a,
			    0xd3, 0x01, 0x0e, 0x49, 0x78, 0xc1, 0x98, 0xdc,
			    0x4d, 0x8f, 0xd0, 0x14, 0xe5, 0x82, 0x82, 0x3a,
			    0x89, 0xe1, 0x6f, 0x9b, 0x2a, 0x7b, 0xbc, 0x1a,
			    0xc9, 0x38, 0xe2, 0xd1, 0x99, 0xe8, 0xbe, 0xa4 },
		.np	= 4,
		.tap	= { 26, 26, 26, 26 }
	},
};

/*
 * SHA512 test vectors from from NIST and kerneli
 */
#define SHA512_TEST_VECTORS	4

struct hash_testvec sha512_tv_template[] = {
	{ 
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xdd, 0xaf, 0x35, 0xa1, 0x93, 0x61, 0x7a, 0xba,
			    0xcc, 0x41, 0x73, 0x49, 0xae, 0x20, 0x41, 0x31,
			    0x12, 0xe6, 0xfa, 0x4e, 0x89, 0xa9, 0x7e, 0xa2,
			    0x0a, 0x9e, 0xee, 0xe6, 0x4b, 0x55, 0xd3, 0x9a,
			    0x21, 0x92, 0x99, 0x2a, 0x27, 0x4f, 0xc1, 0xa8,
			    0x36, 0xba, 0x3c, 0x23, 0xa3, 0xfe, 0xeb, 0xbd,
			    0x45, 0x4d, 0x44, 0x23, 0x64, 0x3c, 0xe8, 0x0e,
			    0x2a, 0x9a, 0xc9, 0x4f, 0xa5, 0x4c, 0xa4, 0x9f },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x20, 0x4a, 0x8f, 0xc6, 0xdd, 0xa8, 0x2f, 0x0a,
			    0x0c, 0xed, 0x7b, 0xeb, 0x8e, 0x08, 0xa4, 0x16,
			    0x57, 0xc1, 0x6e, 0xf4, 0x68, 0xb2, 0x28, 0xa8,
			    0x27, 0x9b, 0xe3, 0x31, 0xa7, 0x03, 0xc3, 0x35,
			    0x96, 0xfd, 0x15, 0xc1, 0x3b, 0x1b, 0x07, 0xf9,
			    0xaa, 0x1d, 0x3b, 0xea, 0x57, 0x78, 0x9c, 0xa0,
			    0x31, 0xad, 0x85, 0xc7, 0xa7, 0x1d, 0xd7, 0x03,
			    0x54, 0xec, 0x63, 0x12, 0x38, 0xca, 0x34, 0x45 },
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
			     "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= { 0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
			    0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
			    0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
			    0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
			    0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
			    0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
			    0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
			    0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09 },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcd"
			     "efghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz",
		.psize	= 104,
		.digest	= { 0x93, 0x0d, 0x0c, 0xef, 0xcb, 0x30, 0xff, 0x11,
			    0x33, 0xb6, 0x89, 0x81, 0x21, 0xf1, 0xcf, 0x3d,
			    0x27, 0x57, 0x8a, 0xfc, 0xaf, 0xe8, 0x67, 0x7c,
			    0x52, 0x57, 0xcf, 0x06, 0x99, 0x11, 0xf7, 0x5d,
			    0x8f, 0x58, 0x31, 0xb5, 0x6e, 0xbf, 0xda, 0x67,
			    0xb2, 0x78, 0xe6, 0x6d, 0xff, 0x8b, 0x84, 0xfe,
			    0x2b, 0x28, 0x70, 0xf7, 0x42, 0xa5, 0x80, 0xd8,
			    0xed, 0xb4, 0x19, 0x87, 0x23, 0x28, 0x50, 0xc9 },
		.np	= 4,
		.tap	= { 26, 26, 26, 26 }
	},
};

#ifdef CONFIG_CRYPTO_HMAC
/*
 * HMAC-MD5 test vectors from RFC2202
 * (These need to be fixed to not use strlen).
 */
#define HMAC_MD5_TEST_VECTORS	7

struct hmac_testvec hmac_md5_tv_template[] =
{	
	{
		.key	= { [0 ... 15] =  0x0b },
		.ksize	= 16,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= { 0x92, 0x94, 0x72, 0x7a, 0x36, 0x38, 0xbb, 0x1c,
			    0x13, 0xf4, 0x8e, 0xf8, 0x15, 0x8b, 0xfc, 0x9d },
	}, {
		.key	= { 'J', 'e', 'f', 'e' },
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= { 0x75, 0x0c, 0x78, 0x3e, 0x6a, 0xb0, 0xb5, 0x03,
			    0xea, 0xa8, 0x6e, 0x31, 0x0a, 0x5d, 0xb7, 0x38 },
		.np	= 2,
		.tap	= {14, 14}
	}, {
		.key	= { [0 ... 15] = 0xaa },
		.ksize	= 16,
		.plaintext = { [0 ... 49] =  0xdd },
		.psize	= 50,
		.digest	= { 0x56, 0xbe, 0x34, 0x52, 0x1d, 0x14, 0x4c, 0x88,
			    0xdb, 0xb8, 0xc7, 0x33, 0xf0, 0xe8, 0xb3, 0xf6 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, },
		.ksize	= 25,
		.plaintext = { [0 ... 49] =  0xcd },
		.psize	= 50,
		.digest	= { 0x69, 0x7e, 0xaf, 0x0a, 0xca, 0x3a, 0x3a, 0xea,
			    0x3a, 0x75, 0x16, 0x47, 0x46, 0xff, 0xaa, 0x79 },
	}, {
		.key	= { [0 ... 15] = 0x0c },
		.ksize	= 16,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= { 0x56, 0x46, 0x1e, 0xf2, 0x34, 0x2e, 0xdc, 0x00,
			    0xf9, 0xba, 0xb9, 0x95, 0x69, 0x0e, 0xfd, 0x4c },
	}, {
		.key	= { [0 ... 79] =  0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= { 0x6b, 0x1a, 0xb7, 0xfe, 0x4b, 0xd7, 0xbf, 0x8f,
			    0x0b, 0x62, 0xe6, 0xce, 0x61, 0xb9, 0xd0, 0xcd },
	}, {
		.key	= { [0 ... 79] =  0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			     "Block-Size Data",
		.psize	= 73,
		.digest	= { 0x6f, 0x63, 0x0f, 0xad, 0x67, 0xcd, 0xa0, 0xee,
			    0x1f, 0xb1, 0xf5, 0x62, 0xdb, 0x3a, 0xa5, 0x3e },
	},
};

/*
 * HMAC-SHA1 test vectors from RFC2202
 */
#define HMAC_SHA1_TEST_VECTORS	7

struct hmac_testvec hmac_sha1_tv_template[] = {	
	{
		.key	= { [0 ... 19] = 0x0b },
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= { 0xb6, 0x17, 0x31, 0x86, 0x55, 0x05, 0x72, 0x64,
			    0xe2, 0x8b, 0xc0, 0xb6, 0xfb, 0x37, 0x8c, 0x8e, 0xf1,
			    0x46, 0xbe },
	}, {
		.key	= { 'J', 'e', 'f', 'e' },		
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= { 0xef, 0xfc, 0xdf, 0x6a, 0xe5, 0xeb, 0x2f, 0xa2, 0xd2, 0x74, 
			    0x16, 0xd5, 0xf1, 0x84, 0xdf, 0x9c, 0x25, 0x9a, 0x7c, 0x79 },
		.np	= 2,
		.tap	= { 14, 14 }
	}, {
		.key	= { [0 ... 19] = 0xaa },
		.ksize	= 20,
		.plaintext = { [0 ... 49] = 0xdd },
		.psize	= 50,
		.digest	= { 0x12, 0x5d, 0x73, 0x42, 0xb9, 0xac, 0x11, 0xcd, 0x91, 0xa3, 
			    0x9a, 0xf4, 0x8a, 0xa1, 0x7b, 0x4f, 0x63, 0xf1, 0x75, 0xd3 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19 },
		.ksize	= 25,
		.plaintext = { [0 ... 49] = 0xcd },
		.psize	= 50,
		.digest	= { 0x4c, 0x90, 0x07, 0xf4, 0x02, 0x62, 0x50, 0xc6, 0xbc, 0x84, 
			    0x14, 0xf9, 0xbf, 0x50, 0xc8, 0x6c, 0x2d, 0x72, 0x35, 0xda },
	}, {
		.key	= { [0 ... 19] = 0x0c },
		.ksize	= 20,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= { 0x4c, 0x1a, 0x03, 0x42, 0x4b, 0x55, 0xe0, 0x7f, 0xe7, 0xf2, 
			    0x7b, 0xe1, 0xd5, 0x8b, 0xb9, 0x32, 0x4a, 0x9a, 0x5a, 0x04 },
	}, {
		.key	= { [0 ... 79] = 0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= { 0xaa, 0x4a, 0xe5, 0xe1, 0x52, 0x72, 0xd0, 0x0e, 0x95, 0x70, 
			    0x56, 0x37, 0xce, 0x8a, 0x3b, 0x55, 0xed, 0x40, 0x21, 0x12 },
	}, {
		.key	= { [0 ... 79] = 0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			     "Block-Size Data",
		.psize	= 73,
		.digest	= { 0xe8, 0xe9, 0x9d, 0x0f, 0x45, 0x23, 0x7d, 0x78, 0x6d, 0x6b, 
			    0xba, 0xa7, 0x96, 0x5c, 0x78, 0x08, 0xbb, 0xff, 0x1a, 0x91 },
	},
};

/*
 * HMAC-SHA256 test vectors from
 * draft-ietf-ipsec-ciph-sha-256-01.txt
 */
#define HMAC_SHA256_TEST_VECTORS	10

struct hmac_testvec hmac_sha256_tv_template[] = {
	{
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20},
		.ksize	= 32,
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xa2, 0x1b, 0x1f, 0x5d, 0x4c, 0xf4, 0xf7, 0x3a,
			    0x4d, 0xd9, 0x39, 0x75, 0x0f, 0x7a, 0x06, 0x6a,
			    0x7f, 0x98, 0xcc, 0x13, 0x1c, 0xb1, 0x6a, 0x66,
			    0x92, 0x75, 0x90, 0x21, 0xcf, 0xab, 0x81, 0x81 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20 },
		.ksize	= 32,
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x10, 0x4f, 0xdc, 0x12, 0x57, 0x32, 0x8f, 0x08,
			    0x18, 0x4b, 0xa7, 0x31, 0x31, 0xc5, 0x3c, 0xae,
			    0xe6, 0x98, 0xe3, 0x61, 0x19, 0x42, 0x11, 0x49,
			    0xea, 0x8c, 0x71, 0x24, 0x56, 0x69, 0x7d, 0x30 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20 },
		.ksize	= 32,
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
			     "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 112,
		.digest	= { 0x47, 0x03, 0x05, 0xfc, 0x7e, 0x40, 0xfe, 0x34,
			    0xd3, 0xee, 0xb3, 0xe7, 0x73, 0xd9, 0x5a, 0xab,
			    0x73, 0xac, 0xf0, 0xfd, 0x06, 0x04, 0x47, 0xa5,
			    0xeb, 0x45, 0x95, 0xbf, 0x33, 0xa9, 0xd1, 0xa3 },
	}, {
		.key	= { [0 ... 31] = 0x0b },
		.ksize	= 32,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= { 0x19, 0x8a, 0x60, 0x7e, 0xb4, 0x4b, 0xfb, 0xc6,
			    0x99, 0x03, 0xa0, 0xf1, 0xcf, 0x2b, 0xbd, 0xc5,
			    0xba, 0x0a, 0xa3, 0xf3, 0xd9, 0xae, 0x3c, 0x1c,
			    0x7a, 0x3b, 0x16, 0x96, 0xa0, 0xb6, 0x8c, 0xf7 },
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= { 0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e,
			    0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xc7,
			    0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83,
			    0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec, 0x38, 0x43 },
		.np	= 2,
		.tap	= { 14, 14 }
	}, {
		.key	= { [0 ... 31] = 0xaa },
		.ksize	= 32,
		.plaintext = { [0 ... 49] = 0xdd },
		.psize	= 50,
		.digest	= { 0xcd, 0xcb, 0x12, 0x20, 0xd1, 0xec, 0xcc, 0xea,
			    0x91, 0xe5, 0x3a, 0xba, 0x30, 0x92, 0xf9, 0x62,
			    0xe5, 0x49, 0xfe, 0x6c, 0xe9, 0xed, 0x7f, 0xdc,
			    0x43, 0x19, 0x1f, 0xbd, 0xe4, 0x5c, 0x30, 0xb0 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
			    0x21, 0x22, 0x23, 0x24, 0x25 },
		.ksize	= 37,
		.plaintext = { [0 ... 49] = 0xcd },
		.psize	= 50,
		.digest	= { 0xd4, 0x63, 0x3c, 0x17, 0xf6, 0xfb, 0x8d, 0x74,
			    0x4c, 0x66, 0xde, 0xe0, 0xf8, 0xf0, 0x74, 0x55,
			    0x6e, 0xc4, 0xaf, 0x55, 0xef, 0x07, 0x99, 0x85,
			    0x41, 0x46, 0x8e, 0xb4, 0x9b, 0xd2, 0xe9, 0x17 },
	}, {
		.key	= { [0 ... 31] = 0x0c },
		.ksize	= 32,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= { 0x75, 0x46, 0xaf, 0x01, 0x84, 0x1f, 0xc0, 0x9b,
			    0x1a, 0xb9, 0xc3, 0x74, 0x9a, 0x5f, 0x1c, 0x17,
			    0xd4, 0xf5, 0x89, 0x66, 0x8a, 0x58, 0x7b, 0x27,
			    0x00, 0xa9, 0xc9, 0x7c, 0x11, 0x93, 0xcf, 0x42 },
	}, {
		.key	= { [0 ... 79] = 0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= { 0x69, 0x53, 0x02, 0x5e, 0xd9, 0x6f, 0x0c, 0x09,
			    0xf8, 0x0a, 0x96, 0xf7, 0x8e, 0x65, 0x38, 0xdb,
			    0xe2, 0xe7, 0xb8, 0x20, 0xe3, 0xdd, 0x97, 0x0e,
			    0x7d, 0xdd, 0x39, 0x09, 0x1b, 0x32, 0x35, 0x2f },
	}, {
		.key	= { [0 ... 79] = 0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than "
			     "One Block-Size Data",
		.psize	= 73,
		.digest	= { 0x63, 0x55, 0xac, 0x22, 0xe8, 0x90, 0xd0, 0xa3,
			    0xc8, 0x48, 0x1a, 0x5c, 0xa4, 0x82, 0x5b, 0xc8,
			    0x84, 0xd3, 0xe7, 0xa1, 0xff, 0x98, 0xa2, 0xfc,
			    0x2a, 0xc7, 0xd8, 0xe0, 0x64, 0xc3, 0xb2, 0xe6 },
	},
};

#endif	/* CONFIG_CRYPTO_HMAC */

/*
 * WHIRLPOOL test vectors from Whirlpool package 
 * by Vincent Rijmen and Paulo S. L. M. Barreto as part of the NESSIE
 * submission
 */
#define WP512_TEST_VECTORS	8

struct hash_testvec wp512_tv_template[] = {
	{ 
		.plaintext = "",
		.psize	= 0,
		.digest	= { 0x19, 0xFA, 0x61, 0xD7, 0x55, 0x22, 0xA4, 0x66,
			    0x9B, 0x44, 0xE3, 0x9C, 0x1D, 0x2E, 0x17, 0x26,
			    0xC5, 0x30, 0x23, 0x21, 0x30, 0xD4, 0x07, 0xF8,
			    0x9A, 0xFE, 0xE0, 0x96, 0x49, 0x97, 0xF7, 0xA7,
			    0x3E, 0x83, 0xBE, 0x69, 0x8B, 0x28, 0x8F, 0xEB,
			    0xCF, 0x88, 0xE3, 0xE0, 0x3C, 0x4F, 0x07, 0x57,
			    0xEA, 0x89, 0x64, 0xE5, 0x9B, 0x63, 0xD9, 0x37,
			    0x08, 0xB1, 0x38, 0xCC, 0x42, 0xA6, 0x6E, 0xB3 },


	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0x8A, 0xCA, 0x26, 0x02, 0x79, 0x2A, 0xEC, 0x6F,
			    0x11, 0xA6, 0x72, 0x06, 0x53, 0x1F, 0xB7, 0xD7,
			    0xF0, 0xDF, 0xF5, 0x94, 0x13, 0x14, 0x5E, 0x69,
			    0x73, 0xC4, 0x50, 0x01, 0xD0, 0x08, 0x7B, 0x42,
			    0xD1, 0x1B, 0xC6, 0x45, 0x41, 0x3A, 0xEF, 0xF6,
			    0x3A, 0x42, 0x39, 0x1A, 0x39, 0x14, 0x5A, 0x59,
			    0x1A, 0x92, 0x20, 0x0D, 0x56, 0x01, 0x95, 0xE5,
			    0x3B, 0x47, 0x85, 0x84, 0xFD, 0xAE, 0x23, 0x1A },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0x4E, 0x24, 0x48, 0xA4, 0xC6, 0xF4, 0x86, 0xBB,
			    0x16, 0xB6, 0x56, 0x2C, 0x73, 0xB4, 0x02, 0x0B,
			    0xF3, 0x04, 0x3E, 0x3A, 0x73, 0x1B, 0xCE, 0x72,
			    0x1A, 0xE1, 0xB3, 0x03, 0xD9, 0x7E, 0x6D, 0x4C,
			    0x71, 0x81, 0xEE, 0xBD, 0xB6, 0xC5, 0x7E, 0x27,
			    0x7D, 0x0E, 0x34, 0x95, 0x71, 0x14, 0xCB, 0xD6,
			    0xC7, 0x97, 0xFC, 0x9D, 0x95, 0xD8, 0xB5, 0x82,
			    0xD2, 0x25, 0x29, 0x20, 0x76, 0xD4, 0xEE, 0xF5 },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0x37, 0x8C, 0x84, 0xA4, 0x12, 0x6E, 0x2D, 0xC6, 
			    0xE5, 0x6D, 0xCC, 0x74, 0x58, 0x37, 0x7A, 0xAC, 
			    0x83, 0x8D, 0x00, 0x03, 0x22, 0x30, 0xF5, 0x3C, 
			    0xE1, 0xF5, 0x70, 0x0C, 0x0F, 0xFB, 0x4D, 0x3B, 
			    0x84, 0x21, 0x55, 0x76, 0x59, 0xEF, 0x55, 0xC1, 
			    0x06, 0xB4, 0xB5, 0x2A, 0xC5, 0xA4, 0xAA, 0xA6, 
			    0x92, 0xED, 0x92, 0x00, 0x52, 0x83, 0x8F, 0x33, 
			    0x62, 0xE8, 0x6D, 0xBD, 0x37, 0xA8, 0x90, 0x3E },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xF1, 0xD7, 0x54, 0x66, 0x26, 0x36, 0xFF, 0xE9,
			    0x2C, 0x82, 0xEB, 0xB9, 0x21, 0x2A, 0x48, 0x4A,
			    0x8D, 0x38, 0x63, 0x1E, 0xAD, 0x42, 0x38, 0xF5,
			    0x44, 0x2E, 0xE1, 0x3B, 0x80, 0x54, 0xE4, 0x1B,
			    0x08, 0xBF, 0x2A, 0x92, 0x51, 0xC3, 0x0B, 0x6A,
			    0x0B, 0x8A, 0xAE, 0x86, 0x17, 0x7A, 0xB4, 0xA6,
			    0xF6, 0x8F, 0x67, 0x3E, 0x72, 0x07, 0x86, 0x5D,
			    0x5D, 0x98, 0x19, 0xA3, 0xDB, 0xA4, 0xEB, 0x3B },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			     "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0xDC, 0x37, 0xE0, 0x08, 0xCF, 0x9E, 0xE6, 0x9B,
			    0xF1, 0x1F, 0x00, 0xED, 0x9A, 0xBA, 0x26, 0x90,
			    0x1D, 0xD7, 0xC2, 0x8C, 0xDE, 0xC0, 0x66, 0xCC,
			    0x6A, 0xF4, 0x2E, 0x40, 0xF8, 0x2F, 0x3A, 0x1E,
			    0x08, 0xEB, 0xA2, 0x66, 0x29, 0x12, 0x9D, 0x8F,
			    0xB7, 0xCB, 0x57, 0x21, 0x1B, 0x92, 0x81, 0xA6,
			    0x55, 0x17, 0xCC, 0x87, 0x9D, 0x7B, 0x96, 0x21,
			    0x42, 0xC6, 0x5F, 0x5A, 0x7A, 0xF0, 0x14, 0x67 },
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= { 0x46, 0x6E, 0xF1, 0x8B, 0xAB, 0xB0, 0x15, 0x4D,
			    0x25, 0xB9, 0xD3, 0x8A, 0x64, 0x14, 0xF5, 0xC0,
			    0x87, 0x84, 0x37, 0x2B, 0xCC, 0xB2, 0x04, 0xD6,
			    0x54, 0x9C, 0x4A, 0xFA, 0xDB, 0x60, 0x14, 0x29,
			    0x4D, 0x5B, 0xD8, 0xDF, 0x2A, 0x6C, 0x44, 0xE5,
			    0x38, 0xCD, 0x04, 0x7B, 0x26, 0x81, 0xA5, 0x1A,
			    0x2C, 0x60, 0x48, 0x1E, 0x88, 0xC5, 0xA2, 0x0B,
			    0x2C, 0x2A, 0x80, 0xCF, 0x3A, 0x9A, 0x08, 0x3B },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= { 0x2A, 0x98, 0x7E, 0xA4, 0x0F, 0x91, 0x70, 0x61, 
			    0xF5, 0xD6, 0xF0, 0xA0, 0xE4, 0x64, 0x4F, 0x48,
			    0x8A, 0x7A, 0x5A, 0x52, 0xDE, 0xEE, 0x65, 0x62,
			    0x07, 0xC5, 0x62, 0xF9, 0x88, 0xE9, 0x5C, 0x69,
			    0x16, 0xBD, 0xC8, 0x03, 0x1B, 0xC5, 0xBE, 0x1B,
			    0x7B, 0x94, 0x76, 0x39, 0xFE, 0x05, 0x0B, 0x56,
			    0x93, 0x9B, 0xAA, 0xA0, 0xAD, 0xFF, 0x9A, 0xE6,
			    0x74, 0x5B, 0x7B, 0x18, 0x1C, 0x3B, 0xE3, 0xFD },
	},
};

#define WP384_TEST_VECTORS	8

struct hash_testvec wp384_tv_template[] = {
	{ 
		.plaintext = "",
		.psize	= 0,
		.digest	= { 0x19, 0xFA, 0x61, 0xD7, 0x55, 0x22, 0xA4, 0x66,
			    0x9B, 0x44, 0xE3, 0x9C, 0x1D, 0x2E, 0x17, 0x26,
			    0xC5, 0x30, 0x23, 0x21, 0x30, 0xD4, 0x07, 0xF8,
			    0x9A, 0xFE, 0xE0, 0x96, 0x49, 0x97, 0xF7, 0xA7,
			    0x3E, 0x83, 0xBE, 0x69, 0x8B, 0x28, 0x8F, 0xEB,
			    0xCF, 0x88, 0xE3, 0xE0, 0x3C, 0x4F, 0x07, 0x57 },


	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0x8A, 0xCA, 0x26, 0x02, 0x79, 0x2A, 0xEC, 0x6F,
			    0x11, 0xA6, 0x72, 0x06, 0x53, 0x1F, 0xB7, 0xD7,
			    0xF0, 0xDF, 0xF5, 0x94, 0x13, 0x14, 0x5E, 0x69,
			    0x73, 0xC4, 0x50, 0x01, 0xD0, 0x08, 0x7B, 0x42,
			    0xD1, 0x1B, 0xC6, 0x45, 0x41, 0x3A, 0xEF, 0xF6,
			    0x3A, 0x42, 0x39, 0x1A, 0x39, 0x14, 0x5A, 0x59 },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0x4E, 0x24, 0x48, 0xA4, 0xC6, 0xF4, 0x86, 0xBB,
			    0x16, 0xB6, 0x56, 0x2C, 0x73, 0xB4, 0x02, 0x0B,
			    0xF3, 0x04, 0x3E, 0x3A, 0x73, 0x1B, 0xCE, 0x72,
			    0x1A, 0xE1, 0xB3, 0x03, 0xD9, 0x7E, 0x6D, 0x4C,
			    0x71, 0x81, 0xEE, 0xBD, 0xB6, 0xC5, 0x7E, 0x27,
			    0x7D, 0x0E, 0x34, 0x95, 0x71, 0x14, 0xCB, 0xD6 },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0x37, 0x8C, 0x84, 0xA4, 0x12, 0x6E, 0x2D, 0xC6, 
			    0xE5, 0x6D, 0xCC, 0x74, 0x58, 0x37, 0x7A, 0xAC, 
			    0x83, 0x8D, 0x00, 0x03, 0x22, 0x30, 0xF5, 0x3C, 
			    0xE1, 0xF5, 0x70, 0x0C, 0x0F, 0xFB, 0x4D, 0x3B, 
			    0x84, 0x21, 0x55, 0x76, 0x59, 0xEF, 0x55, 0xC1, 
			    0x06, 0xB4, 0xB5, 0x2A, 0xC5, 0xA4, 0xAA, 0xA6 },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xF1, 0xD7, 0x54, 0x66, 0x26, 0x36, 0xFF, 0xE9,
			    0x2C, 0x82, 0xEB, 0xB9, 0x21, 0x2A, 0x48, 0x4A,
			    0x8D, 0x38, 0x63, 0x1E, 0xAD, 0x42, 0x38, 0xF5,
			    0x44, 0x2E, 0xE1, 0x3B, 0x80, 0x54, 0xE4, 0x1B,
			    0x08, 0xBF, 0x2A, 0x92, 0x51, 0xC3, 0x0B, 0x6A,
			    0x0B, 0x8A, 0xAE, 0x86, 0x17, 0x7A, 0xB4, 0xA6 },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			     "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0xDC, 0x37, 0xE0, 0x08, 0xCF, 0x9E, 0xE6, 0x9B,
			    0xF1, 0x1F, 0x00, 0xED, 0x9A, 0xBA, 0x26, 0x90,
			    0x1D, 0xD7, 0xC2, 0x8C, 0xDE, 0xC0, 0x66, 0xCC,
			    0x6A, 0xF4, 0x2E, 0x40, 0xF8, 0x2F, 0x3A, 0x1E,
			    0x08, 0xEB, 0xA2, 0x66, 0x29, 0x12, 0x9D, 0x8F,
			    0xB7, 0xCB, 0x57, 0x21, 0x1B, 0x92, 0x81, 0xA6 },
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= { 0x46, 0x6E, 0xF1, 0x8B, 0xAB, 0xB0, 0x15, 0x4D,
			    0x25, 0xB9, 0xD3, 0x8A, 0x64, 0x14, 0xF5, 0xC0,
			    0x87, 0x84, 0x37, 0x2B, 0xCC, 0xB2, 0x04, 0xD6,
			    0x54, 0x9C, 0x4A, 0xFA, 0xDB, 0x60, 0x14, 0x29,
			    0x4D, 0x5B, 0xD8, 0xDF, 0x2A, 0x6C, 0x44, 0xE5,
			    0x38, 0xCD, 0x04, 0x7B, 0x26, 0x81, 0xA5, 0x1A },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= { 0x2A, 0x98, 0x7E, 0xA4, 0x0F, 0x91, 0x70, 0x61, 
			    0xF5, 0xD6, 0xF0, 0xA0, 0xE4, 0x64, 0x4F, 0x48,
			    0x8A, 0x7A, 0x5A, 0x52, 0xDE, 0xEE, 0x65, 0x62,
			    0x07, 0xC5, 0x62, 0xF9, 0x88, 0xE9, 0x5C, 0x69,
			    0x16, 0xBD, 0xC8, 0x03, 0x1B, 0xC5, 0xBE, 0x1B,
			    0x7B, 0x94, 0x76, 0x39, 0xFE, 0x05, 0x0B, 0x56 },
	},
};

#define WP256_TEST_VECTORS	8

struct hash_testvec wp256_tv_template[] = {
	{ 
		.plaintext = "",
		.psize	= 0,
		.digest	= { 0x19, 0xFA, 0x61, 0xD7, 0x55, 0x22, 0xA4, 0x66,
			    0x9B, 0x44, 0xE3, 0x9C, 0x1D, 0x2E, 0x17, 0x26,
			    0xC5, 0x30, 0x23, 0x21, 0x30, 0xD4, 0x07, 0xF8,
			    0x9A, 0xFE, 0xE0, 0x96, 0x49, 0x97, 0xF7, 0xA7 },


	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0x8A, 0xCA, 0x26, 0x02, 0x79, 0x2A, 0xEC, 0x6F,
			    0x11, 0xA6, 0x72, 0x06, 0x53, 0x1F, 0xB7, 0xD7,
			    0xF0, 0xDF, 0xF5, 0x94, 0x13, 0x14, 0x5E, 0x69,
			    0x73, 0xC4, 0x50, 0x01, 0xD0, 0x08, 0x7B, 0x42 },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0x4E, 0x24, 0x48, 0xA4, 0xC6, 0xF4, 0x86, 0xBB,
			    0x16, 0xB6, 0x56, 0x2C, 0x73, 0xB4, 0x02, 0x0B,
			    0xF3, 0x04, 0x3E, 0x3A, 0x73, 0x1B, 0xCE, 0x72,
			    0x1A, 0xE1, 0xB3, 0x03, 0xD9, 0x7E, 0x6D, 0x4C },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0x37, 0x8C, 0x84, 0xA4, 0x12, 0x6E, 0x2D, 0xC6, 
			    0xE5, 0x6D, 0xCC, 0x74, 0x58, 0x37, 0x7A, 0xAC, 
			    0x83, 0x8D, 0x00, 0x03, 0x22, 0x30, 0xF5, 0x3C, 
			    0xE1, 0xF5, 0x70, 0x0C, 0x0F, 0xFB, 0x4D, 0x3B },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xF1, 0xD7, 0x54, 0x66, 0x26, 0x36, 0xFF, 0xE9,
			    0x2C, 0x82, 0xEB, 0xB9, 0x21, 0x2A, 0x48, 0x4A,
			    0x8D, 0x38, 0x63, 0x1E, 0xAD, 0x42, 0x38, 0xF5,
			    0x44, 0x2E, 0xE1, 0x3B, 0x80, 0x54, 0xE4, 0x1B },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			     "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0xDC, 0x37, 0xE0, 0x08, 0xCF, 0x9E, 0xE6, 0x9B,
			    0xF1, 0x1F, 0x00, 0xED, 0x9A, 0xBA, 0x26, 0x90,
			    0x1D, 0xD7, 0xC2, 0x8C, 0xDE, 0xC0, 0x66, 0xCC,
			    0x6A, 0xF4, 0x2E, 0x40, 0xF8, 0x2F, 0x3A, 0x1E },
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= { 0x46, 0x6E, 0xF1, 0x8B, 0xAB, 0xB0, 0x15, 0x4D,
			    0x25, 0xB9, 0xD3, 0x8A, 0x64, 0x14, 0xF5, 0xC0,
			    0x87, 0x84, 0x37, 0x2B, 0xCC, 0xB2, 0x04, 0xD6,
			    0x54, 0x9C, 0x4A, 0xFA, 0xDB, 0x60, 0x14, 0x29 },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= { 0x2A, 0x98, 0x7E, 0xA4, 0x0F, 0x91, 0x70, 0x61, 
			    0xF5, 0xD6, 0xF0, 0xA0, 0xE4, 0x64, 0x4F, 0x48,
			    0x8A, 0x7A, 0x5A, 0x52, 0xDE, 0xEE, 0x65, 0x62,
			    0x07, 0xC5, 0x62, 0xF9, 0x88, 0xE9, 0x5C, 0x69 },
	},
};

/*
 * DES test vectors.
 */
#define DES_ENC_TEST_VECTORS		10
#define DES_DEC_TEST_VECTORS		4
#define DES_CBC_ENC_TEST_VECTORS	5
#define DES_CBC_DEC_TEST_VECTORS	4
#define DES3_EDE_ENC_TEST_VECTORS	3
#define DES3_EDE_DEC_TEST_VECTORS	3

struct cipher_testvec des_enc_tv_template[] = {
	{ /* From Applied Cryptography */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7 },
		.ilen	= 8,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d },
		.rlen	= 8,
	}, { /* Same key, different plaintext block */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 },
		.ilen	= 8,
		.result	= { 0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b },
		.rlen	= 8,
	}, { /* Sbox test from NBS */
		.key	= { 0x7c, 0xa1, 0x10, 0x45, 0x4a, 0x1a, 0x6e, 0x57 },
		.klen	= 8,
		.input	= { 0x01, 0xa1, 0xd6, 0xd0, 0x39, 0x77, 0x67, 0x42 },
		.ilen	= 8,
		.result	= { 0x69, 0x0f, 0x5b, 0x0d, 0x9a, 0x26, 0x93, 0x9b },
		.rlen	= 8,
	}, { /* Three blocks */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
			    0xca, 0xfe, 0xba, 0xbe, 0xfe, 0xed, 0xbe, 0xef },
		.ilen	= 24,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b,
			    0xb4, 0x99, 0x26, 0xf7, 0x1f, 0xe1, 0xd4, 0x90 },
		.rlen	= 24,
	}, { /* Weak key */
		.fail	= 1,
		.wk	= 1,
		.key	= { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7 },
		.ilen	= 8,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d },
		.rlen	= 8,
	}, { /* Two blocks -- for testing encryption across pages */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 },
		.ilen	= 16,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b },
		.rlen	= 16,
		.np	= 2,
		.tap	= { 8, 8 }
	}, { /* Four blocks -- for testing encryption with chunking */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
			    0xca, 0xfe, 0xba, 0xbe, 0xfe, 0xed, 0xbe, 0xef,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 },
		.ilen	= 32,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b,
			    0xb4, 0x99, 0x26, 0xf7, 0x1f, 0xe1, 0xd4, 0x90,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b },
		.rlen	= 32,
		.np	= 3,
		.tap	= { 14, 10, 8 } 
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
			    0xca, 0xfe, 0xba, 0xbe, 0xfe, 0xed, 0xbe, 0xef },
		.ilen	= 24,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b,
			    0xb4, 0x99, 0x26, 0xf7, 0x1f, 0xe1, 0xd4, 0x90 },
		.rlen	= 24,
		.np	= 4,
		.tap	= { 2, 1, 3, 18 } 
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 },
		.ilen	= 16,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b },
		.rlen	= 16,
		.np	= 5,
		.tap	= { 2, 2, 2, 2, 8 } 
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7 },
		.ilen	= 8,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d },
		.rlen	= 8,
		.np	= 8,
		.tap	= { 1, 1, 1, 1, 1, 1, 1, 1 } 
	},
};

struct cipher_testvec des_dec_tv_template[] = {
	{ /* From Applied Cryptography */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7 },
		.rlen	= 8,
	}, { /* Sbox test from NBS */
		.key	= { 0x7c, 0xa1, 0x10, 0x45, 0x4a, 0x1a, 0x6e, 0x57 },		
		.klen	= 8,
		.input	= { 0x69, 0x0f, 0x5b, 0x0d, 0x9a, 0x26, 0x93, 0x9b },
		.ilen	= 8,
		.result	= { 0x01, 0xa1, 0xd6, 0xd0, 0x39, 0x77, 0x67, 0x42 },
		.rlen	= 8,
	}, { /* Two blocks, for chunking test */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0x69, 0x0f, 0x5b, 0x0d, 0x9a, 0x26, 0x93, 0x9b },
		.ilen	= 16,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0xa3, 0x99, 0x7b, 0xca, 0xaf, 0x69, 0xa0, 0xf5 },
		.rlen	= 16,
		.np	= 2,
		.tap	= { 8, 8 }
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0x69, 0x0f, 0x5b, 0x0d, 0x9a, 0x26, 0x93, 0x9b },
		.ilen	= 16,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0xa3, 0x99, 0x7b, 0xca, 0xaf, 0x69, 0xa0, 0xf5 },
		.rlen	= 16,
		.np	= 3,
		.tap	= { 3, 12, 1 }
	},
};

struct cipher_testvec des_cbc_enc_tv_template[] = {
	{ /* From OpenSSL */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
		.klen	= 8,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},		
		.input	= { 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20, 
			    0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74, 
			    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20 }, 
		.ilen	= 24,
		.result	= { 0xcc, 0xd1, 0x73, 0xff, 0xab, 0x20, 0x39, 0xf4, 
			    0xac, 0xd8, 0xae, 0xfd, 0xdf, 0xd8, 0xa1, 0xeb, 
			    0x46, 0x8e, 0x91, 0x15, 0x78, 0x88, 0xba, 0x68 }, 
		.rlen	= 24,
	}, { /* FIPS Pub 81 */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef },		
		.input	= { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 },
		.ilen	= 8,
		.result	= { 0xe5, 0xc7, 0xcd, 0xde, 0x87, 0x2b, 0xf2, 0x7c },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0xe5, 0xc7, 0xcd, 0xde, 0x87, 0x2b, 0xf2, 0x7c },
		.input	= { 0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20 },
		.ilen	= 8,
		.result	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.rlen	= 8,
	}, {	
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.input	= { 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6c, 0x6c, 0x20 },
		.ilen	= 8,
		.result	= { 0x68, 0x37, 0x88, 0x49, 0x9a, 0x7c, 0x05, 0xf6 },
		.rlen	= 8,
	}, { /* Copy of openssl vector for chunk testing */	
	     /* From OpenSSL */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
		.klen	= 8,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},		
		.input	= { 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20, 
			    0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74, 
			    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20 }, 
		.ilen	= 24,
		.result	= { 0xcc, 0xd1, 0x73, 0xff, 0xab, 0x20, 0x39, 0xf4, 
			    0xac, 0xd8, 0xae, 0xfd, 0xdf, 0xd8, 0xa1, 0xeb, 
			    0x46, 0x8e, 0x91, 0x15, 0x78, 0x88, 0xba, 0x68 }, 
		.rlen	= 24,
		.np	= 2,
		.tap	= { 13, 11 }
	},
};

struct cipher_testvec des_cbc_dec_tv_template[] = {
	{ /* FIPS Pub 81 */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef },
		.input	= { 0xe5, 0xc7, 0xcd, 0xde, 0x87, 0x2b, 0xf2, 0x7c },
		.ilen	= 8,
		.result	= { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0xe5, 0xc7, 0xcd, 0xde, 0x87, 0x2b, 0xf2, 0x7c },		
		.input	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.ilen	= 8,
		.result	= { 0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20 }, 
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.input	= { 0x68, 0x37, 0x88, 0x49, 0x9a, 0x7c, 0x05, 0xf6 }, 
		.ilen	= 8,
		.result	= { 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6c, 0x6c, 0x20 },
		.rlen	= 8,
	}, { /* Copy of above, for chunk testing */	
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.input	= { 0x68, 0x37, 0x88, 0x49, 0x9a, 0x7c, 0x05, 0xf6 }, 
		.ilen	= 8,
		.result	= { 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6c, 0x6c, 0x20 },
		.rlen	= 8,
		.np	= 2,
		.tap	= { 4, 4 }
	},
};

/*
 * We really need some more test vectors, especially for DES3 CBC.
 */
struct cipher_testvec des3_ede_enc_tv_template[] = {
	{ /* These are from openssl */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},
		.klen	= 24,
		.input	= { 0x73, 0x6f, 0x6d, 0x65, 0x64, 0x61, 0x74, 0x61 },
		.ilen	= 8,
		.result	= { 0x18, 0xd7, 0x48, 0xe5, 0x63, 0x62, 0x05, 0x72 },
		.rlen	= 8,
	}, {
		.key	= { 0x03, 0x52, 0x02, 0x07, 0x67, 0x20, 0x82, 0x17,
			    0x86, 0x02, 0x87, 0x66, 0x59, 0x08, 0x21, 0x98,
			    0x64, 0x05, 0x6a, 0xbd, 0xfe, 0xa9, 0x34, 0x57 },
		.klen	= 24,
		.input	= { 0x73, 0x71, 0x75, 0x69, 0x67, 0x67, 0x6c, 0x65 },
		.ilen	= 8,
		.result	= { 0xc0, 0x7d, 0x2a, 0x0f, 0xa5, 0x66, 0xfa, 0x30 },
		.rlen	= 8,
	}, {
		.key	= { 0x10, 0x46, 0x10, 0x34, 0x89, 0x98, 0x80, 0x20,
			    0x91, 0x07, 0xd0, 0x15, 0x89, 0x19, 0x01, 0x01,
			    0x19, 0x07, 0x92, 0x10, 0x98, 0x1a, 0x01, 0x01 },
		.klen	= 24,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0xe1, 0xef, 0x62, 0xc3, 0x32, 0xfe, 0x82, 0x5b },
		.rlen	= 8,
	},
};

struct cipher_testvec des3_ede_dec_tv_template[] = {
	{ /* These are from openssl */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},
		.klen	= 24,
		.input	= { 0x18, 0xd7, 0x48, 0xe5, 0x63, 0x62, 0x05, 0x72 },
		.ilen	= 8,
		.result	= { 0x73, 0x6f, 0x6d, 0x65, 0x64, 0x61, 0x74, 0x61 },
		.rlen	= 8,
	}, {
		.key	= { 0x03, 0x52, 0x02, 0x07, 0x67, 0x20, 0x82, 0x17,
			    0x86, 0x02, 0x87, 0x66, 0x59, 0x08, 0x21, 0x98,
			    0x64, 0x05, 0x6a, 0xbd, 0xfe, 0xa9, 0x34, 0x57 },
		.klen	= 24,
		.input	= { 0xc0, 0x7d, 0x2a, 0x0f, 0xa5, 0x66, 0xfa, 0x30 },
		.ilen	= 8,
		.result	= { 0x73, 0x71, 0x75, 0x69, 0x67, 0x67, 0x6c, 0x65 },
		.rlen	= 8,
	}, {
		.key	= { 0x10, 0x46, 0x10, 0x34, 0x89, 0x98, 0x80, 0x20,
			    0x91, 0x07, 0xd0, 0x15, 0x89, 0x19, 0x01, 0x01,
			    0x19, 0x07, 0x92, 0x10, 0x98, 0x1a, 0x01, 0x01 },
		.klen	= 24,
		.input	= { 0xe1, 0xef, 0x62, 0xc3, 0x32, 0xfe, 0x82, 0x5b },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	},
};

/*
 * Blowfish test vectors.
 */
#define BF_ENC_TEST_VECTORS	6
#define BF_DEC_TEST_VECTORS	6
#define BF_CBC_ENC_TEST_VECTORS	1
#define BF_CBC_DEC_TEST_VECTORS	1

struct cipher_testvec bf_enc_tv_template[] = {
	{ /* DES test vectors from OpenSSL */
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, },
		.klen	= 8,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0x4e, 0xf9, 0x97, 0x45, 0x61, 0x98, 0xdd, 0x78 },
		.rlen	= 8,
	}, {
		.key	= { 0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0xa7, 0x90, 0x79, 0x51, 0x08, 0xea, 0x3c, 0xae },
		.rlen	= 8,
	}, {
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 8,
		.input	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 8,
		.result	= { 0xe8, 0x7a, 0x24, 0x4e, 0x2c, 0xc8, 0x5e, 0x82 },
		.rlen	= 8,
	}, { /* Vary the keylength... */	
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f },
		.klen	= 16,
		.input	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 8,
		.result	= { 0x93, 0x14, 0x28, 0x87, 0xee, 0x3b, 0xe1, 0x5c },
		.rlen	= 8,
	}, {
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f,
			    0x00, 0x11, 0x22, 0x33, 0x44 },
		.klen	= 21,
		.input	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 8,
		.result	= { 0xe6, 0xf5, 0x1e, 0xd7, 0x9b, 0x9d, 0xb2, 0x1f },
		.rlen	= 8,
	}, { /* Generated with bf488 */
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x04, 0x68, 0x91, 0x04, 0xc2, 0xfd, 0x3b, 0x2f, 
			    0x58, 0x40, 0x23, 0x64, 0x1a, 0xba, 0x61, 0x76, 
			    0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e, 
			    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
		.klen	= 56,
		.input	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 8,
		.result	= { 0xc0, 0x45, 0x04, 0x01, 0x2e, 0x4e, 0x1f, 0x53 },
		.rlen	= 8,
	},
};

struct cipher_testvec bf_dec_tv_template[] = {
	{ /* DES test vectors from OpenSSL */
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 8,
		.input	= { 0x4e, 0xf9, 0x97, 0x45, 0x61, 0x98, 0xdd, 0x78 },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e },
		.klen	= 8,
		.input	= { 0xa7, 0x90, 0x79, 0x51, 0x08, 0xea, 0x3c, 0xae },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	}, {
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 8,
		.input	= { 0xe8, 0x7a, 0x24, 0x4e, 0x2c, 0xc8, 0x5e, 0x82 },
		.ilen	= 8,
		.result	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 8,
	}, { /* Vary the keylength... */	
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f },
		.klen	= 16,
		.input	= { 0x93, 0x14, 0x28, 0x87, 0xee, 0x3b, 0xe1, 0x5c },
		.ilen	= 8,
		.result	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 8,
	}, {
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f,
			    0x00, 0x11, 0x22, 0x33, 0x44 },
		.klen	= 21,
		.input	= { 0xe6, 0xf5, 0x1e, 0xd7, 0x9b, 0x9d, 0xb2, 0x1f },
		.ilen	= 8,
		.result	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 8,
	}, { /* Generated with bf488, using OpenSSL, Libgcrypt and Nettle */
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x04, 0x68, 0x91, 0x04, 0xc2, 0xfd, 0x3b, 0x2f, 
			    0x58, 0x40, 0x23, 0x64, 0x1a, 0xba, 0x61, 0x76, 
			    0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e, 
			    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
		.klen	= 56,
		.input	= { 0xc0, 0x45, 0x04, 0x01, 0x2e, 0x4e, 0x1f, 0x53 },
		.ilen	= 8,
		.result	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 8,
	},
};

struct cipher_testvec bf_cbc_enc_tv_template[] = {
	{ /* From OpenSSL */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 16,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },		
		.input	= { 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20,
			    0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
			    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20,
			    0x66, 0x6f, 0x72, 0x20, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 32,
		.result	= { 0x6b, 0x77, 0xb4, 0xd6, 0x30, 0x06, 0xde, 0xe6,
			    0x05, 0xb1, 0x56, 0xe2, 0x74, 0x03, 0x97, 0x93,
			    0x58, 0xde, 0xb9, 0xe7, 0x15, 0x46, 0x16, 0xd9,
			    0x59, 0xf1, 0x65, 0x2b, 0xd5, 0xff, 0x92, 0xcc },
		.rlen	= 32,
	},
};

struct cipher_testvec bf_cbc_dec_tv_template[] = {
	{ /* From OpenSSL */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 16,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },		
		.input	= { 0x6b, 0x77, 0xb4, 0xd6, 0x30, 0x06, 0xde, 0xe6,
			    0x05, 0xb1, 0x56, 0xe2, 0x74, 0x03, 0x97, 0x93,
			    0x58, 0xde, 0xb9, 0xe7, 0x15, 0x46, 0x16, 0xd9,
			    0x59, 0xf1, 0x65, 0x2b, 0xd5, 0xff, 0x92, 0xcc },
		.ilen	= 32,
		.result	= { 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20,
			    0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
			    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20,
			    0x66, 0x6f, 0x72, 0x20, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 32,
	},
};

/*
 * Twofish test vectors.
 */
#define TF_ENC_TEST_VECTORS		3
#define TF_DEC_TEST_VECTORS		3
#define TF_CBC_ENC_TEST_VECTORS		4
#define TF_CBC_DEC_TEST_VECTORS		4

struct cipher_testvec tf_enc_tv_template[] = {
	{
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77 },
		.klen	= 24,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xcf, 0xd1, 0xd2, 0xe5, 0xa9, 0xbe, 0x9c, 0xdf,
			    0x50, 0x1f, 0x13, 0xb8, 0x92, 0xbd, 0x22, 0x48 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.klen	= 32,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x37, 0x52, 0x7b, 0xe0, 0x05, 0x23, 0x34, 0xb8,
			    0x9f, 0x0c, 0xfc, 0xca, 0xe8, 0x7c, 0xfa, 0x20 },
		.rlen	= 16,
	},
};

struct cipher_testvec tf_dec_tv_template[] = {
	{
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77 },
		.klen	= 24,
		.input	= { 0xcf, 0xd1, 0xd2, 0xe5, 0xa9, 0xbe, 0x9c, 0xdf,
			    0x50, 0x1f, 0x13, 0xb8, 0x92, 0xbd, 0x22, 0x48 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.klen	= 32,
		.input	= { 0x37, 0x52, 0x7b, 0xe0, 0x05, 0x23, 0x34, 0xb8,
			    0x9f, 0x0c, 0xfc, 0xca, 0xe8, 0x7c, 0xfa, 0x20 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	},
};

struct cipher_testvec tf_cbc_enc_tv_template[] = {
	{ /* Generated with Nettle */
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { [0 ... 15] = 0x00 },
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },		
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { 0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19 },
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x05, 0xef, 0x8c, 0x61, 0xa8, 0x11, 0x58, 0x26,
			    0x34, 0xba, 0x5c, 0xb7, 0x10, 0x6a, 0xa6, 0x41 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { [0 ... 15] = 0x00 },
		.input	= { [0 ... 47] = 0x00 },
		.ilen	= 48,
		.result	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a,
			    0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19,
			    0x05, 0xef, 0x8c, 0x61, 0xa8, 0x11, 0x58, 0x26,
			    0x34, 0xba, 0x5c, 0xb7, 0x10, 0x6a, 0xa6, 0x41 },
		.rlen	= 48,
	},
};

struct cipher_testvec tf_cbc_dec_tv_template[] = {
	{ /* Reverse of the first four above */
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { [0 ... 15] = 0x00 },
		.input	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },		  
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.input	= { 0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { 0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19 },
		.input	= { 0x05, 0xef, 0x8c, 0x61, 0xa8, 0x11, 0x58, 0x26,
			    0x34, 0xba, 0x5c, 0xb7, 0x10, 0x6a, 0xa6, 0x41 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { [0 ... 15] = 0x00 },
		.input	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a,
			    0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19,
			    0x05, 0xef, 0x8c, 0x61, 0xa8, 0x11, 0x58, 0x26,
			    0x34, 0xba, 0x5c, 0xb7, 0x10, 0x6a, 0xa6, 0x41 },
		.ilen	= 48,
		.result	= { [0 ... 47] = 0x00 },
		.rlen	= 48,
	},
};

/*
 * Serpent test vectors.  These are backwards because Serpent writes
 * octet sequences in right-to-left mode.
 */
#define SERPENT_ENC_TEST_VECTORS	4
#define SERPENT_DEC_TEST_VECTORS	4

#define TNEPRES_ENC_TEST_VECTORS	4
#define TNEPRES_DEC_TEST_VECTORS	4

struct cipher_testvec serpent_enc_tv_template[] = 
{
	{
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.ilen	= 16,
		.result	= { 0x12, 0x07, 0xfc, 0xce, 0x9b, 0xd0, 0xd6, 0x47,
			    0x6a, 0xe9, 0x8f, 0xbe, 0xd1, 0x43, 0xa0, 0xe2 },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.ilen	= 16,
		.result	= { 0x4c, 0x7d, 0x8a, 0x32, 0x80, 0x72, 0xa2, 0x2c,
			    0x82, 0x3e, 0x4a, 0x1f, 0x3a, 0xcd, 0xa1, 0x6d },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.ilen	= 16,
		.result	= { 0xde, 0x26, 0x9f, 0xf8, 0x33, 0xe4, 0x32, 0xb8,
			    0x5b, 0x2e, 0x88, 0xd2, 0x70, 0x1c, 0xe7, 0x5c },
		.rlen	= 16,
	}, {
		.key	= { [15] = 0x80 },
		.klen	= 16,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xdd, 0xd2, 0x6b, 0x98, 0xa5, 0xff, 0xd8, 0x2c,
			    0x05, 0x34, 0x5a, 0x9d, 0xad, 0xbf, 0xaf, 0x49},
		.rlen	= 16,
	},
};

struct cipher_testvec tnepres_enc_tv_template[] = 
{
	{ /* KeySize=128, PT=0, I=1 */
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.key    = { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen   = 16,
		.ilen	= 16,
		.result	= { 0x49, 0xaf, 0xbf, 0xad, 0x9d, 0x5a, 0x34, 0x05, 
			    0x2c, 0xd8, 0xff, 0xa5, 0x98, 0x6b, 0xd2, 0xdd },
		.rlen	= 16,
	}, { /* KeySize=192, PT=0, I=1 */
		.key	= { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 24,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 16,
		.result	= { 0xe7, 0x8e, 0x54, 0x02, 0xc7, 0x19, 0x55, 0x68, 
			    0xac, 0x36, 0x78, 0xf7, 0xa3, 0xf6, 0x0c, 0x66 },
		.rlen	= 16,
	}, { /* KeySize=256, PT=0, I=1 */
		.key	= { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 32,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 16,
		.result	= { 0xab, 0xed, 0x96, 0xe7, 0x66, 0xbf, 0x28, 0xcb, 
			    0xc0, 0xeb, 0xd2, 0x1a, 0x82, 0xef, 0x08, 0x19 },
		.rlen	= 16,
	}, { /* KeySize=256, I=257 */
	        .key	= { 0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18,
			    0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10,
			    0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08,
			    0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00 },
		.klen	= 32,
		.input	= { 0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08,
			    0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00 },
		.ilen	= 16,
		.result	= { 0x5c, 0xe7, 0x1c, 0x70, 0xd2, 0x88, 0x2e, 0x5b, 
			    0xb8, 0x32, 0xe4, 0x33, 0xf8, 0x9f, 0x26, 0xde },
		.rlen	= 16,
	},
};


struct cipher_testvec serpent_dec_tv_template[] = 
{
	{
		.input	= { 0x12, 0x07, 0xfc, 0xce, 0x9b, 0xd0, 0xd6, 0x47,
			    0x6a, 0xe9, 0x8f, 0xbe, 0xd1, 0x43, 0xa0, 0xe2 },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0x4c, 0x7d, 0x8a, 0x32, 0x80, 0x72, 0xa2, 0x2c,
			    0x82, 0x3e, 0x4a, 0x1f, 0x3a, 0xcd, 0xa1, 0x6d },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0xde, 0x26, 0x9f, 0xf8, 0x33, 0xe4, 0x32, 0xb8,
			    0x5b, 0x2e, 0x88, 0xd2, 0x70, 0x1c, 0xe7, 0x5c },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { [15] = 0x80 },
		.klen	= 16,
		.input	= { 0xdd, 0xd2, 0x6b, 0x98, 0xa5, 0xff, 0xd8, 0x2c,
			    0x05, 0x34, 0x5a, 0x9d, 0xad, 0xbf, 0xaf, 0x49},
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	},
};

struct cipher_testvec tnepres_dec_tv_template[] =
{
	{
		.input	= { 0x41, 0xcc, 0x6b, 0x31, 0x59, 0x31, 0x45, 0x97,
			    0x6d, 0x6f, 0xbb, 0x38, 0x4b, 0x37, 0x21, 0x28 },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0xea, 0xf4, 0xd7, 0xfc, 0xd8, 0x01, 0x34, 0x47, 
			    0x81, 0x45, 0x0b, 0xfa, 0x0c, 0xd6, 0xad, 0x6e },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0x64, 0xa9, 0x1a, 0x37, 0xed, 0x9f, 0xe7, 0x49, 
			    0xa8, 0x4e, 0x76, 0xd6, 0xf5, 0x0d, 0x78, 0xee },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, { /* KeySize=128, I=121 */
		.key	= { [15] = 0x80 },
		.klen	= 16,
		.input	= { 0x3d, 0xda, 0xbf, 0xc0, 0x06, 0xda, 0xab, 0x06, 
			    0x46, 0x2a, 0xf4, 0xef, 0x81, 0x54, 0x4e, 0x26 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	},
};


/* Cast6 test vectors from RFC 2612 */
#define CAST6_ENC_TEST_VECTORS	3
#define CAST6_DEC_TEST_VECTORS  3

struct cipher_testvec cast6_enc_tv_template[] = 
{
	{
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c, 
			    0x0a, 0xf7, 0x56, 0x47, 0xf2, 0x9f, 0x61, 0x5d },
		.klen	= 16,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xc8, 0x42, 0xa0, 0x89, 0x72, 0xb4, 0x3d, 0x20, 
			    0x83, 0x6c, 0x91, 0xd1, 0xb7, 0x53, 0x0f, 0x6b },
		.rlen	= 16,
	}, {
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c, 
			    0xbe, 0xd0, 0xac, 0x83, 0x94, 0x0a, 0xc2, 0x98, 
			    0xba, 0xc7, 0x7a, 0x77, 0x17, 0x94, 0x28, 0x63 },
		.klen	= 24,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x1b, 0x38, 0x6c, 0x02, 0x10, 0xdc, 0xad, 0xcb, 
			    0xdd, 0x0e, 0x41, 0xaa, 0x08, 0xa7, 0xa7, 0xe8 },
		.rlen	= 16,
	}, {
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c,
			    0xbe, 0xd0, 0xac, 0x83, 0x94, 0x0a, 0xc2, 0x98,
			    0x8d, 0x7c, 0x47, 0xce, 0x26, 0x49, 0x08, 0x46,
			    0x1c, 0xc1, 0xb5, 0x13, 0x7a, 0xe6, 0xb6, 0x04 },				  
		.klen	= 32,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x4f, 0x6a, 0x20, 0x38, 0x28, 0x68, 0x97, 0xb9, 
			    0xc9, 0x87, 0x01, 0x36, 0x55, 0x33, 0x17, 0xfa },
		.rlen	= 16,
	},
};

struct cipher_testvec cast6_dec_tv_template[] = 
{
	{
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c, 
			    0x0a, 0xf7, 0x56, 0x47, 0xf2, 0x9f, 0x61, 0x5d },
		.klen	= 16,
		.input	= { 0xc8, 0x42, 0xa0, 0x89, 0x72, 0xb4, 0x3d, 0x20, 
			    0x83, 0x6c, 0x91, 0xd1, 0xb7, 0x53, 0x0f, 0x6b },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c, 
			    0xbe, 0xd0, 0xac, 0x83, 0x94, 0x0a, 0xc2, 0x98, 
			    0xba, 0xc7, 0x7a, 0x77, 0x17, 0x94, 0x28, 0x63 },
		.klen	= 24,
		.input	= { 0x1b, 0x38, 0x6c, 0x02, 0x10, 0xdc, 0xad, 0xcb, 
			    0xdd, 0x0e, 0x41, 0xaa, 0x08, 0xa7, 0xa7, 0xe8 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c,
			    0xbe, 0xd0, 0xac, 0x83, 0x94, 0x0a, 0xc2, 0x98,
			    0x8d, 0x7c, 0x47, 0xce, 0x26, 0x49, 0x08, 0x46,
			    0x1c, 0xc1, 0xb5, 0x13, 0x7a, 0xe6, 0xb6, 0x04 },				  
		.klen	= 32,
		.input	= { 0x4f, 0x6a, 0x20, 0x38, 0x28, 0x68, 0x97, 0xb9, 
			    0xc9, 0x87, 0x01, 0x36, 0x55, 0x33, 0x17, 0xfa },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	},
};


/*
 * AES test vectors.
 */
#define AES_ENC_TEST_VECTORS 3
#define AES_DEC_TEST_VECTORS 3

struct cipher_testvec aes_enc_tv_template[] = { 
	{ /* From FIPS-197 */
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.ilen	= 16,
		.result	= { 0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
			    0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 },
		.klen	= 24,
		.input	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.ilen	= 16,
		.result	= { 0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
			    0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91 },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.ilen	= 16,
		.result	= { 0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
			    0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89 },
		.rlen	= 16,
	},
};

struct cipher_testvec aes_dec_tv_template[] = { 
	{ /* From FIPS-197 */
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
			    0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a },
		.ilen	= 16,
		.result	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 },
		.klen	= 24,
		.input	= { 0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
			    0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91 },
		.ilen	= 16,
		.result	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },  
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
			    0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89 },
		.ilen	= 16,
		.result	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.rlen	= 16,
	},
};

/* Cast5 test vectors from RFC 2144 */
#define CAST5_ENC_TEST_VECTORS	3
#define CAST5_DEC_TEST_VECTORS	3

struct cipher_testvec cast5_enc_tv_template[] =
{
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
			    0x23, 0x45, 0x67, 0x89, 0x34, 0x56, 0x78, 0x9a },
		.klen	= 16,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0x23, 0x8b, 0x4f, 0xe5, 0x84, 0x7e, 0x44, 0xb2 },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
			    0x23, 0x45 },
		.klen	= 10,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0xeb, 0x6a, 0x71, 0x1a, 0x2c, 0x02, 0x27, 0x1b },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12 },
		.klen	= 5,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0x7a, 0xc8, 0x16, 0xd1, 0x6e, 0x9b, 0x30, 0x2e },
		.rlen	= 8,
	},
};

struct cipher_testvec cast5_dec_tv_template[] =
{
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
			    0x23, 0x45, 0x67, 0x89, 0x34, 0x56, 0x78, 0x9a },
		.klen	= 16,
		.input	= { 0x23, 0x8b, 0x4f, 0xe5, 0x84, 0x7e, 0x44, 0xb2 },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
			    0x23, 0x45 },
		.klen	= 10,
		.input	= { 0xeb, 0x6a, 0x71, 0x1a, 0x2c, 0x02, 0x27, 0x1b },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12 },
		.klen	= 5,
		.input	= { 0x7a, 0xc8, 0x16, 0xd1, 0x6e, 0x9b, 0x30, 0x2e },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	},
};

/* 
 * ARC4 test vectors from OpenSSL 
 */
#define ARC4_ENC_TEST_VECTORS	7
#define ARC4_DEC_TEST_VECTORS	7

struct cipher_testvec arc4_enc_tv_template[] =
{
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0x75, 0xb7, 0x87, 0x80, 0x99, 0xe0, 0xc5, 0x96 },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0x74, 0x94, 0xc2, 0xe7, 0x10, 0x4b, 0x08, 0x79 },
		.rlen	= 8,
	}, {
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 8,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0xde, 0x18, 0x89, 0x41, 0xa3, 0x37, 0x5d, 0x3a },
		.rlen	= 8,
	}, {
		.key	= { 0xef, 0x01, 0x23, 0x45},
		.klen	= 4,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00 },
		.ilen	= 20,
		.result	= { 0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			    0xbd, 0x61, 0x5a, 0x11, 0x62, 0xe1, 0xc7, 0xba,
			    0x36, 0xb6, 0x78, 0x58 },
		.rlen	= 20,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78 },
		.ilen	= 28,
		.result	= { 0x66, 0xa0, 0x94, 0x9f, 0x8a, 0xf7, 0xd6, 0x89,
			    0x1f, 0x7f, 0x83, 0x2b, 0xa8, 0x33, 0xc0, 0x0c,
			    0x89, 0x2e, 0xbe, 0x30, 0x14, 0x3c, 0xe2, 0x87,
			    0x40, 0x01, 0x1e, 0xcf },
		.rlen	= 28,
	}, {
		.key	= { 0xef, 0x01, 0x23, 0x45 },
		.klen	= 4,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00 },
		.ilen	= 10,
		.result	= { 0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			    0xbd, 0x61 },
		.rlen	= 10,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
		            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 16,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF },
		.ilen	= 8,
		.result	= { 0x69, 0x72, 0x36, 0x59, 0x1B, 0x52, 0x42, 0xB1 },
		.rlen	= 8,
	},
};

struct cipher_testvec arc4_dec_tv_template[] =
{
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x75, 0xb7, 0x87, 0x80, 0x99, 0xe0, 0xc5, 0x96 },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x74, 0x94, 0xc2, 0xe7, 0x10, 0x4b, 0x08, 0x79 },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 8,
		.input	= { 0xde, 0x18, 0x89, 0x41, 0xa3, 0x37, 0x5d, 0x3a },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0xef, 0x01, 0x23, 0x45},
		.klen	= 4,
		.input	= { 0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			    0xbd, 0x61, 0x5a, 0x11, 0x62, 0xe1, 0xc7, 0xba,
			    0x36, 0xb6, 0x78, 0x58 },
		.ilen	= 20,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00 },
		.rlen	= 20,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x66, 0xa0, 0x94, 0x9f, 0x8a, 0xf7, 0xd6, 0x89,
			    0x1f, 0x7f, 0x83, 0x2b, 0xa8, 0x33, 0xc0, 0x0c,
			    0x89, 0x2e, 0xbe, 0x30, 0x14, 0x3c, 0xe2, 0x87,
			    0x40, 0x01, 0x1e, 0xcf },
		.ilen	= 28,
		.result	= { 0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78 },
		.rlen	= 28,
	}, {
		.key	= { 0xef, 0x01, 0x23, 0x45 },
		.klen	= 4,
		.input	= { 0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			    0xbd, 0x61 },
		.ilen	= 10,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00 },
		.rlen	= 10,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
		            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 16,
		.input	= { 0x69, 0x72, 0x36, 0x59, 0x1B, 0x52, 0x42, 0xB1 },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF },
		.rlen	= 8,
	},
};

/* 
 * TEA test vectors
 */
#define TEA_ENC_TEST_VECTORS	4
#define TEA_DEC_TEST_VECTORS	4

struct cipher_testvec tea_enc_tv_template[] =
{
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input  = { [0 ... 8] = 0x00 },
		.ilen	= 8,
		.result	= { 0x0a, 0x3a, 0xea, 0x41, 0x40, 0xa9, 0xba, 0x94 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.ilen	= 8,
		.result	= { 0x77, 0x5d, 0x2a, 0x6a, 0xf6, 0xce, 0x92, 0x09 },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74, 
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.ilen	= 16,
		.result	= { 0xbe, 0x7a, 0xbb, 0x81, 0x95, 0x2d, 0x1f, 0x1e, 
			    0xdd, 0x89, 0xa1, 0x25, 0x04, 0x21, 0xdf, 0x95 },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67, 
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72, 
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.ilen	= 32,
		.result	= { 0xe0, 0x4d, 0x5d, 0x3c, 0xb7, 0x8c, 0x36, 0x47, 
			    0x94, 0x18, 0x95, 0x91, 0xa9, 0xfc, 0x49, 0xf8, 
			    0x44, 0xd1, 0x2d, 0xc2, 0x99, 0xb8, 0x08, 0x2a, 
			    0x07, 0x89, 0x73, 0xc2, 0x45, 0x92, 0xc6, 0x90 },
		.rlen	= 32,
	}
};

struct cipher_testvec tea_dec_tv_template[] =
{
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { 0x0a, 0x3a, 0xea, 0x41, 0x40, 0xa9, 0xba, 0x94 },
		.ilen	= 8,
		.result = { [0 ... 8] = 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x77, 0x5d, 0x2a, 0x6a, 0xf6, 0xce, 0x92, 0x09 },
		.ilen	= 8,
		.result	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0xbe, 0x7a, 0xbb, 0x81, 0x95, 0x2d, 0x1f, 0x1e,
                            0xdd, 0x89, 0xa1, 0x25, 0x04, 0x21, 0xdf, 0x95 },
                .ilen   = 16,
		.result	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74, 
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0xe0, 0x4d, 0x5d, 0x3c, 0xb7, 0x8c, 0x36, 0x47,
                            0x94, 0x18, 0x95, 0x91, 0xa9, 0xfc, 0x49, 0xf8,
                            0x44, 0xd1, 0x2d, 0xc2, 0x99, 0xb8, 0x08, 0x2a,
                            0x07, 0x89, 0x73, 0xc2, 0x45, 0x92, 0xc6, 0x90 },
		.ilen	= 32,
		.result	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67, 
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72, 
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.rlen	= 32,
	}
};

/* 
 * XTEA test vectors 
 */
#define XTEA_ENC_TEST_VECTORS	4
#define XTEA_DEC_TEST_VECTORS	4

struct cipher_testvec xtea_enc_tv_template[] =
{
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input  = { [0 ... 8] = 0x00 },
		.ilen	= 8,
		.result	= { 0xaa, 0x22, 0x96, 0xe5, 0x6c, 0x61, 0xf3, 0x45 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.ilen	= 8,
		.result	= { 0x82, 0x3e, 0xeb, 0x35, 0xdc, 0xdd, 0xd9, 0xc3 },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74, 
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.ilen	= 16,
		.result	= { 0xe2, 0x04, 0xdb, 0xf2, 0x89, 0x85, 0x9e, 0xea, 
			    0x61, 0x35, 0xaa, 0xed, 0xb5, 0xcb, 0x71, 0x2c },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67, 
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72, 
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.ilen	= 32,
		.result	= { 0x0b, 0x03, 0xcd, 0x8a, 0xbe, 0x95, 0xfd, 0xb1, 
			    0xc1, 0x44, 0x91, 0x0b, 0xa5, 0xc9, 0x1b, 0xb4, 
			    0xa9, 0xda, 0x1e, 0x9e, 0xb1, 0x3e, 0x2a, 0x8f, 
			    0xea, 0xa5, 0x6a, 0x85, 0xd1, 0xf4, 0xa8, 0xa5 },
		.rlen	= 32,
	}
};

struct cipher_testvec xtea_dec_tv_template[] =
{
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { 0xaa, 0x22, 0x96, 0xe5, 0x6c, 0x61, 0xf3, 0x45 },
		.ilen	= 8,
		.result = { [0 ... 8] = 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x82, 0x3e, 0xeb, 0x35, 0xdc, 0xdd, 0xd9, 0xc3 },
		.ilen	= 8,
		.result	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0xe2, 0x04, 0xdb, 0xf2, 0x89, 0x85, 0x9e, 0xea, 
			    0x61, 0x35, 0xaa, 0xed, 0xb5, 0xcb, 0x71, 0x2c },
		.ilen	= 16,
		.result	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74, 
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0x0b, 0x03, 0xcd, 0x8a, 0xbe, 0x95, 0xfd, 0xb1, 
			    0xc1, 0x44, 0x91, 0x0b, 0xa5, 0xc9, 0x1b, 0xb4, 
			    0xa9, 0xda, 0x1e, 0x9e, 0xb1, 0x3e, 0x2a, 0x8f, 
			    0xea, 0xa5, 0x6a, 0x85, 0xd1, 0xf4, 0xa8, 0xa5 },
		.ilen	= 32,
		.result	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67, 
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72, 
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.rlen	= 32,
	}
};

/*
 * KHAZAD test vectors.
 */
#define KHAZAD_ENC_TEST_VECTORS 5
#define KHAZAD_DEC_TEST_VECTORS 5

struct cipher_testvec khazad_enc_tv_template[] = { 
	{ 
		.key	= { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 16,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0x49, 0xa4, 0xce, 0x32, 0xac, 0x19, 0x0e, 0x3f },
		.rlen	= 8,
	}, {
		.key	= { 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,
			    0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38 },
		.klen	= 16,
		.input	= { 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38 },
		.ilen	= 8,
		.result	= { 0x7e, 0x82, 0x12, 0xa1, 0Xd9, 0X5b, 0Xe4, 0Xf9 },
		.rlen	= 8,
	}, {
		.key	= { 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2,
			    0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2 },
		.klen	= 16,
		.input	= { 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2 },
		.ilen	= 8,
		.result	= { 0Xaa, 0Xbe, 0Xc1, 0X95, 0Xc5, 0X94, 0X1a, 0X9c },
		.rlen	= 8,
	}, {
		.key	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.klen	= 16,
		.input	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.ilen	= 8,
		.result = { 0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 },
		.rlen	= 8,
	}, {
		.key	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.klen	= 16,
		.input	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f ,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.ilen	= 16,
		.result = { 0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 ,
			    0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 },
		.rlen	= 16,
	},
};

struct cipher_testvec khazad_dec_tv_template[] = { 
	{
		.key	= { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 16,
		.input	= { 0X49, 0Xa4, 0Xce, 0X32, 0Xac, 0X19, 0X0e, 0X3f },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,
			    0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38 },
		.klen	= 16,
		.input	= { 0X7e, 0X82, 0X12, 0Xa1, 0Xd9, 0X5b, 0Xe4, 0Xf9 },
		.ilen	= 8,
		.result	= { 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38 },
		.rlen	= 8,
	}, {
		.key	= { 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2,
			    0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2 },
		.klen	= 16,
		.input	= { 0Xaa, 0Xbe, 0Xc1, 0X95, 0Xc5, 0X94, 0X1a, 0X9c },
		.ilen	= 8,
		.result	= { 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2 },
		.rlen	= 8,
	}, {
		.key	= { 0x2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.klen	= 16,
		.input = { 0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 },
		.ilen	= 8,
		.result	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.rlen	= 8,
	}, {
		.key	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.klen	= 16,
		.input = { 0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 ,
			    0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 },
		.ilen	= 16,
		.result	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f ,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.rlen	= 16,
	},
};

/*
 * Anubis test vectors.
 */

#define ANUBIS_ENC_TEST_VECTORS			5
#define ANUBIS_DEC_TEST_VECTORS			5
#define ANUBIS_CBC_ENC_TEST_VECTORS		2
#define ANUBIS_CBC_DEC_TEST_VECTORS		2

struct cipher_testvec anubis_enc_tv_template[] = {
	{
		.key	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.klen	= 16,
		.input	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.ilen	= 16,
		.result	= { 0x6d, 0xc5, 0xda, 0xa2, 0x26, 0x7d, 0x62, 0x6f, 
			    0x08, 0xb7, 0x52, 0x8e, 0x6e, 0x6e, 0x86, 0x90 },
		.rlen	= 16,
	}, {

		.key	= { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 
			    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 
			    0x03, 0x03, 0x03, 0x03 },
		.klen	= 20,
		.input	= { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 
			    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03 },
		.ilen	= 16,
		.result	= { 0xdb, 0xf1, 0x42, 0xf4, 0xd1, 0x8a, 0xc7, 0x49, 
			    0x87, 0x41, 0x6f, 0x82, 0x0a, 0x98, 0x64, 0xae },
		.rlen	= 16,
	}, {
		.key	= { 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 
			    0x24, 0x24, 0x24, 0x24 },
		.klen	= 28,
		.input	= { 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24 },
		.ilen	= 16,
		.result	= { 0xfd, 0x1b, 0x4a, 0xe3, 0xbf, 0xf0, 0xad, 0x3d, 
			    0x06, 0xd3, 0x61, 0x27, 0xfd, 0x13, 0x9e, 0xde },
		.rlen	= 16,
	}, {
		.key	= { 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25 },
		.klen	= 32,
		.input	= { 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25 },
		.ilen	= 16,
		.result	= { 0x1a, 0x91, 0xfb, 0x2b, 0xb7, 0x78, 0x6b, 0xc4, 
		            0x17, 0xd9, 0xff, 0x40, 0x3b, 0x0e, 0xe5, 0xfe },
		.rlen	= 16,
	}, {
		.key	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.klen	= 40,
		.input	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.ilen	= 16,
		.result = { 0xa5, 0x2c, 0x85, 0x6f, 0x9c, 0xba, 0xa0, 0x97, 
			    0x9e, 0xc6, 0x84, 0x0f, 0x17, 0x21, 0x07, 0xee },
		.rlen	= 16,
	},
};

struct cipher_testvec anubis_dec_tv_template[] = {
	{
		.key	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.klen	= 16,
		.input	= { 0x6d, 0xc5, 0xda, 0xa2, 0x26, 0x7d, 0x62, 0x6f, 
			    0x08, 0xb7, 0x52, 0x8e, 0x6e, 0x6e, 0x86, 0x90 },
		.ilen	= 16,
		.result	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.rlen	= 16,
	}, {

		.key	= { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 
			    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 
			    0x03, 0x03, 0x03, 0x03 },
		.klen	= 20,
		.input	= { 0xdb, 0xf1, 0x42, 0xf4, 0xd1, 0x8a, 0xc7, 0x49, 
			    0x87, 0x41, 0x6f, 0x82, 0x0a, 0x98, 0x64, 0xae },
		.ilen	= 16,
		.result	= { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 
			    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03 },
		.rlen	= 16,
	}, {
		.key	= { 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 
			    0x24, 0x24, 0x24, 0x24 },
		.klen	= 28,
		.input	= { 0xfd, 0x1b, 0x4a, 0xe3, 0xbf, 0xf0, 0xad, 0x3d, 
			    0x06, 0xd3, 0x61, 0x27, 0xfd, 0x13, 0x9e, 0xde },
		.ilen	= 16,
		.result	= { 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24 },
		.rlen	= 16,
	}, {
		.key	= { 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25 },
		.klen	= 32,
		.input	= { 0x1a, 0x91, 0xfb, 0x2b, 0xb7, 0x78, 0x6b, 0xc4, 
		            0x17, 0xd9, 0xff, 0x40, 0x3b, 0x0e, 0xe5, 0xfe },
		.ilen	= 16,
		.result	= { 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25 },
		.rlen	= 16,
	}, {
		.key	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.input = {  0xa5, 0x2c, 0x85, 0x6f, 0x9c, 0xba, 0xa0, 0x97, 
			    0x9e, 0xc6, 0x84, 0x0f, 0x17, 0x21, 0x07, 0xee },
		.klen	= 40,
		.ilen	= 16,
		.result	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.rlen	= 16,
	},
};

struct cipher_testvec anubis_cbc_enc_tv_template[] = {
	{
		.key	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.klen	= 16,
		.input	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.ilen	= 32,
		.result	= { 0x6d, 0xc5, 0xda, 0xa2, 0x26, 0x7d, 0x62, 0x6f, 
			    0x08, 0xb7, 0x52, 0x8e, 0x6e, 0x6e, 0x86, 0x90, 
			    0x86, 0xd8, 0xb5, 0x6f, 0x98, 0x5e, 0x8a, 0x66,
			    0x4f, 0x1f, 0x78, 0xa1, 0xbb, 0x37, 0xf1, 0xbe },
		.rlen	= 32,
	}, {
		.key	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.klen	= 40,
		.input	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.ilen	= 32,
		.result = { 0xa5, 0x2c, 0x85, 0x6f, 0x9c, 0xba, 0xa0, 0x97, 
			    0x9e, 0xc6, 0x84, 0x0f, 0x17, 0x21, 0x07, 0xee, 
			    0xa2, 0xbc, 0x06, 0x98, 0xc6, 0x4b, 0xda, 0x75, 
			    0x2e, 0xaa, 0xbe, 0x58, 0xce, 0x01, 0x5b, 0xc7 },
		.rlen	= 32,
	},
};

struct cipher_testvec anubis_cbc_dec_tv_template[] = {
	{
		.key	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.klen	= 16,
		.input	= { 0x6d, 0xc5, 0xda, 0xa2, 0x26, 0x7d, 0x62, 0x6f, 
			    0x08, 0xb7, 0x52, 0x8e, 0x6e, 0x6e, 0x86, 0x90, 
			    0x86, 0xd8, 0xb5, 0x6f, 0x98, 0x5e, 0x8a, 0x66,
			    0x4f, 0x1f, 0x78, 0xa1, 0xbb, 0x37, 0xf1, 0xbe },
		.ilen	= 32,
		.result	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.rlen	= 32,
	}, {
		.key	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.klen	= 40,
		.input = { 0xa5, 0x2c, 0x85, 0x6f, 0x9c, 0xba, 0xa0, 0x97, 
			    0x9e, 0xc6, 0x84, 0x0f, 0x17, 0x21, 0x07, 0xee, 
			    0xa2, 0xbc, 0x06, 0x98, 0xc6, 0x4b, 0xda, 0x75, 
			    0x2e, 0xaa, 0xbe, 0x58, 0xce, 0x01, 0x5b, 0xc7 },
		.ilen	= 32,
		.result	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.rlen	= 32,
	},
};

/*
 * Compression stuff.
 */
#define COMP_BUF_SIZE           512

struct comp_testvec {
	int inlen, outlen;
	char input[COMP_BUF_SIZE];
	char output[COMP_BUF_SIZE];
};

/*
 * Deflate test vectors (null-terminated strings).
 * Params: winbits=11, Z_DEFAULT_COMPRESSION, MAX_MEM_LEVEL.
 */
#define DEFLATE_COMP_TEST_VECTORS 2
#define DEFLATE_DECOMP_TEST_VECTORS 2

struct comp_testvec deflate_comp_tv_template[] = {
	{
		.inlen	= 70,
		.outlen	= 38,
	  	.input	= "Join us now and share the software "
			  "Join us now and share the software ",
		.output	= { 0xf3, 0xca, 0xcf, 0xcc, 0x53, 0x28, 0x2d, 0x56,
			    0xc8, 0xcb, 0x2f, 0x57, 0x48, 0xcc, 0x4b, 0x51,
			    0x28, 0xce, 0x48, 0x2c, 0x4a, 0x55, 0x28, 0xc9,
			    0x48, 0x55, 0x28, 0xce, 0x4f, 0x2b, 0x29, 0x07,
			    0x71, 0xbc, 0x08, 0x2b, 0x01, 0x00 },
	}, {
		.inlen	= 191,
		.outlen	= 122,
		.input	= "This document describes a compression method based on the DEFLATE"
			  "compression algorithm.  This document defines the application of "
			  "the DEFLATE algorithm to the IP Payload Compression Protocol.",
		.output	= { 0x5d, 0x8d, 0x31, 0x0e, 0xc2, 0x30, 0x10, 0x04,
			    0xbf, 0xb2, 0x2f, 0xc8, 0x1f, 0x10, 0x04, 0x09,
			    0x89, 0xc2, 0x85, 0x3f, 0x70, 0xb1, 0x2f, 0xf8,
			    0x24, 0xdb, 0x67, 0xd9, 0x47, 0xc1, 0xef, 0x49,
			    0x68, 0x12, 0x51, 0xae, 0x76, 0x67, 0xd6, 0x27,
			    0x19, 0x88, 0x1a, 0xde, 0x85, 0xab, 0x21, 0xf2,
			    0x08, 0x5d, 0x16, 0x1e, 0x20, 0x04, 0x2d, 0xad,
			    0xf3, 0x18, 0xa2, 0x15, 0x85, 0x2d, 0x69, 0xc4,
			    0x42, 0x83, 0x23, 0xb6, 0x6c, 0x89, 0x71, 0x9b,
			    0xef, 0xcf, 0x8b, 0x9f, 0xcf, 0x33, 0xca, 0x2f,
			    0xed, 0x62, 0xa9, 0x4c, 0x80, 0xff, 0x13, 0xaf,
			    0x52, 0x37, 0xed, 0x0e, 0x52, 0x6b, 0x59, 0x02,
			    0xd9, 0x4e, 0xe8, 0x7a, 0x76, 0x1d, 0x02, 0x98,
			    0xfe, 0x8a, 0x87, 0x83, 0xa3, 0x4f, 0x56, 0x8a,
			    0xb8, 0x9e, 0x8e, 0x5c, 0x57, 0xd3, 0xa0, 0x79,
			    0xfa, 0x02 },
	},
};

struct comp_testvec deflate_decomp_tv_template[] = {
	{
		.inlen	= 122,
		.outlen	= 191,
		.input	= { 0x5d, 0x8d, 0x31, 0x0e, 0xc2, 0x30, 0x10, 0x04,
			    0xbf, 0xb2, 0x2f, 0xc8, 0x1f, 0x10, 0x04, 0x09,
			    0x89, 0xc2, 0x85, 0x3f, 0x70, 0xb1, 0x2f, 0xf8,
			    0x24, 0xdb, 0x67, 0xd9, 0x47, 0xc1, 0xef, 0x49,
			    0x68, 0x12, 0x51, 0xae, 0x76, 0x67, 0xd6, 0x27,
			    0x19, 0x88, 0x1a, 0xde, 0x85, 0xab, 0x21, 0xf2,
			    0x08, 0x5d, 0x16, 0x1e, 0x20, 0x04, 0x2d, 0xad,
			    0xf3, 0x18, 0xa2, 0x15, 0x85, 0x2d, 0x69, 0xc4,
			    0x42, 0x83, 0x23, 0xb6, 0x6c, 0x89, 0x71, 0x9b,
			    0xef, 0xcf, 0x8b, 0x9f, 0xcf, 0x33, 0xca, 0x2f,
			    0xed, 0x62, 0xa9, 0x4c, 0x80, 0xff, 0x13, 0xaf,
			    0x52, 0x37, 0xed, 0x0e, 0x52, 0x6b, 0x59, 0x02,
			    0xd9, 0x4e, 0xe8, 0x7a, 0x76, 0x1d, 0x02, 0x98,
			    0xfe, 0x8a, 0x87, 0x83, 0xa3, 0x4f, 0x56, 0x8a,
			    0xb8, 0x9e, 0x8e, 0x5c, 0x57, 0xd3, 0xa0, 0x79,
			    0xfa, 0x02 },
		.output	= "This document describes a compression method based on the DEFLATE"
			  "compression algorithm.  This document defines the application of "
			  "the DEFLATE algorithm to the IP Payload Compression Protocol.",
	}, {
		.inlen	= 38,
		.outlen	= 70,
		.input	= { 0xf3, 0xca, 0xcf, 0xcc, 0x53, 0x28, 0x2d, 0x56,
			    0xc8, 0xcb, 0x2f, 0x57, 0x48, 0xcc, 0x4b, 0x51,
			    0x28, 0xce, 0x48, 0x2c, 0x4a, 0x55, 0x28, 0xc9,
			    0x48, 0x55, 0x28, 0xce, 0x4f, 0x2b, 0x29, 0x07,
			    0x71, 0xbc, 0x08, 0x2b, 0x01, 0x00 },
		.output	= "Join us now and share the software "
			  "Join us now and share the software ",
	},
};

/*
 * Michael MIC test vectors from IEEE 802.11i
 */
#define MICHAEL_MIC_TEST_VECTORS 6

struct hash_testvec michael_mic_tv_template[] =
{
	{
		.key = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ksize = 8,
		.plaintext = { },
		.psize = 0,
		.digest = { 0x82, 0x92, 0x5c, 0x1c, 0xa1, 0xd1, 0x30, 0xb8 }
	},
	{
		.key = { 0x82, 0x92, 0x5c, 0x1c, 0xa1, 0xd1, 0x30, 0xb8 },
		.ksize = 8,
		.plaintext = { 'M' },
		.psize = 1,
		.digest = { 0x43, 0x47, 0x21, 0xca, 0x40, 0x63, 0x9b, 0x3f }
	},
	{
		.key = { 0x43, 0x47, 0x21, 0xca, 0x40, 0x63, 0x9b, 0x3f },
		.ksize = 8,
		.plaintext = { 'M', 'i' },
		.psize = 2,
		.digest = { 0xe8, 0xf9, 0xbe, 0xca, 0xe9, 0x7e, 0x5d, 0x29 }
	},
	{
		.key = { 0xe8, 0xf9, 0xbe, 0xca, 0xe9, 0x7e, 0x5d, 0x29 },
		.ksize = 8,
		.plaintext = { 'M', 'i', 'c' },
		.psize = 3,
		.digest = { 0x90, 0x03, 0x8f, 0xc6, 0xcf, 0x13, 0xc1, 0xdb }
	},
	{
		.key = { 0x90, 0x03, 0x8f, 0xc6, 0xcf, 0x13, 0xc1, 0xdb },
		.ksize = 8,
		.plaintext = { 'M', 'i', 'c', 'h' },
		.psize = 4,
		.digest = { 0xd5, 0x5e, 0x10, 0x05, 0x10, 0x12, 0x89, 0x86 }
	},
	{
		.key = { 0xd5, 0x5e, 0x10, 0x05, 0x10, 0x12, 0x89, 0x86 },
		.ksize = 8,
		.plaintext = { 'M', 'i', 'c', 'h', 'a', 'e', 'l' },
		.psize = 7,
		.digest = { 0x0a, 0x94, 0x2b, 0x12, 0x4e, 0xca, 0xa5, 0x46 },
	}
};

#endif	/* _CRYPTO_TCRYPT_H */
