#!/usr/bin/env python3

import rsa
from rsa import PrivateKey

import json

"""
{
  "algorithm" : "RSAES-PKCS1-v1_5",
  "generatorVersion" : "0.8r12",
  "numberOfTests" : 65,
  "header" : [
    "Test vectors of type RsaesPkcs1Decrypt are intended to check the decryption",
    "of RSA encrypted ciphertexts."
  ],
  "notes" : {
    "InvalidPkcs1Padding" : "This is a test vector with an invalid PKCS #1 padding. Implementations must ensure that different error conditions cannot be distinguished, since the information about the error condition can be used for a padding oracle attack. (RFC 8017 Section 7.2.2)"
  },
  "schema" : "rsaes_pkcs1_decrypt_schema.json",
  "testGroups" : [
    {
      "d" : "00d3519bd86edf5dbe9c64a3781713dd049b747f5651fd918cce8a9b80b6d6a2fb6aab334ab569a89b58982e0aae373090299fa227f9652b802d23ee5ca6824c433d01d40bb0473b16190a8bfb137c0a704b4f49b45015bfbee1f670446f07595259d70c7d79ae95f9b2543b0162ebb763ebb81c4b6edff9d197dede1dbe57e40466d156d4dfd5d7634c45534fb2bbbd1bf257817a173c0795fdbcc533976f4a0d04160f299af5dc272a814b36d5bae5116b9542178f1d554cab7a6453035482b981add4bd8ffd5d500839b23e30ebd3f3868f07584cbd12ba089325982860f69f2bdc7077b6c60f58a2272823379e1023704165f7c14f64a18fd5b245a18149b7eda3385a56ba9e79502a27bf13865cde35be15de03d06aa6f08b172b7eeb4b73cbc157019c5d933584a8d1add61b7dbca3b253ca0cf93dc9a8a3aac2f50227022e692f7b4767556620cc928d63e31078c360e2ab4b71a917e19e7eb09380482e5b4ea88202871c2a29ca6f66b2fe304af609520e4f81d64c26768bb812a86679366611f4cf6e89e2bd1d9d7e28729c7f0e4e3152d7ad7f186fa32f01e169ff06a12001bc179dfd0dc942cbabc555f67f5fd043e0a3354340fa490d2f123367be926eb3e1dffe70c3151c87450f3217dc2aa0b5a7069c5a17d55614c9994139f7b37b49e9fe780af65a0e89e2e58feaaffe163bc9d19eafd9cd29b0ef3cef3d01",
      "e" : "010001",
      "keysize" : 4096,
      "n" : "00f601be0dccd04aa40b12f3f191ae17c1f9c8c0b68e7a77e14be25c3c7907cb1d33a6ef418ef41852f32c98392bc5c9aed91c1a1501c503eab89b3ee6f4f8eb2e0fcfc41bd03609cf6a8eb3aa6f0fbe23187b33db4d34b66d128a8aba0a2abf40bb9d13d8e2554569a57ab1d8c61b8cad2dc88599ae0da5346e15dace1bac7bf69737c22f083be9b46bb8b1eab5957b2da740275e96c87195b96fe11452159dafcfd916cee5d749a77bc3905a5ebd387ae445e8fe70f16e9a086639779ceffbfd41557bd99aea6a371a6b4b160615a1a12bc6958d34bce0c85adcbd8392fa10ceca52209d56196ba3d273ce228f1f111192aa92de2a039798a17bcecb4dc6100e6f8ae8c2643f2ae768b2255f082c978e95ca551555f10608231cf8003bbf807969fff1e51914b9a8c9b8f4564645b9e5d705ffad29663f5dae3d76652b422e43f13e6c1491090805c2d1268a74a251177427e33a9a91175c3670b91746008bce1fd231e6e4f2ad70cb43aca5f07600a6d31dd02915243dfdd943a02165da367a6b7e4dae1dd2e8b836903080795d2585076cc1c15dd9e8d2e5e047526569b1bfd395d957eb9fde325d342d14426e71efdc1887515e53cdea5834921f928629e748eed097ac4024e2bf255d70411f87373948cf8e8aa7effa2b0ab47d5166091e1aedec60568b155bd9c27bc55f3ece35f83d636dbcd5abf4853a051db94d5045",
      "privateKeyJwk" : {
        "alg" : "RSA1_5",
        "d" : "01Gb2G7fXb6cZKN4FxPdBJt0f1ZR_ZGMzoqbgLbWovtqqzNKtWmom1iYLgquNzCQKZ-iJ_llK4AtI-5cpoJMQz0B1AuwRzsWGQqL-xN8CnBLT0m0UBW_vuH2cERvB1lSWdcMfXmulfmyVDsBYuu3Y-u4HEtu3_nRl97eHb5X5ARm0VbU39XXY0xFU0-yu70b8leBehc8B5X9vMUzl29KDQQWDyma9dwnKoFLNtW65RFrlUIXjx1VTKt6ZFMDVIK5ga3UvY_9XVAIObI-MOvT84aPB1hMvRK6CJMlmChg9p8r3HB3tsYPWKInKCM3nhAjcEFl98FPZKGP1bJFoYFJt-2jOFpWup55UConvxOGXN41vhXeA9BqpvCLFyt-60tzy8FXAZxdkzWEqNGt1ht9vKOyU8oM-T3JqKOqwvUCJwIuaS97R2dVZiDMko1j4xB4w2Diq0txqRfhnn6wk4BILltOqIIChxwqKcpvZrL-MEr2CVIOT4HWTCZ2i7gSqGZ5NmYR9M9uieK9HZ1-KHKcfw5OMVLXrX8Yb6MvAeFp_wahIAG8F539DclCy6vFVfZ_X9BD4KM1Q0D6SQ0vEjNnvpJus-Hf_nDDFRyHRQ8yF9wqoLWnBpxaF9VWFMmZQTn3s3tJ6f54CvZaDoni5Y_qr_4WO8nRnq_ZzSmw7zzvPQE",
        "dp" : "gVSGqrCgiWv5fxPj6x9_XEkZW0nMO2J3QSo2iHmLGPRkIt9HnLlBs7VOJZZKPWm4l7zINVFg5YtK8p8XRd0sq7Zw9jS5wFjms1FJR_LCfeXtQk9zseHxvkoYiRGgMz86Zohliz7o4yZaUS5N6srcRw7jBOu1IkEjr7RhmE_oUk_gtrMNMqWfbtLcdKlrx8v9G7ROWKcJIjXF1icuEqLIYsuMjPXRCapPscZHKHWhRGDB7VIHxLIrxJTHlH63ymOoyv0xNh0ADd8WotefE92RQNl5FJtIjL9ElFpbaq8TIhv0SR67t_yifKIOIh9Jw8N7ifzy3A4stj-Pipt6FCJQWQ",
        "dq" : "th2E_5NKTkN7Fu4bS5_fSuEzcLU4W956VGShI8A0PfV1-eEo7535RCMNOcyc9dwO2yi350C2nvAkwb_uOfzVNA_66gAQFgxTXcCSDnzYG-Uz0A-lVKH8TT4CxGFWn158p4fxUV7fRbGWt1mITeZSw41ZNM-SUk6Ae007WQvDm8QX7kiFp2HSjdrc5sj9s7lh0-f9SAZN-TQKln-LeZl0OIQfSFeaR23bVQiMMI9o8rKdAcZZelp8jQZihPY-N6aMOHnDKqODZnX9DrJxmIOpGURWHp3X6KprsXFX8IxI-Ob65cPlortrXVgO7GyX3c2b4KSe8oOnAxrXq6jUON9OlQ",
        "e" : "AQAB",
        "kid" : "none",
        "kty" : "RSA",
        "n" : "9gG-DczQSqQLEvPxka4XwfnIwLaOenfhS-JcPHkHyx0zpu9BjvQYUvMsmDkrxcmu2RwaFQHFA-q4mz7m9PjrLg_PxBvQNgnPao6zqm8PviMYezPbTTS2bRKKiroKKr9Au50T2OJVRWmlerHYxhuMrS3IhZmuDaU0bhXazhuse_aXN8IvCDvptGu4seq1lXstp0AnXpbIcZW5b-EUUhWdr8_ZFs7l10mne8OQWl69OHrkRej-cPFumghmOXec7_v9QVV72Zrqajcaa0sWBhWhoSvGlY00vODIWty9g5L6EM7KUiCdVhlro9JzziKPHxERkqqS3ioDl5ihe87LTcYQDm-K6MJkPyrnaLIlXwgsl46VylUVVfEGCCMc-AA7v4B5af_x5RkUuajJuPRWRkW55dcF_60pZj9drj12ZStCLkPxPmwUkQkIBcLRJop0olEXdCfjOpqRF1w2cLkXRgCLzh_SMebk8q1wy0OspfB2AKbTHdApFSQ9_dlDoCFl2jZ6a35Nrh3S6Lg2kDCAeV0lhQdswcFd2ejS5eBHUmVpsb_TldlX65_eMl00LRRCbnHv3BiHUV5TzepYNJIfkoYp50ju0JesQCTivyVdcEEfhzc5SM-Oiqfv-isKtH1RZgkeGu3sYFaLFVvZwnvFXz7ONfg9Y2281av0hToFHblNUEU",
        "p" : "_CG4VcWtTKK2lwUWQG9xxuee_EEm5lmHctseCC3msN3aqiopUfBBSOhuC94oITt_YA-YcwgwHqzqE0Biuww932KNqav5PvHOPnWwlTpITb01VL1cBkmTPdd-UnVj6Q8FqAE_3ayVjDKTeOlDA7MEvl-d8f5bBDp_3ZRwCj8LHLvQUWt82UxXypbZ_SqMqXOZEhjLozocI9gQ91GdH3cCq3Kv_bP4ShsqiBFuQDO8TQz8eYnGV-D-lOlkR2rli65reHbzbAnTKxpj-MR8lKdMku7fdfwnz_4PhFI2PkvI92U_PLVer2k87HDRPIdd6TWosgQ5q36T92mBxZV_xbtE2Q",
        "q" : "-cf3SKUF0j7O-ahfgJfIz31wKO9skOIqM2URWC0sw2NuNOrTcgTb0i8UKj-x1fhXsDEMekM_Ua4U1GCLAbQ6qMeuZ4Nff74LnZeUiznpui06FoftuLVu5w_wU22rTQVR9x7Q2u6eQSRJ9fCZvMFeTvBVTcefh_7FoN6nF8cFQ5K_REYTk3QBu-88Ivv35zjFh3m5gWCaH5wR3W8LvpmW4nc0WeTO8kewKp_CEpasV6WxBWGCQxDPvezJDgZZg3DjaYcT_b4lKOxO89zKrnAe7cPlStbnr05o47Ob0ul6yRGZNsZHpQNRHLKD35hM_XwH8PVqqK4xZpSO8_QbCFmTTQ",
        "qi" : "IvuOX82bdnEE5xJE21MFjBgGHhsNH2O3Pi1ZqV4qEM2HQmoz2hPCh83vgTbl5H6T-5swrZJiintUP0jrARqGNWqzy0gPJ-ORsBjKGH2Xrz2C4xhh7K-mY9t4qonDvUaOaq3vs6Q_eLwAuAFMldtU6dIaAX6PIfZxVF7d6all6jLf_0XNo3_KGqUTL2yO7SIr0B_tWm59Y5WAxZVXd6hlRMLEyTm9uLTEht2lMHKGGgM0NZvbN1hHXknZDQU5lE54z8_Y__Vbsxoc68ZbKPUeeQcBsveRIYiYTwNObpbhxSUeM_44-yIbznqQqGhXxfVrbKdzB8RdUpCx8Iit4IKzSQ"
      },
      "privateKeyPem" : "-----BEGIN RSA PRIVATE KEY-----\nMIIJKgIBAAKCAgEA9gG+DczQSqQLEvPxka4XwfnIwLaOenfhS+JcPHkHyx0zpu9B\njvQYUvMsmDkrxcmu2RwaFQHFA+q4mz7m9PjrLg/PxBvQNgnPao6zqm8PviMYezPb\nTTS2bRKKiroKKr9Au50T2OJVRWmlerHYxhuMrS3IhZmuDaU0bhXazhuse/aXN8Iv\nCDvptGu4seq1lXstp0AnXpbIcZW5b+EUUhWdr8/ZFs7l10mne8OQWl69OHrkRej+\ncPFumghmOXec7/v9QVV72Zrqajcaa0sWBhWhoSvGlY00vODIWty9g5L6EM7KUiCd\nVhlro9JzziKPHxERkqqS3ioDl5ihe87LTcYQDm+K6MJkPyrnaLIlXwgsl46VylUV\nVfEGCCMc+AA7v4B5af/x5RkUuajJuPRWRkW55dcF/60pZj9drj12ZStCLkPxPmwU\nkQkIBcLRJop0olEXdCfjOpqRF1w2cLkXRgCLzh/SMebk8q1wy0OspfB2AKbTHdAp\nFSQ9/dlDoCFl2jZ6a35Nrh3S6Lg2kDCAeV0lhQdswcFd2ejS5eBHUmVpsb/TldlX\n65/eMl00LRRCbnHv3BiHUV5TzepYNJIfkoYp50ju0JesQCTivyVdcEEfhzc5SM+O\niqfv+isKtH1RZgkeGu3sYFaLFVvZwnvFXz7ONfg9Y2281av0hToFHblNUEUCAwEA\nAQKCAgEA01Gb2G7fXb6cZKN4FxPdBJt0f1ZR/ZGMzoqbgLbWovtqqzNKtWmom1iY\nLgquNzCQKZ+iJ/llK4AtI+5cpoJMQz0B1AuwRzsWGQqL+xN8CnBLT0m0UBW/vuH2\ncERvB1lSWdcMfXmulfmyVDsBYuu3Y+u4HEtu3/nRl97eHb5X5ARm0VbU39XXY0xF\nU0+yu70b8leBehc8B5X9vMUzl29KDQQWDyma9dwnKoFLNtW65RFrlUIXjx1VTKt6\nZFMDVIK5ga3UvY/9XVAIObI+MOvT84aPB1hMvRK6CJMlmChg9p8r3HB3tsYPWKIn\nKCM3nhAjcEFl98FPZKGP1bJFoYFJt+2jOFpWup55UConvxOGXN41vhXeA9BqpvCL\nFyt+60tzy8FXAZxdkzWEqNGt1ht9vKOyU8oM+T3JqKOqwvUCJwIuaS97R2dVZiDM\nko1j4xB4w2Diq0txqRfhnn6wk4BILltOqIIChxwqKcpvZrL+MEr2CVIOT4HWTCZ2\ni7gSqGZ5NmYR9M9uieK9HZ1+KHKcfw5OMVLXrX8Yb6MvAeFp/wahIAG8F539DclC\ny6vFVfZ/X9BD4KM1Q0D6SQ0vEjNnvpJus+Hf/nDDFRyHRQ8yF9wqoLWnBpxaF9VW\nFMmZQTn3s3tJ6f54CvZaDoni5Y/qr/4WO8nRnq/ZzSmw7zzvPQECggEBAPwhuFXF\nrUyitpcFFkBvccbnnvxBJuZZh3LbHggt5rDd2qoqKVHwQUjobgveKCE7f2APmHMI\nMB6s6hNAYrsMPd9ijamr+T7xzj51sJU6SE29NVS9XAZJkz3XflJ1Y+kPBagBP92s\nlYwyk3jpQwOzBL5fnfH+WwQ6f92UcAo/Cxy70FFrfNlMV8qW2f0qjKlzmRIYy6M6\nHCPYEPdRnR93Aqtyr/2z+EobKogRbkAzvE0M/HmJxlfg/pTpZEdq5Yuua3h282wJ\n0ysaY/jEfJSnTJLu33X8J8/+D4RSNj5LyPdlPzy1Xq9pPOxw0TyHXek1qLIEOat+\nk/dpgcWVf8W7RNkCggEBAPnH90ilBdI+zvmoX4CXyM99cCjvbJDiKjNlEVgtLMNj\nbjTq03IE29IvFCo/sdX4V7AxDHpDP1GuFNRgiwG0OqjHrmeDX3++C52XlIs56bot\nOhaH7bi1bucP8FNtq00FUfce0NrunkEkSfXwmbzBXk7wVU3Hn4f+xaDepxfHBUOS\nv0RGE5N0AbvvPCL79+c4xYd5uYFgmh+cEd1vC76ZluJ3NFnkzvJHsCqfwhKWrFel\nsQVhgkMQz73syQ4GWYNw42mHE/2+JSjsTvPcyq5wHu3D5UrW569OaOOzm9LpeskR\nmTbGR6UDURyyg9+YTP18B/D1aqiuMWaUjvP0GwhZk00CggEBAIFUhqqwoIlr+X8T\n4+sff1xJGVtJzDtid0EqNoh5ixj0ZCLfR5y5QbO1TiWWSj1puJe8yDVRYOWLSvKf\nF0XdLKu2cPY0ucBY5rNRSUfywn3l7UJPc7Hh8b5KGIkRoDM/OmaIZYs+6OMmWlEu\nTerK3EcO4wTrtSJBI6+0YZhP6FJP4LazDTKln27S3HSpa8fL/Ru0TlinCSI1xdYn\nLhKiyGLLjIz10QmqT7HGRyh1oURgwe1SB8SyK8SUx5R+t8pjqMr9MTYdAA3fFqLX\nnxPdkUDZeRSbSIy/RJRaW2qvEyIb9Ekeu7f8onyiDiIfScPDe4n88twOLLY/j4qb\nehQiUFkCggEBALYdhP+TSk5DexbuG0uf30rhM3C1OFveelRkoSPAND31dfnhKO+d\n+UQjDTnMnPXcDtsot+dAtp7wJMG/7jn81TQP+uoAEBYMU13Akg582BvlM9APpVSh\n/E0+AsRhVp9efKeH8VFe30WxlrdZiE3mUsONWTTPklJOgHtNO1kLw5vEF+5Ihadh\n0o3a3ObI/bO5YdPn/UgGTfk0CpZ/i3mZdDiEH0hXmkdt21UIjDCPaPKynQHGWXpa\nfI0GYoT2PjemjDh5wyqjg2Z1/Q6ycZiDqRlEVh6d1+iqa7FxV/CMSPjm+uXD5aK7\na11YDuxsl93Nm+CknvKDpwMa16uo1DjfTpUCggEAIvuOX82bdnEE5xJE21MFjBgG\nHhsNH2O3Pi1ZqV4qEM2HQmoz2hPCh83vgTbl5H6T+5swrZJiintUP0jrARqGNWqz\ny0gPJ+ORsBjKGH2Xrz2C4xhh7K+mY9t4qonDvUaOaq3vs6Q/eLwAuAFMldtU6dIa\nAX6PIfZxVF7d6all6jLf/0XNo3/KGqUTL2yO7SIr0B/tWm59Y5WAxZVXd6hlRMLE\nyTm9uLTEht2lMHKGGgM0NZvbN1hHXknZDQU5lE54z8/Y//Vbsxoc68ZbKPUeeQcB\nsveRIYiYTwNObpbhxSUeM/44+yIbznqQqGhXxfVrbKdzB8RdUpCx8Iit4IKzSQ==\n-----END RSA PRIVATE KEY-----",
      "privateKeyPkcs8" : "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",
      "type" : "RsaesPkcs1Decrypt",

        {
          "tcId" : 31,
          "comment" : "prepended bytes to ciphertext",
          "msg" : "54657374",
          "ct" : "000004327a40b02bf671557124f963a57b3860e92cff62c439c0425b48b4346fc60c0ebf7a7584f94d34450d20cbd877c8d5dca12f517b486c2cccb8e1f467276ac03aadc94a97fcd224994d81672eb577cf0bbd6aa948d3dc4d7f06456f6650e5620435c078787db0f36124b292349ebe011ce54b3e932fac6525a37c793846a4f08ac3694d649f4a04e24e1f5e50f11a0492a68a509cf30e565ceb9931565b4aa5c3514b2ba87c4c0937ebcd6bb2b8248abb0970d30480059daea4c6ae556f6e91b25ffa5a4f723a9bf98a0bff668a1f0c799d3b0c85b19190dbfb5d894f84fa5d72d261dd2c09013dc0981cd0c46d7a08710801590aac8ff17b237387427ab3c6d6f2a59434b37f123bc7fc0a83d5ca5793540cba582e41b262859d36eabd0aa8203ca05d4c16aaf2a7b2bc7f251497d4c8f8654deae3cfffc5d3599ab4779585bf1673196782075a91ee7c3296a2edc6ee6c30344dd0c0a82274ae17982eeb23eca5c39c7d11a2dd171c70108b0a33164fc175425586f714deb5552e90e561f7882211d3f01c07867256d0cca511e61b0cb51189d8e5124e8cacdab6042bd421447ffa7fe6cad8e7f17dd3e599bae061f85bb5181726d1c0c5bbf2c2a5c1e60f486a81782e58d90ccd5a769f98361765441de142bb0a7f7bd406a537d5be0c2773e847b1df1d49ac1daa963feec84954b72a695b74281159647a62a3c19acdda",
          "result" : "invalid",
          "flags" : []
        },


"""

print("\nTest with prepended bytes to ciphertext\n")

privateKeyPem = "-----BEGIN RSA PRIVATE KEY-----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\n-----END RSA PRIVATE KEY-----"
private_key = PrivateKey.load_pkcs1(privateKeyPem, format='PEM')

ct = bytes.fromhex("000004327a40b02bf671557124f963a57b3860e92cff62c439c0425b48b4346fc60c0ebf7a7584f94d34450d20cbd877c8d5dca12f517b486c2cccb8e1f467276ac03aadc94a97fcd224994d81672eb577cf0bbd6aa948d3dc4d7f06456f6650e5620435c078787db0f36124b292349ebe011ce54b3e932fac6525a37c793846a4f08ac3694d649f4a04e24e1f5e50f11a0492a68a509cf30e565ceb9931565b4aa5c3514b2ba87c4c0937ebcd6bb2b8248abb0970d30480059daea4c6ae556f6e91b25ffa5a4f723a9bf98a0bff668a1f0c799d3b0c85b19190dbfb5d894f84fa5d72d261dd2c09013dc0981cd0c46d7a08710801590aac8ff17b237387427ab3c6d6f2a59434b37f123bc7fc0a83d5ca5793540cba582e41b262859d36eabd0aa8203ca05d4c16aaf2a7b2bc7f251497d4c8f8654deae3cfffc5d3599ab4779585bf1673196782075a91ee7c3296a2edc6ee6c30344dd0c0a82274ae17982eeb23eca5c39c7d11a2dd171c70108b0a33164fc175425586f714deb5552e90e561f7882211d3f01c07867256d0cca511e61b0cb51189d8e5124e8cacdab6042bd421447ffa7fe6cad8e7f17dd3e599bae061f85bb5181726d1c0c5bbf2c2a5c1e60f486a81782e58d90ccd5a769f98361765441de142bb0a7f7bd406a537d5be0c2773e847b1df1d49ac1daa963feec84954b72a695b74281159647a62a3c19acdda")
pt = bytes.fromhex("54657374")

try:
  message = rsa.decrypt(ct, private_key)
except:
  print("[*] Invalid decryption")
else:
  print("[!] No errors in decryption")
  print("message == pt?", message == pt)

print("\nTest with original ciphertext\n")

privateKeyPem = "-----BEGIN RSA PRIVATE KEY-----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\n-----END RSA PRIVATE KEY-----"
private_key = PrivateKey.load_pkcs1(privateKeyPem, format='PEM')

ct = bytes.fromhex("04327a40b02bf671557124f963a57b3860e92cff62c439c0425b48b4346fc60c0ebf7a7584f94d34450d20cbd877c8d5dca12f517b486c2cccb8e1f467276ac03aadc94a97fcd224994d81672eb577cf0bbd6aa948d3dc4d7f06456f6650e5620435c078787db0f36124b292349ebe011ce54b3e932fac6525a37c793846a4f08ac3694d649f4a04e24e1f5e50f11a0492a68a509cf30e565ceb9931565b4aa5c3514b2ba87c4c0937ebcd6bb2b8248abb0970d30480059daea4c6ae556f6e91b25ffa5a4f723a9bf98a0bff668a1f0c799d3b0c85b19190dbfb5d894f84fa5d72d261dd2c09013dc0981cd0c46d7a08710801590aac8ff17b237387427ab3c6d6f2a59434b37f123bc7fc0a83d5ca5793540cba582e41b262859d36eabd0aa8203ca05d4c16aaf2a7b2bc7f251497d4c8f8654deae3cfffc5d3599ab4779585bf1673196782075a91ee7c3296a2edc6ee6c30344dd0c0a82274ae17982eeb23eca5c39c7d11a2dd171c70108b0a33164fc175425586f714deb5552e90e561f7882211d3f01c07867256d0cca511e61b0cb51189d8e5124e8cacdab6042bd421447ffa7fe6cad8e7f17dd3e599bae061f85bb5181726d1c0c5bbf2c2a5c1e60f486a81782e58d90ccd5a769f98361765441de142bb0a7f7bd406a537d5be0c2773e847b1df1d49ac1daa963feec84954b72a695b74281159647a62a3c19acdda")
pt = bytes.fromhex("54657374")

try:
  message = rsa.decrypt(ct, private_key)
except:
  print("[*] Invalid decryption")
else:
  print("[!] No errors in decryption")
  print("message == pt?", message == pt)
