/*******************************************************************************
 * Copyright (c) 2000, 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Syntevo GmbH    - major redesign
 *******************************************************************************/
package com.syntevo.q.swt;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;

public final class QTabFolderItem extends Item {

	// Fields =================================================================

	private final QTabFolder parent;

	private int x;
	private int y;
	private int width;
	private int height;
	private Control control;

	private String toolTipText;
	private String shortenedText;
	private int shortenedTextWidth;

	private Rectangle closeRect = new Rectangle(0, 0, 0, 0);
	private int closeImageState = SWT.BACKGROUND;
	private int state = SWT.NONE;
	private boolean showClose;
	private boolean showing;

	// Setup ==================================================================

	public QTabFolderItem(Control control, QTabFolder parent, int style) {
		super(parent, style | SWT.NO_FOCUS);
		if (control.isDisposed()) {
			SWT.error(SWT.ERROR_WIDGET_DISPOSED);
		}
		if (control.getParent() != parent) {
			SWT.error(SWT.ERROR_UNSPECIFIED);
		}

		this.parent = parent;
		this.control = control;

		showClose = (style & SWT.CLOSE) != 0;
		parent.createItem(this, parent.getItemCount());

		final int index = parent.indexOf(this);
		if (index == parent.getSelectionIndex()) {
			control.setBounds(parent.getClientArea());
			control.setVisible(true);
		}
		else {
			final QTabFolderItem selectedItem = parent.getSelection();
			if (selectedItem == null || control != selectedItem.control) {
				control.setVisible(false);
			}
		}
	}

	// Implemented ============================================================

	@Override
	public void dispose() {
		if (isDisposed()) {
			return;
		}

		parent.destroyItem(this);
		super.dispose();
		control = null;
		toolTipText = null;
		shortenedText = null;
	}

	@Override
	public void setImage(Image image) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void setText(String string) {
		checkWidget();
		if (string == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}
		if (string.equals(getText())) {
			return;
		}

		super.setText(string);
		shortenedText = null;
		shortenedTextWidth = 0;
		if (!parent.updateTabHeight(false)) {
			parent.updateItems();
			parent.redrawTabs();
		}
	}

	// Accessing ==============================================================

	boolean isZeroSize() {
		return width == 0 || height == 0;
	}

	Rectangle getBounds() {
		return new Rectangle(x, y, width, height);
	}

	public Control getControl() {
		checkWidget();
		return control;
	}

	String getToolTipText() {
		checkWidget();
		if (toolTipText == null && shortenedText != null) {
			final String text = getText();
			if (!shortenedText.equals(text)) {
				return text;
			}
		}
		return toolTipText;
	}

	boolean isShowing() {
		checkWidget();
		return showing;
	}

	public void setShowClose(boolean showClose) {
		checkWidget();
		if (this.showClose == showClose) {
			return;
		}
		this.showClose = showClose;
		parent.updateItems();
		parent.redrawTabs();
	}

	public void setToolTipText(String string) {
		checkWidget();
		toolTipText = string;
	}

	public QTabFolder getParent() {
		return parent;
	}

	int getX() {
		return x;
	}

	void setX(int x) {
		this.x = x;
	}

	int getY() {
		return y;
	}

	void setY(int y) {
		this.y = y;
	}

	int getWidth() {
		return width;
	}

	void setWidth(int width) {
		this.width = width;
	}

	int getHeight() {
		return height;
	}

	void setHeight(int height) {
		this.height = height;
	}

	String getShortenedText() {
		return shortenedText;
	}

	void setShortenedText(String shortenedText) {
		this.shortenedText = shortenedText;
	}

	int getShortenedTextWidth() {
		return shortenedTextWidth;
	}

	void setShortenedTextWidth(int shortenedTextWidth) {
		this.shortenedTextWidth = shortenedTextWidth;
	}

	public Rectangle getCloseRect() {
		return closeRect;
	}

	void setCloseRect(Rectangle closeRect) {
		this.closeRect = closeRect;
	}

	int getCloseImageState() {
		return closeImageState;
	}

	void setCloseImageState(int closeImageState) {
		this.closeImageState = closeImageState;
	}

	int getState() {
		return state;
	}

	void setState(int state) {
		this.state = state;
	}

	boolean isShowClose() {
		return showClose;
	}

	void setShowing(boolean showing) {
		this.showing = showing;
	}
}
