/* 
   - The contents of this file are subject to the Mozilla Public
   - License Version 1.1 (the "License"); you may not use this file
   - except in compliance with the License. You may obtain a copy of
   - the License at http://www.mozilla.org/MPL/
   -
   - Software distributed under the License is distributed on an "AS
   - IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
   - implied. See the License for the specific language governing
   - rights and limitations under the License.
   -
   - The Original Code is this file as it was released on
   - August 30, 2003 (rewrite from old source files).
   -
   - The Initial Developer of the Original Code is HJ van Rantwijk.
   - Portions created by HJ van Rantwijk are Copyright (C) 2003-2007
   - HJ van Rantwijk.  All Rights Reserved.
   - 
   - Contributor(s):
   - 	Michael Vincent van Rantwijk <mv_van_rantwijk@yahoo.com>
   -
*/

var gDebug = false; // change this to true to enable debug info on the console

var mzRDFStore = null; // global pointer to multizilla.rdf

if (!RDF) {
  var RDF = Components.classes[ "@mozilla.org/rdf/rdf-service;1" ]
                      .getService(Components.interfaces.nsIRDFService);
}
var mzRDFCUtils = Components.classes["@mozilla.org/rdf/container-utils;1"]
                            .getService(Components.interfaces.nsIRDFContainerUtils);

var mzTabIndex = 0; // global index to number of tabs that have been restored
var gSessionName = "";

/**
 * File mode bits
 * #define PR_IRWXU 00700  read, write, execute/search by owner
 * #define PR_IRUSR 00400  read permission, owner
 * #define PR_IWUSR 00200  write permission, owner
 * #define PR_IXUSR 00100  execute/search permission, owner
 * #define PR_IRWXG 00070  read, write, execute/search by group
 * #define PR_IRGRP 00040  read permission, group
 * #define PR_IWGRP 00020  write permission, group
 * #define PR_IXGRP 00010  execute/search permission, group
 * #define PR_IRWXO 00007  read, write, execute/search by others
 * #define PR_IROTH 00004  read permission, others
 * #define PR_IWOTH 00002  write permission, others
 * #define PR_IXOTH 00001  execute/search permission, others
 * 
 * File open bits
 * #define PR_RDONLY       0x01
 * #define PR_WRONLY       0x02
 * #define PR_RDWR         0x04
 * #define PR_CREATE_FILE  0x08
 * #define PR_APPEND       0x10
 * #define PR_TRUNCATE     0x20
 * #define PR_SYNC         0x40
 * #define PR_EXCL         0x80
*/

function getNavigatorWindow(aOpenFlag)
{
  var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1']
                                .getService(Components.interfaces.nsIWindowMediator);
  var navigatorWindow = windowManager.getMostRecentWindow("navigator:browser");

  if (!navigatorWindow) {
    if (!aOpenFlag)
      return null;
    else {
      window.open();
      navigatorWindow = windowManager.getMostRecentWindow("navigator:browser");
    }
  }
  return navigatorWindow;
}

function mzGetDirByKey(aKey)
{
  var fl = Components.classes["@mozilla.org/file/directory_service;1"]
                     .getService(Components.interfaces.nsIProperties);
  return fl.get(aKey, Components.interfaces.nsILocalFile);
}

function mzGetProfile()
{
  return mzGetDirByKey("ProfD");
}

function mzGetProfileURI()
{
  var profile = mzGetProfile();
  var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                            .getService(Components.interfaces.nsIIOService);
  return ioService.newFileURI(profile);
}

function mzInitializeRDFStorageFile() 
{
  mzRDFStore = mzGetNewOrCurrentDSByFilename("multizilla.rdf", false, null, false);  // "quickprefs.rdf" for stand-alone versions of QPM
}

function mzGetNewOrCurrentDSByFilename(aFileName, createWhenMissing, RDFData, hasPath)
{
  var fileName = hasPath ? aFileName : (mzGetProfileURI().spec + aFileName);
  var ioService = Components.classes['@mozilla.org/network/io-service;1']
                            .getService(Components.interfaces.nsIIOService);
  var fileHandler = ioService.getProtocolHandler("file")
                             .QueryInterface(Components.interfaces.nsIFileProtocolHandler);
  var file = fileHandler.getFileFromURLSpec(fileName)
                        .QueryInterface(Components.interfaces.nsILocalFile);
  fileName = fileHandler.getURLSpecFromFile(file);

  if (!file.exists()) {
    if (createWhenMissing) {
      if (!RDFData) {
        RDFData = "<?xml version=\"1.0\"?>\n" +
                  "<RDF:RDF xmlns:MZ=\"http://multizilla.mozdev.org/rdf#\"\n" +
                  "         xmlns:RDF=\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\">\n" +
                  "  <!-- Empty -->\n" +
                  "</RDF:RDF>\n";
      }
      mzWriteToFile(file, RDFData);
    }
    else {
      var dirKey, targetPath = "";
      var appInfo = Components.classes["@mozilla.org/xre/app-info;1"]
                              .getService(Components.interfaces.nsIXULAppInfo);

      if ('nsIXULRuntime' in appInfo) {
        dirKey = "ProfD";
        targetPath = "extensions/multizilla@multizilla.mozdev.org/defaults/";
      }
      else
        dirKey = "DefRt";

      var file = mzGetDirByKey(dirKey);
      var url = ioService.newFileURI(file).spec;
      targetPath = targetPath + "profile/";
      fileName = url + targetPath + aFileName;
      var sourceFile = fileHandler.getFileFromURLSpec(fileName)
                                  .QueryInterface(Components.interfaces.nsILocalFile);
      fileName = fileHandler.getURLSpecFromFile(sourceFile);
      var profile = mzGetProfile();
      sourceFile.copyTo(profile, "");
      // Retry with copied file
      fileName = mzGetProfileURI().spec + aFileName;
      file = fileHandler.getFileFromURLSpec(fileName)
                        .QueryInterface(Components.interfaces.nsILocalFile);
      fileName = fileHandler.getURLSpecFromFile(file);
    }
  }
  return RDF.GetDataSourceBlocking(fileName) || null;
}
////////////////////////////////////////////////////////////////////////////////
// This function returns a RDFContainer based on the given datasource 
// 'aDatasource', the given type 'aType' and the given resource 'aResource'. It 
// will creates a new container when you call this function with 'true' for 
// 'aInit'. It returns null on failure.
////////////////////////////////////////////////////////////////////////////////
function mzConstructRDFContainer(aDataSource, aType, aResource, aSkipInitFlag)
{
  var container = Components.classes["@mozilla.org/rdf/container;1"]
                            .createInstance(Components.interfaces.nsIRDFContainer);

  if (aDataSource && aType && aResource) {
    try {
      var resource = (typeof(aResource) == "string") ? RDF.GetResource(aResource)
                                                     : aResource;

      switch(aType) {
        default:
        case "seq": mzRDFCUtils.MakeSeq(aDataSource, resource);
          break;
        case "bag": mzRDFCUtils.MakeBag(aDataSource, resource);
          break;
        case "alt": mzRDFCUtils.MakeAlt(aDataSource, resource);
          break;
      }
      if (!aSkipInitFlag)
        container.Init(aDataSource, resource);
    } catch (ex) {
      mzDebug(ex);
      return null;
    }
  }
  return container;
}

function mzGetNextSessionNumber(aSessionNumber, aWindowID)
{
  while (aSessionNumber < 15) {
    var resource = "urn:session-" + aSessionNumber + ":" + aWindowID + ":tabs:root";

    if (!mzCheckResource(mzRDFStore, resource))
      return aSessionNumber;
    aSessionNumber++;
  }
  return 0; // Maximum number of sessions reached!
}

function mzGetNextWindowNumber(aSessionID, aWindowNumber)
{
  while (aWindowNumber < 15) {
    var resource = "urn:" + aSessionID + ":window-" + aWindowNumber + ":tabs:root";

    if (!mzCheckResource(mzRDFStore, resource))
      return aWindowNumber;
    aWindowNumber++;
  }
  return 0; // Maximum number of sessions reached!
}

function mzGetNextSessionID(aSessionNumber, aWindowID)
{
  while (aSessionNumber < 15) {
    var resource = "urn:session-" + aSessionNumber + ":" + aWindowID + ":tabs:root";

    if (!mzCheckResource(mzRDFStore, resource))
      return "session-" + aSessionNumber;
    aSessionNumber++;
  }
  return "session-0"; // Maximum number of sessions reached!
}

function mzGetNumberOfWindowsByType(aType)
{
  var defaultType = "navigator:browser";

  if (!aType) // default check for browser window
    aType = defaultType;

  var windowCounter = 0;
  var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1']
                                .getService(Components.interfaces.nsIWindowMediator);
  var enumerator = windowManager.getEnumerator(aType);

  while (enumerator.hasMoreElements()) {
    var aWindow = enumerator.getNext();
    windowCounter++;
  }
  return windowCounter;
}

function mzGetWindowResourceID(aWindow)
{
  // dump("\nmzGetWindowResourceID: " + aWindow);
  var windowID = aWindow.windowID;
  // var windowID = aWindow.document.documentElement.getAttribute("windowID");
  // dump("\nwindowID: " + windowID);

  if (windowID) {
   return windowID;
  }
  else {
    var windowIndex = 0;
    var checkedBrowserWindows = 0;
    var browserWindows = mzGetNumberOfWindowsByType();
    var windowManagerDS = Components.classes['@mozilla.org/rdf/datasource;1?name=window-mediator']
                                    .getService(nsIWindowDataSource);
    var quicklaunchEnabled = 0;

    while (checkedBrowserWindows <= browserWindows) {
      var resourceWindowID = "window-" + windowIndex;
      var resourceWindow = windowManagerDS.getWindowForResource(resourceWindowID);

      if (resourceWindow) {
        if (resourceWindow == aWindow) {
          // aWindow.document.documentElement.setAttribute("windowID", resourceWindowID);
          aWindow.windowID = resourceWindowID;
          return "window-" + resourceWindowID;
        }
        else if ('getBrowser' in resourceWindow)
          checkedBrowserWindows++;
      }
      windowIndex++;
    }
  }
  return null;
}
////////////////////////////////////////////////////////////////////////////////
// This function returns an array with datasource ID's of all open browser 
// (navigator:browser) windows, or null when it fails to locate any of them, 
// but that should never happen (just to prevent a JS strict error).
////////////////////////////////////////////////////////////////////////////////
function mzGetWindowResourceIDList()
{
  var windowArray = new Array();
  var windowIndex = 0;
  var checkedWindow = 0;
  var numberOfWindows = mzGetNumberOfWindowsByType();
  var windowManagerDS = Components.classes['@mozilla.org/rdf/datasource;1?name=window-mediator']
                                  .getService(Components.interfaces.nsIWindowDataSource);

  while (checkedWindow <= numberOfWindows) {
    var resourceWindowID = "window-" + windowIndex;
    var resourceWindow = windowManagerDS.getWindowForResource(resourceWindowID);

    if (resourceWindow) {
      if ('getBrowser' in resourceWindow) {
        windowArray.push(resourceWindowID);
        checkedWindow++;

        if (checkedWindow == numberOfWindows)
          break;
      }
    }
    windowIndex++;
  }
  if (windowArray.length)
    return windowArray;
  return null;
}

function mzGetWindowXULWindowIndex(aWindow)
{
  dump("\nmzGetWindowXULWIndowIndex: " + aWindow);
  var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1']
                                .getService(Components.interfaces.nsIWindowMediator);
  var enumerator = windowManager.getXULWindowEnumerator(null);
  var wIndex = 0;

  while (enumerator.hasMoreElements()) {
    wIndex++;
    // dump("\nwIndex: " + wIndex);
    var targetWindow = enumerator.getNext();
    // dump("\ntargetWindow: " + targetWindow);
    if (targetWindow == aWindow) {
      dump("\nreturn wIndex = " + wIndex);
      return wIndex
    }
  }
  dump("\nreturn wIndex = -1");
  return -1;
}

function mzGetIndexNumberForWindow(aWindow)
{
  dump("\nmzGetIndexNumberForWindow: " + aWindow);
  var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1']
                                .getService(Components.interfaces.nsIWindowMediator);

  var enumerator = windowManager.getEnumerator(null);
  var wIndex = 0;

  while (enumerator.hasMoreElements()) {
    wIndex++;
    // dump("\nwIndex: " + wIndex);
    var targetWindow = enumerator.getNext();
    // dump("\ntargetWindow: " + targetWindow);
    if (targetWindow == aWindow) {
      dump("\nreturn wIndex = " + wIndex);
      return wIndex
    }
  }
  dump("\nreturn wIndex = -1");
  return -1;
}

var gHighestWindowID = "window-0";

function mzGetWindowNumberFor(aWindow, aClosedWindowFlag)
{
  dump("\n\nmzGetIndexNumberForWindow...");
  var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1']
                                .getService(Components.interfaces.nsIWindowMediator);
  var enumerator = windowManager.getEnumerator(null);
  var wIndex = (aClosedWindowFlag) ? -1 : 0;
  var targetWindow, numberOfWindows = 0;
  var closedWindows = 0;

  while (enumerator.hasMoreElements()) {
    wIndex++;
    if (wIndex > 15)
      return -1;
    dump("\nwIndex: " + wIndex);
    if (aClosedWindowFlag) {
       if (gWindowManagerDS.getWindowForResource("window-" + wIndex) == null) {
         // targetWindow = enumerator.getNext();
         closedWindows++;
         dump("\ncontinue");
         continue;
       }
       else {
         // dump("\nelse");
         targetWindow = enumerator.getNext();
         numberOfWindows++;
         // dump("\nwIndex         : " + wIndex);
         // dump("\nnumberOfWindows: " + numberOfWindows);

         if (targetWindow == aWindow)
           return (numberOfWindows - closedWindows);

       }
    }
    else {
      targetWindow = enumerator.getNext();

      if (targetWindow == aWindow)
        return wIndex;
    }
  }
  return -1;
}

function mzGetRealWindowResourceIDList()
{
  var windowArray = new Array();
  var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1']
                                .getService(Components.interfaces.nsIWindowMediator);
  var enumerator = windowManager.getEnumerator("navigator:browser");
  
  while (enumerator.hasMoreElements()) {
    var browserWindow = enumerator.getNext();
    var windowID = mzGetWindowResourceID(browserWindow);
    windowArray.push(windowID);
  }
  if (windowArray.length)
    return windowArray;
  return null;
}
////////////////////////////////////////////////////////////////////////////////
// This function returns a number (window-X) for the given window. This number 
// is based on the position of the datasource ID in the array of open windows. 
// It should never returns -1 but that's only to prevent a JS strict warning.
////////////////////////////////////////////////////////////////////////////////
function mzGetWindowIndexNumber(aWindow)
{
  var thisWindowID = mzGetWindowResourceID(aWindow);
  var allWindowIDs = mzGetWindowResourceIDList();

  for (index in allWindowIDs) {
    if (allWindowIDs[index] == thisWindowID)
      return index;
  }
  return -1;
}

function mzGetRealWindowIndexNumber(aWindow)
{
  // dump("\nmzGetRealWindowIndexNumber: " + aWindow);
  var thisWindowID = mzGetWindowResourceID(aWindow);
  var allWindowIDs = mzGetRealWindowResourceIDList();

  for (index in allWindowIDs) {
    if (allWindowIDs[index] == thisWindowID) {
      // dump("\nthisWindowID: " + thisWindowID);
      return thisWindowID.replace(/window-/, "");
    }
  }
  return -1;
}
////////////////////////////////////////////////////////////////////////////////
// This function returns a datasource ID (window-x) for the given window. This 
// ID is based on the position in the array of all open browser windows. It 
// returns 'window-x', or null (just to prevent a JS strict warning).
////////////////////////////////////////////////////////////////////////////////
function mzGetRelativeWindowResourceID(aWindow)
{
  var thisWindowID = mzGetWindowResourceID(aWindow);
  var allWindowIDs = mzGetRealWindowResourceIDList(); // mzGetWindowResourceIDList();

  for (index in allWindowIDs) {
    if (allWindowIDs[index] == thisWindowID)
      return ('window-' + index);
  }
  return null;
}

function mzGetSessionContainer(aDataSource)
{
  return mzConstructRDFContainer(aDataSource, "seq", "urn:sessions:root", false);
}

function mzAddSessionToSessionContainer(aDataSource, aSessionContainerResource)
{
  if (!aDataSource)
    return;
  if (typeof(aSessionContainerResource) == "string")
    aSessionContainerResource = RDF.GetResource(aSessionContainerResource);

  if (!mzCheckResourceForValue(aDataSource, "urn:sessions:root", aSessionContainerResource)) {
    var sessionContainer = mzGetSessionContainer(aDataSource);
    sessionContainer.AppendElement(aSessionContainerResource);
    mzFlushDataSource(aDataSource);
  }
}
////////////////////////////////////////////////////////////////////////////////
// This function returns an array with all matching windows for the given 
// session. IE "session-x" where 'x' is the session number to lookup for.
////////////////////////////////////////////////////////////////////////////////
function mzGetWindowResourcesForSession(aDataSource, aSession)
{
  var matchingResources = new Array();
  var sessionContainer = mzRDFCUtils.MakeSeq(aDataSource, RDF.GetResource("urn:sessions:root")).GetElements();
  
  while (sessionContainer.hasMoreElements()) {
    var resource = sessionContainer.getNext().QueryInterface(Components.interfaces.nsIRDFResource);

    if (resource.Value.match(aSession))
      matchingResources.push(resource.Value);
  }
  return matchingResources.sort();
}

function openWindowAndReturnWindowID()
{
  window.open();
  var allWindowIDs = mzGetWindowResourceIDList();
  return allWindowIDs[allWindowIDs.length-1];
}

function mzGetNewWindowObject(aWinObj)
{
  // This function fails when it is called from a modal window!
  var winArgs = "";

  if (aWinObj.width && aWinObj.height)
    winArgs = "width=" + aWinObj.width + "height=" + aWinObj.height + "screenX=" + aWinObj.screenX + "screenY=" + aWinObj.screenY;
  window.open(winArgs);

  if (aWinObj == undefined)
    aWinObj = new Object();

  aWinObj.windowIDs = mzGetWindowResourceIDList();
  aWinObj.windowID = aWinObj.windowIDs[aWinObj.windowIDs.length-1];
  var windowManagerDS = Components.classes['@mozilla.org/rdf/datasource;1?name=window-mediator']
                                  .getService(Components.interfaces.nsIWindowDataSource);
  aWinObj.browserWindow = windowManagerDS.getWindowForResource(aWinObj.windowID) || null;
  aWinObj.tabbrowser = aWinObj.browserWindow.getBrowser();
  return aWinObj;
}

function openWindowAndReturnBrowser()
{
  var obj = new Object();
  mzGetNewWindowObject(obj);

  if (obj.browserWindow) {
    var tabBrowser = obj.tabbrowser;
    obj = null;
    return tabBrowser;
  }
  return null;
}

function setSessionIDs()
{
  var index = 0;
  var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1']
                                .getService(Components.interfaces.nsIWindowMediator);
  var enumerator = windowManager.getEnumerator("navigator:browser");
  
  while (enumerator.hasMoreElements()) {
    var browserWindow = enumerator.getNext();
    browserWindow.windowID = "window-" + index;
    index++;
  }
}

var gSessionObject = null;

function mzSaveTabSession(aForcedSaveFlag, aCloseWSFlag, aSessionObject, aQuitApplicationFlag)
{
  var tabbrowser = ('getBrowser' in window) ? getBrowser() : aSessionObject.tabbrowser;

  if (tabbrowser.mChromeHidden) // Skip chromeless windows
    return aCloseWSFlag;
  /***
    * All callers but the Tab Manager calls this function with 'aSessionObject' set to null
    */
  if (gSessionObject == null) {
    var saveSessionPref = gMultiZilla.prefs.readInteger("multizilla.browsersession.save.behavior", 3);
    dump("\nsaveSessionPref: " + saveSessionPref);
    var shouldSave = (saveSessionPref > 0 || aForcedSaveFlag);
    var numberOfTabs = tabbrowser.mTabContainer.childNodes.length;
    // Skip if we're not enabled, all tabs are blank
    if (!shouldSave || gMultiZilla.tabs.areAllTabsBlank(tabbrowser) ||
        (numberOfTabs == 1 &&
        gMultiZilla.prefs.readBoolean("multizilla.browsersession.skip.save-single-tab", false)))
      return aCloseWSFlag;

    gSessionObject = new Object();
    gSessionObject.window = window;
    gSessionObject.windowID = ('windowID' in window) ? window.windowID
                                                     : mzGetRelativeWindowResourceID(window);
    gSessionObject.tabbrowser = tabbrowser;
    gSessionObject.resource = "urn:session-0:" + gSessionObject.windowID + ":tabs:root";
    gSessionObject.previouslyStored = mzCheckResource(mzRDFStore, gSessionObject.resource);
    gSessionObject.saveAsNewSession = (saveSessionPref == 2);
    gSessionObject.askConfirmation = (aCloseWSFlag && saveSessionPref == 3);
    gSessionObject.saveHistory = true;
    gSessionObject.isCancelled = false;
    gSessionObject.quitApplication = (aQuitApplicationFlag == undefined) ? false : aQuitApplicationFlag;
  }
  gSessionObject.datasource = mzRDFStore;
  gSessionObject.sessionID = 0;
  gSessionObject.closeWSFlag = aCloseWSFlag;
  gSessionObject.selectedTabStyle = gMultiZilla.prefs.readString("multizilla.session-manager.selectedtab-styling", "bold");

  if (gSessionObject.askConfirmation) {
    var parentWindow = (gSessionObject.parentWindow == undefined) ? window: gSessionObject.parentWindow;
    parentWindow.openDialog("chrome://multiviews/content/session-manager/miscellaneous/confirmSessionSaving.xul", 
                            "", "chrome, dialog, dependent, modal, centerscreen", gMultiZilla.utility.getPlatformExtras());

    if (gSessionObject.isCancelled) {
      gSessionObject = null;
      return false;
    }
    else { // We only need this for SeaMonkey, not for Mozilla!
      var profileSwitch = (mzGetNumberOfWindowsByType("mozilla:profileSelection") == 1);

      if (mzGetNumberOfWindowsByType() == 1 && profileSwitch == false) {
        function delayedExit() {
          window.tryToClose = function() { return true; };

          if (gSessionObject.quitApplication)
            goQuitApplication();
          else
            window.close();
        }
        setTimeout(delayedExit, 0);
        return 2; // false;
      }
    }
    return aCloseWSFlag;
  }
  var retValue = mzWriteTabSession(gSessionObject);
  /***
    * 'aForcedSaveFlag' is only true when we're called from tabbrowser.xml
    * and we must clear 'gSessionObject' or it won't save sessions on exit 
    * after you've used menu item 'Save open tab(s) as new Tab Session' once.
    */
  if (aForcedSaveFlag)
    gSessionObject = null;
  return retValue;
}
/***
  * Function mzWriteTabSession will be called from function 'mzSaveTabSession'  
  * in this file, and from function 'doButton' in confirmSessionSaving.xul
  *
  * Note: we call function 'mzWriteTabSession' with 'aSessionObject', 
  * instead of using the global 'gSessionObject' object, because this 
  * object will be lost when you close the Tab Manager parent window!
  */
function mzWriteTabSession(aSessionObject)
{
  /***
    * The 'isCancelled' property will only be set to Boolean(true) if
    * you select the 'cancel' button from the session saver window!
    */
  if (!aSessionObject.isCancelled) {
    var dummy;
    // We're not cancelled, but need to determine what to do next
    if (aSessionObject.mergeSessions) {
      dummy = addTabDataSeqToDS(aSessionObject.datasource, aSessionObject.resource, aSessionObject.startindex, 
                                aSessionObject.tabbrowser, true, aSessionObject.selectedTabStyle, aSessionObject.saveHistory);
    }
    else { // overwrite current (matching session) or add new session
      if (aSessionObject.saveAsNewSession) {
        if (!mzCheckResourceForValue(aSessionObject.datasource, "urn:sessions:root", aSessionObject.resource)) {
          dump("\nNo session located for: " + aSessionObject.resource);
        }
        var sessionID = mzGetNextSessionID(aSessionObject.sessionID, aSessionObject.windowID);
        aSessionObject.resource = "urn:" + sessionID + ":" + aSessionObject.windowID + ":tabs:root";
      }
      // Do the actual writing to multizilla.rdf
      dummy = mzStoreGroupData(aSessionObject.datasource, aSessionObject.resource, aSessionObject.tabbrowser, 
                               aSessionObject.closeWSFlag, true, aSessionObject.selectedTabStyle, true, aSessionObject.window);
    }
    /***
     * Function 'mzSaveTabSession' always returns false when the users confirmation  
     * is being asked, but we might need to close the browser window, like when  
     * it is being called from function 'confirmCloseWS' in multizilla.js
     */
    if (aSessionObject.askConfirmation) {
      /***
        * The users wants to quit this browser session, so all windows should be closed and 
        * we have to do that ourself, because we returned false in function 'mzSaveTabSession'
        */		
      // mzCloseApplicationWindows(aSessionObject);
    }
  }
  /***
    * If the 'closeWSFlag' property is set to true, and we're being called from function 
    * 'mzSaveTabSession' in multiviewsInit.js, then this return will close our window!
    * 
    * If the 'closeWSFlag' property is set to true, and we're being called from function
    * 'doButton' in confirmSessionSaving.xul, then this return won't do any thing!
    */
  return aSessionObject.closeWSFlag;
}

function mzCloseApplicationWindows(aSessionObject)
{
  dump("\nmzCloseApplicationWindows");
  if (aSessionObject.quitApplication) {
    var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1']
                                  .getService(Components.interfaces.nsIWindowMediator);
    var enumerator = windowManager.getEnumerator(null);
    var appStartup = Components.classes['@mozilla.org/toolkit/app-startup;1']
                               .getService(Components.interfaces.nsIAppStartup);
    var browserWindows = new Array();

    while (enumerator.hasMoreElements()) {
      var domWindow = enumerator.getNext();
      var windowType = domWindow.document.documentElement.getAttribute("windowtype");
      /***
        * browser windows have a tryToClose method since mozilla 1.5 (build 20032406 and up)
        * and attribute 'wintype' is set to 'navigator:window' so we simply check these.
        */
      if (windowType == "navigator:browser" && ("tryToClose" in domWindow)) {
        if (domWindow != aSessionObject.window)
          browserWindows.push(domWindow); // keep this browser window.
      }
      else {
        // Close all non-browser window like: ChatZilla, Addressbook, MailNews etc.
        domWindow.close();
      }
    }
    // Are there more browser window to be closed?
    if (browserWindows.length >= 1)
      browserWindows[0].mzQuitApplication(aSessionObject.window); // Yes, let's mzQuitApplication handle it.
  }
  else {
    aSessionObject.window.close(); // No, so let's close this browser window (which should quit this session).
  }
}

function mzQuitApplication(aWindow)
{
  aWindow.close(); // close (calling) browser window.
  // There are more browser windows to be closed, so let Mozilla handle it.
  goQuitApplication();
}

function mzStoreGroupData(aDataSource, aResource, aTabbrowser, aCloseWSFlag, skipAboutBlank, aSelectedTabStyle, aSaveHistory, aWindow)
{
  if (skipAboutBlank == undefined)
    skipAboutBlank = true;

  mzRemoveAllResourcesFor(aDataSource, aResource);
  var sessionName = addTabDataSeqToDS(aDataSource, aResource, 0, aTabbrowser, skipAboutBlank, aSelectedTabStyle, aSaveHistory);
  // Add resourceID for this tab session only when it is missing!
  if (!mzCheckResourceForValue(aDataSource, "urn:sessions:root", aResource)) {
    var sessionContainer = mzGetSessionContainer(aDataSource);
    sessionContainer.AppendElement(RDF.GetResource(aResource));
    mzFlushDataSource(aDataSource);
  }
  var windowManagerDS = Components.classes['@mozilla.org/rdf/datasource;1?name=window-mediator'].getService(nsIWindowDataSource);
  var windowID = aResource.replace(/urn:/, '').replace(/session-\w*:/, '').replace(/:tabs:root/, '');
  var browserWindow = (aWindow) ? aWindow : windowManagerDS.getWindowForResource(windowID);
  var documentElementID = document.documentElement.getAttribute("id");

  if (documentElementID == "tabManager-window" && aSelectedTabStyle && (!aResource.match(/session-/))) {
    /***
      * transform 'urn:window-N:tabs:root' into 'window-N' (tree refresh)
      * transform 'urn:session-N:window-N:tabs:root' into 'window-N' (Session Button)
      */
    browserWindow = windowManagerDS.getWindowForResource(windowID);
    var windowIndex = mzGetIndexNumberForWindow(browserWindow);
    sessionName = "[ " + windowIndex + " ] " + sessionName;
  }
  else {
    sessionName = aResource;
    var windowAttributes = ["width", "height", "screenX", "screenY"];
    windowID = mzGetWindowResourceID(browserWindow)
    // dump("\nwindowID: " + windowID + "\nbrowserWindow: " + browserWindow);

    for (i in windowAttributes) {
      var value = browserWindow.document.documentElement.getAttribute(windowAttributes[i]);
      windowAttributes[i] = (value >= 0) ? String(value) : "0";
    }
    var windowState = (browserWindow.fullScreen) ? 4 : browserWindow.windowState; // http://lxr.mozilla.org/seamonkey/source/dom/public/idl/base/nsIDOMChromeWindow.idl
    windowAttributes.push(windowState);
    mzAddOrChangeProperty(aDataSource, aResource, "sizePosState", String(windowAttributes));
    var hiddenToolbars = "";
    var XULNS = "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul";
    var tagNames = ["toolbar", "tabmodetoolbar", "statusbar"];

    for (var e = 0; e < tagNames.length; e++) {
      var elements = browserWindow.document.getElementsByTagNameNS(XULNS, tagNames[e]);

      for (var i = 0; i < elements.length; i++) {
        var id = elements[i].getAttribute("id");

        if (elements[i].getAttribute("hidden") == "true")
          hiddenToolbars += (hiddenToolbars == "") ? id : "," + id;
      }
    }
    if (browserWindow.document.getElementById("component-bar").getAttribute("hidden") == "true")
      hiddenToolbars += (hiddenToolbars == "") ? "component-bar" : ",component-bar";
    if (hiddenToolbars)
      mzAddOrChangeProperty(aDataSource, aResource, "hiddenToolbars", hiddenToolbars);

    var sidebarBox = browserWindow.document.getElementById("sidebar-box");

    if (sidebarBox) {
      if (!browserWindow.sidebar_is_hidden() && !browserWindow.sidebar_is_collapsed()) {
        var width = sidebarBox.getAttribute("width");
        var currentPanel = browserWindow.document.getElementById('sidebar-panels').getAttribute("last-selected-panel");
        mzAddOrChangeProperty(aDataSource, aResource, "sidebar", String([width, currentPanel]));
      }
    }
    mzSetModifiedDate(aDataSource, aResource, "savedOn");
  }
  mzWriteSessionName(aDataSource, aResource, sessionName);
  mzFlushDataSource(aDataSource);

  if (!aCloseWSFlag)
    renameTabSession(aDataSource, aResource);
}

function mzGetWindowSettings(aDataSource, aResource)
{
  var windowSettings = new Object();
  var mzRDF = "http://multizilla.mozdev.org/rdf#";
  var propertyArray = ["sizePosState", "hiddenToolbars", "sidebar"];

  for (var i = 0; i < propertyArray.length; i++) {
    var resource = mzRDF + propertyArray[i];
    var property = RDF.GetResource(resource);
    var values;
    var data = aDataSource.GetTarget(RDF.GetResource(aResource), property, true);

    if (data instanceof Components.interfaces.nsIRDFNode) {
      data = data.QueryInterface(Components.interfaces.nsIRDFLiteral)

      if (data) {
        var value = data.Value;

        if (value) {
          if (i == 0) {
            values = value.split(',');
            windowSettings.width = Number(values[0]);
            windowSettings.height = Number(values[1]);
            windowSettings.screenX = Number(values[2]);
            windowSettings.screenY = Number(values[3]);
            windowSettings.windowState = Number(values[4]);
          }
          else if (i == 1)
            windowSettings.hiddenToolbars = value;
          else if (i == 2) {
            values = value.split(',');
            windowSettings.sidebarWidth = Number(values[0]);
            windowSettings.sidebarPanel = values[1];
          }
        }
      }
    }
  }
  /* dump("\nwidth         : " + windowSettings.width +
       "\nheight        : " + windowSettings.height +
       "\nscreenX       : " + windowSettings.screenX +
       "\nscreenY       : " + windowSettings.screenY +
       "\nwindowState   : " + windowSettings.windowState +
       "\nhiddenToolbars: " + windowSettings.hiddenToolbars +
       "\nsidebarWidth  : " + windowSettings.sidebarWidth +
       "\nsidebarPanel  : " + windowSettings.sidebarPanel); */
  return(windowSettings);
}
/***
  * This function restores the window size, position and (full screen) state of the target
  * window (written by HJ while he was waiting for a breathtaking sandstorm to pass by).
  * 
  * TODO: The GoogleBox should also be checked!
  */
function mzRestoreWindowSettings(aDataCollection)
{
  // The Tab Session Manager and selection window should also be able to restore the window settings!
  if (gMultiZilla.prefs.readInteger("multizilla.browsersession.window.restore-behavior", 0) == 0)
    return;

  var toolbar, cmdElement;
  var hiddenToolbars = aDataCollection.hiddenToolbars;
  var toolbarTargets = (hiddenToolbars == undefined) ? "" : hiddenToolbars.split(',');
  var toolbars = ["nav-bar", "PersonalToolbar", "MultiZillaToolbar", /* "internalToolBox", */ "status-bar", "component-bar"];
  var tbarCmds = ["navbar", "personaltoolbar", "MultiZillaToolbar", /* "IntegratedToolBox", */ "taskbar", "componentbar"];

  if (aDataCollection.width > 100)
    window.outerWidth = aDataCollection.width;
  if (aDataCollection.height > 100)
    window.outerHeight = aDataCollection.height;
  window.moveTo(aDataCollection.screenX, aDataCollection.screenY);

  switch (aDataCollection.windowState) {
    case 1: // aWindow.fullScreen = false;
            window.maximize();
            break;
    case 2: // aWindow.fullScreen = false;
            window.minimize();
            break; 
    case 3: // aWindow.fullScreen = false;
            window.restore();
            break;
    case 4: window.fullScreen = true;
  }

  for (t in toolbarTargets) {
    toolbar = document.getElementById(toolbarTargets[t]);

    if (toolbar) {
      if (toolbar.getAttribute("hidden") != "true")
        toolbar.setAttribute("hidden", "true");

      for (x in toolbars) {
        if (toolbars[x] == toolbarTargets[t]) {
          cmdElement = document.getElementById("cmd_view" + tbarCmds[x]);

          if (cmdElement && cmdElement.getAttribute("checked") != "false")
            cmdElement.setAttribute("checked", "false");

          toolbars.splice(x, 1);
          tbarCmds.splice(x, 1);
        }
      }
    }
    // Default toolbars should only be hidden when specified for this window
    for (t in toolbars) {
      toolbar = document.getElementById(toolbars[t]);

      if (toolbar && toolbar.getAttribute("hidden") == "true") {
        toolbar.setAttribute("hidden", "false");
        cmdElement = document.getElementById("cmd_view" + tbarCmds[t]);

        if (cmdElement && cmdElement.getAttribute("checked") != "true")
          cmdElement.setAttribute("checked", "true");
      }
    }
  }
  if (aDataCollection.sidebarPanel) {
    var sidebarPanels = document.getElementById("sidebar-panels");
    sidebarPanels.setAttribute("last-selected-panel", aDataCollection.sidebarPanel);

    if (sidebar_is_hidden())
      SidebarShowHide();
    if (sidebar_is_collapsed())
      SidebarExpandCollapse();

    var sidebarBox = document.getElementById("sidebar-box");
    sidebarBox.setAttribute("width", aDataCollection.sidebarWidth);

    /* if (currentPanel)
      SidebarSelectPanel(currentPanel, true, true);
      setTimeout(SidebarSelectPanel, 0, currentPanel, true, true); */
  }
}

function addTabDataSeqToDS(aDataSource, aSessionContainerResource, startIndex, aTabBrowser, skipAboutBlank, aSelectedTabStyle, aSaveHistoryFlag, aShouldAppend)
{
  if (!aSessionContainerResource)
    return -1;

  if (startIndex)
    aSessionContainerResource = aSessionContainerResource.replace(/:tabs:tab-\w*/, ":tabs:root");

  if (aShouldAppend == 'undefined')
    aShouldAppend = false;

  if (startIndex == 'undefined')
    startIndex = 0;

  var tabBrowser = aTabBrowser ? aTabBrowser : getBrowser();
  var tabs = tabBrowser.mTabContainer.childNodes.length;
  var resourcePrefix = aSessionContainerResource.replace(/tabs:root/, "tabs");
  var sessionName = aSessionContainerResource;
  var tab, browser, url;

  for (var i = 0; i < tabs; i++) {
    tab = tabBrowser.mTabs[i];
    url = "about:blank";
    browser = tab.linkedBrowser;

    try {
      url = browser.webNavigation.currentURI.spec;
    } catch(ex) {
      try {
        url = tab.getAttribute("firstPage");
      } catch(ex) {
        return i;
      }
    }
    if (skipAboutBlank && url == "about:blank")
      continue;

    var document = browser.contentDocument;
    var isWebFeed = ('mzFeedHandler' in window) ? mzFeedHandler.isDocumentTransformedFeed(document) : false;
    var title = tab.getAttribute("label");

    if (isWebFeed) {
      var mzBundle = Components.classes["@mozilla.org/intl/stringbundle;1"]
                               .getService(Components.interfaces.nsIStringBundleService)
                               .createBundle("chrome://multiviews/locale/multizilla.properties");

      var feedTitle = mzBundle.GetStringFromName("feedTitle").replace(/%S/, "");

      if (title.match(feedTitle)) {
        title = title.substring(feedTitle.length, title.length);
      }
    }
    // tab data collecting
    var propertyValues = new Array();
    propertyValues.push(title);
    propertyValues.push(tab.hasAttribute("customLabel") ? "true" : "false");
    propertyValues.push(tab.getAttribute("firstPage"));
    var iconURL = isWebFeed ? "chrome://multiviews/content/images/icons/page-livemarks.png" : tab.getAttribute("image");
    propertyValues.push(iconURL);
    propertyValues.push(url);
    propertyValues.push(tab.mReloadInterval);
    var activeTab = (!aShouldAppend && tab == tabBrowser.selectedTab);
    propertyValues.push((activeTab ? "true" : "false"));
    // Adjust the 'ordinal' when we append items to a session!
    var tabIndex = (aShouldAppend && startIndex) ? startIndex : i; // .ordinal;
    propertyValues.push(tabIndex);
    propertyValues.push(browser.docShell["allowPlugins"] ? "true" : "false");
    propertyValues.push(browser.docShell["allowJavascript"] ? "true" : "false");
    propertyValues.push(browser.docShell["allowMetaRedirects"] ? "true" : "false");
    propertyValues.push(browser.docShell["allowSubframes"] ? "true" : "false");
    propertyValues.push(browser.docShell["allowImages"] ? "true" : "false");
    var selectedTabStyle = (activeTab && aSelectedTabStyle) ? aSelectedTabStyle : "";
    propertyValues.push(selectedTabStyle);

    if (aSaveHistoryFlag == undefined)
      aSaveHistoryFlag = true; // Force callers to save history

    if (aSaveHistoryFlag) {
      var historyItemsToKeep = gMultiZilla.prefs.readInteger("multizilla.browsersession.history-items", 0);

      if (historyItemsToKeep > 0) {
        var history = tabBrowser.getTabHistory(tab);

        if (history)
          propertyValues.push(history);

      }
    }
    var tabResource = resourcePrefix + ":tab-" + (i + startIndex);
    mzAddOrInsertTabToSession(aDataSource, aSessionContainerResource, tabResource, false, propertyValues, selectedTabStyle);

    if (activeTab)
      sessionName = propertyValues[0];
  }
  return sessionName;
}

function mzAddOrInsertTabToSession(aDataSource, aSessionContainerResource, aTabResource, aInsertIndex, propertyValues, aSelectedTabStyle)
{
  var sessionContainer = mzConstructRDFContainer(aDataSource, "seq", aSessionContainerResource, false);

  if (!sessionContainer)
    return;

  var newNode = RDF.GetResource(aTabResource);

  if (!newNode)
    return;

  var property;
  var mzRDF = "http://multizilla.mozdev.org/rdf#";
  var propertyArray = [ "label", "custom", "firstpage", "image", "location", "interval", "selected", 
                        "ordinal", "allowPlugins", "allowJavascript", "allowMetaRedirects", 
                        "allowSubframes", "allowImages", "selectedTabStyle", "hisindex", "history-" ];
  var defaultValues = [ "about:blank", "false", "about:blank", "", "about:blank", "0", "false", 
                        -1, "true", "true", "true", "true", "true", "", "" ];

  // Store general properties
  for (var i = 0; i < 13; i++) {
    // Keep the RDF as compact as possible by only adding modified/none default values
    if (propertyValues[i] != undefined && propertyValues[i] != defaultValues[i]) {
      property = mzRDF + propertyArray[i];
      aDataSource.Assert(newNode, RDF.GetResource(property), RDF.GetLiteral(propertyValues[i]), true);
    }
  }
  /***
    * The Tab Manager enables you to select a different styling (CSS) for the selected tab
    * Note that 'aSelectedTabStyle' should only be used when called from the Tab Manager
    * FYI: we didn't change 'selected' into 'selectedTab' to preserve backward compatibility
    */
  if (aSelectedTabStyle && propertyValues[6] == "true") {
    property = mzRDF + propertyArray[i];
    aDataSource.Assert(newNode, RDF.GetResource(property), RDF.GetLiteral(aSelectedTabStyle), true);
  }
  /***
    * We need to check the length of 'propertyValues' and check the type of the 
    * 14th value to find out if we have a history object (because the history 
    * items are put into a single object). 
    */
  if (propertyValues.length == 15 && typeof(propertyValues[14]) == "object") {
    var entry = 0;
    var devider = "|mzRDF-DiViDeR|";
    var history = propertyValues[14];
    // Append history data to 'historyItem' and assert it one after the other
    for (entry = 0; entry < history.tabHistoryEntries.length; entry++) {
      var historyItem = history.tabHistoryEntries[entry].uri.spec + devider + 
                        history.tabHistoryEntries[entry].isSubFrame + devider + 
                        history.tabHistoryEntries[entry].x + devider + 
                        history.tabHistoryEntries[entry].y + devider +
                        history.tabHistoryEntries[entry].title;
      property = mzRDF + propertyArray[15] + entry;
      aDataSource.Assert(newNode, RDF.GetResource(property), RDF.GetLiteral(historyItem), true);
    }
    if (entry > 0) { // Assert the history index
      property = mzRDF + propertyArray[14];
      aDataSource.Assert(newNode, RDF.GetResource(property), RDF.GetLiteral(history.tabHistoryIndex), true);
    }
  }
  if (aInsertIndex)
    sessionContainer.InsertElementAt(newNode, aInsertIndex, true);
  else
    sessionContainer.AppendElement(newNode, true);
}

function mzWriteSessionName(aDataSource, aResource, aName)
{
  mzAddOrChangeProperty(aDataSource, aResource, "name", aName);
  mzFlushDataSource(aDataSource);
}

function mzAddOrChangeStyleProperty(aDataSource, aResource, aName)
{
  mzAddOrChangeProperty(aDataSource, aResource, "selectedTabStyle", aName);
  mzFlushDataSource(aDataSource);
}

function mzAddOrChangeProperty(aDataSource, aResource, aProperty, aValue)
{
  var property = RDF.GetResource("http://multizilla.mozdev.org/rdf#" + aProperty);
  var currentValue = aDataSource.GetTarget(RDF.GetResource(aResource), property, true) || null;
  var newValue = RDF.GetLiteral(aValue);

  mzAddOrChangeValue(aDataSource, RDF.GetResource(aResource), property, currentValue, newValue);
}

function renameTabSession(aDataSource, aResource, aWindow)
{
  if (aWindow == undefined)
    aWindow = window;

  var sessionName = aResource;
  var property = RDF.GetResource("http://multizilla.mozdev.org/rdf#name");
  var data = aDataSource.GetTarget(RDF.GetResource(aResource), property, true);

  if (data) {
    data = data.QueryInterface(Components.interfaces.nsIRDFLiteral);
    sessionName = data.Value;
  }
  aWindow.openDialog("chrome://multiviews/content/session-manager/miscellaneous/renameSessionWindow.xul", "",
                     "chrome, modal, dialog, centerscreen", aWindow, aResource,
                     sessionName, gMultiZilla.utility.getPlatformExtras());

  if (gSessionName) { // This was gSessionName but that somehow no longer works!
    if (gSessionName != sessionName)
      mzSetModifiedDate(aDataSource, aResource);
    mzWriteSessionName(aDataSource, aResource, gSessionName);
  }
}

// This function is called from menu item 'Load previously stored Tab Session' in tabbrowser.xml only!
function mzRestoreTabSession()
{
  var windowResourceID = mzGetRelativeWindowResourceID(window); // mzGetWindowResourceID(window);

  if (windowResourceID == null) {
    alert("Error in mzRestoreTabSession()");
    return;
  }
  var sessionID = 0;
  // Use a relative datasource ID to build the resource
  var resource = "urn:session-" + sessionID + ":" + windowResourceID + ":tabs:root";
  // window.document.documentElement.setAttribute("sessionID", resource); // Checked when saving tab sessions

  if (mzCheckResource(mzRDFStore, resource)) {
    var windowManagerDS = Components.classes['@mozilla.org/rdf/datasource;1?name=window-mediator']
                                    .getService(nsIWindowDataSource);  
    // We used a relative datasource ID to build the resource but we need to use the 
    // original datasource ID of our window or 'mzGetWindowResourceID' will return null!
    windowResourceID = mzGetWindowResourceID(window);
    var browserWin = windowManagerDS.getWindowForResource(windowResourceID);
    var browser = browserWin.getBrowser();
    mzRestoreTabsFromResource(mzRDFStore, browser, resource, false);
  }
  else {
    var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                  .getService(Components.interfaces.nsIPromptService);
    promptService.alert(window,
                        gMultiZilla.utility.getTextFromBundle("mzWizard"),
                        gMultiZilla.utility.getTextFromBundle("noTabSessionAssociated"));
  }
}

// This function is called from method 'initMultiZilla' in tabbrowser.xml only!
function mzRestoreBrowserSession(aCheckFirstWindow)
{
  if ((!gMultiZilla.utility.isFirstWindow() && aCheckFirstWindow) || !mzRDFStore)
    return;
  // We can only check for duplicates *after* the startup page(s) is/are loaded!
  if (gMultiZilla.prefs.readBoolean("multizilla.prevent.double", true) || 
      gMultiZilla.prefs.readInteger("browser.startup.page", 0) > 0) {
    var delayed = false;

    if (pref.getIntPref("browser.startup.page") == 1) {
      var pageCount = pref.getIntPref("browser.startup.homepage.count");
      var tab = gBrowser.mTabContainer.childNodes;

      if (pageCount != tab.length)
        delayed = true;
      for (var i = 0; i < tab.length; i++) {
        if (tab[i].hasAttribute("busy")) {
          delayed = true;
          break;
        }
      }
    }
    if (delayed && aCheckFirstWindow) {
      setTimeout(mzRestoreBrowserSession, 1000, true); // Delayed recursive call
      return;
    }
  }
  var restoreSessionPref = gMultiZilla.prefs.readInteger("multizilla.browsersession.restore.behavior", 3);
  var restoreWindowBehavior = gMultiZilla.prefs.readInteger("multizilla.browsersession.window.restore-behavior", 0);
  var skipSingleWindowConfirmation = gMultiZilla.prefs.readBoolean("multizilla.browsersession.skip.confirm-single-window", false);
  // This fails when "session-0" bas been removed!
  var windowResources = mzGetWindowResourcesForSession(mzRDFStore, "session-0");
  var restoreSingleWindow = (windowResources.length == 1);
  var restoreAllWindows = (windowResources.length > 1);
  var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                .getService(Components.interfaces.nsIPromptService);

  if (restoreSessionPref == 3 && (restoreAllWindows || (!skipSingleWindowConfirmation && restoreSingleWindow))) {
    var checkbox = {value:0};
    var text, windowButtonLabel, allWindowsButtonLabel;
    var title = gMultiZilla.utility.getTextFromBundle("restoreTabSessionTitle");
    var dontRestoreButtonLabel = gMultiZilla.utility.getTextFromBundle("restoreNoneButtonLabel");

    // Are we restoring a single window?
    if (restoreSingleWindow) {
      // Yes, select text and button label for single window restore
      text = gMultiZilla.utility.getTextFromBundle("restoreTabSessionText");
      var windowButtonProperty = (restoreWindowBehavior == 0) ? "restoreTabsButtonLabel"
                                                              : "restoreSessionButtonLabel";
      windowButtonLabel = gMultiZilla.utility.getTextFromBundle(windowButtonProperty);
      allWindowsButtonLabel = null; // Hide 'All Windows' button
    }
    else { // restoreAllWindows
      // Select text and button label for multiple windows
      text = gMultiZilla.utility.getTextFromBundle("restoreAllSessionsText");
      text = text.replace(/%S/, windowResources.length);
      windowButtonLabel = gMultiZilla.utility.getTextFromBundle("restoreFirstWindowButtonLabel");
      allWindowsButtonLabel = gMultiZilla.utility.getTextFromBundle("restoreAllWindowsButtonLabel");
    }
    dump("\nwindowButtonLabel: " + windowButtonLabel);
    dump("\ndontRestoreButtonLabel: " + dontRestoreButtonLabel);
    dump("\nallWindowsButtonLabel: " + allWindowsButtonLabel);
    var result = promptService.confirmEx(window, title, text, promptService.BUTTON_POS_1_DEFAULT +
                                         (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_IS_STRING) +
                                         (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_IS_STRING) +
                                         (promptService.BUTTON_POS_2 * promptService.BUTTON_TITLE_IS_STRING),
                                         windowButtonLabel, dontRestoreButtonLabel, allWindowsButtonLabel, null, checkbox);
    if (result == 1) // 0 = First Window Only / 1 = Don't Restore, Esc or Close Window / 2 = All Windows
      return;
    else if (result == 0)
      restoreSessionPref = 1;
  }
  var browserWindow = window;
  var tabbrowser = getBrowser();
  var restoredTabs = 0;
  // Restore all tabs from a previously saved browser session
  for (index in windowResources) {
    var resource = windowResources[index];

    if (mzCheckResource(mzRDFStore, resource)) {
      var windowSettings = mzGetWindowSettings(mzRDFStore, resource);

      if (index > 0) {
        mzGetNewWindowObject(windowSettings);
        browserWindow = windowSettings.browserWindow;
        tabbrowser = windowSettings.tabbrowser;
      }
      browserWindow.mzRestoreWindowSettings(windowSettings);
      restoredTabs += mzRestoreTabsFromResource(mzRDFStore, tabbrowser, resource, true);
      mzSetModifiedDate(mzRDFStore, resource, "lastRestore");

      if (restoreSessionPref == 1) // Restore first window only!
        break;
    }
  }
  var shouldNotify = gMultiZilla.prefs.readBoolean("multizilla.browsersession.restore.notify", false)

  if (shouldNotify && windowResources.length >= 1) {
    var paramArray = [restoredTabs, index];
    var message = gMultiZilla.utility.getTextFromBundle("restoreTabSessionNotifyText", paramArray, null);
    message = message + " " + (index > 0 ? gMultiZilla.utility.getTextFromBundle("windowText")
                                         : gMultiZilla.utility.getTextFromBundle("windowsText"));
    var notifyText = gMultiZilla.utility.getTextFromBundle("mzWizard");
    var mostRecentWin = gMultiZilla.utility._getMostRecentWindow();
    setTimeout(promptService.alert, 1000, mostRecentWin, notifyText, message);
  }
}

function mzRestoreTabsFromResourceList(aDataSource, aBrowser, aResourceList)
{
  var resources = aResourceList.split(',');
  resources[0] = resources[0].replace(/sm:resources:/, '');

  for (i in resources) {
    var result;

    if (resources[i].match(/:tabs:root/)) { // Window
      result = mzRestoreTabsFromResource(aDataSource, aBrowser, resources[i], true);
    }
    else { // Tab
	  var resource = RDF.GetResource(resources[i]);
      result = mzRestoreTabFromResource(aDataSource, aBrowser, resource, true, false);
    }
    if (result == -1)
      break;
  }
  return result;
}

function mzRestoreTabsFromResource(aDataSource, aBrowser, aResource, aStartupFlag)
{
  var tabContainer = mzRDFCUtils.MakeSeq(aDataSource, RDF.GetResource(aResource)).GetElements();

  if (!tabContainer)
    return -1;

  var addTab = true; // default is to add new tabs
  // The next two vars are used when we're called from mzRestoreTabSession()
  var newTabs = true;
  var closeRemaining = false;

  if (aStartupFlag) { // called from mzRestoreBrowserSession()
    // Do we need to use unused (blank) tabs before opening new tabs?
    if (useBlankTabs) {
      var useBlankTabs = gMultiZilla.prefs.readBoolean("multizilla.tabs.blank-tab-first", true);
      /*** 
        * The user wants to use unused (blank) tabs before opening new tabs. We *must* 
        * check the preference to find out if mozilla had to load one or more pages.
        * We can't simply use the first unused (blank) tab if that's the case!
        */	
      addTab = (gMultiZilla.prefs.readInteger("browser.startup.page", 0) > 0);
    }
  }
  else { // called from mzRestoreTabSession() or session manager
    var prefValue = gMultiZilla.prefs.readInteger("multizilla.browsersession.restore.tabs.behavior", 0); // 0 = reuse / 1 = new / 2 = asks / 4 = close remaining tabs
    closeRemaining = (prefValue & 4);

    if (prefValue == 2) {
      window.openDialog("chrome://multiviews/content/session-manager/miscellaneous/tabBehaviorSelection.xul", 
                        "", "chrome,modal,centerscreen", aBrowser,
                        closeRemaining, gMultiZilla.utility.getPlatformExtras());

      if (aBrowser.mNewTabs == -1)
        return -1;
      newTabs = (aBrowser.mNewTabs == 0) ? false : true;
      closeRemaining = aBrowser.mRemoveRemaining;
    }
    else 
      newTabs = (prefValue == 1);
  }
  mzTabIndex = 0; // global var!
  var numberOfTabs = 0;
  var addedTabs = 0;
  // addTab is now either true or false, based on the users pref settings
  while (tabContainer.hasMoreElements()) {

    try {
      var resource = tabContainer.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      mzTabIndex++;

      if (aStartupFlag)
        addTab = (mzTabIndex > 1) ? true : addTab
      else if (newTabs == false) {
        numberOfTabs = aBrowser.mTabs.length;
        addTab = (mzTabIndex > numberOfTabs);

        if (addTab)
          addedTabs++;
      }
      mzRestoreTabFromResource(aDataSource, aBrowser, resource, addTab, aStartupFlag);
    } catch(ex) {
      mzDebug(ex);
      continue;
    }
  }
  if (addedTabs)
    mzSetModifiedDate(aDataSource, aResource);
  // Do we need to remove any remaining tabs? 
  // Skip this section for invalid resources (mzTabIndex == 0)
  if (closeRemaining && mzTabIndex > 0) {
    numberOfTabs = aBrowser.mTabs.length;
    addedTabs = numberOfTabs - mzTabIndex;

    // Do we have any new tabs?
    if (addedTabs == 0)
      return mzTabIndex; // Nope, nothing to do
    if (aStartupFlag) { // mzRestoreBrowserSession (at startup) only!
      // var startTab = newTabs ? (addedTabs+1) : 0;
      var i = 0;

      if (newTabs) {
        /* startTab = aBrowser.mTabList[startTab].ordinal-1;
        for (i = startTab; i >= 0; i--)
          aBrowser.removeTab(aBrowser.mTabList[i]); */
      }
      else {
        for (i = numberOfTabs; i >= mzTabIndex; i--)
          aBrowser.removeTab(aBrowser.mTabs[i]);
      }    
    }
  }
  return mzTabIndex;
}

function mzAddOrChangeValue(aDataSource, aResource, aProperty, currentValue, aValue)
{
  if (currentValue)
    aDataSource.Change(aResource, aProperty, currentValue, aValue);
  else
    aDataSource.Assert(aResource, aProperty, aValue, true);
  mzFlushDataSource(aDataSource);
}

/* function mzMoveTab(aDataSource, aSourceResource, aTargetResource, aSourceTabResource, aTargetTabResource)
{
  aDataSource.Move(RDF.GetResource(aSourceResource), RDF.GetResource(aTargetResource), 
                   RDF.GetResource(aSourceTabResource), RDF.GetResource(aTargetTabResource));
  mzFlushDataSource(aDataSource);
} */

function mzRestoreTabFromResource(aDataSource, aTabbrowser, aResource, aNewTabFlag, aStartupFlag)
{
  var dataCollection = mzGetTabDataFromResource(aDataSource, aResource);

  if (aTabbrowser)
    mzAddTabFromCollection(aTabbrowser, dataCollection, aNewTabFlag, aStartupFlag);
}

function mzGetTabDataFromResource(aDataSource, aResource)
{
  var dataCollection = new Array;
  var mzRDF = "http://multizilla.mozdev.org/rdf#";
  var propertyArray = [ "label", "custom", "firstpage", "image", "location", "interval", "selected", 
                        "ordinal", "allowPlugins", "allowJavascript", "allowMetaRedirects", 
                        "allowSubframes", "allowImages", "selectedTabStyle" ];
  var defaultValues = [ "about:blank", "false", "about:blank", "", "about:blank", "0", "false", "0", 
                        "true", "true", "true", "true", "true", "" ];
  // const addMissingProperties = false;
  var resource, property, data;

  for (var i = 0; i < propertyArray.length; i++) {
    if (i == 7) // Skip ordinal!
      continue;
    resource = mzRDF + propertyArray[i];
    property = RDF.GetResource(resource);

    try {
      data = aDataSource.GetTarget(aResource, property, true);
      data = data.QueryInterface(Components.interfaces.nsIRDFLiteral);
      dataCollection[i] = data.Value;
    }
    catch(ex) {
      dataCollection[i] = defaultValues[i];

      /* if (addMissingProperties) {
        alert("Error: "+propertyArray[i]+" is missing...adding with default value: "+dataCollection[i]);
        aDataSource.Assert(aResource, property, RDF.GetLiteral(dataCollection[i]), true);
        aDataSource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
      } */
    }
  }
  // Retrieve history, starting with the history index number
  var historyObj = mzGetHistoryFromResource(aDataSource, aResource);

  if (historyObj) {
    /* dump("\ntabHistoryEntries: " + historyObj.tabHistoryEntries +
         "\ntabHistoryIndex  : " + historyObj.tabHistoryIndex); */
   dataCollection.push(historyObj);
  }
  return dataCollection;
}
//////////////////////////////////////////////////////////////////////////
// This was part of 'mzGetTabDataFromResource' but this makes it easier
// for us to also use it from the property window of the Session Manager 
//////////////////////////////////////////////////////////////////////////
function mzGetHistoryFromResource(aDataSource, aResource)
{
  if (typeof(aResource) == "string")
    aResource = RDF.GetResource(aResource);
  // Retrieve history, starting with the history index number
  var mzRDF = "http://multizilla.mozdev.org/rdf#";
  var property = RDF.GetResource(mzRDF + "hisindex");
  var historyIndex = aDataSource.GetTarget(aResource, property, true);
  // dump("\nhistoryIndex: " + historyIndex);
  // Do we have a history index number?
  if (historyIndex instanceof Components.interfaces.nsIRDFNode) {
    // Yes we do, get its value
    historyIndex = Number(historyIndex.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);
    var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                              .getService(Components.interfaces.nsIIOService);
    var entries = new Array();
    /***
      * We 'only' store/retrieve the 15 most recent history items
      */
    for (var hIndex = 0; hIndex < 15; hIndex++) {
      property = RDF.GetResource(mzRDF + "history-" + hIndex);

      try {
        var data = aDataSource.GetTarget(aResource, property, true);
        data = data.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
        /***
          * Construct a new history object from the raw data
          */
        var historyData = data.split("|mzRDF-DiViDeR|");
        var URI = ioService.newURI(historyData[0], null, null);
        entries[hIndex] = { index: Number(hIndex),
                            uri: URI,
                            title: historyData[4],
                            isSubFrame: Boolean(historyData[1]),
                            x: Number(historyData[2]),
                            y: Number(historyData[3]) };
      }
      catch(ex) {
        // die silently
        // dump("\nFailed to read: " + mzRDF + "history-" + hIndex);
      }
    }
    if (entries.length) {
      if (historyIndex < 0 || historyIndex > (entries.length - 1))
        historyIndex = 0;

      var historyObj = { tabHistoryEntries: entries,
                         tabHistoryIndex: historyIndex }
      return historyObj;
    }
  }
  return null;
}
//////////////////////////////////////////////////////////////////////////
// This function loads the given URL in a new or unused (about:blank) tab
// and sets all attributes and docShell properties from 'aDataCollection'
//////////////////////////////////////////////////////////////////////////
function mzAddTabFromCollection(aTabbrowser, aDataCollection, newTabFlag, aStartupFlag)
{
  var tabbrowser = aTabbrowser;
  var browser, tab = tabbrowser.selectedTab;

  if (newTabFlag) {
    browser = gMultiZilla.tabs.checkForBlankTab(tabbrowser);

    if (!browser) {
      // 'restoreTab' skips setting the default permissions in addOrInsertTab
      tab = tabbrowser.addTab(null, null, null, false, null, "restoreTab");
    }
  }
  else if (aStartupFlag == false) {
    tab = tabbrowser.mTabs[mzTabIndex-1];
  }
  if (!browser) 
    browser = tab.linkedBrowser;

  // Set browser properties before we load the URL for this tab
  browser.docShell.allowPlugins = (aDataCollection[8] == "true");
  browser.docShell.allowJavascript = (aDataCollection[9] == "true");
  browser.docShell.allowMetaRedirects = (aDataCollection[10] == "true");
  browser.docShell.allowSubframes = (aDataCollection[11] == "true");
  browser.docShell.allowImages = (aDataCollection[12] == "true");

  // Do we have a custom label?
  if (aDataCollection[1] == "true") { // Yes, restore custom label
    tab.setAttribute("customLabel", true);
    tab.setAttribute("label", aDataCollection[0]);
  }
  tab.setAttribute("firstPage", aDataCollection[2]);
  var interval = aDataCollection[5];
  /***
    * The 14th element is an object when tab history can be retrieved
    */
  if (aDataCollection.length == 15)
    var dummy = tabbrowser.setTabHistory(tab, aDataCollection[14]);
  else
    browser.loadURI(aDataCollection[4]);

  if (interval >= 30000) // We only honor intervals of 30+ seconds
    tabbrowser.setReloadInterval(tab, interval);
  // Activate the (new) tab
  if (aDataCollection[6] == "true")
    tabbrowser.selectedTab = tab;
}

function mzRemoveAllResourcesFor(aDataSource, aResource)
{
  mzRemoveResources(aDataSource, aResource);
  mzRemoveDataResources(aDataSource, aResource);
  mzRemoveSession(aDataSource, aResource);
  mzFlushDataSource(aDataSource);
}

function mzRemoveTabResourcesFor(aDataSource, aResource, aRemoveSessionContainerFlag) 
{
  if (aRemoveSessionContainerFlag == undefined)
    aRemoveSessionContainerFlag = true;

  mzRemoveResources(aDataSource, aResource);
  mzRemoveDataResources(aDataSource, aResource);

  var tabContainerResource = aResource.replace(/:tabs:tab-\w*/, ":tabs:root");
  var tabContainer = mzConstructRDFContainer(aDataSource, "seq", tabContainerResource, false);

  tabContainer.RemoveElement(RDF.GetResource(aResource), true);

  if (aRemoveSessionContainerFlag)
    mzRemoveSession(aDataSource, aResource);
  mzFlushDataSource(aDataSource);
}

function mzRemoveResources(aDataSource, aResource)
{
  var container = mzRDFCUtils.MakeSeq(aDataSource, RDF.GetResource(aResource)).GetElements();

  if (!container)
    return;
  mzDebug("mzRemoveResources");
  /*** 
  * EXAMPLE: mzRemoveResources('urn:session-0:window-0:tabs:tab-0');
  * removes:
    <RDF:Description about="urn:session-0:window-0:tabs:tab-0"
                     MZ:label="mozdev.org - multizilla: index"
                     MZ:custom="false"
                     MZ:firstpage="false"
                     MZ:image="http://www.mozilla.org/images/mozilla-16.png"
                     MZ:location="http://multizilla.mozdev.org/index.html"
                     MZ:interval="0"
                     MZ:selected="false"
                     MZ:allowPlugins="true"
                     MZ:allowJavascript="true"
                     MZ:allowMetaRedirects="true"
                     MZ:allowSubframes="true"
                     MZ:allowImages="true" /> */

  while (container.hasMoreElements()) {
    try {
      var resource = container.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      var labels = aDataSource.ArcLabelsOut(resource);

      while (labels.hasMoreElements()) {
        try {
          var label = labels.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
          var data = aDataSource.GetTarget(resource, label, true);
          aDataSource.Unassert(resource, label, data);
        } 
        catch(ex) { 
          mzDebug(ex);
        }
	  }
    } catch(ex) {
      mzDebug(ex);
      continue;
    }
  }
}

function mzRemoveDataResources(aDataSource, aResource)
{
  var resource = RDF.GetResource(aResource);
  var labels = aDataSource.ArcLabelsOut(resource);
  /*** 
  * EXAMPLE: mzRemoveDataResources('urn:session-0:window-0:tabs:root');
  * removes:
    <RDF:Seq about="urn:session-0:window-0:tabs:root">
      <RDF:li resource="urn:session-0:window-0:tabs:tab-0"/>
    </RDF:Seq> */

  while (labels.hasMoreElements()) {
    try {
      var label = labels.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      var data = aDataSource.GetTarget(resource, label, true);
      aDataSource.Unassert(resource, label, data);
    }
    catch(ex) {
      mzDebug(ex);
    }
  }
}

function mzRemoveSession(aDataSource, aResource)
{
  var sessionsContainer = mzGetSessionContainer(aDataSource);
  sessionsContainer.RemoveElement(RDF.GetResource(aResource), true);
  /***
    * We need to cleanup multizilla.rdf when we remove the last window:
    *   <RDF:Seq about="urn:sessions:root">
    *   </RDF:Seq>
    * ...or we keep the above 'garbage' alive in multizilla.rdf
    */
  if (sessionsContainer.GetCount() < 1) {
    mzRemoveDataResources(aDataSource, "urn:sessions:root");
  }
}

function mzCheckResource(aDataSource, aResource)
{
  var tabContainer = mzRDFCUtils.MakeSeq(aDataSource, RDF.GetResource(aResource)).GetElements();
  var found = tabContainer.hasMoreElements();

  if (!found)
    mzRemoveDataResources(aDataSource, aResource); // cleanup junk resource
  return found;
}
////////////////////////////////////////////////////////////////////////////////
// example: 
// mzCheckResourceForValue("urn:sessions:root", "urn:session-0:window-1:tabs:root")
////////////////////////////////////////////////////////////////////////////////
function mzCheckResourceForValue(aDataSource, aResourceID, aValue)
{
  var container = mzRDFCUtils.MakeSeq(aDataSource, RDF.GetResource(aResourceID)).GetElements();

  if (!aDataSource || !container)
    return false;

  while (container.hasMoreElements()) {
    try {
      var resource = container.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      if (resource.Value == aValue)
        return resource.Value;
    } catch(ex) {
      mzDebug(ex);
    }
  }
  return false;
}

function mzGetResourcesFromContainer(aDataSource, aContainerResource)
{
  if (aContainerResource == undefined)
    throw("aResource == undefined in mzGetResourcesFromContainer()");
  else if (typeof aContainerResource == 'string')
    aContainerResource = RDF.GetResource(aContainerResource);

  var container = mzRDFCUtils.MakeSeq(aDataSource, aContainerResource).GetElements();

  if (!container)
    return -1;

  var resourceList = new Array();

  while (container.hasMoreElements()) {
    var resource = container.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
    resourceList.push(resource.Value);
  }
  return resourceList;
}

function mzGetLastResourceFromContainer(aDataSource, aResource)
{
  if (aResource == undefined)
    throw("aResource == undefined in mzGetLastResourceFromContainer()");
  else if (typeof aResource == 'string')
    aResource = RDF.GetResource(aResource);

  var resource = null;
  var container = mzRDFCUtils.MakeSeq(aDataSource, aResource).GetElements();

  if (!container)
    return -1;

  while (container.hasMoreElements()) {
    resource = container.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
  }
  return (resource == null) ? null : resource.Value;

  /***
    * We might also use the following code; but it might fail after we've removed/inserted tabs!
    *
    * var count = container.GetCount();
    * var lastResource = "http://www.w3.org/1999/02/22-rdf-syntax-ns#_" + (count-1);
    * aDataSource.GetTarget(aResource, RDF.GetResource(lastResource), true);
    * return (lastResource == null) ? null : lastResource;
    */
}

function mzGetParentOfContainer(aDataSource, aContainer)
{
  aContainer = RDF.GetResource(aContainer);
  var arcsIn = aDataSource.ArcLabelsIn(aContainer);

  while (arcsIn.hasMoreElements()) {
    var containerArc = arcsIn.getNext();
    if (mzRDFCUtils.IsOrdinalProperty(containerArc)) {
      /***
        * "urn:session-x:window-x:tabs:tab-x" -> "urn:session-x:window-x:tabs:root"
        * "urn:session-x:window-x:tabs:root" -> "urn:sessions:root"
        */
      var resource = aDataSource.GetSources(containerArc, aContainer, true).getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      // alert(resource.Value);
    }
  }
  return resource.Value || null;
}
////////////////////////////////////////////////////////////////////////////////
// This function reads from a given file (aFile) and returns a pointer the 
// content or null on failure
////////////////////////////////////////////////////////////////////////////////	
function mzReadFromFile(aFile) 
{
  if (aFile) {
    var inputStream = Components.classes["@mozilla.org/network/file-input-stream;1"]
                                .createInstance(Components.interfaces.nsIFileInputStream);

    var scriptableStream = Components.classes["@mozilla.org/scriptableinputstream;1"]
                                     .createInstance(Components.interfaces.nsIScriptableInputStream);

    try {
      /**
       * inputStream.init( file, ioFlags, perm, behaviorFlags );
       *
       * @param file          - file to read from (must QI to nsILocalFile)
       * @param ioFlags       - file open flags (see File open bits)
       * @param permission    - file mode bits (see File mode bits)
       * @param behaviorFlags - flags specifying various behaviors of the class (see enumerations in the class)
      */
      inputStream.init(aFile, 1, 0, 0);
      scriptableStream.init(inputStream);
      var content = scriptableStream.read(scriptableStream.available());
      scriptableStream.close();
      inputStream.close();
    } catch(ex) {
      mzDebug(ex);
      return null;
    }
    return content;
  }
  return null;
}
////////////////////////////////////////////////////////////////////////////////
// This function write the given content (aContent) to a given file (aFile) and 
// it returns false on failure and true on success.
////////////////////////////////////////////////////////////////////////////////
function mzWriteToFile(aFile, aContent)
{
  if (aFile && aContent) {
    if (aFile.exists()) 
      aFile.remove(true);

    var outputStream = Components.classes["@mozilla.org/network/file-output-stream;1"]
                                 .createInstance(Components.interfaces.nsIFileOutputStream);

    try {
      aFile.create(aFile.NORMAL_FILE_TYPE, 0666);
      /**
       * outputStream.init( file, ioFlags, perm, behaviorFlags );
       *
       * @param file          - file to write to (must QI to nsILocalFile)
       * @param ioFlags       - file open flags (see File open bits)
       * @param permission    - file mode bits (see File mode bits)
       * @param behaviorFlags - currently none supported
      */
      outputStream.init(aFile, 2, 0x200, 0);
      outputStream.write(aContent, aContent.length);
      outputStream.close();
    } catch(ex) {
      mzDebug(ex);
      return false;
    }
    return true;
  }
  return false;
}

function mzFlushDataSource(aDataSource)
{
  try {
    aDataSource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
  } catch(ex) {
    // die silently
  }
}

function mzRemoveProperties(aContainerResource, aPropertList)
{
  var resourceList = mzGetResourcesFromContainer(mzRDFStore, RDF.GetResource(aContainerResource));

  for (rIndex in resourceList) {
    var resource = RDF.GetResource(resourceList[rIndex]);

    for (i in aPropertList) {
      var property = RDF.GetResource(aPropertList[i]);
      var data = mzRDFStore.GetTarget(resource, property, true);

      if (data)
        mzRDFStore.Unassert(resource, property, data);
    }
  }
}

function mzSetModifiedDate(aDataSource, aResource, aProperty)
{
  if (aProperty == undefined)
    aProperty = "lastModified";

  var now = new Date().getTime();
  mzAddOrChangeProperty(aDataSource, aResource, aProperty, String(now));
}
////////////////////////////////////////////////////////////////////////
// Not in use yet, still work in progress
////////////////////////////////////////////////////////////////////////
function mzRemoveHistoryByObjects(aHistoryObject)
{
  var hisindex_property = RDF.GetResource("http://multizilla.mozdev.org/rdf#hisindex");

  for (i in aHistoryObject) {
    var obj = aHistoryObject[i];
    var index = Number(obj.index);
    var resource = RDF.GetResource(obj.resource);
    var property = RDF.GetResource(obj.property);
    var data = mzRDFStore.GetTarget(resource, property, true);

    if (data) {
      var historyObj = mzGetHistoryFromResource(mzRDFStore, resource);
      mzRDFStore.Unassert(resource, property, data);

      if (historyObj.tabHistoryEntries.length == 1) {
        mzRDFStore.Unassert(resource, hisindex_property, data);
      }
      else {
        var hisindex = Number(data.QueryInterface(Components.interfaces.nsIRDFLiteral).Value);

        if (index == hisindex) {
          dump("\nHistory index will be LOST after this removal!");
        }
        if (index < hisindex || index > hisindex)
          dump("\nHistory index will be OK after this removal!");
      }
    }
    data = mzRDFStore.GetTarget(resource, property, true);
    // mzRDFStore.Unassert(resource, property, data);
    mzFlushDataSource(mzRDFStore);
  }
}

function mzCheckAvailabilityOfFeedPanel()
{
  const NC_URI = "http://home.netscape.com/NC-rdf#";
  var panelDatasource = mzGetNewOrCurrentDSByFilename("panels.rdf", false, null, false);
  var resource = panelDatasource.GetTarget(RDF.GetResource("urn:sidebar:current-panel-list"), 
                                           RDF.GetResource(NC_URI + "panel-list"), true);
  var panels = mzConstructRDFContainer(panelDatasource, "seq", resource, false);
  var newPanelResource = RDF.GetResource("urn:sidebar:panel:webfeeds");
  var panelResource = panelDatasource.GetTarget(resource, newPanelResource, true);
  
  if (panels.IndexOf(newPanelResource) == -1) {
    panelDatasource.Assert(newPanelResource, RDF.GetResource(NC_URI + "title"), 
                                             RDF.GetLiteral("Web Feeds"), true);
    panelDatasource.Assert(newPanelResource, RDF.GetResource(NC_URI + "content"), 
                                             RDF.GetLiteral("chrome://multiviews/content/feedpanel/feed-panel.xul"), true);
    panels.AppendElement(newPanelResource);
    mzFlushDataSource(panelDatasource);
  }
}

function mzDebug(txt)
{
  if (gDebug)
    dump("MultiZilla: " + txt + "\n");
}
