/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System;
using System.Collections.Generic;

namespace LocaleInpector
{

internal sealed class AccessKey
{

private AccessKey()
{
}

public static Status Check(string originalDir, string translatedDir)
{
    Output.OpenH2("Access Keys (underlined chars)");
    Status stat = new Status();


    List<string> fileList = BPIO.GetFileList(originalDir, "*");
    for (int idx = 0; idx < fileList.Count; idx++) {
        stat.Add(CompareKeyFile(originalDir, translatedDir, fileList[idx]));
    }

    if (stat.Empty) {
        Output.Clear();
    }

    Output.Close();
    return stat;
}

private static Status CompareKeyFile(string originalDir, string translatedDir, string file)
{
    Status stat = new Status();
    NameValueFile originalFile = NameValueFile.Load(originalDir + file);
    if (originalFile == null) {
        // unknown format (binary, html, ...)
        return stat;
    }

    Dictionary<string, string> originalKeys = originalFile.LoadHashtable(); // h:wtf
    stat.Files++;


    NameValueFile translatedFile = NameValueFile.LoadTranslated(translatedDir + file);
    Dictionary<string, string> translatedKeys;
    if (translatedFile == null) {
        // translated file is empty
        translatedKeys = new Dictionary<string, string>();
    } else {
        translatedKeys = translatedFile.LoadHashtable();
    }


    int maxPos = GetMaxPos();

    // loop em translatedFile: alguma no existe em originalFile?
    Dictionary<string, string>.Enumerator translatedKeyNames = translatedKeys.GetEnumerator();
    while (translatedKeyNames.MoveNext()) {

        string txt = translatedKeyNames.Current.Value;
        if (txt.IndexOf('&') > -1) {
            if (EntityTest.HasEntity(txt)) {
                continue;
            }
        }

        if (txt.IndexOf('%') > -1) {
            if (CStyleFormatTest.IsCFormat(txt)) {
                continue;
            }
        }

        string ak = GetAccesskeyKey(translatedKeys, translatedKeyNames.Current.Key);
        if (ak == null) {
            continue;
        }

        stat.Comparisons++;

        // A  bbbabc
        // A  bbbAbc
        string ak2 = translatedKeys[ak];
        int pos = txt.IndexOf(ak2);
        if (pos < 0) {
            int lowPos = txt.IndexOf(ak2.ToUpper());
            int uppPos = txt.IndexOf(ak2.ToLower());
            pos = System.Math.Max(lowPos, uppPos);
        }
        if (pos == -1) {
            Output.WriteLine("File:  " + file + " => " + translatedKeyNames.Current.Key + " + " + ak);
            Output.WriteLine("Error: " + "'" + txt + "' doesn't contain accesskey '" + ak2 + "'");
            Output.WriteLine(new string('-', 79));
            stat.Warnings++;
            continue;
        }

        // has maxposition?
        if (maxPos < 0) {
            continue;
        }
        if (pos > maxPos) {
            if (txt[pos - 1] == ' ') {
                continue;
            }
            Output.WriteLine("File:  " + file + " => " + translatedKeyNames.Current.Key + " + " + ak);
            Output.WriteLine("Error: " + "accesskey (" + ak2 + ") position is greater than maxposition (" + maxPos.ToString() + ").");
            Output.WriteLine("Text:  " + txt);
            Output.WriteLine(new string('-', 79));
            stat.Warnings++;
        }

    }

    return stat;
}

private static int GetMaxPos()
{
    string max = Config.Current.GetPref("accesskey", "maxposition");
    if (max.Length == 0) {
        return -1;
    }
    int maxPos;
    try {
        maxPos = int.Parse(max);
    } catch {
        return -1;
    }
    maxPos--;
    return (maxPos < 0) ? -1 : maxPos;
}


/*
<!ENTITY  changepassword.button   "Change Password...">
<!ENTITY  changepassword.accesskey  "C">
<!ENTITY  resetpassword.label    "Reset Master Password">
<!ENTITY  resetpassword.text     "If you reset your master password, all of your stored Web and e-mail passwords, form data, personal certificates, and private keys will be lost.">
<!ENTITY  resetpassword.button   "Reset Password">
<!ENTITY  resetpassword.accesskey   "R">



mismatchCheckNow=Check Now
mismatchCheckNowAccesskey=C
EnterLinkText=Enter text to display for the link:
EnterLinkTextAccessKey=T
cmd_bm_newbookmark = New Bookmark...
cmd_bm_newbookmark_accesskey = B
framesTab             "Frames">
framesTab.accesskey   "r">
showDetails=Show Details
hideDetails=Hide Details
detailsAccessKey=T

addAccountButton.label "Add Account...">
addAccountButton.accesskey "A">
livehttpheaders.pageinfo.tab		"Headers" >
livehttpheaders.pageinfo.accesskey	"h" >
cancelButton=Cancel
cancelAccess=C
webdeveloper.zoom.out.label "Diminuir">
webdeveloper.zoom.out.key "D">
generalTab            "General">
generalAccesskey      "G">
*/
public static string GetAccesskeyKey(Dictionary<string, string> allKeys, string labelKey)
{
    string accesskeyKey = GetAccesskeyKeyCore(allKeys, labelKey);
    if (accesskeyKey == null) {
        return null;
    }
    if (allKeys[accesskeyKey].Length > 1) {
        return null;
    }
    return accesskeyKey;
}

//#: messenger/messenger.dtd/previousUnreadMsgCmd.key+previousUnreadMsgCmd.accesskey

public static bool IsAccessKey(string key)
{
    string[] accesskeySufix = new string[] { ".accesskey", ".accessKey", "AccessKey", "Accesskey", "_accesskey", "_accessKey" };
    foreach (string accesskey in accesskeySufix) {
        if (key.EndsWith(accesskey)) {
            return true;
        }
    }
    return false;
}

public static string GetAccesskeyKeyCore(Dictionary<string, string> allKeys, string labelKey)
{
    string[] accesskeySufix = new string[] { ".accesskey", ".accessKey", "AccessKey", "Accesskey", "_accesskey", "_accessKey" };
    foreach (string accesskey in accesskeySufix) {
        string matchKey = labelKey + accesskey;
        if (allKeys.ContainsKey(matchKey)) {
            return matchKey;
        }
    }

    string key;
    key = TestAccessKey(allKeys, labelKey, ".label", new string[] { ".accesskey", ".accessKey" });
    if (key != null) {
        return key;
    }

    key = TestAccessKey(allKeys, labelKey, ".tab", new string[] { ".accesskey", ".accessKey" });
    if (key != null) {
        return key;
    }

    key = TestAccessKey(allKeys, labelKey, "Button", new string[] { "Access" });
    if (key != null) {
        return key;
    }

    key = TestAccessKey(allKeys, labelKey, "Tab", new string[] { "Accesskey" });
    if (key != null) {
        return key;
    }

    return null;
}


private static string TestAccessKey(Dictionary<string, string> allKeys, string labelKey, string labelSufix, string[] allowedAccesskeySufix)
{
    if (labelKey.EndsWith(labelSufix) == false) {
        return null;
    }
    int len = labelKey.Length - labelSufix.Length;
    string root = labelKey.Substring(0, len);
    foreach (string accesskey in allowedAccesskeySufix) {
        string matchKey = root + accesskey;
        if (allKeys.ContainsKey(matchKey)) {
            return matchKey;
        }
    }
    return null;
}


}//class
}//ns
