/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Text;

namespace LocaleInpector
{

internal sealed class Config
{

public static Config Current;
private Dictionary<string, string> m_contentIni;
private readonly string     m_filenameIni = null;
private readonly  DateTime m_startDate = DateTime.UtcNow;


public Config()
{
    Current = this;

    System.Threading.Thread t = System.Threading.Thread.CurrentThread;
    t.CurrentCulture = System.Globalization.CultureInfo.InvariantCulture;

    string[] arguments = Environment.GetCommandLineArgs();
    if (arguments.Length > 1) {
        m_filenameIni = arguments[1];
        m_filenameIni = System.IO.Path.GetFullPath(m_filenameIni);
    } else {
        m_filenameIni = null;
    }
}

public List<string> InitIni()
{
    if (m_filenameIni == null) {
        return null;
    }
    IniFile ini = new IniFile(m_filenameIni);
    m_contentIni = ini.LoadHashtable();
    if (ini.Exists == false) {
        return null;
    }

    this.SetIniDefault("files", "del.extra", "");
    this.SetIniDefault("general", "warnings", "true");

    // enable
    this.SetIniDefault("general", "tests", "true");
    this.SetIniDefault("directories", "enabled", "true");
    this.SetIniDefault("files", "enabled", "true");
    this.SetIniDefault("bom", "enabled", "true");
    this.SetIniDefault("phrases", "enabled", "true");
    this.SetIniDefault("keybinding", "enabled", "true");
    this.SetIniDefault("markup", "enabled", "true");
    this.SetIniDefault("c-format", "enabled", "true");
    this.SetIniDefault("entities", "enabled", "true");
    this.SetIniDefault("placeholders", "enabled", "true");
    this.SetIniDefault("accesskey", "enabled", "true");
    this.SetIniDefault("encoding", "enabled", "true");

    this.SetIniDefault("untranslated", "enabled", "false");

    // Find all sections [module XXX]
    List<string> sections = ini.LoadSections();
    for (int idx = sections.Count - 1; idx > -1; idx--) {
        if (sections[idx].StartsWith("module")) {
            this.SetIniDefault(sections[idx], "enabled", "true");
            if (this.GetPref(sections[idx], "enabled") == "true") {
                continue;
            }
        }
        sections.RemoveAt(idx);
    }


    return sections;
}

private void SetIniDefault(string section, string key, string value)
{
    string sk = section + "->" + key;
    if (m_contentIni.ContainsKey(sk) == false) {
        m_contentIni[sk] = value;
    }
}

public string GetPref(string section, string name)
{
    string key = section + "->" + name;
    if (m_contentIni.ContainsKey(key)) {
        return m_contentIni[key];
    } else {
        return string.Empty;
    }
}

public static string FixDirPath(string path)
{
    return System.IO.Path.GetFullPath(FixDirPathCore(path));
}

private static string FixDirPathCore(string path)
{
    if (path.Length == 0) {
        return System.IO.Path.DirectorySeparatorChar.ToString();
    }
    if (path[path.Length - 1] == System.IO.Path.DirectorySeparatorChar) {
        return path;
    }
    return path + System.IO.Path.DirectorySeparatorChar.ToString();
}

public string ValidateDir(string module, string type)
{
    string name = module + "->" + type;
    if (m_contentIni.ContainsKey(name) == false) {
        Output.Write(@"[ERROR] """ + type + @""" entry not declared in ini file.");
        return null;
    }
    string path = m_contentIni[name];
    if (path.Length == 0) {
        Output.Write(@"[ERROR] """ + type + @""" not declared in ini file.");
        return null;
    }
    path = FixDirPath(path);

    if (BPIO.IsDir(path) == false) {
        Output.Write("[NOT FOUND] " + path);
        return null;
    }

    Output.Write(path);
    return path;
}

public void Start()
{
    Output.WriteLine("Loading configuration from: " + m_filenameIni);
    Output.WriteLine(m_startDate.ToString("u") + " (UTC)");
}

public void End(Status stat, int mod)
{
    Output.OpenH1("Done. " +
                    ((stat.Errors > 0)
                        ? ("Failed, " + stat.Errors.ToString() + " errors")
                        : "No errors, " + mod.ToString() + " modules")
                    +  ". (" + (DateTime.UtcNow - m_startDate).TotalSeconds.ToString("n2") + "s)");
    Output.Close();
}


public static string GetVersion()
{
    System.Reflection.Assembly assembly = System.Reflection.Assembly.GetExecutingAssembly();
    System.Reflection.AssemblyName name = assembly.GetName();
    System.Version ver = name.Version;
    return ver.ToString(3);
}

}//class
}//ns