/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

namespace Hultmann.Zip
{

// Overall .ZIP file format:
//
//   [file 1]
//      [local file header]                     signature: 0x04034b50
//      [file data]
//      [data descriptor]                       crc/compressed/size (todo)
//   [file n]
//      [local file header]
//      [file data]
//      [data descriptor]
//
//   [archive decryption header]                (todo)
//   [archive extra data record]                (todo)
//      archive extra data signature            signature: 0x08064b50
//      extra field length
//      extra field data

//
//   [central directory]
//      [file header 1]                         signature: 0x02014b50
//      [file header n]
//      [digital signature]                     (todo)
//         header signature                     signature: 0x05054b50
//         size of data
//         signature data
//
//   [zip64 end of central directory record]    signature: 0x06064b50
//
//   [zip64 end of central directory locator]   signature: 0x07064b50
//
//   [end of central directory record]          signature: 0x06054b50


internal sealed class ZipEntryWriter
{

private readonly byte[] m_header;
private          int    m_offset;


public ZipEntryWriter(System.IO.Stream archive, ZipEntry infoNextEntry, System.IO.MemoryStream binNextEntry)
{
    switch (infoNextEntry.CompressionMethod) {
        case CompressionMethod.Deflated:
            m_header = this.WriteDeflated(archive, infoNextEntry, binNextEntry);
            break;

        default:
            m_header = this.WriteStored(archive, infoNextEntry, binNextEntry);
            break;
    }
}


private byte[] WriteDeflated(System.IO.Stream archive, ZipEntry infoNextEntry, System.IO.MemoryStream binNextEntry)
{
    System.IO.MemoryStream compressed = new System.IO.MemoryStream((int) binNextEntry.Length);
    System.IO.Compression.DeflateStream deflate
        = new System.IO.Compression.DeflateStream(compressed,
                                                  System.IO.Compression.CompressionMode.Compress,
                                                  true); // LeaveUnderlyingStreamOpen

    binNextEntry.Position = 0;

    const int LenBuffer = 512;
    byte[] buffer = new byte[LenBuffer];
    int count;


    MyCrc myCrc = new MyCrc();
    do {
        count = binNextEntry.Read(buffer, 0, LenBuffer);
        deflate.Write(buffer, 0, count);
        myCrc.Update(buffer, count);
    } while (count > 0);


    // finaliza deflate -- salva alguns bytes para compressed
    //deflate.Flush(); // doesn't work?
    deflate.Close();


    infoNextEntry.Crc            = myCrc.Value;
    infoNextEntry.Size           = binNextEntry.Length;
    infoNextEntry.CompressedSize = compressed.Length;


    if (infoNextEntry.CompressedSize >= infoNextEntry.Size) { // painless optimization
        compressed.Close();
        binNextEntry.Position = 0L;
        infoNextEntry.CompressionMethod = CompressionMethod.Stored;
        return this.WriteStored(archive, infoNextEntry, binNextEntry);
    }



    byte[] header = BuildLocalFileHeader(infoNextEntry);
    m_offset = (int) archive.Length;


    archive.Write(header, 0, header.Length);


    compressed.Position = 0;
    do {
        count = compressed.Read(buffer, 0, LenBuffer);
        archive.Write(buffer, 0, count);
    } while (count > 0);

    //deflate.Close();
    compressed.Close();

    return header;
}


private byte[] WriteStored(System.IO.Stream archive, ZipEntry infoNextEntry, System.IO.MemoryStream binNextEntry)
{
    binNextEntry.Position = 0L;

    const int LenBuffer = 512;
    byte[] buffer = new byte[LenBuffer];
    int count;


    MyCrc myCrc = new MyCrc();
    do {
        count = binNextEntry.Read(buffer, 0, LenBuffer);
        myCrc.Update(buffer, count);
    } while (count > 0);

    infoNextEntry.Crc            = myCrc.Value;
    infoNextEntry.Size           = binNextEntry.Length;
    infoNextEntry.CompressedSize = infoNextEntry.Size;


    byte[] header = BuildLocalFileHeader(infoNextEntry);
    m_offset = (int) archive.Length;


    archive.Write(header, 0, header.Length);


    binNextEntry.Position = 0L;
    do {
        count = binNextEntry.Read(buffer, 0, LenBuffer);
        archive.Write(buffer, 0, count);
    } while (count > 0);

    return header;
}



// A.  Local file header:
//
//       local file header signature     4 bytes  (0x04034b50)
//       version needed to extract       2 bytes
//       general purpose bit flag        2 bytes
//       compression method              2 bytes
//       last mod file time              2 bytes
//       last mod file date              2 bytes
//       crc-32                          4 bytes
//       compressed size                 4 bytes
//       uncompressed size               4 bytes
//       file name length                2 bytes
//       extra field length              2 bytes
//
//       file name (variable size)
//       extra field (variable size)
//
private static byte[] BuildLocalFileHeader(ZipEntry infoNextEntry)
{
    System.Text.Encoding enc = System.Text.Encoding.GetEncoding(850);
    byte[] binName = enc.GetBytes(infoNextEntry.Name);
    byte[] header = new byte[30 + binName.Length];


    // local file header signature     4 bytes
    header[0] = 0x50;
    header[1] = 0x4b;
    header[2] = 0x03;
    header[3] = 0x04;


    // version needed to extract       2 bytes
    switch (infoNextEntry.CompressionMethod) {
        case CompressionMethod.Deflated:
            header[4] = 20;
            break;
        default:
            header[4] = 10;
            break;
    }
    header[5] = 0x00;


    // general purpose bit flag        2 bytes
    header[6] = 0x00;
    header[7] = 0x00;

    // compression method              2 bytes
    switch (infoNextEntry.CompressionMethod) {
        case CompressionMethod.Deflated:
        case CompressionMethod.Stored:
            header[8] = (byte) infoNextEntry.CompressionMethod;
            break;
        default:
            header[8] = (byte) CompressionMethod.Stored;
            break;
    }
    header[9] = 0x00;


    // last mod file time              2 bytes
    header[10] = (byte)  (infoNextEntry.DosTime & 0x00ff);
    header[11] = (byte) ((infoNextEntry.DosTime & 0xff00) >> 8);


    // last mod file date              2 bytes
    header[12] = (byte) ((infoNextEntry.DosTime & 0x00ff0000) >> 16);
    header[13] = (byte) ((infoNextEntry.DosTime & 0xff000000) >> 24);


    // crc-32                          4 bytes
    uint crc = (uint) infoNextEntry.Crc;
    header[14] = (byte)  (crc & 0x000000ff);
    header[15] = (byte) ((crc & 0x0000ff00) >> 8);
    header[16] = (byte) ((crc & 0x00ff0000) >> 16);
    header[17] = (byte) ((crc & 0xff000000) >> 24);


    // compressed size                 4 bytes
    long compressed = infoNextEntry.CompressedSize;
    header[18] = (byte)  (compressed & 0x000000ff);
    header[19] = (byte) ((compressed & 0x0000ff00) >> 8);
    header[20] = (byte) ((compressed & 0x00ff0000) >> 16);
    header[21] = (byte) ((compressed & 0xff000000) >> 24);


    // uncompressed size               4 bytes
    long uncompressed = infoNextEntry.Size;
    header[22] = (byte)  (uncompressed & 0x000000ff);
    header[23] = (byte) ((uncompressed & 0x0000ff00) >> 8);
    header[24] = (byte) ((uncompressed & 0x00ff0000) >> 16);
    header[25] = (byte) ((uncompressed & 0xff000000) >> 24);


    // file name length                2 bytes
    int length = infoNextEntry.Name.Length;
    header[26] = (byte)  (length & 0x00ff);
    header[27] = (byte) ((length & 0xff00) >> 8);


    // extra field length              2 bytes
    header[28] = 0x00;
    header[29] = 0x00;


    // file name (variable size)
    System.Array.Copy(binName, 0, header, 30, binName.Length);

    return header;
}


// F.  Central directory structure:
//
//     [file header 1]
//     .
//     .
//     .
//     [file header n]
//     [digital signature]
//
//     File header:
//
//       central file header signature   4 bytes  (0x02014b50)
//       version made by                 2 bytes
//       version needed to extract       2 bytes
//       general purpose bit flag        2 bytes
//       compression method              2 bytes
//       last mod file time              2 bytes
//       last mod file date              2 bytes
//       crc-32                          4 bytes
//       compressed size                 4 bytes
//       uncompressed size               4 bytes
//       file name length                2 bytes
//       extra field length              2 bytes
//       file comment length             2 bytes
//       disk number start               2 bytes
//       internal file attributes        2 bytes
//       external file attributes        4 bytes
//       relative offset of local header 4 bytes
//
//       file name (variable size)
//       extra field (variable size)
//       file comment (variable size)
//
public void WriteCentralDirectoryStructureFileHeader(System.IO.Stream archive)
{
    int lenName = System.BitConverter.ToInt16(m_header, 26);
    byte[] header = new byte[46 + lenName];

    // central file header signature   4 bytes  (0x02014b50)
    header[0] = 0x50;
    header[1] = 0x4b;
    header[2] = 0x01;
    header[3] = 0x02;

    // version made by                 2 bytes
    header[4] = 0x00; // 0 - MS-DOS and OS/2 (FAT / VFAT / FAT32 file systems)
    header[5] = 0x00;

    // version needed to extract       2 bytes
    // general purpose bit flag        2 bytes
    // compression method              2 bytes
    // last mod file time              2 bytes
    // last mod file date              2 bytes
    // crc-32                          4 bytes
    // compressed size                 4 bytes
    // uncompressed size               4 bytes
    // file name length                2 bytes
    // extra field length              2 bytes
    System.Array.Copy(m_header, 4, header, 6, 26);

    // file comment length             2 bytes
    header[32] = 0x00;
    header[33] = 0x00;

    // disk number start               2 bytes
    header[34] = 0x00;
    header[35] = 0x00;

    // internal file attributes        2 bytes
    header[36] = 0x00;
    header[37] = 0x00;

    // external file attributes        4 bytes
    header[38] = 0x20;
    header[39] = 0x00;
    header[40] = 0xb6;
    header[41] = 0x81;

    // relative offset of local header 4 bytes
    header[42] = (byte)  (m_offset & 0x000000ff);
    header[43] = (byte) ((m_offset & 0x0000ff00) >> 8);
    header[44] = (byte) ((m_offset & 0x00ff0000) >> 16);
    header[45] = (byte) ((m_offset & 0xff000000) >> 24);

    // file name (variable size)
    System.Array.Copy(m_header, 30, header, 46, lenName);

    archive.Write(header, 0, header.Length);
}


// ==================================================================================================================================
}//class
}//ns
