/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

namespace Hultmann.Zip
{

internal sealed class ZipEntry
{

private const long Mask4bits = 0x0f;
private const long Mask5bits = 0x1f;
private const long Mask6bits = 0x3f;
private const long Mask7bits = 0x7f;
private const long Mask32bits = 0xffffffff;

private readonly string            m_name;
//private readonly int               m_madeBy;
private          long              m_dosTime;
private          long              m_size;
private          CompressionMethod m_method = CompressionMethod.Deflated;
private readonly int               m_version = 20;
private          long              m_crc;
private          long              m_compressed;
private          int               m_offset;
private          int               m_flags = 0;
private          byte[]            m_extra;
private          int               m_index = -1;


public ZipEntry(string name, int version)
{
    m_name = CleanName(name);
    m_version = version;
    this.DateTime = System.DateTime.Now;
}

public ZipEntry(string name)
{
    m_name = CleanName(name);
    this.DateTime = System.DateTime.Now;
}


public ZipEntry Clone()
{
    return this.MemberwiseClone() as ZipEntry;
}


// c:\dir\x.txt ==> dir/x.txt
// /home/user/x.txt ==> home/user/x.txt
// \\server\a\b\c\x.txt ==> "b/c/x.txt"
public static string CleanName(string name)
{
    string root = System.IO.Path.GetPathRoot(name); // "", "x:\", "\", "x:", "\\server\a"
    if (root.Length > 0) {
        name = name.Substring(root.Length);
    }
    return name.Replace('\\', '/');
}


public string Name
{
    get {
        return m_name;
    }
}


public bool IsDirectory
{
    get {
        int len = m_name.Length;
        return len > 0 && m_name[len - 1] == '/';
    }
}


// version needed to extract
public int Version
{
    get {
        return m_version;
    }
}

/*
public int HostSystem
{
    get {
        return m_madeBy;
    }
}
*/

public int ZipFileIndex
{
    get {
        return m_index;
    }
    set {
        m_index = value;
    }
}


public long DosTime
{
    get {
        return m_dosTime;
    }
    set {
        m_dosTime = value & Mask32bits;
    }
}


public System.DateTime DateTime
{
	get {
        long dosTime = m_dosTime;

        int sec = (int) ((dosTime & Mask5bits) << 1);
        if (sec > 59) {
            sec = 59;
        }

        dosTime >>= 5;
        int min = (int) (dosTime & Mask6bits);
        if (min > 59) {
            min = 59;
        }

        dosTime >>= 6;
        int h = (int) (dosTime & Mask5bits);
        if (h > 23) {
            h = 23;
        }

        dosTime >>= 5;
        int dd = (int) (dosTime & Mask5bits);
        if (dd == 0) {
            dd = 1;
        }

        dosTime >>= 5;
        int mm = (int) (dosTime & Mask4bits);
        if (mm > 12) {
            mm = 12;
        } else if (mm == 0) {
            mm = 1;
        }

        dosTime >>= 4;
        int yy = 1980 + (int) (dosTime & Mask7bits);

        return new System.DateTime(yy, mm, dd, h, min, sec);
	}
	set {
		m_dosTime = (value.Year - 1980) & Mask7bits;

		m_dosTime <<= 4;
		m_dosTime |= (byte) value.Month;

		m_dosTime <<= 5;
		m_dosTime |= (byte) value.Day;

		m_dosTime <<= 5;
		m_dosTime |= (byte) value.Hour;

		m_dosTime <<= 6;
		m_dosTime |= (byte) value.Minute;

		m_dosTime <<= 5;
		m_dosTime |= (byte) (value.Second >> 1);
	}
}



public CompressionMethod CompressionMethod
{
    get {
        return m_method;
    }
    set {
        switch (value) {
            case CompressionMethod.Deflated:
            case CompressionMethod.Stored:
                m_method = value;
                break;
            default:
                m_method = CompressionMethod.Unsupported;
                break;
        }
    }
}


public long CompressedSize
{
    get {
        return m_compressed;
    }
    set {
        m_compressed = value;
    }
}


public long Size
{
    get {
        return m_size;
    }
    set {
        m_size = value;
    }
}


public long Crc
{
    set {
        m_crc = value;
    }
    get {
        return m_crc;
    }
}


// probably this value is wrong
public int Offset
{
    get {
        return m_offset;
    }
    set {
        m_offset = value;
    }
}


// general purpose bit flag: (2 bytes)
//
//     Bit 0: If set, indicates that the file is encrypted.
//
//     (For Method 6 - Imploding)
//     Bit 1: If the compression method used was type 6,
//            Imploding, then this bit, if set, indicates
//            an 8K sliding dictionary was used.  If clear,
//            then a 4K sliding dictionary was used.
//     Bit 2: If the compression method used was type 6,
//            Imploding, then this bit, if set, indicates
//            3 Shannon-Fano trees were used to encode the
//            sliding dictionary output.  If clear, then 2
//            Shannon-Fano trees were used.
//
//     (For Methods 8 and 9 - Deflating)
//     Bit 2  Bit 1
//       0      0    Normal (-en) compression option was used.
//       0      1    Maximum (-exx/-ex) compression option was used.
//       1      0    Fast (-ef) compression option was used.
//       1      1    Super Fast (-es) compression option was used.
//
//     Note:  Bits 1 and 2 are undefined if the compression
//            method is any other.
//
//     Bit 3: If this bit is set, the fields crc-32, compressed
//            size and uncompressed size are set to zero in the
//            local header.  The correct values are put in the
//            data descriptor immediately following the compressed
//            data.  (Note: PKZIP version 2.04g for DOS only
//            recognizes this bit for method 8 compression, newer
//            versions of PKZIP recognize this bit for any
//            compression method.)
//
//     Bit 4: Reserved for use with method 8, for enhanced
//            deflating.
//
//     Bit 5: If this bit is set, this indicates that the file is
//            compressed patched data.  (Note: Requires PKZIP
//            version 2.70 or greater)
//
//     Bit 6: Strong encryption.  If this bit is set, you should
//            set the version needed to extract value to at least
//            50 and you must also set bit 0.  If AES encryption
//            is used, the version needed to extract value must
//            be at least 51.
//
//     Bit 7: Currently unused.
//     Bit 8: Currently unused.
//     Bit 9: Currently unused.
//     Bit 10: Currently unused.
//     Bit 11: Currently unused.
//
//     Bit 12: Reserved by PKWARE for enhanced compression.
//
//     Bit 13: Used when encrypting the Central Directory to indicate
//             selected data values in the Local Header are masked to
//             hide their actual values.  See the section describing
//             the Strong Encryption Specification for details.
//
//     Bit 14: Reserved by PKWARE.
//     Bit 15: Reserved by PKWARE.
//
public int Flags
{
    get {
        return m_flags;
    }
    set {
        m_flags = value;
    }
}



public byte[] ExtraData
{
    get {
        return m_extra;
    }
    set {
        m_extra = value;
    }
}


// ==================================================================================================================================
}//class
}//ns
