/* GNU Proxyknife -- customizable proxy hunter
 * Copyright (C) 2005-2006 Jia Wang <skyroam@gmail.com>
 *
 *
 *
 * This file is part of GNU Proxyknife.
 * GNU Proxyknife is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the 
 * Free Software Foundation; either version 2 of the License, or (at your 
 * option) any later version.
 *
 * GNU Proxyknife is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 * General Public License for more details.
 *
 *
 * You should have received a copy of the GNU General Public License 
 * along with Proxyknife; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* all functions are declared in proxyknife.h */
#include <proxyknife.h>
/* for open */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* for dup2 */
#include <unistd.h>

/* for perror */
#include <errno.h>

/* strlen */
#include <string.h>

#include <signal.h>

/* Global mutex */
pthread_mutex_t counter_mutex_in = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t counter_mutex_out = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t counter_mutex_create = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t counter_mutex_malloc = PTHREAD_MUTEX_INITIALIZER;

/* these functions are defined in this file */
void INIT (int argc, char **argv);
void THREAD (void);
void END (void);

/* Set the limit on number of  threads */
enum
{ MAXTHREADS = 65535 };

int
main (int argc, char **argv)
{
  progname = argv[0];
  setlocale (LC_CTYPE, ""); 
  setlocale (LC_MESSAGES, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  INIT (argc, argv);
  THREAD ();
  END ();
  /* clean memory */
  xexit (EXIT_SUCCESS);
}

/* Read configuration file ,init global vars.
 * Open list file and output file.
 * Close buffering of STDIN ,STDOUT and STDERR */
void
INIT (int argc, char **argv)
{
  struct hostent *h;
  size_t len;

  FILE *reqfp;

  /* init , used by END() */
  logfd = -1;

  /* debug level */
  debug = 0;

  /* input mode */
  proxyknife_in_type = IN_FILE;

  mem_init (); /* target.checkreqin = NULL */
  READCONF (argc, argv);	/* Target (checksite) was parsed in this function. */


  /* check httptestmethod */
  if (test.httptestmethod == HTTP_GET)
    {
      /* Build GET request sent to testproxy(if it is HTTP proxy) */
      build_req_get_via_http_proxy ();
    }
  else if (test.httptestmethod == HTTP_CONNECT)
    {
      build_con_via_http_proxy ();
    }
  else
    {
      fprintf (stderr, "%s %s: INIT: "
	       "HTTP Test Method is not supported now: %d!\n",
	       progname, __FILE__, test.httptestmethod);
      xexit (EXIT_FAILURE);
    }
  /* check socks5testmethod */
  if (test.socks5testmethod != SOCKS5_CONNECT)
    {
      fprintf (stderr,
	       "%s: INIT:Socks5 Test Method %d is not supported now\n",
	       progname, test.socks5testmethod);
      xexit (EXIT_FAILURE);
    }
  /* check socks4testmethod */
  if (test.socks4testmethod != SOCKS4_CONNECT)
    {
      fprintf (stderr,
	       "%s: INIT:Socks4 Test Method %d is not supported now\n",
	       progname, test.socks4testmethod);
      xexit (EXIT_FAILURE);
    }

  /* Read customized request. */
  if (target.checkreqin != NULL)
    {
      /* read data to target.req,NULL or more than 1 chars. */
      target.req = read_binary_file (target.checkreqin, &len);
      if (target.req == NULL)
	{
	  perror (target.checkreqin);
	  if (debug)
	    fprintf (stderr, __FILE__ ":Do not use %s!\n", target.checkreqin);
	  xfree ((void **) &target.checkreqin);
	  xexit (EXIT_FAILURE);
	}
      else if (len == 0)
	{			/* NULL file */
	  if (debug)
	    fprintf (stderr, __FILE__ ":%s is null!\n", target.checkreqin);
	  xfree ((void **) &target.req);	/* target.req == NULL */
	  xfree ((void **) &target.checkreqin);
	  xexit (EXIT_FAILURE);
	}
      else
	{
	  /* Implement debug level in the future. 
	     May binary, so output this only for high level debuging. */
	  if (debug)
	    printf ("Using request from %s:%s\n",
		    target.checkreqin, target.req);
	}


    }
  else
    {
            /* Notice, if target.protocol is not HTTP, 
               we will still send http request. 
               Implement other protocol if needed. Prefer protocl.req. 
               */
      http_build_get ();	/* Otherwise,build common HTTP GET request. */
    }

  /* myproxy's authentication data */
  if (my.mytype == HTTP_CONNECT_AUTH)
    {
      http_build_auth ();
    }
  else if (my.mytype == SOCKS5_CONNECT_AUTH)
    {
      s5_build_auth ();
    }

  /* myproxy's ip */
  if (my.mytype != DIRECT)
    {
      h = gethostbyname ((const char *) my.myhost);
      if (h == NULL)
	{
	  herror (my.myhost);
	  xfree ((void *) &(my.myhost));
	  xexit (EXIT_FAILURE);
	}
      else
	{
	  my.myip = *(struct in_addr *) (h->h_addr);
	  xfree ((void *) &(my.myhost));
	}
/* debug */
      /* printf ("addr=%s\n", inet_ntoa (my.myip)); */
      my.myaddr.sin_family = AF_INET;
      my.myaddr.sin_addr = my.myip;
      my.myaddr.sin_port = htons (my.myport);
      memset (&(my.myaddr.sin_zero), 0, sizeof (my.myaddr.sin_zero));

    }
  if (proxyknife_in_type == IN_FILE)
    {
      if (!strcmp (proxyknife_in, "-"))
	{
	  in = stdin;
	}
      else
	{
	  in = fopen (proxyknife_in, "r");
	  if (in == NULL)
	    {
	      fprintf (stderr, __FILE__ " INIT:open %s ", proxyknife_in);
	      perror ("fopen");
	      xfree ((void **) &proxyknife_in);
	      xexit (EXIT_FAILURE);
	    }
	  else
	    xfree ((void **) &proxyknife_in);
	}
    }
  else if (proxyknife_in_type == IN_HTTP)
    {
      /* store all proxylist into proxyknife_in_buffer */
      fetchproxylist (proxyknife_in);
      proxyknife_in_buffer_cur = proxyknife_in_buffer;
    }
  else
    {
      fprintf (stderr, __FILE__ " INIT:proxyknife_in_type unknown: %d !\n",
	       proxyknife_in_type);
      xfree ((void **) &proxyknife_in);
      xexit (EXIT_FAILURE);
    }

  out = fopen (proxyknife_out, "w");
  if (out == NULL)
    {
      fprintf (stderr, __FILE__ " INIT:open %s ", proxyknife_out);
      perror ("fopen");
      xfree ((void **) &proxyknife_out);
      xexit (EXIT_FAILURE);
    }
  else
    xfree ((void **) &proxyknife_out);

  /* Make log possilbe after INIT */
  setvbuf (out, (char *) NULL, _IONBF, 0);
  setvbuf (stdout, (char *) NULL, _IONBF, 0);
  setvbuf (stderr, (char *) NULL, _IONBF, 0);

  if (logfilename != NULL)
    {
      len = strlen (LOGSTDERR) + 1;	/* including trailing '\0' */
      if (strncmp (logfilename, LOGSTDERR, len) != 0)
	{
	  logfd = open (logfilename, O_CREAT | O_WRONLY, S_IRUSR | S_IWUSR);
	  if (logfd == -1)
	    {
	      perror ("INIT:open");
	      xexit (EXIT_FAILURE);
	    }
	  else
	    {
	      if (debug)
		{
		  printf ("Dup stderr to log file %d (%s)now!\n", logfd,
			  logfilename);
		}
	      /* close(2) if necessary) */
	      if (dup2 (logfd, 2) == -1)
		{
		  perror ("INIT:dup2");
		  xexit (EXIT_FAILURE);
		}
	    }
	}
    }
}

/* Create thread loop and wait for thread termination.
 * */
void
THREAD ()
{
  int tidnum[MAXTHREADS], mainret, i;
  pthread_attr_t attr;
  pthread_t tid[MAXTHREADS];
  void *status;

  if ((threads > MAXTHREADS) || (threads < 1))
    {
      fprintf (stderr, "%s: THREAD:Threads between [1,%d] is permitted."
	       "Please modify \"thread=\" in .conf" "then  try again\n",
	       progname, MAXTHREADS);
      xexit (EXIT_FAILURE);
    }

  /* Ignore SIGPIPE. */
  /* Write() will return with EPIPE error when socket is closed. */
  if (SIG_ERR == signal (SIGPIPE, SIG_IGN))
    {
      fprintf (stderr, "Signal redefine failed\n");
    }


  /* Initialize and set thread detached attribute */
  pthread_attr_init (&attr);
  /* If a thread requires joining, consider explicitly creating it as i
     joinable.  This provides portability as not all implementations 
     may create threads as joinable by default. */
  pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_JOINABLE);

  for (i = 0; i < threads; i++)
    {
      tidnum[i] = i;
      if ((mainret = pthread_create (tid + i, &attr, loop, tidnum + i)) == 0)
	{
	  if(debug)printf("%s %s: THREAD:thread %d create %d\n",
                      progname,__FILE__,i,tid[i]); 
	}
      else
	{
	  perror ("THREAD:pthread_create");
      break;
      /* After return, clean memory and exit. Do not exit here. */
	  /* xexit (EXIT_FAILURE); */
	}
    }

  /* Free attribute and wait for the other threads */
  pthread_attr_destroy (&attr);
  for (i = 0; i < threads; i++)
    {
      if (pthread_join (tid[i], &status) == 0)
	{
	  if (debug)
	    printf ("join thread %d\n", i, *(int *) status);
	}
      else
	{
	  perror ("THREAD:pthread_join");
      break;
	  /* xexit (EXIT_FAILURE);*//* After return , clean memory 
      and exit. Do not exit here.  */
	}
    }
  pthread_mutex_destroy (&counter_mutex_in);
  pthread_mutex_destroy (&counter_mutex_out);
  pthread_mutex_destroy (&counter_mutex_create);
  pthread_mutex_destroy (&counter_mutex_malloc);
  /* Deallocate the mutexes */
}

/* Do some cleaning: close file opened. */
void
END ()
{
  if (in != NULL)
    fclose (in);
  fclose (out);
  if (logfd != -1)
    close (logfd);
}
