/*-------------------------------------------------------------------------*/
/* GNU Prolog                                                              */
/*                                                                         */
/* Part  : line-edit library                                               */
/* File  : stty.c                                                          */
/* Descr.: TTY features                                                    */
/* Author: Daniel Diaz                                                     */
/*                                                                         */
/* Copyright (C) 1999 Daniel Diaz                                          */
/*                                                                         */
/* GNU Prolog is free software; you can redistribute it and/or modify it   */
/* under the terms of the GNU General Public License as published by the   */
/* Free Software Foundation; either version 2, or any later version.       */
/*                                                                         */
/* GNU Prolog is distributed in the hope that it will be useful, but       */
/* WITHOUT ANY WARRANTY; without even the implied warranty of              */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        */
/* General Public License for more details.                                */
/*                                                                         */
/* You should have received a copy of the GNU General Public License along */
/* with this program; if not, write to the Free Software Foundation, Inc.  */
/* 59 Temple Place - Suite 330, Boston, MA 02111, USA.                     */
/*-------------------------------------------------------------------------*/
#include <stdio.h>
#include <signal.h>

#if defined(__unix__) || defined(__CYGWIN__)

#include <termio.h>
#include <unistd.h>

#if defined(sun) && defined(sparc)

#include <sys/filio.h>
int ioctl(int d, int request, ...);

#else

#include <sys/ioctl.h>

#endif

#else

#include <conio.h>

#endif


#include "char_io.h"




/*---------------------------------*/
/* Constants                       */
/*---------------------------------*/

/*---------------------------------*/
/* Type Definitions                */
/*---------------------------------*/

/*---------------------------------*/
/* Global Variables                */
/*---------------------------------*/

#if defined(__unix__) || defined(__CYGWIN__)

static int           is_tty_in;
static int           is_tty_out;
static struct termio old_stty_in;
static struct termio new_stty_in;
static struct termio old_stty_out;
static struct termio new_stty_out;

#endif

static int    interrupt_key;
static int    quit_key;




/*---------------------------------*/
/* Function Prototypes             */
/*---------------------------------*/


#if defined(__unix__) || defined(__CYGWIN__)

static
void      Set_TTY_Mode          (struct termio *old,struct termio *new);

#define Gtty(fd,s) ioctl(fd,TCGETA,s)
#define Stty(fd,s) ioctl(fd,TCSETA,s)

#endif



/*-------------------------------------------------------------------------*/
/* LE_INIT                                                                 */
/*                                                                         */
/* Mode cbreak (raw mode).                                                 */
/*-------------------------------------------------------------------------*/
void LE_Init(int fd_in,int fd_out)

{
#if defined(__unix__) || defined(__CYGWIN__)
 is_tty_in= !Gtty(fd_in,&old_stty_in);
 is_tty_out=!Gtty(fd_out,&old_stty_out);

 if (is_tty_in)
    {
     interrupt_key=old_stty_in.c_cc[VINTR];
     quit_key     =old_stty_in.c_cc[VQUIT];
     Set_TTY_Mode(&old_stty_in,&new_stty_in);
     Stty(fd_in,&new_stty_in);
    }
  else
    {
     interrupt_key=KEY_CTRL('C');
     quit_key=KEY_CTRL('\\');
    }

 if (is_tty_out)
    {
     Set_TTY_Mode(&old_stty_out,&new_stty_out);
     Stty(fd_out,&new_stty_out);
    }
#else
 
 interrupt_key=KEY_CTRL('C');                 /* msdos: interrupt = CTRL+C */
 quit_key= -2;                                /*        quit      = nothing*/

#endif

 fflush(stdout);
 fflush(stderr);
}




/*-------------------------------------------------------------------------*/
/* LE_TERM                                                                 */
/*                                                                         */
/* Restore initial mode (cooked mode).                                     */
/*-------------------------------------------------------------------------*/
void LE_Term(int fd_in,int fd_out)

{
#if defined(__unix__) || defined(__CYGWIN__)

 if (is_tty_in)
     Stty(fd_in,&old_stty_in);

 if (is_tty_out)
     Stty(fd_out,&old_stty_out);

#endif
}




/*-------------------------------------------------------------------------*/
/* LE_SCREEN_SIZE                                                          */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void LE_Screen_Size(int fd_out,int *row,int *col)

{
#if defined(__unix__) || defined(__CYGWIN__)
 struct winsize ws;

 if (!is_tty_out)
    {
     row=col=0;
     return;
    }


 ioctl(fd_out,TIOCGWINSZ,&ws);
 *row=ws.ws_row;
 *col=ws.ws_col;

#else

 struct text_info ti;

 gettextinfo(&ti);
 *row=ti.screenheight;
 *col=ti.screenwidth;

#endif
}




/*-------------------------------------------------------------------------*/
/* LE_WHICH_SIGNAL                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
int LE_Which_Signal(int c)

{
 if (c==interrupt_key)
     return SIGINT;

 if (c==quit_key)
     return SIGQUIT;

 return 0;
}




/*-------------------------------------------------------------------------*/
/* LE_INPUT_IS_NOT_EMPTY                                                   */
/*                                                                         */
/*-------------------------------------------------------------------------*/
int LE_Input_Is_Not_Empty(int fd_in)

{
#if defined(__unix__) /* || defined(__CYGWIN__) */
 int nb_not_read;
 ioctl(0,FIONREAD,&nb_not_read);
 return nb_not_read!=0;
#else
#ifndef __CYGWIN__
 printf("LE_Is_Input_Empty not implemented\n");
#endif
 return 0;
#endif
}




#if defined(__unix__) || defined(__CYGWIN__)

/*-------------------------------------------------------------------------*/
/* SET_TTY_MODE                                                            */
/*                                                                         */
/* Mode cbreak (raw mode).                                                 */
/*-------------------------------------------------------------------------*/
static void Set_TTY_Mode(struct termio *old,struct termio *new)

{
 *new = *old;

 new->c_iflag &= ~(INLCR|IGNCR|ICRNL|IXON|IXOFF);
 new->c_oflag =  OPOST | ONLCR;
 new->c_lflag &= ~(ICANON|ECHO|ECHONL);

 new->c_cc[VMIN]=1;                                /* MIN # of chars */
 new->c_cc[VTIME]=1;                                         /* TIME */

 new->c_cc[VINTR]=-1;                  /* desactivate SIGINT  signal */
 new->c_cc[VQUIT]=-1;                  /* desactivate SIGQUIT signal */
}

#endif
