/* Mapping tables for GBK handling.
   Copyright (C) 1999, 2000, 2001 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Sean Chen <sean.chen@turbolinux.com>, 1999.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.  */

#include <dlfcn.h>
#include <gconv.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <wchar.h>
#include <assert.h>

/* Unicode 3.0.1 does not contain all the characters in GBK.  Define
   USE_PRIVATE_AREA to 1 in order to use mappings from/to the Unicode
   Private Use area.  Until we see other systems using the same mappings,
   it is disabled.  */
#define USE_PRIVATE_AREA 0

/* The conversion table to UCS4 has almost no holes.  It can be generated with:

   perl tab.pl < gbk.txt

   where tab.pl is:
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($gbk, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int(($g - 0x8140) / 256) * 192 + (($g - 0x8140) & 0xff), $u);
   }
   printf ("\n");

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 */
static const uint16_t __gbk_to_ucs[] =
{
  [0x0000] = 0x4e02, [0x0001] = 0x4e04, [0x0002] = 0x4e05, [0x0003] = 0x4e06,
  [0x0004] = 0x4e0f, [0x0005] = 0x4e12, [0x0006] = 0x4e17, [0x0007] = 0x4e1f,
  [0x0008] = 0x4e20, [0x0009] = 0x4e21, [0x000a] = 0x4e23, [0x000b] = 0x4e26,
  [0x000c] = 0x4e29, [0x000d] = 0x4e2e, [0x000e] = 0x4e2f, [0x000f] = 0x4e31,
  [0x0010] = 0x4e33, [0x0011] = 0x4e35, [0x0012] = 0x4e37, [0x0013] = 0x4e3c,
  [0x0014] = 0x4e40, [0x0015] = 0x4e41, [0x0016] = 0x4e42, [0x0017] = 0x4e44,
  [0x0018] = 0x4e46, [0x0019] = 0x4e4a, [0x001a] = 0x4e51, [0x001b] = 0x4e55,
  [0x001c] = 0x4e57, [0x001d] = 0x4e5a, [0x001e] = 0x4e5b, [0x001f] = 0x4e62,
  [0x0020] = 0x4e63, [0x0021] = 0x4e64, [0x0022] = 0x4e65, [0x0023] = 0x4e67,
  [0x0024] = 0x4e68, [0x0025] = 0x4e6a, [0x0026] = 0x4e6b, [0x0027] = 0x4e6c,
  [0x0028] = 0x4e6d, [0x0029] = 0x4e6e, [0x002a] = 0x4e6f, [0x002b] = 0x4e72,
  [0x002c] = 0x4e74, [0x002d] = 0x4e75, [0x002e] = 0x4e76, [0x002f] = 0x4e77,
  [0x0030] = 0x4e78, [0x0031] = 0x4e79, [0x0032] = 0x4e7a, [0x0033] = 0x4e7b,
  [0x0034] = 0x4e7c, [0x0035] = 0x4e7d, [0x0036] = 0x4e7f, [0x0037] = 0x4e80,
  [0x0038] = 0x4e81, [0x0039] = 0x4e82, [0x003a] = 0x4e83, [0x003b] = 0x4e84,
  [0x003c] = 0x4e85, [0x003d] = 0x4e87, [0x003e] = 0x4e8a, [0x0040] = 0x4e90,
  [0x0041] = 0x4e96, [0x0042] = 0x4e97, [0x0043] = 0x4e99, [0x0044] = 0x4e9c,
  [0x0045] = 0x4e9d, [0x0046] = 0x4e9e, [0x0047] = 0x4ea3, [0x0048] = 0x4eaa,
  [0x0049] = 0x4eaf, [0x004a] = 0x4eb0, [0x004b] = 0x4eb1, [0x004c] = 0x4eb4,
  [0x004d] = 0x4eb6, [0x004e] = 0x4eb7, [0x004f] = 0x4eb8, [0x0050] = 0x4eb9,
  [0x0051] = 0x4ebc, [0x0052] = 0x4ebd, [0x0053] = 0x4ebe, [0x0054] = 0x4ec8,
  [0x0055] = 0x4ecc, [0x0056] = 0x4ecf, [0x0057] = 0x4ed0, [0x0058] = 0x4ed2,
  [0x0059] = 0x4eda, [0x005a] = 0x4edb, [0x005b] = 0x4edc, [0x005c] = 0x4ee0,
  [0x005d] = 0x4ee2, [0x005e] = 0x4ee6, [0x005f] = 0x4ee7, [0x0060] = 0x4ee9,
  [0x0061] = 0x4eed, [0x0062] = 0x4eee, [0x0063] = 0x4eef, [0x0064] = 0x4ef1,
  [0x0065] = 0x4ef4, [0x0066] = 0x4ef8, [0x0067] = 0x4ef9, [0x0068] = 0x4efa,
  [0x0069] = 0x4efc, [0x006a] = 0x4efe, [0x006b] = 0x4f00, [0x006c] = 0x4f02,
  [0x006d] = 0x4f03, [0x006e] = 0x4f04, [0x006f] = 0x4f05, [0x0070] = 0x4f06,
  [0x0071] = 0x4f07, [0x0072] = 0x4f08, [0x0073] = 0x4f0b, [0x0074] = 0x4f0c,
  [0x0075] = 0x4f12, [0x0076] = 0x4f13, [0x0077] = 0x4f14, [0x0078] = 0x4f15,
  [0x0079] = 0x4f16, [0x007a] = 0x4f1c, [0x007b] = 0x4f1d, [0x007c] = 0x4f21,
  [0x007d] = 0x4f23, [0x007e] = 0x4f28, [0x007f] = 0x4f29, [0x0080] = 0x4f2c,
  [0x0081] = 0x4f2d, [0x0082] = 0x4f2e, [0x0083] = 0x4f31, [0x0084] = 0x4f33,
  [0x0085] = 0x4f35, [0x0086] = 0x4f37, [0x0087] = 0x4f39, [0x0088] = 0x4f3b,
  [0x0089] = 0x4f3e, [0x008a] = 0x4f3f, [0x008b] = 0x4f40, [0x008c] = 0x4f41,
  [0x008d] = 0x4f42, [0x008e] = 0x4f44, [0x008f] = 0x4f45, [0x0090] = 0x4f47,
  [0x0091] = 0x4f48, [0x0092] = 0x4f49, [0x0093] = 0x4f4a, [0x0094] = 0x4f4b,
  [0x0095] = 0x4f4c, [0x0096] = 0x4f52, [0x0097] = 0x4f54, [0x0098] = 0x4f56,
  [0x0099] = 0x4f61, [0x009a] = 0x4f62, [0x009b] = 0x4f66, [0x009c] = 0x4f68,
  [0x009d] = 0x4f6a, [0x009e] = 0x4f6b, [0x009f] = 0x4f6d, [0x00a0] = 0x4f6e,
  [0x00a1] = 0x4f71, [0x00a2] = 0x4f72, [0x00a3] = 0x4f75, [0x00a4] = 0x4f77,
  [0x00a5] = 0x4f78, [0x00a6] = 0x4f79, [0x00a7] = 0x4f7a, [0x00a8] = 0x4f7d,
  [0x00a9] = 0x4f80, [0x00aa] = 0x4f81, [0x00ab] = 0x4f82, [0x00ac] = 0x4f85,
  [0x00ad] = 0x4f86, [0x00ae] = 0x4f87, [0x00af] = 0x4f8a, [0x00b0] = 0x4f8c,
  [0x00b1] = 0x4f8e, [0x00b2] = 0x4f90, [0x00b3] = 0x4f92, [0x00b4] = 0x4f93,
  [0x00b5] = 0x4f95, [0x00b6] = 0x4f96, [0x00b7] = 0x4f98, [0x00b8] = 0x4f99,
  [0x00b9] = 0x4f9a, [0x00ba] = 0x4f9c, [0x00bb] = 0x4f9e, [0x00bc] = 0x4f9f,
  [0x00bd] = 0x4fa1, [0x00be] = 0x4fa2, [0x00c0] = 0x4fa4, [0x00c1] = 0x4fab,
  [0x00c2] = 0x4fad, [0x00c3] = 0x4fb0, [0x00c4] = 0x4fb1, [0x00c5] = 0x4fb2,
  [0x00c6] = 0x4fb3, [0x00c7] = 0x4fb4, [0x00c8] = 0x4fb6, [0x00c9] = 0x4fb7,
  [0x00ca] = 0x4fb8, [0x00cb] = 0x4fb9, [0x00cc] = 0x4fba, [0x00cd] = 0x4fbb,
  [0x00ce] = 0x4fbc, [0x00cf] = 0x4fbd, [0x00d0] = 0x4fbe, [0x00d1] = 0x4fc0,
  [0x00d2] = 0x4fc1, [0x00d3] = 0x4fc2, [0x00d4] = 0x4fc6, [0x00d5] = 0x4fc7,
  [0x00d6] = 0x4fc8, [0x00d7] = 0x4fc9, [0x00d8] = 0x4fcb, [0x00d9] = 0x4fcc,
  [0x00da] = 0x4fcd, [0x00db] = 0x4fd2, [0x00dc] = 0x4fd3, [0x00dd] = 0x4fd4,
  [0x00de] = 0x4fd5, [0x00df] = 0x4fd6, [0x00e0] = 0x4fd9, [0x00e1] = 0x4fdb,
  [0x00e2] = 0x4fe0, [0x00e3] = 0x4fe2, [0x00e4] = 0x4fe4, [0x00e5] = 0x4fe5,
  [0x00e6] = 0x4fe7, [0x00e7] = 0x4feb, [0x00e8] = 0x4fec, [0x00e9] = 0x4ff0,
  [0x00ea] = 0x4ff2, [0x00eb] = 0x4ff4, [0x00ec] = 0x4ff5, [0x00ed] = 0x4ff6,
  [0x00ee] = 0x4ff7, [0x00ef] = 0x4ff9, [0x00f0] = 0x4ffb, [0x00f1] = 0x4ffc,
  [0x00f2] = 0x4ffd, [0x00f3] = 0x4fff, [0x00f4] = 0x5000, [0x00f5] = 0x5001,
  [0x00f6] = 0x5002, [0x00f7] = 0x5003, [0x00f8] = 0x5004, [0x00f9] = 0x5005,
  [0x00fa] = 0x5006, [0x00fb] = 0x5007, [0x00fc] = 0x5008, [0x00fd] = 0x5009,
  [0x00fe] = 0x500a, [0x0100] = 0x500b, [0x0101] = 0x500e, [0x0102] = 0x5010,
  [0x0103] = 0x5011, [0x0104] = 0x5013, [0x0105] = 0x5015, [0x0106] = 0x5016,
  [0x0107] = 0x5017, [0x0108] = 0x501b, [0x0109] = 0x501d, [0x010a] = 0x501e,
  [0x010b] = 0x5020, [0x010c] = 0x5022, [0x010d] = 0x5023, [0x010e] = 0x5024,
  [0x010f] = 0x5027, [0x0110] = 0x502b, [0x0111] = 0x502f, [0x0112] = 0x5030,
  [0x0113] = 0x5031, [0x0114] = 0x5032, [0x0115] = 0x5033, [0x0116] = 0x5034,
  [0x0117] = 0x5035, [0x0118] = 0x5036, [0x0119] = 0x5037, [0x011a] = 0x5038,
  [0x011b] = 0x5039, [0x011c] = 0x503b, [0x011d] = 0x503d, [0x011e] = 0x503f,
  [0x011f] = 0x5040, [0x0120] = 0x5041, [0x0121] = 0x5042, [0x0122] = 0x5044,
  [0x0123] = 0x5045, [0x0124] = 0x5046, [0x0125] = 0x5049, [0x0126] = 0x504a,
  [0x0127] = 0x504b, [0x0128] = 0x504d, [0x0129] = 0x5050, [0x012a] = 0x5051,
  [0x012b] = 0x5052, [0x012c] = 0x5053, [0x012d] = 0x5054, [0x012e] = 0x5056,
  [0x012f] = 0x5057, [0x0130] = 0x5058, [0x0131] = 0x5059, [0x0132] = 0x505b,
  [0x0133] = 0x505d, [0x0134] = 0x505e, [0x0135] = 0x505f, [0x0136] = 0x5060,
  [0x0137] = 0x5061, [0x0138] = 0x5062, [0x0139] = 0x5063, [0x013a] = 0x5064,
  [0x013b] = 0x5066, [0x013c] = 0x5067, [0x013d] = 0x5068, [0x013e] = 0x5069,
  [0x013f] = 0x506a, [0x0140] = 0x506b, [0x0141] = 0x506d, [0x0142] = 0x506e,
  [0x0143] = 0x506f, [0x0144] = 0x5070, [0x0145] = 0x5071, [0x0146] = 0x5072,
  [0x0147] = 0x5073, [0x0148] = 0x5074, [0x0149] = 0x5075, [0x014a] = 0x5078,
  [0x014b] = 0x5079, [0x014c] = 0x507a, [0x014d] = 0x507c, [0x014e] = 0x507d,
  [0x014f] = 0x5081, [0x0150] = 0x5082, [0x0151] = 0x5083, [0x0152] = 0x5084,
  [0x0153] = 0x5086, [0x0154] = 0x5087, [0x0155] = 0x5089, [0x0156] = 0x508a,
  [0x0157] = 0x508b, [0x0158] = 0x508c, [0x0159] = 0x508e, [0x015a] = 0x508f,
  [0x015b] = 0x5090, [0x015c] = 0x5091, [0x015d] = 0x5092, [0x015e] = 0x5093,
  [0x015f] = 0x5094, [0x0160] = 0x5095, [0x0161] = 0x5096, [0x0162] = 0x5097,
  [0x0163] = 0x5098, [0x0164] = 0x5099, [0x0165] = 0x509a, [0x0166] = 0x509b,
  [0x0167] = 0x509c, [0x0168] = 0x509d, [0x0169] = 0x509e, [0x016a] = 0x509f,
  [0x016b] = 0x50a0, [0x016c] = 0x50a1, [0x016d] = 0x50a2, [0x016e] = 0x50a4,
  [0x016f] = 0x50a6, [0x0170] = 0x50aa, [0x0171] = 0x50ab, [0x0172] = 0x50ad,
  [0x0173] = 0x50ae, [0x0174] = 0x50af, [0x0175] = 0x50b0, [0x0176] = 0x50b1,
  [0x0177] = 0x50b3, [0x0178] = 0x50b4, [0x0179] = 0x50b5, [0x017a] = 0x50b6,
  [0x017b] = 0x50b7, [0x017c] = 0x50b8, [0x017d] = 0x50b9, [0x017e] = 0x50bc,
  [0x0180] = 0x50bd, [0x0181] = 0x50be, [0x0182] = 0x50bf, [0x0183] = 0x50c0,
  [0x0184] = 0x50c1, [0x0185] = 0x50c2, [0x0186] = 0x50c3, [0x0187] = 0x50c4,
  [0x0188] = 0x50c5, [0x0189] = 0x50c6, [0x018a] = 0x50c7, [0x018b] = 0x50c8,
  [0x018c] = 0x50c9, [0x018d] = 0x50ca, [0x018e] = 0x50cb, [0x018f] = 0x50cc,
  [0x0190] = 0x50cd, [0x0191] = 0x50ce, [0x0192] = 0x50d0, [0x0193] = 0x50d1,
  [0x0194] = 0x50d2, [0x0195] = 0x50d3, [0x0196] = 0x50d4, [0x0197] = 0x50d5,
  [0x0198] = 0x50d7, [0x0199] = 0x50d8, [0x019a] = 0x50d9, [0x019b] = 0x50db,
  [0x019c] = 0x50dc, [0x019d] = 0x50dd, [0x019e] = 0x50de, [0x019f] = 0x50df,
  [0x01a0] = 0x50e0, [0x01a1] = 0x50e1, [0x01a2] = 0x50e2, [0x01a3] = 0x50e3,
  [0x01a4] = 0x50e4, [0x01a5] = 0x50e5, [0x01a6] = 0x50e8, [0x01a7] = 0x50e9,
  [0x01a8] = 0x50ea, [0x01a9] = 0x50eb, [0x01aa] = 0x50ef, [0x01ab] = 0x50f0,
  [0x01ac] = 0x50f1, [0x01ad] = 0x50f2, [0x01ae] = 0x50f4, [0x01af] = 0x50f6,
  [0x01b0] = 0x50f7, [0x01b1] = 0x50f8, [0x01b2] = 0x50f9, [0x01b3] = 0x50fa,
  [0x01b4] = 0x50fc, [0x01b5] = 0x50fd, [0x01b6] = 0x50fe, [0x01b7] = 0x50ff,
  [0x01b8] = 0x5100, [0x01b9] = 0x5101, [0x01ba] = 0x5102, [0x01bb] = 0x5103,
  [0x01bc] = 0x5104, [0x01bd] = 0x5105, [0x01be] = 0x5108, [0x01c0] = 0x5109,
  [0x01c1] = 0x510a, [0x01c2] = 0x510c, [0x01c3] = 0x510d, [0x01c4] = 0x510e,
  [0x01c5] = 0x510f, [0x01c6] = 0x5110, [0x01c7] = 0x5111, [0x01c8] = 0x5113,
  [0x01c9] = 0x5114, [0x01ca] = 0x5115, [0x01cb] = 0x5116, [0x01cc] = 0x5117,
  [0x01cd] = 0x5118, [0x01ce] = 0x5119, [0x01cf] = 0x511a, [0x01d0] = 0x511b,
  [0x01d1] = 0x511c, [0x01d2] = 0x511d, [0x01d3] = 0x511e, [0x01d4] = 0x511f,
  [0x01d5] = 0x5120, [0x01d6] = 0x5122, [0x01d7] = 0x5123, [0x01d8] = 0x5124,
  [0x01d9] = 0x5125, [0x01da] = 0x5126, [0x01db] = 0x5127, [0x01dc] = 0x5128,
  [0x01dd] = 0x5129, [0x01de] = 0x512a, [0x01df] = 0x512b, [0x01e0] = 0x512c,
  [0x01e1] = 0x512d, [0x01e2] = 0x512e, [0x01e3] = 0x512f, [0x01e4] = 0x5130,
  [0x01e5] = 0x5131, [0x01e6] = 0x5132, [0x01e7] = 0x5133, [0x01e8] = 0x5134,
  [0x01e9] = 0x5135, [0x01ea] = 0x5136, [0x01eb] = 0x5137, [0x01ec] = 0x5138,
  [0x01ed] = 0x5139, [0x01ee] = 0x513a, [0x01ef] = 0x513b, [0x01f0] = 0x513c,
  [0x01f1] = 0x513d, [0x01f2] = 0x513e, [0x01f3] = 0x5142, [0x01f4] = 0x5147,
  [0x01f5] = 0x514a, [0x01f6] = 0x514c, [0x01f7] = 0x514e, [0x01f8] = 0x514f,
  [0x01f9] = 0x5150, [0x01fa] = 0x5152, [0x01fb] = 0x5153, [0x01fc] = 0x5157,
  [0x01fd] = 0x5158, [0x01fe] = 0x5159, [0x01ff] = 0x515b, [0x0200] = 0x515d,
  [0x0201] = 0x515e, [0x0202] = 0x515f, [0x0203] = 0x5160, [0x0204] = 0x5161,
  [0x0205] = 0x5163, [0x0206] = 0x5164, [0x0207] = 0x5166, [0x0208] = 0x5167,
  [0x0209] = 0x5169, [0x020a] = 0x516a, [0x020b] = 0x516f, [0x020c] = 0x5172,
  [0x020d] = 0x517a, [0x020e] = 0x517e, [0x020f] = 0x517f, [0x0210] = 0x5183,
  [0x0211] = 0x5184, [0x0212] = 0x5186, [0x0213] = 0x5187, [0x0214] = 0x518a,
  [0x0215] = 0x518b, [0x0216] = 0x518e, [0x0217] = 0x518f, [0x0218] = 0x5190,
  [0x0219] = 0x5191, [0x021a] = 0x5193, [0x021b] = 0x5194, [0x021c] = 0x5198,
  [0x021d] = 0x519a, [0x021e] = 0x519d, [0x021f] = 0x519e, [0x0220] = 0x519f,
  [0x0221] = 0x51a1, [0x0222] = 0x51a3, [0x0223] = 0x51a6, [0x0224] = 0x51a7,
  [0x0225] = 0x51a8, [0x0226] = 0x51a9, [0x0227] = 0x51aa, [0x0228] = 0x51ad,
  [0x0229] = 0x51ae, [0x022a] = 0x51b4, [0x022b] = 0x51b8, [0x022c] = 0x51b9,
  [0x022d] = 0x51ba, [0x022e] = 0x51be, [0x022f] = 0x51bf, [0x0230] = 0x51c1,
  [0x0231] = 0x51c2, [0x0232] = 0x51c3, [0x0233] = 0x51c5, [0x0234] = 0x51c8,
  [0x0235] = 0x51ca, [0x0236] = 0x51cd, [0x0237] = 0x51ce, [0x0238] = 0x51d0,
  [0x0239] = 0x51d2, [0x023a] = 0x51d3, [0x023b] = 0x51d4, [0x023c] = 0x51d5,
  [0x023d] = 0x51d6, [0x023e] = 0x51d7, [0x0240] = 0x51d8, [0x0241] = 0x51d9,
  [0x0242] = 0x51da, [0x0243] = 0x51dc, [0x0244] = 0x51de, [0x0245] = 0x51df,
  [0x0246] = 0x51e2, [0x0247] = 0x51e3, [0x0248] = 0x51e5, [0x0249] = 0x51e6,
  [0x024a] = 0x51e7, [0x024b] = 0x51e8, [0x024c] = 0x51e9, [0x024d] = 0x51ea,
  [0x024e] = 0x51ec, [0x024f] = 0x51ee, [0x0250] = 0x51f1, [0x0251] = 0x51f2,
  [0x0252] = 0x51f4, [0x0253] = 0x51f7, [0x0254] = 0x51fe, [0x0255] = 0x5204,
  [0x0256] = 0x5205, [0x0257] = 0x5209, [0x0258] = 0x520b, [0x0259] = 0x520c,
  [0x025a] = 0x520f, [0x025b] = 0x5210, [0x025c] = 0x5213, [0x025d] = 0x5214,
  [0x025e] = 0x5215, [0x025f] = 0x521c, [0x0260] = 0x521e, [0x0261] = 0x521f,
  [0x0262] = 0x5221, [0x0263] = 0x5222, [0x0264] = 0x5223, [0x0265] = 0x5225,
  [0x0266] = 0x5226, [0x0267] = 0x5227, [0x0268] = 0x522a, [0x0269] = 0x522c,
  [0x026a] = 0x522f, [0x026b] = 0x5231, [0x026c] = 0x5232, [0x026d] = 0x5234,
  [0x026e] = 0x5235, [0x026f] = 0x523c, [0x0270] = 0x523e, [0x0271] = 0x5244,
  [0x0272] = 0x5245, [0x0273] = 0x5246, [0x0274] = 0x5247, [0x0275] = 0x5248,
  [0x0276] = 0x5249, [0x0277] = 0x524b, [0x0278] = 0x524e, [0x0279] = 0x524f,
  [0x027a] = 0x5252, [0x027b] = 0x5253, [0x027c] = 0x5255, [0x027d] = 0x5257,
  [0x027e] = 0x5258, [0x0280] = 0x5259, [0x0281] = 0x525a, [0x0282] = 0x525b,
  [0x0283] = 0x525d, [0x0284] = 0x525f, [0x0285] = 0x5260, [0x0286] = 0x5262,
  [0x0287] = 0x5263, [0x0288] = 0x5264, [0x0289] = 0x5266, [0x028a] = 0x5268,
  [0x028b] = 0x526b, [0x028c] = 0x526c, [0x028d] = 0x526d, [0x028e] = 0x526e,
  [0x028f] = 0x5270, [0x0290] = 0x5271, [0x0291] = 0x5273, [0x0292] = 0x5274,
  [0x0293] = 0x5275, [0x0294] = 0x5276, [0x0295] = 0x5277, [0x0296] = 0x5278,
  [0x0297] = 0x5279, [0x0298] = 0x527a, [0x0299] = 0x527b, [0x029a] = 0x527c,
  [0x029b] = 0x527e, [0x029c] = 0x5280, [0x029d] = 0x5283, [0x029e] = 0x5284,
  [0x029f] = 0x5285, [0x02a0] = 0x5286, [0x02a1] = 0x5287, [0x02a2] = 0x5289,
  [0x02a3] = 0x528a, [0x02a4] = 0x528b, [0x02a5] = 0x528c, [0x02a6] = 0x528d,
  [0x02a7] = 0x528e, [0x02a8] = 0x528f, [0x02a9] = 0x5291, [0x02aa] = 0x5292,
  [0x02ab] = 0x5294, [0x02ac] = 0x5295, [0x02ad] = 0x5296, [0x02ae] = 0x5297,
  [0x02af] = 0x5298, [0x02b0] = 0x5299, [0x02b1] = 0x529a, [0x02b2] = 0x529c,
  [0x02b3] = 0x52a4, [0x02b4] = 0x52a5, [0x02b5] = 0x52a6, [0x02b6] = 0x52a7,
  [0x02b7] = 0x52ae, [0x02b8] = 0x52af, [0x02b9] = 0x52b0, [0x02ba] = 0x52b4,
  [0x02bb] = 0x52b5, [0x02bc] = 0x52b6, [0x02bd] = 0x52b7, [0x02be] = 0x52b8,
  [0x02bf] = 0x52b9, [0x02c0] = 0x52ba, [0x02c1] = 0x52bb, [0x02c2] = 0x52bc,
  [0x02c3] = 0x52bd, [0x02c4] = 0x52c0, [0x02c5] = 0x52c1, [0x02c6] = 0x52c2,
  [0x02c7] = 0x52c4, [0x02c8] = 0x52c5, [0x02c9] = 0x52c6, [0x02ca] = 0x52c8,
  [0x02cb] = 0x52ca, [0x02cc] = 0x52cc, [0x02cd] = 0x52cd, [0x02ce] = 0x52ce,
  [0x02cf] = 0x52cf, [0x02d0] = 0x52d1, [0x02d1] = 0x52d3, [0x02d2] = 0x52d4,
  [0x02d3] = 0x52d5, [0x02d4] = 0x52d7, [0x02d5] = 0x52d9, [0x02d6] = 0x52da,
  [0x02d7] = 0x52db, [0x02d8] = 0x52dc, [0x02d9] = 0x52dd, [0x02da] = 0x52de,
  [0x02db] = 0x52e0, [0x02dc] = 0x52e1, [0x02dd] = 0x52e2, [0x02de] = 0x52e3,
  [0x02df] = 0x52e5, [0x02e0] = 0x52e6, [0x02e1] = 0x52e7, [0x02e2] = 0x52e8,
  [0x02e3] = 0x52e9, [0x02e4] = 0x52ea, [0x02e5] = 0x52eb, [0x02e6] = 0x52ec,
  [0x02e7] = 0x52ed, [0x02e8] = 0x52ee, [0x02e9] = 0x52ef, [0x02ea] = 0x52f1,
  [0x02eb] = 0x52f2, [0x02ec] = 0x52f3, [0x02ed] = 0x52f4, [0x02ee] = 0x52f5,
  [0x02ef] = 0x52f6, [0x02f0] = 0x52f7, [0x02f1] = 0x52f8, [0x02f2] = 0x52fb,
  [0x02f3] = 0x52fc, [0x02f4] = 0x52fd, [0x02f5] = 0x5301, [0x02f6] = 0x5302,
  [0x02f7] = 0x5303, [0x02f8] = 0x5304, [0x02f9] = 0x5307, [0x02fa] = 0x5309,
  [0x02fb] = 0x530a, [0x02fc] = 0x530b, [0x02fd] = 0x530c, [0x02fe] = 0x530e,
  [0x0300] = 0x5311, [0x0301] = 0x5312, [0x0302] = 0x5313, [0x0303] = 0x5314,
  [0x0304] = 0x5318, [0x0305] = 0x531b, [0x0306] = 0x531c, [0x0307] = 0x531e,
  [0x0308] = 0x531f, [0x0309] = 0x5322, [0x030a] = 0x5324, [0x030b] = 0x5325,
  [0x030c] = 0x5327, [0x030d] = 0x5328, [0x030e] = 0x5329, [0x030f] = 0x532b,
  [0x0310] = 0x532c, [0x0311] = 0x532d, [0x0312] = 0x532f, [0x0313] = 0x5330,
  [0x0314] = 0x5331, [0x0315] = 0x5332, [0x0316] = 0x5333, [0x0317] = 0x5334,
  [0x0318] = 0x5335, [0x0319] = 0x5336, [0x031a] = 0x5337, [0x031b] = 0x5338,
  [0x031c] = 0x533c, [0x031d] = 0x533d, [0x031e] = 0x5340, [0x031f] = 0x5342,
  [0x0320] = 0x5344, [0x0321] = 0x5346, [0x0322] = 0x534b, [0x0323] = 0x534c,
  [0x0324] = 0x534d, [0x0325] = 0x5350, [0x0326] = 0x5354, [0x0327] = 0x5358,
  [0x0328] = 0x5359, [0x0329] = 0x535b, [0x032a] = 0x535d, [0x032b] = 0x5365,
  [0x032c] = 0x5368, [0x032d] = 0x536a, [0x032e] = 0x536c, [0x032f] = 0x536d,
  [0x0330] = 0x5372, [0x0331] = 0x5376, [0x0332] = 0x5379, [0x0333] = 0x537b,
  [0x0334] = 0x537c, [0x0335] = 0x537d, [0x0336] = 0x537e, [0x0337] = 0x5380,
  [0x0338] = 0x5381, [0x0339] = 0x5383, [0x033a] = 0x5387, [0x033b] = 0x5388,
  [0x033c] = 0x538a, [0x033d] = 0x538e, [0x033e] = 0x538f, [0x0340] = 0x5390,
  [0x0341] = 0x5391, [0x0342] = 0x5392, [0x0343] = 0x5393, [0x0344] = 0x5394,
  [0x0345] = 0x5396, [0x0346] = 0x5397, [0x0347] = 0x5399, [0x0348] = 0x539b,
  [0x0349] = 0x539c, [0x034a] = 0x539e, [0x034b] = 0x53a0, [0x034c] = 0x53a1,
  [0x034d] = 0x53a4, [0x034e] = 0x53a7, [0x034f] = 0x53aa, [0x0350] = 0x53ab,
  [0x0351] = 0x53ac, [0x0352] = 0x53ad, [0x0353] = 0x53af, [0x0354] = 0x53b0,
  [0x0355] = 0x53b1, [0x0356] = 0x53b2, [0x0357] = 0x53b3, [0x0358] = 0x53b4,
  [0x0359] = 0x53b5, [0x035a] = 0x53b7, [0x035b] = 0x53b8, [0x035c] = 0x53b9,
  [0x035d] = 0x53ba, [0x035e] = 0x53bc, [0x035f] = 0x53bd, [0x0360] = 0x53be,
  [0x0361] = 0x53c0, [0x0362] = 0x53c3, [0x0363] = 0x53c4, [0x0364] = 0x53c5,
  [0x0365] = 0x53c6, [0x0366] = 0x53c7, [0x0367] = 0x53ce, [0x0368] = 0x53cf,
  [0x0369] = 0x53d0, [0x036a] = 0x53d2, [0x036b] = 0x53d3, [0x036c] = 0x53d5,
  [0x036d] = 0x53da, [0x036e] = 0x53dc, [0x036f] = 0x53dd, [0x0370] = 0x53de,
  [0x0371] = 0x53e1, [0x0372] = 0x53e2, [0x0373] = 0x53e7, [0x0374] = 0x53f4,
  [0x0375] = 0x53fa, [0x0376] = 0x53fe, [0x0377] = 0x53ff, [0x0378] = 0x5400,
  [0x0379] = 0x5402, [0x037a] = 0x5405, [0x037b] = 0x5407, [0x037c] = 0x540b,
  [0x037d] = 0x5414, [0x037e] = 0x5418, [0x037f] = 0x5419, [0x0380] = 0x541a,
  [0x0381] = 0x541c, [0x0382] = 0x5422, [0x0383] = 0x5424, [0x0384] = 0x5425,
  [0x0385] = 0x542a, [0x0386] = 0x5430, [0x0387] = 0x5433, [0x0388] = 0x5436,
  [0x0389] = 0x5437, [0x038a] = 0x543a, [0x038b] = 0x543d, [0x038c] = 0x543f,
  [0x038d] = 0x5441, [0x038e] = 0x5442, [0x038f] = 0x5444, [0x0390] = 0x5445,
  [0x0391] = 0x5447, [0x0392] = 0x5449, [0x0393] = 0x544c, [0x0394] = 0x544d,
  [0x0395] = 0x544e, [0x0396] = 0x544f, [0x0397] = 0x5451, [0x0398] = 0x545a,
  [0x0399] = 0x545d, [0x039a] = 0x545e, [0x039b] = 0x545f, [0x039c] = 0x5460,
  [0x039d] = 0x5461, [0x039e] = 0x5463, [0x039f] = 0x5465, [0x03a0] = 0x5467,
  [0x03a1] = 0x5469, [0x03a2] = 0x546a, [0x03a3] = 0x546b, [0x03a4] = 0x546c,
  [0x03a5] = 0x546d, [0x03a6] = 0x546e, [0x03a7] = 0x546f, [0x03a8] = 0x5470,
  [0x03a9] = 0x5474, [0x03aa] = 0x5479, [0x03ab] = 0x547a, [0x03ac] = 0x547e,
  [0x03ad] = 0x547f, [0x03ae] = 0x5481, [0x03af] = 0x5483, [0x03b0] = 0x5485,
  [0x03b1] = 0x5487, [0x03b2] = 0x5488, [0x03b3] = 0x5489, [0x03b4] = 0x548a,
  [0x03b5] = 0x548d, [0x03b6] = 0x5491, [0x03b7] = 0x5493, [0x03b8] = 0x5497,
  [0x03b9] = 0x5498, [0x03ba] = 0x549c, [0x03bb] = 0x549e, [0x03bc] = 0x549f,
  [0x03bd] = 0x54a0, [0x03be] = 0x54a1, [0x03c0] = 0x54a2, [0x03c1] = 0x54a5,
  [0x03c2] = 0x54ae, [0x03c3] = 0x54b0, [0x03c4] = 0x54b2, [0x03c5] = 0x54b5,
  [0x03c6] = 0x54b6, [0x03c7] = 0x54b7, [0x03c8] = 0x54b9, [0x03c9] = 0x54ba,
  [0x03ca] = 0x54bc, [0x03cb] = 0x54be, [0x03cc] = 0x54c3, [0x03cd] = 0x54c5,
  [0x03ce] = 0x54ca, [0x03cf] = 0x54cb, [0x03d0] = 0x54d6, [0x03d1] = 0x54d8,
  [0x03d2] = 0x54db, [0x03d3] = 0x54e0, [0x03d4] = 0x54e1, [0x03d5] = 0x54e2,
  [0x03d6] = 0x54e3, [0x03d7] = 0x54e4, [0x03d8] = 0x54eb, [0x03d9] = 0x54ec,
  [0x03da] = 0x54ef, [0x03db] = 0x54f0, [0x03dc] = 0x54f1, [0x03dd] = 0x54f4,
  [0x03de] = 0x54f5, [0x03df] = 0x54f6, [0x03e0] = 0x54f7, [0x03e1] = 0x54f8,
  [0x03e2] = 0x54f9, [0x03e3] = 0x54fb, [0x03e4] = 0x54fe, [0x03e5] = 0x5500,
  [0x03e6] = 0x5502, [0x03e7] = 0x5503, [0x03e8] = 0x5504, [0x03e9] = 0x5505,
  [0x03ea] = 0x5508, [0x03eb] = 0x550a, [0x03ec] = 0x550b, [0x03ed] = 0x550c,
  [0x03ee] = 0x550d, [0x03ef] = 0x550e, [0x03f0] = 0x5512, [0x03f1] = 0x5513,
  [0x03f2] = 0x5515, [0x03f3] = 0x5516, [0x03f4] = 0x5517, [0x03f5] = 0x5518,
  [0x03f6] = 0x5519, [0x03f7] = 0x551a, [0x03f8] = 0x551c, [0x03f9] = 0x551d,
  [0x03fa] = 0x551e, [0x03fb] = 0x551f, [0x03fc] = 0x5521, [0x03fd] = 0x5525,
  [0x03fe] = 0x5526, [0x0400] = 0x5528, [0x0401] = 0x5529, [0x0402] = 0x552b,
  [0x0403] = 0x552d, [0x0404] = 0x5532, [0x0405] = 0x5534, [0x0406] = 0x5535,
  [0x0407] = 0x5536, [0x0408] = 0x5538, [0x0409] = 0x5539, [0x040a] = 0x553a,
  [0x040b] = 0x553b, [0x040c] = 0x553d, [0x040d] = 0x5540, [0x040e] = 0x5542,
  [0x040f] = 0x5545, [0x0410] = 0x5547, [0x0411] = 0x5548, [0x0412] = 0x554b,
  [0x0413] = 0x554c, [0x0414] = 0x554d, [0x0415] = 0x554e, [0x0416] = 0x554f,
  [0x0417] = 0x5551, [0x0418] = 0x5552, [0x0419] = 0x5553, [0x041a] = 0x5554,
  [0x041b] = 0x5557, [0x041c] = 0x5558, [0x041d] = 0x5559, [0x041e] = 0x555a,
  [0x041f] = 0x555b, [0x0420] = 0x555d, [0x0421] = 0x555e, [0x0422] = 0x555f,
  [0x0423] = 0x5560, [0x0424] = 0x5562, [0x0425] = 0x5563, [0x0426] = 0x5568,
  [0x0427] = 0x5569, [0x0428] = 0x556b, [0x0429] = 0x556f, [0x042a] = 0x5570,
  [0x042b] = 0x5571, [0x042c] = 0x5572, [0x042d] = 0x5573, [0x042e] = 0x5574,
  [0x042f] = 0x5579, [0x0430] = 0x557a, [0x0431] = 0x557d, [0x0432] = 0x557f,
  [0x0433] = 0x5585, [0x0434] = 0x5586, [0x0435] = 0x558c, [0x0436] = 0x558d,
  [0x0437] = 0x558e, [0x0438] = 0x5590, [0x0439] = 0x5592, [0x043a] = 0x5593,
  [0x043b] = 0x5595, [0x043c] = 0x5596, [0x043d] = 0x5597, [0x043e] = 0x559a,
  [0x043f] = 0x559b, [0x0440] = 0x559e, [0x0441] = 0x55a0, [0x0442] = 0x55a1,
  [0x0443] = 0x55a2, [0x0444] = 0x55a3, [0x0445] = 0x55a4, [0x0446] = 0x55a5,
  [0x0447] = 0x55a6, [0x0448] = 0x55a8, [0x0449] = 0x55a9, [0x044a] = 0x55aa,
  [0x044b] = 0x55ab, [0x044c] = 0x55ac, [0x044d] = 0x55ad, [0x044e] = 0x55ae,
  [0x044f] = 0x55af, [0x0450] = 0x55b0, [0x0451] = 0x55b2, [0x0452] = 0x55b4,
  [0x0453] = 0x55b6, [0x0454] = 0x55b8, [0x0455] = 0x55ba, [0x0456] = 0x55bc,
  [0x0457] = 0x55bf, [0x0458] = 0x55c0, [0x0459] = 0x55c1, [0x045a] = 0x55c2,
  [0x045b] = 0x55c3, [0x045c] = 0x55c6, [0x045d] = 0x55c7, [0x045e] = 0x55c8,
  [0x045f] = 0x55ca, [0x0460] = 0x55cb, [0x0461] = 0x55ce, [0x0462] = 0x55cf,
  [0x0463] = 0x55d0, [0x0464] = 0x55d5, [0x0465] = 0x55d7, [0x0466] = 0x55d8,
  [0x0467] = 0x55d9, [0x0468] = 0x55da, [0x0469] = 0x55db, [0x046a] = 0x55de,
  [0x046b] = 0x55e0, [0x046c] = 0x55e2, [0x046d] = 0x55e7, [0x046e] = 0x55e9,
  [0x046f] = 0x55ed, [0x0470] = 0x55ee, [0x0471] = 0x55f0, [0x0472] = 0x55f1,
  [0x0473] = 0x55f4, [0x0474] = 0x55f6, [0x0475] = 0x55f8, [0x0476] = 0x55f9,
  [0x0477] = 0x55fa, [0x0478] = 0x55fb, [0x0479] = 0x55fc, [0x047a] = 0x55ff,
  [0x047b] = 0x5602, [0x047c] = 0x5603, [0x047d] = 0x5604, [0x047e] = 0x5605,
  [0x0480] = 0x5606, [0x0481] = 0x5607, [0x0482] = 0x560a, [0x0483] = 0x560b,
  [0x0484] = 0x560d, [0x0485] = 0x5610, [0x0486] = 0x5611, [0x0487] = 0x5612,
  [0x0488] = 0x5613, [0x0489] = 0x5614, [0x048a] = 0x5615, [0x048b] = 0x5616,
  [0x048c] = 0x5617, [0x048d] = 0x5619, [0x048e] = 0x561a, [0x048f] = 0x561c,
  [0x0490] = 0x561d, [0x0491] = 0x5620, [0x0492] = 0x5621, [0x0493] = 0x5622,
  [0x0494] = 0x5625, [0x0495] = 0x5626, [0x0496] = 0x5628, [0x0497] = 0x5629,
  [0x0498] = 0x562a, [0x0499] = 0x562b, [0x049a] = 0x562e, [0x049b] = 0x562f,
  [0x049c] = 0x5630, [0x049d] = 0x5633, [0x049e] = 0x5635, [0x049f] = 0x5637,
  [0x04a0] = 0x5638, [0x04a1] = 0x563a, [0x04a2] = 0x563c, [0x04a3] = 0x563d,
  [0x04a4] = 0x563e, [0x04a5] = 0x5640, [0x04a6] = 0x5641, [0x04a7] = 0x5642,
  [0x04a8] = 0x5643, [0x04a9] = 0x5644, [0x04aa] = 0x5645, [0x04ab] = 0x5646,
  [0x04ac] = 0x5647, [0x04ad] = 0x5648, [0x04ae] = 0x5649, [0x04af] = 0x564a,
  [0x04b0] = 0x564b, [0x04b1] = 0x564f, [0x04b2] = 0x5650, [0x04b3] = 0x5651,
  [0x04b4] = 0x5652, [0x04b5] = 0x5653, [0x04b6] = 0x5655, [0x04b7] = 0x5656,
  [0x04b8] = 0x565a, [0x04b9] = 0x565b, [0x04ba] = 0x565d, [0x04bb] = 0x565e,
  [0x04bc] = 0x565f, [0x04bd] = 0x5660, [0x04be] = 0x5661, [0x04c0] = 0x5663,
  [0x04c1] = 0x5665, [0x04c2] = 0x5666, [0x04c3] = 0x5667, [0x04c4] = 0x566d,
  [0x04c5] = 0x566e, [0x04c6] = 0x566f, [0x04c7] = 0x5670, [0x04c8] = 0x5672,
  [0x04c9] = 0x5673, [0x04ca] = 0x5674, [0x04cb] = 0x5675, [0x04cc] = 0x5677,
  [0x04cd] = 0x5678, [0x04ce] = 0x5679, [0x04cf] = 0x567a, [0x04d0] = 0x567d,
  [0x04d1] = 0x567e, [0x04d2] = 0x567f, [0x04d3] = 0x5680, [0x04d4] = 0x5681,
  [0x04d5] = 0x5682, [0x04d6] = 0x5683, [0x04d7] = 0x5684, [0x04d8] = 0x5687,
  [0x04d9] = 0x5688, [0x04da] = 0x5689, [0x04db] = 0x568a, [0x04dc] = 0x568b,
  [0x04dd] = 0x568c, [0x04de] = 0x568d, [0x04df] = 0x5690, [0x04e0] = 0x5691,
  [0x04e1] = 0x5692, [0x04e2] = 0x5694, [0x04e3] = 0x5695, [0x04e4] = 0x5696,
  [0x04e5] = 0x5697, [0x04e6] = 0x5698, [0x04e7] = 0x5699, [0x04e8] = 0x569a,
  [0x04e9] = 0x569b, [0x04ea] = 0x569c, [0x04eb] = 0x569d, [0x04ec] = 0x569e,
  [0x04ed] = 0x569f, [0x04ee] = 0x56a0, [0x04ef] = 0x56a1, [0x04f0] = 0x56a2,
  [0x04f1] = 0x56a4, [0x04f2] = 0x56a5, [0x04f3] = 0x56a6, [0x04f4] = 0x56a7,
  [0x04f5] = 0x56a8, [0x04f6] = 0x56a9, [0x04f7] = 0x56aa, [0x04f8] = 0x56ab,
  [0x04f9] = 0x56ac, [0x04fa] = 0x56ad, [0x04fb] = 0x56ae, [0x04fc] = 0x56b0,
  [0x04fd] = 0x56b1, [0x04fe] = 0x56b2, [0x04ff] = 0x56b3, [0x0500] = 0x56b4,
  [0x0501] = 0x56b5, [0x0502] = 0x56b6, [0x0503] = 0x56b8, [0x0504] = 0x56b9,
  [0x0505] = 0x56ba, [0x0506] = 0x56bb, [0x0507] = 0x56bd, [0x0508] = 0x56be,
  [0x0509] = 0x56bf, [0x050a] = 0x56c0, [0x050b] = 0x56c1, [0x050c] = 0x56c2,
  [0x050d] = 0x56c3, [0x050e] = 0x56c4, [0x050f] = 0x56c5, [0x0510] = 0x56c6,
  [0x0511] = 0x56c7, [0x0512] = 0x56c8, [0x0513] = 0x56c9, [0x0514] = 0x56cb,
  [0x0515] = 0x56cc, [0x0516] = 0x56cd, [0x0517] = 0x56ce, [0x0518] = 0x56cf,
  [0x0519] = 0x56d0, [0x051a] = 0x56d1, [0x051b] = 0x56d2, [0x051c] = 0x56d3,
  [0x051d] = 0x56d5, [0x051e] = 0x56d6, [0x051f] = 0x56d8, [0x0520] = 0x56d9,
  [0x0521] = 0x56dc, [0x0522] = 0x56e3, [0x0523] = 0x56e5, [0x0524] = 0x56e6,
  [0x0525] = 0x56e7, [0x0526] = 0x56e8, [0x0527] = 0x56e9, [0x0528] = 0x56ea,
  [0x0529] = 0x56ec, [0x052a] = 0x56ee, [0x052b] = 0x56ef, [0x052c] = 0x56f2,
  [0x052d] = 0x56f3, [0x052e] = 0x56f6, [0x052f] = 0x56f7, [0x0530] = 0x56f8,
  [0x0531] = 0x56fb, [0x0532] = 0x56fc, [0x0533] = 0x5700, [0x0534] = 0x5701,
  [0x0535] = 0x5702, [0x0536] = 0x5705, [0x0537] = 0x5707, [0x0538] = 0x570b,
  [0x0539] = 0x570c, [0x053a] = 0x570d, [0x053b] = 0x570e, [0x053c] = 0x570f,
  [0x053d] = 0x5710, [0x053e] = 0x5711, [0x0540] = 0x5712, [0x0541] = 0x5713,
  [0x0542] = 0x5714, [0x0543] = 0x5715, [0x0544] = 0x5716, [0x0545] = 0x5717,
  [0x0546] = 0x5718, [0x0547] = 0x5719, [0x0548] = 0x571a, [0x0549] = 0x571b,
  [0x054a] = 0x571d, [0x054b] = 0x571e, [0x054c] = 0x5720, [0x054d] = 0x5721,
  [0x054e] = 0x5722, [0x054f] = 0x5724, [0x0550] = 0x5725, [0x0551] = 0x5726,
  [0x0552] = 0x5727, [0x0553] = 0x572b, [0x0554] = 0x5731, [0x0555] = 0x5732,
  [0x0556] = 0x5734, [0x0557] = 0x5735, [0x0558] = 0x5736, [0x0559] = 0x5737,
  [0x055a] = 0x5738, [0x055b] = 0x573c, [0x055c] = 0x573d, [0x055d] = 0x573f,
  [0x055e] = 0x5741, [0x055f] = 0x5743, [0x0560] = 0x5744, [0x0561] = 0x5745,
  [0x0562] = 0x5746, [0x0563] = 0x5748, [0x0564] = 0x5749, [0x0565] = 0x574b,
  [0x0566] = 0x5752, [0x0567] = 0x5753, [0x0568] = 0x5754, [0x0569] = 0x5755,
  [0x056a] = 0x5756, [0x056b] = 0x5758, [0x056c] = 0x5759, [0x056d] = 0x5762,
  [0x056e] = 0x5763, [0x056f] = 0x5765, [0x0570] = 0x5767, [0x0571] = 0x576c,
  [0x0572] = 0x576e, [0x0573] = 0x5770, [0x0574] = 0x5771, [0x0575] = 0x5772,
  [0x0576] = 0x5774, [0x0577] = 0x5775, [0x0578] = 0x5778, [0x0579] = 0x5779,
  [0x057a] = 0x577a, [0x057b] = 0x577d, [0x057c] = 0x577e, [0x057d] = 0x577f,
  [0x057e] = 0x5780, [0x0580] = 0x5781, [0x0581] = 0x5787, [0x0582] = 0x5788,
  [0x0583] = 0x5789, [0x0584] = 0x578a, [0x0585] = 0x578d, [0x0586] = 0x578e,
  [0x0587] = 0x578f, [0x0588] = 0x5790, [0x0589] = 0x5791, [0x058a] = 0x5794,
  [0x058b] = 0x5795, [0x058c] = 0x5796, [0x058d] = 0x5797, [0x058e] = 0x5798,
  [0x058f] = 0x5799, [0x0590] = 0x579a, [0x0591] = 0x579c, [0x0592] = 0x579d,
  [0x0593] = 0x579e, [0x0594] = 0x579f, [0x0595] = 0x57a5, [0x0596] = 0x57a8,
  [0x0597] = 0x57aa, [0x0598] = 0x57ac, [0x0599] = 0x57af, [0x059a] = 0x57b0,
  [0x059b] = 0x57b1, [0x059c] = 0x57b3, [0x059d] = 0x57b5, [0x059e] = 0x57b6,
  [0x059f] = 0x57b7, [0x05a0] = 0x57b9, [0x05a1] = 0x57ba, [0x05a2] = 0x57bb,
  [0x05a3] = 0x57bc, [0x05a4] = 0x57bd, [0x05a5] = 0x57be, [0x05a6] = 0x57bf,
  [0x05a7] = 0x57c0, [0x05a8] = 0x57c1, [0x05a9] = 0x57c4, [0x05aa] = 0x57c5,
  [0x05ab] = 0x57c6, [0x05ac] = 0x57c7, [0x05ad] = 0x57c8, [0x05ae] = 0x57c9,
  [0x05af] = 0x57ca, [0x05b0] = 0x57cc, [0x05b1] = 0x57cd, [0x05b2] = 0x57d0,
  [0x05b3] = 0x57d1, [0x05b4] = 0x57d3, [0x05b5] = 0x57d6, [0x05b6] = 0x57d7,
  [0x05b7] = 0x57db, [0x05b8] = 0x57dc, [0x05b9] = 0x57de, [0x05ba] = 0x57e1,
  [0x05bb] = 0x57e2, [0x05bc] = 0x57e3, [0x05bd] = 0x57e5, [0x05be] = 0x57e6,
  [0x05bf] = 0x57e7, [0x05c0] = 0x57e8, [0x05c1] = 0x57e9, [0x05c2] = 0x57ea,
  [0x05c3] = 0x57eb, [0x05c4] = 0x57ec, [0x05c5] = 0x57ee, [0x05c6] = 0x57f0,
  [0x05c7] = 0x57f1, [0x05c8] = 0x57f2, [0x05c9] = 0x57f3, [0x05ca] = 0x57f5,
  [0x05cb] = 0x57f6, [0x05cc] = 0x57f7, [0x05cd] = 0x57fb, [0x05ce] = 0x57fc,
  [0x05cf] = 0x57fe, [0x05d0] = 0x57ff, [0x05d1] = 0x5801, [0x05d2] = 0x5803,
  [0x05d3] = 0x5804, [0x05d4] = 0x5805, [0x05d5] = 0x5808, [0x05d6] = 0x5809,
  [0x05d7] = 0x580a, [0x05d8] = 0x580c, [0x05d9] = 0x580e, [0x05da] = 0x580f,
  [0x05db] = 0x5810, [0x05dc] = 0x5812, [0x05dd] = 0x5813, [0x05de] = 0x5814,
  [0x05df] = 0x5816, [0x05e0] = 0x5817, [0x05e1] = 0x5818, [0x05e2] = 0x581a,
  [0x05e3] = 0x581b, [0x05e4] = 0x581c, [0x05e5] = 0x581d, [0x05e6] = 0x581f,
  [0x05e7] = 0x5822, [0x05e8] = 0x5823, [0x05e9] = 0x5825, [0x05ea] = 0x5826,
  [0x05eb] = 0x5827, [0x05ec] = 0x5828, [0x05ed] = 0x5829, [0x05ee] = 0x582b,
  [0x05ef] = 0x582c, [0x05f0] = 0x582d, [0x05f1] = 0x582e, [0x05f2] = 0x582f,
  [0x05f3] = 0x5831, [0x05f4] = 0x5832, [0x05f5] = 0x5833, [0x05f6] = 0x5834,
  [0x05f7] = 0x5836, [0x05f8] = 0x5837, [0x05f9] = 0x5838, [0x05fa] = 0x5839,
  [0x05fb] = 0x583a, [0x05fc] = 0x583b, [0x05fd] = 0x583c, [0x05fe] = 0x583d,
  [0x0600] = 0x583e, [0x0601] = 0x583f, [0x0602] = 0x5840, [0x0603] = 0x5841,
  [0x0604] = 0x5842, [0x0605] = 0x5843, [0x0606] = 0x5845, [0x0607] = 0x5846,
  [0x0608] = 0x5847, [0x0609] = 0x5848, [0x060a] = 0x5849, [0x060b] = 0x584a,
  [0x060c] = 0x584b, [0x060d] = 0x584e, [0x060e] = 0x584f, [0x060f] = 0x5850,
  [0x0610] = 0x5852, [0x0611] = 0x5853, [0x0612] = 0x5855, [0x0613] = 0x5856,
  [0x0614] = 0x5857, [0x0615] = 0x5859, [0x0616] = 0x585a, [0x0617] = 0x585b,
  [0x0618] = 0x585c, [0x0619] = 0x585d, [0x061a] = 0x585f, [0x061b] = 0x5860,
  [0x061c] = 0x5861, [0x061d] = 0x5862, [0x061e] = 0x5863, [0x061f] = 0x5864,
  [0x0620] = 0x5866, [0x0621] = 0x5867, [0x0622] = 0x5868, [0x0623] = 0x5869,
  [0x0624] = 0x586a, [0x0625] = 0x586d, [0x0626] = 0x586e, [0x0627] = 0x586f,
  [0x0628] = 0x5870, [0x0629] = 0x5871, [0x062a] = 0x5872, [0x062b] = 0x5873,
  [0x062c] = 0x5874, [0x062d] = 0x5875, [0x062e] = 0x5876, [0x062f] = 0x5877,
  [0x0630] = 0x5878, [0x0631] = 0x5879, [0x0632] = 0x587a, [0x0633] = 0x587b,
  [0x0634] = 0x587c, [0x0635] = 0x587d, [0x0636] = 0x587f, [0x0637] = 0x5882,
  [0x0638] = 0x5884, [0x0639] = 0x5886, [0x063a] = 0x5887, [0x063b] = 0x5888,
  [0x063c] = 0x588a, [0x063d] = 0x588b, [0x063e] = 0x588c, [0x0640] = 0x588d,
  [0x0641] = 0x588e, [0x0642] = 0x588f, [0x0643] = 0x5890, [0x0644] = 0x5891,
  [0x0645] = 0x5894, [0x0646] = 0x5895, [0x0647] = 0x5896, [0x0648] = 0x5897,
  [0x0649] = 0x5898, [0x064a] = 0x589b, [0x064b] = 0x589c, [0x064c] = 0x589d,
  [0x064d] = 0x58a0, [0x064e] = 0x58a1, [0x064f] = 0x58a2, [0x0650] = 0x58a3,
  [0x0651] = 0x58a4, [0x0652] = 0x58a5, [0x0653] = 0x58a6, [0x0654] = 0x58a7,
  [0x0655] = 0x58aa, [0x0656] = 0x58ab, [0x0657] = 0x58ac, [0x0658] = 0x58ad,
  [0x0659] = 0x58ae, [0x065a] = 0x58af, [0x065b] = 0x58b0, [0x065c] = 0x58b1,
  [0x065d] = 0x58b2, [0x065e] = 0x58b3, [0x065f] = 0x58b4, [0x0660] = 0x58b5,
  [0x0661] = 0x58b6, [0x0662] = 0x58b7, [0x0663] = 0x58b8, [0x0664] = 0x58b9,
  [0x0665] = 0x58ba, [0x0666] = 0x58bb, [0x0667] = 0x58bd, [0x0668] = 0x58be,
  [0x0669] = 0x58bf, [0x066a] = 0x58c0, [0x066b] = 0x58c2, [0x066c] = 0x58c3,
  [0x066d] = 0x58c4, [0x066e] = 0x58c6, [0x066f] = 0x58c7, [0x0670] = 0x58c8,
  [0x0671] = 0x58c9, [0x0672] = 0x58ca, [0x0673] = 0x58cb, [0x0674] = 0x58cc,
  [0x0675] = 0x58cd, [0x0676] = 0x58ce, [0x0677] = 0x58cf, [0x0678] = 0x58d0,
  [0x0679] = 0x58d2, [0x067a] = 0x58d3, [0x067b] = 0x58d4, [0x067c] = 0x58d6,
  [0x067d] = 0x58d7, [0x067e] = 0x58d8, [0x067f] = 0x58d9, [0x0680] = 0x58da,
  [0x0681] = 0x58db, [0x0682] = 0x58dc, [0x0683] = 0x58dd, [0x0684] = 0x58de,
  [0x0685] = 0x58df, [0x0686] = 0x58e0, [0x0687] = 0x58e1, [0x0688] = 0x58e2,
  [0x0689] = 0x58e3, [0x068a] = 0x58e5, [0x068b] = 0x58e6, [0x068c] = 0x58e7,
  [0x068d] = 0x58e8, [0x068e] = 0x58e9, [0x068f] = 0x58ea, [0x0690] = 0x58ed,
  [0x0691] = 0x58ef, [0x0692] = 0x58f1, [0x0693] = 0x58f2, [0x0694] = 0x58f4,
  [0x0695] = 0x58f5, [0x0696] = 0x58f7, [0x0697] = 0x58f8, [0x0698] = 0x58fa,
  [0x0699] = 0x58fb, [0x069a] = 0x58fc, [0x069b] = 0x58fd, [0x069c] = 0x58fe,
  [0x069d] = 0x58ff, [0x069e] = 0x5900, [0x069f] = 0x5901, [0x06a0] = 0x5903,
  [0x06a1] = 0x5905, [0x06a2] = 0x5906, [0x06a3] = 0x5908, [0x06a4] = 0x5909,
  [0x06a5] = 0x590a, [0x06a6] = 0x590b, [0x06a7] = 0x590c, [0x06a8] = 0x590e,
  [0x06a9] = 0x5910, [0x06aa] = 0x5911, [0x06ab] = 0x5912, [0x06ac] = 0x5913,
  [0x06ad] = 0x5917, [0x06ae] = 0x5918, [0x06af] = 0x591b, [0x06b0] = 0x591d,
  [0x06b1] = 0x591e, [0x06b2] = 0x5920, [0x06b3] = 0x5921, [0x06b4] = 0x5922,
  [0x06b5] = 0x5923, [0x06b6] = 0x5926, [0x06b7] = 0x5928, [0x06b8] = 0x592c,
  [0x06b9] = 0x5930, [0x06ba] = 0x5932, [0x06bb] = 0x5933, [0x06bc] = 0x5935,
  [0x06bd] = 0x5936, [0x06be] = 0x593b, [0x06c0] = 0x593d, [0x06c1] = 0x593e,
  [0x06c2] = 0x593f, [0x06c3] = 0x5940, [0x06c4] = 0x5943, [0x06c5] = 0x5945,
  [0x06c6] = 0x5946, [0x06c7] = 0x594a, [0x06c8] = 0x594c, [0x06c9] = 0x594d,
  [0x06ca] = 0x5950, [0x06cb] = 0x5952, [0x06cc] = 0x5953, [0x06cd] = 0x5959,
  [0x06ce] = 0x595b, [0x06cf] = 0x595c, [0x06d0] = 0x595d, [0x06d1] = 0x595e,
  [0x06d2] = 0x595f, [0x06d3] = 0x5961, [0x06d4] = 0x5963, [0x06d5] = 0x5964,
  [0x06d6] = 0x5966, [0x06d7] = 0x5967, [0x06d8] = 0x5968, [0x06d9] = 0x5969,
  [0x06da] = 0x596a, [0x06db] = 0x596b, [0x06dc] = 0x596c, [0x06dd] = 0x596d,
  [0x06de] = 0x596e, [0x06df] = 0x596f, [0x06e0] = 0x5970, [0x06e1] = 0x5971,
  [0x06e2] = 0x5972, [0x06e3] = 0x5975, [0x06e4] = 0x5977, [0x06e5] = 0x597a,
  [0x06e6] = 0x597b, [0x06e7] = 0x597c, [0x06e8] = 0x597e, [0x06e9] = 0x597f,
  [0x06ea] = 0x5980, [0x06eb] = 0x5985, [0x06ec] = 0x5989, [0x06ed] = 0x598b,
  [0x06ee] = 0x598c, [0x06ef] = 0x598e, [0x06f0] = 0x598f, [0x06f1] = 0x5990,
  [0x06f2] = 0x5991, [0x06f3] = 0x5994, [0x06f4] = 0x5995, [0x06f5] = 0x5998,
  [0x06f6] = 0x599a, [0x06f7] = 0x599b, [0x06f8] = 0x599c, [0x06f9] = 0x599d,
  [0x06fa] = 0x599f, [0x06fb] = 0x59a0, [0x06fc] = 0x59a1, [0x06fd] = 0x59a2,
  [0x06fe] = 0x59a6, [0x0700] = 0x59a7, [0x0701] = 0x59ac, [0x0702] = 0x59ad,
  [0x0703] = 0x59b0, [0x0704] = 0x59b1, [0x0705] = 0x59b3, [0x0706] = 0x59b4,
  [0x0707] = 0x59b5, [0x0708] = 0x59b6, [0x0709] = 0x59b7, [0x070a] = 0x59b8,
  [0x070b] = 0x59ba, [0x070c] = 0x59bc, [0x070d] = 0x59bd, [0x070e] = 0x59bf,
  [0x070f] = 0x59c0, [0x0710] = 0x59c1, [0x0711] = 0x59c2, [0x0712] = 0x59c3,
  [0x0713] = 0x59c4, [0x0714] = 0x59c5, [0x0715] = 0x59c7, [0x0716] = 0x59c8,
  [0x0717] = 0x59c9, [0x0718] = 0x59cc, [0x0719] = 0x59cd, [0x071a] = 0x59ce,
  [0x071b] = 0x59cf, [0x071c] = 0x59d5, [0x071d] = 0x59d6, [0x071e] = 0x59d9,
  [0x071f] = 0x59db, [0x0720] = 0x59de, [0x0721] = 0x59df, [0x0722] = 0x59e0,
  [0x0723] = 0x59e1, [0x0724] = 0x59e2, [0x0725] = 0x59e4, [0x0726] = 0x59e6,
  [0x0727] = 0x59e7, [0x0728] = 0x59e9, [0x0729] = 0x59ea, [0x072a] = 0x59eb,
  [0x072b] = 0x59ed, [0x072c] = 0x59ee, [0x072d] = 0x59ef, [0x072e] = 0x59f0,
  [0x072f] = 0x59f1, [0x0730] = 0x59f2, [0x0731] = 0x59f3, [0x0732] = 0x59f4,
  [0x0733] = 0x59f5, [0x0734] = 0x59f6, [0x0735] = 0x59f7, [0x0736] = 0x59f8,
  [0x0737] = 0x59fa, [0x0738] = 0x59fc, [0x0739] = 0x59fd, [0x073a] = 0x59fe,
  [0x073b] = 0x5a00, [0x073c] = 0x5a02, [0x073d] = 0x5a0a, [0x073e] = 0x5a0b,
  [0x073f] = 0x5a0d, [0x0740] = 0x5a0e, [0x0741] = 0x5a0f, [0x0742] = 0x5a10,
  [0x0743] = 0x5a12, [0x0744] = 0x5a14, [0x0745] = 0x5a15, [0x0746] = 0x5a16,
  [0x0747] = 0x5a17, [0x0748] = 0x5a19, [0x0749] = 0x5a1a, [0x074a] = 0x5a1b,
  [0x074b] = 0x5a1d, [0x074c] = 0x5a1e, [0x074d] = 0x5a21, [0x074e] = 0x5a22,
  [0x074f] = 0x5a24, [0x0750] = 0x5a26, [0x0751] = 0x5a27, [0x0752] = 0x5a28,
  [0x0753] = 0x5a2a, [0x0754] = 0x5a2b, [0x0755] = 0x5a2c, [0x0756] = 0x5a2d,
  [0x0757] = 0x5a2e, [0x0758] = 0x5a2f, [0x0759] = 0x5a30, [0x075a] = 0x5a33,
  [0x075b] = 0x5a35, [0x075c] = 0x5a37, [0x075d] = 0x5a38, [0x075e] = 0x5a39,
  [0x075f] = 0x5a3a, [0x0760] = 0x5a3b, [0x0761] = 0x5a3d, [0x0762] = 0x5a3e,
  [0x0763] = 0x5a3f, [0x0764] = 0x5a41, [0x0765] = 0x5a42, [0x0766] = 0x5a43,
  [0x0767] = 0x5a44, [0x0768] = 0x5a45, [0x0769] = 0x5a47, [0x076a] = 0x5a48,
  [0x076b] = 0x5a4b, [0x076c] = 0x5a4c, [0x076d] = 0x5a4d, [0x076e] = 0x5a4e,
  [0x076f] = 0x5a4f, [0x0770] = 0x5a50, [0x0771] = 0x5a51, [0x0772] = 0x5a52,
  [0x0773] = 0x5a53, [0x0774] = 0x5a54, [0x0775] = 0x5a56, [0x0776] = 0x5a57,
  [0x0777] = 0x5a58, [0x0778] = 0x5a59, [0x0779] = 0x5a5b, [0x077a] = 0x5a5c,
  [0x077b] = 0x5a5d, [0x077c] = 0x5a5e, [0x077d] = 0x5a5f, [0x077e] = 0x5a60,
  [0x0780] = 0x5a61, [0x0781] = 0x5a63, [0x0782] = 0x5a64, [0x0783] = 0x5a65,
  [0x0784] = 0x5a66, [0x0785] = 0x5a68, [0x0786] = 0x5a69, [0x0787] = 0x5a6b,
  [0x0788] = 0x5a6c, [0x0789] = 0x5a6d, [0x078a] = 0x5a6e, [0x078b] = 0x5a6f,
  [0x078c] = 0x5a70, [0x078d] = 0x5a71, [0x078e] = 0x5a72, [0x078f] = 0x5a73,
  [0x0790] = 0x5a78, [0x0791] = 0x5a79, [0x0792] = 0x5a7b, [0x0793] = 0x5a7c,
  [0x0794] = 0x5a7d, [0x0795] = 0x5a7e, [0x0796] = 0x5a80, [0x0797] = 0x5a81,
  [0x0798] = 0x5a82, [0x0799] = 0x5a83, [0x079a] = 0x5a84, [0x079b] = 0x5a85,
  [0x079c] = 0x5a86, [0x079d] = 0x5a87, [0x079e] = 0x5a88, [0x079f] = 0x5a89,
  [0x07a0] = 0x5a8a, [0x07a1] = 0x5a8b, [0x07a2] = 0x5a8c, [0x07a3] = 0x5a8d,
  [0x07a4] = 0x5a8e, [0x07a5] = 0x5a8f, [0x07a6] = 0x5a90, [0x07a7] = 0x5a91,
  [0x07a8] = 0x5a93, [0x07a9] = 0x5a94, [0x07aa] = 0x5a95, [0x07ab] = 0x5a96,
  [0x07ac] = 0x5a97, [0x07ad] = 0x5a98, [0x07ae] = 0x5a99, [0x07af] = 0x5a9c,
  [0x07b0] = 0x5a9d, [0x07b1] = 0x5a9e, [0x07b2] = 0x5a9f, [0x07b3] = 0x5aa0,
  [0x07b4] = 0x5aa1, [0x07b5] = 0x5aa2, [0x07b6] = 0x5aa3, [0x07b7] = 0x5aa4,
  [0x07b8] = 0x5aa5, [0x07b9] = 0x5aa6, [0x07ba] = 0x5aa7, [0x07bb] = 0x5aa8,
  [0x07bc] = 0x5aa9, [0x07bd] = 0x5aab, [0x07be] = 0x5aac, [0x07c0] = 0x5aad,
  [0x07c1] = 0x5aae, [0x07c2] = 0x5aaf, [0x07c3] = 0x5ab0, [0x07c4] = 0x5ab1,
  [0x07c5] = 0x5ab4, [0x07c6] = 0x5ab6, [0x07c7] = 0x5ab7, [0x07c8] = 0x5ab9,
  [0x07c9] = 0x5aba, [0x07ca] = 0x5abb, [0x07cb] = 0x5abc, [0x07cc] = 0x5abd,
  [0x07cd] = 0x5abf, [0x07ce] = 0x5ac0, [0x07cf] = 0x5ac3, [0x07d0] = 0x5ac4,
  [0x07d1] = 0x5ac5, [0x07d2] = 0x5ac6, [0x07d3] = 0x5ac7, [0x07d4] = 0x5ac8,
  [0x07d5] = 0x5aca, [0x07d6] = 0x5acb, [0x07d7] = 0x5acd, [0x07d8] = 0x5ace,
  [0x07d9] = 0x5acf, [0x07da] = 0x5ad0, [0x07db] = 0x5ad1, [0x07dc] = 0x5ad3,
  [0x07dd] = 0x5ad5, [0x07de] = 0x5ad7, [0x07df] = 0x5ad9, [0x07e0] = 0x5ada,
  [0x07e1] = 0x5adb, [0x07e2] = 0x5add, [0x07e3] = 0x5ade, [0x07e4] = 0x5adf,
  [0x07e5] = 0x5ae2, [0x07e6] = 0x5ae4, [0x07e7] = 0x5ae5, [0x07e8] = 0x5ae7,
  [0x07e9] = 0x5ae8, [0x07ea] = 0x5aea, [0x07eb] = 0x5aec, [0x07ec] = 0x5aed,
  [0x07ed] = 0x5aee, [0x07ee] = 0x5aef, [0x07ef] = 0x5af0, [0x07f0] = 0x5af2,
  [0x07f1] = 0x5af3, [0x07f2] = 0x5af4, [0x07f3] = 0x5af5, [0x07f4] = 0x5af6,
  [0x07f5] = 0x5af7, [0x07f6] = 0x5af8, [0x07f7] = 0x5af9, [0x07f8] = 0x5afa,
  [0x07f9] = 0x5afb, [0x07fa] = 0x5afc, [0x07fb] = 0x5afd, [0x07fc] = 0x5afe,
  [0x07fd] = 0x5aff, [0x07fe] = 0x5b00, [0x07ff] = 0x5b01, [0x0800] = 0x5b02,
  [0x0801] = 0x5b03, [0x0802] = 0x5b04, [0x0803] = 0x5b05, [0x0804] = 0x5b06,
  [0x0805] = 0x5b07, [0x0806] = 0x5b08, [0x0807] = 0x5b0a, [0x0808] = 0x5b0b,
  [0x0809] = 0x5b0c, [0x080a] = 0x5b0d, [0x080b] = 0x5b0e, [0x080c] = 0x5b0f,
  [0x080d] = 0x5b10, [0x080e] = 0x5b11, [0x080f] = 0x5b12, [0x0810] = 0x5b13,
  [0x0811] = 0x5b14, [0x0812] = 0x5b15, [0x0813] = 0x5b18, [0x0814] = 0x5b19,
  [0x0815] = 0x5b1a, [0x0816] = 0x5b1b, [0x0817] = 0x5b1c, [0x0818] = 0x5b1d,
  [0x0819] = 0x5b1e, [0x081a] = 0x5b1f, [0x081b] = 0x5b20, [0x081c] = 0x5b21,
  [0x081d] = 0x5b22, [0x081e] = 0x5b23, [0x081f] = 0x5b24, [0x0820] = 0x5b25,
  [0x0821] = 0x5b26, [0x0822] = 0x5b27, [0x0823] = 0x5b28, [0x0824] = 0x5b29,
  [0x0825] = 0x5b2a, [0x0826] = 0x5b2b, [0x0827] = 0x5b2c, [0x0828] = 0x5b2d,
  [0x0829] = 0x5b2e, [0x082a] = 0x5b2f, [0x082b] = 0x5b30, [0x082c] = 0x5b31,
  [0x082d] = 0x5b33, [0x082e] = 0x5b35, [0x082f] = 0x5b36, [0x0830] = 0x5b38,
  [0x0831] = 0x5b39, [0x0832] = 0x5b3a, [0x0833] = 0x5b3b, [0x0834] = 0x5b3c,
  [0x0835] = 0x5b3d, [0x0836] = 0x5b3e, [0x0837] = 0x5b3f, [0x0838] = 0x5b41,
  [0x0839] = 0x5b42, [0x083a] = 0x5b43, [0x083b] = 0x5b44, [0x083c] = 0x5b45,
  [0x083d] = 0x5b46, [0x083e] = 0x5b47, [0x0840] = 0x5b48, [0x0841] = 0x5b49,
  [0x0842] = 0x5b4a, [0x0843] = 0x5b4b, [0x0844] = 0x5b4c, [0x0845] = 0x5b4d,
  [0x0846] = 0x5b4e, [0x0847] = 0x5b4f, [0x0848] = 0x5b52, [0x0849] = 0x5b56,
  [0x084a] = 0x5b5e, [0x084b] = 0x5b60, [0x084c] = 0x5b61, [0x084d] = 0x5b67,
  [0x084e] = 0x5b68, [0x084f] = 0x5b6b, [0x0850] = 0x5b6d, [0x0851] = 0x5b6e,
  [0x0852] = 0x5b6f, [0x0853] = 0x5b72, [0x0854] = 0x5b74, [0x0855] = 0x5b76,
  [0x0856] = 0x5b77, [0x0857] = 0x5b78, [0x0858] = 0x5b79, [0x0859] = 0x5b7b,
  [0x085a] = 0x5b7c, [0x085b] = 0x5b7e, [0x085c] = 0x5b7f, [0x085d] = 0x5b82,
  [0x085e] = 0x5b86, [0x085f] = 0x5b8a, [0x0860] = 0x5b8d, [0x0861] = 0x5b8e,
  [0x0862] = 0x5b90, [0x0863] = 0x5b91, [0x0864] = 0x5b92, [0x0865] = 0x5b94,
  [0x0866] = 0x5b96, [0x0867] = 0x5b9f, [0x0868] = 0x5ba7, [0x0869] = 0x5ba8,
  [0x086a] = 0x5ba9, [0x086b] = 0x5bac, [0x086c] = 0x5bad, [0x086d] = 0x5bae,
  [0x086e] = 0x5baf, [0x086f] = 0x5bb1, [0x0870] = 0x5bb2, [0x0871] = 0x5bb7,
  [0x0872] = 0x5bba, [0x0873] = 0x5bbb, [0x0874] = 0x5bbc, [0x0875] = 0x5bc0,
  [0x0876] = 0x5bc1, [0x0877] = 0x5bc3, [0x0878] = 0x5bc8, [0x0879] = 0x5bc9,
  [0x087a] = 0x5bca, [0x087b] = 0x5bcb, [0x087c] = 0x5bcd, [0x087d] = 0x5bce,
  [0x087e] = 0x5bcf, [0x0880] = 0x5bd1, [0x0881] = 0x5bd4, [0x0882] = 0x5bd5,
  [0x0883] = 0x5bd6, [0x0884] = 0x5bd7, [0x0885] = 0x5bd8, [0x0886] = 0x5bd9,
  [0x0887] = 0x5bda, [0x0888] = 0x5bdb, [0x0889] = 0x5bdc, [0x088a] = 0x5be0,
  [0x088b] = 0x5be2, [0x088c] = 0x5be3, [0x088d] = 0x5be6, [0x088e] = 0x5be7,
  [0x088f] = 0x5be9, [0x0890] = 0x5bea, [0x0891] = 0x5beb, [0x0892] = 0x5bec,
  [0x0893] = 0x5bed, [0x0894] = 0x5bef, [0x0895] = 0x5bf1, [0x0896] = 0x5bf2,
  [0x0897] = 0x5bf3, [0x0898] = 0x5bf4, [0x0899] = 0x5bf5, [0x089a] = 0x5bf6,
  [0x089b] = 0x5bf7, [0x089c] = 0x5bfd, [0x089d] = 0x5bfe, [0x089e] = 0x5c00,
  [0x089f] = 0x5c02, [0x08a0] = 0x5c03, [0x08a1] = 0x5c05, [0x08a2] = 0x5c07,
  [0x08a3] = 0x5c08, [0x08a4] = 0x5c0b, [0x08a5] = 0x5c0c, [0x08a6] = 0x5c0d,
  [0x08a7] = 0x5c0e, [0x08a8] = 0x5c10, [0x08a9] = 0x5c12, [0x08aa] = 0x5c13,
  [0x08ab] = 0x5c17, [0x08ac] = 0x5c19, [0x08ad] = 0x5c1b, [0x08ae] = 0x5c1e,
  [0x08af] = 0x5c1f, [0x08b0] = 0x5c20, [0x08b1] = 0x5c21, [0x08b2] = 0x5c23,
  [0x08b3] = 0x5c26, [0x08b4] = 0x5c28, [0x08b5] = 0x5c29, [0x08b6] = 0x5c2a,
  [0x08b7] = 0x5c2b, [0x08b8] = 0x5c2d, [0x08b9] = 0x5c2e, [0x08ba] = 0x5c2f,
  [0x08bb] = 0x5c30, [0x08bc] = 0x5c32, [0x08bd] = 0x5c33, [0x08be] = 0x5c35,
  [0x08bf] = 0x5c36, [0x08c0] = 0x5c37, [0x08c1] = 0x5c43, [0x08c2] = 0x5c44,
  [0x08c3] = 0x5c46, [0x08c4] = 0x5c47, [0x08c5] = 0x5c4c, [0x08c6] = 0x5c4d,
  [0x08c7] = 0x5c52, [0x08c8] = 0x5c53, [0x08c9] = 0x5c54, [0x08ca] = 0x5c56,
  [0x08cb] = 0x5c57, [0x08cc] = 0x5c58, [0x08cd] = 0x5c5a, [0x08ce] = 0x5c5b,
  [0x08cf] = 0x5c5c, [0x08d0] = 0x5c5d, [0x08d1] = 0x5c5f, [0x08d2] = 0x5c62,
  [0x08d3] = 0x5c64, [0x08d4] = 0x5c67, [0x08d5] = 0x5c68, [0x08d6] = 0x5c69,
  [0x08d7] = 0x5c6a, [0x08d8] = 0x5c6b, [0x08d9] = 0x5c6c, [0x08da] = 0x5c6d,
  [0x08db] = 0x5c70, [0x08dc] = 0x5c72, [0x08dd] = 0x5c73, [0x08de] = 0x5c74,
  [0x08df] = 0x5c75, [0x08e0] = 0x5c76, [0x08e1] = 0x5c77, [0x08e2] = 0x5c78,
  [0x08e3] = 0x5c7b, [0x08e4] = 0x5c7c, [0x08e5] = 0x5c7d, [0x08e6] = 0x5c7e,
  [0x08e7] = 0x5c80, [0x08e8] = 0x5c83, [0x08e9] = 0x5c84, [0x08ea] = 0x5c85,
  [0x08eb] = 0x5c86, [0x08ec] = 0x5c87, [0x08ed] = 0x5c89, [0x08ee] = 0x5c8a,
  [0x08ef] = 0x5c8b, [0x08f0] = 0x5c8e, [0x08f1] = 0x5c8f, [0x08f2] = 0x5c92,
  [0x08f3] = 0x5c93, [0x08f4] = 0x5c95, [0x08f5] = 0x5c9d, [0x08f6] = 0x5c9e,
  [0x08f7] = 0x5c9f, [0x08f8] = 0x5ca0, [0x08f9] = 0x5ca1, [0x08fa] = 0x5ca4,
  [0x08fb] = 0x5ca5, [0x08fc] = 0x5ca6, [0x08fd] = 0x5ca7, [0x08fe] = 0x5ca8,
  [0x0900] = 0x5caa, [0x0901] = 0x5cae, [0x0902] = 0x5caf, [0x0903] = 0x5cb0,
  [0x0904] = 0x5cb2, [0x0905] = 0x5cb4, [0x0906] = 0x5cb6, [0x0907] = 0x5cb9,
  [0x0908] = 0x5cba, [0x0909] = 0x5cbb, [0x090a] = 0x5cbc, [0x090b] = 0x5cbe,
  [0x090c] = 0x5cc0, [0x090d] = 0x5cc2, [0x090e] = 0x5cc3, [0x090f] = 0x5cc5,
  [0x0910] = 0x5cc6, [0x0911] = 0x5cc7, [0x0912] = 0x5cc8, [0x0913] = 0x5cc9,
  [0x0914] = 0x5cca, [0x0915] = 0x5ccc, [0x0916] = 0x5ccd, [0x0917] = 0x5cce,
  [0x0918] = 0x5ccf, [0x0919] = 0x5cd0, [0x091a] = 0x5cd1, [0x091b] = 0x5cd3,
  [0x091c] = 0x5cd4, [0x091d] = 0x5cd5, [0x091e] = 0x5cd6, [0x091f] = 0x5cd7,
  [0x0920] = 0x5cd8, [0x0921] = 0x5cda, [0x0922] = 0x5cdb, [0x0923] = 0x5cdc,
  [0x0924] = 0x5cdd, [0x0925] = 0x5cde, [0x0926] = 0x5cdf, [0x0927] = 0x5ce0,
  [0x0928] = 0x5ce2, [0x0929] = 0x5ce3, [0x092a] = 0x5ce7, [0x092b] = 0x5ce9,
  [0x092c] = 0x5ceb, [0x092d] = 0x5cec, [0x092e] = 0x5cee, [0x092f] = 0x5cef,
  [0x0930] = 0x5cf1, [0x0931] = 0x5cf2, [0x0932] = 0x5cf3, [0x0933] = 0x5cf4,
  [0x0934] = 0x5cf5, [0x0935] = 0x5cf6, [0x0936] = 0x5cf7, [0x0937] = 0x5cf8,
  [0x0938] = 0x5cf9, [0x0939] = 0x5cfa, [0x093a] = 0x5cfc, [0x093b] = 0x5cfd,
  [0x093c] = 0x5cfe, [0x093d] = 0x5cff, [0x093e] = 0x5d00, [0x0940] = 0x5d01,
  [0x0941] = 0x5d04, [0x0942] = 0x5d05, [0x0943] = 0x5d08, [0x0944] = 0x5d09,
  [0x0945] = 0x5d0a, [0x0946] = 0x5d0b, [0x0947] = 0x5d0c, [0x0948] = 0x5d0d,
  [0x0949] = 0x5d0f, [0x094a] = 0x5d10, [0x094b] = 0x5d11, [0x094c] = 0x5d12,
  [0x094d] = 0x5d13, [0x094e] = 0x5d15, [0x094f] = 0x5d17, [0x0950] = 0x5d18,
  [0x0951] = 0x5d19, [0x0952] = 0x5d1a, [0x0953] = 0x5d1c, [0x0954] = 0x5d1d,
  [0x0955] = 0x5d1f, [0x0956] = 0x5d20, [0x0957] = 0x5d21, [0x0958] = 0x5d22,
  [0x0959] = 0x5d23, [0x095a] = 0x5d25, [0x095b] = 0x5d28, [0x095c] = 0x5d2a,
  [0x095d] = 0x5d2b, [0x095e] = 0x5d2c, [0x095f] = 0x5d2f, [0x0960] = 0x5d30,
  [0x0961] = 0x5d31, [0x0962] = 0x5d32, [0x0963] = 0x5d33, [0x0964] = 0x5d35,
  [0x0965] = 0x5d36, [0x0966] = 0x5d37, [0x0967] = 0x5d38, [0x0968] = 0x5d39,
  [0x0969] = 0x5d3a, [0x096a] = 0x5d3b, [0x096b] = 0x5d3c, [0x096c] = 0x5d3f,
  [0x096d] = 0x5d40, [0x096e] = 0x5d41, [0x096f] = 0x5d42, [0x0970] = 0x5d43,
  [0x0971] = 0x5d44, [0x0972] = 0x5d45, [0x0973] = 0x5d46, [0x0974] = 0x5d48,
  [0x0975] = 0x5d49, [0x0976] = 0x5d4d, [0x0977] = 0x5d4e, [0x0978] = 0x5d4f,
  [0x0979] = 0x5d50, [0x097a] = 0x5d51, [0x097b] = 0x5d52, [0x097c] = 0x5d53,
  [0x097d] = 0x5d54, [0x097e] = 0x5d55, [0x097f] = 0x5d56, [0x0980] = 0x5d57,
  [0x0981] = 0x5d59, [0x0982] = 0x5d5a, [0x0983] = 0x5d5c, [0x0984] = 0x5d5e,
  [0x0985] = 0x5d5f, [0x0986] = 0x5d60, [0x0987] = 0x5d61, [0x0988] = 0x5d62,
  [0x0989] = 0x5d63, [0x098a] = 0x5d64, [0x098b] = 0x5d65, [0x098c] = 0x5d66,
  [0x098d] = 0x5d67, [0x098e] = 0x5d68, [0x098f] = 0x5d6a, [0x0990] = 0x5d6d,
  [0x0991] = 0x5d6e, [0x0992] = 0x5d70, [0x0993] = 0x5d71, [0x0994] = 0x5d72,
  [0x0995] = 0x5d73, [0x0996] = 0x5d75, [0x0997] = 0x5d76, [0x0998] = 0x5d77,
  [0x0999] = 0x5d78, [0x099a] = 0x5d79, [0x099b] = 0x5d7a, [0x099c] = 0x5d7b,
  [0x099d] = 0x5d7c, [0x099e] = 0x5d7d, [0x099f] = 0x5d7e, [0x09a0] = 0x5d7f,
  [0x09a1] = 0x5d80, [0x09a2] = 0x5d81, [0x09a3] = 0x5d83, [0x09a4] = 0x5d84,
  [0x09a5] = 0x5d85, [0x09a6] = 0x5d86, [0x09a7] = 0x5d87, [0x09a8] = 0x5d88,
  [0x09a9] = 0x5d89, [0x09aa] = 0x5d8a, [0x09ab] = 0x5d8b, [0x09ac] = 0x5d8c,
  [0x09ad] = 0x5d8d, [0x09ae] = 0x5d8e, [0x09af] = 0x5d8f, [0x09b0] = 0x5d90,
  [0x09b1] = 0x5d91, [0x09b2] = 0x5d92, [0x09b3] = 0x5d93, [0x09b4] = 0x5d94,
  [0x09b5] = 0x5d95, [0x09b6] = 0x5d96, [0x09b7] = 0x5d97, [0x09b8] = 0x5d98,
  [0x09b9] = 0x5d9a, [0x09ba] = 0x5d9b, [0x09bb] = 0x5d9c, [0x09bc] = 0x5d9e,
  [0x09bd] = 0x5d9f, [0x09be] = 0x5da0, [0x09c0] = 0x5da1, [0x09c1] = 0x5da2,
  [0x09c2] = 0x5da3, [0x09c3] = 0x5da4, [0x09c4] = 0x5da5, [0x09c5] = 0x5da6,
  [0x09c6] = 0x5da7, [0x09c7] = 0x5da8, [0x09c8] = 0x5da9, [0x09c9] = 0x5daa,
  [0x09ca] = 0x5dab, [0x09cb] = 0x5dac, [0x09cc] = 0x5dad, [0x09cd] = 0x5dae,
  [0x09ce] = 0x5daf, [0x09cf] = 0x5db0, [0x09d0] = 0x5db1, [0x09d1] = 0x5db2,
  [0x09d2] = 0x5db3, [0x09d3] = 0x5db4, [0x09d4] = 0x5db5, [0x09d5] = 0x5db6,
  [0x09d6] = 0x5db8, [0x09d7] = 0x5db9, [0x09d8] = 0x5dba, [0x09d9] = 0x5dbb,
  [0x09da] = 0x5dbc, [0x09db] = 0x5dbd, [0x09dc] = 0x5dbe, [0x09dd] = 0x5dbf,
  [0x09de] = 0x5dc0, [0x09df] = 0x5dc1, [0x09e0] = 0x5dc2, [0x09e1] = 0x5dc3,
  [0x09e2] = 0x5dc4, [0x09e3] = 0x5dc6, [0x09e4] = 0x5dc7, [0x09e5] = 0x5dc8,
  [0x09e6] = 0x5dc9, [0x09e7] = 0x5dca, [0x09e8] = 0x5dcb, [0x09e9] = 0x5dcc,
  [0x09ea] = 0x5dce, [0x09eb] = 0x5dcf, [0x09ec] = 0x5dd0, [0x09ed] = 0x5dd1,
  [0x09ee] = 0x5dd2, [0x09ef] = 0x5dd3, [0x09f0] = 0x5dd4, [0x09f1] = 0x5dd5,
  [0x09f2] = 0x5dd6, [0x09f3] = 0x5dd7, [0x09f4] = 0x5dd8, [0x09f5] = 0x5dd9,
  [0x09f6] = 0x5dda, [0x09f7] = 0x5ddc, [0x09f8] = 0x5ddf, [0x09f9] = 0x5de0,
  [0x09fa] = 0x5de3, [0x09fb] = 0x5de4, [0x09fc] = 0x5dea, [0x09fd] = 0x5dec,
  [0x09fe] = 0x5ded, [0x0a00] = 0x5df0, [0x0a01] = 0x5df5, [0x0a02] = 0x5df6,
  [0x0a03] = 0x5df8, [0x0a04] = 0x5df9, [0x0a05] = 0x5dfa, [0x0a06] = 0x5dfb,
  [0x0a07] = 0x5dfc, [0x0a08] = 0x5dff, [0x0a09] = 0x5e00, [0x0a0a] = 0x5e04,
  [0x0a0b] = 0x5e07, [0x0a0c] = 0x5e09, [0x0a0d] = 0x5e0a, [0x0a0e] = 0x5e0b,
  [0x0a0f] = 0x5e0d, [0x0a10] = 0x5e0e, [0x0a11] = 0x5e12, [0x0a12] = 0x5e13,
  [0x0a13] = 0x5e17, [0x0a14] = 0x5e1e, [0x0a15] = 0x5e1f, [0x0a16] = 0x5e20,
  [0x0a17] = 0x5e21, [0x0a18] = 0x5e22, [0x0a19] = 0x5e23, [0x0a1a] = 0x5e24,
  [0x0a1b] = 0x5e25, [0x0a1c] = 0x5e28, [0x0a1d] = 0x5e29, [0x0a1e] = 0x5e2a,
  [0x0a1f] = 0x5e2b, [0x0a20] = 0x5e2c, [0x0a21] = 0x5e2f, [0x0a22] = 0x5e30,
  [0x0a23] = 0x5e32, [0x0a24] = 0x5e33, [0x0a25] = 0x5e34, [0x0a26] = 0x5e35,
  [0x0a27] = 0x5e36, [0x0a28] = 0x5e39, [0x0a29] = 0x5e3a, [0x0a2a] = 0x5e3e,
  [0x0a2b] = 0x5e3f, [0x0a2c] = 0x5e40, [0x0a2d] = 0x5e41, [0x0a2e] = 0x5e43,
  [0x0a2f] = 0x5e46, [0x0a30] = 0x5e47, [0x0a31] = 0x5e48, [0x0a32] = 0x5e49,
  [0x0a33] = 0x5e4a, [0x0a34] = 0x5e4b, [0x0a35] = 0x5e4d, [0x0a36] = 0x5e4e,
  [0x0a37] = 0x5e4f, [0x0a38] = 0x5e50, [0x0a39] = 0x5e51, [0x0a3a] = 0x5e52,
  [0x0a3b] = 0x5e53, [0x0a3c] = 0x5e56, [0x0a3d] = 0x5e57, [0x0a3e] = 0x5e58,
  [0x0a3f] = 0x5e59, [0x0a40] = 0x5e5a, [0x0a41] = 0x5e5c, [0x0a42] = 0x5e5d,
  [0x0a43] = 0x5e5f, [0x0a44] = 0x5e60, [0x0a45] = 0x5e63, [0x0a46] = 0x5e64,
  [0x0a47] = 0x5e65, [0x0a48] = 0x5e66, [0x0a49] = 0x5e67, [0x0a4a] = 0x5e68,
  [0x0a4b] = 0x5e69, [0x0a4c] = 0x5e6a, [0x0a4d] = 0x5e6b, [0x0a4e] = 0x5e6c,
  [0x0a4f] = 0x5e6d, [0x0a50] = 0x5e6e, [0x0a51] = 0x5e6f, [0x0a52] = 0x5e70,
  [0x0a53] = 0x5e71, [0x0a54] = 0x5e75, [0x0a55] = 0x5e77, [0x0a56] = 0x5e79,
  [0x0a57] = 0x5e7e, [0x0a58] = 0x5e81, [0x0a59] = 0x5e82, [0x0a5a] = 0x5e83,
  [0x0a5b] = 0x5e85, [0x0a5c] = 0x5e88, [0x0a5d] = 0x5e89, [0x0a5e] = 0x5e8c,
  [0x0a5f] = 0x5e8d, [0x0a60] = 0x5e8e, [0x0a61] = 0x5e92, [0x0a62] = 0x5e98,
  [0x0a63] = 0x5e9b, [0x0a64] = 0x5e9d, [0x0a65] = 0x5ea1, [0x0a66] = 0x5ea2,
  [0x0a67] = 0x5ea3, [0x0a68] = 0x5ea4, [0x0a69] = 0x5ea8, [0x0a6a] = 0x5ea9,
  [0x0a6b] = 0x5eaa, [0x0a6c] = 0x5eab, [0x0a6d] = 0x5eac, [0x0a6e] = 0x5eae,
  [0x0a6f] = 0x5eaf, [0x0a70] = 0x5eb0, [0x0a71] = 0x5eb1, [0x0a72] = 0x5eb2,
  [0x0a73] = 0x5eb4, [0x0a74] = 0x5eba, [0x0a75] = 0x5ebb, [0x0a76] = 0x5ebc,
  [0x0a77] = 0x5ebd, [0x0a78] = 0x5ebf, [0x0a79] = 0x5ec0, [0x0a7a] = 0x5ec1,
  [0x0a7b] = 0x5ec2, [0x0a7c] = 0x5ec3, [0x0a7d] = 0x5ec4, [0x0a7e] = 0x5ec5,
  [0x0a80] = 0x5ec6, [0x0a81] = 0x5ec7, [0x0a82] = 0x5ec8, [0x0a83] = 0x5ecb,
  [0x0a84] = 0x5ecc, [0x0a85] = 0x5ecd, [0x0a86] = 0x5ece, [0x0a87] = 0x5ecf,
  [0x0a88] = 0x5ed0, [0x0a89] = 0x5ed4, [0x0a8a] = 0x5ed5, [0x0a8b] = 0x5ed7,
  [0x0a8c] = 0x5ed8, [0x0a8d] = 0x5ed9, [0x0a8e] = 0x5eda, [0x0a8f] = 0x5edc,
  [0x0a90] = 0x5edd, [0x0a91] = 0x5ede, [0x0a92] = 0x5edf, [0x0a93] = 0x5ee0,
  [0x0a94] = 0x5ee1, [0x0a95] = 0x5ee2, [0x0a96] = 0x5ee3, [0x0a97] = 0x5ee4,
  [0x0a98] = 0x5ee5, [0x0a99] = 0x5ee6, [0x0a9a] = 0x5ee7, [0x0a9b] = 0x5ee9,
  [0x0a9c] = 0x5eeb, [0x0a9d] = 0x5eec, [0x0a9e] = 0x5eed, [0x0a9f] = 0x5eee,
  [0x0aa0] = 0x5eef, [0x0aa1] = 0x5ef0, [0x0aa2] = 0x5ef1, [0x0aa3] = 0x5ef2,
  [0x0aa4] = 0x5ef3, [0x0aa5] = 0x5ef5, [0x0aa6] = 0x5ef8, [0x0aa7] = 0x5ef9,
  [0x0aa8] = 0x5efb, [0x0aa9] = 0x5efc, [0x0aaa] = 0x5efd, [0x0aab] = 0x5f05,
  [0x0aac] = 0x5f06, [0x0aad] = 0x5f07, [0x0aae] = 0x5f09, [0x0aaf] = 0x5f0c,
  [0x0ab0] = 0x5f0d, [0x0ab1] = 0x5f0e, [0x0ab2] = 0x5f10, [0x0ab3] = 0x5f12,
  [0x0ab4] = 0x5f14, [0x0ab5] = 0x5f16, [0x0ab6] = 0x5f19, [0x0ab7] = 0x5f1a,
  [0x0ab8] = 0x5f1c, [0x0ab9] = 0x5f1d, [0x0aba] = 0x5f1e, [0x0abb] = 0x5f21,
  [0x0abc] = 0x5f22, [0x0abd] = 0x5f23, [0x0abe] = 0x5f24, [0x0ac0] = 0x5f28,
  [0x0ac1] = 0x5f2b, [0x0ac2] = 0x5f2c, [0x0ac3] = 0x5f2e, [0x0ac4] = 0x5f30,
  [0x0ac5] = 0x5f32, [0x0ac6] = 0x5f33, [0x0ac7] = 0x5f34, [0x0ac8] = 0x5f35,
  [0x0ac9] = 0x5f36, [0x0aca] = 0x5f37, [0x0acb] = 0x5f38, [0x0acc] = 0x5f3b,
  [0x0acd] = 0x5f3d, [0x0ace] = 0x5f3e, [0x0acf] = 0x5f3f, [0x0ad0] = 0x5f41,
  [0x0ad1] = 0x5f42, [0x0ad2] = 0x5f43, [0x0ad3] = 0x5f44, [0x0ad4] = 0x5f45,
  [0x0ad5] = 0x5f46, [0x0ad6] = 0x5f47, [0x0ad7] = 0x5f48, [0x0ad8] = 0x5f49,
  [0x0ad9] = 0x5f4a, [0x0ada] = 0x5f4b, [0x0adb] = 0x5f4c, [0x0adc] = 0x5f4d,
  [0x0add] = 0x5f4e, [0x0ade] = 0x5f4f, [0x0adf] = 0x5f51, [0x0ae0] = 0x5f54,
  [0x0ae1] = 0x5f59, [0x0ae2] = 0x5f5a, [0x0ae3] = 0x5f5b, [0x0ae4] = 0x5f5c,
  [0x0ae5] = 0x5f5e, [0x0ae6] = 0x5f5f, [0x0ae7] = 0x5f60, [0x0ae8] = 0x5f63,
  [0x0ae9] = 0x5f65, [0x0aea] = 0x5f67, [0x0aeb] = 0x5f68, [0x0aec] = 0x5f6b,
  [0x0aed] = 0x5f6e, [0x0aee] = 0x5f6f, [0x0aef] = 0x5f72, [0x0af0] = 0x5f74,
  [0x0af1] = 0x5f75, [0x0af2] = 0x5f76, [0x0af3] = 0x5f78, [0x0af4] = 0x5f7a,
  [0x0af5] = 0x5f7d, [0x0af6] = 0x5f7e, [0x0af7] = 0x5f7f, [0x0af8] = 0x5f83,
  [0x0af9] = 0x5f86, [0x0afa] = 0x5f8d, [0x0afb] = 0x5f8e, [0x0afc] = 0x5f8f,
  [0x0afd] = 0x5f91, [0x0afe] = 0x5f93, [0x0aff] = 0x5f94, [0x0b00] = 0x5f96,
  [0x0b01] = 0x5f9a, [0x0b02] = 0x5f9b, [0x0b03] = 0x5f9d, [0x0b04] = 0x5f9e,
  [0x0b05] = 0x5f9f, [0x0b06] = 0x5fa0, [0x0b07] = 0x5fa2, [0x0b08] = 0x5fa3,
  [0x0b09] = 0x5fa4, [0x0b0a] = 0x5fa5, [0x0b0b] = 0x5fa6, [0x0b0c] = 0x5fa7,
  [0x0b0d] = 0x5fa9, [0x0b0e] = 0x5fab, [0x0b0f] = 0x5fac, [0x0b10] = 0x5faf,
  [0x0b11] = 0x5fb0, [0x0b12] = 0x5fb1, [0x0b13] = 0x5fb2, [0x0b14] = 0x5fb3,
  [0x0b15] = 0x5fb4, [0x0b16] = 0x5fb6, [0x0b17] = 0x5fb8, [0x0b18] = 0x5fb9,
  [0x0b19] = 0x5fba, [0x0b1a] = 0x5fbb, [0x0b1b] = 0x5fbe, [0x0b1c] = 0x5fbf,
  [0x0b1d] = 0x5fc0, [0x0b1e] = 0x5fc1, [0x0b1f] = 0x5fc2, [0x0b20] = 0x5fc7,
  [0x0b21] = 0x5fc8, [0x0b22] = 0x5fca, [0x0b23] = 0x5fcb, [0x0b24] = 0x5fce,
  [0x0b25] = 0x5fd3, [0x0b26] = 0x5fd4, [0x0b27] = 0x5fd5, [0x0b28] = 0x5fda,
  [0x0b29] = 0x5fdb, [0x0b2a] = 0x5fdc, [0x0b2b] = 0x5fde, [0x0b2c] = 0x5fdf,
  [0x0b2d] = 0x5fe2, [0x0b2e] = 0x5fe3, [0x0b2f] = 0x5fe5, [0x0b30] = 0x5fe6,
  [0x0b31] = 0x5fe8, [0x0b32] = 0x5fe9, [0x0b33] = 0x5fec, [0x0b34] = 0x5fef,
  [0x0b35] = 0x5ff0, [0x0b36] = 0x5ff2, [0x0b37] = 0x5ff3, [0x0b38] = 0x5ff4,
  [0x0b39] = 0x5ff6, [0x0b3a] = 0x5ff7, [0x0b3b] = 0x5ff9, [0x0b3c] = 0x5ffa,
  [0x0b3d] = 0x5ffc, [0x0b3e] = 0x6007, [0x0b40] = 0x6008, [0x0b41] = 0x6009,
  [0x0b42] = 0x600b, [0x0b43] = 0x600c, [0x0b44] = 0x6010, [0x0b45] = 0x6011,
  [0x0b46] = 0x6013, [0x0b47] = 0x6017, [0x0b48] = 0x6018, [0x0b49] = 0x601a,
  [0x0b4a] = 0x601e, [0x0b4b] = 0x601f, [0x0b4c] = 0x6022, [0x0b4d] = 0x6023,
  [0x0b4e] = 0x6024, [0x0b4f] = 0x602c, [0x0b50] = 0x602d, [0x0b51] = 0x602e,
  [0x0b52] = 0x6030, [0x0b53] = 0x6031, [0x0b54] = 0x6032, [0x0b55] = 0x6033,
  [0x0b56] = 0x6034, [0x0b57] = 0x6036, [0x0b58] = 0x6037, [0x0b59] = 0x6038,
  [0x0b5a] = 0x6039, [0x0b5b] = 0x603a, [0x0b5c] = 0x603d, [0x0b5d] = 0x603e,
  [0x0b5e] = 0x6040, [0x0b5f] = 0x6044, [0x0b60] = 0x6045, [0x0b61] = 0x6046,
  [0x0b62] = 0x6047, [0x0b63] = 0x6048, [0x0b64] = 0x6049, [0x0b65] = 0x604a,
  [0x0b66] = 0x604c, [0x0b67] = 0x604e, [0x0b68] = 0x604f, [0x0b69] = 0x6051,
  [0x0b6a] = 0x6053, [0x0b6b] = 0x6054, [0x0b6c] = 0x6056, [0x0b6d] = 0x6057,
  [0x0b6e] = 0x6058, [0x0b6f] = 0x605b, [0x0b70] = 0x605c, [0x0b71] = 0x605e,
  [0x0b72] = 0x605f, [0x0b73] = 0x6060, [0x0b74] = 0x6061, [0x0b75] = 0x6065,
  [0x0b76] = 0x6066, [0x0b77] = 0x606e, [0x0b78] = 0x6071, [0x0b79] = 0x6072,
  [0x0b7a] = 0x6074, [0x0b7b] = 0x6075, [0x0b7c] = 0x6077, [0x0b7d] = 0x607e,
  [0x0b7e] = 0x6080, [0x0b80] = 0x6081, [0x0b81] = 0x6082, [0x0b82] = 0x6085,
  [0x0b83] = 0x6086, [0x0b84] = 0x6087, [0x0b85] = 0x6088, [0x0b86] = 0x608a,
  [0x0b87] = 0x608b, [0x0b88] = 0x608e, [0x0b89] = 0x608f, [0x0b8a] = 0x6090,
  [0x0b8b] = 0x6091, [0x0b8c] = 0x6093, [0x0b8d] = 0x6095, [0x0b8e] = 0x6097,
  [0x0b8f] = 0x6098, [0x0b90] = 0x6099, [0x0b91] = 0x609c, [0x0b92] = 0x609e,
  [0x0b93] = 0x60a1, [0x0b94] = 0x60a2, [0x0b95] = 0x60a4, [0x0b96] = 0x60a5,
  [0x0b97] = 0x60a7, [0x0b98] = 0x60a9, [0x0b99] = 0x60aa, [0x0b9a] = 0x60ae,
  [0x0b9b] = 0x60b0, [0x0b9c] = 0x60b3, [0x0b9d] = 0x60b5, [0x0b9e] = 0x60b6,
  [0x0b9f] = 0x60b7, [0x0ba0] = 0x60b9, [0x0ba1] = 0x60ba, [0x0ba2] = 0x60bd,
  [0x0ba3] = 0x60be, [0x0ba4] = 0x60bf, [0x0ba5] = 0x60c0, [0x0ba6] = 0x60c1,
  [0x0ba7] = 0x60c2, [0x0ba8] = 0x60c3, [0x0ba9] = 0x60c4, [0x0baa] = 0x60c7,
  [0x0bab] = 0x60c8, [0x0bac] = 0x60c9, [0x0bad] = 0x60cc, [0x0bae] = 0x60cd,
  [0x0baf] = 0x60ce, [0x0bb0] = 0x60cf, [0x0bb1] = 0x60d0, [0x0bb2] = 0x60d2,
  [0x0bb3] = 0x60d3, [0x0bb4] = 0x60d4, [0x0bb5] = 0x60d6, [0x0bb6] = 0x60d7,
  [0x0bb7] = 0x60d9, [0x0bb8] = 0x60db, [0x0bb9] = 0x60de, [0x0bba] = 0x60e1,
  [0x0bbb] = 0x60e2, [0x0bbc] = 0x60e3, [0x0bbd] = 0x60e4, [0x0bbe] = 0x60e5,
  [0x0bbf] = 0x60ea, [0x0bc0] = 0x60f1, [0x0bc1] = 0x60f2, [0x0bc2] = 0x60f5,
  [0x0bc3] = 0x60f7, [0x0bc4] = 0x60f8, [0x0bc5] = 0x60fb, [0x0bc6] = 0x60fc,
  [0x0bc7] = 0x60fd, [0x0bc8] = 0x60fe, [0x0bc9] = 0x60ff, [0x0bca] = 0x6102,
  [0x0bcb] = 0x6103, [0x0bcc] = 0x6104, [0x0bcd] = 0x6105, [0x0bce] = 0x6107,
  [0x0bcf] = 0x610a, [0x0bd0] = 0x610b, [0x0bd1] = 0x610c, [0x0bd2] = 0x6110,
  [0x0bd3] = 0x6111, [0x0bd4] = 0x6112, [0x0bd5] = 0x6113, [0x0bd6] = 0x6114,
  [0x0bd7] = 0x6116, [0x0bd8] = 0x6117, [0x0bd9] = 0x6118, [0x0bda] = 0x6119,
  [0x0bdb] = 0x611b, [0x0bdc] = 0x611c, [0x0bdd] = 0x611d, [0x0bde] = 0x611e,
  [0x0bdf] = 0x6121, [0x0be0] = 0x6122, [0x0be1] = 0x6125, [0x0be2] = 0x6128,
  [0x0be3] = 0x6129, [0x0be4] = 0x612a, [0x0be5] = 0x612c, [0x0be6] = 0x612d,
  [0x0be7] = 0x612e, [0x0be8] = 0x612f, [0x0be9] = 0x6130, [0x0bea] = 0x6131,
  [0x0beb] = 0x6132, [0x0bec] = 0x6133, [0x0bed] = 0x6134, [0x0bee] = 0x6135,
  [0x0bef] = 0x6136, [0x0bf0] = 0x6137, [0x0bf1] = 0x6138, [0x0bf2] = 0x6139,
  [0x0bf3] = 0x613a, [0x0bf4] = 0x613b, [0x0bf5] = 0x613c, [0x0bf6] = 0x613d,
  [0x0bf7] = 0x613e, [0x0bf8] = 0x6140, [0x0bf9] = 0x6141, [0x0bfa] = 0x6142,
  [0x0bfb] = 0x6143, [0x0bfc] = 0x6144, [0x0bfd] = 0x6145, [0x0bfe] = 0x6146,
  [0x0c00] = 0x6147, [0x0c01] = 0x6149, [0x0c02] = 0x614b, [0x0c03] = 0x614d,
  [0x0c04] = 0x614f, [0x0c05] = 0x6150, [0x0c06] = 0x6152, [0x0c07] = 0x6153,
  [0x0c08] = 0x6154, [0x0c09] = 0x6156, [0x0c0a] = 0x6157, [0x0c0b] = 0x6158,
  [0x0c0c] = 0x6159, [0x0c0d] = 0x615a, [0x0c0e] = 0x615b, [0x0c0f] = 0x615c,
  [0x0c10] = 0x615e, [0x0c11] = 0x615f, [0x0c12] = 0x6160, [0x0c13] = 0x6161,
  [0x0c14] = 0x6163, [0x0c15] = 0x6164, [0x0c16] = 0x6165, [0x0c17] = 0x6166,
  [0x0c18] = 0x6169, [0x0c19] = 0x616a, [0x0c1a] = 0x616b, [0x0c1b] = 0x616c,
  [0x0c1c] = 0x616d, [0x0c1d] = 0x616e, [0x0c1e] = 0x616f, [0x0c1f] = 0x6171,
  [0x0c20] = 0x6172, [0x0c21] = 0x6173, [0x0c22] = 0x6174, [0x0c23] = 0x6176,
  [0x0c24] = 0x6178, [0x0c25] = 0x6179, [0x0c26] = 0x617a, [0x0c27] = 0x617b,
  [0x0c28] = 0x617c, [0x0c29] = 0x617d, [0x0c2a] = 0x617e, [0x0c2b] = 0x617f,
  [0x0c2c] = 0x6180, [0x0c2d] = 0x6181, [0x0c2e] = 0x6182, [0x0c2f] = 0x6183,
  [0x0c30] = 0x6184, [0x0c31] = 0x6185, [0x0c32] = 0x6186, [0x0c33] = 0x6187,
  [0x0c34] = 0x6188, [0x0c35] = 0x6189, [0x0c36] = 0x618a, [0x0c37] = 0x618c,
  [0x0c38] = 0x618d, [0x0c39] = 0x618f, [0x0c3a] = 0x6190, [0x0c3b] = 0x6191,
  [0x0c3c] = 0x6192, [0x0c3d] = 0x6193, [0x0c3e] = 0x6195, [0x0c40] = 0x6196,
  [0x0c41] = 0x6197, [0x0c42] = 0x6198, [0x0c43] = 0x6199, [0x0c44] = 0x619a,
  [0x0c45] = 0x619b, [0x0c46] = 0x619c, [0x0c47] = 0x619e, [0x0c48] = 0x619f,
  [0x0c49] = 0x61a0, [0x0c4a] = 0x61a1, [0x0c4b] = 0x61a2, [0x0c4c] = 0x61a3,
  [0x0c4d] = 0x61a4, [0x0c4e] = 0x61a5, [0x0c4f] = 0x61a6, [0x0c50] = 0x61aa,
  [0x0c51] = 0x61ab, [0x0c52] = 0x61ad, [0x0c53] = 0x61ae, [0x0c54] = 0x61af,
  [0x0c55] = 0x61b0, [0x0c56] = 0x61b1, [0x0c57] = 0x61b2, [0x0c58] = 0x61b3,
  [0x0c59] = 0x61b4, [0x0c5a] = 0x61b5, [0x0c5b] = 0x61b6, [0x0c5c] = 0x61b8,
  [0x0c5d] = 0x61b9, [0x0c5e] = 0x61ba, [0x0c5f] = 0x61bb, [0x0c60] = 0x61bc,
  [0x0c61] = 0x61bd, [0x0c62] = 0x61bf, [0x0c63] = 0x61c0, [0x0c64] = 0x61c1,
  [0x0c65] = 0x61c3, [0x0c66] = 0x61c4, [0x0c67] = 0x61c5, [0x0c68] = 0x61c6,
  [0x0c69] = 0x61c7, [0x0c6a] = 0x61c9, [0x0c6b] = 0x61cc, [0x0c6c] = 0x61cd,
  [0x0c6d] = 0x61ce, [0x0c6e] = 0x61cf, [0x0c6f] = 0x61d0, [0x0c70] = 0x61d3,
  [0x0c71] = 0x61d5, [0x0c72] = 0x61d6, [0x0c73] = 0x61d7, [0x0c74] = 0x61d8,
  [0x0c75] = 0x61d9, [0x0c76] = 0x61da, [0x0c77] = 0x61db, [0x0c78] = 0x61dc,
  [0x0c79] = 0x61dd, [0x0c7a] = 0x61de, [0x0c7b] = 0x61df, [0x0c7c] = 0x61e0,
  [0x0c7d] = 0x61e1, [0x0c7e] = 0x61e2, [0x0c7f] = 0x61e3, [0x0c80] = 0x61e4,
  [0x0c81] = 0x61e5, [0x0c82] = 0x61e7, [0x0c83] = 0x61e8, [0x0c84] = 0x61e9,
  [0x0c85] = 0x61ea, [0x0c86] = 0x61eb, [0x0c87] = 0x61ec, [0x0c88] = 0x61ed,
  [0x0c89] = 0x61ee, [0x0c8a] = 0x61ef, [0x0c8b] = 0x61f0, [0x0c8c] = 0x61f1,
  [0x0c8d] = 0x61f2, [0x0c8e] = 0x61f3, [0x0c8f] = 0x61f4, [0x0c90] = 0x61f6,
  [0x0c91] = 0x61f7, [0x0c92] = 0x61f8, [0x0c93] = 0x61f9, [0x0c94] = 0x61fa,
  [0x0c95] = 0x61fb, [0x0c96] = 0x61fc, [0x0c97] = 0x61fd, [0x0c98] = 0x61fe,
  [0x0c99] = 0x6200, [0x0c9a] = 0x6201, [0x0c9b] = 0x6202, [0x0c9c] = 0x6203,
  [0x0c9d] = 0x6204, [0x0c9e] = 0x6205, [0x0c9f] = 0x6207, [0x0ca0] = 0x6209,
  [0x0ca1] = 0x6213, [0x0ca2] = 0x6214, [0x0ca3] = 0x6219, [0x0ca4] = 0x621c,
  [0x0ca5] = 0x621d, [0x0ca6] = 0x621e, [0x0ca7] = 0x6220, [0x0ca8] = 0x6223,
  [0x0ca9] = 0x6226, [0x0caa] = 0x6227, [0x0cab] = 0x6228, [0x0cac] = 0x6229,
  [0x0cad] = 0x622b, [0x0cae] = 0x622d, [0x0caf] = 0x622f, [0x0cb0] = 0x6230,
  [0x0cb1] = 0x6231, [0x0cb2] = 0x6232, [0x0cb3] = 0x6235, [0x0cb4] = 0x6236,
  [0x0cb5] = 0x6238, [0x0cb6] = 0x6239, [0x0cb7] = 0x623a, [0x0cb8] = 0x623b,
  [0x0cb9] = 0x623c, [0x0cba] = 0x6242, [0x0cbb] = 0x6244, [0x0cbc] = 0x6245,
  [0x0cbd] = 0x6246, [0x0cbe] = 0x624a, [0x0cc0] = 0x624f, [0x0cc1] = 0x6250,
  [0x0cc2] = 0x6255, [0x0cc3] = 0x6256, [0x0cc4] = 0x6257, [0x0cc5] = 0x6259,
  [0x0cc6] = 0x625a, [0x0cc7] = 0x625c, [0x0cc8] = 0x625d, [0x0cc9] = 0x625e,
  [0x0cca] = 0x625f, [0x0ccb] = 0x6260, [0x0ccc] = 0x6261, [0x0ccd] = 0x6262,
  [0x0cce] = 0x6264, [0x0ccf] = 0x6265, [0x0cd0] = 0x6268, [0x0cd1] = 0x6271,
  [0x0cd2] = 0x6272, [0x0cd3] = 0x6274, [0x0cd4] = 0x6275, [0x0cd5] = 0x6277,
  [0x0cd6] = 0x6278, [0x0cd7] = 0x627a, [0x0cd8] = 0x627b, [0x0cd9] = 0x627d,
  [0x0cda] = 0x6281, [0x0cdb] = 0x6282, [0x0cdc] = 0x6283, [0x0cdd] = 0x6285,
  [0x0cde] = 0x6286, [0x0cdf] = 0x6287, [0x0ce0] = 0x6288, [0x0ce1] = 0x628b,
  [0x0ce2] = 0x628c, [0x0ce3] = 0x628d, [0x0ce4] = 0x628e, [0x0ce5] = 0x628f,
  [0x0ce6] = 0x6290, [0x0ce7] = 0x6294, [0x0ce8] = 0x6299, [0x0ce9] = 0x629c,
  [0x0cea] = 0x629d, [0x0ceb] = 0x629e, [0x0cec] = 0x62a3, [0x0ced] = 0x62a6,
  [0x0cee] = 0x62a7, [0x0cef] = 0x62a9, [0x0cf0] = 0x62aa, [0x0cf1] = 0x62ad,
  [0x0cf2] = 0x62ae, [0x0cf3] = 0x62af, [0x0cf4] = 0x62b0, [0x0cf5] = 0x62b2,
  [0x0cf6] = 0x62b3, [0x0cf7] = 0x62b4, [0x0cf8] = 0x62b6, [0x0cf9] = 0x62b7,
  [0x0cfa] = 0x62b8, [0x0cfb] = 0x62ba, [0x0cfc] = 0x62be, [0x0cfd] = 0x62c0,
  [0x0cfe] = 0x62c1, [0x0d00] = 0x62c3, [0x0d01] = 0x62cb, [0x0d02] = 0x62cf,
  [0x0d03] = 0x62d1, [0x0d04] = 0x62d5, [0x0d05] = 0x62dd, [0x0d06] = 0x62de,
  [0x0d07] = 0x62e0, [0x0d08] = 0x62e1, [0x0d09] = 0x62e4, [0x0d0a] = 0x62ea,
  [0x0d0b] = 0x62eb, [0x0d0c] = 0x62f0, [0x0d0d] = 0x62f2, [0x0d0e] = 0x62f5,
  [0x0d0f] = 0x62f8, [0x0d10] = 0x62f9, [0x0d11] = 0x62fa, [0x0d12] = 0x62fb,
  [0x0d13] = 0x6300, [0x0d14] = 0x6303, [0x0d15] = 0x6304, [0x0d16] = 0x6305,
  [0x0d17] = 0x6306, [0x0d18] = 0x630a, [0x0d19] = 0x630b, [0x0d1a] = 0x630c,
  [0x0d1b] = 0x630d, [0x0d1c] = 0x630f, [0x0d1d] = 0x6310, [0x0d1e] = 0x6312,
  [0x0d1f] = 0x6313, [0x0d20] = 0x6314, [0x0d21] = 0x6315, [0x0d22] = 0x6317,
  [0x0d23] = 0x6318, [0x0d24] = 0x6319, [0x0d25] = 0x631c, [0x0d26] = 0x6326,
  [0x0d27] = 0x6327, [0x0d28] = 0x6329, [0x0d29] = 0x632c, [0x0d2a] = 0x632d,
  [0x0d2b] = 0x632e, [0x0d2c] = 0x6330, [0x0d2d] = 0x6331, [0x0d2e] = 0x6333,
  [0x0d2f] = 0x6334, [0x0d30] = 0x6335, [0x0d31] = 0x6336, [0x0d32] = 0x6337,
  [0x0d33] = 0x6338, [0x0d34] = 0x633b, [0x0d35] = 0x633c, [0x0d36] = 0x633e,
  [0x0d37] = 0x633f, [0x0d38] = 0x6340, [0x0d39] = 0x6341, [0x0d3a] = 0x6344,
  [0x0d3b] = 0x6347, [0x0d3c] = 0x6348, [0x0d3d] = 0x634a, [0x0d3e] = 0x6351,
  [0x0d3f] = 0x6352, [0x0d40] = 0x6353, [0x0d41] = 0x6354, [0x0d42] = 0x6356,
  [0x0d43] = 0x6357, [0x0d44] = 0x6358, [0x0d45] = 0x6359, [0x0d46] = 0x635a,
  [0x0d47] = 0x635b, [0x0d48] = 0x635c, [0x0d49] = 0x635d, [0x0d4a] = 0x6360,
  [0x0d4b] = 0x6364, [0x0d4c] = 0x6365, [0x0d4d] = 0x6366, [0x0d4e] = 0x6368,
  [0x0d4f] = 0x636a, [0x0d50] = 0x636b, [0x0d51] = 0x636c, [0x0d52] = 0x636f,
  [0x0d53] = 0x6370, [0x0d54] = 0x6372, [0x0d55] = 0x6373, [0x0d56] = 0x6374,
  [0x0d57] = 0x6375, [0x0d58] = 0x6378, [0x0d59] = 0x6379, [0x0d5a] = 0x637c,
  [0x0d5b] = 0x637d, [0x0d5c] = 0x637e, [0x0d5d] = 0x637f, [0x0d5e] = 0x6381,
  [0x0d5f] = 0x6383, [0x0d60] = 0x6384, [0x0d61] = 0x6385, [0x0d62] = 0x6386,
  [0x0d63] = 0x638b, [0x0d64] = 0x638d, [0x0d65] = 0x6391, [0x0d66] = 0x6393,
  [0x0d67] = 0x6394, [0x0d68] = 0x6395, [0x0d69] = 0x6397, [0x0d6a] = 0x6399,
  [0x0d6b] = 0x639a, [0x0d6c] = 0x639b, [0x0d6d] = 0x639c, [0x0d6e] = 0x639d,
  [0x0d6f] = 0x639e, [0x0d70] = 0x639f, [0x0d71] = 0x63a1, [0x0d72] = 0x63a4,
  [0x0d73] = 0x63a6, [0x0d74] = 0x63ab, [0x0d75] = 0x63af, [0x0d76] = 0x63b1,
  [0x0d77] = 0x63b2, [0x0d78] = 0x63b5, [0x0d79] = 0x63b6, [0x0d7a] = 0x63b9,
  [0x0d7b] = 0x63bb, [0x0d7c] = 0x63bd, [0x0d7d] = 0x63bf, [0x0d7e] = 0x63c0,
  [0x0d80] = 0x63c1, [0x0d81] = 0x63c2, [0x0d82] = 0x63c3, [0x0d83] = 0x63c5,
  [0x0d84] = 0x63c7, [0x0d85] = 0x63c8, [0x0d86] = 0x63ca, [0x0d87] = 0x63cb,
  [0x0d88] = 0x63cc, [0x0d89] = 0x63d1, [0x0d8a] = 0x63d3, [0x0d8b] = 0x63d4,
  [0x0d8c] = 0x63d5, [0x0d8d] = 0x63d7, [0x0d8e] = 0x63d8, [0x0d8f] = 0x63d9,
  [0x0d90] = 0x63da, [0x0d91] = 0x63db, [0x0d92] = 0x63dc, [0x0d93] = 0x63dd,
  [0x0d94] = 0x63df, [0x0d95] = 0x63e2, [0x0d96] = 0x63e4, [0x0d97] = 0x63e5,
  [0x0d98] = 0x63e6, [0x0d99] = 0x63e7, [0x0d9a] = 0x63e8, [0x0d9b] = 0x63eb,
  [0x0d9c] = 0x63ec, [0x0d9d] = 0x63ee, [0x0d9e] = 0x63ef, [0x0d9f] = 0x63f0,
  [0x0da0] = 0x63f1, [0x0da1] = 0x63f3, [0x0da2] = 0x63f5, [0x0da3] = 0x63f7,
  [0x0da4] = 0x63f9, [0x0da5] = 0x63fa, [0x0da6] = 0x63fb, [0x0da7] = 0x63fc,
  [0x0da8] = 0x63fe, [0x0da9] = 0x6403, [0x0daa] = 0x6404, [0x0dab] = 0x6406,
  [0x0dac] = 0x6407, [0x0dad] = 0x6408, [0x0dae] = 0x6409, [0x0daf] = 0x640a,
  [0x0db0] = 0x640d, [0x0db1] = 0x640e, [0x0db2] = 0x6411, [0x0db3] = 0x6412,
  [0x0db4] = 0x6415, [0x0db5] = 0x6416, [0x0db6] = 0x6417, [0x0db7] = 0x6418,
  [0x0db8] = 0x6419, [0x0db9] = 0x641a, [0x0dba] = 0x641d, [0x0dbb] = 0x641f,
  [0x0dbc] = 0x6422, [0x0dbd] = 0x6423, [0x0dbe] = 0x6424, [0x0dc0] = 0x6425,
  [0x0dc1] = 0x6427, [0x0dc2] = 0x6428, [0x0dc3] = 0x6429, [0x0dc4] = 0x642b,
  [0x0dc5] = 0x642e, [0x0dc6] = 0x642f, [0x0dc7] = 0x6430, [0x0dc8] = 0x6431,
  [0x0dc9] = 0x6432, [0x0dca] = 0x6433, [0x0dcb] = 0x6435, [0x0dcc] = 0x6436,
  [0x0dcd] = 0x6437, [0x0dce] = 0x6438, [0x0dcf] = 0x6439, [0x0dd0] = 0x643b,
  [0x0dd1] = 0x643c, [0x0dd2] = 0x643e, [0x0dd3] = 0x6440, [0x0dd4] = 0x6442,
  [0x0dd5] = 0x6443, [0x0dd6] = 0x6449, [0x0dd7] = 0x644b, [0x0dd8] = 0x644c,
  [0x0dd9] = 0x644d, [0x0dda] = 0x644e, [0x0ddb] = 0x644f, [0x0ddc] = 0x6450,
  [0x0ddd] = 0x6451, [0x0dde] = 0x6453, [0x0ddf] = 0x6455, [0x0de0] = 0x6456,
  [0x0de1] = 0x6457, [0x0de2] = 0x6459, [0x0de3] = 0x645a, [0x0de4] = 0x645b,
  [0x0de5] = 0x645c, [0x0de6] = 0x645d, [0x0de7] = 0x645f, [0x0de8] = 0x6460,
  [0x0de9] = 0x6461, [0x0dea] = 0x6462, [0x0deb] = 0x6463, [0x0dec] = 0x6464,
  [0x0ded] = 0x6465, [0x0dee] = 0x6466, [0x0def] = 0x6468, [0x0df0] = 0x646a,
  [0x0df1] = 0x646b, [0x0df2] = 0x646c, [0x0df3] = 0x646e, [0x0df4] = 0x646f,
  [0x0df5] = 0x6470, [0x0df6] = 0x6471, [0x0df7] = 0x6472, [0x0df8] = 0x6473,
  [0x0df9] = 0x6474, [0x0dfa] = 0x6475, [0x0dfb] = 0x6476, [0x0dfc] = 0x6477,
  [0x0dfd] = 0x647b, [0x0dfe] = 0x647c, [0x0dff] = 0x647d, [0x0e00] = 0x647e,
  [0x0e01] = 0x647f, [0x0e02] = 0x6480, [0x0e03] = 0x6481, [0x0e04] = 0x6483,
  [0x0e05] = 0x6486, [0x0e06] = 0x6488, [0x0e07] = 0x6489, [0x0e08] = 0x648a,
  [0x0e09] = 0x648b, [0x0e0a] = 0x648c, [0x0e0b] = 0x648d, [0x0e0c] = 0x648e,
  [0x0e0d] = 0x648f, [0x0e0e] = 0x6490, [0x0e0f] = 0x6493, [0x0e10] = 0x6494,
  [0x0e11] = 0x6497, [0x0e12] = 0x6498, [0x0e13] = 0x649a, [0x0e14] = 0x649b,
  [0x0e15] = 0x649c, [0x0e16] = 0x649d, [0x0e17] = 0x649f, [0x0e18] = 0x64a0,
  [0x0e19] = 0x64a1, [0x0e1a] = 0x64a2, [0x0e1b] = 0x64a3, [0x0e1c] = 0x64a5,
  [0x0e1d] = 0x64a6, [0x0e1e] = 0x64a7, [0x0e1f] = 0x64a8, [0x0e20] = 0x64aa,
  [0x0e21] = 0x64ab, [0x0e22] = 0x64af, [0x0e23] = 0x64b1, [0x0e24] = 0x64b2,
  [0x0e25] = 0x64b3, [0x0e26] = 0x64b4, [0x0e27] = 0x64b6, [0x0e28] = 0x64b9,
  [0x0e29] = 0x64bb, [0x0e2a] = 0x64bd, [0x0e2b] = 0x64be, [0x0e2c] = 0x64bf,
  [0x0e2d] = 0x64c1, [0x0e2e] = 0x64c3, [0x0e2f] = 0x64c4, [0x0e30] = 0x64c6,
  [0x0e31] = 0x64c7, [0x0e32] = 0x64c8, [0x0e33] = 0x64c9, [0x0e34] = 0x64ca,
  [0x0e35] = 0x64cb, [0x0e36] = 0x64cc, [0x0e37] = 0x64cf, [0x0e38] = 0x64d1,
  [0x0e39] = 0x64d3, [0x0e3a] = 0x64d4, [0x0e3b] = 0x64d5, [0x0e3c] = 0x64d6,
  [0x0e3d] = 0x64d9, [0x0e3e] = 0x64da, [0x0e40] = 0x64db, [0x0e41] = 0x64dc,
  [0x0e42] = 0x64dd, [0x0e43] = 0x64df, [0x0e44] = 0x64e0, [0x0e45] = 0x64e1,
  [0x0e46] = 0x64e3, [0x0e47] = 0x64e5, [0x0e48] = 0x64e7, [0x0e49] = 0x64e8,
  [0x0e4a] = 0x64e9, [0x0e4b] = 0x64ea, [0x0e4c] = 0x64eb, [0x0e4d] = 0x64ec,
  [0x0e4e] = 0x64ed, [0x0e4f] = 0x64ee, [0x0e50] = 0x64ef, [0x0e51] = 0x64f0,
  [0x0e52] = 0x64f1, [0x0e53] = 0x64f2, [0x0e54] = 0x64f3, [0x0e55] = 0x64f4,
  [0x0e56] = 0x64f5, [0x0e57] = 0x64f6, [0x0e58] = 0x64f7, [0x0e59] = 0x64f8,
  [0x0e5a] = 0x64f9, [0x0e5b] = 0x64fa, [0x0e5c] = 0x64fb, [0x0e5d] = 0x64fc,
  [0x0e5e] = 0x64fd, [0x0e5f] = 0x64fe, [0x0e60] = 0x64ff, [0x0e61] = 0x6501,
  [0x0e62] = 0x6502, [0x0e63] = 0x6503, [0x0e64] = 0x6504, [0x0e65] = 0x6505,
  [0x0e66] = 0x6506, [0x0e67] = 0x6507, [0x0e68] = 0x6508, [0x0e69] = 0x650a,
  [0x0e6a] = 0x650b, [0x0e6b] = 0x650c, [0x0e6c] = 0x650d, [0x0e6d] = 0x650e,
  [0x0e6e] = 0x650f, [0x0e6f] = 0x6510, [0x0e70] = 0x6511, [0x0e71] = 0x6513,
  [0x0e72] = 0x6514, [0x0e73] = 0x6515, [0x0e74] = 0x6516, [0x0e75] = 0x6517,
  [0x0e76] = 0x6519, [0x0e77] = 0x651a, [0x0e78] = 0x651b, [0x0e79] = 0x651c,
  [0x0e7a] = 0x651d, [0x0e7b] = 0x651e, [0x0e7c] = 0x651f, [0x0e7d] = 0x6520,
  [0x0e7e] = 0x6521, [0x0e80] = 0x6522, [0x0e81] = 0x6523, [0x0e82] = 0x6524,
  [0x0e83] = 0x6526, [0x0e84] = 0x6527, [0x0e85] = 0x6528, [0x0e86] = 0x6529,
  [0x0e87] = 0x652a, [0x0e88] = 0x652c, [0x0e89] = 0x652d, [0x0e8a] = 0x6530,
  [0x0e8b] = 0x6531, [0x0e8c] = 0x6532, [0x0e8d] = 0x6533, [0x0e8e] = 0x6537,
  [0x0e8f] = 0x653a, [0x0e90] = 0x653c, [0x0e91] = 0x653d, [0x0e92] = 0x6540,
  [0x0e93] = 0x6541, [0x0e94] = 0x6542, [0x0e95] = 0x6543, [0x0e96] = 0x6544,
  [0x0e97] = 0x6546, [0x0e98] = 0x6547, [0x0e99] = 0x654a, [0x0e9a] = 0x654b,
  [0x0e9b] = 0x654d, [0x0e9c] = 0x654e, [0x0e9d] = 0x6550, [0x0e9e] = 0x6552,
  [0x0e9f] = 0x6553, [0x0ea0] = 0x6554, [0x0ea1] = 0x6557, [0x0ea2] = 0x6558,
  [0x0ea3] = 0x655a, [0x0ea4] = 0x655c, [0x0ea5] = 0x655f, [0x0ea6] = 0x6560,
  [0x0ea7] = 0x6561, [0x0ea8] = 0x6564, [0x0ea9] = 0x6565, [0x0eaa] = 0x6567,
  [0x0eab] = 0x6568, [0x0eac] = 0x6569, [0x0ead] = 0x656a, [0x0eae] = 0x656d,
  [0x0eaf] = 0x656e, [0x0eb0] = 0x656f, [0x0eb1] = 0x6571, [0x0eb2] = 0x6573,
  [0x0eb3] = 0x6575, [0x0eb4] = 0x6576, [0x0eb5] = 0x6578, [0x0eb6] = 0x6579,
  [0x0eb7] = 0x657a, [0x0eb8] = 0x657b, [0x0eb9] = 0x657c, [0x0eba] = 0x657d,
  [0x0ebb] = 0x657e, [0x0ebc] = 0x657f, [0x0ebd] = 0x6580, [0x0ebe] = 0x6581,
  [0x0ebf] = 0x6582, [0x0ec0] = 0x6583, [0x0ec1] = 0x6584, [0x0ec2] = 0x6585,
  [0x0ec3] = 0x6586, [0x0ec4] = 0x6588, [0x0ec5] = 0x6589, [0x0ec6] = 0x658a,
  [0x0ec7] = 0x658d, [0x0ec8] = 0x658e, [0x0ec9] = 0x658f, [0x0eca] = 0x6592,
  [0x0ecb] = 0x6594, [0x0ecc] = 0x6595, [0x0ecd] = 0x6596, [0x0ece] = 0x6598,
  [0x0ecf] = 0x659a, [0x0ed0] = 0x659d, [0x0ed1] = 0x659e, [0x0ed2] = 0x65a0,
  [0x0ed3] = 0x65a2, [0x0ed4] = 0x65a3, [0x0ed5] = 0x65a6, [0x0ed6] = 0x65a8,
  [0x0ed7] = 0x65aa, [0x0ed8] = 0x65ac, [0x0ed9] = 0x65ae, [0x0eda] = 0x65b1,
  [0x0edb] = 0x65b2, [0x0edc] = 0x65b3, [0x0edd] = 0x65b4, [0x0ede] = 0x65b5,
  [0x0edf] = 0x65b6, [0x0ee0] = 0x65b7, [0x0ee1] = 0x65b8, [0x0ee2] = 0x65ba,
  [0x0ee3] = 0x65bb, [0x0ee4] = 0x65be, [0x0ee5] = 0x65bf, [0x0ee6] = 0x65c0,
  [0x0ee7] = 0x65c2, [0x0ee8] = 0x65c7, [0x0ee9] = 0x65c8, [0x0eea] = 0x65c9,
  [0x0eeb] = 0x65ca, [0x0eec] = 0x65cd, [0x0eed] = 0x65d0, [0x0eee] = 0x65d1,
  [0x0eef] = 0x65d3, [0x0ef0] = 0x65d4, [0x0ef1] = 0x65d5, [0x0ef2] = 0x65d8,
  [0x0ef3] = 0x65d9, [0x0ef4] = 0x65da, [0x0ef5] = 0x65db, [0x0ef6] = 0x65dc,
  [0x0ef7] = 0x65dd, [0x0ef8] = 0x65de, [0x0ef9] = 0x65df, [0x0efa] = 0x65e1,
  [0x0efb] = 0x65e3, [0x0efc] = 0x65e4, [0x0efd] = 0x65ea, [0x0efe] = 0x65eb,
  [0x0f00] = 0x65f2, [0x0f01] = 0x65f3, [0x0f02] = 0x65f4, [0x0f03] = 0x65f5,
  [0x0f04] = 0x65f8, [0x0f05] = 0x65f9, [0x0f06] = 0x65fb, [0x0f07] = 0x65fc,
  [0x0f08] = 0x65fd, [0x0f09] = 0x65fe, [0x0f0a] = 0x65ff, [0x0f0b] = 0x6601,
  [0x0f0c] = 0x6604, [0x0f0d] = 0x6605, [0x0f0e] = 0x6607, [0x0f0f] = 0x6608,
  [0x0f10] = 0x6609, [0x0f11] = 0x660b, [0x0f12] = 0x660d, [0x0f13] = 0x6610,
  [0x0f14] = 0x6611, [0x0f15] = 0x6612, [0x0f16] = 0x6616, [0x0f17] = 0x6617,
  [0x0f18] = 0x6618, [0x0f19] = 0x661a, [0x0f1a] = 0x661b, [0x0f1b] = 0x661c,
  [0x0f1c] = 0x661e, [0x0f1d] = 0x6621, [0x0f1e] = 0x6622, [0x0f1f] = 0x6623,
  [0x0f20] = 0x6624, [0x0f21] = 0x6626, [0x0f22] = 0x6629, [0x0f23] = 0x662a,
  [0x0f24] = 0x662b, [0x0f25] = 0x662c, [0x0f26] = 0x662e, [0x0f27] = 0x6630,
  [0x0f28] = 0x6632, [0x0f29] = 0x6633, [0x0f2a] = 0x6637, [0x0f2b] = 0x6638,
  [0x0f2c] = 0x6639, [0x0f2d] = 0x663a, [0x0f2e] = 0x663b, [0x0f2f] = 0x663d,
  [0x0f30] = 0x663f, [0x0f31] = 0x6640, [0x0f32] = 0x6642, [0x0f33] = 0x6644,
  [0x0f34] = 0x6645, [0x0f35] = 0x6646, [0x0f36] = 0x6647, [0x0f37] = 0x6648,
  [0x0f38] = 0x6649, [0x0f39] = 0x664a, [0x0f3a] = 0x664d, [0x0f3b] = 0x664e,
  [0x0f3c] = 0x6650, [0x0f3d] = 0x6651, [0x0f3e] = 0x6658, [0x0f40] = 0x6659,
  [0x0f41] = 0x665b, [0x0f42] = 0x665c, [0x0f43] = 0x665d, [0x0f44] = 0x665e,
  [0x0f45] = 0x6660, [0x0f46] = 0x6662, [0x0f47] = 0x6663, [0x0f48] = 0x6665,
  [0x0f49] = 0x6667, [0x0f4a] = 0x6669, [0x0f4b] = 0x666a, [0x0f4c] = 0x666b,
  [0x0f4d] = 0x666c, [0x0f4e] = 0x666d, [0x0f4f] = 0x6671, [0x0f50] = 0x6672,
  [0x0f51] = 0x6673, [0x0f52] = 0x6675, [0x0f53] = 0x6678, [0x0f54] = 0x6679,
  [0x0f55] = 0x667b, [0x0f56] = 0x667c, [0x0f57] = 0x667d, [0x0f58] = 0x667f,
  [0x0f59] = 0x6680, [0x0f5a] = 0x6681, [0x0f5b] = 0x6683, [0x0f5c] = 0x6685,
  [0x0f5d] = 0x6686, [0x0f5e] = 0x6688, [0x0f5f] = 0x6689, [0x0f60] = 0x668a,
  [0x0f61] = 0x668b, [0x0f62] = 0x668d, [0x0f63] = 0x668e, [0x0f64] = 0x668f,
  [0x0f65] = 0x6690, [0x0f66] = 0x6692, [0x0f67] = 0x6693, [0x0f68] = 0x6694,
  [0x0f69] = 0x6695, [0x0f6a] = 0x6698, [0x0f6b] = 0x6699, [0x0f6c] = 0x669a,
  [0x0f6d] = 0x669b, [0x0f6e] = 0x669c, [0x0f6f] = 0x669e, [0x0f70] = 0x669f,
  [0x0f71] = 0x66a0, [0x0f72] = 0x66a1, [0x0f73] = 0x66a2, [0x0f74] = 0x66a3,
  [0x0f75] = 0x66a4, [0x0f76] = 0x66a5, [0x0f77] = 0x66a6, [0x0f78] = 0x66a9,
  [0x0f79] = 0x66aa, [0x0f7a] = 0x66ab, [0x0f7b] = 0x66ac, [0x0f7c] = 0x66ad,
  [0x0f7d] = 0x66af, [0x0f7e] = 0x66b0, [0x0f7f] = 0x66b1, [0x0f80] = 0x66b2,
  [0x0f81] = 0x66b3, [0x0f82] = 0x66b5, [0x0f83] = 0x66b6, [0x0f84] = 0x66b7,
  [0x0f85] = 0x66b8, [0x0f86] = 0x66ba, [0x0f87] = 0x66bb, [0x0f88] = 0x66bc,
  [0x0f89] = 0x66bd, [0x0f8a] = 0x66bf, [0x0f8b] = 0x66c0, [0x0f8c] = 0x66c1,
  [0x0f8d] = 0x66c2, [0x0f8e] = 0x66c3, [0x0f8f] = 0x66c4, [0x0f90] = 0x66c5,
  [0x0f91] = 0x66c6, [0x0f92] = 0x66c7, [0x0f93] = 0x66c8, [0x0f94] = 0x66c9,
  [0x0f95] = 0x66ca, [0x0f96] = 0x66cb, [0x0f97] = 0x66cc, [0x0f98] = 0x66cd,
  [0x0f99] = 0x66ce, [0x0f9a] = 0x66cf, [0x0f9b] = 0x66d0, [0x0f9c] = 0x66d1,
  [0x0f9d] = 0x66d2, [0x0f9e] = 0x66d3, [0x0f9f] = 0x66d4, [0x0fa0] = 0x66d5,
  [0x0fa1] = 0x66d6, [0x0fa2] = 0x66d7, [0x0fa3] = 0x66d8, [0x0fa4] = 0x66da,
  [0x0fa5] = 0x66de, [0x0fa6] = 0x66df, [0x0fa7] = 0x66e0, [0x0fa8] = 0x66e1,
  [0x0fa9] = 0x66e2, [0x0faa] = 0x66e3, [0x0fab] = 0x66e4, [0x0fac] = 0x66e5,
  [0x0fad] = 0x66e7, [0x0fae] = 0x66e8, [0x0faf] = 0x66ea, [0x0fb0] = 0x66eb,
  [0x0fb1] = 0x66ec, [0x0fb2] = 0x66ed, [0x0fb3] = 0x66ee, [0x0fb4] = 0x66ef,
  [0x0fb5] = 0x66f1, [0x0fb6] = 0x66f5, [0x0fb7] = 0x66f6, [0x0fb8] = 0x66f8,
  [0x0fb9] = 0x66fa, [0x0fba] = 0x66fb, [0x0fbb] = 0x66fd, [0x0fbc] = 0x6701,
  [0x0fbd] = 0x6702, [0x0fbe] = 0x6703, [0x0fc0] = 0x6704, [0x0fc1] = 0x6705,
  [0x0fc2] = 0x6706, [0x0fc3] = 0x6707, [0x0fc4] = 0x670c, [0x0fc5] = 0x670e,
  [0x0fc6] = 0x670f, [0x0fc7] = 0x6711, [0x0fc8] = 0x6712, [0x0fc9] = 0x6713,
  [0x0fca] = 0x6716, [0x0fcb] = 0x6718, [0x0fcc] = 0x6719, [0x0fcd] = 0x671a,
  [0x0fce] = 0x671c, [0x0fcf] = 0x671e, [0x0fd0] = 0x6720, [0x0fd1] = 0x6721,
  [0x0fd2] = 0x6722, [0x0fd3] = 0x6723, [0x0fd4] = 0x6724, [0x0fd5] = 0x6725,
  [0x0fd6] = 0x6727, [0x0fd7] = 0x6729, [0x0fd8] = 0x672e, [0x0fd9] = 0x6730,
  [0x0fda] = 0x6732, [0x0fdb] = 0x6733, [0x0fdc] = 0x6736, [0x0fdd] = 0x6737,
  [0x0fde] = 0x6738, [0x0fdf] = 0x6739, [0x0fe0] = 0x673b, [0x0fe1] = 0x673c,
  [0x0fe2] = 0x673e, [0x0fe3] = 0x673f, [0x0fe4] = 0x6741, [0x0fe5] = 0x6744,
  [0x0fe6] = 0x6745, [0x0fe7] = 0x6747, [0x0fe8] = 0x674a, [0x0fe9] = 0x674b,
  [0x0fea] = 0x674d, [0x0feb] = 0x6752, [0x0fec] = 0x6754, [0x0fed] = 0x6755,
  [0x0fee] = 0x6757, [0x0fef] = 0x6758, [0x0ff0] = 0x6759, [0x0ff1] = 0x675a,
  [0x0ff2] = 0x675b, [0x0ff3] = 0x675d, [0x0ff4] = 0x6762, [0x0ff5] = 0x6763,
  [0x0ff6] = 0x6764, [0x0ff7] = 0x6766, [0x0ff8] = 0x6767, [0x0ff9] = 0x676b,
  [0x0ffa] = 0x676c, [0x0ffb] = 0x676e, [0x0ffc] = 0x6771, [0x0ffd] = 0x6774,
  [0x0ffe] = 0x6776, [0x1000] = 0x6778, [0x1001] = 0x6779, [0x1002] = 0x677a,
  [0x1003] = 0x677b, [0x1004] = 0x677d, [0x1005] = 0x6780, [0x1006] = 0x6782,
  [0x1007] = 0x6783, [0x1008] = 0x6785, [0x1009] = 0x6786, [0x100a] = 0x6788,
  [0x100b] = 0x678a, [0x100c] = 0x678c, [0x100d] = 0x678d, [0x100e] = 0x678e,
  [0x100f] = 0x678f, [0x1010] = 0x6791, [0x1011] = 0x6792, [0x1012] = 0x6793,
  [0x1013] = 0x6794, [0x1014] = 0x6796, [0x1015] = 0x6799, [0x1016] = 0x679b,
  [0x1017] = 0x679f, [0x1018] = 0x67a0, [0x1019] = 0x67a1, [0x101a] = 0x67a4,
  [0x101b] = 0x67a6, [0x101c] = 0x67a9, [0x101d] = 0x67ac, [0x101e] = 0x67ae,
  [0x101f] = 0x67b1, [0x1020] = 0x67b2, [0x1021] = 0x67b4, [0x1022] = 0x67b9,
  [0x1023] = 0x67ba, [0x1024] = 0x67bb, [0x1025] = 0x67bc, [0x1026] = 0x67bd,
  [0x1027] = 0x67be, [0x1028] = 0x67bf, [0x1029] = 0x67c0, [0x102a] = 0x67c2,
  [0x102b] = 0x67c5, [0x102c] = 0x67c6, [0x102d] = 0x67c7, [0x102e] = 0x67c8,
  [0x102f] = 0x67c9, [0x1030] = 0x67ca, [0x1031] = 0x67cb, [0x1032] = 0x67cc,
  [0x1033] = 0x67cd, [0x1034] = 0x67ce, [0x1035] = 0x67d5, [0x1036] = 0x67d6,
  [0x1037] = 0x67d7, [0x1038] = 0x67db, [0x1039] = 0x67df, [0x103a] = 0x67e1,
  [0x103b] = 0x67e3, [0x103c] = 0x67e4, [0x103d] = 0x67e6, [0x103e] = 0x67e7,
  [0x103f] = 0x67e8, [0x1040] = 0x67ea, [0x1041] = 0x67eb, [0x1042] = 0x67ed,
  [0x1043] = 0x67ee, [0x1044] = 0x67f2, [0x1045] = 0x67f5, [0x1046] = 0x67f6,
  [0x1047] = 0x67f7, [0x1048] = 0x67f8, [0x1049] = 0x67f9, [0x104a] = 0x67fa,
  [0x104b] = 0x67fb, [0x104c] = 0x67fc, [0x104d] = 0x67fe, [0x104e] = 0x6801,
  [0x104f] = 0x6802, [0x1050] = 0x6803, [0x1051] = 0x6804, [0x1052] = 0x6806,
  [0x1053] = 0x680d, [0x1054] = 0x6810, [0x1055] = 0x6812, [0x1056] = 0x6814,
  [0x1057] = 0x6815, [0x1058] = 0x6818, [0x1059] = 0x6819, [0x105a] = 0x681a,
  [0x105b] = 0x681b, [0x105c] = 0x681c, [0x105d] = 0x681e, [0x105e] = 0x681f,
  [0x105f] = 0x6820, [0x1060] = 0x6822, [0x1061] = 0x6823, [0x1062] = 0x6824,
  [0x1063] = 0x6825, [0x1064] = 0x6826, [0x1065] = 0x6827, [0x1066] = 0x6828,
  [0x1067] = 0x682b, [0x1068] = 0x682c, [0x1069] = 0x682d, [0x106a] = 0x682e,
  [0x106b] = 0x682f, [0x106c] = 0x6830, [0x106d] = 0x6831, [0x106e] = 0x6834,
  [0x106f] = 0x6835, [0x1070] = 0x6836, [0x1071] = 0x683a, [0x1072] = 0x683b,
  [0x1073] = 0x683f, [0x1074] = 0x6847, [0x1075] = 0x684b, [0x1076] = 0x684d,
  [0x1077] = 0x684f, [0x1078] = 0x6852, [0x1079] = 0x6856, [0x107a] = 0x6857,
  [0x107b] = 0x6858, [0x107c] = 0x6859, [0x107d] = 0x685a, [0x107e] = 0x685b,
  [0x1080] = 0x685c, [0x1081] = 0x685d, [0x1082] = 0x685e, [0x1083] = 0x685f,
  [0x1084] = 0x686a, [0x1085] = 0x686c, [0x1086] = 0x686d, [0x1087] = 0x686e,
  [0x1088] = 0x686f, [0x1089] = 0x6870, [0x108a] = 0x6871, [0x108b] = 0x6872,
  [0x108c] = 0x6873, [0x108d] = 0x6875, [0x108e] = 0x6878, [0x108f] = 0x6879,
  [0x1090] = 0x687a, [0x1091] = 0x687b, [0x1092] = 0x687c, [0x1093] = 0x687d,
  [0x1094] = 0x687e, [0x1095] = 0x687f, [0x1096] = 0x6880, [0x1097] = 0x6882,
  [0x1098] = 0x6884, [0x1099] = 0x6887, [0x109a] = 0x6888, [0x109b] = 0x6889,
  [0x109c] = 0x688a, [0x109d] = 0x688b, [0x109e] = 0x688c, [0x109f] = 0x688d,
  [0x10a0] = 0x688e, [0x10a1] = 0x6890, [0x10a2] = 0x6891, [0x10a3] = 0x6892,
  [0x10a4] = 0x6894, [0x10a5] = 0x6895, [0x10a6] = 0x6896, [0x10a7] = 0x6898,
  [0x10a8] = 0x6899, [0x10a9] = 0x689a, [0x10aa] = 0x689b, [0x10ab] = 0x689c,
  [0x10ac] = 0x689d, [0x10ad] = 0x689e, [0x10ae] = 0x689f, [0x10af] = 0x68a0,
  [0x10b0] = 0x68a1, [0x10b1] = 0x68a3, [0x10b2] = 0x68a4, [0x10b3] = 0x68a5,
  [0x10b4] = 0x68a9, [0x10b5] = 0x68aa, [0x10b6] = 0x68ab, [0x10b7] = 0x68ac,
  [0x10b8] = 0x68ae, [0x10b9] = 0x68b1, [0x10ba] = 0x68b2, [0x10bb] = 0x68b4,
  [0x10bc] = 0x68b6, [0x10bd] = 0x68b7, [0x10be] = 0x68b8, [0x10c0] = 0x68b9,
  [0x10c1] = 0x68ba, [0x10c2] = 0x68bb, [0x10c3] = 0x68bc, [0x10c4] = 0x68bd,
  [0x10c5] = 0x68be, [0x10c6] = 0x68bf, [0x10c7] = 0x68c1, [0x10c8] = 0x68c3,
  [0x10c9] = 0x68c4, [0x10ca] = 0x68c5, [0x10cb] = 0x68c6, [0x10cc] = 0x68c7,
  [0x10cd] = 0x68c8, [0x10ce] = 0x68ca, [0x10cf] = 0x68cc, [0x10d0] = 0x68ce,
  [0x10d1] = 0x68cf, [0x10d2] = 0x68d0, [0x10d3] = 0x68d1, [0x10d4] = 0x68d3,
  [0x10d5] = 0x68d4, [0x10d6] = 0x68d6, [0x10d7] = 0x68d7, [0x10d8] = 0x68d9,
  [0x10d9] = 0x68db, [0x10da] = 0x68dc, [0x10db] = 0x68dd, [0x10dc] = 0x68de,
  [0x10dd] = 0x68df, [0x10de] = 0x68e1, [0x10df] = 0x68e2, [0x10e0] = 0x68e4,
  [0x10e1] = 0x68e5, [0x10e2] = 0x68e6, [0x10e3] = 0x68e7, [0x10e4] = 0x68e8,
  [0x10e5] = 0x68e9, [0x10e6] = 0x68ea, [0x10e7] = 0x68eb, [0x10e8] = 0x68ec,
  [0x10e9] = 0x68ed, [0x10ea] = 0x68ef, [0x10eb] = 0x68f2, [0x10ec] = 0x68f3,
  [0x10ed] = 0x68f4, [0x10ee] = 0x68f6, [0x10ef] = 0x68f7, [0x10f0] = 0x68f8,
  [0x10f1] = 0x68fb, [0x10f2] = 0x68fd, [0x10f3] = 0x68fe, [0x10f4] = 0x68ff,
  [0x10f5] = 0x6900, [0x10f6] = 0x6902, [0x10f7] = 0x6903, [0x10f8] = 0x6904,
  [0x10f9] = 0x6906, [0x10fa] = 0x6907, [0x10fb] = 0x6908, [0x10fc] = 0x6909,
  [0x10fd] = 0x690a, [0x10fe] = 0x690c, [0x10ff] = 0x690f, [0x1100] = 0x6911,
  [0x1101] = 0x6913, [0x1102] = 0x6914, [0x1103] = 0x6915, [0x1104] = 0x6916,
  [0x1105] = 0x6917, [0x1106] = 0x6918, [0x1107] = 0x6919, [0x1108] = 0x691a,
  [0x1109] = 0x691b, [0x110a] = 0x691c, [0x110b] = 0x691d, [0x110c] = 0x691e,
  [0x110d] = 0x6921, [0x110e] = 0x6922, [0x110f] = 0x6923, [0x1110] = 0x6925,
  [0x1111] = 0x6926, [0x1112] = 0x6927, [0x1113] = 0x6928, [0x1114] = 0x6929,
  [0x1115] = 0x692a, [0x1116] = 0x692b, [0x1117] = 0x692c, [0x1118] = 0x692e,
  [0x1119] = 0x692f, [0x111a] = 0x6931, [0x111b] = 0x6932, [0x111c] = 0x6933,
  [0x111d] = 0x6935, [0x111e] = 0x6936, [0x111f] = 0x6937, [0x1120] = 0x6938,
  [0x1121] = 0x693a, [0x1122] = 0x693b, [0x1123] = 0x693c, [0x1124] = 0x693e,
  [0x1125] = 0x6940, [0x1126] = 0x6941, [0x1127] = 0x6943, [0x1128] = 0x6944,
  [0x1129] = 0x6945, [0x112a] = 0x6946, [0x112b] = 0x6947, [0x112c] = 0x6948,
  [0x112d] = 0x6949, [0x112e] = 0x694a, [0x112f] = 0x694b, [0x1130] = 0x694c,
  [0x1131] = 0x694d, [0x1132] = 0x694e, [0x1133] = 0x694f, [0x1134] = 0x6950,
  [0x1135] = 0x6951, [0x1136] = 0x6952, [0x1137] = 0x6953, [0x1138] = 0x6955,
  [0x1139] = 0x6956, [0x113a] = 0x6958, [0x113b] = 0x6959, [0x113c] = 0x695b,
  [0x113d] = 0x695c, [0x113e] = 0x695f, [0x1140] = 0x6961, [0x1141] = 0x6962,
  [0x1142] = 0x6964, [0x1143] = 0x6965, [0x1144] = 0x6967, [0x1145] = 0x6968,
  [0x1146] = 0x6969, [0x1147] = 0x696a, [0x1148] = 0x696c, [0x1149] = 0x696d,
  [0x114a] = 0x696f, [0x114b] = 0x6970, [0x114c] = 0x6972, [0x114d] = 0x6973,
  [0x114e] = 0x6974, [0x114f] = 0x6975, [0x1150] = 0x6976, [0x1151] = 0x697a,
  [0x1152] = 0x697b, [0x1153] = 0x697d, [0x1154] = 0x697e, [0x1155] = 0x697f,
  [0x1156] = 0x6981, [0x1157] = 0x6983, [0x1158] = 0x6985, [0x1159] = 0x698a,
  [0x115a] = 0x698b, [0x115b] = 0x698c, [0x115c] = 0x698e, [0x115d] = 0x698f,
  [0x115e] = 0x6990, [0x115f] = 0x6991, [0x1160] = 0x6992, [0x1161] = 0x6993,
  [0x1162] = 0x6996, [0x1163] = 0x6997, [0x1164] = 0x6999, [0x1165] = 0x699a,
  [0x1166] = 0x699d, [0x1167] = 0x699e, [0x1168] = 0x699f, [0x1169] = 0x69a0,
  [0x116a] = 0x69a1, [0x116b] = 0x69a2, [0x116c] = 0x69a3, [0x116d] = 0x69a4,
  [0x116e] = 0x69a5, [0x116f] = 0x69a6, [0x1170] = 0x69a9, [0x1171] = 0x69aa,
  [0x1172] = 0x69ac, [0x1173] = 0x69ae, [0x1174] = 0x69af, [0x1175] = 0x69b0,
  [0x1176] = 0x69b2, [0x1177] = 0x69b3, [0x1178] = 0x69b5, [0x1179] = 0x69b6,
  [0x117a] = 0x69b8, [0x117b] = 0x69b9, [0x117c] = 0x69ba, [0x117d] = 0x69bc,
  [0x117e] = 0x69bd, [0x1180] = 0x69be, [0x1181] = 0x69bf, [0x1182] = 0x69c0,
  [0x1183] = 0x69c2, [0x1184] = 0x69c3, [0x1185] = 0x69c4, [0x1186] = 0x69c5,
  [0x1187] = 0x69c6, [0x1188] = 0x69c7, [0x1189] = 0x69c8, [0x118a] = 0x69c9,
  [0x118b] = 0x69cb, [0x118c] = 0x69cd, [0x118d] = 0x69cf, [0x118e] = 0x69d1,
  [0x118f] = 0x69d2, [0x1190] = 0x69d3, [0x1191] = 0x69d5, [0x1192] = 0x69d6,
  [0x1193] = 0x69d7, [0x1194] = 0x69d8, [0x1195] = 0x69d9, [0x1196] = 0x69da,
  [0x1197] = 0x69dc, [0x1198] = 0x69dd, [0x1199] = 0x69de, [0x119a] = 0x69e1,
  [0x119b] = 0x69e2, [0x119c] = 0x69e3, [0x119d] = 0x69e4, [0x119e] = 0x69e5,
  [0x119f] = 0x69e6, [0x11a0] = 0x69e7, [0x11a1] = 0x69e8, [0x11a2] = 0x69e9,
  [0x11a3] = 0x69ea, [0x11a4] = 0x69eb, [0x11a5] = 0x69ec, [0x11a6] = 0x69ee,
  [0x11a7] = 0x69ef, [0x11a8] = 0x69f0, [0x11a9] = 0x69f1, [0x11aa] = 0x69f3,
  [0x11ab] = 0x69f4, [0x11ac] = 0x69f5, [0x11ad] = 0x69f6, [0x11ae] = 0x69f7,
  [0x11af] = 0x69f8, [0x11b0] = 0x69f9, [0x11b1] = 0x69fa, [0x11b2] = 0x69fb,
  [0x11b3] = 0x69fc, [0x11b4] = 0x69fe, [0x11b5] = 0x6a00, [0x11b6] = 0x6a01,
  [0x11b7] = 0x6a02, [0x11b8] = 0x6a03, [0x11b9] = 0x6a04, [0x11ba] = 0x6a05,
  [0x11bb] = 0x6a06, [0x11bc] = 0x6a07, [0x11bd] = 0x6a08, [0x11be] = 0x6a09,
  [0x11bf] = 0x6a0b, [0x11c0] = 0x6a0c, [0x11c1] = 0x6a0d, [0x11c2] = 0x6a0e,
  [0x11c3] = 0x6a0f, [0x11c4] = 0x6a10, [0x11c5] = 0x6a11, [0x11c6] = 0x6a12,
  [0x11c7] = 0x6a13, [0x11c8] = 0x6a14, [0x11c9] = 0x6a15, [0x11ca] = 0x6a16,
  [0x11cb] = 0x6a19, [0x11cc] = 0x6a1a, [0x11cd] = 0x6a1b, [0x11ce] = 0x6a1c,
  [0x11cf] = 0x6a1d, [0x11d0] = 0x6a1e, [0x11d1] = 0x6a20, [0x11d2] = 0x6a22,
  [0x11d3] = 0x6a23, [0x11d4] = 0x6a24, [0x11d5] = 0x6a25, [0x11d6] = 0x6a26,
  [0x11d7] = 0x6a27, [0x11d8] = 0x6a29, [0x11d9] = 0x6a2b, [0x11da] = 0x6a2c,
  [0x11db] = 0x6a2d, [0x11dc] = 0x6a2e, [0x11dd] = 0x6a30, [0x11de] = 0x6a32,
  [0x11df] = 0x6a33, [0x11e0] = 0x6a34, [0x11e1] = 0x6a36, [0x11e2] = 0x6a37,
  [0x11e3] = 0x6a38, [0x11e4] = 0x6a39, [0x11e5] = 0x6a3a, [0x11e6] = 0x6a3b,
  [0x11e7] = 0x6a3c, [0x11e8] = 0x6a3f, [0x11e9] = 0x6a40, [0x11ea] = 0x6a41,
  [0x11eb] = 0x6a42, [0x11ec] = 0x6a43, [0x11ed] = 0x6a45, [0x11ee] = 0x6a46,
  [0x11ef] = 0x6a48, [0x11f0] = 0x6a49, [0x11f1] = 0x6a4a, [0x11f2] = 0x6a4b,
  [0x11f3] = 0x6a4c, [0x11f4] = 0x6a4d, [0x11f5] = 0x6a4e, [0x11f6] = 0x6a4f,
  [0x11f7] = 0x6a51, [0x11f8] = 0x6a52, [0x11f9] = 0x6a53, [0x11fa] = 0x6a54,
  [0x11fb] = 0x6a55, [0x11fc] = 0x6a56, [0x11fd] = 0x6a57, [0x11fe] = 0x6a5a,
  [0x1200] = 0x6a5c, [0x1201] = 0x6a5d, [0x1202] = 0x6a5e, [0x1203] = 0x6a5f,
  [0x1204] = 0x6a60, [0x1205] = 0x6a62, [0x1206] = 0x6a63, [0x1207] = 0x6a64,
  [0x1208] = 0x6a66, [0x1209] = 0x6a67, [0x120a] = 0x6a68, [0x120b] = 0x6a69,
  [0x120c] = 0x6a6a, [0x120d] = 0x6a6b, [0x120e] = 0x6a6c, [0x120f] = 0x6a6d,
  [0x1210] = 0x6a6e, [0x1211] = 0x6a6f, [0x1212] = 0x6a70, [0x1213] = 0x6a72,
  [0x1214] = 0x6a73, [0x1215] = 0x6a74, [0x1216] = 0x6a75, [0x1217] = 0x6a76,
  [0x1218] = 0x6a77, [0x1219] = 0x6a78, [0x121a] = 0x6a7a, [0x121b] = 0x6a7b,
  [0x121c] = 0x6a7d, [0x121d] = 0x6a7e, [0x121e] = 0x6a7f, [0x121f] = 0x6a81,
  [0x1220] = 0x6a82, [0x1221] = 0x6a83, [0x1222] = 0x6a85, [0x1223] = 0x6a86,
  [0x1224] = 0x6a87, [0x1225] = 0x6a88, [0x1226] = 0x6a89, [0x1227] = 0x6a8a,
  [0x1228] = 0x6a8b, [0x1229] = 0x6a8c, [0x122a] = 0x6a8d, [0x122b] = 0x6a8f,
  [0x122c] = 0x6a92, [0x122d] = 0x6a93, [0x122e] = 0x6a94, [0x122f] = 0x6a95,
  [0x1230] = 0x6a96, [0x1231] = 0x6a98, [0x1232] = 0x6a99, [0x1233] = 0x6a9a,
  [0x1234] = 0x6a9b, [0x1235] = 0x6a9c, [0x1236] = 0x6a9d, [0x1237] = 0x6a9e,
  [0x1238] = 0x6a9f, [0x1239] = 0x6aa1, [0x123a] = 0x6aa2, [0x123b] = 0x6aa3,
  [0x123c] = 0x6aa4, [0x123d] = 0x6aa5, [0x123e] = 0x6aa6, [0x1240] = 0x6aa7,
  [0x1241] = 0x6aa8, [0x1242] = 0x6aaa, [0x1243] = 0x6aad, [0x1244] = 0x6aae,
  [0x1245] = 0x6aaf, [0x1246] = 0x6ab0, [0x1247] = 0x6ab1, [0x1248] = 0x6ab2,
  [0x1249] = 0x6ab3, [0x124a] = 0x6ab4, [0x124b] = 0x6ab5, [0x124c] = 0x6ab6,
  [0x124d] = 0x6ab7, [0x124e] = 0x6ab8, [0x124f] = 0x6ab9, [0x1250] = 0x6aba,
  [0x1251] = 0x6abb, [0x1252] = 0x6abc, [0x1253] = 0x6abd, [0x1254] = 0x6abe,
  [0x1255] = 0x6abf, [0x1256] = 0x6ac0, [0x1257] = 0x6ac1, [0x1258] = 0x6ac2,
  [0x1259] = 0x6ac3, [0x125a] = 0x6ac4, [0x125b] = 0x6ac5, [0x125c] = 0x6ac6,
  [0x125d] = 0x6ac7, [0x125e] = 0x6ac8, [0x125f] = 0x6ac9, [0x1260] = 0x6aca,
  [0x1261] = 0x6acb, [0x1262] = 0x6acc, [0x1263] = 0x6acd, [0x1264] = 0x6ace,
  [0x1265] = 0x6acf, [0x1266] = 0x6ad0, [0x1267] = 0x6ad1, [0x1268] = 0x6ad2,
  [0x1269] = 0x6ad3, [0x126a] = 0x6ad4, [0x126b] = 0x6ad5, [0x126c] = 0x6ad6,
  [0x126d] = 0x6ad7, [0x126e] = 0x6ad8, [0x126f] = 0x6ad9, [0x1270] = 0x6ada,
  [0x1271] = 0x6adb, [0x1272] = 0x6adc, [0x1273] = 0x6add, [0x1274] = 0x6ade,
  [0x1275] = 0x6adf, [0x1276] = 0x6ae0, [0x1277] = 0x6ae1, [0x1278] = 0x6ae2,
  [0x1279] = 0x6ae3, [0x127a] = 0x6ae4, [0x127b] = 0x6ae5, [0x127c] = 0x6ae6,
  [0x127d] = 0x6ae7, [0x127e] = 0x6ae8, [0x127f] = 0x6ae9, [0x1280] = 0x6aea,
  [0x1281] = 0x6aeb, [0x1282] = 0x6aec, [0x1283] = 0x6aed, [0x1284] = 0x6aee,
  [0x1285] = 0x6aef, [0x1286] = 0x6af0, [0x1287] = 0x6af1, [0x1288] = 0x6af2,
  [0x1289] = 0x6af3, [0x128a] = 0x6af4, [0x128b] = 0x6af5, [0x128c] = 0x6af6,
  [0x128d] = 0x6af7, [0x128e] = 0x6af8, [0x128f] = 0x6af9, [0x1290] = 0x6afa,
  [0x1291] = 0x6afb, [0x1292] = 0x6afc, [0x1293] = 0x6afd, [0x1294] = 0x6afe,
  [0x1295] = 0x6aff, [0x1296] = 0x6b00, [0x1297] = 0x6b01, [0x1298] = 0x6b02,
  [0x1299] = 0x6b03, [0x129a] = 0x6b04, [0x129b] = 0x6b05, [0x129c] = 0x6b06,
  [0x129d] = 0x6b07, [0x129e] = 0x6b08, [0x129f] = 0x6b09, [0x12a0] = 0x6b0a,
  [0x12a1] = 0x6b0b, [0x12a2] = 0x6b0c, [0x12a3] = 0x6b0d, [0x12a4] = 0x6b0e,
  [0x12a5] = 0x6b0f, [0x12a6] = 0x6b10, [0x12a7] = 0x6b11, [0x12a8] = 0x6b12,
  [0x12a9] = 0x6b13, [0x12aa] = 0x6b14, [0x12ab] = 0x6b15, [0x12ac] = 0x6b16,
  [0x12ad] = 0x6b17, [0x12ae] = 0x6b18, [0x12af] = 0x6b19, [0x12b0] = 0x6b1a,
  [0x12b1] = 0x6b1b, [0x12b2] = 0x6b1c, [0x12b3] = 0x6b1d, [0x12b4] = 0x6b1e,
  [0x12b5] = 0x6b1f, [0x12b6] = 0x6b25, [0x12b7] = 0x6b26, [0x12b8] = 0x6b28,
  [0x12b9] = 0x6b29, [0x12ba] = 0x6b2a, [0x12bb] = 0x6b2b, [0x12bc] = 0x6b2c,
  [0x12bd] = 0x6b2d, [0x12be] = 0x6b2e, [0x12c0] = 0x6b2f, [0x12c1] = 0x6b30,
  [0x12c2] = 0x6b31, [0x12c3] = 0x6b33, [0x12c4] = 0x6b34, [0x12c5] = 0x6b35,
  [0x12c6] = 0x6b36, [0x12c7] = 0x6b38, [0x12c8] = 0x6b3b, [0x12c9] = 0x6b3c,
  [0x12ca] = 0x6b3d, [0x12cb] = 0x6b3f, [0x12cc] = 0x6b40, [0x12cd] = 0x6b41,
  [0x12ce] = 0x6b42, [0x12cf] = 0x6b44, [0x12d0] = 0x6b45, [0x12d1] = 0x6b48,
  [0x12d2] = 0x6b4a, [0x12d3] = 0x6b4b, [0x12d4] = 0x6b4d, [0x12d5] = 0x6b4e,
  [0x12d6] = 0x6b4f, [0x12d7] = 0x6b50, [0x12d8] = 0x6b51, [0x12d9] = 0x6b52,
  [0x12da] = 0x6b53, [0x12db] = 0x6b54, [0x12dc] = 0x6b55, [0x12dd] = 0x6b56,
  [0x12de] = 0x6b57, [0x12df] = 0x6b58, [0x12e0] = 0x6b5a, [0x12e1] = 0x6b5b,
  [0x12e2] = 0x6b5c, [0x12e3] = 0x6b5d, [0x12e4] = 0x6b5e, [0x12e5] = 0x6b5f,
  [0x12e6] = 0x6b60, [0x12e7] = 0x6b61, [0x12e8] = 0x6b68, [0x12e9] = 0x6b69,
  [0x12ea] = 0x6b6b, [0x12eb] = 0x6b6c, [0x12ec] = 0x6b6d, [0x12ed] = 0x6b6e,
  [0x12ee] = 0x6b6f, [0x12ef] = 0x6b70, [0x12f0] = 0x6b71, [0x12f1] = 0x6b72,
  [0x12f2] = 0x6b73, [0x12f3] = 0x6b74, [0x12f4] = 0x6b75, [0x12f5] = 0x6b76,
  [0x12f6] = 0x6b77, [0x12f7] = 0x6b78, [0x12f8] = 0x6b7a, [0x12f9] = 0x6b7d,
  [0x12fa] = 0x6b7e, [0x12fb] = 0x6b7f, [0x12fc] = 0x6b80, [0x12fd] = 0x6b85,
  [0x12fe] = 0x6b88, [0x1300] = 0x6b8c, [0x1301] = 0x6b8e, [0x1302] = 0x6b8f,
  [0x1303] = 0x6b90, [0x1304] = 0x6b91, [0x1305] = 0x6b94, [0x1306] = 0x6b95,
  [0x1307] = 0x6b97, [0x1308] = 0x6b98, [0x1309] = 0x6b99, [0x130a] = 0x6b9c,
  [0x130b] = 0x6b9d, [0x130c] = 0x6b9e, [0x130d] = 0x6b9f, [0x130e] = 0x6ba0,
  [0x130f] = 0x6ba2, [0x1310] = 0x6ba3, [0x1311] = 0x6ba4, [0x1312] = 0x6ba5,
  [0x1313] = 0x6ba6, [0x1314] = 0x6ba7, [0x1315] = 0x6ba8, [0x1316] = 0x6ba9,
  [0x1317] = 0x6bab, [0x1318] = 0x6bac, [0x1319] = 0x6bad, [0x131a] = 0x6bae,
  [0x131b] = 0x6baf, [0x131c] = 0x6bb0, [0x131d] = 0x6bb1, [0x131e] = 0x6bb2,
  [0x131f] = 0x6bb6, [0x1320] = 0x6bb8, [0x1321] = 0x6bb9, [0x1322] = 0x6bba,
  [0x1323] = 0x6bbb, [0x1324] = 0x6bbc, [0x1325] = 0x6bbd, [0x1326] = 0x6bbe,
  [0x1327] = 0x6bc0, [0x1328] = 0x6bc3, [0x1329] = 0x6bc4, [0x132a] = 0x6bc6,
  [0x132b] = 0x6bc7, [0x132c] = 0x6bc8, [0x132d] = 0x6bc9, [0x132e] = 0x6bca,
  [0x132f] = 0x6bcc, [0x1330] = 0x6bce, [0x1331] = 0x6bd0, [0x1332] = 0x6bd1,
  [0x1333] = 0x6bd8, [0x1334] = 0x6bda, [0x1335] = 0x6bdc, [0x1336] = 0x6bdd,
  [0x1337] = 0x6bde, [0x1338] = 0x6bdf, [0x1339] = 0x6be0, [0x133a] = 0x6be2,
  [0x133b] = 0x6be3, [0x133c] = 0x6be4, [0x133d] = 0x6be5, [0x133e] = 0x6be6,
  [0x133f] = 0x6be7, [0x1340] = 0x6be8, [0x1341] = 0x6be9, [0x1342] = 0x6bec,
  [0x1343] = 0x6bed, [0x1344] = 0x6bee, [0x1345] = 0x6bf0, [0x1346] = 0x6bf1,
  [0x1347] = 0x6bf2, [0x1348] = 0x6bf4, [0x1349] = 0x6bf6, [0x134a] = 0x6bf7,
  [0x134b] = 0x6bf8, [0x134c] = 0x6bfa, [0x134d] = 0x6bfb, [0x134e] = 0x6bfc,
  [0x134f] = 0x6bfe, [0x1350] = 0x6bff, [0x1351] = 0x6c00, [0x1352] = 0x6c01,
  [0x1353] = 0x6c02, [0x1354] = 0x6c03, [0x1355] = 0x6c04, [0x1356] = 0x6c08,
  [0x1357] = 0x6c09, [0x1358] = 0x6c0a, [0x1359] = 0x6c0b, [0x135a] = 0x6c0c,
  [0x135b] = 0x6c0e, [0x135c] = 0x6c12, [0x135d] = 0x6c17, [0x135e] = 0x6c1c,
  [0x135f] = 0x6c1d, [0x1360] = 0x6c1e, [0x1361] = 0x6c20, [0x1362] = 0x6c23,
  [0x1363] = 0x6c25, [0x1364] = 0x6c2b, [0x1365] = 0x6c2c, [0x1366] = 0x6c2d,
  [0x1367] = 0x6c31, [0x1368] = 0x6c33, [0x1369] = 0x6c36, [0x136a] = 0x6c37,
  [0x136b] = 0x6c39, [0x136c] = 0x6c3a, [0x136d] = 0x6c3b, [0x136e] = 0x6c3c,
  [0x136f] = 0x6c3e, [0x1370] = 0x6c3f, [0x1371] = 0x6c43, [0x1372] = 0x6c44,
  [0x1373] = 0x6c45, [0x1374] = 0x6c48, [0x1375] = 0x6c4b, [0x1376] = 0x6c4c,
  [0x1377] = 0x6c4d, [0x1378] = 0x6c4e, [0x1379] = 0x6c4f, [0x137a] = 0x6c51,
  [0x137b] = 0x6c52, [0x137c] = 0x6c53, [0x137d] = 0x6c56, [0x137e] = 0x6c58,
  [0x1380] = 0x6c59, [0x1381] = 0x6c5a, [0x1382] = 0x6c62, [0x1383] = 0x6c63,
  [0x1384] = 0x6c65, [0x1385] = 0x6c66, [0x1386] = 0x6c67, [0x1387] = 0x6c6b,
  [0x1388] = 0x6c6c, [0x1389] = 0x6c6d, [0x138a] = 0x6c6e, [0x138b] = 0x6c6f,
  [0x138c] = 0x6c71, [0x138d] = 0x6c73, [0x138e] = 0x6c75, [0x138f] = 0x6c77,
  [0x1390] = 0x6c78, [0x1391] = 0x6c7a, [0x1392] = 0x6c7b, [0x1393] = 0x6c7c,
  [0x1394] = 0x6c7f, [0x1395] = 0x6c80, [0x1396] = 0x6c84, [0x1397] = 0x6c87,
  [0x1398] = 0x6c8a, [0x1399] = 0x6c8b, [0x139a] = 0x6c8d, [0x139b] = 0x6c8e,
  [0x139c] = 0x6c91, [0x139d] = 0x6c92, [0x139e] = 0x6c95, [0x139f] = 0x6c96,
  [0x13a0] = 0x6c97, [0x13a1] = 0x6c98, [0x13a2] = 0x6c9a, [0x13a3] = 0x6c9c,
  [0x13a4] = 0x6c9d, [0x13a5] = 0x6c9e, [0x13a6] = 0x6ca0, [0x13a7] = 0x6ca2,
  [0x13a8] = 0x6ca8, [0x13a9] = 0x6cac, [0x13aa] = 0x6caf, [0x13ab] = 0x6cb0,
  [0x13ac] = 0x6cb4, [0x13ad] = 0x6cb5, [0x13ae] = 0x6cb6, [0x13af] = 0x6cb7,
  [0x13b0] = 0x6cba, [0x13b1] = 0x6cc0, [0x13b2] = 0x6cc1, [0x13b3] = 0x6cc2,
  [0x13b4] = 0x6cc3, [0x13b5] = 0x6cc6, [0x13b6] = 0x6cc7, [0x13b7] = 0x6cc8,
  [0x13b8] = 0x6ccb, [0x13b9] = 0x6ccd, [0x13ba] = 0x6cce, [0x13bb] = 0x6ccf,
  [0x13bc] = 0x6cd1, [0x13bd] = 0x6cd2, [0x13be] = 0x6cd8, [0x13c0] = 0x6cd9,
  [0x13c1] = 0x6cda, [0x13c2] = 0x6cdc, [0x13c3] = 0x6cdd, [0x13c4] = 0x6cdf,
  [0x13c5] = 0x6ce4, [0x13c6] = 0x6ce6, [0x13c7] = 0x6ce7, [0x13c8] = 0x6ce9,
  [0x13c9] = 0x6cec, [0x13ca] = 0x6ced, [0x13cb] = 0x6cf2, [0x13cc] = 0x6cf4,
  [0x13cd] = 0x6cf9, [0x13ce] = 0x6cff, [0x13cf] = 0x6d00, [0x13d0] = 0x6d02,
  [0x13d1] = 0x6d03, [0x13d2] = 0x6d05, [0x13d3] = 0x6d06, [0x13d4] = 0x6d08,
  [0x13d5] = 0x6d09, [0x13d6] = 0x6d0a, [0x13d7] = 0x6d0d, [0x13d8] = 0x6d0f,
  [0x13d9] = 0x6d10, [0x13da] = 0x6d11, [0x13db] = 0x6d13, [0x13dc] = 0x6d14,
  [0x13dd] = 0x6d15, [0x13de] = 0x6d16, [0x13df] = 0x6d18, [0x13e0] = 0x6d1c,
  [0x13e1] = 0x6d1d, [0x13e2] = 0x6d1f, [0x13e3] = 0x6d20, [0x13e4] = 0x6d21,
  [0x13e5] = 0x6d22, [0x13e6] = 0x6d23, [0x13e7] = 0x6d24, [0x13e8] = 0x6d26,
  [0x13e9] = 0x6d28, [0x13ea] = 0x6d29, [0x13eb] = 0x6d2c, [0x13ec] = 0x6d2d,
  [0x13ed] = 0x6d2f, [0x13ee] = 0x6d30, [0x13ef] = 0x6d34, [0x13f0] = 0x6d36,
  [0x13f1] = 0x6d37, [0x13f2] = 0x6d38, [0x13f3] = 0x6d3a, [0x13f4] = 0x6d3f,
  [0x13f5] = 0x6d40, [0x13f6] = 0x6d42, [0x13f7] = 0x6d44, [0x13f8] = 0x6d49,
  [0x13f9] = 0x6d4c, [0x13fa] = 0x6d50, [0x13fb] = 0x6d55, [0x13fc] = 0x6d56,
  [0x13fd] = 0x6d57, [0x13fe] = 0x6d58, [0x13ff] = 0x6d5b, [0x1400] = 0x6d5d,
  [0x1401] = 0x6d5f, [0x1402] = 0x6d61, [0x1403] = 0x6d62, [0x1404] = 0x6d64,
  [0x1405] = 0x6d65, [0x1406] = 0x6d67, [0x1407] = 0x6d68, [0x1408] = 0x6d6b,
  [0x1409] = 0x6d6c, [0x140a] = 0x6d6d, [0x140b] = 0x6d70, [0x140c] = 0x6d71,
  [0x140d] = 0x6d72, [0x140e] = 0x6d73, [0x140f] = 0x6d75, [0x1410] = 0x6d76,
  [0x1411] = 0x6d79, [0x1412] = 0x6d7a, [0x1413] = 0x6d7b, [0x1414] = 0x6d7d,
  [0x1415] = 0x6d7e, [0x1416] = 0x6d7f, [0x1417] = 0x6d80, [0x1418] = 0x6d81,
  [0x1419] = 0x6d83, [0x141a] = 0x6d84, [0x141b] = 0x6d86, [0x141c] = 0x6d87,
  [0x141d] = 0x6d8a, [0x141e] = 0x6d8b, [0x141f] = 0x6d8d, [0x1420] = 0x6d8f,
  [0x1421] = 0x6d90, [0x1422] = 0x6d92, [0x1423] = 0x6d96, [0x1424] = 0x6d97,
  [0x1425] = 0x6d98, [0x1426] = 0x6d99, [0x1427] = 0x6d9a, [0x1428] = 0x6d9c,
  [0x1429] = 0x6da2, [0x142a] = 0x6da5, [0x142b] = 0x6dac, [0x142c] = 0x6dad,
  [0x142d] = 0x6db0, [0x142e] = 0x6db1, [0x142f] = 0x6db3, [0x1430] = 0x6db4,
  [0x1431] = 0x6db6, [0x1432] = 0x6db7, [0x1433] = 0x6db9, [0x1434] = 0x6dba,
  [0x1435] = 0x6dbb, [0x1436] = 0x6dbc, [0x1437] = 0x6dbd, [0x1438] = 0x6dbe,
  [0x1439] = 0x6dc1, [0x143a] = 0x6dc2, [0x143b] = 0x6dc3, [0x143c] = 0x6dc8,
  [0x143d] = 0x6dc9, [0x143e] = 0x6dca, [0x1440] = 0x6dcd, [0x1441] = 0x6dce,
  [0x1442] = 0x6dcf, [0x1443] = 0x6dd0, [0x1444] = 0x6dd2, [0x1445] = 0x6dd3,
  [0x1446] = 0x6dd4, [0x1447] = 0x6dd5, [0x1448] = 0x6dd7, [0x1449] = 0x6dda,
  [0x144a] = 0x6ddb, [0x144b] = 0x6ddc, [0x144c] = 0x6ddf, [0x144d] = 0x6de2,
  [0x144e] = 0x6de3, [0x144f] = 0x6de5, [0x1450] = 0x6de7, [0x1451] = 0x6de8,
  [0x1452] = 0x6de9, [0x1453] = 0x6dea, [0x1454] = 0x6ded, [0x1455] = 0x6def,
  [0x1456] = 0x6df0, [0x1457] = 0x6df2, [0x1458] = 0x6df4, [0x1459] = 0x6df5,
  [0x145a] = 0x6df6, [0x145b] = 0x6df8, [0x145c] = 0x6dfa, [0x145d] = 0x6dfd,
  [0x145e] = 0x6dfe, [0x145f] = 0x6dff, [0x1460] = 0x6e00, [0x1461] = 0x6e01,
  [0x1462] = 0x6e02, [0x1463] = 0x6e03, [0x1464] = 0x6e04, [0x1465] = 0x6e06,
  [0x1466] = 0x6e07, [0x1467] = 0x6e08, [0x1468] = 0x6e09, [0x1469] = 0x6e0b,
  [0x146a] = 0x6e0f, [0x146b] = 0x6e12, [0x146c] = 0x6e13, [0x146d] = 0x6e15,
  [0x146e] = 0x6e18, [0x146f] = 0x6e19, [0x1470] = 0x6e1b, [0x1471] = 0x6e1c,
  [0x1472] = 0x6e1e, [0x1473] = 0x6e1f, [0x1474] = 0x6e22, [0x1475] = 0x6e26,
  [0x1476] = 0x6e27, [0x1477] = 0x6e28, [0x1478] = 0x6e2a, [0x1479] = 0x6e2c,
  [0x147a] = 0x6e2e, [0x147b] = 0x6e30, [0x147c] = 0x6e31, [0x147d] = 0x6e33,
  [0x147e] = 0x6e35, [0x1480] = 0x6e36, [0x1481] = 0x6e37, [0x1482] = 0x6e39,
  [0x1483] = 0x6e3b, [0x1484] = 0x6e3c, [0x1485] = 0x6e3d, [0x1486] = 0x6e3e,
  [0x1487] = 0x6e3f, [0x1488] = 0x6e40, [0x1489] = 0x6e41, [0x148a] = 0x6e42,
  [0x148b] = 0x6e45, [0x148c] = 0x6e46, [0x148d] = 0x6e47, [0x148e] = 0x6e48,
  [0x148f] = 0x6e49, [0x1490] = 0x6e4a, [0x1491] = 0x6e4b, [0x1492] = 0x6e4c,
  [0x1493] = 0x6e4f, [0x1494] = 0x6e50, [0x1495] = 0x6e51, [0x1496] = 0x6e52,
  [0x1497] = 0x6e55, [0x1498] = 0x6e57, [0x1499] = 0x6e59, [0x149a] = 0x6e5a,
  [0x149b] = 0x6e5c, [0x149c] = 0x6e5d, [0x149d] = 0x6e5e, [0x149e] = 0x6e60,
  [0x149f] = 0x6e61, [0x14a0] = 0x6e62, [0x14a1] = 0x6e63, [0x14a2] = 0x6e64,
  [0x14a3] = 0x6e65, [0x14a4] = 0x6e66, [0x14a5] = 0x6e67, [0x14a6] = 0x6e68,
  [0x14a7] = 0x6e69, [0x14a8] = 0x6e6a, [0x14a9] = 0x6e6c, [0x14aa] = 0x6e6d,
  [0x14ab] = 0x6e6f, [0x14ac] = 0x6e70, [0x14ad] = 0x6e71, [0x14ae] = 0x6e72,
  [0x14af] = 0x6e73, [0x14b0] = 0x6e74, [0x14b1] = 0x6e75, [0x14b2] = 0x6e76,
  [0x14b3] = 0x6e77, [0x14b4] = 0x6e78, [0x14b5] = 0x6e79, [0x14b6] = 0x6e7a,
  [0x14b7] = 0x6e7b, [0x14b8] = 0x6e7c, [0x14b9] = 0x6e7d, [0x14ba] = 0x6e80,
  [0x14bb] = 0x6e81, [0x14bc] = 0x6e82, [0x14bd] = 0x6e84, [0x14be] = 0x6e87,
  [0x14bf] = 0x6e88, [0x14c0] = 0x6e8a, [0x14c1] = 0x6e8b, [0x14c2] = 0x6e8c,
  [0x14c3] = 0x6e8d, [0x14c4] = 0x6e8e, [0x14c5] = 0x6e91, [0x14c6] = 0x6e92,
  [0x14c7] = 0x6e93, [0x14c8] = 0x6e94, [0x14c9] = 0x6e95, [0x14ca] = 0x6e96,
  [0x14cb] = 0x6e97, [0x14cc] = 0x6e99, [0x14cd] = 0x6e9a, [0x14ce] = 0x6e9b,
  [0x14cf] = 0x6e9d, [0x14d0] = 0x6e9e, [0x14d1] = 0x6ea0, [0x14d2] = 0x6ea1,
  [0x14d3] = 0x6ea3, [0x14d4] = 0x6ea4, [0x14d5] = 0x6ea6, [0x14d6] = 0x6ea8,
  [0x14d7] = 0x6ea9, [0x14d8] = 0x6eab, [0x14d9] = 0x6eac, [0x14da] = 0x6ead,
  [0x14db] = 0x6eae, [0x14dc] = 0x6eb0, [0x14dd] = 0x6eb3, [0x14de] = 0x6eb5,
  [0x14df] = 0x6eb8, [0x14e0] = 0x6eb9, [0x14e1] = 0x6ebc, [0x14e2] = 0x6ebe,
  [0x14e3] = 0x6ebf, [0x14e4] = 0x6ec0, [0x14e5] = 0x6ec3, [0x14e6] = 0x6ec4,
  [0x14e7] = 0x6ec5, [0x14e8] = 0x6ec6, [0x14e9] = 0x6ec8, [0x14ea] = 0x6ec9,
  [0x14eb] = 0x6eca, [0x14ec] = 0x6ecc, [0x14ed] = 0x6ecd, [0x14ee] = 0x6ece,
  [0x14ef] = 0x6ed0, [0x14f0] = 0x6ed2, [0x14f1] = 0x6ed6, [0x14f2] = 0x6ed8,
  [0x14f3] = 0x6ed9, [0x14f4] = 0x6edb, [0x14f5] = 0x6edc, [0x14f6] = 0x6edd,
  [0x14f7] = 0x6ee3, [0x14f8] = 0x6ee7, [0x14f9] = 0x6eea, [0x14fa] = 0x6eeb,
  [0x14fb] = 0x6eec, [0x14fc] = 0x6eed, [0x14fd] = 0x6eee, [0x14fe] = 0x6eef,
  [0x1500] = 0x6ef0, [0x1501] = 0x6ef1, [0x1502] = 0x6ef2, [0x1503] = 0x6ef3,
  [0x1504] = 0x6ef5, [0x1505] = 0x6ef6, [0x1506] = 0x6ef7, [0x1507] = 0x6ef8,
  [0x1508] = 0x6efa, [0x1509] = 0x6efb, [0x150a] = 0x6efc, [0x150b] = 0x6efd,
  [0x150c] = 0x6efe, [0x150d] = 0x6eff, [0x150e] = 0x6f00, [0x150f] = 0x6f01,
  [0x1510] = 0x6f03, [0x1511] = 0x6f04, [0x1512] = 0x6f05, [0x1513] = 0x6f07,
  [0x1514] = 0x6f08, [0x1515] = 0x6f0a, [0x1516] = 0x6f0b, [0x1517] = 0x6f0c,
  [0x1518] = 0x6f0d, [0x1519] = 0x6f0e, [0x151a] = 0x6f10, [0x151b] = 0x6f11,
  [0x151c] = 0x6f12, [0x151d] = 0x6f16, [0x151e] = 0x6f17, [0x151f] = 0x6f18,
  [0x1520] = 0x6f19, [0x1521] = 0x6f1a, [0x1522] = 0x6f1b, [0x1523] = 0x6f1c,
  [0x1524] = 0x6f1d, [0x1525] = 0x6f1e, [0x1526] = 0x6f1f, [0x1527] = 0x6f21,
  [0x1528] = 0x6f22, [0x1529] = 0x6f23, [0x152a] = 0x6f25, [0x152b] = 0x6f26,
  [0x152c] = 0x6f27, [0x152d] = 0x6f28, [0x152e] = 0x6f2c, [0x152f] = 0x6f2e,
  [0x1530] = 0x6f30, [0x1531] = 0x6f32, [0x1532] = 0x6f34, [0x1533] = 0x6f35,
  [0x1534] = 0x6f37, [0x1535] = 0x6f38, [0x1536] = 0x6f39, [0x1537] = 0x6f3a,
  [0x1538] = 0x6f3b, [0x1539] = 0x6f3c, [0x153a] = 0x6f3d, [0x153b] = 0x6f3f,
  [0x153c] = 0x6f40, [0x153d] = 0x6f41, [0x153e] = 0x6f42, [0x1540] = 0x6f43,
  [0x1541] = 0x6f44, [0x1542] = 0x6f45, [0x1543] = 0x6f48, [0x1544] = 0x6f49,
  [0x1545] = 0x6f4a, [0x1546] = 0x6f4c, [0x1547] = 0x6f4e, [0x1548] = 0x6f4f,
  [0x1549] = 0x6f50, [0x154a] = 0x6f51, [0x154b] = 0x6f52, [0x154c] = 0x6f53,
  [0x154d] = 0x6f54, [0x154e] = 0x6f55, [0x154f] = 0x6f56, [0x1550] = 0x6f57,
  [0x1551] = 0x6f59, [0x1552] = 0x6f5a, [0x1553] = 0x6f5b, [0x1554] = 0x6f5d,
  [0x1555] = 0x6f5f, [0x1556] = 0x6f60, [0x1557] = 0x6f61, [0x1558] = 0x6f63,
  [0x1559] = 0x6f64, [0x155a] = 0x6f65, [0x155b] = 0x6f67, [0x155c] = 0x6f68,
  [0x155d] = 0x6f69, [0x155e] = 0x6f6a, [0x155f] = 0x6f6b, [0x1560] = 0x6f6c,
  [0x1561] = 0x6f6f, [0x1562] = 0x6f70, [0x1563] = 0x6f71, [0x1564] = 0x6f73,
  [0x1565] = 0x6f75, [0x1566] = 0x6f76, [0x1567] = 0x6f77, [0x1568] = 0x6f79,
  [0x1569] = 0x6f7b, [0x156a] = 0x6f7d, [0x156b] = 0x6f7e, [0x156c] = 0x6f7f,
  [0x156d] = 0x6f80, [0x156e] = 0x6f81, [0x156f] = 0x6f82, [0x1570] = 0x6f83,
  [0x1571] = 0x6f85, [0x1572] = 0x6f86, [0x1573] = 0x6f87, [0x1574] = 0x6f8a,
  [0x1575] = 0x6f8b, [0x1576] = 0x6f8f, [0x1577] = 0x6f90, [0x1578] = 0x6f91,
  [0x1579] = 0x6f92, [0x157a] = 0x6f93, [0x157b] = 0x6f94, [0x157c] = 0x6f95,
  [0x157d] = 0x6f96, [0x157e] = 0x6f97, [0x157f] = 0x6f98, [0x1580] = 0x6f99,
  [0x1581] = 0x6f9a, [0x1582] = 0x6f9b, [0x1583] = 0x6f9d, [0x1584] = 0x6f9e,
  [0x1585] = 0x6f9f, [0x1586] = 0x6fa0, [0x1587] = 0x6fa2, [0x1588] = 0x6fa3,
  [0x1589] = 0x6fa4, [0x158a] = 0x6fa5, [0x158b] = 0x6fa6, [0x158c] = 0x6fa8,
  [0x158d] = 0x6fa9, [0x158e] = 0x6faa, [0x158f] = 0x6fab, [0x1590] = 0x6fac,
  [0x1591] = 0x6fad, [0x1592] = 0x6fae, [0x1593] = 0x6faf, [0x1594] = 0x6fb0,
  [0x1595] = 0x6fb1, [0x1596] = 0x6fb2, [0x1597] = 0x6fb4, [0x1598] = 0x6fb5,
  [0x1599] = 0x6fb7, [0x159a] = 0x6fb8, [0x159b] = 0x6fba, [0x159c] = 0x6fbb,
  [0x159d] = 0x6fbc, [0x159e] = 0x6fbd, [0x159f] = 0x6fbe, [0x15a0] = 0x6fbf,
  [0x15a1] = 0x6fc1, [0x15a2] = 0x6fc3, [0x15a3] = 0x6fc4, [0x15a4] = 0x6fc5,
  [0x15a5] = 0x6fc6, [0x15a6] = 0x6fc7, [0x15a7] = 0x6fc8, [0x15a8] = 0x6fca,
  [0x15a9] = 0x6fcb, [0x15aa] = 0x6fcc, [0x15ab] = 0x6fcd, [0x15ac] = 0x6fce,
  [0x15ad] = 0x6fcf, [0x15ae] = 0x6fd0, [0x15af] = 0x6fd3, [0x15b0] = 0x6fd4,
  [0x15b1] = 0x6fd5, [0x15b2] = 0x6fd6, [0x15b3] = 0x6fd7, [0x15b4] = 0x6fd8,
  [0x15b5] = 0x6fd9, [0x15b6] = 0x6fda, [0x15b7] = 0x6fdb, [0x15b8] = 0x6fdc,
  [0x15b9] = 0x6fdd, [0x15ba] = 0x6fdf, [0x15bb] = 0x6fe2, [0x15bc] = 0x6fe3,
  [0x15bd] = 0x6fe4, [0x15be] = 0x6fe5, [0x15c0] = 0x6fe6, [0x15c1] = 0x6fe7,
  [0x15c2] = 0x6fe8, [0x15c3] = 0x6fe9, [0x15c4] = 0x6fea, [0x15c5] = 0x6feb,
  [0x15c6] = 0x6fec, [0x15c7] = 0x6fed, [0x15c8] = 0x6ff0, [0x15c9] = 0x6ff1,
  [0x15ca] = 0x6ff2, [0x15cb] = 0x6ff3, [0x15cc] = 0x6ff4, [0x15cd] = 0x6ff5,
  [0x15ce] = 0x6ff6, [0x15cf] = 0x6ff7, [0x15d0] = 0x6ff8, [0x15d1] = 0x6ff9,
  [0x15d2] = 0x6ffa, [0x15d3] = 0x6ffb, [0x15d4] = 0x6ffc, [0x15d5] = 0x6ffd,
  [0x15d6] = 0x6ffe, [0x15d7] = 0x6fff, [0x15d8] = 0x7000, [0x15d9] = 0x7001,
  [0x15da] = 0x7002, [0x15db] = 0x7003, [0x15dc] = 0x7004, [0x15dd] = 0x7005,
  [0x15de] = 0x7006, [0x15df] = 0x7007, [0x15e0] = 0x7008, [0x15e1] = 0x7009,
  [0x15e2] = 0x700a, [0x15e3] = 0x700b, [0x15e4] = 0x700c, [0x15e5] = 0x700d,
  [0x15e6] = 0x700e, [0x15e7] = 0x700f, [0x15e8] = 0x7010, [0x15e9] = 0x7012,
  [0x15ea] = 0x7013, [0x15eb] = 0x7014, [0x15ec] = 0x7015, [0x15ed] = 0x7016,
  [0x15ee] = 0x7017, [0x15ef] = 0x7018, [0x15f0] = 0x7019, [0x15f1] = 0x701c,
  [0x15f2] = 0x701d, [0x15f3] = 0x701e, [0x15f4] = 0x701f, [0x15f5] = 0x7020,
  [0x15f6] = 0x7021, [0x15f7] = 0x7022, [0x15f8] = 0x7024, [0x15f9] = 0x7025,
  [0x15fa] = 0x7026, [0x15fb] = 0x7027, [0x15fc] = 0x7028, [0x15fd] = 0x7029,
  [0x15fe] = 0x702a, [0x1600] = 0x702b, [0x1601] = 0x702c, [0x1602] = 0x702d,
  [0x1603] = 0x702e, [0x1604] = 0x702f, [0x1605] = 0x7030, [0x1606] = 0x7031,
  [0x1607] = 0x7032, [0x1608] = 0x7033, [0x1609] = 0x7034, [0x160a] = 0x7036,
  [0x160b] = 0x7037, [0x160c] = 0x7038, [0x160d] = 0x703a, [0x160e] = 0x703b,
  [0x160f] = 0x703c, [0x1610] = 0x703d, [0x1611] = 0x703e, [0x1612] = 0x703f,
  [0x1613] = 0x7040, [0x1614] = 0x7041, [0x1615] = 0x7042, [0x1616] = 0x7043,
  [0x1617] = 0x7044, [0x1618] = 0x7045, [0x1619] = 0x7046, [0x161a] = 0x7047,
  [0x161b] = 0x7048, [0x161c] = 0x7049, [0x161d] = 0x704a, [0x161e] = 0x704b,
  [0x161f] = 0x704d, [0x1620] = 0x704e, [0x1621] = 0x7050, [0x1622] = 0x7051,
  [0x1623] = 0x7052, [0x1624] = 0x7053, [0x1625] = 0x7054, [0x1626] = 0x7055,
  [0x1627] = 0x7056, [0x1628] = 0x7057, [0x1629] = 0x7058, [0x162a] = 0x7059,
  [0x162b] = 0x705a, [0x162c] = 0x705b, [0x162d] = 0x705c, [0x162e] = 0x705d,
  [0x162f] = 0x705f, [0x1630] = 0x7060, [0x1631] = 0x7061, [0x1632] = 0x7062,
  [0x1633] = 0x7063, [0x1634] = 0x7064, [0x1635] = 0x7065, [0x1636] = 0x7066,
  [0x1637] = 0x7067, [0x1638] = 0x7068, [0x1639] = 0x7069, [0x163a] = 0x706a,
  [0x163b] = 0x706e, [0x163c] = 0x7071, [0x163d] = 0x7072, [0x163e] = 0x7073,
  [0x163f] = 0x7074, [0x1640] = 0x7077, [0x1641] = 0x7079, [0x1642] = 0x707a,
  [0x1643] = 0x707b, [0x1644] = 0x707d, [0x1645] = 0x7081, [0x1646] = 0x7082,
  [0x1647] = 0x7083, [0x1648] = 0x7084, [0x1649] = 0x7086, [0x164a] = 0x7087,
  [0x164b] = 0x7088, [0x164c] = 0x708b, [0x164d] = 0x708c, [0x164e] = 0x708d,
  [0x164f] = 0x708f, [0x1650] = 0x7090, [0x1651] = 0x7091, [0x1652] = 0x7093,
  [0x1653] = 0x7097, [0x1654] = 0x7098, [0x1655] = 0x709a, [0x1656] = 0x709b,
  [0x1657] = 0x709e, [0x1658] = 0x709f, [0x1659] = 0x70a0, [0x165a] = 0x70a1,
  [0x165b] = 0x70a2, [0x165c] = 0x70a3, [0x165d] = 0x70a4, [0x165e] = 0x70a5,
  [0x165f] = 0x70a6, [0x1660] = 0x70a7, [0x1661] = 0x70a8, [0x1662] = 0x70a9,
  [0x1663] = 0x70aa, [0x1664] = 0x70b0, [0x1665] = 0x70b2, [0x1666] = 0x70b4,
  [0x1667] = 0x70b5, [0x1668] = 0x70b6, [0x1669] = 0x70ba, [0x166a] = 0x70be,
  [0x166b] = 0x70bf, [0x166c] = 0x70c4, [0x166d] = 0x70c5, [0x166e] = 0x70c6,
  [0x166f] = 0x70c7, [0x1670] = 0x70c9, [0x1671] = 0x70cb, [0x1672] = 0x70cc,
  [0x1673] = 0x70cd, [0x1674] = 0x70ce, [0x1675] = 0x70cf, [0x1676] = 0x70d0,
  [0x1677] = 0x70d1, [0x1678] = 0x70d2, [0x1679] = 0x70d3, [0x167a] = 0x70d4,
  [0x167b] = 0x70d5, [0x167c] = 0x70d6, [0x167d] = 0x70d7, [0x167e] = 0x70da,
  [0x1680] = 0x70dc, [0x1681] = 0x70dd, [0x1682] = 0x70de, [0x1683] = 0x70e0,
  [0x1684] = 0x70e1, [0x1685] = 0x70e2, [0x1686] = 0x70e3, [0x1687] = 0x70e5,
  [0x1688] = 0x70ea, [0x1689] = 0x70ee, [0x168a] = 0x70f0, [0x168b] = 0x70f1,
  [0x168c] = 0x70f2, [0x168d] = 0x70f3, [0x168e] = 0x70f4, [0x168f] = 0x70f5,
  [0x1690] = 0x70f6, [0x1691] = 0x70f8, [0x1692] = 0x70fa, [0x1693] = 0x70fb,
  [0x1694] = 0x70fc, [0x1695] = 0x70fe, [0x1696] = 0x70ff, [0x1697] = 0x7100,
  [0x1698] = 0x7101, [0x1699] = 0x7102, [0x169a] = 0x7103, [0x169b] = 0x7104,
  [0x169c] = 0x7105, [0x169d] = 0x7106, [0x169e] = 0x7107, [0x169f] = 0x7108,
  [0x16a0] = 0x710b, [0x16a1] = 0x710c, [0x16a2] = 0x710d, [0x16a3] = 0x710e,
  [0x16a4] = 0x710f, [0x16a5] = 0x7111, [0x16a6] = 0x7112, [0x16a7] = 0x7114,
  [0x16a8] = 0x7117, [0x16a9] = 0x711b, [0x16aa] = 0x711c, [0x16ab] = 0x711d,
  [0x16ac] = 0x711e, [0x16ad] = 0x711f, [0x16ae] = 0x7120, [0x16af] = 0x7121,
  [0x16b0] = 0x7122, [0x16b1] = 0x7123, [0x16b2] = 0x7124, [0x16b3] = 0x7125,
  [0x16b4] = 0x7127, [0x16b5] = 0x7128, [0x16b6] = 0x7129, [0x16b7] = 0x712a,
  [0x16b8] = 0x712b, [0x16b9] = 0x712c, [0x16ba] = 0x712d, [0x16bb] = 0x712e,
  [0x16bc] = 0x7132, [0x16bd] = 0x7133, [0x16be] = 0x7134, [0x16c0] = 0x7135,
  [0x16c1] = 0x7137, [0x16c2] = 0x7138, [0x16c3] = 0x7139, [0x16c4] = 0x713a,
  [0x16c5] = 0x713b, [0x16c6] = 0x713c, [0x16c7] = 0x713d, [0x16c8] = 0x713e,
  [0x16c9] = 0x713f, [0x16ca] = 0x7140, [0x16cb] = 0x7141, [0x16cc] = 0x7142,
  [0x16cd] = 0x7143, [0x16ce] = 0x7144, [0x16cf] = 0x7146, [0x16d0] = 0x7147,
  [0x16d1] = 0x7148, [0x16d2] = 0x7149, [0x16d3] = 0x714b, [0x16d4] = 0x714d,
  [0x16d5] = 0x714f, [0x16d6] = 0x7150, [0x16d7] = 0x7151, [0x16d8] = 0x7152,
  [0x16d9] = 0x7153, [0x16da] = 0x7154, [0x16db] = 0x7155, [0x16dc] = 0x7156,
  [0x16dd] = 0x7157, [0x16de] = 0x7158, [0x16df] = 0x7159, [0x16e0] = 0x715a,
  [0x16e1] = 0x715b, [0x16e2] = 0x715d, [0x16e3] = 0x715f, [0x16e4] = 0x7160,
  [0x16e5] = 0x7161, [0x16e6] = 0x7162, [0x16e7] = 0x7163, [0x16e8] = 0x7165,
  [0x16e9] = 0x7169, [0x16ea] = 0x716a, [0x16eb] = 0x716b, [0x16ec] = 0x716c,
  [0x16ed] = 0x716d, [0x16ee] = 0x716f, [0x16ef] = 0x7170, [0x16f0] = 0x7171,
  [0x16f1] = 0x7174, [0x16f2] = 0x7175, [0x16f3] = 0x7176, [0x16f4] = 0x7177,
  [0x16f5] = 0x7179, [0x16f6] = 0x717b, [0x16f7] = 0x717c, [0x16f8] = 0x717e,
  [0x16f9] = 0x717f, [0x16fa] = 0x7180, [0x16fb] = 0x7181, [0x16fc] = 0x7182,
  [0x16fd] = 0x7183, [0x16fe] = 0x7185, [0x16ff] = 0x7186, [0x1700] = 0x7187,
  [0x1701] = 0x7188, [0x1702] = 0x7189, [0x1703] = 0x718b, [0x1704] = 0x718c,
  [0x1705] = 0x718d, [0x1706] = 0x718e, [0x1707] = 0x7190, [0x1708] = 0x7191,
  [0x1709] = 0x7192, [0x170a] = 0x7193, [0x170b] = 0x7195, [0x170c] = 0x7196,
  [0x170d] = 0x7197, [0x170e] = 0x719a, [0x170f] = 0x719b, [0x1710] = 0x719c,
  [0x1711] = 0x719d, [0x1712] = 0x719e, [0x1713] = 0x71a1, [0x1714] = 0x71a2,
  [0x1715] = 0x71a3, [0x1716] = 0x71a4, [0x1717] = 0x71a5, [0x1718] = 0x71a6,
  [0x1719] = 0x71a7, [0x171a] = 0x71a9, [0x171b] = 0x71aa, [0x171c] = 0x71ab,
  [0x171d] = 0x71ad, [0x171e] = 0x71ae, [0x171f] = 0x71af, [0x1720] = 0x71b0,
  [0x1721] = 0x71b1, [0x1722] = 0x71b2, [0x1723] = 0x71b4, [0x1724] = 0x71b6,
  [0x1725] = 0x71b7, [0x1726] = 0x71b8, [0x1727] = 0x71ba, [0x1728] = 0x71bb,
  [0x1729] = 0x71bc, [0x172a] = 0x71bd, [0x172b] = 0x71be, [0x172c] = 0x71bf,
  [0x172d] = 0x71c0, [0x172e] = 0x71c1, [0x172f] = 0x71c2, [0x1730] = 0x71c4,
  [0x1731] = 0x71c5, [0x1732] = 0x71c6, [0x1733] = 0x71c7, [0x1734] = 0x71c8,
  [0x1735] = 0x71c9, [0x1736] = 0x71ca, [0x1737] = 0x71cb, [0x1738] = 0x71cc,
  [0x1739] = 0x71cd, [0x173a] = 0x71cf, [0x173b] = 0x71d0, [0x173c] = 0x71d1,
  [0x173d] = 0x71d2, [0x173e] = 0x71d3, [0x1740] = 0x71d6, [0x1741] = 0x71d7,
  [0x1742] = 0x71d8, [0x1743] = 0x71d9, [0x1744] = 0x71da, [0x1745] = 0x71db,
  [0x1746] = 0x71dc, [0x1747] = 0x71dd, [0x1748] = 0x71de, [0x1749] = 0x71df,
  [0x174a] = 0x71e1, [0x174b] = 0x71e2, [0x174c] = 0x71e3, [0x174d] = 0x71e4,
  [0x174e] = 0x71e6, [0x174f] = 0x71e8, [0x1750] = 0x71e9, [0x1751] = 0x71ea,
  [0x1752] = 0x71eb, [0x1753] = 0x71ec, [0x1754] = 0x71ed, [0x1755] = 0x71ef,
  [0x1756] = 0x71f0, [0x1757] = 0x71f1, [0x1758] = 0x71f2, [0x1759] = 0x71f3,
  [0x175a] = 0x71f4, [0x175b] = 0x71f5, [0x175c] = 0x71f6, [0x175d] = 0x71f7,
  [0x175e] = 0x71f8, [0x175f] = 0x71fa, [0x1760] = 0x71fb, [0x1761] = 0x71fc,
  [0x1762] = 0x71fd, [0x1763] = 0x71fe, [0x1764] = 0x71ff, [0x1765] = 0x7200,
  [0x1766] = 0x7201, [0x1767] = 0x7202, [0x1768] = 0x7203, [0x1769] = 0x7204,
  [0x176a] = 0x7205, [0x176b] = 0x7207, [0x176c] = 0x7208, [0x176d] = 0x7209,
  [0x176e] = 0x720a, [0x176f] = 0x720b, [0x1770] = 0x720c, [0x1771] = 0x720d,
  [0x1772] = 0x720e, [0x1773] = 0x720f, [0x1774] = 0x7210, [0x1775] = 0x7211,
  [0x1776] = 0x7212, [0x1777] = 0x7213, [0x1778] = 0x7214, [0x1779] = 0x7215,
  [0x177a] = 0x7216, [0x177b] = 0x7217, [0x177c] = 0x7218, [0x177d] = 0x7219,
  [0x177e] = 0x721a, [0x1780] = 0x721b, [0x1781] = 0x721c, [0x1782] = 0x721e,
  [0x1783] = 0x721f, [0x1784] = 0x7220, [0x1785] = 0x7221, [0x1786] = 0x7222,
  [0x1787] = 0x7223, [0x1788] = 0x7224, [0x1789] = 0x7225, [0x178a] = 0x7226,
  [0x178b] = 0x7227, [0x178c] = 0x7229, [0x178d] = 0x722b, [0x178e] = 0x722d,
  [0x178f] = 0x722e, [0x1790] = 0x722f, [0x1791] = 0x7232, [0x1792] = 0x7233,
  [0x1793] = 0x7234, [0x1794] = 0x723a, [0x1795] = 0x723c, [0x1796] = 0x723e,
  [0x1797] = 0x7240, [0x1798] = 0x7241, [0x1799] = 0x7242, [0x179a] = 0x7243,
  [0x179b] = 0x7244, [0x179c] = 0x7245, [0x179d] = 0x7246, [0x179e] = 0x7249,
  [0x179f] = 0x724a, [0x17a0] = 0x724b, [0x17a1] = 0x724e, [0x17a2] = 0x724f,
  [0x17a3] = 0x7250, [0x17a4] = 0x7251, [0x17a5] = 0x7253, [0x17a6] = 0x7254,
  [0x17a7] = 0x7255, [0x17a8] = 0x7257, [0x17a9] = 0x7258, [0x17aa] = 0x725a,
  [0x17ab] = 0x725c, [0x17ac] = 0x725e, [0x17ad] = 0x7260, [0x17ae] = 0x7263,
  [0x17af] = 0x7264, [0x17b0] = 0x7265, [0x17b1] = 0x7268, [0x17b2] = 0x726a,
  [0x17b3] = 0x726b, [0x17b4] = 0x726c, [0x17b5] = 0x726d, [0x17b6] = 0x7270,
  [0x17b7] = 0x7271, [0x17b8] = 0x7273, [0x17b9] = 0x7274, [0x17ba] = 0x7276,
  [0x17bb] = 0x7277, [0x17bc] = 0x7278, [0x17bd] = 0x727b, [0x17be] = 0x727c,
  [0x17bf] = 0x727d, [0x17c0] = 0x7282, [0x17c1] = 0x7283, [0x17c2] = 0x7285,
  [0x17c3] = 0x7286, [0x17c4] = 0x7287, [0x17c5] = 0x7288, [0x17c6] = 0x7289,
  [0x17c7] = 0x728c, [0x17c8] = 0x728e, [0x17c9] = 0x7290, [0x17ca] = 0x7291,
  [0x17cb] = 0x7293, [0x17cc] = 0x7294, [0x17cd] = 0x7295, [0x17ce] = 0x7296,
  [0x17cf] = 0x7297, [0x17d0] = 0x7298, [0x17d1] = 0x7299, [0x17d2] = 0x729a,
  [0x17d3] = 0x729b, [0x17d4] = 0x729c, [0x17d5] = 0x729d, [0x17d6] = 0x729e,
  [0x17d7] = 0x72a0, [0x17d8] = 0x72a1, [0x17d9] = 0x72a2, [0x17da] = 0x72a3,
  [0x17db] = 0x72a4, [0x17dc] = 0x72a5, [0x17dd] = 0x72a6, [0x17de] = 0x72a7,
  [0x17df] = 0x72a8, [0x17e0] = 0x72a9, [0x17e1] = 0x72aa, [0x17e2] = 0x72ab,
  [0x17e3] = 0x72ae, [0x17e4] = 0x72b1, [0x17e5] = 0x72b2, [0x17e6] = 0x72b3,
  [0x17e7] = 0x72b5, [0x17e8] = 0x72ba, [0x17e9] = 0x72bb, [0x17ea] = 0x72bc,
  [0x17eb] = 0x72bd, [0x17ec] = 0x72be, [0x17ed] = 0x72bf, [0x17ee] = 0x72c0,
  [0x17ef] = 0x72c5, [0x17f0] = 0x72c6, [0x17f1] = 0x72c7, [0x17f2] = 0x72c9,
  [0x17f3] = 0x72ca, [0x17f4] = 0x72cb, [0x17f5] = 0x72cc, [0x17f6] = 0x72cf,
  [0x17f7] = 0x72d1, [0x17f8] = 0x72d3, [0x17f9] = 0x72d4, [0x17fa] = 0x72d5,
  [0x17fb] = 0x72d6, [0x17fc] = 0x72d8, [0x17fd] = 0x72da, [0x17fe] = 0x72db,
  [0x1861] = 0x3000, [0x1862] = 0x3001, [0x1863] = 0x3002, [0x1864] = 0x00b7,
  [0x1865] = 0x02c9, [0x1866] = 0x02c7, [0x1867] = 0x00a8, [0x1868] = 0x3003,
  [0x1869] = 0x3005, [0x186a] = 0x2014, [0x186b] = 0xff5e, [0x186c] = 0x2016,
  [0x186d] = 0x2026, [0x186e] = 0x2018, [0x186f] = 0x2019, [0x1870] = 0x201c,
  [0x1871] = 0x201d, [0x1872] = 0x3014, [0x1873] = 0x3015, [0x1874] = 0x3008,
  [0x1875] = 0x3009, [0x1876] = 0x300a, [0x1877] = 0x300b, [0x1878] = 0x300c,
  [0x1879] = 0x300d, [0x187a] = 0x300e, [0x187b] = 0x300f, [0x187c] = 0x3016,
  [0x187d] = 0x3017, [0x187e] = 0x3010, [0x187f] = 0x3011, [0x1880] = 0x00b1,
  [0x1881] = 0x00d7, [0x1882] = 0x00f7, [0x1883] = 0x2236, [0x1884] = 0x2227,
  [0x1885] = 0x2228, [0x1886] = 0x2211, [0x1887] = 0x220f, [0x1888] = 0x222a,
  [0x1889] = 0x2229, [0x188a] = 0x2208, [0x188b] = 0x2237, [0x188c] = 0x221a,
  [0x188d] = 0x22a5, [0x188e] = 0x2225, [0x188f] = 0x2220, [0x1890] = 0x2312,
  [0x1891] = 0x2299, [0x1892] = 0x222b, [0x1893] = 0x222e, [0x1894] = 0x2261,
  [0x1895] = 0x224c, [0x1896] = 0x2248, [0x1897] = 0x223d, [0x1898] = 0x221d,
  [0x1899] = 0x2260, [0x189a] = 0x226e, [0x189b] = 0x226f, [0x189c] = 0x2264,
  [0x189d] = 0x2265, [0x189e] = 0x221e, [0x189f] = 0x2235, [0x18a0] = 0x2234,
  [0x18a1] = 0x2642, [0x18a2] = 0x2640, [0x18a3] = 0x00b0, [0x18a4] = 0x2032,
  [0x18a5] = 0x2033, [0x18a6] = 0x2103, [0x18a7] = 0xff04, [0x18a8] = 0x00a4,
  [0x18a9] = 0xffe0, [0x18aa] = 0xffe1, [0x18ab] = 0x2030, [0x18ac] = 0x00a7,
  [0x18ad] = 0x2116, [0x18ae] = 0x2606, [0x18af] = 0x2605, [0x18b0] = 0x25cb,
  [0x18b1] = 0x25cf, [0x18b2] = 0x25ce, [0x18b3] = 0x25c7, [0x18b4] = 0x25c6,
  [0x18b5] = 0x25a1, [0x18b6] = 0x25a0, [0x18b7] = 0x25b3, [0x18b8] = 0x25b2,
  [0x18b9] = 0x203b, [0x18ba] = 0x2192, [0x18bb] = 0x2190, [0x18bc] = 0x2191,
  [0x18bd] = 0x2193, [0x18be] = 0x3013, [0x1921] = 0x2170, [0x1922] = 0x2171,
  [0x1923] = 0x2172, [0x1924] = 0x2173, [0x1925] = 0x2174, [0x1926] = 0x2175,
  [0x1927] = 0x2176, [0x1928] = 0x2177, [0x1929] = 0x2178, [0x192a] = 0x2179,
  [0x1931] = 0x2488, [0x1932] = 0x2489, [0x1933] = 0x248a, [0x1934] = 0x248b,
  [0x1935] = 0x248c, [0x1936] = 0x248d, [0x1937] = 0x248e, [0x1938] = 0x248f,
  [0x1939] = 0x2490, [0x193a] = 0x2491, [0x193b] = 0x2492, [0x193c] = 0x2493,
  [0x193d] = 0x2494, [0x193e] = 0x2495, [0x193f] = 0x2496, [0x1940] = 0x2497,
  [0x1941] = 0x2498, [0x1942] = 0x2499, [0x1943] = 0x249a, [0x1944] = 0x249b,
  [0x1945] = 0x2474, [0x1946] = 0x2475, [0x1947] = 0x2476, [0x1948] = 0x2477,
  [0x1949] = 0x2478, [0x194a] = 0x2479, [0x194b] = 0x247a, [0x194c] = 0x247b,
  [0x194d] = 0x247c, [0x194e] = 0x247d, [0x194f] = 0x247e, [0x1950] = 0x247f,
  [0x1951] = 0x2480, [0x1952] = 0x2481, [0x1953] = 0x2482, [0x1954] = 0x2483,
  [0x1955] = 0x2484, [0x1956] = 0x2485, [0x1957] = 0x2486, [0x1958] = 0x2487,
  [0x1959] = 0x2460, [0x195a] = 0x2461, [0x195b] = 0x2462, [0x195c] = 0x2463,
  [0x195d] = 0x2464, [0x195e] = 0x2465, [0x195f] = 0x2466, [0x1960] = 0x2467,
  [0x1961] = 0x2468, [0x1962] = 0x2469, [0x1965] = 0x3220, [0x1966] = 0x3221,
  [0x1967] = 0x3222, [0x1968] = 0x3223, [0x1969] = 0x3224, [0x196a] = 0x3225,
  [0x196b] = 0x3226, [0x196c] = 0x3227, [0x196d] = 0x3228, [0x196e] = 0x3229,
  [0x1971] = 0x2160, [0x1972] = 0x2161, [0x1973] = 0x2162, [0x1974] = 0x2163,
  [0x1975] = 0x2164, [0x1976] = 0x2165, [0x1977] = 0x2166, [0x1978] = 0x2167,
  [0x1979] = 0x2168, [0x197a] = 0x2169, [0x197b] = 0x216a, [0x197c] = 0x216b,
  [0x19e1] = 0xff01, [0x19e2] = 0xff02, [0x19e3] = 0xff03, [0x19e4] = 0xffe5,
  [0x19e5] = 0xff05, [0x19e6] = 0xff06, [0x19e7] = 0xff07, [0x19e8] = 0xff08,
  [0x19e9] = 0xff09, [0x19ea] = 0xff0a, [0x19eb] = 0xff0b, [0x19ec] = 0xff0c,
  [0x19ed] = 0xff0d, [0x19ee] = 0xff0e, [0x19ef] = 0xff0f, [0x19f0] = 0xff10,
  [0x19f1] = 0xff11, [0x19f2] = 0xff12, [0x19f3] = 0xff13, [0x19f4] = 0xff14,
  [0x19f5] = 0xff15, [0x19f6] = 0xff16, [0x19f7] = 0xff17, [0x19f8] = 0xff18,
  [0x19f9] = 0xff19, [0x19fa] = 0xff1a, [0x19fb] = 0xff1b, [0x19fc] = 0xff1c,
  [0x19fd] = 0xff1d, [0x19fe] = 0xff1e, [0x19ff] = 0xff1f, [0x1a00] = 0xff20,
  [0x1a01] = 0xff21, [0x1a02] = 0xff22, [0x1a03] = 0xff23, [0x1a04] = 0xff24,
  [0x1a05] = 0xff25, [0x1a06] = 0xff26, [0x1a07] = 0xff27, [0x1a08] = 0xff28,
  [0x1a09] = 0xff29, [0x1a0a] = 0xff2a, [0x1a0b] = 0xff2b, [0x1a0c] = 0xff2c,
  [0x1a0d] = 0xff2d, [0x1a0e] = 0xff2e, [0x1a0f] = 0xff2f, [0x1a10] = 0xff30,
  [0x1a11] = 0xff31, [0x1a12] = 0xff32, [0x1a13] = 0xff33, [0x1a14] = 0xff34,
  [0x1a15] = 0xff35, [0x1a16] = 0xff36, [0x1a17] = 0xff37, [0x1a18] = 0xff38,
  [0x1a19] = 0xff39, [0x1a1a] = 0xff3a, [0x1a1b] = 0xff3b, [0x1a1c] = 0xff3c,
  [0x1a1d] = 0xff3d, [0x1a1e] = 0xff3e, [0x1a1f] = 0xff3f, [0x1a20] = 0xff40,
  [0x1a21] = 0xff41, [0x1a22] = 0xff42, [0x1a23] = 0xff43, [0x1a24] = 0xff44,
  [0x1a25] = 0xff45, [0x1a26] = 0xff46, [0x1a27] = 0xff47, [0x1a28] = 0xff48,
  [0x1a29] = 0xff49, [0x1a2a] = 0xff4a, [0x1a2b] = 0xff4b, [0x1a2c] = 0xff4c,
  [0x1a2d] = 0xff4d, [0x1a2e] = 0xff4e, [0x1a2f] = 0xff4f, [0x1a30] = 0xff50,
  [0x1a31] = 0xff51, [0x1a32] = 0xff52, [0x1a33] = 0xff53, [0x1a34] = 0xff54,
  [0x1a35] = 0xff55, [0x1a36] = 0xff56, [0x1a37] = 0xff57, [0x1a38] = 0xff58,
  [0x1a39] = 0xff59, [0x1a3a] = 0xff5a, [0x1a3b] = 0xff5b, [0x1a3c] = 0xff5c,
  [0x1a3d] = 0xff5d, [0x1a3e] = 0xffe3, [0x1aa1] = 0x3041, [0x1aa2] = 0x3042,
  [0x1aa3] = 0x3043, [0x1aa4] = 0x3044, [0x1aa5] = 0x3045, [0x1aa6] = 0x3046,
  [0x1aa7] = 0x3047, [0x1aa8] = 0x3048, [0x1aa9] = 0x3049, [0x1aaa] = 0x304a,
  [0x1aab] = 0x304b, [0x1aac] = 0x304c, [0x1aad] = 0x304d, [0x1aae] = 0x304e,
  [0x1aaf] = 0x304f, [0x1ab0] = 0x3050, [0x1ab1] = 0x3051, [0x1ab2] = 0x3052,
  [0x1ab3] = 0x3053, [0x1ab4] = 0x3054, [0x1ab5] = 0x3055, [0x1ab6] = 0x3056,
  [0x1ab7] = 0x3057, [0x1ab8] = 0x3058, [0x1ab9] = 0x3059, [0x1aba] = 0x305a,
  [0x1abb] = 0x305b, [0x1abc] = 0x305c, [0x1abd] = 0x305d, [0x1abe] = 0x305e,
  [0x1abf] = 0x305f, [0x1ac0] = 0x3060, [0x1ac1] = 0x3061, [0x1ac2] = 0x3062,
  [0x1ac3] = 0x3063, [0x1ac4] = 0x3064, [0x1ac5] = 0x3065, [0x1ac6] = 0x3066,
  [0x1ac7] = 0x3067, [0x1ac8] = 0x3068, [0x1ac9] = 0x3069, [0x1aca] = 0x306a,
  [0x1acb] = 0x306b, [0x1acc] = 0x306c, [0x1acd] = 0x306d, [0x1ace] = 0x306e,
  [0x1acf] = 0x306f, [0x1ad0] = 0x3070, [0x1ad1] = 0x3071, [0x1ad2] = 0x3072,
  [0x1ad3] = 0x3073, [0x1ad4] = 0x3074, [0x1ad5] = 0x3075, [0x1ad6] = 0x3076,
  [0x1ad7] = 0x3077, [0x1ad8] = 0x3078, [0x1ad9] = 0x3079, [0x1ada] = 0x307a,
  [0x1adb] = 0x307b, [0x1adc] = 0x307c, [0x1add] = 0x307d, [0x1ade] = 0x307e,
  [0x1adf] = 0x307f, [0x1ae0] = 0x3080, [0x1ae1] = 0x3081, [0x1ae2] = 0x3082,
  [0x1ae3] = 0x3083, [0x1ae4] = 0x3084, [0x1ae5] = 0x3085, [0x1ae6] = 0x3086,
  [0x1ae7] = 0x3087, [0x1ae8] = 0x3088, [0x1ae9] = 0x3089, [0x1aea] = 0x308a,
  [0x1aeb] = 0x308b, [0x1aec] = 0x308c, [0x1aed] = 0x308d, [0x1aee] = 0x308e,
  [0x1aef] = 0x308f, [0x1af0] = 0x3090, [0x1af1] = 0x3091, [0x1af2] = 0x3092,
  [0x1af3] = 0x3093, [0x1b61] = 0x30a1, [0x1b62] = 0x30a2, [0x1b63] = 0x30a3,
  [0x1b64] = 0x30a4, [0x1b65] = 0x30a5, [0x1b66] = 0x30a6, [0x1b67] = 0x30a7,
  [0x1b68] = 0x30a8, [0x1b69] = 0x30a9, [0x1b6a] = 0x30aa, [0x1b6b] = 0x30ab,
  [0x1b6c] = 0x30ac, [0x1b6d] = 0x30ad, [0x1b6e] = 0x30ae, [0x1b6f] = 0x30af,
  [0x1b70] = 0x30b0, [0x1b71] = 0x30b1, [0x1b72] = 0x30b2, [0x1b73] = 0x30b3,
  [0x1b74] = 0x30b4, [0x1b75] = 0x30b5, [0x1b76] = 0x30b6, [0x1b77] = 0x30b7,
  [0x1b78] = 0x30b8, [0x1b79] = 0x30b9, [0x1b7a] = 0x30ba, [0x1b7b] = 0x30bb,
  [0x1b7c] = 0x30bc, [0x1b7d] = 0x30bd, [0x1b7e] = 0x30be, [0x1b7f] = 0x30bf,
  [0x1b80] = 0x30c0, [0x1b81] = 0x30c1, [0x1b82] = 0x30c2, [0x1b83] = 0x30c3,
  [0x1b84] = 0x30c4, [0x1b85] = 0x30c5, [0x1b86] = 0x30c6, [0x1b87] = 0x30c7,
  [0x1b88] = 0x30c8, [0x1b89] = 0x30c9, [0x1b8a] = 0x30ca, [0x1b8b] = 0x30cb,
  [0x1b8c] = 0x30cc, [0x1b8d] = 0x30cd, [0x1b8e] = 0x30ce, [0x1b8f] = 0x30cf,
  [0x1b90] = 0x30d0, [0x1b91] = 0x30d1, [0x1b92] = 0x30d2, [0x1b93] = 0x30d3,
  [0x1b94] = 0x30d4, [0x1b95] = 0x30d5, [0x1b96] = 0x30d6, [0x1b97] = 0x30d7,
  [0x1b98] = 0x30d8, [0x1b99] = 0x30d9, [0x1b9a] = 0x30da, [0x1b9b] = 0x30db,
  [0x1b9c] = 0x30dc, [0x1b9d] = 0x30dd, [0x1b9e] = 0x30de, [0x1b9f] = 0x30df,
  [0x1ba0] = 0x30e0, [0x1ba1] = 0x30e1, [0x1ba2] = 0x30e2, [0x1ba3] = 0x30e3,
  [0x1ba4] = 0x30e4, [0x1ba5] = 0x30e5, [0x1ba6] = 0x30e6, [0x1ba7] = 0x30e7,
  [0x1ba8] = 0x30e8, [0x1ba9] = 0x30e9, [0x1baa] = 0x30ea, [0x1bab] = 0x30eb,
  [0x1bac] = 0x30ec, [0x1bad] = 0x30ed, [0x1bae] = 0x30ee, [0x1baf] = 0x30ef,
  [0x1bb0] = 0x30f0, [0x1bb1] = 0x30f1, [0x1bb2] = 0x30f2, [0x1bb3] = 0x30f3,
  [0x1bb4] = 0x30f4, [0x1bb5] = 0x30f5, [0x1bb6] = 0x30f6, [0x1c21] = 0x0391,
  [0x1c22] = 0x0392, [0x1c23] = 0x0393, [0x1c24] = 0x0394, [0x1c25] = 0x0395,
  [0x1c26] = 0x0396, [0x1c27] = 0x0397, [0x1c28] = 0x0398, [0x1c29] = 0x0399,
  [0x1c2a] = 0x039a, [0x1c2b] = 0x039b, [0x1c2c] = 0x039c, [0x1c2d] = 0x039d,
  [0x1c2e] = 0x039e, [0x1c2f] = 0x039f, [0x1c30] = 0x03a0, [0x1c31] = 0x03a1,
  [0x1c32] = 0x03a3, [0x1c33] = 0x03a4, [0x1c34] = 0x03a5, [0x1c35] = 0x03a6,
  [0x1c36] = 0x03a7, [0x1c37] = 0x03a8, [0x1c38] = 0x03a9, [0x1c41] = 0x03b1,
  [0x1c42] = 0x03b2, [0x1c43] = 0x03b3, [0x1c44] = 0x03b4, [0x1c45] = 0x03b5,
  [0x1c46] = 0x03b6, [0x1c47] = 0x03b7, [0x1c48] = 0x03b8, [0x1c49] = 0x03b9,
  [0x1c4a] = 0x03ba, [0x1c4b] = 0x03bb, [0x1c4c] = 0x03bc, [0x1c4d] = 0x03bd,
  [0x1c4e] = 0x03be, [0x1c4f] = 0x03bf, [0x1c50] = 0x03c0, [0x1c51] = 0x03c1,
  [0x1c52] = 0x03c3, [0x1c53] = 0x03c4, [0x1c54] = 0x03c5, [0x1c55] = 0x03c6,
  [0x1c56] = 0x03c7, [0x1c57] = 0x03c8, [0x1c58] = 0x03c9, [0x1c60] = 0xfe35,
  [0x1c61] = 0xfe36, [0x1c62] = 0xfe39, [0x1c63] = 0xfe3a, [0x1c64] = 0xfe3f,
  [0x1c65] = 0xfe40, [0x1c66] = 0xfe3d, [0x1c67] = 0xfe3e, [0x1c68] = 0xfe41,
  [0x1c69] = 0xfe42, [0x1c6a] = 0xfe43, [0x1c6b] = 0xfe44, [0x1c6e] = 0xfe3b,
  [0x1c6f] = 0xfe3c, [0x1c70] = 0xfe37, [0x1c71] = 0xfe38, [0x1c72] = 0xfe31,
  [0x1c74] = 0xfe33, [0x1c75] = 0xfe34, [0x1ce1] = 0x0410, [0x1ce2] = 0x0411,
  [0x1ce3] = 0x0412, [0x1ce4] = 0x0413, [0x1ce5] = 0x0414, [0x1ce6] = 0x0415,
  [0x1ce7] = 0x0401, [0x1ce8] = 0x0416, [0x1ce9] = 0x0417, [0x1cea] = 0x0418,
  [0x1ceb] = 0x0419, [0x1cec] = 0x041a, [0x1ced] = 0x041b, [0x1cee] = 0x041c,
  [0x1cef] = 0x041d, [0x1cf0] = 0x041e, [0x1cf1] = 0x041f, [0x1cf2] = 0x0420,
  [0x1cf3] = 0x0421, [0x1cf4] = 0x0422, [0x1cf5] = 0x0423, [0x1cf6] = 0x0424,
  [0x1cf7] = 0x0425, [0x1cf8] = 0x0426, [0x1cf9] = 0x0427, [0x1cfa] = 0x0428,
  [0x1cfb] = 0x0429, [0x1cfc] = 0x042a, [0x1cfd] = 0x042b, [0x1cfe] = 0x042c,
  [0x1cff] = 0x042d, [0x1d00] = 0x042e, [0x1d01] = 0x042f, [0x1d11] = 0x0430,
  [0x1d12] = 0x0431, [0x1d13] = 0x0432, [0x1d14] = 0x0433, [0x1d15] = 0x0434,
  [0x1d16] = 0x0435, [0x1d17] = 0x0451, [0x1d18] = 0x0436, [0x1d19] = 0x0437,
  [0x1d1a] = 0x0438, [0x1d1b] = 0x0439, [0x1d1c] = 0x043a, [0x1d1d] = 0x043b,
  [0x1d1e] = 0x043c, [0x1d1f] = 0x043d, [0x1d20] = 0x043e, [0x1d21] = 0x043f,
  [0x1d22] = 0x0440, [0x1d23] = 0x0441, [0x1d24] = 0x0442, [0x1d25] = 0x0443,
  [0x1d26] = 0x0444, [0x1d27] = 0x0445, [0x1d28] = 0x0446, [0x1d29] = 0x0447,
  [0x1d2a] = 0x0448, [0x1d2b] = 0x0449, [0x1d2c] = 0x044a, [0x1d2d] = 0x044b,
  [0x1d2e] = 0x044c, [0x1d2f] = 0x044d, [0x1d30] = 0x044e, [0x1d31] = 0x044f,
  [0x1d40] = 0x02ca, [0x1d41] = 0x02cb, [0x1d42] = 0x02d9, [0x1d43] = 0x2013,
  [0x1d44] = 0x2015, [0x1d45] = 0x2025, [0x1d46] = 0x2035, [0x1d47] = 0x2105,
  [0x1d48] = 0x2109, [0x1d49] = 0x2196, [0x1d4a] = 0x2197, [0x1d4b] = 0x2198,
  [0x1d4c] = 0x2199, [0x1d4d] = 0x2215, [0x1d4e] = 0x221f, [0x1d4f] = 0x2223,
  [0x1d50] = 0x2252, [0x1d51] = 0x2266, [0x1d52] = 0x2267, [0x1d53] = 0x22bf,
  [0x1d54] = 0x2550, [0x1d55] = 0x2551, [0x1d56] = 0x2552, [0x1d57] = 0x2553,
  [0x1d58] = 0x2554, [0x1d59] = 0x2555, [0x1d5a] = 0x2556, [0x1d5b] = 0x2557,
  [0x1d5c] = 0x2558, [0x1d5d] = 0x2559, [0x1d5e] = 0x255a, [0x1d5f] = 0x255b,
  [0x1d60] = 0x255c, [0x1d61] = 0x255d, [0x1d62] = 0x255e, [0x1d63] = 0x255f,
  [0x1d64] = 0x2560, [0x1d65] = 0x2561, [0x1d66] = 0x2562, [0x1d67] = 0x2563,
  [0x1d68] = 0x2564, [0x1d69] = 0x2565, [0x1d6a] = 0x2566, [0x1d6b] = 0x2567,
  [0x1d6c] = 0x2568, [0x1d6d] = 0x2569, [0x1d6e] = 0x256a, [0x1d6f] = 0x256b,
  [0x1d70] = 0x256c, [0x1d71] = 0x256d, [0x1d72] = 0x256e, [0x1d73] = 0x256f,
  [0x1d74] = 0x2570, [0x1d75] = 0x2571, [0x1d76] = 0x2572, [0x1d77] = 0x2573,
  [0x1d78] = 0x2581, [0x1d79] = 0x2582, [0x1d7a] = 0x2583, [0x1d7b] = 0x2584,
  [0x1d7c] = 0x2585, [0x1d7d] = 0x2586, [0x1d7e] = 0x2587, [0x1d80] = 0x2588,
  [0x1d81] = 0x2589, [0x1d82] = 0x258a, [0x1d83] = 0x258b, [0x1d84] = 0x258c,
  [0x1d85] = 0x258d, [0x1d86] = 0x258e, [0x1d87] = 0x258f, [0x1d88] = 0x2593,
  [0x1d89] = 0x2594, [0x1d8a] = 0x2595, [0x1d8b] = 0x25bc, [0x1d8c] = 0x25bd,
  [0x1d8d] = 0x25e2, [0x1d8e] = 0x25e3, [0x1d8f] = 0x25e4, [0x1d90] = 0x25e5,
  [0x1d91] = 0x2609, [0x1d92] = 0x2295, [0x1d93] = 0x3012, [0x1d94] = 0x301d,
  [0x1d95] = 0x301e, [0x1da1] = 0x0101, [0x1da2] = 0x00e1, [0x1da3] = 0x01ce,
  [0x1da4] = 0x00e0, [0x1da5] = 0x0113, [0x1da6] = 0x00e9, [0x1da7] = 0x011b,
  [0x1da8] = 0x00e8, [0x1da9] = 0x012b, [0x1daa] = 0x00ed, [0x1dab] = 0x01d0,
  [0x1dac] = 0x00ec, [0x1dad] = 0x014d, [0x1dae] = 0x00f3, [0x1daf] = 0x01d2,
  [0x1db0] = 0x00f2, [0x1db1] = 0x016b, [0x1db2] = 0x00fa, [0x1db3] = 0x01d4,
  [0x1db4] = 0x00f9, [0x1db5] = 0x01d6, [0x1db6] = 0x01d8, [0x1db7] = 0x01da,
  [0x1db8] = 0x01dc, [0x1db9] = 0x00fc, [0x1dba] = 0x00ea, [0x1dbb] = 0x0251,
#if USE_PRIVATE_AREA
  [0x1dbc] = 0xe7c7,
#endif
                     [0x1dbd] = 0x0144, [0x1dbe] = 0x0148,
#if USE_PRIVATE_AREA
                                                           [0x1dbf] = 0xe7c8,
#endif
  [0x1dc0] = 0x0261, [0x1dc5] = 0x3105, [0x1dc6] = 0x3106, [0x1dc7] = 0x3107,
  [0x1dc8] = 0x3108, [0x1dc9] = 0x3109, [0x1dca] = 0x310a, [0x1dcb] = 0x310b,
  [0x1dcc] = 0x310c, [0x1dcd] = 0x310d, [0x1dce] = 0x310e, [0x1dcf] = 0x310f,
  [0x1dd0] = 0x3110, [0x1dd1] = 0x3111, [0x1dd2] = 0x3112, [0x1dd3] = 0x3113,
  [0x1dd4] = 0x3114, [0x1dd5] = 0x3115, [0x1dd6] = 0x3116, [0x1dd7] = 0x3117,
  [0x1dd8] = 0x3118, [0x1dd9] = 0x3119, [0x1dda] = 0x311a, [0x1ddb] = 0x311b,
  [0x1ddc] = 0x311c, [0x1ddd] = 0x311d, [0x1dde] = 0x311e, [0x1ddf] = 0x311f,
  [0x1de0] = 0x3120, [0x1de1] = 0x3121, [0x1de2] = 0x3122, [0x1de3] = 0x3123,
  [0x1de4] = 0x3124, [0x1de5] = 0x3125, [0x1de6] = 0x3126, [0x1de7] = 0x3127,
  [0x1de8] = 0x3128, [0x1de9] = 0x3129, [0x1e00] = 0x3021, [0x1e01] = 0x3022,
  [0x1e02] = 0x3023, [0x1e03] = 0x3024, [0x1e04] = 0x3025, [0x1e05] = 0x3026,
  [0x1e06] = 0x3027, [0x1e07] = 0x3028, [0x1e08] = 0x3029, [0x1e09] = 0x32a3,
  [0x1e0a] = 0x338e, [0x1e0b] = 0x338f, [0x1e0c] = 0x339c, [0x1e0d] = 0x339d,
  [0x1e0e] = 0x339e, [0x1e0f] = 0x33a1, [0x1e10] = 0x33c4, [0x1e11] = 0x33ce,
  [0x1e12] = 0x33d1, [0x1e13] = 0x33d2, [0x1e14] = 0x33d5, [0x1e15] = 0xfe30,
  [0x1e16] = 0xffe2, [0x1e17] = 0xffe4, [0x1e19] = 0x2121, [0x1e1a] = 0x3231,
  [0x1e1c] = 0x2010, [0x1e20] = 0x30fc, [0x1e21] = 0x309b, [0x1e22] = 0x309c,
  [0x1e23] = 0x30fd, [0x1e24] = 0x30fe, [0x1e25] = 0x3006, [0x1e26] = 0x309d,
  [0x1e27] = 0x309e, [0x1e28] = 0xfe49, [0x1e29] = 0xfe4a, [0x1e2a] = 0xfe4b,
  [0x1e2b] = 0xfe4c, [0x1e2c] = 0xfe4d, [0x1e2d] = 0xfe4e, [0x1e2e] = 0xfe4f,
  [0x1e2f] = 0xfe50, [0x1e30] = 0xfe51, [0x1e31] = 0xfe52, [0x1e32] = 0xfe54,
  [0x1e33] = 0xfe55, [0x1e34] = 0xfe56, [0x1e35] = 0xfe57, [0x1e36] = 0xfe59,
  [0x1e37] = 0xfe5a, [0x1e38] = 0xfe5b, [0x1e39] = 0xfe5c, [0x1e3a] = 0xfe5d,
  [0x1e3b] = 0xfe5e, [0x1e3c] = 0xfe5f, [0x1e3d] = 0xfe60, [0x1e3e] = 0xfe61,
  [0x1e40] = 0xfe62, [0x1e41] = 0xfe63, [0x1e42] = 0xfe64, [0x1e43] = 0xfe65,
  [0x1e44] = 0xfe66, [0x1e45] = 0xfe68, [0x1e46] = 0xfe69, [0x1e47] = 0xfe6a,
  [0x1e48] = 0xfe6b,
#if USE_PRIVATE_AREA
                     [0x1e49] = 0xe7e7, [0x1e4a] = 0xe7e8, [0x1e4b] = 0xe7e9,
  [0x1e4c] = 0xe7ea, [0x1e4d] = 0xe7eb, [0x1e4e] = 0xe7ec, [0x1e4f] = 0xe7ed,
  [0x1e50] = 0xe7ee, [0x1e51] = 0xe7ef, [0x1e52] = 0xe7f0, [0x1e53] = 0xe7f1,
  [0x1e54] = 0xe7f2, [0x1e55] = 0xe7f3,
#endif
                                        [0x1e56] = 0x3007, [0x1e64] = 0x2500,
  [0x1e65] = 0x2501, [0x1e66] = 0x2502, [0x1e67] = 0x2503, [0x1e68] = 0x2504,
  [0x1e69] = 0x2505, [0x1e6a] = 0x2506, [0x1e6b] = 0x2507, [0x1e6c] = 0x2508,
  [0x1e6d] = 0x2509, [0x1e6e] = 0x250a, [0x1e6f] = 0x250b, [0x1e70] = 0x250c,
  [0x1e71] = 0x250d, [0x1e72] = 0x250e, [0x1e73] = 0x250f, [0x1e74] = 0x2510,
  [0x1e75] = 0x2511, [0x1e76] = 0x2512, [0x1e77] = 0x2513, [0x1e78] = 0x2514,
  [0x1e79] = 0x2515, [0x1e7a] = 0x2516, [0x1e7b] = 0x2517, [0x1e7c] = 0x2518,
  [0x1e7d] = 0x2519, [0x1e7e] = 0x251a, [0x1e7f] = 0x251b, [0x1e80] = 0x251c,
  [0x1e81] = 0x251d, [0x1e82] = 0x251e, [0x1e83] = 0x251f, [0x1e84] = 0x2520,
  [0x1e85] = 0x2521, [0x1e86] = 0x2522, [0x1e87] = 0x2523, [0x1e88] = 0x2524,
  [0x1e89] = 0x2525, [0x1e8a] = 0x2526, [0x1e8b] = 0x2527, [0x1e8c] = 0x2528,
  [0x1e8d] = 0x2529, [0x1e8e] = 0x252a, [0x1e8f] = 0x252b, [0x1e90] = 0x252c,
  [0x1e91] = 0x252d, [0x1e92] = 0x252e, [0x1e93] = 0x252f, [0x1e94] = 0x2530,
  [0x1e95] = 0x2531, [0x1e96] = 0x2532, [0x1e97] = 0x2533, [0x1e98] = 0x2534,
  [0x1e99] = 0x2535, [0x1e9a] = 0x2536, [0x1e9b] = 0x2537, [0x1e9c] = 0x2538,
  [0x1e9d] = 0x2539, [0x1e9e] = 0x253a, [0x1e9f] = 0x253b, [0x1ea0] = 0x253c,
  [0x1ea1] = 0x253d, [0x1ea2] = 0x253e, [0x1ea3] = 0x253f, [0x1ea4] = 0x2540,
  [0x1ea5] = 0x2541, [0x1ea6] = 0x2542, [0x1ea7] = 0x2543, [0x1ea8] = 0x2544,
  [0x1ea9] = 0x2545, [0x1eaa] = 0x2546, [0x1eab] = 0x2547, [0x1eac] = 0x2548,
  [0x1ead] = 0x2549, [0x1eae] = 0x254a, [0x1eaf] = 0x254b, [0x1ec0] = 0x72dc,
  [0x1ec1] = 0x72dd, [0x1ec2] = 0x72df, [0x1ec3] = 0x72e2, [0x1ec4] = 0x72e3,
  [0x1ec5] = 0x72e4, [0x1ec6] = 0x72e5, [0x1ec7] = 0x72e6, [0x1ec8] = 0x72e7,
  [0x1ec9] = 0x72ea, [0x1eca] = 0x72eb, [0x1ecb] = 0x72f5, [0x1ecc] = 0x72f6,
  [0x1ecd] = 0x72f9, [0x1ece] = 0x72fd, [0x1ecf] = 0x72fe, [0x1ed0] = 0x72ff,
  [0x1ed1] = 0x7300, [0x1ed2] = 0x7302, [0x1ed3] = 0x7304, [0x1ed4] = 0x7305,
  [0x1ed5] = 0x7306, [0x1ed6] = 0x7307, [0x1ed7] = 0x7308, [0x1ed8] = 0x7309,
  [0x1ed9] = 0x730b, [0x1eda] = 0x730c, [0x1edb] = 0x730d, [0x1edc] = 0x730f,
  [0x1edd] = 0x7310, [0x1ede] = 0x7311, [0x1edf] = 0x7312, [0x1ee0] = 0x7314,
  [0x1ee1] = 0x7318, [0x1ee2] = 0x7319, [0x1ee3] = 0x731a, [0x1ee4] = 0x731f,
  [0x1ee5] = 0x7320, [0x1ee6] = 0x7323, [0x1ee7] = 0x7324, [0x1ee8] = 0x7326,
  [0x1ee9] = 0x7327, [0x1eea] = 0x7328, [0x1eeb] = 0x732d, [0x1eec] = 0x732f,
  [0x1eed] = 0x7330, [0x1eee] = 0x7332, [0x1eef] = 0x7333, [0x1ef0] = 0x7335,
  [0x1ef1] = 0x7336, [0x1ef2] = 0x733a, [0x1ef3] = 0x733b, [0x1ef4] = 0x733c,
  [0x1ef5] = 0x733d, [0x1ef6] = 0x7340, [0x1ef7] = 0x7341, [0x1ef8] = 0x7342,
  [0x1ef9] = 0x7343, [0x1efa] = 0x7344, [0x1efb] = 0x7345, [0x1efc] = 0x7346,
  [0x1efd] = 0x7347, [0x1efe] = 0x7348, [0x1f00] = 0x7349, [0x1f01] = 0x734a,
  [0x1f02] = 0x734b, [0x1f03] = 0x734c, [0x1f04] = 0x734e, [0x1f05] = 0x734f,
  [0x1f06] = 0x7351, [0x1f07] = 0x7353, [0x1f08] = 0x7354, [0x1f09] = 0x7355,
  [0x1f0a] = 0x7356, [0x1f0b] = 0x7358, [0x1f0c] = 0x7359, [0x1f0d] = 0x735a,
  [0x1f0e] = 0x735b, [0x1f0f] = 0x735c, [0x1f10] = 0x735d, [0x1f11] = 0x735e,
  [0x1f12] = 0x735f, [0x1f13] = 0x7361, [0x1f14] = 0x7362, [0x1f15] = 0x7363,
  [0x1f16] = 0x7364, [0x1f17] = 0x7365, [0x1f18] = 0x7366, [0x1f19] = 0x7367,
  [0x1f1a] = 0x7368, [0x1f1b] = 0x7369, [0x1f1c] = 0x736a, [0x1f1d] = 0x736b,
  [0x1f1e] = 0x736e, [0x1f1f] = 0x7370, [0x1f20] = 0x7371, [0x1f80] = 0x7372,
  [0x1f81] = 0x7373, [0x1f82] = 0x7374, [0x1f83] = 0x7375, [0x1f84] = 0x7376,
  [0x1f85] = 0x7377, [0x1f86] = 0x7378, [0x1f87] = 0x7379, [0x1f88] = 0x737a,
  [0x1f89] = 0x737b, [0x1f8a] = 0x737c, [0x1f8b] = 0x737d, [0x1f8c] = 0x737f,
  [0x1f8d] = 0x7380, [0x1f8e] = 0x7381, [0x1f8f] = 0x7382, [0x1f90] = 0x7383,
  [0x1f91] = 0x7385, [0x1f92] = 0x7386, [0x1f93] = 0x7388, [0x1f94] = 0x738a,
  [0x1f95] = 0x738c, [0x1f96] = 0x738d, [0x1f97] = 0x738f, [0x1f98] = 0x7390,
  [0x1f99] = 0x7392, [0x1f9a] = 0x7393, [0x1f9b] = 0x7394, [0x1f9c] = 0x7395,
  [0x1f9d] = 0x7397, [0x1f9e] = 0x7398, [0x1f9f] = 0x7399, [0x1fa0] = 0x739a,
  [0x1fa1] = 0x739c, [0x1fa2] = 0x739d, [0x1fa3] = 0x739e, [0x1fa4] = 0x73a0,
  [0x1fa5] = 0x73a1, [0x1fa6] = 0x73a3, [0x1fa7] = 0x73a4, [0x1fa8] = 0x73a5,
  [0x1fa9] = 0x73a6, [0x1faa] = 0x73a7, [0x1fab] = 0x73a8, [0x1fac] = 0x73aa,
  [0x1fad] = 0x73ac, [0x1fae] = 0x73ad, [0x1faf] = 0x73b1, [0x1fb0] = 0x73b4,
  [0x1fb1] = 0x73b5, [0x1fb2] = 0x73b6, [0x1fb3] = 0x73b8, [0x1fb4] = 0x73b9,
  [0x1fb5] = 0x73bc, [0x1fb6] = 0x73bd, [0x1fb7] = 0x73be, [0x1fb8] = 0x73bf,
  [0x1fb9] = 0x73c1, [0x1fba] = 0x73c3, [0x1fbb] = 0x73c4, [0x1fbc] = 0x73c5,
  [0x1fbd] = 0x73c6, [0x1fbe] = 0x73c7, [0x1fc0] = 0x73cb, [0x1fc1] = 0x73cc,
  [0x1fc2] = 0x73ce, [0x1fc3] = 0x73d2, [0x1fc4] = 0x73d3, [0x1fc5] = 0x73d4,
  [0x1fc6] = 0x73d5, [0x1fc7] = 0x73d6, [0x1fc8] = 0x73d7, [0x1fc9] = 0x73d8,
  [0x1fca] = 0x73da, [0x1fcb] = 0x73db, [0x1fcc] = 0x73dc, [0x1fcd] = 0x73dd,
  [0x1fce] = 0x73df, [0x1fcf] = 0x73e1, [0x1fd0] = 0x73e2, [0x1fd1] = 0x73e3,
  [0x1fd2] = 0x73e4, [0x1fd3] = 0x73e6, [0x1fd4] = 0x73e8, [0x1fd5] = 0x73ea,
  [0x1fd6] = 0x73eb, [0x1fd7] = 0x73ec, [0x1fd8] = 0x73ee, [0x1fd9] = 0x73ef,
  [0x1fda] = 0x73f0, [0x1fdb] = 0x73f1, [0x1fdc] = 0x73f3, [0x1fdd] = 0x73f4,
  [0x1fde] = 0x73f5, [0x1fdf] = 0x73f6, [0x1fe0] = 0x73f7, [0x2040] = 0x73f8,
  [0x2041] = 0x73f9, [0x2042] = 0x73fa, [0x2043] = 0x73fb, [0x2044] = 0x73fc,
  [0x2045] = 0x73fd, [0x2046] = 0x73fe, [0x2047] = 0x73ff, [0x2048] = 0x7400,
  [0x2049] = 0x7401, [0x204a] = 0x7402, [0x204b] = 0x7404, [0x204c] = 0x7407,
  [0x204d] = 0x7408, [0x204e] = 0x740b, [0x204f] = 0x740c, [0x2050] = 0x740d,
  [0x2051] = 0x740e, [0x2052] = 0x7411, [0x2053] = 0x7412, [0x2054] = 0x7413,
  [0x2055] = 0x7414, [0x2056] = 0x7415, [0x2057] = 0x7416, [0x2058] = 0x7417,
  [0x2059] = 0x7418, [0x205a] = 0x7419, [0x205b] = 0x741c, [0x205c] = 0x741d,
  [0x205d] = 0x741e, [0x205e] = 0x741f, [0x205f] = 0x7420, [0x2060] = 0x7421,
  [0x2061] = 0x7423, [0x2062] = 0x7424, [0x2063] = 0x7427, [0x2064] = 0x7429,
  [0x2065] = 0x742b, [0x2066] = 0x742d, [0x2067] = 0x742f, [0x2068] = 0x7431,
  [0x2069] = 0x7432, [0x206a] = 0x7437, [0x206b] = 0x7438, [0x206c] = 0x7439,
  [0x206d] = 0x743a, [0x206e] = 0x743b, [0x206f] = 0x743d, [0x2070] = 0x743e,
  [0x2071] = 0x743f, [0x2072] = 0x7440, [0x2073] = 0x7442, [0x2074] = 0x7443,
  [0x2075] = 0x7444, [0x2076] = 0x7445, [0x2077] = 0x7446, [0x2078] = 0x7447,
  [0x2079] = 0x7448, [0x207a] = 0x7449, [0x207b] = 0x744a, [0x207c] = 0x744b,
  [0x207d] = 0x744c, [0x207e] = 0x744d, [0x2080] = 0x744e, [0x2081] = 0x744f,
  [0x2082] = 0x7450, [0x2083] = 0x7451, [0x2084] = 0x7452, [0x2085] = 0x7453,
  [0x2086] = 0x7454, [0x2087] = 0x7456, [0x2088] = 0x7458, [0x2089] = 0x745d,
  [0x208a] = 0x7460, [0x208b] = 0x7461, [0x208c] = 0x7462, [0x208d] = 0x7463,
  [0x208e] = 0x7464, [0x208f] = 0x7465, [0x2090] = 0x7466, [0x2091] = 0x7467,
  [0x2092] = 0x7468, [0x2093] = 0x7469, [0x2094] = 0x746a, [0x2095] = 0x746b,
  [0x2096] = 0x746c, [0x2097] = 0x746e, [0x2098] = 0x746f, [0x2099] = 0x7471,
  [0x209a] = 0x7472, [0x209b] = 0x7473, [0x209c] = 0x7474, [0x209d] = 0x7475,
  [0x209e] = 0x7478, [0x209f] = 0x7479, [0x20a0] = 0x747a, [0x2100] = 0x747b,
  [0x2101] = 0x747c, [0x2102] = 0x747d, [0x2103] = 0x747f, [0x2104] = 0x7482,
  [0x2105] = 0x7484, [0x2106] = 0x7485, [0x2107] = 0x7486, [0x2108] = 0x7488,
  [0x2109] = 0x7489, [0x210a] = 0x748a, [0x210b] = 0x748c, [0x210c] = 0x748d,
  [0x210d] = 0x748f, [0x210e] = 0x7491, [0x210f] = 0x7492, [0x2110] = 0x7493,
  [0x2111] = 0x7494, [0x2112] = 0x7495, [0x2113] = 0x7496, [0x2114] = 0x7497,
  [0x2115] = 0x7498, [0x2116] = 0x7499, [0x2117] = 0x749a, [0x2118] = 0x749b,
  [0x2119] = 0x749d, [0x211a] = 0x749f, [0x211b] = 0x74a0, [0x211c] = 0x74a1,
  [0x211d] = 0x74a2, [0x211e] = 0x74a3, [0x211f] = 0x74a4, [0x2120] = 0x74a5,
  [0x2121] = 0x74a6, [0x2122] = 0x74aa, [0x2123] = 0x74ab, [0x2124] = 0x74ac,
  [0x2125] = 0x74ad, [0x2126] = 0x74ae, [0x2127] = 0x74af, [0x2128] = 0x74b0,
  [0x2129] = 0x74b1, [0x212a] = 0x74b2, [0x212b] = 0x74b3, [0x212c] = 0x74b4,
  [0x212d] = 0x74b5, [0x212e] = 0x74b6, [0x212f] = 0x74b7, [0x2130] = 0x74b8,
  [0x2131] = 0x74b9, [0x2132] = 0x74bb, [0x2133] = 0x74bc, [0x2134] = 0x74bd,
  [0x2135] = 0x74be, [0x2136] = 0x74bf, [0x2137] = 0x74c0, [0x2138] = 0x74c1,
  [0x2139] = 0x74c2, [0x213a] = 0x74c3, [0x213b] = 0x74c4, [0x213c] = 0x74c5,
  [0x213d] = 0x74c6, [0x213e] = 0x74c7, [0x2140] = 0x74c8, [0x2141] = 0x74c9,
  [0x2142] = 0x74ca, [0x2143] = 0x74cb, [0x2144] = 0x74cc, [0x2145] = 0x74cd,
  [0x2146] = 0x74ce, [0x2147] = 0x74cf, [0x2148] = 0x74d0, [0x2149] = 0x74d1,
  [0x214a] = 0x74d3, [0x214b] = 0x74d4, [0x214c] = 0x74d5, [0x214d] = 0x74d6,
  [0x214e] = 0x74d7, [0x214f] = 0x74d8, [0x2150] = 0x74d9, [0x2151] = 0x74da,
  [0x2152] = 0x74db, [0x2153] = 0x74dd, [0x2154] = 0x74df, [0x2155] = 0x74e1,
  [0x2156] = 0x74e5, [0x2157] = 0x74e7, [0x2158] = 0x74e8, [0x2159] = 0x74e9,
  [0x215a] = 0x74ea, [0x215b] = 0x74eb, [0x215c] = 0x74ec, [0x215d] = 0x74ed,
  [0x215e] = 0x74f0, [0x215f] = 0x74f1, [0x2160] = 0x74f2, [0x21c0] = 0x74f3,
  [0x21c1] = 0x74f5, [0x21c2] = 0x74f8, [0x21c3] = 0x74f9, [0x21c4] = 0x74fa,
  [0x21c5] = 0x74fb, [0x21c6] = 0x74fc, [0x21c7] = 0x74fd, [0x21c8] = 0x74fe,
  [0x21c9] = 0x7500, [0x21ca] = 0x7501, [0x21cb] = 0x7502, [0x21cc] = 0x7503,
  [0x21cd] = 0x7505, [0x21ce] = 0x7506, [0x21cf] = 0x7507, [0x21d0] = 0x7508,
  [0x21d1] = 0x7509, [0x21d2] = 0x750a, [0x21d3] = 0x750b, [0x21d4] = 0x750c,
  [0x21d5] = 0x750e, [0x21d6] = 0x7510, [0x21d7] = 0x7512, [0x21d8] = 0x7514,
  [0x21d9] = 0x7515, [0x21da] = 0x7516, [0x21db] = 0x7517, [0x21dc] = 0x751b,
  [0x21dd] = 0x751d, [0x21de] = 0x751e, [0x21df] = 0x7520, [0x21e0] = 0x7521,
  [0x21e1] = 0x7522, [0x21e2] = 0x7523, [0x21e3] = 0x7524, [0x21e4] = 0x7526,
  [0x21e5] = 0x7527, [0x21e6] = 0x752a, [0x21e7] = 0x752e, [0x21e8] = 0x7534,
  [0x21e9] = 0x7536, [0x21ea] = 0x7539, [0x21eb] = 0x753c, [0x21ec] = 0x753d,
  [0x21ed] = 0x753f, [0x21ee] = 0x7541, [0x21ef] = 0x7542, [0x21f0] = 0x7543,
  [0x21f1] = 0x7544, [0x21f2] = 0x7546, [0x21f3] = 0x7547, [0x21f4] = 0x7549,
  [0x21f5] = 0x754a, [0x21f6] = 0x754d, [0x21f7] = 0x7550, [0x21f8] = 0x7551,
  [0x21f9] = 0x7552, [0x21fa] = 0x7553, [0x21fb] = 0x7555, [0x21fc] = 0x7556,
  [0x21fd] = 0x7557, [0x21fe] = 0x7558, [0x2200] = 0x755d, [0x2201] = 0x755e,
  [0x2202] = 0x755f, [0x2203] = 0x7560, [0x2204] = 0x7561, [0x2205] = 0x7562,
  [0x2206] = 0x7563, [0x2207] = 0x7564, [0x2208] = 0x7567, [0x2209] = 0x7568,
  [0x220a] = 0x7569, [0x220b] = 0x756b, [0x220c] = 0x756c, [0x220d] = 0x756d,
  [0x220e] = 0x756e, [0x220f] = 0x756f, [0x2210] = 0x7570, [0x2211] = 0x7571,
  [0x2212] = 0x7573, [0x2213] = 0x7575, [0x2214] = 0x7576, [0x2215] = 0x7577,
  [0x2216] = 0x757a, [0x2217] = 0x757b, [0x2218] = 0x757c, [0x2219] = 0x757d,
  [0x221a] = 0x757e, [0x221b] = 0x7580, [0x221c] = 0x7581, [0x221d] = 0x7582,
  [0x221e] = 0x7584, [0x221f] = 0x7585, [0x2220] = 0x7587, [0x2280] = 0x7588,
  [0x2281] = 0x7589, [0x2282] = 0x758a, [0x2283] = 0x758c, [0x2284] = 0x758d,
  [0x2285] = 0x758e, [0x2286] = 0x7590, [0x2287] = 0x7593, [0x2288] = 0x7595,
  [0x2289] = 0x7598, [0x228a] = 0x759b, [0x228b] = 0x759c, [0x228c] = 0x759e,
  [0x228d] = 0x75a2, [0x228e] = 0x75a6, [0x228f] = 0x75a7, [0x2290] = 0x75a8,
  [0x2291] = 0x75a9, [0x2292] = 0x75aa, [0x2293] = 0x75ad, [0x2294] = 0x75b6,
  [0x2295] = 0x75b7, [0x2296] = 0x75ba, [0x2297] = 0x75bb, [0x2298] = 0x75bf,
  [0x2299] = 0x75c0, [0x229a] = 0x75c1, [0x229b] = 0x75c6, [0x229c] = 0x75cb,
  [0x229d] = 0x75cc, [0x229e] = 0x75ce, [0x229f] = 0x75cf, [0x22a0] = 0x75d0,
  [0x22a1] = 0x75d1, [0x22a2] = 0x75d3, [0x22a3] = 0x75d7, [0x22a4] = 0x75d9,
  [0x22a5] = 0x75da, [0x22a6] = 0x75dc, [0x22a7] = 0x75dd, [0x22a8] = 0x75df,
  [0x22a9] = 0x75e0, [0x22aa] = 0x75e1, [0x22ab] = 0x75e5, [0x22ac] = 0x75e9,
  [0x22ad] = 0x75ec, [0x22ae] = 0x75ed, [0x22af] = 0x75ee, [0x22b0] = 0x75ef,
  [0x22b1] = 0x75f2, [0x22b2] = 0x75f3, [0x22b3] = 0x75f5, [0x22b4] = 0x75f6,
  [0x22b5] = 0x75f7, [0x22b6] = 0x75f8, [0x22b7] = 0x75fa, [0x22b8] = 0x75fb,
  [0x22b9] = 0x75fd, [0x22ba] = 0x75fe, [0x22bb] = 0x7602, [0x22bc] = 0x7604,
  [0x22bd] = 0x7606, [0x22be] = 0x7607, [0x22c0] = 0x7608, [0x22c1] = 0x7609,
  [0x22c2] = 0x760b, [0x22c3] = 0x760d, [0x22c4] = 0x760e, [0x22c5] = 0x760f,
  [0x22c6] = 0x7611, [0x22c7] = 0x7612, [0x22c8] = 0x7613, [0x22c9] = 0x7614,
  [0x22ca] = 0x7616, [0x22cb] = 0x761a, [0x22cc] = 0x761c, [0x22cd] = 0x761d,
  [0x22ce] = 0x761e, [0x22cf] = 0x7621, [0x22d0] = 0x7623, [0x22d1] = 0x7627,
  [0x22d2] = 0x7628, [0x22d3] = 0x762c, [0x22d4] = 0x762e, [0x22d5] = 0x762f,
  [0x22d6] = 0x7631, [0x22d7] = 0x7632, [0x22d8] = 0x7636, [0x22d9] = 0x7637,
  [0x22da] = 0x7639, [0x22db] = 0x763a, [0x22dc] = 0x763b, [0x22dd] = 0x763d,
  [0x22de] = 0x7641, [0x22df] = 0x7642, [0x22e0] = 0x7644, [0x2340] = 0x7645,
  [0x2341] = 0x7646, [0x2342] = 0x7647, [0x2343] = 0x7648, [0x2344] = 0x7649,
  [0x2345] = 0x764a, [0x2346] = 0x764b, [0x2347] = 0x764e, [0x2348] = 0x764f,
  [0x2349] = 0x7650, [0x234a] = 0x7651, [0x234b] = 0x7652, [0x234c] = 0x7653,
  [0x234d] = 0x7655, [0x234e] = 0x7657, [0x234f] = 0x7658, [0x2350] = 0x7659,
  [0x2351] = 0x765a, [0x2352] = 0x765b, [0x2353] = 0x765d, [0x2354] = 0x765f,
  [0x2355] = 0x7660, [0x2356] = 0x7661, [0x2357] = 0x7662, [0x2358] = 0x7664,
  [0x2359] = 0x7665, [0x235a] = 0x7666, [0x235b] = 0x7667, [0x235c] = 0x7668,
  [0x235d] = 0x7669, [0x235e] = 0x766a, [0x235f] = 0x766c, [0x2360] = 0x766d,
  [0x2361] = 0x766e, [0x2362] = 0x7670, [0x2363] = 0x7671, [0x2364] = 0x7672,
  [0x2365] = 0x7673, [0x2366] = 0x7674, [0x2367] = 0x7675, [0x2368] = 0x7676,
  [0x2369] = 0x7677, [0x236a] = 0x7679, [0x236b] = 0x767a, [0x236c] = 0x767c,
  [0x236d] = 0x767f, [0x236e] = 0x7680, [0x236f] = 0x7681, [0x2370] = 0x7683,
  [0x2371] = 0x7685, [0x2372] = 0x7689, [0x2373] = 0x768a, [0x2374] = 0x768c,
  [0x2375] = 0x768d, [0x2376] = 0x768f, [0x2377] = 0x7690, [0x2378] = 0x7692,
  [0x2379] = 0x7694, [0x237a] = 0x7695, [0x237b] = 0x7697, [0x237c] = 0x7698,
  [0x237d] = 0x769a, [0x237e] = 0x769b, [0x2380] = 0x769c, [0x2381] = 0x769d,
  [0x2382] = 0x769e, [0x2383] = 0x769f, [0x2384] = 0x76a0, [0x2385] = 0x76a1,
  [0x2386] = 0x76a2, [0x2387] = 0x76a3, [0x2388] = 0x76a5, [0x2389] = 0x76a6,
  [0x238a] = 0x76a7, [0x238b] = 0x76a8, [0x238c] = 0x76a9, [0x238d] = 0x76aa,
  [0x238e] = 0x76ab, [0x238f] = 0x76ac, [0x2390] = 0x76ad, [0x2391] = 0x76af,
  [0x2392] = 0x76b0, [0x2393] = 0x76b3, [0x2394] = 0x76b5, [0x2395] = 0x76b6,
  [0x2396] = 0x76b7, [0x2397] = 0x76b8, [0x2398] = 0x76b9, [0x2399] = 0x76ba,
  [0x239a] = 0x76bb, [0x239b] = 0x76bc, [0x239c] = 0x76bd, [0x239d] = 0x76be,
  [0x239e] = 0x76c0, [0x239f] = 0x76c1, [0x23a0] = 0x76c3, [0x23a1] = 0x554a,
  [0x23a2] = 0x963f, [0x23a3] = 0x57c3, [0x23a4] = 0x6328, [0x23a5] = 0x54ce,
  [0x23a6] = 0x5509, [0x23a7] = 0x54c0, [0x23a8] = 0x7691, [0x23a9] = 0x764c,
  [0x23aa] = 0x853c, [0x23ab] = 0x77ee, [0x23ac] = 0x827e, [0x23ad] = 0x788d,
  [0x23ae] = 0x7231, [0x23af] = 0x9698, [0x23b0] = 0x978d, [0x23b1] = 0x6c28,
  [0x23b2] = 0x5b89, [0x23b3] = 0x4ffa, [0x23b4] = 0x6309, [0x23b5] = 0x6697,
  [0x23b6] = 0x5cb8, [0x23b7] = 0x80fa, [0x23b8] = 0x6848, [0x23b9] = 0x80ae,
  [0x23ba] = 0x6602, [0x23bb] = 0x76ce, [0x23bc] = 0x51f9, [0x23bd] = 0x6556,
  [0x23be] = 0x71ac, [0x23bf] = 0x7ff1, [0x23c0] = 0x8884, [0x23c1] = 0x50b2,
  [0x23c2] = 0x5965, [0x23c3] = 0x61ca, [0x23c4] = 0x6fb3, [0x23c5] = 0x82ad,
  [0x23c6] = 0x634c, [0x23c7] = 0x6252, [0x23c8] = 0x53ed, [0x23c9] = 0x5427,
  [0x23ca] = 0x7b06, [0x23cb] = 0x516b, [0x23cc] = 0x75a4, [0x23cd] = 0x5df4,
  [0x23ce] = 0x62d4, [0x23cf] = 0x8dcb, [0x23d0] = 0x9776, [0x23d1] = 0x628a,
  [0x23d2] = 0x8019, [0x23d3] = 0x575d, [0x23d4] = 0x9738, [0x23d5] = 0x7f62,
  [0x23d6] = 0x7238, [0x23d7] = 0x767d, [0x23d8] = 0x67cf, [0x23d9] = 0x767e,
  [0x23da] = 0x6446, [0x23db] = 0x4f70, [0x23dc] = 0x8d25, [0x23dd] = 0x62dc,
  [0x23de] = 0x7a17, [0x23df] = 0x6591, [0x23e0] = 0x73ed, [0x23e1] = 0x642c,
  [0x23e2] = 0x6273, [0x23e3] = 0x822c, [0x23e4] = 0x9881, [0x23e5] = 0x677f,
  [0x23e6] = 0x7248, [0x23e7] = 0x626e, [0x23e8] = 0x62cc, [0x23e9] = 0x4f34,
  [0x23ea] = 0x74e3, [0x23eb] = 0x534a, [0x23ec] = 0x529e, [0x23ed] = 0x7eca,
  [0x23ee] = 0x90a6, [0x23ef] = 0x5e2e, [0x23f0] = 0x6886, [0x23f1] = 0x699c,
  [0x23f2] = 0x8180, [0x23f3] = 0x7ed1, [0x23f4] = 0x68d2, [0x23f5] = 0x78c5,
  [0x23f6] = 0x868c, [0x23f7] = 0x9551, [0x23f8] = 0x508d, [0x23f9] = 0x8c24,
  [0x23fa] = 0x82de, [0x23fb] = 0x80de, [0x23fc] = 0x5305, [0x23fd] = 0x8912,
  [0x23fe] = 0x5265, [0x2400] = 0x76c4, [0x2401] = 0x76c7, [0x2402] = 0x76c9,
  [0x2403] = 0x76cb, [0x2404] = 0x76cc, [0x2405] = 0x76d3, [0x2406] = 0x76d5,
  [0x2407] = 0x76d9, [0x2408] = 0x76da, [0x2409] = 0x76dc, [0x240a] = 0x76dd,
  [0x240b] = 0x76de, [0x240c] = 0x76e0, [0x240d] = 0x76e1, [0x240e] = 0x76e2,
  [0x240f] = 0x76e3, [0x2410] = 0x76e4, [0x2411] = 0x76e6, [0x2412] = 0x76e7,
  [0x2413] = 0x76e8, [0x2414] = 0x76e9, [0x2415] = 0x76ea, [0x2416] = 0x76eb,
  [0x2417] = 0x76ec, [0x2418] = 0x76ed, [0x2419] = 0x76f0, [0x241a] = 0x76f3,
  [0x241b] = 0x76f5, [0x241c] = 0x76f6, [0x241d] = 0x76f7, [0x241e] = 0x76fa,
  [0x241f] = 0x76fb, [0x2420] = 0x76fd, [0x2421] = 0x76ff, [0x2422] = 0x7700,
  [0x2423] = 0x7702, [0x2424] = 0x7703, [0x2425] = 0x7705, [0x2426] = 0x7706,
  [0x2427] = 0x770a, [0x2428] = 0x770c, [0x2429] = 0x770e, [0x242a] = 0x770f,
  [0x242b] = 0x7710, [0x242c] = 0x7711, [0x242d] = 0x7712, [0x242e] = 0x7713,
  [0x242f] = 0x7714, [0x2430] = 0x7715, [0x2431] = 0x7716, [0x2432] = 0x7717,
  [0x2433] = 0x7718, [0x2434] = 0x771b, [0x2435] = 0x771c, [0x2436] = 0x771d,
  [0x2437] = 0x771e, [0x2438] = 0x7721, [0x2439] = 0x7723, [0x243a] = 0x7724,
  [0x243b] = 0x7725, [0x243c] = 0x7727, [0x243d] = 0x772a, [0x243e] = 0x772b,
  [0x2440] = 0x772c, [0x2441] = 0x772e, [0x2442] = 0x7730, [0x2443] = 0x7731,
  [0x2444] = 0x7732, [0x2445] = 0x7733, [0x2446] = 0x7734, [0x2447] = 0x7739,
  [0x2448] = 0x773b, [0x2449] = 0x773d, [0x244a] = 0x773e, [0x244b] = 0x773f,
  [0x244c] = 0x7742, [0x244d] = 0x7744, [0x244e] = 0x7745, [0x244f] = 0x7746,
  [0x2450] = 0x7748, [0x2451] = 0x7749, [0x2452] = 0x774a, [0x2453] = 0x774b,
  [0x2454] = 0x774c, [0x2455] = 0x774d, [0x2456] = 0x774e, [0x2457] = 0x774f,
  [0x2458] = 0x7752, [0x2459] = 0x7753, [0x245a] = 0x7754, [0x245b] = 0x7755,
  [0x245c] = 0x7756, [0x245d] = 0x7757, [0x245e] = 0x7758, [0x245f] = 0x7759,
  [0x2460] = 0x775c, [0x2461] = 0x8584, [0x2462] = 0x96f9, [0x2463] = 0x4fdd,
  [0x2464] = 0x5821, [0x2465] = 0x9971, [0x2466] = 0x5b9d, [0x2467] = 0x62b1,
  [0x2468] = 0x62a5, [0x2469] = 0x66b4, [0x246a] = 0x8c79, [0x246b] = 0x9c8d,
  [0x246c] = 0x7206, [0x246d] = 0x676f, [0x246e] = 0x7891, [0x246f] = 0x60b2,
  [0x2470] = 0x5351, [0x2471] = 0x5317, [0x2472] = 0x8f88, [0x2473] = 0x80cc,
  [0x2474] = 0x8d1d, [0x2475] = 0x94a1, [0x2476] = 0x500d, [0x2477] = 0x72c8,
  [0x2478] = 0x5907, [0x2479] = 0x60eb, [0x247a] = 0x7119, [0x247b] = 0x88ab,
  [0x247c] = 0x5954, [0x247d] = 0x82ef, [0x247e] = 0x672c, [0x247f] = 0x7b28,
  [0x2480] = 0x5d29, [0x2481] = 0x7ef7, [0x2482] = 0x752d, [0x2483] = 0x6cf5,
  [0x2484] = 0x8e66, [0x2485] = 0x8ff8, [0x2486] = 0x903c, [0x2487] = 0x9f3b,
  [0x2488] = 0x6bd4, [0x2489] = 0x9119, [0x248a] = 0x7b14, [0x248b] = 0x5f7c,
  [0x248c] = 0x78a7, [0x248d] = 0x84d6, [0x248e] = 0x853d, [0x248f] = 0x6bd5,
  [0x2490] = 0x6bd9, [0x2491] = 0x6bd6, [0x2492] = 0x5e01, [0x2493] = 0x5e87,
  [0x2494] = 0x75f9, [0x2495] = 0x95ed, [0x2496] = 0x655d, [0x2497] = 0x5f0a,
  [0x2498] = 0x5fc5, [0x2499] = 0x8f9f, [0x249a] = 0x58c1, [0x249b] = 0x81c2,
  [0x249c] = 0x907f, [0x249d] = 0x965b, [0x249e] = 0x97ad, [0x249f] = 0x8fb9,
  [0x24a0] = 0x7f16, [0x24a1] = 0x8d2c, [0x24a2] = 0x6241, [0x24a3] = 0x4fbf,
  [0x24a4] = 0x53d8, [0x24a5] = 0x535e, [0x24a6] = 0x8fa8, [0x24a7] = 0x8fa9,
  [0x24a8] = 0x8fab, [0x24a9] = 0x904d, [0x24aa] = 0x6807, [0x24ab] = 0x5f6a,
  [0x24ac] = 0x8198, [0x24ad] = 0x8868, [0x24ae] = 0x9cd6, [0x24af] = 0x618b,
  [0x24b0] = 0x522b, [0x24b1] = 0x762a, [0x24b2] = 0x5f6c, [0x24b3] = 0x658c,
  [0x24b4] = 0x6fd2, [0x24b5] = 0x6ee8, [0x24b6] = 0x5bbe, [0x24b7] = 0x6448,
  [0x24b8] = 0x5175, [0x24b9] = 0x51b0, [0x24ba] = 0x67c4, [0x24bb] = 0x4e19,
  [0x24bc] = 0x79c9, [0x24bd] = 0x997c, [0x24be] = 0x70b3, [0x24c0] = 0x775d,
  [0x24c1] = 0x775e, [0x24c2] = 0x775f, [0x24c3] = 0x7760, [0x24c4] = 0x7764,
  [0x24c5] = 0x7767, [0x24c6] = 0x7769, [0x24c7] = 0x776a, [0x24c8] = 0x776d,
  [0x24c9] = 0x776e, [0x24ca] = 0x776f, [0x24cb] = 0x7770, [0x24cc] = 0x7771,
  [0x24cd] = 0x7772, [0x24ce] = 0x7773, [0x24cf] = 0x7774, [0x24d0] = 0x7775,
  [0x24d1] = 0x7776, [0x24d2] = 0x7777, [0x24d3] = 0x7778, [0x24d4] = 0x777a,
  [0x24d5] = 0x777b, [0x24d6] = 0x777c, [0x24d7] = 0x7781, [0x24d8] = 0x7782,
  [0x24d9] = 0x7783, [0x24da] = 0x7786, [0x24db] = 0x7787, [0x24dc] = 0x7788,
  [0x24dd] = 0x7789, [0x24de] = 0x778a, [0x24df] = 0x778b, [0x24e0] = 0x778f,
  [0x24e1] = 0x7790, [0x24e2] = 0x7793, [0x24e3] = 0x7794, [0x24e4] = 0x7795,
  [0x24e5] = 0x7796, [0x24e6] = 0x7797, [0x24e7] = 0x7798, [0x24e8] = 0x7799,
  [0x24e9] = 0x779a, [0x24ea] = 0x779b, [0x24eb] = 0x779c, [0x24ec] = 0x779d,
  [0x24ed] = 0x779e, [0x24ee] = 0x77a1, [0x24ef] = 0x77a3, [0x24f0] = 0x77a4,
  [0x24f1] = 0x77a6, [0x24f2] = 0x77a8, [0x24f3] = 0x77ab, [0x24f4] = 0x77ad,
  [0x24f5] = 0x77ae, [0x24f6] = 0x77af, [0x24f7] = 0x77b1, [0x24f8] = 0x77b2,
  [0x24f9] = 0x77b4, [0x24fa] = 0x77b6, [0x24fb] = 0x77b7, [0x24fc] = 0x77b8,
  [0x24fd] = 0x77b9, [0x24fe] = 0x77ba, [0x2500] = 0x77bc, [0x2501] = 0x77be,
  [0x2502] = 0x77c0, [0x2503] = 0x77c1, [0x2504] = 0x77c2, [0x2505] = 0x77c3,
  [0x2506] = 0x77c4, [0x2507] = 0x77c5, [0x2508] = 0x77c6, [0x2509] = 0x77c7,
  [0x250a] = 0x77c8, [0x250b] = 0x77c9, [0x250c] = 0x77ca, [0x250d] = 0x77cb,
  [0x250e] = 0x77cc, [0x250f] = 0x77ce, [0x2510] = 0x77cf, [0x2511] = 0x77d0,
  [0x2512] = 0x77d1, [0x2513] = 0x77d2, [0x2514] = 0x77d3, [0x2515] = 0x77d4,
  [0x2516] = 0x77d5, [0x2517] = 0x77d6, [0x2518] = 0x77d8, [0x2519] = 0x77d9,
  [0x251a] = 0x77da, [0x251b] = 0x77dd, [0x251c] = 0x77de, [0x251d] = 0x77df,
  [0x251e] = 0x77e0, [0x251f] = 0x77e1, [0x2520] = 0x77e4, [0x2521] = 0x75c5,
  [0x2522] = 0x5e76, [0x2523] = 0x73bb, [0x2524] = 0x83e0, [0x2525] = 0x64ad,
  [0x2526] = 0x62e8, [0x2527] = 0x94b5, [0x2528] = 0x6ce2, [0x2529] = 0x535a,
  [0x252a] = 0x52c3, [0x252b] = 0x640f, [0x252c] = 0x94c2, [0x252d] = 0x7b94,
  [0x252e] = 0x4f2f, [0x252f] = 0x5e1b, [0x2530] = 0x8236, [0x2531] = 0x8116,
  [0x2532] = 0x818a, [0x2533] = 0x6e24, [0x2534] = 0x6cca, [0x2535] = 0x9a73,
  [0x2536] = 0x6355, [0x2537] = 0x535c, [0x2538] = 0x54fa, [0x2539] = 0x8865,
  [0x253a] = 0x57e0, [0x253b] = 0x4e0d, [0x253c] = 0x5e03, [0x253d] = 0x6b65,
  [0x253e] = 0x7c3f, [0x253f] = 0x90e8, [0x2540] = 0x6016, [0x2541] = 0x64e6,
  [0x2542] = 0x731c, [0x2543] = 0x88c1, [0x2544] = 0x6750, [0x2545] = 0x624d,
  [0x2546] = 0x8d22, [0x2547] = 0x776c, [0x2548] = 0x8e29, [0x2549] = 0x91c7,
  [0x254a] = 0x5f69, [0x254b] = 0x83dc, [0x254c] = 0x8521, [0x254d] = 0x9910,
  [0x254e] = 0x53c2, [0x254f] = 0x8695, [0x2550] = 0x6b8b, [0x2551] = 0x60ed,
  [0x2552] = 0x60e8, [0x2553] = 0x707f, [0x2554] = 0x82cd, [0x2555] = 0x8231,
  [0x2556] = 0x4ed3, [0x2557] = 0x6ca7, [0x2558] = 0x85cf, [0x2559] = 0x64cd,
  [0x255a] = 0x7cd9, [0x255b] = 0x69fd, [0x255c] = 0x66f9, [0x255d] = 0x8349,
  [0x255e] = 0x5395, [0x255f] = 0x7b56, [0x2560] = 0x4fa7, [0x2561] = 0x518c,
  [0x2562] = 0x6d4b, [0x2563] = 0x5c42, [0x2564] = 0x8e6d, [0x2565] = 0x63d2,
  [0x2566] = 0x53c9, [0x2567] = 0x832c, [0x2568] = 0x8336, [0x2569] = 0x67e5,
  [0x256a] = 0x78b4, [0x256b] = 0x643d, [0x256c] = 0x5bdf, [0x256d] = 0x5c94,
  [0x256e] = 0x5dee, [0x256f] = 0x8be7, [0x2570] = 0x62c6, [0x2571] = 0x67f4,
  [0x2572] = 0x8c7a, [0x2573] = 0x6400, [0x2574] = 0x63ba, [0x2575] = 0x8749,
  [0x2576] = 0x998b, [0x2577] = 0x8c17, [0x2578] = 0x7f20, [0x2579] = 0x94f2,
  [0x257a] = 0x4ea7, [0x257b] = 0x9610, [0x257c] = 0x98a4, [0x257d] = 0x660c,
  [0x257e] = 0x7316, [0x2580] = 0x77e6, [0x2581] = 0x77e8, [0x2582] = 0x77ea,
  [0x2583] = 0x77ef, [0x2584] = 0x77f0, [0x2585] = 0x77f1, [0x2586] = 0x77f2,
  [0x2587] = 0x77f4, [0x2588] = 0x77f5, [0x2589] = 0x77f7, [0x258a] = 0x77f9,
  [0x258b] = 0x77fa, [0x258c] = 0x77fb, [0x258d] = 0x77fc, [0x258e] = 0x7803,
  [0x258f] = 0x7804, [0x2590] = 0x7805, [0x2591] = 0x7806, [0x2592] = 0x7807,
  [0x2593] = 0x7808, [0x2594] = 0x780a, [0x2595] = 0x780b, [0x2596] = 0x780e,
  [0x2597] = 0x780f, [0x2598] = 0x7810, [0x2599] = 0x7813, [0x259a] = 0x7815,
  [0x259b] = 0x7819, [0x259c] = 0x781b, [0x259d] = 0x781e, [0x259e] = 0x7820,
  [0x259f] = 0x7821, [0x25a0] = 0x7822, [0x25a1] = 0x7824, [0x25a2] = 0x7828,
  [0x25a3] = 0x782a, [0x25a4] = 0x782b, [0x25a5] = 0x782e, [0x25a6] = 0x782f,
  [0x25a7] = 0x7831, [0x25a8] = 0x7832, [0x25a9] = 0x7833, [0x25aa] = 0x7835,
  [0x25ab] = 0x7836, [0x25ac] = 0x783d, [0x25ad] = 0x783f, [0x25ae] = 0x7841,
  [0x25af] = 0x7842, [0x25b0] = 0x7843, [0x25b1] = 0x7844, [0x25b2] = 0x7846,
  [0x25b3] = 0x7848, [0x25b4] = 0x7849, [0x25b5] = 0x784a, [0x25b6] = 0x784b,
  [0x25b7] = 0x784d, [0x25b8] = 0x784f, [0x25b9] = 0x7851, [0x25ba] = 0x7853,
  [0x25bb] = 0x7854, [0x25bc] = 0x7858, [0x25bd] = 0x7859, [0x25be] = 0x785a,
  [0x25c0] = 0x785b, [0x25c1] = 0x785c, [0x25c2] = 0x785e, [0x25c3] = 0x785f,
  [0x25c4] = 0x7860, [0x25c5] = 0x7861, [0x25c6] = 0x7862, [0x25c7] = 0x7863,
  [0x25c8] = 0x7864, [0x25c9] = 0x7865, [0x25ca] = 0x7866, [0x25cb] = 0x7867,
  [0x25cc] = 0x7868, [0x25cd] = 0x7869, [0x25ce] = 0x786f, [0x25cf] = 0x7870,
  [0x25d0] = 0x7871, [0x25d1] = 0x7872, [0x25d2] = 0x7873, [0x25d3] = 0x7874,
  [0x25d4] = 0x7875, [0x25d5] = 0x7876, [0x25d6] = 0x7878, [0x25d7] = 0x7879,
  [0x25d8] = 0x787a, [0x25d9] = 0x787b, [0x25da] = 0x787d, [0x25db] = 0x787e,
  [0x25dc] = 0x787f, [0x25dd] = 0x7880, [0x25de] = 0x7881, [0x25df] = 0x7882,
  [0x25e0] = 0x7883, [0x25e1] = 0x573a, [0x25e2] = 0x5c1d, [0x25e3] = 0x5e38,
  [0x25e4] = 0x957f, [0x25e5] = 0x507f, [0x25e6] = 0x80a0, [0x25e7] = 0x5382,
  [0x25e8] = 0x655e, [0x25e9] = 0x7545, [0x25ea] = 0x5531, [0x25eb] = 0x5021,
  [0x25ec] = 0x8d85, [0x25ed] = 0x6284, [0x25ee] = 0x949e, [0x25ef] = 0x671d,
  [0x25f0] = 0x5632, [0x25f1] = 0x6f6e, [0x25f2] = 0x5de2, [0x25f3] = 0x5435,
  [0x25f4] = 0x7092, [0x25f5] = 0x8f66, [0x25f6] = 0x626f, [0x25f7] = 0x64a4,
  [0x25f8] = 0x63a3, [0x25f9] = 0x5f7b, [0x25fa] = 0x6f88, [0x25fb] = 0x90f4,
  [0x25fc] = 0x81e3, [0x25fd] = 0x8fb0, [0x25fe] = 0x5c18, [0x25ff] = 0x6668,
  [0x2600] = 0x5ff1, [0x2601] = 0x6c89, [0x2602] = 0x9648, [0x2603] = 0x8d81,
  [0x2604] = 0x886c, [0x2605] = 0x6491, [0x2606] = 0x79f0, [0x2607] = 0x57ce,
  [0x2608] = 0x6a59, [0x2609] = 0x6210, [0x260a] = 0x5448, [0x260b] = 0x4e58,
  [0x260c] = 0x7a0b, [0x260d] = 0x60e9, [0x260e] = 0x6f84, [0x260f] = 0x8bda,
  [0x2610] = 0x627f, [0x2611] = 0x901e, [0x2612] = 0x9a8b, [0x2613] = 0x79e4,
  [0x2614] = 0x5403, [0x2615] = 0x75f4, [0x2616] = 0x6301, [0x2617] = 0x5319,
  [0x2618] = 0x6c60, [0x2619] = 0x8fdf, [0x261a] = 0x5f1b, [0x261b] = 0x9a70,
  [0x261c] = 0x803b, [0x261d] = 0x9f7f, [0x261e] = 0x4f88, [0x261f] = 0x5c3a,
  [0x2620] = 0x8d64, [0x2621] = 0x7fc5, [0x2622] = 0x65a5, [0x2623] = 0x70bd,
  [0x2624] = 0x5145, [0x2625] = 0x51b2, [0x2626] = 0x866b, [0x2627] = 0x5d07,
  [0x2628] = 0x5ba0, [0x2629] = 0x62bd, [0x262a] = 0x916c, [0x262b] = 0x7574,
  [0x262c] = 0x8e0c, [0x262d] = 0x7a20, [0x262e] = 0x6101, [0x262f] = 0x7b79,
  [0x2630] = 0x4ec7, [0x2631] = 0x7ef8, [0x2632] = 0x7785, [0x2633] = 0x4e11,
  [0x2634] = 0x81ed, [0x2635] = 0x521d, [0x2636] = 0x51fa, [0x2637] = 0x6a71,
  [0x2638] = 0x53a8, [0x2639] = 0x8e87, [0x263a] = 0x9504, [0x263b] = 0x96cf,
  [0x263c] = 0x6ec1, [0x263d] = 0x9664, [0x263e] = 0x695a, [0x2640] = 0x7884,
  [0x2641] = 0x7885, [0x2642] = 0x7886, [0x2643] = 0x7888, [0x2644] = 0x788a,
  [0x2645] = 0x788b, [0x2646] = 0x788f, [0x2647] = 0x7890, [0x2648] = 0x7892,
  [0x2649] = 0x7894, [0x264a] = 0x7895, [0x264b] = 0x7896, [0x264c] = 0x7899,
  [0x264d] = 0x789d, [0x264e] = 0x789e, [0x264f] = 0x78a0, [0x2650] = 0x78a2,
  [0x2651] = 0x78a4, [0x2652] = 0x78a6, [0x2653] = 0x78a8, [0x2654] = 0x78a9,
  [0x2655] = 0x78aa, [0x2656] = 0x78ab, [0x2657] = 0x78ac, [0x2658] = 0x78ad,
  [0x2659] = 0x78ae, [0x265a] = 0x78af, [0x265b] = 0x78b5, [0x265c] = 0x78b6,
  [0x265d] = 0x78b7, [0x265e] = 0x78b8, [0x265f] = 0x78ba, [0x2660] = 0x78bb,
  [0x2661] = 0x78bc, [0x2662] = 0x78bd, [0x2663] = 0x78bf, [0x2664] = 0x78c0,
  [0x2665] = 0x78c2, [0x2666] = 0x78c3, [0x2667] = 0x78c4, [0x2668] = 0x78c6,
  [0x2669] = 0x78c7, [0x266a] = 0x78c8, [0x266b] = 0x78cc, [0x266c] = 0x78cd,
  [0x266d] = 0x78ce, [0x266e] = 0x78cf, [0x266f] = 0x78d1, [0x2670] = 0x78d2,
  [0x2671] = 0x78d3, [0x2672] = 0x78d6, [0x2673] = 0x78d7, [0x2674] = 0x78d8,
  [0x2675] = 0x78da, [0x2676] = 0x78db, [0x2677] = 0x78dc, [0x2678] = 0x78dd,
  [0x2679] = 0x78de, [0x267a] = 0x78df, [0x267b] = 0x78e0, [0x267c] = 0x78e1,
  [0x267d] = 0x78e2, [0x267e] = 0x78e3, [0x2680] = 0x78e4, [0x2681] = 0x78e5,
  [0x2682] = 0x78e6, [0x2683] = 0x78e7, [0x2684] = 0x78e9, [0x2685] = 0x78ea,
  [0x2686] = 0x78eb, [0x2687] = 0x78ed, [0x2688] = 0x78ee, [0x2689] = 0x78ef,
  [0x268a] = 0x78f0, [0x268b] = 0x78f1, [0x268c] = 0x78f3, [0x268d] = 0x78f5,
  [0x268e] = 0x78f6, [0x268f] = 0x78f8, [0x2690] = 0x78f9, [0x2691] = 0x78fb,
  [0x2692] = 0x78fc, [0x2693] = 0x78fd, [0x2694] = 0x78fe, [0x2695] = 0x78ff,
  [0x2696] = 0x7900, [0x2697] = 0x7902, [0x2698] = 0x7903, [0x2699] = 0x7904,
  [0x269a] = 0x7906, [0x269b] = 0x7907, [0x269c] = 0x7908, [0x269d] = 0x7909,
  [0x269e] = 0x790a, [0x269f] = 0x790b, [0x26a0] = 0x790c, [0x26a1] = 0x7840,
  [0x26a2] = 0x50a8, [0x26a3] = 0x77d7, [0x26a4] = 0x6410, [0x26a5] = 0x89e6,
  [0x26a6] = 0x5904, [0x26a7] = 0x63e3, [0x26a8] = 0x5ddd, [0x26a9] = 0x7a7f,
  [0x26aa] = 0x693d, [0x26ab] = 0x4f20, [0x26ac] = 0x8239, [0x26ad] = 0x5598,
  [0x26ae] = 0x4e32, [0x26af] = 0x75ae, [0x26b0] = 0x7a97, [0x26b1] = 0x5e62,
  [0x26b2] = 0x5e8a, [0x26b3] = 0x95ef, [0x26b4] = 0x521b, [0x26b5] = 0x5439,
  [0x26b6] = 0x708a, [0x26b7] = 0x6376, [0x26b8] = 0x9524, [0x26b9] = 0x5782,
  [0x26ba] = 0x6625, [0x26bb] = 0x693f, [0x26bc] = 0x9187, [0x26bd] = 0x5507,
  [0x26be] = 0x6df3, [0x26bf] = 0x7eaf, [0x26c0] = 0x8822, [0x26c1] = 0x6233,
  [0x26c2] = 0x7ef0, [0x26c3] = 0x75b5, [0x26c4] = 0x8328, [0x26c5] = 0x78c1,
  [0x26c6] = 0x96cc, [0x26c7] = 0x8f9e, [0x26c8] = 0x6148, [0x26c9] = 0x74f7,
  [0x26ca] = 0x8bcd, [0x26cb] = 0x6b64, [0x26cc] = 0x523a, [0x26cd] = 0x8d50,
  [0x26ce] = 0x6b21, [0x26cf] = 0x806a, [0x26d0] = 0x8471, [0x26d1] = 0x56f1,
  [0x26d2] = 0x5306, [0x26d3] = 0x4ece, [0x26d4] = 0x4e1b, [0x26d5] = 0x51d1,
  [0x26d6] = 0x7c97, [0x26d7] = 0x918b, [0x26d8] = 0x7c07, [0x26d9] = 0x4fc3,
  [0x26da] = 0x8e7f, [0x26db] = 0x7be1, [0x26dc] = 0x7a9c, [0x26dd] = 0x6467,
  [0x26de] = 0x5d14, [0x26df] = 0x50ac, [0x26e0] = 0x8106, [0x26e1] = 0x7601,
  [0x26e2] = 0x7cb9, [0x26e3] = 0x6dec, [0x26e4] = 0x7fe0, [0x26e5] = 0x6751,
  [0x26e6] = 0x5b58, [0x26e7] = 0x5bf8, [0x26e8] = 0x78cb, [0x26e9] = 0x64ae,
  [0x26ea] = 0x6413, [0x26eb] = 0x63aa, [0x26ec] = 0x632b, [0x26ed] = 0x9519,
  [0x26ee] = 0x642d, [0x26ef] = 0x8fbe, [0x26f0] = 0x7b54, [0x26f1] = 0x7629,
  [0x26f2] = 0x6253, [0x26f3] = 0x5927, [0x26f4] = 0x5446, [0x26f5] = 0x6b79,
  [0x26f6] = 0x50a3, [0x26f7] = 0x6234, [0x26f8] = 0x5e26, [0x26f9] = 0x6b86,
  [0x26fa] = 0x4ee3, [0x26fb] = 0x8d37, [0x26fc] = 0x888b, [0x26fd] = 0x5f85,
  [0x26fe] = 0x902e, [0x2700] = 0x790d, [0x2701] = 0x790e, [0x2702] = 0x790f,
  [0x2703] = 0x7910, [0x2704] = 0x7911, [0x2705] = 0x7912, [0x2706] = 0x7914,
  [0x2707] = 0x7915, [0x2708] = 0x7916, [0x2709] = 0x7917, [0x270a] = 0x7918,
  [0x270b] = 0x7919, [0x270c] = 0x791a, [0x270d] = 0x791b, [0x270e] = 0x791c,
  [0x270f] = 0x791d, [0x2710] = 0x791f, [0x2711] = 0x7920, [0x2712] = 0x7921,
  [0x2713] = 0x7922, [0x2714] = 0x7923, [0x2715] = 0x7925, [0x2716] = 0x7926,
  [0x2717] = 0x7927, [0x2718] = 0x7928, [0x2719] = 0x7929, [0x271a] = 0x792a,
  [0x271b] = 0x792b, [0x271c] = 0x792c, [0x271d] = 0x792d, [0x271e] = 0x792e,
  [0x271f] = 0x792f, [0x2720] = 0x7930, [0x2721] = 0x7931, [0x2722] = 0x7932,
  [0x2723] = 0x7933, [0x2724] = 0x7935, [0x2725] = 0x7936, [0x2726] = 0x7937,
  [0x2727] = 0x7938, [0x2728] = 0x7939, [0x2729] = 0x793d, [0x272a] = 0x793f,
  [0x272b] = 0x7942, [0x272c] = 0x7943, [0x272d] = 0x7944, [0x272e] = 0x7945,
  [0x272f] = 0x7947, [0x2730] = 0x794a, [0x2731] = 0x794b, [0x2732] = 0x794c,
  [0x2733] = 0x794d, [0x2734] = 0x794e, [0x2735] = 0x794f, [0x2736] = 0x7950,
  [0x2737] = 0x7951, [0x2738] = 0x7952, [0x2739] = 0x7954, [0x273a] = 0x7955,
  [0x273b] = 0x7958, [0x273c] = 0x7959, [0x273d] = 0x7961, [0x273e] = 0x7963,
  [0x2740] = 0x7964, [0x2741] = 0x7966, [0x2742] = 0x7969, [0x2743] = 0x796a,
  [0x2744] = 0x796b, [0x2745] = 0x796c, [0x2746] = 0x796e, [0x2747] = 0x7970,
  [0x2748] = 0x7971, [0x2749] = 0x7972, [0x274a] = 0x7973, [0x274b] = 0x7974,
  [0x274c] = 0x7975, [0x274d] = 0x7976, [0x274e] = 0x7979, [0x274f] = 0x797b,
  [0x2750] = 0x797c, [0x2751] = 0x797d, [0x2752] = 0x797e, [0x2753] = 0x797f,
  [0x2754] = 0x7982, [0x2755] = 0x7983, [0x2756] = 0x7986, [0x2757] = 0x7987,
  [0x2758] = 0x7988, [0x2759] = 0x7989, [0x275a] = 0x798b, [0x275b] = 0x798c,
  [0x275c] = 0x798d, [0x275d] = 0x798e, [0x275e] = 0x7990, [0x275f] = 0x7991,
  [0x2760] = 0x7992, [0x2761] = 0x6020, [0x2762] = 0x803d, [0x2763] = 0x62c5,
  [0x2764] = 0x4e39, [0x2765] = 0x5355, [0x2766] = 0x90f8, [0x2767] = 0x63b8,
  [0x2768] = 0x80c6, [0x2769] = 0x65e6, [0x276a] = 0x6c2e, [0x276b] = 0x4f46,
  [0x276c] = 0x60ee, [0x276d] = 0x6de1, [0x276e] = 0x8bde, [0x276f] = 0x5f39,
  [0x2770] = 0x86cb, [0x2771] = 0x5f53, [0x2772] = 0x6321, [0x2773] = 0x515a,
  [0x2774] = 0x8361, [0x2775] = 0x6863, [0x2776] = 0x5200, [0x2777] = 0x6363,
  [0x2778] = 0x8e48, [0x2779] = 0x5012, [0x277a] = 0x5c9b, [0x277b] = 0x7977,
  [0x277c] = 0x5bfc, [0x277d] = 0x5230, [0x277e] = 0x7a3b, [0x277f] = 0x60bc,
  [0x2780] = 0x9053, [0x2781] = 0x76d7, [0x2782] = 0x5fb7, [0x2783] = 0x5f97,
  [0x2784] = 0x7684, [0x2785] = 0x8e6c, [0x2786] = 0x706f, [0x2787] = 0x767b,
  [0x2788] = 0x7b49, [0x2789] = 0x77aa, [0x278a] = 0x51f3, [0x278b] = 0x9093,
  [0x278c] = 0x5824, [0x278d] = 0x4f4e, [0x278e] = 0x6ef4, [0x278f] = 0x8fea,
  [0x2790] = 0x654c, [0x2791] = 0x7b1b, [0x2792] = 0x72c4, [0x2793] = 0x6da4,
  [0x2794] = 0x7fdf, [0x2795] = 0x5ae1, [0x2796] = 0x62b5, [0x2797] = 0x5e95,
  [0x2798] = 0x5730, [0x2799] = 0x8482, [0x279a] = 0x7b2c, [0x279b] = 0x5e1d,
  [0x279c] = 0x5f1f, [0x279d] = 0x9012, [0x279e] = 0x7f14, [0x279f] = 0x98a0,
  [0x27a0] = 0x6382, [0x27a1] = 0x6ec7, [0x27a2] = 0x7898, [0x27a3] = 0x70b9,
  [0x27a4] = 0x5178, [0x27a5] = 0x975b, [0x27a6] = 0x57ab, [0x27a7] = 0x7535,
  [0x27a8] = 0x4f43, [0x27a9] = 0x7538, [0x27aa] = 0x5e97, [0x27ab] = 0x60e6,
  [0x27ac] = 0x5960, [0x27ad] = 0x6dc0, [0x27ae] = 0x6bbf, [0x27af] = 0x7889,
  [0x27b0] = 0x53fc, [0x27b1] = 0x96d5, [0x27b2] = 0x51cb, [0x27b3] = 0x5201,
  [0x27b4] = 0x6389, [0x27b5] = 0x540a, [0x27b6] = 0x9493, [0x27b7] = 0x8c03,
  [0x27b8] = 0x8dcc, [0x27b9] = 0x7239, [0x27ba] = 0x789f, [0x27bb] = 0x8776,
  [0x27bc] = 0x8fed, [0x27bd] = 0x8c0d, [0x27be] = 0x53e0, [0x27c0] = 0x7993,
  [0x27c1] = 0x7994, [0x27c2] = 0x7995, [0x27c3] = 0x7996, [0x27c4] = 0x7997,
  [0x27c5] = 0x7998, [0x27c6] = 0x7999, [0x27c7] = 0x799b, [0x27c8] = 0x799c,
  [0x27c9] = 0x799d, [0x27ca] = 0x799e, [0x27cb] = 0x799f, [0x27cc] = 0x79a0,
  [0x27cd] = 0x79a1, [0x27ce] = 0x79a2, [0x27cf] = 0x79a3, [0x27d0] = 0x79a4,
  [0x27d1] = 0x79a5, [0x27d2] = 0x79a6, [0x27d3] = 0x79a8, [0x27d4] = 0x79a9,
  [0x27d5] = 0x79aa, [0x27d6] = 0x79ab, [0x27d7] = 0x79ac, [0x27d8] = 0x79ad,
  [0x27d9] = 0x79ae, [0x27da] = 0x79af, [0x27db] = 0x79b0, [0x27dc] = 0x79b1,
  [0x27dd] = 0x79b2, [0x27de] = 0x79b4, [0x27df] = 0x79b5, [0x27e0] = 0x79b6,
  [0x27e1] = 0x79b7, [0x27e2] = 0x79b8, [0x27e3] = 0x79bc, [0x27e4] = 0x79bf,
  [0x27e5] = 0x79c2, [0x27e6] = 0x79c4, [0x27e7] = 0x79c5, [0x27e8] = 0x79c7,
  [0x27e9] = 0x79c8, [0x27ea] = 0x79ca, [0x27eb] = 0x79cc, [0x27ec] = 0x79ce,
  [0x27ed] = 0x79cf, [0x27ee] = 0x79d0, [0x27ef] = 0x79d3, [0x27f0] = 0x79d4,
  [0x27f1] = 0x79d6, [0x27f2] = 0x79d7, [0x27f3] = 0x79d9, [0x27f4] = 0x79da,
  [0x27f5] = 0x79db, [0x27f6] = 0x79dc, [0x27f7] = 0x79dd, [0x27f8] = 0x79de,
  [0x27f9] = 0x79e0, [0x27fa] = 0x79e1, [0x27fb] = 0x79e2, [0x27fc] = 0x79e5,
  [0x27fd] = 0x79e8, [0x27fe] = 0x79ea, [0x2800] = 0x79ec, [0x2801] = 0x79ee,
  [0x2802] = 0x79f1, [0x2803] = 0x79f2, [0x2804] = 0x79f3, [0x2805] = 0x79f4,
  [0x2806] = 0x79f5, [0x2807] = 0x79f6, [0x2808] = 0x79f7, [0x2809] = 0x79f9,
  [0x280a] = 0x79fa, [0x280b] = 0x79fc, [0x280c] = 0x79fe, [0x280d] = 0x79ff,
  [0x280e] = 0x7a01, [0x280f] = 0x7a04, [0x2810] = 0x7a05, [0x2811] = 0x7a07,
  [0x2812] = 0x7a08, [0x2813] = 0x7a09, [0x2814] = 0x7a0a, [0x2815] = 0x7a0c,
  [0x2816] = 0x7a0f, [0x2817] = 0x7a10, [0x2818] = 0x7a11, [0x2819] = 0x7a12,
  [0x281a] = 0x7a13, [0x281b] = 0x7a15, [0x281c] = 0x7a16, [0x281d] = 0x7a18,
  [0x281e] = 0x7a19, [0x281f] = 0x7a1b, [0x2820] = 0x7a1c, [0x2821] = 0x4e01,
  [0x2822] = 0x76ef, [0x2823] = 0x53ee, [0x2824] = 0x9489, [0x2825] = 0x9876,
  [0x2826] = 0x9f0e, [0x2827] = 0x952d, [0x2828] = 0x5b9a, [0x2829] = 0x8ba2,
  [0x282a] = 0x4e22, [0x282b] = 0x4e1c, [0x282c] = 0x51ac, [0x282d] = 0x8463,
  [0x282e] = 0x61c2, [0x282f] = 0x52a8, [0x2830] = 0x680b, [0x2831] = 0x4f97,
  [0x2832] = 0x606b, [0x2833] = 0x51bb, [0x2834] = 0x6d1e, [0x2835] = 0x515c,
  [0x2836] = 0x6296, [0x2837] = 0x6597, [0x2838] = 0x9661, [0x2839] = 0x8c46,
  [0x283a] = 0x9017, [0x283b] = 0x75d8, [0x283c] = 0x90fd, [0x283d] = 0x7763,
  [0x283e] = 0x6bd2, [0x283f] = 0x728a, [0x2840] = 0x72ec, [0x2841] = 0x8bfb,
  [0x2842] = 0x5835, [0x2843] = 0x7779, [0x2844] = 0x8d4c, [0x2845] = 0x675c,
  [0x2846] = 0x9540, [0x2847] = 0x809a, [0x2848] = 0x5ea6, [0x2849] = 0x6e21,
  [0x284a] = 0x5992, [0x284b] = 0x7aef, [0x284c] = 0x77ed, [0x284d] = 0x953b,
  [0x284e] = 0x6bb5, [0x284f] = 0x65ad, [0x2850] = 0x7f0e, [0x2851] = 0x5806,
  [0x2852] = 0x5151, [0x2853] = 0x961f, [0x2854] = 0x5bf9, [0x2855] = 0x58a9,
  [0x2856] = 0x5428, [0x2857] = 0x8e72, [0x2858] = 0x6566, [0x2859] = 0x987f,
  [0x285a] = 0x56e4, [0x285b] = 0x949d, [0x285c] = 0x76fe, [0x285d] = 0x9041,
  [0x285e] = 0x6387, [0x285f] = 0x54c6, [0x2860] = 0x591a, [0x2861] = 0x593a,
  [0x2862] = 0x579b, [0x2863] = 0x8eb2, [0x2864] = 0x6735, [0x2865] = 0x8dfa,
  [0x2866] = 0x8235, [0x2867] = 0x5241, [0x2868] = 0x60f0, [0x2869] = 0x5815,
  [0x286a] = 0x86fe, [0x286b] = 0x5ce8, [0x286c] = 0x9e45, [0x286d] = 0x4fc4,
  [0x286e] = 0x989d, [0x286f] = 0x8bb9, [0x2870] = 0x5a25, [0x2871] = 0x6076,
  [0x2872] = 0x5384, [0x2873] = 0x627c, [0x2874] = 0x904f, [0x2875] = 0x9102,
  [0x2876] = 0x997f, [0x2877] = 0x6069, [0x2878] = 0x800c, [0x2879] = 0x513f,
  [0x287a] = 0x8033, [0x287b] = 0x5c14, [0x287c] = 0x9975, [0x287d] = 0x6d31,
  [0x287e] = 0x4e8c, [0x2880] = 0x7a1d, [0x2881] = 0x7a1f, [0x2882] = 0x7a21,
  [0x2883] = 0x7a22, [0x2884] = 0x7a24, [0x2885] = 0x7a25, [0x2886] = 0x7a26,
  [0x2887] = 0x7a27, [0x2888] = 0x7a28, [0x2889] = 0x7a29, [0x288a] = 0x7a2a,
  [0x288b] = 0x7a2b, [0x288c] = 0x7a2c, [0x288d] = 0x7a2d, [0x288e] = 0x7a2e,
  [0x288f] = 0x7a2f, [0x2890] = 0x7a30, [0x2891] = 0x7a31, [0x2892] = 0x7a32,
  [0x2893] = 0x7a34, [0x2894] = 0x7a35, [0x2895] = 0x7a36, [0x2896] = 0x7a38,
  [0x2897] = 0x7a3a, [0x2898] = 0x7a3e, [0x2899] = 0x7a40, [0x289a] = 0x7a41,
  [0x289b] = 0x7a42, [0x289c] = 0x7a43, [0x289d] = 0x7a44, [0x289e] = 0x7a45,
  [0x289f] = 0x7a47, [0x28a0] = 0x7a48, [0x28a1] = 0x7a49, [0x28a2] = 0x7a4a,
  [0x28a3] = 0x7a4b, [0x28a4] = 0x7a4c, [0x28a5] = 0x7a4d, [0x28a6] = 0x7a4e,
  [0x28a7] = 0x7a4f, [0x28a8] = 0x7a50, [0x28a9] = 0x7a52, [0x28aa] = 0x7a53,
  [0x28ab] = 0x7a54, [0x28ac] = 0x7a55, [0x28ad] = 0x7a56, [0x28ae] = 0x7a58,
  [0x28af] = 0x7a59, [0x28b0] = 0x7a5a, [0x28b1] = 0x7a5b, [0x28b2] = 0x7a5c,
  [0x28b3] = 0x7a5d, [0x28b4] = 0x7a5e, [0x28b5] = 0x7a5f, [0x28b6] = 0x7a60,
  [0x28b7] = 0x7a61, [0x28b8] = 0x7a62, [0x28b9] = 0x7a63, [0x28ba] = 0x7a64,
  [0x28bb] = 0x7a65, [0x28bc] = 0x7a66, [0x28bd] = 0x7a67, [0x28be] = 0x7a68,
  [0x28c0] = 0x7a69, [0x28c1] = 0x7a6a, [0x28c2] = 0x7a6b, [0x28c3] = 0x7a6c,
  [0x28c4] = 0x7a6d, [0x28c5] = 0x7a6e, [0x28c6] = 0x7a6f, [0x28c7] = 0x7a71,
  [0x28c8] = 0x7a72, [0x28c9] = 0x7a73, [0x28ca] = 0x7a75, [0x28cb] = 0x7a7b,
  [0x28cc] = 0x7a7c, [0x28cd] = 0x7a7d, [0x28ce] = 0x7a7e, [0x28cf] = 0x7a82,
  [0x28d0] = 0x7a85, [0x28d1] = 0x7a87, [0x28d2] = 0x7a89, [0x28d3] = 0x7a8a,
  [0x28d4] = 0x7a8b, [0x28d5] = 0x7a8c, [0x28d6] = 0x7a8e, [0x28d7] = 0x7a8f,
  [0x28d8] = 0x7a90, [0x28d9] = 0x7a93, [0x28da] = 0x7a94, [0x28db] = 0x7a99,
  [0x28dc] = 0x7a9a, [0x28dd] = 0x7a9b, [0x28de] = 0x7a9e, [0x28df] = 0x7aa1,
  [0x28e0] = 0x7aa2, [0x28e1] = 0x8d30, [0x28e2] = 0x53d1, [0x28e3] = 0x7f5a,
  [0x28e4] = 0x7b4f, [0x28e5] = 0x4f10, [0x28e6] = 0x4e4f, [0x28e7] = 0x9600,
  [0x28e8] = 0x6cd5, [0x28e9] = 0x73d0, [0x28ea] = 0x85e9, [0x28eb] = 0x5e06,
  [0x28ec] = 0x756a, [0x28ed] = 0x7ffb, [0x28ee] = 0x6a0a, [0x28ef] = 0x77fe,
  [0x28f0] = 0x9492, [0x28f1] = 0x7e41, [0x28f2] = 0x51e1, [0x28f3] = 0x70e6,
  [0x28f4] = 0x53cd, [0x28f5] = 0x8fd4, [0x28f6] = 0x8303, [0x28f7] = 0x8d29,
  [0x28f8] = 0x72af, [0x28f9] = 0x996d, [0x28fa] = 0x6cdb, [0x28fb] = 0x574a,
  [0x28fc] = 0x82b3, [0x28fd] = 0x65b9, [0x28fe] = 0x80aa, [0x28ff] = 0x623f,
  [0x2900] = 0x9632, [0x2901] = 0x59a8, [0x2902] = 0x4eff, [0x2903] = 0x8bbf,
  [0x2904] = 0x7eba, [0x2905] = 0x653e, [0x2906] = 0x83f2, [0x2907] = 0x975e,
  [0x2908] = 0x5561, [0x2909] = 0x98de, [0x290a] = 0x80a5, [0x290b] = 0x532a,
  [0x290c] = 0x8bfd, [0x290d] = 0x5420, [0x290e] = 0x80ba, [0x290f] = 0x5e9f,
  [0x2910] = 0x6cb8, [0x2911] = 0x8d39, [0x2912] = 0x82ac, [0x2913] = 0x915a,
  [0x2914] = 0x5429, [0x2915] = 0x6c1b, [0x2916] = 0x5206, [0x2917] = 0x7eb7,
  [0x2918] = 0x575f, [0x2919] = 0x711a, [0x291a] = 0x6c7e, [0x291b] = 0x7c89,
  [0x291c] = 0x594b, [0x291d] = 0x4efd, [0x291e] = 0x5fff, [0x291f] = 0x6124,
  [0x2920] = 0x7caa, [0x2921] = 0x4e30, [0x2922] = 0x5c01, [0x2923] = 0x67ab,
  [0x2924] = 0x8702, [0x2925] = 0x5cf0, [0x2926] = 0x950b, [0x2927] = 0x98ce,
  [0x2928] = 0x75af, [0x2929] = 0x70fd, [0x292a] = 0x9022, [0x292b] = 0x51af,
  [0x292c] = 0x7f1d, [0x292d] = 0x8bbd, [0x292e] = 0x5949, [0x292f] = 0x51e4,
  [0x2930] = 0x4f5b, [0x2931] = 0x5426, [0x2932] = 0x592b, [0x2933] = 0x6577,
  [0x2934] = 0x80a4, [0x2935] = 0x5b75, [0x2936] = 0x6276, [0x2937] = 0x62c2,
  [0x2938] = 0x8f90, [0x2939] = 0x5e45, [0x293a] = 0x6c1f, [0x293b] = 0x7b26,
  [0x293c] = 0x4f0f, [0x293d] = 0x4fd8, [0x293e] = 0x670d, [0x2940] = 0x7aa3,
  [0x2941] = 0x7aa4, [0x2942] = 0x7aa7, [0x2943] = 0x7aa9, [0x2944] = 0x7aaa,
  [0x2945] = 0x7aab, [0x2946] = 0x7aae, [0x2947] = 0x7aaf, [0x2948] = 0x7ab0,
  [0x2949] = 0x7ab1, [0x294a] = 0x7ab2, [0x294b] = 0x7ab4, [0x294c] = 0x7ab5,
  [0x294d] = 0x7ab6, [0x294e] = 0x7ab7, [0x294f] = 0x7ab8, [0x2950] = 0x7ab9,
  [0x2951] = 0x7aba, [0x2952] = 0x7abb, [0x2953] = 0x7abc, [0x2954] = 0x7abd,
  [0x2955] = 0x7abe, [0x2956] = 0x7ac0, [0x2957] = 0x7ac1, [0x2958] = 0x7ac2,
  [0x2959] = 0x7ac3, [0x295a] = 0x7ac4, [0x295b] = 0x7ac5, [0x295c] = 0x7ac6,
  [0x295d] = 0x7ac7, [0x295e] = 0x7ac8, [0x295f] = 0x7ac9, [0x2960] = 0x7aca,
  [0x2961] = 0x7acc, [0x2962] = 0x7acd, [0x2963] = 0x7ace, [0x2964] = 0x7acf,
  [0x2965] = 0x7ad0, [0x2966] = 0x7ad1, [0x2967] = 0x7ad2, [0x2968] = 0x7ad3,
  [0x2969] = 0x7ad4, [0x296a] = 0x7ad5, [0x296b] = 0x7ad7, [0x296c] = 0x7ad8,
  [0x296d] = 0x7ada, [0x296e] = 0x7adb, [0x296f] = 0x7adc, [0x2970] = 0x7add,
  [0x2971] = 0x7ae1, [0x2972] = 0x7ae2, [0x2973] = 0x7ae4, [0x2974] = 0x7ae7,
  [0x2975] = 0x7ae8, [0x2976] = 0x7ae9, [0x2977] = 0x7aea, [0x2978] = 0x7aeb,
  [0x2979] = 0x7aec, [0x297a] = 0x7aee, [0x297b] = 0x7af0, [0x297c] = 0x7af1,
  [0x297d] = 0x7af2, [0x297e] = 0x7af3, [0x2980] = 0x7af4, [0x2981] = 0x7af5,
  [0x2982] = 0x7af6, [0x2983] = 0x7af7, [0x2984] = 0x7af8, [0x2985] = 0x7afb,
  [0x2986] = 0x7afc, [0x2987] = 0x7afe, [0x2988] = 0x7b00, [0x2989] = 0x7b01,
  [0x298a] = 0x7b02, [0x298b] = 0x7b05, [0x298c] = 0x7b07, [0x298d] = 0x7b09,
  [0x298e] = 0x7b0c, [0x298f] = 0x7b0d, [0x2990] = 0x7b0e, [0x2991] = 0x7b10,
  [0x2992] = 0x7b12, [0x2993] = 0x7b13, [0x2994] = 0x7b16, [0x2995] = 0x7b17,
  [0x2996] = 0x7b18, [0x2997] = 0x7b1a, [0x2998] = 0x7b1c, [0x2999] = 0x7b1d,
  [0x299a] = 0x7b1f, [0x299b] = 0x7b21, [0x299c] = 0x7b22, [0x299d] = 0x7b23,
  [0x299e] = 0x7b27, [0x299f] = 0x7b29, [0x29a0] = 0x7b2d, [0x29a1] = 0x6d6e,
  [0x29a2] = 0x6daa, [0x29a3] = 0x798f, [0x29a4] = 0x88b1, [0x29a5] = 0x5f17,
  [0x29a6] = 0x752b, [0x29a7] = 0x629a, [0x29a8] = 0x8f85, [0x29a9] = 0x4fef,
  [0x29aa] = 0x91dc, [0x29ab] = 0x65a7, [0x29ac] = 0x812f, [0x29ad] = 0x8151,
  [0x29ae] = 0x5e9c, [0x29af] = 0x8150, [0x29b0] = 0x8d74, [0x29b1] = 0x526f,
  [0x29b2] = 0x8986, [0x29b3] = 0x8d4b, [0x29b4] = 0x590d, [0x29b5] = 0x5085,
  [0x29b6] = 0x4ed8, [0x29b7] = 0x961c, [0x29b8] = 0x7236, [0x29b9] = 0x8179,
  [0x29ba] = 0x8d1f, [0x29bb] = 0x5bcc, [0x29bc] = 0x8ba3, [0x29bd] = 0x9644,
  [0x29be] = 0x5987, [0x29bf] = 0x7f1a, [0x29c0] = 0x5490, [0x29c1] = 0x5676,
  [0x29c2] = 0x560e, [0x29c3] = 0x8be5, [0x29c4] = 0x6539, [0x29c5] = 0x6982,
  [0x29c6] = 0x9499, [0x29c7] = 0x76d6, [0x29c8] = 0x6e89, [0x29c9] = 0x5e72,
  [0x29ca] = 0x7518, [0x29cb] = 0x6746, [0x29cc] = 0x67d1, [0x29cd] = 0x7aff,
  [0x29ce] = 0x809d, [0x29cf] = 0x8d76, [0x29d0] = 0x611f, [0x29d1] = 0x79c6,
  [0x29d2] = 0x6562, [0x29d3] = 0x8d63, [0x29d4] = 0x5188, [0x29d5] = 0x521a,
  [0x29d6] = 0x94a2, [0x29d7] = 0x7f38, [0x29d8] = 0x809b, [0x29d9] = 0x7eb2,
  [0x29da] = 0x5c97, [0x29db] = 0x6e2f, [0x29dc] = 0x6760, [0x29dd] = 0x7bd9,
  [0x29de] = 0x768b, [0x29df] = 0x9ad8, [0x29e0] = 0x818f, [0x29e1] = 0x7f94,
  [0x29e2] = 0x7cd5, [0x29e3] = 0x641e, [0x29e4] = 0x9550, [0x29e5] = 0x7a3f,
  [0x29e6] = 0x544a, [0x29e7] = 0x54e5, [0x29e8] = 0x6b4c, [0x29e9] = 0x6401,
  [0x29ea] = 0x6208, [0x29eb] = 0x9e3d, [0x29ec] = 0x80f3, [0x29ed] = 0x7599,
  [0x29ee] = 0x5272, [0x29ef] = 0x9769, [0x29f0] = 0x845b, [0x29f1] = 0x683c,
  [0x29f2] = 0x86e4, [0x29f3] = 0x9601, [0x29f4] = 0x9694, [0x29f5] = 0x94ec,
  [0x29f6] = 0x4e2a, [0x29f7] = 0x5404, [0x29f8] = 0x7ed9, [0x29f9] = 0x6839,
  [0x29fa] = 0x8ddf, [0x29fb] = 0x8015, [0x29fc] = 0x66f4, [0x29fd] = 0x5e9a,
  [0x29fe] = 0x7fb9, [0x2a00] = 0x7b2f, [0x2a01] = 0x7b30, [0x2a02] = 0x7b32,
  [0x2a03] = 0x7b34, [0x2a04] = 0x7b35, [0x2a05] = 0x7b36, [0x2a06] = 0x7b37,
  [0x2a07] = 0x7b39, [0x2a08] = 0x7b3b, [0x2a09] = 0x7b3d, [0x2a0a] = 0x7b3f,
  [0x2a0b] = 0x7b40, [0x2a0c] = 0x7b41, [0x2a0d] = 0x7b42, [0x2a0e] = 0x7b43,
  [0x2a0f] = 0x7b44, [0x2a10] = 0x7b46, [0x2a11] = 0x7b48, [0x2a12] = 0x7b4a,
  [0x2a13] = 0x7b4d, [0x2a14] = 0x7b4e, [0x2a15] = 0x7b53, [0x2a16] = 0x7b55,
  [0x2a17] = 0x7b57, [0x2a18] = 0x7b59, [0x2a19] = 0x7b5c, [0x2a1a] = 0x7b5e,
  [0x2a1b] = 0x7b5f, [0x2a1c] = 0x7b61, [0x2a1d] = 0x7b63, [0x2a1e] = 0x7b64,
  [0x2a1f] = 0x7b65, [0x2a20] = 0x7b66, [0x2a21] = 0x7b67, [0x2a22] = 0x7b68,
  [0x2a23] = 0x7b69, [0x2a24] = 0x7b6a, [0x2a25] = 0x7b6b, [0x2a26] = 0x7b6c,
  [0x2a27] = 0x7b6d, [0x2a28] = 0x7b6f, [0x2a29] = 0x7b70, [0x2a2a] = 0x7b73,
  [0x2a2b] = 0x7b74, [0x2a2c] = 0x7b76, [0x2a2d] = 0x7b78, [0x2a2e] = 0x7b7a,
  [0x2a2f] = 0x7b7c, [0x2a30] = 0x7b7d, [0x2a31] = 0x7b7f, [0x2a32] = 0x7b81,
  [0x2a33] = 0x7b82, [0x2a34] = 0x7b83, [0x2a35] = 0x7b84, [0x2a36] = 0x7b86,
  [0x2a37] = 0x7b87, [0x2a38] = 0x7b88, [0x2a39] = 0x7b89, [0x2a3a] = 0x7b8a,
  [0x2a3b] = 0x7b8b, [0x2a3c] = 0x7b8c, [0x2a3d] = 0x7b8e, [0x2a3e] = 0x7b8f,
  [0x2a40] = 0x7b91, [0x2a41] = 0x7b92, [0x2a42] = 0x7b93, [0x2a43] = 0x7b96,
  [0x2a44] = 0x7b98, [0x2a45] = 0x7b99, [0x2a46] = 0x7b9a, [0x2a47] = 0x7b9b,
  [0x2a48] = 0x7b9e, [0x2a49] = 0x7b9f, [0x2a4a] = 0x7ba0, [0x2a4b] = 0x7ba3,
  [0x2a4c] = 0x7ba4, [0x2a4d] = 0x7ba5, [0x2a4e] = 0x7bae, [0x2a4f] = 0x7baf,
  [0x2a50] = 0x7bb0, [0x2a51] = 0x7bb2, [0x2a52] = 0x7bb3, [0x2a53] = 0x7bb5,
  [0x2a54] = 0x7bb6, [0x2a55] = 0x7bb7, [0x2a56] = 0x7bb9, [0x2a57] = 0x7bba,
  [0x2a58] = 0x7bbb, [0x2a59] = 0x7bbc, [0x2a5a] = 0x7bbd, [0x2a5b] = 0x7bbe,
  [0x2a5c] = 0x7bbf, [0x2a5d] = 0x7bc0, [0x2a5e] = 0x7bc2, [0x2a5f] = 0x7bc3,
  [0x2a60] = 0x7bc4, [0x2a61] = 0x57c2, [0x2a62] = 0x803f, [0x2a63] = 0x6897,
  [0x2a64] = 0x5de5, [0x2a65] = 0x653b, [0x2a66] = 0x529f, [0x2a67] = 0x606d,
  [0x2a68] = 0x9f9a, [0x2a69] = 0x4f9b, [0x2a6a] = 0x8eac, [0x2a6b] = 0x516c,
  [0x2a6c] = 0x5bab, [0x2a6d] = 0x5f13, [0x2a6e] = 0x5de9, [0x2a6f] = 0x6c5e,
  [0x2a70] = 0x62f1, [0x2a71] = 0x8d21, [0x2a72] = 0x5171, [0x2a73] = 0x94a9,
  [0x2a74] = 0x52fe, [0x2a75] = 0x6c9f, [0x2a76] = 0x82df, [0x2a77] = 0x72d7,
  [0x2a78] = 0x57a2, [0x2a79] = 0x6784, [0x2a7a] = 0x8d2d, [0x2a7b] = 0x591f,
  [0x2a7c] = 0x8f9c, [0x2a7d] = 0x83c7, [0x2a7e] = 0x5495, [0x2a7f] = 0x7b8d,
  [0x2a80] = 0x4f30, [0x2a81] = 0x6cbd, [0x2a82] = 0x5b64, [0x2a83] = 0x59d1,
  [0x2a84] = 0x9f13, [0x2a85] = 0x53e4, [0x2a86] = 0x86ca, [0x2a87] = 0x9aa8,
  [0x2a88] = 0x8c37, [0x2a89] = 0x80a1, [0x2a8a] = 0x6545, [0x2a8b] = 0x987e,
  [0x2a8c] = 0x56fa, [0x2a8d] = 0x96c7, [0x2a8e] = 0x522e, [0x2a8f] = 0x74dc,
  [0x2a90] = 0x5250, [0x2a91] = 0x5be1, [0x2a92] = 0x6302, [0x2a93] = 0x8902,
  [0x2a94] = 0x4e56, [0x2a95] = 0x62d0, [0x2a96] = 0x602a, [0x2a97] = 0x68fa,
  [0x2a98] = 0x5173, [0x2a99] = 0x5b98, [0x2a9a] = 0x51a0, [0x2a9b] = 0x89c2,
  [0x2a9c] = 0x7ba1, [0x2a9d] = 0x9986, [0x2a9e] = 0x7f50, [0x2a9f] = 0x60ef,
  [0x2aa0] = 0x704c, [0x2aa1] = 0x8d2f, [0x2aa2] = 0x5149, [0x2aa3] = 0x5e7f,
  [0x2aa4] = 0x901b, [0x2aa5] = 0x7470, [0x2aa6] = 0x89c4, [0x2aa7] = 0x572d,
  [0x2aa8] = 0x7845, [0x2aa9] = 0x5f52, [0x2aaa] = 0x9f9f, [0x2aab] = 0x95fa,
  [0x2aac] = 0x8f68, [0x2aad] = 0x9b3c, [0x2aae] = 0x8be1, [0x2aaf] = 0x7678,
  [0x2ab0] = 0x6842, [0x2ab1] = 0x67dc, [0x2ab2] = 0x8dea, [0x2ab3] = 0x8d35,
  [0x2ab4] = 0x523d, [0x2ab5] = 0x8f8a, [0x2ab6] = 0x6eda, [0x2ab7] = 0x68cd,
  [0x2ab8] = 0x9505, [0x2ab9] = 0x90ed, [0x2aba] = 0x56fd, [0x2abb] = 0x679c,
  [0x2abc] = 0x88f9, [0x2abd] = 0x8fc7, [0x2abe] = 0x54c8, [0x2ac0] = 0x7bc5,
  [0x2ac1] = 0x7bc8, [0x2ac2] = 0x7bc9, [0x2ac3] = 0x7bca, [0x2ac4] = 0x7bcb,
  [0x2ac5] = 0x7bcd, [0x2ac6] = 0x7bce, [0x2ac7] = 0x7bcf, [0x2ac8] = 0x7bd0,
  [0x2ac9] = 0x7bd2, [0x2aca] = 0x7bd4, [0x2acb] = 0x7bd5, [0x2acc] = 0x7bd6,
  [0x2acd] = 0x7bd7, [0x2ace] = 0x7bd8, [0x2acf] = 0x7bdb, [0x2ad0] = 0x7bdc,
  [0x2ad1] = 0x7bde, [0x2ad2] = 0x7bdf, [0x2ad3] = 0x7be0, [0x2ad4] = 0x7be2,
  [0x2ad5] = 0x7be3, [0x2ad6] = 0x7be4, [0x2ad7] = 0x7be7, [0x2ad8] = 0x7be8,
  [0x2ad9] = 0x7be9, [0x2ada] = 0x7beb, [0x2adb] = 0x7bec, [0x2adc] = 0x7bed,
  [0x2add] = 0x7bef, [0x2ade] = 0x7bf0, [0x2adf] = 0x7bf2, [0x2ae0] = 0x7bf3,
  [0x2ae1] = 0x7bf4, [0x2ae2] = 0x7bf5, [0x2ae3] = 0x7bf6, [0x2ae4] = 0x7bf8,
  [0x2ae5] = 0x7bf9, [0x2ae6] = 0x7bfa, [0x2ae7] = 0x7bfb, [0x2ae8] = 0x7bfd,
  [0x2ae9] = 0x7bff, [0x2aea] = 0x7c00, [0x2aeb] = 0x7c01, [0x2aec] = 0x7c02,
  [0x2aed] = 0x7c03, [0x2aee] = 0x7c04, [0x2aef] = 0x7c05, [0x2af0] = 0x7c06,
  [0x2af1] = 0x7c08, [0x2af2] = 0x7c09, [0x2af3] = 0x7c0a, [0x2af4] = 0x7c0d,
  [0x2af5] = 0x7c0e, [0x2af6] = 0x7c10, [0x2af7] = 0x7c11, [0x2af8] = 0x7c12,
  [0x2af9] = 0x7c13, [0x2afa] = 0x7c14, [0x2afb] = 0x7c15, [0x2afc] = 0x7c17,
  [0x2afd] = 0x7c18, [0x2afe] = 0x7c19, [0x2b00] = 0x7c1a, [0x2b01] = 0x7c1b,
  [0x2b02] = 0x7c1c, [0x2b03] = 0x7c1d, [0x2b04] = 0x7c1e, [0x2b05] = 0x7c20,
  [0x2b06] = 0x7c21, [0x2b07] = 0x7c22, [0x2b08] = 0x7c23, [0x2b09] = 0x7c24,
  [0x2b0a] = 0x7c25, [0x2b0b] = 0x7c28, [0x2b0c] = 0x7c29, [0x2b0d] = 0x7c2b,
  [0x2b0e] = 0x7c2c, [0x2b0f] = 0x7c2d, [0x2b10] = 0x7c2e, [0x2b11] = 0x7c2f,
  [0x2b12] = 0x7c30, [0x2b13] = 0x7c31, [0x2b14] = 0x7c32, [0x2b15] = 0x7c33,
  [0x2b16] = 0x7c34, [0x2b17] = 0x7c35, [0x2b18] = 0x7c36, [0x2b19] = 0x7c37,
  [0x2b1a] = 0x7c39, [0x2b1b] = 0x7c3a, [0x2b1c] = 0x7c3b, [0x2b1d] = 0x7c3c,
  [0x2b1e] = 0x7c3d, [0x2b1f] = 0x7c3e, [0x2b20] = 0x7c42, [0x2b21] = 0x9ab8,
  [0x2b22] = 0x5b69, [0x2b23] = 0x6d77, [0x2b24] = 0x6c26, [0x2b25] = 0x4ea5,
  [0x2b26] = 0x5bb3, [0x2b27] = 0x9a87, [0x2b28] = 0x9163, [0x2b29] = 0x61a8,
  [0x2b2a] = 0x90af, [0x2b2b] = 0x97e9, [0x2b2c] = 0x542b, [0x2b2d] = 0x6db5,
  [0x2b2e] = 0x5bd2, [0x2b2f] = 0x51fd, [0x2b30] = 0x558a, [0x2b31] = 0x7f55,
  [0x2b32] = 0x7ff0, [0x2b33] = 0x64bc, [0x2b34] = 0x634d, [0x2b35] = 0x65f1,
  [0x2b36] = 0x61be, [0x2b37] = 0x608d, [0x2b38] = 0x710a, [0x2b39] = 0x6c57,
  [0x2b3a] = 0x6c49, [0x2b3b] = 0x592f, [0x2b3c] = 0x676d, [0x2b3d] = 0x822a,
  [0x2b3e] = 0x58d5, [0x2b3f] = 0x568e, [0x2b40] = 0x8c6a, [0x2b41] = 0x6beb,
  [0x2b42] = 0x90dd, [0x2b43] = 0x597d, [0x2b44] = 0x8017, [0x2b45] = 0x53f7,
  [0x2b46] = 0x6d69, [0x2b47] = 0x5475, [0x2b48] = 0x559d, [0x2b49] = 0x8377,
  [0x2b4a] = 0x83cf, [0x2b4b] = 0x6838, [0x2b4c] = 0x79be, [0x2b4d] = 0x548c,
  [0x2b4e] = 0x4f55, [0x2b4f] = 0x5408, [0x2b50] = 0x76d2, [0x2b51] = 0x8c89,
  [0x2b52] = 0x9602, [0x2b53] = 0x6cb3, [0x2b54] = 0x6db8, [0x2b55] = 0x8d6b,
  [0x2b56] = 0x8910, [0x2b57] = 0x9e64, [0x2b58] = 0x8d3a, [0x2b59] = 0x563f,
  [0x2b5a] = 0x9ed1, [0x2b5b] = 0x75d5, [0x2b5c] = 0x5f88, [0x2b5d] = 0x72e0,
  [0x2b5e] = 0x6068, [0x2b5f] = 0x54fc, [0x2b60] = 0x4ea8, [0x2b61] = 0x6a2a,
  [0x2b62] = 0x8861, [0x2b63] = 0x6052, [0x2b64] = 0x8f70, [0x2b65] = 0x54c4,
  [0x2b66] = 0x70d8, [0x2b67] = 0x8679, [0x2b68] = 0x9e3f, [0x2b69] = 0x6d2a,
  [0x2b6a] = 0x5b8f, [0x2b6b] = 0x5f18, [0x2b6c] = 0x7ea2, [0x2b6d] = 0x5589,
  [0x2b6e] = 0x4faf, [0x2b6f] = 0x7334, [0x2b70] = 0x543c, [0x2b71] = 0x539a,
  [0x2b72] = 0x5019, [0x2b73] = 0x540e, [0x2b74] = 0x547c, [0x2b75] = 0x4e4e,
  [0x2b76] = 0x5ffd, [0x2b77] = 0x745a, [0x2b78] = 0x58f6, [0x2b79] = 0x846b,
  [0x2b7a] = 0x80e1, [0x2b7b] = 0x8774, [0x2b7c] = 0x72d0, [0x2b7d] = 0x7cca,
  [0x2b7e] = 0x6e56, [0x2b80] = 0x7c43, [0x2b81] = 0x7c44, [0x2b82] = 0x7c45,
  [0x2b83] = 0x7c46, [0x2b84] = 0x7c47, [0x2b85] = 0x7c48, [0x2b86] = 0x7c49,
  [0x2b87] = 0x7c4a, [0x2b88] = 0x7c4b, [0x2b89] = 0x7c4c, [0x2b8a] = 0x7c4e,
  [0x2b8b] = 0x7c4f, [0x2b8c] = 0x7c50, [0x2b8d] = 0x7c51, [0x2b8e] = 0x7c52,
  [0x2b8f] = 0x7c53, [0x2b90] = 0x7c54, [0x2b91] = 0x7c55, [0x2b92] = 0x7c56,
  [0x2b93] = 0x7c57, [0x2b94] = 0x7c58, [0x2b95] = 0x7c59, [0x2b96] = 0x7c5a,
  [0x2b97] = 0x7c5b, [0x2b98] = 0x7c5c, [0x2b99] = 0x7c5d, [0x2b9a] = 0x7c5e,
  [0x2b9b] = 0x7c5f, [0x2b9c] = 0x7c60, [0x2b9d] = 0x7c61, [0x2b9e] = 0x7c62,
  [0x2b9f] = 0x7c63, [0x2ba0] = 0x7c64, [0x2ba1] = 0x7c65, [0x2ba2] = 0x7c66,
  [0x2ba3] = 0x7c67, [0x2ba4] = 0x7c68, [0x2ba5] = 0x7c69, [0x2ba6] = 0x7c6a,
  [0x2ba7] = 0x7c6b, [0x2ba8] = 0x7c6c, [0x2ba9] = 0x7c6d, [0x2baa] = 0x7c6e,
  [0x2bab] = 0x7c6f, [0x2bac] = 0x7c70, [0x2bad] = 0x7c71, [0x2bae] = 0x7c72,
  [0x2baf] = 0x7c75, [0x2bb0] = 0x7c76, [0x2bb1] = 0x7c77, [0x2bb2] = 0x7c78,
  [0x2bb3] = 0x7c79, [0x2bb4] = 0x7c7a, [0x2bb5] = 0x7c7e, [0x2bb6] = 0x7c7f,
  [0x2bb7] = 0x7c80, [0x2bb8] = 0x7c81, [0x2bb9] = 0x7c82, [0x2bba] = 0x7c83,
  [0x2bbb] = 0x7c84, [0x2bbc] = 0x7c85, [0x2bbd] = 0x7c86, [0x2bbe] = 0x7c87,
  [0x2bc0] = 0x7c88, [0x2bc1] = 0x7c8a, [0x2bc2] = 0x7c8b, [0x2bc3] = 0x7c8c,
  [0x2bc4] = 0x7c8d, [0x2bc5] = 0x7c8e, [0x2bc6] = 0x7c8f, [0x2bc7] = 0x7c90,
  [0x2bc8] = 0x7c93, [0x2bc9] = 0x7c94, [0x2bca] = 0x7c96, [0x2bcb] = 0x7c99,
  [0x2bcc] = 0x7c9a, [0x2bcd] = 0x7c9b, [0x2bce] = 0x7ca0, [0x2bcf] = 0x7ca1,
  [0x2bd0] = 0x7ca3, [0x2bd1] = 0x7ca6, [0x2bd2] = 0x7ca7, [0x2bd3] = 0x7ca8,
  [0x2bd4] = 0x7ca9, [0x2bd5] = 0x7cab, [0x2bd6] = 0x7cac, [0x2bd7] = 0x7cad,
  [0x2bd8] = 0x7caf, [0x2bd9] = 0x7cb0, [0x2bda] = 0x7cb4, [0x2bdb] = 0x7cb5,
  [0x2bdc] = 0x7cb6, [0x2bdd] = 0x7cb7, [0x2bde] = 0x7cb8, [0x2bdf] = 0x7cba,
  [0x2be0] = 0x7cbb, [0x2be1] = 0x5f27, [0x2be2] = 0x864e, [0x2be3] = 0x552c,
  [0x2be4] = 0x62a4, [0x2be5] = 0x4e92, [0x2be6] = 0x6caa, [0x2be7] = 0x6237,
  [0x2be8] = 0x82b1, [0x2be9] = 0x54d7, [0x2bea] = 0x534e, [0x2beb] = 0x733e,
  [0x2bec] = 0x6ed1, [0x2bed] = 0x753b, [0x2bee] = 0x5212, [0x2bef] = 0x5316,
  [0x2bf0] = 0x8bdd, [0x2bf1] = 0x69d0, [0x2bf2] = 0x5f8a, [0x2bf3] = 0x6000,
  [0x2bf4] = 0x6dee, [0x2bf5] = 0x574f, [0x2bf6] = 0x6b22, [0x2bf7] = 0x73af,
  [0x2bf8] = 0x6853, [0x2bf9] = 0x8fd8, [0x2bfa] = 0x7f13, [0x2bfb] = 0x6362,
  [0x2bfc] = 0x60a3, [0x2bfd] = 0x5524, [0x2bfe] = 0x75ea, [0x2bff] = 0x8c62,
  [0x2c00] = 0x7115, [0x2c01] = 0x6da3, [0x2c02] = 0x5ba6, [0x2c03] = 0x5e7b,
  [0x2c04] = 0x8352, [0x2c05] = 0x614c, [0x2c06] = 0x9ec4, [0x2c07] = 0x78fa,
  [0x2c08] = 0x8757, [0x2c09] = 0x7c27, [0x2c0a] = 0x7687, [0x2c0b] = 0x51f0,
  [0x2c0c] = 0x60f6, [0x2c0d] = 0x714c, [0x2c0e] = 0x6643, [0x2c0f] = 0x5e4c,
  [0x2c10] = 0x604d, [0x2c11] = 0x8c0e, [0x2c12] = 0x7070, [0x2c13] = 0x6325,
  [0x2c14] = 0x8f89, [0x2c15] = 0x5fbd, [0x2c16] = 0x6062, [0x2c17] = 0x86d4,
  [0x2c18] = 0x56de, [0x2c19] = 0x6bc1, [0x2c1a] = 0x6094, [0x2c1b] = 0x6167,
  [0x2c1c] = 0x5349, [0x2c1d] = 0x60e0, [0x2c1e] = 0x6666, [0x2c1f] = 0x8d3f,
  [0x2c20] = 0x79fd, [0x2c21] = 0x4f1a, [0x2c22] = 0x70e9, [0x2c23] = 0x6c47,
  [0x2c24] = 0x8bb3, [0x2c25] = 0x8bf2, [0x2c26] = 0x7ed8, [0x2c27] = 0x8364,
  [0x2c28] = 0x660f, [0x2c29] = 0x5a5a, [0x2c2a] = 0x9b42, [0x2c2b] = 0x6d51,
  [0x2c2c] = 0x6df7, [0x2c2d] = 0x8c41, [0x2c2e] = 0x6d3b, [0x2c2f] = 0x4f19,
  [0x2c30] = 0x706b, [0x2c31] = 0x83b7, [0x2c32] = 0x6216, [0x2c33] = 0x60d1,
  [0x2c34] = 0x970d, [0x2c35] = 0x8d27, [0x2c36] = 0x7978, [0x2c37] = 0x51fb,
  [0x2c38] = 0x573e, [0x2c39] = 0x57fa, [0x2c3a] = 0x673a, [0x2c3b] = 0x7578,
  [0x2c3c] = 0x7a3d, [0x2c3d] = 0x79ef, [0x2c3e] = 0x7b95, [0x2c40] = 0x7cbf,
  [0x2c41] = 0x7cc0, [0x2c42] = 0x7cc2, [0x2c43] = 0x7cc3, [0x2c44] = 0x7cc4,
  [0x2c45] = 0x7cc6, [0x2c46] = 0x7cc9, [0x2c47] = 0x7ccb, [0x2c48] = 0x7cce,
  [0x2c49] = 0x7ccf, [0x2c4a] = 0x7cd0, [0x2c4b] = 0x7cd1, [0x2c4c] = 0x7cd2,
  [0x2c4d] = 0x7cd3, [0x2c4e] = 0x7cd4, [0x2c4f] = 0x7cd8, [0x2c50] = 0x7cda,
  [0x2c51] = 0x7cdb, [0x2c52] = 0x7cdd, [0x2c53] = 0x7cde, [0x2c54] = 0x7ce1,
  [0x2c55] = 0x7ce2, [0x2c56] = 0x7ce3, [0x2c57] = 0x7ce4, [0x2c58] = 0x7ce5,
  [0x2c59] = 0x7ce6, [0x2c5a] = 0x7ce7, [0x2c5b] = 0x7ce9, [0x2c5c] = 0x7cea,
  [0x2c5d] = 0x7ceb, [0x2c5e] = 0x7cec, [0x2c5f] = 0x7ced, [0x2c60] = 0x7cee,
  [0x2c61] = 0x7cf0, [0x2c62] = 0x7cf1, [0x2c63] = 0x7cf2, [0x2c64] = 0x7cf3,
  [0x2c65] = 0x7cf4, [0x2c66] = 0x7cf5, [0x2c67] = 0x7cf6, [0x2c68] = 0x7cf7,
  [0x2c69] = 0x7cf9, [0x2c6a] = 0x7cfa, [0x2c6b] = 0x7cfc, [0x2c6c] = 0x7cfd,
  [0x2c6d] = 0x7cfe, [0x2c6e] = 0x7cff, [0x2c6f] = 0x7d00, [0x2c70] = 0x7d01,
  [0x2c71] = 0x7d02, [0x2c72] = 0x7d03, [0x2c73] = 0x7d04, [0x2c74] = 0x7d05,
  [0x2c75] = 0x7d06, [0x2c76] = 0x7d07, [0x2c77] = 0x7d08, [0x2c78] = 0x7d09,
  [0x2c79] = 0x7d0b, [0x2c7a] = 0x7d0c, [0x2c7b] = 0x7d0d, [0x2c7c] = 0x7d0e,
  [0x2c7d] = 0x7d0f, [0x2c7e] = 0x7d10, [0x2c80] = 0x7d11, [0x2c81] = 0x7d12,
  [0x2c82] = 0x7d13, [0x2c83] = 0x7d14, [0x2c84] = 0x7d15, [0x2c85] = 0x7d16,
  [0x2c86] = 0x7d17, [0x2c87] = 0x7d18, [0x2c88] = 0x7d19, [0x2c89] = 0x7d1a,
  [0x2c8a] = 0x7d1b, [0x2c8b] = 0x7d1c, [0x2c8c] = 0x7d1d, [0x2c8d] = 0x7d1e,
  [0x2c8e] = 0x7d1f, [0x2c8f] = 0x7d21, [0x2c90] = 0x7d23, [0x2c91] = 0x7d24,
  [0x2c92] = 0x7d25, [0x2c93] = 0x7d26, [0x2c94] = 0x7d28, [0x2c95] = 0x7d29,
  [0x2c96] = 0x7d2a, [0x2c97] = 0x7d2c, [0x2c98] = 0x7d2d, [0x2c99] = 0x7d2e,
  [0x2c9a] = 0x7d30, [0x2c9b] = 0x7d31, [0x2c9c] = 0x7d32, [0x2c9d] = 0x7d33,
  [0x2c9e] = 0x7d34, [0x2c9f] = 0x7d35, [0x2ca0] = 0x7d36, [0x2ca1] = 0x808c,
  [0x2ca2] = 0x9965, [0x2ca3] = 0x8ff9, [0x2ca4] = 0x6fc0, [0x2ca5] = 0x8ba5,
  [0x2ca6] = 0x9e21, [0x2ca7] = 0x59ec, [0x2ca8] = 0x7ee9, [0x2ca9] = 0x7f09,
  [0x2caa] = 0x5409, [0x2cab] = 0x6781, [0x2cac] = 0x68d8, [0x2cad] = 0x8f91,
  [0x2cae] = 0x7c4d, [0x2caf] = 0x96c6, [0x2cb0] = 0x53ca, [0x2cb1] = 0x6025,
  [0x2cb2] = 0x75be, [0x2cb3] = 0x6c72, [0x2cb4] = 0x5373, [0x2cb5] = 0x5ac9,
  [0x2cb6] = 0x7ea7, [0x2cb7] = 0x6324, [0x2cb8] = 0x51e0, [0x2cb9] = 0x810a,
  [0x2cba] = 0x5df1, [0x2cbb] = 0x84df, [0x2cbc] = 0x6280, [0x2cbd] = 0x5180,
  [0x2cbe] = 0x5b63, [0x2cbf] = 0x4f0e, [0x2cc0] = 0x796d, [0x2cc1] = 0x5242,
  [0x2cc2] = 0x60b8, [0x2cc3] = 0x6d4e, [0x2cc4] = 0x5bc4, [0x2cc5] = 0x5bc2,
  [0x2cc6] = 0x8ba1, [0x2cc7] = 0x8bb0, [0x2cc8] = 0x65e2, [0x2cc9] = 0x5fcc,
  [0x2cca] = 0x9645, [0x2ccb] = 0x5993, [0x2ccc] = 0x7ee7, [0x2ccd] = 0x7eaa,
  [0x2cce] = 0x5609, [0x2ccf] = 0x67b7, [0x2cd0] = 0x5939, [0x2cd1] = 0x4f73,
  [0x2cd2] = 0x5bb6, [0x2cd3] = 0x52a0, [0x2cd4] = 0x835a, [0x2cd5] = 0x988a,
  [0x2cd6] = 0x8d3e, [0x2cd7] = 0x7532, [0x2cd8] = 0x94be, [0x2cd9] = 0x5047,
  [0x2cda] = 0x7a3c, [0x2cdb] = 0x4ef7, [0x2cdc] = 0x67b6, [0x2cdd] = 0x9a7e,
  [0x2cde] = 0x5ac1, [0x2cdf] = 0x6b7c, [0x2ce0] = 0x76d1, [0x2ce1] = 0x575a,
  [0x2ce2] = 0x5c16, [0x2ce3] = 0x7b3a, [0x2ce4] = 0x95f4, [0x2ce5] = 0x714e,
  [0x2ce6] = 0x517c, [0x2ce7] = 0x80a9, [0x2ce8] = 0x8270, [0x2ce9] = 0x5978,
  [0x2cea] = 0x7f04, [0x2ceb] = 0x8327, [0x2cec] = 0x68c0, [0x2ced] = 0x67ec,
  [0x2cee] = 0x78b1, [0x2cef] = 0x7877, [0x2cf0] = 0x62e3, [0x2cf1] = 0x6361,
  [0x2cf2] = 0x7b80, [0x2cf3] = 0x4fed, [0x2cf4] = 0x526a, [0x2cf5] = 0x51cf,
  [0x2cf6] = 0x8350, [0x2cf7] = 0x69db, [0x2cf8] = 0x9274, [0x2cf9] = 0x8df5,
  [0x2cfa] = 0x8d31, [0x2cfb] = 0x89c1, [0x2cfc] = 0x952e, [0x2cfd] = 0x7bad,
  [0x2cfe] = 0x4ef6, [0x2d00] = 0x7d37, [0x2d01] = 0x7d38, [0x2d02] = 0x7d39,
  [0x2d03] = 0x7d3a, [0x2d04] = 0x7d3b, [0x2d05] = 0x7d3c, [0x2d06] = 0x7d3d,
  [0x2d07] = 0x7d3e, [0x2d08] = 0x7d3f, [0x2d09] = 0x7d40, [0x2d0a] = 0x7d41,
  [0x2d0b] = 0x7d42, [0x2d0c] = 0x7d43, [0x2d0d] = 0x7d44, [0x2d0e] = 0x7d45,
  [0x2d0f] = 0x7d46, [0x2d10] = 0x7d47, [0x2d11] = 0x7d48, [0x2d12] = 0x7d49,
  [0x2d13] = 0x7d4a, [0x2d14] = 0x7d4b, [0x2d15] = 0x7d4c, [0x2d16] = 0x7d4d,
  [0x2d17] = 0x7d4e, [0x2d18] = 0x7d4f, [0x2d19] = 0x7d50, [0x2d1a] = 0x7d51,
  [0x2d1b] = 0x7d52, [0x2d1c] = 0x7d53, [0x2d1d] = 0x7d54, [0x2d1e] = 0x7d55,
  [0x2d1f] = 0x7d56, [0x2d20] = 0x7d57, [0x2d21] = 0x7d58, [0x2d22] = 0x7d59,
  [0x2d23] = 0x7d5a, [0x2d24] = 0x7d5b, [0x2d25] = 0x7d5c, [0x2d26] = 0x7d5d,
  [0x2d27] = 0x7d5e, [0x2d28] = 0x7d5f, [0x2d29] = 0x7d60, [0x2d2a] = 0x7d61,
  [0x2d2b] = 0x7d62, [0x2d2c] = 0x7d63, [0x2d2d] = 0x7d64, [0x2d2e] = 0x7d65,
  [0x2d2f] = 0x7d66, [0x2d30] = 0x7d67, [0x2d31] = 0x7d68, [0x2d32] = 0x7d69,
  [0x2d33] = 0x7d6a, [0x2d34] = 0x7d6b, [0x2d35] = 0x7d6c, [0x2d36] = 0x7d6d,
  [0x2d37] = 0x7d6f, [0x2d38] = 0x7d70, [0x2d39] = 0x7d71, [0x2d3a] = 0x7d72,
  [0x2d3b] = 0x7d73, [0x2d3c] = 0x7d74, [0x2d3d] = 0x7d75, [0x2d3e] = 0x7d76,
  [0x2d40] = 0x7d78, [0x2d41] = 0x7d79, [0x2d42] = 0x7d7a, [0x2d43] = 0x7d7b,
  [0x2d44] = 0x7d7c, [0x2d45] = 0x7d7d, [0x2d46] = 0x7d7e, [0x2d47] = 0x7d7f,
  [0x2d48] = 0x7d80, [0x2d49] = 0x7d81, [0x2d4a] = 0x7d82, [0x2d4b] = 0x7d83,
  [0x2d4c] = 0x7d84, [0x2d4d] = 0x7d85, [0x2d4e] = 0x7d86, [0x2d4f] = 0x7d87,
  [0x2d50] = 0x7d88, [0x2d51] = 0x7d89, [0x2d52] = 0x7d8a, [0x2d53] = 0x7d8b,
  [0x2d54] = 0x7d8c, [0x2d55] = 0x7d8d, [0x2d56] = 0x7d8e, [0x2d57] = 0x7d8f,
  [0x2d58] = 0x7d90, [0x2d59] = 0x7d91, [0x2d5a] = 0x7d92, [0x2d5b] = 0x7d93,
  [0x2d5c] = 0x7d94, [0x2d5d] = 0x7d95, [0x2d5e] = 0x7d96, [0x2d5f] = 0x7d97,
  [0x2d60] = 0x7d98, [0x2d61] = 0x5065, [0x2d62] = 0x8230, [0x2d63] = 0x5251,
  [0x2d64] = 0x996f, [0x2d65] = 0x6e10, [0x2d66] = 0x6e85, [0x2d67] = 0x6da7,
  [0x2d68] = 0x5efa, [0x2d69] = 0x50f5, [0x2d6a] = 0x59dc, [0x2d6b] = 0x5c06,
  [0x2d6c] = 0x6d46, [0x2d6d] = 0x6c5f, [0x2d6e] = 0x7586, [0x2d6f] = 0x848b,
  [0x2d70] = 0x6868, [0x2d71] = 0x5956, [0x2d72] = 0x8bb2, [0x2d73] = 0x5320,
  [0x2d74] = 0x9171, [0x2d75] = 0x964d, [0x2d76] = 0x8549, [0x2d77] = 0x6912,
  [0x2d78] = 0x7901, [0x2d79] = 0x7126, [0x2d7a] = 0x80f6, [0x2d7b] = 0x4ea4,
  [0x2d7c] = 0x90ca, [0x2d7d] = 0x6d47, [0x2d7e] = 0x9a84, [0x2d7f] = 0x5a07,
  [0x2d80] = 0x56bc, [0x2d81] = 0x6405, [0x2d82] = 0x94f0, [0x2d83] = 0x77eb,
  [0x2d84] = 0x4fa5, [0x2d85] = 0x811a, [0x2d86] = 0x72e1, [0x2d87] = 0x89d2,
  [0x2d88] = 0x997a, [0x2d89] = 0x7f34, [0x2d8a] = 0x7ede, [0x2d8b] = 0x527f,
  [0x2d8c] = 0x6559, [0x2d8d] = 0x9175, [0x2d8e] = 0x8f7f, [0x2d8f] = 0x8f83,
  [0x2d90] = 0x53eb, [0x2d91] = 0x7a96, [0x2d92] = 0x63ed, [0x2d93] = 0x63a5,
  [0x2d94] = 0x7686, [0x2d95] = 0x79f8, [0x2d96] = 0x8857, [0x2d97] = 0x9636,
  [0x2d98] = 0x622a, [0x2d99] = 0x52ab, [0x2d9a] = 0x8282, [0x2d9b] = 0x6854,
  [0x2d9c] = 0x6770, [0x2d9d] = 0x6377, [0x2d9e] = 0x776b, [0x2d9f] = 0x7aed,
  [0x2da0] = 0x6d01, [0x2da1] = 0x7ed3, [0x2da2] = 0x89e3, [0x2da3] = 0x59d0,
  [0x2da4] = 0x6212, [0x2da5] = 0x85c9, [0x2da6] = 0x82a5, [0x2da7] = 0x754c,
  [0x2da8] = 0x501f, [0x2da9] = 0x4ecb, [0x2daa] = 0x75a5, [0x2dab] = 0x8beb,
  [0x2dac] = 0x5c4a, [0x2dad] = 0x5dfe, [0x2dae] = 0x7b4b, [0x2daf] = 0x65a4,
  [0x2db0] = 0x91d1, [0x2db1] = 0x4eca, [0x2db2] = 0x6d25, [0x2db3] = 0x895f,
  [0x2db4] = 0x7d27, [0x2db5] = 0x9526, [0x2db6] = 0x4ec5, [0x2db7] = 0x8c28,
  [0x2db8] = 0x8fdb, [0x2db9] = 0x9773, [0x2dba] = 0x664b, [0x2dbb] = 0x7981,
  [0x2dbc] = 0x8fd1, [0x2dbd] = 0x70ec, [0x2dbe] = 0x6d78, [0x2dc0] = 0x7d99,
  [0x2dc1] = 0x7d9a, [0x2dc2] = 0x7d9b, [0x2dc3] = 0x7d9c, [0x2dc4] = 0x7d9d,
  [0x2dc5] = 0x7d9e, [0x2dc6] = 0x7d9f, [0x2dc7] = 0x7da0, [0x2dc8] = 0x7da1,
  [0x2dc9] = 0x7da2, [0x2dca] = 0x7da3, [0x2dcb] = 0x7da4, [0x2dcc] = 0x7da5,
  [0x2dcd] = 0x7da7, [0x2dce] = 0x7da8, [0x2dcf] = 0x7da9, [0x2dd0] = 0x7daa,
  [0x2dd1] = 0x7dab, [0x2dd2] = 0x7dac, [0x2dd3] = 0x7dad, [0x2dd4] = 0x7daf,
  [0x2dd5] = 0x7db0, [0x2dd6] = 0x7db1, [0x2dd7] = 0x7db2, [0x2dd8] = 0x7db3,
  [0x2dd9] = 0x7db4, [0x2dda] = 0x7db5, [0x2ddb] = 0x7db6, [0x2ddc] = 0x7db7,
  [0x2ddd] = 0x7db8, [0x2dde] = 0x7db9, [0x2ddf] = 0x7dba, [0x2de0] = 0x7dbb,
  [0x2de1] = 0x7dbc, [0x2de2] = 0x7dbd, [0x2de3] = 0x7dbe, [0x2de4] = 0x7dbf,
  [0x2de5] = 0x7dc0, [0x2de6] = 0x7dc1, [0x2de7] = 0x7dc2, [0x2de8] = 0x7dc3,
  [0x2de9] = 0x7dc4, [0x2dea] = 0x7dc5, [0x2deb] = 0x7dc6, [0x2dec] = 0x7dc7,
  [0x2ded] = 0x7dc8, [0x2dee] = 0x7dc9, [0x2def] = 0x7dca, [0x2df0] = 0x7dcb,
  [0x2df1] = 0x7dcc, [0x2df2] = 0x7dcd, [0x2df3] = 0x7dce, [0x2df4] = 0x7dcf,
  [0x2df5] = 0x7dd0, [0x2df6] = 0x7dd1, [0x2df7] = 0x7dd2, [0x2df8] = 0x7dd3,
  [0x2df9] = 0x7dd4, [0x2dfa] = 0x7dd5, [0x2dfb] = 0x7dd6, [0x2dfc] = 0x7dd7,
  [0x2dfd] = 0x7dd8, [0x2dfe] = 0x7dd9, [0x2e00] = 0x7dda, [0x2e01] = 0x7ddb,
  [0x2e02] = 0x7ddc, [0x2e03] = 0x7ddd, [0x2e04] = 0x7dde, [0x2e05] = 0x7ddf,
  [0x2e06] = 0x7de0, [0x2e07] = 0x7de1, [0x2e08] = 0x7de2, [0x2e09] = 0x7de3,
  [0x2e0a] = 0x7de4, [0x2e0b] = 0x7de5, [0x2e0c] = 0x7de6, [0x2e0d] = 0x7de7,
  [0x2e0e] = 0x7de8, [0x2e0f] = 0x7de9, [0x2e10] = 0x7dea, [0x2e11] = 0x7deb,
  [0x2e12] = 0x7dec, [0x2e13] = 0x7ded, [0x2e14] = 0x7dee, [0x2e15] = 0x7def,
  [0x2e16] = 0x7df0, [0x2e17] = 0x7df1, [0x2e18] = 0x7df2, [0x2e19] = 0x7df3,
  [0x2e1a] = 0x7df4, [0x2e1b] = 0x7df5, [0x2e1c] = 0x7df6, [0x2e1d] = 0x7df7,
  [0x2e1e] = 0x7df8, [0x2e1f] = 0x7df9, [0x2e20] = 0x7dfa, [0x2e21] = 0x5c3d,
  [0x2e22] = 0x52b2, [0x2e23] = 0x8346, [0x2e24] = 0x5162, [0x2e25] = 0x830e,
  [0x2e26] = 0x775b, [0x2e27] = 0x6676, [0x2e28] = 0x9cb8, [0x2e29] = 0x4eac,
  [0x2e2a] = 0x60ca, [0x2e2b] = 0x7cbe, [0x2e2c] = 0x7cb3, [0x2e2d] = 0x7ecf,
  [0x2e2e] = 0x4e95, [0x2e2f] = 0x8b66, [0x2e30] = 0x666f, [0x2e31] = 0x9888,
  [0x2e32] = 0x9759, [0x2e33] = 0x5883, [0x2e34] = 0x656c, [0x2e35] = 0x955c,
  [0x2e36] = 0x5f84, [0x2e37] = 0x75c9, [0x2e38] = 0x9756, [0x2e39] = 0x7adf,
  [0x2e3a] = 0x7ade, [0x2e3b] = 0x51c0, [0x2e3c] = 0x70af, [0x2e3d] = 0x7a98,
  [0x2e3e] = 0x63ea, [0x2e3f] = 0x7a76, [0x2e40] = 0x7ea0, [0x2e41] = 0x7396,
  [0x2e42] = 0x97ed, [0x2e43] = 0x4e45, [0x2e44] = 0x7078, [0x2e45] = 0x4e5d,
  [0x2e46] = 0x9152, [0x2e47] = 0x53a9, [0x2e48] = 0x6551, [0x2e49] = 0x65e7,
  [0x2e4a] = 0x81fc, [0x2e4b] = 0x8205, [0x2e4c] = 0x548e, [0x2e4d] = 0x5c31,
  [0x2e4e] = 0x759a, [0x2e4f] = 0x97a0, [0x2e50] = 0x62d8, [0x2e51] = 0x72d9,
  [0x2e52] = 0x75bd, [0x2e53] = 0x5c45, [0x2e54] = 0x9a79, [0x2e55] = 0x83ca,
  [0x2e56] = 0x5c40, [0x2e57] = 0x5480, [0x2e58] = 0x77e9, [0x2e59] = 0x4e3e,
  [0x2e5a] = 0x6cae, [0x2e5b] = 0x805a, [0x2e5c] = 0x62d2, [0x2e5d] = 0x636e,
  [0x2e5e] = 0x5de8, [0x2e5f] = 0x5177, [0x2e60] = 0x8ddd, [0x2e61] = 0x8e1e,
  [0x2e62] = 0x952f, [0x2e63] = 0x4ff1, [0x2e64] = 0x53e5, [0x2e65] = 0x60e7,
  [0x2e66] = 0x70ac, [0x2e67] = 0x5267, [0x2e68] = 0x6350, [0x2e69] = 0x9e43,
  [0x2e6a] = 0x5a1f, [0x2e6b] = 0x5026, [0x2e6c] = 0x7737, [0x2e6d] = 0x5377,
  [0x2e6e] = 0x7ee2, [0x2e6f] = 0x6485, [0x2e70] = 0x652b, [0x2e71] = 0x6289,
  [0x2e72] = 0x6398, [0x2e73] = 0x5014, [0x2e74] = 0x7235, [0x2e75] = 0x89c9,
  [0x2e76] = 0x51b3, [0x2e77] = 0x8bc0, [0x2e78] = 0x7edd, [0x2e79] = 0x5747,
  [0x2e7a] = 0x83cc, [0x2e7b] = 0x94a7, [0x2e7c] = 0x519b, [0x2e7d] = 0x541b,
  [0x2e7e] = 0x5cfb, [0x2e80] = 0x7dfb, [0x2e81] = 0x7dfc, [0x2e82] = 0x7dfd,
  [0x2e83] = 0x7dfe, [0x2e84] = 0x7dff, [0x2e85] = 0x7e00, [0x2e86] = 0x7e01,
  [0x2e87] = 0x7e02, [0x2e88] = 0x7e03, [0x2e89] = 0x7e04, [0x2e8a] = 0x7e05,
  [0x2e8b] = 0x7e06, [0x2e8c] = 0x7e07, [0x2e8d] = 0x7e08, [0x2e8e] = 0x7e09,
  [0x2e8f] = 0x7e0a, [0x2e90] = 0x7e0b, [0x2e91] = 0x7e0c, [0x2e92] = 0x7e0d,
  [0x2e93] = 0x7e0e, [0x2e94] = 0x7e0f, [0x2e95] = 0x7e10, [0x2e96] = 0x7e11,
  [0x2e97] = 0x7e12, [0x2e98] = 0x7e13, [0x2e99] = 0x7e14, [0x2e9a] = 0x7e15,
  [0x2e9b] = 0x7e16, [0x2e9c] = 0x7e17, [0x2e9d] = 0x7e18, [0x2e9e] = 0x7e19,
  [0x2e9f] = 0x7e1a, [0x2ea0] = 0x7e1b, [0x2ea1] = 0x7e1c, [0x2ea2] = 0x7e1d,
  [0x2ea3] = 0x7e1e, [0x2ea4] = 0x7e1f, [0x2ea5] = 0x7e20, [0x2ea6] = 0x7e21,
  [0x2ea7] = 0x7e22, [0x2ea8] = 0x7e23, [0x2ea9] = 0x7e24, [0x2eaa] = 0x7e25,
  [0x2eab] = 0x7e26, [0x2eac] = 0x7e27, [0x2ead] = 0x7e28, [0x2eae] = 0x7e29,
  [0x2eaf] = 0x7e2a, [0x2eb0] = 0x7e2b, [0x2eb1] = 0x7e2c, [0x2eb2] = 0x7e2d,
  [0x2eb3] = 0x7e2e, [0x2eb4] = 0x7e2f, [0x2eb5] = 0x7e30, [0x2eb6] = 0x7e31,
  [0x2eb7] = 0x7e32, [0x2eb8] = 0x7e33, [0x2eb9] = 0x7e34, [0x2eba] = 0x7e35,
  [0x2ebb] = 0x7e36, [0x2ebc] = 0x7e37, [0x2ebd] = 0x7e38, [0x2ebe] = 0x7e39,
  [0x2ec0] = 0x7e3a, [0x2ec1] = 0x7e3c, [0x2ec2] = 0x7e3d, [0x2ec3] = 0x7e3e,
  [0x2ec4] = 0x7e3f, [0x2ec5] = 0x7e40, [0x2ec6] = 0x7e42, [0x2ec7] = 0x7e43,
  [0x2ec8] = 0x7e44, [0x2ec9] = 0x7e45, [0x2eca] = 0x7e46, [0x2ecb] = 0x7e48,
  [0x2ecc] = 0x7e49, [0x2ecd] = 0x7e4a, [0x2ece] = 0x7e4b, [0x2ecf] = 0x7e4c,
  [0x2ed0] = 0x7e4d, [0x2ed1] = 0x7e4e, [0x2ed2] = 0x7e4f, [0x2ed3] = 0x7e50,
  [0x2ed4] = 0x7e51, [0x2ed5] = 0x7e52, [0x2ed6] = 0x7e53, [0x2ed7] = 0x7e54,
  [0x2ed8] = 0x7e55, [0x2ed9] = 0x7e56, [0x2eda] = 0x7e57, [0x2edb] = 0x7e58,
  [0x2edc] = 0x7e59, [0x2edd] = 0x7e5a, [0x2ede] = 0x7e5b, [0x2edf] = 0x7e5c,
  [0x2ee0] = 0x7e5d, [0x2ee1] = 0x4fca, [0x2ee2] = 0x7ae3, [0x2ee3] = 0x6d5a,
  [0x2ee4] = 0x90e1, [0x2ee5] = 0x9a8f, [0x2ee6] = 0x5580, [0x2ee7] = 0x5496,
  [0x2ee8] = 0x5361, [0x2ee9] = 0x54af, [0x2eea] = 0x5f00, [0x2eeb] = 0x63e9,
  [0x2eec] = 0x6977, [0x2eed] = 0x51ef, [0x2eee] = 0x6168, [0x2eef] = 0x520a,
  [0x2ef0] = 0x582a, [0x2ef1] = 0x52d8, [0x2ef2] = 0x574e, [0x2ef3] = 0x780d,
  [0x2ef4] = 0x770b, [0x2ef5] = 0x5eb7, [0x2ef6] = 0x6177, [0x2ef7] = 0x7ce0,
  [0x2ef8] = 0x625b, [0x2ef9] = 0x6297, [0x2efa] = 0x4ea2, [0x2efb] = 0x7095,
  [0x2efc] = 0x8003, [0x2efd] = 0x62f7, [0x2efe] = 0x70e4, [0x2eff] = 0x9760,
  [0x2f00] = 0x5777, [0x2f01] = 0x82db, [0x2f02] = 0x67ef, [0x2f03] = 0x68f5,
  [0x2f04] = 0x78d5, [0x2f05] = 0x9897, [0x2f06] = 0x79d1, [0x2f07] = 0x58f3,
  [0x2f08] = 0x54b3, [0x2f09] = 0x53ef, [0x2f0a] = 0x6e34, [0x2f0b] = 0x514b,
  [0x2f0c] = 0x523b, [0x2f0d] = 0x5ba2, [0x2f0e] = 0x8bfe, [0x2f0f] = 0x80af,
  [0x2f10] = 0x5543, [0x2f11] = 0x57a6, [0x2f12] = 0x6073, [0x2f13] = 0x5751,
  [0x2f14] = 0x542d, [0x2f15] = 0x7a7a, [0x2f16] = 0x6050, [0x2f17] = 0x5b54,
  [0x2f18] = 0x63a7, [0x2f19] = 0x62a0, [0x2f1a] = 0x53e3, [0x2f1b] = 0x6263,
  [0x2f1c] = 0x5bc7, [0x2f1d] = 0x67af, [0x2f1e] = 0x54ed, [0x2f1f] = 0x7a9f,
  [0x2f20] = 0x82e6, [0x2f21] = 0x9177, [0x2f22] = 0x5e93, [0x2f23] = 0x88e4,
  [0x2f24] = 0x5938, [0x2f25] = 0x57ae, [0x2f26] = 0x630e, [0x2f27] = 0x8de8,
  [0x2f28] = 0x80ef, [0x2f29] = 0x5757, [0x2f2a] = 0x7b77, [0x2f2b] = 0x4fa9,
  [0x2f2c] = 0x5feb, [0x2f2d] = 0x5bbd, [0x2f2e] = 0x6b3e, [0x2f2f] = 0x5321,
  [0x2f30] = 0x7b50, [0x2f31] = 0x72c2, [0x2f32] = 0x6846, [0x2f33] = 0x77ff,
  [0x2f34] = 0x7736, [0x2f35] = 0x65f7, [0x2f36] = 0x51b5, [0x2f37] = 0x4e8f,
  [0x2f38] = 0x76d4, [0x2f39] = 0x5cbf, [0x2f3a] = 0x7aa5, [0x2f3b] = 0x8475,
  [0x2f3c] = 0x594e, [0x2f3d] = 0x9b41, [0x2f3e] = 0x5080, [0x2f40] = 0x7e5e,
  [0x2f41] = 0x7e5f, [0x2f42] = 0x7e60, [0x2f43] = 0x7e61, [0x2f44] = 0x7e62,
  [0x2f45] = 0x7e63, [0x2f46] = 0x7e64, [0x2f47] = 0x7e65, [0x2f48] = 0x7e66,
  [0x2f49] = 0x7e67, [0x2f4a] = 0x7e68, [0x2f4b] = 0x7e69, [0x2f4c] = 0x7e6a,
  [0x2f4d] = 0x7e6b, [0x2f4e] = 0x7e6c, [0x2f4f] = 0x7e6d, [0x2f50] = 0x7e6e,
  [0x2f51] = 0x7e6f, [0x2f52] = 0x7e70, [0x2f53] = 0x7e71, [0x2f54] = 0x7e72,
  [0x2f55] = 0x7e73, [0x2f56] = 0x7e74, [0x2f57] = 0x7e75, [0x2f58] = 0x7e76,
  [0x2f59] = 0x7e77, [0x2f5a] = 0x7e78, [0x2f5b] = 0x7e79, [0x2f5c] = 0x7e7a,
  [0x2f5d] = 0x7e7b, [0x2f5e] = 0x7e7c, [0x2f5f] = 0x7e7d, [0x2f60] = 0x7e7e,
  [0x2f61] = 0x7e7f, [0x2f62] = 0x7e80, [0x2f63] = 0x7e81, [0x2f64] = 0x7e83,
  [0x2f65] = 0x7e84, [0x2f66] = 0x7e85, [0x2f67] = 0x7e86, [0x2f68] = 0x7e87,
  [0x2f69] = 0x7e88, [0x2f6a] = 0x7e89, [0x2f6b] = 0x7e8a, [0x2f6c] = 0x7e8b,
  [0x2f6d] = 0x7e8c, [0x2f6e] = 0x7e8d, [0x2f6f] = 0x7e8e, [0x2f70] = 0x7e8f,
  [0x2f71] = 0x7e90, [0x2f72] = 0x7e91, [0x2f73] = 0x7e92, [0x2f74] = 0x7e93,
  [0x2f75] = 0x7e94, [0x2f76] = 0x7e95, [0x2f77] = 0x7e96, [0x2f78] = 0x7e97,
  [0x2f79] = 0x7e98, [0x2f7a] = 0x7e99, [0x2f7b] = 0x7e9a, [0x2f7c] = 0x7e9c,
  [0x2f7d] = 0x7e9d, [0x2f7e] = 0x7e9e, [0x2f80] = 0x7eae, [0x2f81] = 0x7eb4,
  [0x2f82] = 0x7ebb, [0x2f83] = 0x7ebc, [0x2f84] = 0x7ed6, [0x2f85] = 0x7ee4,
  [0x2f86] = 0x7eec, [0x2f87] = 0x7ef9, [0x2f88] = 0x7f0a, [0x2f89] = 0x7f10,
  [0x2f8a] = 0x7f1e, [0x2f8b] = 0x7f37, [0x2f8c] = 0x7f39, [0x2f8d] = 0x7f3b,
  [0x2f8e] = 0x7f3c, [0x2f8f] = 0x7f3d, [0x2f90] = 0x7f3e, [0x2f91] = 0x7f3f,
  [0x2f92] = 0x7f40, [0x2f93] = 0x7f41, [0x2f94] = 0x7f43, [0x2f95] = 0x7f46,
  [0x2f96] = 0x7f47, [0x2f97] = 0x7f48, [0x2f98] = 0x7f49, [0x2f99] = 0x7f4a,
  [0x2f9a] = 0x7f4b, [0x2f9b] = 0x7f4c, [0x2f9c] = 0x7f4d, [0x2f9d] = 0x7f4e,
  [0x2f9e] = 0x7f4f, [0x2f9f] = 0x7f52, [0x2fa0] = 0x7f53, [0x2fa1] = 0x9988,
  [0x2fa2] = 0x6127, [0x2fa3] = 0x6e83, [0x2fa4] = 0x5764, [0x2fa5] = 0x6606,
  [0x2fa6] = 0x6346, [0x2fa7] = 0x56f0, [0x2fa8] = 0x62ec, [0x2fa9] = 0x6269,
  [0x2faa] = 0x5ed3, [0x2fab] = 0x9614, [0x2fac] = 0x5783, [0x2fad] = 0x62c9,
  [0x2fae] = 0x5587, [0x2faf] = 0x8721, [0x2fb0] = 0x814a, [0x2fb1] = 0x8fa3,
  [0x2fb2] = 0x5566, [0x2fb3] = 0x83b1, [0x2fb4] = 0x6765, [0x2fb5] = 0x8d56,
  [0x2fb6] = 0x84dd, [0x2fb7] = 0x5a6a, [0x2fb8] = 0x680f, [0x2fb9] = 0x62e6,
  [0x2fba] = 0x7bee, [0x2fbb] = 0x9611, [0x2fbc] = 0x5170, [0x2fbd] = 0x6f9c,
  [0x2fbe] = 0x8c30, [0x2fbf] = 0x63fd, [0x2fc0] = 0x89c8, [0x2fc1] = 0x61d2,
  [0x2fc2] = 0x7f06, [0x2fc3] = 0x70c2, [0x2fc4] = 0x6ee5, [0x2fc5] = 0x7405,
  [0x2fc6] = 0x6994, [0x2fc7] = 0x72fc, [0x2fc8] = 0x5eca, [0x2fc9] = 0x90ce,
  [0x2fca] = 0x6717, [0x2fcb] = 0x6d6a, [0x2fcc] = 0x635e, [0x2fcd] = 0x52b3,
  [0x2fce] = 0x7262, [0x2fcf] = 0x8001, [0x2fd0] = 0x4f6c, [0x2fd1] = 0x59e5,
  [0x2fd2] = 0x916a, [0x2fd3] = 0x70d9, [0x2fd4] = 0x6d9d, [0x2fd5] = 0x52d2,
  [0x2fd6] = 0x4e50, [0x2fd7] = 0x96f7, [0x2fd8] = 0x956d, [0x2fd9] = 0x857e,
  [0x2fda] = 0x78ca, [0x2fdb] = 0x7d2f, [0x2fdc] = 0x5121, [0x2fdd] = 0x5792,
  [0x2fde] = 0x64c2, [0x2fdf] = 0x808b, [0x2fe0] = 0x7c7b, [0x2fe1] = 0x6cea,
  [0x2fe2] = 0x68f1, [0x2fe3] = 0x695e, [0x2fe4] = 0x51b7, [0x2fe5] = 0x5398,
  [0x2fe6] = 0x68a8, [0x2fe7] = 0x7281, [0x2fe8] = 0x9ece, [0x2fe9] = 0x7bf1,
  [0x2fea] = 0x72f8, [0x2feb] = 0x79bb, [0x2fec] = 0x6f13, [0x2fed] = 0x7406,
  [0x2fee] = 0x674e, [0x2fef] = 0x91cc, [0x2ff0] = 0x9ca4, [0x2ff1] = 0x793c,
  [0x2ff2] = 0x8389, [0x2ff3] = 0x8354, [0x2ff4] = 0x540f, [0x2ff5] = 0x6817,
  [0x2ff6] = 0x4e3d, [0x2ff7] = 0x5389, [0x2ff8] = 0x52b1, [0x2ff9] = 0x783e,
  [0x2ffa] = 0x5386, [0x2ffb] = 0x5229, [0x2ffc] = 0x5088, [0x2ffd] = 0x4f8b,
  [0x2ffe] = 0x4fd0, [0x3000] = 0x7f56, [0x3001] = 0x7f59, [0x3002] = 0x7f5b,
  [0x3003] = 0x7f5c, [0x3004] = 0x7f5d, [0x3005] = 0x7f5e, [0x3006] = 0x7f60,
  [0x3007] = 0x7f63, [0x3008] = 0x7f64, [0x3009] = 0x7f65, [0x300a] = 0x7f66,
  [0x300b] = 0x7f67, [0x300c] = 0x7f6b, [0x300d] = 0x7f6c, [0x300e] = 0x7f6d,
  [0x300f] = 0x7f6f, [0x3010] = 0x7f70, [0x3011] = 0x7f73, [0x3012] = 0x7f75,
  [0x3013] = 0x7f76, [0x3014] = 0x7f77, [0x3015] = 0x7f78, [0x3016] = 0x7f7a,
  [0x3017] = 0x7f7b, [0x3018] = 0x7f7c, [0x3019] = 0x7f7d, [0x301a] = 0x7f7f,
  [0x301b] = 0x7f80, [0x301c] = 0x7f82, [0x301d] = 0x7f83, [0x301e] = 0x7f84,
  [0x301f] = 0x7f85, [0x3020] = 0x7f86, [0x3021] = 0x7f87, [0x3022] = 0x7f88,
  [0x3023] = 0x7f89, [0x3024] = 0x7f8b, [0x3025] = 0x7f8d, [0x3026] = 0x7f8f,
  [0x3027] = 0x7f90, [0x3028] = 0x7f91, [0x3029] = 0x7f92, [0x302a] = 0x7f93,
  [0x302b] = 0x7f95, [0x302c] = 0x7f96, [0x302d] = 0x7f97, [0x302e] = 0x7f98,
  [0x302f] = 0x7f99, [0x3030] = 0x7f9b, [0x3031] = 0x7f9c, [0x3032] = 0x7fa0,
  [0x3033] = 0x7fa2, [0x3034] = 0x7fa3, [0x3035] = 0x7fa5, [0x3036] = 0x7fa6,
  [0x3037] = 0x7fa8, [0x3038] = 0x7fa9, [0x3039] = 0x7faa, [0x303a] = 0x7fab,
  [0x303b] = 0x7fac, [0x303c] = 0x7fad, [0x303d] = 0x7fae, [0x303e] = 0x7fb1,
  [0x3040] = 0x7fb3, [0x3041] = 0x7fb4, [0x3042] = 0x7fb5, [0x3043] = 0x7fb6,
  [0x3044] = 0x7fb7, [0x3045] = 0x7fba, [0x3046] = 0x7fbb, [0x3047] = 0x7fbe,
  [0x3048] = 0x7fc0, [0x3049] = 0x7fc2, [0x304a] = 0x7fc3, [0x304b] = 0x7fc4,
  [0x304c] = 0x7fc6, [0x304d] = 0x7fc7, [0x304e] = 0x7fc8, [0x304f] = 0x7fc9,
  [0x3050] = 0x7fcb, [0x3051] = 0x7fcd, [0x3052] = 0x7fcf, [0x3053] = 0x7fd0,
  [0x3054] = 0x7fd1, [0x3055] = 0x7fd2, [0x3056] = 0x7fd3, [0x3057] = 0x7fd6,
  [0x3058] = 0x7fd7, [0x3059] = 0x7fd9, [0x305a] = 0x7fda, [0x305b] = 0x7fdb,
  [0x305c] = 0x7fdc, [0x305d] = 0x7fdd, [0x305e] = 0x7fde, [0x305f] = 0x7fe2,
  [0x3060] = 0x7fe3, [0x3061] = 0x75e2, [0x3062] = 0x7acb, [0x3063] = 0x7c92,
  [0x3064] = 0x6ca5, [0x3065] = 0x96b6, [0x3066] = 0x529b, [0x3067] = 0x7483,
  [0x3068] = 0x54e9, [0x3069] = 0x4fe9, [0x306a] = 0x8054, [0x306b] = 0x83b2,
  [0x306c] = 0x8fde, [0x306d] = 0x9570, [0x306e] = 0x5ec9, [0x306f] = 0x601c,
  [0x3070] = 0x6d9f, [0x3071] = 0x5e18, [0x3072] = 0x655b, [0x3073] = 0x8138,
  [0x3074] = 0x94fe, [0x3075] = 0x604b, [0x3076] = 0x70bc, [0x3077] = 0x7ec3,
  [0x3078] = 0x7cae, [0x3079] = 0x51c9, [0x307a] = 0x6881, [0x307b] = 0x7cb1,
  [0x307c] = 0x826f, [0x307d] = 0x4e24, [0x307e] = 0x8f86, [0x307f] = 0x91cf,
  [0x3080] = 0x667e, [0x3081] = 0x4eae, [0x3082] = 0x8c05, [0x3083] = 0x64a9,
  [0x3084] = 0x804a, [0x3085] = 0x50da, [0x3086] = 0x7597, [0x3087] = 0x71ce,
  [0x3088] = 0x5be5, [0x3089] = 0x8fbd, [0x308a] = 0x6f66, [0x308b] = 0x4e86,
  [0x308c] = 0x6482, [0x308d] = 0x9563, [0x308e] = 0x5ed6, [0x308f] = 0x6599,
  [0x3090] = 0x5217, [0x3091] = 0x88c2, [0x3092] = 0x70c8, [0x3093] = 0x52a3,
  [0x3094] = 0x730e, [0x3095] = 0x7433, [0x3096] = 0x6797, [0x3097] = 0x78f7,
  [0x3098] = 0x9716, [0x3099] = 0x4e34, [0x309a] = 0x90bb, [0x309b] = 0x9cde,
  [0x309c] = 0x6dcb, [0x309d] = 0x51db, [0x309e] = 0x8d41, [0x309f] = 0x541d,
  [0x30a0] = 0x62ce, [0x30a1] = 0x73b2, [0x30a2] = 0x83f1, [0x30a3] = 0x96f6,
  [0x30a4] = 0x9f84, [0x30a5] = 0x94c3, [0x30a6] = 0x4f36, [0x30a7] = 0x7f9a,
  [0x30a8] = 0x51cc, [0x30a9] = 0x7075, [0x30aa] = 0x9675, [0x30ab] = 0x5cad,
  [0x30ac] = 0x9886, [0x30ad] = 0x53e6, [0x30ae] = 0x4ee4, [0x30af] = 0x6e9c,
  [0x30b0] = 0x7409, [0x30b1] = 0x69b4, [0x30b2] = 0x786b, [0x30b3] = 0x998f,
  [0x30b4] = 0x7559, [0x30b5] = 0x5218, [0x30b6] = 0x7624, [0x30b7] = 0x6d41,
  [0x30b8] = 0x67f3, [0x30b9] = 0x516d, [0x30ba] = 0x9f99, [0x30bb] = 0x804b,
  [0x30bc] = 0x5499, [0x30bd] = 0x7b3c, [0x30be] = 0x7abf, [0x30c0] = 0x7fe4,
  [0x30c1] = 0x7fe7, [0x30c2] = 0x7fe8, [0x30c3] = 0x7fea, [0x30c4] = 0x7feb,
  [0x30c5] = 0x7fec, [0x30c6] = 0x7fed, [0x30c7] = 0x7fef, [0x30c8] = 0x7ff2,
  [0x30c9] = 0x7ff4, [0x30ca] = 0x7ff5, [0x30cb] = 0x7ff6, [0x30cc] = 0x7ff7,
  [0x30cd] = 0x7ff8, [0x30ce] = 0x7ff9, [0x30cf] = 0x7ffa, [0x30d0] = 0x7ffd,
  [0x30d1] = 0x7ffe, [0x30d2] = 0x7fff, [0x30d3] = 0x8002, [0x30d4] = 0x8007,
  [0x30d5] = 0x8008, [0x30d6] = 0x8009, [0x30d7] = 0x800a, [0x30d8] = 0x800e,
  [0x30d9] = 0x800f, [0x30da] = 0x8011, [0x30db] = 0x8013, [0x30dc] = 0x801a,
  [0x30dd] = 0x801b, [0x30de] = 0x801d, [0x30df] = 0x801e, [0x30e0] = 0x801f,
  [0x30e1] = 0x8021, [0x30e2] = 0x8023, [0x30e3] = 0x8024, [0x30e4] = 0x802b,
  [0x30e5] = 0x802c, [0x30e6] = 0x802d, [0x30e7] = 0x802e, [0x30e8] = 0x802f,
  [0x30e9] = 0x8030, [0x30ea] = 0x8032, [0x30eb] = 0x8034, [0x30ec] = 0x8039,
  [0x30ed] = 0x803a, [0x30ee] = 0x803c, [0x30ef] = 0x803e, [0x30f0] = 0x8040,
  [0x30f1] = 0x8041, [0x30f2] = 0x8044, [0x30f3] = 0x8045, [0x30f4] = 0x8047,
  [0x30f5] = 0x8048, [0x30f6] = 0x8049, [0x30f7] = 0x804e, [0x30f8] = 0x804f,
  [0x30f9] = 0x8050, [0x30fa] = 0x8051, [0x30fb] = 0x8053, [0x30fc] = 0x8055,
  [0x30fd] = 0x8056, [0x30fe] = 0x8057, [0x3100] = 0x8059, [0x3101] = 0x805b,
  [0x3102] = 0x805c, [0x3103] = 0x805d, [0x3104] = 0x805e, [0x3105] = 0x805f,
  [0x3106] = 0x8060, [0x3107] = 0x8061, [0x3108] = 0x8062, [0x3109] = 0x8063,
  [0x310a] = 0x8064, [0x310b] = 0x8065, [0x310c] = 0x8066, [0x310d] = 0x8067,
  [0x310e] = 0x8068, [0x310f] = 0x806b, [0x3110] = 0x806c, [0x3111] = 0x806d,
  [0x3112] = 0x806e, [0x3113] = 0x806f, [0x3114] = 0x8070, [0x3115] = 0x8072,
  [0x3116] = 0x8073, [0x3117] = 0x8074, [0x3118] = 0x8075, [0x3119] = 0x8076,
  [0x311a] = 0x8077, [0x311b] = 0x8078, [0x311c] = 0x8079, [0x311d] = 0x807a,
  [0x311e] = 0x807b, [0x311f] = 0x807c, [0x3120] = 0x807d, [0x3121] = 0x9686,
  [0x3122] = 0x5784, [0x3123] = 0x62e2, [0x3124] = 0x9647, [0x3125] = 0x697c,
  [0x3126] = 0x5a04, [0x3127] = 0x6402, [0x3128] = 0x7bd3, [0x3129] = 0x6f0f,
  [0x312a] = 0x964b, [0x312b] = 0x82a6, [0x312c] = 0x5362, [0x312d] = 0x9885,
  [0x312e] = 0x5e90, [0x312f] = 0x7089, [0x3130] = 0x63b3, [0x3131] = 0x5364,
  [0x3132] = 0x864f, [0x3133] = 0x9c81, [0x3134] = 0x9e93, [0x3135] = 0x788c,
  [0x3136] = 0x9732, [0x3137] = 0x8def, [0x3138] = 0x8d42, [0x3139] = 0x9e7f,
  [0x313a] = 0x6f5e, [0x313b] = 0x7984, [0x313c] = 0x5f55, [0x313d] = 0x9646,
  [0x313e] = 0x622e, [0x313f] = 0x9a74, [0x3140] = 0x5415, [0x3141] = 0x94dd,
  [0x3142] = 0x4fa3, [0x3143] = 0x65c5, [0x3144] = 0x5c65, [0x3145] = 0x5c61,
  [0x3146] = 0x7f15, [0x3147] = 0x8651, [0x3148] = 0x6c2f, [0x3149] = 0x5f8b,
  [0x314a] = 0x7387, [0x314b] = 0x6ee4, [0x314c] = 0x7eff, [0x314d] = 0x5ce6,
  [0x314e] = 0x631b, [0x314f] = 0x5b6a, [0x3150] = 0x6ee6, [0x3151] = 0x5375,
  [0x3152] = 0x4e71, [0x3153] = 0x63a0, [0x3154] = 0x7565, [0x3155] = 0x62a1,
  [0x3156] = 0x8f6e, [0x3157] = 0x4f26, [0x3158] = 0x4ed1, [0x3159] = 0x6ca6,
  [0x315a] = 0x7eb6, [0x315b] = 0x8bba, [0x315c] = 0x841d, [0x315d] = 0x87ba,
  [0x315e] = 0x7f57, [0x315f] = 0x903b, [0x3160] = 0x9523, [0x3161] = 0x7ba9,
  [0x3162] = 0x9aa1, [0x3163] = 0x88f8, [0x3164] = 0x843d, [0x3165] = 0x6d1b,
  [0x3166] = 0x9a86, [0x3167] = 0x7edc, [0x3168] = 0x5988, [0x3169] = 0x9ebb,
  [0x316a] = 0x739b, [0x316b] = 0x7801, [0x316c] = 0x8682, [0x316d] = 0x9a6c,
  [0x316e] = 0x9a82, [0x316f] = 0x561b, [0x3170] = 0x5417, [0x3171] = 0x57cb,
  [0x3172] = 0x4e70, [0x3173] = 0x9ea6, [0x3174] = 0x5356, [0x3175] = 0x8fc8,
  [0x3176] = 0x8109, [0x3177] = 0x7792, [0x3178] = 0x9992, [0x3179] = 0x86ee,
  [0x317a] = 0x6ee1, [0x317b] = 0x8513, [0x317c] = 0x66fc, [0x317d] = 0x6162,
  [0x317e] = 0x6f2b, [0x3180] = 0x807e, [0x3181] = 0x8081, [0x3182] = 0x8082,
  [0x3183] = 0x8085, [0x3184] = 0x8088, [0x3185] = 0x808a, [0x3186] = 0x808d,
  [0x3187] = 0x808e, [0x3188] = 0x808f, [0x3189] = 0x8090, [0x318a] = 0x8091,
  [0x318b] = 0x8092, [0x318c] = 0x8094, [0x318d] = 0x8095, [0x318e] = 0x8097,
  [0x318f] = 0x8099, [0x3190] = 0x809e, [0x3191] = 0x80a3, [0x3192] = 0x80a6,
  [0x3193] = 0x80a7, [0x3194] = 0x80a8, [0x3195] = 0x80ac, [0x3196] = 0x80b0,
  [0x3197] = 0x80b3, [0x3198] = 0x80b5, [0x3199] = 0x80b6, [0x319a] = 0x80b8,
  [0x319b] = 0x80b9, [0x319c] = 0x80bb, [0x319d] = 0x80c5, [0x319e] = 0x80c7,
  [0x319f] = 0x80c8, [0x31a0] = 0x80c9, [0x31a1] = 0x80ca, [0x31a2] = 0x80cb,
  [0x31a3] = 0x80cf, [0x31a4] = 0x80d0, [0x31a5] = 0x80d1, [0x31a6] = 0x80d2,
  [0x31a7] = 0x80d3, [0x31a8] = 0x80d4, [0x31a9] = 0x80d5, [0x31aa] = 0x80d8,
  [0x31ab] = 0x80df, [0x31ac] = 0x80e0, [0x31ad] = 0x80e2, [0x31ae] = 0x80e3,
  [0x31af] = 0x80e6, [0x31b0] = 0x80ee, [0x31b1] = 0x80f5, [0x31b2] = 0x80f7,
  [0x31b3] = 0x80f9, [0x31b4] = 0x80fb, [0x31b5] = 0x80fe, [0x31b6] = 0x80ff,
  [0x31b7] = 0x8100, [0x31b8] = 0x8101, [0x31b9] = 0x8103, [0x31ba] = 0x8104,
  [0x31bb] = 0x8105, [0x31bc] = 0x8107, [0x31bd] = 0x8108, [0x31be] = 0x810b,
  [0x31c0] = 0x810c, [0x31c1] = 0x8115, [0x31c2] = 0x8117, [0x31c3] = 0x8119,
  [0x31c4] = 0x811b, [0x31c5] = 0x811c, [0x31c6] = 0x811d, [0x31c7] = 0x811f,
  [0x31c8] = 0x8120, [0x31c9] = 0x8121, [0x31ca] = 0x8122, [0x31cb] = 0x8123,
  [0x31cc] = 0x8124, [0x31cd] = 0x8125, [0x31ce] = 0x8126, [0x31cf] = 0x8127,
  [0x31d0] = 0x8128, [0x31d1] = 0x8129, [0x31d2] = 0x812a, [0x31d3] = 0x812b,
  [0x31d4] = 0x812d, [0x31d5] = 0x812e, [0x31d6] = 0x8130, [0x31d7] = 0x8133,
  [0x31d8] = 0x8134, [0x31d9] = 0x8135, [0x31da] = 0x8137, [0x31db] = 0x8139,
  [0x31dc] = 0x813a, [0x31dd] = 0x813b, [0x31de] = 0x813c, [0x31df] = 0x813d,
  [0x31e0] = 0x813f, [0x31e1] = 0x8c29, [0x31e2] = 0x8292, [0x31e3] = 0x832b,
  [0x31e4] = 0x76f2, [0x31e5] = 0x6c13, [0x31e6] = 0x5fd9, [0x31e7] = 0x83bd,
  [0x31e8] = 0x732b, [0x31e9] = 0x8305, [0x31ea] = 0x951a, [0x31eb] = 0x6bdb,
  [0x31ec] = 0x77db, [0x31ed] = 0x94c6, [0x31ee] = 0x536f, [0x31ef] = 0x8302,
  [0x31f0] = 0x5192, [0x31f1] = 0x5e3d, [0x31f2] = 0x8c8c, [0x31f3] = 0x8d38,
  [0x31f4] = 0x4e48, [0x31f5] = 0x73ab, [0x31f6] = 0x679a, [0x31f7] = 0x6885,
  [0x31f8] = 0x9176, [0x31f9] = 0x9709, [0x31fa] = 0x7164, [0x31fb] = 0x6ca1,
  [0x31fc] = 0x7709, [0x31fd] = 0x5a92, [0x31fe] = 0x9541, [0x31ff] = 0x6bcf,
  [0x3200] = 0x7f8e, [0x3201] = 0x6627, [0x3202] = 0x5bd0, [0x3203] = 0x59b9,
  [0x3204] = 0x5a9a, [0x3205] = 0x95e8, [0x3206] = 0x95f7, [0x3207] = 0x4eec,
  [0x3208] = 0x840c, [0x3209] = 0x8499, [0x320a] = 0x6aac, [0x320b] = 0x76df,
  [0x320c] = 0x9530, [0x320d] = 0x731b, [0x320e] = 0x68a6, [0x320f] = 0x5b5f,
  [0x3210] = 0x772f, [0x3211] = 0x919a, [0x3212] = 0x9761, [0x3213] = 0x7cdc,
  [0x3214] = 0x8ff7, [0x3215] = 0x8c1c, [0x3216] = 0x5f25, [0x3217] = 0x7c73,
  [0x3218] = 0x79d8, [0x3219] = 0x89c5, [0x321a] = 0x6ccc, [0x321b] = 0x871c,
  [0x321c] = 0x5bc6, [0x321d] = 0x5e42, [0x321e] = 0x68c9, [0x321f] = 0x7720,
  [0x3220] = 0x7ef5, [0x3221] = 0x5195, [0x3222] = 0x514d, [0x3223] = 0x52c9,
  [0x3224] = 0x5a29, [0x3225] = 0x7f05, [0x3226] = 0x9762, [0x3227] = 0x82d7,
  [0x3228] = 0x63cf, [0x3229] = 0x7784, [0x322a] = 0x85d0, [0x322b] = 0x79d2,
  [0x322c] = 0x6e3a, [0x322d] = 0x5e99, [0x322e] = 0x5999, [0x322f] = 0x8511,
  [0x3230] = 0x706d, [0x3231] = 0x6c11, [0x3232] = 0x62bf, [0x3233] = 0x76bf,
  [0x3234] = 0x654f, [0x3235] = 0x60af, [0x3236] = 0x95fd, [0x3237] = 0x660e,
  [0x3238] = 0x879f, [0x3239] = 0x9e23, [0x323a] = 0x94ed, [0x323b] = 0x540d,
  [0x323c] = 0x547d, [0x323d] = 0x8c2c, [0x323e] = 0x6478, [0x3240] = 0x8140,
  [0x3241] = 0x8141, [0x3242] = 0x8142, [0x3243] = 0x8143, [0x3244] = 0x8144,
  [0x3245] = 0x8145, [0x3246] = 0x8147, [0x3247] = 0x8149, [0x3248] = 0x814d,
  [0x3249] = 0x814e, [0x324a] = 0x814f, [0x324b] = 0x8152, [0x324c] = 0x8156,
  [0x324d] = 0x8157, [0x324e] = 0x8158, [0x324f] = 0x815b, [0x3250] = 0x815c,
  [0x3251] = 0x815d, [0x3252] = 0x815e, [0x3253] = 0x815f, [0x3254] = 0x8161,
  [0x3255] = 0x8162, [0x3256] = 0x8163, [0x3257] = 0x8164, [0x3258] = 0x8166,
  [0x3259] = 0x8168, [0x325a] = 0x816a, [0x325b] = 0x816b, [0x325c] = 0x816c,
  [0x325d] = 0x816f, [0x325e] = 0x8172, [0x325f] = 0x8173, [0x3260] = 0x8175,
  [0x3261] = 0x8176, [0x3262] = 0x8177, [0x3263] = 0x8178, [0x3264] = 0x8181,
  [0x3265] = 0x8183, [0x3266] = 0x8184, [0x3267] = 0x8185, [0x3268] = 0x8186,
  [0x3269] = 0x8187, [0x326a] = 0x8189, [0x326b] = 0x818b, [0x326c] = 0x818c,
  [0x326d] = 0x818d, [0x326e] = 0x818e, [0x326f] = 0x8190, [0x3270] = 0x8192,
  [0x3271] = 0x8193, [0x3272] = 0x8194, [0x3273] = 0x8195, [0x3274] = 0x8196,
  [0x3275] = 0x8197, [0x3276] = 0x8199, [0x3277] = 0x819a, [0x3278] = 0x819e,
  [0x3279] = 0x819f, [0x327a] = 0x81a0, [0x327b] = 0x81a1, [0x327c] = 0x81a2,
  [0x327d] = 0x81a4, [0x327e] = 0x81a5, [0x3280] = 0x81a7, [0x3281] = 0x81a9,
  [0x3282] = 0x81ab, [0x3283] = 0x81ac, [0x3284] = 0x81ad, [0x3285] = 0x81ae,
  [0x3286] = 0x81af, [0x3287] = 0x81b0, [0x3288] = 0x81b1, [0x3289] = 0x81b2,
  [0x328a] = 0x81b4, [0x328b] = 0x81b5, [0x328c] = 0x81b6, [0x328d] = 0x81b7,
  [0x328e] = 0x81b8, [0x328f] = 0x81b9, [0x3290] = 0x81bc, [0x3291] = 0x81bd,
  [0x3292] = 0x81be, [0x3293] = 0x81bf, [0x3294] = 0x81c4, [0x3295] = 0x81c5,
  [0x3296] = 0x81c7, [0x3297] = 0x81c8, [0x3298] = 0x81c9, [0x3299] = 0x81cb,
  [0x329a] = 0x81cd, [0x329b] = 0x81ce, [0x329c] = 0x81cf, [0x329d] = 0x81d0,
  [0x329e] = 0x81d1, [0x329f] = 0x81d2, [0x32a0] = 0x81d3, [0x32a1] = 0x6479,
  [0x32a2] = 0x8611, [0x32a3] = 0x6a21, [0x32a4] = 0x819c, [0x32a5] = 0x78e8,
  [0x32a6] = 0x6469, [0x32a7] = 0x9b54, [0x32a8] = 0x62b9, [0x32a9] = 0x672b,
  [0x32aa] = 0x83ab, [0x32ab] = 0x58a8, [0x32ac] = 0x9ed8, [0x32ad] = 0x6cab,
  [0x32ae] = 0x6f20, [0x32af] = 0x5bde, [0x32b0] = 0x964c, [0x32b1] = 0x8c0b,
  [0x32b2] = 0x725f, [0x32b3] = 0x67d0, [0x32b4] = 0x62c7, [0x32b5] = 0x7261,
  [0x32b6] = 0x4ea9, [0x32b7] = 0x59c6, [0x32b8] = 0x6bcd, [0x32b9] = 0x5893,
  [0x32ba] = 0x66ae, [0x32bb] = 0x5e55, [0x32bc] = 0x52df, [0x32bd] = 0x6155,
  [0x32be] = 0x6728, [0x32bf] = 0x76ee, [0x32c0] = 0x7766, [0x32c1] = 0x7267,
  [0x32c2] = 0x7a46, [0x32c3] = 0x62ff, [0x32c4] = 0x54ea, [0x32c5] = 0x5450,
  [0x32c6] = 0x94a0, [0x32c7] = 0x90a3, [0x32c8] = 0x5a1c, [0x32c9] = 0x7eb3,
  [0x32ca] = 0x6c16, [0x32cb] = 0x4e43, [0x32cc] = 0x5976, [0x32cd] = 0x8010,
  [0x32ce] = 0x5948, [0x32cf] = 0x5357, [0x32d0] = 0x7537, [0x32d1] = 0x96be,
  [0x32d2] = 0x56ca, [0x32d3] = 0x6320, [0x32d4] = 0x8111, [0x32d5] = 0x607c,
  [0x32d6] = 0x95f9, [0x32d7] = 0x6dd6, [0x32d8] = 0x5462, [0x32d9] = 0x9981,
  [0x32da] = 0x5185, [0x32db] = 0x5ae9, [0x32dc] = 0x80fd, [0x32dd] = 0x59ae,
  [0x32de] = 0x9713, [0x32df] = 0x502a, [0x32e0] = 0x6ce5, [0x32e1] = 0x5c3c,
  [0x32e2] = 0x62df, [0x32e3] = 0x4f60, [0x32e4] = 0x533f, [0x32e5] = 0x817b,
  [0x32e6] = 0x9006, [0x32e7] = 0x6eba, [0x32e8] = 0x852b, [0x32e9] = 0x62c8,
  [0x32ea] = 0x5e74, [0x32eb] = 0x78be, [0x32ec] = 0x64b5, [0x32ed] = 0x637b,
  [0x32ee] = 0x5ff5, [0x32ef] = 0x5a18, [0x32f0] = 0x917f, [0x32f1] = 0x9e1f,
  [0x32f2] = 0x5c3f, [0x32f3] = 0x634f, [0x32f4] = 0x8042, [0x32f5] = 0x5b7d,
  [0x32f6] = 0x556e, [0x32f7] = 0x954a, [0x32f8] = 0x954d, [0x32f9] = 0x6d85,
  [0x32fa] = 0x60a8, [0x32fb] = 0x67e0, [0x32fc] = 0x72de, [0x32fd] = 0x51dd,
  [0x32fe] = 0x5b81, [0x3300] = 0x81d4, [0x3301] = 0x81d5, [0x3302] = 0x81d6,
  [0x3303] = 0x81d7, [0x3304] = 0x81d8, [0x3305] = 0x81d9, [0x3306] = 0x81da,
  [0x3307] = 0x81db, [0x3308] = 0x81dc, [0x3309] = 0x81dd, [0x330a] = 0x81de,
  [0x330b] = 0x81df, [0x330c] = 0x81e0, [0x330d] = 0x81e1, [0x330e] = 0x81e2,
  [0x330f] = 0x81e4, [0x3310] = 0x81e5, [0x3311] = 0x81e6, [0x3312] = 0x81e8,
  [0x3313] = 0x81e9, [0x3314] = 0x81eb, [0x3315] = 0x81ee, [0x3316] = 0x81ef,
  [0x3317] = 0x81f0, [0x3318] = 0x81f1, [0x3319] = 0x81f2, [0x331a] = 0x81f5,
  [0x331b] = 0x81f6, [0x331c] = 0x81f7, [0x331d] = 0x81f8, [0x331e] = 0x81f9,
  [0x331f] = 0x81fa, [0x3320] = 0x81fd, [0x3321] = 0x81ff, [0x3322] = 0x8203,
  [0x3323] = 0x8207, [0x3324] = 0x8208, [0x3325] = 0x8209, [0x3326] = 0x820a,
  [0x3327] = 0x820b, [0x3328] = 0x820e, [0x3329] = 0x820f, [0x332a] = 0x8211,
  [0x332b] = 0x8213, [0x332c] = 0x8215, [0x332d] = 0x8216, [0x332e] = 0x8217,
  [0x332f] = 0x8218, [0x3330] = 0x8219, [0x3331] = 0x821a, [0x3332] = 0x821d,
  [0x3333] = 0x8220, [0x3334] = 0x8224, [0x3335] = 0x8225, [0x3336] = 0x8226,
  [0x3337] = 0x8227, [0x3338] = 0x8229, [0x3339] = 0x822e, [0x333a] = 0x8232,
  [0x333b] = 0x823a, [0x333c] = 0x823c, [0x333d] = 0x823d, [0x333e] = 0x823f,
  [0x3340] = 0x8240, [0x3341] = 0x8241, [0x3342] = 0x8242, [0x3343] = 0x8243,
  [0x3344] = 0x8245, [0x3345] = 0x8246, [0x3346] = 0x8248, [0x3347] = 0x824a,
  [0x3348] = 0x824c, [0x3349] = 0x824d, [0x334a] = 0x824e, [0x334b] = 0x8250,
  [0x334c] = 0x8251, [0x334d] = 0x8252, [0x334e] = 0x8253, [0x334f] = 0x8254,
  [0x3350] = 0x8255, [0x3351] = 0x8256, [0x3352] = 0x8257, [0x3353] = 0x8259,
  [0x3354] = 0x825b, [0x3355] = 0x825c, [0x3356] = 0x825d, [0x3357] = 0x825e,
  [0x3358] = 0x8260, [0x3359] = 0x8261, [0x335a] = 0x8262, [0x335b] = 0x8263,
  [0x335c] = 0x8264, [0x335d] = 0x8265, [0x335e] = 0x8266, [0x335f] = 0x8267,
  [0x3360] = 0x8269, [0x3361] = 0x62e7, [0x3362] = 0x6cde, [0x3363] = 0x725b,
  [0x3364] = 0x626d, [0x3365] = 0x94ae, [0x3366] = 0x7ebd, [0x3367] = 0x8113,
  [0x3368] = 0x6d53, [0x3369] = 0x519c, [0x336a] = 0x5f04, [0x336b] = 0x5974,
  [0x336c] = 0x52aa, [0x336d] = 0x6012, [0x336e] = 0x5973, [0x336f] = 0x6696,
  [0x3370] = 0x8650, [0x3371] = 0x759f, [0x3372] = 0x632a, [0x3373] = 0x61e6,
  [0x3374] = 0x7cef, [0x3375] = 0x8bfa, [0x3376] = 0x54e6, [0x3377] = 0x6b27,
  [0x3378] = 0x9e25, [0x3379] = 0x6bb4, [0x337a] = 0x85d5, [0x337b] = 0x5455,
  [0x337c] = 0x5076, [0x337d] = 0x6ca4, [0x337e] = 0x556a, [0x337f] = 0x8db4,
  [0x3380] = 0x722c, [0x3381] = 0x5e15, [0x3382] = 0x6015, [0x3383] = 0x7436,
  [0x3384] = 0x62cd, [0x3385] = 0x6392, [0x3386] = 0x724c, [0x3387] = 0x5f98,
  [0x3388] = 0x6e43, [0x3389] = 0x6d3e, [0x338a] = 0x6500, [0x338b] = 0x6f58,
  [0x338c] = 0x76d8, [0x338d] = 0x78d0, [0x338e] = 0x76fc, [0x338f] = 0x7554,
  [0x3390] = 0x5224, [0x3391] = 0x53db, [0x3392] = 0x4e53, [0x3393] = 0x5e9e,
  [0x3394] = 0x65c1, [0x3395] = 0x802a, [0x3396] = 0x80d6, [0x3397] = 0x629b,
  [0x3398] = 0x5486, [0x3399] = 0x5228, [0x339a] = 0x70ae, [0x339b] = 0x888d,
  [0x339c] = 0x8dd1, [0x339d] = 0x6ce1, [0x339e] = 0x5478, [0x339f] = 0x80da,
  [0x33a0] = 0x57f9, [0x33a1] = 0x88f4, [0x33a2] = 0x8d54, [0x33a3] = 0x966a,
  [0x33a4] = 0x914d, [0x33a5] = 0x4f69, [0x33a6] = 0x6c9b, [0x33a7] = 0x55b7,
  [0x33a8] = 0x76c6, [0x33a9] = 0x7830, [0x33aa] = 0x62a8, [0x33ab] = 0x70f9,
  [0x33ac] = 0x6f8e, [0x33ad] = 0x5f6d, [0x33ae] = 0x84ec, [0x33af] = 0x68da,
  [0x33b0] = 0x787c, [0x33b1] = 0x7bf7, [0x33b2] = 0x81a8, [0x33b3] = 0x670b,
  [0x33b4] = 0x9e4f, [0x33b5] = 0x6367, [0x33b6] = 0x78b0, [0x33b7] = 0x576f,
  [0x33b8] = 0x7812, [0x33b9] = 0x9739, [0x33ba] = 0x6279, [0x33bb] = 0x62ab,
  [0x33bc] = 0x5288, [0x33bd] = 0x7435, [0x33be] = 0x6bd7, [0x33c0] = 0x826a,
  [0x33c1] = 0x826b, [0x33c2] = 0x826c, [0x33c3] = 0x826d, [0x33c4] = 0x8271,
  [0x33c5] = 0x8275, [0x33c6] = 0x8276, [0x33c7] = 0x8277, [0x33c8] = 0x8278,
  [0x33c9] = 0x827b, [0x33ca] = 0x827c, [0x33cb] = 0x8280, [0x33cc] = 0x8281,
  [0x33cd] = 0x8283, [0x33ce] = 0x8285, [0x33cf] = 0x8286, [0x33d0] = 0x8287,
  [0x33d1] = 0x8289, [0x33d2] = 0x828c, [0x33d3] = 0x8290, [0x33d4] = 0x8293,
  [0x33d5] = 0x8294, [0x33d6] = 0x8295, [0x33d7] = 0x8296, [0x33d8] = 0x829a,
  [0x33d9] = 0x829b, [0x33da] = 0x829e, [0x33db] = 0x82a0, [0x33dc] = 0x82a2,
  [0x33dd] = 0x82a3, [0x33de] = 0x82a7, [0x33df] = 0x82b2, [0x33e0] = 0x82b5,
  [0x33e1] = 0x82b6, [0x33e2] = 0x82ba, [0x33e3] = 0x82bb, [0x33e4] = 0x82bc,
  [0x33e5] = 0x82bf, [0x33e6] = 0x82c0, [0x33e7] = 0x82c2, [0x33e8] = 0x82c3,
  [0x33e9] = 0x82c5, [0x33ea] = 0x82c6, [0x33eb] = 0x82c9, [0x33ec] = 0x82d0,
  [0x33ed] = 0x82d6, [0x33ee] = 0x82d9, [0x33ef] = 0x82da, [0x33f0] = 0x82dd,
  [0x33f1] = 0x82e2, [0x33f2] = 0x82e7, [0x33f3] = 0x82e8, [0x33f4] = 0x82e9,
  [0x33f5] = 0x82ea, [0x33f6] = 0x82ec, [0x33f7] = 0x82ed, [0x33f8] = 0x82ee,
  [0x33f9] = 0x82f0, [0x33fa] = 0x82f2, [0x33fb] = 0x82f3, [0x33fc] = 0x82f5,
  [0x33fd] = 0x82f6, [0x33fe] = 0x82f8, [0x3400] = 0x82fa, [0x3401] = 0x82fc,
  [0x3402] = 0x82fd, [0x3403] = 0x82fe, [0x3404] = 0x82ff, [0x3405] = 0x8300,
  [0x3406] = 0x830a, [0x3407] = 0x830b, [0x3408] = 0x830d, [0x3409] = 0x8310,
  [0x340a] = 0x8312, [0x340b] = 0x8313, [0x340c] = 0x8316, [0x340d] = 0x8318,
  [0x340e] = 0x8319, [0x340f] = 0x831d, [0x3410] = 0x831e, [0x3411] = 0x831f,
  [0x3412] = 0x8320, [0x3413] = 0x8321, [0x3414] = 0x8322, [0x3415] = 0x8323,
  [0x3416] = 0x8324, [0x3417] = 0x8325, [0x3418] = 0x8326, [0x3419] = 0x8329,
  [0x341a] = 0x832a, [0x341b] = 0x832e, [0x341c] = 0x8330, [0x341d] = 0x8332,
  [0x341e] = 0x8337, [0x341f] = 0x833b, [0x3420] = 0x833d, [0x3421] = 0x5564,
  [0x3422] = 0x813e, [0x3423] = 0x75b2, [0x3424] = 0x76ae, [0x3425] = 0x5339,
  [0x3426] = 0x75de, [0x3427] = 0x50fb, [0x3428] = 0x5c41, [0x3429] = 0x8b6c,
  [0x342a] = 0x7bc7, [0x342b] = 0x504f, [0x342c] = 0x7247, [0x342d] = 0x9a97,
  [0x342e] = 0x98d8, [0x342f] = 0x6f02, [0x3430] = 0x74e2, [0x3431] = 0x7968,
  [0x3432] = 0x6487, [0x3433] = 0x77a5, [0x3434] = 0x62fc, [0x3435] = 0x9891,
  [0x3436] = 0x8d2b, [0x3437] = 0x54c1, [0x3438] = 0x8058, [0x3439] = 0x4e52,
  [0x343a] = 0x576a, [0x343b] = 0x82f9, [0x343c] = 0x840d, [0x343d] = 0x5e73,
  [0x343e] = 0x51ed, [0x343f] = 0x74f6, [0x3440] = 0x8bc4, [0x3441] = 0x5c4f,
  [0x3442] = 0x5761, [0x3443] = 0x6cfc, [0x3444] = 0x9887, [0x3445] = 0x5a46,
  [0x3446] = 0x7834, [0x3447] = 0x9b44, [0x3448] = 0x8feb, [0x3449] = 0x7c95,
  [0x344a] = 0x5256, [0x344b] = 0x6251, [0x344c] = 0x94fa, [0x344d] = 0x4ec6,
  [0x344e] = 0x8386, [0x344f] = 0x8461, [0x3450] = 0x83e9, [0x3451] = 0x84b2,
  [0x3452] = 0x57d4, [0x3453] = 0x6734, [0x3454] = 0x5703, [0x3455] = 0x666e,
  [0x3456] = 0x6d66, [0x3457] = 0x8c31, [0x3458] = 0x66dd, [0x3459] = 0x7011,
  [0x345a] = 0x671f, [0x345b] = 0x6b3a, [0x345c] = 0x6816, [0x345d] = 0x621a,
  [0x345e] = 0x59bb, [0x345f] = 0x4e03, [0x3460] = 0x51c4, [0x3461] = 0x6f06,
  [0x3462] = 0x67d2, [0x3463] = 0x6c8f, [0x3464] = 0x5176, [0x3465] = 0x68cb,
  [0x3466] = 0x5947, [0x3467] = 0x6b67, [0x3468] = 0x7566, [0x3469] = 0x5d0e,
  [0x346a] = 0x8110, [0x346b] = 0x9f50, [0x346c] = 0x65d7, [0x346d] = 0x7948,
  [0x346e] = 0x7941, [0x346f] = 0x9a91, [0x3470] = 0x8d77, [0x3471] = 0x5c82,
  [0x3472] = 0x4e5e, [0x3473] = 0x4f01, [0x3474] = 0x542f, [0x3475] = 0x5951,
  [0x3476] = 0x780c, [0x3477] = 0x5668, [0x3478] = 0x6c14, [0x3479] = 0x8fc4,
  [0x347a] = 0x5f03, [0x347b] = 0x6c7d, [0x347c] = 0x6ce3, [0x347d] = 0x8bab,
  [0x347e] = 0x6390, [0x3480] = 0x833e, [0x3481] = 0x833f, [0x3482] = 0x8341,
  [0x3483] = 0x8342, [0x3484] = 0x8344, [0x3485] = 0x8345, [0x3486] = 0x8348,
  [0x3487] = 0x834a, [0x3488] = 0x834b, [0x3489] = 0x834c, [0x348a] = 0x834d,
  [0x348b] = 0x834e, [0x348c] = 0x8353, [0x348d] = 0x8355, [0x348e] = 0x8356,
  [0x348f] = 0x8357, [0x3490] = 0x8358, [0x3491] = 0x8359, [0x3492] = 0x835d,
  [0x3493] = 0x8362, [0x3494] = 0x8370, [0x3495] = 0x8371, [0x3496] = 0x8372,
  [0x3497] = 0x8373, [0x3498] = 0x8374, [0x3499] = 0x8375, [0x349a] = 0x8376,
  [0x349b] = 0x8379, [0x349c] = 0x837a, [0x349d] = 0x837e, [0x349e] = 0x837f,
  [0x349f] = 0x8380, [0x34a0] = 0x8381, [0x34a1] = 0x8382, [0x34a2] = 0x8383,
  [0x34a3] = 0x8384, [0x34a4] = 0x8387, [0x34a5] = 0x8388, [0x34a6] = 0x838a,
  [0x34a7] = 0x838b, [0x34a8] = 0x838c, [0x34a9] = 0x838d, [0x34aa] = 0x838f,
  [0x34ab] = 0x8390, [0x34ac] = 0x8391, [0x34ad] = 0x8394, [0x34ae] = 0x8395,
  [0x34af] = 0x8396, [0x34b0] = 0x8397, [0x34b1] = 0x8399, [0x34b2] = 0x839a,
  [0x34b3] = 0x839d, [0x34b4] = 0x839f, [0x34b5] = 0x83a1, [0x34b6] = 0x83a2,
  [0x34b7] = 0x83a3, [0x34b8] = 0x83a4, [0x34b9] = 0x83a5, [0x34ba] = 0x83a6,
  [0x34bb] = 0x83a7, [0x34bc] = 0x83ac, [0x34bd] = 0x83ad, [0x34be] = 0x83ae,
  [0x34c0] = 0x83af, [0x34c1] = 0x83b5, [0x34c2] = 0x83bb, [0x34c3] = 0x83be,
  [0x34c4] = 0x83bf, [0x34c5] = 0x83c2, [0x34c6] = 0x83c3, [0x34c7] = 0x83c4,
  [0x34c8] = 0x83c6, [0x34c9] = 0x83c8, [0x34ca] = 0x83c9, [0x34cb] = 0x83cb,
  [0x34cc] = 0x83cd, [0x34cd] = 0x83ce, [0x34ce] = 0x83d0, [0x34cf] = 0x83d1,
  [0x34d0] = 0x83d2, [0x34d1] = 0x83d3, [0x34d2] = 0x83d5, [0x34d3] = 0x83d7,
  [0x34d4] = 0x83d9, [0x34d5] = 0x83da, [0x34d6] = 0x83db, [0x34d7] = 0x83de,
  [0x34d8] = 0x83e2, [0x34d9] = 0x83e3, [0x34da] = 0x83e4, [0x34db] = 0x83e6,
  [0x34dc] = 0x83e7, [0x34dd] = 0x83e8, [0x34de] = 0x83eb, [0x34df] = 0x83ec,
  [0x34e0] = 0x83ed, [0x34e1] = 0x6070, [0x34e2] = 0x6d3d, [0x34e3] = 0x7275,
  [0x34e4] = 0x6266, [0x34e5] = 0x948e, [0x34e6] = 0x94c5, [0x34e7] = 0x5343,
  [0x34e8] = 0x8fc1, [0x34e9] = 0x7b7e, [0x34ea] = 0x4edf, [0x34eb] = 0x8c26,
  [0x34ec] = 0x4e7e, [0x34ed] = 0x9ed4, [0x34ee] = 0x94b1, [0x34ef] = 0x94b3,
  [0x34f0] = 0x524d, [0x34f1] = 0x6f5c, [0x34f2] = 0x9063, [0x34f3] = 0x6d45,
  [0x34f4] = 0x8c34, [0x34f5] = 0x5811, [0x34f6] = 0x5d4c, [0x34f7] = 0x6b20,
  [0x34f8] = 0x6b49, [0x34f9] = 0x67aa, [0x34fa] = 0x545b, [0x34fb] = 0x8154,
  [0x34fc] = 0x7f8c, [0x34fd] = 0x5899, [0x34fe] = 0x8537, [0x34ff] = 0x5f3a,
  [0x3500] = 0x62a2, [0x3501] = 0x6a47, [0x3502] = 0x9539, [0x3503] = 0x6572,
  [0x3504] = 0x6084, [0x3505] = 0x6865, [0x3506] = 0x77a7, [0x3507] = 0x4e54,
  [0x3508] = 0x4fa8, [0x3509] = 0x5de7, [0x350a] = 0x9798, [0x350b] = 0x64ac,
  [0x350c] = 0x7fd8, [0x350d] = 0x5ced, [0x350e] = 0x4fcf, [0x350f] = 0x7a8d,
  [0x3510] = 0x5207, [0x3511] = 0x8304, [0x3512] = 0x4e14, [0x3513] = 0x602f,
  [0x3514] = 0x7a83, [0x3515] = 0x94a6, [0x3516] = 0x4fb5, [0x3517] = 0x4eb2,
  [0x3518] = 0x79e6, [0x3519] = 0x7434, [0x351a] = 0x52e4, [0x351b] = 0x82b9,
  [0x351c] = 0x64d2, [0x351d] = 0x79bd, [0x351e] = 0x5bdd, [0x351f] = 0x6c81,
  [0x3520] = 0x9752, [0x3521] = 0x8f7b, [0x3522] = 0x6c22, [0x3523] = 0x503e,
  [0x3524] = 0x537f, [0x3525] = 0x6e05, [0x3526] = 0x64ce, [0x3527] = 0x6674,
  [0x3528] = 0x6c30, [0x3529] = 0x60c5, [0x352a] = 0x9877, [0x352b] = 0x8bf7,
  [0x352c] = 0x5e86, [0x352d] = 0x743c, [0x352e] = 0x7a77, [0x352f] = 0x79cb,
  [0x3530] = 0x4e18, [0x3531] = 0x90b1, [0x3532] = 0x7403, [0x3533] = 0x6c42,
  [0x3534] = 0x56da, [0x3535] = 0x914b, [0x3536] = 0x6cc5, [0x3537] = 0x8d8b,
  [0x3538] = 0x533a, [0x3539] = 0x86c6, [0x353a] = 0x66f2, [0x353b] = 0x8eaf,
  [0x353c] = 0x5c48, [0x353d] = 0x9a71, [0x353e] = 0x6e20, [0x3540] = 0x83ee,
  [0x3541] = 0x83ef, [0x3542] = 0x83f3, [0x3543] = 0x83f4, [0x3544] = 0x83f5,
  [0x3545] = 0x83f6, [0x3546] = 0x83f7, [0x3547] = 0x83fa, [0x3548] = 0x83fb,
  [0x3549] = 0x83fc, [0x354a] = 0x83fe, [0x354b] = 0x83ff, [0x354c] = 0x8400,
  [0x354d] = 0x8402, [0x354e] = 0x8405, [0x354f] = 0x8407, [0x3550] = 0x8408,
  [0x3551] = 0x8409, [0x3552] = 0x840a, [0x3553] = 0x8410, [0x3554] = 0x8412,
  [0x3555] = 0x8413, [0x3556] = 0x8414, [0x3557] = 0x8415, [0x3558] = 0x8416,
  [0x3559] = 0x8417, [0x355a] = 0x8419, [0x355b] = 0x841a, [0x355c] = 0x841b,
  [0x355d] = 0x841e, [0x355e] = 0x841f, [0x355f] = 0x8420, [0x3560] = 0x8421,
  [0x3561] = 0x8422, [0x3562] = 0x8423, [0x3563] = 0x8429, [0x3564] = 0x842a,
  [0x3565] = 0x842b, [0x3566] = 0x842c, [0x3567] = 0x842d, [0x3568] = 0x842e,
  [0x3569] = 0x842f, [0x356a] = 0x8430, [0x356b] = 0x8432, [0x356c] = 0x8433,
  [0x356d] = 0x8434, [0x356e] = 0x8435, [0x356f] = 0x8436, [0x3570] = 0x8437,
  [0x3571] = 0x8439, [0x3572] = 0x843a, [0x3573] = 0x843b, [0x3574] = 0x843e,
  [0x3575] = 0x843f, [0x3576] = 0x8440, [0x3577] = 0x8441, [0x3578] = 0x8442,
  [0x3579] = 0x8443, [0x357a] = 0x8444, [0x357b] = 0x8445, [0x357c] = 0x8447,
  [0x357d] = 0x8448, [0x357e] = 0x8449, [0x3580] = 0x844a, [0x3581] = 0x844b,
  [0x3582] = 0x844c, [0x3583] = 0x844d, [0x3584] = 0x844e, [0x3585] = 0x844f,
  [0x3586] = 0x8450, [0x3587] = 0x8452, [0x3588] = 0x8453, [0x3589] = 0x8454,
  [0x358a] = 0x8455, [0x358b] = 0x8456, [0x358c] = 0x8458, [0x358d] = 0x845d,
  [0x358e] = 0x845e, [0x358f] = 0x845f, [0x3590] = 0x8460, [0x3591] = 0x8462,
  [0x3592] = 0x8464, [0x3593] = 0x8465, [0x3594] = 0x8466, [0x3595] = 0x8467,
  [0x3596] = 0x8468, [0x3597] = 0x846a, [0x3598] = 0x846e, [0x3599] = 0x846f,
  [0x359a] = 0x8470, [0x359b] = 0x8472, [0x359c] = 0x8474, [0x359d] = 0x8477,
  [0x359e] = 0x8479, [0x359f] = 0x847b, [0x35a0] = 0x847c, [0x35a1] = 0x53d6,
  [0x35a2] = 0x5a36, [0x35a3] = 0x9f8b, [0x35a4] = 0x8da3, [0x35a5] = 0x53bb,
  [0x35a6] = 0x5708, [0x35a7] = 0x98a7, [0x35a8] = 0x6743, [0x35a9] = 0x919b,
  [0x35aa] = 0x6cc9, [0x35ab] = 0x5168, [0x35ac] = 0x75ca, [0x35ad] = 0x62f3,
  [0x35ae] = 0x72ac, [0x35af] = 0x5238, [0x35b0] = 0x529d, [0x35b1] = 0x7f3a,
  [0x35b2] = 0x7094, [0x35b3] = 0x7638, [0x35b4] = 0x5374, [0x35b5] = 0x9e4a,
  [0x35b6] = 0x69b7, [0x35b7] = 0x786e, [0x35b8] = 0x96c0, [0x35b9] = 0x88d9,
  [0x35ba] = 0x7fa4, [0x35bb] = 0x7136, [0x35bc] = 0x71c3, [0x35bd] = 0x5189,
  [0x35be] = 0x67d3, [0x35bf] = 0x74e4, [0x35c0] = 0x58e4, [0x35c1] = 0x6518,
  [0x35c2] = 0x56b7, [0x35c3] = 0x8ba9, [0x35c4] = 0x9976, [0x35c5] = 0x6270,
  [0x35c6] = 0x7ed5, [0x35c7] = 0x60f9, [0x35c8] = 0x70ed, [0x35c9] = 0x58ec,
  [0x35ca] = 0x4ec1, [0x35cb] = 0x4eba, [0x35cc] = 0x5fcd, [0x35cd] = 0x97e7,
  [0x35ce] = 0x4efb, [0x35cf] = 0x8ba4, [0x35d0] = 0x5203, [0x35d1] = 0x598a,
  [0x35d2] = 0x7eab, [0x35d3] = 0x6254, [0x35d4] = 0x4ecd, [0x35d5] = 0x65e5,
  [0x35d6] = 0x620e, [0x35d7] = 0x8338, [0x35d8] = 0x84c9, [0x35d9] = 0x8363,
  [0x35da] = 0x878d, [0x35db] = 0x7194, [0x35dc] = 0x6eb6, [0x35dd] = 0x5bb9,
  [0x35de] = 0x7ed2, [0x35df] = 0x5197, [0x35e0] = 0x63c9, [0x35e1] = 0x67d4,
  [0x35e2] = 0x8089, [0x35e3] = 0x8339, [0x35e4] = 0x8815, [0x35e5] = 0x5112,
  [0x35e6] = 0x5b7a, [0x35e7] = 0x5982, [0x35e8] = 0x8fb1, [0x35e9] = 0x4e73,
  [0x35ea] = 0x6c5d, [0x35eb] = 0x5165, [0x35ec] = 0x8925, [0x35ed] = 0x8f6f,
  [0x35ee] = 0x962e, [0x35ef] = 0x854a, [0x35f0] = 0x745e, [0x35f1] = 0x9510,
  [0x35f2] = 0x95f0, [0x35f3] = 0x6da6, [0x35f4] = 0x82e5, [0x35f5] = 0x5f31,
  [0x35f6] = 0x6492, [0x35f7] = 0x6d12, [0x35f8] = 0x8428, [0x35f9] = 0x816e,
  [0x35fa] = 0x9cc3, [0x35fb] = 0x585e, [0x35fc] = 0x8d5b, [0x35fd] = 0x4e09,
  [0x35fe] = 0x53c1, [0x3600] = 0x847d, [0x3601] = 0x847e, [0x3602] = 0x847f,
  [0x3603] = 0x8480, [0x3604] = 0x8481, [0x3605] = 0x8483, [0x3606] = 0x8484,
  [0x3607] = 0x8485, [0x3608] = 0x8486, [0x3609] = 0x848a, [0x360a] = 0x848d,
  [0x360b] = 0x848f, [0x360c] = 0x8490, [0x360d] = 0x8491, [0x360e] = 0x8492,
  [0x360f] = 0x8493, [0x3610] = 0x8494, [0x3611] = 0x8495, [0x3612] = 0x8496,
  [0x3613] = 0x8498, [0x3614] = 0x849a, [0x3615] = 0x849b, [0x3616] = 0x849d,
  [0x3617] = 0x849e, [0x3618] = 0x849f, [0x3619] = 0x84a0, [0x361a] = 0x84a2,
  [0x361b] = 0x84a3, [0x361c] = 0x84a4, [0x361d] = 0x84a5, [0x361e] = 0x84a6,
  [0x361f] = 0x84a7, [0x3620] = 0x84a8, [0x3621] = 0x84a9, [0x3622] = 0x84aa,
  [0x3623] = 0x84ab, [0x3624] = 0x84ac, [0x3625] = 0x84ad, [0x3626] = 0x84ae,
  [0x3627] = 0x84b0, [0x3628] = 0x84b1, [0x3629] = 0x84b3, [0x362a] = 0x84b5,
  [0x362b] = 0x84b6, [0x362c] = 0x84b7, [0x362d] = 0x84bb, [0x362e] = 0x84bc,
  [0x362f] = 0x84be, [0x3630] = 0x84c0, [0x3631] = 0x84c2, [0x3632] = 0x84c3,
  [0x3633] = 0x84c5, [0x3634] = 0x84c6, [0x3635] = 0x84c7, [0x3636] = 0x84c8,
  [0x3637] = 0x84cb, [0x3638] = 0x84cc, [0x3639] = 0x84ce, [0x363a] = 0x84cf,
  [0x363b] = 0x84d2, [0x363c] = 0x84d4, [0x363d] = 0x84d5, [0x363e] = 0x84d7,
  [0x3640] = 0x84d8, [0x3641] = 0x84d9, [0x3642] = 0x84da, [0x3643] = 0x84db,
  [0x3644] = 0x84dc, [0x3645] = 0x84de, [0x3646] = 0x84e1, [0x3647] = 0x84e2,
  [0x3648] = 0x84e4, [0x3649] = 0x84e7, [0x364a] = 0x84e8, [0x364b] = 0x84e9,
  [0x364c] = 0x84ea, [0x364d] = 0x84eb, [0x364e] = 0x84ed, [0x364f] = 0x84ee,
  [0x3650] = 0x84ef, [0x3651] = 0x84f1, [0x3652] = 0x84f2, [0x3653] = 0x84f3,
  [0x3654] = 0x84f4, [0x3655] = 0x84f5, [0x3656] = 0x84f6, [0x3657] = 0x84f7,
  [0x3658] = 0x84f8, [0x3659] = 0x84f9, [0x365a] = 0x84fa, [0x365b] = 0x84fb,
  [0x365c] = 0x84fd, [0x365d] = 0x84fe, [0x365e] = 0x8500, [0x365f] = 0x8501,
  [0x3660] = 0x8502, [0x3661] = 0x4f1e, [0x3662] = 0x6563, [0x3663] = 0x6851,
  [0x3664] = 0x55d3, [0x3665] = 0x4e27, [0x3666] = 0x6414, [0x3667] = 0x9a9a,
  [0x3668] = 0x626b, [0x3669] = 0x5ac2, [0x366a] = 0x745f, [0x366b] = 0x8272,
  [0x366c] = 0x6da9, [0x366d] = 0x68ee, [0x366e] = 0x50e7, [0x366f] = 0x838e,
  [0x3670] = 0x7802, [0x3671] = 0x6740, [0x3672] = 0x5239, [0x3673] = 0x6c99,
  [0x3674] = 0x7eb1, [0x3675] = 0x50bb, [0x3676] = 0x5565, [0x3677] = 0x715e,
  [0x3678] = 0x7b5b, [0x3679] = 0x6652, [0x367a] = 0x73ca, [0x367b] = 0x82eb,
  [0x367c] = 0x6749, [0x367d] = 0x5c71, [0x367e] = 0x5220, [0x367f] = 0x717d,
  [0x3680] = 0x886b, [0x3681] = 0x95ea, [0x3682] = 0x9655, [0x3683] = 0x64c5,
  [0x3684] = 0x8d61, [0x3685] = 0x81b3, [0x3686] = 0x5584, [0x3687] = 0x6c55,
  [0x3688] = 0x6247, [0x3689] = 0x7f2e, [0x368a] = 0x5892, [0x368b] = 0x4f24,
  [0x368c] = 0x5546, [0x368d] = 0x8d4f, [0x368e] = 0x664c, [0x368f] = 0x4e0a,
  [0x3690] = 0x5c1a, [0x3691] = 0x88f3, [0x3692] = 0x68a2, [0x3693] = 0x634e,
  [0x3694] = 0x7a0d, [0x3695] = 0x70e7, [0x3696] = 0x828d, [0x3697] = 0x52fa,
  [0x3698] = 0x97f6, [0x3699] = 0x5c11, [0x369a] = 0x54e8, [0x369b] = 0x90b5,
  [0x369c] = 0x7ecd, [0x369d] = 0x5962, [0x369e] = 0x8d4a, [0x369f] = 0x86c7,
  [0x36a0] = 0x820c, [0x36a1] = 0x820d, [0x36a2] = 0x8d66, [0x36a3] = 0x6444,
  [0x36a4] = 0x5c04, [0x36a5] = 0x6151, [0x36a6] = 0x6d89, [0x36a7] = 0x793e,
  [0x36a8] = 0x8bbe, [0x36a9] = 0x7837, [0x36aa] = 0x7533, [0x36ab] = 0x547b,
  [0x36ac] = 0x4f38, [0x36ad] = 0x8eab, [0x36ae] = 0x6df1, [0x36af] = 0x5a20,
  [0x36b0] = 0x7ec5, [0x36b1] = 0x795e, [0x36b2] = 0x6c88, [0x36b3] = 0x5ba1,
  [0x36b4] = 0x5a76, [0x36b5] = 0x751a, [0x36b6] = 0x80be, [0x36b7] = 0x614e,
  [0x36b8] = 0x6e17, [0x36b9] = 0x58f0, [0x36ba] = 0x751f, [0x36bb] = 0x7525,
  [0x36bc] = 0x7272, [0x36bd] = 0x5347, [0x36be] = 0x7ef3, [0x36c0] = 0x8503,
  [0x36c1] = 0x8504, [0x36c2] = 0x8505, [0x36c3] = 0x8506, [0x36c4] = 0x8507,
  [0x36c5] = 0x8508, [0x36c6] = 0x8509, [0x36c7] = 0x850a, [0x36c8] = 0x850b,
  [0x36c9] = 0x850d, [0x36ca] = 0x850e, [0x36cb] = 0x850f, [0x36cc] = 0x8510,
  [0x36cd] = 0x8512, [0x36ce] = 0x8514, [0x36cf] = 0x8515, [0x36d0] = 0x8516,
  [0x36d1] = 0x8518, [0x36d2] = 0x8519, [0x36d3] = 0x851b, [0x36d4] = 0x851c,
  [0x36d5] = 0x851d, [0x36d6] = 0x851e, [0x36d7] = 0x8520, [0x36d8] = 0x8522,
  [0x36d9] = 0x8523, [0x36da] = 0x8524, [0x36db] = 0x8525, [0x36dc] = 0x8526,
  [0x36dd] = 0x8527, [0x36de] = 0x8528, [0x36df] = 0x8529, [0x36e0] = 0x852a,
  [0x36e1] = 0x852d, [0x36e2] = 0x852e, [0x36e3] = 0x852f, [0x36e4] = 0x8530,
  [0x36e5] = 0x8531, [0x36e6] = 0x8532, [0x36e7] = 0x8533, [0x36e8] = 0x8534,
  [0x36e9] = 0x8535, [0x36ea] = 0x8536, [0x36eb] = 0x853e, [0x36ec] = 0x853f,
  [0x36ed] = 0x8540, [0x36ee] = 0x8541, [0x36ef] = 0x8542, [0x36f0] = 0x8544,
  [0x36f1] = 0x8545, [0x36f2] = 0x8546, [0x36f3] = 0x8547, [0x36f4] = 0x854b,
  [0x36f5] = 0x854c, [0x36f6] = 0x854d, [0x36f7] = 0x854e, [0x36f8] = 0x854f,
  [0x36f9] = 0x8550, [0x36fa] = 0x8551, [0x36fb] = 0x8552, [0x36fc] = 0x8553,
  [0x36fd] = 0x8554, [0x36fe] = 0x8555, [0x3700] = 0x8557, [0x3701] = 0x8558,
  [0x3702] = 0x855a, [0x3703] = 0x855b, [0x3704] = 0x855c, [0x3705] = 0x855d,
  [0x3706] = 0x855f, [0x3707] = 0x8560, [0x3708] = 0x8561, [0x3709] = 0x8562,
  [0x370a] = 0x8563, [0x370b] = 0x8565, [0x370c] = 0x8566, [0x370d] = 0x8567,
  [0x370e] = 0x8569, [0x370f] = 0x856a, [0x3710] = 0x856b, [0x3711] = 0x856c,
  [0x3712] = 0x856d, [0x3713] = 0x856e, [0x3714] = 0x856f, [0x3715] = 0x8570,
  [0x3716] = 0x8571, [0x3717] = 0x8573, [0x3718] = 0x8575, [0x3719] = 0x8576,
  [0x371a] = 0x8577, [0x371b] = 0x8578, [0x371c] = 0x857c, [0x371d] = 0x857d,
  [0x371e] = 0x857f, [0x371f] = 0x8580, [0x3720] = 0x8581, [0x3721] = 0x7701,
  [0x3722] = 0x76db, [0x3723] = 0x5269, [0x3724] = 0x80dc, [0x3725] = 0x5723,
  [0x3726] = 0x5e08, [0x3727] = 0x5931, [0x3728] = 0x72ee, [0x3729] = 0x65bd,
  [0x372a] = 0x6e7f, [0x372b] = 0x8bd7, [0x372c] = 0x5c38, [0x372d] = 0x8671,
  [0x372e] = 0x5341, [0x372f] = 0x77f3, [0x3730] = 0x62fe, [0x3731] = 0x65f6,
  [0x3732] = 0x4ec0, [0x3733] = 0x98df, [0x3734] = 0x8680, [0x3735] = 0x5b9e,
  [0x3736] = 0x8bc6, [0x3737] = 0x53f2, [0x3738] = 0x77e2, [0x3739] = 0x4f7f,
  [0x373a] = 0x5c4e, [0x373b] = 0x9a76, [0x373c] = 0x59cb, [0x373d] = 0x5f0f,
  [0x373e] = 0x793a, [0x373f] = 0x58eb, [0x3740] = 0x4e16, [0x3741] = 0x67ff,
  [0x3742] = 0x4e8b, [0x3743] = 0x62ed, [0x3744] = 0x8a93, [0x3745] = 0x901d,
  [0x3746] = 0x52bf, [0x3747] = 0x662f, [0x3748] = 0x55dc, [0x3749] = 0x566c,
  [0x374a] = 0x9002, [0x374b] = 0x4ed5, [0x374c] = 0x4f8d, [0x374d] = 0x91ca,
  [0x374e] = 0x9970, [0x374f] = 0x6c0f, [0x3750] = 0x5e02, [0x3751] = 0x6043,
  [0x3752] = 0x5ba4, [0x3753] = 0x89c6, [0x3754] = 0x8bd5, [0x3755] = 0x6536,
  [0x3756] = 0x624b, [0x3757] = 0x9996, [0x3758] = 0x5b88, [0x3759] = 0x5bff,
  [0x375a] = 0x6388, [0x375b] = 0x552e, [0x375c] = 0x53d7, [0x375d] = 0x7626,
  [0x375e] = 0x517d, [0x375f] = 0x852c, [0x3760] = 0x67a2, [0x3761] = 0x68b3,
  [0x3762] = 0x6b8a, [0x3763] = 0x6292, [0x3764] = 0x8f93, [0x3765] = 0x53d4,
  [0x3766] = 0x8212, [0x3767] = 0x6dd1, [0x3768] = 0x758f, [0x3769] = 0x4e66,
  [0x376a] = 0x8d4e, [0x376b] = 0x5b70, [0x376c] = 0x719f, [0x376d] = 0x85af,
  [0x376e] = 0x6691, [0x376f] = 0x66d9, [0x3770] = 0x7f72, [0x3771] = 0x8700,
  [0x3772] = 0x9ecd, [0x3773] = 0x9f20, [0x3774] = 0x5c5e, [0x3775] = 0x672f,
  [0x3776] = 0x8ff0, [0x3777] = 0x6811, [0x3778] = 0x675f, [0x3779] = 0x620d,
  [0x377a] = 0x7ad6, [0x377b] = 0x5885, [0x377c] = 0x5eb6, [0x377d] = 0x6570,
  [0x377e] = 0x6f31, [0x3780] = 0x8582, [0x3781] = 0x8583, [0x3782] = 0x8586,
  [0x3783] = 0x8588, [0x3784] = 0x8589, [0x3785] = 0x858a, [0x3786] = 0x858b,
  [0x3787] = 0x858c, [0x3788] = 0x858d, [0x3789] = 0x858e, [0x378a] = 0x8590,
  [0x378b] = 0x8591, [0x378c] = 0x8592, [0x378d] = 0x8593, [0x378e] = 0x8594,
  [0x378f] = 0x8595, [0x3790] = 0x8596, [0x3791] = 0x8597, [0x3792] = 0x8598,
  [0x3793] = 0x8599, [0x3794] = 0x859a, [0x3795] = 0x859d, [0x3796] = 0x859e,
  [0x3797] = 0x859f, [0x3798] = 0x85a0, [0x3799] = 0x85a1, [0x379a] = 0x85a2,
  [0x379b] = 0x85a3, [0x379c] = 0x85a5, [0x379d] = 0x85a6, [0x379e] = 0x85a7,
  [0x379f] = 0x85a9, [0x37a0] = 0x85ab, [0x37a1] = 0x85ac, [0x37a2] = 0x85ad,
  [0x37a3] = 0x85b1, [0x37a4] = 0x85b2, [0x37a5] = 0x85b3, [0x37a6] = 0x85b4,
  [0x37a7] = 0x85b5, [0x37a8] = 0x85b6, [0x37a9] = 0x85b8, [0x37aa] = 0x85ba,
  [0x37ab] = 0x85bb, [0x37ac] = 0x85bc, [0x37ad] = 0x85bd, [0x37ae] = 0x85be,
  [0x37af] = 0x85bf, [0x37b0] = 0x85c0, [0x37b1] = 0x85c2, [0x37b2] = 0x85c3,
  [0x37b3] = 0x85c4, [0x37b4] = 0x85c5, [0x37b5] = 0x85c6, [0x37b6] = 0x85c7,
  [0x37b7] = 0x85c8, [0x37b8] = 0x85ca, [0x37b9] = 0x85cb, [0x37ba] = 0x85cc,
  [0x37bb] = 0x85cd, [0x37bc] = 0x85ce, [0x37bd] = 0x85d1, [0x37be] = 0x85d2,
  [0x37c0] = 0x85d4, [0x37c1] = 0x85d6, [0x37c2] = 0x85d7, [0x37c3] = 0x85d8,
  [0x37c4] = 0x85d9, [0x37c5] = 0x85da, [0x37c6] = 0x85db, [0x37c7] = 0x85dd,
  [0x37c8] = 0x85de, [0x37c9] = 0x85df, [0x37ca] = 0x85e0, [0x37cb] = 0x85e1,
  [0x37cc] = 0x85e2, [0x37cd] = 0x85e3, [0x37ce] = 0x85e5, [0x37cf] = 0x85e6,
  [0x37d0] = 0x85e7, [0x37d1] = 0x85e8, [0x37d2] = 0x85ea, [0x37d3] = 0x85eb,
  [0x37d4] = 0x85ec, [0x37d5] = 0x85ed, [0x37d6] = 0x85ee, [0x37d7] = 0x85ef,
  [0x37d8] = 0x85f0, [0x37d9] = 0x85f1, [0x37da] = 0x85f2, [0x37db] = 0x85f3,
  [0x37dc] = 0x85f4, [0x37dd] = 0x85f5, [0x37de] = 0x85f6, [0x37df] = 0x85f7,
  [0x37e0] = 0x85f8, [0x37e1] = 0x6055, [0x37e2] = 0x5237, [0x37e3] = 0x800d,
  [0x37e4] = 0x6454, [0x37e5] = 0x8870, [0x37e6] = 0x7529, [0x37e7] = 0x5e05,
  [0x37e8] = 0x6813, [0x37e9] = 0x62f4, [0x37ea] = 0x971c, [0x37eb] = 0x53cc,
  [0x37ec] = 0x723d, [0x37ed] = 0x8c01, [0x37ee] = 0x6c34, [0x37ef] = 0x7761,
  [0x37f0] = 0x7a0e, [0x37f1] = 0x542e, [0x37f2] = 0x77ac, [0x37f3] = 0x987a,
  [0x37f4] = 0x821c, [0x37f5] = 0x8bf4, [0x37f6] = 0x7855, [0x37f7] = 0x6714,
  [0x37f8] = 0x70c1, [0x37f9] = 0x65af, [0x37fa] = 0x6495, [0x37fb] = 0x5636,
  [0x37fc] = 0x601d, [0x37fd] = 0x79c1, [0x37fe] = 0x53f8, [0x37ff] = 0x4e1d,
  [0x3800] = 0x6b7b, [0x3801] = 0x8086, [0x3802] = 0x5bfa, [0x3803] = 0x55e3,
  [0x3804] = 0x56db, [0x3805] = 0x4f3a, [0x3806] = 0x4f3c, [0x3807] = 0x9972,
  [0x3808] = 0x5df3, [0x3809] = 0x677e, [0x380a] = 0x8038, [0x380b] = 0x6002,
  [0x380c] = 0x9882, [0x380d] = 0x9001, [0x380e] = 0x5b8b, [0x380f] = 0x8bbc,
  [0x3810] = 0x8bf5, [0x3811] = 0x641c, [0x3812] = 0x8258, [0x3813] = 0x64de,
  [0x3814] = 0x55fd, [0x3815] = 0x82cf, [0x3816] = 0x9165, [0x3817] = 0x4fd7,
  [0x3818] = 0x7d20, [0x3819] = 0x901f, [0x381a] = 0x7c9f, [0x381b] = 0x50f3,
  [0x381c] = 0x5851, [0x381d] = 0x6eaf, [0x381e] = 0x5bbf, [0x381f] = 0x8bc9,
  [0x3820] = 0x8083, [0x3821] = 0x9178, [0x3822] = 0x849c, [0x3823] = 0x7b97,
  [0x3824] = 0x867d, [0x3825] = 0x968b, [0x3826] = 0x968f, [0x3827] = 0x7ee5,
  [0x3828] = 0x9ad3, [0x3829] = 0x788e, [0x382a] = 0x5c81, [0x382b] = 0x7a57,
  [0x382c] = 0x9042, [0x382d] = 0x96a7, [0x382e] = 0x795f, [0x382f] = 0x5b59,
  [0x3830] = 0x635f, [0x3831] = 0x7b0b, [0x3832] = 0x84d1, [0x3833] = 0x68ad,
  [0x3834] = 0x5506, [0x3835] = 0x7f29, [0x3836] = 0x7410, [0x3837] = 0x7d22,
  [0x3838] = 0x9501, [0x3839] = 0x6240, [0x383a] = 0x584c, [0x383b] = 0x4ed6,
  [0x383c] = 0x5b83, [0x383d] = 0x5979, [0x383e] = 0x5854, [0x3840] = 0x85f9,
  [0x3841] = 0x85fa, [0x3842] = 0x85fc, [0x3843] = 0x85fd, [0x3844] = 0x85fe,
  [0x3845] = 0x8600, [0x3846] = 0x8601, [0x3847] = 0x8602, [0x3848] = 0x8603,
  [0x3849] = 0x8604, [0x384a] = 0x8606, [0x384b] = 0x8607, [0x384c] = 0x8608,
  [0x384d] = 0x8609, [0x384e] = 0x860a, [0x384f] = 0x860b, [0x3850] = 0x860c,
  [0x3851] = 0x860d, [0x3852] = 0x860e, [0x3853] = 0x860f, [0x3854] = 0x8610,
  [0x3855] = 0x8612, [0x3856] = 0x8613, [0x3857] = 0x8614, [0x3858] = 0x8615,
  [0x3859] = 0x8617, [0x385a] = 0x8618, [0x385b] = 0x8619, [0x385c] = 0x861a,
  [0x385d] = 0x861b, [0x385e] = 0x861c, [0x385f] = 0x861d, [0x3860] = 0x861e,
  [0x3861] = 0x861f, [0x3862] = 0x8620, [0x3863] = 0x8621, [0x3864] = 0x8622,
  [0x3865] = 0x8623, [0x3866] = 0x8624, [0x3867] = 0x8625, [0x3868] = 0x8626,
  [0x3869] = 0x8628, [0x386a] = 0x862a, [0x386b] = 0x862b, [0x386c] = 0x862c,
  [0x386d] = 0x862d, [0x386e] = 0x862e, [0x386f] = 0x862f, [0x3870] = 0x8630,
  [0x3871] = 0x8631, [0x3872] = 0x8632, [0x3873] = 0x8633, [0x3874] = 0x8634,
  [0x3875] = 0x8635, [0x3876] = 0x8636, [0x3877] = 0x8637, [0x3878] = 0x8639,
  [0x3879] = 0x863a, [0x387a] = 0x863b, [0x387b] = 0x863d, [0x387c] = 0x863e,
  [0x387d] = 0x863f, [0x387e] = 0x8640, [0x3880] = 0x8641, [0x3881] = 0x8642,
  [0x3882] = 0x8643, [0x3883] = 0x8644, [0x3884] = 0x8645, [0x3885] = 0x8646,
  [0x3886] = 0x8647, [0x3887] = 0x8648, [0x3888] = 0x8649, [0x3889] = 0x864a,
  [0x388a] = 0x864b, [0x388b] = 0x864c, [0x388c] = 0x8652, [0x388d] = 0x8653,
  [0x388e] = 0x8655, [0x388f] = 0x8656, [0x3890] = 0x8657, [0x3891] = 0x8658,
  [0x3892] = 0x8659, [0x3893] = 0x865b, [0x3894] = 0x865c, [0x3895] = 0x865d,
  [0x3896] = 0x865f, [0x3897] = 0x8660, [0x3898] = 0x8661, [0x3899] = 0x8663,
  [0x389a] = 0x8664, [0x389b] = 0x8665, [0x389c] = 0x8666, [0x389d] = 0x8667,
  [0x389e] = 0x8668, [0x389f] = 0x8669, [0x38a0] = 0x866a, [0x38a1] = 0x736d,
  [0x38a2] = 0x631e, [0x38a3] = 0x8e4b, [0x38a4] = 0x8e0f, [0x38a5] = 0x80ce,
  [0x38a6] = 0x82d4, [0x38a7] = 0x62ac, [0x38a8] = 0x53f0, [0x38a9] = 0x6cf0,
  [0x38aa] = 0x915e, [0x38ab] = 0x592a, [0x38ac] = 0x6001, [0x38ad] = 0x6c70,
  [0x38ae] = 0x574d, [0x38af] = 0x644a, [0x38b0] = 0x8d2a, [0x38b1] = 0x762b,
  [0x38b2] = 0x6ee9, [0x38b3] = 0x575b, [0x38b4] = 0x6a80, [0x38b5] = 0x75f0,
  [0x38b6] = 0x6f6d, [0x38b7] = 0x8c2d, [0x38b8] = 0x8c08, [0x38b9] = 0x5766,
  [0x38ba] = 0x6bef, [0x38bb] = 0x8892, [0x38bc] = 0x78b3, [0x38bd] = 0x63a2,
  [0x38be] = 0x53f9, [0x38bf] = 0x70ad, [0x38c0] = 0x6c64, [0x38c1] = 0x5858,
  [0x38c2] = 0x642a, [0x38c3] = 0x5802, [0x38c4] = 0x68e0, [0x38c5] = 0x819b,
  [0x38c6] = 0x5510, [0x38c7] = 0x7cd6, [0x38c8] = 0x5018, [0x38c9] = 0x8eba,
  [0x38ca] = 0x6dcc, [0x38cb] = 0x8d9f, [0x38cc] = 0x70eb, [0x38cd] = 0x638f,
  [0x38ce] = 0x6d9b, [0x38cf] = 0x6ed4, [0x38d0] = 0x7ee6, [0x38d1] = 0x8404,
  [0x38d2] = 0x6843, [0x38d3] = 0x9003, [0x38d4] = 0x6dd8, [0x38d5] = 0x9676,
  [0x38d6] = 0x8ba8, [0x38d7] = 0x5957, [0x38d8] = 0x7279, [0x38d9] = 0x85e4,
  [0x38da] = 0x817e, [0x38db] = 0x75bc, [0x38dc] = 0x8a8a, [0x38dd] = 0x68af,
  [0x38de] = 0x5254, [0x38df] = 0x8e22, [0x38e0] = 0x9511, [0x38e1] = 0x63d0,
  [0x38e2] = 0x9898, [0x38e3] = 0x8e44, [0x38e4] = 0x557c, [0x38e5] = 0x4f53,
  [0x38e6] = 0x66ff, [0x38e7] = 0x568f, [0x38e8] = 0x60d5, [0x38e9] = 0x6d95,
  [0x38ea] = 0x5243, [0x38eb] = 0x5c49, [0x38ec] = 0x5929, [0x38ed] = 0x6dfb,
  [0x38ee] = 0x586b, [0x38ef] = 0x7530, [0x38f0] = 0x751c, [0x38f1] = 0x606c,
  [0x38f2] = 0x8214, [0x38f3] = 0x8146, [0x38f4] = 0x6311, [0x38f5] = 0x6761,
  [0x38f6] = 0x8fe2, [0x38f7] = 0x773a, [0x38f8] = 0x8df3, [0x38f9] = 0x8d34,
  [0x38fa] = 0x94c1, [0x38fb] = 0x5e16, [0x38fc] = 0x5385, [0x38fd] = 0x542c,
  [0x38fe] = 0x70c3, [0x3900] = 0x866d, [0x3901] = 0x866f, [0x3902] = 0x8670,
  [0x3903] = 0x8672, [0x3904] = 0x8673, [0x3905] = 0x8674, [0x3906] = 0x8675,
  [0x3907] = 0x8676, [0x3908] = 0x8677, [0x3909] = 0x8678, [0x390a] = 0x8683,
  [0x390b] = 0x8684, [0x390c] = 0x8685, [0x390d] = 0x8686, [0x390e] = 0x8687,
  [0x390f] = 0x8688, [0x3910] = 0x8689, [0x3911] = 0x868e, [0x3912] = 0x868f,
  [0x3913] = 0x8690, [0x3914] = 0x8691, [0x3915] = 0x8692, [0x3916] = 0x8694,
  [0x3917] = 0x8696, [0x3918] = 0x8697, [0x3919] = 0x8698, [0x391a] = 0x8699,
  [0x391b] = 0x869a, [0x391c] = 0x869b, [0x391d] = 0x869e, [0x391e] = 0x869f,
  [0x391f] = 0x86a0, [0x3920] = 0x86a1, [0x3921] = 0x86a2, [0x3922] = 0x86a5,
  [0x3923] = 0x86a6, [0x3924] = 0x86ab, [0x3925] = 0x86ad, [0x3926] = 0x86ae,
  [0x3927] = 0x86b2, [0x3928] = 0x86b3, [0x3929] = 0x86b7, [0x392a] = 0x86b8,
  [0x392b] = 0x86b9, [0x392c] = 0x86bb, [0x392d] = 0x86bc, [0x392e] = 0x86bd,
  [0x392f] = 0x86be, [0x3930] = 0x86bf, [0x3931] = 0x86c1, [0x3932] = 0x86c2,
  [0x3933] = 0x86c3, [0x3934] = 0x86c5, [0x3935] = 0x86c8, [0x3936] = 0x86cc,
  [0x3937] = 0x86cd, [0x3938] = 0x86d2, [0x3939] = 0x86d3, [0x393a] = 0x86d5,
  [0x393b] = 0x86d6, [0x393c] = 0x86d7, [0x393d] = 0x86da, [0x393e] = 0x86dc,
  [0x3940] = 0x86dd, [0x3941] = 0x86e0, [0x3942] = 0x86e1, [0x3943] = 0x86e2,
  [0x3944] = 0x86e3, [0x3945] = 0x86e5, [0x3946] = 0x86e6, [0x3947] = 0x86e7,
  [0x3948] = 0x86e8, [0x3949] = 0x86ea, [0x394a] = 0x86eb, [0x394b] = 0x86ec,
  [0x394c] = 0x86ef, [0x394d] = 0x86f5, [0x394e] = 0x86f6, [0x394f] = 0x86f7,
  [0x3950] = 0x86fa, [0x3951] = 0x86fb, [0x3952] = 0x86fc, [0x3953] = 0x86fd,
  [0x3954] = 0x86ff, [0x3955] = 0x8701, [0x3956] = 0x8704, [0x3957] = 0x8705,
  [0x3958] = 0x8706, [0x3959] = 0x870b, [0x395a] = 0x870c, [0x395b] = 0x870e,
  [0x395c] = 0x870f, [0x395d] = 0x8710, [0x395e] = 0x8711, [0x395f] = 0x8714,
  [0x3960] = 0x8716, [0x3961] = 0x6c40, [0x3962] = 0x5ef7, [0x3963] = 0x505c,
  [0x3964] = 0x4ead, [0x3965] = 0x5ead, [0x3966] = 0x633a, [0x3967] = 0x8247,
  [0x3968] = 0x901a, [0x3969] = 0x6850, [0x396a] = 0x916e, [0x396b] = 0x77b3,
  [0x396c] = 0x540c, [0x396d] = 0x94dc, [0x396e] = 0x5f64, [0x396f] = 0x7ae5,
  [0x3970] = 0x6876, [0x3971] = 0x6345, [0x3972] = 0x7b52, [0x3973] = 0x7edf,
  [0x3974] = 0x75db, [0x3975] = 0x5077, [0x3976] = 0x6295, [0x3977] = 0x5934,
  [0x3978] = 0x900f, [0x3979] = 0x51f8, [0x397a] = 0x79c3, [0x397b] = 0x7a81,
  [0x397c] = 0x56fe, [0x397d] = 0x5f92, [0x397e] = 0x9014, [0x397f] = 0x6d82,
  [0x3980] = 0x5c60, [0x3981] = 0x571f, [0x3982] = 0x5410, [0x3983] = 0x5154,
  [0x3984] = 0x6e4d, [0x3985] = 0x56e2, [0x3986] = 0x63a8, [0x3987] = 0x9893,
  [0x3988] = 0x817f, [0x3989] = 0x8715, [0x398a] = 0x892a, [0x398b] = 0x9000,
  [0x398c] = 0x541e, [0x398d] = 0x5c6f, [0x398e] = 0x81c0, [0x398f] = 0x62d6,
  [0x3990] = 0x6258, [0x3991] = 0x8131, [0x3992] = 0x9e35, [0x3993] = 0x9640,
  [0x3994] = 0x9a6e, [0x3995] = 0x9a7c, [0x3996] = 0x692d, [0x3997] = 0x59a5,
  [0x3998] = 0x62d3, [0x3999] = 0x553e, [0x399a] = 0x6316, [0x399b] = 0x54c7,
  [0x399c] = 0x86d9, [0x399d] = 0x6d3c, [0x399e] = 0x5a03, [0x399f] = 0x74e6,
  [0x39a0] = 0x889c, [0x39a1] = 0x6b6a, [0x39a2] = 0x5916, [0x39a3] = 0x8c4c,
  [0x39a4] = 0x5f2f, [0x39a5] = 0x6e7e, [0x39a6] = 0x73a9, [0x39a7] = 0x987d,
  [0x39a8] = 0x4e38, [0x39a9] = 0x70f7, [0x39aa] = 0x5b8c, [0x39ab] = 0x7897,
  [0x39ac] = 0x633d, [0x39ad] = 0x665a, [0x39ae] = 0x7696, [0x39af] = 0x60cb,
  [0x39b0] = 0x5b9b, [0x39b1] = 0x5a49, [0x39b2] = 0x4e07, [0x39b3] = 0x8155,
  [0x39b4] = 0x6c6a, [0x39b5] = 0x738b, [0x39b6] = 0x4ea1, [0x39b7] = 0x6789,
  [0x39b8] = 0x7f51, [0x39b9] = 0x5f80, [0x39ba] = 0x65fa, [0x39bb] = 0x671b,
  [0x39bc] = 0x5fd8, [0x39bd] = 0x5984, [0x39be] = 0x5a01, [0x39c0] = 0x8719,
  [0x39c1] = 0x871b, [0x39c2] = 0x871d, [0x39c3] = 0x871f, [0x39c4] = 0x8720,
  [0x39c5] = 0x8724, [0x39c6] = 0x8726, [0x39c7] = 0x8727, [0x39c8] = 0x8728,
  [0x39c9] = 0x872a, [0x39ca] = 0x872b, [0x39cb] = 0x872c, [0x39cc] = 0x872d,
  [0x39cd] = 0x872f, [0x39ce] = 0x8730, [0x39cf] = 0x8732, [0x39d0] = 0x8733,
  [0x39d1] = 0x8735, [0x39d2] = 0x8736, [0x39d3] = 0x8738, [0x39d4] = 0x8739,
  [0x39d5] = 0x873a, [0x39d6] = 0x873c, [0x39d7] = 0x873d, [0x39d8] = 0x8740,
  [0x39d9] = 0x8741, [0x39da] = 0x8742, [0x39db] = 0x8743, [0x39dc] = 0x8744,
  [0x39dd] = 0x8745, [0x39de] = 0x8746, [0x39df] = 0x874a, [0x39e0] = 0x874b,
  [0x39e1] = 0x874d, [0x39e2] = 0x874f, [0x39e3] = 0x8750, [0x39e4] = 0x8751,
  [0x39e5] = 0x8752, [0x39e6] = 0x8754, [0x39e7] = 0x8755, [0x39e8] = 0x8756,
  [0x39e9] = 0x8758, [0x39ea] = 0x875a, [0x39eb] = 0x875b, [0x39ec] = 0x875c,
  [0x39ed] = 0x875d, [0x39ee] = 0x875e, [0x39ef] = 0x875f, [0x39f0] = 0x8761,
  [0x39f1] = 0x8762, [0x39f2] = 0x8766, [0x39f3] = 0x8767, [0x39f4] = 0x8768,
  [0x39f5] = 0x8769, [0x39f6] = 0x876a, [0x39f7] = 0x876b, [0x39f8] = 0x876c,
  [0x39f9] = 0x876d, [0x39fa] = 0x876f, [0x39fb] = 0x8771, [0x39fc] = 0x8772,
  [0x39fd] = 0x8773, [0x39fe] = 0x8775, [0x3a00] = 0x8777, [0x3a01] = 0x8778,
  [0x3a02] = 0x8779, [0x3a03] = 0x877a, [0x3a04] = 0x877f, [0x3a05] = 0x8780,
  [0x3a06] = 0x8781, [0x3a07] = 0x8784, [0x3a08] = 0x8786, [0x3a09] = 0x8787,
  [0x3a0a] = 0x8789, [0x3a0b] = 0x878a, [0x3a0c] = 0x878c, [0x3a0d] = 0x878e,
  [0x3a0e] = 0x878f, [0x3a0f] = 0x8790, [0x3a10] = 0x8791, [0x3a11] = 0x8792,
  [0x3a12] = 0x8794, [0x3a13] = 0x8795, [0x3a14] = 0x8796, [0x3a15] = 0x8798,
  [0x3a16] = 0x8799, [0x3a17] = 0x879a, [0x3a18] = 0x879b, [0x3a19] = 0x879c,
  [0x3a1a] = 0x879d, [0x3a1b] = 0x879e, [0x3a1c] = 0x87a0, [0x3a1d] = 0x87a1,
  [0x3a1e] = 0x87a2, [0x3a1f] = 0x87a3, [0x3a20] = 0x87a4, [0x3a21] = 0x5dcd,
  [0x3a22] = 0x5fae, [0x3a23] = 0x5371, [0x3a24] = 0x97e6, [0x3a25] = 0x8fdd,
  [0x3a26] = 0x6845, [0x3a27] = 0x56f4, [0x3a28] = 0x552f, [0x3a29] = 0x60df,
  [0x3a2a] = 0x4e3a, [0x3a2b] = 0x6f4d, [0x3a2c] = 0x7ef4, [0x3a2d] = 0x82c7,
  [0x3a2e] = 0x840e, [0x3a2f] = 0x59d4, [0x3a30] = 0x4f1f, [0x3a31] = 0x4f2a,
  [0x3a32] = 0x5c3e, [0x3a33] = 0x7eac, [0x3a34] = 0x672a, [0x3a35] = 0x851a,
  [0x3a36] = 0x5473, [0x3a37] = 0x754f, [0x3a38] = 0x80c3, [0x3a39] = 0x5582,
  [0x3a3a] = 0x9b4f, [0x3a3b] = 0x4f4d, [0x3a3c] = 0x6e2d, [0x3a3d] = 0x8c13,
  [0x3a3e] = 0x5c09, [0x3a3f] = 0x6170, [0x3a40] = 0x536b, [0x3a41] = 0x761f,
  [0x3a42] = 0x6e29, [0x3a43] = 0x868a, [0x3a44] = 0x6587, [0x3a45] = 0x95fb,
  [0x3a46] = 0x7eb9, [0x3a47] = 0x543b, [0x3a48] = 0x7a33, [0x3a49] = 0x7d0a,
  [0x3a4a] = 0x95ee, [0x3a4b] = 0x55e1, [0x3a4c] = 0x7fc1, [0x3a4d] = 0x74ee,
  [0x3a4e] = 0x631d, [0x3a4f] = 0x8717, [0x3a50] = 0x6da1, [0x3a51] = 0x7a9d,
  [0x3a52] = 0x6211, [0x3a53] = 0x65a1, [0x3a54] = 0x5367, [0x3a55] = 0x63e1,
  [0x3a56] = 0x6c83, [0x3a57] = 0x5deb, [0x3a58] = 0x545c, [0x3a59] = 0x94a8,
  [0x3a5a] = 0x4e4c, [0x3a5b] = 0x6c61, [0x3a5c] = 0x8bec, [0x3a5d] = 0x5c4b,
  [0x3a5e] = 0x65e0, [0x3a5f] = 0x829c, [0x3a60] = 0x68a7, [0x3a61] = 0x543e,
  [0x3a62] = 0x5434, [0x3a63] = 0x6bcb, [0x3a64] = 0x6b66, [0x3a65] = 0x4e94,
  [0x3a66] = 0x6342, [0x3a67] = 0x5348, [0x3a68] = 0x821e, [0x3a69] = 0x4f0d,
  [0x3a6a] = 0x4fae, [0x3a6b] = 0x575e, [0x3a6c] = 0x620a, [0x3a6d] = 0x96fe,
  [0x3a6e] = 0x6664, [0x3a6f] = 0x7269, [0x3a70] = 0x52ff, [0x3a71] = 0x52a1,
  [0x3a72] = 0x609f, [0x3a73] = 0x8bef, [0x3a74] = 0x6614, [0x3a75] = 0x7199,
  [0x3a76] = 0x6790, [0x3a77] = 0x897f, [0x3a78] = 0x7852, [0x3a79] = 0x77fd,
  [0x3a7a] = 0x6670, [0x3a7b] = 0x563b, [0x3a7c] = 0x5438, [0x3a7d] = 0x9521,
  [0x3a7e] = 0x727a, [0x3a80] = 0x87a5, [0x3a81] = 0x87a6, [0x3a82] = 0x87a7,
  [0x3a83] = 0x87a9, [0x3a84] = 0x87aa, [0x3a85] = 0x87ae, [0x3a86] = 0x87b0,
  [0x3a87] = 0x87b1, [0x3a88] = 0x87b2, [0x3a89] = 0x87b4, [0x3a8a] = 0x87b6,
  [0x3a8b] = 0x87b7, [0x3a8c] = 0x87b8, [0x3a8d] = 0x87b9, [0x3a8e] = 0x87bb,
  [0x3a8f] = 0x87bc, [0x3a90] = 0x87be, [0x3a91] = 0x87bf, [0x3a92] = 0x87c1,
  [0x3a93] = 0x87c2, [0x3a94] = 0x87c3, [0x3a95] = 0x87c4, [0x3a96] = 0x87c5,
  [0x3a97] = 0x87c7, [0x3a98] = 0x87c8, [0x3a99] = 0x87c9, [0x3a9a] = 0x87cc,
  [0x3a9b] = 0x87cd, [0x3a9c] = 0x87ce, [0x3a9d] = 0x87cf, [0x3a9e] = 0x87d0,
  [0x3a9f] = 0x87d4, [0x3aa0] = 0x87d5, [0x3aa1] = 0x87d6, [0x3aa2] = 0x87d7,
  [0x3aa3] = 0x87d8, [0x3aa4] = 0x87d9, [0x3aa5] = 0x87da, [0x3aa6] = 0x87dc,
  [0x3aa7] = 0x87dd, [0x3aa8] = 0x87de, [0x3aa9] = 0x87df, [0x3aaa] = 0x87e1,
  [0x3aab] = 0x87e2, [0x3aac] = 0x87e3, [0x3aad] = 0x87e4, [0x3aae] = 0x87e6,
  [0x3aaf] = 0x87e7, [0x3ab0] = 0x87e8, [0x3ab1] = 0x87e9, [0x3ab2] = 0x87eb,
  [0x3ab3] = 0x87ec, [0x3ab4] = 0x87ed, [0x3ab5] = 0x87ef, [0x3ab6] = 0x87f0,
  [0x3ab7] = 0x87f1, [0x3ab8] = 0x87f2, [0x3ab9] = 0x87f3, [0x3aba] = 0x87f4,
  [0x3abb] = 0x87f5, [0x3abc] = 0x87f6, [0x3abd] = 0x87f7, [0x3abe] = 0x87f8,
  [0x3ac0] = 0x87fa, [0x3ac1] = 0x87fb, [0x3ac2] = 0x87fc, [0x3ac3] = 0x87fd,
  [0x3ac4] = 0x87ff, [0x3ac5] = 0x8800, [0x3ac6] = 0x8801, [0x3ac7] = 0x8802,
  [0x3ac8] = 0x8804, [0x3ac9] = 0x8805, [0x3aca] = 0x8806, [0x3acb] = 0x8807,
  [0x3acc] = 0x8808, [0x3acd] = 0x8809, [0x3ace] = 0x880b, [0x3acf] = 0x880c,
  [0x3ad0] = 0x880d, [0x3ad1] = 0x880e, [0x3ad2] = 0x880f, [0x3ad3] = 0x8810,
  [0x3ad4] = 0x8811, [0x3ad5] = 0x8812, [0x3ad6] = 0x8814, [0x3ad7] = 0x8817,
  [0x3ad8] = 0x8818, [0x3ad9] = 0x8819, [0x3ada] = 0x881a, [0x3adb] = 0x881c,
  [0x3adc] = 0x881d, [0x3add] = 0x881e, [0x3ade] = 0x881f, [0x3adf] = 0x8820,
  [0x3ae0] = 0x8823, [0x3ae1] = 0x7a00, [0x3ae2] = 0x606f, [0x3ae3] = 0x5e0c,
  [0x3ae4] = 0x6089, [0x3ae5] = 0x819d, [0x3ae6] = 0x5915, [0x3ae7] = 0x60dc,
  [0x3ae8] = 0x7184, [0x3ae9] = 0x70ef, [0x3aea] = 0x6eaa, [0x3aeb] = 0x6c50,
  [0x3aec] = 0x7280, [0x3aed] = 0x6a84, [0x3aee] = 0x88ad, [0x3aef] = 0x5e2d,
  [0x3af0] = 0x4e60, [0x3af1] = 0x5ab3, [0x3af2] = 0x559c, [0x3af3] = 0x94e3,
  [0x3af4] = 0x6d17, [0x3af5] = 0x7cfb, [0x3af6] = 0x9699, [0x3af7] = 0x620f,
  [0x3af8] = 0x7ec6, [0x3af9] = 0x778e, [0x3afa] = 0x867e, [0x3afb] = 0x5323,
  [0x3afc] = 0x971e, [0x3afd] = 0x8f96, [0x3afe] = 0x6687, [0x3aff] = 0x5ce1,
  [0x3b00] = 0x4fa0, [0x3b01] = 0x72ed, [0x3b02] = 0x4e0b, [0x3b03] = 0x53a6,
  [0x3b04] = 0x590f, [0x3b05] = 0x5413, [0x3b06] = 0x6380, [0x3b07] = 0x9528,
  [0x3b08] = 0x5148, [0x3b09] = 0x4ed9, [0x3b0a] = 0x9c9c, [0x3b0b] = 0x7ea4,
  [0x3b0c] = 0x54b8, [0x3b0d] = 0x8d24, [0x3b0e] = 0x8854, [0x3b0f] = 0x8237,
  [0x3b10] = 0x95f2, [0x3b11] = 0x6d8e, [0x3b12] = 0x5f26, [0x3b13] = 0x5acc,
  [0x3b14] = 0x663e, [0x3b15] = 0x9669, [0x3b16] = 0x73b0, [0x3b17] = 0x732e,
  [0x3b18] = 0x53bf, [0x3b19] = 0x817a, [0x3b1a] = 0x9985, [0x3b1b] = 0x7fa1,
  [0x3b1c] = 0x5baa, [0x3b1d] = 0x9677, [0x3b1e] = 0x9650, [0x3b1f] = 0x7ebf,
  [0x3b20] = 0x76f8, [0x3b21] = 0x53a2, [0x3b22] = 0x9576, [0x3b23] = 0x9999,
  [0x3b24] = 0x7bb1, [0x3b25] = 0x8944, [0x3b26] = 0x6e58, [0x3b27] = 0x4e61,
  [0x3b28] = 0x7fd4, [0x3b29] = 0x7965, [0x3b2a] = 0x8be6, [0x3b2b] = 0x60f3,
  [0x3b2c] = 0x54cd, [0x3b2d] = 0x4eab, [0x3b2e] = 0x9879, [0x3b2f] = 0x5df7,
  [0x3b30] = 0x6a61, [0x3b31] = 0x50cf, [0x3b32] = 0x5411, [0x3b33] = 0x8c61,
  [0x3b34] = 0x8427, [0x3b35] = 0x785d, [0x3b36] = 0x9704, [0x3b37] = 0x524a,
  [0x3b38] = 0x54ee, [0x3b39] = 0x56a3, [0x3b3a] = 0x9500, [0x3b3b] = 0x6d88,
  [0x3b3c] = 0x5bb5, [0x3b3d] = 0x6dc6, [0x3b3e] = 0x6653, [0x3b40] = 0x8824,
  [0x3b41] = 0x8825, [0x3b42] = 0x8826, [0x3b43] = 0x8827, [0x3b44] = 0x8828,
  [0x3b45] = 0x8829, [0x3b46] = 0x882a, [0x3b47] = 0x882b, [0x3b48] = 0x882c,
  [0x3b49] = 0x882d, [0x3b4a] = 0x882e, [0x3b4b] = 0x882f, [0x3b4c] = 0x8830,
  [0x3b4d] = 0x8831, [0x3b4e] = 0x8833, [0x3b4f] = 0x8834, [0x3b50] = 0x8835,
  [0x3b51] = 0x8836, [0x3b52] = 0x8837, [0x3b53] = 0x8838, [0x3b54] = 0x883a,
  [0x3b55] = 0x883b, [0x3b56] = 0x883d, [0x3b57] = 0x883e, [0x3b58] = 0x883f,
  [0x3b59] = 0x8841, [0x3b5a] = 0x8842, [0x3b5b] = 0x8843, [0x3b5c] = 0x8846,
  [0x3b5d] = 0x8847, [0x3b5e] = 0x8848, [0x3b5f] = 0x8849, [0x3b60] = 0x884a,
  [0x3b61] = 0x884b, [0x3b62] = 0x884e, [0x3b63] = 0x884f, [0x3b64] = 0x8850,
  [0x3b65] = 0x8851, [0x3b66] = 0x8852, [0x3b67] = 0x8853, [0x3b68] = 0x8855,
  [0x3b69] = 0x8856, [0x3b6a] = 0x8858, [0x3b6b] = 0x885a, [0x3b6c] = 0x885b,
  [0x3b6d] = 0x885c, [0x3b6e] = 0x885d, [0x3b6f] = 0x885e, [0x3b70] = 0x885f,
  [0x3b71] = 0x8860, [0x3b72] = 0x8866, [0x3b73] = 0x8867, [0x3b74] = 0x886a,
  [0x3b75] = 0x886d, [0x3b76] = 0x886f, [0x3b77] = 0x8871, [0x3b78] = 0x8873,
  [0x3b79] = 0x8874, [0x3b7a] = 0x8875, [0x3b7b] = 0x8876, [0x3b7c] = 0x8878,
  [0x3b7d] = 0x8879, [0x3b7e] = 0x887a, [0x3b80] = 0x887b, [0x3b81] = 0x887c,
  [0x3b82] = 0x8880, [0x3b83] = 0x8883, [0x3b84] = 0x8886, [0x3b85] = 0x8887,
  [0x3b86] = 0x8889, [0x3b87] = 0x888a, [0x3b88] = 0x888c, [0x3b89] = 0x888e,
  [0x3b8a] = 0x888f, [0x3b8b] = 0x8890, [0x3b8c] = 0x8891, [0x3b8d] = 0x8893,
  [0x3b8e] = 0x8894, [0x3b8f] = 0x8895, [0x3b90] = 0x8897, [0x3b91] = 0x8898,
  [0x3b92] = 0x8899, [0x3b93] = 0x889a, [0x3b94] = 0x889b, [0x3b95] = 0x889d,
  [0x3b96] = 0x889e, [0x3b97] = 0x889f, [0x3b98] = 0x88a0, [0x3b99] = 0x88a1,
  [0x3b9a] = 0x88a3, [0x3b9b] = 0x88a5, [0x3b9c] = 0x88a6, [0x3b9d] = 0x88a7,
  [0x3b9e] = 0x88a8, [0x3b9f] = 0x88a9, [0x3ba0] = 0x88aa, [0x3ba1] = 0x5c0f,
  [0x3ba2] = 0x5b5d, [0x3ba3] = 0x6821, [0x3ba4] = 0x8096, [0x3ba5] = 0x5578,
  [0x3ba6] = 0x7b11, [0x3ba7] = 0x6548, [0x3ba8] = 0x6954, [0x3ba9] = 0x4e9b,
  [0x3baa] = 0x6b47, [0x3bab] = 0x874e, [0x3bac] = 0x978b, [0x3bad] = 0x534f,
  [0x3bae] = 0x631f, [0x3baf] = 0x643a, [0x3bb0] = 0x90aa, [0x3bb1] = 0x659c,
  [0x3bb2] = 0x80c1, [0x3bb3] = 0x8c10, [0x3bb4] = 0x5199, [0x3bb5] = 0x68b0,
  [0x3bb6] = 0x5378, [0x3bb7] = 0x87f9, [0x3bb8] = 0x61c8, [0x3bb9] = 0x6cc4,
  [0x3bba] = 0x6cfb, [0x3bbb] = 0x8c22, [0x3bbc] = 0x5c51, [0x3bbd] = 0x85aa,
  [0x3bbe] = 0x82af, [0x3bbf] = 0x950c, [0x3bc0] = 0x6b23, [0x3bc1] = 0x8f9b,
  [0x3bc2] = 0x65b0, [0x3bc3] = 0x5ffb, [0x3bc4] = 0x5fc3, [0x3bc5] = 0x4fe1,
  [0x3bc6] = 0x8845, [0x3bc7] = 0x661f, [0x3bc8] = 0x8165, [0x3bc9] = 0x7329,
  [0x3bca] = 0x60fa, [0x3bcb] = 0x5174, [0x3bcc] = 0x5211, [0x3bcd] = 0x578b,
  [0x3bce] = 0x5f62, [0x3bcf] = 0x90a2, [0x3bd0] = 0x884c, [0x3bd1] = 0x9192,
  [0x3bd2] = 0x5e78, [0x3bd3] = 0x674f, [0x3bd4] = 0x6027, [0x3bd5] = 0x59d3,
  [0x3bd6] = 0x5144, [0x3bd7] = 0x51f6, [0x3bd8] = 0x80f8, [0x3bd9] = 0x5308,
  [0x3bda] = 0x6c79, [0x3bdb] = 0x96c4, [0x3bdc] = 0x718a, [0x3bdd] = 0x4f11,
  [0x3bde] = 0x4fee, [0x3bdf] = 0x7f9e, [0x3be0] = 0x673d, [0x3be1] = 0x55c5,
  [0x3be2] = 0x9508, [0x3be3] = 0x79c0, [0x3be4] = 0x8896, [0x3be5] = 0x7ee3,
  [0x3be6] = 0x589f, [0x3be7] = 0x620c, [0x3be8] = 0x9700, [0x3be9] = 0x865a,
  [0x3bea] = 0x5618, [0x3beb] = 0x987b, [0x3bec] = 0x5f90, [0x3bed] = 0x8bb8,
  [0x3bee] = 0x84c4, [0x3bef] = 0x9157, [0x3bf0] = 0x53d9, [0x3bf1] = 0x65ed,
  [0x3bf2] = 0x5e8f, [0x3bf3] = 0x755c, [0x3bf4] = 0x6064, [0x3bf5] = 0x7d6e,
  [0x3bf6] = 0x5a7f, [0x3bf7] = 0x7eea, [0x3bf8] = 0x7eed, [0x3bf9] = 0x8f69,
  [0x3bfa] = 0x55a7, [0x3bfb] = 0x5ba3, [0x3bfc] = 0x60ac, [0x3bfd] = 0x65cb,
  [0x3bfe] = 0x7384, [0x3c00] = 0x88ac, [0x3c01] = 0x88ae, [0x3c02] = 0x88af,
  [0x3c03] = 0x88b0, [0x3c04] = 0x88b2, [0x3c05] = 0x88b3, [0x3c06] = 0x88b4,
  [0x3c07] = 0x88b5, [0x3c08] = 0x88b6, [0x3c09] = 0x88b8, [0x3c0a] = 0x88b9,
  [0x3c0b] = 0x88ba, [0x3c0c] = 0x88bb, [0x3c0d] = 0x88bd, [0x3c0e] = 0x88be,
  [0x3c0f] = 0x88bf, [0x3c10] = 0x88c0, [0x3c11] = 0x88c3, [0x3c12] = 0x88c4,
  [0x3c13] = 0x88c7, [0x3c14] = 0x88c8, [0x3c15] = 0x88ca, [0x3c16] = 0x88cb,
  [0x3c17] = 0x88cc, [0x3c18] = 0x88cd, [0x3c19] = 0x88cf, [0x3c1a] = 0x88d0,
  [0x3c1b] = 0x88d1, [0x3c1c] = 0x88d3, [0x3c1d] = 0x88d6, [0x3c1e] = 0x88d7,
  [0x3c1f] = 0x88da, [0x3c20] = 0x88db, [0x3c21] = 0x88dc, [0x3c22] = 0x88dd,
  [0x3c23] = 0x88de, [0x3c24] = 0x88e0, [0x3c25] = 0x88e1, [0x3c26] = 0x88e6,
  [0x3c27] = 0x88e7, [0x3c28] = 0x88e9, [0x3c29] = 0x88ea, [0x3c2a] = 0x88eb,
  [0x3c2b] = 0x88ec, [0x3c2c] = 0x88ed, [0x3c2d] = 0x88ee, [0x3c2e] = 0x88ef,
  [0x3c2f] = 0x88f2, [0x3c30] = 0x88f5, [0x3c31] = 0x88f6, [0x3c32] = 0x88f7,
  [0x3c33] = 0x88fa, [0x3c34] = 0x88fb, [0x3c35] = 0x88fd, [0x3c36] = 0x88ff,
  [0x3c37] = 0x8900, [0x3c38] = 0x8901, [0x3c39] = 0x8903, [0x3c3a] = 0x8904,
  [0x3c3b] = 0x8905, [0x3c3c] = 0x8906, [0x3c3d] = 0x8907, [0x3c3e] = 0x8908,
  [0x3c40] = 0x8909, [0x3c41] = 0x890b, [0x3c42] = 0x890c, [0x3c43] = 0x890d,
  [0x3c44] = 0x890e, [0x3c45] = 0x890f, [0x3c46] = 0x8911, [0x3c47] = 0x8914,
  [0x3c48] = 0x8915, [0x3c49] = 0x8916, [0x3c4a] = 0x8917, [0x3c4b] = 0x8918,
  [0x3c4c] = 0x891c, [0x3c4d] = 0x891d, [0x3c4e] = 0x891e, [0x3c4f] = 0x891f,
  [0x3c50] = 0x8920, [0x3c51] = 0x8922, [0x3c52] = 0x8923, [0x3c53] = 0x8924,
  [0x3c54] = 0x8926, [0x3c55] = 0x8927, [0x3c56] = 0x8928, [0x3c57] = 0x8929,
  [0x3c58] = 0x892c, [0x3c59] = 0x892d, [0x3c5a] = 0x892e, [0x3c5b] = 0x892f,
  [0x3c5c] = 0x8931, [0x3c5d] = 0x8932, [0x3c5e] = 0x8933, [0x3c5f] = 0x8935,
  [0x3c60] = 0x8937, [0x3c61] = 0x9009, [0x3c62] = 0x7663, [0x3c63] = 0x7729,
  [0x3c64] = 0x7eda, [0x3c65] = 0x9774, [0x3c66] = 0x859b, [0x3c67] = 0x5b66,
  [0x3c68] = 0x7a74, [0x3c69] = 0x96ea, [0x3c6a] = 0x8840, [0x3c6b] = 0x52cb,
  [0x3c6c] = 0x718f, [0x3c6d] = 0x5faa, [0x3c6e] = 0x65ec, [0x3c6f] = 0x8be2,
  [0x3c70] = 0x5bfb, [0x3c71] = 0x9a6f, [0x3c72] = 0x5de1, [0x3c73] = 0x6b89,
  [0x3c74] = 0x6c5b, [0x3c75] = 0x8bad, [0x3c76] = 0x8baf, [0x3c77] = 0x900a,
  [0x3c78] = 0x8fc5, [0x3c79] = 0x538b, [0x3c7a] = 0x62bc, [0x3c7b] = 0x9e26,
  [0x3c7c] = 0x9e2d, [0x3c7d] = 0x5440, [0x3c7e] = 0x4e2b, [0x3c7f] = 0x82bd,
  [0x3c80] = 0x7259, [0x3c81] = 0x869c, [0x3c82] = 0x5d16, [0x3c83] = 0x8859,
  [0x3c84] = 0x6daf, [0x3c85] = 0x96c5, [0x3c86] = 0x54d1, [0x3c87] = 0x4e9a,
  [0x3c88] = 0x8bb6, [0x3c89] = 0x7109, [0x3c8a] = 0x54bd, [0x3c8b] = 0x9609,
  [0x3c8c] = 0x70df, [0x3c8d] = 0x6df9, [0x3c8e] = 0x76d0, [0x3c8f] = 0x4e25,
  [0x3c90] = 0x7814, [0x3c91] = 0x8712, [0x3c92] = 0x5ca9, [0x3c93] = 0x5ef6,
  [0x3c94] = 0x8a00, [0x3c95] = 0x989c, [0x3c96] = 0x960e, [0x3c97] = 0x708e,
  [0x3c98] = 0x6cbf, [0x3c99] = 0x5944, [0x3c9a] = 0x63a9, [0x3c9b] = 0x773c,
  [0x3c9c] = 0x884d, [0x3c9d] = 0x6f14, [0x3c9e] = 0x8273, [0x3c9f] = 0x5830,
  [0x3ca0] = 0x71d5, [0x3ca1] = 0x538c, [0x3ca2] = 0x781a, [0x3ca3] = 0x96c1,
  [0x3ca4] = 0x5501, [0x3ca5] = 0x5f66, [0x3ca6] = 0x7130, [0x3ca7] = 0x5bb4,
  [0x3ca8] = 0x8c1a, [0x3ca9] = 0x9a8c, [0x3caa] = 0x6b83, [0x3cab] = 0x592e,
  [0x3cac] = 0x9e2f, [0x3cad] = 0x79e7, [0x3cae] = 0x6768, [0x3caf] = 0x626c,
  [0x3cb0] = 0x4f6f, [0x3cb1] = 0x75a1, [0x3cb2] = 0x7f8a, [0x3cb3] = 0x6d0b,
  [0x3cb4] = 0x9633, [0x3cb5] = 0x6c27, [0x3cb6] = 0x4ef0, [0x3cb7] = 0x75d2,
  [0x3cb8] = 0x517b, [0x3cb9] = 0x6837, [0x3cba] = 0x6f3e, [0x3cbb] = 0x9080,
  [0x3cbc] = 0x8170, [0x3cbd] = 0x5996, [0x3cbe] = 0x7476, [0x3cc0] = 0x8938,
  [0x3cc1] = 0x8939, [0x3cc2] = 0x893a, [0x3cc3] = 0x893b, [0x3cc4] = 0x893c,
  [0x3cc5] = 0x893d, [0x3cc6] = 0x893e, [0x3cc7] = 0x893f, [0x3cc8] = 0x8940,
  [0x3cc9] = 0x8942, [0x3cca] = 0x8943, [0x3ccb] = 0x8945, [0x3ccc] = 0x8946,
  [0x3ccd] = 0x8947, [0x3cce] = 0x8948, [0x3ccf] = 0x8949, [0x3cd0] = 0x894a,
  [0x3cd1] = 0x894b, [0x3cd2] = 0x894c, [0x3cd3] = 0x894d, [0x3cd4] = 0x894e,
  [0x3cd5] = 0x894f, [0x3cd6] = 0x8950, [0x3cd7] = 0x8951, [0x3cd8] = 0x8952,
  [0x3cd9] = 0x8953, [0x3cda] = 0x8954, [0x3cdb] = 0x8955, [0x3cdc] = 0x8956,
  [0x3cdd] = 0x8957, [0x3cde] = 0x8958, [0x3cdf] = 0x8959, [0x3ce0] = 0x895a,
  [0x3ce1] = 0x895b, [0x3ce2] = 0x895c, [0x3ce3] = 0x895d, [0x3ce4] = 0x8960,
  [0x3ce5] = 0x8961, [0x3ce6] = 0x8962, [0x3ce7] = 0x8963, [0x3ce8] = 0x8964,
  [0x3ce9] = 0x8965, [0x3cea] = 0x8967, [0x3ceb] = 0x8968, [0x3cec] = 0x8969,
  [0x3ced] = 0x896a, [0x3cee] = 0x896b, [0x3cef] = 0x896c, [0x3cf0] = 0x896d,
  [0x3cf1] = 0x896e, [0x3cf2] = 0x896f, [0x3cf3] = 0x8970, [0x3cf4] = 0x8971,
  [0x3cf5] = 0x8972, [0x3cf6] = 0x8973, [0x3cf7] = 0x8974, [0x3cf8] = 0x8975,
  [0x3cf9] = 0x8976, [0x3cfa] = 0x8977, [0x3cfb] = 0x8978, [0x3cfc] = 0x8979,
  [0x3cfd] = 0x897a, [0x3cfe] = 0x897c, [0x3d00] = 0x897d, [0x3d01] = 0x897e,
  [0x3d02] = 0x8980, [0x3d03] = 0x8982, [0x3d04] = 0x8984, [0x3d05] = 0x8985,
  [0x3d06] = 0x8987, [0x3d07] = 0x8988, [0x3d08] = 0x8989, [0x3d09] = 0x898a,
  [0x3d0a] = 0x898b, [0x3d0b] = 0x898c, [0x3d0c] = 0x898d, [0x3d0d] = 0x898e,
  [0x3d0e] = 0x898f, [0x3d0f] = 0x8990, [0x3d10] = 0x8991, [0x3d11] = 0x8992,
  [0x3d12] = 0x8993, [0x3d13] = 0x8994, [0x3d14] = 0x8995, [0x3d15] = 0x8996,
  [0x3d16] = 0x8997, [0x3d17] = 0x8998, [0x3d18] = 0x8999, [0x3d19] = 0x899a,
  [0x3d1a] = 0x899b, [0x3d1b] = 0x899c, [0x3d1c] = 0x899d, [0x3d1d] = 0x899e,
  [0x3d1e] = 0x899f, [0x3d1f] = 0x89a0, [0x3d20] = 0x89a1, [0x3d21] = 0x6447,
  [0x3d22] = 0x5c27, [0x3d23] = 0x9065, [0x3d24] = 0x7a91, [0x3d25] = 0x8c23,
  [0x3d26] = 0x59da, [0x3d27] = 0x54ac, [0x3d28] = 0x8200, [0x3d29] = 0x836f,
  [0x3d2a] = 0x8981, [0x3d2b] = 0x8000, [0x3d2c] = 0x6930, [0x3d2d] = 0x564e,
  [0x3d2e] = 0x8036, [0x3d2f] = 0x7237, [0x3d30] = 0x91ce, [0x3d31] = 0x51b6,
  [0x3d32] = 0x4e5f, [0x3d33] = 0x9875, [0x3d34] = 0x6396, [0x3d35] = 0x4e1a,
  [0x3d36] = 0x53f6, [0x3d37] = 0x66f3, [0x3d38] = 0x814b, [0x3d39] = 0x591c,
  [0x3d3a] = 0x6db2, [0x3d3b] = 0x4e00, [0x3d3c] = 0x58f9, [0x3d3d] = 0x533b,
  [0x3d3e] = 0x63d6, [0x3d3f] = 0x94f1, [0x3d40] = 0x4f9d, [0x3d41] = 0x4f0a,
  [0x3d42] = 0x8863, [0x3d43] = 0x9890, [0x3d44] = 0x5937, [0x3d45] = 0x9057,
  [0x3d46] = 0x79fb, [0x3d47] = 0x4eea, [0x3d48] = 0x80f0, [0x3d49] = 0x7591,
  [0x3d4a] = 0x6c82, [0x3d4b] = 0x5b9c, [0x3d4c] = 0x59e8, [0x3d4d] = 0x5f5d,
  [0x3d4e] = 0x6905, [0x3d4f] = 0x8681, [0x3d50] = 0x501a, [0x3d51] = 0x5df2,
  [0x3d52] = 0x4e59, [0x3d53] = 0x77e3, [0x3d54] = 0x4ee5, [0x3d55] = 0x827a,
  [0x3d56] = 0x6291, [0x3d57] = 0x6613, [0x3d58] = 0x9091, [0x3d59] = 0x5c79,
  [0x3d5a] = 0x4ebf, [0x3d5b] = 0x5f79, [0x3d5c] = 0x81c6, [0x3d5d] = 0x9038,
  [0x3d5e] = 0x8084, [0x3d5f] = 0x75ab, [0x3d60] = 0x4ea6, [0x3d61] = 0x88d4,
  [0x3d62] = 0x610f, [0x3d63] = 0x6bc5, [0x3d64] = 0x5fc6, [0x3d65] = 0x4e49,
  [0x3d66] = 0x76ca, [0x3d67] = 0x6ea2, [0x3d68] = 0x8be3, [0x3d69] = 0x8bae,
  [0x3d6a] = 0x8c0a, [0x3d6b] = 0x8bd1, [0x3d6c] = 0x5f02, [0x3d6d] = 0x7ffc,
  [0x3d6e] = 0x7fcc, [0x3d6f] = 0x7ece, [0x3d70] = 0x8335, [0x3d71] = 0x836b,
  [0x3d72] = 0x56e0, [0x3d73] = 0x6bb7, [0x3d74] = 0x97f3, [0x3d75] = 0x9634,
  [0x3d76] = 0x59fb, [0x3d77] = 0x541f, [0x3d78] = 0x94f6, [0x3d79] = 0x6deb,
  [0x3d7a] = 0x5bc5, [0x3d7b] = 0x996e, [0x3d7c] = 0x5c39, [0x3d7d] = 0x5f15,
  [0x3d7e] = 0x9690, [0x3d80] = 0x89a2, [0x3d81] = 0x89a3, [0x3d82] = 0x89a4,
  [0x3d83] = 0x89a5, [0x3d84] = 0x89a6, [0x3d85] = 0x89a7, [0x3d86] = 0x89a8,
  [0x3d87] = 0x89a9, [0x3d88] = 0x89aa, [0x3d89] = 0x89ab, [0x3d8a] = 0x89ac,
  [0x3d8b] = 0x89ad, [0x3d8c] = 0x89ae, [0x3d8d] = 0x89af, [0x3d8e] = 0x89b0,
  [0x3d8f] = 0x89b1, [0x3d90] = 0x89b2, [0x3d91] = 0x89b3, [0x3d92] = 0x89b4,
  [0x3d93] = 0x89b5, [0x3d94] = 0x89b6, [0x3d95] = 0x89b7, [0x3d96] = 0x89b8,
  [0x3d97] = 0x89b9, [0x3d98] = 0x89ba, [0x3d99] = 0x89bb, [0x3d9a] = 0x89bc,
  [0x3d9b] = 0x89bd, [0x3d9c] = 0x89be, [0x3d9d] = 0x89bf, [0x3d9e] = 0x89c0,
  [0x3d9f] = 0x89c3, [0x3da0] = 0x89cd, [0x3da1] = 0x89d3, [0x3da2] = 0x89d4,
  [0x3da3] = 0x89d5, [0x3da4] = 0x89d7, [0x3da5] = 0x89d8, [0x3da6] = 0x89d9,
  [0x3da7] = 0x89db, [0x3da8] = 0x89dd, [0x3da9] = 0x89df, [0x3daa] = 0x89e0,
  [0x3dab] = 0x89e1, [0x3dac] = 0x89e2, [0x3dad] = 0x89e4, [0x3dae] = 0x89e7,
  [0x3daf] = 0x89e8, [0x3db0] = 0x89e9, [0x3db1] = 0x89ea, [0x3db2] = 0x89ec,
  [0x3db3] = 0x89ed, [0x3db4] = 0x89ee, [0x3db5] = 0x89f0, [0x3db6] = 0x89f1,
  [0x3db7] = 0x89f2, [0x3db8] = 0x89f4, [0x3db9] = 0x89f5, [0x3dba] = 0x89f6,
  [0x3dbb] = 0x89f7, [0x3dbc] = 0x89f8, [0x3dbd] = 0x89f9, [0x3dbe] = 0x89fa,
  [0x3dc0] = 0x89fb, [0x3dc1] = 0x89fc, [0x3dc2] = 0x89fd, [0x3dc3] = 0x89fe,
  [0x3dc4] = 0x89ff, [0x3dc5] = 0x8a01, [0x3dc6] = 0x8a02, [0x3dc7] = 0x8a03,
  [0x3dc8] = 0x8a04, [0x3dc9] = 0x8a05, [0x3dca] = 0x8a06, [0x3dcb] = 0x8a08,
  [0x3dcc] = 0x8a09, [0x3dcd] = 0x8a0a, [0x3dce] = 0x8a0b, [0x3dcf] = 0x8a0c,
  [0x3dd0] = 0x8a0d, [0x3dd1] = 0x8a0e, [0x3dd2] = 0x8a0f, [0x3dd3] = 0x8a10,
  [0x3dd4] = 0x8a11, [0x3dd5] = 0x8a12, [0x3dd6] = 0x8a13, [0x3dd7] = 0x8a14,
  [0x3dd8] = 0x8a15, [0x3dd9] = 0x8a16, [0x3dda] = 0x8a17, [0x3ddb] = 0x8a18,
  [0x3ddc] = 0x8a19, [0x3ddd] = 0x8a1a, [0x3dde] = 0x8a1b, [0x3ddf] = 0x8a1c,
  [0x3de0] = 0x8a1d, [0x3de1] = 0x5370, [0x3de2] = 0x82f1, [0x3de3] = 0x6a31,
  [0x3de4] = 0x5a74, [0x3de5] = 0x9e70, [0x3de6] = 0x5e94, [0x3de7] = 0x7f28,
  [0x3de8] = 0x83b9, [0x3de9] = 0x8424, [0x3dea] = 0x8425, [0x3deb] = 0x8367,
  [0x3dec] = 0x8747, [0x3ded] = 0x8fce, [0x3dee] = 0x8d62, [0x3def] = 0x76c8,
  [0x3df0] = 0x5f71, [0x3df1] = 0x9896, [0x3df2] = 0x786c, [0x3df3] = 0x6620,
  [0x3df4] = 0x54df, [0x3df5] = 0x62e5, [0x3df6] = 0x4f63, [0x3df7] = 0x81c3,
  [0x3df8] = 0x75c8, [0x3df9] = 0x5eb8, [0x3dfa] = 0x96cd, [0x3dfb] = 0x8e0a,
  [0x3dfc] = 0x86f9, [0x3dfd] = 0x548f, [0x3dfe] = 0x6cf3, [0x3dff] = 0x6d8c,
  [0x3e00] = 0x6c38, [0x3e01] = 0x607f, [0x3e02] = 0x52c7, [0x3e03] = 0x7528,
  [0x3e04] = 0x5e7d, [0x3e05] = 0x4f18, [0x3e06] = 0x60a0, [0x3e07] = 0x5fe7,
  [0x3e08] = 0x5c24, [0x3e09] = 0x7531, [0x3e0a] = 0x90ae, [0x3e0b] = 0x94c0,
  [0x3e0c] = 0x72b9, [0x3e0d] = 0x6cb9, [0x3e0e] = 0x6e38, [0x3e0f] = 0x9149,
  [0x3e10] = 0x6709, [0x3e11] = 0x53cb, [0x3e12] = 0x53f3, [0x3e13] = 0x4f51,
  [0x3e14] = 0x91c9, [0x3e15] = 0x8bf1, [0x3e16] = 0x53c8, [0x3e17] = 0x5e7c,
  [0x3e18] = 0x8fc2, [0x3e19] = 0x6de4, [0x3e1a] = 0x4e8e, [0x3e1b] = 0x76c2,
  [0x3e1c] = 0x6986, [0x3e1d] = 0x865e, [0x3e1e] = 0x611a, [0x3e1f] = 0x8206,
  [0x3e20] = 0x4f59, [0x3e21] = 0x4fde, [0x3e22] = 0x903e, [0x3e23] = 0x9c7c,
  [0x3e24] = 0x6109, [0x3e25] = 0x6e1d, [0x3e26] = 0x6e14, [0x3e27] = 0x9685,
  [0x3e28] = 0x4e88, [0x3e29] = 0x5a31, [0x3e2a] = 0x96e8, [0x3e2b] = 0x4e0e,
  [0x3e2c] = 0x5c7f, [0x3e2d] = 0x79b9, [0x3e2e] = 0x5b87, [0x3e2f] = 0x8bed,
  [0x3e30] = 0x7fbd, [0x3e31] = 0x7389, [0x3e32] = 0x57df, [0x3e33] = 0x828b,
  [0x3e34] = 0x90c1, [0x3e35] = 0x5401, [0x3e36] = 0x9047, [0x3e37] = 0x55bb,
  [0x3e38] = 0x5cea, [0x3e39] = 0x5fa1, [0x3e3a] = 0x6108, [0x3e3b] = 0x6b32,
  [0x3e3c] = 0x72f1, [0x3e3d] = 0x80b2, [0x3e3e] = 0x8a89, [0x3e40] = 0x8a1e,
  [0x3e41] = 0x8a1f, [0x3e42] = 0x8a20, [0x3e43] = 0x8a21, [0x3e44] = 0x8a22,
  [0x3e45] = 0x8a23, [0x3e46] = 0x8a24, [0x3e47] = 0x8a25, [0x3e48] = 0x8a26,
  [0x3e49] = 0x8a27, [0x3e4a] = 0x8a28, [0x3e4b] = 0x8a29, [0x3e4c] = 0x8a2a,
  [0x3e4d] = 0x8a2b, [0x3e4e] = 0x8a2c, [0x3e4f] = 0x8a2d, [0x3e50] = 0x8a2e,
  [0x3e51] = 0x8a2f, [0x3e52] = 0x8a30, [0x3e53] = 0x8a31, [0x3e54] = 0x8a32,
  [0x3e55] = 0x8a33, [0x3e56] = 0x8a34, [0x3e57] = 0x8a35, [0x3e58] = 0x8a36,
  [0x3e59] = 0x8a37, [0x3e5a] = 0x8a38, [0x3e5b] = 0x8a39, [0x3e5c] = 0x8a3a,
  [0x3e5d] = 0x8a3b, [0x3e5e] = 0x8a3c, [0x3e5f] = 0x8a3d, [0x3e60] = 0x8a3f,
  [0x3e61] = 0x8a40, [0x3e62] = 0x8a41, [0x3e63] = 0x8a42, [0x3e64] = 0x8a43,
  [0x3e65] = 0x8a44, [0x3e66] = 0x8a45, [0x3e67] = 0x8a46, [0x3e68] = 0x8a47,
  [0x3e69] = 0x8a49, [0x3e6a] = 0x8a4a, [0x3e6b] = 0x8a4b, [0x3e6c] = 0x8a4c,
  [0x3e6d] = 0x8a4d, [0x3e6e] = 0x8a4e, [0x3e6f] = 0x8a4f, [0x3e70] = 0x8a50,
  [0x3e71] = 0x8a51, [0x3e72] = 0x8a52, [0x3e73] = 0x8a53, [0x3e74] = 0x8a54,
  [0x3e75] = 0x8a55, [0x3e76] = 0x8a56, [0x3e77] = 0x8a57, [0x3e78] = 0x8a58,
  [0x3e79] = 0x8a59, [0x3e7a] = 0x8a5a, [0x3e7b] = 0x8a5b, [0x3e7c] = 0x8a5c,
  [0x3e7d] = 0x8a5d, [0x3e7e] = 0x8a5e, [0x3e80] = 0x8a5f, [0x3e81] = 0x8a60,
  [0x3e82] = 0x8a61, [0x3e83] = 0x8a62, [0x3e84] = 0x8a63, [0x3e85] = 0x8a64,
  [0x3e86] = 0x8a65, [0x3e87] = 0x8a66, [0x3e88] = 0x8a67, [0x3e89] = 0x8a68,
  [0x3e8a] = 0x8a69, [0x3e8b] = 0x8a6a, [0x3e8c] = 0x8a6b, [0x3e8d] = 0x8a6c,
  [0x3e8e] = 0x8a6d, [0x3e8f] = 0x8a6e, [0x3e90] = 0x8a6f, [0x3e91] = 0x8a70,
  [0x3e92] = 0x8a71, [0x3e93] = 0x8a72, [0x3e94] = 0x8a73, [0x3e95] = 0x8a74,
  [0x3e96] = 0x8a75, [0x3e97] = 0x8a76, [0x3e98] = 0x8a77, [0x3e99] = 0x8a78,
  [0x3e9a] = 0x8a7a, [0x3e9b] = 0x8a7b, [0x3e9c] = 0x8a7c, [0x3e9d] = 0x8a7d,
  [0x3e9e] = 0x8a7e, [0x3e9f] = 0x8a7f, [0x3ea0] = 0x8a80, [0x3ea1] = 0x6d74,
  [0x3ea2] = 0x5bd3, [0x3ea3] = 0x88d5, [0x3ea4] = 0x9884, [0x3ea5] = 0x8c6b,
  [0x3ea6] = 0x9a6d, [0x3ea7] = 0x9e33, [0x3ea8] = 0x6e0a, [0x3ea9] = 0x51a4,
  [0x3eaa] = 0x5143, [0x3eab] = 0x57a3, [0x3eac] = 0x8881, [0x3ead] = 0x539f,
  [0x3eae] = 0x63f4, [0x3eaf] = 0x8f95, [0x3eb0] = 0x56ed, [0x3eb1] = 0x5458,
  [0x3eb2] = 0x5706, [0x3eb3] = 0x733f, [0x3eb4] = 0x6e90, [0x3eb5] = 0x7f18,
  [0x3eb6] = 0x8fdc, [0x3eb7] = 0x82d1, [0x3eb8] = 0x613f, [0x3eb9] = 0x6028,
  [0x3eba] = 0x9662, [0x3ebb] = 0x66f0, [0x3ebc] = 0x7ea6, [0x3ebd] = 0x8d8a,
  [0x3ebe] = 0x8dc3, [0x3ebf] = 0x94a5, [0x3ec0] = 0x5cb3, [0x3ec1] = 0x7ca4,
  [0x3ec2] = 0x6708, [0x3ec3] = 0x60a6, [0x3ec4] = 0x9605, [0x3ec5] = 0x8018,
  [0x3ec6] = 0x4e91, [0x3ec7] = 0x90e7, [0x3ec8] = 0x5300, [0x3ec9] = 0x9668,
  [0x3eca] = 0x5141, [0x3ecb] = 0x8fd0, [0x3ecc] = 0x8574, [0x3ecd] = 0x915d,
  [0x3ece] = 0x6655, [0x3ecf] = 0x97f5, [0x3ed0] = 0x5b55, [0x3ed1] = 0x531d,
  [0x3ed2] = 0x7838, [0x3ed3] = 0x6742, [0x3ed4] = 0x683d, [0x3ed5] = 0x54c9,
  [0x3ed6] = 0x707e, [0x3ed7] = 0x5bb0, [0x3ed8] = 0x8f7d, [0x3ed9] = 0x518d,
  [0x3eda] = 0x5728, [0x3edb] = 0x54b1, [0x3edc] = 0x6512, [0x3edd] = 0x6682,
  [0x3ede] = 0x8d5e, [0x3edf] = 0x8d43, [0x3ee0] = 0x810f, [0x3ee1] = 0x846c,
  [0x3ee2] = 0x906d, [0x3ee3] = 0x7cdf, [0x3ee4] = 0x51ff, [0x3ee5] = 0x85fb,
  [0x3ee6] = 0x67a3, [0x3ee7] = 0x65e9, [0x3ee8] = 0x6fa1, [0x3ee9] = 0x86a4,
  [0x3eea] = 0x8e81, [0x3eeb] = 0x566a, [0x3eec] = 0x9020, [0x3eed] = 0x7682,
  [0x3eee] = 0x7076, [0x3eef] = 0x71e5, [0x3ef0] = 0x8d23, [0x3ef1] = 0x62e9,
  [0x3ef2] = 0x5219, [0x3ef3] = 0x6cfd, [0x3ef4] = 0x8d3c, [0x3ef5] = 0x600e,
  [0x3ef6] = 0x589e, [0x3ef7] = 0x618e, [0x3ef8] = 0x66fe, [0x3ef9] = 0x8d60,
  [0x3efa] = 0x624e, [0x3efb] = 0x55b3, [0x3efc] = 0x6e23, [0x3efd] = 0x672d,
  [0x3efe] = 0x8f67, [0x3f00] = 0x8a81, [0x3f01] = 0x8a82, [0x3f02] = 0x8a83,
  [0x3f03] = 0x8a84, [0x3f04] = 0x8a85, [0x3f05] = 0x8a86, [0x3f06] = 0x8a87,
  [0x3f07] = 0x8a88, [0x3f08] = 0x8a8b, [0x3f09] = 0x8a8c, [0x3f0a] = 0x8a8d,
  [0x3f0b] = 0x8a8e, [0x3f0c] = 0x8a8f, [0x3f0d] = 0x8a90, [0x3f0e] = 0x8a91,
  [0x3f0f] = 0x8a92, [0x3f10] = 0x8a94, [0x3f11] = 0x8a95, [0x3f12] = 0x8a96,
  [0x3f13] = 0x8a97, [0x3f14] = 0x8a98, [0x3f15] = 0x8a99, [0x3f16] = 0x8a9a,
  [0x3f17] = 0x8a9b, [0x3f18] = 0x8a9c, [0x3f19] = 0x8a9d, [0x3f1a] = 0x8a9e,
  [0x3f1b] = 0x8a9f, [0x3f1c] = 0x8aa0, [0x3f1d] = 0x8aa1, [0x3f1e] = 0x8aa2,
  [0x3f1f] = 0x8aa3, [0x3f20] = 0x8aa4, [0x3f21] = 0x8aa5, [0x3f22] = 0x8aa6,
  [0x3f23] = 0x8aa7, [0x3f24] = 0x8aa8, [0x3f25] = 0x8aa9, [0x3f26] = 0x8aaa,
  [0x3f27] = 0x8aab, [0x3f28] = 0x8aac, [0x3f29] = 0x8aad, [0x3f2a] = 0x8aae,
  [0x3f2b] = 0x8aaf, [0x3f2c] = 0x8ab0, [0x3f2d] = 0x8ab1, [0x3f2e] = 0x8ab2,
  [0x3f2f] = 0x8ab3, [0x3f30] = 0x8ab4, [0x3f31] = 0x8ab5, [0x3f32] = 0x8ab6,
  [0x3f33] = 0x8ab7, [0x3f34] = 0x8ab8, [0x3f35] = 0x8ab9, [0x3f36] = 0x8aba,
  [0x3f37] = 0x8abb, [0x3f38] = 0x8abc, [0x3f39] = 0x8abd, [0x3f3a] = 0x8abe,
  [0x3f3b] = 0x8abf, [0x3f3c] = 0x8ac0, [0x3f3d] = 0x8ac1, [0x3f3e] = 0x8ac2,
  [0x3f40] = 0x8ac3, [0x3f41] = 0x8ac4, [0x3f42] = 0x8ac5, [0x3f43] = 0x8ac6,
  [0x3f44] = 0x8ac7, [0x3f45] = 0x8ac8, [0x3f46] = 0x8ac9, [0x3f47] = 0x8aca,
  [0x3f48] = 0x8acb, [0x3f49] = 0x8acc, [0x3f4a] = 0x8acd, [0x3f4b] = 0x8ace,
  [0x3f4c] = 0x8acf, [0x3f4d] = 0x8ad0, [0x3f4e] = 0x8ad1, [0x3f4f] = 0x8ad2,
  [0x3f50] = 0x8ad3, [0x3f51] = 0x8ad4, [0x3f52] = 0x8ad5, [0x3f53] = 0x8ad6,
  [0x3f54] = 0x8ad7, [0x3f55] = 0x8ad8, [0x3f56] = 0x8ad9, [0x3f57] = 0x8ada,
  [0x3f58] = 0x8adb, [0x3f59] = 0x8adc, [0x3f5a] = 0x8add, [0x3f5b] = 0x8ade,
  [0x3f5c] = 0x8adf, [0x3f5d] = 0x8ae0, [0x3f5e] = 0x8ae1, [0x3f5f] = 0x8ae2,
  [0x3f60] = 0x8ae3, [0x3f61] = 0x94e1, [0x3f62] = 0x95f8, [0x3f63] = 0x7728,
  [0x3f64] = 0x6805, [0x3f65] = 0x69a8, [0x3f66] = 0x548b, [0x3f67] = 0x4e4d,
  [0x3f68] = 0x70b8, [0x3f69] = 0x8bc8, [0x3f6a] = 0x6458, [0x3f6b] = 0x658b,
  [0x3f6c] = 0x5b85, [0x3f6d] = 0x7a84, [0x3f6e] = 0x503a, [0x3f6f] = 0x5be8,
  [0x3f70] = 0x77bb, [0x3f71] = 0x6be1, [0x3f72] = 0x8a79, [0x3f73] = 0x7c98,
  [0x3f74] = 0x6cbe, [0x3f75] = 0x76cf, [0x3f76] = 0x65a9, [0x3f77] = 0x8f97,
  [0x3f78] = 0x5d2d, [0x3f79] = 0x5c55, [0x3f7a] = 0x8638, [0x3f7b] = 0x6808,
  [0x3f7c] = 0x5360, [0x3f7d] = 0x6218, [0x3f7e] = 0x7ad9, [0x3f7f] = 0x6e5b,
  [0x3f80] = 0x7efd, [0x3f81] = 0x6a1f, [0x3f82] = 0x7ae0, [0x3f83] = 0x5f70,
  [0x3f84] = 0x6f33, [0x3f85] = 0x5f20, [0x3f86] = 0x638c, [0x3f87] = 0x6da8,
  [0x3f88] = 0x6756, [0x3f89] = 0x4e08, [0x3f8a] = 0x5e10, [0x3f8b] = 0x8d26,
  [0x3f8c] = 0x4ed7, [0x3f8d] = 0x80c0, [0x3f8e] = 0x7634, [0x3f8f] = 0x969c,
  [0x3f90] = 0x62db, [0x3f91] = 0x662d, [0x3f92] = 0x627e, [0x3f93] = 0x6cbc,
  [0x3f94] = 0x8d75, [0x3f95] = 0x7167, [0x3f96] = 0x7f69, [0x3f97] = 0x5146,
  [0x3f98] = 0x8087, [0x3f99] = 0x53ec, [0x3f9a] = 0x906e, [0x3f9b] = 0x6298,
  [0x3f9c] = 0x54f2, [0x3f9d] = 0x86f0, [0x3f9e] = 0x8f99, [0x3f9f] = 0x8005,
  [0x3fa0] = 0x9517, [0x3fa1] = 0x8517, [0x3fa2] = 0x8fd9, [0x3fa3] = 0x6d59,
  [0x3fa4] = 0x73cd, [0x3fa5] = 0x659f, [0x3fa6] = 0x771f, [0x3fa7] = 0x7504,
  [0x3fa8] = 0x7827, [0x3fa9] = 0x81fb, [0x3faa] = 0x8d1e, [0x3fab] = 0x9488,
  [0x3fac] = 0x4fa6, [0x3fad] = 0x6795, [0x3fae] = 0x75b9, [0x3faf] = 0x8bca,
  [0x3fb0] = 0x9707, [0x3fb1] = 0x632f, [0x3fb2] = 0x9547, [0x3fb3] = 0x9635,
  [0x3fb4] = 0x84b8, [0x3fb5] = 0x6323, [0x3fb6] = 0x7741, [0x3fb7] = 0x5f81,
  [0x3fb8] = 0x72f0, [0x3fb9] = 0x4e89, [0x3fba] = 0x6014, [0x3fbb] = 0x6574,
  [0x3fbc] = 0x62ef, [0x3fbd] = 0x6b63, [0x3fbe] = 0x653f, [0x3fc0] = 0x8ae4,
  [0x3fc1] = 0x8ae5, [0x3fc2] = 0x8ae6, [0x3fc3] = 0x8ae7, [0x3fc4] = 0x8ae8,
  [0x3fc5] = 0x8ae9, [0x3fc6] = 0x8aea, [0x3fc7] = 0x8aeb, [0x3fc8] = 0x8aec,
  [0x3fc9] = 0x8aed, [0x3fca] = 0x8aee, [0x3fcb] = 0x8aef, [0x3fcc] = 0x8af0,
  [0x3fcd] = 0x8af1, [0x3fce] = 0x8af2, [0x3fcf] = 0x8af3, [0x3fd0] = 0x8af4,
  [0x3fd1] = 0x8af5, [0x3fd2] = 0x8af6, [0x3fd3] = 0x8af7, [0x3fd4] = 0x8af8,
  [0x3fd5] = 0x8af9, [0x3fd6] = 0x8afa, [0x3fd7] = 0x8afb, [0x3fd8] = 0x8afc,
  [0x3fd9] = 0x8afd, [0x3fda] = 0x8afe, [0x3fdb] = 0x8aff, [0x3fdc] = 0x8b00,
  [0x3fdd] = 0x8b01, [0x3fde] = 0x8b02, [0x3fdf] = 0x8b03, [0x3fe0] = 0x8b04,
  [0x3fe1] = 0x8b05, [0x3fe2] = 0x8b06, [0x3fe3] = 0x8b08, [0x3fe4] = 0x8b09,
  [0x3fe5] = 0x8b0a, [0x3fe6] = 0x8b0b, [0x3fe7] = 0x8b0c, [0x3fe8] = 0x8b0d,
  [0x3fe9] = 0x8b0e, [0x3fea] = 0x8b0f, [0x3feb] = 0x8b10, [0x3fec] = 0x8b11,
  [0x3fed] = 0x8b12, [0x3fee] = 0x8b13, [0x3fef] = 0x8b14, [0x3ff0] = 0x8b15,
  [0x3ff1] = 0x8b16, [0x3ff2] = 0x8b17, [0x3ff3] = 0x8b18, [0x3ff4] = 0x8b19,
  [0x3ff5] = 0x8b1a, [0x3ff6] = 0x8b1b, [0x3ff7] = 0x8b1c, [0x3ff8] = 0x8b1d,
  [0x3ff9] = 0x8b1e, [0x3ffa] = 0x8b1f, [0x3ffb] = 0x8b20, [0x3ffc] = 0x8b21,
  [0x3ffd] = 0x8b22, [0x3ffe] = 0x8b23, [0x4000] = 0x8b24, [0x4001] = 0x8b25,
  [0x4002] = 0x8b27, [0x4003] = 0x8b28, [0x4004] = 0x8b29, [0x4005] = 0x8b2a,
  [0x4006] = 0x8b2b, [0x4007] = 0x8b2c, [0x4008] = 0x8b2d, [0x4009] = 0x8b2e,
  [0x400a] = 0x8b2f, [0x400b] = 0x8b30, [0x400c] = 0x8b31, [0x400d] = 0x8b32,
  [0x400e] = 0x8b33, [0x400f] = 0x8b34, [0x4010] = 0x8b35, [0x4011] = 0x8b36,
  [0x4012] = 0x8b37, [0x4013] = 0x8b38, [0x4014] = 0x8b39, [0x4015] = 0x8b3a,
  [0x4016] = 0x8b3b, [0x4017] = 0x8b3c, [0x4018] = 0x8b3d, [0x4019] = 0x8b3e,
  [0x401a] = 0x8b3f, [0x401b] = 0x8b40, [0x401c] = 0x8b41, [0x401d] = 0x8b42,
  [0x401e] = 0x8b43, [0x401f] = 0x8b44, [0x4020] = 0x8b45, [0x4021] = 0x5e27,
  [0x4022] = 0x75c7, [0x4023] = 0x90d1, [0x4024] = 0x8bc1, [0x4025] = 0x829d,
  [0x4026] = 0x679d, [0x4027] = 0x652f, [0x4028] = 0x5431, [0x4029] = 0x8718,
  [0x402a] = 0x77e5, [0x402b] = 0x80a2, [0x402c] = 0x8102, [0x402d] = 0x6c41,
  [0x402e] = 0x4e4b, [0x402f] = 0x7ec7, [0x4030] = 0x804c, [0x4031] = 0x76f4,
  [0x4032] = 0x690d, [0x4033] = 0x6b96, [0x4034] = 0x6267, [0x4035] = 0x503c,
  [0x4036] = 0x4f84, [0x4037] = 0x5740, [0x4038] = 0x6307, [0x4039] = 0x6b62,
  [0x403a] = 0x8dbe, [0x403b] = 0x53ea, [0x403c] = 0x65e8, [0x403d] = 0x7eb8,
  [0x403e] = 0x5fd7, [0x403f] = 0x631a, [0x4040] = 0x63b7, [0x4041] = 0x81f3,
  [0x4042] = 0x81f4, [0x4043] = 0x7f6e, [0x4044] = 0x5e1c, [0x4045] = 0x5cd9,
  [0x4046] = 0x5236, [0x4047] = 0x667a, [0x4048] = 0x79e9, [0x4049] = 0x7a1a,
  [0x404a] = 0x8d28, [0x404b] = 0x7099, [0x404c] = 0x75d4, [0x404d] = 0x6ede,
  [0x404e] = 0x6cbb, [0x404f] = 0x7a92, [0x4050] = 0x4e2d, [0x4051] = 0x76c5,
  [0x4052] = 0x5fe0, [0x4053] = 0x949f, [0x4054] = 0x8877, [0x4055] = 0x7ec8,
  [0x4056] = 0x79cd, [0x4057] = 0x80bf, [0x4058] = 0x91cd, [0x4059] = 0x4ef2,
  [0x405a] = 0x4f17, [0x405b] = 0x821f, [0x405c] = 0x5468, [0x405d] = 0x5dde,
  [0x405e] = 0x6d32, [0x405f] = 0x8bcc, [0x4060] = 0x7ca5, [0x4061] = 0x8f74,
  [0x4062] = 0x8098, [0x4063] = 0x5e1a, [0x4064] = 0x5492, [0x4065] = 0x76b1,
  [0x4066] = 0x5b99, [0x4067] = 0x663c, [0x4068] = 0x9aa4, [0x4069] = 0x73e0,
  [0x406a] = 0x682a, [0x406b] = 0x86db, [0x406c] = 0x6731, [0x406d] = 0x732a,
  [0x406e] = 0x8bf8, [0x406f] = 0x8bdb, [0x4070] = 0x9010, [0x4071] = 0x7af9,
  [0x4072] = 0x70db, [0x4073] = 0x716e, [0x4074] = 0x62c4, [0x4075] = 0x77a9,
  [0x4076] = 0x5631, [0x4077] = 0x4e3b, [0x4078] = 0x8457, [0x4079] = 0x67f1,
  [0x407a] = 0x52a9, [0x407b] = 0x86c0, [0x407c] = 0x8d2e, [0x407d] = 0x94f8,
  [0x407e] = 0x7b51, [0x4080] = 0x8b46, [0x4081] = 0x8b47, [0x4082] = 0x8b48,
  [0x4083] = 0x8b49, [0x4084] = 0x8b4a, [0x4085] = 0x8b4b, [0x4086] = 0x8b4c,
  [0x4087] = 0x8b4d, [0x4088] = 0x8b4e, [0x4089] = 0x8b4f, [0x408a] = 0x8b50,
  [0x408b] = 0x8b51, [0x408c] = 0x8b52, [0x408d] = 0x8b53, [0x408e] = 0x8b54,
  [0x408f] = 0x8b55, [0x4090] = 0x8b56, [0x4091] = 0x8b57, [0x4092] = 0x8b58,
  [0x4093] = 0x8b59, [0x4094] = 0x8b5a, [0x4095] = 0x8b5b, [0x4096] = 0x8b5c,
  [0x4097] = 0x8b5d, [0x4098] = 0x8b5e, [0x4099] = 0x8b5f, [0x409a] = 0x8b60,
  [0x409b] = 0x8b61, [0x409c] = 0x8b62, [0x409d] = 0x8b63, [0x409e] = 0x8b64,
  [0x409f] = 0x8b65, [0x40a0] = 0x8b67, [0x40a1] = 0x8b68, [0x40a2] = 0x8b69,
  [0x40a3] = 0x8b6a, [0x40a4] = 0x8b6b, [0x40a5] = 0x8b6d, [0x40a6] = 0x8b6e,
  [0x40a7] = 0x8b6f, [0x40a8] = 0x8b70, [0x40a9] = 0x8b71, [0x40aa] = 0x8b72,
  [0x40ab] = 0x8b73, [0x40ac] = 0x8b74, [0x40ad] = 0x8b75, [0x40ae] = 0x8b76,
  [0x40af] = 0x8b77, [0x40b0] = 0x8b78, [0x40b1] = 0x8b79, [0x40b2] = 0x8b7a,
  [0x40b3] = 0x8b7b, [0x40b4] = 0x8b7c, [0x40b5] = 0x8b7d, [0x40b6] = 0x8b7e,
  [0x40b7] = 0x8b7f, [0x40b8] = 0x8b80, [0x40b9] = 0x8b81, [0x40ba] = 0x8b82,
  [0x40bb] = 0x8b83, [0x40bc] = 0x8b84, [0x40bd] = 0x8b85, [0x40be] = 0x8b86,
  [0x40c0] = 0x8b87, [0x40c1] = 0x8b88, [0x40c2] = 0x8b89, [0x40c3] = 0x8b8a,
  [0x40c4] = 0x8b8b, [0x40c5] = 0x8b8c, [0x40c6] = 0x8b8d, [0x40c7] = 0x8b8e,
  [0x40c8] = 0x8b8f, [0x40c9] = 0x8b90, [0x40ca] = 0x8b91, [0x40cb] = 0x8b92,
  [0x40cc] = 0x8b93, [0x40cd] = 0x8b94, [0x40ce] = 0x8b95, [0x40cf] = 0x8b96,
  [0x40d0] = 0x8b97, [0x40d1] = 0x8b98, [0x40d2] = 0x8b99, [0x40d3] = 0x8b9a,
  [0x40d4] = 0x8b9b, [0x40d5] = 0x8b9c, [0x40d6] = 0x8b9d, [0x40d7] = 0x8b9e,
  [0x40d8] = 0x8b9f, [0x40d9] = 0x8bac, [0x40da] = 0x8bb1, [0x40db] = 0x8bbb,
  [0x40dc] = 0x8bc7, [0x40dd] = 0x8bd0, [0x40de] = 0x8bea, [0x40df] = 0x8c09,
  [0x40e0] = 0x8c1e, [0x40e1] = 0x4f4f, [0x40e2] = 0x6ce8, [0x40e3] = 0x795d,
  [0x40e4] = 0x9a7b, [0x40e5] = 0x6293, [0x40e6] = 0x722a, [0x40e7] = 0x62fd,
  [0x40e8] = 0x4e13, [0x40e9] = 0x7816, [0x40ea] = 0x8f6c, [0x40eb] = 0x64b0,
  [0x40ec] = 0x8d5a, [0x40ed] = 0x7bc6, [0x40ee] = 0x6869, [0x40ef] = 0x5e84,
  [0x40f0] = 0x88c5, [0x40f1] = 0x5986, [0x40f2] = 0x649e, [0x40f3] = 0x58ee,
  [0x40f4] = 0x72b6, [0x40f5] = 0x690e, [0x40f6] = 0x9525, [0x40f7] = 0x8ffd,
  [0x40f8] = 0x8d58, [0x40f9] = 0x5760, [0x40fa] = 0x7f00, [0x40fb] = 0x8c06,
  [0x40fc] = 0x51c6, [0x40fd] = 0x6349, [0x40fe] = 0x62d9, [0x40ff] = 0x5353,
  [0x4100] = 0x684c, [0x4101] = 0x7422, [0x4102] = 0x8301, [0x4103] = 0x914c,
  [0x4104] = 0x5544, [0x4105] = 0x7740, [0x4106] = 0x707c, [0x4107] = 0x6d4a,
  [0x4108] = 0x5179, [0x4109] = 0x54a8, [0x410a] = 0x8d44, [0x410b] = 0x59ff,
  [0x410c] = 0x6ecb, [0x410d] = 0x6dc4, [0x410e] = 0x5b5c, [0x410f] = 0x7d2b,
  [0x4110] = 0x4ed4, [0x4111] = 0x7c7d, [0x4112] = 0x6ed3, [0x4113] = 0x5b50,
  [0x4114] = 0x81ea, [0x4115] = 0x6e0d, [0x4116] = 0x5b57, [0x4117] = 0x9b03,
  [0x4118] = 0x68d5, [0x4119] = 0x8e2a, [0x411a] = 0x5b97, [0x411b] = 0x7efc,
  [0x411c] = 0x603b, [0x411d] = 0x7eb5, [0x411e] = 0x90b9, [0x411f] = 0x8d70,
  [0x4120] = 0x594f, [0x4121] = 0x63cd, [0x4122] = 0x79df, [0x4123] = 0x8db3,
  [0x4124] = 0x5352, [0x4125] = 0x65cf, [0x4126] = 0x7956, [0x4127] = 0x8bc5,
  [0x4128] = 0x963b, [0x4129] = 0x7ec4, [0x412a] = 0x94bb, [0x412b] = 0x7e82,
  [0x412c] = 0x5634, [0x412d] = 0x9189, [0x412e] = 0x6700, [0x412f] = 0x7f6a,
  [0x4130] = 0x5c0a, [0x4131] = 0x9075, [0x4132] = 0x6628, [0x4133] = 0x5de6,
  [0x4134] = 0x4f50, [0x4135] = 0x67de, [0x4136] = 0x505a, [0x4137] = 0x4f5c,
  [0x4138] = 0x5750, [0x4139] = 0x5ea7, [0x4140] = 0x8c38, [0x4141] = 0x8c39,
  [0x4142] = 0x8c3a, [0x4143] = 0x8c3b, [0x4144] = 0x8c3c, [0x4145] = 0x8c3d,
  [0x4146] = 0x8c3e, [0x4147] = 0x8c3f, [0x4148] = 0x8c40, [0x4149] = 0x8c42,
  [0x414a] = 0x8c43, [0x414b] = 0x8c44, [0x414c] = 0x8c45, [0x414d] = 0x8c48,
  [0x414e] = 0x8c4a, [0x414f] = 0x8c4b, [0x4150] = 0x8c4d, [0x4151] = 0x8c4e,
  [0x4152] = 0x8c4f, [0x4153] = 0x8c50, [0x4154] = 0x8c51, [0x4155] = 0x8c52,
  [0x4156] = 0x8c53, [0x4157] = 0x8c54, [0x4158] = 0x8c56, [0x4159] = 0x8c57,
  [0x415a] = 0x8c58, [0x415b] = 0x8c59, [0x415c] = 0x8c5b, [0x415d] = 0x8c5c,
  [0x415e] = 0x8c5d, [0x415f] = 0x8c5e, [0x4160] = 0x8c5f, [0x4161] = 0x8c60,
  [0x4162] = 0x8c63, [0x4163] = 0x8c64, [0x4164] = 0x8c65, [0x4165] = 0x8c66,
  [0x4166] = 0x8c67, [0x4167] = 0x8c68, [0x4168] = 0x8c69, [0x4169] = 0x8c6c,
  [0x416a] = 0x8c6d, [0x416b] = 0x8c6e, [0x416c] = 0x8c6f, [0x416d] = 0x8c70,
  [0x416e] = 0x8c71, [0x416f] = 0x8c72, [0x4170] = 0x8c74, [0x4171] = 0x8c75,
  [0x4172] = 0x8c76, [0x4173] = 0x8c77, [0x4174] = 0x8c7b, [0x4175] = 0x8c7c,
  [0x4176] = 0x8c7d, [0x4177] = 0x8c7e, [0x4178] = 0x8c7f, [0x4179] = 0x8c80,
  [0x417a] = 0x8c81, [0x417b] = 0x8c83, [0x417c] = 0x8c84, [0x417d] = 0x8c86,
  [0x417e] = 0x8c87, [0x4180] = 0x8c88, [0x4181] = 0x8c8b, [0x4182] = 0x8c8d,
  [0x4183] = 0x8c8e, [0x4184] = 0x8c8f, [0x4185] = 0x8c90, [0x4186] = 0x8c91,
  [0x4187] = 0x8c92, [0x4188] = 0x8c93, [0x4189] = 0x8c95, [0x418a] = 0x8c96,
  [0x418b] = 0x8c97, [0x418c] = 0x8c99, [0x418d] = 0x8c9a, [0x418e] = 0x8c9b,
  [0x418f] = 0x8c9c, [0x4190] = 0x8c9d, [0x4191] = 0x8c9e, [0x4192] = 0x8c9f,
  [0x4193] = 0x8ca0, [0x4194] = 0x8ca1, [0x4195] = 0x8ca2, [0x4196] = 0x8ca3,
  [0x4197] = 0x8ca4, [0x4198] = 0x8ca5, [0x4199] = 0x8ca6, [0x419a] = 0x8ca7,
  [0x419b] = 0x8ca8, [0x419c] = 0x8ca9, [0x419d] = 0x8caa, [0x419e] = 0x8cab,
  [0x419f] = 0x8cac, [0x41a0] = 0x8cad, [0x41a1] = 0x4e8d, [0x41a2] = 0x4e0c,
  [0x41a3] = 0x5140, [0x41a4] = 0x4e10, [0x41a5] = 0x5eff, [0x41a6] = 0x5345,
  [0x41a7] = 0x4e15, [0x41a8] = 0x4e98, [0x41a9] = 0x4e1e, [0x41aa] = 0x9b32,
  [0x41ab] = 0x5b6c, [0x41ac] = 0x5669, [0x41ad] = 0x4e28, [0x41ae] = 0x79ba,
  [0x41af] = 0x4e3f, [0x41b0] = 0x5315, [0x41b1] = 0x4e47, [0x41b2] = 0x592d,
  [0x41b3] = 0x723b, [0x41b4] = 0x536e, [0x41b5] = 0x6c10, [0x41b6] = 0x56df,
  [0x41b7] = 0x80e4, [0x41b8] = 0x9997, [0x41b9] = 0x6bd3, [0x41ba] = 0x777e,
  [0x41bb] = 0x9f17, [0x41bc] = 0x4e36, [0x41bd] = 0x4e9f, [0x41be] = 0x9f10,
  [0x41bf] = 0x4e5c, [0x41c0] = 0x4e69, [0x41c1] = 0x4e93, [0x41c2] = 0x8288,
  [0x41c3] = 0x5b5b, [0x41c4] = 0x556c, [0x41c5] = 0x560f, [0x41c6] = 0x4ec4,
  [0x41c7] = 0x538d, [0x41c8] = 0x539d, [0x41c9] = 0x53a3, [0x41ca] = 0x53a5,
  [0x41cb] = 0x53ae, [0x41cc] = 0x9765, [0x41cd] = 0x8d5d, [0x41ce] = 0x531a,
  [0x41cf] = 0x53f5, [0x41d0] = 0x5326, [0x41d1] = 0x532e, [0x41d2] = 0x533e,
  [0x41d3] = 0x8d5c, [0x41d4] = 0x5366, [0x41d5] = 0x5363, [0x41d6] = 0x5202,
  [0x41d7] = 0x5208, [0x41d8] = 0x520e, [0x41d9] = 0x522d, [0x41da] = 0x5233,
  [0x41db] = 0x523f, [0x41dc] = 0x5240, [0x41dd] = 0x524c, [0x41de] = 0x525e,
  [0x41df] = 0x5261, [0x41e0] = 0x525c, [0x41e1] = 0x84af, [0x41e2] = 0x527d,
  [0x41e3] = 0x5282, [0x41e4] = 0x5281, [0x41e5] = 0x5290, [0x41e6] = 0x5293,
  [0x41e7] = 0x5182, [0x41e8] = 0x7f54, [0x41e9] = 0x4ebb, [0x41ea] = 0x4ec3,
  [0x41eb] = 0x4ec9, [0x41ec] = 0x4ec2, [0x41ed] = 0x4ee8, [0x41ee] = 0x4ee1,
  [0x41ef] = 0x4eeb, [0x41f0] = 0x4ede, [0x41f1] = 0x4f1b, [0x41f2] = 0x4ef3,
  [0x41f3] = 0x4f22, [0x41f4] = 0x4f64, [0x41f5] = 0x4ef5, [0x41f6] = 0x4f25,
  [0x41f7] = 0x4f27, [0x41f8] = 0x4f09, [0x41f9] = 0x4f2b, [0x41fa] = 0x4f5e,
  [0x41fb] = 0x4f67, [0x41fc] = 0x6538, [0x41fd] = 0x4f5a, [0x41fe] = 0x4f5d,
  [0x4200] = 0x8cae, [0x4201] = 0x8caf, [0x4202] = 0x8cb0, [0x4203] = 0x8cb1,
  [0x4204] = 0x8cb2, [0x4205] = 0x8cb3, [0x4206] = 0x8cb4, [0x4207] = 0x8cb5,
  [0x4208] = 0x8cb6, [0x4209] = 0x8cb7, [0x420a] = 0x8cb8, [0x420b] = 0x8cb9,
  [0x420c] = 0x8cba, [0x420d] = 0x8cbb, [0x420e] = 0x8cbc, [0x420f] = 0x8cbd,
  [0x4210] = 0x8cbe, [0x4211] = 0x8cbf, [0x4212] = 0x8cc0, [0x4213] = 0x8cc1,
  [0x4214] = 0x8cc2, [0x4215] = 0x8cc3, [0x4216] = 0x8cc4, [0x4217] = 0x8cc5,
  [0x4218] = 0x8cc6, [0x4219] = 0x8cc7, [0x421a] = 0x8cc8, [0x421b] = 0x8cc9,
  [0x421c] = 0x8cca, [0x421d] = 0x8ccb, [0x421e] = 0x8ccc, [0x421f] = 0x8ccd,
  [0x4220] = 0x8cce, [0x4221] = 0x8ccf, [0x4222] = 0x8cd0, [0x4223] = 0x8cd1,
  [0x4224] = 0x8cd2, [0x4225] = 0x8cd3, [0x4226] = 0x8cd4, [0x4227] = 0x8cd5,
  [0x4228] = 0x8cd6, [0x4229] = 0x8cd7, [0x422a] = 0x8cd8, [0x422b] = 0x8cd9,
  [0x422c] = 0x8cda, [0x422d] = 0x8cdb, [0x422e] = 0x8cdc, [0x422f] = 0x8cdd,
  [0x4230] = 0x8cde, [0x4231] = 0x8cdf, [0x4232] = 0x8ce0, [0x4233] = 0x8ce1,
  [0x4234] = 0x8ce2, [0x4235] = 0x8ce3, [0x4236] = 0x8ce4, [0x4237] = 0x8ce5,
  [0x4238] = 0x8ce6, [0x4239] = 0x8ce7, [0x423a] = 0x8ce8, [0x423b] = 0x8ce9,
  [0x423c] = 0x8cea, [0x423d] = 0x8ceb, [0x423e] = 0x8cec, [0x4240] = 0x8ced,
  [0x4241] = 0x8cee, [0x4242] = 0x8cef, [0x4243] = 0x8cf0, [0x4244] = 0x8cf1,
  [0x4245] = 0x8cf2, [0x4246] = 0x8cf3, [0x4247] = 0x8cf4, [0x4248] = 0x8cf5,
  [0x4249] = 0x8cf6, [0x424a] = 0x8cf7, [0x424b] = 0x8cf8, [0x424c] = 0x8cf9,
  [0x424d] = 0x8cfa, [0x424e] = 0x8cfb, [0x424f] = 0x8cfc, [0x4250] = 0x8cfd,
  [0x4251] = 0x8cfe, [0x4252] = 0x8cff, [0x4253] = 0x8d00, [0x4254] = 0x8d01,
  [0x4255] = 0x8d02, [0x4256] = 0x8d03, [0x4257] = 0x8d04, [0x4258] = 0x8d05,
  [0x4259] = 0x8d06, [0x425a] = 0x8d07, [0x425b] = 0x8d08, [0x425c] = 0x8d09,
  [0x425d] = 0x8d0a, [0x425e] = 0x8d0b, [0x425f] = 0x8d0c, [0x4260] = 0x8d0d,
  [0x4261] = 0x4f5f, [0x4262] = 0x4f57, [0x4263] = 0x4f32, [0x4264] = 0x4f3d,
  [0x4265] = 0x4f76, [0x4266] = 0x4f74, [0x4267] = 0x4f91, [0x4268] = 0x4f89,
  [0x4269] = 0x4f83, [0x426a] = 0x4f8f, [0x426b] = 0x4f7e, [0x426c] = 0x4f7b,
  [0x426d] = 0x4faa, [0x426e] = 0x4f7c, [0x426f] = 0x4fac, [0x4270] = 0x4f94,
  [0x4271] = 0x4fe6, [0x4272] = 0x4fe8, [0x4273] = 0x4fea, [0x4274] = 0x4fc5,
  [0x4275] = 0x4fda, [0x4276] = 0x4fe3, [0x4277] = 0x4fdc, [0x4278] = 0x4fd1,
  [0x4279] = 0x4fdf, [0x427a] = 0x4ff8, [0x427b] = 0x5029, [0x427c] = 0x504c,
  [0x427d] = 0x4ff3, [0x427e] = 0x502c, [0x427f] = 0x500f, [0x4280] = 0x502e,
  [0x4281] = 0x502d, [0x4282] = 0x4ffe, [0x4283] = 0x501c, [0x4284] = 0x500c,
  [0x4285] = 0x5025, [0x4286] = 0x5028, [0x4287] = 0x507e, [0x4288] = 0x5043,
  [0x4289] = 0x5055, [0x428a] = 0x5048, [0x428b] = 0x504e, [0x428c] = 0x506c,
  [0x428d] = 0x507b, [0x428e] = 0x50a5, [0x428f] = 0x50a7, [0x4290] = 0x50a9,
  [0x4291] = 0x50ba, [0x4292] = 0x50d6, [0x4293] = 0x5106, [0x4294] = 0x50ed,
  [0x4295] = 0x50ec, [0x4296] = 0x50e6, [0x4297] = 0x50ee, [0x4298] = 0x5107,
  [0x4299] = 0x510b, [0x429a] = 0x4edd, [0x429b] = 0x6c3d, [0x429c] = 0x4f58,
  [0x429d] = 0x4f65, [0x429e] = 0x4fce, [0x429f] = 0x9fa0, [0x42a0] = 0x6c46,
  [0x42a1] = 0x7c74, [0x42a2] = 0x516e, [0x42a3] = 0x5dfd, [0x42a4] = 0x9ec9,
  [0x42a5] = 0x9998, [0x42a6] = 0x5181, [0x42a7] = 0x5914, [0x42a8] = 0x52f9,
  [0x42a9] = 0x530d, [0x42aa] = 0x8a07, [0x42ab] = 0x5310, [0x42ac] = 0x51eb,
  [0x42ad] = 0x5919, [0x42ae] = 0x5155, [0x42af] = 0x4ea0, [0x42b0] = 0x5156,
  [0x42b1] = 0x4eb3, [0x42b2] = 0x886e, [0x42b3] = 0x88a4, [0x42b4] = 0x4eb5,
  [0x42b5] = 0x8114, [0x42b6] = 0x88d2, [0x42b7] = 0x7980, [0x42b8] = 0x5b34,
  [0x42b9] = 0x8803, [0x42ba] = 0x7fb8, [0x42bb] = 0x51ab, [0x42bc] = 0x51b1,
  [0x42bd] = 0x51bd, [0x42be] = 0x51bc, [0x42c0] = 0x8d0e, [0x42c1] = 0x8d0f,
  [0x42c2] = 0x8d10, [0x42c3] = 0x8d11, [0x42c4] = 0x8d12, [0x42c5] = 0x8d13,
  [0x42c6] = 0x8d14, [0x42c7] = 0x8d15, [0x42c8] = 0x8d16, [0x42c9] = 0x8d17,
  [0x42ca] = 0x8d18, [0x42cb] = 0x8d19, [0x42cc] = 0x8d1a, [0x42cd] = 0x8d1b,
  [0x42ce] = 0x8d1c, [0x42cf] = 0x8d20, [0x42d0] = 0x8d51, [0x42d1] = 0x8d52,
  [0x42d2] = 0x8d57, [0x42d3] = 0x8d5f, [0x42d4] = 0x8d65, [0x42d5] = 0x8d68,
  [0x42d6] = 0x8d69, [0x42d7] = 0x8d6a, [0x42d8] = 0x8d6c, [0x42d9] = 0x8d6e,
  [0x42da] = 0x8d6f, [0x42db] = 0x8d71, [0x42dc] = 0x8d72, [0x42dd] = 0x8d78,
  [0x42de] = 0x8d79, [0x42df] = 0x8d7a, [0x42e0] = 0x8d7b, [0x42e1] = 0x8d7c,
  [0x42e2] = 0x8d7d, [0x42e3] = 0x8d7e, [0x42e4] = 0x8d7f, [0x42e5] = 0x8d80,
  [0x42e6] = 0x8d82, [0x42e7] = 0x8d83, [0x42e8] = 0x8d86, [0x42e9] = 0x8d87,
  [0x42ea] = 0x8d88, [0x42eb] = 0x8d89, [0x42ec] = 0x8d8c, [0x42ed] = 0x8d8d,
  [0x42ee] = 0x8d8e, [0x42ef] = 0x8d8f, [0x42f0] = 0x8d90, [0x42f1] = 0x8d92,
  [0x42f2] = 0x8d93, [0x42f3] = 0x8d95, [0x42f4] = 0x8d96, [0x42f5] = 0x8d97,
  [0x42f6] = 0x8d98, [0x42f7] = 0x8d99, [0x42f8] = 0x8d9a, [0x42f9] = 0x8d9b,
  [0x42fa] = 0x8d9c, [0x42fb] = 0x8d9d, [0x42fc] = 0x8d9e, [0x42fd] = 0x8da0,
  [0x42fe] = 0x8da1, [0x4300] = 0x8da2, [0x4301] = 0x8da4, [0x4302] = 0x8da5,
  [0x4303] = 0x8da6, [0x4304] = 0x8da7, [0x4305] = 0x8da8, [0x4306] = 0x8da9,
  [0x4307] = 0x8daa, [0x4308] = 0x8dab, [0x4309] = 0x8dac, [0x430a] = 0x8dad,
  [0x430b] = 0x8dae, [0x430c] = 0x8daf, [0x430d] = 0x8db0, [0x430e] = 0x8db2,
  [0x430f] = 0x8db6, [0x4310] = 0x8db7, [0x4311] = 0x8db9, [0x4312] = 0x8dbb,
  [0x4313] = 0x8dbd, [0x4314] = 0x8dc0, [0x4315] = 0x8dc1, [0x4316] = 0x8dc2,
  [0x4317] = 0x8dc5, [0x4318] = 0x8dc7, [0x4319] = 0x8dc8, [0x431a] = 0x8dc9,
  [0x431b] = 0x8dca, [0x431c] = 0x8dcd, [0x431d] = 0x8dd0, [0x431e] = 0x8dd2,
  [0x431f] = 0x8dd3, [0x4320] = 0x8dd4, [0x4321] = 0x51c7, [0x4322] = 0x5196,
  [0x4323] = 0x51a2, [0x4324] = 0x51a5, [0x4325] = 0x8ba0, [0x4326] = 0x8ba6,
  [0x4327] = 0x8ba7, [0x4328] = 0x8baa, [0x4329] = 0x8bb4, [0x432a] = 0x8bb5,
  [0x432b] = 0x8bb7, [0x432c] = 0x8bc2, [0x432d] = 0x8bc3, [0x432e] = 0x8bcb,
  [0x432f] = 0x8bcf, [0x4330] = 0x8bce, [0x4331] = 0x8bd2, [0x4332] = 0x8bd3,
  [0x4333] = 0x8bd4, [0x4334] = 0x8bd6, [0x4335] = 0x8bd8, [0x4336] = 0x8bd9,
  [0x4337] = 0x8bdc, [0x4338] = 0x8bdf, [0x4339] = 0x8be0, [0x433a] = 0x8be4,
  [0x433b] = 0x8be8, [0x433c] = 0x8be9, [0x433d] = 0x8bee, [0x433e] = 0x8bf0,
  [0x433f] = 0x8bf3, [0x4340] = 0x8bf6, [0x4341] = 0x8bf9, [0x4342] = 0x8bfc,
  [0x4343] = 0x8bff, [0x4344] = 0x8c00, [0x4345] = 0x8c02, [0x4346] = 0x8c04,
  [0x4347] = 0x8c07, [0x4348] = 0x8c0c, [0x4349] = 0x8c0f, [0x434a] = 0x8c11,
  [0x434b] = 0x8c12, [0x434c] = 0x8c14, [0x434d] = 0x8c15, [0x434e] = 0x8c16,
  [0x434f] = 0x8c19, [0x4350] = 0x8c1b, [0x4351] = 0x8c18, [0x4352] = 0x8c1d,
  [0x4353] = 0x8c1f, [0x4354] = 0x8c20, [0x4355] = 0x8c21, [0x4356] = 0x8c25,
  [0x4357] = 0x8c27, [0x4358] = 0x8c2a, [0x4359] = 0x8c2b, [0x435a] = 0x8c2e,
  [0x435b] = 0x8c2f, [0x435c] = 0x8c32, [0x435d] = 0x8c33, [0x435e] = 0x8c35,
  [0x435f] = 0x8c36, [0x4360] = 0x5369, [0x4361] = 0x537a, [0x4362] = 0x961d,
  [0x4363] = 0x9622, [0x4364] = 0x9621, [0x4365] = 0x9631, [0x4366] = 0x962a,
  [0x4367] = 0x963d, [0x4368] = 0x963c, [0x4369] = 0x9642, [0x436a] = 0x9649,
  [0x436b] = 0x9654, [0x436c] = 0x965f, [0x436d] = 0x9667, [0x436e] = 0x966c,
  [0x436f] = 0x9672, [0x4370] = 0x9674, [0x4371] = 0x9688, [0x4372] = 0x968d,
  [0x4373] = 0x9697, [0x4374] = 0x96b0, [0x4375] = 0x9097, [0x4376] = 0x909b,
  [0x4377] = 0x909d, [0x4378] = 0x9099, [0x4379] = 0x90ac, [0x437a] = 0x90a1,
  [0x437b] = 0x90b4, [0x437c] = 0x90b3, [0x437d] = 0x90b6, [0x437e] = 0x90ba,
  [0x4380] = 0x8dd5, [0x4381] = 0x8dd8, [0x4382] = 0x8dd9, [0x4383] = 0x8ddc,
  [0x4384] = 0x8de0, [0x4385] = 0x8de1, [0x4386] = 0x8de2, [0x4387] = 0x8de5,
  [0x4388] = 0x8de6, [0x4389] = 0x8de7, [0x438a] = 0x8de9, [0x438b] = 0x8ded,
  [0x438c] = 0x8dee, [0x438d] = 0x8df0, [0x438e] = 0x8df1, [0x438f] = 0x8df2,
  [0x4390] = 0x8df4, [0x4391] = 0x8df6, [0x4392] = 0x8dfc, [0x4393] = 0x8dfe,
  [0x4394] = 0x8dff, [0x4395] = 0x8e00, [0x4396] = 0x8e01, [0x4397] = 0x8e02,
  [0x4398] = 0x8e03, [0x4399] = 0x8e04, [0x439a] = 0x8e06, [0x439b] = 0x8e07,
  [0x439c] = 0x8e08, [0x439d] = 0x8e0b, [0x439e] = 0x8e0d, [0x439f] = 0x8e0e,
  [0x43a0] = 0x8e10, [0x43a1] = 0x8e11, [0x43a2] = 0x8e12, [0x43a3] = 0x8e13,
  [0x43a4] = 0x8e15, [0x43a5] = 0x8e16, [0x43a6] = 0x8e17, [0x43a7] = 0x8e18,
  [0x43a8] = 0x8e19, [0x43a9] = 0x8e1a, [0x43aa] = 0x8e1b, [0x43ab] = 0x8e1c,
  [0x43ac] = 0x8e20, [0x43ad] = 0x8e21, [0x43ae] = 0x8e24, [0x43af] = 0x8e25,
  [0x43b0] = 0x8e26, [0x43b1] = 0x8e27, [0x43b2] = 0x8e28, [0x43b3] = 0x8e2b,
  [0x43b4] = 0x8e2d, [0x43b5] = 0x8e30, [0x43b6] = 0x8e32, [0x43b7] = 0x8e33,
  [0x43b8] = 0x8e34, [0x43b9] = 0x8e36, [0x43ba] = 0x8e37, [0x43bb] = 0x8e38,
  [0x43bc] = 0x8e3b, [0x43bd] = 0x8e3c, [0x43be] = 0x8e3e, [0x43c0] = 0x8e3f,
  [0x43c1] = 0x8e43, [0x43c2] = 0x8e45, [0x43c3] = 0x8e46, [0x43c4] = 0x8e4c,
  [0x43c5] = 0x8e4d, [0x43c6] = 0x8e4e, [0x43c7] = 0x8e4f, [0x43c8] = 0x8e50,
  [0x43c9] = 0x8e53, [0x43ca] = 0x8e54, [0x43cb] = 0x8e55, [0x43cc] = 0x8e56,
  [0x43cd] = 0x8e57, [0x43ce] = 0x8e58, [0x43cf] = 0x8e5a, [0x43d0] = 0x8e5b,
  [0x43d1] = 0x8e5c, [0x43d2] = 0x8e5d, [0x43d3] = 0x8e5e, [0x43d4] = 0x8e5f,
  [0x43d5] = 0x8e60, [0x43d6] = 0x8e61, [0x43d7] = 0x8e62, [0x43d8] = 0x8e63,
  [0x43d9] = 0x8e64, [0x43da] = 0x8e65, [0x43db] = 0x8e67, [0x43dc] = 0x8e68,
  [0x43dd] = 0x8e6a, [0x43de] = 0x8e6b, [0x43df] = 0x8e6e, [0x43e0] = 0x8e71,
  [0x43e1] = 0x90b8, [0x43e2] = 0x90b0, [0x43e3] = 0x90cf, [0x43e4] = 0x90c5,
  [0x43e5] = 0x90be, [0x43e6] = 0x90d0, [0x43e7] = 0x90c4, [0x43e8] = 0x90c7,
  [0x43e9] = 0x90d3, [0x43ea] = 0x90e6, [0x43eb] = 0x90e2, [0x43ec] = 0x90dc,
  [0x43ed] = 0x90d7, [0x43ee] = 0x90db, [0x43ef] = 0x90eb, [0x43f0] = 0x90ef,
  [0x43f1] = 0x90fe, [0x43f2] = 0x9104, [0x43f3] = 0x9122, [0x43f4] = 0x911e,
  [0x43f5] = 0x9123, [0x43f6] = 0x9131, [0x43f7] = 0x912f, [0x43f8] = 0x9139,
  [0x43f9] = 0x9143, [0x43fa] = 0x9146, [0x43fb] = 0x520d, [0x43fc] = 0x5942,
  [0x43fd] = 0x52a2, [0x43fe] = 0x52ac, [0x43ff] = 0x52ad, [0x4400] = 0x52be,
  [0x4401] = 0x54ff, [0x4402] = 0x52d0, [0x4403] = 0x52d6, [0x4404] = 0x52f0,
  [0x4405] = 0x53df, [0x4406] = 0x71ee, [0x4407] = 0x77cd, [0x4408] = 0x5ef4,
  [0x4409] = 0x51f5, [0x440a] = 0x51fc, [0x440b] = 0x9b2f, [0x440c] = 0x53b6,
  [0x440d] = 0x5f01, [0x440e] = 0x755a, [0x440f] = 0x5def, [0x4410] = 0x574c,
  [0x4411] = 0x57a9, [0x4412] = 0x57a1, [0x4413] = 0x587e, [0x4414] = 0x58bc,
  [0x4415] = 0x58c5, [0x4416] = 0x58d1, [0x4417] = 0x5729, [0x4418] = 0x572c,
  [0x4419] = 0x572a, [0x441a] = 0x5733, [0x441b] = 0x5739, [0x441c] = 0x572e,
  [0x441d] = 0x572f, [0x441e] = 0x575c, [0x441f] = 0x573b, [0x4420] = 0x5742,
  [0x4421] = 0x5769, [0x4422] = 0x5785, [0x4423] = 0x576b, [0x4424] = 0x5786,
  [0x4425] = 0x577c, [0x4426] = 0x577b, [0x4427] = 0x5768, [0x4428] = 0x576d,
  [0x4429] = 0x5776, [0x442a] = 0x5773, [0x442b] = 0x57ad, [0x442c] = 0x57a4,
  [0x442d] = 0x578c, [0x442e] = 0x57b2, [0x442f] = 0x57cf, [0x4430] = 0x57a7,
  [0x4431] = 0x57b4, [0x4432] = 0x5793, [0x4433] = 0x57a0, [0x4434] = 0x57d5,
  [0x4435] = 0x57d8, [0x4436] = 0x57da, [0x4437] = 0x57d9, [0x4438] = 0x57d2,
  [0x4439] = 0x57b8, [0x443a] = 0x57f4, [0x443b] = 0x57ef, [0x443c] = 0x57f8,
  [0x443d] = 0x57e4, [0x443e] = 0x57dd, [0x4440] = 0x8e73, [0x4441] = 0x8e75,
  [0x4442] = 0x8e77, [0x4443] = 0x8e78, [0x4444] = 0x8e79, [0x4445] = 0x8e7a,
  [0x4446] = 0x8e7b, [0x4447] = 0x8e7d, [0x4448] = 0x8e7e, [0x4449] = 0x8e80,
  [0x444a] = 0x8e82, [0x444b] = 0x8e83, [0x444c] = 0x8e84, [0x444d] = 0x8e86,
  [0x444e] = 0x8e88, [0x444f] = 0x8e89, [0x4450] = 0x8e8a, [0x4451] = 0x8e8b,
  [0x4452] = 0x8e8c, [0x4453] = 0x8e8d, [0x4454] = 0x8e8e, [0x4455] = 0x8e91,
  [0x4456] = 0x8e92, [0x4457] = 0x8e93, [0x4458] = 0x8e95, [0x4459] = 0x8e96,
  [0x445a] = 0x8e97, [0x445b] = 0x8e98, [0x445c] = 0x8e99, [0x445d] = 0x8e9a,
  [0x445e] = 0x8e9b, [0x445f] = 0x8e9d, [0x4460] = 0x8e9f, [0x4461] = 0x8ea0,
  [0x4462] = 0x8ea1, [0x4463] = 0x8ea2, [0x4464] = 0x8ea3, [0x4465] = 0x8ea4,
  [0x4466] = 0x8ea5, [0x4467] = 0x8ea6, [0x4468] = 0x8ea7, [0x4469] = 0x8ea8,
  [0x446a] = 0x8ea9, [0x446b] = 0x8eaa, [0x446c] = 0x8ead, [0x446d] = 0x8eae,
  [0x446e] = 0x8eb0, [0x446f] = 0x8eb1, [0x4470] = 0x8eb3, [0x4471] = 0x8eb4,
  [0x4472] = 0x8eb5, [0x4473] = 0x8eb6, [0x4474] = 0x8eb7, [0x4475] = 0x8eb8,
  [0x4476] = 0x8eb9, [0x4477] = 0x8ebb, [0x4478] = 0x8ebc, [0x4479] = 0x8ebd,
  [0x447a] = 0x8ebe, [0x447b] = 0x8ebf, [0x447c] = 0x8ec0, [0x447d] = 0x8ec1,
  [0x447e] = 0x8ec2, [0x4480] = 0x8ec3, [0x4481] = 0x8ec4, [0x4482] = 0x8ec5,
  [0x4483] = 0x8ec6, [0x4484] = 0x8ec7, [0x4485] = 0x8ec8, [0x4486] = 0x8ec9,
  [0x4487] = 0x8eca, [0x4488] = 0x8ecb, [0x4489] = 0x8ecc, [0x448a] = 0x8ecd,
  [0x448b] = 0x8ecf, [0x448c] = 0x8ed0, [0x448d] = 0x8ed1, [0x448e] = 0x8ed2,
  [0x448f] = 0x8ed3, [0x4490] = 0x8ed4, [0x4491] = 0x8ed5, [0x4492] = 0x8ed6,
  [0x4493] = 0x8ed7, [0x4494] = 0x8ed8, [0x4495] = 0x8ed9, [0x4496] = 0x8eda,
  [0x4497] = 0x8edb, [0x4498] = 0x8edc, [0x4499] = 0x8edd, [0x449a] = 0x8ede,
  [0x449b] = 0x8edf, [0x449c] = 0x8ee0, [0x449d] = 0x8ee1, [0x449e] = 0x8ee2,
  [0x449f] = 0x8ee3, [0x44a0] = 0x8ee4, [0x44a1] = 0x580b, [0x44a2] = 0x580d,
  [0x44a3] = 0x57fd, [0x44a4] = 0x57ed, [0x44a5] = 0x5800, [0x44a6] = 0x581e,
  [0x44a7] = 0x5819, [0x44a8] = 0x5844, [0x44a9] = 0x5820, [0x44aa] = 0x5865,
  [0x44ab] = 0x586c, [0x44ac] = 0x5881, [0x44ad] = 0x5889, [0x44ae] = 0x589a,
  [0x44af] = 0x5880, [0x44b0] = 0x99a8, [0x44b1] = 0x9f19, [0x44b2] = 0x61ff,
  [0x44b3] = 0x8279, [0x44b4] = 0x827d, [0x44b5] = 0x827f, [0x44b6] = 0x828f,
  [0x44b7] = 0x828a, [0x44b8] = 0x82a8, [0x44b9] = 0x8284, [0x44ba] = 0x828e,
  [0x44bb] = 0x8291, [0x44bc] = 0x8297, [0x44bd] = 0x8299, [0x44be] = 0x82ab,
  [0x44bf] = 0x82b8, [0x44c0] = 0x82be, [0x44c1] = 0x82b0, [0x44c2] = 0x82c8,
  [0x44c3] = 0x82ca, [0x44c4] = 0x82e3, [0x44c5] = 0x8298, [0x44c6] = 0x82b7,
  [0x44c7] = 0x82ae, [0x44c8] = 0x82cb, [0x44c9] = 0x82cc, [0x44ca] = 0x82c1,
  [0x44cb] = 0x82a9, [0x44cc] = 0x82b4, [0x44cd] = 0x82a1, [0x44ce] = 0x82aa,
  [0x44cf] = 0x829f, [0x44d0] = 0x82c4, [0x44d1] = 0x82ce, [0x44d2] = 0x82a4,
  [0x44d3] = 0x82e1, [0x44d4] = 0x8309, [0x44d5] = 0x82f7, [0x44d6] = 0x82e4,
  [0x44d7] = 0x830f, [0x44d8] = 0x8307, [0x44d9] = 0x82dc, [0x44da] = 0x82f4,
  [0x44db] = 0x82d2, [0x44dc] = 0x82d8, [0x44dd] = 0x830c, [0x44de] = 0x82fb,
  [0x44df] = 0x82d3, [0x44e0] = 0x8311, [0x44e1] = 0x831a, [0x44e2] = 0x8306,
  [0x44e3] = 0x8314, [0x44e4] = 0x8315, [0x44e5] = 0x82e0, [0x44e6] = 0x82d5,
  [0x44e7] = 0x831c, [0x44e8] = 0x8351, [0x44e9] = 0x835b, [0x44ea] = 0x835c,
  [0x44eb] = 0x8308, [0x44ec] = 0x8392, [0x44ed] = 0x833c, [0x44ee] = 0x8334,
  [0x44ef] = 0x8331, [0x44f0] = 0x839b, [0x44f1] = 0x835e, [0x44f2] = 0x832f,
  [0x44f3] = 0x834f, [0x44f4] = 0x8347, [0x44f5] = 0x8343, [0x44f6] = 0x835f,
  [0x44f7] = 0x8340, [0x44f8] = 0x8317, [0x44f9] = 0x8360, [0x44fa] = 0x832d,
  [0x44fb] = 0x833a, [0x44fc] = 0x8333, [0x44fd] = 0x8366, [0x44fe] = 0x8365,
  [0x4500] = 0x8ee5, [0x4501] = 0x8ee6, [0x4502] = 0x8ee7, [0x4503] = 0x8ee8,
  [0x4504] = 0x8ee9, [0x4505] = 0x8eea, [0x4506] = 0x8eeb, [0x4507] = 0x8eec,
  [0x4508] = 0x8eed, [0x4509] = 0x8eee, [0x450a] = 0x8eef, [0x450b] = 0x8ef0,
  [0x450c] = 0x8ef1, [0x450d] = 0x8ef2, [0x450e] = 0x8ef3, [0x450f] = 0x8ef4,
  [0x4510] = 0x8ef5, [0x4511] = 0x8ef6, [0x4512] = 0x8ef7, [0x4513] = 0x8ef8,
  [0x4514] = 0x8ef9, [0x4515] = 0x8efa, [0x4516] = 0x8efb, [0x4517] = 0x8efc,
  [0x4518] = 0x8efd, [0x4519] = 0x8efe, [0x451a] = 0x8eff, [0x451b] = 0x8f00,
  [0x451c] = 0x8f01, [0x451d] = 0x8f02, [0x451e] = 0x8f03, [0x451f] = 0x8f04,
  [0x4520] = 0x8f05, [0x4521] = 0x8f06, [0x4522] = 0x8f07, [0x4523] = 0x8f08,
  [0x4524] = 0x8f09, [0x4525] = 0x8f0a, [0x4526] = 0x8f0b, [0x4527] = 0x8f0c,
  [0x4528] = 0x8f0d, [0x4529] = 0x8f0e, [0x452a] = 0x8f0f, [0x452b] = 0x8f10,
  [0x452c] = 0x8f11, [0x452d] = 0x8f12, [0x452e] = 0x8f13, [0x452f] = 0x8f14,
  [0x4530] = 0x8f15, [0x4531] = 0x8f16, [0x4532] = 0x8f17, [0x4533] = 0x8f18,
  [0x4534] = 0x8f19, [0x4535] = 0x8f1a, [0x4536] = 0x8f1b, [0x4537] = 0x8f1c,
  [0x4538] = 0x8f1d, [0x4539] = 0x8f1e, [0x453a] = 0x8f1f, [0x453b] = 0x8f20,
  [0x453c] = 0x8f21, [0x453d] = 0x8f22, [0x453e] = 0x8f23, [0x4540] = 0x8f24,
  [0x4541] = 0x8f25, [0x4542] = 0x8f26, [0x4543] = 0x8f27, [0x4544] = 0x8f28,
  [0x4545] = 0x8f29, [0x4546] = 0x8f2a, [0x4547] = 0x8f2b, [0x4548] = 0x8f2c,
  [0x4549] = 0x8f2d, [0x454a] = 0x8f2e, [0x454b] = 0x8f2f, [0x454c] = 0x8f30,
  [0x454d] = 0x8f31, [0x454e] = 0x8f32, [0x454f] = 0x8f33, [0x4550] = 0x8f34,
  [0x4551] = 0x8f35, [0x4552] = 0x8f36, [0x4553] = 0x8f37, [0x4554] = 0x8f38,
  [0x4555] = 0x8f39, [0x4556] = 0x8f3a, [0x4557] = 0x8f3b, [0x4558] = 0x8f3c,
  [0x4559] = 0x8f3d, [0x455a] = 0x8f3e, [0x455b] = 0x8f3f, [0x455c] = 0x8f40,
  [0x455d] = 0x8f41, [0x455e] = 0x8f42, [0x455f] = 0x8f43, [0x4560] = 0x8f44,
  [0x4561] = 0x8368, [0x4562] = 0x831b, [0x4563] = 0x8369, [0x4564] = 0x836c,
  [0x4565] = 0x836a, [0x4566] = 0x836d, [0x4567] = 0x836e, [0x4568] = 0x83b0,
  [0x4569] = 0x8378, [0x456a] = 0x83b3, [0x456b] = 0x83b4, [0x456c] = 0x83a0,
  [0x456d] = 0x83aa, [0x456e] = 0x8393, [0x456f] = 0x839c, [0x4570] = 0x8385,
  [0x4571] = 0x837c, [0x4572] = 0x83b6, [0x4573] = 0x83a9, [0x4574] = 0x837d,
  [0x4575] = 0x83b8, [0x4576] = 0x837b, [0x4577] = 0x8398, [0x4578] = 0x839e,
  [0x4579] = 0x83a8, [0x457a] = 0x83ba, [0x457b] = 0x83bc, [0x457c] = 0x83c1,
  [0x457d] = 0x8401, [0x457e] = 0x83e5, [0x457f] = 0x83d8, [0x4580] = 0x5807,
  [0x4581] = 0x8418, [0x4582] = 0x840b, [0x4583] = 0x83dd, [0x4584] = 0x83fd,
  [0x4585] = 0x83d6, [0x4586] = 0x841c, [0x4587] = 0x8438, [0x4588] = 0x8411,
  [0x4589] = 0x8406, [0x458a] = 0x83d4, [0x458b] = 0x83df, [0x458c] = 0x840f,
  [0x458d] = 0x8403, [0x458e] = 0x83f8, [0x458f] = 0x83f9, [0x4590] = 0x83ea,
  [0x4591] = 0x83c5, [0x4592] = 0x83c0, [0x4593] = 0x8426, [0x4594] = 0x83f0,
  [0x4595] = 0x83e1, [0x4596] = 0x845c, [0x4597] = 0x8451, [0x4598] = 0x845a,
  [0x4599] = 0x8459, [0x459a] = 0x8473, [0x459b] = 0x8487, [0x459c] = 0x8488,
  [0x459d] = 0x847a, [0x459e] = 0x8489, [0x459f] = 0x8478, [0x45a0] = 0x843c,
  [0x45a1] = 0x8446, [0x45a2] = 0x8469, [0x45a3] = 0x8476, [0x45a4] = 0x848c,
  [0x45a5] = 0x848e, [0x45a6] = 0x8431, [0x45a7] = 0x846d, [0x45a8] = 0x84c1,
  [0x45a9] = 0x84cd, [0x45aa] = 0x84d0, [0x45ab] = 0x84e6, [0x45ac] = 0x84bd,
  [0x45ad] = 0x84d3, [0x45ae] = 0x84ca, [0x45af] = 0x84bf, [0x45b0] = 0x84ba,
  [0x45b1] = 0x84e0, [0x45b2] = 0x84a1, [0x45b3] = 0x84b9, [0x45b4] = 0x84b4,
  [0x45b5] = 0x8497, [0x45b6] = 0x84e5, [0x45b7] = 0x84e3, [0x45b8] = 0x850c,
  [0x45b9] = 0x750d, [0x45ba] = 0x8538, [0x45bb] = 0x84f0, [0x45bc] = 0x8539,
  [0x45bd] = 0x851f, [0x45be] = 0x853a, [0x45c0] = 0x8f45, [0x45c1] = 0x8f46,
  [0x45c2] = 0x8f47, [0x45c3] = 0x8f48, [0x45c4] = 0x8f49, [0x45c5] = 0x8f4a,
  [0x45c6] = 0x8f4b, [0x45c7] = 0x8f4c, [0x45c8] = 0x8f4d, [0x45c9] = 0x8f4e,
  [0x45ca] = 0x8f4f, [0x45cb] = 0x8f50, [0x45cc] = 0x8f51, [0x45cd] = 0x8f52,
  [0x45ce] = 0x8f53, [0x45cf] = 0x8f54, [0x45d0] = 0x8f55, [0x45d1] = 0x8f56,
  [0x45d2] = 0x8f57, [0x45d3] = 0x8f58, [0x45d4] = 0x8f59, [0x45d5] = 0x8f5a,
  [0x45d6] = 0x8f5b, [0x45d7] = 0x8f5c, [0x45d8] = 0x8f5d, [0x45d9] = 0x8f5e,
  [0x45da] = 0x8f5f, [0x45db] = 0x8f60, [0x45dc] = 0x8f61, [0x45dd] = 0x8f62,
  [0x45de] = 0x8f63, [0x45df] = 0x8f64, [0x45e0] = 0x8f65, [0x45e1] = 0x8f6a,
  [0x45e2] = 0x8f80, [0x45e3] = 0x8f8c, [0x45e4] = 0x8f92, [0x45e5] = 0x8f9d,
  [0x45e6] = 0x8fa0, [0x45e7] = 0x8fa1, [0x45e8] = 0x8fa2, [0x45e9] = 0x8fa4,
  [0x45ea] = 0x8fa5, [0x45eb] = 0x8fa6, [0x45ec] = 0x8fa7, [0x45ed] = 0x8faa,
  [0x45ee] = 0x8fac, [0x45ef] = 0x8fad, [0x45f0] = 0x8fae, [0x45f1] = 0x8faf,
  [0x45f2] = 0x8fb2, [0x45f3] = 0x8fb3, [0x45f4] = 0x8fb4, [0x45f5] = 0x8fb5,
  [0x45f6] = 0x8fb7, [0x45f7] = 0x8fb8, [0x45f8] = 0x8fba, [0x45f9] = 0x8fbb,
  [0x45fa] = 0x8fbc, [0x45fb] = 0x8fbf, [0x45fc] = 0x8fc0, [0x45fd] = 0x8fc3,
  [0x45fe] = 0x8fc6, [0x4600] = 0x8fc9, [0x4601] = 0x8fca, [0x4602] = 0x8fcb,
  [0x4603] = 0x8fcc, [0x4604] = 0x8fcd, [0x4605] = 0x8fcf, [0x4606] = 0x8fd2,
  [0x4607] = 0x8fd6, [0x4608] = 0x8fd7, [0x4609] = 0x8fda, [0x460a] = 0x8fe0,
  [0x460b] = 0x8fe1, [0x460c] = 0x8fe3, [0x460d] = 0x8fe7, [0x460e] = 0x8fec,
  [0x460f] = 0x8fef, [0x4610] = 0x8ff1, [0x4611] = 0x8ff2, [0x4612] = 0x8ff4,
  [0x4613] = 0x8ff5, [0x4614] = 0x8ff6, [0x4615] = 0x8ffa, [0x4616] = 0x8ffb,
  [0x4617] = 0x8ffc, [0x4618] = 0x8ffe, [0x4619] = 0x8fff, [0x461a] = 0x9007,
  [0x461b] = 0x9008, [0x461c] = 0x900c, [0x461d] = 0x900e, [0x461e] = 0x9013,
  [0x461f] = 0x9015, [0x4620] = 0x9018, [0x4621] = 0x8556, [0x4622] = 0x853b,
  [0x4623] = 0x84ff, [0x4624] = 0x84fc, [0x4625] = 0x8559, [0x4626] = 0x8548,
  [0x4627] = 0x8568, [0x4628] = 0x8564, [0x4629] = 0x855e, [0x462a] = 0x857a,
  [0x462b] = 0x77a2, [0x462c] = 0x8543, [0x462d] = 0x8572, [0x462e] = 0x857b,
  [0x462f] = 0x85a4, [0x4630] = 0x85a8, [0x4631] = 0x8587, [0x4632] = 0x858f,
  [0x4633] = 0x8579, [0x4634] = 0x85ae, [0x4635] = 0x859c, [0x4636] = 0x8585,
  [0x4637] = 0x85b9, [0x4638] = 0x85b7, [0x4639] = 0x85b0, [0x463a] = 0x85d3,
  [0x463b] = 0x85c1, [0x463c] = 0x85dc, [0x463d] = 0x85ff, [0x463e] = 0x8627,
  [0x463f] = 0x8605, [0x4640] = 0x8629, [0x4641] = 0x8616, [0x4642] = 0x863c,
  [0x4643] = 0x5efe, [0x4644] = 0x5f08, [0x4645] = 0x593c, [0x4646] = 0x5941,
  [0x4647] = 0x8037, [0x4648] = 0x5955, [0x4649] = 0x595a, [0x464a] = 0x5958,
  [0x464b] = 0x530f, [0x464c] = 0x5c22, [0x464d] = 0x5c25, [0x464e] = 0x5c2c,
  [0x464f] = 0x5c34, [0x4650] = 0x624c, [0x4651] = 0x626a, [0x4652] = 0x629f,
  [0x4653] = 0x62bb, [0x4654] = 0x62ca, [0x4655] = 0x62da, [0x4656] = 0x62d7,
  [0x4657] = 0x62ee, [0x4658] = 0x6322, [0x4659] = 0x62f6, [0x465a] = 0x6339,
  [0x465b] = 0x634b, [0x465c] = 0x6343, [0x465d] = 0x63ad, [0x465e] = 0x63f6,
  [0x465f] = 0x6371, [0x4660] = 0x637a, [0x4661] = 0x638e, [0x4662] = 0x63b4,
  [0x4663] = 0x636d, [0x4664] = 0x63ac, [0x4665] = 0x638a, [0x4666] = 0x6369,
  [0x4667] = 0x63ae, [0x4668] = 0x63bc, [0x4669] = 0x63f2, [0x466a] = 0x63f8,
  [0x466b] = 0x63e0, [0x466c] = 0x63ff, [0x466d] = 0x63c4, [0x466e] = 0x63de,
  [0x466f] = 0x63ce, [0x4670] = 0x6452, [0x4671] = 0x63c6, [0x4672] = 0x63be,
  [0x4673] = 0x6445, [0x4674] = 0x6441, [0x4675] = 0x640b, [0x4676] = 0x641b,
  [0x4677] = 0x6420, [0x4678] = 0x640c, [0x4679] = 0x6426, [0x467a] = 0x6421,
  [0x467b] = 0x645e, [0x467c] = 0x6484, [0x467d] = 0x646d, [0x467e] = 0x6496,
  [0x4680] = 0x9019, [0x4681] = 0x901c, [0x4682] = 0x9023, [0x4683] = 0x9024,
  [0x4684] = 0x9025, [0x4685] = 0x9027, [0x4686] = 0x9028, [0x4687] = 0x9029,
  [0x4688] = 0x902a, [0x4689] = 0x902b, [0x468a] = 0x902c, [0x468b] = 0x9030,
  [0x468c] = 0x9031, [0x468d] = 0x9032, [0x468e] = 0x9033, [0x468f] = 0x9034,
  [0x4690] = 0x9037, [0x4691] = 0x9039, [0x4692] = 0x903a, [0x4693] = 0x903d,
  [0x4694] = 0x903f, [0x4695] = 0x9040, [0x4696] = 0x9043, [0x4697] = 0x9045,
  [0x4698] = 0x9046, [0x4699] = 0x9048, [0x469a] = 0x9049, [0x469b] = 0x904a,
  [0x469c] = 0x904b, [0x469d] = 0x904c, [0x469e] = 0x904e, [0x469f] = 0x9054,
  [0x46a0] = 0x9055, [0x46a1] = 0x9056, [0x46a2] = 0x9059, [0x46a3] = 0x905a,
  [0x46a4] = 0x905c, [0x46a5] = 0x905d, [0x46a6] = 0x905e, [0x46a7] = 0x905f,
  [0x46a8] = 0x9060, [0x46a9] = 0x9061, [0x46aa] = 0x9064, [0x46ab] = 0x9066,
  [0x46ac] = 0x9067, [0x46ad] = 0x9069, [0x46ae] = 0x906a, [0x46af] = 0x906b,
  [0x46b0] = 0x906c, [0x46b1] = 0x906f, [0x46b2] = 0x9070, [0x46b3] = 0x9071,
  [0x46b4] = 0x9072, [0x46b5] = 0x9073, [0x46b6] = 0x9076, [0x46b7] = 0x9077,
  [0x46b8] = 0x9078, [0x46b9] = 0x9079, [0x46ba] = 0x907a, [0x46bb] = 0x907b,
  [0x46bc] = 0x907c, [0x46bd] = 0x907e, [0x46be] = 0x9081, [0x46c0] = 0x9084,
  [0x46c1] = 0x9085, [0x46c2] = 0x9086, [0x46c3] = 0x9087, [0x46c4] = 0x9089,
  [0x46c5] = 0x908a, [0x46c6] = 0x908c, [0x46c7] = 0x908d, [0x46c8] = 0x908e,
  [0x46c9] = 0x908f, [0x46ca] = 0x9090, [0x46cb] = 0x9092, [0x46cc] = 0x9094,
  [0x46cd] = 0x9096, [0x46ce] = 0x9098, [0x46cf] = 0x909a, [0x46d0] = 0x909c,
  [0x46d1] = 0x909e, [0x46d2] = 0x909f, [0x46d3] = 0x90a0, [0x46d4] = 0x90a4,
  [0x46d5] = 0x90a5, [0x46d6] = 0x90a7, [0x46d7] = 0x90a8, [0x46d8] = 0x90a9,
  [0x46d9] = 0x90ab, [0x46da] = 0x90ad, [0x46db] = 0x90b2, [0x46dc] = 0x90b7,
  [0x46dd] = 0x90bc, [0x46de] = 0x90bd, [0x46df] = 0x90bf, [0x46e0] = 0x90c0,
  [0x46e1] = 0x647a, [0x46e2] = 0x64b7, [0x46e3] = 0x64b8, [0x46e4] = 0x6499,
  [0x46e5] = 0x64ba, [0x46e6] = 0x64c0, [0x46e7] = 0x64d0, [0x46e8] = 0x64d7,
  [0x46e9] = 0x64e4, [0x46ea] = 0x64e2, [0x46eb] = 0x6509, [0x46ec] = 0x6525,
  [0x46ed] = 0x652e, [0x46ee] = 0x5f0b, [0x46ef] = 0x5fd2, [0x46f0] = 0x7519,
  [0x46f1] = 0x5f11, [0x46f2] = 0x535f, [0x46f3] = 0x53f1, [0x46f4] = 0x53fd,
  [0x46f5] = 0x53e9, [0x46f6] = 0x53e8, [0x46f7] = 0x53fb, [0x46f8] = 0x5412,
  [0x46f9] = 0x5416, [0x46fa] = 0x5406, [0x46fb] = 0x544b, [0x46fc] = 0x5452,
  [0x46fd] = 0x5453, [0x46fe] = 0x5454, [0x46ff] = 0x5456, [0x4700] = 0x5443,
  [0x4701] = 0x5421, [0x4702] = 0x5457, [0x4703] = 0x5459, [0x4704] = 0x5423,
  [0x4705] = 0x5432, [0x4706] = 0x5482, [0x4707] = 0x5494, [0x4708] = 0x5477,
  [0x4709] = 0x5471, [0x470a] = 0x5464, [0x470b] = 0x549a, [0x470c] = 0x549b,
  [0x470d] = 0x5484, [0x470e] = 0x5476, [0x470f] = 0x5466, [0x4710] = 0x549d,
  [0x4711] = 0x54d0, [0x4712] = 0x54ad, [0x4713] = 0x54c2, [0x4714] = 0x54b4,
  [0x4715] = 0x54d2, [0x4716] = 0x54a7, [0x4717] = 0x54a6, [0x4718] = 0x54d3,
  [0x4719] = 0x54d4, [0x471a] = 0x5472, [0x471b] = 0x54a3, [0x471c] = 0x54d5,
  [0x471d] = 0x54bb, [0x471e] = 0x54bf, [0x471f] = 0x54cc, [0x4720] = 0x54d9,
  [0x4721] = 0x54da, [0x4722] = 0x54dc, [0x4723] = 0x54a9, [0x4724] = 0x54aa,
  [0x4725] = 0x54a4, [0x4726] = 0x54dd, [0x4727] = 0x54cf, [0x4728] = 0x54de,
  [0x4729] = 0x551b, [0x472a] = 0x54e7, [0x472b] = 0x5520, [0x472c] = 0x54fd,
  [0x472d] = 0x5514, [0x472e] = 0x54f3, [0x472f] = 0x5522, [0x4730] = 0x5523,
  [0x4731] = 0x550f, [0x4732] = 0x5511, [0x4733] = 0x5527, [0x4734] = 0x552a,
  [0x4735] = 0x5567, [0x4736] = 0x558f, [0x4737] = 0x55b5, [0x4738] = 0x5549,
  [0x4739] = 0x556d, [0x473a] = 0x5541, [0x473b] = 0x5555, [0x473c] = 0x553f,
  [0x473d] = 0x5550, [0x473e] = 0x553c, [0x4740] = 0x90c2, [0x4741] = 0x90c3,
  [0x4742] = 0x90c6, [0x4743] = 0x90c8, [0x4744] = 0x90c9, [0x4745] = 0x90cb,
  [0x4746] = 0x90cc, [0x4747] = 0x90cd, [0x4748] = 0x90d2, [0x4749] = 0x90d4,
  [0x474a] = 0x90d5, [0x474b] = 0x90d6, [0x474c] = 0x90d8, [0x474d] = 0x90d9,
  [0x474e] = 0x90da, [0x474f] = 0x90de, [0x4750] = 0x90df, [0x4751] = 0x90e0,
  [0x4752] = 0x90e3, [0x4753] = 0x90e4, [0x4754] = 0x90e5, [0x4755] = 0x90e9,
  [0x4756] = 0x90ea, [0x4757] = 0x90ec, [0x4758] = 0x90ee, [0x4759] = 0x90f0,
  [0x475a] = 0x90f1, [0x475b] = 0x90f2, [0x475c] = 0x90f3, [0x475d] = 0x90f5,
  [0x475e] = 0x90f6, [0x475f] = 0x90f7, [0x4760] = 0x90f9, [0x4761] = 0x90fa,
  [0x4762] = 0x90fb, [0x4763] = 0x90fc, [0x4764] = 0x90ff, [0x4765] = 0x9100,
  [0x4766] = 0x9101, [0x4767] = 0x9103, [0x4768] = 0x9105, [0x4769] = 0x9106,
  [0x476a] = 0x9107, [0x476b] = 0x9108, [0x476c] = 0x9109, [0x476d] = 0x910a,
  [0x476e] = 0x910b, [0x476f] = 0x910c, [0x4770] = 0x910d, [0x4771] = 0x910e,
  [0x4772] = 0x910f, [0x4773] = 0x9110, [0x4774] = 0x9111, [0x4775] = 0x9112,
  [0x4776] = 0x9113, [0x4777] = 0x9114, [0x4778] = 0x9115, [0x4779] = 0x9116,
  [0x477a] = 0x9117, [0x477b] = 0x9118, [0x477c] = 0x911a, [0x477d] = 0x911b,
  [0x477e] = 0x911c, [0x4780] = 0x911d, [0x4781] = 0x911f, [0x4782] = 0x9120,
  [0x4783] = 0x9121, [0x4784] = 0x9124, [0x4785] = 0x9125, [0x4786] = 0x9126,
  [0x4787] = 0x9127, [0x4788] = 0x9128, [0x4789] = 0x9129, [0x478a] = 0x912a,
  [0x478b] = 0x912b, [0x478c] = 0x912c, [0x478d] = 0x912d, [0x478e] = 0x912e,
  [0x478f] = 0x9130, [0x4790] = 0x9132, [0x4791] = 0x9133, [0x4792] = 0x9134,
  [0x4793] = 0x9135, [0x4794] = 0x9136, [0x4795] = 0x9137, [0x4796] = 0x9138,
  [0x4797] = 0x913a, [0x4798] = 0x913b, [0x4799] = 0x913c, [0x479a] = 0x913d,
  [0x479b] = 0x913e, [0x479c] = 0x913f, [0x479d] = 0x9140, [0x479e] = 0x9141,
  [0x479f] = 0x9142, [0x47a0] = 0x9144, [0x47a1] = 0x5537, [0x47a2] = 0x5556,
  [0x47a3] = 0x5575, [0x47a4] = 0x5576, [0x47a5] = 0x5577, [0x47a6] = 0x5533,
  [0x47a7] = 0x5530, [0x47a8] = 0x555c, [0x47a9] = 0x558b, [0x47aa] = 0x55d2,
  [0x47ab] = 0x5583, [0x47ac] = 0x55b1, [0x47ad] = 0x55b9, [0x47ae] = 0x5588,
  [0x47af] = 0x5581, [0x47b0] = 0x559f, [0x47b1] = 0x557e, [0x47b2] = 0x55d6,
  [0x47b3] = 0x5591, [0x47b4] = 0x557b, [0x47b5] = 0x55df, [0x47b6] = 0x55bd,
  [0x47b7] = 0x55be, [0x47b8] = 0x5594, [0x47b9] = 0x5599, [0x47ba] = 0x55ea,
  [0x47bb] = 0x55f7, [0x47bc] = 0x55c9, [0x47bd] = 0x561f, [0x47be] = 0x55d1,
  [0x47bf] = 0x55eb, [0x47c0] = 0x55ec, [0x47c1] = 0x55d4, [0x47c2] = 0x55e6,
  [0x47c3] = 0x55dd, [0x47c4] = 0x55c4, [0x47c5] = 0x55ef, [0x47c6] = 0x55e5,
  [0x47c7] = 0x55f2, [0x47c8] = 0x55f3, [0x47c9] = 0x55cc, [0x47ca] = 0x55cd,
  [0x47cb] = 0x55e8, [0x47cc] = 0x55f5, [0x47cd] = 0x55e4, [0x47ce] = 0x8f94,
  [0x47cf] = 0x561e, [0x47d0] = 0x5608, [0x47d1] = 0x560c, [0x47d2] = 0x5601,
  [0x47d3] = 0x5624, [0x47d4] = 0x5623, [0x47d5] = 0x55fe, [0x47d6] = 0x5600,
  [0x47d7] = 0x5627, [0x47d8] = 0x562d, [0x47d9] = 0x5658, [0x47da] = 0x5639,
  [0x47db] = 0x5657, [0x47dc] = 0x562c, [0x47dd] = 0x564d, [0x47de] = 0x5662,
  [0x47df] = 0x5659, [0x47e0] = 0x565c, [0x47e1] = 0x564c, [0x47e2] = 0x5654,
  [0x47e3] = 0x5686, [0x47e4] = 0x5664, [0x47e5] = 0x5671, [0x47e6] = 0x566b,
  [0x47e7] = 0x567b, [0x47e8] = 0x567c, [0x47e9] = 0x5685, [0x47ea] = 0x5693,
  [0x47eb] = 0x56af, [0x47ec] = 0x56d4, [0x47ed] = 0x56d7, [0x47ee] = 0x56dd,
  [0x47ef] = 0x56e1, [0x47f0] = 0x56f5, [0x47f1] = 0x56eb, [0x47f2] = 0x56f9,
  [0x47f3] = 0x56ff, [0x47f4] = 0x5704, [0x47f5] = 0x570a, [0x47f6] = 0x5709,
  [0x47f7] = 0x571c, [0x47f8] = 0x5e0f, [0x47f9] = 0x5e19, [0x47fa] = 0x5e14,
  [0x47fb] = 0x5e11, [0x47fc] = 0x5e31, [0x47fd] = 0x5e3b, [0x47fe] = 0x5e3c,
  [0x4800] = 0x9145, [0x4801] = 0x9147, [0x4802] = 0x9148, [0x4803] = 0x9151,
  [0x4804] = 0x9153, [0x4805] = 0x9154, [0x4806] = 0x9155, [0x4807] = 0x9156,
  [0x4808] = 0x9158, [0x4809] = 0x9159, [0x480a] = 0x915b, [0x480b] = 0x915c,
  [0x480c] = 0x915f, [0x480d] = 0x9160, [0x480e] = 0x9166, [0x480f] = 0x9167,
  [0x4810] = 0x9168, [0x4811] = 0x916b, [0x4812] = 0x916d, [0x4813] = 0x9173,
  [0x4814] = 0x917a, [0x4815] = 0x917b, [0x4816] = 0x917c, [0x4817] = 0x9180,
  [0x4818] = 0x9181, [0x4819] = 0x9182, [0x481a] = 0x9183, [0x481b] = 0x9184,
  [0x481c] = 0x9186, [0x481d] = 0x9188, [0x481e] = 0x918a, [0x481f] = 0x918e,
  [0x4820] = 0x918f, [0x4821] = 0x9193, [0x4822] = 0x9194, [0x4823] = 0x9195,
  [0x4824] = 0x9196, [0x4825] = 0x9197, [0x4826] = 0x9198, [0x4827] = 0x9199,
  [0x4828] = 0x919c, [0x4829] = 0x919d, [0x482a] = 0x919e, [0x482b] = 0x919f,
  [0x482c] = 0x91a0, [0x482d] = 0x91a1, [0x482e] = 0x91a4, [0x482f] = 0x91a5,
  [0x4830] = 0x91a6, [0x4831] = 0x91a7, [0x4832] = 0x91a8, [0x4833] = 0x91a9,
  [0x4834] = 0x91ab, [0x4835] = 0x91ac, [0x4836] = 0x91b0, [0x4837] = 0x91b1,
  [0x4838] = 0x91b2, [0x4839] = 0x91b3, [0x483a] = 0x91b6, [0x483b] = 0x91b7,
  [0x483c] = 0x91b8, [0x483d] = 0x91b9, [0x483e] = 0x91bb, [0x4840] = 0x91bc,
  [0x4841] = 0x91bd, [0x4842] = 0x91be, [0x4843] = 0x91bf, [0x4844] = 0x91c0,
  [0x4845] = 0x91c1, [0x4846] = 0x91c2, [0x4847] = 0x91c3, [0x4848] = 0x91c4,
  [0x4849] = 0x91c5, [0x484a] = 0x91c6, [0x484b] = 0x91c8, [0x484c] = 0x91cb,
  [0x484d] = 0x91d0, [0x484e] = 0x91d2, [0x484f] = 0x91d3, [0x4850] = 0x91d4,
  [0x4851] = 0x91d5, [0x4852] = 0x91d6, [0x4853] = 0x91d7, [0x4854] = 0x91d8,
  [0x4855] = 0x91d9, [0x4856] = 0x91da, [0x4857] = 0x91db, [0x4858] = 0x91dd,
  [0x4859] = 0x91de, [0x485a] = 0x91df, [0x485b] = 0x91e0, [0x485c] = 0x91e1,
  [0x485d] = 0x91e2, [0x485e] = 0x91e3, [0x485f] = 0x91e4, [0x4860] = 0x91e5,
  [0x4861] = 0x5e37, [0x4862] = 0x5e44, [0x4863] = 0x5e54, [0x4864] = 0x5e5b,
  [0x4865] = 0x5e5e, [0x4866] = 0x5e61, [0x4867] = 0x5c8c, [0x4868] = 0x5c7a,
  [0x4869] = 0x5c8d, [0x486a] = 0x5c90, [0x486b] = 0x5c96, [0x486c] = 0x5c88,
  [0x486d] = 0x5c98, [0x486e] = 0x5c99, [0x486f] = 0x5c91, [0x4870] = 0x5c9a,
  [0x4871] = 0x5c9c, [0x4872] = 0x5cb5, [0x4873] = 0x5ca2, [0x4874] = 0x5cbd,
  [0x4875] = 0x5cac, [0x4876] = 0x5cab, [0x4877] = 0x5cb1, [0x4878] = 0x5ca3,
  [0x4879] = 0x5cc1, [0x487a] = 0x5cb7, [0x487b] = 0x5cc4, [0x487c] = 0x5cd2,
  [0x487d] = 0x5ce4, [0x487e] = 0x5ccb, [0x487f] = 0x5ce5, [0x4880] = 0x5d02,
  [0x4881] = 0x5d03, [0x4882] = 0x5d27, [0x4883] = 0x5d26, [0x4884] = 0x5d2e,
  [0x4885] = 0x5d24, [0x4886] = 0x5d1e, [0x4887] = 0x5d06, [0x4888] = 0x5d1b,
  [0x4889] = 0x5d58, [0x488a] = 0x5d3e, [0x488b] = 0x5d34, [0x488c] = 0x5d3d,
  [0x488d] = 0x5d6c, [0x488e] = 0x5d5b, [0x488f] = 0x5d6f, [0x4890] = 0x5d5d,
  [0x4891] = 0x5d6b, [0x4892] = 0x5d4b, [0x4893] = 0x5d4a, [0x4894] = 0x5d69,
  [0x4895] = 0x5d74, [0x4896] = 0x5d82, [0x4897] = 0x5d99, [0x4898] = 0x5d9d,
  [0x4899] = 0x8c73, [0x489a] = 0x5db7, [0x489b] = 0x5dc5, [0x489c] = 0x5f73,
  [0x489d] = 0x5f77, [0x489e] = 0x5f82, [0x489f] = 0x5f87, [0x48a0] = 0x5f89,
  [0x48a1] = 0x5f8c, [0x48a2] = 0x5f95, [0x48a3] = 0x5f99, [0x48a4] = 0x5f9c,
  [0x48a5] = 0x5fa8, [0x48a6] = 0x5fad, [0x48a7] = 0x5fb5, [0x48a8] = 0x5fbc,
  [0x48a9] = 0x8862, [0x48aa] = 0x5f61, [0x48ab] = 0x72ad, [0x48ac] = 0x72b0,
  [0x48ad] = 0x72b4, [0x48ae] = 0x72b7, [0x48af] = 0x72b8, [0x48b0] = 0x72c3,
  [0x48b1] = 0x72c1, [0x48b2] = 0x72ce, [0x48b3] = 0x72cd, [0x48b4] = 0x72d2,
  [0x48b5] = 0x72e8, [0x48b6] = 0x72ef, [0x48b7] = 0x72e9, [0x48b8] = 0x72f2,
  [0x48b9] = 0x72f4, [0x48ba] = 0x72f7, [0x48bb] = 0x7301, [0x48bc] = 0x72f3,
  [0x48bd] = 0x7303, [0x48be] = 0x72fa, [0x48c0] = 0x91e6, [0x48c1] = 0x91e7,
  [0x48c2] = 0x91e8, [0x48c3] = 0x91e9, [0x48c4] = 0x91ea, [0x48c5] = 0x91eb,
  [0x48c6] = 0x91ec, [0x48c7] = 0x91ed, [0x48c8] = 0x91ee, [0x48c9] = 0x91ef,
  [0x48ca] = 0x91f0, [0x48cb] = 0x91f1, [0x48cc] = 0x91f2, [0x48cd] = 0x91f3,
  [0x48ce] = 0x91f4, [0x48cf] = 0x91f5, [0x48d0] = 0x91f6, [0x48d1] = 0x91f7,
  [0x48d2] = 0x91f8, [0x48d3] = 0x91f9, [0x48d4] = 0x91fa, [0x48d5] = 0x91fb,
  [0x48d6] = 0x91fc, [0x48d7] = 0x91fd, [0x48d8] = 0x91fe, [0x48d9] = 0x91ff,
  [0x48da] = 0x9200, [0x48db] = 0x9201, [0x48dc] = 0x9202, [0x48dd] = 0x9203,
  [0x48de] = 0x9204, [0x48df] = 0x9205, [0x48e0] = 0x9206, [0x48e1] = 0x9207,
  [0x48e2] = 0x9208, [0x48e3] = 0x9209, [0x48e4] = 0x920a, [0x48e5] = 0x920b,
  [0x48e6] = 0x920c, [0x48e7] = 0x920d, [0x48e8] = 0x920e, [0x48e9] = 0x920f,
  [0x48ea] = 0x9210, [0x48eb] = 0x9211, [0x48ec] = 0x9212, [0x48ed] = 0x9213,
  [0x48ee] = 0x9214, [0x48ef] = 0x9215, [0x48f0] = 0x9216, [0x48f1] = 0x9217,
  [0x48f2] = 0x9218, [0x48f3] = 0x9219, [0x48f4] = 0x921a, [0x48f5] = 0x921b,
  [0x48f6] = 0x921c, [0x48f7] = 0x921d, [0x48f8] = 0x921e, [0x48f9] = 0x921f,
  [0x48fa] = 0x9220, [0x48fb] = 0x9221, [0x48fc] = 0x9222, [0x48fd] = 0x9223,
  [0x48fe] = 0x9224, [0x4900] = 0x9225, [0x4901] = 0x9226, [0x4902] = 0x9227,
  [0x4903] = 0x9228, [0x4904] = 0x9229, [0x4905] = 0x922a, [0x4906] = 0x922b,
  [0x4907] = 0x922c, [0x4908] = 0x922d, [0x4909] = 0x922e, [0x490a] = 0x922f,
  [0x490b] = 0x9230, [0x490c] = 0x9231, [0x490d] = 0x9232, [0x490e] = 0x9233,
  [0x490f] = 0x9234, [0x4910] = 0x9235, [0x4911] = 0x9236, [0x4912] = 0x9237,
  [0x4913] = 0x9238, [0x4914] = 0x9239, [0x4915] = 0x923a, [0x4916] = 0x923b,
  [0x4917] = 0x923c, [0x4918] = 0x923d, [0x4919] = 0x923e, [0x491a] = 0x923f,
  [0x491b] = 0x9240, [0x491c] = 0x9241, [0x491d] = 0x9242, [0x491e] = 0x9243,
  [0x491f] = 0x9244, [0x4920] = 0x9245, [0x4921] = 0x72fb, [0x4922] = 0x7317,
  [0x4923] = 0x7313, [0x4924] = 0x7321, [0x4925] = 0x730a, [0x4926] = 0x731e,
  [0x4927] = 0x731d, [0x4928] = 0x7315, [0x4929] = 0x7322, [0x492a] = 0x7339,
  [0x492b] = 0x7325, [0x492c] = 0x732c, [0x492d] = 0x7338, [0x492e] = 0x7331,
  [0x492f] = 0x7350, [0x4930] = 0x734d, [0x4931] = 0x7357, [0x4932] = 0x7360,
  [0x4933] = 0x736c, [0x4934] = 0x736f, [0x4935] = 0x737e, [0x4936] = 0x821b,
  [0x4937] = 0x5925, [0x4938] = 0x98e7, [0x4939] = 0x5924, [0x493a] = 0x5902,
  [0x493b] = 0x9963, [0x493c] = 0x9967, [0x493d] = 0x9968, [0x493e] = 0x9969,
  [0x493f] = 0x996a, [0x4940] = 0x996b, [0x4941] = 0x996c, [0x4942] = 0x9974,
  [0x4943] = 0x9977, [0x4944] = 0x997d, [0x4945] = 0x9980, [0x4946] = 0x9984,
  [0x4947] = 0x9987, [0x4948] = 0x998a, [0x4949] = 0x998d, [0x494a] = 0x9990,
  [0x494b] = 0x9991, [0x494c] = 0x9993, [0x494d] = 0x9994, [0x494e] = 0x9995,
  [0x494f] = 0x5e80, [0x4950] = 0x5e91, [0x4951] = 0x5e8b, [0x4952] = 0x5e96,
  [0x4953] = 0x5ea5, [0x4954] = 0x5ea0, [0x4955] = 0x5eb9, [0x4956] = 0x5eb5,
  [0x4957] = 0x5ebe, [0x4958] = 0x5eb3, [0x4959] = 0x8d53, [0x495a] = 0x5ed2,
  [0x495b] = 0x5ed1, [0x495c] = 0x5edb, [0x495d] = 0x5ee8, [0x495e] = 0x5eea,
  [0x495f] = 0x81ba, [0x4960] = 0x5fc4, [0x4961] = 0x5fc9, [0x4962] = 0x5fd6,
  [0x4963] = 0x5fcf, [0x4964] = 0x6003, [0x4965] = 0x5fee, [0x4966] = 0x6004,
  [0x4967] = 0x5fe1, [0x4968] = 0x5fe4, [0x4969] = 0x5ffe, [0x496a] = 0x6005,
  [0x496b] = 0x6006, [0x496c] = 0x5fea, [0x496d] = 0x5fed, [0x496e] = 0x5ff8,
  [0x496f] = 0x6019, [0x4970] = 0x6035, [0x4971] = 0x6026, [0x4972] = 0x601b,
  [0x4973] = 0x600f, [0x4974] = 0x600d, [0x4975] = 0x6029, [0x4976] = 0x602b,
  [0x4977] = 0x600a, [0x4978] = 0x603f, [0x4979] = 0x6021, [0x497a] = 0x6078,
  [0x497b] = 0x6079, [0x497c] = 0x607b, [0x497d] = 0x607a, [0x497e] = 0x6042,
  [0x4980] = 0x9246, [0x4981] = 0x9247, [0x4982] = 0x9248, [0x4983] = 0x9249,
  [0x4984] = 0x924a, [0x4985] = 0x924b, [0x4986] = 0x924c, [0x4987] = 0x924d,
  [0x4988] = 0x924e, [0x4989] = 0x924f, [0x498a] = 0x9250, [0x498b] = 0x9251,
  [0x498c] = 0x9252, [0x498d] = 0x9253, [0x498e] = 0x9254, [0x498f] = 0x9255,
  [0x4990] = 0x9256, [0x4991] = 0x9257, [0x4992] = 0x9258, [0x4993] = 0x9259,
  [0x4994] = 0x925a, [0x4995] = 0x925b, [0x4996] = 0x925c, [0x4997] = 0x925d,
  [0x4998] = 0x925e, [0x4999] = 0x925f, [0x499a] = 0x9260, [0x499b] = 0x9261,
  [0x499c] = 0x9262, [0x499d] = 0x9263, [0x499e] = 0x9264, [0x499f] = 0x9265,
  [0x49a0] = 0x9266, [0x49a1] = 0x9267, [0x49a2] = 0x9268, [0x49a3] = 0x9269,
  [0x49a4] = 0x926a, [0x49a5] = 0x926b, [0x49a6] = 0x926c, [0x49a7] = 0x926d,
  [0x49a8] = 0x926e, [0x49a9] = 0x926f, [0x49aa] = 0x9270, [0x49ab] = 0x9271,
  [0x49ac] = 0x9272, [0x49ad] = 0x9273, [0x49ae] = 0x9275, [0x49af] = 0x9276,
  [0x49b0] = 0x9277, [0x49b1] = 0x9278, [0x49b2] = 0x9279, [0x49b3] = 0x927a,
  [0x49b4] = 0x927b, [0x49b5] = 0x927c, [0x49b6] = 0x927d, [0x49b7] = 0x927e,
  [0x49b8] = 0x927f, [0x49b9] = 0x9280, [0x49ba] = 0x9281, [0x49bb] = 0x9282,
  [0x49bc] = 0x9283, [0x49bd] = 0x9284, [0x49be] = 0x9285, [0x49c0] = 0x9286,
  [0x49c1] = 0x9287, [0x49c2] = 0x9288, [0x49c3] = 0x9289, [0x49c4] = 0x928a,
  [0x49c5] = 0x928b, [0x49c6] = 0x928c, [0x49c7] = 0x928d, [0x49c8] = 0x928f,
  [0x49c9] = 0x9290, [0x49ca] = 0x9291, [0x49cb] = 0x9292, [0x49cc] = 0x9293,
  [0x49cd] = 0x9294, [0x49ce] = 0x9295, [0x49cf] = 0x9296, [0x49d0] = 0x9297,
  [0x49d1] = 0x9298, [0x49d2] = 0x9299, [0x49d3] = 0x929a, [0x49d4] = 0x929b,
  [0x49d5] = 0x929c, [0x49d6] = 0x929d, [0x49d7] = 0x929e, [0x49d8] = 0x929f,
  [0x49d9] = 0x92a0, [0x49da] = 0x92a1, [0x49db] = 0x92a2, [0x49dc] = 0x92a3,
  [0x49dd] = 0x92a4, [0x49de] = 0x92a5, [0x49df] = 0x92a6, [0x49e0] = 0x92a7,
  [0x49e1] = 0x606a, [0x49e2] = 0x607d, [0x49e3] = 0x6096, [0x49e4] = 0x609a,
  [0x49e5] = 0x60ad, [0x49e6] = 0x609d, [0x49e7] = 0x6083, [0x49e8] = 0x6092,
  [0x49e9] = 0x608c, [0x49ea] = 0x609b, [0x49eb] = 0x60ec, [0x49ec] = 0x60bb,
  [0x49ed] = 0x60b1, [0x49ee] = 0x60dd, [0x49ef] = 0x60d8, [0x49f0] = 0x60c6,
  [0x49f1] = 0x60da, [0x49f2] = 0x60b4, [0x49f3] = 0x6120, [0x49f4] = 0x6126,
  [0x49f5] = 0x6115, [0x49f6] = 0x6123, [0x49f7] = 0x60f4, [0x49f8] = 0x6100,
  [0x49f9] = 0x610e, [0x49fa] = 0x612b, [0x49fb] = 0x614a, [0x49fc] = 0x6175,
  [0x49fd] = 0x61ac, [0x49fe] = 0x6194, [0x49ff] = 0x61a7, [0x4a00] = 0x61b7,
  [0x4a01] = 0x61d4, [0x4a02] = 0x61f5, [0x4a03] = 0x5fdd, [0x4a04] = 0x96b3,
  [0x4a05] = 0x95e9, [0x4a06] = 0x95eb, [0x4a07] = 0x95f1, [0x4a08] = 0x95f3,
  [0x4a09] = 0x95f5, [0x4a0a] = 0x95f6, [0x4a0b] = 0x95fc, [0x4a0c] = 0x95fe,
  [0x4a0d] = 0x9603, [0x4a0e] = 0x9604, [0x4a0f] = 0x9606, [0x4a10] = 0x9608,
  [0x4a11] = 0x960a, [0x4a12] = 0x960b, [0x4a13] = 0x960c, [0x4a14] = 0x960d,
  [0x4a15] = 0x960f, [0x4a16] = 0x9612, [0x4a17] = 0x9615, [0x4a18] = 0x9616,
  [0x4a19] = 0x9617, [0x4a1a] = 0x9619, [0x4a1b] = 0x961a, [0x4a1c] = 0x4e2c,
  [0x4a1d] = 0x723f, [0x4a1e] = 0x6215, [0x4a1f] = 0x6c35, [0x4a20] = 0x6c54,
  [0x4a21] = 0x6c5c, [0x4a22] = 0x6c4a, [0x4a23] = 0x6ca3, [0x4a24] = 0x6c85,
  [0x4a25] = 0x6c90, [0x4a26] = 0x6c94, [0x4a27] = 0x6c8c, [0x4a28] = 0x6c68,
  [0x4a29] = 0x6c69, [0x4a2a] = 0x6c74, [0x4a2b] = 0x6c76, [0x4a2c] = 0x6c86,
  [0x4a2d] = 0x6ca9, [0x4a2e] = 0x6cd0, [0x4a2f] = 0x6cd4, [0x4a30] = 0x6cad,
  [0x4a31] = 0x6cf7, [0x4a32] = 0x6cf8, [0x4a33] = 0x6cf1, [0x4a34] = 0x6cd7,
  [0x4a35] = 0x6cb2, [0x4a36] = 0x6ce0, [0x4a37] = 0x6cd6, [0x4a38] = 0x6cfa,
  [0x4a39] = 0x6ceb, [0x4a3a] = 0x6cee, [0x4a3b] = 0x6cb1, [0x4a3c] = 0x6cd3,
  [0x4a3d] = 0x6cef, [0x4a3e] = 0x6cfe, [0x4a40] = 0x92a8, [0x4a41] = 0x92a9,
  [0x4a42] = 0x92aa, [0x4a43] = 0x92ab, [0x4a44] = 0x92ac, [0x4a45] = 0x92ad,
  [0x4a46] = 0x92af, [0x4a47] = 0x92b0, [0x4a48] = 0x92b1, [0x4a49] = 0x92b2,
  [0x4a4a] = 0x92b3, [0x4a4b] = 0x92b4, [0x4a4c] = 0x92b5, [0x4a4d] = 0x92b6,
  [0x4a4e] = 0x92b7, [0x4a4f] = 0x92b8, [0x4a50] = 0x92b9, [0x4a51] = 0x92ba,
  [0x4a52] = 0x92bb, [0x4a53] = 0x92bc, [0x4a54] = 0x92bd, [0x4a55] = 0x92be,
  [0x4a56] = 0x92bf, [0x4a57] = 0x92c0, [0x4a58] = 0x92c1, [0x4a59] = 0x92c2,
  [0x4a5a] = 0x92c3, [0x4a5b] = 0x92c4, [0x4a5c] = 0x92c5, [0x4a5d] = 0x92c6,
  [0x4a5e] = 0x92c7, [0x4a5f] = 0x92c9, [0x4a60] = 0x92ca, [0x4a61] = 0x92cb,
  [0x4a62] = 0x92cc, [0x4a63] = 0x92cd, [0x4a64] = 0x92ce, [0x4a65] = 0x92cf,
  [0x4a66] = 0x92d0, [0x4a67] = 0x92d1, [0x4a68] = 0x92d2, [0x4a69] = 0x92d3,
  [0x4a6a] = 0x92d4, [0x4a6b] = 0x92d5, [0x4a6c] = 0x92d6, [0x4a6d] = 0x92d7,
  [0x4a6e] = 0x92d8, [0x4a6f] = 0x92d9, [0x4a70] = 0x92da, [0x4a71] = 0x92db,
  [0x4a72] = 0x92dc, [0x4a73] = 0x92dd, [0x4a74] = 0x92de, [0x4a75] = 0x92df,
  [0x4a76] = 0x92e0, [0x4a77] = 0x92e1, [0x4a78] = 0x92e2, [0x4a79] = 0x92e3,
  [0x4a7a] = 0x92e4, [0x4a7b] = 0x92e5, [0x4a7c] = 0x92e6, [0x4a7d] = 0x92e7,
  [0x4a7e] = 0x92e8, [0x4a80] = 0x92e9, [0x4a81] = 0x92ea, [0x4a82] = 0x92eb,
  [0x4a83] = 0x92ec, [0x4a84] = 0x92ed, [0x4a85] = 0x92ee, [0x4a86] = 0x92ef,
  [0x4a87] = 0x92f0, [0x4a88] = 0x92f1, [0x4a89] = 0x92f2, [0x4a8a] = 0x92f3,
  [0x4a8b] = 0x92f4, [0x4a8c] = 0x92f5, [0x4a8d] = 0x92f6, [0x4a8e] = 0x92f7,
  [0x4a8f] = 0x92f8, [0x4a90] = 0x92f9, [0x4a91] = 0x92fa, [0x4a92] = 0x92fb,
  [0x4a93] = 0x92fc, [0x4a94] = 0x92fd, [0x4a95] = 0x92fe, [0x4a96] = 0x92ff,
  [0x4a97] = 0x9300, [0x4a98] = 0x9301, [0x4a99] = 0x9302, [0x4a9a] = 0x9303,
  [0x4a9b] = 0x9304, [0x4a9c] = 0x9305, [0x4a9d] = 0x9306, [0x4a9e] = 0x9307,
  [0x4a9f] = 0x9308, [0x4aa0] = 0x9309, [0x4aa1] = 0x6d39, [0x4aa2] = 0x6d27,
  [0x4aa3] = 0x6d0c, [0x4aa4] = 0x6d43, [0x4aa5] = 0x6d48, [0x4aa6] = 0x6d07,
  [0x4aa7] = 0x6d04, [0x4aa8] = 0x6d19, [0x4aa9] = 0x6d0e, [0x4aaa] = 0x6d2b,
  [0x4aab] = 0x6d4d, [0x4aac] = 0x6d2e, [0x4aad] = 0x6d35, [0x4aae] = 0x6d1a,
  [0x4aaf] = 0x6d4f, [0x4ab0] = 0x6d52, [0x4ab1] = 0x6d54, [0x4ab2] = 0x6d33,
  [0x4ab3] = 0x6d91, [0x4ab4] = 0x6d6f, [0x4ab5] = 0x6d9e, [0x4ab6] = 0x6da0,
  [0x4ab7] = 0x6d5e, [0x4ab8] = 0x6d93, [0x4ab9] = 0x6d94, [0x4aba] = 0x6d5c,
  [0x4abb] = 0x6d60, [0x4abc] = 0x6d7c, [0x4abd] = 0x6d63, [0x4abe] = 0x6e1a,
  [0x4abf] = 0x6dc7, [0x4ac0] = 0x6dc5, [0x4ac1] = 0x6dde, [0x4ac2] = 0x6e0e,
  [0x4ac3] = 0x6dbf, [0x4ac4] = 0x6de0, [0x4ac5] = 0x6e11, [0x4ac6] = 0x6de6,
  [0x4ac7] = 0x6ddd, [0x4ac8] = 0x6dd9, [0x4ac9] = 0x6e16, [0x4aca] = 0x6dab,
  [0x4acb] = 0x6e0c, [0x4acc] = 0x6dae, [0x4acd] = 0x6e2b, [0x4ace] = 0x6e6e,
  [0x4acf] = 0x6e4e, [0x4ad0] = 0x6e6b, [0x4ad1] = 0x6eb2, [0x4ad2] = 0x6e5f,
  [0x4ad3] = 0x6e86, [0x4ad4] = 0x6e53, [0x4ad5] = 0x6e54, [0x4ad6] = 0x6e32,
  [0x4ad7] = 0x6e25, [0x4ad8] = 0x6e44, [0x4ad9] = 0x6edf, [0x4ada] = 0x6eb1,
  [0x4adb] = 0x6e98, [0x4adc] = 0x6ee0, [0x4add] = 0x6f2d, [0x4ade] = 0x6ee2,
  [0x4adf] = 0x6ea5, [0x4ae0] = 0x6ea7, [0x4ae1] = 0x6ebd, [0x4ae2] = 0x6ebb,
  [0x4ae3] = 0x6eb7, [0x4ae4] = 0x6ed7, [0x4ae5] = 0x6eb4, [0x4ae6] = 0x6ecf,
  [0x4ae7] = 0x6e8f, [0x4ae8] = 0x6ec2, [0x4ae9] = 0x6e9f, [0x4aea] = 0x6f62,
  [0x4aeb] = 0x6f46, [0x4aec] = 0x6f47, [0x4aed] = 0x6f24, [0x4aee] = 0x6f15,
  [0x4aef] = 0x6ef9, [0x4af0] = 0x6f2f, [0x4af1] = 0x6f36, [0x4af2] = 0x6f4b,
  [0x4af3] = 0x6f74, [0x4af4] = 0x6f2a, [0x4af5] = 0x6f09, [0x4af6] = 0x6f29,
  [0x4af7] = 0x6f89, [0x4af8] = 0x6f8d, [0x4af9] = 0x6f8c, [0x4afa] = 0x6f78,
  [0x4afb] = 0x6f72, [0x4afc] = 0x6f7c, [0x4afd] = 0x6f7a, [0x4afe] = 0x6fd1,
  [0x4b00] = 0x930a, [0x4b01] = 0x930b, [0x4b02] = 0x930c, [0x4b03] = 0x930d,
  [0x4b04] = 0x930e, [0x4b05] = 0x930f, [0x4b06] = 0x9310, [0x4b07] = 0x9311,
  [0x4b08] = 0x9312, [0x4b09] = 0x9313, [0x4b0a] = 0x9314, [0x4b0b] = 0x9315,
  [0x4b0c] = 0x9316, [0x4b0d] = 0x9317, [0x4b0e] = 0x9318, [0x4b0f] = 0x9319,
  [0x4b10] = 0x931a, [0x4b11] = 0x931b, [0x4b12] = 0x931c, [0x4b13] = 0x931d,
  [0x4b14] = 0x931e, [0x4b15] = 0x931f, [0x4b16] = 0x9320, [0x4b17] = 0x9321,
  [0x4b18] = 0x9322, [0x4b19] = 0x9323, [0x4b1a] = 0x9324, [0x4b1b] = 0x9325,
  [0x4b1c] = 0x9326, [0x4b1d] = 0x9327, [0x4b1e] = 0x9328, [0x4b1f] = 0x9329,
  [0x4b20] = 0x932a, [0x4b21] = 0x932b, [0x4b22] = 0x932c, [0x4b23] = 0x932d,
  [0x4b24] = 0x932e, [0x4b25] = 0x932f, [0x4b26] = 0x9330, [0x4b27] = 0x9331,
  [0x4b28] = 0x9332, [0x4b29] = 0x9333, [0x4b2a] = 0x9334, [0x4b2b] = 0x9335,
  [0x4b2c] = 0x9336, [0x4b2d] = 0x9337, [0x4b2e] = 0x9338, [0x4b2f] = 0x9339,
  [0x4b30] = 0x933a, [0x4b31] = 0x933b, [0x4b32] = 0x933c, [0x4b33] = 0x933d,
  [0x4b34] = 0x933f, [0x4b35] = 0x9340, [0x4b36] = 0x9341, [0x4b37] = 0x9342,
  [0x4b38] = 0x9343, [0x4b39] = 0x9344, [0x4b3a] = 0x9345, [0x4b3b] = 0x9346,
  [0x4b3c] = 0x9347, [0x4b3d] = 0x9348, [0x4b3e] = 0x9349, [0x4b40] = 0x934a,
  [0x4b41] = 0x934b, [0x4b42] = 0x934c, [0x4b43] = 0x934d, [0x4b44] = 0x934e,
  [0x4b45] = 0x934f, [0x4b46] = 0x9350, [0x4b47] = 0x9351, [0x4b48] = 0x9352,
  [0x4b49] = 0x9353, [0x4b4a] = 0x9354, [0x4b4b] = 0x9355, [0x4b4c] = 0x9356,
  [0x4b4d] = 0x9357, [0x4b4e] = 0x9358, [0x4b4f] = 0x9359, [0x4b50] = 0x935a,
  [0x4b51] = 0x935b, [0x4b52] = 0x935c, [0x4b53] = 0x935d, [0x4b54] = 0x935e,
  [0x4b55] = 0x935f, [0x4b56] = 0x9360, [0x4b57] = 0x9361, [0x4b58] = 0x9362,
  [0x4b59] = 0x9363, [0x4b5a] = 0x9364, [0x4b5b] = 0x9365, [0x4b5c] = 0x9366,
  [0x4b5d] = 0x9367, [0x4b5e] = 0x9368, [0x4b5f] = 0x9369, [0x4b60] = 0x936b,
  [0x4b61] = 0x6fc9, [0x4b62] = 0x6fa7, [0x4b63] = 0x6fb9, [0x4b64] = 0x6fb6,
  [0x4b65] = 0x6fc2, [0x4b66] = 0x6fe1, [0x4b67] = 0x6fee, [0x4b68] = 0x6fde,
  [0x4b69] = 0x6fe0, [0x4b6a] = 0x6fef, [0x4b6b] = 0x701a, [0x4b6c] = 0x7023,
  [0x4b6d] = 0x701b, [0x4b6e] = 0x7039, [0x4b6f] = 0x7035, [0x4b70] = 0x704f,
  [0x4b71] = 0x705e, [0x4b72] = 0x5b80, [0x4b73] = 0x5b84, [0x4b74] = 0x5b95,
  [0x4b75] = 0x5b93, [0x4b76] = 0x5ba5, [0x4b77] = 0x5bb8, [0x4b78] = 0x752f,
  [0x4b79] = 0x9a9e, [0x4b7a] = 0x6434, [0x4b7b] = 0x5be4, [0x4b7c] = 0x5bee,
  [0x4b7d] = 0x8930, [0x4b7e] = 0x5bf0, [0x4b7f] = 0x8e47, [0x4b80] = 0x8b07,
  [0x4b81] = 0x8fb6, [0x4b82] = 0x8fd3, [0x4b83] = 0x8fd5, [0x4b84] = 0x8fe5,
  [0x4b85] = 0x8fee, [0x4b86] = 0x8fe4, [0x4b87] = 0x8fe9, [0x4b88] = 0x8fe6,
  [0x4b89] = 0x8ff3, [0x4b8a] = 0x8fe8, [0x4b8b] = 0x9005, [0x4b8c] = 0x9004,
  [0x4b8d] = 0x900b, [0x4b8e] = 0x9026, [0x4b8f] = 0x9011, [0x4b90] = 0x900d,
  [0x4b91] = 0x9016, [0x4b92] = 0x9021, [0x4b93] = 0x9035, [0x4b94] = 0x9036,
  [0x4b95] = 0x902d, [0x4b96] = 0x902f, [0x4b97] = 0x9044, [0x4b98] = 0x9051,
  [0x4b99] = 0x9052, [0x4b9a] = 0x9050, [0x4b9b] = 0x9068, [0x4b9c] = 0x9058,
  [0x4b9d] = 0x9062, [0x4b9e] = 0x905b, [0x4b9f] = 0x66b9, [0x4ba0] = 0x9074,
  [0x4ba1] = 0x907d, [0x4ba2] = 0x9082, [0x4ba3] = 0x9088, [0x4ba4] = 0x9083,
  [0x4ba5] = 0x908b, [0x4ba6] = 0x5f50, [0x4ba7] = 0x5f57, [0x4ba8] = 0x5f56,
  [0x4ba9] = 0x5f58, [0x4baa] = 0x5c3b, [0x4bab] = 0x54ab, [0x4bac] = 0x5c50,
  [0x4bad] = 0x5c59, [0x4bae] = 0x5b71, [0x4baf] = 0x5c63, [0x4bb0] = 0x5c66,
  [0x4bb1] = 0x7fbc, [0x4bb2] = 0x5f2a, [0x4bb3] = 0x5f29, [0x4bb4] = 0x5f2d,
  [0x4bb5] = 0x8274, [0x4bb6] = 0x5f3c, [0x4bb7] = 0x9b3b, [0x4bb8] = 0x5c6e,
  [0x4bb9] = 0x5981, [0x4bba] = 0x5983, [0x4bbb] = 0x598d, [0x4bbc] = 0x59a9,
  [0x4bbd] = 0x59aa, [0x4bbe] = 0x59a3, [0x4bc0] = 0x936c, [0x4bc1] = 0x936d,
  [0x4bc2] = 0x936e, [0x4bc3] = 0x936f, [0x4bc4] = 0x9370, [0x4bc5] = 0x9371,
  [0x4bc6] = 0x9372, [0x4bc7] = 0x9373, [0x4bc8] = 0x9374, [0x4bc9] = 0x9375,
  [0x4bca] = 0x9376, [0x4bcb] = 0x9377, [0x4bcc] = 0x9378, [0x4bcd] = 0x9379,
  [0x4bce] = 0x937a, [0x4bcf] = 0x937b, [0x4bd0] = 0x937c, [0x4bd1] = 0x937d,
  [0x4bd2] = 0x937e, [0x4bd3] = 0x937f, [0x4bd4] = 0x9380, [0x4bd5] = 0x9381,
  [0x4bd6] = 0x9382, [0x4bd7] = 0x9383, [0x4bd8] = 0x9384, [0x4bd9] = 0x9385,
  [0x4bda] = 0x9386, [0x4bdb] = 0x9387, [0x4bdc] = 0x9388, [0x4bdd] = 0x9389,
  [0x4bde] = 0x938a, [0x4bdf] = 0x938b, [0x4be0] = 0x938c, [0x4be1] = 0x938d,
  [0x4be2] = 0x938e, [0x4be3] = 0x9390, [0x4be4] = 0x9391, [0x4be5] = 0x9392,
  [0x4be6] = 0x9393, [0x4be7] = 0x9394, [0x4be8] = 0x9395, [0x4be9] = 0x9396,
  [0x4bea] = 0x9397, [0x4beb] = 0x9398, [0x4bec] = 0x9399, [0x4bed] = 0x939a,
  [0x4bee] = 0x939b, [0x4bef] = 0x939c, [0x4bf0] = 0x939d, [0x4bf1] = 0x939e,
  [0x4bf2] = 0x939f, [0x4bf3] = 0x93a0, [0x4bf4] = 0x93a1, [0x4bf5] = 0x93a2,
  [0x4bf6] = 0x93a3, [0x4bf7] = 0x93a4, [0x4bf8] = 0x93a5, [0x4bf9] = 0x93a6,
  [0x4bfa] = 0x93a7, [0x4bfb] = 0x93a8, [0x4bfc] = 0x93a9, [0x4bfd] = 0x93aa,
  [0x4bfe] = 0x93ab, [0x4c00] = 0x93ac, [0x4c01] = 0x93ad, [0x4c02] = 0x93ae,
  [0x4c03] = 0x93af, [0x4c04] = 0x93b0, [0x4c05] = 0x93b1, [0x4c06] = 0x93b2,
  [0x4c07] = 0x93b3, [0x4c08] = 0x93b4, [0x4c09] = 0x93b5, [0x4c0a] = 0x93b6,
  [0x4c0b] = 0x93b7, [0x4c0c] = 0x93b8, [0x4c0d] = 0x93b9, [0x4c0e] = 0x93ba,
  [0x4c0f] = 0x93bb, [0x4c10] = 0x93bc, [0x4c11] = 0x93bd, [0x4c12] = 0x93be,
  [0x4c13] = 0x93bf, [0x4c14] = 0x93c0, [0x4c15] = 0x93c1, [0x4c16] = 0x93c2,
  [0x4c17] = 0x93c3, [0x4c18] = 0x93c4, [0x4c19] = 0x93c5, [0x4c1a] = 0x93c6,
  [0x4c1b] = 0x93c7, [0x4c1c] = 0x93c8, [0x4c1d] = 0x93c9, [0x4c1e] = 0x93cb,
  [0x4c1f] = 0x93cc, [0x4c20] = 0x93cd, [0x4c21] = 0x5997, [0x4c22] = 0x59ca,
  [0x4c23] = 0x59ab, [0x4c24] = 0x599e, [0x4c25] = 0x59a4, [0x4c26] = 0x59d2,
  [0x4c27] = 0x59b2, [0x4c28] = 0x59af, [0x4c29] = 0x59d7, [0x4c2a] = 0x59be,
  [0x4c2b] = 0x5a05, [0x4c2c] = 0x5a06, [0x4c2d] = 0x59dd, [0x4c2e] = 0x5a08,
  [0x4c2f] = 0x59e3, [0x4c30] = 0x59d8, [0x4c31] = 0x59f9, [0x4c32] = 0x5a0c,
  [0x4c33] = 0x5a09, [0x4c34] = 0x5a32, [0x4c35] = 0x5a34, [0x4c36] = 0x5a11,
  [0x4c37] = 0x5a23, [0x4c38] = 0x5a13, [0x4c39] = 0x5a40, [0x4c3a] = 0x5a67,
  [0x4c3b] = 0x5a4a, [0x4c3c] = 0x5a55, [0x4c3d] = 0x5a3c, [0x4c3e] = 0x5a62,
  [0x4c3f] = 0x5a75, [0x4c40] = 0x80ec, [0x4c41] = 0x5aaa, [0x4c42] = 0x5a9b,
  [0x4c43] = 0x5a77, [0x4c44] = 0x5a7a, [0x4c45] = 0x5abe, [0x4c46] = 0x5aeb,
  [0x4c47] = 0x5ab2, [0x4c48] = 0x5ad2, [0x4c49] = 0x5ad4, [0x4c4a] = 0x5ab8,
  [0x4c4b] = 0x5ae0, [0x4c4c] = 0x5ae3, [0x4c4d] = 0x5af1, [0x4c4e] = 0x5ad6,
  [0x4c4f] = 0x5ae6, [0x4c50] = 0x5ad8, [0x4c51] = 0x5adc, [0x4c52] = 0x5b09,
  [0x4c53] = 0x5b17, [0x4c54] = 0x5b16, [0x4c55] = 0x5b32, [0x4c56] = 0x5b37,
  [0x4c57] = 0x5b40, [0x4c58] = 0x5c15, [0x4c59] = 0x5c1c, [0x4c5a] = 0x5b5a,
  [0x4c5b] = 0x5b65, [0x4c5c] = 0x5b73, [0x4c5d] = 0x5b51, [0x4c5e] = 0x5b53,
  [0x4c5f] = 0x5b62, [0x4c60] = 0x9a75, [0x4c61] = 0x9a77, [0x4c62] = 0x9a78,
  [0x4c63] = 0x9a7a, [0x4c64] = 0x9a7f, [0x4c65] = 0x9a7d, [0x4c66] = 0x9a80,
  [0x4c67] = 0x9a81, [0x4c68] = 0x9a85, [0x4c69] = 0x9a88, [0x4c6a] = 0x9a8a,
  [0x4c6b] = 0x9a90, [0x4c6c] = 0x9a92, [0x4c6d] = 0x9a93, [0x4c6e] = 0x9a96,
  [0x4c6f] = 0x9a98, [0x4c70] = 0x9a9b, [0x4c71] = 0x9a9c, [0x4c72] = 0x9a9d,
  [0x4c73] = 0x9a9f, [0x4c74] = 0x9aa0, [0x4c75] = 0x9aa2, [0x4c76] = 0x9aa3,
  [0x4c77] = 0x9aa5, [0x4c78] = 0x9aa7, [0x4c79] = 0x7e9f, [0x4c7a] = 0x7ea1,
  [0x4c7b] = 0x7ea3, [0x4c7c] = 0x7ea5, [0x4c7d] = 0x7ea8, [0x4c7e] = 0x7ea9,
  [0x4c80] = 0x93ce, [0x4c81] = 0x93cf, [0x4c82] = 0x93d0, [0x4c83] = 0x93d1,
  [0x4c84] = 0x93d2, [0x4c85] = 0x93d3, [0x4c86] = 0x93d4, [0x4c87] = 0x93d5,
  [0x4c88] = 0x93d7, [0x4c89] = 0x93d8, [0x4c8a] = 0x93d9, [0x4c8b] = 0x93da,
  [0x4c8c] = 0x93db, [0x4c8d] = 0x93dc, [0x4c8e] = 0x93dd, [0x4c8f] = 0x93de,
  [0x4c90] = 0x93df, [0x4c91] = 0x93e0, [0x4c92] = 0x93e1, [0x4c93] = 0x93e2,
  [0x4c94] = 0x93e3, [0x4c95] = 0x93e4, [0x4c96] = 0x93e5, [0x4c97] = 0x93e6,
  [0x4c98] = 0x93e7, [0x4c99] = 0x93e8, [0x4c9a] = 0x93e9, [0x4c9b] = 0x93ea,
  [0x4c9c] = 0x93eb, [0x4c9d] = 0x93ec, [0x4c9e] = 0x93ed, [0x4c9f] = 0x93ee,
  [0x4ca0] = 0x93ef, [0x4ca1] = 0x93f0, [0x4ca2] = 0x93f1, [0x4ca3] = 0x93f2,
  [0x4ca4] = 0x93f3, [0x4ca5] = 0x93f4, [0x4ca6] = 0x93f5, [0x4ca7] = 0x93f6,
  [0x4ca8] = 0x93f7, [0x4ca9] = 0x93f8, [0x4caa] = 0x93f9, [0x4cab] = 0x93fa,
  [0x4cac] = 0x93fb, [0x4cad] = 0x93fc, [0x4cae] = 0x93fd, [0x4caf] = 0x93fe,
  [0x4cb0] = 0x93ff, [0x4cb1] = 0x9400, [0x4cb2] = 0x9401, [0x4cb3] = 0x9402,
  [0x4cb4] = 0x9403, [0x4cb5] = 0x9404, [0x4cb6] = 0x9405, [0x4cb7] = 0x9406,
  [0x4cb8] = 0x9407, [0x4cb9] = 0x9408, [0x4cba] = 0x9409, [0x4cbb] = 0x940a,
  [0x4cbc] = 0x940b, [0x4cbd] = 0x940c, [0x4cbe] = 0x940d, [0x4cc0] = 0x940e,
  [0x4cc1] = 0x940f, [0x4cc2] = 0x9410, [0x4cc3] = 0x9411, [0x4cc4] = 0x9412,
  [0x4cc5] = 0x9413, [0x4cc6] = 0x9414, [0x4cc7] = 0x9415, [0x4cc8] = 0x9416,
  [0x4cc9] = 0x9417, [0x4cca] = 0x9418, [0x4ccb] = 0x9419, [0x4ccc] = 0x941a,
  [0x4ccd] = 0x941b, [0x4cce] = 0x941c, [0x4ccf] = 0x941d, [0x4cd0] = 0x941e,
  [0x4cd1] = 0x941f, [0x4cd2] = 0x9420, [0x4cd3] = 0x9421, [0x4cd4] = 0x9422,
  [0x4cd5] = 0x9423, [0x4cd6] = 0x9424, [0x4cd7] = 0x9425, [0x4cd8] = 0x9426,
  [0x4cd9] = 0x9427, [0x4cda] = 0x9428, [0x4cdb] = 0x9429, [0x4cdc] = 0x942a,
  [0x4cdd] = 0x942b, [0x4cde] = 0x942c, [0x4cdf] = 0x942d, [0x4ce0] = 0x942e,
  [0x4ce1] = 0x7ead, [0x4ce2] = 0x7eb0, [0x4ce3] = 0x7ebe, [0x4ce4] = 0x7ec0,
  [0x4ce5] = 0x7ec1, [0x4ce6] = 0x7ec2, [0x4ce7] = 0x7ec9, [0x4ce8] = 0x7ecb,
  [0x4ce9] = 0x7ecc, [0x4cea] = 0x7ed0, [0x4ceb] = 0x7ed4, [0x4cec] = 0x7ed7,
  [0x4ced] = 0x7edb, [0x4cee] = 0x7ee0, [0x4cef] = 0x7ee1, [0x4cf0] = 0x7ee8,
  [0x4cf1] = 0x7eeb, [0x4cf2] = 0x7eee, [0x4cf3] = 0x7eef, [0x4cf4] = 0x7ef1,
  [0x4cf5] = 0x7ef2, [0x4cf6] = 0x7f0d, [0x4cf7] = 0x7ef6, [0x4cf8] = 0x7efa,
  [0x4cf9] = 0x7efb, [0x4cfa] = 0x7efe, [0x4cfb] = 0x7f01, [0x4cfc] = 0x7f02,
  [0x4cfd] = 0x7f03, [0x4cfe] = 0x7f07, [0x4cff] = 0x7f08, [0x4d00] = 0x7f0b,
  [0x4d01] = 0x7f0c, [0x4d02] = 0x7f0f, [0x4d03] = 0x7f11, [0x4d04] = 0x7f12,
  [0x4d05] = 0x7f17, [0x4d06] = 0x7f19, [0x4d07] = 0x7f1c, [0x4d08] = 0x7f1b,
  [0x4d09] = 0x7f1f, [0x4d0a] = 0x7f21, [0x4d0b] = 0x7f22, [0x4d0c] = 0x7f23,
  [0x4d0d] = 0x7f24, [0x4d0e] = 0x7f25, [0x4d0f] = 0x7f26, [0x4d10] = 0x7f27,
  [0x4d11] = 0x7f2a, [0x4d12] = 0x7f2b, [0x4d13] = 0x7f2c, [0x4d14] = 0x7f2d,
  [0x4d15] = 0x7f2f, [0x4d16] = 0x7f30, [0x4d17] = 0x7f31, [0x4d18] = 0x7f32,
  [0x4d19] = 0x7f33, [0x4d1a] = 0x7f35, [0x4d1b] = 0x5e7a, [0x4d1c] = 0x757f,
  [0x4d1d] = 0x5ddb, [0x4d1e] = 0x753e, [0x4d1f] = 0x9095, [0x4d20] = 0x738e,
  [0x4d21] = 0x7391, [0x4d22] = 0x73ae, [0x4d23] = 0x73a2, [0x4d24] = 0x739f,
  [0x4d25] = 0x73cf, [0x4d26] = 0x73c2, [0x4d27] = 0x73d1, [0x4d28] = 0x73b7,
  [0x4d29] = 0x73b3, [0x4d2a] = 0x73c0, [0x4d2b] = 0x73c9, [0x4d2c] = 0x73c8,
  [0x4d2d] = 0x73e5, [0x4d2e] = 0x73d9, [0x4d2f] = 0x987c, [0x4d30] = 0x740a,
  [0x4d31] = 0x73e9, [0x4d32] = 0x73e7, [0x4d33] = 0x73de, [0x4d34] = 0x73ba,
  [0x4d35] = 0x73f2, [0x4d36] = 0x740f, [0x4d37] = 0x742a, [0x4d38] = 0x745b,
  [0x4d39] = 0x7426, [0x4d3a] = 0x7425, [0x4d3b] = 0x7428, [0x4d3c] = 0x7430,
  [0x4d3d] = 0x742e, [0x4d3e] = 0x742c, [0x4d40] = 0x942f, [0x4d41] = 0x9430,
  [0x4d42] = 0x9431, [0x4d43] = 0x9432, [0x4d44] = 0x9433, [0x4d45] = 0x9434,
  [0x4d46] = 0x9435, [0x4d47] = 0x9436, [0x4d48] = 0x9437, [0x4d49] = 0x9438,
  [0x4d4a] = 0x9439, [0x4d4b] = 0x943a, [0x4d4c] = 0x943b, [0x4d4d] = 0x943c,
  [0x4d4e] = 0x943d, [0x4d4f] = 0x943f, [0x4d50] = 0x9440, [0x4d51] = 0x9441,
  [0x4d52] = 0x9442, [0x4d53] = 0x9443, [0x4d54] = 0x9444, [0x4d55] = 0x9445,
  [0x4d56] = 0x9446, [0x4d57] = 0x9447, [0x4d58] = 0x9448, [0x4d59] = 0x9449,
  [0x4d5a] = 0x944a, [0x4d5b] = 0x944b, [0x4d5c] = 0x944c, [0x4d5d] = 0x944d,
  [0x4d5e] = 0x944e, [0x4d5f] = 0x944f, [0x4d60] = 0x9450, [0x4d61] = 0x9451,
  [0x4d62] = 0x9452, [0x4d63] = 0x9453, [0x4d64] = 0x9454, [0x4d65] = 0x9455,
  [0x4d66] = 0x9456, [0x4d67] = 0x9457, [0x4d68] = 0x9458, [0x4d69] = 0x9459,
  [0x4d6a] = 0x945a, [0x4d6b] = 0x945b, [0x4d6c] = 0x945c, [0x4d6d] = 0x945d,
  [0x4d6e] = 0x945e, [0x4d6f] = 0x945f, [0x4d70] = 0x9460, [0x4d71] = 0x9461,
  [0x4d72] = 0x9462, [0x4d73] = 0x9463, [0x4d74] = 0x9464, [0x4d75] = 0x9465,
  [0x4d76] = 0x9466, [0x4d77] = 0x9467, [0x4d78] = 0x9468, [0x4d79] = 0x9469,
  [0x4d7a] = 0x946a, [0x4d7b] = 0x946c, [0x4d7c] = 0x946d, [0x4d7d] = 0x946e,
  [0x4d7e] = 0x946f, [0x4d80] = 0x9470, [0x4d81] = 0x9471, [0x4d82] = 0x9472,
  [0x4d83] = 0x9473, [0x4d84] = 0x9474, [0x4d85] = 0x9475, [0x4d86] = 0x9476,
  [0x4d87] = 0x9477, [0x4d88] = 0x9478, [0x4d89] = 0x9479, [0x4d8a] = 0x947a,
  [0x4d8b] = 0x947b, [0x4d8c] = 0x947c, [0x4d8d] = 0x947d, [0x4d8e] = 0x947e,
  [0x4d8f] = 0x947f, [0x4d90] = 0x9480, [0x4d91] = 0x9481, [0x4d92] = 0x9482,
  [0x4d93] = 0x9483, [0x4d94] = 0x9484, [0x4d95] = 0x9491, [0x4d96] = 0x9496,
  [0x4d97] = 0x9498, [0x4d98] = 0x94c7, [0x4d99] = 0x94cf, [0x4d9a] = 0x94d3,
  [0x4d9b] = 0x94d4, [0x4d9c] = 0x94da, [0x4d9d] = 0x94e6, [0x4d9e] = 0x94fb,
  [0x4d9f] = 0x951c, [0x4da0] = 0x9520, [0x4da1] = 0x741b, [0x4da2] = 0x741a,
  [0x4da3] = 0x7441, [0x4da4] = 0x745c, [0x4da5] = 0x7457, [0x4da6] = 0x7455,
  [0x4da7] = 0x7459, [0x4da8] = 0x7477, [0x4da9] = 0x746d, [0x4daa] = 0x747e,
  [0x4dab] = 0x749c, [0x4dac] = 0x748e, [0x4dad] = 0x7480, [0x4dae] = 0x7481,
  [0x4daf] = 0x7487, [0x4db0] = 0x748b, [0x4db1] = 0x749e, [0x4db2] = 0x74a8,
  [0x4db3] = 0x74a9, [0x4db4] = 0x7490, [0x4db5] = 0x74a7, [0x4db6] = 0x74d2,
  [0x4db7] = 0x74ba, [0x4db8] = 0x97ea, [0x4db9] = 0x97eb, [0x4dba] = 0x97ec,
  [0x4dbb] = 0x674c, [0x4dbc] = 0x6753, [0x4dbd] = 0x675e, [0x4dbe] = 0x6748,
  [0x4dbf] = 0x6769, [0x4dc0] = 0x67a5, [0x4dc1] = 0x6787, [0x4dc2] = 0x676a,
  [0x4dc3] = 0x6773, [0x4dc4] = 0x6798, [0x4dc5] = 0x67a7, [0x4dc6] = 0x6775,
  [0x4dc7] = 0x67a8, [0x4dc8] = 0x679e, [0x4dc9] = 0x67ad, [0x4dca] = 0x678b,
  [0x4dcb] = 0x6777, [0x4dcc] = 0x677c, [0x4dcd] = 0x67f0, [0x4dce] = 0x6809,
  [0x4dcf] = 0x67d8, [0x4dd0] = 0x680a, [0x4dd1] = 0x67e9, [0x4dd2] = 0x67b0,
  [0x4dd3] = 0x680c, [0x4dd4] = 0x67d9, [0x4dd5] = 0x67b5, [0x4dd6] = 0x67da,
  [0x4dd7] = 0x67b3, [0x4dd8] = 0x67dd, [0x4dd9] = 0x6800, [0x4dda] = 0x67c3,
  [0x4ddb] = 0x67b8, [0x4ddc] = 0x67e2, [0x4ddd] = 0x680e, [0x4dde] = 0x67c1,
  [0x4ddf] = 0x67fd, [0x4de0] = 0x6832, [0x4de1] = 0x6833, [0x4de2] = 0x6860,
  [0x4de3] = 0x6861, [0x4de4] = 0x684e, [0x4de5] = 0x6862, [0x4de6] = 0x6844,
  [0x4de7] = 0x6864, [0x4de8] = 0x6883, [0x4de9] = 0x681d, [0x4dea] = 0x6855,
  [0x4deb] = 0x6866, [0x4dec] = 0x6841, [0x4ded] = 0x6867, [0x4dee] = 0x6840,
  [0x4def] = 0x683e, [0x4df0] = 0x684a, [0x4df1] = 0x6849, [0x4df2] = 0x6829,
  [0x4df3] = 0x68b5, [0x4df4] = 0x688f, [0x4df5] = 0x6874, [0x4df6] = 0x6877,
  [0x4df7] = 0x6893, [0x4df8] = 0x686b, [0x4df9] = 0x68c2, [0x4dfa] = 0x696e,
  [0x4dfb] = 0x68fc, [0x4dfc] = 0x691f, [0x4dfd] = 0x6920, [0x4dfe] = 0x68f9,
  [0x4e00] = 0x9527, [0x4e01] = 0x9533, [0x4e02] = 0x953d, [0x4e03] = 0x9543,
  [0x4e04] = 0x9548, [0x4e05] = 0x954b, [0x4e06] = 0x9555, [0x4e07] = 0x955a,
  [0x4e08] = 0x9560, [0x4e09] = 0x956e, [0x4e0a] = 0x9574, [0x4e0b] = 0x9575,
  [0x4e0c] = 0x9577, [0x4e0d] = 0x9578, [0x4e0e] = 0x9579, [0x4e0f] = 0x957a,
  [0x4e10] = 0x957b, [0x4e11] = 0x957c, [0x4e12] = 0x957d, [0x4e13] = 0x957e,
  [0x4e14] = 0x9580, [0x4e15] = 0x9581, [0x4e16] = 0x9582, [0x4e17] = 0x9583,
  [0x4e18] = 0x9584, [0x4e19] = 0x9585, [0x4e1a] = 0x9586, [0x4e1b] = 0x9587,
  [0x4e1c] = 0x9588, [0x4e1d] = 0x9589, [0x4e1e] = 0x958a, [0x4e1f] = 0x958b,
  [0x4e20] = 0x958c, [0x4e21] = 0x958d, [0x4e22] = 0x958e, [0x4e23] = 0x958f,
  [0x4e24] = 0x9590, [0x4e25] = 0x9591, [0x4e26] = 0x9592, [0x4e27] = 0x9593,
  [0x4e28] = 0x9594, [0x4e29] = 0x9595, [0x4e2a] = 0x9596, [0x4e2b] = 0x9597,
  [0x4e2c] = 0x9598, [0x4e2d] = 0x9599, [0x4e2e] = 0x959a, [0x4e2f] = 0x959b,
  [0x4e30] = 0x959c, [0x4e31] = 0x959d, [0x4e32] = 0x959e, [0x4e33] = 0x959f,
  [0x4e34] = 0x95a0, [0x4e35] = 0x95a1, [0x4e36] = 0x95a2, [0x4e37] = 0x95a3,
  [0x4e38] = 0x95a4, [0x4e39] = 0x95a5, [0x4e3a] = 0x95a6, [0x4e3b] = 0x95a7,
  [0x4e3c] = 0x95a8, [0x4e3d] = 0x95a9, [0x4e3e] = 0x95aa, [0x4e40] = 0x95ab,
  [0x4e41] = 0x95ac, [0x4e42] = 0x95ad, [0x4e43] = 0x95ae, [0x4e44] = 0x95af,
  [0x4e45] = 0x95b0, [0x4e46] = 0x95b1, [0x4e47] = 0x95b2, [0x4e48] = 0x95b3,
  [0x4e49] = 0x95b4, [0x4e4a] = 0x95b5, [0x4e4b] = 0x95b6, [0x4e4c] = 0x95b7,
  [0x4e4d] = 0x95b8, [0x4e4e] = 0x95b9, [0x4e4f] = 0x95ba, [0x4e50] = 0x95bb,
  [0x4e51] = 0x95bc, [0x4e52] = 0x95bd, [0x4e53] = 0x95be, [0x4e54] = 0x95bf,
  [0x4e55] = 0x95c0, [0x4e56] = 0x95c1, [0x4e57] = 0x95c2, [0x4e58] = 0x95c3,
  [0x4e59] = 0x95c4, [0x4e5a] = 0x95c5, [0x4e5b] = 0x95c6, [0x4e5c] = 0x95c7,
  [0x4e5d] = 0x95c8, [0x4e5e] = 0x95c9, [0x4e5f] = 0x95ca, [0x4e60] = 0x95cb,
  [0x4e61] = 0x6924, [0x4e62] = 0x68f0, [0x4e63] = 0x690b, [0x4e64] = 0x6901,
  [0x4e65] = 0x6957, [0x4e66] = 0x68e3, [0x4e67] = 0x6910, [0x4e68] = 0x6971,
  [0x4e69] = 0x6939, [0x4e6a] = 0x6960, [0x4e6b] = 0x6942, [0x4e6c] = 0x695d,
  [0x4e6d] = 0x6984, [0x4e6e] = 0x696b, [0x4e6f] = 0x6980, [0x4e70] = 0x6998,
  [0x4e71] = 0x6978, [0x4e72] = 0x6934, [0x4e73] = 0x69cc, [0x4e74] = 0x6987,
  [0x4e75] = 0x6988, [0x4e76] = 0x69ce, [0x4e77] = 0x6989, [0x4e78] = 0x6966,
  [0x4e79] = 0x6963, [0x4e7a] = 0x6979, [0x4e7b] = 0x699b, [0x4e7c] = 0x69a7,
  [0x4e7d] = 0x69bb, [0x4e7e] = 0x69ab, [0x4e7f] = 0x69ad, [0x4e80] = 0x69d4,
  [0x4e81] = 0x69b1, [0x4e82] = 0x69c1, [0x4e83] = 0x69ca, [0x4e84] = 0x69df,
  [0x4e85] = 0x6995, [0x4e86] = 0x69e0, [0x4e87] = 0x698d, [0x4e88] = 0x69ff,
  [0x4e89] = 0x6a2f, [0x4e8a] = 0x69ed, [0x4e8b] = 0x6a17, [0x4e8c] = 0x6a18,
  [0x4e8d] = 0x6a65, [0x4e8e] = 0x69f2, [0x4e8f] = 0x6a44, [0x4e90] = 0x6a3e,
  [0x4e91] = 0x6aa0, [0x4e92] = 0x6a50, [0x4e93] = 0x6a5b, [0x4e94] = 0x6a35,
  [0x4e95] = 0x6a8e, [0x4e96] = 0x6a79, [0x4e97] = 0x6a3d, [0x4e98] = 0x6a28,
  [0x4e99] = 0x6a58, [0x4e9a] = 0x6a7c, [0x4e9b] = 0x6a91, [0x4e9c] = 0x6a90,
  [0x4e9d] = 0x6aa9, [0x4e9e] = 0x6a97, [0x4e9f] = 0x6aab, [0x4ea0] = 0x7337,
  [0x4ea1] = 0x7352, [0x4ea2] = 0x6b81, [0x4ea3] = 0x6b82, [0x4ea4] = 0x6b87,
  [0x4ea5] = 0x6b84, [0x4ea6] = 0x6b92, [0x4ea7] = 0x6b93, [0x4ea8] = 0x6b8d,
  [0x4ea9] = 0x6b9a, [0x4eaa] = 0x6b9b, [0x4eab] = 0x6ba1, [0x4eac] = 0x6baa,
  [0x4ead] = 0x8f6b, [0x4eae] = 0x8f6d, [0x4eaf] = 0x8f71, [0x4eb0] = 0x8f72,
  [0x4eb1] = 0x8f73, [0x4eb2] = 0x8f75, [0x4eb3] = 0x8f76, [0x4eb4] = 0x8f78,
  [0x4eb5] = 0x8f77, [0x4eb6] = 0x8f79, [0x4eb7] = 0x8f7a, [0x4eb8] = 0x8f7c,
  [0x4eb9] = 0x8f7e, [0x4eba] = 0x8f81, [0x4ebb] = 0x8f82, [0x4ebc] = 0x8f84,
  [0x4ebd] = 0x8f87, [0x4ebe] = 0x8f8b, [0x4ec0] = 0x95cc, [0x4ec1] = 0x95cd,
  [0x4ec2] = 0x95ce, [0x4ec3] = 0x95cf, [0x4ec4] = 0x95d0, [0x4ec5] = 0x95d1,
  [0x4ec6] = 0x95d2, [0x4ec7] = 0x95d3, [0x4ec8] = 0x95d4, [0x4ec9] = 0x95d5,
  [0x4eca] = 0x95d6, [0x4ecb] = 0x95d7, [0x4ecc] = 0x95d8, [0x4ecd] = 0x95d9,
  [0x4ece] = 0x95da, [0x4ecf] = 0x95db, [0x4ed0] = 0x95dc, [0x4ed1] = 0x95dd,
  [0x4ed2] = 0x95de, [0x4ed3] = 0x95df, [0x4ed4] = 0x95e0, [0x4ed5] = 0x95e1,
  [0x4ed6] = 0x95e2, [0x4ed7] = 0x95e3, [0x4ed8] = 0x95e4, [0x4ed9] = 0x95e5,
  [0x4eda] = 0x95e6, [0x4edb] = 0x95e7, [0x4edc] = 0x95ec, [0x4edd] = 0x95ff,
  [0x4ede] = 0x9607, [0x4edf] = 0x9613, [0x4ee0] = 0x9618, [0x4ee1] = 0x961b,
  [0x4ee2] = 0x961e, [0x4ee3] = 0x9620, [0x4ee4] = 0x9623, [0x4ee5] = 0x9624,
  [0x4ee6] = 0x9625, [0x4ee7] = 0x9626, [0x4ee8] = 0x9627, [0x4ee9] = 0x9628,
  [0x4eea] = 0x9629, [0x4eeb] = 0x962b, [0x4eec] = 0x962c, [0x4eed] = 0x962d,
  [0x4eee] = 0x962f, [0x4eef] = 0x9630, [0x4ef0] = 0x9637, [0x4ef1] = 0x9638,
  [0x4ef2] = 0x9639, [0x4ef3] = 0x963a, [0x4ef4] = 0x963e, [0x4ef5] = 0x9641,
  [0x4ef6] = 0x9643, [0x4ef7] = 0x964a, [0x4ef8] = 0x964e, [0x4ef9] = 0x964f,
  [0x4efa] = 0x9651, [0x4efb] = 0x9652, [0x4efc] = 0x9653, [0x4efd] = 0x9656,
  [0x4efe] = 0x9657, [0x4f00] = 0x9658, [0x4f01] = 0x9659, [0x4f02] = 0x965a,
  [0x4f03] = 0x965c, [0x4f04] = 0x965d, [0x4f05] = 0x965e, [0x4f06] = 0x9660,
  [0x4f07] = 0x9663, [0x4f08] = 0x9665, [0x4f09] = 0x9666, [0x4f0a] = 0x966b,
  [0x4f0b] = 0x966d, [0x4f0c] = 0x966e, [0x4f0d] = 0x966f, [0x4f0e] = 0x9670,
  [0x4f0f] = 0x9671, [0x4f10] = 0x9673, [0x4f11] = 0x9678, [0x4f12] = 0x9679,
  [0x4f13] = 0x967a, [0x4f14] = 0x967b, [0x4f15] = 0x967c, [0x4f16] = 0x967d,
  [0x4f17] = 0x967e, [0x4f18] = 0x967f, [0x4f19] = 0x9680, [0x4f1a] = 0x9681,
  [0x4f1b] = 0x9682, [0x4f1c] = 0x9683, [0x4f1d] = 0x9684, [0x4f1e] = 0x9687,
  [0x4f1f] = 0x9689, [0x4f20] = 0x968a, [0x4f21] = 0x8f8d, [0x4f22] = 0x8f8e,
  [0x4f23] = 0x8f8f, [0x4f24] = 0x8f98, [0x4f25] = 0x8f9a, [0x4f26] = 0x8ece,
  [0x4f27] = 0x620b, [0x4f28] = 0x6217, [0x4f29] = 0x621b, [0x4f2a] = 0x621f,
  [0x4f2b] = 0x6222, [0x4f2c] = 0x6221, [0x4f2d] = 0x6225, [0x4f2e] = 0x6224,
  [0x4f2f] = 0x622c, [0x4f30] = 0x81e7, [0x4f31] = 0x74ef, [0x4f32] = 0x74f4,
  [0x4f33] = 0x74ff, [0x4f34] = 0x750f, [0x4f35] = 0x7511, [0x4f36] = 0x7513,
  [0x4f37] = 0x6534, [0x4f38] = 0x65ee, [0x4f39] = 0x65ef, [0x4f3a] = 0x65f0,
  [0x4f3b] = 0x660a, [0x4f3c] = 0x6619, [0x4f3d] = 0x6772, [0x4f3e] = 0x6603,
  [0x4f3f] = 0x6615, [0x4f40] = 0x6600, [0x4f41] = 0x7085, [0x4f42] = 0x66f7,
  [0x4f43] = 0x661d, [0x4f44] = 0x6634, [0x4f45] = 0x6631, [0x4f46] = 0x6636,
  [0x4f47] = 0x6635, [0x4f48] = 0x8006, [0x4f49] = 0x665f, [0x4f4a] = 0x6654,
  [0x4f4b] = 0x6641, [0x4f4c] = 0x664f, [0x4f4d] = 0x6656, [0x4f4e] = 0x6661,
  [0x4f4f] = 0x6657, [0x4f50] = 0x6677, [0x4f51] = 0x6684, [0x4f52] = 0x668c,
  [0x4f53] = 0x66a7, [0x4f54] = 0x669d, [0x4f55] = 0x66be, [0x4f56] = 0x66db,
  [0x4f57] = 0x66dc, [0x4f58] = 0x66e6, [0x4f59] = 0x66e9, [0x4f5a] = 0x8d32,
  [0x4f5b] = 0x8d33, [0x4f5c] = 0x8d36, [0x4f5d] = 0x8d3b, [0x4f5e] = 0x8d3d,
  [0x4f5f] = 0x8d40, [0x4f60] = 0x8d45, [0x4f61] = 0x8d46, [0x4f62] = 0x8d48,
  [0x4f63] = 0x8d49, [0x4f64] = 0x8d47, [0x4f65] = 0x8d4d, [0x4f66] = 0x8d55,
  [0x4f67] = 0x8d59, [0x4f68] = 0x89c7, [0x4f69] = 0x89ca, [0x4f6a] = 0x89cb,
  [0x4f6b] = 0x89cc, [0x4f6c] = 0x89ce, [0x4f6d] = 0x89cf, [0x4f6e] = 0x89d0,
  [0x4f6f] = 0x89d1, [0x4f70] = 0x726e, [0x4f71] = 0x729f, [0x4f72] = 0x725d,
  [0x4f73] = 0x7266, [0x4f74] = 0x726f, [0x4f75] = 0x727e, [0x4f76] = 0x727f,
  [0x4f77] = 0x7284, [0x4f78] = 0x728b, [0x4f79] = 0x728d, [0x4f7a] = 0x728f,
  [0x4f7b] = 0x7292, [0x4f7c] = 0x6308, [0x4f7d] = 0x6332, [0x4f7e] = 0x63b0,
  [0x4f80] = 0x968c, [0x4f81] = 0x968e, [0x4f82] = 0x9691, [0x4f83] = 0x9692,
  [0x4f84] = 0x9693, [0x4f85] = 0x9695, [0x4f86] = 0x9696, [0x4f87] = 0x969a,
  [0x4f88] = 0x969b, [0x4f89] = 0x969d, [0x4f8a] = 0x969e, [0x4f8b] = 0x969f,
  [0x4f8c] = 0x96a0, [0x4f8d] = 0x96a1, [0x4f8e] = 0x96a2, [0x4f8f] = 0x96a3,
  [0x4f90] = 0x96a4, [0x4f91] = 0x96a5, [0x4f92] = 0x96a6, [0x4f93] = 0x96a8,
  [0x4f94] = 0x96a9, [0x4f95] = 0x96aa, [0x4f96] = 0x96ab, [0x4f97] = 0x96ac,
  [0x4f98] = 0x96ad, [0x4f99] = 0x96ae, [0x4f9a] = 0x96af, [0x4f9b] = 0x96b1,
  [0x4f9c] = 0x96b2, [0x4f9d] = 0x96b4, [0x4f9e] = 0x96b5, [0x4f9f] = 0x96b7,
  [0x4fa0] = 0x96b8, [0x4fa1] = 0x96ba, [0x4fa2] = 0x96bb, [0x4fa3] = 0x96bf,
  [0x4fa4] = 0x96c2, [0x4fa5] = 0x96c3, [0x4fa6] = 0x96c8, [0x4fa7] = 0x96ca,
  [0x4fa8] = 0x96cb, [0x4fa9] = 0x96d0, [0x4faa] = 0x96d1, [0x4fab] = 0x96d3,
  [0x4fac] = 0x96d4, [0x4fad] = 0x96d6, [0x4fae] = 0x96d7, [0x4faf] = 0x96d8,
  [0x4fb0] = 0x96d9, [0x4fb1] = 0x96da, [0x4fb2] = 0x96db, [0x4fb3] = 0x96dc,
  [0x4fb4] = 0x96dd, [0x4fb5] = 0x96de, [0x4fb6] = 0x96df, [0x4fb7] = 0x96e1,
  [0x4fb8] = 0x96e2, [0x4fb9] = 0x96e3, [0x4fba] = 0x96e4, [0x4fbb] = 0x96e5,
  [0x4fbc] = 0x96e6, [0x4fbd] = 0x96e7, [0x4fbe] = 0x96eb, [0x4fc0] = 0x96ec,
  [0x4fc1] = 0x96ed, [0x4fc2] = 0x96ee, [0x4fc3] = 0x96f0, [0x4fc4] = 0x96f1,
  [0x4fc5] = 0x96f2, [0x4fc6] = 0x96f4, [0x4fc7] = 0x96f5, [0x4fc8] = 0x96f8,
  [0x4fc9] = 0x96fa, [0x4fca] = 0x96fb, [0x4fcb] = 0x96fc, [0x4fcc] = 0x96fd,
  [0x4fcd] = 0x96ff, [0x4fce] = 0x9702, [0x4fcf] = 0x9703, [0x4fd0] = 0x9705,
  [0x4fd1] = 0x970a, [0x4fd2] = 0x970b, [0x4fd3] = 0x970c, [0x4fd4] = 0x9710,
  [0x4fd5] = 0x9711, [0x4fd6] = 0x9712, [0x4fd7] = 0x9714, [0x4fd8] = 0x9715,
  [0x4fd9] = 0x9717, [0x4fda] = 0x9718, [0x4fdb] = 0x9719, [0x4fdc] = 0x971a,
  [0x4fdd] = 0x971b, [0x4fde] = 0x971d, [0x4fdf] = 0x971f, [0x4fe0] = 0x9720,
  [0x4fe1] = 0x643f, [0x4fe2] = 0x64d8, [0x4fe3] = 0x8004, [0x4fe4] = 0x6bea,
  [0x4fe5] = 0x6bf3, [0x4fe6] = 0x6bfd, [0x4fe7] = 0x6bf5, [0x4fe8] = 0x6bf9,
  [0x4fe9] = 0x6c05, [0x4fea] = 0x6c07, [0x4feb] = 0x6c06, [0x4fec] = 0x6c0d,
  [0x4fed] = 0x6c15, [0x4fee] = 0x6c18, [0x4fef] = 0x6c19, [0x4ff0] = 0x6c1a,
  [0x4ff1] = 0x6c21, [0x4ff2] = 0x6c29, [0x4ff3] = 0x6c24, [0x4ff4] = 0x6c2a,
  [0x4ff5] = 0x6c32, [0x4ff6] = 0x6535, [0x4ff7] = 0x6555, [0x4ff8] = 0x656b,
  [0x4ff9] = 0x724d, [0x4ffa] = 0x7252, [0x4ffb] = 0x7256, [0x4ffc] = 0x7230,
  [0x4ffd] = 0x8662, [0x4ffe] = 0x5216, [0x4fff] = 0x809f, [0x5000] = 0x809c,
  [0x5001] = 0x8093, [0x5002] = 0x80bc, [0x5003] = 0x670a, [0x5004] = 0x80bd,
  [0x5005] = 0x80b1, [0x5006] = 0x80ab, [0x5007] = 0x80ad, [0x5008] = 0x80b4,
  [0x5009] = 0x80b7, [0x500a] = 0x80e7, [0x500b] = 0x80e8, [0x500c] = 0x80e9,
  [0x500d] = 0x80ea, [0x500e] = 0x80db, [0x500f] = 0x80c2, [0x5010] = 0x80c4,
  [0x5011] = 0x80d9, [0x5012] = 0x80cd, [0x5013] = 0x80d7, [0x5014] = 0x6710,
  [0x5015] = 0x80dd, [0x5016] = 0x80eb, [0x5017] = 0x80f1, [0x5018] = 0x80f4,
  [0x5019] = 0x80ed, [0x501a] = 0x810d, [0x501b] = 0x810e, [0x501c] = 0x80f2,
  [0x501d] = 0x80fc, [0x501e] = 0x6715, [0x501f] = 0x8112, [0x5020] = 0x8c5a,
  [0x5021] = 0x8136, [0x5022] = 0x811e, [0x5023] = 0x812c, [0x5024] = 0x8118,
  [0x5025] = 0x8132, [0x5026] = 0x8148, [0x5027] = 0x814c, [0x5028] = 0x8153,
  [0x5029] = 0x8174, [0x502a] = 0x8159, [0x502b] = 0x815a, [0x502c] = 0x8171,
  [0x502d] = 0x8160, [0x502e] = 0x8169, [0x502f] = 0x817c, [0x5030] = 0x817d,
  [0x5031] = 0x816d, [0x5032] = 0x8167, [0x5033] = 0x584d, [0x5034] = 0x5ab5,
  [0x5035] = 0x8188, [0x5036] = 0x8182, [0x5037] = 0x8191, [0x5038] = 0x6ed5,
  [0x5039] = 0x81a3, [0x503a] = 0x81aa, [0x503b] = 0x81cc, [0x503c] = 0x6726,
  [0x503d] = 0x81ca, [0x503e] = 0x81bb, [0x5040] = 0x9721, [0x5041] = 0x9722,
  [0x5042] = 0x9723, [0x5043] = 0x9724, [0x5044] = 0x9725, [0x5045] = 0x9726,
  [0x5046] = 0x9727, [0x5047] = 0x9728, [0x5048] = 0x9729, [0x5049] = 0x972b,
  [0x504a] = 0x972c, [0x504b] = 0x972e, [0x504c] = 0x972f, [0x504d] = 0x9731,
  [0x504e] = 0x9733, [0x504f] = 0x9734, [0x5050] = 0x9735, [0x5051] = 0x9736,
  [0x5052] = 0x9737, [0x5053] = 0x973a, [0x5054] = 0x973b, [0x5055] = 0x973c,
  [0x5056] = 0x973d, [0x5057] = 0x973f, [0x5058] = 0x9740, [0x5059] = 0x9741,
  [0x505a] = 0x9742, [0x505b] = 0x9743, [0x505c] = 0x9744, [0x505d] = 0x9745,
  [0x505e] = 0x9746, [0x505f] = 0x9747, [0x5060] = 0x9748, [0x5061] = 0x9749,
  [0x5062] = 0x974a, [0x5063] = 0x974b, [0x5064] = 0x974c, [0x5065] = 0x974d,
  [0x5066] = 0x974e, [0x5067] = 0x974f, [0x5068] = 0x9750, [0x5069] = 0x9751,
  [0x506a] = 0x9754, [0x506b] = 0x9755, [0x506c] = 0x9757, [0x506d] = 0x9758,
  [0x506e] = 0x975a, [0x506f] = 0x975c, [0x5070] = 0x975d, [0x5071] = 0x975f,
  [0x5072] = 0x9763, [0x5073] = 0x9764, [0x5074] = 0x9766, [0x5075] = 0x9767,
  [0x5076] = 0x9768, [0x5077] = 0x976a, [0x5078] = 0x976b, [0x5079] = 0x976c,
  [0x507a] = 0x976d, [0x507b] = 0x976e, [0x507c] = 0x976f, [0x507d] = 0x9770,
  [0x507e] = 0x9771, [0x5080] = 0x9772, [0x5081] = 0x9775, [0x5082] = 0x9777,
  [0x5083] = 0x9778, [0x5084] = 0x9779, [0x5085] = 0x977a, [0x5086] = 0x977b,
  [0x5087] = 0x977d, [0x5088] = 0x977e, [0x5089] = 0x977f, [0x508a] = 0x9780,
  [0x508b] = 0x9781, [0x508c] = 0x9782, [0x508d] = 0x9783, [0x508e] = 0x9784,
  [0x508f] = 0x9786, [0x5090] = 0x9787, [0x5091] = 0x9788, [0x5092] = 0x9789,
  [0x5093] = 0x978a, [0x5094] = 0x978c, [0x5095] = 0x978e, [0x5096] = 0x978f,
  [0x5097] = 0x9790, [0x5098] = 0x9793, [0x5099] = 0x9795, [0x509a] = 0x9796,
  [0x509b] = 0x9797, [0x509c] = 0x9799, [0x509d] = 0x979a, [0x509e] = 0x979b,
  [0x509f] = 0x979c, [0x50a0] = 0x979d, [0x50a1] = 0x81c1, [0x50a2] = 0x81a6,
  [0x50a3] = 0x6b24, [0x50a4] = 0x6b37, [0x50a5] = 0x6b39, [0x50a6] = 0x6b43,
  [0x50a7] = 0x6b46, [0x50a8] = 0x6b59, [0x50a9] = 0x98d1, [0x50aa] = 0x98d2,
  [0x50ab] = 0x98d3, [0x50ac] = 0x98d5, [0x50ad] = 0x98d9, [0x50ae] = 0x98da,
  [0x50af] = 0x6bb3, [0x50b0] = 0x5f40, [0x50b1] = 0x6bc2, [0x50b2] = 0x89f3,
  [0x50b3] = 0x6590, [0x50b4] = 0x9f51, [0x50b5] = 0x6593, [0x50b6] = 0x65bc,
  [0x50b7] = 0x65c6, [0x50b8] = 0x65c4, [0x50b9] = 0x65c3, [0x50ba] = 0x65cc,
  [0x50bb] = 0x65ce, [0x50bc] = 0x65d2, [0x50bd] = 0x65d6, [0x50be] = 0x7080,
  [0x50bf] = 0x709c, [0x50c0] = 0x7096, [0x50c1] = 0x709d, [0x50c2] = 0x70bb,
  [0x50c3] = 0x70c0, [0x50c4] = 0x70b7, [0x50c5] = 0x70ab, [0x50c6] = 0x70b1,
  [0x50c7] = 0x70e8, [0x50c8] = 0x70ca, [0x50c9] = 0x7110, [0x50ca] = 0x7113,
  [0x50cb] = 0x7116, [0x50cc] = 0x712f, [0x50cd] = 0x7131, [0x50ce] = 0x7173,
  [0x50cf] = 0x715c, [0x50d0] = 0x7168, [0x50d1] = 0x7145, [0x50d2] = 0x7172,
  [0x50d3] = 0x714a, [0x50d4] = 0x7178, [0x50d5] = 0x717a, [0x50d6] = 0x7198,
  [0x50d7] = 0x71b3, [0x50d8] = 0x71b5, [0x50d9] = 0x71a8, [0x50da] = 0x71a0,
  [0x50db] = 0x71e0, [0x50dc] = 0x71d4, [0x50dd] = 0x71e7, [0x50de] = 0x71f9,
  [0x50df] = 0x721d, [0x50e0] = 0x7228, [0x50e1] = 0x706c, [0x50e2] = 0x7118,
  [0x50e3] = 0x7166, [0x50e4] = 0x71b9, [0x50e5] = 0x623e, [0x50e6] = 0x623d,
  [0x50e7] = 0x6243, [0x50e8] = 0x6248, [0x50e9] = 0x6249, [0x50ea] = 0x793b,
  [0x50eb] = 0x7940, [0x50ec] = 0x7946, [0x50ed] = 0x7949, [0x50ee] = 0x795b,
  [0x50ef] = 0x795c, [0x50f0] = 0x7953, [0x50f1] = 0x795a, [0x50f2] = 0x7962,
  [0x50f3] = 0x7957, [0x50f4] = 0x7960, [0x50f5] = 0x796f, [0x50f6] = 0x7967,
  [0x50f7] = 0x797a, [0x50f8] = 0x7985, [0x50f9] = 0x798a, [0x50fa] = 0x799a,
  [0x50fb] = 0x79a7, [0x50fc] = 0x79b3, [0x50fd] = 0x5fd1, [0x50fe] = 0x5fd0,
  [0x5100] = 0x979e, [0x5101] = 0x979f, [0x5102] = 0x97a1, [0x5103] = 0x97a2,
  [0x5104] = 0x97a4, [0x5105] = 0x97a5, [0x5106] = 0x97a6, [0x5107] = 0x97a7,
  [0x5108] = 0x97a8, [0x5109] = 0x97a9, [0x510a] = 0x97aa, [0x510b] = 0x97ac,
  [0x510c] = 0x97ae, [0x510d] = 0x97b0, [0x510e] = 0x97b1, [0x510f] = 0x97b3,
  [0x5110] = 0x97b5, [0x5111] = 0x97b6, [0x5112] = 0x97b7, [0x5113] = 0x97b8,
  [0x5114] = 0x97b9, [0x5115] = 0x97ba, [0x5116] = 0x97bb, [0x5117] = 0x97bc,
  [0x5118] = 0x97bd, [0x5119] = 0x97be, [0x511a] = 0x97bf, [0x511b] = 0x97c0,
  [0x511c] = 0x97c1, [0x511d] = 0x97c2, [0x511e] = 0x97c3, [0x511f] = 0x97c4,
  [0x5120] = 0x97c5, [0x5121] = 0x97c6, [0x5122] = 0x97c7, [0x5123] = 0x97c8,
  [0x5124] = 0x97c9, [0x5125] = 0x97ca, [0x5126] = 0x97cb, [0x5127] = 0x97cc,
  [0x5128] = 0x97cd, [0x5129] = 0x97ce, [0x512a] = 0x97cf, [0x512b] = 0x97d0,
  [0x512c] = 0x97d1, [0x512d] = 0x97d2, [0x512e] = 0x97d3, [0x512f] = 0x97d4,
  [0x5130] = 0x97d5, [0x5131] = 0x97d6, [0x5132] = 0x97d7, [0x5133] = 0x97d8,
  [0x5134] = 0x97d9, [0x5135] = 0x97da, [0x5136] = 0x97db, [0x5137] = 0x97dc,
  [0x5138] = 0x97dd, [0x5139] = 0x97de, [0x513a] = 0x97df, [0x513b] = 0x97e0,
  [0x513c] = 0x97e1, [0x513d] = 0x97e2, [0x513e] = 0x97e3, [0x5140] = 0x97e4,
  [0x5141] = 0x97e5, [0x5142] = 0x97e8, [0x5143] = 0x97ee, [0x5144] = 0x97ef,
  [0x5145] = 0x97f0, [0x5146] = 0x97f1, [0x5147] = 0x97f2, [0x5148] = 0x97f4,
  [0x5149] = 0x97f7, [0x514a] = 0x97f8, [0x514b] = 0x97f9, [0x514c] = 0x97fa,
  [0x514d] = 0x97fb, [0x514e] = 0x97fc, [0x514f] = 0x97fd, [0x5150] = 0x97fe,
  [0x5151] = 0x97ff, [0x5152] = 0x9800, [0x5153] = 0x9801, [0x5154] = 0x9802,
  [0x5155] = 0x9803, [0x5156] = 0x9804, [0x5157] = 0x9805, [0x5158] = 0x9806,
  [0x5159] = 0x9807, [0x515a] = 0x9808, [0x515b] = 0x9809, [0x515c] = 0x980a,
  [0x515d] = 0x980b, [0x515e] = 0x980c, [0x515f] = 0x980d, [0x5160] = 0x980e,
  [0x5161] = 0x603c, [0x5162] = 0x605d, [0x5163] = 0x605a, [0x5164] = 0x6067,
  [0x5165] = 0x6041, [0x5166] = 0x6059, [0x5167] = 0x6063, [0x5168] = 0x60ab,
  [0x5169] = 0x6106, [0x516a] = 0x610d, [0x516b] = 0x615d, [0x516c] = 0x61a9,
  [0x516d] = 0x619d, [0x516e] = 0x61cb, [0x516f] = 0x61d1, [0x5170] = 0x6206,
  [0x5171] = 0x8080, [0x5172] = 0x807f, [0x5173] = 0x6c93, [0x5174] = 0x6cf6,
  [0x5175] = 0x6dfc, [0x5176] = 0x77f6, [0x5177] = 0x77f8, [0x5178] = 0x7800,
  [0x5179] = 0x7809, [0x517a] = 0x7817, [0x517b] = 0x7818, [0x517c] = 0x7811,
  [0x517d] = 0x65ab, [0x517e] = 0x782d, [0x517f] = 0x781c, [0x5180] = 0x781d,
  [0x5181] = 0x7839, [0x5182] = 0x783a, [0x5183] = 0x783b, [0x5184] = 0x781f,
  [0x5185] = 0x783c, [0x5186] = 0x7825, [0x5187] = 0x782c, [0x5188] = 0x7823,
  [0x5189] = 0x7829, [0x518a] = 0x784e, [0x518b] = 0x786d, [0x518c] = 0x7856,
  [0x518d] = 0x7857, [0x518e] = 0x7826, [0x518f] = 0x7850, [0x5190] = 0x7847,
  [0x5191] = 0x784c, [0x5192] = 0x786a, [0x5193] = 0x789b, [0x5194] = 0x7893,
  [0x5195] = 0x789a, [0x5196] = 0x7887, [0x5197] = 0x789c, [0x5198] = 0x78a1,
  [0x5199] = 0x78a3, [0x519a] = 0x78b2, [0x519b] = 0x78b9, [0x519c] = 0x78a5,
  [0x519d] = 0x78d4, [0x519e] = 0x78d9, [0x519f] = 0x78c9, [0x51a0] = 0x78ec,
  [0x51a1] = 0x78f2, [0x51a2] = 0x7905, [0x51a3] = 0x78f4, [0x51a4] = 0x7913,
  [0x51a5] = 0x7924, [0x51a6] = 0x791e, [0x51a7] = 0x7934, [0x51a8] = 0x9f9b,
  [0x51a9] = 0x9ef9, [0x51aa] = 0x9efb, [0x51ab] = 0x9efc, [0x51ac] = 0x76f1,
  [0x51ad] = 0x7704, [0x51ae] = 0x770d, [0x51af] = 0x76f9, [0x51b0] = 0x7707,
  [0x51b1] = 0x7708, [0x51b2] = 0x771a, [0x51b3] = 0x7722, [0x51b4] = 0x7719,
  [0x51b5] = 0x772d, [0x51b6] = 0x7726, [0x51b7] = 0x7735, [0x51b8] = 0x7738,
  [0x51b9] = 0x7750, [0x51ba] = 0x7751, [0x51bb] = 0x7747, [0x51bc] = 0x7743,
  [0x51bd] = 0x775a, [0x51be] = 0x7768, [0x51c0] = 0x980f, [0x51c1] = 0x9810,
  [0x51c2] = 0x9811, [0x51c3] = 0x9812, [0x51c4] = 0x9813, [0x51c5] = 0x9814,
  [0x51c6] = 0x9815, [0x51c7] = 0x9816, [0x51c8] = 0x9817, [0x51c9] = 0x9818,
  [0x51ca] = 0x9819, [0x51cb] = 0x981a, [0x51cc] = 0x981b, [0x51cd] = 0x981c,
  [0x51ce] = 0x981d, [0x51cf] = 0x981e, [0x51d0] = 0x981f, [0x51d1] = 0x9820,
  [0x51d2] = 0x9821, [0x51d3] = 0x9822, [0x51d4] = 0x9823, [0x51d5] = 0x9824,
  [0x51d6] = 0x9825, [0x51d7] = 0x9826, [0x51d8] = 0x9827, [0x51d9] = 0x9828,
  [0x51da] = 0x9829, [0x51db] = 0x982a, [0x51dc] = 0x982b, [0x51dd] = 0x982c,
  [0x51de] = 0x982d, [0x51df] = 0x982e, [0x51e0] = 0x982f, [0x51e1] = 0x9830,
  [0x51e2] = 0x9831, [0x51e3] = 0x9832, [0x51e4] = 0x9833, [0x51e5] = 0x9834,
  [0x51e6] = 0x9835, [0x51e7] = 0x9836, [0x51e8] = 0x9837, [0x51e9] = 0x9838,
  [0x51ea] = 0x9839, [0x51eb] = 0x983a, [0x51ec] = 0x983b, [0x51ed] = 0x983c,
  [0x51ee] = 0x983d, [0x51ef] = 0x983e, [0x51f0] = 0x983f, [0x51f1] = 0x9840,
  [0x51f2] = 0x9841, [0x51f3] = 0x9842, [0x51f4] = 0x9843, [0x51f5] = 0x9844,
  [0x51f6] = 0x9845, [0x51f7] = 0x9846, [0x51f8] = 0x9847, [0x51f9] = 0x9848,
  [0x51fa] = 0x9849, [0x51fb] = 0x984a, [0x51fc] = 0x984b, [0x51fd] = 0x984c,
  [0x51fe] = 0x984d, [0x5200] = 0x984e, [0x5201] = 0x984f, [0x5202] = 0x9850,
  [0x5203] = 0x9851, [0x5204] = 0x9852, [0x5205] = 0x9853, [0x5206] = 0x9854,
  [0x5207] = 0x9855, [0x5208] = 0x9856, [0x5209] = 0x9857, [0x520a] = 0x9858,
  [0x520b] = 0x9859, [0x520c] = 0x985a, [0x520d] = 0x985b, [0x520e] = 0x985c,
  [0x520f] = 0x985d, [0x5210] = 0x985e, [0x5211] = 0x985f, [0x5212] = 0x9860,
  [0x5213] = 0x9861, [0x5214] = 0x9862, [0x5215] = 0x9863, [0x5216] = 0x9864,
  [0x5217] = 0x9865, [0x5218] = 0x9866, [0x5219] = 0x9867, [0x521a] = 0x9868,
  [0x521b] = 0x9869, [0x521c] = 0x986a, [0x521d] = 0x986b, [0x521e] = 0x986c,
  [0x521f] = 0x986d, [0x5220] = 0x986e, [0x5221] = 0x7762, [0x5222] = 0x7765,
  [0x5223] = 0x777f, [0x5224] = 0x778d, [0x5225] = 0x777d, [0x5226] = 0x7780,
  [0x5227] = 0x778c, [0x5228] = 0x7791, [0x5229] = 0x779f, [0x522a] = 0x77a0,
  [0x522b] = 0x77b0, [0x522c] = 0x77b5, [0x522d] = 0x77bd, [0x522e] = 0x753a,
  [0x522f] = 0x7540, [0x5230] = 0x754e, [0x5231] = 0x754b, [0x5232] = 0x7548,
  [0x5233] = 0x755b, [0x5234] = 0x7572, [0x5235] = 0x7579, [0x5236] = 0x7583,
  [0x5237] = 0x7f58, [0x5238] = 0x7f61, [0x5239] = 0x7f5f, [0x523a] = 0x8a48,
  [0x523b] = 0x7f68, [0x523c] = 0x7f74, [0x523d] = 0x7f71, [0x523e] = 0x7f79,
  [0x523f] = 0x7f81, [0x5240] = 0x7f7e, [0x5241] = 0x76cd, [0x5242] = 0x76e5,
  [0x5243] = 0x8832, [0x5244] = 0x9485, [0x5245] = 0x9486, [0x5246] = 0x9487,
  [0x5247] = 0x948b, [0x5248] = 0x948a, [0x5249] = 0x948c, [0x524a] = 0x948d,
  [0x524b] = 0x948f, [0x524c] = 0x9490, [0x524d] = 0x9494, [0x524e] = 0x9497,
  [0x524f] = 0x9495, [0x5250] = 0x949a, [0x5251] = 0x949b, [0x5252] = 0x949c,
  [0x5253] = 0x94a3, [0x5254] = 0x94a4, [0x5255] = 0x94ab, [0x5256] = 0x94aa,
  [0x5257] = 0x94ad, [0x5258] = 0x94ac, [0x5259] = 0x94af, [0x525a] = 0x94b0,
  [0x525b] = 0x94b2, [0x525c] = 0x94b4, [0x525d] = 0x94b6, [0x525e] = 0x94b7,
  [0x525f] = 0x94b8, [0x5260] = 0x94b9, [0x5261] = 0x94ba, [0x5262] = 0x94bc,
  [0x5263] = 0x94bd, [0x5264] = 0x94bf, [0x5265] = 0x94c4, [0x5266] = 0x94c8,
  [0x5267] = 0x94c9, [0x5268] = 0x94ca, [0x5269] = 0x94cb, [0x526a] = 0x94cc,
  [0x526b] = 0x94cd, [0x526c] = 0x94ce, [0x526d] = 0x94d0, [0x526e] = 0x94d1,
  [0x526f] = 0x94d2, [0x5270] = 0x94d5, [0x5271] = 0x94d6, [0x5272] = 0x94d7,
  [0x5273] = 0x94d9, [0x5274] = 0x94d8, [0x5275] = 0x94db, [0x5276] = 0x94de,
  [0x5277] = 0x94df, [0x5278] = 0x94e0, [0x5279] = 0x94e2, [0x527a] = 0x94e4,
  [0x527b] = 0x94e5, [0x527c] = 0x94e7, [0x527d] = 0x94e8, [0x527e] = 0x94ea,
  [0x5280] = 0x986f, [0x5281] = 0x9870, [0x5282] = 0x9871, [0x5283] = 0x9872,
  [0x5284] = 0x9873, [0x5285] = 0x9874, [0x5286] = 0x988b, [0x5287] = 0x988e,
  [0x5288] = 0x9892, [0x5289] = 0x9895, [0x528a] = 0x9899, [0x528b] = 0x98a3,
  [0x528c] = 0x98a8, [0x528d] = 0x98a9, [0x528e] = 0x98aa, [0x528f] = 0x98ab,
  [0x5290] = 0x98ac, [0x5291] = 0x98ad, [0x5292] = 0x98ae, [0x5293] = 0x98af,
  [0x5294] = 0x98b0, [0x5295] = 0x98b1, [0x5296] = 0x98b2, [0x5297] = 0x98b3,
  [0x5298] = 0x98b4, [0x5299] = 0x98b5, [0x529a] = 0x98b6, [0x529b] = 0x98b7,
  [0x529c] = 0x98b8, [0x529d] = 0x98b9, [0x529e] = 0x98ba, [0x529f] = 0x98bb,
  [0x52a0] = 0x98bc, [0x52a1] = 0x98bd, [0x52a2] = 0x98be, [0x52a3] = 0x98bf,
  [0x52a4] = 0x98c0, [0x52a5] = 0x98c1, [0x52a6] = 0x98c2, [0x52a7] = 0x98c3,
  [0x52a8] = 0x98c4, [0x52a9] = 0x98c5, [0x52aa] = 0x98c6, [0x52ab] = 0x98c7,
  [0x52ac] = 0x98c8, [0x52ad] = 0x98c9, [0x52ae] = 0x98ca, [0x52af] = 0x98cb,
  [0x52b0] = 0x98cc, [0x52b1] = 0x98cd, [0x52b2] = 0x98cf, [0x52b3] = 0x98d0,
  [0x52b4] = 0x98d4, [0x52b5] = 0x98d6, [0x52b6] = 0x98d7, [0x52b7] = 0x98db,
  [0x52b8] = 0x98dc, [0x52b9] = 0x98dd, [0x52ba] = 0x98e0, [0x52bb] = 0x98e1,
  [0x52bc] = 0x98e2, [0x52bd] = 0x98e3, [0x52be] = 0x98e4, [0x52c0] = 0x98e5,
  [0x52c1] = 0x98e6, [0x52c2] = 0x98e9, [0x52c3] = 0x98ea, [0x52c4] = 0x98eb,
  [0x52c5] = 0x98ec, [0x52c6] = 0x98ed, [0x52c7] = 0x98ee, [0x52c8] = 0x98ef,
  [0x52c9] = 0x98f0, [0x52ca] = 0x98f1, [0x52cb] = 0x98f2, [0x52cc] = 0x98f3,
  [0x52cd] = 0x98f4, [0x52ce] = 0x98f5, [0x52cf] = 0x98f6, [0x52d0] = 0x98f7,
  [0x52d1] = 0x98f8, [0x52d2] = 0x98f9, [0x52d3] = 0x98fa, [0x52d4] = 0x98fb,
  [0x52d5] = 0x98fc, [0x52d6] = 0x98fd, [0x52d7] = 0x98fe, [0x52d8] = 0x98ff,
  [0x52d9] = 0x9900, [0x52da] = 0x9901, [0x52db] = 0x9902, [0x52dc] = 0x9903,
  [0x52dd] = 0x9904, [0x52de] = 0x9905, [0x52df] = 0x9906, [0x52e0] = 0x9907,
  [0x52e1] = 0x94e9, [0x52e2] = 0x94eb, [0x52e3] = 0x94ee, [0x52e4] = 0x94ef,
  [0x52e5] = 0x94f3, [0x52e6] = 0x94f4, [0x52e7] = 0x94f5, [0x52e8] = 0x94f7,
  [0x52e9] = 0x94f9, [0x52ea] = 0x94fc, [0x52eb] = 0x94fd, [0x52ec] = 0x94ff,
  [0x52ed] = 0x9503, [0x52ee] = 0x9502, [0x52ef] = 0x9506, [0x52f0] = 0x9507,
  [0x52f1] = 0x9509, [0x52f2] = 0x950a, [0x52f3] = 0x950d, [0x52f4] = 0x950e,
  [0x52f5] = 0x950f, [0x52f6] = 0x9512, [0x52f7] = 0x9513, [0x52f8] = 0x9514,
  [0x52f9] = 0x9515, [0x52fa] = 0x9516, [0x52fb] = 0x9518, [0x52fc] = 0x951b,
  [0x52fd] = 0x951d, [0x52fe] = 0x951e, [0x52ff] = 0x951f, [0x5300] = 0x9522,
  [0x5301] = 0x952a, [0x5302] = 0x952b, [0x5303] = 0x9529, [0x5304] = 0x952c,
  [0x5305] = 0x9531, [0x5306] = 0x9532, [0x5307] = 0x9534, [0x5308] = 0x9536,
  [0x5309] = 0x9537, [0x530a] = 0x9538, [0x530b] = 0x953c, [0x530c] = 0x953e,
  [0x530d] = 0x953f, [0x530e] = 0x9542, [0x530f] = 0x9535, [0x5310] = 0x9544,
  [0x5311] = 0x9545, [0x5312] = 0x9546, [0x5313] = 0x9549, [0x5314] = 0x954c,
  [0x5315] = 0x954e, [0x5316] = 0x954f, [0x5317] = 0x9552, [0x5318] = 0x9553,
  [0x5319] = 0x9554, [0x531a] = 0x9556, [0x531b] = 0x9557, [0x531c] = 0x9558,
  [0x531d] = 0x9559, [0x531e] = 0x955b, [0x531f] = 0x955e, [0x5320] = 0x955f,
  [0x5321] = 0x955d, [0x5322] = 0x9561, [0x5323] = 0x9562, [0x5324] = 0x9564,
  [0x5325] = 0x9565, [0x5326] = 0x9566, [0x5327] = 0x9567, [0x5328] = 0x9568,
  [0x5329] = 0x9569, [0x532a] = 0x956a, [0x532b] = 0x956b, [0x532c] = 0x956c,
  [0x532d] = 0x956f, [0x532e] = 0x9571, [0x532f] = 0x9572, [0x5330] = 0x9573,
  [0x5331] = 0x953a, [0x5332] = 0x77e7, [0x5333] = 0x77ec, [0x5334] = 0x96c9,
  [0x5335] = 0x79d5, [0x5336] = 0x79ed, [0x5337] = 0x79e3, [0x5338] = 0x79eb,
  [0x5339] = 0x7a06, [0x533a] = 0x5d47, [0x533b] = 0x7a03, [0x533c] = 0x7a02,
  [0x533d] = 0x7a1e, [0x533e] = 0x7a14, [0x5340] = 0x9908, [0x5341] = 0x9909,
  [0x5342] = 0x990a, [0x5343] = 0x990b, [0x5344] = 0x990c, [0x5345] = 0x990e,
  [0x5346] = 0x990f, [0x5347] = 0x9911, [0x5348] = 0x9912, [0x5349] = 0x9913,
  [0x534a] = 0x9914, [0x534b] = 0x9915, [0x534c] = 0x9916, [0x534d] = 0x9917,
  [0x534e] = 0x9918, [0x534f] = 0x9919, [0x5350] = 0x991a, [0x5351] = 0x991b,
  [0x5352] = 0x991c, [0x5353] = 0x991d, [0x5354] = 0x991e, [0x5355] = 0x991f,
  [0x5356] = 0x9920, [0x5357] = 0x9921, [0x5358] = 0x9922, [0x5359] = 0x9923,
  [0x535a] = 0x9924, [0x535b] = 0x9925, [0x535c] = 0x9926, [0x535d] = 0x9927,
  [0x535e] = 0x9928, [0x535f] = 0x9929, [0x5360] = 0x992a, [0x5361] = 0x992b,
  [0x5362] = 0x992c, [0x5363] = 0x992d, [0x5364] = 0x992f, [0x5365] = 0x9930,
  [0x5366] = 0x9931, [0x5367] = 0x9932, [0x5368] = 0x9933, [0x5369] = 0x9934,
  [0x536a] = 0x9935, [0x536b] = 0x9936, [0x536c] = 0x9937, [0x536d] = 0x9938,
  [0x536e] = 0x9939, [0x536f] = 0x993a, [0x5370] = 0x993b, [0x5371] = 0x993c,
  [0x5372] = 0x993d, [0x5373] = 0x993e, [0x5374] = 0x993f, [0x5375] = 0x9940,
  [0x5376] = 0x9941, [0x5377] = 0x9942, [0x5378] = 0x9943, [0x5379] = 0x9944,
  [0x537a] = 0x9945, [0x537b] = 0x9946, [0x537c] = 0x9947, [0x537d] = 0x9948,
  [0x537e] = 0x9949, [0x5380] = 0x994a, [0x5381] = 0x994b, [0x5382] = 0x994c,
  [0x5383] = 0x994d, [0x5384] = 0x994e, [0x5385] = 0x994f, [0x5386] = 0x9950,
  [0x5387] = 0x9951, [0x5388] = 0x9952, [0x5389] = 0x9953, [0x538a] = 0x9956,
  [0x538b] = 0x9957, [0x538c] = 0x9958, [0x538d] = 0x9959, [0x538e] = 0x995a,
  [0x538f] = 0x995b, [0x5390] = 0x995c, [0x5391] = 0x995d, [0x5392] = 0x995e,
  [0x5393] = 0x995f, [0x5394] = 0x9960, [0x5395] = 0x9961, [0x5396] = 0x9962,
  [0x5397] = 0x9964, [0x5398] = 0x9966, [0x5399] = 0x9973, [0x539a] = 0x9978,
  [0x539b] = 0x9979, [0x539c] = 0x997b, [0x539d] = 0x997e, [0x539e] = 0x9982,
  [0x539f] = 0x9983, [0x53a0] = 0x9989, [0x53a1] = 0x7a39, [0x53a2] = 0x7a37,
  [0x53a3] = 0x7a51, [0x53a4] = 0x9ecf, [0x53a5] = 0x99a5, [0x53a6] = 0x7a70,
  [0x53a7] = 0x7688, [0x53a8] = 0x768e, [0x53a9] = 0x7693, [0x53aa] = 0x7699,
  [0x53ab] = 0x76a4, [0x53ac] = 0x74de, [0x53ad] = 0x74e0, [0x53ae] = 0x752c,
  [0x53af] = 0x9e20, [0x53b0] = 0x9e22, [0x53b1] = 0x9e28, [0x53b2] = 0x9e29,
  [0x53b3] = 0x9e2a, [0x53b4] = 0x9e2b, [0x53b5] = 0x9e2c, [0x53b6] = 0x9e32,
  [0x53b7] = 0x9e31, [0x53b8] = 0x9e36, [0x53b9] = 0x9e38, [0x53ba] = 0x9e37,
  [0x53bb] = 0x9e39, [0x53bc] = 0x9e3a, [0x53bd] = 0x9e3e, [0x53be] = 0x9e41,
  [0x53bf] = 0x9e42, [0x53c0] = 0x9e44, [0x53c1] = 0x9e46, [0x53c2] = 0x9e47,
  [0x53c3] = 0x9e48, [0x53c4] = 0x9e49, [0x53c5] = 0x9e4b, [0x53c6] = 0x9e4c,
  [0x53c7] = 0x9e4e, [0x53c8] = 0x9e51, [0x53c9] = 0x9e55, [0x53ca] = 0x9e57,
  [0x53cb] = 0x9e5a, [0x53cc] = 0x9e5b, [0x53cd] = 0x9e5c, [0x53ce] = 0x9e5e,
  [0x53cf] = 0x9e63, [0x53d0] = 0x9e66, [0x53d1] = 0x9e67, [0x53d2] = 0x9e68,
  [0x53d3] = 0x9e69, [0x53d4] = 0x9e6a, [0x53d5] = 0x9e6b, [0x53d6] = 0x9e6c,
  [0x53d7] = 0x9e71, [0x53d8] = 0x9e6d, [0x53d9] = 0x9e73, [0x53da] = 0x7592,
  [0x53db] = 0x7594, [0x53dc] = 0x7596, [0x53dd] = 0x75a0, [0x53de] = 0x759d,
  [0x53df] = 0x75ac, [0x53e0] = 0x75a3, [0x53e1] = 0x75b3, [0x53e2] = 0x75b4,
  [0x53e3] = 0x75b8, [0x53e4] = 0x75c4, [0x53e5] = 0x75b1, [0x53e6] = 0x75b0,
  [0x53e7] = 0x75c3, [0x53e8] = 0x75c2, [0x53e9] = 0x75d6, [0x53ea] = 0x75cd,
  [0x53eb] = 0x75e3, [0x53ec] = 0x75e8, [0x53ed] = 0x75e6, [0x53ee] = 0x75e4,
  [0x53ef] = 0x75eb, [0x53f0] = 0x75e7, [0x53f1] = 0x7603, [0x53f2] = 0x75f1,
  [0x53f3] = 0x75fc, [0x53f4] = 0x75ff, [0x53f5] = 0x7610, [0x53f6] = 0x7600,
  [0x53f7] = 0x7605, [0x53f8] = 0x760c, [0x53f9] = 0x7617, [0x53fa] = 0x760a,
  [0x53fb] = 0x7625, [0x53fc] = 0x7618, [0x53fd] = 0x7615, [0x53fe] = 0x7619,
  [0x5400] = 0x998c, [0x5401] = 0x998e, [0x5402] = 0x999a, [0x5403] = 0x999b,
  [0x5404] = 0x999c, [0x5405] = 0x999d, [0x5406] = 0x999e, [0x5407] = 0x999f,
  [0x5408] = 0x99a0, [0x5409] = 0x99a1, [0x540a] = 0x99a2, [0x540b] = 0x99a3,
  [0x540c] = 0x99a4, [0x540d] = 0x99a6, [0x540e] = 0x99a7, [0x540f] = 0x99a9,
  [0x5410] = 0x99aa, [0x5411] = 0x99ab, [0x5412] = 0x99ac, [0x5413] = 0x99ad,
  [0x5414] = 0x99ae, [0x5415] = 0x99af, [0x5416] = 0x99b0, [0x5417] = 0x99b1,
  [0x5418] = 0x99b2, [0x5419] = 0x99b3, [0x541a] = 0x99b4, [0x541b] = 0x99b5,
  [0x541c] = 0x99b6, [0x541d] = 0x99b7, [0x541e] = 0x99b8, [0x541f] = 0x99b9,
  [0x5420] = 0x99ba, [0x5421] = 0x99bb, [0x5422] = 0x99bc, [0x5423] = 0x99bd,
  [0x5424] = 0x99be, [0x5425] = 0x99bf, [0x5426] = 0x99c0, [0x5427] = 0x99c1,
  [0x5428] = 0x99c2, [0x5429] = 0x99c3, [0x542a] = 0x99c4, [0x542b] = 0x99c5,
  [0x542c] = 0x99c6, [0x542d] = 0x99c7, [0x542e] = 0x99c8, [0x542f] = 0x99c9,
  [0x5430] = 0x99ca, [0x5431] = 0x99cb, [0x5432] = 0x99cc, [0x5433] = 0x99cd,
  [0x5434] = 0x99ce, [0x5435] = 0x99cf, [0x5436] = 0x99d0, [0x5437] = 0x99d1,
  [0x5438] = 0x99d2, [0x5439] = 0x99d3, [0x543a] = 0x99d4, [0x543b] = 0x99d5,
  [0x543c] = 0x99d6, [0x543d] = 0x99d7, [0x543e] = 0x99d8, [0x5440] = 0x99d9,
  [0x5441] = 0x99da, [0x5442] = 0x99db, [0x5443] = 0x99dc, [0x5444] = 0x99dd,
  [0x5445] = 0x99de, [0x5446] = 0x99df, [0x5447] = 0x99e0, [0x5448] = 0x99e1,
  [0x5449] = 0x99e2, [0x544a] = 0x99e3, [0x544b] = 0x99e4, [0x544c] = 0x99e5,
  [0x544d] = 0x99e6, [0x544e] = 0x99e7, [0x544f] = 0x99e8, [0x5450] = 0x99e9,
  [0x5451] = 0x99ea, [0x5452] = 0x99eb, [0x5453] = 0x99ec, [0x5454] = 0x99ed,
  [0x5455] = 0x99ee, [0x5456] = 0x99ef, [0x5457] = 0x99f0, [0x5458] = 0x99f1,
  [0x5459] = 0x99f2, [0x545a] = 0x99f3, [0x545b] = 0x99f4, [0x545c] = 0x99f5,
  [0x545d] = 0x99f6, [0x545e] = 0x99f7, [0x545f] = 0x99f8, [0x5460] = 0x99f9,
  [0x5461] = 0x761b, [0x5462] = 0x763c, [0x5463] = 0x7622, [0x5464] = 0x7620,
  [0x5465] = 0x7640, [0x5466] = 0x762d, [0x5467] = 0x7630, [0x5468] = 0x763f,
  [0x5469] = 0x7635, [0x546a] = 0x7643, [0x546b] = 0x763e, [0x546c] = 0x7633,
  [0x546d] = 0x764d, [0x546e] = 0x765e, [0x546f] = 0x7654, [0x5470] = 0x765c,
  [0x5471] = 0x7656, [0x5472] = 0x766b, [0x5473] = 0x766f, [0x5474] = 0x7fca,
  [0x5475] = 0x7ae6, [0x5476] = 0x7a78, [0x5477] = 0x7a79, [0x5478] = 0x7a80,
  [0x5479] = 0x7a86, [0x547a] = 0x7a88, [0x547b] = 0x7a95, [0x547c] = 0x7aa6,
  [0x547d] = 0x7aa0, [0x547e] = 0x7aac, [0x547f] = 0x7aa8, [0x5480] = 0x7aad,
  [0x5481] = 0x7ab3, [0x5482] = 0x8864, [0x5483] = 0x8869, [0x5484] = 0x8872,
  [0x5485] = 0x887d, [0x5486] = 0x887f, [0x5487] = 0x8882, [0x5488] = 0x88a2,
  [0x5489] = 0x88c6, [0x548a] = 0x88b7, [0x548b] = 0x88bc, [0x548c] = 0x88c9,
  [0x548d] = 0x88e2, [0x548e] = 0x88ce, [0x548f] = 0x88e3, [0x5490] = 0x88e5,
  [0x5491] = 0x88f1, [0x5492] = 0x891a, [0x5493] = 0x88fc, [0x5494] = 0x88e8,
  [0x5495] = 0x88fe, [0x5496] = 0x88f0, [0x5497] = 0x8921, [0x5498] = 0x8919,
  [0x5499] = 0x8913, [0x549a] = 0x891b, [0x549b] = 0x890a, [0x549c] = 0x8934,
  [0x549d] = 0x892b, [0x549e] = 0x8936, [0x549f] = 0x8941, [0x54a0] = 0x8966,
  [0x54a1] = 0x897b, [0x54a2] = 0x758b, [0x54a3] = 0x80e5, [0x54a4] = 0x76b2,
  [0x54a5] = 0x76b4, [0x54a6] = 0x77dc, [0x54a7] = 0x8012, [0x54a8] = 0x8014,
  [0x54a9] = 0x8016, [0x54aa] = 0x801c, [0x54ab] = 0x8020, [0x54ac] = 0x8022,
  [0x54ad] = 0x8025, [0x54ae] = 0x8026, [0x54af] = 0x8027, [0x54b0] = 0x8029,
  [0x54b1] = 0x8028, [0x54b2] = 0x8031, [0x54b3] = 0x800b, [0x54b4] = 0x8035,
  [0x54b5] = 0x8043, [0x54b6] = 0x8046, [0x54b7] = 0x804d, [0x54b8] = 0x8052,
  [0x54b9] = 0x8069, [0x54ba] = 0x8071, [0x54bb] = 0x8983, [0x54bc] = 0x9878,
  [0x54bd] = 0x9880, [0x54be] = 0x9883, [0x54c0] = 0x99fa, [0x54c1] = 0x99fb,
  [0x54c2] = 0x99fc, [0x54c3] = 0x99fd, [0x54c4] = 0x99fe, [0x54c5] = 0x99ff,
  [0x54c6] = 0x9a00, [0x54c7] = 0x9a01, [0x54c8] = 0x9a02, [0x54c9] = 0x9a03,
  [0x54ca] = 0x9a04, [0x54cb] = 0x9a05, [0x54cc] = 0x9a06, [0x54cd] = 0x9a07,
  [0x54ce] = 0x9a08, [0x54cf] = 0x9a09, [0x54d0] = 0x9a0a, [0x54d1] = 0x9a0b,
  [0x54d2] = 0x9a0c, [0x54d3] = 0x9a0d, [0x54d4] = 0x9a0e, [0x54d5] = 0x9a0f,
  [0x54d6] = 0x9a10, [0x54d7] = 0x9a11, [0x54d8] = 0x9a12, [0x54d9] = 0x9a13,
  [0x54da] = 0x9a14, [0x54db] = 0x9a15, [0x54dc] = 0x9a16, [0x54dd] = 0x9a17,
  [0x54de] = 0x9a18, [0x54df] = 0x9a19, [0x54e0] = 0x9a1a, [0x54e1] = 0x9a1b,
  [0x54e2] = 0x9a1c, [0x54e3] = 0x9a1d, [0x54e4] = 0x9a1e, [0x54e5] = 0x9a1f,
  [0x54e6] = 0x9a20, [0x54e7] = 0x9a21, [0x54e8] = 0x9a22, [0x54e9] = 0x9a23,
  [0x54ea] = 0x9a24, [0x54eb] = 0x9a25, [0x54ec] = 0x9a26, [0x54ed] = 0x9a27,
  [0x54ee] = 0x9a28, [0x54ef] = 0x9a29, [0x54f0] = 0x9a2a, [0x54f1] = 0x9a2b,
  [0x54f2] = 0x9a2c, [0x54f3] = 0x9a2d, [0x54f4] = 0x9a2e, [0x54f5] = 0x9a2f,
  [0x54f6] = 0x9a30, [0x54f7] = 0x9a31, [0x54f8] = 0x9a32, [0x54f9] = 0x9a33,
  [0x54fa] = 0x9a34, [0x54fb] = 0x9a35, [0x54fc] = 0x9a36, [0x54fd] = 0x9a37,
  [0x54fe] = 0x9a38, [0x5500] = 0x9a39, [0x5501] = 0x9a3a, [0x5502] = 0x9a3b,
  [0x5503] = 0x9a3c, [0x5504] = 0x9a3d, [0x5505] = 0x9a3e, [0x5506] = 0x9a3f,
  [0x5507] = 0x9a40, [0x5508] = 0x9a41, [0x5509] = 0x9a42, [0x550a] = 0x9a43,
  [0x550b] = 0x9a44, [0x550c] = 0x9a45, [0x550d] = 0x9a46, [0x550e] = 0x9a47,
  [0x550f] = 0x9a48, [0x5510] = 0x9a49, [0x5511] = 0x9a4a, [0x5512] = 0x9a4b,
  [0x5513] = 0x9a4c, [0x5514] = 0x9a4d, [0x5515] = 0x9a4e, [0x5516] = 0x9a4f,
  [0x5517] = 0x9a50, [0x5518] = 0x9a51, [0x5519] = 0x9a52, [0x551a] = 0x9a53,
  [0x551b] = 0x9a54, [0x551c] = 0x9a55, [0x551d] = 0x9a56, [0x551e] = 0x9a57,
  [0x551f] = 0x9a58, [0x5520] = 0x9a59, [0x5521] = 0x9889, [0x5522] = 0x988c,
  [0x5523] = 0x988d, [0x5524] = 0x988f, [0x5525] = 0x9894, [0x5526] = 0x989a,
  [0x5527] = 0x989b, [0x5528] = 0x989e, [0x5529] = 0x989f, [0x552a] = 0x98a1,
  [0x552b] = 0x98a2, [0x552c] = 0x98a5, [0x552d] = 0x98a6, [0x552e] = 0x864d,
  [0x552f] = 0x8654, [0x5530] = 0x866c, [0x5531] = 0x866e, [0x5532] = 0x867f,
  [0x5533] = 0x867a, [0x5534] = 0x867c, [0x5535] = 0x867b, [0x5536] = 0x86a8,
  [0x5537] = 0x868d, [0x5538] = 0x868b, [0x5539] = 0x86ac, [0x553a] = 0x869d,
  [0x553b] = 0x86a7, [0x553c] = 0x86a3, [0x553d] = 0x86aa, [0x553e] = 0x8693,
  [0x553f] = 0x86a9, [0x5540] = 0x86b6, [0x5541] = 0x86c4, [0x5542] = 0x86b5,
  [0x5543] = 0x86ce, [0x5544] = 0x86b0, [0x5545] = 0x86ba, [0x5546] = 0x86b1,
  [0x5547] = 0x86af, [0x5548] = 0x86c9, [0x5549] = 0x86cf, [0x554a] = 0x86b4,
  [0x554b] = 0x86e9, [0x554c] = 0x86f1, [0x554d] = 0x86f2, [0x554e] = 0x86ed,
  [0x554f] = 0x86f3, [0x5550] = 0x86d0, [0x5551] = 0x8713, [0x5552] = 0x86de,
  [0x5553] = 0x86f4, [0x5554] = 0x86df, [0x5555] = 0x86d8, [0x5556] = 0x86d1,
  [0x5557] = 0x8703, [0x5558] = 0x8707, [0x5559] = 0x86f8, [0x555a] = 0x8708,
  [0x555b] = 0x870a, [0x555c] = 0x870d, [0x555d] = 0x8709, [0x555e] = 0x8723,
  [0x555f] = 0x873b, [0x5560] = 0x871e, [0x5561] = 0x8725, [0x5562] = 0x872e,
  [0x5563] = 0x871a, [0x5564] = 0x873e, [0x5565] = 0x8748, [0x5566] = 0x8734,
  [0x5567] = 0x8731, [0x5568] = 0x8729, [0x5569] = 0x8737, [0x556a] = 0x873f,
  [0x556b] = 0x8782, [0x556c] = 0x8722, [0x556d] = 0x877d, [0x556e] = 0x877e,
  [0x556f] = 0x877b, [0x5570] = 0x8760, [0x5571] = 0x8770, [0x5572] = 0x874c,
  [0x5573] = 0x876e, [0x5574] = 0x878b, [0x5575] = 0x8753, [0x5576] = 0x8763,
  [0x5577] = 0x877c, [0x5578] = 0x8764, [0x5579] = 0x8759, [0x557a] = 0x8765,
  [0x557b] = 0x8793, [0x557c] = 0x87af, [0x557d] = 0x87a8, [0x557e] = 0x87d2,
  [0x5580] = 0x9a5a, [0x5581] = 0x9a5b, [0x5582] = 0x9a5c, [0x5583] = 0x9a5d,
  [0x5584] = 0x9a5e, [0x5585] = 0x9a5f, [0x5586] = 0x9a60, [0x5587] = 0x9a61,
  [0x5588] = 0x9a62, [0x5589] = 0x9a63, [0x558a] = 0x9a64, [0x558b] = 0x9a65,
  [0x558c] = 0x9a66, [0x558d] = 0x9a67, [0x558e] = 0x9a68, [0x558f] = 0x9a69,
  [0x5590] = 0x9a6a, [0x5591] = 0x9a6b, [0x5592] = 0x9a72, [0x5593] = 0x9a83,
  [0x5594] = 0x9a89, [0x5595] = 0x9a8d, [0x5596] = 0x9a8e, [0x5597] = 0x9a94,
  [0x5598] = 0x9a95, [0x5599] = 0x9a99, [0x559a] = 0x9aa6, [0x559b] = 0x9aa9,
  [0x559c] = 0x9aaa, [0x559d] = 0x9aab, [0x559e] = 0x9aac, [0x559f] = 0x9aad,
  [0x55a0] = 0x9aae, [0x55a1] = 0x9aaf, [0x55a2] = 0x9ab2, [0x55a3] = 0x9ab3,
  [0x55a4] = 0x9ab4, [0x55a5] = 0x9ab5, [0x55a6] = 0x9ab9, [0x55a7] = 0x9abb,
  [0x55a8] = 0x9abd, [0x55a9] = 0x9abe, [0x55aa] = 0x9abf, [0x55ab] = 0x9ac3,
  [0x55ac] = 0x9ac4, [0x55ad] = 0x9ac6, [0x55ae] = 0x9ac7, [0x55af] = 0x9ac8,
  [0x55b0] = 0x9ac9, [0x55b1] = 0x9aca, [0x55b2] = 0x9acd, [0x55b3] = 0x9ace,
  [0x55b4] = 0x9acf, [0x55b5] = 0x9ad0, [0x55b6] = 0x9ad2, [0x55b7] = 0x9ad4,
  [0x55b8] = 0x9ad5, [0x55b9] = 0x9ad6, [0x55ba] = 0x9ad7, [0x55bb] = 0x9ad9,
  [0x55bc] = 0x9ada, [0x55bd] = 0x9adb, [0x55be] = 0x9adc, [0x55c0] = 0x9add,
  [0x55c1] = 0x9ade, [0x55c2] = 0x9ae0, [0x55c3] = 0x9ae2, [0x55c4] = 0x9ae3,
  [0x55c5] = 0x9ae4, [0x55c6] = 0x9ae5, [0x55c7] = 0x9ae7, [0x55c8] = 0x9ae8,
  [0x55c9] = 0x9ae9, [0x55ca] = 0x9aea, [0x55cb] = 0x9aec, [0x55cc] = 0x9aee,
  [0x55cd] = 0x9af0, [0x55ce] = 0x9af1, [0x55cf] = 0x9af2, [0x55d0] = 0x9af3,
  [0x55d1] = 0x9af4, [0x55d2] = 0x9af5, [0x55d3] = 0x9af6, [0x55d4] = 0x9af7,
  [0x55d5] = 0x9af8, [0x55d6] = 0x9afa, [0x55d7] = 0x9afc, [0x55d8] = 0x9afd,
  [0x55d9] = 0x9afe, [0x55da] = 0x9aff, [0x55db] = 0x9b00, [0x55dc] = 0x9b01,
  [0x55dd] = 0x9b02, [0x55de] = 0x9b04, [0x55df] = 0x9b05, [0x55e0] = 0x9b06,
  [0x55e1] = 0x87c6, [0x55e2] = 0x8788, [0x55e3] = 0x8785, [0x55e4] = 0x87ad,
  [0x55e5] = 0x8797, [0x55e6] = 0x8783, [0x55e7] = 0x87ab, [0x55e8] = 0x87e5,
  [0x55e9] = 0x87ac, [0x55ea] = 0x87b5, [0x55eb] = 0x87b3, [0x55ec] = 0x87cb,
  [0x55ed] = 0x87d3, [0x55ee] = 0x87bd, [0x55ef] = 0x87d1, [0x55f0] = 0x87c0,
  [0x55f1] = 0x87ca, [0x55f2] = 0x87db, [0x55f3] = 0x87ea, [0x55f4] = 0x87e0,
  [0x55f5] = 0x87ee, [0x55f6] = 0x8816, [0x55f7] = 0x8813, [0x55f8] = 0x87fe,
  [0x55f9] = 0x880a, [0x55fa] = 0x881b, [0x55fb] = 0x8821, [0x55fc] = 0x8839,
  [0x55fd] = 0x883c, [0x55fe] = 0x7f36, [0x55ff] = 0x7f42, [0x5600] = 0x7f44,
  [0x5601] = 0x7f45, [0x5602] = 0x8210, [0x5603] = 0x7afa, [0x5604] = 0x7afd,
  [0x5605] = 0x7b08, [0x5606] = 0x7b03, [0x5607] = 0x7b04, [0x5608] = 0x7b15,
  [0x5609] = 0x7b0a, [0x560a] = 0x7b2b, [0x560b] = 0x7b0f, [0x560c] = 0x7b47,
  [0x560d] = 0x7b38, [0x560e] = 0x7b2a, [0x560f] = 0x7b19, [0x5610] = 0x7b2e,
  [0x5611] = 0x7b31, [0x5612] = 0x7b20, [0x5613] = 0x7b25, [0x5614] = 0x7b24,
  [0x5615] = 0x7b33, [0x5616] = 0x7b3e, [0x5617] = 0x7b1e, [0x5618] = 0x7b58,
  [0x5619] = 0x7b5a, [0x561a] = 0x7b45, [0x561b] = 0x7b75, [0x561c] = 0x7b4c,
  [0x561d] = 0x7b5d, [0x561e] = 0x7b60, [0x561f] = 0x7b6e, [0x5620] = 0x7b7b,
  [0x5621] = 0x7b62, [0x5622] = 0x7b72, [0x5623] = 0x7b71, [0x5624] = 0x7b90,
  [0x5625] = 0x7ba6, [0x5626] = 0x7ba7, [0x5627] = 0x7bb8, [0x5628] = 0x7bac,
  [0x5629] = 0x7b9d, [0x562a] = 0x7ba8, [0x562b] = 0x7b85, [0x562c] = 0x7baa,
  [0x562d] = 0x7b9c, [0x562e] = 0x7ba2, [0x562f] = 0x7bab, [0x5630] = 0x7bb4,
  [0x5631] = 0x7bd1, [0x5632] = 0x7bc1, [0x5633] = 0x7bcc, [0x5634] = 0x7bdd,
  [0x5635] = 0x7bda, [0x5636] = 0x7be5, [0x5637] = 0x7be6, [0x5638] = 0x7bea,
  [0x5639] = 0x7c0c, [0x563a] = 0x7bfe, [0x563b] = 0x7bfc, [0x563c] = 0x7c0f,
  [0x563d] = 0x7c16, [0x563e] = 0x7c0b, [0x5640] = 0x9b07, [0x5641] = 0x9b09,
  [0x5642] = 0x9b0a, [0x5643] = 0x9b0b, [0x5644] = 0x9b0c, [0x5645] = 0x9b0d,
  [0x5646] = 0x9b0e, [0x5647] = 0x9b10, [0x5648] = 0x9b11, [0x5649] = 0x9b12,
  [0x564a] = 0x9b14, [0x564b] = 0x9b15, [0x564c] = 0x9b16, [0x564d] = 0x9b17,
  [0x564e] = 0x9b18, [0x564f] = 0x9b19, [0x5650] = 0x9b1a, [0x5651] = 0x9b1b,
  [0x5652] = 0x9b1c, [0x5653] = 0x9b1d, [0x5654] = 0x9b1e, [0x5655] = 0x9b20,
  [0x5656] = 0x9b21, [0x5657] = 0x9b22, [0x5658] = 0x9b24, [0x5659] = 0x9b25,
  [0x565a] = 0x9b26, [0x565b] = 0x9b27, [0x565c] = 0x9b28, [0x565d] = 0x9b29,
  [0x565e] = 0x9b2a, [0x565f] = 0x9b2b, [0x5660] = 0x9b2c, [0x5661] = 0x9b2d,
  [0x5662] = 0x9b2e, [0x5663] = 0x9b30, [0x5664] = 0x9b31, [0x5665] = 0x9b33,
  [0x5666] = 0x9b34, [0x5667] = 0x9b35, [0x5668] = 0x9b36, [0x5669] = 0x9b37,
  [0x566a] = 0x9b38, [0x566b] = 0x9b39, [0x566c] = 0x9b3a, [0x566d] = 0x9b3d,
  [0x566e] = 0x9b3e, [0x566f] = 0x9b3f, [0x5670] = 0x9b40, [0x5671] = 0x9b46,
  [0x5672] = 0x9b4a, [0x5673] = 0x9b4b, [0x5674] = 0x9b4c, [0x5675] = 0x9b4e,
  [0x5676] = 0x9b50, [0x5677] = 0x9b52, [0x5678] = 0x9b53, [0x5679] = 0x9b55,
  [0x567a] = 0x9b56, [0x567b] = 0x9b57, [0x567c] = 0x9b58, [0x567d] = 0x9b59,
  [0x567e] = 0x9b5a, [0x5680] = 0x9b5b, [0x5681] = 0x9b5c, [0x5682] = 0x9b5d,
  [0x5683] = 0x9b5e, [0x5684] = 0x9b5f, [0x5685] = 0x9b60, [0x5686] = 0x9b61,
  [0x5687] = 0x9b62, [0x5688] = 0x9b63, [0x5689] = 0x9b64, [0x568a] = 0x9b65,
  [0x568b] = 0x9b66, [0x568c] = 0x9b67, [0x568d] = 0x9b68, [0x568e] = 0x9b69,
  [0x568f] = 0x9b6a, [0x5690] = 0x9b6b, [0x5691] = 0x9b6c, [0x5692] = 0x9b6d,
  [0x5693] = 0x9b6e, [0x5694] = 0x9b6f, [0x5695] = 0x9b70, [0x5696] = 0x9b71,
  [0x5697] = 0x9b72, [0x5698] = 0x9b73, [0x5699] = 0x9b74, [0x569a] = 0x9b75,
  [0x569b] = 0x9b76, [0x569c] = 0x9b77, [0x569d] = 0x9b78, [0x569e] = 0x9b79,
  [0x569f] = 0x9b7a, [0x56a0] = 0x9b7b, [0x56a1] = 0x7c1f, [0x56a2] = 0x7c2a,
  [0x56a3] = 0x7c26, [0x56a4] = 0x7c38, [0x56a5] = 0x7c41, [0x56a6] = 0x7c40,
  [0x56a7] = 0x81fe, [0x56a8] = 0x8201, [0x56a9] = 0x8202, [0x56aa] = 0x8204,
  [0x56ab] = 0x81ec, [0x56ac] = 0x8844, [0x56ad] = 0x8221, [0x56ae] = 0x8222,
  [0x56af] = 0x8223, [0x56b0] = 0x822d, [0x56b1] = 0x822f, [0x56b2] = 0x8228,
  [0x56b3] = 0x822b, [0x56b4] = 0x8238, [0x56b5] = 0x823b, [0x56b6] = 0x8233,
  [0x56b7] = 0x8234, [0x56b8] = 0x823e, [0x56b9] = 0x8244, [0x56ba] = 0x8249,
  [0x56bb] = 0x824b, [0x56bc] = 0x824f, [0x56bd] = 0x825a, [0x56be] = 0x825f,
  [0x56bf] = 0x8268, [0x56c0] = 0x887e, [0x56c1] = 0x8885, [0x56c2] = 0x8888,
  [0x56c3] = 0x88d8, [0x56c4] = 0x88df, [0x56c5] = 0x895e, [0x56c6] = 0x7f9d,
  [0x56c7] = 0x7f9f, [0x56c8] = 0x7fa7, [0x56c9] = 0x7faf, [0x56ca] = 0x7fb0,
  [0x56cb] = 0x7fb2, [0x56cc] = 0x7c7c, [0x56cd] = 0x6549, [0x56ce] = 0x7c91,
  [0x56cf] = 0x7c9d, [0x56d0] = 0x7c9c, [0x56d1] = 0x7c9e, [0x56d2] = 0x7ca2,
  [0x56d3] = 0x7cb2, [0x56d4] = 0x7cbc, [0x56d5] = 0x7cbd, [0x56d6] = 0x7cc1,
  [0x56d7] = 0x7cc7, [0x56d8] = 0x7ccc, [0x56d9] = 0x7ccd, [0x56da] = 0x7cc8,
  [0x56db] = 0x7cc5, [0x56dc] = 0x7cd7, [0x56dd] = 0x7ce8, [0x56de] = 0x826e,
  [0x56df] = 0x66a8, [0x56e0] = 0x7fbf, [0x56e1] = 0x7fce, [0x56e2] = 0x7fd5,
  [0x56e3] = 0x7fe5, [0x56e4] = 0x7fe1, [0x56e5] = 0x7fe6, [0x56e6] = 0x7fe9,
  [0x56e7] = 0x7fee, [0x56e8] = 0x7ff3, [0x56e9] = 0x7cf8, [0x56ea] = 0x7d77,
  [0x56eb] = 0x7da6, [0x56ec] = 0x7dae, [0x56ed] = 0x7e47, [0x56ee] = 0x7e9b,
  [0x56ef] = 0x9eb8, [0x56f0] = 0x9eb4, [0x56f1] = 0x8d73, [0x56f2] = 0x8d84,
  [0x56f3] = 0x8d94, [0x56f4] = 0x8d91, [0x56f5] = 0x8db1, [0x56f6] = 0x8d67,
  [0x56f7] = 0x8d6d, [0x56f8] = 0x8c47, [0x56f9] = 0x8c49, [0x56fa] = 0x914a,
  [0x56fb] = 0x9150, [0x56fc] = 0x914e, [0x56fd] = 0x914f, [0x56fe] = 0x9164,
  [0x5700] = 0x9b7c, [0x5701] = 0x9b7d, [0x5702] = 0x9b7e, [0x5703] = 0x9b7f,
  [0x5704] = 0x9b80, [0x5705] = 0x9b81, [0x5706] = 0x9b82, [0x5707] = 0x9b83,
  [0x5708] = 0x9b84, [0x5709] = 0x9b85, [0x570a] = 0x9b86, [0x570b] = 0x9b87,
  [0x570c] = 0x9b88, [0x570d] = 0x9b89, [0x570e] = 0x9b8a, [0x570f] = 0x9b8b,
  [0x5710] = 0x9b8c, [0x5711] = 0x9b8d, [0x5712] = 0x9b8e, [0x5713] = 0x9b8f,
  [0x5714] = 0x9b90, [0x5715] = 0x9b91, [0x5716] = 0x9b92, [0x5717] = 0x9b93,
  [0x5718] = 0x9b94, [0x5719] = 0x9b95, [0x571a] = 0x9b96, [0x571b] = 0x9b97,
  [0x571c] = 0x9b98, [0x571d] = 0x9b99, [0x571e] = 0x9b9a, [0x571f] = 0x9b9b,
  [0x5720] = 0x9b9c, [0x5721] = 0x9b9d, [0x5722] = 0x9b9e, [0x5723] = 0x9b9f,
  [0x5724] = 0x9ba0, [0x5725] = 0x9ba1, [0x5726] = 0x9ba2, [0x5727] = 0x9ba3,
  [0x5728] = 0x9ba4, [0x5729] = 0x9ba5, [0x572a] = 0x9ba6, [0x572b] = 0x9ba7,
  [0x572c] = 0x9ba8, [0x572d] = 0x9ba9, [0x572e] = 0x9baa, [0x572f] = 0x9bab,
  [0x5730] = 0x9bac, [0x5731] = 0x9bad, [0x5732] = 0x9bae, [0x5733] = 0x9baf,
  [0x5734] = 0x9bb0, [0x5735] = 0x9bb1, [0x5736] = 0x9bb2, [0x5737] = 0x9bb3,
  [0x5738] = 0x9bb4, [0x5739] = 0x9bb5, [0x573a] = 0x9bb6, [0x573b] = 0x9bb7,
  [0x573c] = 0x9bb8, [0x573d] = 0x9bb9, [0x573e] = 0x9bba, [0x5740] = 0x9bbb,
  [0x5741] = 0x9bbc, [0x5742] = 0x9bbd, [0x5743] = 0x9bbe, [0x5744] = 0x9bbf,
  [0x5745] = 0x9bc0, [0x5746] = 0x9bc1, [0x5747] = 0x9bc2, [0x5748] = 0x9bc3,
  [0x5749] = 0x9bc4, [0x574a] = 0x9bc5, [0x574b] = 0x9bc6, [0x574c] = 0x9bc7,
  [0x574d] = 0x9bc8, [0x574e] = 0x9bc9, [0x574f] = 0x9bca, [0x5750] = 0x9bcb,
  [0x5751] = 0x9bcc, [0x5752] = 0x9bcd, [0x5753] = 0x9bce, [0x5754] = 0x9bcf,
  [0x5755] = 0x9bd0, [0x5756] = 0x9bd1, [0x5757] = 0x9bd2, [0x5758] = 0x9bd3,
  [0x5759] = 0x9bd4, [0x575a] = 0x9bd5, [0x575b] = 0x9bd6, [0x575c] = 0x9bd7,
  [0x575d] = 0x9bd8, [0x575e] = 0x9bd9, [0x575f] = 0x9bda, [0x5760] = 0x9bdb,
  [0x5761] = 0x9162, [0x5762] = 0x9161, [0x5763] = 0x9170, [0x5764] = 0x9169,
  [0x5765] = 0x916f, [0x5766] = 0x917d, [0x5767] = 0x917e, [0x5768] = 0x9172,
  [0x5769] = 0x9174, [0x576a] = 0x9179, [0x576b] = 0x918c, [0x576c] = 0x9185,
  [0x576d] = 0x9190, [0x576e] = 0x918d, [0x576f] = 0x9191, [0x5770] = 0x91a2,
  [0x5771] = 0x91a3, [0x5772] = 0x91aa, [0x5773] = 0x91ad, [0x5774] = 0x91ae,
  [0x5775] = 0x91af, [0x5776] = 0x91b5, [0x5777] = 0x91b4, [0x5778] = 0x91ba,
  [0x5779] = 0x8c55, [0x577a] = 0x9e7e, [0x577b] = 0x8db8, [0x577c] = 0x8deb,
  [0x577d] = 0x8e05, [0x577e] = 0x8e59, [0x577f] = 0x8e69, [0x5780] = 0x8db5,
  [0x5781] = 0x8dbf, [0x5782] = 0x8dbc, [0x5783] = 0x8dba, [0x5784] = 0x8dc4,
  [0x5785] = 0x8dd6, [0x5786] = 0x8dd7, [0x5787] = 0x8dda, [0x5788] = 0x8dde,
  [0x5789] = 0x8dce, [0x578a] = 0x8dcf, [0x578b] = 0x8ddb, [0x578c] = 0x8dc6,
  [0x578d] = 0x8dec, [0x578e] = 0x8df7, [0x578f] = 0x8df8, [0x5790] = 0x8de3,
  [0x5791] = 0x8df9, [0x5792] = 0x8dfb, [0x5793] = 0x8de4, [0x5794] = 0x8e09,
  [0x5795] = 0x8dfd, [0x5796] = 0x8e14, [0x5797] = 0x8e1d, [0x5798] = 0x8e1f,
  [0x5799] = 0x8e2c, [0x579a] = 0x8e2e, [0x579b] = 0x8e23, [0x579c] = 0x8e2f,
  [0x579d] = 0x8e3a, [0x579e] = 0x8e40, [0x579f] = 0x8e39, [0x57a0] = 0x8e35,
  [0x57a1] = 0x8e3d, [0x57a2] = 0x8e31, [0x57a3] = 0x8e49, [0x57a4] = 0x8e41,
  [0x57a5] = 0x8e42, [0x57a6] = 0x8e51, [0x57a7] = 0x8e52, [0x57a8] = 0x8e4a,
  [0x57a9] = 0x8e70, [0x57aa] = 0x8e76, [0x57ab] = 0x8e7c, [0x57ac] = 0x8e6f,
  [0x57ad] = 0x8e74, [0x57ae] = 0x8e85, [0x57af] = 0x8e8f, [0x57b0] = 0x8e94,
  [0x57b1] = 0x8e90, [0x57b2] = 0x8e9c, [0x57b3] = 0x8e9e, [0x57b4] = 0x8c78,
  [0x57b5] = 0x8c82, [0x57b6] = 0x8c8a, [0x57b7] = 0x8c85, [0x57b8] = 0x8c98,
  [0x57b9] = 0x8c94, [0x57ba] = 0x659b, [0x57bb] = 0x89d6, [0x57bc] = 0x89de,
  [0x57bd] = 0x89da, [0x57be] = 0x89dc, [0x57c0] = 0x9bdc, [0x57c1] = 0x9bdd,
  [0x57c2] = 0x9bde, [0x57c3] = 0x9bdf, [0x57c4] = 0x9be0, [0x57c5] = 0x9be1,
  [0x57c6] = 0x9be2, [0x57c7] = 0x9be3, [0x57c8] = 0x9be4, [0x57c9] = 0x9be5,
  [0x57ca] = 0x9be6, [0x57cb] = 0x9be7, [0x57cc] = 0x9be8, [0x57cd] = 0x9be9,
  [0x57ce] = 0x9bea, [0x57cf] = 0x9beb, [0x57d0] = 0x9bec, [0x57d1] = 0x9bed,
  [0x57d2] = 0x9bee, [0x57d3] = 0x9bef, [0x57d4] = 0x9bf0, [0x57d5] = 0x9bf1,
  [0x57d6] = 0x9bf2, [0x57d7] = 0x9bf3, [0x57d8] = 0x9bf4, [0x57d9] = 0x9bf5,
  [0x57da] = 0x9bf6, [0x57db] = 0x9bf7, [0x57dc] = 0x9bf8, [0x57dd] = 0x9bf9,
  [0x57de] = 0x9bfa, [0x57df] = 0x9bfb, [0x57e0] = 0x9bfc, [0x57e1] = 0x9bfd,
  [0x57e2] = 0x9bfe, [0x57e3] = 0x9bff, [0x57e4] = 0x9c00, [0x57e5] = 0x9c01,
  [0x57e6] = 0x9c02, [0x57e7] = 0x9c03, [0x57e8] = 0x9c04, [0x57e9] = 0x9c05,
  [0x57ea] = 0x9c06, [0x57eb] = 0x9c07, [0x57ec] = 0x9c08, [0x57ed] = 0x9c09,
  [0x57ee] = 0x9c0a, [0x57ef] = 0x9c0b, [0x57f0] = 0x9c0c, [0x57f1] = 0x9c0d,
  [0x57f2] = 0x9c0e, [0x57f3] = 0x9c0f, [0x57f4] = 0x9c10, [0x57f5] = 0x9c11,
  [0x57f6] = 0x9c12, [0x57f7] = 0x9c13, [0x57f8] = 0x9c14, [0x57f9] = 0x9c15,
  [0x57fa] = 0x9c16, [0x57fb] = 0x9c17, [0x57fc] = 0x9c18, [0x57fd] = 0x9c19,
  [0x57fe] = 0x9c1a, [0x5800] = 0x9c1b, [0x5801] = 0x9c1c, [0x5802] = 0x9c1d,
  [0x5803] = 0x9c1e, [0x5804] = 0x9c1f, [0x5805] = 0x9c20, [0x5806] = 0x9c21,
  [0x5807] = 0x9c22, [0x5808] = 0x9c23, [0x5809] = 0x9c24, [0x580a] = 0x9c25,
  [0x580b] = 0x9c26, [0x580c] = 0x9c27, [0x580d] = 0x9c28, [0x580e] = 0x9c29,
  [0x580f] = 0x9c2a, [0x5810] = 0x9c2b, [0x5811] = 0x9c2c, [0x5812] = 0x9c2d,
  [0x5813] = 0x9c2e, [0x5814] = 0x9c2f, [0x5815] = 0x9c30, [0x5816] = 0x9c31,
  [0x5817] = 0x9c32, [0x5818] = 0x9c33, [0x5819] = 0x9c34, [0x581a] = 0x9c35,
  [0x581b] = 0x9c36, [0x581c] = 0x9c37, [0x581d] = 0x9c38, [0x581e] = 0x9c39,
  [0x581f] = 0x9c3a, [0x5820] = 0x9c3b, [0x5821] = 0x89e5, [0x5822] = 0x89eb,
  [0x5823] = 0x89ef, [0x5824] = 0x8a3e, [0x5825] = 0x8b26, [0x5826] = 0x9753,
  [0x5827] = 0x96e9, [0x5828] = 0x96f3, [0x5829] = 0x96ef, [0x582a] = 0x9706,
  [0x582b] = 0x9701, [0x582c] = 0x9708, [0x582d] = 0x970f, [0x582e] = 0x970e,
  [0x582f] = 0x972a, [0x5830] = 0x972d, [0x5831] = 0x9730, [0x5832] = 0x973e,
  [0x5833] = 0x9f80, [0x5834] = 0x9f83, [0x5835] = 0x9f85, [0x5836] = 0x9f86,
  [0x5837] = 0x9f87, [0x5838] = 0x9f88, [0x5839] = 0x9f89, [0x583a] = 0x9f8a,
  [0x583b] = 0x9f8c, [0x583c] = 0x9efe, [0x583d] = 0x9f0b, [0x583e] = 0x9f0d,
  [0x583f] = 0x96b9, [0x5840] = 0x96bc, [0x5841] = 0x96bd, [0x5842] = 0x96ce,
  [0x5843] = 0x96d2, [0x5844] = 0x77bf, [0x5845] = 0x96e0, [0x5846] = 0x928e,
  [0x5847] = 0x92ae, [0x5848] = 0x92c8, [0x5849] = 0x933e, [0x584a] = 0x936a,
  [0x584b] = 0x93ca, [0x584c] = 0x938f, [0x584d] = 0x943e, [0x584e] = 0x946b,
  [0x584f] = 0x9c7f, [0x5850] = 0x9c82, [0x5851] = 0x9c85, [0x5852] = 0x9c86,
  [0x5853] = 0x9c87, [0x5854] = 0x9c88, [0x5855] = 0x7a23, [0x5856] = 0x9c8b,
  [0x5857] = 0x9c8e, [0x5858] = 0x9c90, [0x5859] = 0x9c91, [0x585a] = 0x9c92,
  [0x585b] = 0x9c94, [0x585c] = 0x9c95, [0x585d] = 0x9c9a, [0x585e] = 0x9c9b,
  [0x585f] = 0x9c9e, [0x5860] = 0x9c9f, [0x5861] = 0x9ca0, [0x5862] = 0x9ca1,
  [0x5863] = 0x9ca2, [0x5864] = 0x9ca3, [0x5865] = 0x9ca5, [0x5866] = 0x9ca6,
  [0x5867] = 0x9ca7, [0x5868] = 0x9ca8, [0x5869] = 0x9ca9, [0x586a] = 0x9cab,
  [0x586b] = 0x9cad, [0x586c] = 0x9cae, [0x586d] = 0x9cb0, [0x586e] = 0x9cb1,
  [0x586f] = 0x9cb2, [0x5870] = 0x9cb3, [0x5871] = 0x9cb4, [0x5872] = 0x9cb5,
  [0x5873] = 0x9cb6, [0x5874] = 0x9cb7, [0x5875] = 0x9cba, [0x5876] = 0x9cbb,
  [0x5877] = 0x9cbc, [0x5878] = 0x9cbd, [0x5879] = 0x9cc4, [0x587a] = 0x9cc5,
  [0x587b] = 0x9cc6, [0x587c] = 0x9cc7, [0x587d] = 0x9cca, [0x587e] = 0x9ccb,
  [0x5880] = 0x9c3c, [0x5881] = 0x9c3d, [0x5882] = 0x9c3e, [0x5883] = 0x9c3f,
  [0x5884] = 0x9c40, [0x5885] = 0x9c41, [0x5886] = 0x9c42, [0x5887] = 0x9c43,
  [0x5888] = 0x9c44, [0x5889] = 0x9c45, [0x588a] = 0x9c46, [0x588b] = 0x9c47,
  [0x588c] = 0x9c48, [0x588d] = 0x9c49, [0x588e] = 0x9c4a, [0x588f] = 0x9c4b,
  [0x5890] = 0x9c4c, [0x5891] = 0x9c4d, [0x5892] = 0x9c4e, [0x5893] = 0x9c4f,
  [0x5894] = 0x9c50, [0x5895] = 0x9c51, [0x5896] = 0x9c52, [0x5897] = 0x9c53,
  [0x5898] = 0x9c54, [0x5899] = 0x9c55, [0x589a] = 0x9c56, [0x589b] = 0x9c57,
  [0x589c] = 0x9c58, [0x589d] = 0x9c59, [0x589e] = 0x9c5a, [0x589f] = 0x9c5b,
  [0x58a0] = 0x9c5c, [0x58a1] = 0x9c5d, [0x58a2] = 0x9c5e, [0x58a3] = 0x9c5f,
  [0x58a4] = 0x9c60, [0x58a5] = 0x9c61, [0x58a6] = 0x9c62, [0x58a7] = 0x9c63,
  [0x58a8] = 0x9c64, [0x58a9] = 0x9c65, [0x58aa] = 0x9c66, [0x58ab] = 0x9c67,
  [0x58ac] = 0x9c68, [0x58ad] = 0x9c69, [0x58ae] = 0x9c6a, [0x58af] = 0x9c6b,
  [0x58b0] = 0x9c6c, [0x58b1] = 0x9c6d, [0x58b2] = 0x9c6e, [0x58b3] = 0x9c6f,
  [0x58b4] = 0x9c70, [0x58b5] = 0x9c71, [0x58b6] = 0x9c72, [0x58b7] = 0x9c73,
  [0x58b8] = 0x9c74, [0x58b9] = 0x9c75, [0x58ba] = 0x9c76, [0x58bb] = 0x9c77,
  [0x58bc] = 0x9c78, [0x58bd] = 0x9c79, [0x58be] = 0x9c7a, [0x58c0] = 0x9c7b,
  [0x58c1] = 0x9c7d, [0x58c2] = 0x9c7e, [0x58c3] = 0x9c80, [0x58c4] = 0x9c83,
  [0x58c5] = 0x9c84, [0x58c6] = 0x9c89, [0x58c7] = 0x9c8a, [0x58c8] = 0x9c8c,
  [0x58c9] = 0x9c8f, [0x58ca] = 0x9c93, [0x58cb] = 0x9c96, [0x58cc] = 0x9c97,
  [0x58cd] = 0x9c98, [0x58ce] = 0x9c99, [0x58cf] = 0x9c9d, [0x58d0] = 0x9caa,
  [0x58d1] = 0x9cac, [0x58d2] = 0x9caf, [0x58d3] = 0x9cb9, [0x58d4] = 0x9cbe,
  [0x58d5] = 0x9cbf, [0x58d6] = 0x9cc0, [0x58d7] = 0x9cc1, [0x58d8] = 0x9cc2,
  [0x58d9] = 0x9cc8, [0x58da] = 0x9cc9, [0x58db] = 0x9cd1, [0x58dc] = 0x9cd2,
  [0x58dd] = 0x9cda, [0x58de] = 0x9cdb, [0x58df] = 0x9ce0, [0x58e0] = 0x9ce1,
  [0x58e1] = 0x9ccc, [0x58e2] = 0x9ccd, [0x58e3] = 0x9cce, [0x58e4] = 0x9ccf,
  [0x58e5] = 0x9cd0, [0x58e6] = 0x9cd3, [0x58e7] = 0x9cd4, [0x58e8] = 0x9cd5,
  [0x58e9] = 0x9cd7, [0x58ea] = 0x9cd8, [0x58eb] = 0x9cd9, [0x58ec] = 0x9cdc,
  [0x58ed] = 0x9cdd, [0x58ee] = 0x9cdf, [0x58ef] = 0x9ce2, [0x58f0] = 0x977c,
  [0x58f1] = 0x9785, [0x58f2] = 0x9791, [0x58f3] = 0x9792, [0x58f4] = 0x9794,
  [0x58f5] = 0x97af, [0x58f6] = 0x97ab, [0x58f7] = 0x97a3, [0x58f8] = 0x97b2,
  [0x58f9] = 0x97b4, [0x58fa] = 0x9ab1, [0x58fb] = 0x9ab0, [0x58fc] = 0x9ab7,
  [0x58fd] = 0x9e58, [0x58fe] = 0x9ab6, [0x58ff] = 0x9aba, [0x5900] = 0x9abc,
  [0x5901] = 0x9ac1, [0x5902] = 0x9ac0, [0x5903] = 0x9ac5, [0x5904] = 0x9ac2,
  [0x5905] = 0x9acb, [0x5906] = 0x9acc, [0x5907] = 0x9ad1, [0x5908] = 0x9b45,
  [0x5909] = 0x9b43, [0x590a] = 0x9b47, [0x590b] = 0x9b49, [0x590c] = 0x9b48,
  [0x590d] = 0x9b4d, [0x590e] = 0x9b51, [0x590f] = 0x98e8, [0x5910] = 0x990d,
  [0x5911] = 0x992e, [0x5912] = 0x9955, [0x5913] = 0x9954, [0x5914] = 0x9adf,
  [0x5915] = 0x9ae1, [0x5916] = 0x9ae6, [0x5917] = 0x9aef, [0x5918] = 0x9aeb,
  [0x5919] = 0x9afb, [0x591a] = 0x9aed, [0x591b] = 0x9af9, [0x591c] = 0x9b08,
  [0x591d] = 0x9b0f, [0x591e] = 0x9b13, [0x591f] = 0x9b1f, [0x5920] = 0x9b23,
  [0x5921] = 0x9ebd, [0x5922] = 0x9ebe, [0x5923] = 0x7e3b, [0x5924] = 0x9e82,
  [0x5925] = 0x9e87, [0x5926] = 0x9e88, [0x5927] = 0x9e8b, [0x5928] = 0x9e92,
  [0x5929] = 0x93d6, [0x592a] = 0x9e9d, [0x592b] = 0x9e9f, [0x592c] = 0x9edb,
  [0x592d] = 0x9edc, [0x592e] = 0x9edd, [0x592f] = 0x9ee0, [0x5930] = 0x9edf,
  [0x5931] = 0x9ee2, [0x5932] = 0x9ee9, [0x5933] = 0x9ee7, [0x5934] = 0x9ee5,
  [0x5935] = 0x9eea, [0x5936] = 0x9eef, [0x5937] = 0x9f22, [0x5938] = 0x9f2c,
  [0x5939] = 0x9f2f, [0x593a] = 0x9f39, [0x593b] = 0x9f37, [0x593c] = 0x9f3d,
  [0x593d] = 0x9f3e, [0x593e] = 0x9f44, [0x5940] = 0x9ce3, [0x5941] = 0x9ce4,
  [0x5942] = 0x9ce5, [0x5943] = 0x9ce6, [0x5944] = 0x9ce7, [0x5945] = 0x9ce8,
  [0x5946] = 0x9ce9, [0x5947] = 0x9cea, [0x5948] = 0x9ceb, [0x5949] = 0x9cec,
  [0x594a] = 0x9ced, [0x594b] = 0x9cee, [0x594c] = 0x9cef, [0x594d] = 0x9cf0,
  [0x594e] = 0x9cf1, [0x594f] = 0x9cf2, [0x5950] = 0x9cf3, [0x5951] = 0x9cf4,
  [0x5952] = 0x9cf5, [0x5953] = 0x9cf6, [0x5954] = 0x9cf7, [0x5955] = 0x9cf8,
  [0x5956] = 0x9cf9, [0x5957] = 0x9cfa, [0x5958] = 0x9cfb, [0x5959] = 0x9cfc,
  [0x595a] = 0x9cfd, [0x595b] = 0x9cfe, [0x595c] = 0x9cff, [0x595d] = 0x9d00,
  [0x595e] = 0x9d01, [0x595f] = 0x9d02, [0x5960] = 0x9d03, [0x5961] = 0x9d04,
  [0x5962] = 0x9d05, [0x5963] = 0x9d06, [0x5964] = 0x9d07, [0x5965] = 0x9d08,
  [0x5966] = 0x9d09, [0x5967] = 0x9d0a, [0x5968] = 0x9d0b, [0x5969] = 0x9d0c,
  [0x596a] = 0x9d0d, [0x596b] = 0x9d0e, [0x596c] = 0x9d0f, [0x596d] = 0x9d10,
  [0x596e] = 0x9d11, [0x596f] = 0x9d12, [0x5970] = 0x9d13, [0x5971] = 0x9d14,
  [0x5972] = 0x9d15, [0x5973] = 0x9d16, [0x5974] = 0x9d17, [0x5975] = 0x9d18,
  [0x5976] = 0x9d19, [0x5977] = 0x9d1a, [0x5978] = 0x9d1b, [0x5979] = 0x9d1c,
  [0x597a] = 0x9d1d, [0x597b] = 0x9d1e, [0x597c] = 0x9d1f, [0x597d] = 0x9d20,
  [0x597e] = 0x9d21, [0x5980] = 0x9d22, [0x5981] = 0x9d23, [0x5982] = 0x9d24,
  [0x5983] = 0x9d25, [0x5984] = 0x9d26, [0x5985] = 0x9d27, [0x5986] = 0x9d28,
  [0x5987] = 0x9d29, [0x5988] = 0x9d2a, [0x5989] = 0x9d2b, [0x598a] = 0x9d2c,
  [0x598b] = 0x9d2d, [0x598c] = 0x9d2e, [0x598d] = 0x9d2f, [0x598e] = 0x9d30,
  [0x598f] = 0x9d31, [0x5990] = 0x9d32, [0x5991] = 0x9d33, [0x5992] = 0x9d34,
  [0x5993] = 0x9d35, [0x5994] = 0x9d36, [0x5995] = 0x9d37, [0x5996] = 0x9d38,
  [0x5997] = 0x9d39, [0x5998] = 0x9d3a, [0x5999] = 0x9d3b, [0x599a] = 0x9d3c,
  [0x599b] = 0x9d3d, [0x599c] = 0x9d3e, [0x599d] = 0x9d3f, [0x599e] = 0x9d40,
  [0x599f] = 0x9d41, [0x59a0] = 0x9d42, [0x5a00] = 0x9d43, [0x5a01] = 0x9d44,
  [0x5a02] = 0x9d45, [0x5a03] = 0x9d46, [0x5a04] = 0x9d47, [0x5a05] = 0x9d48,
  [0x5a06] = 0x9d49, [0x5a07] = 0x9d4a, [0x5a08] = 0x9d4b, [0x5a09] = 0x9d4c,
  [0x5a0a] = 0x9d4d, [0x5a0b] = 0x9d4e, [0x5a0c] = 0x9d4f, [0x5a0d] = 0x9d50,
  [0x5a0e] = 0x9d51, [0x5a0f] = 0x9d52, [0x5a10] = 0x9d53, [0x5a11] = 0x9d54,
  [0x5a12] = 0x9d55, [0x5a13] = 0x9d56, [0x5a14] = 0x9d57, [0x5a15] = 0x9d58,
  [0x5a16] = 0x9d59, [0x5a17] = 0x9d5a, [0x5a18] = 0x9d5b, [0x5a19] = 0x9d5c,
  [0x5a1a] = 0x9d5d, [0x5a1b] = 0x9d5e, [0x5a1c] = 0x9d5f, [0x5a1d] = 0x9d60,
  [0x5a1e] = 0x9d61, [0x5a1f] = 0x9d62, [0x5a20] = 0x9d63, [0x5a21] = 0x9d64,
  [0x5a22] = 0x9d65, [0x5a23] = 0x9d66, [0x5a24] = 0x9d67, [0x5a25] = 0x9d68,
  [0x5a26] = 0x9d69, [0x5a27] = 0x9d6a, [0x5a28] = 0x9d6b, [0x5a29] = 0x9d6c,
  [0x5a2a] = 0x9d6d, [0x5a2b] = 0x9d6e, [0x5a2c] = 0x9d6f, [0x5a2d] = 0x9d70,
  [0x5a2e] = 0x9d71, [0x5a2f] = 0x9d72, [0x5a30] = 0x9d73, [0x5a31] = 0x9d74,
  [0x5a32] = 0x9d75, [0x5a33] = 0x9d76, [0x5a34] = 0x9d77, [0x5a35] = 0x9d78,
  [0x5a36] = 0x9d79, [0x5a37] = 0x9d7a, [0x5a38] = 0x9d7b, [0x5a39] = 0x9d7c,
  [0x5a3a] = 0x9d7d, [0x5a3b] = 0x9d7e, [0x5a3c] = 0x9d7f, [0x5a3d] = 0x9d80,
  [0x5a3e] = 0x9d81, [0x5a40] = 0x9d82, [0x5a41] = 0x9d83, [0x5a42] = 0x9d84,
  [0x5a43] = 0x9d85, [0x5a44] = 0x9d86, [0x5a45] = 0x9d87, [0x5a46] = 0x9d88,
  [0x5a47] = 0x9d89, [0x5a48] = 0x9d8a, [0x5a49] = 0x9d8b, [0x5a4a] = 0x9d8c,
  [0x5a4b] = 0x9d8d, [0x5a4c] = 0x9d8e, [0x5a4d] = 0x9d8f, [0x5a4e] = 0x9d90,
  [0x5a4f] = 0x9d91, [0x5a50] = 0x9d92, [0x5a51] = 0x9d93, [0x5a52] = 0x9d94,
  [0x5a53] = 0x9d95, [0x5a54] = 0x9d96, [0x5a55] = 0x9d97, [0x5a56] = 0x9d98,
  [0x5a57] = 0x9d99, [0x5a58] = 0x9d9a, [0x5a59] = 0x9d9b, [0x5a5a] = 0x9d9c,
  [0x5a5b] = 0x9d9d, [0x5a5c] = 0x9d9e, [0x5a5d] = 0x9d9f, [0x5a5e] = 0x9da0,
  [0x5a5f] = 0x9da1, [0x5a60] = 0x9da2, [0x5ac0] = 0x9da3, [0x5ac1] = 0x9da4,
  [0x5ac2] = 0x9da5, [0x5ac3] = 0x9da6, [0x5ac4] = 0x9da7, [0x5ac5] = 0x9da8,
  [0x5ac6] = 0x9da9, [0x5ac7] = 0x9daa, [0x5ac8] = 0x9dab, [0x5ac9] = 0x9dac,
  [0x5aca] = 0x9dad, [0x5acb] = 0x9dae, [0x5acc] = 0x9daf, [0x5acd] = 0x9db0,
  [0x5ace] = 0x9db1, [0x5acf] = 0x9db2, [0x5ad0] = 0x9db3, [0x5ad1] = 0x9db4,
  [0x5ad2] = 0x9db5, [0x5ad3] = 0x9db6, [0x5ad4] = 0x9db7, [0x5ad5] = 0x9db8,
  [0x5ad6] = 0x9db9, [0x5ad7] = 0x9dba, [0x5ad8] = 0x9dbb, [0x5ad9] = 0x9dbc,
  [0x5ada] = 0x9dbd, [0x5adb] = 0x9dbe, [0x5adc] = 0x9dbf, [0x5add] = 0x9dc0,
  [0x5ade] = 0x9dc1, [0x5adf] = 0x9dc2, [0x5ae0] = 0x9dc3, [0x5ae1] = 0x9dc4,
  [0x5ae2] = 0x9dc5, [0x5ae3] = 0x9dc6, [0x5ae4] = 0x9dc7, [0x5ae5] = 0x9dc8,
  [0x5ae6] = 0x9dc9, [0x5ae7] = 0x9dca, [0x5ae8] = 0x9dcb, [0x5ae9] = 0x9dcc,
  [0x5aea] = 0x9dcd, [0x5aeb] = 0x9dce, [0x5aec] = 0x9dcf, [0x5aed] = 0x9dd0,
  [0x5aee] = 0x9dd1, [0x5aef] = 0x9dd2, [0x5af0] = 0x9dd3, [0x5af1] = 0x9dd4,
  [0x5af2] = 0x9dd5, [0x5af3] = 0x9dd6, [0x5af4] = 0x9dd7, [0x5af5] = 0x9dd8,
  [0x5af6] = 0x9dd9, [0x5af7] = 0x9dda, [0x5af8] = 0x9ddb, [0x5af9] = 0x9ddc,
  [0x5afa] = 0x9ddd, [0x5afb] = 0x9dde, [0x5afc] = 0x9ddf, [0x5afd] = 0x9de0,
  [0x5afe] = 0x9de1, [0x5b00] = 0x9de2, [0x5b01] = 0x9de3, [0x5b02] = 0x9de4,
  [0x5b03] = 0x9de5, [0x5b04] = 0x9de6, [0x5b05] = 0x9de7, [0x5b06] = 0x9de8,
  [0x5b07] = 0x9de9, [0x5b08] = 0x9dea, [0x5b09] = 0x9deb, [0x5b0a] = 0x9dec,
  [0x5b0b] = 0x9ded, [0x5b0c] = 0x9dee, [0x5b0d] = 0x9def, [0x5b0e] = 0x9df0,
  [0x5b0f] = 0x9df1, [0x5b10] = 0x9df2, [0x5b11] = 0x9df3, [0x5b12] = 0x9df4,
  [0x5b13] = 0x9df5, [0x5b14] = 0x9df6, [0x5b15] = 0x9df7, [0x5b16] = 0x9df8,
  [0x5b17] = 0x9df9, [0x5b18] = 0x9dfa, [0x5b19] = 0x9dfb, [0x5b1a] = 0x9dfc,
  [0x5b1b] = 0x9dfd, [0x5b1c] = 0x9dfe, [0x5b1d] = 0x9dff, [0x5b1e] = 0x9e00,
  [0x5b1f] = 0x9e01, [0x5b20] = 0x9e02, [0x5b80] = 0x9e03, [0x5b81] = 0x9e04,
  [0x5b82] = 0x9e05, [0x5b83] = 0x9e06, [0x5b84] = 0x9e07, [0x5b85] = 0x9e08,
  [0x5b86] = 0x9e09, [0x5b87] = 0x9e0a, [0x5b88] = 0x9e0b, [0x5b89] = 0x9e0c,
  [0x5b8a] = 0x9e0d, [0x5b8b] = 0x9e0e, [0x5b8c] = 0x9e0f, [0x5b8d] = 0x9e10,
  [0x5b8e] = 0x9e11, [0x5b8f] = 0x9e12, [0x5b90] = 0x9e13, [0x5b91] = 0x9e14,
  [0x5b92] = 0x9e15, [0x5b93] = 0x9e16, [0x5b94] = 0x9e17, [0x5b95] = 0x9e18,
  [0x5b96] = 0x9e19, [0x5b97] = 0x9e1a, [0x5b98] = 0x9e1b, [0x5b99] = 0x9e1c,
  [0x5b9a] = 0x9e1d, [0x5b9b] = 0x9e1e, [0x5b9c] = 0x9e24, [0x5b9d] = 0x9e27,
  [0x5b9e] = 0x9e2e, [0x5b9f] = 0x9e30, [0x5ba0] = 0x9e34, [0x5ba1] = 0x9e3b,
  [0x5ba2] = 0x9e3c, [0x5ba3] = 0x9e40, [0x5ba4] = 0x9e4d, [0x5ba5] = 0x9e50,
  [0x5ba6] = 0x9e52, [0x5ba7] = 0x9e53, [0x5ba8] = 0x9e54, [0x5ba9] = 0x9e56,
  [0x5baa] = 0x9e59, [0x5bab] = 0x9e5d, [0x5bac] = 0x9e5f, [0x5bad] = 0x9e60,
  [0x5bae] = 0x9e61, [0x5baf] = 0x9e62, [0x5bb0] = 0x9e65, [0x5bb1] = 0x9e6e,
  [0x5bb2] = 0x9e6f, [0x5bb3] = 0x9e72, [0x5bb4] = 0x9e74, [0x5bb5] = 0x9e75,
  [0x5bb6] = 0x9e76, [0x5bb7] = 0x9e77, [0x5bb8] = 0x9e78, [0x5bb9] = 0x9e79,
  [0x5bba] = 0x9e7a, [0x5bbb] = 0x9e7b, [0x5bbc] = 0x9e7c, [0x5bbd] = 0x9e7d,
  [0x5bbe] = 0x9e80, [0x5bc0] = 0x9e81, [0x5bc1] = 0x9e83, [0x5bc2] = 0x9e84,
  [0x5bc3] = 0x9e85, [0x5bc4] = 0x9e86, [0x5bc5] = 0x9e89, [0x5bc6] = 0x9e8a,
  [0x5bc7] = 0x9e8c, [0x5bc8] = 0x9e8d, [0x5bc9] = 0x9e8e, [0x5bca] = 0x9e8f,
  [0x5bcb] = 0x9e90, [0x5bcc] = 0x9e91, [0x5bcd] = 0x9e94, [0x5bce] = 0x9e95,
  [0x5bcf] = 0x9e96, [0x5bd0] = 0x9e97, [0x5bd1] = 0x9e98, [0x5bd2] = 0x9e99,
  [0x5bd3] = 0x9e9a, [0x5bd4] = 0x9e9b, [0x5bd5] = 0x9e9c, [0x5bd6] = 0x9e9e,
  [0x5bd7] = 0x9ea0, [0x5bd8] = 0x9ea1, [0x5bd9] = 0x9ea2, [0x5bda] = 0x9ea3,
  [0x5bdb] = 0x9ea4, [0x5bdc] = 0x9ea5, [0x5bdd] = 0x9ea7, [0x5bde] = 0x9ea8,
  [0x5bdf] = 0x9ea9, [0x5be0] = 0x9eaa, [0x5c40] = 0x9eab, [0x5c41] = 0x9eac,
  [0x5c42] = 0x9ead, [0x5c43] = 0x9eae, [0x5c44] = 0x9eaf, [0x5c45] = 0x9eb0,
  [0x5c46] = 0x9eb1, [0x5c47] = 0x9eb2, [0x5c48] = 0x9eb3, [0x5c49] = 0x9eb5,
  [0x5c4a] = 0x9eb6, [0x5c4b] = 0x9eb7, [0x5c4c] = 0x9eb9, [0x5c4d] = 0x9eba,
  [0x5c4e] = 0x9ebc, [0x5c4f] = 0x9ebf, [0x5c50] = 0x9ec0, [0x5c51] = 0x9ec1,
  [0x5c52] = 0x9ec2, [0x5c53] = 0x9ec3, [0x5c54] = 0x9ec5, [0x5c55] = 0x9ec6,
  [0x5c56] = 0x9ec7, [0x5c57] = 0x9ec8, [0x5c58] = 0x9eca, [0x5c59] = 0x9ecb,
  [0x5c5a] = 0x9ecc, [0x5c5b] = 0x9ed0, [0x5c5c] = 0x9ed2, [0x5c5d] = 0x9ed3,
  [0x5c5e] = 0x9ed5, [0x5c5f] = 0x9ed6, [0x5c60] = 0x9ed7, [0x5c61] = 0x9ed9,
  [0x5c62] = 0x9eda, [0x5c63] = 0x9ede, [0x5c64] = 0x9ee1, [0x5c65] = 0x9ee3,
  [0x5c66] = 0x9ee4, [0x5c67] = 0x9ee6, [0x5c68] = 0x9ee8, [0x5c69] = 0x9eeb,
  [0x5c6a] = 0x9eec, [0x5c6b] = 0x9eed, [0x5c6c] = 0x9eee, [0x5c6d] = 0x9ef0,
  [0x5c6e] = 0x9ef1, [0x5c6f] = 0x9ef2, [0x5c70] = 0x9ef3, [0x5c71] = 0x9ef4,
  [0x5c72] = 0x9ef5, [0x5c73] = 0x9ef6, [0x5c74] = 0x9ef7, [0x5c75] = 0x9ef8,
  [0x5c76] = 0x9efa, [0x5c77] = 0x9efd, [0x5c78] = 0x9eff, [0x5c79] = 0x9f00,
  [0x5c7a] = 0x9f01, [0x5c7b] = 0x9f02, [0x5c7c] = 0x9f03, [0x5c7d] = 0x9f04,
  [0x5c7e] = 0x9f05, [0x5c80] = 0x9f06, [0x5c81] = 0x9f07, [0x5c82] = 0x9f08,
  [0x5c83] = 0x9f09, [0x5c84] = 0x9f0a, [0x5c85] = 0x9f0c, [0x5c86] = 0x9f0f,
  [0x5c87] = 0x9f11, [0x5c88] = 0x9f12, [0x5c89] = 0x9f14, [0x5c8a] = 0x9f15,
  [0x5c8b] = 0x9f16, [0x5c8c] = 0x9f18, [0x5c8d] = 0x9f1a, [0x5c8e] = 0x9f1b,
  [0x5c8f] = 0x9f1c, [0x5c90] = 0x9f1d, [0x5c91] = 0x9f1e, [0x5c92] = 0x9f1f,
  [0x5c93] = 0x9f21, [0x5c94] = 0x9f23, [0x5c95] = 0x9f24, [0x5c96] = 0x9f25,
  [0x5c97] = 0x9f26, [0x5c98] = 0x9f27, [0x5c99] = 0x9f28, [0x5c9a] = 0x9f29,
  [0x5c9b] = 0x9f2a, [0x5c9c] = 0x9f2b, [0x5c9d] = 0x9f2d, [0x5c9e] = 0x9f2e,
  [0x5c9f] = 0x9f30, [0x5ca0] = 0x9f31, [0x5d00] = 0x9f32, [0x5d01] = 0x9f33,
  [0x5d02] = 0x9f34, [0x5d03] = 0x9f35, [0x5d04] = 0x9f36, [0x5d05] = 0x9f38,
  [0x5d06] = 0x9f3a, [0x5d07] = 0x9f3c, [0x5d08] = 0x9f3f, [0x5d09] = 0x9f40,
  [0x5d0a] = 0x9f41, [0x5d0b] = 0x9f42, [0x5d0c] = 0x9f43, [0x5d0d] = 0x9f45,
  [0x5d0e] = 0x9f46, [0x5d0f] = 0x9f47, [0x5d10] = 0x9f48, [0x5d11] = 0x9f49,
  [0x5d12] = 0x9f4a, [0x5d13] = 0x9f4b, [0x5d14] = 0x9f4c, [0x5d15] = 0x9f4d,
  [0x5d16] = 0x9f4e, [0x5d17] = 0x9f4f, [0x5d18] = 0x9f52, [0x5d19] = 0x9f53,
  [0x5d1a] = 0x9f54, [0x5d1b] = 0x9f55, [0x5d1c] = 0x9f56, [0x5d1d] = 0x9f57,
  [0x5d1e] = 0x9f58, [0x5d1f] = 0x9f59, [0x5d20] = 0x9f5a, [0x5d21] = 0x9f5b,
  [0x5d22] = 0x9f5c, [0x5d23] = 0x9f5d, [0x5d24] = 0x9f5e, [0x5d25] = 0x9f5f,
  [0x5d26] = 0x9f60, [0x5d27] = 0x9f61, [0x5d28] = 0x9f62, [0x5d29] = 0x9f63,
  [0x5d2a] = 0x9f64, [0x5d2b] = 0x9f65, [0x5d2c] = 0x9f66, [0x5d2d] = 0x9f67,
  [0x5d2e] = 0x9f68, [0x5d2f] = 0x9f69, [0x5d30] = 0x9f6a, [0x5d31] = 0x9f6b,
  [0x5d32] = 0x9f6c, [0x5d33] = 0x9f6d, [0x5d34] = 0x9f6e, [0x5d35] = 0x9f6f,
  [0x5d36] = 0x9f70, [0x5d37] = 0x9f71, [0x5d38] = 0x9f72, [0x5d39] = 0x9f73,
  [0x5d3a] = 0x9f74, [0x5d3b] = 0x9f75, [0x5d3c] = 0x9f76, [0x5d3d] = 0x9f77,
  [0x5d3e] = 0x9f78, [0x5d40] = 0x9f79, [0x5d41] = 0x9f7a, [0x5d42] = 0x9f7b,
  [0x5d43] = 0x9f7c, [0x5d44] = 0x9f7d, [0x5d45] = 0x9f7e, [0x5d46] = 0x9f81,
  [0x5d47] = 0x9f82, [0x5d48] = 0x9f8d, [0x5d49] = 0x9f8e, [0x5d4a] = 0x9f8f,
  [0x5d4b] = 0x9f90, [0x5d4c] = 0x9f91, [0x5d4d] = 0x9f92, [0x5d4e] = 0x9f93,
  [0x5d4f] = 0x9f94, [0x5d50] = 0x9f95, [0x5d51] = 0x9f96, [0x5d52] = 0x9f97,
  [0x5d53] = 0x9f98, [0x5d54] = 0x9f9c, [0x5d55] = 0x9f9d, [0x5d56] = 0x9f9e,
  [0x5d57] = 0x9fa1, [0x5d58] = 0x9fa2, [0x5d59] = 0x9fa3, [0x5d5a] = 0x9fa4,
  [0x5d5b] = 0x9fa5, [0x5d5c] = 0xf92c, [0x5d5d] = 0xf979, [0x5d5e] = 0xf995,
  [0x5d5f] = 0xf9e7, [0x5d60] = 0xf9f1, [0x5dc0] = 0xfa0c, [0x5dc1] = 0xfa0d,
  [0x5dc2] = 0xfa0e, [0x5dc3] = 0xfa0f, [0x5dc4] = 0xfa11, [0x5dc5] = 0xfa13,
  [0x5dc6] = 0xfa14, [0x5dc7] = 0xfa18, [0x5dc8] = 0xfa1f, [0x5dc9] = 0xfa20,
  [0x5dca] = 0xfa21, [0x5dcb] = 0xfa23, [0x5dcc] = 0xfa24, [0x5dcd] = 0xfa27,
  [0x5dce] = 0xfa28, [0x5dcf] = 0xfa29,
#if USE_PRIVATE_AREA
                                        [0x5dd0] = 0xe815, [0x5dd1] = 0xe816,
  [0x5dd2] = 0xe817, [0x5dd3] = 0xe818, [0x5dd4] = 0xe819, [0x5dd5] = 0xe81a,
  [0x5dd6] = 0xe81b, [0x5dd7] = 0xe81c, [0x5dd8] = 0xe81d, [0x5dd9] = 0xe81e,
  [0x5dda] = 0xe81f, [0x5ddb] = 0xe820, [0x5ddc] = 0xe821, [0x5ddd] = 0xe822,
  [0x5dde] = 0xe823, [0x5ddf] = 0xe824, [0x5de0] = 0xe825, [0x5de1] = 0xe826,
  [0x5de2] = 0xe827, [0x5de3] = 0xe828, [0x5de4] = 0xe829, [0x5de5] = 0xe82a,
  [0x5de6] = 0xe82b, [0x5de7] = 0xe82c, [0x5de8] = 0xe82d, [0x5de9] = 0xe82e,
  [0x5dea] = 0xe82f, [0x5deb] = 0xe830, [0x5dec] = 0xe831, [0x5ded] = 0xe832,
  [0x5dee] = 0xe833, [0x5def] = 0xe834, [0x5df0] = 0xe835, [0x5df1] = 0xe836,
  [0x5df2] = 0xe837, [0x5df3] = 0xe838, [0x5df4] = 0xe839, [0x5df5] = 0xe83a,
  [0x5df6] = 0xe83b, [0x5df7] = 0xe83c, [0x5df8] = 0xe83d, [0x5df9] = 0xe83e,
  [0x5dfa] = 0xe83f, [0x5dfb] = 0xe840, [0x5dfc] = 0xe841, [0x5dfd] = 0xe842,
  [0x5dfe] = 0xe843, [0x5e00] = 0xe844, [0x5e01] = 0xe845, [0x5e02] = 0xe846,
  [0x5e03] = 0xe847, [0x5e04] = 0xe848, [0x5e05] = 0xe849, [0x5e06] = 0xe84a,
  [0x5e07] = 0xe84b, [0x5e08] = 0xe84c, [0x5e09] = 0xe84d, [0x5e0a] = 0xe84e,
  [0x5e0b] = 0xe84f, [0x5e0c] = 0xe850, [0x5e0d] = 0xe851, [0x5e0e] = 0xe852,
  [0x5e0f] = 0xe853, [0x5e10] = 0xe854, [0x5e11] = 0xe855, [0x5e12] = 0xe856,
  [0x5e13] = 0xe857, [0x5e14] = 0xe858, [0x5e15] = 0xe859, [0x5e16] = 0xe85a,
  [0x5e17] = 0xe85b, [0x5e18] = 0xe85c, [0x5e19] = 0xe85d, [0x5e1a] = 0xe85e,
  [0x5e1b] = 0xe85f, [0x5e1c] = 0xe860, [0x5e1d] = 0xe861, [0x5e1e] = 0xe862,
  [0x5e1f] = 0xe863, [0x5e20] = 0xe864,
#else
  [0x5e20] = 0x0000,
#endif
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0x00' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0xa4,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab1[][2] =
{
  [0x0000] = "\xa1\xe8", [0x0003] = "\xa1\xec", [0x0004] = "\xa1\xa7",
  [0x000c] = "\xa1\xe3", [0x000d] = "\xa1\xc0", [0x0013] = "\xa1\xa4",
  [0x0033] = "\xa1\xc1", [0x003c] = "\xa8\xa4", [0x003d] = "\xa8\xa2",
  [0x0044] = "\xa8\xa8", [0x0045] = "\xa8\xa6", [0x0046] = "\xa8\xba",
  [0x0048] = "\xa8\xac", [0x0049] = "\xa8\xaa", [0x004e] = "\xa8\xb0",
  [0x004f] = "\xa8\xae", [0x0053] = "\xa1\xc2", [0x0055] = "\xa8\xb4",
  [0x0056] = "\xa8\xb2", [0x0058] = "\xa8\xb9", [0x005d] = "\xa8\xa1"
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0x03' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x391,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab2[][2] =
{
  [0x0000] = "\xa6\xa1", [0x0001] = "\xa6\xa2", [0x0002] = "\xa6\xa3",
  [0x0003] = "\xa6\xa4", [0x0004] = "\xa6\xa5", [0x0005] = "\xa6\xa6",
  [0x0006] = "\xa6\xa7", [0x0007] = "\xa6\xa8", [0x0008] = "\xa6\xa9",
  [0x0009] = "\xa6\xaa", [0x000a] = "\xa6\xab", [0x000b] = "\xa6\xac",
  [0x000c] = "\xa6\xad", [0x000d] = "\xa6\xae", [0x000e] = "\xa6\xaf",
  [0x000f] = "\xa6\xb0", [0x0010] = "\xa6\xb1", [0x0012] = "\xa6\xb2",
  [0x0013] = "\xa6\xb3", [0x0014] = "\xa6\xb4", [0x0015] = "\xa6\xb5",
  [0x0016] = "\xa6\xb6", [0x0017] = "\xa6\xb7", [0x0018] = "\xa6\xb8",
  [0x0020] = "\xa6\xc1", [0x0021] = "\xa6\xc2", [0x0022] = "\xa6\xc3",
  [0x0023] = "\xa6\xc4", [0x0024] = "\xa6\xc5", [0x0025] = "\xa6\xc6",
  [0x0026] = "\xa6\xc7", [0x0027] = "\xa6\xc8", [0x0028] = "\xa6\xc9",
  [0x0029] = "\xa6\xca", [0x002a] = "\xa6\xcb", [0x002b] = "\xa6\xcc",
  [0x002c] = "\xa6\xcd", [0x002d] = "\xa6\xce", [0x002e] = "\xa6\xcf",
  [0x002f] = "\xa6\xd0", [0x0030] = "\xa6\xd1", [0x0032] = "\xa6\xd2",
  [0x0033] = "\xa6\xd3", [0x0034] = "\xa6\xd4", [0x0035] = "\xa6\xd5",
  [0x0036] = "\xa6\xd6", [0x0037] = "\xa6\xd7", [0x0038] = "\xa6\xd8",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0x04' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x401,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab3[][2] =
{
  [0x0000] = "\xa7\xa7", [0x000f] = "\xa7\xa1", [0x0010] = "\xa7\xa2",
  [0x0011] = "\xa7\xa3", [0x0012] = "\xa7\xa4", [0x0013] = "\xa7\xa5",
  [0x0014] = "\xa7\xa6", [0x0015] = "\xa7\xa8", [0x0016] = "\xa7\xa9",
  [0x0017] = "\xa7\xaa", [0x0018] = "\xa7\xab", [0x0019] = "\xa7\xac",
  [0x001a] = "\xa7\xad", [0x001b] = "\xa7\xae", [0x001c] = "\xa7\xaf",
  [0x001d] = "\xa7\xb0", [0x001e] = "\xa7\xb1", [0x001f] = "\xa7\xb2",
  [0x0020] = "\xa7\xb3", [0x0021] = "\xa7\xb4", [0x0022] = "\xa7\xb5",
  [0x0023] = "\xa7\xb6", [0x0024] = "\xa7\xb7", [0x0025] = "\xa7\xb8",
  [0x0026] = "\xa7\xb9", [0x0027] = "\xa7\xba", [0x0028] = "\xa7\xbb",
  [0x0029] = "\xa7\xbc", [0x002a] = "\xa7\xbd", [0x002b] = "\xa7\xbe",
  [0x002c] = "\xa7\xbf", [0x002d] = "\xa7\xc0", [0x002e] = "\xa7\xc1",
  [0x002f] = "\xa7\xd1", [0x0030] = "\xa7\xd2", [0x0031] = "\xa7\xd3",
  [0x0032] = "\xa7\xd4", [0x0033] = "\xa7\xd5", [0x0034] = "\xa7\xd6",
  [0x0035] = "\xa7\xd8", [0x0036] = "\xa7\xd9", [0x0037] = "\xa7\xda",
  [0x0038] = "\xa7\xdb", [0x0039] = "\xa7\xdc", [0x003a] = "\xa7\xdd",
  [0x003b] = "\xa7\xde", [0x003c] = "\xa7\xdf", [0x003d] = "\xa7\xe0",
  [0x003e] = "\xa7\xe1", [0x003f] = "\xa7\xe2", [0x0040] = "\xa7\xe3",
  [0x0041] = "\xa7\xe4", [0x0042] = "\xa7\xe5", [0x0043] = "\xa7\xe6",
  [0x0044] = "\xa7\xe7", [0x0045] = "\xa7\xe8", [0x0046] = "\xa7\xe9",
  [0x0047] = "\xa7\xea", [0x0048] = "\xa7\xeb", [0x0049] = "\xa7\xec",
  [0x004a] = "\xa7\xed", [0x004b] = "\xa7\xee", [0x004c] = "\xa7\xef",
  [0x004d] = "\xa7\xf0", [0x004e] = "\xa7\xf1", [0x0050] = "\xa7\xd7",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0x20' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x2010,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab4[][2] =
{
  [0x0000] = "\xa9\x5c", [0x0003] = "\xa8\x43", [0x0004] = "\xa1\xaa",
  [0x0005] = "\xa8\x44", [0x0006] = "\xa1\xac", [0x0008] = "\xa1\xae",
  [0x0009] = "\xa1\xaf", [0x000c] = "\xa1\xb0", [0x000d] = "\xa1\xb1",
  [0x0015] = "\xa8\x45", [0x0016] = "\xa1\xad", [0x0020] = "\xa1\xeb",
  [0x0022] = "\xa1\xe4", [0x0023] = "\xa1\xe5", [0x0025] = "\xa8\x46",
  [0x002b] = "\xa1\xf9",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0x2[12]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x2103,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab5[][2] =
{
  [0x0000] = "\xa1\xe6", [0x0002] = "\xa8\x47", [0x0006] = "\xa8\x48",
  [0x0013] = "\xa1\xed", [0x001e] = "\xa9\x59", [0x005d] = "\xa2\xf1",
  [0x005e] = "\xa2\xf2", [0x005f] = "\xa2\xf3", [0x0060] = "\xa2\xf4",
  [0x0061] = "\xa2\xf5", [0x0062] = "\xa2\xf6", [0x0063] = "\xa2\xf7",
  [0x0064] = "\xa2\xf8", [0x0065] = "\xa2\xf9", [0x0066] = "\xa2\xfa",
  [0x0067] = "\xa2\xfb", [0x0068] = "\xa2\xfc", [0x006d] = "\xa2\xa1",
  [0x006e] = "\xa2\xa2", [0x006f] = "\xa2\xa3", [0x0070] = "\xa2\xa4",
  [0x0071] = "\xa2\xa5", [0x0072] = "\xa2\xa6", [0x0073] = "\xa2\xa7",
  [0x0074] = "\xa2\xa8", [0x0075] = "\xa2\xa9", [0x0076] = "\xa2\xaa",
  [0x008d] = "\xa1\xfb", [0x008e] = "\xa1\xfc", [0x008f] = "\xa1\xfa",
  [0x0090] = "\xa1\xfd", [0x0093] = "\xa8\x49", [0x0094] = "\xa8\x4a",
  [0x0095] = "\xa8\x4b", [0x0096] = "\xa8\x4c", [0x0105] = "\xa1\xca",
  [0x010c] = "\xa1\xc7", [0x010e] = "\xa1\xc6", [0x0112] = "\xa8\x4d",
  [0x0117] = "\xa1\xcc", [0x011a] = "\xa1\xd8", [0x011b] = "\xa1\xde",
  [0x011c] = "\xa8\x4e", [0x011d] = "\xa1\xcf", [0x0120] = "\xa8\x4f",
  [0x0122] = "\xa1\xce", [0x0124] = "\xa1\xc4", [0x0125] = "\xa1\xc5",
  [0x0126] = "\xa1\xc9", [0x0127] = "\xa1\xc8", [0x0128] = "\xa1\xd2",
  [0x012b] = "\xa1\xd3", [0x0131] = "\xa1\xe0", [0x0132] = "\xa1\xdf",
  [0x0133] = "\xa1\xc3", [0x0134] = "\xa1\xcb", [0x013a] = "\xa1\xd7",
  [0x0145] = "\xa1\xd6", [0x0149] = "\xa1\xd5", [0x014f] = "\xa8\x50",
  [0x015d] = "\xa1\xd9", [0x015e] = "\xa1\xd4", [0x0161] = "\xa1\xdc",
  [0x0162] = "\xa1\xdd", [0x0163] = "\xa8\x51", [0x0164] = "\xa8\x52",
  [0x016b] = "\xa1\xda", [0x016c] = "\xa1\xdb", [0x0192] = "\xa8\x92",
  [0x0196] = "\xa1\xd1", [0x01a2] = "\xa1\xcd", [0x01bc] = "\xa8\x53",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0x2[4-6]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x2460,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab6[][2] =
{
  [0x0000] = "\xa2\xd9", [0x0001] = "\xa2\xda", [0x0002] = "\xa2\xdb",
  [0x0003] = "\xa2\xdc", [0x0004] = "\xa2\xdd", [0x0005] = "\xa2\xde",
  [0x0006] = "\xa2\xdf", [0x0007] = "\xa2\xe0", [0x0008] = "\xa2\xe1",
  [0x0009] = "\xa2\xe2", [0x0014] = "\xa2\xc5", [0x0015] = "\xa2\xc6",
  [0x0016] = "\xa2\xc7", [0x0017] = "\xa2\xc8", [0x0018] = "\xa2\xc9",
  [0x0019] = "\xa2\xca", [0x001a] = "\xa2\xcb", [0x001b] = "\xa2\xcc",
  [0x001c] = "\xa2\xcd", [0x001d] = "\xa2\xce", [0x001e] = "\xa2\xcf",
  [0x001f] = "\xa2\xd0", [0x0020] = "\xa2\xd1", [0x0021] = "\xa2\xd2",
  [0x0022] = "\xa2\xd3", [0x0023] = "\xa2\xd4", [0x0024] = "\xa2\xd5",
  [0x0025] = "\xa2\xd6", [0x0026] = "\xa2\xd7", [0x0027] = "\xa2\xd8",
  [0x0028] = "\xa2\xb1", [0x0029] = "\xa2\xb2", [0x002a] = "\xa2\xb3",
  [0x002b] = "\xa2\xb4", [0x002c] = "\xa2\xb5", [0x002d] = "\xa2\xb6",
  [0x002e] = "\xa2\xb7", [0x002f] = "\xa2\xb8", [0x0030] = "\xa2\xb9",
  [0x0031] = "\xa2\xba", [0x0032] = "\xa2\xbb", [0x0033] = "\xa2\xbc",
  [0x0034] = "\xa2\xbd", [0x0035] = "\xa2\xbe", [0x0036] = "\xa2\xbf",
  [0x0037] = "\xa2\xc0", [0x0038] = "\xa2\xc1", [0x0039] = "\xa2\xc2",
  [0x003a] = "\xa2\xc3", [0x003b] = "\xa2\xc4", [0x00a0] = "\xa9\xa4",
  [0x00a1] = "\xa9\xa5", [0x00a2] = "\xa9\xa6", [0x00a3] = "\xa9\xa7",
  [0x00a4] = "\xa9\xa8", [0x00a5] = "\xa9\xa9", [0x00a6] = "\xa9\xaa",
  [0x00a7] = "\xa9\xab", [0x00a8] = "\xa9\xac", [0x00a9] = "\xa9\xad",
  [0x00aa] = "\xa9\xae", [0x00ab] = "\xa9\xaf", [0x00ac] = "\xa9\xb0",
  [0x00ad] = "\xa9\xb1", [0x00ae] = "\xa9\xb2", [0x00af] = "\xa9\xb3",
  [0x00b0] = "\xa9\xb4", [0x00b1] = "\xa9\xb5", [0x00b2] = "\xa9\xb6",
  [0x00b3] = "\xa9\xb7", [0x00b4] = "\xa9\xb8", [0x00b5] = "\xa9\xb9",
  [0x00b6] = "\xa9\xba", [0x00b7] = "\xa9\xbb", [0x00b8] = "\xa9\xbc",
  [0x00b9] = "\xa9\xbd", [0x00ba] = "\xa9\xbe", [0x00bb] = "\xa9\xbf",
  [0x00bc] = "\xa9\xc0", [0x00bd] = "\xa9\xc1", [0x00be] = "\xa9\xc2",
  [0x00bf] = "\xa9\xc3", [0x00c0] = "\xa9\xc4", [0x00c1] = "\xa9\xc5",
  [0x00c2] = "\xa9\xc6", [0x00c3] = "\xa9\xc7", [0x00c4] = "\xa9\xc8",
  [0x00c5] = "\xa9\xc9", [0x00c6] = "\xa9\xca", [0x00c7] = "\xa9\xcb",
  [0x00c8] = "\xa9\xcc", [0x00c9] = "\xa9\xcd", [0x00ca] = "\xa9\xce",
  [0x00cb] = "\xa9\xcf", [0x00cc] = "\xa9\xd0", [0x00cd] = "\xa9\xd1",
  [0x00ce] = "\xa9\xd2", [0x00cf] = "\xa9\xd3", [0x00d0] = "\xa9\xd4",
  [0x00d1] = "\xa9\xd5", [0x00d2] = "\xa9\xd6", [0x00d3] = "\xa9\xd7",
  [0x00d4] = "\xa9\xd8", [0x00d5] = "\xa9\xd9", [0x00d6] = "\xa9\xda",
  [0x00d7] = "\xa9\xdb", [0x00d8] = "\xa9\xdc", [0x00d9] = "\xa9\xdd",
  [0x00da] = "\xa9\xde", [0x00db] = "\xa9\xdf", [0x00dc] = "\xa9\xe0",
  [0x00dd] = "\xa9\xe1", [0x00de] = "\xa9\xe2", [0x00df] = "\xa9\xe3",
  [0x00e0] = "\xa9\xe4", [0x00e1] = "\xa9\xe5", [0x00e2] = "\xa9\xe6",
  [0x00e3] = "\xa9\xe7", [0x00e4] = "\xa9\xe8", [0x00e5] = "\xa9\xe9",
  [0x00e6] = "\xa9\xea", [0x00e7] = "\xa9\xeb", [0x00e8] = "\xa9\xec",
  [0x00e9] = "\xa9\xed", [0x00ea] = "\xa9\xee", [0x00eb] = "\xa9\xef",
  [0x00f0] = "\xa8\x54", [0x00f1] = "\xa8\x55", [0x00f2] = "\xa8\x56",
  [0x00f3] = "\xa8\x57", [0x00f4] = "\xa8\x58", [0x00f5] = "\xa8\x59",
  [0x00f6] = "\xa8\x5a", [0x00f7] = "\xa8\x5b", [0x00f8] = "\xa8\x5c",
  [0x00f9] = "\xa8\x5d", [0x00fa] = "\xa8\x5e", [0x00fb] = "\xa8\x5f",
  [0x00fc] = "\xa8\x60", [0x00fd] = "\xa8\x61", [0x00fe] = "\xa8\x62",
  [0x00ff] = "\xa8\x63", [0x0100] = "\xa8\x64", [0x0101] = "\xa8\x65",
  [0x0102] = "\xa8\x66", [0x0103] = "\xa8\x67", [0x0104] = "\xa8\x68",
  [0x0105] = "\xa8\x69", [0x0106] = "\xa8\x6a", [0x0107] = "\xa8\x6b",
  [0x0108] = "\xa8\x6c", [0x0109] = "\xa8\x6d", [0x010a] = "\xa8\x6e",
  [0x010b] = "\xa8\x6f", [0x010c] = "\xa8\x70", [0x010d] = "\xa8\x71",
  [0x010e] = "\xa8\x72", [0x010f] = "\xa8\x73", [0x0110] = "\xa8\x74",
  [0x0111] = "\xa8\x75", [0x0112] = "\xa8\x76", [0x0113] = "\xa8\x77",
  [0x0121] = "\xa8\x78", [0x0122] = "\xa8\x79", [0x0123] = "\xa8\x7a",
  [0x0124] = "\xa8\x7b", [0x0125] = "\xa8\x7c", [0x0126] = "\xa8\x7d",
  [0x0127] = "\xa8\x7e", [0x0128] = "\xa8\x80", [0x0129] = "\xa8\x81",
  [0x012a] = "\xa8\x82", [0x012b] = "\xa8\x83", [0x012c] = "\xa8\x84",
  [0x012d] = "\xa8\x85", [0x012e] = "\xa8\x86", [0x012f] = "\xa8\x87",
  [0x0133] = "\xa8\x88", [0x0134] = "\xa8\x89", [0x0135] = "\xa8\x8a",
  [0x0140] = "\xa1\xf6", [0x0141] = "\xa1\xf5", [0x0152] = "\xa1\xf8",
  [0x0153] = "\xa1\xf7", [0x015c] = "\xa8\x8b", [0x015d] = "\xa8\x8c",
  [0x0166] = "\xa1\xf4", [0x0167] = "\xa1\xf3", [0x016b] = "\xa1\xf0",
  [0x016e] = "\xa1\xf2", [0x016f] = "\xa1\xf1", [0x0182] = "\xa8\x8d",
  [0x0183] = "\xa8\x8e", [0x0184] = "\xa8\x8f", [0x0185] = "\xa8\x90",
  [0x01a5] = "\xa1\xef", [0x01a6] = "\xa1\xee", [0x01a9] = "\xa8\x91",
  [0x01e0] = "\xa1\xe2", [0x01e2] = "\xa1\xe1",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0x3[0-3]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x3000,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab7[][2] =
{
  [0x0000] = "\xa1\xa1", [0x0001] = "\xa1\xa2", [0x0002] = "\xa1\xa3",
  [0x0003] = "\xa1\xa8", [0x0005] = "\xa1\xa9", [0x0006] = "\xa9\x65",
  [0x0007] = "\xa9\x96", [0x0008] = "\xa1\xb4", [0x0009] = "\xa1\xb5",
  [0x000a] = "\xa1\xb6", [0x000b] = "\xa1\xb7", [0x000c] = "\xa1\xb8",
  [0x000d] = "\xa1\xb9", [0x000e] = "\xa1\xba", [0x000f] = "\xa1\xbb",
  [0x0010] = "\xa1\xbe", [0x0011] = "\xa1\xbf", [0x0012] = "\xa8\x93",
  [0x0013] = "\xa1\xfe", [0x0014] = "\xa1\xb2", [0x0015] = "\xa1\xb3",
  [0x0016] = "\xa1\xbc", [0x0017] = "\xa1\xbd", [0x001d] = "\xa8\x94",
  [0x001e] = "\xa8\x95", [0x0021] = "\xa9\x40", [0x0022] = "\xa9\x41",
  [0x0023] = "\xa9\x42", [0x0024] = "\xa9\x43", [0x0025] = "\xa9\x44",
  [0x0026] = "\xa9\x45", [0x0027] = "\xa9\x46", [0x0028] = "\xa9\x47",
  [0x0029] = "\xa9\x48", [0x0041] = "\xa4\xa1", [0x0042] = "\xa4\xa2",
  [0x0043] = "\xa4\xa3", [0x0044] = "\xa4\xa4", [0x0045] = "\xa4\xa5",
  [0x0046] = "\xa4\xa6", [0x0047] = "\xa4\xa7", [0x0048] = "\xa4\xa8",
  [0x0049] = "\xa4\xa9", [0x004a] = "\xa4\xaa", [0x004b] = "\xa4\xab",
  [0x004c] = "\xa4\xac", [0x004d] = "\xa4\xad", [0x004e] = "\xa4\xae",
  [0x004f] = "\xa4\xaf", [0x0050] = "\xa4\xb0", [0x0051] = "\xa4\xb1",
  [0x0052] = "\xa4\xb2", [0x0053] = "\xa4\xb3", [0x0054] = "\xa4\xb4",
  [0x0055] = "\xa4\xb5", [0x0056] = "\xa4\xb6", [0x0057] = "\xa4\xb7",
  [0x0058] = "\xa4\xb8", [0x0059] = "\xa4\xb9", [0x005a] = "\xa4\xba",
  [0x005b] = "\xa4\xbb", [0x005c] = "\xa4\xbc", [0x005d] = "\xa4\xbd",
  [0x005e] = "\xa4\xbe", [0x005f] = "\xa4\xbf", [0x0060] = "\xa4\xc0",
  [0x0061] = "\xa4\xc1", [0x0062] = "\xa4\xc2", [0x0063] = "\xa4\xc3",
  [0x0064] = "\xa4\xc4", [0x0065] = "\xa4\xc5", [0x0066] = "\xa4\xc6",
  [0x0067] = "\xa4\xc7", [0x0068] = "\xa4\xc8", [0x0069] = "\xa4\xc9",
  [0x006a] = "\xa4\xca", [0x006b] = "\xa4\xcb", [0x006c] = "\xa4\xcc",
  [0x006d] = "\xa4\xcd", [0x006e] = "\xa4\xce", [0x006f] = "\xa4\xcf",
  [0x0070] = "\xa4\xd0", [0x0071] = "\xa4\xd1", [0x0072] = "\xa4\xd2",
  [0x0073] = "\xa4\xd3", [0x0074] = "\xa4\xd4", [0x0075] = "\xa4\xd5",
  [0x0076] = "\xa4\xd6", [0x0077] = "\xa4\xd7", [0x0078] = "\xa4\xd8",
  [0x0079] = "\xa4\xd9", [0x007a] = "\xa4\xda", [0x007b] = "\xa4\xdb",
  [0x007c] = "\xa4\xdc", [0x007d] = "\xa4\xdd", [0x007e] = "\xa4\xde",
  [0x007f] = "\xa4\xdf", [0x0080] = "\xa4\xe0", [0x0081] = "\xa4\xe1",
  [0x0082] = "\xa4\xe2", [0x0083] = "\xa4\xe3", [0x0084] = "\xa4\xe4",
  [0x0085] = "\xa4\xe5", [0x0086] = "\xa4\xe6", [0x0087] = "\xa4\xe7",
  [0x0088] = "\xa4\xe8", [0x0089] = "\xa4\xe9", [0x008a] = "\xa4\xea",
  [0x008b] = "\xa4\xeb", [0x008c] = "\xa4\xec", [0x008d] = "\xa4\xed",
  [0x008e] = "\xa4\xee", [0x008f] = "\xa4\xef", [0x0090] = "\xa4\xf0",
  [0x0091] = "\xa4\xf1", [0x0092] = "\xa4\xf2", [0x0093] = "\xa4\xf3",
  [0x009b] = "\xa9\x61", [0x009c] = "\xa9\x62", [0x009d] = "\xa9\x66",
  [0x009e] = "\xa9\x67", [0x00a1] = "\xa5\xa1", [0x00a2] = "\xa5\xa2",
  [0x00a3] = "\xa5\xa3", [0x00a4] = "\xa5\xa4", [0x00a5] = "\xa5\xa5",
  [0x00a6] = "\xa5\xa6", [0x00a7] = "\xa5\xa7", [0x00a8] = "\xa5\xa8",
  [0x00a9] = "\xa5\xa9", [0x00aa] = "\xa5\xaa", [0x00ab] = "\xa5\xab",
  [0x00ac] = "\xa5\xac", [0x00ad] = "\xa5\xad", [0x00ae] = "\xa5\xae",
  [0x00af] = "\xa5\xaf", [0x00b0] = "\xa5\xb0", [0x00b1] = "\xa5\xb1",
  [0x00b2] = "\xa5\xb2", [0x00b3] = "\xa5\xb3", [0x00b4] = "\xa5\xb4",
  [0x00b5] = "\xa5\xb5", [0x00b6] = "\xa5\xb6", [0x00b7] = "\xa5\xb7",
  [0x00b8] = "\xa5\xb8", [0x00b9] = "\xa5\xb9", [0x00ba] = "\xa5\xba",
  [0x00bb] = "\xa5\xbb", [0x00bc] = "\xa5\xbc", [0x00bd] = "\xa5\xbd",
  [0x00be] = "\xa5\xbe", [0x00bf] = "\xa5\xbf", [0x00c0] = "\xa5\xc0",
  [0x00c1] = "\xa5\xc1", [0x00c2] = "\xa5\xc2", [0x00c3] = "\xa5\xc3",
  [0x00c4] = "\xa5\xc4", [0x00c5] = "\xa5\xc5", [0x00c6] = "\xa5\xc6",
  [0x00c7] = "\xa5\xc7", [0x00c8] = "\xa5\xc8", [0x00c9] = "\xa5\xc9",
  [0x00ca] = "\xa5\xca", [0x00cb] = "\xa5\xcb", [0x00cc] = "\xa5\xcc",
  [0x00cd] = "\xa5\xcd", [0x00ce] = "\xa5\xce", [0x00cf] = "\xa5\xcf",
  [0x00d0] = "\xa5\xd0", [0x00d1] = "\xa5\xd1", [0x00d2] = "\xa5\xd2",
  [0x00d3] = "\xa5\xd3", [0x00d4] = "\xa5\xd4", [0x00d5] = "\xa5\xd5",
  [0x00d6] = "\xa5\xd6", [0x00d7] = "\xa5\xd7", [0x00d8] = "\xa5\xd8",
  [0x00d9] = "\xa5\xd9", [0x00da] = "\xa5\xda", [0x00db] = "\xa5\xdb",
  [0x00dc] = "\xa5\xdc", [0x00dd] = "\xa5\xdd", [0x00de] = "\xa5\xde",
  [0x00df] = "\xa5\xdf", [0x00e0] = "\xa5\xe0", [0x00e1] = "\xa5\xe1",
  [0x00e2] = "\xa5\xe2", [0x00e3] = "\xa5\xe3", [0x00e4] = "\xa5\xe4",
  [0x00e5] = "\xa5\xe5", [0x00e6] = "\xa5\xe6", [0x00e7] = "\xa5\xe7",
  [0x00e8] = "\xa5\xe8", [0x00e9] = "\xa5\xe9", [0x00ea] = "\xa5\xea",
  [0x00eb] = "\xa5\xeb", [0x00ec] = "\xa5\xec", [0x00ed] = "\xa5\xed",
  [0x00ee] = "\xa5\xee", [0x00ef] = "\xa5\xef", [0x00f0] = "\xa5\xf0",
  [0x00f1] = "\xa5\xf1", [0x00f2] = "\xa5\xf2", [0x00f3] = "\xa5\xf3",
  [0x00f4] = "\xa5\xf4", [0x00f5] = "\xa5\xf5", [0x00f6] = "\xa5\xf6",
  [0x00fc] = "\xa9\x60", [0x00fd] = "\xa9\x63", [0x00fe] = "\xa9\x64",
  [0x0105] = "\xa8\xc5", [0x0106] = "\xa8\xc6", [0x0107] = "\xa8\xc7",
  [0x0108] = "\xa8\xc8", [0x0109] = "\xa8\xc9", [0x010a] = "\xa8\xca",
  [0x010b] = "\xa8\xcb", [0x010c] = "\xa8\xcc", [0x010d] = "\xa8\xcd",
  [0x010e] = "\xa8\xce", [0x010f] = "\xa8\xcf", [0x0110] = "\xa8\xd0",
  [0x0111] = "\xa8\xd1", [0x0112] = "\xa8\xd2", [0x0113] = "\xa8\xd3",
  [0x0114] = "\xa8\xd4", [0x0115] = "\xa8\xd5", [0x0116] = "\xa8\xd6",
  [0x0117] = "\xa8\xd7", [0x0118] = "\xa8\xd8", [0x0119] = "\xa8\xd9",
  [0x011a] = "\xa8\xda", [0x011b] = "\xa8\xdb", [0x011c] = "\xa8\xdc",
  [0x011d] = "\xa8\xdd", [0x011e] = "\xa8\xde", [0x011f] = "\xa8\xdf",
  [0x0120] = "\xa8\xe0", [0x0121] = "\xa8\xe1", [0x0122] = "\xa8\xe2",
  [0x0123] = "\xa8\xe3", [0x0124] = "\xa8\xe4", [0x0125] = "\xa8\xe5",
  [0x0126] = "\xa8\xe6", [0x0127] = "\xa8\xe7", [0x0128] = "\xa8\xe8",
  [0x0129] = "\xa8\xe9", [0x0220] = "\xa2\xe5", [0x0221] = "\xa2\xe6",
  [0x0222] = "\xa2\xe7", [0x0223] = "\xa2\xe8", [0x0224] = "\xa2\xe9",
  [0x0225] = "\xa2\xea", [0x0226] = "\xa2\xeb", [0x0227] = "\xa2\xec",
  [0x0228] = "\xa2\xed", [0x0229] = "\xa2\xee", [0x0231] = "\xa9\x5a",
  [0x02a3] = "\xa9\x49", [0x038e] = "\xa9\x4a", [0x038f] = "\xa9\x4b",
  [0x039c] = "\xa9\x4c", [0x039d] = "\xa9\x4d", [0x039e] = "\xa9\x4e",
  [0x03a1] = "\xa9\x4f", [0x03c4] = "\xa9\x50", [0x03ce] = "\xa9\x51",
  [0x03d1] = "\xa9\x52", [0x03d2] = "\xa9\x53", [0x03d5] = "\xa9\x54",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0x[4-9]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0x4e00,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab8[][2] =
{
  [0x0000] = "\xd2\xbb", [0x0001] = "\xb6\xa1", [0x0002] = "\x81\x40",
  [0x0003] = "\xc6\xdf", [0x0004] = "\x81\x41", [0x0005] = "\x81\x42",
  [0x0006] = "\x81\x43", [0x0007] = "\xcd\xf2", [0x0008] = "\xd5\xc9",
  [0x0009] = "\xc8\xfd", [0x000a] = "\xc9\xcf", [0x000b] = "\xcf\xc2",
  [0x000c] = "\xd8\xa2", [0x000d] = "\xb2\xbb", [0x000e] = "\xd3\xeb",
  [0x000f] = "\x81\x44", [0x0010] = "\xd8\xa4", [0x0011] = "\xb3\xf3",
  [0x0012] = "\x81\x45", [0x0013] = "\xd7\xa8", [0x0014] = "\xc7\xd2",
  [0x0015] = "\xd8\xa7", [0x0016] = "\xca\xc0", [0x0017] = "\x81\x46",
  [0x0018] = "\xc7\xf0", [0x0019] = "\xb1\xfb", [0x001a] = "\xd2\xb5",
  [0x001b] = "\xb4\xd4", [0x001c] = "\xb6\xab", [0x001d] = "\xcb\xbf",
  [0x001e] = "\xd8\xa9", [0x001f] = "\x81\x47", [0x0020] = "\x81\x48",
  [0x0021] = "\x81\x49", [0x0022] = "\xb6\xaa", [0x0023] = "\x81\x4a",
  [0x0024] = "\xc1\xbd", [0x0025] = "\xd1\xcf", [0x0026] = "\x81\x4b",
  [0x0027] = "\xc9\xa5", [0x0028] = "\xd8\xad", [0x0029] = "\x81\x4c",
  [0x002a] = "\xb8\xf6", [0x002b] = "\xd1\xbe", [0x002c] = "\xe3\xdc",
  [0x002d] = "\xd6\xd0", [0x002e] = "\x81\x4d", [0x002f] = "\x81\x4e",
  [0x0030] = "\xb7\xe1", [0x0031] = "\x81\x4f", [0x0032] = "\xb4\xae",
  [0x0033] = "\x81\x50", [0x0034] = "\xc1\xd9", [0x0035] = "\x81\x51",
  [0x0036] = "\xd8\xbc", [0x0037] = "\x81\x52", [0x0038] = "\xcd\xe8",
  [0x0039] = "\xb5\xa4", [0x003a] = "\xce\xaa", [0x003b] = "\xd6\xf7",
  [0x003c] = "\x81\x53", [0x003d] = "\xc0\xf6", [0x003e] = "\xbe\xd9",
  [0x003f] = "\xd8\xaf", [0x0040] = "\x81\x54", [0x0041] = "\x81\x55",
  [0x0042] = "\x81\x56", [0x0043] = "\xc4\xcb", [0x0044] = "\x81\x57",
  [0x0045] = "\xbe\xc3", [0x0046] = "\x81\x58", [0x0047] = "\xd8\xb1",
  [0x0048] = "\xc3\xb4", [0x0049] = "\xd2\xe5", [0x004a] = "\x81\x59",
  [0x004b] = "\xd6\xae", [0x004c] = "\xce\xda", [0x004d] = "\xd5\xa7",
  [0x004e] = "\xba\xf5", [0x004f] = "\xb7\xa6", [0x0050] = "\xc0\xd6",
  [0x0051] = "\x81\x5a", [0x0052] = "\xc6\xb9", [0x0053] = "\xc5\xd2",
  [0x0054] = "\xc7\xc7", [0x0055] = "\x81\x5b", [0x0056] = "\xb9\xd4",
  [0x0057] = "\x81\x5c", [0x0058] = "\xb3\xcb", [0x0059] = "\xd2\xd2",
  [0x005a] = "\x81\x5d", [0x005b] = "\x81\x5e", [0x005c] = "\xd8\xbf",
  [0x005d] = "\xbe\xc5", [0x005e] = "\xc6\xf2", [0x005f] = "\xd2\xb2",
  [0x0060] = "\xcf\xb0", [0x0061] = "\xcf\xe7", [0x0062] = "\x81\x5f",
  [0x0063] = "\x81\x60", [0x0064] = "\x81\x61", [0x0065] = "\x81\x62",
  [0x0066] = "\xca\xe9", [0x0067] = "\x81\x63", [0x0068] = "\x81\x64",
  [0x0069] = "\xd8\xc0", [0x006a] = "\x81\x65", [0x006b] = "\x81\x66",
  [0x006c] = "\x81\x67", [0x006d] = "\x81\x68", [0x006e] = "\x81\x69",
  [0x006f] = "\x81\x6a", [0x0070] = "\xc2\xf2", [0x0071] = "\xc2\xd2",
  [0x0072] = "\x81\x6b", [0x0073] = "\xc8\xe9", [0x0074] = "\x81\x6c",
  [0x0075] = "\x81\x6d", [0x0076] = "\x81\x6e", [0x0077] = "\x81\x6f",
  [0x0078] = "\x81\x70", [0x0079] = "\x81\x71", [0x007a] = "\x81\x72",
  [0x007b] = "\x81\x73", [0x007c] = "\x81\x74", [0x007d] = "\x81\x75",
  [0x007e] = "\xc7\xac", [0x007f] = "\x81\x76", [0x0080] = "\x81\x77",
  [0x0081] = "\x81\x78", [0x0082] = "\x81\x79", [0x0083] = "\x81\x7a",
  [0x0084] = "\x81\x7b", [0x0085] = "\x81\x7c", [0x0086] = "\xc1\xcb",
  [0x0087] = "\x81\x7d", [0x0088] = "\xd3\xe8", [0x0089] = "\xd5\xf9",
  [0x008a] = "\x81\x7e", [0x008b] = "\xca\xc2", [0x008c] = "\xb6\xfe",
  [0x008d] = "\xd8\xa1", [0x008e] = "\xd3\xda", [0x008f] = "\xbf\xf7",
  [0x0090] = "\x81\x80", [0x0091] = "\xd4\xc6", [0x0092] = "\xbb\xa5",
  [0x0093] = "\xd8\xc1", [0x0094] = "\xce\xe5", [0x0095] = "\xbe\xae",
  [0x0096] = "\x81\x81", [0x0097] = "\x81\x82", [0x0098] = "\xd8\xa8",
  [0x0099] = "\x81\x83", [0x009a] = "\xd1\xc7", [0x009b] = "\xd0\xa9",
  [0x009c] = "\x81\x84", [0x009d] = "\x81\x85", [0x009e] = "\x81\x86",
  [0x009f] = "\xd8\xbd", [0x00a0] = "\xd9\xef", [0x00a1] = "\xcd\xf6",
  [0x00a2] = "\xbf\xba", [0x00a3] = "\x81\x87", [0x00a4] = "\xbd\xbb",
  [0x00a5] = "\xba\xa5", [0x00a6] = "\xd2\xe0", [0x00a7] = "\xb2\xfa",
  [0x00a8] = "\xba\xe0", [0x00a9] = "\xc4\xb6", [0x00aa] = "\x81\x88",
  [0x00ab] = "\xcf\xed", [0x00ac] = "\xbe\xa9", [0x00ad] = "\xcd\xa4",
  [0x00ae] = "\xc1\xc1", [0x00af] = "\x81\x89", [0x00b0] = "\x81\x8a",
  [0x00b1] = "\x81\x8b", [0x00b2] = "\xc7\xd7", [0x00b3] = "\xd9\xf1",
  [0x00b4] = "\x81\x8c", [0x00b5] = "\xd9\xf4", [0x00b6] = "\x81\x8d",
  [0x00b7] = "\x81\x8e", [0x00b8] = "\x81\x8f", [0x00b9] = "\x81\x90",
  [0x00ba] = "\xc8\xcb", [0x00bb] = "\xd8\xe9", [0x00bc] = "\x81\x91",
  [0x00bd] = "\x81\x92", [0x00be] = "\x81\x93", [0x00bf] = "\xd2\xda",
  [0x00c0] = "\xca\xb2", [0x00c1] = "\xc8\xca", [0x00c2] = "\xd8\xec",
  [0x00c3] = "\xd8\xea", [0x00c4] = "\xd8\xc6", [0x00c5] = "\xbd\xf6",
  [0x00c6] = "\xc6\xcd", [0x00c7] = "\xb3\xf0", [0x00c8] = "\x81\x94",
  [0x00c9] = "\xd8\xeb", [0x00ca] = "\xbd\xf1", [0x00cb] = "\xbd\xe9",
  [0x00cc] = "\x81\x95", [0x00cd] = "\xc8\xd4", [0x00ce] = "\xb4\xd3",
  [0x00cf] = "\x81\x96", [0x00d0] = "\x81\x97", [0x00d1] = "\xc2\xd8",
  [0x00d2] = "\x81\x98", [0x00d3] = "\xb2\xd6", [0x00d4] = "\xd7\xd0",
  [0x00d5] = "\xca\xcb", [0x00d6] = "\xcb\xfb", [0x00d7] = "\xd5\xcc",
  [0x00d8] = "\xb8\xb6", [0x00d9] = "\xcf\xc9", [0x00da] = "\x81\x99",
  [0x00db] = "\x81\x9a", [0x00dc] = "\x81\x9b", [0x00dd] = "\xd9\xda",
  [0x00de] = "\xd8\xf0", [0x00df] = "\xc7\xaa", [0x00e0] = "\x81\x9c",
  [0x00e1] = "\xd8\xee", [0x00e2] = "\x81\x9d", [0x00e3] = "\xb4\xfa",
  [0x00e4] = "\xc1\xee", [0x00e5] = "\xd2\xd4", [0x00e6] = "\x81\x9e",
  [0x00e7] = "\x81\x9f", [0x00e8] = "\xd8\xed", [0x00e9] = "\x81\xa0",
  [0x00ea] = "\xd2\xc7", [0x00eb] = "\xd8\xef", [0x00ec] = "\xc3\xc7",
  [0x00ed] = "\x81\xa1", [0x00ee] = "\x81\xa2", [0x00ef] = "\x81\xa3",
  [0x00f0] = "\xd1\xf6", [0x00f1] = "\x81\xa4", [0x00f2] = "\xd6\xd9",
  [0x00f3] = "\xd8\xf2", [0x00f4] = "\x81\xa5", [0x00f5] = "\xd8\xf5",
  [0x00f6] = "\xbc\xfe", [0x00f7] = "\xbc\xdb", [0x00f8] = "\x81\xa6",
  [0x00f9] = "\x81\xa7", [0x00fa] = "\x81\xa8", [0x00fb] = "\xc8\xce",
  [0x00fc] = "\x81\xa9", [0x00fd] = "\xb7\xdd", [0x00fe] = "\x81\xaa",
  [0x00ff] = "\xb7\xc2", [0x0100] = "\x81\xab", [0x0101] = "\xc6\xf3",
  [0x0102] = "\x81\xac", [0x0103] = "\x81\xad", [0x0104] = "\x81\xae",
  [0x0105] = "\x81\xaf", [0x0106] = "\x81\xb0", [0x0107] = "\x81\xb1",
  [0x0108] = "\x81\xb2", [0x0109] = "\xd8\xf8", [0x010a] = "\xd2\xc1",
  [0x010b] = "\x81\xb3", [0x010c] = "\x81\xb4", [0x010d] = "\xce\xe9",
  [0x010e] = "\xbc\xbf", [0x010f] = "\xb7\xfc", [0x0110] = "\xb7\xa5",
  [0x0111] = "\xd0\xdd", [0x0112] = "\x81\xb5", [0x0113] = "\x81\xb6",
  [0x0114] = "\x81\xb7", [0x0115] = "\x81\xb8", [0x0116] = "\x81\xb9",
  [0x0117] = "\xd6\xda", [0x0118] = "\xd3\xc5", [0x0119] = "\xbb\xef",
  [0x011a] = "\xbb\xe1", [0x011b] = "\xd8\xf1", [0x011c] = "\x81\xba",
  [0x011d] = "\x81\xbb", [0x011e] = "\xc9\xa1", [0x011f] = "\xce\xb0",
  [0x0120] = "\xb4\xab", [0x0121] = "\x81\xbc", [0x0122] = "\xd8\xf3",
  [0x0123] = "\x81\xbd", [0x0124] = "\xc9\xcb", [0x0125] = "\xd8\xf6",
  [0x0126] = "\xc2\xd7", [0x0127] = "\xd8\xf7", [0x0128] = "\x81\xbe",
  [0x0129] = "\x81\xbf", [0x012a] = "\xce\xb1", [0x012b] = "\xd8\xf9",
  [0x012c] = "\x81\xc0", [0x012d] = "\x81\xc1", [0x012e] = "\x81\xc2",
  [0x012f] = "\xb2\xae", [0x0130] = "\xb9\xc0", [0x0131] = "\x81\xc3",
  [0x0132] = "\xd9\xa3", [0x0133] = "\x81\xc4", [0x0134] = "\xb0\xe9",
  [0x0135] = "\x81\xc5", [0x0136] = "\xc1\xe6", [0x0137] = "\x81\xc6",
  [0x0138] = "\xc9\xec", [0x0139] = "\x81\xc7", [0x013a] = "\xcb\xc5",
  [0x013b] = "\x81\xc8", [0x013c] = "\xcb\xc6", [0x013d] = "\xd9\xa4",
  [0x013e] = "\x81\xc9", [0x013f] = "\x81\xca", [0x0140] = "\x81\xcb",
  [0x0141] = "\x81\xcc", [0x0142] = "\x81\xcd", [0x0143] = "\xb5\xe8",
  [0x0144] = "\x81\xce", [0x0145] = "\x81\xcf", [0x0146] = "\xb5\xab",
  [0x0147] = "\x81\xd0", [0x0148] = "\x81\xd1", [0x0149] = "\x81\xd2",
  [0x014a] = "\x81\xd3", [0x014b] = "\x81\xd4", [0x014c] = "\x81\xd5",
  [0x014d] = "\xce\xbb", [0x014e] = "\xb5\xcd", [0x014f] = "\xd7\xa1",
  [0x0150] = "\xd7\xf4", [0x0151] = "\xd3\xd3", [0x0152] = "\x81\xd6",
  [0x0153] = "\xcc\xe5", [0x0154] = "\x81\xd7", [0x0155] = "\xba\xce",
  [0x0156] = "\x81\xd8", [0x0157] = "\xd9\xa2", [0x0158] = "\xd9\xdc",
  [0x0159] = "\xd3\xe0", [0x015a] = "\xd8\xfd", [0x015b] = "\xb7\xf0",
  [0x015c] = "\xd7\xf7", [0x015d] = "\xd8\xfe", [0x015e] = "\xd8\xfa",
  [0x015f] = "\xd9\xa1", [0x0160] = "\xc4\xe3", [0x0161] = "\x81\xd9",
  [0x0162] = "\x81\xda", [0x0163] = "\xd3\xb6", [0x0164] = "\xd8\xf4",
  [0x0165] = "\xd9\xdd", [0x0166] = "\x81\xdb", [0x0167] = "\xd8\xfb",
  [0x0168] = "\x81\xdc", [0x0169] = "\xc5\xe5", [0x016a] = "\x81\xdd",
  [0x016b] = "\x81\xde", [0x016c] = "\xc0\xd0", [0x016d] = "\x81\xdf",
  [0x016e] = "\x81\xe0", [0x016f] = "\xd1\xf0", [0x0170] = "\xb0\xdb",
  [0x0171] = "\x81\xe1", [0x0172] = "\x81\xe2", [0x0173] = "\xbc\xd1",
  [0x0174] = "\xd9\xa6", [0x0175] = "\x81\xe3", [0x0176] = "\xd9\xa5",
  [0x0177] = "\x81\xe4", [0x0178] = "\x81\xe5", [0x0179] = "\x81\xe6",
  [0x017a] = "\x81\xe7", [0x017b] = "\xd9\xac", [0x017c] = "\xd9\xae",
  [0x017d] = "\x81\xe8", [0x017e] = "\xd9\xab", [0x017f] = "\xca\xb9",
  [0x0180] = "\x81\xe9", [0x0181] = "\x81\xea", [0x0182] = "\x81\xeb",
  [0x0183] = "\xd9\xa9", [0x0184] = "\xd6\xb6", [0x0185] = "\x81\xec",
  [0x0186] = "\x81\xed", [0x0187] = "\x81\xee", [0x0188] = "\xb3\xde",
  [0x0189] = "\xd9\xa8", [0x018a] = "\x81\xef", [0x018b] = "\xc0\xfd",
  [0x018c] = "\x81\xf0", [0x018d] = "\xca\xcc", [0x018e] = "\x81\xf1",
  [0x018f] = "\xd9\xaa", [0x0190] = "\x81\xf2", [0x0191] = "\xd9\xa7",
  [0x0192] = "\x81\xf3", [0x0193] = "\x81\xf4", [0x0194] = "\xd9\xb0",
  [0x0195] = "\x81\xf5", [0x0196] = "\x81\xf6", [0x0197] = "\xb6\xb1",
  [0x0198] = "\x81\xf7", [0x0199] = "\x81\xf8", [0x019a] = "\x81\xf9",
  [0x019b] = "\xb9\xa9", [0x019c] = "\x81\xfa", [0x019d] = "\xd2\xc0",
  [0x019e] = "\x81\xfb", [0x019f] = "\x81\xfc", [0x01a0] = "\xcf\xc0",
  [0x01a1] = "\x81\xfd", [0x01a2] = "\x81\xfe", [0x01a3] = "\xc2\xc2",
  [0x01a4] = "\x82\x40", [0x01a5] = "\xbd\xc4", [0x01a6] = "\xd5\xec",
  [0x01a7] = "\xb2\xe0", [0x01a8] = "\xc7\xc8", [0x01a9] = "\xbf\xeb",
  [0x01aa] = "\xd9\xad", [0x01ab] = "\x82\x41", [0x01ac] = "\xd9\xaf",
  [0x01ad] = "\x82\x42", [0x01ae] = "\xce\xea", [0x01af] = "\xba\xee",
  [0x01b0] = "\x82\x43", [0x01b1] = "\x82\x44", [0x01b2] = "\x82\x45",
  [0x01b3] = "\x82\x46", [0x01b4] = "\x82\x47", [0x01b5] = "\xc7\xd6",
  [0x01b6] = "\x82\x48", [0x01b7] = "\x82\x49", [0x01b8] = "\x82\x4a",
  [0x01b9] = "\x82\x4b", [0x01ba] = "\x82\x4c", [0x01bb] = "\x82\x4d",
  [0x01bc] = "\x82\x4e", [0x01bd] = "\x82\x4f", [0x01be] = "\x82\x50",
  [0x01bf] = "\xb1\xe3", [0x01c0] = "\x82\x51", [0x01c1] = "\x82\x52",
  [0x01c2] = "\x82\x53", [0x01c3] = "\xb4\xd9", [0x01c4] = "\xb6\xed",
  [0x01c5] = "\xd9\xb4", [0x01c6] = "\x82\x54", [0x01c7] = "\x82\x55",
  [0x01c8] = "\x82\x56", [0x01c9] = "\x82\x57", [0x01ca] = "\xbf\xa1",
  [0x01cb] = "\x82\x58", [0x01cc] = "\x82\x59", [0x01cd] = "\x82\x5a",
  [0x01ce] = "\xd9\xde", [0x01cf] = "\xc7\xce", [0x01d0] = "\xc0\xfe",
  [0x01d1] = "\xd9\xb8", [0x01d2] = "\x82\x5b", [0x01d3] = "\x82\x5c",
  [0x01d4] = "\x82\x5d", [0x01d5] = "\x82\x5e", [0x01d6] = "\x82\x5f",
  [0x01d7] = "\xcb\xd7", [0x01d8] = "\xb7\xfd", [0x01d9] = "\x82\x60",
  [0x01da] = "\xd9\xb5", [0x01db] = "\x82\x61", [0x01dc] = "\xd9\xb7",
  [0x01dd] = "\xb1\xa3", [0x01de] = "\xd3\xe1", [0x01df] = "\xd9\xb9",
  [0x01e0] = "\x82\x62", [0x01e1] = "\xd0\xc5", [0x01e2] = "\x82\x63",
  [0x01e3] = "\xd9\xb6", [0x01e4] = "\x82\x64", [0x01e5] = "\x82\x65",
  [0x01e6] = "\xd9\xb1", [0x01e7] = "\x82\x66", [0x01e8] = "\xd9\xb2",
  [0x01e9] = "\xc1\xa9", [0x01ea] = "\xd9\xb3", [0x01eb] = "\x82\x67",
  [0x01ec] = "\x82\x68", [0x01ed] = "\xbc\xf3", [0x01ee] = "\xd0\xde",
  [0x01ef] = "\xb8\xa9", [0x01f0] = "\x82\x69", [0x01f1] = "\xbe\xe3",
  [0x01f2] = "\x82\x6a", [0x01f3] = "\xd9\xbd", [0x01f4] = "\x82\x6b",
  [0x01f5] = "\x82\x6c", [0x01f6] = "\x82\x6d", [0x01f7] = "\x82\x6e",
  [0x01f8] = "\xd9\xba", [0x01f9] = "\x82\x6f", [0x01fa] = "\xb0\xb3",
  [0x01fb] = "\x82\x70", [0x01fc] = "\x82\x71", [0x01fd] = "\x82\x72",
  [0x01fe] = "\xd9\xc2", [0x01ff] = "\x82\x73", [0x0200] = "\x82\x74",
  [0x0201] = "\x82\x75", [0x0202] = "\x82\x76", [0x0203] = "\x82\x77",
  [0x0204] = "\x82\x78", [0x0205] = "\x82\x79", [0x0206] = "\x82\x7a",
  [0x0207] = "\x82\x7b", [0x0208] = "\x82\x7c", [0x0209] = "\x82\x7d",
  [0x020a] = "\x82\x7e", [0x020b] = "\x82\x80", [0x020c] = "\xd9\xc4",
  [0x020d] = "\xb1\xb6", [0x020e] = "\x82\x81", [0x020f] = "\xd9\xbf",
  [0x0210] = "\x82\x82", [0x0211] = "\x82\x83", [0x0212] = "\xb5\xb9",
  [0x0213] = "\x82\x84", [0x0214] = "\xbe\xf3", [0x0215] = "\x82\x85",
  [0x0216] = "\x82\x86", [0x0217] = "\x82\x87", [0x0218] = "\xcc\xc8",
  [0x0219] = "\xba\xf2", [0x021a] = "\xd2\xd0", [0x021b] = "\x82\x88",
  [0x021c] = "\xd9\xc3", [0x021d] = "\x82\x89", [0x021e] = "\x82\x8a",
  [0x021f] = "\xbd\xe8", [0x0220] = "\x82\x8b", [0x0221] = "\xb3\xab",
  [0x0222] = "\x82\x8c", [0x0223] = "\x82\x8d", [0x0224] = "\x82\x8e",
  [0x0225] = "\xd9\xc5", [0x0226] = "\xbe\xeb", [0x0227] = "\x82\x8f",
  [0x0228] = "\xd9\xc6", [0x0229] = "\xd9\xbb", [0x022a] = "\xc4\xdf",
  [0x022b] = "\x82\x90", [0x022c] = "\xd9\xbe", [0x022d] = "\xd9\xc1",
  [0x022e] = "\xd9\xc0", [0x022f] = "\x82\x91", [0x0230] = "\x82\x92",
  [0x0231] = "\x82\x93", [0x0232] = "\x82\x94", [0x0233] = "\x82\x95",
  [0x0234] = "\x82\x96", [0x0235] = "\x82\x97", [0x0236] = "\x82\x98",
  [0x0237] = "\x82\x99", [0x0238] = "\x82\x9a", [0x0239] = "\x82\x9b",
  [0x023a] = "\xd5\xae", [0x023b] = "\x82\x9c", [0x023c] = "\xd6\xb5",
  [0x023d] = "\x82\x9d", [0x023e] = "\xc7\xe3", [0x023f] = "\x82\x9e",
  [0x0240] = "\x82\x9f", [0x0241] = "\x82\xa0", [0x0242] = "\x82\xa1",
  [0x0243] = "\xd9\xc8", [0x0244] = "\x82\xa2", [0x0245] = "\x82\xa3",
  [0x0246] = "\x82\xa4", [0x0247] = "\xbc\xd9", [0x0248] = "\xd9\xca",
  [0x0249] = "\x82\xa5", [0x024a] = "\x82\xa6", [0x024b] = "\x82\xa7",
  [0x024c] = "\xd9\xbc", [0x024d] = "\x82\xa8", [0x024e] = "\xd9\xcb",
  [0x024f] = "\xc6\xab", [0x0250] = "\x82\xa9", [0x0251] = "\x82\xaa",
  [0x0252] = "\x82\xab", [0x0253] = "\x82\xac", [0x0254] = "\x82\xad",
  [0x0255] = "\xd9\xc9", [0x0256] = "\x82\xae", [0x0257] = "\x82\xaf",
  [0x0258] = "\x82\xb0", [0x0259] = "\x82\xb1", [0x025a] = "\xd7\xf6",
  [0x025b] = "\x82\xb2", [0x025c] = "\xcd\xa3", [0x025d] = "\x82\xb3",
  [0x025e] = "\x82\xb4", [0x025f] = "\x82\xb5", [0x0260] = "\x82\xb6",
  [0x0261] = "\x82\xb7", [0x0262] = "\x82\xb8", [0x0263] = "\x82\xb9",
  [0x0264] = "\x82\xba", [0x0265] = "\xbd\xa1", [0x0266] = "\x82\xbb",
  [0x0267] = "\x82\xbc", [0x0268] = "\x82\xbd", [0x0269] = "\x82\xbe",
  [0x026a] = "\x82\xbf", [0x026b] = "\x82\xc0", [0x026c] = "\xd9\xcc",
  [0x026d] = "\x82\xc1", [0x026e] = "\x82\xc2", [0x026f] = "\x82\xc3",
  [0x0270] = "\x82\xc4", [0x0271] = "\x82\xc5", [0x0272] = "\x82\xc6",
  [0x0273] = "\x82\xc7", [0x0274] = "\x82\xc8", [0x0275] = "\x82\xc9",
  [0x0276] = "\xc5\xbc", [0x0277] = "\xcd\xb5", [0x0278] = "\x82\xca",
  [0x0279] = "\x82\xcb", [0x027a] = "\x82\xcc", [0x027b] = "\xd9\xcd",
  [0x027c] = "\x82\xcd", [0x027d] = "\x82\xce", [0x027e] = "\xd9\xc7",
  [0x027f] = "\xb3\xa5", [0x0280] = "\xbf\xfe", [0x0281] = "\x82\xcf",
  [0x0282] = "\x82\xd0", [0x0283] = "\x82\xd1", [0x0284] = "\x82\xd2",
  [0x0285] = "\xb8\xb5", [0x0286] = "\x82\xd3", [0x0287] = "\x82\xd4",
  [0x0288] = "\xc0\xfc", [0x0289] = "\x82\xd5", [0x028a] = "\x82\xd6",
  [0x028b] = "\x82\xd7", [0x028c] = "\x82\xd8", [0x028d] = "\xb0\xf8",
  [0x028e] = "\x82\xd9", [0x028f] = "\x82\xda", [0x0290] = "\x82\xdb",
  [0x0291] = "\x82\xdc", [0x0292] = "\x82\xdd", [0x0293] = "\x82\xde",
  [0x0294] = "\x82\xdf", [0x0295] = "\x82\xe0", [0x0296] = "\x82\xe1",
  [0x0297] = "\x82\xe2", [0x0298] = "\x82\xe3", [0x0299] = "\x82\xe4",
  [0x029a] = "\x82\xe5", [0x029b] = "\x82\xe6", [0x029c] = "\x82\xe7",
  [0x029d] = "\x82\xe8", [0x029e] = "\x82\xe9", [0x029f] = "\x82\xea",
  [0x02a0] = "\x82\xeb", [0x02a1] = "\x82\xec", [0x02a2] = "\x82\xed",
  [0x02a3] = "\xb4\xf6", [0x02a4] = "\x82\xee", [0x02a5] = "\xd9\xce",
  [0x02a6] = "\x82\xef", [0x02a7] = "\xd9\xcf", [0x02a8] = "\xb4\xa2",
  [0x02a9] = "\xd9\xd0", [0x02aa] = "\x82\xf0", [0x02ab] = "\x82\xf1",
  [0x02ac] = "\xb4\xdf", [0x02ad] = "\x82\xf2", [0x02ae] = "\x82\xf3",
  [0x02af] = "\x82\xf4", [0x02b0] = "\x82\xf5", [0x02b1] = "\x82\xf6",
  [0x02b2] = "\xb0\xc1", [0x02b3] = "\x82\xf7", [0x02b4] = "\x82\xf8",
  [0x02b5] = "\x82\xf9", [0x02b6] = "\x82\xfa", [0x02b7] = "\x82\xfb",
  [0x02b8] = "\x82\xfc", [0x02b9] = "\x82\xfd", [0x02ba] = "\xd9\xd1",
  [0x02bb] = "\xc9\xb5", [0x02bc] = "\x82\xfe", [0x02bd] = "\x83\x40",
  [0x02be] = "\x83\x41", [0x02bf] = "\x83\x42", [0x02c0] = "\x83\x43",
  [0x02c1] = "\x83\x44", [0x02c2] = "\x83\x45", [0x02c3] = "\x83\x46",
  [0x02c4] = "\x83\x47", [0x02c5] = "\x83\x48", [0x02c6] = "\x83\x49",
  [0x02c7] = "\x83\x4a", [0x02c8] = "\x83\x4b", [0x02c9] = "\x83\x4c",
  [0x02ca] = "\x83\x4d", [0x02cb] = "\x83\x4e", [0x02cc] = "\x83\x4f",
  [0x02cd] = "\x83\x50", [0x02ce] = "\x83\x51", [0x02cf] = "\xcf\xf1",
  [0x02d0] = "\x83\x52", [0x02d1] = "\x83\x53", [0x02d2] = "\x83\x54",
  [0x02d3] = "\x83\x55", [0x02d4] = "\x83\x56", [0x02d5] = "\x83\x57",
  [0x02d6] = "\xd9\xd2", [0x02d7] = "\x83\x58", [0x02d8] = "\x83\x59",
  [0x02d9] = "\x83\x5a", [0x02da] = "\xc1\xc5", [0x02db] = "\x83\x5b",
  [0x02dc] = "\x83\x5c", [0x02dd] = "\x83\x5d", [0x02de] = "\x83\x5e",
  [0x02df] = "\x83\x5f", [0x02e0] = "\x83\x60", [0x02e1] = "\x83\x61",
  [0x02e2] = "\x83\x62", [0x02e3] = "\x83\x63", [0x02e4] = "\x83\x64",
  [0x02e5] = "\x83\x65", [0x02e6] = "\xd9\xd6", [0x02e7] = "\xc9\xae",
  [0x02e8] = "\x83\x66", [0x02e9] = "\x83\x67", [0x02ea] = "\x83\x68",
  [0x02eb] = "\x83\x69", [0x02ec] = "\xd9\xd5", [0x02ed] = "\xd9\xd4",
  [0x02ee] = "\xd9\xd7", [0x02ef] = "\x83\x6a", [0x02f0] = "\x83\x6b",
  [0x02f1] = "\x83\x6c", [0x02f2] = "\x83\x6d", [0x02f3] = "\xcb\xdb",
  [0x02f4] = "\x83\x6e", [0x02f5] = "\xbd\xa9", [0x02f6] = "\x83\x6f",
  [0x02f7] = "\x83\x70", [0x02f8] = "\x83\x71", [0x02f9] = "\x83\x72",
  [0x02fa] = "\x83\x73", [0x02fb] = "\xc6\xa7", [0x02fc] = "\x83\x74",
  [0x02fd] = "\x83\x75", [0x02fe] = "\x83\x76", [0x02ff] = "\x83\x77",
  [0x0300] = "\x83\x78", [0x0301] = "\x83\x79", [0x0302] = "\x83\x7a",
  [0x0303] = "\x83\x7b", [0x0304] = "\x83\x7c", [0x0305] = "\x83\x7d",
  [0x0306] = "\xd9\xd3", [0x0307] = "\xd9\xd8", [0x0308] = "\x83\x7e",
  [0x0309] = "\x83\x80", [0x030a] = "\x83\x81", [0x030b] = "\xd9\xd9",
  [0x030c] = "\x83\x82", [0x030d] = "\x83\x83", [0x030e] = "\x83\x84",
  [0x030f] = "\x83\x85", [0x0310] = "\x83\x86", [0x0311] = "\x83\x87",
  [0x0312] = "\xc8\xe5", [0x0313] = "\x83\x88", [0x0314] = "\x83\x89",
  [0x0315] = "\x83\x8a", [0x0316] = "\x83\x8b", [0x0317] = "\x83\x8c",
  [0x0318] = "\x83\x8d", [0x0319] = "\x83\x8e", [0x031a] = "\x83\x8f",
  [0x031b] = "\x83\x90", [0x031c] = "\x83\x91", [0x031d] = "\x83\x92",
  [0x031e] = "\x83\x93", [0x031f] = "\x83\x94", [0x0320] = "\x83\x95",
  [0x0321] = "\xc0\xdc", [0x0322] = "\x83\x96", [0x0323] = "\x83\x97",
  [0x0324] = "\x83\x98", [0x0325] = "\x83\x99", [0x0326] = "\x83\x9a",
  [0x0327] = "\x83\x9b", [0x0328] = "\x83\x9c", [0x0329] = "\x83\x9d",
  [0x032a] = "\x83\x9e", [0x032b] = "\x83\x9f", [0x032c] = "\x83\xa0",
  [0x032d] = "\x83\xa1", [0x032e] = "\x83\xa2", [0x032f] = "\x83\xa3",
  [0x0330] = "\x83\xa4", [0x0331] = "\x83\xa5", [0x0332] = "\x83\xa6",
  [0x0333] = "\x83\xa7", [0x0334] = "\x83\xa8", [0x0335] = "\x83\xa9",
  [0x0336] = "\x83\xaa", [0x0337] = "\x83\xab", [0x0338] = "\x83\xac",
  [0x0339] = "\x83\xad", [0x033a] = "\x83\xae", [0x033b] = "\x83\xaf",
  [0x033c] = "\x83\xb0", [0x033d] = "\x83\xb1", [0x033e] = "\x83\xb2",
  [0x033f] = "\xb6\xf9", [0x0340] = "\xd8\xa3", [0x0341] = "\xd4\xca",
  [0x0342] = "\x83\xb3", [0x0343] = "\xd4\xaa", [0x0344] = "\xd0\xd6",
  [0x0345] = "\xb3\xe4", [0x0346] = "\xd5\xd7", [0x0347] = "\x83\xb4",
  [0x0348] = "\xcf\xc8", [0x0349] = "\xb9\xe2", [0x034a] = "\x83\xb5",
  [0x034b] = "\xbf\xcb", [0x034c] = "\x83\xb6", [0x034d] = "\xc3\xe2",
  [0x034e] = "\x83\xb7", [0x034f] = "\x83\xb8", [0x0350] = "\x83\xb9",
  [0x0351] = "\xb6\xd2", [0x0352] = "\x83\xba", [0x0353] = "\x83\xbb",
  [0x0354] = "\xcd\xc3", [0x0355] = "\xd9\xee", [0x0356] = "\xd9\xf0",
  [0x0357] = "\x83\xbc", [0x0358] = "\x83\xbd", [0x0359] = "\x83\xbe",
  [0x035a] = "\xb5\xb3", [0x035b] = "\x83\xbf", [0x035c] = "\xb6\xb5",
  [0x035d] = "\x83\xc0", [0x035e] = "\x83\xc1", [0x035f] = "\x83\xc2",
  [0x0360] = "\x83\xc3", [0x0361] = "\x83\xc4", [0x0362] = "\xbe\xa4",
  [0x0363] = "\x83\xc5", [0x0364] = "\x83\xc6", [0x0365] = "\xc8\xeb",
  [0x0366] = "\x83\xc7", [0x0367] = "\x83\xc8", [0x0368] = "\xc8\xab",
  [0x0369] = "\x83\xc9", [0x036a] = "\x83\xca", [0x036b] = "\xb0\xcb",
  [0x036c] = "\xb9\xab", [0x036d] = "\xc1\xf9", [0x036e] = "\xd9\xe2",
  [0x036f] = "\x83\xcb", [0x0370] = "\xc0\xbc", [0x0371] = "\xb9\xb2",
  [0x0372] = "\x83\xcc", [0x0373] = "\xb9\xd8", [0x0374] = "\xd0\xcb",
  [0x0375] = "\xb1\xf8", [0x0376] = "\xc6\xe4", [0x0377] = "\xbe\xdf",
  [0x0378] = "\xb5\xe4", [0x0379] = "\xd7\xc8", [0x037a] = "\x83\xcd",
  [0x037b] = "\xd1\xf8", [0x037c] = "\xbc\xe6", [0x037d] = "\xca\xde",
  [0x037e] = "\x83\xce", [0x037f] = "\x83\xcf", [0x0380] = "\xbc\xbd",
  [0x0381] = "\xd9\xe6", [0x0382] = "\xd8\xe7", [0x0383] = "\x83\xd0",
  [0x0384] = "\x83\xd1", [0x0385] = "\xc4\xda", [0x0386] = "\x83\xd2",
  [0x0387] = "\x83\xd3", [0x0388] = "\xb8\xd4", [0x0389] = "\xc8\xbd",
  [0x038a] = "\x83\xd4", [0x038b] = "\x83\xd5", [0x038c] = "\xb2\xe1",
  [0x038d] = "\xd4\xd9", [0x038e] = "\x83\xd6", [0x038f] = "\x83\xd7",
  [0x0390] = "\x83\xd8", [0x0391] = "\x83\xd9", [0x0392] = "\xc3\xb0",
  [0x0393] = "\x83\xda", [0x0394] = "\x83\xdb", [0x0395] = "\xc3\xe1",
  [0x0396] = "\xda\xa2", [0x0397] = "\xc8\xdf", [0x0398] = "\x83\xdc",
  [0x0399] = "\xd0\xb4", [0x039a] = "\x83\xdd", [0x039b] = "\xbe\xfc",
  [0x039c] = "\xc5\xa9", [0x039d] = "\x83\xde", [0x039e] = "\x83\xdf",
  [0x039f] = "\x83\xe0", [0x03a0] = "\xb9\xda", [0x03a1] = "\x83\xe1",
  [0x03a2] = "\xda\xa3", [0x03a3] = "\x83\xe2", [0x03a4] = "\xd4\xa9",
  [0x03a5] = "\xda\xa4", [0x03a6] = "\x83\xe3", [0x03a7] = "\x83\xe4",
  [0x03a8] = "\x83\xe5", [0x03a9] = "\x83\xe6", [0x03aa] = "\x83\xe7",
  [0x03ab] = "\xd9\xfb", [0x03ac] = "\xb6\xac", [0x03ad] = "\x83\xe8",
  [0x03ae] = "\x83\xe9", [0x03af] = "\xb7\xeb", [0x03b0] = "\xb1\xf9",
  [0x03b1] = "\xd9\xfc", [0x03b2] = "\xb3\xe5", [0x03b3] = "\xbe\xf6",
  [0x03b4] = "\x83\xea", [0x03b5] = "\xbf\xf6", [0x03b6] = "\xd2\xb1",
  [0x03b7] = "\xc0\xe4", [0x03b8] = "\x83\xeb", [0x03b9] = "\x83\xec",
  [0x03ba] = "\x83\xed", [0x03bb] = "\xb6\xb3", [0x03bc] = "\xd9\xfe",
  [0x03bd] = "\xd9\xfd", [0x03be] = "\x83\xee", [0x03bf] = "\x83\xef",
  [0x03c0] = "\xbe\xbb", [0x03c1] = "\x83\xf0", [0x03c2] = "\x83\xf1",
  [0x03c3] = "\x83\xf2", [0x03c4] = "\xc6\xe0", [0x03c5] = "\x83\xf3",
  [0x03c6] = "\xd7\xbc", [0x03c7] = "\xda\xa1", [0x03c8] = "\x83\xf4",
  [0x03c9] = "\xc1\xb9", [0x03ca] = "\x83\xf5", [0x03cb] = "\xb5\xf2",
  [0x03cc] = "\xc1\xe8", [0x03cd] = "\x83\xf6", [0x03ce] = "\x83\xf7",
  [0x03cf] = "\xbc\xf5", [0x03d0] = "\x83\xf8", [0x03d1] = "\xb4\xd5",
  [0x03d2] = "\x83\xf9", [0x03d3] = "\x83\xfa", [0x03d4] = "\x83\xfb",
  [0x03d5] = "\x83\xfc", [0x03d6] = "\x83\xfd", [0x03d7] = "\x83\xfe",
  [0x03d8] = "\x84\x40", [0x03d9] = "\x84\x41", [0x03da] = "\x84\x42",
  [0x03db] = "\xc1\xdd", [0x03dc] = "\x84\x43", [0x03dd] = "\xc4\xfd",
  [0x03de] = "\x84\x44", [0x03df] = "\x84\x45", [0x03e0] = "\xbc\xb8",
  [0x03e1] = "\xb7\xb2", [0x03e2] = "\x84\x46", [0x03e3] = "\x84\x47",
  [0x03e4] = "\xb7\xef", [0x03e5] = "\x84\x48", [0x03e6] = "\x84\x49",
  [0x03e7] = "\x84\x4a", [0x03e8] = "\x84\x4b", [0x03e9] = "\x84\x4c",
  [0x03ea] = "\x84\x4d", [0x03eb] = "\xd9\xec", [0x03ec] = "\x84\x4e",
  [0x03ed] = "\xc6\xbe", [0x03ee] = "\x84\x4f", [0x03ef] = "\xbf\xad",
  [0x03f0] = "\xbb\xcb", [0x03f1] = "\x84\x50", [0x03f2] = "\x84\x51",
  [0x03f3] = "\xb5\xca", [0x03f4] = "\x84\x52", [0x03f5] = "\xdb\xc9",
  [0x03f6] = "\xd0\xd7", [0x03f7] = "\x84\x53", [0x03f8] = "\xcd\xb9",
  [0x03f9] = "\xb0\xbc", [0x03fa] = "\xb3\xf6", [0x03fb] = "\xbb\xf7",
  [0x03fc] = "\xdb\xca", [0x03fd] = "\xba\xaf", [0x03fe] = "\x84\x54",
  [0x03ff] = "\xd4\xe4", [0x0400] = "\xb5\xb6", [0x0401] = "\xb5\xf3",
  [0x0402] = "\xd8\xd6", [0x0403] = "\xc8\xd0", [0x0404] = "\x84\x55",
  [0x0405] = "\x84\x56", [0x0406] = "\xb7\xd6", [0x0407] = "\xc7\xd0",
  [0x0408] = "\xd8\xd7", [0x0409] = "\x84\x57", [0x040a] = "\xbf\xaf",
  [0x040b] = "\x84\x58", [0x040c] = "\x84\x59", [0x040d] = "\xdb\xbb",
  [0x040e] = "\xd8\xd8", [0x040f] = "\x84\x5a", [0x0410] = "\x84\x5b",
  [0x0411] = "\xd0\xcc", [0x0412] = "\xbb\xae", [0x0413] = "\x84\x5c",
  [0x0414] = "\x84\x5d", [0x0415] = "\x84\x5e", [0x0416] = "\xeb\xbe",
  [0x0417] = "\xc1\xd0", [0x0418] = "\xc1\xf5", [0x0419] = "\xd4\xf2",
  [0x041a] = "\xb8\xd5", [0x041b] = "\xb4\xb4", [0x041c] = "\x84\x5f",
  [0x041d] = "\xb3\xf5", [0x041e] = "\x84\x60", [0x041f] = "\x84\x61",
  [0x0420] = "\xc9\xbe", [0x0421] = "\x84\x62", [0x0422] = "\x84\x63",
  [0x0423] = "\x84\x64", [0x0424] = "\xc5\xd0", [0x0425] = "\x84\x65",
  [0x0426] = "\x84\x66", [0x0427] = "\x84\x67", [0x0428] = "\xc5\xd9",
  [0x0429] = "\xc0\xfb", [0x042a] = "\x84\x68", [0x042b] = "\xb1\xf0",
  [0x042c] = "\x84\x69", [0x042d] = "\xd8\xd9", [0x042e] = "\xb9\xce",
  [0x042f] = "\x84\x6a", [0x0430] = "\xb5\xbd", [0x0431] = "\x84\x6b",
  [0x0432] = "\x84\x6c", [0x0433] = "\xd8\xda", [0x0434] = "\x84\x6d",
  [0x0435] = "\x84\x6e", [0x0436] = "\xd6\xc6", [0x0437] = "\xcb\xa2",
  [0x0438] = "\xc8\xaf", [0x0439] = "\xc9\xb2", [0x043a] = "\xb4\xcc",
  [0x043b] = "\xbf\xcc", [0x043c] = "\x84\x6f", [0x043d] = "\xb9\xf4",
  [0x043e] = "\x84\x70", [0x043f] = "\xd8\xdb", [0x0440] = "\xd8\xdc",
  [0x0441] = "\xb6\xe7", [0x0442] = "\xbc\xc1", [0x0443] = "\xcc\xea",
  [0x0444] = "\x84\x71", [0x0445] = "\x84\x72", [0x0446] = "\x84\x73",
  [0x0447] = "\x84\x74", [0x0448] = "\x84\x75", [0x0449] = "\x84\x76",
  [0x044a] = "\xcf\xf7", [0x044b] = "\x84\x77", [0x044c] = "\xd8\xdd",
  [0x044d] = "\xc7\xb0", [0x044e] = "\x84\x78", [0x044f] = "\x84\x79",
  [0x0450] = "\xb9\xd0", [0x0451] = "\xbd\xa3", [0x0452] = "\x84\x7a",
  [0x0453] = "\x84\x7b", [0x0454] = "\xcc\xde", [0x0455] = "\x84\x7c",
  [0x0456] = "\xc6\xca", [0x0457] = "\x84\x7d", [0x0458] = "\x84\x7e",
  [0x0459] = "\x84\x80", [0x045a] = "\x84\x81", [0x045b] = "\x84\x82",
  [0x045c] = "\xd8\xe0", [0x045d] = "\x84\x83", [0x045e] = "\xd8\xde",
  [0x045f] = "\x84\x84", [0x0460] = "\x84\x85", [0x0461] = "\xd8\xdf",
  [0x0462] = "\x84\x86", [0x0463] = "\x84\x87", [0x0464] = "\x84\x88",
  [0x0465] = "\xb0\xfe", [0x0466] = "\x84\x89", [0x0467] = "\xbe\xe7",
  [0x0468] = "\x84\x8a", [0x0469] = "\xca\xa3", [0x046a] = "\xbc\xf4",
  [0x046b] = "\x84\x8b", [0x046c] = "\x84\x8c", [0x046d] = "\x84\x8d",
  [0x046e] = "\x84\x8e", [0x046f] = "\xb8\xb1", [0x0470] = "\x84\x8f",
  [0x0471] = "\x84\x90", [0x0472] = "\xb8\xee", [0x0473] = "\x84\x91",
  [0x0474] = "\x84\x92", [0x0475] = "\x84\x93", [0x0476] = "\x84\x94",
  [0x0477] = "\x84\x95", [0x0478] = "\x84\x96", [0x0479] = "\x84\x97",
  [0x047a] = "\x84\x98", [0x047b] = "\x84\x99", [0x047c] = "\x84\x9a",
  [0x047d] = "\xd8\xe2", [0x047e] = "\x84\x9b", [0x047f] = "\xbd\xcb",
  [0x0480] = "\x84\x9c", [0x0481] = "\xd8\xe4", [0x0482] = "\xd8\xe3",
  [0x0483] = "\x84\x9d", [0x0484] = "\x84\x9e", [0x0485] = "\x84\x9f",
  [0x0486] = "\x84\xa0", [0x0487] = "\x84\xa1", [0x0488] = "\xc5\xfc",
  [0x0489] = "\x84\xa2", [0x048a] = "\x84\xa3", [0x048b] = "\x84\xa4",
  [0x048c] = "\x84\xa5", [0x048d] = "\x84\xa6", [0x048e] = "\x84\xa7",
  [0x048f] = "\x84\xa8", [0x0490] = "\xd8\xe5", [0x0491] = "\x84\xa9",
  [0x0492] = "\x84\xaa", [0x0493] = "\xd8\xe6", [0x0494] = "\x84\xab",
  [0x0495] = "\x84\xac", [0x0496] = "\x84\xad", [0x0497] = "\x84\xae",
  [0x0498] = "\x84\xaf", [0x0499] = "\x84\xb0", [0x049a] = "\x84\xb1",
  [0x049b] = "\xc1\xa6", [0x049c] = "\x84\xb2", [0x049d] = "\xc8\xb0",
  [0x049e] = "\xb0\xec", [0x049f] = "\xb9\xa6", [0x04a0] = "\xbc\xd3",
  [0x04a1] = "\xce\xf1", [0x04a2] = "\xdb\xbd", [0x04a3] = "\xc1\xd3",
  [0x04a4] = "\x84\xb3", [0x04a5] = "\x84\xb4", [0x04a6] = "\x84\xb5",
  [0x04a7] = "\x84\xb6", [0x04a8] = "\xb6\xaf", [0x04a9] = "\xd6\xfa",
  [0x04aa] = "\xc5\xac", [0x04ab] = "\xbd\xd9", [0x04ac] = "\xdb\xbe",
  [0x04ad] = "\xdb\xbf", [0x04ae] = "\x84\xb7", [0x04af] = "\x84\xb8",
  [0x04b0] = "\x84\xb9", [0x04b1] = "\xc0\xf8", [0x04b2] = "\xbe\xa2",
  [0x04b3] = "\xc0\xcd", [0x04b4] = "\x84\xba", [0x04b5] = "\x84\xbb",
  [0x04b6] = "\x84\xbc", [0x04b7] = "\x84\xbd", [0x04b8] = "\x84\xbe",
  [0x04b9] = "\x84\xbf", [0x04ba] = "\x84\xc0", [0x04bb] = "\x84\xc1",
  [0x04bc] = "\x84\xc2", [0x04bd] = "\x84\xc3", [0x04be] = "\xdb\xc0",
  [0x04bf] = "\xca\xc6", [0x04c0] = "\x84\xc4", [0x04c1] = "\x84\xc5",
  [0x04c2] = "\x84\xc6", [0x04c3] = "\xb2\xaa", [0x04c4] = "\x84\xc7",
  [0x04c5] = "\x84\xc8", [0x04c6] = "\x84\xc9", [0x04c7] = "\xd3\xc2",
  [0x04c8] = "\x84\xca", [0x04c9] = "\xc3\xe3", [0x04ca] = "\x84\xcb",
  [0x04cb] = "\xd1\xab", [0x04cc] = "\x84\xcc", [0x04cd] = "\x84\xcd",
  [0x04ce] = "\x84\xce", [0x04cf] = "\x84\xcf", [0x04d0] = "\xdb\xc2",
  [0x04d1] = "\x84\xd0", [0x04d2] = "\xc0\xd5", [0x04d3] = "\x84\xd1",
  [0x04d4] = "\x84\xd2", [0x04d5] = "\x84\xd3", [0x04d6] = "\xdb\xc3",
  [0x04d7] = "\x84\xd4", [0x04d8] = "\xbf\xb1", [0x04d9] = "\x84\xd5",
  [0x04da] = "\x84\xd6", [0x04db] = "\x84\xd7", [0x04dc] = "\x84\xd8",
  [0x04dd] = "\x84\xd9", [0x04de] = "\x84\xda", [0x04df] = "\xc4\xbc",
  [0x04e0] = "\x84\xdb", [0x04e1] = "\x84\xdc", [0x04e2] = "\x84\xdd",
  [0x04e3] = "\x84\xde", [0x04e4] = "\xc7\xda", [0x04e5] = "\x84\xdf",
  [0x04e6] = "\x84\xe0", [0x04e7] = "\x84\xe1", [0x04e8] = "\x84\xe2",
  [0x04e9] = "\x84\xe3", [0x04ea] = "\x84\xe4", [0x04eb] = "\x84\xe5",
  [0x04ec] = "\x84\xe6", [0x04ed] = "\x84\xe7", [0x04ee] = "\x84\xe8",
  [0x04ef] = "\x84\xe9", [0x04f0] = "\xdb\xc4", [0x04f1] = "\x84\xea",
  [0x04f2] = "\x84\xeb", [0x04f3] = "\x84\xec", [0x04f4] = "\x84\xed",
  [0x04f5] = "\x84\xee", [0x04f6] = "\x84\xef", [0x04f7] = "\x84\xf0",
  [0x04f8] = "\x84\xf1", [0x04f9] = "\xd9\xe8", [0x04fa] = "\xc9\xd7",
  [0x04fb] = "\x84\xf2", [0x04fc] = "\x84\xf3", [0x04fd] = "\x84\xf4",
  [0x04fe] = "\xb9\xb4", [0x04ff] = "\xce\xf0", [0x0500] = "\xd4\xc8",
  [0x0501] = "\x84\xf5", [0x0502] = "\x84\xf6", [0x0503] = "\x84\xf7",
  [0x0504] = "\x84\xf8", [0x0505] = "\xb0\xfc", [0x0506] = "\xb4\xd2",
  [0x0507] = "\x84\xf9", [0x0508] = "\xd0\xd9", [0x0509] = "\x84\xfa",
  [0x050a] = "\x84\xfb", [0x050b] = "\x84\xfc", [0x050c] = "\x84\xfd",
  [0x050d] = "\xd9\xe9", [0x050e] = "\x84\xfe", [0x050f] = "\xde\xcb",
  [0x0510] = "\xd9\xeb", [0x0511] = "\x85\x40", [0x0512] = "\x85\x41",
  [0x0513] = "\x85\x42", [0x0514] = "\x85\x43", [0x0515] = "\xd8\xb0",
  [0x0516] = "\xbb\xaf", [0x0517] = "\xb1\xb1", [0x0518] = "\x85\x44",
  [0x0519] = "\xb3\xd7", [0x051a] = "\xd8\xce", [0x051b] = "\x85\x45",
  [0x051c] = "\x85\x46", [0x051d] = "\xd4\xd1", [0x051e] = "\x85\x47",
  [0x051f] = "\x85\x48", [0x0520] = "\xbd\xb3", [0x0521] = "\xbf\xef",
  [0x0522] = "\x85\x49", [0x0523] = "\xcf\xbb", [0x0524] = "\x85\x4a",
  [0x0525] = "\x85\x4b", [0x0526] = "\xd8\xd0", [0x0527] = "\x85\x4c",
  [0x0528] = "\x85\x4d", [0x0529] = "\x85\x4e", [0x052a] = "\xb7\xcb",
  [0x052b] = "\x85\x4f", [0x052c] = "\x85\x50", [0x052d] = "\x85\x51",
  [0x052e] = "\xd8\xd1", [0x052f] = "\x85\x52", [0x0530] = "\x85\x53",
  [0x0531] = "\x85\x54", [0x0532] = "\x85\x55", [0x0533] = "\x85\x56",
  [0x0534] = "\x85\x57", [0x0535] = "\x85\x58", [0x0536] = "\x85\x59",
  [0x0537] = "\x85\x5a", [0x0538] = "\x85\x5b", [0x0539] = "\xc6\xa5",
  [0x053a] = "\xc7\xf8", [0x053b] = "\xd2\xbd", [0x053c] = "\x85\x5c",
  [0x053d] = "\x85\x5d", [0x053e] = "\xd8\xd2", [0x053f] = "\xc4\xe4",
  [0x0540] = "\x85\x5e", [0x0541] = "\xca\xae", [0x0542] = "\x85\x5f",
  [0x0543] = "\xc7\xa7", [0x0544] = "\x85\x60", [0x0545] = "\xd8\xa6",
  [0x0546] = "\x85\x61", [0x0547] = "\xc9\xfd", [0x0548] = "\xce\xe7",
  [0x0549] = "\xbb\xdc", [0x054a] = "\xb0\xeb", [0x054b] = "\x85\x62",
  [0x054c] = "\x85\x63", [0x054d] = "\x85\x64", [0x054e] = "\xbb\xaa",
  [0x054f] = "\xd0\xad", [0x0550] = "\x85\x65", [0x0551] = "\xb1\xb0",
  [0x0552] = "\xd7\xe4", [0x0553] = "\xd7\xbf", [0x0554] = "\x85\x66",
  [0x0555] = "\xb5\xa5", [0x0556] = "\xc2\xf4", [0x0557] = "\xc4\xcf",
  [0x0558] = "\x85\x67", [0x0559] = "\x85\x68", [0x055a] = "\xb2\xa9",
  [0x055b] = "\x85\x69", [0x055c] = "\xb2\xb7", [0x055d] = "\x85\x6a",
  [0x055e] = "\xb1\xe5", [0x055f] = "\xdf\xb2", [0x0560] = "\xd5\xbc",
  [0x0561] = "\xbf\xa8", [0x0562] = "\xc2\xac", [0x0563] = "\xd8\xd5",
  [0x0564] = "\xc2\xb1", [0x0565] = "\x85\x6b", [0x0566] = "\xd8\xd4",
  [0x0567] = "\xce\xd4", [0x0568] = "\x85\x6c", [0x0569] = "\xda\xe0",
  [0x056a] = "\x85\x6d", [0x056b] = "\xce\xc0", [0x056c] = "\x85\x6e",
  [0x056d] = "\x85\x6f", [0x056e] = "\xd8\xb4", [0x056f] = "\xc3\xae",
  [0x0570] = "\xd3\xa1", [0x0571] = "\xce\xa3", [0x0572] = "\x85\x70",
  [0x0573] = "\xbc\xb4", [0x0574] = "\xc8\xb4", [0x0575] = "\xc2\xd1",
  [0x0576] = "\x85\x71", [0x0577] = "\xbe\xed", [0x0578] = "\xd0\xb6",
  [0x0579] = "\x85\x72", [0x057a] = "\xda\xe1", [0x057b] = "\x85\x73",
  [0x057c] = "\x85\x74", [0x057d] = "\x85\x75", [0x057e] = "\x85\x76",
  [0x057f] = "\xc7\xe4", [0x0580] = "\x85\x77", [0x0581] = "\x85\x78",
  [0x0582] = "\xb3\xa7", [0x0583] = "\x85\x79", [0x0584] = "\xb6\xf2",
  [0x0585] = "\xcc\xfc", [0x0586] = "\xc0\xfa", [0x0587] = "\x85\x7a",
  [0x0588] = "\x85\x7b", [0x0589] = "\xc0\xf7", [0x058a] = "\x85\x7c",
  [0x058b] = "\xd1\xb9", [0x058c] = "\xd1\xe1", [0x058d] = "\xd8\xc7",
  [0x058e] = "\x85\x7d", [0x058f] = "\x85\x7e", [0x0590] = "\x85\x80",
  [0x0591] = "\x85\x81", [0x0592] = "\x85\x82", [0x0593] = "\x85\x83",
  [0x0594] = "\x85\x84", [0x0595] = "\xb2\xde", [0x0596] = "\x85\x85",
  [0x0597] = "\x85\x86", [0x0598] = "\xc0\xe5", [0x0599] = "\x85\x87",
  [0x059a] = "\xba\xf1", [0x059b] = "\x85\x88", [0x059c] = "\x85\x89",
  [0x059d] = "\xd8\xc8", [0x059e] = "\x85\x8a", [0x059f] = "\xd4\xad",
  [0x05a0] = "\x85\x8b", [0x05a1] = "\x85\x8c", [0x05a2] = "\xcf\xe1",
  [0x05a3] = "\xd8\xc9", [0x05a4] = "\x85\x8d", [0x05a5] = "\xd8\xca",
  [0x05a6] = "\xcf\xc3", [0x05a7] = "\x85\x8e", [0x05a8] = "\xb3\xf8",
  [0x05a9] = "\xbe\xc7", [0x05aa] = "\x85\x8f", [0x05ab] = "\x85\x90",
  [0x05ac] = "\x85\x91", [0x05ad] = "\x85\x92", [0x05ae] = "\xd8\xcb",
  [0x05af] = "\x85\x93", [0x05b0] = "\x85\x94", [0x05b1] = "\x85\x95",
  [0x05b2] = "\x85\x96", [0x05b3] = "\x85\x97", [0x05b4] = "\x85\x98",
  [0x05b5] = "\x85\x99", [0x05b6] = "\xdb\xcc", [0x05b7] = "\x85\x9a",
  [0x05b8] = "\x85\x9b", [0x05b9] = "\x85\x9c", [0x05ba] = "\x85\x9d",
  [0x05bb] = "\xc8\xa5", [0x05bc] = "\x85\x9e", [0x05bd] = "\x85\x9f",
  [0x05be] = "\x85\xa0", [0x05bf] = "\xcf\xd8", [0x05c0] = "\x85\xa1",
  [0x05c1] = "\xc8\xfe", [0x05c2] = "\xb2\xce", [0x05c3] = "\x85\xa2",
  [0x05c4] = "\x85\xa3", [0x05c5] = "\x85\xa4", [0x05c6] = "\x85\xa5",
  [0x05c7] = "\x85\xa6", [0x05c8] = "\xd3\xd6", [0x05c9] = "\xb2\xe6",
  [0x05ca] = "\xbc\xb0", [0x05cb] = "\xd3\xd1", [0x05cc] = "\xcb\xab",
  [0x05cd] = "\xb7\xb4", [0x05ce] = "\x85\xa7", [0x05cf] = "\x85\xa8",
  [0x05d0] = "\x85\xa9", [0x05d1] = "\xb7\xa2", [0x05d2] = "\x85\xaa",
  [0x05d3] = "\x85\xab", [0x05d4] = "\xca\xe5", [0x05d5] = "\x85\xac",
  [0x05d6] = "\xc8\xa1", [0x05d7] = "\xca\xdc", [0x05d8] = "\xb1\xe4",
  [0x05d9] = "\xd0\xf0", [0x05da] = "\x85\xad", [0x05db] = "\xc5\xd1",
  [0x05dc] = "\x85\xae", [0x05dd] = "\x85\xaf", [0x05de] = "\x85\xb0",
  [0x05df] = "\xdb\xc5", [0x05e0] = "\xb5\xfe", [0x05e1] = "\x85\xb1",
  [0x05e2] = "\x85\xb2", [0x05e3] = "\xbf\xda", [0x05e4] = "\xb9\xc5",
  [0x05e5] = "\xbe\xe4", [0x05e6] = "\xc1\xed", [0x05e7] = "\x85\xb3",
  [0x05e8] = "\xdf\xb6", [0x05e9] = "\xdf\xb5", [0x05ea] = "\xd6\xbb",
  [0x05eb] = "\xbd\xd0", [0x05ec] = "\xd5\xd9", [0x05ed] = "\xb0\xc8",
  [0x05ee] = "\xb6\xa3", [0x05ef] = "\xbf\xc9", [0x05f0] = "\xcc\xa8",
  [0x05f1] = "\xdf\xb3", [0x05f2] = "\xca\xb7", [0x05f3] = "\xd3\xd2",
  [0x05f4] = "\x85\xb4", [0x05f5] = "\xd8\xcf", [0x05f6] = "\xd2\xb6",
  [0x05f7] = "\xba\xc5", [0x05f8] = "\xcb\xbe", [0x05f9] = "\xcc\xbe",
  [0x05fa] = "\x85\xb5", [0x05fb] = "\xdf\xb7", [0x05fc] = "\xb5\xf0",
  [0x05fd] = "\xdf\xb4", [0x05fe] = "\x85\xb6", [0x05ff] = "\x85\xb7",
  [0x0600] = "\x85\xb8", [0x0601] = "\xd3\xf5", [0x0602] = "\x85\xb9",
  [0x0603] = "\xb3\xd4", [0x0604] = "\xb8\xf7", [0x0605] = "\x85\xba",
  [0x0606] = "\xdf\xba", [0x0607] = "\x85\xbb", [0x0608] = "\xba\xcf",
  [0x0609] = "\xbc\xaa", [0x060a] = "\xb5\xf5", [0x060b] = "\x85\xbc",
  [0x060c] = "\xcd\xac", [0x060d] = "\xc3\xfb", [0x060e] = "\xba\xf3",
  [0x060f] = "\xc0\xf4", [0x0610] = "\xcd\xc2", [0x0611] = "\xcf\xf2",
  [0x0612] = "\xdf\xb8", [0x0613] = "\xcf\xc5", [0x0614] = "\x85\xbd",
  [0x0615] = "\xc2\xc0", [0x0616] = "\xdf\xb9", [0x0617] = "\xc2\xf0",
  [0x0618] = "\x85\xbe", [0x0619] = "\x85\xbf", [0x061a] = "\x85\xc0",
  [0x061b] = "\xbe\xfd", [0x061c] = "\x85\xc1", [0x061d] = "\xc1\xdf",
  [0x061e] = "\xcd\xcc", [0x061f] = "\xd2\xf7", [0x0620] = "\xb7\xcd",
  [0x0621] = "\xdf\xc1", [0x0622] = "\x85\xc2", [0x0623] = "\xdf\xc4",
  [0x0624] = "\x85\xc3", [0x0625] = "\x85\xc4", [0x0626] = "\xb7\xf1",
  [0x0627] = "\xb0\xc9", [0x0628] = "\xb6\xd6", [0x0629] = "\xb7\xd4",
  [0x062a] = "\x85\xc5", [0x062b] = "\xba\xac", [0x062c] = "\xcc\xfd",
  [0x062d] = "\xbf\xd4", [0x062e] = "\xcb\xb1", [0x062f] = "\xc6\xf4",
  [0x0630] = "\x85\xc6", [0x0631] = "\xd6\xa8", [0x0632] = "\xdf\xc5",
  [0x0633] = "\x85\xc7", [0x0634] = "\xce\xe2", [0x0635] = "\xb3\xb3",
  [0x0636] = "\x85\xc8", [0x0637] = "\x85\xc9", [0x0638] = "\xce\xfc",
  [0x0639] = "\xb4\xb5", [0x063a] = "\x85\xca", [0x063b] = "\xce\xc7",
  [0x063c] = "\xba\xf0", [0x063d] = "\x85\xcb", [0x063e] = "\xce\xe1",
  [0x063f] = "\x85\xcc", [0x0640] = "\xd1\xbd", [0x0641] = "\x85\xcd",
  [0x0642] = "\x85\xce", [0x0643] = "\xdf\xc0", [0x0644] = "\x85\xcf",
  [0x0645] = "\x85\xd0", [0x0646] = "\xb4\xf4", [0x0647] = "\x85\xd1",
  [0x0648] = "\xb3\xca", [0x0649] = "\x85\xd2", [0x064a] = "\xb8\xe6",
  [0x064b] = "\xdf\xbb", [0x064c] = "\x85\xd3", [0x064d] = "\x85\xd4",
  [0x064e] = "\x85\xd5", [0x064f] = "\x85\xd6", [0x0650] = "\xc4\xc5",
  [0x0651] = "\x85\xd7", [0x0652] = "\xdf\xbc", [0x0653] = "\xdf\xbd",
  [0x0654] = "\xdf\xbe", [0x0655] = "\xc5\xbb", [0x0656] = "\xdf\xbf",
  [0x0657] = "\xdf\xc2", [0x0658] = "\xd4\xb1", [0x0659] = "\xdf\xc3",
  [0x065a] = "\x85\xd8", [0x065b] = "\xc7\xba", [0x065c] = "\xce\xd8",
  [0x065d] = "\x85\xd9", [0x065e] = "\x85\xda", [0x065f] = "\x85\xdb",
  [0x0660] = "\x85\xdc", [0x0661] = "\x85\xdd", [0x0662] = "\xc4\xd8",
  [0x0663] = "\x85\xde", [0x0664] = "\xdf\xca", [0x0665] = "\x85\xdf",
  [0x0666] = "\xdf\xcf", [0x0667] = "\x85\xe0", [0x0668] = "\xd6\xdc",
  [0x0669] = "\x85\xe1", [0x066a] = "\x85\xe2", [0x066b] = "\x85\xe3",
  [0x066c] = "\x85\xe4", [0x066d] = "\x85\xe5", [0x066e] = "\x85\xe6",
  [0x066f] = "\x85\xe7", [0x0670] = "\x85\xe8", [0x0671] = "\xdf\xc9",
  [0x0672] = "\xdf\xda", [0x0673] = "\xce\xb6", [0x0674] = "\x85\xe9",
  [0x0675] = "\xba\xc7", [0x0676] = "\xdf\xce", [0x0677] = "\xdf\xc8",
  [0x0678] = "\xc5\xde", [0x0679] = "\x85\xea", [0x067a] = "\x85\xeb",
  [0x067b] = "\xc9\xeb", [0x067c] = "\xba\xf4", [0x067d] = "\xc3\xfc",
  [0x067e] = "\x85\xec", [0x067f] = "\x85\xed", [0x0680] = "\xbe\xd7",
  [0x0681] = "\x85\xee", [0x0682] = "\xdf\xc6", [0x0683] = "\x85\xef",
  [0x0684] = "\xdf\xcd", [0x0685] = "\x85\xf0", [0x0686] = "\xc5\xd8",
  [0x0687] = "\x85\xf1", [0x0688] = "\x85\xf2", [0x0689] = "\x85\xf3",
  [0x068a] = "\x85\xf4", [0x068b] = "\xd5\xa6", [0x068c] = "\xba\xcd",
  [0x068d] = "\x85\xf5", [0x068e] = "\xbe\xcc", [0x068f] = "\xd3\xbd",
  [0x0690] = "\xb8\xc0", [0x0691] = "\x85\xf6", [0x0692] = "\xd6\xe4",
  [0x0693] = "\x85\xf7", [0x0694] = "\xdf\xc7", [0x0695] = "\xb9\xbe",
  [0x0696] = "\xbf\xa7", [0x0697] = "\x85\xf8", [0x0698] = "\x85\xf9",
  [0x0699] = "\xc1\xfc", [0x069a] = "\xdf\xcb", [0x069b] = "\xdf\xcc",
  [0x069c] = "\x85\xfa", [0x069d] = "\xdf\xd0", [0x069e] = "\x85\xfb",
  [0x069f] = "\x85\xfc", [0x06a0] = "\x85\xfd", [0x06a1] = "\x85\xfe",
  [0x06a2] = "\x86\x40", [0x06a3] = "\xdf\xdb", [0x06a4] = "\xdf\xe5",
  [0x06a5] = "\x86\x41", [0x06a6] = "\xdf\xd7", [0x06a7] = "\xdf\xd6",
  [0x06a8] = "\xd7\xc9", [0x06a9] = "\xdf\xe3", [0x06aa] = "\xdf\xe4",
  [0x06ab] = "\xe5\xeb", [0x06ac] = "\xd2\xa7", [0x06ad] = "\xdf\xd2",
  [0x06ae] = "\x86\x42", [0x06af] = "\xbf\xa9", [0x06b0] = "\x86\x43",
  [0x06b1] = "\xd4\xdb", [0x06b2] = "\x86\x44", [0x06b3] = "\xbf\xc8",
  [0x06b4] = "\xdf\xd4", [0x06b5] = "\x86\x45", [0x06b6] = "\x86\x46",
  [0x06b7] = "\x86\x47", [0x06b8] = "\xcf\xcc", [0x06b9] = "\x86\x48",
  [0x06ba] = "\x86\x49", [0x06bb] = "\xdf\xdd", [0x06bc] = "\x86\x4a",
  [0x06bd] = "\xd1\xca", [0x06be] = "\x86\x4b", [0x06bf] = "\xdf\xde",
  [0x06c0] = "\xb0\xa7", [0x06c1] = "\xc6\xb7", [0x06c2] = "\xdf\xd3",
  [0x06c3] = "\x86\x4c", [0x06c4] = "\xba\xe5", [0x06c5] = "\x86\x4d",
  [0x06c6] = "\xb6\xdf", [0x06c7] = "\xcd\xdb", [0x06c8] = "\xb9\xfe",
  [0x06c9] = "\xd4\xd5", [0x06ca] = "\x86\x4e", [0x06cb] = "\x86\x4f",
  [0x06cc] = "\xdf\xdf", [0x06cd] = "\xcf\xec", [0x06ce] = "\xb0\xa5",
  [0x06cf] = "\xdf\xe7", [0x06d0] = "\xdf\xd1", [0x06d1] = "\xd1\xc6",
  [0x06d2] = "\xdf\xd5", [0x06d3] = "\xdf\xd8", [0x06d4] = "\xdf\xd9",
  [0x06d5] = "\xdf\xdc", [0x06d6] = "\x86\x50", [0x06d7] = "\xbb\xa9",
  [0x06d8] = "\x86\x51", [0x06d9] = "\xdf\xe0", [0x06da] = "\xdf\xe1",
  [0x06db] = "\x86\x52", [0x06dc] = "\xdf\xe2", [0x06dd] = "\xdf\xe6",
  [0x06de] = "\xdf\xe8", [0x06df] = "\xd3\xb4", [0x06e0] = "\x86\x53",
  [0x06e1] = "\x86\x54", [0x06e2] = "\x86\x55", [0x06e3] = "\x86\x56",
  [0x06e4] = "\x86\x57", [0x06e5] = "\xb8\xe7", [0x06e6] = "\xc5\xb6",
  [0x06e7] = "\xdf\xea", [0x06e8] = "\xc9\xda", [0x06e9] = "\xc1\xa8",
  [0x06ea] = "\xc4\xc4", [0x06eb] = "\x86\x58", [0x06ec] = "\x86\x59",
  [0x06ed] = "\xbf\xde", [0x06ee] = "\xcf\xf8", [0x06ef] = "\x86\x5a",
  [0x06f0] = "\x86\x5b", [0x06f1] = "\x86\x5c", [0x06f2] = "\xd5\xdc",
  [0x06f3] = "\xdf\xee", [0x06f4] = "\x86\x5d", [0x06f5] = "\x86\x5e",
  [0x06f6] = "\x86\x5f", [0x06f7] = "\x86\x60", [0x06f8] = "\x86\x61",
  [0x06f9] = "\x86\x62", [0x06fa] = "\xb2\xb8", [0x06fb] = "\x86\x63",
  [0x06fc] = "\xba\xdf", [0x06fd] = "\xdf\xec", [0x06fe] = "\x86\x64",
  [0x06ff] = "\xdb\xc1", [0x0700] = "\x86\x65", [0x0701] = "\xd1\xe4",
  [0x0702] = "\x86\x66", [0x0703] = "\x86\x67", [0x0704] = "\x86\x68",
  [0x0705] = "\x86\x69", [0x0706] = "\xcb\xf4", [0x0707] = "\xb4\xbd",
  [0x0708] = "\x86\x6a", [0x0709] = "\xb0\xa6", [0x070a] = "\x86\x6b",
  [0x070b] = "\x86\x6c", [0x070c] = "\x86\x6d", [0x070d] = "\x86\x6e",
  [0x070e] = "\x86\x6f", [0x070f] = "\xdf\xf1", [0x0710] = "\xcc\xc6",
  [0x0711] = "\xdf\xf2", [0x0712] = "\x86\x70", [0x0713] = "\x86\x71",
  [0x0714] = "\xdf\xed", [0x0715] = "\x86\x72", [0x0716] = "\x86\x73",
  [0x0717] = "\x86\x74", [0x0718] = "\x86\x75", [0x0719] = "\x86\x76",
  [0x071a] = "\x86\x77", [0x071b] = "\xdf\xe9", [0x071c] = "\x86\x78",
  [0x071d] = "\x86\x79", [0x071e] = "\x86\x7a", [0x071f] = "\x86\x7b",
  [0x0720] = "\xdf\xeb", [0x0721] = "\x86\x7c", [0x0722] = "\xdf\xef",
  [0x0723] = "\xdf\xf0", [0x0724] = "\xbb\xbd", [0x0725] = "\x86\x7d",
  [0x0726] = "\x86\x7e", [0x0727] = "\xdf\xf3", [0x0728] = "\x86\x80",
  [0x0729] = "\x86\x81", [0x072a] = "\xdf\xf4", [0x072b] = "\x86\x82",
  [0x072c] = "\xbb\xa3", [0x072d] = "\x86\x83", [0x072e] = "\xca\xdb",
  [0x072f] = "\xce\xa8", [0x0730] = "\xe0\xa7", [0x0731] = "\xb3\xaa",
  [0x0732] = "\x86\x84", [0x0733] = "\xe0\xa6", [0x0734] = "\x86\x85",
  [0x0735] = "\x86\x86", [0x0736] = "\x86\x87", [0x0737] = "\xe0\xa1",
  [0x0738] = "\x86\x88", [0x0739] = "\x86\x89", [0x073a] = "\x86\x8a",
  [0x073b] = "\x86\x8b", [0x073c] = "\xdf\xfe", [0x073d] = "\x86\x8c",
  [0x073e] = "\xcd\xd9", [0x073f] = "\xdf\xfc", [0x0740] = "\x86\x8d",
  [0x0741] = "\xdf\xfa", [0x0742] = "\x86\x8e", [0x0743] = "\xbf\xd0",
  [0x0744] = "\xd7\xc4", [0x0745] = "\x86\x8f", [0x0746] = "\xc9\xcc",
  [0x0747] = "\x86\x90", [0x0748] = "\x86\x91", [0x0749] = "\xdf\xf8",
  [0x074a] = "\xb0\xa1", [0x074b] = "\x86\x92", [0x074c] = "\x86\x93",
  [0x074d] = "\x86\x94", [0x074e] = "\x86\x95", [0x074f] = "\x86\x96",
  [0x0750] = "\xdf\xfd", [0x0751] = "\x86\x97", [0x0752] = "\x86\x98",
  [0x0753] = "\x86\x99", [0x0754] = "\x86\x9a", [0x0755] = "\xdf\xfb",
  [0x0756] = "\xe0\xa2", [0x0757] = "\x86\x9b", [0x0758] = "\x86\x9c",
  [0x0759] = "\x86\x9d", [0x075a] = "\x86\x9e", [0x075b] = "\x86\x9f",
  [0x075c] = "\xe0\xa8", [0x075d] = "\x86\xa0", [0x075e] = "\x86\xa1",
  [0x075f] = "\x86\xa2", [0x0760] = "\x86\xa3", [0x0761] = "\xb7\xc8",
  [0x0762] = "\x86\xa4", [0x0763] = "\x86\xa5", [0x0764] = "\xc6\xa1",
  [0x0765] = "\xc9\xb6", [0x0766] = "\xc0\xb2", [0x0767] = "\xdf\xf5",
  [0x0768] = "\x86\xa6", [0x0769] = "\x86\xa7", [0x076a] = "\xc5\xbe",
  [0x076b] = "\x86\xa8", [0x076c] = "\xd8\xc4", [0x076d] = "\xdf\xf9",
  [0x076e] = "\xc4\xf6", [0x076f] = "\x86\xa9", [0x0770] = "\x86\xaa",
  [0x0771] = "\x86\xab", [0x0772] = "\x86\xac", [0x0773] = "\x86\xad",
  [0x0774] = "\x86\xae", [0x0775] = "\xe0\xa3", [0x0776] = "\xe0\xa4",
  [0x0777] = "\xe0\xa5", [0x0778] = "\xd0\xa5", [0x0779] = "\x86\xaf",
  [0x077a] = "\x86\xb0", [0x077b] = "\xe0\xb4", [0x077c] = "\xcc\xe4",
  [0x077d] = "\x86\xb1", [0x077e] = "\xe0\xb1", [0x077f] = "\x86\xb2",
  [0x0780] = "\xbf\xa6", [0x0781] = "\xe0\xaf", [0x0782] = "\xce\xb9",
  [0x0783] = "\xe0\xab", [0x0784] = "\xc9\xc6", [0x0785] = "\x86\xb3",
  [0x0786] = "\x86\xb4", [0x0787] = "\xc0\xae", [0x0788] = "\xe0\xae",
  [0x0789] = "\xba\xed", [0x078a] = "\xba\xb0", [0x078b] = "\xe0\xa9",
  [0x078c] = "\x86\xb5", [0x078d] = "\x86\xb6", [0x078e] = "\x86\xb7",
  [0x078f] = "\xdf\xf6", [0x0790] = "\x86\xb8", [0x0791] = "\xe0\xb3",
  [0x0792] = "\x86\xb9", [0x0793] = "\x86\xba", [0x0794] = "\xe0\xb8",
  [0x0795] = "\x86\xbb", [0x0796] = "\x86\xbc", [0x0797] = "\x86\xbd",
  [0x0798] = "\xb4\xad", [0x0799] = "\xe0\xb9", [0x079a] = "\x86\xbe",
  [0x079b] = "\x86\xbf", [0x079c] = "\xcf\xb2", [0x079d] = "\xba\xc8",
  [0x079e] = "\x86\xc0", [0x079f] = "\xe0\xb0", [0x07a0] = "\x86\xc1",
  [0x07a1] = "\x86\xc2", [0x07a2] = "\x86\xc3", [0x07a3] = "\x86\xc4",
  [0x07a4] = "\x86\xc5", [0x07a5] = "\x86\xc6", [0x07a6] = "\x86\xc7",
  [0x07a7] = "\xd0\xfa", [0x07a8] = "\x86\xc8", [0x07a9] = "\x86\xc9",
  [0x07aa] = "\x86\xca", [0x07ab] = "\x86\xcb", [0x07ac] = "\x86\xcc",
  [0x07ad] = "\x86\xcd", [0x07ae] = "\x86\xce", [0x07af] = "\x86\xcf",
  [0x07b0] = "\x86\xd0", [0x07b1] = "\xe0\xac", [0x07b2] = "\x86\xd1",
  [0x07b3] = "\xd4\xfb", [0x07b4] = "\x86\xd2", [0x07b5] = "\xdf\xf7",
  [0x07b6] = "\x86\xd3", [0x07b7] = "\xc5\xe7", [0x07b8] = "\x86\xd4",
  [0x07b9] = "\xe0\xad", [0x07ba] = "\x86\xd5", [0x07bb] = "\xd3\xf7",
  [0x07bc] = "\x86\xd6", [0x07bd] = "\xe0\xb6", [0x07be] = "\xe0\xb7",
  [0x07bf] = "\x86\xd7", [0x07c0] = "\x86\xd8", [0x07c1] = "\x86\xd9",
  [0x07c2] = "\x86\xda", [0x07c3] = "\x86\xdb", [0x07c4] = "\xe0\xc4",
  [0x07c5] = "\xd0\xe1", [0x07c6] = "\x86\xdc", [0x07c7] = "\x86\xdd",
  [0x07c8] = "\x86\xde", [0x07c9] = "\xe0\xbc", [0x07ca] = "\x86\xdf",
  [0x07cb] = "\x86\xe0", [0x07cc] = "\xe0\xc9", [0x07cd] = "\xe0\xca",
  [0x07ce] = "\x86\xe1", [0x07cf] = "\x86\xe2", [0x07d0] = "\x86\xe3",
  [0x07d1] = "\xe0\xbe", [0x07d2] = "\xe0\xaa", [0x07d3] = "\xc9\xa4",
  [0x07d4] = "\xe0\xc1", [0x07d5] = "\x86\xe4", [0x07d6] = "\xe0\xb2",
  [0x07d7] = "\x86\xe5", [0x07d8] = "\x86\xe6", [0x07d9] = "\x86\xe7",
  [0x07da] = "\x86\xe8", [0x07db] = "\x86\xe9", [0x07dc] = "\xca\xc8",
  [0x07dd] = "\xe0\xc3", [0x07de] = "\x86\xea", [0x07df] = "\xe0\xb5",
  [0x07e0] = "\x86\xeb", [0x07e1] = "\xce\xcb", [0x07e2] = "\x86\xec",
  [0x07e3] = "\xcb\xc3", [0x07e4] = "\xe0\xcd", [0x07e5] = "\xe0\xc6",
  [0x07e6] = "\xe0\xc2", [0x07e7] = "\x86\xed", [0x07e8] = "\xe0\xcb",
  [0x07e9] = "\x86\xee", [0x07ea] = "\xe0\xba", [0x07eb] = "\xe0\xbf",
  [0x07ec] = "\xe0\xc0", [0x07ed] = "\x86\xef", [0x07ee] = "\x86\xf0",
  [0x07ef] = "\xe0\xc5", [0x07f0] = "\x86\xf1", [0x07f1] = "\x86\xf2",
  [0x07f2] = "\xe0\xc7", [0x07f3] = "\xe0\xc8", [0x07f4] = "\x86\xf3",
  [0x07f5] = "\xe0\xcc", [0x07f6] = "\x86\xf4", [0x07f7] = "\xe0\xbb",
  [0x07f8] = "\x86\xf5", [0x07f9] = "\x86\xf6", [0x07fa] = "\x86\xf7",
  [0x07fb] = "\x86\xf8", [0x07fc] = "\x86\xf9", [0x07fd] = "\xcb\xd4",
  [0x07fe] = "\xe0\xd5", [0x07ff] = "\x86\xfa", [0x0800] = "\xe0\xd6",
  [0x0801] = "\xe0\xd2", [0x0802] = "\x86\xfb", [0x0803] = "\x86\xfc",
  [0x0804] = "\x86\xfd", [0x0805] = "\x86\xfe", [0x0806] = "\x87\x40",
  [0x0807] = "\x87\x41", [0x0808] = "\xe0\xd0", [0x0809] = "\xbc\xce",
  [0x080a] = "\x87\x42", [0x080b] = "\x87\x43", [0x080c] = "\xe0\xd1",
  [0x080d] = "\x87\x44", [0x080e] = "\xb8\xc2", [0x080f] = "\xd8\xc5",
  [0x0810] = "\x87\x45", [0x0811] = "\x87\x46", [0x0812] = "\x87\x47",
  [0x0813] = "\x87\x48", [0x0814] = "\x87\x49", [0x0815] = "\x87\x4a",
  [0x0816] = "\x87\x4b", [0x0817] = "\x87\x4c", [0x0818] = "\xd0\xea",
  [0x0819] = "\x87\x4d", [0x081a] = "\x87\x4e", [0x081b] = "\xc2\xef",
  [0x081c] = "\x87\x4f", [0x081d] = "\x87\x50", [0x081e] = "\xe0\xcf",
  [0x081f] = "\xe0\xbd", [0x0820] = "\x87\x51", [0x0821] = "\x87\x52",
  [0x0822] = "\x87\x53", [0x0823] = "\xe0\xd4", [0x0824] = "\xe0\xd3",
  [0x0825] = "\x87\x54", [0x0826] = "\x87\x55", [0x0827] = "\xe0\xd7",
  [0x0828] = "\x87\x56", [0x0829] = "\x87\x57", [0x082a] = "\x87\x58",
  [0x082b] = "\x87\x59", [0x082c] = "\xe0\xdc", [0x082d] = "\xe0\xd8",
  [0x082e] = "\x87\x5a", [0x082f] = "\x87\x5b", [0x0830] = "\x87\x5c",
  [0x0831] = "\xd6\xf6", [0x0832] = "\xb3\xb0", [0x0833] = "\x87\x5d",
  [0x0834] = "\xd7\xec", [0x0835] = "\x87\x5e", [0x0836] = "\xcb\xbb",
  [0x0837] = "\x87\x5f", [0x0838] = "\x87\x60", [0x0839] = "\xe0\xda",
  [0x083a] = "\x87\x61", [0x083b] = "\xce\xfb", [0x083c] = "\x87\x62",
  [0x083d] = "\x87\x63", [0x083e] = "\x87\x64", [0x083f] = "\xba\xd9",
  [0x0840] = "\x87\x65", [0x0841] = "\x87\x66", [0x0842] = "\x87\x67",
  [0x0843] = "\x87\x68", [0x0844] = "\x87\x69", [0x0845] = "\x87\x6a",
  [0x0846] = "\x87\x6b", [0x0847] = "\x87\x6c", [0x0848] = "\x87\x6d",
  [0x0849] = "\x87\x6e", [0x084a] = "\x87\x6f", [0x084b] = "\x87\x70",
  [0x084c] = "\xe0\xe1", [0x084d] = "\xe0\xdd", [0x084e] = "\xd2\xad",
  [0x084f] = "\x87\x71", [0x0850] = "\x87\x72", [0x0851] = "\x87\x73",
  [0x0852] = "\x87\x74", [0x0853] = "\x87\x75", [0x0854] = "\xe0\xe2",
  [0x0855] = "\x87\x76", [0x0856] = "\x87\x77", [0x0857] = "\xe0\xdb",
  [0x0858] = "\xe0\xd9", [0x0859] = "\xe0\xdf", [0x085a] = "\x87\x78",
  [0x085b] = "\x87\x79", [0x085c] = "\xe0\xe0", [0x085d] = "\x87\x7a",
  [0x085e] = "\x87\x7b", [0x085f] = "\x87\x7c", [0x0860] = "\x87\x7d",
  [0x0861] = "\x87\x7e", [0x0862] = "\xe0\xde", [0x0863] = "\x87\x80",
  [0x0864] = "\xe0\xe4", [0x0865] = "\x87\x81", [0x0866] = "\x87\x82",
  [0x0867] = "\x87\x83", [0x0868] = "\xc6\xf7", [0x0869] = "\xd8\xac",
  [0x086a] = "\xd4\xeb", [0x086b] = "\xe0\xe6", [0x086c] = "\xca\xc9",
  [0x086d] = "\x87\x84", [0x086e] = "\x87\x85", [0x086f] = "\x87\x86",
  [0x0870] = "\x87\x87", [0x0871] = "\xe0\xe5", [0x0872] = "\x87\x88",
  [0x0873] = "\x87\x89", [0x0874] = "\x87\x8a", [0x0875] = "\x87\x8b",
  [0x0876] = "\xb8\xc1", [0x0877] = "\x87\x8c", [0x0878] = "\x87\x8d",
  [0x0879] = "\x87\x8e", [0x087a] = "\x87\x8f", [0x087b] = "\xe0\xe7",
  [0x087c] = "\xe0\xe8", [0x087d] = "\x87\x90", [0x087e] = "\x87\x91",
  [0x087f] = "\x87\x92", [0x0880] = "\x87\x93", [0x0881] = "\x87\x94",
  [0x0882] = "\x87\x95", [0x0883] = "\x87\x96", [0x0884] = "\x87\x97",
  [0x0885] = "\xe0\xe9", [0x0886] = "\xe0\xe3", [0x0887] = "\x87\x98",
  [0x0888] = "\x87\x99", [0x0889] = "\x87\x9a", [0x088a] = "\x87\x9b",
  [0x088b] = "\x87\x9c", [0x088c] = "\x87\x9d", [0x088d] = "\x87\x9e",
  [0x088e] = "\xba\xbf", [0x088f] = "\xcc\xe7", [0x0890] = "\x87\x9f",
  [0x0891] = "\x87\xa0", [0x0892] = "\x87\xa1", [0x0893] = "\xe0\xea",
  [0x0894] = "\x87\xa2", [0x0895] = "\x87\xa3", [0x0896] = "\x87\xa4",
  [0x0897] = "\x87\xa5", [0x0898] = "\x87\xa6", [0x0899] = "\x87\xa7",
  [0x089a] = "\x87\xa8", [0x089b] = "\x87\xa9", [0x089c] = "\x87\xaa",
  [0x089d] = "\x87\xab", [0x089e] = "\x87\xac", [0x089f] = "\x87\xad",
  [0x08a0] = "\x87\xae", [0x08a1] = "\x87\xaf", [0x08a2] = "\x87\xb0",
  [0x08a3] = "\xcf\xf9", [0x08a4] = "\x87\xb1", [0x08a5] = "\x87\xb2",
  [0x08a6] = "\x87\xb3", [0x08a7] = "\x87\xb4", [0x08a8] = "\x87\xb5",
  [0x08a9] = "\x87\xb6", [0x08aa] = "\x87\xb7", [0x08ab] = "\x87\xb8",
  [0x08ac] = "\x87\xb9", [0x08ad] = "\x87\xba", [0x08ae] = "\x87\xbb",
  [0x08af] = "\xe0\xeb", [0x08b0] = "\x87\xbc", [0x08b1] = "\x87\xbd",
  [0x08b2] = "\x87\xbe", [0x08b3] = "\x87\xbf", [0x08b4] = "\x87\xc0",
  [0x08b5] = "\x87\xc1", [0x08b6] = "\x87\xc2", [0x08b7] = "\xc8\xc2",
  [0x08b8] = "\x87\xc3", [0x08b9] = "\x87\xc4", [0x08ba] = "\x87\xc5",
  [0x08bb] = "\x87\xc6", [0x08bc] = "\xbd\xc0", [0x08bd] = "\x87\xc7",
  [0x08be] = "\x87\xc8", [0x08bf] = "\x87\xc9", [0x08c0] = "\x87\xca",
  [0x08c1] = "\x87\xcb", [0x08c2] = "\x87\xcc", [0x08c3] = "\x87\xcd",
  [0x08c4] = "\x87\xce", [0x08c5] = "\x87\xcf", [0x08c6] = "\x87\xd0",
  [0x08c7] = "\x87\xd1", [0x08c8] = "\x87\xd2", [0x08c9] = "\x87\xd3",
  [0x08ca] = "\xc4\xd2", [0x08cb] = "\x87\xd4", [0x08cc] = "\x87\xd5",
  [0x08cd] = "\x87\xd6", [0x08ce] = "\x87\xd7", [0x08cf] = "\x87\xd8",
  [0x08d0] = "\x87\xd9", [0x08d1] = "\x87\xda", [0x08d2] = "\x87\xdb",
  [0x08d3] = "\x87\xdc", [0x08d4] = "\xe0\xec", [0x08d5] = "\x87\xdd",
  [0x08d6] = "\x87\xde", [0x08d7] = "\xe0\xed", [0x08d8] = "\x87\xdf",
  [0x08d9] = "\x87\xe0", [0x08da] = "\xc7\xf4", [0x08db] = "\xcb\xc4",
  [0x08dc] = "\x87\xe1", [0x08dd] = "\xe0\xee", [0x08de] = "\xbb\xd8",
  [0x08df] = "\xd8\xb6", [0x08e0] = "\xd2\xf2", [0x08e1] = "\xe0\xef",
  [0x08e2] = "\xcd\xc5", [0x08e3] = "\x87\xe2", [0x08e4] = "\xb6\xda",
  [0x08e5] = "\x87\xe3", [0x08e6] = "\x87\xe4", [0x08e7] = "\x87\xe5",
  [0x08e8] = "\x87\xe6", [0x08e9] = "\x87\xe7", [0x08ea] = "\x87\xe8",
  [0x08eb] = "\xe0\xf1", [0x08ec] = "\x87\xe9", [0x08ed] = "\xd4\xb0",
  [0x08ee] = "\x87\xea", [0x08ef] = "\x87\xeb", [0x08f0] = "\xc0\xa7",
  [0x08f1] = "\xb4\xd1", [0x08f2] = "\x87\xec", [0x08f3] = "\x87\xed",
  [0x08f4] = "\xce\xa7", [0x08f5] = "\xe0\xf0", [0x08f6] = "\x87\xee",
  [0x08f7] = "\x87\xef", [0x08f8] = "\x87\xf0", [0x08f9] = "\xe0\xf2",
  [0x08fa] = "\xb9\xcc", [0x08fb] = "\x87\xf1", [0x08fc] = "\x87\xf2",
  [0x08fd] = "\xb9\xfa", [0x08fe] = "\xcd\xbc", [0x08ff] = "\xe0\xf3",
  [0x0900] = "\x87\xf3", [0x0901] = "\x87\xf4", [0x0902] = "\x87\xf5",
  [0x0903] = "\xc6\xd4", [0x0904] = "\xe0\xf4", [0x0905] = "\x87\xf6",
  [0x0906] = "\xd4\xb2", [0x0907] = "\x87\xf7", [0x0908] = "\xc8\xa6",
  [0x0909] = "\xe0\xf6", [0x090a] = "\xe0\xf5", [0x090b] = "\x87\xf8",
  [0x090c] = "\x87\xf9", [0x090d] = "\x87\xfa", [0x090e] = "\x87\xfb",
  [0x090f] = "\x87\xfc", [0x0910] = "\x87\xfd", [0x0911] = "\x87\xfe",
  [0x0912] = "\x88\x40", [0x0913] = "\x88\x41", [0x0914] = "\x88\x42",
  [0x0915] = "\x88\x43", [0x0916] = "\x88\x44", [0x0917] = "\x88\x45",
  [0x0918] = "\x88\x46", [0x0919] = "\x88\x47", [0x091a] = "\x88\x48",
  [0x091b] = "\x88\x49", [0x091c] = "\xe0\xf7", [0x091d] = "\x88\x4a",
  [0x091e] = "\x88\x4b", [0x091f] = "\xcd\xc1", [0x0920] = "\x88\x4c",
  [0x0921] = "\x88\x4d", [0x0922] = "\x88\x4e", [0x0923] = "\xca\xa5",
  [0x0924] = "\x88\x4f", [0x0925] = "\x88\x50", [0x0926] = "\x88\x51",
  [0x0927] = "\x88\x52", [0x0928] = "\xd4\xda", [0x0929] = "\xdb\xd7",
  [0x092a] = "\xdb\xd9", [0x092b] = "\x88\x53", [0x092c] = "\xdb\xd8",
  [0x092d] = "\xb9\xe7", [0x092e] = "\xdb\xdc", [0x092f] = "\xdb\xdd",
  [0x0930] = "\xb5\xd8", [0x0931] = "\x88\x54", [0x0932] = "\x88\x55",
  [0x0933] = "\xdb\xda", [0x0934] = "\x88\x56", [0x0935] = "\x88\x57",
  [0x0936] = "\x88\x58", [0x0937] = "\x88\x59", [0x0938] = "\x88\x5a",
  [0x0939] = "\xdb\xdb", [0x093a] = "\xb3\xa1", [0x093b] = "\xdb\xdf",
  [0x093c] = "\x88\x5b", [0x093d] = "\x88\x5c", [0x093e] = "\xbb\xf8",
  [0x093f] = "\x88\x5d", [0x0940] = "\xd6\xb7", [0x0941] = "\x88\x5e",
  [0x0942] = "\xdb\xe0", [0x0943] = "\x88\x5f", [0x0944] = "\x88\x60",
  [0x0945] = "\x88\x61", [0x0946] = "\x88\x62", [0x0947] = "\xbe\xf9",
  [0x0948] = "\x88\x63", [0x0949] = "\x88\x64", [0x094a] = "\xb7\xbb",
  [0x094b] = "\x88\x65", [0x094c] = "\xdb\xd0", [0x094d] = "\xcc\xae",
  [0x094e] = "\xbf\xb2", [0x094f] = "\xbb\xb5", [0x0950] = "\xd7\xf8",
  [0x0951] = "\xbf\xd3", [0x0952] = "\x88\x66", [0x0953] = "\x88\x67",
  [0x0954] = "\x88\x68", [0x0955] = "\x88\x69", [0x0956] = "\x88\x6a",
  [0x0957] = "\xbf\xe9", [0x0958] = "\x88\x6b", [0x0959] = "\x88\x6c",
  [0x095a] = "\xbc\xe1", [0x095b] = "\xcc\xb3", [0x095c] = "\xdb\xde",
  [0x095d] = "\xb0\xd3", [0x095e] = "\xce\xeb", [0x095f] = "\xb7\xd8",
  [0x0960] = "\xd7\xb9", [0x0961] = "\xc6\xc2", [0x0962] = "\x88\x6d",
  [0x0963] = "\x88\x6e", [0x0964] = "\xc0\xa4", [0x0965] = "\x88\x6f",
  [0x0966] = "\xcc\xb9", [0x0967] = "\x88\x70", [0x0968] = "\xdb\xe7",
  [0x0969] = "\xdb\xe1", [0x096a] = "\xc6\xba", [0x096b] = "\xdb\xe3",
  [0x096c] = "\x88\x71", [0x096d] = "\xdb\xe8", [0x096e] = "\x88\x72",
  [0x096f] = "\xc5\xf7", [0x0970] = "\x88\x73", [0x0971] = "\x88\x74",
  [0x0972] = "\x88\x75", [0x0973] = "\xdb\xea", [0x0974] = "\x88\x76",
  [0x0975] = "\x88\x77", [0x0976] = "\xdb\xe9", [0x0977] = "\xbf\xc0",
  [0x0978] = "\x88\x78", [0x0979] = "\x88\x79", [0x097a] = "\x88\x7a",
  [0x097b] = "\xdb\xe6", [0x097c] = "\xdb\xe5", [0x097d] = "\x88\x7b",
  [0x097e] = "\x88\x7c", [0x097f] = "\x88\x7d", [0x0980] = "\x88\x7e",
  [0x0981] = "\x88\x80", [0x0982] = "\xb4\xb9", [0x0983] = "\xc0\xac",
  [0x0984] = "\xc2\xa2", [0x0985] = "\xdb\xe2", [0x0986] = "\xdb\xe4",
  [0x0987] = "\x88\x81", [0x0988] = "\x88\x82", [0x0989] = "\x88\x83",
  [0x098a] = "\x88\x84", [0x098b] = "\xd0\xcd", [0x098c] = "\xdb\xed",
  [0x098d] = "\x88\x85", [0x098e] = "\x88\x86", [0x098f] = "\x88\x87",
  [0x0990] = "\x88\x88", [0x0991] = "\x88\x89", [0x0992] = "\xc0\xdd",
  [0x0993] = "\xdb\xf2", [0x0994] = "\x88\x8a", [0x0995] = "\x88\x8b",
  [0x0996] = "\x88\x8c", [0x0997] = "\x88\x8d", [0x0998] = "\x88\x8e",
  [0x0999] = "\x88\x8f", [0x099a] = "\x88\x90", [0x099b] = "\xb6\xe2",
  [0x099c] = "\x88\x91", [0x099d] = "\x88\x92", [0x099e] = "\x88\x93",
  [0x099f] = "\x88\x94", [0x09a0] = "\xdb\xf3", [0x09a1] = "\xdb\xd2",
  [0x09a2] = "\xb9\xb8", [0x09a3] = "\xd4\xab", [0x09a4] = "\xdb\xec",
  [0x09a5] = "\x88\x95", [0x09a6] = "\xbf\xd1", [0x09a7] = "\xdb\xf0",
  [0x09a8] = "\x88\x96", [0x09a9] = "\xdb\xd1", [0x09aa] = "\x88\x97",
  [0x09ab] = "\xb5\xe6", [0x09ac] = "\x88\x98", [0x09ad] = "\xdb\xeb",
  [0x09ae] = "\xbf\xe5", [0x09af] = "\x88\x99", [0x09b0] = "\x88\x9a",
  [0x09b1] = "\x88\x9b", [0x09b2] = "\xdb\xee", [0x09b3] = "\x88\x9c",
  [0x09b4] = "\xdb\xf1", [0x09b5] = "\x88\x9d", [0x09b6] = "\x88\x9e",
  [0x09b7] = "\x88\x9f", [0x09b8] = "\xdb\xf9", [0x09b9] = "\x88\xa0",
  [0x09ba] = "\x88\xa1", [0x09bb] = "\x88\xa2", [0x09bc] = "\x88\xa3",
  [0x09bd] = "\x88\xa4", [0x09be] = "\x88\xa5", [0x09bf] = "\x88\xa6",
  [0x09c0] = "\x88\xa7", [0x09c1] = "\x88\xa8", [0x09c2] = "\xb9\xa1",
  [0x09c3] = "\xb0\xa3", [0x09c4] = "\x88\xa9", [0x09c5] = "\x88\xaa",
  [0x09c6] = "\x88\xab", [0x09c7] = "\x88\xac", [0x09c8] = "\x88\xad",
  [0x09c9] = "\x88\xae", [0x09ca] = "\x88\xaf", [0x09cb] = "\xc2\xf1",
  [0x09cc] = "\x88\xb0", [0x09cd] = "\x88\xb1", [0x09ce] = "\xb3\xc7",
  [0x09cf] = "\xdb\xef", [0x09d0] = "\x88\xb2", [0x09d1] = "\x88\xb3",
  [0x09d2] = "\xdb\xf8", [0x09d3] = "\x88\xb4", [0x09d4] = "\xc6\xd2",
  [0x09d5] = "\xdb\xf4", [0x09d6] = "\x88\xb5", [0x09d7] = "\x88\xb6",
  [0x09d8] = "\xdb\xf5", [0x09d9] = "\xdb\xf7", [0x09da] = "\xdb\xf6",
  [0x09db] = "\x88\xb7", [0x09dc] = "\x88\xb8", [0x09dd] = "\xdb\xfe",
  [0x09de] = "\x88\xb9", [0x09df] = "\xd3\xf2", [0x09e0] = "\xb2\xba",
  [0x09e1] = "\x88\xba", [0x09e2] = "\x88\xbb", [0x09e3] = "\x88\xbc",
  [0x09e4] = "\xdb\xfd", [0x09e5] = "\x88\xbd", [0x09e6] = "\x88\xbe",
  [0x09e7] = "\x88\xbf", [0x09e8] = "\x88\xc0", [0x09e9] = "\x88\xc1",
  [0x09ea] = "\x88\xc2", [0x09eb] = "\x88\xc3", [0x09ec] = "\x88\xc4",
  [0x09ed] = "\xdc\xa4", [0x09ee] = "\x88\xc5", [0x09ef] = "\xdb\xfb",
  [0x09f0] = "\x88\xc6", [0x09f1] = "\x88\xc7", [0x09f2] = "\x88\xc8",
  [0x09f3] = "\x88\xc9", [0x09f4] = "\xdb\xfa", [0x09f5] = "\x88\xca",
  [0x09f6] = "\x88\xcb", [0x09f7] = "\x88\xcc", [0x09f8] = "\xdb\xfc",
  [0x09f9] = "\xc5\xe0", [0x09fa] = "\xbb\xf9", [0x09fb] = "\x88\xcd",
  [0x09fc] = "\x88\xce", [0x09fd] = "\xdc\xa3", [0x09fe] = "\x88\xcf",
  [0x09ff] = "\x88\xd0", [0x0a00] = "\xdc\xa5", [0x0a01] = "\x88\xd1",
  [0x0a02] = "\xcc\xc3", [0x0a03] = "\x88\xd2", [0x0a04] = "\x88\xd3",
  [0x0a05] = "\x88\xd4", [0x0a06] = "\xb6\xd1", [0x0a07] = "\xdd\xc0",
  [0x0a08] = "\x88\xd5", [0x0a09] = "\x88\xd6", [0x0a0a] = "\x88\xd7",
  [0x0a0b] = "\xdc\xa1", [0x0a0c] = "\x88\xd8", [0x0a0d] = "\xdc\xa2",
  [0x0a0e] = "\x88\xd9", [0x0a0f] = "\x88\xda", [0x0a10] = "\x88\xdb",
  [0x0a11] = "\xc7\xb5", [0x0a12] = "\x88\xdc", [0x0a13] = "\x88\xdd",
  [0x0a14] = "\x88\xde", [0x0a15] = "\xb6\xe9", [0x0a16] = "\x88\xdf",
  [0x0a17] = "\x88\xe0", [0x0a18] = "\x88\xe1", [0x0a19] = "\xdc\xa7",
  [0x0a1a] = "\x88\xe2", [0x0a1b] = "\x88\xe3", [0x0a1c] = "\x88\xe4",
  [0x0a1d] = "\x88\xe5", [0x0a1e] = "\xdc\xa6", [0x0a1f] = "\x88\xe6",
  [0x0a20] = "\xdc\xa9", [0x0a21] = "\xb1\xa4", [0x0a22] = "\x88\xe7",
  [0x0a23] = "\x88\xe8", [0x0a24] = "\xb5\xcc", [0x0a25] = "\x88\xe9",
  [0x0a26] = "\x88\xea", [0x0a27] = "\x88\xeb", [0x0a28] = "\x88\xec",
  [0x0a29] = "\x88\xed", [0x0a2a] = "\xbf\xb0", [0x0a2b] = "\x88\xee",
  [0x0a2c] = "\x88\xef", [0x0a2d] = "\x88\xf0", [0x0a2e] = "\x88\xf1",
  [0x0a2f] = "\x88\xf2", [0x0a30] = "\xd1\xdf", [0x0a31] = "\x88\xf3",
  [0x0a32] = "\x88\xf4", [0x0a33] = "\x88\xf5", [0x0a34] = "\x88\xf6",
  [0x0a35] = "\xb6\xc2", [0x0a36] = "\x88\xf7", [0x0a37] = "\x88\xf8",
  [0x0a38] = "\x88\xf9", [0x0a39] = "\x88\xfa", [0x0a3a] = "\x88\xfb",
  [0x0a3b] = "\x88\xfc", [0x0a3c] = "\x88\xfd", [0x0a3d] = "\x88\xfe",
  [0x0a3e] = "\x89\x40", [0x0a3f] = "\x89\x41", [0x0a40] = "\x89\x42",
  [0x0a41] = "\x89\x43", [0x0a42] = "\x89\x44", [0x0a43] = "\x89\x45",
  [0x0a44] = "\xdc\xa8", [0x0a45] = "\x89\x46", [0x0a46] = "\x89\x47",
  [0x0a47] = "\x89\x48", [0x0a48] = "\x89\x49", [0x0a49] = "\x89\x4a",
  [0x0a4a] = "\x89\x4b", [0x0a4b] = "\x89\x4c", [0x0a4c] = "\xcb\xfa",
  [0x0a4d] = "\xeb\xf3", [0x0a4e] = "\x89\x4d", [0x0a4f] = "\x89\x4e",
  [0x0a50] = "\x89\x4f", [0x0a51] = "\xcb\xdc", [0x0a52] = "\x89\x50",
  [0x0a53] = "\x89\x51", [0x0a54] = "\xcb\xfe", [0x0a55] = "\x89\x52",
  [0x0a56] = "\x89\x53", [0x0a57] = "\x89\x54", [0x0a58] = "\xcc\xc1",
  [0x0a59] = "\x89\x55", [0x0a5a] = "\x89\x56", [0x0a5b] = "\x89\x57",
  [0x0a5c] = "\x89\x58", [0x0a5d] = "\x89\x59", [0x0a5e] = "\xc8\xfb",
  [0x0a5f] = "\x89\x5a", [0x0a60] = "\x89\x5b", [0x0a61] = "\x89\x5c",
  [0x0a62] = "\x89\x5d", [0x0a63] = "\x89\x5e", [0x0a64] = "\x89\x5f",
  [0x0a65] = "\xdc\xaa", [0x0a66] = "\x89\x60", [0x0a67] = "\x89\x61",
  [0x0a68] = "\x89\x62", [0x0a69] = "\x89\x63", [0x0a6a] = "\x89\x64",
  [0x0a6b] = "\xcc\xee", [0x0a6c] = "\xdc\xab", [0x0a6d] = "\x89\x65",
  [0x0a6e] = "\x89\x66", [0x0a6f] = "\x89\x67", [0x0a70] = "\x89\x68",
  [0x0a71] = "\x89\x69", [0x0a72] = "\x89\x6a", [0x0a73] = "\x89\x6b",
  [0x0a74] = "\x89\x6c", [0x0a75] = "\x89\x6d", [0x0a76] = "\x89\x6e",
  [0x0a77] = "\x89\x6f", [0x0a78] = "\x89\x70", [0x0a79] = "\x89\x71",
  [0x0a7a] = "\x89\x72", [0x0a7b] = "\x89\x73", [0x0a7c] = "\x89\x74",
  [0x0a7d] = "\x89\x75", [0x0a7e] = "\xdb\xd3", [0x0a7f] = "\x89\x76",
  [0x0a80] = "\xdc\xaf", [0x0a81] = "\xdc\xac", [0x0a82] = "\x89\x77",
  [0x0a83] = "\xbe\xb3", [0x0a84] = "\x89\x78", [0x0a85] = "\xca\xfb",
  [0x0a86] = "\x89\x79", [0x0a87] = "\x89\x7a", [0x0a88] = "\x89\x7b",
  [0x0a89] = "\xdc\xad", [0x0a8a] = "\x89\x7c", [0x0a8b] = "\x89\x7d",
  [0x0a8c] = "\x89\x7e", [0x0a8d] = "\x89\x80", [0x0a8e] = "\x89\x81",
  [0x0a8f] = "\x89\x82", [0x0a90] = "\x89\x83", [0x0a91] = "\x89\x84",
  [0x0a92] = "\xc9\xca", [0x0a93] = "\xc4\xb9", [0x0a94] = "\x89\x85",
  [0x0a95] = "\x89\x86", [0x0a96] = "\x89\x87", [0x0a97] = "\x89\x88",
  [0x0a98] = "\x89\x89", [0x0a99] = "\xc7\xbd", [0x0a9a] = "\xdc\xae",
  [0x0a9b] = "\x89\x8a", [0x0a9c] = "\x89\x8b", [0x0a9d] = "\x89\x8c",
  [0x0a9e] = "\xd4\xf6", [0x0a9f] = "\xd0\xe6", [0x0aa0] = "\x89\x8d",
  [0x0aa1] = "\x89\x8e", [0x0aa2] = "\x89\x8f", [0x0aa3] = "\x89\x90",
  [0x0aa4] = "\x89\x91", [0x0aa5] = "\x89\x92", [0x0aa6] = "\x89\x93",
  [0x0aa7] = "\x89\x94", [0x0aa8] = "\xc4\xab", [0x0aa9] = "\xb6\xd5",
  [0x0aaa] = "\x89\x95", [0x0aab] = "\x89\x96", [0x0aac] = "\x89\x97",
  [0x0aad] = "\x89\x98", [0x0aae] = "\x89\x99", [0x0aaf] = "\x89\x9a",
  [0x0ab0] = "\x89\x9b", [0x0ab1] = "\x89\x9c", [0x0ab2] = "\x89\x9d",
  [0x0ab3] = "\x89\x9e", [0x0ab4] = "\x89\x9f", [0x0ab5] = "\x89\xa0",
  [0x0ab6] = "\x89\xa1", [0x0ab7] = "\x89\xa2", [0x0ab8] = "\x89\xa3",
  [0x0ab9] = "\x89\xa4", [0x0aba] = "\x89\xa5", [0x0abb] = "\x89\xa6",
  [0x0abc] = "\xdb\xd4", [0x0abd] = "\x89\xa7", [0x0abe] = "\x89\xa8",
  [0x0abf] = "\x89\xa9", [0x0ac0] = "\x89\xaa", [0x0ac1] = "\xb1\xda",
  [0x0ac2] = "\x89\xab", [0x0ac3] = "\x89\xac", [0x0ac4] = "\x89\xad",
  [0x0ac5] = "\xdb\xd5", [0x0ac6] = "\x89\xae", [0x0ac7] = "\x89\xaf",
  [0x0ac8] = "\x89\xb0", [0x0ac9] = "\x89\xb1", [0x0aca] = "\x89\xb2",
  [0x0acb] = "\x89\xb3", [0x0acc] = "\x89\xb4", [0x0acd] = "\x89\xb5",
  [0x0ace] = "\x89\xb6", [0x0acf] = "\x89\xb7", [0x0ad0] = "\x89\xb8",
  [0x0ad1] = "\xdb\xd6", [0x0ad2] = "\x89\xb9", [0x0ad3] = "\x89\xba",
  [0x0ad4] = "\x89\xbb", [0x0ad5] = "\xba\xbe", [0x0ad6] = "\x89\xbc",
  [0x0ad7] = "\x89\xbd", [0x0ad8] = "\x89\xbe", [0x0ad9] = "\x89\xbf",
  [0x0ada] = "\x89\xc0", [0x0adb] = "\x89\xc1", [0x0adc] = "\x89\xc2",
  [0x0add] = "\x89\xc3", [0x0ade] = "\x89\xc4", [0x0adf] = "\x89\xc5",
  [0x0ae0] = "\x89\xc6", [0x0ae1] = "\x89\xc7", [0x0ae2] = "\x89\xc8",
  [0x0ae3] = "\x89\xc9", [0x0ae4] = "\xc8\xc0", [0x0ae5] = "\x89\xca",
  [0x0ae6] = "\x89\xcb", [0x0ae7] = "\x89\xcc", [0x0ae8] = "\x89\xcd",
  [0x0ae9] = "\x89\xce", [0x0aea] = "\x89\xcf", [0x0aeb] = "\xca\xbf",
  [0x0aec] = "\xc8\xc9", [0x0aed] = "\x89\xd0", [0x0aee] = "\xd7\xb3",
  [0x0aef] = "\x89\xd1", [0x0af0] = "\xc9\xf9", [0x0af1] = "\x89\xd2",
  [0x0af2] = "\x89\xd3", [0x0af3] = "\xbf\xc7", [0x0af4] = "\x89\xd4",
  [0x0af5] = "\x89\xd5", [0x0af6] = "\xba\xf8", [0x0af7] = "\x89\xd6",
  [0x0af8] = "\x89\xd7", [0x0af9] = "\xd2\xbc", [0x0afa] = "\x89\xd8",
  [0x0afb] = "\x89\xd9", [0x0afc] = "\x89\xda", [0x0afd] = "\x89\xdb",
  [0x0afe] = "\x89\xdc", [0x0aff] = "\x89\xdd", [0x0b00] = "\x89\xde",
  [0x0b01] = "\x89\xdf", [0x0b02] = "\xe2\xba", [0x0b03] = "\x89\xe0",
  [0x0b04] = "\xb4\xa6", [0x0b05] = "\x89\xe1", [0x0b06] = "\x89\xe2",
  [0x0b07] = "\xb1\xb8", [0x0b08] = "\x89\xe3", [0x0b09] = "\x89\xe4",
  [0x0b0a] = "\x89\xe5", [0x0b0b] = "\x89\xe6", [0x0b0c] = "\x89\xe7",
  [0x0b0d] = "\xb8\xb4", [0x0b0e] = "\x89\xe8", [0x0b0f] = "\xcf\xc4",
  [0x0b10] = "\x89\xe9", [0x0b11] = "\x89\xea", [0x0b12] = "\x89\xeb",
  [0x0b13] = "\x89\xec", [0x0b14] = "\xd9\xe7", [0x0b15] = "\xcf\xa6",
  [0x0b16] = "\xcd\xe2", [0x0b17] = "\x89\xed", [0x0b18] = "\x89\xee",
  [0x0b19] = "\xd9\xed", [0x0b1a] = "\xb6\xe0", [0x0b1b] = "\x89\xef",
  [0x0b1c] = "\xd2\xb9", [0x0b1d] = "\x89\xf0", [0x0b1e] = "\x89\xf1",
  [0x0b1f] = "\xb9\xbb", [0x0b20] = "\x89\xf2", [0x0b21] = "\x89\xf3",
  [0x0b22] = "\x89\xf4", [0x0b23] = "\x89\xf5", [0x0b24] = "\xe2\xb9",
  [0x0b25] = "\xe2\xb7", [0x0b26] = "\x89\xf6", [0x0b27] = "\xb4\xf3",
  [0x0b28] = "\x89\xf7", [0x0b29] = "\xcc\xec", [0x0b2a] = "\xcc\xab",
  [0x0b2b] = "\xb7\xf2", [0x0b2c] = "\x89\xf8", [0x0b2d] = "\xd8\xb2",
  [0x0b2e] = "\xd1\xeb", [0x0b2f] = "\xba\xbb", [0x0b30] = "\x89\xf9",
  [0x0b31] = "\xca\xa7", [0x0b32] = "\x89\xfa", [0x0b33] = "\x89\xfb",
  [0x0b34] = "\xcd\xb7", [0x0b35] = "\x89\xfc", [0x0b36] = "\x89\xfd",
  [0x0b37] = "\xd2\xc4", [0x0b38] = "\xbf\xe4", [0x0b39] = "\xbc\xd0",
  [0x0b3a] = "\xb6\xe1", [0x0b3b] = "\x89\xfe", [0x0b3c] = "\xde\xc5",
  [0x0b3d] = "\x8a\x40", [0x0b3e] = "\x8a\x41", [0x0b3f] = "\x8a\x42",
  [0x0b40] = "\x8a\x43", [0x0b41] = "\xde\xc6", [0x0b42] = "\xdb\xbc",
  [0x0b43] = "\x8a\x44", [0x0b44] = "\xd1\xd9", [0x0b45] = "\x8a\x45",
  [0x0b46] = "\x8a\x46", [0x0b47] = "\xc6\xe6", [0x0b48] = "\xc4\xce",
  [0x0b49] = "\xb7\xee", [0x0b4a] = "\x8a\x47", [0x0b4b] = "\xb7\xdc",
  [0x0b4c] = "\x8a\x48", [0x0b4d] = "\x8a\x49", [0x0b4e] = "\xbf\xfc",
  [0x0b4f] = "\xd7\xe0", [0x0b50] = "\x8a\x4a", [0x0b51] = "\xc6\xf5",
  [0x0b52] = "\x8a\x4b", [0x0b53] = "\x8a\x4c", [0x0b54] = "\xb1\xbc",
  [0x0b55] = "\xde\xc8", [0x0b56] = "\xbd\xb1", [0x0b57] = "\xcc\xd7",
  [0x0b58] = "\xde\xca", [0x0b59] = "\x8a\x4d", [0x0b5a] = "\xde\xc9",
  [0x0b5b] = "\x8a\x4e", [0x0b5c] = "\x8a\x4f", [0x0b5d] = "\x8a\x50",
  [0x0b5e] = "\x8a\x51", [0x0b5f] = "\x8a\x52", [0x0b60] = "\xb5\xec",
  [0x0b61] = "\x8a\x53", [0x0b62] = "\xc9\xdd", [0x0b63] = "\x8a\x54",
  [0x0b64] = "\x8a\x55", [0x0b65] = "\xb0\xc2", [0x0b66] = "\x8a\x56",
  [0x0b67] = "\x8a\x57", [0x0b68] = "\x8a\x58", [0x0b69] = "\x8a\x59",
  [0x0b6a] = "\x8a\x5a", [0x0b6b] = "\x8a\x5b", [0x0b6c] = "\x8a\x5c",
  [0x0b6d] = "\x8a\x5d", [0x0b6e] = "\x8a\x5e", [0x0b6f] = "\x8a\x5f",
  [0x0b70] = "\x8a\x60", [0x0b71] = "\x8a\x61", [0x0b72] = "\x8a\x62",
  [0x0b73] = "\xc5\xae", [0x0b74] = "\xc5\xab", [0x0b75] = "\x8a\x63",
  [0x0b76] = "\xc4\xcc", [0x0b77] = "\x8a\x64", [0x0b78] = "\xbc\xe9",
  [0x0b79] = "\xcb\xfd", [0x0b7a] = "\x8a\x65", [0x0b7b] = "\x8a\x66",
  [0x0b7c] = "\x8a\x67", [0x0b7d] = "\xba\xc3", [0x0b7e] = "\x8a\x68",
  [0x0b7f] = "\x8a\x69", [0x0b80] = "\x8a\x6a", [0x0b81] = "\xe5\xf9",
  [0x0b82] = "\xc8\xe7", [0x0b83] = "\xe5\xfa", [0x0b84] = "\xcd\xfd",
  [0x0b85] = "\x8a\x6b", [0x0b86] = "\xd7\xb1", [0x0b87] = "\xb8\xbe",
  [0x0b88] = "\xc2\xe8", [0x0b89] = "\x8a\x6c", [0x0b8a] = "\xc8\xd1",
  [0x0b8b] = "\x8a\x6d", [0x0b8c] = "\x8a\x6e", [0x0b8d] = "\xe5\xfb",
  [0x0b8e] = "\x8a\x6f", [0x0b8f] = "\x8a\x70", [0x0b90] = "\x8a\x71",
  [0x0b91] = "\x8a\x72", [0x0b92] = "\xb6\xca", [0x0b93] = "\xbc\xcb",
  [0x0b94] = "\x8a\x73", [0x0b95] = "\x8a\x74", [0x0b96] = "\xd1\xfd",
  [0x0b97] = "\xe6\xa1", [0x0b98] = "\x8a\x75", [0x0b99] = "\xc3\xee",
  [0x0b9a] = "\x8a\x76", [0x0b9b] = "\x8a\x77", [0x0b9c] = "\x8a\x78",
  [0x0b9d] = "\x8a\x79", [0x0b9e] = "\xe6\xa4", [0x0b9f] = "\x8a\x7a",
  [0x0ba0] = "\x8a\x7b", [0x0ba1] = "\x8a\x7c", [0x0ba2] = "\x8a\x7d",
  [0x0ba3] = "\xe5\xfe", [0x0ba4] = "\xe6\xa5", [0x0ba5] = "\xcd\xd7",
  [0x0ba6] = "\x8a\x7e", [0x0ba7] = "\x8a\x80", [0x0ba8] = "\xb7\xc1",
  [0x0ba9] = "\xe5\xfc", [0x0baa] = "\xe5\xfd", [0x0bab] = "\xe6\xa3",
  [0x0bac] = "\x8a\x81", [0x0bad] = "\x8a\x82", [0x0bae] = "\xc4\xdd",
  [0x0baf] = "\xe6\xa8", [0x0bb0] = "\x8a\x83", [0x0bb1] = "\x8a\x84",
  [0x0bb2] = "\xe6\xa7", [0x0bb3] = "\x8a\x85", [0x0bb4] = "\x8a\x86",
  [0x0bb5] = "\x8a\x87", [0x0bb6] = "\x8a\x88", [0x0bb7] = "\x8a\x89",
  [0x0bb8] = "\x8a\x8a", [0x0bb9] = "\xc3\xc3", [0x0bba] = "\x8a\x8b",
  [0x0bbb] = "\xc6\xde", [0x0bbc] = "\x8a\x8c", [0x0bbd] = "\x8a\x8d",
  [0x0bbe] = "\xe6\xaa", [0x0bbf] = "\x8a\x8e", [0x0bc0] = "\x8a\x8f",
  [0x0bc1] = "\x8a\x90", [0x0bc2] = "\x8a\x91", [0x0bc3] = "\x8a\x92",
  [0x0bc4] = "\x8a\x93", [0x0bc5] = "\x8a\x94", [0x0bc6] = "\xc4\xb7",
  [0x0bc7] = "\x8a\x95", [0x0bc8] = "\x8a\x96", [0x0bc9] = "\x8a\x97",
  [0x0bca] = "\xe6\xa2", [0x0bcb] = "\xca\xbc", [0x0bcc] = "\x8a\x98",
  [0x0bcd] = "\x8a\x99", [0x0bce] = "\x8a\x9a", [0x0bcf] = "\x8a\x9b",
  [0x0bd0] = "\xbd\xe3", [0x0bd1] = "\xb9\xc3", [0x0bd2] = "\xe6\xa6",
  [0x0bd3] = "\xd0\xd5", [0x0bd4] = "\xce\xaf", [0x0bd5] = "\x8a\x9c",
  [0x0bd6] = "\x8a\x9d", [0x0bd7] = "\xe6\xa9", [0x0bd8] = "\xe6\xb0",
  [0x0bd9] = "\x8a\x9e", [0x0bda] = "\xd2\xa6", [0x0bdb] = "\x8a\x9f",
  [0x0bdc] = "\xbd\xaa", [0x0bdd] = "\xe6\xad", [0x0bde] = "\x8a\xa0",
  [0x0bdf] = "\x8a\xa1", [0x0be0] = "\x8a\xa2", [0x0be1] = "\x8a\xa3",
  [0x0be2] = "\x8a\xa4", [0x0be3] = "\xe6\xaf", [0x0be4] = "\x8a\xa5",
  [0x0be5] = "\xc0\xd1", [0x0be6] = "\x8a\xa6", [0x0be7] = "\x8a\xa7",
  [0x0be8] = "\xd2\xcc", [0x0be9] = "\x8a\xa8", [0x0bea] = "\x8a\xa9",
  [0x0beb] = "\x8a\xaa", [0x0bec] = "\xbc\xa7", [0x0bed] = "\x8a\xab",
  [0x0bee] = "\x8a\xac", [0x0bef] = "\x8a\xad", [0x0bf0] = "\x8a\xae",
  [0x0bf1] = "\x8a\xaf", [0x0bf2] = "\x8a\xb0", [0x0bf3] = "\x8a\xb1",
  [0x0bf4] = "\x8a\xb2", [0x0bf5] = "\x8a\xb3", [0x0bf6] = "\x8a\xb4",
  [0x0bf7] = "\x8a\xb5", [0x0bf8] = "\x8a\xb6", [0x0bf9] = "\xe6\xb1",
  [0x0bfa] = "\x8a\xb7", [0x0bfb] = "\xd2\xf6", [0x0bfc] = "\x8a\xb8",
  [0x0bfd] = "\x8a\xb9", [0x0bfe] = "\x8a\xba", [0x0bff] = "\xd7\xcb",
  [0x0c00] = "\x8a\xbb", [0x0c01] = "\xcd\xfe", [0x0c02] = "\x8a\xbc",
  [0x0c03] = "\xcd\xde", [0x0c04] = "\xc2\xa6", [0x0c05] = "\xe6\xab",
  [0x0c06] = "\xe6\xac", [0x0c07] = "\xbd\xbf", [0x0c08] = "\xe6\xae",
  [0x0c09] = "\xe6\xb3", [0x0c0a] = "\x8a\xbd", [0x0c0b] = "\x8a\xbe",
  [0x0c0c] = "\xe6\xb2", [0x0c0d] = "\x8a\xbf", [0x0c0e] = "\x8a\xc0",
  [0x0c0f] = "\x8a\xc1", [0x0c10] = "\x8a\xc2", [0x0c11] = "\xe6\xb6",
  [0x0c12] = "\x8a\xc3", [0x0c13] = "\xe6\xb8", [0x0c14] = "\x8a\xc4",
  [0x0c15] = "\x8a\xc5", [0x0c16] = "\x8a\xc6", [0x0c17] = "\x8a\xc7",
  [0x0c18] = "\xc4\xef", [0x0c19] = "\x8a\xc8", [0x0c1a] = "\x8a\xc9",
  [0x0c1b] = "\x8a\xca", [0x0c1c] = "\xc4\xc8", [0x0c1d] = "\x8a\xcb",
  [0x0c1e] = "\x8a\xcc", [0x0c1f] = "\xbe\xea", [0x0c20] = "\xc9\xef",
  [0x0c21] = "\x8a\xcd", [0x0c22] = "\x8a\xce", [0x0c23] = "\xe6\xb7",
  [0x0c24] = "\x8a\xcf", [0x0c25] = "\xb6\xf0", [0x0c26] = "\x8a\xd0",
  [0x0c27] = "\x8a\xd1", [0x0c28] = "\x8a\xd2", [0x0c29] = "\xc3\xe4",
  [0x0c2a] = "\x8a\xd3", [0x0c2b] = "\x8a\xd4", [0x0c2c] = "\x8a\xd5",
  [0x0c2d] = "\x8a\xd6", [0x0c2e] = "\x8a\xd7", [0x0c2f] = "\x8a\xd8",
  [0x0c30] = "\x8a\xd9", [0x0c31] = "\xd3\xe9", [0x0c32] = "\xe6\xb4",
  [0x0c33] = "\x8a\xda", [0x0c34] = "\xe6\xb5", [0x0c35] = "\x8a\xdb",
  [0x0c36] = "\xc8\xa2", [0x0c37] = "\x8a\xdc", [0x0c38] = "\x8a\xdd",
  [0x0c39] = "\x8a\xde", [0x0c3a] = "\x8a\xdf", [0x0c3b] = "\x8a\xe0",
  [0x0c3c] = "\xe6\xbd", [0x0c3d] = "\x8a\xe1", [0x0c3e] = "\x8a\xe2",
  [0x0c3f] = "\x8a\xe3", [0x0c40] = "\xe6\xb9", [0x0c41] = "\x8a\xe4",
  [0x0c42] = "\x8a\xe5", [0x0c43] = "\x8a\xe6", [0x0c44] = "\x8a\xe7",
  [0x0c45] = "\x8a\xe8", [0x0c46] = "\xc6\xc5", [0x0c47] = "\x8a\xe9",
  [0x0c48] = "\x8a\xea", [0x0c49] = "\xcd\xf1", [0x0c4a] = "\xe6\xbb",
  [0x0c4b] = "\x8a\xeb", [0x0c4c] = "\x8a\xec", [0x0c4d] = "\x8a\xed",
  [0x0c4e] = "\x8a\xee", [0x0c4f] = "\x8a\xef", [0x0c50] = "\x8a\xf0",
  [0x0c51] = "\x8a\xf1", [0x0c52] = "\x8a\xf2", [0x0c53] = "\x8a\xf3",
  [0x0c54] = "\x8a\xf4", [0x0c55] = "\xe6\xbc", [0x0c56] = "\x8a\xf5",
  [0x0c57] = "\x8a\xf6", [0x0c58] = "\x8a\xf7", [0x0c59] = "\x8a\xf8",
  [0x0c5a] = "\xbb\xe9", [0x0c5b] = "\x8a\xf9", [0x0c5c] = "\x8a\xfa",
  [0x0c5d] = "\x8a\xfb", [0x0c5e] = "\x8a\xfc", [0x0c5f] = "\x8a\xfd",
  [0x0c60] = "\x8a\xfe", [0x0c61] = "\x8b\x40", [0x0c62] = "\xe6\xbe",
  [0x0c63] = "\x8b\x41", [0x0c64] = "\x8b\x42", [0x0c65] = "\x8b\x43",
  [0x0c66] = "\x8b\x44", [0x0c67] = "\xe6\xba", [0x0c68] = "\x8b\x45",
  [0x0c69] = "\x8b\x46", [0x0c6a] = "\xc0\xb7", [0x0c6b] = "\x8b\x47",
  [0x0c6c] = "\x8b\x48", [0x0c6d] = "\x8b\x49", [0x0c6e] = "\x8b\x4a",
  [0x0c6f] = "\x8b\x4b", [0x0c70] = "\x8b\x4c", [0x0c71] = "\x8b\x4d",
  [0x0c72] = "\x8b\x4e", [0x0c73] = "\x8b\x4f", [0x0c74] = "\xd3\xa4",
  [0x0c75] = "\xe6\xbf", [0x0c76] = "\xc9\xf4", [0x0c77] = "\xe6\xc3",
  [0x0c78] = "\x8b\x50", [0x0c79] = "\x8b\x51", [0x0c7a] = "\xe6\xc4",
  [0x0c7b] = "\x8b\x52", [0x0c7c] = "\x8b\x53", [0x0c7d] = "\x8b\x54",
  [0x0c7e] = "\x8b\x55", [0x0c7f] = "\xd0\xf6", [0x0c80] = "\x8b\x56",
  [0x0c81] = "\x8b\x57", [0x0c82] = "\x8b\x58", [0x0c83] = "\x8b\x59",
  [0x0c84] = "\x8b\x5a", [0x0c85] = "\x8b\x5b", [0x0c86] = "\x8b\x5c",
  [0x0c87] = "\x8b\x5d", [0x0c88] = "\x8b\x5e", [0x0c89] = "\x8b\x5f",
  [0x0c8a] = "\x8b\x60", [0x0c8b] = "\x8b\x61", [0x0c8c] = "\x8b\x62",
  [0x0c8d] = "\x8b\x63", [0x0c8e] = "\x8b\x64", [0x0c8f] = "\x8b\x65",
  [0x0c90] = "\x8b\x66", [0x0c91] = "\x8b\x67", [0x0c92] = "\xc3\xbd",
  [0x0c93] = "\x8b\x68", [0x0c94] = "\x8b\x69", [0x0c95] = "\x8b\x6a",
  [0x0c96] = "\x8b\x6b", [0x0c97] = "\x8b\x6c", [0x0c98] = "\x8b\x6d",
  [0x0c99] = "\x8b\x6e", [0x0c9a] = "\xc3\xc4", [0x0c9b] = "\xe6\xc2",
  [0x0c9c] = "\x8b\x6f", [0x0c9d] = "\x8b\x70", [0x0c9e] = "\x8b\x71",
  [0x0c9f] = "\x8b\x72", [0x0ca0] = "\x8b\x73", [0x0ca1] = "\x8b\x74",
  [0x0ca2] = "\x8b\x75", [0x0ca3] = "\x8b\x76", [0x0ca4] = "\x8b\x77",
  [0x0ca5] = "\x8b\x78", [0x0ca6] = "\x8b\x79", [0x0ca7] = "\x8b\x7a",
  [0x0ca8] = "\x8b\x7b", [0x0ca9] = "\x8b\x7c", [0x0caa] = "\xe6\xc1",
  [0x0cab] = "\x8b\x7d", [0x0cac] = "\x8b\x7e", [0x0cad] = "\x8b\x80",
  [0x0cae] = "\x8b\x81", [0x0caf] = "\x8b\x82", [0x0cb0] = "\x8b\x83",
  [0x0cb1] = "\x8b\x84", [0x0cb2] = "\xe6\xc7", [0x0cb3] = "\xcf\xb1",
  [0x0cb4] = "\x8b\x85", [0x0cb5] = "\xeb\xf4", [0x0cb6] = "\x8b\x86",
  [0x0cb7] = "\x8b\x87", [0x0cb8] = "\xe6\xca", [0x0cb9] = "\x8b\x88",
  [0x0cba] = "\x8b\x89", [0x0cbb] = "\x8b\x8a", [0x0cbc] = "\x8b\x8b",
  [0x0cbd] = "\x8b\x8c", [0x0cbe] = "\xe6\xc5", [0x0cbf] = "\x8b\x8d",
  [0x0cc0] = "\x8b\x8e", [0x0cc1] = "\xbc\xde", [0x0cc2] = "\xc9\xa9",
  [0x0cc3] = "\x8b\x8f", [0x0cc4] = "\x8b\x90", [0x0cc5] = "\x8b\x91",
  [0x0cc6] = "\x8b\x92", [0x0cc7] = "\x8b\x93", [0x0cc8] = "\x8b\x94",
  [0x0cc9] = "\xbc\xb5", [0x0cca] = "\x8b\x95", [0x0ccb] = "\x8b\x96",
  [0x0ccc] = "\xcf\xd3", [0x0ccd] = "\x8b\x97", [0x0cce] = "\x8b\x98",
  [0x0ccf] = "\x8b\x99", [0x0cd0] = "\x8b\x9a", [0x0cd1] = "\x8b\x9b",
  [0x0cd2] = "\xe6\xc8", [0x0cd3] = "\x8b\x9c", [0x0cd4] = "\xe6\xc9",
  [0x0cd5] = "\x8b\x9d", [0x0cd6] = "\xe6\xce", [0x0cd7] = "\x8b\x9e",
  [0x0cd8] = "\xe6\xd0", [0x0cd9] = "\x8b\x9f", [0x0cda] = "\x8b\xa0",
  [0x0cdb] = "\x8b\xa1", [0x0cdc] = "\xe6\xd1", [0x0cdd] = "\x8b\xa2",
  [0x0cde] = "\x8b\xa3", [0x0cdf] = "\x8b\xa4", [0x0ce0] = "\xe6\xcb",
  [0x0ce1] = "\xb5\xd5", [0x0ce2] = "\x8b\xa5", [0x0ce3] = "\xe6\xcc",
  [0x0ce4] = "\x8b\xa6", [0x0ce5] = "\x8b\xa7", [0x0ce6] = "\xe6\xcf",
  [0x0ce7] = "\x8b\xa8", [0x0ce8] = "\x8b\xa9", [0x0ce9] = "\xc4\xdb",
  [0x0cea] = "\x8b\xaa", [0x0ceb] = "\xe6\xc6", [0x0cec] = "\x8b\xab",
  [0x0ced] = "\x8b\xac", [0x0cee] = "\x8b\xad", [0x0cef] = "\x8b\xae",
  [0x0cf0] = "\x8b\xaf", [0x0cf1] = "\xe6\xcd", [0x0cf2] = "\x8b\xb0",
  [0x0cf3] = "\x8b\xb1", [0x0cf4] = "\x8b\xb2", [0x0cf5] = "\x8b\xb3",
  [0x0cf6] = "\x8b\xb4", [0x0cf7] = "\x8b\xb5", [0x0cf8] = "\x8b\xb6",
  [0x0cf9] = "\x8b\xb7", [0x0cfa] = "\x8b\xb8", [0x0cfb] = "\x8b\xb9",
  [0x0cfc] = "\x8b\xba", [0x0cfd] = "\x8b\xbb", [0x0cfe] = "\x8b\xbc",
  [0x0cff] = "\x8b\xbd", [0x0d00] = "\x8b\xbe", [0x0d01] = "\x8b\xbf",
  [0x0d02] = "\x8b\xc0", [0x0d03] = "\x8b\xc1", [0x0d04] = "\x8b\xc2",
  [0x0d05] = "\x8b\xc3", [0x0d06] = "\x8b\xc4", [0x0d07] = "\x8b\xc5",
  [0x0d08] = "\x8b\xc6", [0x0d09] = "\xe6\xd2", [0x0d0a] = "\x8b\xc7",
  [0x0d0b] = "\x8b\xc8", [0x0d0c] = "\x8b\xc9", [0x0d0d] = "\x8b\xca",
  [0x0d0e] = "\x8b\xcb", [0x0d0f] = "\x8b\xcc", [0x0d10] = "\x8b\xcd",
  [0x0d11] = "\x8b\xce", [0x0d12] = "\x8b\xcf", [0x0d13] = "\x8b\xd0",
  [0x0d14] = "\x8b\xd1", [0x0d15] = "\x8b\xd2", [0x0d16] = "\xe6\xd4",
  [0x0d17] = "\xe6\xd3", [0x0d18] = "\x8b\xd3", [0x0d19] = "\x8b\xd4",
  [0x0d1a] = "\x8b\xd5", [0x0d1b] = "\x8b\xd6", [0x0d1c] = "\x8b\xd7",
  [0x0d1d] = "\x8b\xd8", [0x0d1e] = "\x8b\xd9", [0x0d1f] = "\x8b\xda",
  [0x0d20] = "\x8b\xdb", [0x0d21] = "\x8b\xdc", [0x0d22] = "\x8b\xdd",
  [0x0d23] = "\x8b\xde", [0x0d24] = "\x8b\xdf", [0x0d25] = "\x8b\xe0",
  [0x0d26] = "\x8b\xe1", [0x0d27] = "\x8b\xe2", [0x0d28] = "\x8b\xe3",
  [0x0d29] = "\x8b\xe4", [0x0d2a] = "\x8b\xe5", [0x0d2b] = "\x8b\xe6",
  [0x0d2c] = "\x8b\xe7", [0x0d2d] = "\x8b\xe8", [0x0d2e] = "\x8b\xe9",
  [0x0d2f] = "\x8b\xea", [0x0d30] = "\x8b\xeb", [0x0d31] = "\x8b\xec",
  [0x0d32] = "\xe6\xd5", [0x0d33] = "\x8b\xed", [0x0d34] = "\xd9\xf8",
  [0x0d35] = "\x8b\xee", [0x0d36] = "\x8b\xef", [0x0d37] = "\xe6\xd6",
  [0x0d38] = "\x8b\xf0", [0x0d39] = "\x8b\xf1", [0x0d3a] = "\x8b\xf2",
  [0x0d3b] = "\x8b\xf3", [0x0d3c] = "\x8b\xf4", [0x0d3d] = "\x8b\xf5",
  [0x0d3e] = "\x8b\xf6", [0x0d3f] = "\x8b\xf7", [0x0d40] = "\xe6\xd7",
  [0x0d41] = "\x8b\xf8", [0x0d42] = "\x8b\xf9", [0x0d43] = "\x8b\xfa",
  [0x0d44] = "\x8b\xfb", [0x0d45] = "\x8b\xfc", [0x0d46] = "\x8b\xfd",
  [0x0d47] = "\x8b\xfe", [0x0d48] = "\x8c\x40", [0x0d49] = "\x8c\x41",
  [0x0d4a] = "\x8c\x42", [0x0d4b] = "\x8c\x43", [0x0d4c] = "\x8c\x44",
  [0x0d4d] = "\x8c\x45", [0x0d4e] = "\x8c\x46", [0x0d4f] = "\x8c\x47",
  [0x0d50] = "\xd7\xd3", [0x0d51] = "\xe6\xdd", [0x0d52] = "\x8c\x48",
  [0x0d53] = "\xe6\xde", [0x0d54] = "\xbf\xd7", [0x0d55] = "\xd4\xd0",
  [0x0d56] = "\x8c\x49", [0x0d57] = "\xd7\xd6", [0x0d58] = "\xb4\xe6",
  [0x0d59] = "\xcb\xef", [0x0d5a] = "\xe6\xda", [0x0d5b] = "\xd8\xc3",
  [0x0d5c] = "\xd7\xce", [0x0d5d] = "\xd0\xa2", [0x0d5e] = "\x8c\x4a",
  [0x0d5f] = "\xc3\xcf", [0x0d60] = "\x8c\x4b", [0x0d61] = "\x8c\x4c",
  [0x0d62] = "\xe6\xdf", [0x0d63] = "\xbc\xbe", [0x0d64] = "\xb9\xc2",
  [0x0d65] = "\xe6\xdb", [0x0d66] = "\xd1\xa7", [0x0d67] = "\x8c\x4d",
  [0x0d68] = "\x8c\x4e", [0x0d69] = "\xba\xa2", [0x0d6a] = "\xc2\xcf",
  [0x0d6b] = "\x8c\x4f", [0x0d6c] = "\xd8\xab", [0x0d6d] = "\x8c\x50",
  [0x0d6e] = "\x8c\x51", [0x0d6f] = "\x8c\x52", [0x0d70] = "\xca\xeb",
  [0x0d71] = "\xe5\xee", [0x0d72] = "\x8c\x53", [0x0d73] = "\xe6\xdc",
  [0x0d74] = "\x8c\x54", [0x0d75] = "\xb7\xf5", [0x0d76] = "\x8c\x55",
  [0x0d77] = "\x8c\x56", [0x0d78] = "\x8c\x57", [0x0d79] = "\x8c\x58",
  [0x0d7a] = "\xc8\xe6", [0x0d7b] = "\x8c\x59", [0x0d7c] = "\x8c\x5a",
  [0x0d7d] = "\xc4\xf5", [0x0d7e] = "\x8c\x5b", [0x0d7f] = "\x8c\x5c",
  [0x0d80] = "\xe5\xb2", [0x0d81] = "\xc4\xfe", [0x0d82] = "\x8c\x5d",
  [0x0d83] = "\xcb\xfc", [0x0d84] = "\xe5\xb3", [0x0d85] = "\xd5\xac",
  [0x0d86] = "\x8c\x5e", [0x0d87] = "\xd3\xee", [0x0d88] = "\xca\xd8",
  [0x0d89] = "\xb0\xb2", [0x0d8a] = "\x8c\x5f", [0x0d8b] = "\xcb\xce",
  [0x0d8c] = "\xcd\xea", [0x0d8d] = "\x8c\x60", [0x0d8e] = "\x8c\x61",
  [0x0d8f] = "\xba\xea", [0x0d90] = "\x8c\x62", [0x0d91] = "\x8c\x63",
  [0x0d92] = "\x8c\x64", [0x0d93] = "\xe5\xb5", [0x0d94] = "\x8c\x65",
  [0x0d95] = "\xe5\xb4", [0x0d96] = "\x8c\x66", [0x0d97] = "\xd7\xda",
  [0x0d98] = "\xb9\xd9", [0x0d99] = "\xd6\xe6", [0x0d9a] = "\xb6\xa8",
  [0x0d9b] = "\xcd\xf0", [0x0d9c] = "\xd2\xcb", [0x0d9d] = "\xb1\xa6",
  [0x0d9e] = "\xca\xb5", [0x0d9f] = "\x8c\x67", [0x0da0] = "\xb3\xe8",
  [0x0da1] = "\xc9\xf3", [0x0da2] = "\xbf\xcd", [0x0da3] = "\xd0\xfb",
  [0x0da4] = "\xca\xd2", [0x0da5] = "\xe5\xb6", [0x0da6] = "\xbb\xc2",
  [0x0da7] = "\x8c\x68", [0x0da8] = "\x8c\x69", [0x0da9] = "\x8c\x6a",
  [0x0daa] = "\xcf\xdc", [0x0dab] = "\xb9\xac", [0x0dac] = "\x8c\x6b",
  [0x0dad] = "\x8c\x6c", [0x0dae] = "\x8c\x6d", [0x0daf] = "\x8c\x6e",
  [0x0db0] = "\xd4\xd7", [0x0db1] = "\x8c\x6f", [0x0db2] = "\x8c\x70",
  [0x0db3] = "\xba\xa6", [0x0db4] = "\xd1\xe7", [0x0db5] = "\xcf\xfc",
  [0x0db6] = "\xbc\xd2", [0x0db7] = "\x8c\x71", [0x0db8] = "\xe5\xb7",
  [0x0db9] = "\xc8\xdd", [0x0dba] = "\x8c\x72", [0x0dbb] = "\x8c\x73",
  [0x0dbc] = "\x8c\x74", [0x0dbd] = "\xbf\xed", [0x0dbe] = "\xb1\xf6",
  [0x0dbf] = "\xcb\xde", [0x0dc0] = "\x8c\x75", [0x0dc1] = "\x8c\x76",
  [0x0dc2] = "\xbc\xc5", [0x0dc3] = "\x8c\x77", [0x0dc4] = "\xbc\xc4",
  [0x0dc5] = "\xd2\xfa", [0x0dc6] = "\xc3\xdc", [0x0dc7] = "\xbf\xdc",
  [0x0dc8] = "\x8c\x78", [0x0dc9] = "\x8c\x79", [0x0dca] = "\x8c\x7a",
  [0x0dcb] = "\x8c\x7b", [0x0dcc] = "\xb8\xbb", [0x0dcd] = "\x8c\x7c",
  [0x0dce] = "\x8c\x7d", [0x0dcf] = "\x8c\x7e", [0x0dd0] = "\xc3\xc2",
  [0x0dd1] = "\x8c\x80", [0x0dd2] = "\xba\xae", [0x0dd3] = "\xd4\xa2",
  [0x0dd4] = "\x8c\x81", [0x0dd5] = "\x8c\x82", [0x0dd6] = "\x8c\x83",
  [0x0dd7] = "\x8c\x84", [0x0dd8] = "\x8c\x85", [0x0dd9] = "\x8c\x86",
  [0x0dda] = "\x8c\x87", [0x0ddb] = "\x8c\x88", [0x0ddc] = "\x8c\x89",
  [0x0ddd] = "\xc7\xde", [0x0dde] = "\xc4\xaf", [0x0ddf] = "\xb2\xec",
  [0x0de0] = "\x8c\x8a", [0x0de1] = "\xb9\xd1", [0x0de2] = "\x8c\x8b",
  [0x0de3] = "\x8c\x8c", [0x0de4] = "\xe5\xbb", [0x0de5] = "\xc1\xc8",
  [0x0de6] = "\x8c\x8d", [0x0de7] = "\x8c\x8e", [0x0de8] = "\xd5\xaf",
  [0x0de9] = "\x8c\x8f", [0x0dea] = "\x8c\x90", [0x0deb] = "\x8c\x91",
  [0x0dec] = "\x8c\x92", [0x0ded] = "\x8c\x93", [0x0dee] = "\xe5\xbc",
  [0x0def] = "\x8c\x94", [0x0df0] = "\xe5\xbe", [0x0df1] = "\x8c\x95",
  [0x0df2] = "\x8c\x96", [0x0df3] = "\x8c\x97", [0x0df4] = "\x8c\x98",
  [0x0df5] = "\x8c\x99", [0x0df6] = "\x8c\x9a", [0x0df7] = "\x8c\x9b",
  [0x0df8] = "\xb4\xe7", [0x0df9] = "\xb6\xd4", [0x0dfa] = "\xcb\xc2",
  [0x0dfb] = "\xd1\xb0", [0x0dfc] = "\xb5\xbc", [0x0dfd] = "\x8c\x9c",
  [0x0dfe] = "\x8c\x9d", [0x0dff] = "\xca\xd9", [0x0e00] = "\x8c\x9e",
  [0x0e01] = "\xb7\xe2", [0x0e02] = "\x8c\x9f", [0x0e03] = "\x8c\xa0",
  [0x0e04] = "\xc9\xe4", [0x0e05] = "\x8c\xa1", [0x0e06] = "\xbd\xab",
  [0x0e07] = "\x8c\xa2", [0x0e08] = "\x8c\xa3", [0x0e09] = "\xce\xbe",
  [0x0e0a] = "\xd7\xf0", [0x0e0b] = "\x8c\xa4", [0x0e0c] = "\x8c\xa5",
  [0x0e0d] = "\x8c\xa6", [0x0e0e] = "\x8c\xa7", [0x0e0f] = "\xd0\xa1",
  [0x0e10] = "\x8c\xa8", [0x0e11] = "\xc9\xd9", [0x0e12] = "\x8c\xa9",
  [0x0e13] = "\x8c\xaa", [0x0e14] = "\xb6\xfb", [0x0e15] = "\xe6\xd8",
  [0x0e16] = "\xbc\xe2", [0x0e17] = "\x8c\xab", [0x0e18] = "\xb3\xbe",
  [0x0e19] = "\x8c\xac", [0x0e1a] = "\xc9\xd0", [0x0e1b] = "\x8c\xad",
  [0x0e1c] = "\xe6\xd9", [0x0e1d] = "\xb3\xa2", [0x0e1e] = "\x8c\xae",
  [0x0e1f] = "\x8c\xaf", [0x0e20] = "\x8c\xb0", [0x0e21] = "\x8c\xb1",
  [0x0e22] = "\xde\xcc", [0x0e23] = "\x8c\xb2", [0x0e24] = "\xd3\xc8",
  [0x0e25] = "\xde\xcd", [0x0e26] = "\x8c\xb3", [0x0e27] = "\xd2\xa2",
  [0x0e28] = "\x8c\xb4", [0x0e29] = "\x8c\xb5", [0x0e2a] = "\x8c\xb6",
  [0x0e2b] = "\x8c\xb7", [0x0e2c] = "\xde\xce", [0x0e2d] = "\x8c\xb8",
  [0x0e2e] = "\x8c\xb9", [0x0e2f] = "\x8c\xba", [0x0e30] = "\x8c\xbb",
  [0x0e31] = "\xbe\xcd", [0x0e32] = "\x8c\xbc", [0x0e33] = "\x8c\xbd",
  [0x0e34] = "\xde\xcf", [0x0e35] = "\x8c\xbe", [0x0e36] = "\x8c\xbf",
  [0x0e37] = "\x8c\xc0", [0x0e38] = "\xca\xac", [0x0e39] = "\xd2\xfc",
  [0x0e3a] = "\xb3\xdf", [0x0e3b] = "\xe5\xea", [0x0e3c] = "\xc4\xe1",
  [0x0e3d] = "\xbe\xa1", [0x0e3e] = "\xce\xb2", [0x0e3f] = "\xc4\xf2",
  [0x0e40] = "\xbe\xd6", [0x0e41] = "\xc6\xa8", [0x0e42] = "\xb2\xe3",
  [0x0e43] = "\x8c\xc1", [0x0e44] = "\x8c\xc2", [0x0e45] = "\xbe\xd3",
  [0x0e46] = "\x8c\xc3", [0x0e47] = "\x8c\xc4", [0x0e48] = "\xc7\xfc",
  [0x0e49] = "\xcc\xeb", [0x0e4a] = "\xbd\xec", [0x0e4b] = "\xce\xdd",
  [0x0e4c] = "\x8c\xc5", [0x0e4d] = "\x8c\xc6", [0x0e4e] = "\xca\xba",
  [0x0e4f] = "\xc6\xc1", [0x0e50] = "\xe5\xec", [0x0e51] = "\xd0\xbc",
  [0x0e52] = "\x8c\xc7", [0x0e53] = "\x8c\xc8", [0x0e54] = "\x8c\xc9",
  [0x0e55] = "\xd5\xb9", [0x0e56] = "\x8c\xca", [0x0e57] = "\x8c\xcb",
  [0x0e58] = "\x8c\xcc", [0x0e59] = "\xe5\xed", [0x0e5a] = "\x8c\xcd",
  [0x0e5b] = "\x8c\xce", [0x0e5c] = "\x8c\xcf", [0x0e5d] = "\x8c\xd0",
  [0x0e5e] = "\xca\xf4", [0x0e5f] = "\x8c\xd1", [0x0e60] = "\xcd\xc0",
  [0x0e61] = "\xc2\xc5", [0x0e62] = "\x8c\xd2", [0x0e63] = "\xe5\xef",
  [0x0e64] = "\x8c\xd3", [0x0e65] = "\xc2\xc4", [0x0e66] = "\xe5\xf0",
  [0x0e67] = "\x8c\xd4", [0x0e68] = "\x8c\xd5", [0x0e69] = "\x8c\xd6",
  [0x0e6a] = "\x8c\xd7", [0x0e6b] = "\x8c\xd8", [0x0e6c] = "\x8c\xd9",
  [0x0e6d] = "\x8c\xda", [0x0e6e] = "\xe5\xf8", [0x0e6f] = "\xcd\xcd",
  [0x0e70] = "\x8c\xdb", [0x0e71] = "\xc9\xbd", [0x0e72] = "\x8c\xdc",
  [0x0e73] = "\x8c\xdd", [0x0e74] = "\x8c\xde", [0x0e75] = "\x8c\xdf",
  [0x0e76] = "\x8c\xe0", [0x0e77] = "\x8c\xe1", [0x0e78] = "\x8c\xe2",
  [0x0e79] = "\xd2\xd9", [0x0e7a] = "\xe1\xa8", [0x0e7b] = "\x8c\xe3",
  [0x0e7c] = "\x8c\xe4", [0x0e7d] = "\x8c\xe5", [0x0e7e] = "\x8c\xe6",
  [0x0e7f] = "\xd3\xec", [0x0e80] = "\x8c\xe7", [0x0e81] = "\xcb\xea",
  [0x0e82] = "\xc6\xf1", [0x0e83] = "\x8c\xe8", [0x0e84] = "\x8c\xe9",
  [0x0e85] = "\x8c\xea", [0x0e86] = "\x8c\xeb", [0x0e87] = "\x8c\xec",
  [0x0e88] = "\xe1\xac", [0x0e89] = "\x8c\xed", [0x0e8a] = "\x8c\xee",
  [0x0e8b] = "\x8c\xef", [0x0e8c] = "\xe1\xa7", [0x0e8d] = "\xe1\xa9",
  [0x0e8e] = "\x8c\xf0", [0x0e8f] = "\x8c\xf1", [0x0e90] = "\xe1\xaa",
  [0x0e91] = "\xe1\xaf", [0x0e92] = "\x8c\xf2", [0x0e93] = "\x8c\xf3",
  [0x0e94] = "\xb2\xed", [0x0e95] = "\x8c\xf4", [0x0e96] = "\xe1\xab",
  [0x0e97] = "\xb8\xda", [0x0e98] = "\xe1\xad", [0x0e99] = "\xe1\xae",
  [0x0e9a] = "\xe1\xb0", [0x0e9b] = "\xb5\xba", [0x0e9c] = "\xe1\xb1",
  [0x0e9d] = "\x8c\xf5", [0x0e9e] = "\x8c\xf6", [0x0e9f] = "\x8c\xf7",
  [0x0ea0] = "\x8c\xf8", [0x0ea1] = "\x8c\xf9", [0x0ea2] = "\xe1\xb3",
  [0x0ea3] = "\xe1\xb8", [0x0ea4] = "\x8c\xfa", [0x0ea5] = "\x8c\xfb",
  [0x0ea6] = "\x8c\xfc", [0x0ea7] = "\x8c\xfd", [0x0ea8] = "\x8c\xfe",
  [0x0ea9] = "\xd1\xd2", [0x0eaa] = "\x8d\x40", [0x0eab] = "\xe1\xb6",
  [0x0eac] = "\xe1\xb5", [0x0ead] = "\xc1\xeb", [0x0eae] = "\x8d\x41",
  [0x0eaf] = "\x8d\x42", [0x0eb0] = "\x8d\x43", [0x0eb1] = "\xe1\xb7",
  [0x0eb2] = "\x8d\x44", [0x0eb3] = "\xd4\xc0", [0x0eb4] = "\x8d\x45",
  [0x0eb5] = "\xe1\xb2", [0x0eb6] = "\x8d\x46", [0x0eb7] = "\xe1\xba",
  [0x0eb8] = "\xb0\xb6", [0x0eb9] = "\x8d\x47", [0x0eba] = "\x8d\x48",
  [0x0ebb] = "\x8d\x49", [0x0ebc] = "\x8d\x4a", [0x0ebd] = "\xe1\xb4",
  [0x0ebe] = "\x8d\x4b", [0x0ebf] = "\xbf\xf9", [0x0ec0] = "\x8d\x4c",
  [0x0ec1] = "\xe1\xb9", [0x0ec2] = "\x8d\x4d", [0x0ec3] = "\x8d\x4e",
  [0x0ec4] = "\xe1\xbb", [0x0ec5] = "\x8d\x4f", [0x0ec6] = "\x8d\x50",
  [0x0ec7] = "\x8d\x51", [0x0ec8] = "\x8d\x52", [0x0ec9] = "\x8d\x53",
  [0x0eca] = "\x8d\x54", [0x0ecb] = "\xe1\xbe", [0x0ecc] = "\x8d\x55",
  [0x0ecd] = "\x8d\x56", [0x0ece] = "\x8d\x57", [0x0ecf] = "\x8d\x58",
  [0x0ed0] = "\x8d\x59", [0x0ed1] = "\x8d\x5a", [0x0ed2] = "\xe1\xbc",
  [0x0ed3] = "\x8d\x5b", [0x0ed4] = "\x8d\x5c", [0x0ed5] = "\x8d\x5d",
  [0x0ed6] = "\x8d\x5e", [0x0ed7] = "\x8d\x5f", [0x0ed8] = "\x8d\x60",
  [0x0ed9] = "\xd6\xc5", [0x0eda] = "\x8d\x61", [0x0edb] = "\x8d\x62",
  [0x0edc] = "\x8d\x63", [0x0edd] = "\x8d\x64", [0x0ede] = "\x8d\x65",
  [0x0edf] = "\x8d\x66", [0x0ee0] = "\x8d\x67", [0x0ee1] = "\xcf\xbf",
  [0x0ee2] = "\x8d\x68", [0x0ee3] = "\x8d\x69", [0x0ee4] = "\xe1\xbd",
  [0x0ee5] = "\xe1\xbf", [0x0ee6] = "\xc2\xcd", [0x0ee7] = "\x8d\x6a",
  [0x0ee8] = "\xb6\xeb", [0x0ee9] = "\x8d\x6b", [0x0eea] = "\xd3\xf8",
  [0x0eeb] = "\x8d\x6c", [0x0eec] = "\x8d\x6d", [0x0eed] = "\xc7\xcd",
  [0x0eee] = "\x8d\x6e", [0x0eef] = "\x8d\x6f", [0x0ef0] = "\xb7\xe5",
  [0x0ef1] = "\x8d\x70", [0x0ef2] = "\x8d\x71", [0x0ef3] = "\x8d\x72",
  [0x0ef4] = "\x8d\x73", [0x0ef5] = "\x8d\x74", [0x0ef6] = "\x8d\x75",
  [0x0ef7] = "\x8d\x76", [0x0ef8] = "\x8d\x77", [0x0ef9] = "\x8d\x78",
  [0x0efa] = "\x8d\x79", [0x0efb] = "\xbe\xfe", [0x0efc] = "\x8d\x7a",
  [0x0efd] = "\x8d\x7b", [0x0efe] = "\x8d\x7c", [0x0eff] = "\x8d\x7d",
  [0x0f00] = "\x8d\x7e", [0x0f01] = "\x8d\x80", [0x0f02] = "\xe1\xc0",
  [0x0f03] = "\xe1\xc1", [0x0f04] = "\x8d\x81", [0x0f05] = "\x8d\x82",
  [0x0f06] = "\xe1\xc7", [0x0f07] = "\xb3\xe7", [0x0f08] = "\x8d\x83",
  [0x0f09] = "\x8d\x84", [0x0f0a] = "\x8d\x85", [0x0f0b] = "\x8d\x86",
  [0x0f0c] = "\x8d\x87", [0x0f0d] = "\x8d\x88", [0x0f0e] = "\xc6\xe9",
  [0x0f0f] = "\x8d\x89", [0x0f10] = "\x8d\x8a", [0x0f11] = "\x8d\x8b",
  [0x0f12] = "\x8d\x8c", [0x0f13] = "\x8d\x8d", [0x0f14] = "\xb4\xde",
  [0x0f15] = "\x8d\x8e", [0x0f16] = "\xd1\xc2", [0x0f17] = "\x8d\x8f",
  [0x0f18] = "\x8d\x90", [0x0f19] = "\x8d\x91", [0x0f1a] = "\x8d\x92",
  [0x0f1b] = "\xe1\xc8", [0x0f1c] = "\x8d\x93", [0x0f1d] = "\x8d\x94",
  [0x0f1e] = "\xe1\xc6", [0x0f1f] = "\x8d\x95", [0x0f20] = "\x8d\x96",
  [0x0f21] = "\x8d\x97", [0x0f22] = "\x8d\x98", [0x0f23] = "\x8d\x99",
  [0x0f24] = "\xe1\xc5", [0x0f25] = "\x8d\x9a", [0x0f26] = "\xe1\xc3",
  [0x0f27] = "\xe1\xc2", [0x0f28] = "\x8d\x9b", [0x0f29] = "\xb1\xc0",
  [0x0f2a] = "\x8d\x9c", [0x0f2b] = "\x8d\x9d", [0x0f2c] = "\x8d\x9e",
  [0x0f2d] = "\xd5\xb8", [0x0f2e] = "\xe1\xc4", [0x0f2f] = "\x8d\x9f",
  [0x0f30] = "\x8d\xa0", [0x0f31] = "\x8d\xa1", [0x0f32] = "\x8d\xa2",
  [0x0f33] = "\x8d\xa3", [0x0f34] = "\xe1\xcb", [0x0f35] = "\x8d\xa4",
  [0x0f36] = "\x8d\xa5", [0x0f37] = "\x8d\xa6", [0x0f38] = "\x8d\xa7",
  [0x0f39] = "\x8d\xa8", [0x0f3a] = "\x8d\xa9", [0x0f3b] = "\x8d\xaa",
  [0x0f3c] = "\x8d\xab", [0x0f3d] = "\xe1\xcc", [0x0f3e] = "\xe1\xca",
  [0x0f3f] = "\x8d\xac", [0x0f40] = "\x8d\xad", [0x0f41] = "\x8d\xae",
  [0x0f42] = "\x8d\xaf", [0x0f43] = "\x8d\xb0", [0x0f44] = "\x8d\xb1",
  [0x0f45] = "\x8d\xb2", [0x0f46] = "\x8d\xb3", [0x0f47] = "\xef\xfa",
  [0x0f48] = "\x8d\xb4", [0x0f49] = "\x8d\xb5", [0x0f4a] = "\xe1\xd3",
  [0x0f4b] = "\xe1\xd2", [0x0f4c] = "\xc7\xb6", [0x0f4d] = "\x8d\xb6",
  [0x0f4e] = "\x8d\xb7", [0x0f4f] = "\x8d\xb8", [0x0f50] = "\x8d\xb9",
  [0x0f51] = "\x8d\xba", [0x0f52] = "\x8d\xbb", [0x0f53] = "\x8d\xbc",
  [0x0f54] = "\x8d\xbd", [0x0f55] = "\x8d\xbe", [0x0f56] = "\x8d\xbf",
  [0x0f57] = "\x8d\xc0", [0x0f58] = "\xe1\xc9", [0x0f59] = "\x8d\xc1",
  [0x0f5a] = "\x8d\xc2", [0x0f5b] = "\xe1\xce", [0x0f5c] = "\x8d\xc3",
  [0x0f5d] = "\xe1\xd0", [0x0f5e] = "\x8d\xc4", [0x0f5f] = "\x8d\xc5",
  [0x0f60] = "\x8d\xc6", [0x0f61] = "\x8d\xc7", [0x0f62] = "\x8d\xc8",
  [0x0f63] = "\x8d\xc9", [0x0f64] = "\x8d\xca", [0x0f65] = "\x8d\xcb",
  [0x0f66] = "\x8d\xcc", [0x0f67] = "\x8d\xcd", [0x0f68] = "\x8d\xce",
  [0x0f69] = "\xe1\xd4", [0x0f6a] = "\x8d\xcf", [0x0f6b] = "\xe1\xd1",
  [0x0f6c] = "\xe1\xcd", [0x0f6d] = "\x8d\xd0", [0x0f6e] = "\x8d\xd1",
  [0x0f6f] = "\xe1\xcf", [0x0f70] = "\x8d\xd2", [0x0f71] = "\x8d\xd3",
  [0x0f72] = "\x8d\xd4", [0x0f73] = "\x8d\xd5", [0x0f74] = "\xe1\xd5",
  [0x0f75] = "\x8d\xd6", [0x0f76] = "\x8d\xd7", [0x0f77] = "\x8d\xd8",
  [0x0f78] = "\x8d\xd9", [0x0f79] = "\x8d\xda", [0x0f7a] = "\x8d\xdb",
  [0x0f7b] = "\x8d\xdc", [0x0f7c] = "\x8d\xdd", [0x0f7d] = "\x8d\xde",
  [0x0f7e] = "\x8d\xdf", [0x0f7f] = "\x8d\xe0", [0x0f80] = "\x8d\xe1",
  [0x0f81] = "\x8d\xe2", [0x0f82] = "\xe1\xd6", [0x0f83] = "\x8d\xe3",
  [0x0f84] = "\x8d\xe4", [0x0f85] = "\x8d\xe5", [0x0f86] = "\x8d\xe6",
  [0x0f87] = "\x8d\xe7", [0x0f88] = "\x8d\xe8", [0x0f89] = "\x8d\xe9",
  [0x0f8a] = "\x8d\xea", [0x0f8b] = "\x8d\xeb", [0x0f8c] = "\x8d\xec",
  [0x0f8d] = "\x8d\xed", [0x0f8e] = "\x8d\xee", [0x0f8f] = "\x8d\xef",
  [0x0f90] = "\x8d\xf0", [0x0f91] = "\x8d\xf1", [0x0f92] = "\x8d\xf2",
  [0x0f93] = "\x8d\xf3", [0x0f94] = "\x8d\xf4", [0x0f95] = "\x8d\xf5",
  [0x0f96] = "\x8d\xf6", [0x0f97] = "\x8d\xf7", [0x0f98] = "\x8d\xf8",
  [0x0f99] = "\xe1\xd7", [0x0f9a] = "\x8d\xf9", [0x0f9b] = "\x8d\xfa",
  [0x0f9c] = "\x8d\xfb", [0x0f9d] = "\xe1\xd8", [0x0f9e] = "\x8d\xfc",
  [0x0f9f] = "\x8d\xfd", [0x0fa0] = "\x8d\xfe", [0x0fa1] = "\x8e\x40",
  [0x0fa2] = "\x8e\x41", [0x0fa3] = "\x8e\x42", [0x0fa4] = "\x8e\x43",
  [0x0fa5] = "\x8e\x44", [0x0fa6] = "\x8e\x45", [0x0fa7] = "\x8e\x46",
  [0x0fa8] = "\x8e\x47", [0x0fa9] = "\x8e\x48", [0x0faa] = "\x8e\x49",
  [0x0fab] = "\x8e\x4a", [0x0fac] = "\x8e\x4b", [0x0fad] = "\x8e\x4c",
  [0x0fae] = "\x8e\x4d", [0x0faf] = "\x8e\x4e", [0x0fb0] = "\x8e\x4f",
  [0x0fb1] = "\x8e\x50", [0x0fb2] = "\x8e\x51", [0x0fb3] = "\x8e\x52",
  [0x0fb4] = "\x8e\x53", [0x0fb5] = "\x8e\x54", [0x0fb6] = "\x8e\x55",
  [0x0fb7] = "\xe1\xda", [0x0fb8] = "\x8e\x56", [0x0fb9] = "\x8e\x57",
  [0x0fba] = "\x8e\x58", [0x0fbb] = "\x8e\x59", [0x0fbc] = "\x8e\x5a",
  [0x0fbd] = "\x8e\x5b", [0x0fbe] = "\x8e\x5c", [0x0fbf] = "\x8e\x5d",
  [0x0fc0] = "\x8e\x5e", [0x0fc1] = "\x8e\x5f", [0x0fc2] = "\x8e\x60",
  [0x0fc3] = "\x8e\x61", [0x0fc4] = "\x8e\x62", [0x0fc5] = "\xe1\xdb",
  [0x0fc6] = "\x8e\x63", [0x0fc7] = "\x8e\x64", [0x0fc8] = "\x8e\x65",
  [0x0fc9] = "\x8e\x66", [0x0fca] = "\x8e\x67", [0x0fcb] = "\x8e\x68",
  [0x0fcc] = "\x8e\x69", [0x0fcd] = "\xce\xa1", [0x0fce] = "\x8e\x6a",
  [0x0fcf] = "\x8e\x6b", [0x0fd0] = "\x8e\x6c", [0x0fd1] = "\x8e\x6d",
  [0x0fd2] = "\x8e\x6e", [0x0fd3] = "\x8e\x6f", [0x0fd4] = "\x8e\x70",
  [0x0fd5] = "\x8e\x71", [0x0fd6] = "\x8e\x72", [0x0fd7] = "\x8e\x73",
  [0x0fd8] = "\x8e\x74", [0x0fd9] = "\x8e\x75", [0x0fda] = "\x8e\x76",
  [0x0fdb] = "\xe7\xdd", [0x0fdc] = "\x8e\x77", [0x0fdd] = "\xb4\xa8",
  [0x0fde] = "\xd6\xdd", [0x0fdf] = "\x8e\x78", [0x0fe0] = "\x8e\x79",
  [0x0fe1] = "\xd1\xb2", [0x0fe2] = "\xb3\xb2", [0x0fe3] = "\x8e\x7a",
  [0x0fe4] = "\x8e\x7b", [0x0fe5] = "\xb9\xa4", [0x0fe6] = "\xd7\xf3",
  [0x0fe7] = "\xc7\xc9", [0x0fe8] = "\xbe\xde", [0x0fe9] = "\xb9\xae",
  [0x0fea] = "\x8e\x7c", [0x0feb] = "\xce\xd7", [0x0fec] = "\x8e\x7d",
  [0x0fed] = "\x8e\x7e", [0x0fee] = "\xb2\xee", [0x0fef] = "\xdb\xcf",
  [0x0ff0] = "\x8e\x80", [0x0ff1] = "\xbc\xba", [0x0ff2] = "\xd2\xd1",
  [0x0ff3] = "\xcb\xc8", [0x0ff4] = "\xb0\xcd", [0x0ff5] = "\x8e\x81",
  [0x0ff6] = "\x8e\x82", [0x0ff7] = "\xcf\xef", [0x0ff8] = "\x8e\x83",
  [0x0ff9] = "\x8e\x84", [0x0ffa] = "\x8e\x85", [0x0ffb] = "\x8e\x86",
  [0x0ffc] = "\x8e\x87", [0x0ffd] = "\xd9\xe3", [0x0ffe] = "\xbd\xed",
  [0x0fff] = "\x8e\x88", [0x1000] = "\x8e\x89", [0x1001] = "\xb1\xd2",
  [0x1002] = "\xca\xd0", [0x1003] = "\xb2\xbc", [0x1004] = "\x8e\x8a",
  [0x1005] = "\xcb\xa7", [0x1006] = "\xb7\xab", [0x1007] = "\x8e\x8b",
  [0x1008] = "\xca\xa6", [0x1009] = "\x8e\x8c", [0x100a] = "\x8e\x8d",
  [0x100b] = "\x8e\x8e", [0x100c] = "\xcf\xa3", [0x100d] = "\x8e\x8f",
  [0x100e] = "\x8e\x90", [0x100f] = "\xe0\xf8", [0x1010] = "\xd5\xca",
  [0x1011] = "\xe0\xfb", [0x1012] = "\x8e\x91", [0x1013] = "\x8e\x92",
  [0x1014] = "\xe0\xfa", [0x1015] = "\xc5\xc1", [0x1016] = "\xcc\xfb",
  [0x1017] = "\x8e\x93", [0x1018] = "\xc1\xb1", [0x1019] = "\xe0\xf9",
  [0x101a] = "\xd6\xe3", [0x101b] = "\xb2\xaf", [0x101c] = "\xd6\xc4",
  [0x101d] = "\xb5\xdb", [0x101e] = "\x8e\x94", [0x101f] = "\x8e\x95",
  [0x1020] = "\x8e\x96", [0x1021] = "\x8e\x97", [0x1022] = "\x8e\x98",
  [0x1023] = "\x8e\x99", [0x1024] = "\x8e\x9a", [0x1025] = "\x8e\x9b",
  [0x1026] = "\xb4\xf8", [0x1027] = "\xd6\xa1", [0x1028] = "\x8e\x9c",
  [0x1029] = "\x8e\x9d", [0x102a] = "\x8e\x9e", [0x102b] = "\x8e\x9f",
  [0x102c] = "\x8e\xa0", [0x102d] = "\xcf\xaf", [0x102e] = "\xb0\xef",
  [0x102f] = "\x8e\xa1", [0x1030] = "\x8e\xa2", [0x1031] = "\xe0\xfc",
  [0x1032] = "\x8e\xa3", [0x1033] = "\x8e\xa4", [0x1034] = "\x8e\xa5",
  [0x1035] = "\x8e\xa6", [0x1036] = "\x8e\xa7", [0x1037] = "\xe1\xa1",
  [0x1038] = "\xb3\xa3", [0x1039] = "\x8e\xa8", [0x103a] = "\x8e\xa9",
  [0x103b] = "\xe0\xfd", [0x103c] = "\xe0\xfe", [0x103d] = "\xc3\xb1",
  [0x103e] = "\x8e\xaa", [0x103f] = "\x8e\xab", [0x1040] = "\x8e\xac",
  [0x1041] = "\x8e\xad", [0x1042] = "\xc3\xdd", [0x1043] = "\x8e\xae",
  [0x1044] = "\xe1\xa2", [0x1045] = "\xb7\xf9", [0x1046] = "\x8e\xaf",
  [0x1047] = "\x8e\xb0", [0x1048] = "\x8e\xb1", [0x1049] = "\x8e\xb2",
  [0x104a] = "\x8e\xb3", [0x104b] = "\x8e\xb4", [0x104c] = "\xbb\xcf",
  [0x104d] = "\x8e\xb5", [0x104e] = "\x8e\xb6", [0x104f] = "\x8e\xb7",
  [0x1050] = "\x8e\xb8", [0x1051] = "\x8e\xb9", [0x1052] = "\x8e\xba",
  [0x1053] = "\x8e\xbb", [0x1054] = "\xe1\xa3", [0x1055] = "\xc4\xbb",
  [0x1056] = "\x8e\xbc", [0x1057] = "\x8e\xbd", [0x1058] = "\x8e\xbe",
  [0x1059] = "\x8e\xbf", [0x105a] = "\x8e\xc0", [0x105b] = "\xe1\xa4",
  [0x105c] = "\x8e\xc1", [0x105d] = "\x8e\xc2", [0x105e] = "\xe1\xa5",
  [0x105f] = "\x8e\xc3", [0x1060] = "\x8e\xc4", [0x1061] = "\xe1\xa6",
  [0x1062] = "\xb4\xb1", [0x1063] = "\x8e\xc5", [0x1064] = "\x8e\xc6",
  [0x1065] = "\x8e\xc7", [0x1066] = "\x8e\xc8", [0x1067] = "\x8e\xc9",
  [0x1068] = "\x8e\xca", [0x1069] = "\x8e\xcb", [0x106a] = "\x8e\xcc",
  [0x106b] = "\x8e\xcd", [0x106c] = "\x8e\xce", [0x106d] = "\x8e\xcf",
  [0x106e] = "\x8e\xd0", [0x106f] = "\x8e\xd1", [0x1070] = "\x8e\xd2",
  [0x1071] = "\x8e\xd3", [0x1072] = "\xb8\xc9", [0x1073] = "\xc6\xbd",
  [0x1074] = "\xc4\xea", [0x1075] = "\x8e\xd4", [0x1076] = "\xb2\xa2",
  [0x1077] = "\x8e\xd5", [0x1078] = "\xd0\xd2", [0x1079] = "\x8e\xd6",
  [0x107a] = "\xe7\xdb", [0x107b] = "\xbb\xc3", [0x107c] = "\xd3\xd7",
  [0x107d] = "\xd3\xc4", [0x107e] = "\x8e\xd7", [0x107f] = "\xb9\xe3",
  [0x1080] = "\xe2\xcf", [0x1081] = "\x8e\xd8", [0x1082] = "\x8e\xd9",
  [0x1083] = "\x8e\xda", [0x1084] = "\xd7\xaf", [0x1085] = "\x8e\xdb",
  [0x1086] = "\xc7\xec", [0x1087] = "\xb1\xd3", [0x1088] = "\x8e\xdc",
  [0x1089] = "\x8e\xdd", [0x108a] = "\xb4\xb2", [0x108b] = "\xe2\xd1",
  [0x108c] = "\x8e\xde", [0x108d] = "\x8e\xdf", [0x108e] = "\x8e\xe0",
  [0x108f] = "\xd0\xf2", [0x1090] = "\xc2\xae", [0x1091] = "\xe2\xd0",
  [0x1092] = "\x8e\xe1", [0x1093] = "\xbf\xe2", [0x1094] = "\xd3\xa6",
  [0x1095] = "\xb5\xd7", [0x1096] = "\xe2\xd2", [0x1097] = "\xb5\xea",
  [0x1098] = "\x8e\xe2", [0x1099] = "\xc3\xed", [0x109a] = "\xb8\xfd",
  [0x109b] = "\x8e\xe3", [0x109c] = "\xb8\xae", [0x109d] = "\x8e\xe4",
  [0x109e] = "\xc5\xd3", [0x109f] = "\xb7\xcf", [0x10a0] = "\xe2\xd4",
  [0x10a1] = "\x8e\xe5", [0x10a2] = "\x8e\xe6", [0x10a3] = "\x8e\xe7",
  [0x10a4] = "\x8e\xe8", [0x10a5] = "\xe2\xd3", [0x10a6] = "\xb6\xc8",
  [0x10a7] = "\xd7\xf9", [0x10a8] = "\x8e\xe9", [0x10a9] = "\x8e\xea",
  [0x10aa] = "\x8e\xeb", [0x10ab] = "\x8e\xec", [0x10ac] = "\x8e\xed",
  [0x10ad] = "\xcd\xa5", [0x10ae] = "\x8e\xee", [0x10af] = "\x8e\xef",
  [0x10b0] = "\x8e\xf0", [0x10b1] = "\x8e\xf1", [0x10b2] = "\x8e\xf2",
  [0x10b3] = "\xe2\xd8", [0x10b4] = "\x8e\xf3", [0x10b5] = "\xe2\xd6",
  [0x10b6] = "\xca\xfc", [0x10b7] = "\xbf\xb5", [0x10b8] = "\xd3\xb9",
  [0x10b9] = "\xe2\xd5", [0x10ba] = "\x8e\xf4", [0x10bb] = "\x8e\xf5",
  [0x10bc] = "\x8e\xf6", [0x10bd] = "\x8e\xf7", [0x10be] = "\xe2\xd7",
  [0x10bf] = "\x8e\xf8", [0x10c0] = "\x8e\xf9", [0x10c1] = "\x8e\xfa",
  [0x10c2] = "\x8e\xfb", [0x10c3] = "\x8e\xfc", [0x10c4] = "\x8e\xfd",
  [0x10c5] = "\x8e\xfe", [0x10c6] = "\x8f\x40", [0x10c7] = "\x8f\x41",
  [0x10c8] = "\x8f\x42", [0x10c9] = "\xc1\xae", [0x10ca] = "\xc0\xc8",
  [0x10cb] = "\x8f\x43", [0x10cc] = "\x8f\x44", [0x10cd] = "\x8f\x45",
  [0x10ce] = "\x8f\x46", [0x10cf] = "\x8f\x47", [0x10d0] = "\x8f\x48",
  [0x10d1] = "\xe2\xdb", [0x10d2] = "\xe2\xda", [0x10d3] = "\xc0\xaa",
  [0x10d4] = "\x8f\x49", [0x10d5] = "\x8f\x4a", [0x10d6] = "\xc1\xce",
  [0x10d7] = "\x8f\x4b", [0x10d8] = "\x8f\x4c", [0x10d9] = "\x8f\x4d",
  [0x10da] = "\x8f\x4e", [0x10db] = "\xe2\xdc", [0x10dc] = "\x8f\x4f",
  [0x10dd] = "\x8f\x50", [0x10de] = "\x8f\x51", [0x10df] = "\x8f\x52",
  [0x10e0] = "\x8f\x53", [0x10e1] = "\x8f\x54", [0x10e2] = "\x8f\x55",
  [0x10e3] = "\x8f\x56", [0x10e4] = "\x8f\x57", [0x10e5] = "\x8f\x58",
  [0x10e6] = "\x8f\x59", [0x10e7] = "\x8f\x5a", [0x10e8] = "\xe2\xdd",
  [0x10e9] = "\x8f\x5b", [0x10ea] = "\xe2\xde", [0x10eb] = "\x8f\x5c",
  [0x10ec] = "\x8f\x5d", [0x10ed] = "\x8f\x5e", [0x10ee] = "\x8f\x5f",
  [0x10ef] = "\x8f\x60", [0x10f0] = "\x8f\x61", [0x10f1] = "\x8f\x62",
  [0x10f2] = "\x8f\x63", [0x10f3] = "\x8f\x64", [0x10f4] = "\xdb\xc8",
  [0x10f5] = "\x8f\x65", [0x10f6] = "\xd1\xd3", [0x10f7] = "\xcd\xa2",
  [0x10f8] = "\x8f\x66", [0x10f9] = "\x8f\x67", [0x10fa] = "\xbd\xa8",
  [0x10fb] = "\x8f\x68", [0x10fc] = "\x8f\x69", [0x10fd] = "\x8f\x6a",
  [0x10fe] = "\xde\xc3", [0x10ff] = "\xd8\xa5", [0x1100] = "\xbf\xaa",
  [0x1101] = "\xdb\xcd", [0x1102] = "\xd2\xec", [0x1103] = "\xc6\xfa",
  [0x1104] = "\xc5\xaa", [0x1105] = "\x8f\x6b", [0x1106] = "\x8f\x6c",
  [0x1107] = "\x8f\x6d", [0x1108] = "\xde\xc4", [0x1109] = "\x8f\x6e",
  [0x110a] = "\xb1\xd7", [0x110b] = "\xdf\xae", [0x110c] = "\x8f\x6f",
  [0x110d] = "\x8f\x70", [0x110e] = "\x8f\x71", [0x110f] = "\xca\xbd",
  [0x1110] = "\x8f\x72", [0x1111] = "\xdf\xb1", [0x1112] = "\x8f\x73",
  [0x1113] = "\xb9\xad", [0x1114] = "\x8f\x74", [0x1115] = "\xd2\xfd",
  [0x1116] = "\x8f\x75", [0x1117] = "\xb8\xa5", [0x1118] = "\xba\xeb",
  [0x1119] = "\x8f\x76", [0x111a] = "\x8f\x77", [0x111b] = "\xb3\xda",
  [0x111c] = "\x8f\x78", [0x111d] = "\x8f\x79", [0x111e] = "\x8f\x7a",
  [0x111f] = "\xb5\xdc", [0x1120] = "\xd5\xc5", [0x1121] = "\x8f\x7b",
  [0x1122] = "\x8f\x7c", [0x1123] = "\x8f\x7d", [0x1124] = "\x8f\x7e",
  [0x1125] = "\xc3\xd6", [0x1126] = "\xcf\xd2", [0x1127] = "\xbb\xa1",
  [0x1128] = "\x8f\x80", [0x1129] = "\xe5\xf3", [0x112a] = "\xe5\xf2",
  [0x112b] = "\x8f\x81", [0x112c] = "\x8f\x82", [0x112d] = "\xe5\xf4",
  [0x112e] = "\x8f\x83", [0x112f] = "\xcd\xe4", [0x1130] = "\x8f\x84",
  [0x1131] = "\xc8\xf5", [0x1132] = "\x8f\x85", [0x1133] = "\x8f\x86",
  [0x1134] = "\x8f\x87", [0x1135] = "\x8f\x88", [0x1136] = "\x8f\x89",
  [0x1137] = "\x8f\x8a", [0x1138] = "\x8f\x8b", [0x1139] = "\xb5\xaf",
  [0x113a] = "\xc7\xbf", [0x113b] = "\x8f\x8c", [0x113c] = "\xe5\xf6",
  [0x113d] = "\x8f\x8d", [0x113e] = "\x8f\x8e", [0x113f] = "\x8f\x8f",
  [0x1140] = "\xec\xb0", [0x1141] = "\x8f\x90", [0x1142] = "\x8f\x91",
  [0x1143] = "\x8f\x92", [0x1144] = "\x8f\x93", [0x1145] = "\x8f\x94",
  [0x1146] = "\x8f\x95", [0x1147] = "\x8f\x96", [0x1148] = "\x8f\x97",
  [0x1149] = "\x8f\x98", [0x114a] = "\x8f\x99", [0x114b] = "\x8f\x9a",
  [0x114c] = "\x8f\x9b", [0x114d] = "\x8f\x9c", [0x114e] = "\x8f\x9d",
  [0x114f] = "\x8f\x9e", [0x1150] = "\xe5\xe6", [0x1151] = "\x8f\x9f",
  [0x1152] = "\xb9\xe9", [0x1153] = "\xb5\xb1", [0x1154] = "\x8f\xa0",
  [0x1155] = "\xc2\xbc", [0x1156] = "\xe5\xe8", [0x1157] = "\xe5\xe7",
  [0x1158] = "\xe5\xe9", [0x1159] = "\x8f\xa1", [0x115a] = "\x8f\xa2",
  [0x115b] = "\x8f\xa3", [0x115c] = "\x8f\xa4", [0x115d] = "\xd2\xcd",
  [0x115e] = "\x8f\xa5", [0x115f] = "\x8f\xa6", [0x1160] = "\x8f\xa7",
  [0x1161] = "\xe1\xea", [0x1162] = "\xd0\xce", [0x1163] = "\x8f\xa8",
  [0x1164] = "\xcd\xae", [0x1165] = "\x8f\xa9", [0x1166] = "\xd1\xe5",
  [0x1167] = "\x8f\xaa", [0x1168] = "\x8f\xab", [0x1169] = "\xb2\xca",
  [0x116a] = "\xb1\xeb", [0x116b] = "\x8f\xac", [0x116c] = "\xb1\xf2",
  [0x116d] = "\xc5\xed", [0x116e] = "\x8f\xad", [0x116f] = "\x8f\xae",
  [0x1170] = "\xd5\xc3", [0x1171] = "\xd3\xb0", [0x1172] = "\x8f\xaf",
  [0x1173] = "\xe1\xdc", [0x1174] = "\x8f\xb0", [0x1175] = "\x8f\xb1",
  [0x1176] = "\x8f\xb2", [0x1177] = "\xe1\xdd", [0x1178] = "\x8f\xb3",
  [0x1179] = "\xd2\xdb", [0x117a] = "\x8f\xb4", [0x117b] = "\xb3\xb9",
  [0x117c] = "\xb1\xcb", [0x117d] = "\x8f\xb5", [0x117e] = "\x8f\xb6",
  [0x117f] = "\x8f\xb7", [0x1180] = "\xcd\xf9", [0x1181] = "\xd5\xf7",
  [0x1182] = "\xe1\xde", [0x1183] = "\x8f\xb8", [0x1184] = "\xbe\xb6",
  [0x1185] = "\xb4\xfd", [0x1186] = "\x8f\xb9", [0x1187] = "\xe1\xdf",
  [0x1188] = "\xba\xdc", [0x1189] = "\xe1\xe0", [0x118a] = "\xbb\xb2",
  [0x118b] = "\xc2\xc9", [0x118c] = "\xe1\xe1", [0x118d] = "\x8f\xba",
  [0x118e] = "\x8f\xbb", [0x118f] = "\x8f\xbc", [0x1190] = "\xd0\xec",
  [0x1191] = "\x8f\xbd", [0x1192] = "\xcd\xbd", [0x1193] = "\x8f\xbe",
  [0x1194] = "\x8f\xbf", [0x1195] = "\xe1\xe2", [0x1196] = "\x8f\xc0",
  [0x1197] = "\xb5\xc3", [0x1198] = "\xc5\xc7", [0x1199] = "\xe1\xe3",
  [0x119a] = "\x8f\xc1", [0x119b] = "\x8f\xc2", [0x119c] = "\xe1\xe4",
  [0x119d] = "\x8f\xc3", [0x119e] = "\x8f\xc4", [0x119f] = "\x8f\xc5",
  [0x11a0] = "\x8f\xc6", [0x11a1] = "\xd3\xf9", [0x11a2] = "\x8f\xc7",
  [0x11a3] = "\x8f\xc8", [0x11a4] = "\x8f\xc9", [0x11a5] = "\x8f\xca",
  [0x11a6] = "\x8f\xcb", [0x11a7] = "\x8f\xcc", [0x11a8] = "\xe1\xe5",
  [0x11a9] = "\x8f\xcd", [0x11aa] = "\xd1\xad", [0x11ab] = "\x8f\xce",
  [0x11ac] = "\x8f\xcf", [0x11ad] = "\xe1\xe6", [0x11ae] = "\xce\xa2",
  [0x11af] = "\x8f\xd0", [0x11b0] = "\x8f\xd1", [0x11b1] = "\x8f\xd2",
  [0x11b2] = "\x8f\xd3", [0x11b3] = "\x8f\xd4", [0x11b4] = "\x8f\xd5",
  [0x11b5] = "\xe1\xe7", [0x11b6] = "\x8f\xd6", [0x11b7] = "\xb5\xc2",
  [0x11b8] = "\x8f\xd7", [0x11b9] = "\x8f\xd8", [0x11ba] = "\x8f\xd9",
  [0x11bb] = "\x8f\xda", [0x11bc] = "\xe1\xe8", [0x11bd] = "\xbb\xd5",
  [0x11be] = "\x8f\xdb", [0x11bf] = "\x8f\xdc", [0x11c0] = "\x8f\xdd",
  [0x11c1] = "\x8f\xde", [0x11c2] = "\x8f\xdf", [0x11c3] = "\xd0\xc4",
  [0x11c4] = "\xe2\xe0", [0x11c5] = "\xb1\xd8", [0x11c6] = "\xd2\xe4",
  [0x11c7] = "\x8f\xe0", [0x11c8] = "\x8f\xe1", [0x11c9] = "\xe2\xe1",
  [0x11ca] = "\x8f\xe2", [0x11cb] = "\x8f\xe3", [0x11cc] = "\xbc\xc9",
  [0x11cd] = "\xc8\xcc", [0x11ce] = "\x8f\xe4", [0x11cf] = "\xe2\xe3",
  [0x11d0] = "\xec\xfe", [0x11d1] = "\xec\xfd", [0x11d2] = "\xdf\xaf",
  [0x11d3] = "\x8f\xe5", [0x11d4] = "\x8f\xe6", [0x11d5] = "\x8f\xe7",
  [0x11d6] = "\xe2\xe2", [0x11d7] = "\xd6\xbe", [0x11d8] = "\xcd\xfc",
  [0x11d9] = "\xc3\xa6", [0x11da] = "\x8f\xe8", [0x11db] = "\x8f\xe9",
  [0x11dc] = "\x8f\xea", [0x11dd] = "\xe3\xc3", [0x11de] = "\x8f\xeb",
  [0x11df] = "\x8f\xec", [0x11e0] = "\xd6\xd2", [0x11e1] = "\xe2\xe7",
  [0x11e2] = "\x8f\xed", [0x11e3] = "\x8f\xee", [0x11e4] = "\xe2\xe8",
  [0x11e5] = "\x8f\xef", [0x11e6] = "\x8f\xf0", [0x11e7] = "\xd3\xc7",
  [0x11e8] = "\x8f\xf1", [0x11e9] = "\x8f\xf2", [0x11ea] = "\xe2\xec",
  [0x11eb] = "\xbf\xec", [0x11ec] = "\x8f\xf3", [0x11ed] = "\xe2\xed",
  [0x11ee] = "\xe2\xe5", [0x11ef] = "\x8f\xf4", [0x11f0] = "\x8f\xf5",
  [0x11f1] = "\xb3\xc0", [0x11f2] = "\x8f\xf6", [0x11f3] = "\x8f\xf7",
  [0x11f4] = "\x8f\xf8", [0x11f5] = "\xc4\xee", [0x11f6] = "\x8f\xf9",
  [0x11f7] = "\x8f\xfa", [0x11f8] = "\xe2\xee", [0x11f9] = "\x8f\xfb",
  [0x11fa] = "\x8f\xfc", [0x11fb] = "\xd0\xc3", [0x11fc] = "\x8f\xfd",
  [0x11fd] = "\xba\xf6", [0x11fe] = "\xe2\xe9", [0x11ff] = "\xb7\xde",
  [0x1200] = "\xbb\xb3", [0x1201] = "\xcc\xac", [0x1202] = "\xcb\xcb",
  [0x1203] = "\xe2\xe4", [0x1204] = "\xe2\xe6", [0x1205] = "\xe2\xea",
  [0x1206] = "\xe2\xeb", [0x1207] = "\x8f\xfe", [0x1208] = "\x90\x40",
  [0x1209] = "\x90\x41", [0x120a] = "\xe2\xf7", [0x120b] = "\x90\x42",
  [0x120c] = "\x90\x43", [0x120d] = "\xe2\xf4", [0x120e] = "\xd4\xf5",
  [0x120f] = "\xe2\xf3", [0x1210] = "\x90\x44", [0x1211] = "\x90\x45",
  [0x1212] = "\xc5\xad", [0x1213] = "\x90\x46", [0x1214] = "\xd5\xfa",
  [0x1215] = "\xc5\xc2", [0x1216] = "\xb2\xc0", [0x1217] = "\x90\x47",
  [0x1218] = "\x90\x48", [0x1219] = "\xe2\xef", [0x121a] = "\x90\x49",
  [0x121b] = "\xe2\xf2", [0x121c] = "\xc1\xaf", [0x121d] = "\xcb\xbc",
  [0x121e] = "\x90\x4a", [0x121f] = "\x90\x4b", [0x1220] = "\xb5\xa1",
  [0x1221] = "\xe2\xf9", [0x1222] = "\x90\x4c", [0x1223] = "\x90\x4d",
  [0x1224] = "\x90\x4e", [0x1225] = "\xbc\xb1", [0x1226] = "\xe2\xf1",
  [0x1227] = "\xd0\xd4", [0x1228] = "\xd4\xb9", [0x1229] = "\xe2\xf5",
  [0x122a] = "\xb9\xd6", [0x122b] = "\xe2\xf6", [0x122c] = "\x90\x4f",
  [0x122d] = "\x90\x50", [0x122e] = "\x90\x51", [0x122f] = "\xc7\xd3",
  [0x1230] = "\x90\x52", [0x1231] = "\x90\x53", [0x1232] = "\x90\x54",
  [0x1233] = "\x90\x55", [0x1234] = "\x90\x56", [0x1235] = "\xe2\xf0",
  [0x1236] = "\x90\x57", [0x1237] = "\x90\x58", [0x1238] = "\x90\x59",
  [0x1239] = "\x90\x5a", [0x123a] = "\x90\x5b", [0x123b] = "\xd7\xdc",
  [0x123c] = "\xed\xa1", [0x123d] = "\x90\x5c", [0x123e] = "\x90\x5d",
  [0x123f] = "\xe2\xf8", [0x1240] = "\x90\x5e", [0x1241] = "\xed\xa5",
  [0x1242] = "\xe2\xfe", [0x1243] = "\xca\xd1", [0x1244] = "\x90\x5f",
  [0x1245] = "\x90\x60", [0x1246] = "\x90\x61", [0x1247] = "\x90\x62",
  [0x1248] = "\x90\x63", [0x1249] = "\x90\x64", [0x124a] = "\x90\x65",
  [0x124b] = "\xc1\xb5", [0x124c] = "\x90\x66", [0x124d] = "\xbb\xd0",
  [0x124e] = "\x90\x67", [0x124f] = "\x90\x68", [0x1250] = "\xbf\xd6",
  [0x1251] = "\x90\x69", [0x1252] = "\xba\xe3", [0x1253] = "\x90\x6a",
  [0x1254] = "\x90\x6b", [0x1255] = "\xcb\xa1", [0x1256] = "\x90\x6c",
  [0x1257] = "\x90\x6d", [0x1258] = "\x90\x6e", [0x1259] = "\xed\xa6",
  [0x125a] = "\xed\xa3", [0x125b] = "\x90\x6f", [0x125c] = "\x90\x70",
  [0x125d] = "\xed\xa2", [0x125e] = "\x90\x71", [0x125f] = "\x90\x72",
  [0x1260] = "\x90\x73", [0x1261] = "\x90\x74", [0x1262] = "\xbb\xd6",
  [0x1263] = "\xed\xa7", [0x1264] = "\xd0\xf4", [0x1265] = "\x90\x75",
  [0x1266] = "\x90\x76", [0x1267] = "\xed\xa4", [0x1268] = "\xba\xde",
  [0x1269] = "\xb6\xf7", [0x126a] = "\xe3\xa1", [0x126b] = "\xb6\xb2",
  [0x126c] = "\xcc\xf1", [0x126d] = "\xb9\xa7", [0x126e] = "\x90\x77",
  [0x126f] = "\xcf\xa2", [0x1270] = "\xc7\xa1", [0x1271] = "\x90\x78",
  [0x1272] = "\x90\x79", [0x1273] = "\xbf\xd2", [0x1274] = "\x90\x7a",
  [0x1275] = "\x90\x7b", [0x1276] = "\xb6\xf1", [0x1277] = "\x90\x7c",
  [0x1278] = "\xe2\xfa", [0x1279] = "\xe2\xfb", [0x127a] = "\xe2\xfd",
  [0x127b] = "\xe2\xfc", [0x127c] = "\xc4\xd5", [0x127d] = "\xe3\xa2",
  [0x127e] = "\x90\x7d", [0x127f] = "\xd3\xc1", [0x1280] = "\x90\x7e",
  [0x1281] = "\x90\x80", [0x1282] = "\x90\x81", [0x1283] = "\xe3\xa7",
  [0x1284] = "\xc7\xc4", [0x1285] = "\x90\x82", [0x1286] = "\x90\x83",
  [0x1287] = "\x90\x84", [0x1288] = "\x90\x85", [0x1289] = "\xcf\xa4",
  [0x128a] = "\x90\x86", [0x128b] = "\x90\x87", [0x128c] = "\xe3\xa9",
  [0x128d] = "\xba\xb7", [0x128e] = "\x90\x88", [0x128f] = "\x90\x89",
  [0x1290] = "\x90\x8a", [0x1291] = "\x90\x8b", [0x1292] = "\xe3\xa8",
  [0x1293] = "\x90\x8c", [0x1294] = "\xbb\xda", [0x1295] = "\x90\x8d",
  [0x1296] = "\xe3\xa3", [0x1297] = "\x90\x8e", [0x1298] = "\x90\x8f",
  [0x1299] = "\x90\x90", [0x129a] = "\xe3\xa4", [0x129b] = "\xe3\xaa",
  [0x129c] = "\x90\x91", [0x129d] = "\xe3\xa6", [0x129e] = "\x90\x92",
  [0x129f] = "\xce\xf2", [0x12a0] = "\xd3\xc6", [0x12a1] = "\x90\x93",
  [0x12a2] = "\x90\x94", [0x12a3] = "\xbb\xbc", [0x12a4] = "\x90\x95",
  [0x12a5] = "\x90\x96", [0x12a6] = "\xd4\xc3", [0x12a7] = "\x90\x97",
  [0x12a8] = "\xc4\xfa", [0x12a9] = "\x90\x98", [0x12aa] = "\x90\x99",
  [0x12ab] = "\xed\xa8", [0x12ac] = "\xd0\xfc", [0x12ad] = "\xe3\xa5",
  [0x12ae] = "\x90\x9a", [0x12af] = "\xc3\xf5", [0x12b0] = "\x90\x9b",
  [0x12b1] = "\xe3\xad", [0x12b2] = "\xb1\xaf", [0x12b3] = "\x90\x9c",
  [0x12b4] = "\xe3\xb2", [0x12b5] = "\x90\x9d", [0x12b6] = "\x90\x9e",
  [0x12b7] = "\x90\x9f", [0x12b8] = "\xbc\xc2", [0x12b9] = "\x90\xa0",
  [0x12ba] = "\x90\xa1", [0x12bb] = "\xe3\xac", [0x12bc] = "\xb5\xbf",
  [0x12bd] = "\x90\xa2", [0x12be] = "\x90\xa3", [0x12bf] = "\x90\xa4",
  [0x12c0] = "\x90\xa5", [0x12c1] = "\x90\xa6", [0x12c2] = "\x90\xa7",
  [0x12c3] = "\x90\xa8", [0x12c4] = "\x90\xa9", [0x12c5] = "\xc7\xe9",
  [0x12c6] = "\xe3\xb0", [0x12c7] = "\x90\xaa", [0x12c8] = "\x90\xab",
  [0x12c9] = "\x90\xac", [0x12ca] = "\xbe\xaa", [0x12cb] = "\xcd\xef",
  [0x12cc] = "\x90\xad", [0x12cd] = "\x90\xae", [0x12ce] = "\x90\xaf",
  [0x12cf] = "\x90\xb0", [0x12d0] = "\x90\xb1", [0x12d1] = "\xbb\xf3",
  [0x12d2] = "\x90\xb2", [0x12d3] = "\x90\xb3", [0x12d4] = "\x90\xb4",
  [0x12d5] = "\xcc\xe8", [0x12d6] = "\x90\xb5", [0x12d7] = "\x90\xb6",
  [0x12d8] = "\xe3\xaf", [0x12d9] = "\x90\xb7", [0x12da] = "\xe3\xb1",
  [0x12db] = "\x90\xb8", [0x12dc] = "\xcf\xa7", [0x12dd] = "\xe3\xae",
  [0x12de] = "\x90\xb9", [0x12df] = "\xce\xa9", [0x12e0] = "\xbb\xdd",
  [0x12e1] = "\x90\xba", [0x12e2] = "\x90\xbb", [0x12e3] = "\x90\xbc",
  [0x12e4] = "\x90\xbd", [0x12e5] = "\x90\xbe", [0x12e6] = "\xb5\xeb",
  [0x12e7] = "\xbe\xe5", [0x12e8] = "\xb2\xd2", [0x12e9] = "\xb3\xcd",
  [0x12ea] = "\x90\xbf", [0x12eb] = "\xb1\xb9", [0x12ec] = "\xe3\xab",
  [0x12ed] = "\xb2\xd1", [0x12ee] = "\xb5\xac", [0x12ef] = "\xb9\xdf",
  [0x12f0] = "\xb6\xe8", [0x12f1] = "\x90\xc0", [0x12f2] = "\x90\xc1",
  [0x12f3] = "\xcf\xeb", [0x12f4] = "\xe3\xb7", [0x12f5] = "\x90\xc2",
  [0x12f6] = "\xbb\xcc", [0x12f7] = "\x90\xc3", [0x12f8] = "\x90\xc4",
  [0x12f9] = "\xc8\xc7", [0x12fa] = "\xd0\xca", [0x12fb] = "\x90\xc5",
  [0x12fc] = "\x90\xc6", [0x12fd] = "\x90\xc7", [0x12fe] = "\x90\xc8",
  [0x12ff] = "\x90\xc9", [0x1300] = "\xe3\xb8", [0x1301] = "\xb3\xee",
  [0x1302] = "\x90\xca", [0x1303] = "\x90\xcb", [0x1304] = "\x90\xcc",
  [0x1305] = "\x90\xcd", [0x1306] = "\xed\xa9", [0x1307] = "\x90\xce",
  [0x1308] = "\xd3\xfa", [0x1309] = "\xd3\xe4", [0x130a] = "\x90\xcf",
  [0x130b] = "\x90\xd0", [0x130c] = "\x90\xd1", [0x130d] = "\xed\xaa",
  [0x130e] = "\xe3\xb9", [0x130f] = "\xd2\xe2", [0x1310] = "\x90\xd2",
  [0x1311] = "\x90\xd3", [0x1312] = "\x90\xd4", [0x1313] = "\x90\xd5",
  [0x1314] = "\x90\xd6", [0x1315] = "\xe3\xb5", [0x1316] = "\x90\xd7",
  [0x1317] = "\x90\xd8", [0x1318] = "\x90\xd9", [0x1319] = "\x90\xda",
  [0x131a] = "\xd3\xde", [0x131b] = "\x90\xdb", [0x131c] = "\x90\xdc",
  [0x131d] = "\x90\xdd", [0x131e] = "\x90\xde", [0x131f] = "\xb8\xd0",
  [0x1320] = "\xe3\xb3", [0x1321] = "\x90\xdf", [0x1322] = "\x90\xe0",
  [0x1323] = "\xe3\xb6", [0x1324] = "\xb7\xdf", [0x1325] = "\x90\xe1",
  [0x1326] = "\xe3\xb4", [0x1327] = "\xc0\xa2", [0x1328] = "\x90\xe2",
  [0x1329] = "\x90\xe3", [0x132a] = "\x90\xe4", [0x132b] = "\xe3\xba",
  [0x132c] = "\x90\xe5", [0x132d] = "\x90\xe6", [0x132e] = "\x90\xe7",
  [0x132f] = "\x90\xe8", [0x1330] = "\x90\xe9", [0x1331] = "\x90\xea",
  [0x1332] = "\x90\xeb", [0x1333] = "\x90\xec", [0x1334] = "\x90\xed",
  [0x1335] = "\x90\xee", [0x1336] = "\x90\xef", [0x1337] = "\x90\xf0",
  [0x1338] = "\x90\xf1", [0x1339] = "\x90\xf2", [0x133a] = "\x90\xf3",
  [0x133b] = "\x90\xf4", [0x133c] = "\x90\xf5", [0x133d] = "\x90\xf6",
  [0x133e] = "\x90\xf7", [0x133f] = "\xd4\xb8", [0x1340] = "\x90\xf8",
  [0x1341] = "\x90\xf9", [0x1342] = "\x90\xfa", [0x1343] = "\x90\xfb",
  [0x1344] = "\x90\xfc", [0x1345] = "\x90\xfd", [0x1346] = "\x90\xfe",
  [0x1347] = "\x91\x40", [0x1348] = "\xb4\xc8", [0x1349] = "\x91\x41",
  [0x134a] = "\xe3\xbb", [0x134b] = "\x91\x42", [0x134c] = "\xbb\xc5",
  [0x134d] = "\x91\x43", [0x134e] = "\xc9\xf7", [0x134f] = "\x91\x44",
  [0x1350] = "\x91\x45", [0x1351] = "\xc9\xe5", [0x1352] = "\x91\x46",
  [0x1353] = "\x91\x47", [0x1354] = "\x91\x48", [0x1355] = "\xc4\xbd",
  [0x1356] = "\x91\x49", [0x1357] = "\x91\x4a", [0x1358] = "\x91\x4b",
  [0x1359] = "\x91\x4c", [0x135a] = "\x91\x4d", [0x135b] = "\x91\x4e",
  [0x135c] = "\x91\x4f", [0x135d] = "\xed\xab", [0x135e] = "\x91\x50",
  [0x135f] = "\x91\x51", [0x1360] = "\x91\x52", [0x1361] = "\x91\x53",
  [0x1362] = "\xc2\xfd", [0x1363] = "\x91\x54", [0x1364] = "\x91\x55",
  [0x1365] = "\x91\x56", [0x1366] = "\x91\x57", [0x1367] = "\xbb\xdb",
  [0x1368] = "\xbf\xae", [0x1369] = "\x91\x58", [0x136a] = "\x91\x59",
  [0x136b] = "\x91\x5a", [0x136c] = "\x91\x5b", [0x136d] = "\x91\x5c",
  [0x136e] = "\x91\x5d", [0x136f] = "\x91\x5e", [0x1370] = "\xce\xbf",
  [0x1371] = "\x91\x5f", [0x1372] = "\x91\x60", [0x1373] = "\x91\x61",
  [0x1374] = "\x91\x62", [0x1375] = "\xe3\xbc", [0x1376] = "\x91\x63",
  [0x1377] = "\xbf\xb6", [0x1378] = "\x91\x64", [0x1379] = "\x91\x65",
  [0x137a] = "\x91\x66", [0x137b] = "\x91\x67", [0x137c] = "\x91\x68",
  [0x137d] = "\x91\x69", [0x137e] = "\x91\x6a", [0x137f] = "\x91\x6b",
  [0x1380] = "\x91\x6c", [0x1381] = "\x91\x6d", [0x1382] = "\x91\x6e",
  [0x1383] = "\x91\x6f", [0x1384] = "\x91\x70", [0x1385] = "\x91\x71",
  [0x1386] = "\x91\x72", [0x1387] = "\x91\x73", [0x1388] = "\x91\x74",
  [0x1389] = "\x91\x75", [0x138a] = "\x91\x76", [0x138b] = "\xb1\xef",
  [0x138c] = "\x91\x77", [0x138d] = "\x91\x78", [0x138e] = "\xd4\xf7",
  [0x138f] = "\x91\x79", [0x1390] = "\x91\x7a", [0x1391] = "\x91\x7b",
  [0x1392] = "\x91\x7c", [0x1393] = "\x91\x7d", [0x1394] = "\xe3\xbe",
  [0x1395] = "\x91\x7e", [0x1396] = "\x91\x80", [0x1397] = "\x91\x81",
  [0x1398] = "\x91\x82", [0x1399] = "\x91\x83", [0x139a] = "\x91\x84",
  [0x139b] = "\x91\x85", [0x139c] = "\x91\x86", [0x139d] = "\xed\xad",
  [0x139e] = "\x91\x87", [0x139f] = "\x91\x88", [0x13a0] = "\x91\x89",
  [0x13a1] = "\x91\x8a", [0x13a2] = "\x91\x8b", [0x13a3] = "\x91\x8c",
  [0x13a4] = "\x91\x8d", [0x13a5] = "\x91\x8e", [0x13a6] = "\x91\x8f",
  [0x13a7] = "\xe3\xbf", [0x13a8] = "\xba\xa9", [0x13a9] = "\xed\xac",
  [0x13aa] = "\x91\x90", [0x13ab] = "\x91\x91", [0x13ac] = "\xe3\xbd",
  [0x13ad] = "\x91\x92", [0x13ae] = "\x91\x93", [0x13af] = "\x91\x94",
  [0x13b0] = "\x91\x95", [0x13b1] = "\x91\x96", [0x13b2] = "\x91\x97",
  [0x13b3] = "\x91\x98", [0x13b4] = "\x91\x99", [0x13b5] = "\x91\x9a",
  [0x13b6] = "\x91\x9b", [0x13b7] = "\xe3\xc0", [0x13b8] = "\x91\x9c",
  [0x13b9] = "\x91\x9d", [0x13ba] = "\x91\x9e", [0x13bb] = "\x91\x9f",
  [0x13bc] = "\x91\xa0", [0x13bd] = "\x91\xa1", [0x13be] = "\xba\xb6",
  [0x13bf] = "\x91\xa2", [0x13c0] = "\x91\xa3", [0x13c1] = "\x91\xa4",
  [0x13c2] = "\xb6\xae", [0x13c3] = "\x91\xa5", [0x13c4] = "\x91\xa6",
  [0x13c5] = "\x91\xa7", [0x13c6] = "\x91\xa8", [0x13c7] = "\x91\xa9",
  [0x13c8] = "\xd0\xb8", [0x13c9] = "\x91\xaa", [0x13ca] = "\xb0\xc3",
  [0x13cb] = "\xed\xae", [0x13cc] = "\x91\xab", [0x13cd] = "\x91\xac",
  [0x13ce] = "\x91\xad", [0x13cf] = "\x91\xae", [0x13d0] = "\x91\xaf",
  [0x13d1] = "\xed\xaf", [0x13d2] = "\xc0\xc1", [0x13d3] = "\x91\xb0",
  [0x13d4] = "\xe3\xc1", [0x13d5] = "\x91\xb1", [0x13d6] = "\x91\xb2",
  [0x13d7] = "\x91\xb3", [0x13d8] = "\x91\xb4", [0x13d9] = "\x91\xb5",
  [0x13da] = "\x91\xb6", [0x13db] = "\x91\xb7", [0x13dc] = "\x91\xb8",
  [0x13dd] = "\x91\xb9", [0x13de] = "\x91\xba", [0x13df] = "\x91\xbb",
  [0x13e0] = "\x91\xbc", [0x13e1] = "\x91\xbd", [0x13e2] = "\x91\xbe",
  [0x13e3] = "\x91\xbf", [0x13e4] = "\x91\xc0", [0x13e5] = "\x91\xc1",
  [0x13e6] = "\xc5\xb3", [0x13e7] = "\x91\xc2", [0x13e8] = "\x91\xc3",
  [0x13e9] = "\x91\xc4", [0x13ea] = "\x91\xc5", [0x13eb] = "\x91\xc6",
  [0x13ec] = "\x91\xc7", [0x13ed] = "\x91\xc8", [0x13ee] = "\x91\xc9",
  [0x13ef] = "\x91\xca", [0x13f0] = "\x91\xcb", [0x13f1] = "\x91\xcc",
  [0x13f2] = "\x91\xcd", [0x13f3] = "\x91\xce", [0x13f4] = "\x91\xcf",
  [0x13f5] = "\xe3\xc2", [0x13f6] = "\x91\xd0", [0x13f7] = "\x91\xd1",
  [0x13f8] = "\x91\xd2", [0x13f9] = "\x91\xd3", [0x13fa] = "\x91\xd4",
  [0x13fb] = "\x91\xd5", [0x13fc] = "\x91\xd6", [0x13fd] = "\x91\xd7",
  [0x13fe] = "\x91\xd8", [0x13ff] = "\xdc\xb2", [0x1400] = "\x91\xd9",
  [0x1401] = "\x91\xda", [0x1402] = "\x91\xdb", [0x1403] = "\x91\xdc",
  [0x1404] = "\x91\xdd", [0x1405] = "\x91\xde", [0x1406] = "\xed\xb0",
  [0x1407] = "\x91\xdf", [0x1408] = "\xb8\xea", [0x1409] = "\x91\xe0",
  [0x140a] = "\xce\xec", [0x140b] = "\xea\xa7", [0x140c] = "\xd0\xe7",
  [0x140d] = "\xca\xf9", [0x140e] = "\xc8\xd6", [0x140f] = "\xcf\xb7",
  [0x1410] = "\xb3\xc9", [0x1411] = "\xce\xd2", [0x1412] = "\xbd\xe4",
  [0x1413] = "\x91\xe1", [0x1414] = "\x91\xe2", [0x1415] = "\xe3\xde",
  [0x1416] = "\xbb\xf2", [0x1417] = "\xea\xa8", [0x1418] = "\xd5\xbd",
  [0x1419] = "\x91\xe3", [0x141a] = "\xc6\xdd", [0x141b] = "\xea\xa9",
  [0x141c] = "\x91\xe4", [0x141d] = "\x91\xe5", [0x141e] = "\x91\xe6",
  [0x141f] = "\xea\xaa", [0x1420] = "\x91\xe7", [0x1421] = "\xea\xac",
  [0x1422] = "\xea\xab", [0x1423] = "\x91\xe8", [0x1424] = "\xea\xae",
  [0x1425] = "\xea\xad", [0x1426] = "\x91\xe9", [0x1427] = "\x91\xea",
  [0x1428] = "\x91\xeb", [0x1429] = "\x91\xec", [0x142a] = "\xbd\xd8",
  [0x142b] = "\x91\xed", [0x142c] = "\xea\xaf", [0x142d] = "\x91\xee",
  [0x142e] = "\xc2\xbe", [0x142f] = "\x91\xef", [0x1430] = "\x91\xf0",
  [0x1431] = "\x91\xf1", [0x1432] = "\x91\xf2", [0x1433] = "\xb4\xc1",
  [0x1434] = "\xb4\xf7", [0x1435] = "\x91\xf3", [0x1436] = "\x91\xf4",
  [0x1437] = "\xbb\xa7", [0x1438] = "\x91\xf5", [0x1439] = "\x91\xf6",
  [0x143a] = "\x91\xf7", [0x143b] = "\x91\xf8", [0x143c] = "\x91\xf9",
  [0x143d] = "\xec\xe6", [0x143e] = "\xec\xe5", [0x143f] = "\xb7\xbf",
  [0x1440] = "\xcb\xf9", [0x1441] = "\xb1\xe2", [0x1442] = "\x91\xfa",
  [0x1443] = "\xec\xe7", [0x1444] = "\x91\xfb", [0x1445] = "\x91\xfc",
  [0x1446] = "\x91\xfd", [0x1447] = "\xc9\xc8", [0x1448] = "\xec\xe8",
  [0x1449] = "\xec\xe9", [0x144a] = "\x91\xfe", [0x144b] = "\xca\xd6",
  [0x144c] = "\xde\xd0", [0x144d] = "\xb2\xc5", [0x144e] = "\xd4\xfa",
  [0x144f] = "\x92\x40", [0x1450] = "\x92\x41", [0x1451] = "\xc6\xcb",
  [0x1452] = "\xb0\xc7", [0x1453] = "\xb4\xf2", [0x1454] = "\xc8\xd3",
  [0x1455] = "\x92\x42", [0x1456] = "\x92\x43", [0x1457] = "\x92\x44",
  [0x1458] = "\xcd\xd0", [0x1459] = "\x92\x45", [0x145a] = "\x92\x46",
  [0x145b] = "\xbf\xb8", [0x145c] = "\x92\x47", [0x145d] = "\x92\x48",
  [0x145e] = "\x92\x49", [0x145f] = "\x92\x4a", [0x1460] = "\x92\x4b",
  [0x1461] = "\x92\x4c", [0x1462] = "\x92\x4d", [0x1463] = "\xbf\xdb",
  [0x1464] = "\x92\x4e", [0x1465] = "\x92\x4f", [0x1466] = "\xc7\xa4",
  [0x1467] = "\xd6\xb4", [0x1468] = "\x92\x50", [0x1469] = "\xc0\xa9",
  [0x146a] = "\xde\xd1", [0x146b] = "\xc9\xa8", [0x146c] = "\xd1\xef",
  [0x146d] = "\xc5\xa4", [0x146e] = "\xb0\xe7", [0x146f] = "\xb3\xb6",
  [0x1470] = "\xc8\xc5", [0x1471] = "\x92\x51", [0x1472] = "\x92\x52",
  [0x1473] = "\xb0\xe2", [0x1474] = "\x92\x53", [0x1475] = "\x92\x54",
  [0x1476] = "\xb7\xf6", [0x1477] = "\x92\x55", [0x1478] = "\x92\x56",
  [0x1479] = "\xc5\xfa", [0x147a] = "\x92\x57", [0x147b] = "\x92\x58",
  [0x147c] = "\xb6\xf3", [0x147d] = "\x92\x59", [0x147e] = "\xd5\xd2",
  [0x147f] = "\xb3\xd0", [0x1480] = "\xbc\xbc", [0x1481] = "\x92\x5a",
  [0x1482] = "\x92\x5b", [0x1483] = "\x92\x5c", [0x1484] = "\xb3\xad",
  [0x1485] = "\x92\x5d", [0x1486] = "\x92\x5e", [0x1487] = "\x92\x5f",
  [0x1488] = "\x92\x60", [0x1489] = "\xbe\xf1", [0x148a] = "\xb0\xd1",
  [0x148b] = "\x92\x61", [0x148c] = "\x92\x62", [0x148d] = "\x92\x63",
  [0x148e] = "\x92\x64", [0x148f] = "\x92\x65", [0x1490] = "\x92\x66",
  [0x1491] = "\xd2\xd6", [0x1492] = "\xca\xe3", [0x1493] = "\xd7\xa5",
  [0x1494] = "\x92\x67", [0x1495] = "\xcd\xb6", [0x1496] = "\xb6\xb6",
  [0x1497] = "\xbf\xb9", [0x1498] = "\xd5\xdb", [0x1499] = "\x92\x68",
  [0x149a] = "\xb8\xa7", [0x149b] = "\xc5\xd7", [0x149c] = "\x92\x69",
  [0x149d] = "\x92\x6a", [0x149e] = "\x92\x6b", [0x149f] = "\xde\xd2",
  [0x14a0] = "\xbf\xd9", [0x14a1] = "\xc2\xd5", [0x14a2] = "\xc7\xc0",
  [0x14a3] = "\x92\x6c", [0x14a4] = "\xbb\xa4", [0x14a5] = "\xb1\xa8",
  [0x14a6] = "\x92\x6d", [0x14a7] = "\x92\x6e", [0x14a8] = "\xc5\xea",
  [0x14a9] = "\x92\x6f", [0x14aa] = "\x92\x70", [0x14ab] = "\xc5\xfb",
  [0x14ac] = "\xcc\xa7", [0x14ad] = "\x92\x71", [0x14ae] = "\x92\x72",
  [0x14af] = "\x92\x73", [0x14b0] = "\x92\x74", [0x14b1] = "\xb1\xa7",
  [0x14b2] = "\x92\x75", [0x14b3] = "\x92\x76", [0x14b4] = "\x92\x77",
  [0x14b5] = "\xb5\xd6", [0x14b6] = "\x92\x78", [0x14b7] = "\x92\x79",
  [0x14b8] = "\x92\x7a", [0x14b9] = "\xc4\xa8", [0x14ba] = "\x92\x7b",
  [0x14bb] = "\xde\xd3", [0x14bc] = "\xd1\xba", [0x14bd] = "\xb3\xe9",
  [0x14be] = "\x92\x7c", [0x14bf] = "\xc3\xf2", [0x14c0] = "\x92\x7d",
  [0x14c1] = "\x92\x7e", [0x14c2] = "\xb7\xf7", [0x14c3] = "\x92\x80",
  [0x14c4] = "\xd6\xf4", [0x14c5] = "\xb5\xa3", [0x14c6] = "\xb2\xf0",
  [0x14c7] = "\xc4\xb4", [0x14c8] = "\xc4\xe9", [0x14c9] = "\xc0\xad",
  [0x14ca] = "\xde\xd4", [0x14cb] = "\x92\x81", [0x14cc] = "\xb0\xe8",
  [0x14cd] = "\xc5\xc4", [0x14ce] = "\xc1\xe0", [0x14cf] = "\x92\x82",
  [0x14d0] = "\xb9\xd5", [0x14d1] = "\x92\x83", [0x14d2] = "\xbe\xdc",
  [0x14d3] = "\xcd\xd8", [0x14d4] = "\xb0\xce", [0x14d5] = "\x92\x84",
  [0x14d6] = "\xcd\xcf", [0x14d7] = "\xde\xd6", [0x14d8] = "\xbe\xd0",
  [0x14d9] = "\xd7\xbe", [0x14da] = "\xde\xd5", [0x14db] = "\xd5\xd0",
  [0x14dc] = "\xb0\xdd", [0x14dd] = "\x92\x85", [0x14de] = "\x92\x86",
  [0x14df] = "\xc4\xe2", [0x14e0] = "\x92\x87", [0x14e1] = "\x92\x88",
  [0x14e2] = "\xc2\xa3", [0x14e3] = "\xbc\xf0", [0x14e4] = "\x92\x89",
  [0x14e5] = "\xd3\xb5", [0x14e6] = "\xc0\xb9", [0x14e7] = "\xc5\xa1",
  [0x14e8] = "\xb2\xa6", [0x14e9] = "\xd4\xf1", [0x14ea] = "\x92\x8a",
  [0x14eb] = "\x92\x8b", [0x14ec] = "\xc0\xa8", [0x14ed] = "\xca\xc3",
  [0x14ee] = "\xde\xd7", [0x14ef] = "\xd5\xfc", [0x14f0] = "\x92\x8c",
  [0x14f1] = "\xb9\xb0", [0x14f2] = "\x92\x8d", [0x14f3] = "\xc8\xad",
  [0x14f4] = "\xcb\xa9", [0x14f5] = "\x92\x8e", [0x14f6] = "\xde\xd9",
  [0x14f7] = "\xbf\xbd", [0x14f8] = "\x92\x8f", [0x14f9] = "\x92\x90",
  [0x14fa] = "\x92\x91", [0x14fb] = "\x92\x92", [0x14fc] = "\xc6\xb4",
  [0x14fd] = "\xd7\xa7", [0x14fe] = "\xca\xb0", [0x14ff] = "\xc4\xc3",
  [0x1500] = "\x92\x93", [0x1501] = "\xb3\xd6", [0x1502] = "\xb9\xd2",
  [0x1503] = "\x92\x94", [0x1504] = "\x92\x95", [0x1505] = "\x92\x96",
  [0x1506] = "\x92\x97", [0x1507] = "\xd6\xb8", [0x1508] = "\xea\xfc",
  [0x1509] = "\xb0\xb4", [0x150a] = "\x92\x98", [0x150b] = "\x92\x99",
  [0x150c] = "\x92\x9a", [0x150d] = "\x92\x9b", [0x150e] = "\xbf\xe6",
  [0x150f] = "\x92\x9c", [0x1510] = "\x92\x9d", [0x1511] = "\xcc\xf4",
  [0x1512] = "\x92\x9e", [0x1513] = "\x92\x9f", [0x1514] = "\x92\xa0",
  [0x1515] = "\x92\xa1", [0x1516] = "\xcd\xda", [0x1517] = "\x92\xa2",
  [0x1518] = "\x92\xa3", [0x1519] = "\x92\xa4", [0x151a] = "\xd6\xbf",
  [0x151b] = "\xc2\xce", [0x151c] = "\x92\xa5", [0x151d] = "\xce\xce",
  [0x151e] = "\xcc\xa2", [0x151f] = "\xd0\xae", [0x1520] = "\xc4\xd3",
  [0x1521] = "\xb5\xb2", [0x1522] = "\xde\xd8", [0x1523] = "\xd5\xf5",
  [0x1524] = "\xbc\xb7", [0x1525] = "\xbb\xd3", [0x1526] = "\x92\xa6",
  [0x1527] = "\x92\xa7", [0x1528] = "\xb0\xa4", [0x1529] = "\x92\xa8",
  [0x152a] = "\xc5\xb2", [0x152b] = "\xb4\xec", [0x152c] = "\x92\xa9",
  [0x152d] = "\x92\xaa", [0x152e] = "\x92\xab", [0x152f] = "\xd5\xf1",
  [0x1530] = "\x92\xac", [0x1531] = "\x92\xad", [0x1532] = "\xea\xfd",
  [0x1533] = "\x92\xae", [0x1534] = "\x92\xaf", [0x1535] = "\x92\xb0",
  [0x1536] = "\x92\xb1", [0x1537] = "\x92\xb2", [0x1538] = "\x92\xb3",
  [0x1539] = "\xde\xda", [0x153a] = "\xcd\xa6", [0x153b] = "\x92\xb4",
  [0x153c] = "\x92\xb5", [0x153d] = "\xcd\xec", [0x153e] = "\x92\xb6",
  [0x153f] = "\x92\xb7", [0x1540] = "\x92\xb8", [0x1541] = "\x92\xb9",
  [0x1542] = "\xce\xe6", [0x1543] = "\xde\xdc", [0x1544] = "\x92\xba",
  [0x1545] = "\xcd\xb1", [0x1546] = "\xc0\xa6", [0x1547] = "\x92\xbb",
  [0x1548] = "\x92\xbc", [0x1549] = "\xd7\xbd", [0x154a] = "\x92\xbd",
  [0x154b] = "\xde\xdb", [0x154c] = "\xb0\xc6", [0x154d] = "\xba\xb4",
  [0x154e] = "\xc9\xd3", [0x154f] = "\xc4\xf3", [0x1550] = "\xbe\xe8",
  [0x1551] = "\x92\xbe", [0x1552] = "\x92\xbf", [0x1553] = "\x92\xc0",
  [0x1554] = "\x92\xc1", [0x1555] = "\xb2\xb6", [0x1556] = "\x92\xc2",
  [0x1557] = "\x92\xc3", [0x1558] = "\x92\xc4", [0x1559] = "\x92\xc5",
  [0x155a] = "\x92\xc6", [0x155b] = "\x92\xc7", [0x155c] = "\x92\xc8",
  [0x155d] = "\x92\xc9", [0x155e] = "\xc0\xcc", [0x155f] = "\xcb\xf0",
  [0x1560] = "\x92\xca", [0x1561] = "\xbc\xf1", [0x1562] = "\xbb\xbb",
  [0x1563] = "\xb5\xb7", [0x1564] = "\x92\xcb", [0x1565] = "\x92\xcc",
  [0x1566] = "\x92\xcd", [0x1567] = "\xc5\xf5", [0x1568] = "\x92\xce",
  [0x1569] = "\xde\xe6", [0x156a] = "\x92\xcf", [0x156b] = "\x92\xd0",
  [0x156c] = "\x92\xd1", [0x156d] = "\xde\xe3", [0x156e] = "\xbe\xdd",
  [0x156f] = "\x92\xd2", [0x1570] = "\x92\xd3", [0x1571] = "\xde\xdf",
  [0x1572] = "\x92\xd4", [0x1573] = "\x92\xd5", [0x1574] = "\x92\xd6",
  [0x1575] = "\x92\xd7", [0x1576] = "\xb4\xb7", [0x1577] = "\xbd\xdd",
  [0x1578] = "\x92\xd8", [0x1579] = "\x92\xd9", [0x157a] = "\xde\xe0",
  [0x157b] = "\xc4\xed", [0x157c] = "\x92\xda", [0x157d] = "\x92\xdb",
  [0x157e] = "\x92\xdc", [0x157f] = "\x92\xdd", [0x1580] = "\xcf\xc6",
  [0x1581] = "\x92\xde", [0x1582] = "\xb5\xe0", [0x1583] = "\x92\xdf",
  [0x1584] = "\x92\xe0", [0x1585] = "\x92\xe1", [0x1586] = "\x92\xe2",
  [0x1587] = "\xb6\xde", [0x1588] = "\xca\xda", [0x1589] = "\xb5\xf4",
  [0x158a] = "\xde\xe5", [0x158b] = "\x92\xe3", [0x158c] = "\xd5\xc6",
  [0x158d] = "\x92\xe4", [0x158e] = "\xde\xe1", [0x158f] = "\xcc\xcd",
  [0x1590] = "\xc6\xfe", [0x1591] = "\x92\xe5", [0x1592] = "\xc5\xc5",
  [0x1593] = "\x92\xe6", [0x1594] = "\x92\xe7", [0x1595] = "\x92\xe8",
  [0x1596] = "\xd2\xb4", [0x1597] = "\x92\xe9", [0x1598] = "\xbe\xf2",
  [0x1599] = "\x92\xea", [0x159a] = "\x92\xeb", [0x159b] = "\x92\xec",
  [0x159c] = "\x92\xed", [0x159d] = "\x92\xee", [0x159e] = "\x92\xef",
  [0x159f] = "\x92\xf0", [0x15a0] = "\xc2\xd3", [0x15a1] = "\x92\xf1",
  [0x15a2] = "\xcc\xbd", [0x15a3] = "\xb3\xb8", [0x15a4] = "\x92\xf2",
  [0x15a5] = "\xbd\xd3", [0x15a6] = "\x92\xf3", [0x15a7] = "\xbf\xd8",
  [0x15a8] = "\xcd\xc6", [0x15a9] = "\xd1\xda", [0x15aa] = "\xb4\xeb",
  [0x15ab] = "\x92\xf4", [0x15ac] = "\xde\xe4", [0x15ad] = "\xde\xdd",
  [0x15ae] = "\xde\xe7", [0x15af] = "\x92\xf5", [0x15b0] = "\xea\xfe",
  [0x15b1] = "\x92\xf6", [0x15b2] = "\x92\xf7", [0x15b3] = "\xc2\xb0",
  [0x15b4] = "\xde\xe2", [0x15b5] = "\x92\xf8", [0x15b6] = "\x92\xf9",
  [0x15b7] = "\xd6\xc0", [0x15b8] = "\xb5\xa7", [0x15b9] = "\x92\xfa",
  [0x15ba] = "\xb2\xf4", [0x15bb] = "\x92\xfb", [0x15bc] = "\xde\xe8",
  [0x15bd] = "\x92\xfc", [0x15be] = "\xde\xf2", [0x15bf] = "\x92\xfd",
  [0x15c0] = "\x92\xfe", [0x15c1] = "\x93\x40", [0x15c2] = "\x93\x41",
  [0x15c3] = "\x93\x42", [0x15c4] = "\xde\xed", [0x15c5] = "\x93\x43",
  [0x15c6] = "\xde\xf1", [0x15c7] = "\x93\x44", [0x15c8] = "\x93\x45",
  [0x15c9] = "\xc8\xe0", [0x15ca] = "\x93\x46", [0x15cb] = "\x93\x47",
  [0x15cc] = "\x93\x48", [0x15cd] = "\xd7\xe1", [0x15ce] = "\xde\xef",
  [0x15cf] = "\xc3\xe8", [0x15d0] = "\xcc\xe1", [0x15d1] = "\x93\x49",
  [0x15d2] = "\xb2\xe5", [0x15d3] = "\x93\x4a", [0x15d4] = "\x93\x4b",
  [0x15d5] = "\x93\x4c", [0x15d6] = "\xd2\xbe", [0x15d7] = "\x93\x4d",
  [0x15d8] = "\x93\x4e", [0x15d9] = "\x93\x4f", [0x15da] = "\x93\x50",
  [0x15db] = "\x93\x51", [0x15dc] = "\x93\x52", [0x15dd] = "\x93\x53",
  [0x15de] = "\xde\xee", [0x15df] = "\x93\x54", [0x15e0] = "\xde\xeb",
  [0x15e1] = "\xce\xd5", [0x15e2] = "\x93\x55", [0x15e3] = "\xb4\xa7",
  [0x15e4] = "\x93\x56", [0x15e5] = "\x93\x57", [0x15e6] = "\x93\x58",
  [0x15e7] = "\x93\x59", [0x15e8] = "\x93\x5a", [0x15e9] = "\xbf\xab",
  [0x15ea] = "\xbe\xbe", [0x15eb] = "\x93\x5b", [0x15ec] = "\x93\x5c",
  [0x15ed] = "\xbd\xd2", [0x15ee] = "\x93\x5d", [0x15ef] = "\x93\x5e",
  [0x15f0] = "\x93\x5f", [0x15f1] = "\x93\x60", [0x15f2] = "\xde\xe9",
  [0x15f3] = "\x93\x61", [0x15f4] = "\xd4\xae", [0x15f5] = "\x93\x62",
  [0x15f6] = "\xde\xde", [0x15f7] = "\x93\x63", [0x15f8] = "\xde\xea",
  [0x15f9] = "\x93\x64", [0x15fa] = "\x93\x65", [0x15fb] = "\x93\x66",
  [0x15fc] = "\x93\x67", [0x15fd] = "\xc0\xbf", [0x15fe] = "\x93\x68",
  [0x15ff] = "\xde\xec", [0x1600] = "\xb2\xf3", [0x1601] = "\xb8\xe9",
  [0x1602] = "\xc2\xa7", [0x1603] = "\x93\x69", [0x1604] = "\x93\x6a",
  [0x1605] = "\xbd\xc1", [0x1606] = "\x93\x6b", [0x1607] = "\x93\x6c",
  [0x1608] = "\x93\x6d", [0x1609] = "\x93\x6e", [0x160a] = "\x93\x6f",
  [0x160b] = "\xde\xf5", [0x160c] = "\xde\xf8", [0x160d] = "\x93\x70",
  [0x160e] = "\x93\x71", [0x160f] = "\xb2\xab", [0x1610] = "\xb4\xa4",
  [0x1611] = "\x93\x72", [0x1612] = "\x93\x73", [0x1613] = "\xb4\xea",
  [0x1614] = "\xc9\xa6", [0x1615] = "\x93\x74", [0x1616] = "\x93\x75",
  [0x1617] = "\x93\x76", [0x1618] = "\x93\x77", [0x1619] = "\x93\x78",
  [0x161a] = "\x93\x79", [0x161b] = "\xde\xf6", [0x161c] = "\xcb\xd1",
  [0x161d] = "\x93\x7a", [0x161e] = "\xb8\xe3", [0x161f] = "\x93\x7b",
  [0x1620] = "\xde\xf7", [0x1621] = "\xde\xfa", [0x1622] = "\x93\x7c",
  [0x1623] = "\x93\x7d", [0x1624] = "\x93\x7e", [0x1625] = "\x93\x80",
  [0x1626] = "\xde\xf9", [0x1627] = "\x93\x81", [0x1628] = "\x93\x82",
  [0x1629] = "\x93\x83", [0x162a] = "\xcc\xc2", [0x162b] = "\x93\x84",
  [0x162c] = "\xb0\xe1", [0x162d] = "\xb4\xee", [0x162e] = "\x93\x85",
  [0x162f] = "\x93\x86", [0x1630] = "\x93\x87", [0x1631] = "\x93\x88",
  [0x1632] = "\x93\x89", [0x1633] = "\x93\x8a", [0x1634] = "\xe5\xba",
  [0x1635] = "\x93\x8b", [0x1636] = "\x93\x8c", [0x1637] = "\x93\x8d",
  [0x1638] = "\x93\x8e", [0x1639] = "\x93\x8f", [0x163a] = "\xd0\xaf",
  [0x163b] = "\x93\x90", [0x163c] = "\x93\x91", [0x163d] = "\xb2\xeb",
  [0x163e] = "\x93\x92", [0x163f] = "\xeb\xa1", [0x1640] = "\x93\x93",
  [0x1641] = "\xde\xf4", [0x1642] = "\x93\x94", [0x1643] = "\x93\x95",
  [0x1644] = "\xc9\xe3", [0x1645] = "\xde\xf3", [0x1646] = "\xb0\xda",
  [0x1647] = "\xd2\xa1", [0x1648] = "\xb1\xf7", [0x1649] = "\x93\x96",
  [0x164a] = "\xcc\xaf", [0x164b] = "\x93\x97", [0x164c] = "\x93\x98",
  [0x164d] = "\x93\x99", [0x164e] = "\x93\x9a", [0x164f] = "\x93\x9b",
  [0x1650] = "\x93\x9c", [0x1651] = "\x93\x9d", [0x1652] = "\xde\xf0",
  [0x1653] = "\x93\x9e", [0x1654] = "\xcb\xa4", [0x1655] = "\x93\x9f",
  [0x1656] = "\x93\xa0", [0x1657] = "\x93\xa1", [0x1658] = "\xd5\xaa",
  [0x1659] = "\x93\xa2", [0x165a] = "\x93\xa3", [0x165b] = "\x93\xa4",
  [0x165c] = "\x93\xa5", [0x165d] = "\x93\xa6", [0x165e] = "\xde\xfb",
  [0x165f] = "\x93\xa7", [0x1660] = "\x93\xa8", [0x1661] = "\x93\xa9",
  [0x1662] = "\x93\xaa", [0x1663] = "\x93\xab", [0x1664] = "\x93\xac",
  [0x1665] = "\x93\xad", [0x1666] = "\x93\xae", [0x1667] = "\xb4\xdd",
  [0x1668] = "\x93\xaf", [0x1669] = "\xc4\xa6", [0x166a] = "\x93\xb0",
  [0x166b] = "\x93\xb1", [0x166c] = "\x93\xb2", [0x166d] = "\xde\xfd",
  [0x166e] = "\x93\xb3", [0x166f] = "\x93\xb4", [0x1670] = "\x93\xb5",
  [0x1671] = "\x93\xb6", [0x1672] = "\x93\xb7", [0x1673] = "\x93\xb8",
  [0x1674] = "\x93\xb9", [0x1675] = "\x93\xba", [0x1676] = "\x93\xbb",
  [0x1677] = "\x93\xbc", [0x1678] = "\xc3\xfe", [0x1679] = "\xc4\xa1",
  [0x167a] = "\xdf\xa1", [0x167b] = "\x93\xbd", [0x167c] = "\x93\xbe",
  [0x167d] = "\x93\xbf", [0x167e] = "\x93\xc0", [0x167f] = "\x93\xc1",
  [0x1680] = "\x93\xc2", [0x1681] = "\x93\xc3", [0x1682] = "\xc1\xcc",
  [0x1683] = "\x93\xc4", [0x1684] = "\xde\xfc", [0x1685] = "\xbe\xef",
  [0x1686] = "\x93\xc5", [0x1687] = "\xc6\xb2", [0x1688] = "\x93\xc6",
  [0x1689] = "\x93\xc7", [0x168a] = "\x93\xc8", [0x168b] = "\x93\xc9",
  [0x168c] = "\x93\xca", [0x168d] = "\x93\xcb", [0x168e] = "\x93\xcc",
  [0x168f] = "\x93\xcd", [0x1690] = "\x93\xce", [0x1691] = "\xb3\xc5",
  [0x1692] = "\xc8\xf6", [0x1693] = "\x93\xcf", [0x1694] = "\x93\xd0",
  [0x1695] = "\xcb\xba", [0x1696] = "\xde\xfe", [0x1697] = "\x93\xd1",
  [0x1698] = "\x93\xd2", [0x1699] = "\xdf\xa4", [0x169a] = "\x93\xd3",
  [0x169b] = "\x93\xd4", [0x169c] = "\x93\xd5", [0x169d] = "\x93\xd6",
  [0x169e] = "\xd7\xb2", [0x169f] = "\x93\xd7", [0x16a0] = "\x93\xd8",
  [0x16a1] = "\x93\xd9", [0x16a2] = "\x93\xda", [0x16a3] = "\x93\xdb",
  [0x16a4] = "\xb3\xb7", [0x16a5] = "\x93\xdc", [0x16a6] = "\x93\xdd",
  [0x16a7] = "\x93\xde", [0x16a8] = "\x93\xdf", [0x16a9] = "\xc1\xc3",
  [0x16aa] = "\x93\xe0", [0x16ab] = "\x93\xe1", [0x16ac] = "\xc7\xcb",
  [0x16ad] = "\xb2\xa5", [0x16ae] = "\xb4\xe9", [0x16af] = "\x93\xe2",
  [0x16b0] = "\xd7\xab", [0x16b1] = "\x93\xe3", [0x16b2] = "\x93\xe4",
  [0x16b3] = "\x93\xe5", [0x16b4] = "\x93\xe6", [0x16b5] = "\xc4\xec",
  [0x16b6] = "\x93\xe7", [0x16b7] = "\xdf\xa2", [0x16b8] = "\xdf\xa3",
  [0x16b9] = "\x93\xe8", [0x16ba] = "\xdf\xa5", [0x16bb] = "\x93\xe9",
  [0x16bc] = "\xba\xb3", [0x16bd] = "\x93\xea", [0x16be] = "\x93\xeb",
  [0x16bf] = "\x93\xec", [0x16c0] = "\xdf\xa6", [0x16c1] = "\x93\xed",
  [0x16c2] = "\xc0\xde", [0x16c3] = "\x93\xee", [0x16c4] = "\x93\xef",
  [0x16c5] = "\xc9\xc3", [0x16c6] = "\x93\xf0", [0x16c7] = "\x93\xf1",
  [0x16c8] = "\x93\xf2", [0x16c9] = "\x93\xf3", [0x16ca] = "\x93\xf4",
  [0x16cb] = "\x93\xf5", [0x16cc] = "\x93\xf6", [0x16cd] = "\xb2\xd9",
  [0x16ce] = "\xc7\xe6", [0x16cf] = "\x93\xf7", [0x16d0] = "\xdf\xa7",
  [0x16d1] = "\x93\xf8", [0x16d2] = "\xc7\xdc", [0x16d3] = "\x93\xf9",
  [0x16d4] = "\x93\xfa", [0x16d5] = "\x93\xfb", [0x16d6] = "\x93\xfc",
  [0x16d7] = "\xdf\xa8", [0x16d8] = "\xeb\xa2", [0x16d9] = "\x93\xfd",
  [0x16da] = "\x93\xfe", [0x16db] = "\x94\x40", [0x16dc] = "\x94\x41",
  [0x16dd] = "\x94\x42", [0x16de] = "\xcb\xd3", [0x16df] = "\x94\x43",
  [0x16e0] = "\x94\x44", [0x16e1] = "\x94\x45", [0x16e2] = "\xdf\xaa",
  [0x16e3] = "\x94\x46", [0x16e4] = "\xdf\xa9", [0x16e5] = "\x94\x47",
  [0x16e6] = "\xb2\xc1", [0x16e7] = "\x94\x48", [0x16e8] = "\x94\x49",
  [0x16e9] = "\x94\x4a", [0x16ea] = "\x94\x4b", [0x16eb] = "\x94\x4c",
  [0x16ec] = "\x94\x4d", [0x16ed] = "\x94\x4e", [0x16ee] = "\x94\x4f",
  [0x16ef] = "\x94\x50", [0x16f0] = "\x94\x51", [0x16f1] = "\x94\x52",
  [0x16f2] = "\x94\x53", [0x16f3] = "\x94\x54", [0x16f4] = "\x94\x55",
  [0x16f5] = "\x94\x56", [0x16f6] = "\x94\x57", [0x16f7] = "\x94\x58",
  [0x16f8] = "\x94\x59", [0x16f9] = "\x94\x5a", [0x16fa] = "\x94\x5b",
  [0x16fb] = "\x94\x5c", [0x16fc] = "\x94\x5d", [0x16fd] = "\x94\x5e",
  [0x16fe] = "\x94\x5f", [0x16ff] = "\x94\x60", [0x1700] = "\xc5\xca",
  [0x1701] = "\x94\x61", [0x1702] = "\x94\x62", [0x1703] = "\x94\x63",
  [0x1704] = "\x94\x64", [0x1705] = "\x94\x65", [0x1706] = "\x94\x66",
  [0x1707] = "\x94\x67", [0x1708] = "\x94\x68", [0x1709] = "\xdf\xab",
  [0x170a] = "\x94\x69", [0x170b] = "\x94\x6a", [0x170c] = "\x94\x6b",
  [0x170d] = "\x94\x6c", [0x170e] = "\x94\x6d", [0x170f] = "\x94\x6e",
  [0x1710] = "\x94\x6f", [0x1711] = "\x94\x70", [0x1712] = "\xd4\xdc",
  [0x1713] = "\x94\x71", [0x1714] = "\x94\x72", [0x1715] = "\x94\x73",
  [0x1716] = "\x94\x74", [0x1717] = "\x94\x75", [0x1718] = "\xc8\xc1",
  [0x1719] = "\x94\x76", [0x171a] = "\x94\x77", [0x171b] = "\x94\x78",
  [0x171c] = "\x94\x79", [0x171d] = "\x94\x7a", [0x171e] = "\x94\x7b",
  [0x171f] = "\x94\x7c", [0x1720] = "\x94\x7d", [0x1721] = "\x94\x7e",
  [0x1722] = "\x94\x80", [0x1723] = "\x94\x81", [0x1724] = "\x94\x82",
  [0x1725] = "\xdf\xac", [0x1726] = "\x94\x83", [0x1727] = "\x94\x84",
  [0x1728] = "\x94\x85", [0x1729] = "\x94\x86", [0x172a] = "\x94\x87",
  [0x172b] = "\xbe\xf0", [0x172c] = "\x94\x88", [0x172d] = "\x94\x89",
  [0x172e] = "\xdf\xad", [0x172f] = "\xd6\xa7", [0x1730] = "\x94\x8a",
  [0x1731] = "\x94\x8b", [0x1732] = "\x94\x8c", [0x1733] = "\x94\x8d",
  [0x1734] = "\xea\xb7", [0x1735] = "\xeb\xb6", [0x1736] = "\xca\xd5",
  [0x1737] = "\x94\x8e", [0x1738] = "\xd8\xfc", [0x1739] = "\xb8\xc4",
  [0x173a] = "\x94\x8f", [0x173b] = "\xb9\xa5", [0x173c] = "\x94\x90",
  [0x173d] = "\x94\x91", [0x173e] = "\xb7\xc5", [0x173f] = "\xd5\xfe",
  [0x1740] = "\x94\x92", [0x1741] = "\x94\x93", [0x1742] = "\x94\x94",
  [0x1743] = "\x94\x95", [0x1744] = "\x94\x96", [0x1745] = "\xb9\xca",
  [0x1746] = "\x94\x97", [0x1747] = "\x94\x98", [0x1748] = "\xd0\xa7",
  [0x1749] = "\xf4\xcd", [0x174a] = "\x94\x99", [0x174b] = "\x94\x9a",
  [0x174c] = "\xb5\xd0", [0x174d] = "\x94\x9b", [0x174e] = "\x94\x9c",
  [0x174f] = "\xc3\xf4", [0x1750] = "\x94\x9d", [0x1751] = "\xbe\xc8",
  [0x1752] = "\x94\x9e", [0x1753] = "\x94\x9f", [0x1754] = "\x94\xa0",
  [0x1755] = "\xeb\xb7", [0x1756] = "\xb0\xbd", [0x1757] = "\x94\xa1",
  [0x1758] = "\x94\xa2", [0x1759] = "\xbd\xcc", [0x175a] = "\x94\xa3",
  [0x175b] = "\xc1\xb2", [0x175c] = "\x94\xa4", [0x175d] = "\xb1\xd6",
  [0x175e] = "\xb3\xa8", [0x175f] = "\x94\xa5", [0x1760] = "\x94\xa6",
  [0x1761] = "\x94\xa7", [0x1762] = "\xb8\xd2", [0x1763] = "\xc9\xa2",
  [0x1764] = "\x94\xa8", [0x1765] = "\x94\xa9", [0x1766] = "\xb6\xd8",
  [0x1767] = "\x94\xaa", [0x1768] = "\x94\xab", [0x1769] = "\x94\xac",
  [0x176a] = "\x94\xad", [0x176b] = "\xeb\xb8", [0x176c] = "\xbe\xb4",
  [0x176d] = "\x94\xae", [0x176e] = "\x94\xaf", [0x176f] = "\x94\xb0",
  [0x1770] = "\xca\xfd", [0x1771] = "\x94\xb1", [0x1772] = "\xc7\xc3",
  [0x1773] = "\x94\xb2", [0x1774] = "\xd5\xfb", [0x1775] = "\x94\xb3",
  [0x1776] = "\x94\xb4", [0x1777] = "\xb7\xf3", [0x1778] = "\x94\xb5",
  [0x1779] = "\x94\xb6", [0x177a] = "\x94\xb7", [0x177b] = "\x94\xb8",
  [0x177c] = "\x94\xb9", [0x177d] = "\x94\xba", [0x177e] = "\x94\xbb",
  [0x177f] = "\x94\xbc", [0x1780] = "\x94\xbd", [0x1781] = "\x94\xbe",
  [0x1782] = "\x94\xbf", [0x1783] = "\x94\xc0", [0x1784] = "\x94\xc1",
  [0x1785] = "\x94\xc2", [0x1786] = "\x94\xc3", [0x1787] = "\xce\xc4",
  [0x1788] = "\x94\xc4", [0x1789] = "\x94\xc5", [0x178a] = "\x94\xc6",
  [0x178b] = "\xd5\xab", [0x178c] = "\xb1\xf3", [0x178d] = "\x94\xc7",
  [0x178e] = "\x94\xc8", [0x178f] = "\x94\xc9", [0x1790] = "\xec\xb3",
  [0x1791] = "\xb0\xdf", [0x1792] = "\x94\xca", [0x1793] = "\xec\xb5",
  [0x1794] = "\x94\xcb", [0x1795] = "\x94\xcc", [0x1796] = "\x94\xcd",
  [0x1797] = "\xb6\xb7", [0x1798] = "\x94\xce", [0x1799] = "\xc1\xcf",
  [0x179a] = "\x94\xcf", [0x179b] = "\xf5\xfa", [0x179c] = "\xd0\xb1",
  [0x179d] = "\x94\xd0", [0x179e] = "\x94\xd1", [0x179f] = "\xd5\xe5",
  [0x17a0] = "\x94\xd2", [0x17a1] = "\xce\xd3", [0x17a2] = "\x94\xd3",
  [0x17a3] = "\x94\xd4", [0x17a4] = "\xbd\xef", [0x17a5] = "\xb3\xe2",
  [0x17a6] = "\x94\xd5", [0x17a7] = "\xb8\xab", [0x17a8] = "\x94\xd6",
  [0x17a9] = "\xd5\xb6", [0x17aa] = "\x94\xd7", [0x17ab] = "\xed\xbd",
  [0x17ac] = "\x94\xd8", [0x17ad] = "\xb6\xcf", [0x17ae] = "\x94\xd9",
  [0x17af] = "\xcb\xb9", [0x17b0] = "\xd0\xc2", [0x17b1] = "\x94\xda",
  [0x17b2] = "\x94\xdb", [0x17b3] = "\x94\xdc", [0x17b4] = "\x94\xdd",
  [0x17b5] = "\x94\xde", [0x17b6] = "\x94\xdf", [0x17b7] = "\x94\xe0",
  [0x17b8] = "\x94\xe1", [0x17b9] = "\xb7\xbd", [0x17ba] = "\x94\xe2",
  [0x17bb] = "\x94\xe3", [0x17bc] = "\xec\xb6", [0x17bd] = "\xca\xa9",
  [0x17be] = "\x94\xe4", [0x17bf] = "\x94\xe5", [0x17c0] = "\x94\xe6",
  [0x17c1] = "\xc5\xd4", [0x17c2] = "\x94\xe7", [0x17c3] = "\xec\xb9",
  [0x17c4] = "\xec\xb8", [0x17c5] = "\xc2\xc3", [0x17c6] = "\xec\xb7",
  [0x17c7] = "\x94\xe8", [0x17c8] = "\x94\xe9", [0x17c9] = "\x94\xea",
  [0x17ca] = "\x94\xeb", [0x17cb] = "\xd0\xfd", [0x17cc] = "\xec\xba",
  [0x17cd] = "\x94\xec", [0x17ce] = "\xec\xbb", [0x17cf] = "\xd7\xe5",
  [0x17d0] = "\x94\xed", [0x17d1] = "\x94\xee", [0x17d2] = "\xec\xbc",
  [0x17d3] = "\x94\xef", [0x17d4] = "\x94\xf0", [0x17d5] = "\x94\xf1",
  [0x17d6] = "\xec\xbd", [0x17d7] = "\xc6\xec", [0x17d8] = "\x94\xf2",
  [0x17d9] = "\x94\xf3", [0x17da] = "\x94\xf4", [0x17db] = "\x94\xf5",
  [0x17dc] = "\x94\xf6", [0x17dd] = "\x94\xf7", [0x17de] = "\x94\xf8",
  [0x17df] = "\x94\xf9", [0x17e0] = "\xce\xde", [0x17e1] = "\x94\xfa",
  [0x17e2] = "\xbc\xc8", [0x17e3] = "\x94\xfb", [0x17e4] = "\x94\xfc",
  [0x17e5] = "\xc8\xd5", [0x17e6] = "\xb5\xa9", [0x17e7] = "\xbe\xc9",
  [0x17e8] = "\xd6\xbc", [0x17e9] = "\xd4\xe7", [0x17ea] = "\x94\xfd",
  [0x17eb] = "\x94\xfe", [0x17ec] = "\xd1\xae", [0x17ed] = "\xd0\xf1",
  [0x17ee] = "\xea\xb8", [0x17ef] = "\xea\xb9", [0x17f0] = "\xea\xba",
  [0x17f1] = "\xba\xb5", [0x17f2] = "\x95\x40", [0x17f3] = "\x95\x41",
  [0x17f4] = "\x95\x42", [0x17f5] = "\x95\x43", [0x17f6] = "\xca\xb1",
  [0x17f7] = "\xbf\xf5", [0x17f8] = "\x95\x44", [0x17f9] = "\x95\x45",
  [0x17fa] = "\xcd\xfa", [0x17fb] = "\x95\x46", [0x17fc] = "\x95\x47",
  [0x17fd] = "\x95\x48", [0x17fe] = "\x95\x49", [0x17ff] = "\x95\x4a",
  [0x1800] = "\xea\xc0", [0x1801] = "\x95\x4b", [0x1802] = "\xb0\xba",
  [0x1803] = "\xea\xbe", [0x1804] = "\x95\x4c", [0x1805] = "\x95\x4d",
  [0x1806] = "\xc0\xa5", [0x1807] = "\x95\x4e", [0x1808] = "\x95\x4f",
  [0x1809] = "\x95\x50", [0x180a] = "\xea\xbb", [0x180b] = "\x95\x51",
  [0x180c] = "\xb2\xfd", [0x180d] = "\x95\x52", [0x180e] = "\xc3\xf7",
  [0x180f] = "\xbb\xe8", [0x1810] = "\x95\x53", [0x1811] = "\x95\x54",
  [0x1812] = "\x95\x55", [0x1813] = "\xd2\xd7", [0x1814] = "\xce\xf4",
  [0x1815] = "\xea\xbf", [0x1816] = "\x95\x56", [0x1817] = "\x95\x57",
  [0x1818] = "\x95\x58", [0x1819] = "\xea\xbc", [0x181a] = "\x95\x59",
  [0x181b] = "\x95\x5a", [0x181c] = "\x95\x5b", [0x181d] = "\xea\xc3",
  [0x181e] = "\x95\x5c", [0x181f] = "\xd0\xc7", [0x1820] = "\xd3\xb3",
  [0x1821] = "\x95\x5d", [0x1822] = "\x95\x5e", [0x1823] = "\x95\x5f",
  [0x1824] = "\x95\x60", [0x1825] = "\xb4\xba", [0x1826] = "\x95\x61",
  [0x1827] = "\xc3\xc1", [0x1828] = "\xd7\xf2", [0x1829] = "\x95\x62",
  [0x182a] = "\x95\x63", [0x182b] = "\x95\x64", [0x182c] = "\x95\x65",
  [0x182d] = "\xd5\xd1", [0x182e] = "\x95\x66", [0x182f] = "\xca\xc7",
  [0x1830] = "\x95\x67", [0x1831] = "\xea\xc5", [0x1832] = "\x95\x68",
  [0x1833] = "\x95\x69", [0x1834] = "\xea\xc4", [0x1835] = "\xea\xc7",
  [0x1836] = "\xea\xc6", [0x1837] = "\x95\x6a", [0x1838] = "\x95\x6b",
  [0x1839] = "\x95\x6c", [0x183a] = "\x95\x6d", [0x183b] = "\x95\x6e",
  [0x183c] = "\xd6\xe7", [0x183d] = "\x95\x6f", [0x183e] = "\xcf\xd4",
  [0x183f] = "\x95\x70", [0x1840] = "\x95\x71", [0x1841] = "\xea\xcb",
  [0x1842] = "\x95\x72", [0x1843] = "\xbb\xce", [0x1844] = "\x95\x73",
  [0x1845] = "\x95\x74", [0x1846] = "\x95\x75", [0x1847] = "\x95\x76",
  [0x1848] = "\x95\x77", [0x1849] = "\x95\x78", [0x184a] = "\x95\x79",
  [0x184b] = "\xbd\xfa", [0x184c] = "\xc9\xce", [0x184d] = "\x95\x7a",
  [0x184e] = "\x95\x7b", [0x184f] = "\xea\xcc", [0x1850] = "\x95\x7c",
  [0x1851] = "\x95\x7d", [0x1852] = "\xc9\xb9", [0x1853] = "\xcf\xfe",
  [0x1854] = "\xea\xca", [0x1855] = "\xd4\xce", [0x1856] = "\xea\xcd",
  [0x1857] = "\xea\xcf", [0x1858] = "\x95\x7e", [0x1859] = "\x95\x80",
  [0x185a] = "\xcd\xed", [0x185b] = "\x95\x81", [0x185c] = "\x95\x82",
  [0x185d] = "\x95\x83", [0x185e] = "\x95\x84", [0x185f] = "\xea\xc9",
  [0x1860] = "\x95\x85", [0x1861] = "\xea\xce", [0x1862] = "\x95\x86",
  [0x1863] = "\x95\x87", [0x1864] = "\xce\xee", [0x1865] = "\x95\x88",
  [0x1866] = "\xbb\xde", [0x1867] = "\x95\x89", [0x1868] = "\xb3\xbf",
  [0x1869] = "\x95\x8a", [0x186a] = "\x95\x8b", [0x186b] = "\x95\x8c",
  [0x186c] = "\x95\x8d", [0x186d] = "\x95\x8e", [0x186e] = "\xc6\xd5",
  [0x186f] = "\xbe\xb0", [0x1870] = "\xce\xfa", [0x1871] = "\x95\x8f",
  [0x1872] = "\x95\x90", [0x1873] = "\x95\x91", [0x1874] = "\xc7\xe7",
  [0x1875] = "\x95\x92", [0x1876] = "\xbe\xa7", [0x1877] = "\xea\xd0",
  [0x1878] = "\x95\x93", [0x1879] = "\x95\x94", [0x187a] = "\xd6\xc7",
  [0x187b] = "\x95\x95", [0x187c] = "\x95\x96", [0x187d] = "\x95\x97",
  [0x187e] = "\xc1\xc0", [0x187f] = "\x95\x98", [0x1880] = "\x95\x99",
  [0x1881] = "\x95\x9a", [0x1882] = "\xd4\xdd", [0x1883] = "\x95\x9b",
  [0x1884] = "\xea\xd1", [0x1885] = "\x95\x9c", [0x1886] = "\x95\x9d",
  [0x1887] = "\xcf\xbe", [0x1888] = "\x95\x9e", [0x1889] = "\x95\x9f",
  [0x188a] = "\x95\xa0", [0x188b] = "\x95\xa1", [0x188c] = "\xea\xd2",
  [0x188d] = "\x95\xa2", [0x188e] = "\x95\xa3", [0x188f] = "\x95\xa4",
  [0x1890] = "\x95\xa5", [0x1891] = "\xca\xee", [0x1892] = "\x95\xa6",
  [0x1893] = "\x95\xa7", [0x1894] = "\x95\xa8", [0x1895] = "\x95\xa9",
  [0x1896] = "\xc5\xaf", [0x1897] = "\xb0\xb5", [0x1898] = "\x95\xaa",
  [0x1899] = "\x95\xab", [0x189a] = "\x95\xac", [0x189b] = "\x95\xad",
  [0x189c] = "\x95\xae", [0x189d] = "\xea\xd4", [0x189e] = "\x95\xaf",
  [0x189f] = "\x95\xb0", [0x18a0] = "\x95\xb1", [0x18a1] = "\x95\xb2",
  [0x18a2] = "\x95\xb3", [0x18a3] = "\x95\xb4", [0x18a4] = "\x95\xb5",
  [0x18a5] = "\x95\xb6", [0x18a6] = "\x95\xb7", [0x18a7] = "\xea\xd3",
  [0x18a8] = "\xf4\xdf", [0x18a9] = "\x95\xb8", [0x18aa] = "\x95\xb9",
  [0x18ab] = "\x95\xba", [0x18ac] = "\x95\xbb", [0x18ad] = "\x95\xbc",
  [0x18ae] = "\xc4\xba", [0x18af] = "\x95\xbd", [0x18b0] = "\x95\xbe",
  [0x18b1] = "\x95\xbf", [0x18b2] = "\x95\xc0", [0x18b3] = "\x95\xc1",
  [0x18b4] = "\xb1\xa9", [0x18b5] = "\x95\xc2", [0x18b6] = "\x95\xc3",
  [0x18b7] = "\x95\xc4", [0x18b8] = "\x95\xc5", [0x18b9] = "\xe5\xdf",
  [0x18ba] = "\x95\xc6", [0x18bb] = "\x95\xc7", [0x18bc] = "\x95\xc8",
  [0x18bd] = "\x95\xc9", [0x18be] = "\xea\xd5", [0x18bf] = "\x95\xca",
  [0x18c0] = "\x95\xcb", [0x18c1] = "\x95\xcc", [0x18c2] = "\x95\xcd",
  [0x18c3] = "\x95\xce", [0x18c4] = "\x95\xcf", [0x18c5] = "\x95\xd0",
  [0x18c6] = "\x95\xd1", [0x18c7] = "\x95\xd2", [0x18c8] = "\x95\xd3",
  [0x18c9] = "\x95\xd4", [0x18ca] = "\x95\xd5", [0x18cb] = "\x95\xd6",
  [0x18cc] = "\x95\xd7", [0x18cd] = "\x95\xd8", [0x18ce] = "\x95\xd9",
  [0x18cf] = "\x95\xda", [0x18d0] = "\x95\xdb", [0x18d1] = "\x95\xdc",
  [0x18d2] = "\x95\xdd", [0x18d3] = "\x95\xde", [0x18d4] = "\x95\xdf",
  [0x18d5] = "\x95\xe0", [0x18d6] = "\x95\xe1", [0x18d7] = "\x95\xe2",
  [0x18d8] = "\x95\xe3", [0x18d9] = "\xca\xef", [0x18da] = "\x95\xe4",
  [0x18db] = "\xea\xd6", [0x18dc] = "\xea\xd7", [0x18dd] = "\xc6\xd8",
  [0x18de] = "\x95\xe5", [0x18df] = "\x95\xe6", [0x18e0] = "\x95\xe7",
  [0x18e1] = "\x95\xe8", [0x18e2] = "\x95\xe9", [0x18e3] = "\x95\xea",
  [0x18e4] = "\x95\xeb", [0x18e5] = "\x95\xec", [0x18e6] = "\xea\xd8",
  [0x18e7] = "\x95\xed", [0x18e8] = "\x95\xee", [0x18e9] = "\xea\xd9",
  [0x18ea] = "\x95\xef", [0x18eb] = "\x95\xf0", [0x18ec] = "\x95\xf1",
  [0x18ed] = "\x95\xf2", [0x18ee] = "\x95\xf3", [0x18ef] = "\x95\xf4",
  [0x18f0] = "\xd4\xbb", [0x18f1] = "\x95\xf5", [0x18f2] = "\xc7\xfa",
  [0x18f3] = "\xd2\xb7", [0x18f4] = "\xb8\xfc", [0x18f5] = "\x95\xf6",
  [0x18f6] = "\x95\xf7", [0x18f7] = "\xea\xc2", [0x18f8] = "\x95\xf8",
  [0x18f9] = "\xb2\xdc", [0x18fa] = "\x95\xf9", [0x18fb] = "\x95\xfa",
  [0x18fc] = "\xc2\xfc", [0x18fd] = "\x95\xfb", [0x18fe] = "\xd4\xf8",
  [0x18ff] = "\xcc\xe6", [0x1900] = "\xd7\xee", [0x1901] = "\x95\xfc",
  [0x1902] = "\x95\xfd", [0x1903] = "\x95\xfe", [0x1904] = "\x96\x40",
  [0x1905] = "\x96\x41", [0x1906] = "\x96\x42", [0x1907] = "\x96\x43",
  [0x1908] = "\xd4\xc2", [0x1909] = "\xd3\xd0", [0x190a] = "\xeb\xc3",
  [0x190b] = "\xc5\xf3", [0x190c] = "\x96\x44", [0x190d] = "\xb7\xfe",
  [0x190e] = "\x96\x45", [0x190f] = "\x96\x46", [0x1910] = "\xeb\xd4",
  [0x1911] = "\x96\x47", [0x1912] = "\x96\x48", [0x1913] = "\x96\x49",
  [0x1914] = "\xcb\xb7", [0x1915] = "\xeb\xde", [0x1916] = "\x96\x4a",
  [0x1917] = "\xc0\xca", [0x1918] = "\x96\x4b", [0x1919] = "\x96\x4c",
  [0x191a] = "\x96\x4d", [0x191b] = "\xcd\xfb", [0x191c] = "\x96\x4e",
  [0x191d] = "\xb3\xaf", [0x191e] = "\x96\x4f", [0x191f] = "\xc6\xda",
  [0x1920] = "\x96\x50", [0x1921] = "\x96\x51", [0x1922] = "\x96\x52",
  [0x1923] = "\x96\x53", [0x1924] = "\x96\x54", [0x1925] = "\x96\x55",
  [0x1926] = "\xeb\xfc", [0x1927] = "\x96\x56", [0x1928] = "\xc4\xbe",
  [0x1929] = "\x96\x57", [0x192a] = "\xce\xb4", [0x192b] = "\xc4\xa9",
  [0x192c] = "\xb1\xbe", [0x192d] = "\xd4\xfd", [0x192e] = "\x96\x58",
  [0x192f] = "\xca\xf5", [0x1930] = "\x96\x59", [0x1931] = "\xd6\xec",
  [0x1932] = "\x96\x5a", [0x1933] = "\x96\x5b", [0x1934] = "\xc6\xd3",
  [0x1935] = "\xb6\xe4", [0x1936] = "\x96\x5c", [0x1937] = "\x96\x5d",
  [0x1938] = "\x96\x5e", [0x1939] = "\x96\x5f", [0x193a] = "\xbb\xfa",
  [0x193b] = "\x96\x60", [0x193c] = "\x96\x61", [0x193d] = "\xd0\xe0",
  [0x193e] = "\x96\x62", [0x193f] = "\x96\x63", [0x1940] = "\xc9\xb1",
  [0x1941] = "\x96\x64", [0x1942] = "\xd4\xd3", [0x1943] = "\xc8\xa8",
  [0x1944] = "\x96\x65", [0x1945] = "\x96\x66", [0x1946] = "\xb8\xcb",
  [0x1947] = "\x96\x67", [0x1948] = "\xe8\xbe", [0x1949] = "\xc9\xbc",
  [0x194a] = "\x96\x68", [0x194b] = "\x96\x69", [0x194c] = "\xe8\xbb",
  [0x194d] = "\x96\x6a", [0x194e] = "\xc0\xee", [0x194f] = "\xd0\xd3",
  [0x1950] = "\xb2\xc4", [0x1951] = "\xb4\xe5", [0x1952] = "\x96\x6b",
  [0x1953] = "\xe8\xbc", [0x1954] = "\x96\x6c", [0x1955] = "\x96\x6d",
  [0x1956] = "\xd5\xc8", [0x1957] = "\x96\x6e", [0x1958] = "\x96\x6f",
  [0x1959] = "\x96\x70", [0x195a] = "\x96\x71", [0x195b] = "\x96\x72",
  [0x195c] = "\xb6\xc5", [0x195d] = "\x96\x73", [0x195e] = "\xe8\xbd",
  [0x195f] = "\xca\xf8", [0x1960] = "\xb8\xdc", [0x1961] = "\xcc\xf5",
  [0x1962] = "\x96\x74", [0x1963] = "\x96\x75", [0x1964] = "\x96\x76",
  [0x1965] = "\xc0\xb4", [0x1966] = "\x96\x77", [0x1967] = "\x96\x78",
  [0x1968] = "\xd1\xee", [0x1969] = "\xe8\xbf", [0x196a] = "\xe8\xc2",
  [0x196b] = "\x96\x79", [0x196c] = "\x96\x7a", [0x196d] = "\xba\xbc",
  [0x196e] = "\x96\x7b", [0x196f] = "\xb1\xad", [0x1970] = "\xbd\xdc",
  [0x1971] = "\x96\x7c", [0x1972] = "\xea\xbd", [0x1973] = "\xe8\xc3",
  [0x1974] = "\x96\x7d", [0x1975] = "\xe8\xc6", [0x1976] = "\x96\x7e",
  [0x1977] = "\xe8\xcb", [0x1978] = "\x96\x80", [0x1979] = "\x96\x81",
  [0x197a] = "\x96\x82", [0x197b] = "\x96\x83", [0x197c] = "\xe8\xcc",
  [0x197d] = "\x96\x84", [0x197e] = "\xcb\xc9", [0x197f] = "\xb0\xe5",
  [0x1980] = "\x96\x85", [0x1981] = "\xbc\xab", [0x1982] = "\x96\x86",
  [0x1983] = "\x96\x87", [0x1984] = "\xb9\xb9", [0x1985] = "\x96\x88",
  [0x1986] = "\x96\x89", [0x1987] = "\xe8\xc1", [0x1988] = "\x96\x8a",
  [0x1989] = "\xcd\xf7", [0x198a] = "\x96\x8b", [0x198b] = "\xe8\xca",
  [0x198c] = "\x96\x8c", [0x198d] = "\x96\x8d", [0x198e] = "\x96\x8e",
  [0x198f] = "\x96\x8f", [0x1990] = "\xce\xf6", [0x1991] = "\x96\x90",
  [0x1992] = "\x96\x91", [0x1993] = "\x96\x92", [0x1994] = "\x96\x93",
  [0x1995] = "\xd5\xed", [0x1996] = "\x96\x94", [0x1997] = "\xc1\xd6",
  [0x1998] = "\xe8\xc4", [0x1999] = "\x96\x95", [0x199a] = "\xc3\xb6",
  [0x199b] = "\x96\x96", [0x199c] = "\xb9\xfb", [0x199d] = "\xd6\xa6",
  [0x199e] = "\xe8\xc8", [0x199f] = "\x96\x97", [0x19a0] = "\x96\x98",
  [0x19a1] = "\x96\x99", [0x19a2] = "\xca\xe0", [0x19a3] = "\xd4\xe6",
  [0x19a4] = "\x96\x9a", [0x19a5] = "\xe8\xc0", [0x19a6] = "\x96\x9b",
  [0x19a7] = "\xe8\xc5", [0x19a8] = "\xe8\xc7", [0x19a9] = "\x96\x9c",
  [0x19aa] = "\xc7\xb9", [0x19ab] = "\xb7\xe3", [0x19ac] = "\x96\x9d",
  [0x19ad] = "\xe8\xc9", [0x19ae] = "\x96\x9e", [0x19af] = "\xbf\xdd",
  [0x19b0] = "\xe8\xd2", [0x19b1] = "\x96\x9f", [0x19b2] = "\x96\xa0",
  [0x19b3] = "\xe8\xd7", [0x19b4] = "\x96\xa1", [0x19b5] = "\xe8\xd5",
  [0x19b6] = "\xbc\xdc", [0x19b7] = "\xbc\xcf", [0x19b8] = "\xe8\xdb",
  [0x19b9] = "\x96\xa2", [0x19ba] = "\x96\xa3", [0x19bb] = "\x96\xa4",
  [0x19bc] = "\x96\xa5", [0x19bd] = "\x96\xa6", [0x19be] = "\x96\xa7",
  [0x19bf] = "\x96\xa8", [0x19c0] = "\x96\xa9", [0x19c1] = "\xe8\xde",
  [0x19c2] = "\x96\xaa", [0x19c3] = "\xe8\xda", [0x19c4] = "\xb1\xfa",
  [0x19c5] = "\x96\xab", [0x19c6] = "\x96\xac", [0x19c7] = "\x96\xad",
  [0x19c8] = "\x96\xae", [0x19c9] = "\x96\xaf", [0x19ca] = "\x96\xb0",
  [0x19cb] = "\x96\xb1", [0x19cc] = "\x96\xb2", [0x19cd] = "\x96\xb3",
  [0x19ce] = "\x96\xb4", [0x19cf] = "\xb0\xd8", [0x19d0] = "\xc4\xb3",
  [0x19d1] = "\xb8\xcc", [0x19d2] = "\xc6\xe2", [0x19d3] = "\xc8\xbe",
  [0x19d4] = "\xc8\xe1", [0x19d5] = "\x96\xb5", [0x19d6] = "\x96\xb6",
  [0x19d7] = "\x96\xb7", [0x19d8] = "\xe8\xcf", [0x19d9] = "\xe8\xd4",
  [0x19da] = "\xe8\xd6", [0x19db] = "\x96\xb8", [0x19dc] = "\xb9\xf1",
  [0x19dd] = "\xe8\xd8", [0x19de] = "\xd7\xf5", [0x19df] = "\x96\xb9",
  [0x19e0] = "\xc4\xfb", [0x19e1] = "\x96\xba", [0x19e2] = "\xe8\xdc",
  [0x19e3] = "\x96\xbb", [0x19e4] = "\x96\xbc", [0x19e5] = "\xb2\xe9",
  [0x19e6] = "\x96\xbd", [0x19e7] = "\x96\xbe", [0x19e8] = "\x96\xbf",
  [0x19e9] = "\xe8\xd1", [0x19ea] = "\x96\xc0", [0x19eb] = "\x96\xc1",
  [0x19ec] = "\xbc\xed", [0x19ed] = "\x96\xc2", [0x19ee] = "\x96\xc3",
  [0x19ef] = "\xbf\xc2", [0x19f0] = "\xe8\xcd", [0x19f1] = "\xd6\xf9",
  [0x19f2] = "\x96\xc4", [0x19f3] = "\xc1\xf8", [0x19f4] = "\xb2\xf1",
  [0x19f5] = "\x96\xc5", [0x19f6] = "\x96\xc6", [0x19f7] = "\x96\xc7",
  [0x19f8] = "\x96\xc8", [0x19f9] = "\x96\xc9", [0x19fa] = "\x96\xca",
  [0x19fb] = "\x96\xcb", [0x19fc] = "\x96\xcc", [0x19fd] = "\xe8\xdf",
  [0x19fe] = "\x96\xcd", [0x19ff] = "\xca\xc1", [0x1a00] = "\xe8\xd9",
  [0x1a01] = "\x96\xce", [0x1a02] = "\x96\xcf", [0x1a03] = "\x96\xd0",
  [0x1a04] = "\x96\xd1", [0x1a05] = "\xd5\xa4", [0x1a06] = "\x96\xd2",
  [0x1a07] = "\xb1\xea", [0x1a08] = "\xd5\xbb", [0x1a09] = "\xe8\xce",
  [0x1a0a] = "\xe8\xd0", [0x1a0b] = "\xb6\xb0", [0x1a0c] = "\xe8\xd3",
  [0x1a0d] = "\x96\xd3", [0x1a0e] = "\xe8\xdd", [0x1a0f] = "\xc0\xb8",
  [0x1a10] = "\x96\xd4", [0x1a11] = "\xca\xf7", [0x1a12] = "\x96\xd5",
  [0x1a13] = "\xcb\xa8", [0x1a14] = "\x96\xd6", [0x1a15] = "\x96\xd7",
  [0x1a16] = "\xc6\xdc", [0x1a17] = "\xc0\xf5", [0x1a18] = "\x96\xd8",
  [0x1a19] = "\x96\xd9", [0x1a1a] = "\x96\xda", [0x1a1b] = "\x96\xdb",
  [0x1a1c] = "\x96\xdc", [0x1a1d] = "\xe8\xe9", [0x1a1e] = "\x96\xdd",
  [0x1a1f] = "\x96\xde", [0x1a20] = "\x96\xdf", [0x1a21] = "\xd0\xa3",
  [0x1a22] = "\x96\xe0", [0x1a23] = "\x96\xe1", [0x1a24] = "\x96\xe2",
  [0x1a25] = "\x96\xe3", [0x1a26] = "\x96\xe4", [0x1a27] = "\x96\xe5",
  [0x1a28] = "\x96\xe6", [0x1a29] = "\xe8\xf2", [0x1a2a] = "\xd6\xea",
  [0x1a2b] = "\x96\xe7", [0x1a2c] = "\x96\xe8", [0x1a2d] = "\x96\xe9",
  [0x1a2e] = "\x96\xea", [0x1a2f] = "\x96\xeb", [0x1a30] = "\x96\xec",
  [0x1a31] = "\x96\xed", [0x1a32] = "\xe8\xe0", [0x1a33] = "\xe8\xe1",
  [0x1a34] = "\x96\xee", [0x1a35] = "\x96\xef", [0x1a36] = "\x96\xf0",
  [0x1a37] = "\xd1\xf9", [0x1a38] = "\xba\xcb", [0x1a39] = "\xb8\xf9",
  [0x1a3a] = "\x96\xf1", [0x1a3b] = "\x96\xf2", [0x1a3c] = "\xb8\xf1",
  [0x1a3d] = "\xd4\xd4", [0x1a3e] = "\xe8\xef", [0x1a3f] = "\x96\xf3",
  [0x1a40] = "\xe8\xee", [0x1a41] = "\xe8\xec", [0x1a42] = "\xb9\xf0",
  [0x1a43] = "\xcc\xd2", [0x1a44] = "\xe8\xe6", [0x1a45] = "\xce\xa6",
  [0x1a46] = "\xbf\xf2", [0x1a47] = "\x96\xf4", [0x1a48] = "\xb0\xb8",
  [0x1a49] = "\xe8\xf1", [0x1a4a] = "\xe8\xf0", [0x1a4b] = "\x96\xf5",
  [0x1a4c] = "\xd7\xc0", [0x1a4d] = "\x96\xf6", [0x1a4e] = "\xe8\xe4",
  [0x1a4f] = "\x96\xf7", [0x1a50] = "\xcd\xa9", [0x1a51] = "\xc9\xa3",
  [0x1a52] = "\x96\xf8", [0x1a53] = "\xbb\xb8", [0x1a54] = "\xbd\xdb",
  [0x1a55] = "\xe8\xea", [0x1a56] = "\x96\xf9", [0x1a57] = "\x96\xfa",
  [0x1a58] = "\x96\xfb", [0x1a59] = "\x96\xfc", [0x1a5a] = "\x96\xfd",
  [0x1a5b] = "\x96\xfe", [0x1a5c] = "\x97\x40", [0x1a5d] = "\x97\x41",
  [0x1a5e] = "\x97\x42", [0x1a5f] = "\x97\x43", [0x1a60] = "\xe8\xe2",
  [0x1a61] = "\xe8\xe3", [0x1a62] = "\xe8\xe5", [0x1a63] = "\xb5\xb5",
  [0x1a64] = "\xe8\xe7", [0x1a65] = "\xc7\xc5", [0x1a66] = "\xe8\xeb",
  [0x1a67] = "\xe8\xed", [0x1a68] = "\xbd\xb0", [0x1a69] = "\xd7\xae",
  [0x1a6a] = "\x97\x44", [0x1a6b] = "\xe8\xf8", [0x1a6c] = "\x97\x45",
  [0x1a6d] = "\x97\x46", [0x1a6e] = "\x97\x47", [0x1a6f] = "\x97\x48",
  [0x1a70] = "\x97\x49", [0x1a71] = "\x97\x4a", [0x1a72] = "\x97\x4b",
  [0x1a73] = "\x97\x4c", [0x1a74] = "\xe8\xf5", [0x1a75] = "\x97\x4d",
  [0x1a76] = "\xcd\xb0", [0x1a77] = "\xe8\xf6", [0x1a78] = "\x97\x4e",
  [0x1a79] = "\x97\x4f", [0x1a7a] = "\x97\x50", [0x1a7b] = "\x97\x51",
  [0x1a7c] = "\x97\x52", [0x1a7d] = "\x97\x53", [0x1a7e] = "\x97\x54",
  [0x1a7f] = "\x97\x55", [0x1a80] = "\x97\x56", [0x1a81] = "\xc1\xba",
  [0x1a82] = "\x97\x57", [0x1a83] = "\xe8\xe8", [0x1a84] = "\x97\x58",
  [0x1a85] = "\xc3\xb7", [0x1a86] = "\xb0\xf0", [0x1a87] = "\x97\x59",
  [0x1a88] = "\x97\x5a", [0x1a89] = "\x97\x5b", [0x1a8a] = "\x97\x5c",
  [0x1a8b] = "\x97\x5d", [0x1a8c] = "\x97\x5e", [0x1a8d] = "\x97\x5f",
  [0x1a8e] = "\x97\x60", [0x1a8f] = "\xe8\xf4", [0x1a90] = "\x97\x61",
  [0x1a91] = "\x97\x62", [0x1a92] = "\x97\x63", [0x1a93] = "\xe8\xf7",
  [0x1a94] = "\x97\x64", [0x1a95] = "\x97\x65", [0x1a96] = "\x97\x66",
  [0x1a97] = "\xb9\xa3", [0x1a98] = "\x97\x67", [0x1a99] = "\x97\x68",
  [0x1a9a] = "\x97\x69", [0x1a9b] = "\x97\x6a", [0x1a9c] = "\x97\x6b",
  [0x1a9d] = "\x97\x6c", [0x1a9e] = "\x97\x6d", [0x1a9f] = "\x97\x6e",
  [0x1aa0] = "\x97\x6f", [0x1aa1] = "\x97\x70", [0x1aa2] = "\xc9\xd2",
  [0x1aa3] = "\x97\x71", [0x1aa4] = "\x97\x72", [0x1aa5] = "\x97\x73",
  [0x1aa6] = "\xc3\xce", [0x1aa7] = "\xce\xe0", [0x1aa8] = "\xc0\xe6",
  [0x1aa9] = "\x97\x74", [0x1aaa] = "\x97\x75", [0x1aab] = "\x97\x76",
  [0x1aac] = "\x97\x77", [0x1aad] = "\xcb\xf3", [0x1aae] = "\x97\x78",
  [0x1aaf] = "\xcc\xdd", [0x1ab0] = "\xd0\xb5", [0x1ab1] = "\x97\x79",
  [0x1ab2] = "\x97\x7a", [0x1ab3] = "\xca\xe1", [0x1ab4] = "\x97\x7b",
  [0x1ab5] = "\xe8\xf3", [0x1ab6] = "\x97\x7c", [0x1ab7] = "\x97\x7d",
  [0x1ab8] = "\x97\x7e", [0x1ab9] = "\x97\x80", [0x1aba] = "\x97\x81",
  [0x1abb] = "\x97\x82", [0x1abc] = "\x97\x83", [0x1abd] = "\x97\x84",
  [0x1abe] = "\x97\x85", [0x1abf] = "\x97\x86", [0x1ac0] = "\xbc\xec",
  [0x1ac1] = "\x97\x87", [0x1ac2] = "\xe8\xf9", [0x1ac3] = "\x97\x88",
  [0x1ac4] = "\x97\x89", [0x1ac5] = "\x97\x8a", [0x1ac6] = "\x97\x8b",
  [0x1ac7] = "\x97\x8c", [0x1ac8] = "\x97\x8d", [0x1ac9] = "\xc3\xde",
  [0x1aca] = "\x97\x8e", [0x1acb] = "\xc6\xe5", [0x1acc] = "\x97\x8f",
  [0x1acd] = "\xb9\xf7", [0x1ace] = "\x97\x90", [0x1acf] = "\x97\x91",
  [0x1ad0] = "\x97\x92", [0x1ad1] = "\x97\x93", [0x1ad2] = "\xb0\xf4",
  [0x1ad3] = "\x97\x94", [0x1ad4] = "\x97\x95", [0x1ad5] = "\xd7\xd8",
  [0x1ad6] = "\x97\x96", [0x1ad7] = "\x97\x97", [0x1ad8] = "\xbc\xac",
  [0x1ad9] = "\x97\x98", [0x1ada] = "\xc5\xef", [0x1adb] = "\x97\x99",
  [0x1adc] = "\x97\x9a", [0x1add] = "\x97\x9b", [0x1ade] = "\x97\x9c",
  [0x1adf] = "\x97\x9d", [0x1ae0] = "\xcc\xc4", [0x1ae1] = "\x97\x9e",
  [0x1ae2] = "\x97\x9f", [0x1ae3] = "\xe9\xa6", [0x1ae4] = "\x97\xa0",
  [0x1ae5] = "\x97\xa1", [0x1ae6] = "\x97\xa2", [0x1ae7] = "\x97\xa3",
  [0x1ae8] = "\x97\xa4", [0x1ae9] = "\x97\xa5", [0x1aea] = "\x97\xa6",
  [0x1aeb] = "\x97\xa7", [0x1aec] = "\x97\xa8", [0x1aed] = "\x97\xa9",
  [0x1aee] = "\xc9\xad", [0x1aef] = "\x97\xaa", [0x1af0] = "\xe9\xa2",
  [0x1af1] = "\xc0\xe2", [0x1af2] = "\x97\xab", [0x1af3] = "\x97\xac",
  [0x1af4] = "\x97\xad", [0x1af5] = "\xbf\xc3", [0x1af6] = "\x97\xae",
  [0x1af7] = "\x97\xaf", [0x1af8] = "\x97\xb0", [0x1af9] = "\xe8\xfe",
  [0x1afa] = "\xb9\xd7", [0x1afb] = "\x97\xb1", [0x1afc] = "\xe8\xfb",
  [0x1afd] = "\x97\xb2", [0x1afe] = "\x97\xb3", [0x1aff] = "\x97\xb4",
  [0x1b00] = "\x97\xb5", [0x1b01] = "\xe9\xa4", [0x1b02] = "\x97\xb6",
  [0x1b03] = "\x97\xb7", [0x1b04] = "\x97\xb8", [0x1b05] = "\xd2\xce",
  [0x1b06] = "\x97\xb9", [0x1b07] = "\x97\xba", [0x1b08] = "\x97\xbb",
  [0x1b09] = "\x97\xbc", [0x1b0a] = "\x97\xbd", [0x1b0b] = "\xe9\xa3",
  [0x1b0c] = "\x97\xbe", [0x1b0d] = "\xd6\xb2", [0x1b0e] = "\xd7\xb5",
  [0x1b0f] = "\x97\xbf", [0x1b10] = "\xe9\xa7", [0x1b11] = "\x97\xc0",
  [0x1b12] = "\xbd\xb7", [0x1b13] = "\x97\xc1", [0x1b14] = "\x97\xc2",
  [0x1b15] = "\x97\xc3", [0x1b16] = "\x97\xc4", [0x1b17] = "\x97\xc5",
  [0x1b18] = "\x97\xc6", [0x1b19] = "\x97\xc7", [0x1b1a] = "\x97\xc8",
  [0x1b1b] = "\x97\xc9", [0x1b1c] = "\x97\xca", [0x1b1d] = "\x97\xcb",
  [0x1b1e] = "\x97\xcc", [0x1b1f] = "\xe8\xfc", [0x1b20] = "\xe8\xfd",
  [0x1b21] = "\x97\xcd", [0x1b22] = "\x97\xce", [0x1b23] = "\x97\xcf",
  [0x1b24] = "\xe9\xa1", [0x1b25] = "\x97\xd0", [0x1b26] = "\x97\xd1",
  [0x1b27] = "\x97\xd2", [0x1b28] = "\x97\xd3", [0x1b29] = "\x97\xd4",
  [0x1b2a] = "\x97\xd5", [0x1b2b] = "\x97\xd6", [0x1b2c] = "\x97\xd7",
  [0x1b2d] = "\xcd\xd6", [0x1b2e] = "\x97\xd8", [0x1b2f] = "\x97\xd9",
  [0x1b30] = "\xd2\xac", [0x1b31] = "\x97\xda", [0x1b32] = "\x97\xdb",
  [0x1b33] = "\x97\xdc", [0x1b34] = "\xe9\xb2", [0x1b35] = "\x97\xdd",
  [0x1b36] = "\x97\xde", [0x1b37] = "\x97\xdf", [0x1b38] = "\x97\xe0",
  [0x1b39] = "\xe9\xa9", [0x1b3a] = "\x97\xe1", [0x1b3b] = "\x97\xe2",
  [0x1b3c] = "\x97\xe3", [0x1b3d] = "\xb4\xaa", [0x1b3e] = "\x97\xe4",
  [0x1b3f] = "\xb4\xbb", [0x1b40] = "\x97\xe5", [0x1b41] = "\x97\xe6",
  [0x1b42] = "\xe9\xab", [0x1b43] = "\x97\xe7", [0x1b44] = "\x97\xe8",
  [0x1b45] = "\x97\xe9", [0x1b46] = "\x97\xea", [0x1b47] = "\x97\xeb",
  [0x1b48] = "\x97\xec", [0x1b49] = "\x97\xed", [0x1b4a] = "\x97\xee",
  [0x1b4b] = "\x97\xef", [0x1b4c] = "\x97\xf0", [0x1b4d] = "\x97\xf1",
  [0x1b4e] = "\x97\xf2", [0x1b4f] = "\x97\xf3", [0x1b50] = "\x97\xf4",
  [0x1b51] = "\x97\xf5", [0x1b52] = "\x97\xf6", [0x1b53] = "\x97\xf7",
  [0x1b54] = "\xd0\xa8", [0x1b55] = "\x97\xf8", [0x1b56] = "\x97\xf9",
  [0x1b57] = "\xe9\xa5", [0x1b58] = "\x97\xfa", [0x1b59] = "\x97\xfb",
  [0x1b5a] = "\xb3\xfe", [0x1b5b] = "\x97\xfc", [0x1b5c] = "\x97\xfd",
  [0x1b5d] = "\xe9\xac", [0x1b5e] = "\xc0\xe3", [0x1b5f] = "\x97\xfe",
  [0x1b60] = "\xe9\xaa", [0x1b61] = "\x98\x40", [0x1b62] = "\x98\x41",
  [0x1b63] = "\xe9\xb9", [0x1b64] = "\x98\x42", [0x1b65] = "\x98\x43",
  [0x1b66] = "\xe9\xb8", [0x1b67] = "\x98\x44", [0x1b68] = "\x98\x45",
  [0x1b69] = "\x98\x46", [0x1b6a] = "\x98\x47", [0x1b6b] = "\xe9\xae",
  [0x1b6c] = "\x98\x48", [0x1b6d] = "\x98\x49", [0x1b6e] = "\xe8\xfa",
  [0x1b6f] = "\x98\x4a", [0x1b70] = "\x98\x4b", [0x1b71] = "\xe9\xa8",
  [0x1b72] = "\x98\x4c", [0x1b73] = "\x98\x4d", [0x1b74] = "\x98\x4e",
  [0x1b75] = "\x98\x4f", [0x1b76] = "\x98\x50", [0x1b77] = "\xbf\xac",
  [0x1b78] = "\xe9\xb1", [0x1b79] = "\xe9\xba", [0x1b7a] = "\x98\x51",
  [0x1b7b] = "\x98\x52", [0x1b7c] = "\xc2\xa5", [0x1b7d] = "\x98\x53",
  [0x1b7e] = "\x98\x54", [0x1b7f] = "\x98\x55", [0x1b80] = "\xe9\xaf",
  [0x1b81] = "\x98\x56", [0x1b82] = "\xb8\xc5", [0x1b83] = "\x98\x57",
  [0x1b84] = "\xe9\xad", [0x1b85] = "\x98\x58", [0x1b86] = "\xd3\xdc",
  [0x1b87] = "\xe9\xb4", [0x1b88] = "\xe9\xb5", [0x1b89] = "\xe9\xb7",
  [0x1b8a] = "\x98\x59", [0x1b8b] = "\x98\x5a", [0x1b8c] = "\x98\x5b",
  [0x1b8d] = "\xe9\xc7", [0x1b8e] = "\x98\x5c", [0x1b8f] = "\x98\x5d",
  [0x1b90] = "\x98\x5e", [0x1b91] = "\x98\x5f", [0x1b92] = "\x98\x60",
  [0x1b93] = "\x98\x61", [0x1b94] = "\xc0\xc6", [0x1b95] = "\xe9\xc5",
  [0x1b96] = "\x98\x62", [0x1b97] = "\x98\x63", [0x1b98] = "\xe9\xb0",
  [0x1b99] = "\x98\x64", [0x1b9a] = "\x98\x65", [0x1b9b] = "\xe9\xbb",
  [0x1b9c] = "\xb0\xf1", [0x1b9d] = "\x98\x66", [0x1b9e] = "\x98\x67",
  [0x1b9f] = "\x98\x68", [0x1ba0] = "\x98\x69", [0x1ba1] = "\x98\x6a",
  [0x1ba2] = "\x98\x6b", [0x1ba3] = "\x98\x6c", [0x1ba4] = "\x98\x6d",
  [0x1ba5] = "\x98\x6e", [0x1ba6] = "\x98\x6f", [0x1ba7] = "\xe9\xbc",
  [0x1ba8] = "\xd5\xa5", [0x1ba9] = "\x98\x70", [0x1baa] = "\x98\x71",
  [0x1bab] = "\xe9\xbe", [0x1bac] = "\x98\x72", [0x1bad] = "\xe9\xbf",
  [0x1bae] = "\x98\x73", [0x1baf] = "\x98\x74", [0x1bb0] = "\x98\x75",
  [0x1bb1] = "\xe9\xc1", [0x1bb2] = "\x98\x76", [0x1bb3] = "\x98\x77",
  [0x1bb4] = "\xc1\xf1", [0x1bb5] = "\x98\x78", [0x1bb6] = "\x98\x79",
  [0x1bb7] = "\xc8\xb6", [0x1bb8] = "\x98\x7a", [0x1bb9] = "\x98\x7b",
  [0x1bba] = "\x98\x7c", [0x1bbb] = "\xe9\xbd", [0x1bbc] = "\x98\x7d",
  [0x1bbd] = "\x98\x7e", [0x1bbe] = "\x98\x80", [0x1bbf] = "\x98\x81",
  [0x1bc0] = "\x98\x82", [0x1bc1] = "\xe9\xc2", [0x1bc2] = "\x98\x83",
  [0x1bc3] = "\x98\x84", [0x1bc4] = "\x98\x85", [0x1bc5] = "\x98\x86",
  [0x1bc6] = "\x98\x87", [0x1bc7] = "\x98\x88", [0x1bc8] = "\x98\x89",
  [0x1bc9] = "\x98\x8a", [0x1bca] = "\xe9\xc3", [0x1bcb] = "\x98\x8b",
  [0x1bcc] = "\xe9\xb3", [0x1bcd] = "\x98\x8c", [0x1bce] = "\xe9\xb6",
  [0x1bcf] = "\x98\x8d", [0x1bd0] = "\xbb\xb1", [0x1bd1] = "\x98\x8e",
  [0x1bd2] = "\x98\x8f", [0x1bd3] = "\x98\x90", [0x1bd4] = "\xe9\xc0",
  [0x1bd5] = "\x98\x91", [0x1bd6] = "\x98\x92", [0x1bd7] = "\x98\x93",
  [0x1bd8] = "\x98\x94", [0x1bd9] = "\x98\x95", [0x1bda] = "\x98\x96",
  [0x1bdb] = "\xbc\xf7", [0x1bdc] = "\x98\x97", [0x1bdd] = "\x98\x98",
  [0x1bde] = "\x98\x99", [0x1bdf] = "\xe9\xc4", [0x1be0] = "\xe9\xc6",
  [0x1be1] = "\x98\x9a", [0x1be2] = "\x98\x9b", [0x1be3] = "\x98\x9c",
  [0x1be4] = "\x98\x9d", [0x1be5] = "\x98\x9e", [0x1be6] = "\x98\x9f",
  [0x1be7] = "\x98\xa0", [0x1be8] = "\x98\xa1", [0x1be9] = "\x98\xa2",
  [0x1bea] = "\x98\xa3", [0x1beb] = "\x98\xa4", [0x1bec] = "\x98\xa5",
  [0x1bed] = "\xe9\xca", [0x1bee] = "\x98\xa6", [0x1bef] = "\x98\xa7",
  [0x1bf0] = "\x98\xa8", [0x1bf1] = "\x98\xa9", [0x1bf2] = "\xe9\xce",
  [0x1bf3] = "\x98\xaa", [0x1bf4] = "\x98\xab", [0x1bf5] = "\x98\xac",
  [0x1bf6] = "\x98\xad", [0x1bf7] = "\x98\xae", [0x1bf8] = "\x98\xaf",
  [0x1bf9] = "\x98\xb0", [0x1bfa] = "\x98\xb1", [0x1bfb] = "\x98\xb2",
  [0x1bfc] = "\x98\xb3", [0x1bfd] = "\xb2\xdb", [0x1bfe] = "\x98\xb4",
  [0x1bff] = "\xe9\xc8", [0x1c00] = "\x98\xb5", [0x1c01] = "\x98\xb6",
  [0x1c02] = "\x98\xb7", [0x1c03] = "\x98\xb8", [0x1c04] = "\x98\xb9",
  [0x1c05] = "\x98\xba", [0x1c06] = "\x98\xbb", [0x1c07] = "\x98\xbc",
  [0x1c08] = "\x98\xbd", [0x1c09] = "\x98\xbe", [0x1c0a] = "\xb7\xae",
  [0x1c0b] = "\x98\xbf", [0x1c0c] = "\x98\xc0", [0x1c0d] = "\x98\xc1",
  [0x1c0e] = "\x98\xc2", [0x1c0f] = "\x98\xc3", [0x1c10] = "\x98\xc4",
  [0x1c11] = "\x98\xc5", [0x1c12] = "\x98\xc6", [0x1c13] = "\x98\xc7",
  [0x1c14] = "\x98\xc8", [0x1c15] = "\x98\xc9", [0x1c16] = "\x98\xca",
  [0x1c17] = "\xe9\xcb", [0x1c18] = "\xe9\xcc", [0x1c19] = "\x98\xcb",
  [0x1c1a] = "\x98\xcc", [0x1c1b] = "\x98\xcd", [0x1c1c] = "\x98\xce",
  [0x1c1d] = "\x98\xcf", [0x1c1e] = "\x98\xd0", [0x1c1f] = "\xd5\xc1",
  [0x1c20] = "\x98\xd1", [0x1c21] = "\xc4\xa3", [0x1c22] = "\x98\xd2",
  [0x1c23] = "\x98\xd3", [0x1c24] = "\x98\xd4", [0x1c25] = "\x98\xd5",
  [0x1c26] = "\x98\xd6", [0x1c27] = "\x98\xd7", [0x1c28] = "\xe9\xd8",
  [0x1c29] = "\x98\xd8", [0x1c2a] = "\xba\xe1", [0x1c2b] = "\x98\xd9",
  [0x1c2c] = "\x98\xda", [0x1c2d] = "\x98\xdb", [0x1c2e] = "\x98\xdc",
  [0x1c2f] = "\xe9\xc9", [0x1c30] = "\x98\xdd", [0x1c31] = "\xd3\xa3",
  [0x1c32] = "\x98\xde", [0x1c33] = "\x98\xdf", [0x1c34] = "\x98\xe0",
  [0x1c35] = "\xe9\xd4", [0x1c36] = "\x98\xe1", [0x1c37] = "\x98\xe2",
  [0x1c38] = "\x98\xe3", [0x1c39] = "\x98\xe4", [0x1c3a] = "\x98\xe5",
  [0x1c3b] = "\x98\xe6", [0x1c3c] = "\x98\xe7", [0x1c3d] = "\xe9\xd7",
  [0x1c3e] = "\xe9\xd0", [0x1c3f] = "\x98\xe8", [0x1c40] = "\x98\xe9",
  [0x1c41] = "\x98\xea", [0x1c42] = "\x98\xeb", [0x1c43] = "\x98\xec",
  [0x1c44] = "\xe9\xcf", [0x1c45] = "\x98\xed", [0x1c46] = "\x98\xee",
  [0x1c47] = "\xc7\xc1", [0x1c48] = "\x98\xef", [0x1c49] = "\x98\xf0",
  [0x1c4a] = "\x98\xf1", [0x1c4b] = "\x98\xf2", [0x1c4c] = "\x98\xf3",
  [0x1c4d] = "\x98\xf4", [0x1c4e] = "\x98\xf5", [0x1c4f] = "\x98\xf6",
  [0x1c50] = "\xe9\xd2", [0x1c51] = "\x98\xf7", [0x1c52] = "\x98\xf8",
  [0x1c53] = "\x98\xf9", [0x1c54] = "\x98\xfa", [0x1c55] = "\x98\xfb",
  [0x1c56] = "\x98\xfc", [0x1c57] = "\x98\xfd", [0x1c58] = "\xe9\xd9",
  [0x1c59] = "\xb3\xc8", [0x1c5a] = "\x98\xfe", [0x1c5b] = "\xe9\xd3",
  [0x1c5c] = "\x99\x40", [0x1c5d] = "\x99\x41", [0x1c5e] = "\x99\x42",
  [0x1c5f] = "\x99\x43", [0x1c60] = "\x99\x44", [0x1c61] = "\xcf\xf0",
  [0x1c62] = "\x99\x45", [0x1c63] = "\x99\x46", [0x1c64] = "\x99\x47",
  [0x1c65] = "\xe9\xcd", [0x1c66] = "\x99\x48", [0x1c67] = "\x99\x49",
  [0x1c68] = "\x99\x4a", [0x1c69] = "\x99\x4b", [0x1c6a] = "\x99\x4c",
  [0x1c6b] = "\x99\x4d", [0x1c6c] = "\x99\x4e", [0x1c6d] = "\x99\x4f",
  [0x1c6e] = "\x99\x50", [0x1c6f] = "\x99\x51", [0x1c70] = "\x99\x52",
  [0x1c71] = "\xb3\xf7", [0x1c72] = "\x99\x53", [0x1c73] = "\x99\x54",
  [0x1c74] = "\x99\x55", [0x1c75] = "\x99\x56", [0x1c76] = "\x99\x57",
  [0x1c77] = "\x99\x58", [0x1c78] = "\x99\x59", [0x1c79] = "\xe9\xd6",
  [0x1c7a] = "\x99\x5a", [0x1c7b] = "\x99\x5b", [0x1c7c] = "\xe9\xda",
  [0x1c7d] = "\x99\x5c", [0x1c7e] = "\x99\x5d", [0x1c7f] = "\x99\x5e",
  [0x1c80] = "\xcc\xb4", [0x1c81] = "\x99\x5f", [0x1c82] = "\x99\x60",
  [0x1c83] = "\x99\x61", [0x1c84] = "\xcf\xad", [0x1c85] = "\x99\x62",
  [0x1c86] = "\x99\x63", [0x1c87] = "\x99\x64", [0x1c88] = "\x99\x65",
  [0x1c89] = "\x99\x66", [0x1c8a] = "\x99\x67", [0x1c8b] = "\x99\x68",
  [0x1c8c] = "\x99\x69", [0x1c8d] = "\x99\x6a", [0x1c8e] = "\xe9\xd5",
  [0x1c8f] = "\x99\x6b", [0x1c90] = "\xe9\xdc", [0x1c91] = "\xe9\xdb",
  [0x1c92] = "\x99\x6c", [0x1c93] = "\x99\x6d", [0x1c94] = "\x99\x6e",
  [0x1c95] = "\x99\x6f", [0x1c96] = "\x99\x70", [0x1c97] = "\xe9\xde",
  [0x1c98] = "\x99\x71", [0x1c99] = "\x99\x72", [0x1c9a] = "\x99\x73",
  [0x1c9b] = "\x99\x74", [0x1c9c] = "\x99\x75", [0x1c9d] = "\x99\x76",
  [0x1c9e] = "\x99\x77", [0x1c9f] = "\x99\x78", [0x1ca0] = "\xe9\xd1",
  [0x1ca1] = "\x99\x79", [0x1ca2] = "\x99\x7a", [0x1ca3] = "\x99\x7b",
  [0x1ca4] = "\x99\x7c", [0x1ca5] = "\x99\x7d", [0x1ca6] = "\x99\x7e",
  [0x1ca7] = "\x99\x80", [0x1ca8] = "\x99\x81", [0x1ca9] = "\xe9\xdd",
  [0x1caa] = "\x99\x82", [0x1cab] = "\xe9\xdf", [0x1cac] = "\xc3\xca",
  [0x1cad] = "\x99\x83", [0x1cae] = "\x99\x84", [0x1caf] = "\x99\x85",
  [0x1cb0] = "\x99\x86", [0x1cb1] = "\x99\x87", [0x1cb2] = "\x99\x88",
  [0x1cb3] = "\x99\x89", [0x1cb4] = "\x99\x8a", [0x1cb5] = "\x99\x8b",
  [0x1cb6] = "\x99\x8c", [0x1cb7] = "\x99\x8d", [0x1cb8] = "\x99\x8e",
  [0x1cb9] = "\x99\x8f", [0x1cba] = "\x99\x90", [0x1cbb] = "\x99\x91",
  [0x1cbc] = "\x99\x92", [0x1cbd] = "\x99\x93", [0x1cbe] = "\x99\x94",
  [0x1cbf] = "\x99\x95", [0x1cc0] = "\x99\x96", [0x1cc1] = "\x99\x97",
  [0x1cc2] = "\x99\x98", [0x1cc3] = "\x99\x99", [0x1cc4] = "\x99\x9a",
  [0x1cc5] = "\x99\x9b", [0x1cc6] = "\x99\x9c", [0x1cc7] = "\x99\x9d",
  [0x1cc8] = "\x99\x9e", [0x1cc9] = "\x99\x9f", [0x1cca] = "\x99\xa0",
  [0x1ccb] = "\x99\xa1", [0x1ccc] = "\x99\xa2", [0x1ccd] = "\x99\xa3",
  [0x1cce] = "\x99\xa4", [0x1ccf] = "\x99\xa5", [0x1cd0] = "\x99\xa6",
  [0x1cd1] = "\x99\xa7", [0x1cd2] = "\x99\xa8", [0x1cd3] = "\x99\xa9",
  [0x1cd4] = "\x99\xaa", [0x1cd5] = "\x99\xab", [0x1cd6] = "\x99\xac",
  [0x1cd7] = "\x99\xad", [0x1cd8] = "\x99\xae", [0x1cd9] = "\x99\xaf",
  [0x1cda] = "\x99\xb0", [0x1cdb] = "\x99\xb1", [0x1cdc] = "\x99\xb2",
  [0x1cdd] = "\x99\xb3", [0x1cde] = "\x99\xb4", [0x1cdf] = "\x99\xb5",
  [0x1ce0] = "\x99\xb6", [0x1ce1] = "\x99\xb7", [0x1ce2] = "\x99\xb8",
  [0x1ce3] = "\x99\xb9", [0x1ce4] = "\x99\xba", [0x1ce5] = "\x99\xbb",
  [0x1ce6] = "\x99\xbc", [0x1ce7] = "\x99\xbd", [0x1ce8] = "\x99\xbe",
  [0x1ce9] = "\x99\xbf", [0x1cea] = "\x99\xc0", [0x1ceb] = "\x99\xc1",
  [0x1cec] = "\x99\xc2", [0x1ced] = "\x99\xc3", [0x1cee] = "\x99\xc4",
  [0x1cef] = "\x99\xc5", [0x1cf0] = "\x99\xc6", [0x1cf1] = "\x99\xc7",
  [0x1cf2] = "\x99\xc8", [0x1cf3] = "\x99\xc9", [0x1cf4] = "\x99\xca",
  [0x1cf5] = "\x99\xcb", [0x1cf6] = "\x99\xcc", [0x1cf7] = "\x99\xcd",
  [0x1cf8] = "\x99\xce", [0x1cf9] = "\x99\xcf", [0x1cfa] = "\x99\xd0",
  [0x1cfb] = "\x99\xd1", [0x1cfc] = "\x99\xd2", [0x1cfd] = "\x99\xd3",
  [0x1cfe] = "\x99\xd4", [0x1cff] = "\x99\xd5", [0x1d00] = "\x99\xd6",
  [0x1d01] = "\x99\xd7", [0x1d02] = "\x99\xd8", [0x1d03] = "\x99\xd9",
  [0x1d04] = "\x99\xda", [0x1d05] = "\x99\xdb", [0x1d06] = "\x99\xdc",
  [0x1d07] = "\x99\xdd", [0x1d08] = "\x99\xde", [0x1d09] = "\x99\xdf",
  [0x1d0a] = "\x99\xe0", [0x1d0b] = "\x99\xe1", [0x1d0c] = "\x99\xe2",
  [0x1d0d] = "\x99\xe3", [0x1d0e] = "\x99\xe4", [0x1d0f] = "\x99\xe5",
  [0x1d10] = "\x99\xe6", [0x1d11] = "\x99\xe7", [0x1d12] = "\x99\xe8",
  [0x1d13] = "\x99\xe9", [0x1d14] = "\x99\xea", [0x1d15] = "\x99\xeb",
  [0x1d16] = "\x99\xec", [0x1d17] = "\x99\xed", [0x1d18] = "\x99\xee",
  [0x1d19] = "\x99\xef", [0x1d1a] = "\x99\xf0", [0x1d1b] = "\x99\xf1",
  [0x1d1c] = "\x99\xf2", [0x1d1d] = "\x99\xf3", [0x1d1e] = "\x99\xf4",
  [0x1d1f] = "\x99\xf5", [0x1d20] = "\xc7\xb7", [0x1d21] = "\xb4\xce",
  [0x1d22] = "\xbb\xb6", [0x1d23] = "\xd0\xc0", [0x1d24] = "\xec\xa3",
  [0x1d25] = "\x99\xf6", [0x1d26] = "\x99\xf7", [0x1d27] = "\xc5\xb7",
  [0x1d28] = "\x99\xf8", [0x1d29] = "\x99\xf9", [0x1d2a] = "\x99\xfa",
  [0x1d2b] = "\x99\xfb", [0x1d2c] = "\x99\xfc", [0x1d2d] = "\x99\xfd",
  [0x1d2e] = "\x99\xfe", [0x1d2f] = "\x9a\x40", [0x1d30] = "\x9a\x41",
  [0x1d31] = "\x9a\x42", [0x1d32] = "\xd3\xfb", [0x1d33] = "\x9a\x43",
  [0x1d34] = "\x9a\x44", [0x1d35] = "\x9a\x45", [0x1d36] = "\x9a\x46",
  [0x1d37] = "\xec\xa4", [0x1d38] = "\x9a\x47", [0x1d39] = "\xec\xa5",
  [0x1d3a] = "\xc6\xdb", [0x1d3b] = "\x9a\x48", [0x1d3c] = "\x9a\x49",
  [0x1d3d] = "\x9a\x4a", [0x1d3e] = "\xbf\xee", [0x1d3f] = "\x9a\x4b",
  [0x1d40] = "\x9a\x4c", [0x1d41] = "\x9a\x4d", [0x1d42] = "\x9a\x4e",
  [0x1d43] = "\xec\xa6", [0x1d44] = "\x9a\x4f", [0x1d45] = "\x9a\x50",
  [0x1d46] = "\xec\xa7", [0x1d47] = "\xd0\xaa", [0x1d48] = "\x9a\x51",
  [0x1d49] = "\xc7\xb8", [0x1d4a] = "\x9a\x52", [0x1d4b] = "\x9a\x53",
  [0x1d4c] = "\xb8\xe8", [0x1d4d] = "\x9a\x54", [0x1d4e] = "\x9a\x55",
  [0x1d4f] = "\x9a\x56", [0x1d50] = "\x9a\x57", [0x1d51] = "\x9a\x58",
  [0x1d52] = "\x9a\x59", [0x1d53] = "\x9a\x5a", [0x1d54] = "\x9a\x5b",
  [0x1d55] = "\x9a\x5c", [0x1d56] = "\x9a\x5d", [0x1d57] = "\x9a\x5e",
  [0x1d58] = "\x9a\x5f", [0x1d59] = "\xec\xa8", [0x1d5a] = "\x9a\x60",
  [0x1d5b] = "\x9a\x61", [0x1d5c] = "\x9a\x62", [0x1d5d] = "\x9a\x63",
  [0x1d5e] = "\x9a\x64", [0x1d5f] = "\x9a\x65", [0x1d60] = "\x9a\x66",
  [0x1d61] = "\x9a\x67", [0x1d62] = "\xd6\xb9", [0x1d63] = "\xd5\xfd",
  [0x1d64] = "\xb4\xcb", [0x1d65] = "\xb2\xbd", [0x1d66] = "\xce\xe4",
  [0x1d67] = "\xc6\xe7", [0x1d68] = "\x9a\x68", [0x1d69] = "\x9a\x69",
  [0x1d6a] = "\xcd\xe1", [0x1d6b] = "\x9a\x6a", [0x1d6c] = "\x9a\x6b",
  [0x1d6d] = "\x9a\x6c", [0x1d6e] = "\x9a\x6d", [0x1d6f] = "\x9a\x6e",
  [0x1d70] = "\x9a\x6f", [0x1d71] = "\x9a\x70", [0x1d72] = "\x9a\x71",
  [0x1d73] = "\x9a\x72", [0x1d74] = "\x9a\x73", [0x1d75] = "\x9a\x74",
  [0x1d76] = "\x9a\x75", [0x1d77] = "\x9a\x76", [0x1d78] = "\x9a\x77",
  [0x1d79] = "\xb4\xf5", [0x1d7a] = "\x9a\x78", [0x1d7b] = "\xcb\xc0",
  [0x1d7c] = "\xbc\xdf", [0x1d7d] = "\x9a\x79", [0x1d7e] = "\x9a\x7a",
  [0x1d7f] = "\x9a\x7b", [0x1d80] = "\x9a\x7c", [0x1d81] = "\xe9\xe2",
  [0x1d82] = "\xe9\xe3", [0x1d83] = "\xd1\xea", [0x1d84] = "\xe9\xe5",
  [0x1d85] = "\x9a\x7d", [0x1d86] = "\xb4\xf9", [0x1d87] = "\xe9\xe4",
  [0x1d88] = "\x9a\x7e", [0x1d89] = "\xd1\xb3", [0x1d8a] = "\xca\xe2",
  [0x1d8b] = "\xb2\xd0", [0x1d8c] = "\x9a\x80", [0x1d8d] = "\xe9\xe8",
  [0x1d8e] = "\x9a\x81", [0x1d8f] = "\x9a\x82", [0x1d90] = "\x9a\x83",
  [0x1d91] = "\x9a\x84", [0x1d92] = "\xe9\xe6", [0x1d93] = "\xe9\xe7",
  [0x1d94] = "\x9a\x85", [0x1d95] = "\x9a\x86", [0x1d96] = "\xd6\xb3",
  [0x1d97] = "\x9a\x87", [0x1d98] = "\x9a\x88", [0x1d99] = "\x9a\x89",
  [0x1d9a] = "\xe9\xe9", [0x1d9b] = "\xe9\xea", [0x1d9c] = "\x9a\x8a",
  [0x1d9d] = "\x9a\x8b", [0x1d9e] = "\x9a\x8c", [0x1d9f] = "\x9a\x8d",
  [0x1da0] = "\x9a\x8e", [0x1da1] = "\xe9\xeb", [0x1da2] = "\x9a\x8f",
  [0x1da3] = "\x9a\x90", [0x1da4] = "\x9a\x91", [0x1da5] = "\x9a\x92",
  [0x1da6] = "\x9a\x93", [0x1da7] = "\x9a\x94", [0x1da8] = "\x9a\x95",
  [0x1da9] = "\x9a\x96", [0x1daa] = "\xe9\xec", [0x1dab] = "\x9a\x97",
  [0x1dac] = "\x9a\x98", [0x1dad] = "\x9a\x99", [0x1dae] = "\x9a\x9a",
  [0x1daf] = "\x9a\x9b", [0x1db0] = "\x9a\x9c", [0x1db1] = "\x9a\x9d",
  [0x1db2] = "\x9a\x9e", [0x1db3] = "\xec\xaf", [0x1db4] = "\xc5\xb9",
  [0x1db5] = "\xb6\xce", [0x1db6] = "\x9a\x9f", [0x1db7] = "\xd2\xf3",
  [0x1db8] = "\x9a\xa0", [0x1db9] = "\x9a\xa1", [0x1dba] = "\x9a\xa2",
  [0x1dbb] = "\x9a\xa3", [0x1dbc] = "\x9a\xa4", [0x1dbd] = "\x9a\xa5",
  [0x1dbe] = "\x9a\xa6", [0x1dbf] = "\xb5\xee", [0x1dc0] = "\x9a\xa7",
  [0x1dc1] = "\xbb\xd9", [0x1dc2] = "\xec\xb1", [0x1dc3] = "\x9a\xa8",
  [0x1dc4] = "\x9a\xa9", [0x1dc5] = "\xd2\xe3", [0x1dc6] = "\x9a\xaa",
  [0x1dc7] = "\x9a\xab", [0x1dc8] = "\x9a\xac", [0x1dc9] = "\x9a\xad",
  [0x1dca] = "\x9a\xae", [0x1dcb] = "\xce\xe3", [0x1dcc] = "\x9a\xaf",
  [0x1dcd] = "\xc4\xb8", [0x1dce] = "\x9a\xb0", [0x1dcf] = "\xc3\xbf",
  [0x1dd0] = "\x9a\xb1", [0x1dd1] = "\x9a\xb2", [0x1dd2] = "\xb6\xbe",
  [0x1dd3] = "\xd8\xb9", [0x1dd4] = "\xb1\xc8", [0x1dd5] = "\xb1\xcf",
  [0x1dd6] = "\xb1\xd1", [0x1dd7] = "\xc5\xfe", [0x1dd8] = "\x9a\xb3",
  [0x1dd9] = "\xb1\xd0", [0x1dda] = "\x9a\xb4", [0x1ddb] = "\xc3\xab",
  [0x1ddc] = "\x9a\xb5", [0x1ddd] = "\x9a\xb6", [0x1dde] = "\x9a\xb7",
  [0x1ddf] = "\x9a\xb8", [0x1de0] = "\x9a\xb9", [0x1de1] = "\xd5\xb1",
  [0x1de2] = "\x9a\xba", [0x1de3] = "\x9a\xbb", [0x1de4] = "\x9a\xbc",
  [0x1de5] = "\x9a\xbd", [0x1de6] = "\x9a\xbe", [0x1de7] = "\x9a\xbf",
  [0x1de8] = "\x9a\xc0", [0x1de9] = "\x9a\xc1", [0x1dea] = "\xeb\xa4",
  [0x1deb] = "\xba\xc1", [0x1dec] = "\x9a\xc2", [0x1ded] = "\x9a\xc3",
  [0x1dee] = "\x9a\xc4", [0x1def] = "\xcc\xba", [0x1df0] = "\x9a\xc5",
  [0x1df1] = "\x9a\xc6", [0x1df2] = "\x9a\xc7", [0x1df3] = "\xeb\xa5",
  [0x1df4] = "\x9a\xc8", [0x1df5] = "\xeb\xa7", [0x1df6] = "\x9a\xc9",
  [0x1df7] = "\x9a\xca", [0x1df8] = "\x9a\xcb", [0x1df9] = "\xeb\xa8",
  [0x1dfa] = "\x9a\xcc", [0x1dfb] = "\x9a\xcd", [0x1dfc] = "\x9a\xce",
  [0x1dfd] = "\xeb\xa6", [0x1dfe] = "\x9a\xcf", [0x1dff] = "\x9a\xd0",
  [0x1e00] = "\x9a\xd1", [0x1e01] = "\x9a\xd2", [0x1e02] = "\x9a\xd3",
  [0x1e03] = "\x9a\xd4", [0x1e04] = "\x9a\xd5", [0x1e05] = "\xeb\xa9",
  [0x1e06] = "\xeb\xab", [0x1e07] = "\xeb\xaa", [0x1e08] = "\x9a\xd6",
  [0x1e09] = "\x9a\xd7", [0x1e0a] = "\x9a\xd8", [0x1e0b] = "\x9a\xd9",
  [0x1e0c] = "\x9a\xda", [0x1e0d] = "\xeb\xac", [0x1e0e] = "\x9a\xdb",
  [0x1e0f] = "\xca\xcf", [0x1e10] = "\xd8\xb5", [0x1e11] = "\xc3\xf1",
  [0x1e12] = "\x9a\xdc", [0x1e13] = "\xc3\xa5", [0x1e14] = "\xc6\xf8",
  [0x1e15] = "\xeb\xad", [0x1e16] = "\xc4\xca", [0x1e17] = "\x9a\xdd",
  [0x1e18] = "\xeb\xae", [0x1e19] = "\xeb\xaf", [0x1e1a] = "\xeb\xb0",
  [0x1e1b] = "\xb7\xd5", [0x1e1c] = "\x9a\xde", [0x1e1d] = "\x9a\xdf",
  [0x1e1e] = "\x9a\xe0", [0x1e1f] = "\xb7\xfa", [0x1e20] = "\x9a\xe1",
  [0x1e21] = "\xeb\xb1", [0x1e22] = "\xc7\xe2", [0x1e23] = "\x9a\xe2",
  [0x1e24] = "\xeb\xb3", [0x1e25] = "\x9a\xe3", [0x1e26] = "\xba\xa4",
  [0x1e27] = "\xd1\xf5", [0x1e28] = "\xb0\xb1", [0x1e29] = "\xeb\xb2",
  [0x1e2a] = "\xeb\xb4", [0x1e2b] = "\x9a\xe4", [0x1e2c] = "\x9a\xe5",
  [0x1e2d] = "\x9a\xe6", [0x1e2e] = "\xb5\xaa", [0x1e2f] = "\xc2\xc8",
  [0x1e30] = "\xc7\xe8", [0x1e31] = "\x9a\xe7", [0x1e32] = "\xeb\xb5",
  [0x1e33] = "\x9a\xe8", [0x1e34] = "\xcb\xae", [0x1e35] = "\xe3\xdf",
  [0x1e36] = "\x9a\xe9", [0x1e37] = "\x9a\xea", [0x1e38] = "\xd3\xc0",
  [0x1e39] = "\x9a\xeb", [0x1e3a] = "\x9a\xec", [0x1e3b] = "\x9a\xed",
  [0x1e3c] = "\x9a\xee", [0x1e3d] = "\xd9\xdb", [0x1e3e] = "\x9a\xef",
  [0x1e3f] = "\x9a\xf0", [0x1e40] = "\xcd\xa1", [0x1e41] = "\xd6\xad",
  [0x1e42] = "\xc7\xf3", [0x1e43] = "\x9a\xf1", [0x1e44] = "\x9a\xf2",
  [0x1e45] = "\x9a\xf3", [0x1e46] = "\xd9\xe0", [0x1e47] = "\xbb\xe3",
  [0x1e48] = "\x9a\xf4", [0x1e49] = "\xba\xba", [0x1e4a] = "\xe3\xe2",
  [0x1e4b] = "\x9a\xf5", [0x1e4c] = "\x9a\xf6", [0x1e4d] = "\x9a\xf7",
  [0x1e4e] = "\x9a\xf8", [0x1e4f] = "\x9a\xf9", [0x1e50] = "\xcf\xab",
  [0x1e51] = "\x9a\xfa", [0x1e52] = "\x9a\xfb", [0x1e53] = "\x9a\xfc",
  [0x1e54] = "\xe3\xe0", [0x1e55] = "\xc9\xc7", [0x1e56] = "\x9a\xfd",
  [0x1e57] = "\xba\xb9", [0x1e58] = "\x9a\xfe", [0x1e59] = "\x9b\x40",
  [0x1e5a] = "\x9b\x41", [0x1e5b] = "\xd1\xb4", [0x1e5c] = "\xe3\xe1",
  [0x1e5d] = "\xc8\xea", [0x1e5e] = "\xb9\xaf", [0x1e5f] = "\xbd\xad",
  [0x1e60] = "\xb3\xd8", [0x1e61] = "\xce\xdb", [0x1e62] = "\x9b\x42",
  [0x1e63] = "\x9b\x43", [0x1e64] = "\xcc\xc0", [0x1e65] = "\x9b\x44",
  [0x1e66] = "\x9b\x45", [0x1e67] = "\x9b\x46", [0x1e68] = "\xe3\xe8",
  [0x1e69] = "\xe3\xe9", [0x1e6a] = "\xcd\xf4", [0x1e6b] = "\x9b\x47",
  [0x1e6c] = "\x9b\x48", [0x1e6d] = "\x9b\x49", [0x1e6e] = "\x9b\x4a",
  [0x1e6f] = "\x9b\x4b", [0x1e70] = "\xcc\xad", [0x1e71] = "\x9b\x4c",
  [0x1e72] = "\xbc\xb3", [0x1e73] = "\x9b\x4d", [0x1e74] = "\xe3\xea",
  [0x1e75] = "\x9b\x4e", [0x1e76] = "\xe3\xeb", [0x1e77] = "\x9b\x4f",
  [0x1e78] = "\x9b\x50", [0x1e79] = "\xd0\xda", [0x1e7a] = "\x9b\x51",
  [0x1e7b] = "\x9b\x52", [0x1e7c] = "\x9b\x53", [0x1e7d] = "\xc6\xfb",
  [0x1e7e] = "\xb7\xda", [0x1e7f] = "\x9b\x54", [0x1e80] = "\x9b\x55",
  [0x1e81] = "\xc7\xdf", [0x1e82] = "\xd2\xca", [0x1e83] = "\xce\xd6",
  [0x1e84] = "\x9b\x56", [0x1e85] = "\xe3\xe4", [0x1e86] = "\xe3\xec",
  [0x1e87] = "\x9b\x57", [0x1e88] = "\xc9\xf2", [0x1e89] = "\xb3\xc1",
  [0x1e8a] = "\x9b\x58", [0x1e8b] = "\x9b\x59", [0x1e8c] = "\xe3\xe7",
  [0x1e8d] = "\x9b\x5a", [0x1e8e] = "\x9b\x5b", [0x1e8f] = "\xc6\xe3",
  [0x1e90] = "\xe3\xe5", [0x1e91] = "\x9b\x5c", [0x1e92] = "\x9b\x5d",
  [0x1e93] = "\xed\xb3", [0x1e94] = "\xe3\xe6", [0x1e95] = "\x9b\x5e",
  [0x1e96] = "\x9b\x5f", [0x1e97] = "\x9b\x60", [0x1e98] = "\x9b\x61",
  [0x1e99] = "\xc9\xb3", [0x1e9a] = "\x9b\x62", [0x1e9b] = "\xc5\xe6",
  [0x1e9c] = "\x9b\x63", [0x1e9d] = "\x9b\x64", [0x1e9e] = "\x9b\x65",
  [0x1e9f] = "\xb9\xb5", [0x1ea0] = "\x9b\x66", [0x1ea1] = "\xc3\xbb",
  [0x1ea2] = "\x9b\x67", [0x1ea3] = "\xe3\xe3", [0x1ea4] = "\xc5\xbd",
  [0x1ea5] = "\xc1\xa4", [0x1ea6] = "\xc2\xd9", [0x1ea7] = "\xb2\xd7",
  [0x1ea8] = "\x9b\x68", [0x1ea9] = "\xe3\xed", [0x1eaa] = "\xbb\xa6",
  [0x1eab] = "\xc4\xad", [0x1eac] = "\x9b\x69", [0x1ead] = "\xe3\xf0",
  [0x1eae] = "\xbe\xda", [0x1eaf] = "\x9b\x6a", [0x1eb0] = "\x9b\x6b",
  [0x1eb1] = "\xe3\xfb", [0x1eb2] = "\xe3\xf5", [0x1eb3] = "\xba\xd3",
  [0x1eb4] = "\x9b\x6c", [0x1eb5] = "\x9b\x6d", [0x1eb6] = "\x9b\x6e",
  [0x1eb7] = "\x9b\x6f", [0x1eb8] = "\xb7\xd0", [0x1eb9] = "\xd3\xcd",
  [0x1eba] = "\x9b\x70", [0x1ebb] = "\xd6\xce", [0x1ebc] = "\xd5\xd3",
  [0x1ebd] = "\xb9\xc1", [0x1ebe] = "\xd5\xb4", [0x1ebf] = "\xd1\xd8",
  [0x1ec0] = "\x9b\x71", [0x1ec1] = "\x9b\x72", [0x1ec2] = "\x9b\x73",
  [0x1ec3] = "\x9b\x74", [0x1ec4] = "\xd0\xb9", [0x1ec5] = "\xc7\xf6",
  [0x1ec6] = "\x9b\x75", [0x1ec7] = "\x9b\x76", [0x1ec8] = "\x9b\x77",
  [0x1ec9] = "\xc8\xaa", [0x1eca] = "\xb2\xb4", [0x1ecb] = "\x9b\x78",
  [0x1ecc] = "\xc3\xda", [0x1ecd] = "\x9b\x79", [0x1ece] = "\x9b\x7a",
  [0x1ecf] = "\x9b\x7b", [0x1ed0] = "\xe3\xee", [0x1ed1] = "\x9b\x7c",
  [0x1ed2] = "\x9b\x7d", [0x1ed3] = "\xe3\xfc", [0x1ed4] = "\xe3\xef",
  [0x1ed5] = "\xb7\xa8", [0x1ed6] = "\xe3\xf7", [0x1ed7] = "\xe3\xf4",
  [0x1ed8] = "\x9b\x7e", [0x1ed9] = "\x9b\x80", [0x1eda] = "\x9b\x81",
  [0x1edb] = "\xb7\xba", [0x1edc] = "\x9b\x82", [0x1edd] = "\x9b\x83",
  [0x1ede] = "\xc5\xa2", [0x1edf] = "\x9b\x84", [0x1ee0] = "\xe3\xf6",
  [0x1ee1] = "\xc5\xdd", [0x1ee2] = "\xb2\xa8", [0x1ee3] = "\xc6\xfc",
  [0x1ee4] = "\x9b\x85", [0x1ee5] = "\xc4\xe0", [0x1ee6] = "\x9b\x86",
  [0x1ee7] = "\x9b\x87", [0x1ee8] = "\xd7\xa2", [0x1ee9] = "\x9b\x88",
  [0x1eea] = "\xc0\xe1", [0x1eeb] = "\xe3\xf9", [0x1eec] = "\x9b\x89",
  [0x1eed] = "\x9b\x8a", [0x1eee] = "\xe3\xfa", [0x1eef] = "\xe3\xfd",
  [0x1ef0] = "\xcc\xa9", [0x1ef1] = "\xe3\xf3", [0x1ef2] = "\x9b\x8b",
  [0x1ef3] = "\xd3\xbe", [0x1ef4] = "\x9b\x8c", [0x1ef5] = "\xb1\xc3",
  [0x1ef6] = "\xed\xb4", [0x1ef7] = "\xe3\xf1", [0x1ef8] = "\xe3\xf2",
  [0x1ef9] = "\x9b\x8d", [0x1efa] = "\xe3\xf8", [0x1efb] = "\xd0\xba",
  [0x1efc] = "\xc6\xc3", [0x1efd] = "\xd4\xf3", [0x1efe] = "\xe3\xfe",
  [0x1eff] = "\x9b\x8e", [0x1f00] = "\x9b\x8f", [0x1f01] = "\xbd\xe0",
  [0x1f02] = "\x9b\x90", [0x1f03] = "\x9b\x91", [0x1f04] = "\xe4\xa7",
  [0x1f05] = "\x9b\x92", [0x1f06] = "\x9b\x93", [0x1f07] = "\xe4\xa6",
  [0x1f08] = "\x9b\x94", [0x1f09] = "\x9b\x95", [0x1f0a] = "\x9b\x96",
  [0x1f0b] = "\xd1\xf3", [0x1f0c] = "\xe4\xa3", [0x1f0d] = "\x9b\x97",
  [0x1f0e] = "\xe4\xa9", [0x1f0f] = "\x9b\x98", [0x1f10] = "\x9b\x99",
  [0x1f11] = "\x9b\x9a", [0x1f12] = "\xc8\xf7", [0x1f13] = "\x9b\x9b",
  [0x1f14] = "\x9b\x9c", [0x1f15] = "\x9b\x9d", [0x1f16] = "\x9b\x9e",
  [0x1f17] = "\xcf\xb4", [0x1f18] = "\x9b\x9f", [0x1f19] = "\xe4\xa8",
  [0x1f1a] = "\xe4\xae", [0x1f1b] = "\xc2\xe5", [0x1f1c] = "\x9b\xa0",
  [0x1f1d] = "\x9b\xa1", [0x1f1e] = "\xb6\xb4", [0x1f1f] = "\x9b\xa2",
  [0x1f20] = "\x9b\xa3", [0x1f21] = "\x9b\xa4", [0x1f22] = "\x9b\xa5",
  [0x1f23] = "\x9b\xa6", [0x1f24] = "\x9b\xa7", [0x1f25] = "\xbd\xf2",
  [0x1f26] = "\x9b\xa8", [0x1f27] = "\xe4\xa2", [0x1f28] = "\x9b\xa9",
  [0x1f29] = "\x9b\xaa", [0x1f2a] = "\xba\xe9", [0x1f2b] = "\xe4\xaa",
  [0x1f2c] = "\x9b\xab", [0x1f2d] = "\x9b\xac", [0x1f2e] = "\xe4\xac",
  [0x1f2f] = "\x9b\xad", [0x1f30] = "\x9b\xae", [0x1f31] = "\xb6\xfd",
  [0x1f32] = "\xd6\xde", [0x1f33] = "\xe4\xb2", [0x1f34] = "\x9b\xaf",
  [0x1f35] = "\xe4\xad", [0x1f36] = "\x9b\xb0", [0x1f37] = "\x9b\xb1",
  [0x1f38] = "\x9b\xb2", [0x1f39] = "\xe4\xa1", [0x1f3a] = "\x9b\xb3",
  [0x1f3b] = "\xbb\xee", [0x1f3c] = "\xcd\xdd", [0x1f3d] = "\xc7\xa2",
  [0x1f3e] = "\xc5\xc9", [0x1f3f] = "\x9b\xb4", [0x1f40] = "\x9b\xb5",
  [0x1f41] = "\xc1\xf7", [0x1f42] = "\x9b\xb6", [0x1f43] = "\xe4\xa4",
  [0x1f44] = "\x9b\xb7", [0x1f45] = "\xc7\xb3", [0x1f46] = "\xbd\xac",
  [0x1f47] = "\xbd\xbd", [0x1f48] = "\xe4\xa5", [0x1f49] = "\x9b\xb8",
  [0x1f4a] = "\xd7\xc7", [0x1f4b] = "\xb2\xe2", [0x1f4c] = "\x9b\xb9",
  [0x1f4d] = "\xe4\xab", [0x1f4e] = "\xbc\xc3", [0x1f4f] = "\xe4\xaf",
  [0x1f50] = "\x9b\xba", [0x1f51] = "\xbb\xeb", [0x1f52] = "\xe4\xb0",
  [0x1f53] = "\xc5\xa8", [0x1f54] = "\xe4\xb1", [0x1f55] = "\x9b\xbb",
  [0x1f56] = "\x9b\xbc", [0x1f57] = "\x9b\xbd", [0x1f58] = "\x9b\xbe",
  [0x1f59] = "\xd5\xe3", [0x1f5a] = "\xbf\xa3", [0x1f5b] = "\x9b\xbf",
  [0x1f5c] = "\xe4\xba", [0x1f5d] = "\x9b\xc0", [0x1f5e] = "\xe4\xb7",
  [0x1f5f] = "\x9b\xc1", [0x1f60] = "\xe4\xbb", [0x1f61] = "\x9b\xc2",
  [0x1f62] = "\x9b\xc3", [0x1f63] = "\xe4\xbd", [0x1f64] = "\x9b\xc4",
  [0x1f65] = "\x9b\xc5", [0x1f66] = "\xc6\xd6", [0x1f67] = "\x9b\xc6",
  [0x1f68] = "\x9b\xc7", [0x1f69] = "\xba\xc6", [0x1f6a] = "\xc0\xcb",
  [0x1f6b] = "\x9b\xc8", [0x1f6c] = "\x9b\xc9", [0x1f6d] = "\x9b\xca",
  [0x1f6e] = "\xb8\xa1", [0x1f6f] = "\xe4\xb4", [0x1f70] = "\x9b\xcb",
  [0x1f71] = "\x9b\xcc", [0x1f72] = "\x9b\xcd", [0x1f73] = "\x9b\xce",
  [0x1f74] = "\xd4\xa1", [0x1f75] = "\x9b\xcf", [0x1f76] = "\x9b\xd0",
  [0x1f77] = "\xba\xa3", [0x1f78] = "\xbd\xfe", [0x1f79] = "\x9b\xd1",
  [0x1f7a] = "\x9b\xd2", [0x1f7b] = "\x9b\xd3", [0x1f7c] = "\xe4\xbc",
  [0x1f7d] = "\x9b\xd4", [0x1f7e] = "\x9b\xd5", [0x1f7f] = "\x9b\xd6",
  [0x1f80] = "\x9b\xd7", [0x1f81] = "\x9b\xd8", [0x1f82] = "\xcd\xbf",
  [0x1f83] = "\x9b\xd9", [0x1f84] = "\x9b\xda", [0x1f85] = "\xc4\xf9",
  [0x1f86] = "\x9b\xdb", [0x1f87] = "\x9b\xdc", [0x1f88] = "\xcf\xfb",
  [0x1f89] = "\xc9\xe6", [0x1f8a] = "\x9b\xdd", [0x1f8b] = "\x9b\xde",
  [0x1f8c] = "\xd3\xbf", [0x1f8d] = "\x9b\xdf", [0x1f8e] = "\xcf\xd1",
  [0x1f8f] = "\x9b\xe0", [0x1f90] = "\x9b\xe1", [0x1f91] = "\xe4\xb3",
  [0x1f92] = "\x9b\xe2", [0x1f93] = "\xe4\xb8", [0x1f94] = "\xe4\xb9",
  [0x1f95] = "\xcc\xe9", [0x1f96] = "\x9b\xe3", [0x1f97] = "\x9b\xe4",
  [0x1f98] = "\x9b\xe5", [0x1f99] = "\x9b\xe6", [0x1f9a] = "\x9b\xe7",
  [0x1f9b] = "\xcc\xce", [0x1f9c] = "\x9b\xe8", [0x1f9d] = "\xc0\xd4",
  [0x1f9e] = "\xe4\xb5", [0x1f9f] = "\xc1\xb0", [0x1fa0] = "\xe4\xb6",
  [0x1fa1] = "\xce\xd0", [0x1fa2] = "\x9b\xe9", [0x1fa3] = "\xbb\xc1",
  [0x1fa4] = "\xb5\xd3", [0x1fa5] = "\x9b\xea", [0x1fa6] = "\xc8\xf3",
  [0x1fa7] = "\xbd\xa7", [0x1fa8] = "\xd5\xc7", [0x1fa9] = "\xc9\xac",
  [0x1faa] = "\xb8\xa2", [0x1fab] = "\xe4\xca", [0x1fac] = "\x9b\xeb",
  [0x1fad] = "\x9b\xec", [0x1fae] = "\xe4\xcc", [0x1faf] = "\xd1\xc4",
  [0x1fb0] = "\x9b\xed", [0x1fb1] = "\x9b\xee", [0x1fb2] = "\xd2\xba",
  [0x1fb3] = "\x9b\xef", [0x1fb4] = "\x9b\xf0", [0x1fb5] = "\xba\xad",
  [0x1fb6] = "\x9b\xf1", [0x1fb7] = "\x9b\xf2", [0x1fb8] = "\xba\xd4",
  [0x1fb9] = "\x9b\xf3", [0x1fba] = "\x9b\xf4", [0x1fbb] = "\x9b\xf5",
  [0x1fbc] = "\x9b\xf6", [0x1fbd] = "\x9b\xf7", [0x1fbe] = "\x9b\xf8",
  [0x1fbf] = "\xe4\xc3", [0x1fc0] = "\xb5\xed", [0x1fc1] = "\x9b\xf9",
  [0x1fc2] = "\x9b\xfa", [0x1fc3] = "\x9b\xfb", [0x1fc4] = "\xd7\xcd",
  [0x1fc5] = "\xe4\xc0", [0x1fc6] = "\xcf\xfd", [0x1fc7] = "\xe4\xbf",
  [0x1fc8] = "\x9b\xfc", [0x1fc9] = "\x9b\xfd", [0x1fca] = "\x9b\xfe",
  [0x1fcb] = "\xc1\xdc", [0x1fcc] = "\xcc\xca", [0x1fcd] = "\x9c\x40",
  [0x1fce] = "\x9c\x41", [0x1fcf] = "\x9c\x42", [0x1fd0] = "\x9c\x43",
  [0x1fd1] = "\xca\xe7", [0x1fd2] = "\x9c\x44", [0x1fd3] = "\x9c\x45",
  [0x1fd4] = "\x9c\x46", [0x1fd5] = "\x9c\x47", [0x1fd6] = "\xc4\xd7",
  [0x1fd7] = "\x9c\x48", [0x1fd8] = "\xcc\xd4", [0x1fd9] = "\xe4\xc8",
  [0x1fda] = "\x9c\x49", [0x1fdb] = "\x9c\x4a", [0x1fdc] = "\x9c\x4b",
  [0x1fdd] = "\xe4\xc7", [0x1fde] = "\xe4\xc1", [0x1fdf] = "\x9c\x4c",
  [0x1fe0] = "\xe4\xc4", [0x1fe1] = "\xb5\xad", [0x1fe2] = "\x9c\x4d",
  [0x1fe3] = "\x9c\x4e", [0x1fe4] = "\xd3\xd9", [0x1fe5] = "\x9c\x4f",
  [0x1fe6] = "\xe4\xc6", [0x1fe7] = "\x9c\x50", [0x1fe8] = "\x9c\x51",
  [0x1fe9] = "\x9c\x52", [0x1fea] = "\x9c\x53", [0x1feb] = "\xd2\xf9",
  [0x1fec] = "\xb4\xe3", [0x1fed] = "\x9c\x54", [0x1fee] = "\xbb\xb4",
  [0x1fef] = "\x9c\x55", [0x1ff0] = "\x9c\x56", [0x1ff1] = "\xc9\xee",
  [0x1ff2] = "\x9c\x57", [0x1ff3] = "\xb4\xbe", [0x1ff4] = "\x9c\x58",
  [0x1ff5] = "\x9c\x59", [0x1ff6] = "\x9c\x5a", [0x1ff7] = "\xbb\xec",
  [0x1ff8] = "\x9c\x5b", [0x1ff9] = "\xd1\xcd", [0x1ffa] = "\x9c\x5c",
  [0x1ffb] = "\xcc\xed", [0x1ffc] = "\xed\xb5", [0x1ffd] = "\x9c\x5d",
  [0x1ffe] = "\x9c\x5e", [0x1fff] = "\x9c\x5f", [0x2000] = "\x9c\x60",
  [0x2001] = "\x9c\x61", [0x2002] = "\x9c\x62", [0x2003] = "\x9c\x63",
  [0x2004] = "\x9c\x64", [0x2005] = "\xc7\xe5", [0x2006] = "\x9c\x65",
  [0x2007] = "\x9c\x66", [0x2008] = "\x9c\x67", [0x2009] = "\x9c\x68",
  [0x200a] = "\xd4\xa8", [0x200b] = "\x9c\x69", [0x200c] = "\xe4\xcb",
  [0x200d] = "\xd7\xd5", [0x200e] = "\xe4\xc2", [0x200f] = "\x9c\x6a",
  [0x2010] = "\xbd\xa5", [0x2011] = "\xe4\xc5", [0x2012] = "\x9c\x6b",
  [0x2013] = "\x9c\x6c", [0x2014] = "\xd3\xe6", [0x2015] = "\x9c\x6d",
  [0x2016] = "\xe4\xc9", [0x2017] = "\xc9\xf8", [0x2018] = "\x9c\x6e",
  [0x2019] = "\x9c\x6f", [0x201a] = "\xe4\xbe", [0x201b] = "\x9c\x70",
  [0x201c] = "\x9c\x71", [0x201d] = "\xd3\xe5", [0x201e] = "\x9c\x72",
  [0x201f] = "\x9c\x73", [0x2020] = "\xc7\xfe", [0x2021] = "\xb6\xc9",
  [0x2022] = "\x9c\x74", [0x2023] = "\xd4\xfc", [0x2024] = "\xb2\xb3",
  [0x2025] = "\xe4\xd7", [0x2026] = "\x9c\x75", [0x2027] = "\x9c\x76",
  [0x2028] = "\x9c\x77", [0x2029] = "\xce\xc2", [0x202a] = "\x9c\x78",
  [0x202b] = "\xe4\xcd", [0x202c] = "\x9c\x79", [0x202d] = "\xce\xbc",
  [0x202e] = "\x9c\x7a", [0x202f] = "\xb8\xdb", [0x2030] = "\x9c\x7b",
  [0x2031] = "\x9c\x7c", [0x2032] = "\xe4\xd6", [0x2033] = "\x9c\x7d",
  [0x2034] = "\xbf\xca", [0x2035] = "\x9c\x7e", [0x2036] = "\x9c\x80",
  [0x2037] = "\x9c\x81", [0x2038] = "\xd3\xce", [0x2039] = "\x9c\x82",
  [0x203a] = "\xc3\xec", [0x203b] = "\x9c\x83", [0x203c] = "\x9c\x84",
  [0x203d] = "\x9c\x85", [0x203e] = "\x9c\x86", [0x203f] = "\x9c\x87",
  [0x2040] = "\x9c\x88", [0x2041] = "\x9c\x89", [0x2042] = "\x9c\x8a",
  [0x2043] = "\xc5\xc8", [0x2044] = "\xe4\xd8", [0x2045] = "\x9c\x8b",
  [0x2046] = "\x9c\x8c", [0x2047] = "\x9c\x8d", [0x2048] = "\x9c\x8e",
  [0x2049] = "\x9c\x8f", [0x204a] = "\x9c\x90", [0x204b] = "\x9c\x91",
  [0x204c] = "\x9c\x92", [0x204d] = "\xcd\xc4", [0x204e] = "\xe4\xcf",
  [0x204f] = "\x9c\x93", [0x2050] = "\x9c\x94", [0x2051] = "\x9c\x95",
  [0x2052] = "\x9c\x96", [0x2053] = "\xe4\xd4", [0x2054] = "\xe4\xd5",
  [0x2055] = "\x9c\x97", [0x2056] = "\xba\xfe", [0x2057] = "\x9c\x98",
  [0x2058] = "\xcf\xe6", [0x2059] = "\x9c\x99", [0x205a] = "\x9c\x9a",
  [0x205b] = "\xd5\xbf", [0x205c] = "\x9c\x9b", [0x205d] = "\x9c\x9c",
  [0x205e] = "\x9c\x9d", [0x205f] = "\xe4\xd2", [0x2060] = "\x9c\x9e",
  [0x2061] = "\x9c\x9f", [0x2062] = "\x9c\xa0", [0x2063] = "\x9c\xa1",
  [0x2064] = "\x9c\xa2", [0x2065] = "\x9c\xa3", [0x2066] = "\x9c\xa4",
  [0x2067] = "\x9c\xa5", [0x2068] = "\x9c\xa6", [0x2069] = "\x9c\xa7",
  [0x206a] = "\x9c\xa8", [0x206b] = "\xe4\xd0", [0x206c] = "\x9c\xa9",
  [0x206d] = "\x9c\xaa", [0x206e] = "\xe4\xce", [0x206f] = "\x9c\xab",
  [0x2070] = "\x9c\xac", [0x2071] = "\x9c\xad", [0x2072] = "\x9c\xae",
  [0x2073] = "\x9c\xaf", [0x2074] = "\x9c\xb0", [0x2075] = "\x9c\xb1",
  [0x2076] = "\x9c\xb2", [0x2077] = "\x9c\xb3", [0x2078] = "\x9c\xb4",
  [0x2079] = "\x9c\xb5", [0x207a] = "\x9c\xb6", [0x207b] = "\x9c\xb7",
  [0x207c] = "\x9c\xb8", [0x207d] = "\x9c\xb9", [0x207e] = "\xcd\xe5",
  [0x207f] = "\xca\xaa", [0x2080] = "\x9c\xba", [0x2081] = "\x9c\xbb",
  [0x2082] = "\x9c\xbc", [0x2083] = "\xc0\xa3", [0x2084] = "\x9c\xbd",
  [0x2085] = "\xbd\xa6", [0x2086] = "\xe4\xd3", [0x2087] = "\x9c\xbe",
  [0x2088] = "\x9c\xbf", [0x2089] = "\xb8\xc8", [0x208a] = "\x9c\xc0",
  [0x208b] = "\x9c\xc1", [0x208c] = "\x9c\xc2", [0x208d] = "\x9c\xc3",
  [0x208e] = "\x9c\xc4", [0x208f] = "\xe4\xe7", [0x2090] = "\xd4\xb4",
  [0x2091] = "\x9c\xc5", [0x2092] = "\x9c\xc6", [0x2093] = "\x9c\xc7",
  [0x2094] = "\x9c\xc8", [0x2095] = "\x9c\xc9", [0x2096] = "\x9c\xca",
  [0x2097] = "\x9c\xcb", [0x2098] = "\xe4\xdb", [0x2099] = "\x9c\xcc",
  [0x209a] = "\x9c\xcd", [0x209b] = "\x9c\xce", [0x209c] = "\xc1\xef",
  [0x209d] = "\x9c\xcf", [0x209e] = "\x9c\xd0", [0x209f] = "\xe4\xe9",
  [0x20a0] = "\x9c\xd1", [0x20a1] = "\x9c\xd2", [0x20a2] = "\xd2\xe7",
  [0x20a3] = "\x9c\xd3", [0x20a4] = "\x9c\xd4", [0x20a5] = "\xe4\xdf",
  [0x20a6] = "\x9c\xd5", [0x20a7] = "\xe4\xe0", [0x20a8] = "\x9c\xd6",
  [0x20a9] = "\x9c\xd7", [0x20aa] = "\xcf\xaa", [0x20ab] = "\x9c\xd8",
  [0x20ac] = "\x9c\xd9", [0x20ad] = "\x9c\xda", [0x20ae] = "\x9c\xdb",
  [0x20af] = "\xcb\xdd", [0x20b0] = "\x9c\xdc", [0x20b1] = "\xe4\xda",
  [0x20b2] = "\xe4\xd1", [0x20b3] = "\x9c\xdd", [0x20b4] = "\xe4\xe5",
  [0x20b5] = "\x9c\xde", [0x20b6] = "\xc8\xdc", [0x20b7] = "\xe4\xe3",
  [0x20b8] = "\x9c\xdf", [0x20b9] = "\x9c\xe0", [0x20ba] = "\xc4\xe7",
  [0x20bb] = "\xe4\xe2", [0x20bc] = "\x9c\xe1", [0x20bd] = "\xe4\xe1",
  [0x20be] = "\x9c\xe2", [0x20bf] = "\x9c\xe3", [0x20c0] = "\x9c\xe4",
  [0x20c1] = "\xb3\xfc", [0x20c2] = "\xe4\xe8", [0x20c3] = "\x9c\xe5",
  [0x20c4] = "\x9c\xe6", [0x20c5] = "\x9c\xe7", [0x20c6] = "\x9c\xe8",
  [0x20c7] = "\xb5\xe1", [0x20c8] = "\x9c\xe9", [0x20c9] = "\x9c\xea",
  [0x20ca] = "\x9c\xeb", [0x20cb] = "\xd7\xcc", [0x20cc] = "\x9c\xec",
  [0x20cd] = "\x9c\xed", [0x20ce] = "\x9c\xee", [0x20cf] = "\xe4\xe6",
  [0x20d0] = "\x9c\xef", [0x20d1] = "\xbb\xac", [0x20d2] = "\x9c\xf0",
  [0x20d3] = "\xd7\xd2", [0x20d4] = "\xcc\xcf", [0x20d5] = "\xeb\xf8",
  [0x20d6] = "\x9c\xf1", [0x20d7] = "\xe4\xe4", [0x20d8] = "\x9c\xf2",
  [0x20d9] = "\x9c\xf3", [0x20da] = "\xb9\xf6", [0x20db] = "\x9c\xf4",
  [0x20dc] = "\x9c\xf5", [0x20dd] = "\x9c\xf6", [0x20de] = "\xd6\xcd",
  [0x20df] = "\xe4\xd9", [0x20e0] = "\xe4\xdc", [0x20e1] = "\xc2\xfa",
  [0x20e2] = "\xe4\xde", [0x20e3] = "\x9c\xf7", [0x20e4] = "\xc2\xcb",
  [0x20e5] = "\xc0\xc4", [0x20e6] = "\xc2\xd0", [0x20e7] = "\x9c\xf8",
  [0x20e8] = "\xb1\xf5", [0x20e9] = "\xcc\xb2", [0x20ea] = "\x9c\xf9",
  [0x20eb] = "\x9c\xfa", [0x20ec] = "\x9c\xfb", [0x20ed] = "\x9c\xfc",
  [0x20ee] = "\x9c\xfd", [0x20ef] = "\x9c\xfe", [0x20f0] = "\x9d\x40",
  [0x20f1] = "\x9d\x41", [0x20f2] = "\x9d\x42", [0x20f3] = "\x9d\x43",
  [0x20f4] = "\xb5\xce", [0x20f5] = "\x9d\x44", [0x20f6] = "\x9d\x45",
  [0x20f7] = "\x9d\x46", [0x20f8] = "\x9d\x47", [0x20f9] = "\xe4\xef",
  [0x20fa] = "\x9d\x48", [0x20fb] = "\x9d\x49", [0x20fc] = "\x9d\x4a",
  [0x20fd] = "\x9d\x4b", [0x20fe] = "\x9d\x4c", [0x20ff] = "\x9d\x4d",
  [0x2100] = "\x9d\x4e", [0x2101] = "\x9d\x4f", [0x2102] = "\xc6\xaf",
  [0x2103] = "\x9d\x50", [0x2104] = "\x9d\x51", [0x2105] = "\x9d\x52",
  [0x2106] = "\xc6\xe1", [0x2107] = "\x9d\x53", [0x2108] = "\x9d\x54",
  [0x2109] = "\xe4\xf5", [0x210a] = "\x9d\x55", [0x210b] = "\x9d\x56",
  [0x210c] = "\x9d\x57", [0x210d] = "\x9d\x58", [0x210e] = "\x9d\x59",
  [0x210f] = "\xc2\xa9", [0x2110] = "\x9d\x5a", [0x2111] = "\x9d\x5b",
  [0x2112] = "\x9d\x5c", [0x2113] = "\xc0\xec", [0x2114] = "\xd1\xdd",
  [0x2115] = "\xe4\xee", [0x2116] = "\x9d\x5d", [0x2117] = "\x9d\x5e",
  [0x2118] = "\x9d\x5f", [0x2119] = "\x9d\x60", [0x211a] = "\x9d\x61",
  [0x211b] = "\x9d\x62", [0x211c] = "\x9d\x63", [0x211d] = "\x9d\x64",
  [0x211e] = "\x9d\x65", [0x211f] = "\x9d\x66", [0x2120] = "\xc4\xae",
  [0x2121] = "\x9d\x67", [0x2122] = "\x9d\x68", [0x2123] = "\x9d\x69",
  [0x2124] = "\xe4\xed", [0x2125] = "\x9d\x6a", [0x2126] = "\x9d\x6b",
  [0x2127] = "\x9d\x6c", [0x2128] = "\x9d\x6d", [0x2129] = "\xe4\xf6",
  [0x212a] = "\xe4\xf4", [0x212b] = "\xc2\xfe", [0x212c] = "\x9d\x6e",
  [0x212d] = "\xe4\xdd", [0x212e] = "\x9d\x6f", [0x212f] = "\xe4\xf0",
  [0x2130] = "\x9d\x70", [0x2131] = "\xca\xfe", [0x2132] = "\x9d\x71",
  [0x2133] = "\xd5\xc4", [0x2134] = "\x9d\x72", [0x2135] = "\x9d\x73",
  [0x2136] = "\xe4\xf1", [0x2137] = "\x9d\x74", [0x2138] = "\x9d\x75",
  [0x2139] = "\x9d\x76", [0x213a] = "\x9d\x77", [0x213b] = "\x9d\x78",
  [0x213c] = "\x9d\x79", [0x213d] = "\x9d\x7a", [0x213e] = "\xd1\xfa",
  [0x213f] = "\x9d\x7b", [0x2140] = "\x9d\x7c", [0x2141] = "\x9d\x7d",
  [0x2142] = "\x9d\x7e", [0x2143] = "\x9d\x80", [0x2144] = "\x9d\x81",
  [0x2145] = "\x9d\x82", [0x2146] = "\xe4\xeb", [0x2147] = "\xe4\xec",
  [0x2148] = "\x9d\x83", [0x2149] = "\x9d\x84", [0x214a] = "\x9d\x85",
  [0x214b] = "\xe4\xf2", [0x214c] = "\x9d\x86", [0x214d] = "\xce\xab",
  [0x214e] = "\x9d\x87", [0x214f] = "\x9d\x88", [0x2150] = "\x9d\x89",
  [0x2151] = "\x9d\x8a", [0x2152] = "\x9d\x8b", [0x2153] = "\x9d\x8c",
  [0x2154] = "\x9d\x8d", [0x2155] = "\x9d\x8e", [0x2156] = "\x9d\x8f",
  [0x2157] = "\x9d\x90", [0x2158] = "\xc5\xcb", [0x2159] = "\x9d\x91",
  [0x215a] = "\x9d\x92", [0x215b] = "\x9d\x93", [0x215c] = "\xc7\xb1",
  [0x215d] = "\x9d\x94", [0x215e] = "\xc2\xba", [0x215f] = "\x9d\x95",
  [0x2160] = "\x9d\x96", [0x2161] = "\x9d\x97", [0x2162] = "\xe4\xea",
  [0x2163] = "\x9d\x98", [0x2164] = "\x9d\x99", [0x2165] = "\x9d\x9a",
  [0x2166] = "\xc1\xca", [0x2167] = "\x9d\x9b", [0x2168] = "\x9d\x9c",
  [0x2169] = "\x9d\x9d", [0x216a] = "\x9d\x9e", [0x216b] = "\x9d\x9f",
  [0x216c] = "\x9d\xa0", [0x216d] = "\xcc\xb6", [0x216e] = "\xb3\xb1",
  [0x216f] = "\x9d\xa1", [0x2170] = "\x9d\xa2", [0x2171] = "\x9d\xa3",
  [0x2172] = "\xe4\xfb", [0x2173] = "\x9d\xa4", [0x2174] = "\xe4\xf3",
  [0x2175] = "\x9d\xa5", [0x2176] = "\x9d\xa6", [0x2177] = "\x9d\xa7",
  [0x2178] = "\xe4\xfa", [0x2179] = "\x9d\xa8", [0x217a] = "\xe4\xfd",
  [0x217b] = "\x9d\xa9", [0x217c] = "\xe4\xfc", [0x217d] = "\x9d\xaa",
  [0x217e] = "\x9d\xab", [0x217f] = "\x9d\xac", [0x2180] = "\x9d\xad",
  [0x2181] = "\x9d\xae", [0x2182] = "\x9d\xaf", [0x2183] = "\x9d\xb0",
  [0x2184] = "\xb3\xce", [0x2185] = "\x9d\xb1", [0x2186] = "\x9d\xb2",
  [0x2187] = "\x9d\xb3", [0x2188] = "\xb3\xba", [0x2189] = "\xe4\xf7",
  [0x218a] = "\x9d\xb4", [0x218b] = "\x9d\xb5", [0x218c] = "\xe4\xf9",
  [0x218d] = "\xe4\xf8", [0x218e] = "\xc5\xec", [0x218f] = "\x9d\xb6",
  [0x2190] = "\x9d\xb7", [0x2191] = "\x9d\xb8", [0x2192] = "\x9d\xb9",
  [0x2193] = "\x9d\xba", [0x2194] = "\x9d\xbb", [0x2195] = "\x9d\xbc",
  [0x2196] = "\x9d\xbd", [0x2197] = "\x9d\xbe", [0x2198] = "\x9d\xbf",
  [0x2199] = "\x9d\xc0", [0x219a] = "\x9d\xc1", [0x219b] = "\x9d\xc2",
  [0x219c] = "\xc0\xbd", [0x219d] = "\x9d\xc3", [0x219e] = "\x9d\xc4",
  [0x219f] = "\x9d\xc5", [0x21a0] = "\x9d\xc6", [0x21a1] = "\xd4\xe8",
  [0x21a2] = "\x9d\xc7", [0x21a3] = "\x9d\xc8", [0x21a4] = "\x9d\xc9",
  [0x21a5] = "\x9d\xca", [0x21a6] = "\x9d\xcb", [0x21a7] = "\xe5\xa2",
  [0x21a8] = "\x9d\xcc", [0x21a9] = "\x9d\xcd", [0x21aa] = "\x9d\xce",
  [0x21ab] = "\x9d\xcf", [0x21ac] = "\x9d\xd0", [0x21ad] = "\x9d\xd1",
  [0x21ae] = "\x9d\xd2", [0x21af] = "\x9d\xd3", [0x21b0] = "\x9d\xd4",
  [0x21b1] = "\x9d\xd5", [0x21b2] = "\x9d\xd6", [0x21b3] = "\xb0\xc4",
  [0x21b4] = "\x9d\xd7", [0x21b5] = "\x9d\xd8", [0x21b6] = "\xe5\xa4",
  [0x21b7] = "\x9d\xd9", [0x21b8] = "\x9d\xda", [0x21b9] = "\xe5\xa3",
  [0x21ba] = "\x9d\xdb", [0x21bb] = "\x9d\xdc", [0x21bc] = "\x9d\xdd",
  [0x21bd] = "\x9d\xde", [0x21be] = "\x9d\xdf", [0x21bf] = "\x9d\xe0",
  [0x21c0] = "\xbc\xa4", [0x21c1] = "\x9d\xe1", [0x21c2] = "\xe5\xa5",
  [0x21c3] = "\x9d\xe2", [0x21c4] = "\x9d\xe3", [0x21c5] = "\x9d\xe4",
  [0x21c6] = "\x9d\xe5", [0x21c7] = "\x9d\xe6", [0x21c8] = "\x9d\xe7",
  [0x21c9] = "\xe5\xa1", [0x21ca] = "\x9d\xe8", [0x21cb] = "\x9d\xe9",
  [0x21cc] = "\x9d\xea", [0x21cd] = "\x9d\xeb", [0x21ce] = "\x9d\xec",
  [0x21cf] = "\x9d\xed", [0x21d0] = "\x9d\xee", [0x21d1] = "\xe4\xfe",
  [0x21d2] = "\xb1\xf4", [0x21d3] = "\x9d\xef", [0x21d4] = "\x9d\xf0",
  [0x21d5] = "\x9d\xf1", [0x21d6] = "\x9d\xf2", [0x21d7] = "\x9d\xf3",
  [0x21d8] = "\x9d\xf4", [0x21d9] = "\x9d\xf5", [0x21da] = "\x9d\xf6",
  [0x21db] = "\x9d\xf7", [0x21dc] = "\x9d\xf8", [0x21dd] = "\x9d\xf9",
  [0x21de] = "\xe5\xa8", [0x21df] = "\x9d\xfa", [0x21e0] = "\xe5\xa9",
  [0x21e1] = "\xe5\xa6", [0x21e2] = "\x9d\xfb", [0x21e3] = "\x9d\xfc",
  [0x21e4] = "\x9d\xfd", [0x21e5] = "\x9d\xfe", [0x21e6] = "\x9e\x40",
  [0x21e7] = "\x9e\x41", [0x21e8] = "\x9e\x42", [0x21e9] = "\x9e\x43",
  [0x21ea] = "\x9e\x44", [0x21eb] = "\x9e\x45", [0x21ec] = "\x9e\x46",
  [0x21ed] = "\x9e\x47", [0x21ee] = "\xe5\xa7", [0x21ef] = "\xe5\xaa",
  [0x21f0] = "\x9e\x48", [0x21f1] = "\x9e\x49", [0x21f2] = "\x9e\x4a",
  [0x21f3] = "\x9e\x4b", [0x21f4] = "\x9e\x4c", [0x21f5] = "\x9e\x4d",
  [0x21f6] = "\x9e\x4e", [0x21f7] = "\x9e\x4f", [0x21f8] = "\x9e\x50",
  [0x21f9] = "\x9e\x51", [0x21fa] = "\x9e\x52", [0x21fb] = "\x9e\x53",
  [0x21fc] = "\x9e\x54", [0x21fd] = "\x9e\x55", [0x21fe] = "\x9e\x56",
  [0x21ff] = "\x9e\x57", [0x2200] = "\x9e\x58", [0x2201] = "\x9e\x59",
  [0x2202] = "\x9e\x5a", [0x2203] = "\x9e\x5b", [0x2204] = "\x9e\x5c",
  [0x2205] = "\x9e\x5d", [0x2206] = "\x9e\x5e", [0x2207] = "\x9e\x5f",
  [0x2208] = "\x9e\x60", [0x2209] = "\x9e\x61", [0x220a] = "\x9e\x62",
  [0x220b] = "\x9e\x63", [0x220c] = "\x9e\x64", [0x220d] = "\x9e\x65",
  [0x220e] = "\x9e\x66", [0x220f] = "\x9e\x67", [0x2210] = "\x9e\x68",
  [0x2211] = "\xc6\xd9", [0x2212] = "\x9e\x69", [0x2213] = "\x9e\x6a",
  [0x2214] = "\x9e\x6b", [0x2215] = "\x9e\x6c", [0x2216] = "\x9e\x6d",
  [0x2217] = "\x9e\x6e", [0x2218] = "\x9e\x6f", [0x2219] = "\x9e\x70",
  [0x221a] = "\xe5\xab", [0x221b] = "\xe5\xad", [0x221c] = "\x9e\x71",
  [0x221d] = "\x9e\x72", [0x221e] = "\x9e\x73", [0x221f] = "\x9e\x74",
  [0x2220] = "\x9e\x75", [0x2221] = "\x9e\x76", [0x2222] = "\x9e\x77",
  [0x2223] = "\xe5\xac", [0x2224] = "\x9e\x78", [0x2225] = "\x9e\x79",
  [0x2226] = "\x9e\x7a", [0x2227] = "\x9e\x7b", [0x2228] = "\x9e\x7c",
  [0x2229] = "\x9e\x7d", [0x222a] = "\x9e\x7e", [0x222b] = "\x9e\x80",
  [0x222c] = "\x9e\x81", [0x222d] = "\x9e\x82", [0x222e] = "\x9e\x83",
  [0x222f] = "\x9e\x84", [0x2230] = "\x9e\x85", [0x2231] = "\x9e\x86",
  [0x2232] = "\x9e\x87", [0x2233] = "\x9e\x88", [0x2234] = "\x9e\x89",
  [0x2235] = "\xe5\xaf", [0x2236] = "\x9e\x8a", [0x2237] = "\x9e\x8b",
  [0x2238] = "\x9e\x8c", [0x2239] = "\xe5\xae", [0x223a] = "\x9e\x8d",
  [0x223b] = "\x9e\x8e", [0x223c] = "\x9e\x8f", [0x223d] = "\x9e\x90",
  [0x223e] = "\x9e\x91", [0x223f] = "\x9e\x92", [0x2240] = "\x9e\x93",
  [0x2241] = "\x9e\x94", [0x2242] = "\x9e\x95", [0x2243] = "\x9e\x96",
  [0x2244] = "\x9e\x97", [0x2245] = "\x9e\x98", [0x2246] = "\x9e\x99",
  [0x2247] = "\x9e\x9a", [0x2248] = "\x9e\x9b", [0x2249] = "\x9e\x9c",
  [0x224a] = "\x9e\x9d", [0x224b] = "\x9e\x9e", [0x224c] = "\xb9\xe0",
  [0x224d] = "\x9e\x9f", [0x224e] = "\x9e\xa0", [0x224f] = "\xe5\xb0",
  [0x2250] = "\x9e\xa1", [0x2251] = "\x9e\xa2", [0x2252] = "\x9e\xa3",
  [0x2253] = "\x9e\xa4", [0x2254] = "\x9e\xa5", [0x2255] = "\x9e\xa6",
  [0x2256] = "\x9e\xa7", [0x2257] = "\x9e\xa8", [0x2258] = "\x9e\xa9",
  [0x2259] = "\x9e\xaa", [0x225a] = "\x9e\xab", [0x225b] = "\x9e\xac",
  [0x225c] = "\x9e\xad", [0x225d] = "\x9e\xae", [0x225e] = "\xe5\xb1",
  [0x225f] = "\x9e\xaf", [0x2260] = "\x9e\xb0", [0x2261] = "\x9e\xb1",
  [0x2262] = "\x9e\xb2", [0x2263] = "\x9e\xb3", [0x2264] = "\x9e\xb4",
  [0x2265] = "\x9e\xb5", [0x2266] = "\x9e\xb6", [0x2267] = "\x9e\xb7",
  [0x2268] = "\x9e\xb8", [0x2269] = "\x9e\xb9", [0x226a] = "\x9e\xba",
  [0x226b] = "\xbb\xf0", [0x226c] = "\xec\xe1", [0x226d] = "\xc3\xf0",
  [0x226e] = "\x9e\xbb", [0x226f] = "\xb5\xc6", [0x2270] = "\xbb\xd2",
  [0x2271] = "\x9e\xbc", [0x2272] = "\x9e\xbd", [0x2273] = "\x9e\xbe",
  [0x2274] = "\x9e\xbf", [0x2275] = "\xc1\xe9", [0x2276] = "\xd4\xee",
  [0x2277] = "\x9e\xc0", [0x2278] = "\xbe\xc4", [0x2279] = "\x9e\xc1",
  [0x227a] = "\x9e\xc2", [0x227b] = "\x9e\xc3", [0x227c] = "\xd7\xc6",
  [0x227d] = "\x9e\xc4", [0x227e] = "\xd4\xd6", [0x227f] = "\xb2\xd3",
  [0x2280] = "\xec\xbe", [0x2281] = "\x9e\xc5", [0x2282] = "\x9e\xc6",
  [0x2283] = "\x9e\xc7", [0x2284] = "\x9e\xc8", [0x2285] = "\xea\xc1",
  [0x2286] = "\x9e\xc9", [0x2287] = "\x9e\xca", [0x2288] = "\x9e\xcb",
  [0x2289] = "\xc2\xaf", [0x228a] = "\xb4\xb6", [0x228b] = "\x9e\xcc",
  [0x228c] = "\x9e\xcd", [0x228d] = "\x9e\xce", [0x228e] = "\xd1\xd7",
  [0x228f] = "\x9e\xcf", [0x2290] = "\x9e\xd0", [0x2291] = "\x9e\xd1",
  [0x2292] = "\xb3\xb4", [0x2293] = "\x9e\xd2", [0x2294] = "\xc8\xb2",
  [0x2295] = "\xbf\xbb", [0x2296] = "\xec\xc0", [0x2297] = "\x9e\xd3",
  [0x2298] = "\x9e\xd4", [0x2299] = "\xd6\xcb", [0x229a] = "\x9e\xd5",
  [0x229b] = "\x9e\xd6", [0x229c] = "\xec\xbf", [0x229d] = "\xec\xc1",
  [0x229e] = "\x9e\xd7", [0x229f] = "\x9e\xd8", [0x22a0] = "\x9e\xd9",
  [0x22a1] = "\x9e\xda", [0x22a2] = "\x9e\xdb", [0x22a3] = "\x9e\xdc",
  [0x22a4] = "\x9e\xdd", [0x22a5] = "\x9e\xde", [0x22a6] = "\x9e\xdf",
  [0x22a7] = "\x9e\xe0", [0x22a8] = "\x9e\xe1", [0x22a9] = "\x9e\xe2",
  [0x22aa] = "\x9e\xe3", [0x22ab] = "\xec\xc5", [0x22ac] = "\xbe\xe6",
  [0x22ad] = "\xcc\xbf", [0x22ae] = "\xc5\xda", [0x22af] = "\xbe\xbc",
  [0x22b0] = "\x9e\xe4", [0x22b1] = "\xec\xc6", [0x22b2] = "\x9e\xe5",
  [0x22b3] = "\xb1\xfe", [0x22b4] = "\x9e\xe6", [0x22b5] = "\x9e\xe7",
  [0x22b6] = "\x9e\xe8", [0x22b7] = "\xec\xc4", [0x22b8] = "\xd5\xa8",
  [0x22b9] = "\xb5\xe3", [0x22ba] = "\x9e\xe9", [0x22bb] = "\xec\xc2",
  [0x22bc] = "\xc1\xb6", [0x22bd] = "\xb3\xe3", [0x22be] = "\x9e\xea",
  [0x22bf] = "\x9e\xeb", [0x22c0] = "\xec\xc3", [0x22c1] = "\xcb\xb8",
  [0x22c2] = "\xc0\xc3", [0x22c3] = "\xcc\xfe", [0x22c4] = "\x9e\xec",
  [0x22c5] = "\x9e\xed", [0x22c6] = "\x9e\xee", [0x22c7] = "\x9e\xef",
  [0x22c8] = "\xc1\xd2", [0x22c9] = "\x9e\xf0", [0x22ca] = "\xec\xc8",
  [0x22cb] = "\x9e\xf1", [0x22cc] = "\x9e\xf2", [0x22cd] = "\x9e\xf3",
  [0x22ce] = "\x9e\xf4", [0x22cf] = "\x9e\xf5", [0x22d0] = "\x9e\xf6",
  [0x22d1] = "\x9e\xf7", [0x22d2] = "\x9e\xf8", [0x22d3] = "\x9e\xf9",
  [0x22d4] = "\x9e\xfa", [0x22d5] = "\x9e\xfb", [0x22d6] = "\x9e\xfc",
  [0x22d7] = "\x9e\xfd", [0x22d8] = "\xba\xe6", [0x22d9] = "\xc0\xd3",
  [0x22da] = "\x9e\xfe", [0x22db] = "\xd6\xf2", [0x22dc] = "\x9f\x40",
  [0x22dd] = "\x9f\x41", [0x22de] = "\x9f\x42", [0x22df] = "\xd1\xcc",
  [0x22e0] = "\x9f\x43", [0x22e1] = "\x9f\x44", [0x22e2] = "\x9f\x45",
  [0x22e3] = "\x9f\x46", [0x22e4] = "\xbf\xbe", [0x22e5] = "\x9f\x47",
  [0x22e6] = "\xb7\xb3", [0x22e7] = "\xc9\xd5", [0x22e8] = "\xec\xc7",
  [0x22e9] = "\xbb\xe2", [0x22ea] = "\x9f\x48", [0x22eb] = "\xcc\xcc",
  [0x22ec] = "\xbd\xfd", [0x22ed] = "\xc8\xc8", [0x22ee] = "\x9f\x49",
  [0x22ef] = "\xcf\xa9", [0x22f0] = "\x9f\x4a", [0x22f1] = "\x9f\x4b",
  [0x22f2] = "\x9f\x4c", [0x22f3] = "\x9f\x4d", [0x22f4] = "\x9f\x4e",
  [0x22f5] = "\x9f\x4f", [0x22f6] = "\x9f\x50", [0x22f7] = "\xcd\xe9",
  [0x22f8] = "\x9f\x51", [0x22f9] = "\xc5\xeb", [0x22fa] = "\x9f\x52",
  [0x22fb] = "\x9f\x53", [0x22fc] = "\x9f\x54", [0x22fd] = "\xb7\xe9",
  [0x22fe] = "\x9f\x55", [0x22ff] = "\x9f\x56", [0x2300] = "\x9f\x57",
  [0x2301] = "\x9f\x58", [0x2302] = "\x9f\x59", [0x2303] = "\x9f\x5a",
  [0x2304] = "\x9f\x5b", [0x2305] = "\x9f\x5c", [0x2306] = "\x9f\x5d",
  [0x2307] = "\x9f\x5e", [0x2308] = "\x9f\x5f", [0x2309] = "\xd1\xc9",
  [0x230a] = "\xba\xb8", [0x230b] = "\x9f\x60", [0x230c] = "\x9f\x61",
  [0x230d] = "\x9f\x62", [0x230e] = "\x9f\x63", [0x230f] = "\x9f\x64",
  [0x2310] = "\xec\xc9", [0x2311] = "\x9f\x65", [0x2312] = "\x9f\x66",
  [0x2313] = "\xec\xca", [0x2314] = "\x9f\x67", [0x2315] = "\xbb\xc0",
  [0x2316] = "\xec\xcb", [0x2317] = "\x9f\x68", [0x2318] = "\xec\xe2",
  [0x2319] = "\xb1\xba", [0x231a] = "\xb7\xd9", [0x231b] = "\x9f\x69",
  [0x231c] = "\x9f\x6a", [0x231d] = "\x9f\x6b", [0x231e] = "\x9f\x6c",
  [0x231f] = "\x9f\x6d", [0x2320] = "\x9f\x6e", [0x2321] = "\x9f\x6f",
  [0x2322] = "\x9f\x70", [0x2323] = "\x9f\x71", [0x2324] = "\x9f\x72",
  [0x2325] = "\x9f\x73", [0x2326] = "\xbd\xb9", [0x2327] = "\x9f\x74",
  [0x2328] = "\x9f\x75", [0x2329] = "\x9f\x76", [0x232a] = "\x9f\x77",
  [0x232b] = "\x9f\x78", [0x232c] = "\x9f\x79", [0x232d] = "\x9f\x7a",
  [0x232e] = "\x9f\x7b", [0x232f] = "\xec\xcc", [0x2330] = "\xd1\xe6",
  [0x2331] = "\xec\xcd", [0x2332] = "\x9f\x7c", [0x2333] = "\x9f\x7d",
  [0x2334] = "\x9f\x7e", [0x2335] = "\x9f\x80", [0x2336] = "\xc8\xbb",
  [0x2337] = "\x9f\x81", [0x2338] = "\x9f\x82", [0x2339] = "\x9f\x83",
  [0x233a] = "\x9f\x84", [0x233b] = "\x9f\x85", [0x233c] = "\x9f\x86",
  [0x233d] = "\x9f\x87", [0x233e] = "\x9f\x88", [0x233f] = "\x9f\x89",
  [0x2340] = "\x9f\x8a", [0x2341] = "\x9f\x8b", [0x2342] = "\x9f\x8c",
  [0x2343] = "\x9f\x8d", [0x2344] = "\x9f\x8e", [0x2345] = "\xec\xd1",
  [0x2346] = "\x9f\x8f", [0x2347] = "\x9f\x90", [0x2348] = "\x9f\x91",
  [0x2349] = "\x9f\x92", [0x234a] = "\xec\xd3", [0x234b] = "\x9f\x93",
  [0x234c] = "\xbb\xcd", [0x234d] = "\x9f\x94", [0x234e] = "\xbc\xe5",
  [0x234f] = "\x9f\x95", [0x2350] = "\x9f\x96", [0x2351] = "\x9f\x97",
  [0x2352] = "\x9f\x98", [0x2353] = "\x9f\x99", [0x2354] = "\x9f\x9a",
  [0x2355] = "\x9f\x9b", [0x2356] = "\x9f\x9c", [0x2357] = "\x9f\x9d",
  [0x2358] = "\x9f\x9e", [0x2359] = "\x9f\x9f", [0x235a] = "\x9f\xa0",
  [0x235b] = "\x9f\xa1", [0x235c] = "\xec\xcf", [0x235d] = "\x9f\xa2",
  [0x235e] = "\xc9\xb7", [0x235f] = "\x9f\xa3", [0x2360] = "\x9f\xa4",
  [0x2361] = "\x9f\xa5", [0x2362] = "\x9f\xa6", [0x2363] = "\x9f\xa7",
  [0x2364] = "\xc3\xba", [0x2365] = "\x9f\xa8", [0x2366] = "\xec\xe3",
  [0x2367] = "\xd5\xd5", [0x2368] = "\xec\xd0", [0x2369] = "\x9f\xa9",
  [0x236a] = "\x9f\xaa", [0x236b] = "\x9f\xab", [0x236c] = "\x9f\xac",
  [0x236d] = "\x9f\xad", [0x236e] = "\xd6\xf3", [0x236f] = "\x9f\xae",
  [0x2370] = "\x9f\xaf", [0x2371] = "\x9f\xb0", [0x2372] = "\xec\xd2",
  [0x2373] = "\xec\xce", [0x2374] = "\x9f\xb1", [0x2375] = "\x9f\xb2",
  [0x2376] = "\x9f\xb3", [0x2377] = "\x9f\xb4", [0x2378] = "\xec\xd4",
  [0x2379] = "\x9f\xb5", [0x237a] = "\xec\xd5", [0x237b] = "\x9f\xb6",
  [0x237c] = "\x9f\xb7", [0x237d] = "\xc9\xbf", [0x237e] = "\x9f\xb8",
  [0x237f] = "\x9f\xb9", [0x2380] = "\x9f\xba", [0x2381] = "\x9f\xbb",
  [0x2382] = "\x9f\xbc", [0x2383] = "\x9f\xbd", [0x2384] = "\xcf\xa8",
  [0x2385] = "\x9f\xbe", [0x2386] = "\x9f\xbf", [0x2387] = "\x9f\xc0",
  [0x2388] = "\x9f\xc1", [0x2389] = "\x9f\xc2", [0x238a] = "\xd0\xdc",
  [0x238b] = "\x9f\xc3", [0x238c] = "\x9f\xc4", [0x238d] = "\x9f\xc5",
  [0x238e] = "\x9f\xc6", [0x238f] = "\xd1\xac", [0x2390] = "\x9f\xc7",
  [0x2391] = "\x9f\xc8", [0x2392] = "\x9f\xc9", [0x2393] = "\x9f\xca",
  [0x2394] = "\xc8\xdb", [0x2395] = "\x9f\xcb", [0x2396] = "\x9f\xcc",
  [0x2397] = "\x9f\xcd", [0x2398] = "\xec\xd6", [0x2399] = "\xce\xf5",
  [0x239a] = "\x9f\xce", [0x239b] = "\x9f\xcf", [0x239c] = "\x9f\xd0",
  [0x239d] = "\x9f\xd1", [0x239e] = "\x9f\xd2", [0x239f] = "\xca\xec",
  [0x23a0] = "\xec\xda", [0x23a1] = "\x9f\xd3", [0x23a2] = "\x9f\xd4",
  [0x23a3] = "\x9f\xd5", [0x23a4] = "\x9f\xd6", [0x23a5] = "\x9f\xd7",
  [0x23a6] = "\x9f\xd8", [0x23a7] = "\x9f\xd9", [0x23a8] = "\xec\xd9",
  [0x23a9] = "\x9f\xda", [0x23aa] = "\x9f\xdb", [0x23ab] = "\x9f\xdc",
  [0x23ac] = "\xb0\xbe", [0x23ad] = "\x9f\xdd", [0x23ae] = "\x9f\xde",
  [0x23af] = "\x9f\xdf", [0x23b0] = "\x9f\xe0", [0x23b1] = "\x9f\xe1",
  [0x23b2] = "\x9f\xe2", [0x23b3] = "\xec\xd7", [0x23b4] = "\x9f\xe3",
  [0x23b5] = "\xec\xd8", [0x23b6] = "\x9f\xe4", [0x23b7] = "\x9f\xe5",
  [0x23b8] = "\x9f\xe6", [0x23b9] = "\xec\xe4", [0x23ba] = "\x9f\xe7",
  [0x23bb] = "\x9f\xe8", [0x23bc] = "\x9f\xe9", [0x23bd] = "\x9f\xea",
  [0x23be] = "\x9f\xeb", [0x23bf] = "\x9f\xec", [0x23c0] = "\x9f\xed",
  [0x23c1] = "\x9f\xee", [0x23c2] = "\x9f\xef", [0x23c3] = "\xc8\xbc",
  [0x23c4] = "\x9f\xf0", [0x23c5] = "\x9f\xf1", [0x23c6] = "\x9f\xf2",
  [0x23c7] = "\x9f\xf3", [0x23c8] = "\x9f\xf4", [0x23c9] = "\x9f\xf5",
  [0x23ca] = "\x9f\xf6", [0x23cb] = "\x9f\xf7", [0x23cc] = "\x9f\xf8",
  [0x23cd] = "\x9f\xf9", [0x23ce] = "\xc1\xc7", [0x23cf] = "\x9f\xfa",
  [0x23d0] = "\x9f\xfb", [0x23d1] = "\x9f\xfc", [0x23d2] = "\x9f\xfd",
  [0x23d3] = "\x9f\xfe", [0x23d4] = "\xec\xdc", [0x23d5] = "\xd1\xe0",
  [0x23d6] = "\xa0\x40", [0x23d7] = "\xa0\x41", [0x23d8] = "\xa0\x42",
  [0x23d9] = "\xa0\x43", [0x23da] = "\xa0\x44", [0x23db] = "\xa0\x45",
  [0x23dc] = "\xa0\x46", [0x23dd] = "\xa0\x47", [0x23de] = "\xa0\x48",
  [0x23df] = "\xa0\x49", [0x23e0] = "\xec\xdb", [0x23e1] = "\xa0\x4a",
  [0x23e2] = "\xa0\x4b", [0x23e3] = "\xa0\x4c", [0x23e4] = "\xa0\x4d",
  [0x23e5] = "\xd4\xef", [0x23e6] = "\xa0\x4e", [0x23e7] = "\xec\xdd",
  [0x23e8] = "\xa0\x4f", [0x23e9] = "\xa0\x50", [0x23ea] = "\xa0\x51",
  [0x23eb] = "\xa0\x52", [0x23ec] = "\xa0\x53", [0x23ed] = "\xa0\x54",
  [0x23ee] = "\xdb\xc6", [0x23ef] = "\xa0\x55", [0x23f0] = "\xa0\x56",
  [0x23f1] = "\xa0\x57", [0x23f2] = "\xa0\x58", [0x23f3] = "\xa0\x59",
  [0x23f4] = "\xa0\x5a", [0x23f5] = "\xa0\x5b", [0x23f6] = "\xa0\x5c",
  [0x23f7] = "\xa0\x5d", [0x23f8] = "\xa0\x5e", [0x23f9] = "\xec\xde",
  [0x23fa] = "\xa0\x5f", [0x23fb] = "\xa0\x60", [0x23fc] = "\xa0\x61",
  [0x23fd] = "\xa0\x62", [0x23fe] = "\xa0\x63", [0x23ff] = "\xa0\x64",
  [0x2400] = "\xa0\x65", [0x2401] = "\xa0\x66", [0x2402] = "\xa0\x67",
  [0x2403] = "\xa0\x68", [0x2404] = "\xa0\x69", [0x2405] = "\xa0\x6a",
  [0x2406] = "\xb1\xac", [0x2407] = "\xa0\x6b", [0x2408] = "\xa0\x6c",
  [0x2409] = "\xa0\x6d", [0x240a] = "\xa0\x6e", [0x240b] = "\xa0\x6f",
  [0x240c] = "\xa0\x70", [0x240d] = "\xa0\x71", [0x240e] = "\xa0\x72",
  [0x240f] = "\xa0\x73", [0x2410] = "\xa0\x74", [0x2411] = "\xa0\x75",
  [0x2412] = "\xa0\x76", [0x2413] = "\xa0\x77", [0x2414] = "\xa0\x78",
  [0x2415] = "\xa0\x79", [0x2416] = "\xa0\x7a", [0x2417] = "\xa0\x7b",
  [0x2418] = "\xa0\x7c", [0x2419] = "\xa0\x7d", [0x241a] = "\xa0\x7e",
  [0x241b] = "\xa0\x80", [0x241c] = "\xa0\x81", [0x241d] = "\xec\xdf",
  [0x241e] = "\xa0\x82", [0x241f] = "\xa0\x83", [0x2420] = "\xa0\x84",
  [0x2421] = "\xa0\x85", [0x2422] = "\xa0\x86", [0x2423] = "\xa0\x87",
  [0x2424] = "\xa0\x88", [0x2425] = "\xa0\x89", [0x2426] = "\xa0\x8a",
  [0x2427] = "\xa0\x8b", [0x2428] = "\xec\xe0", [0x2429] = "\xa0\x8c",
  [0x242a] = "\xd7\xa6", [0x242b] = "\xa0\x8d", [0x242c] = "\xc5\xc0",
  [0x242d] = "\xa0\x8e", [0x242e] = "\xa0\x8f", [0x242f] = "\xa0\x90",
  [0x2430] = "\xeb\xbc", [0x2431] = "\xb0\xae", [0x2432] = "\xa0\x91",
  [0x2433] = "\xa0\x92", [0x2434] = "\xa0\x93", [0x2435] = "\xbe\xf4",
  [0x2436] = "\xb8\xb8", [0x2437] = "\xd2\xaf", [0x2438] = "\xb0\xd6",
  [0x2439] = "\xb5\xf9", [0x243a] = "\xa0\x94", [0x243b] = "\xd8\xb3",
  [0x243c] = "\xa0\x95", [0x243d] = "\xcb\xac", [0x243e] = "\xa0\x96",
  [0x243f] = "\xe3\xdd", [0x2440] = "\xa0\x97", [0x2441] = "\xa0\x98",
  [0x2442] = "\xa0\x99", [0x2443] = "\xa0\x9a", [0x2444] = "\xa0\x9b",
  [0x2445] = "\xa0\x9c", [0x2446] = "\xa0\x9d", [0x2447] = "\xc6\xac",
  [0x2448] = "\xb0\xe6", [0x2449] = "\xa0\x9e", [0x244a] = "\xa0\x9f",
  [0x244b] = "\xa0\xa0", [0x244c] = "\xc5\xc6", [0x244d] = "\xeb\xb9",
  [0x244e] = "\xa0\xa1", [0x244f] = "\xa0\xa2", [0x2450] = "\xa0\xa3",
  [0x2451] = "\xa0\xa4", [0x2452] = "\xeb\xba", [0x2453] = "\xa0\xa5",
  [0x2454] = "\xa0\xa6", [0x2455] = "\xa0\xa7", [0x2456] = "\xeb\xbb",
  [0x2457] = "\xa0\xa8", [0x2458] = "\xa0\xa9", [0x2459] = "\xd1\xc0",
  [0x245a] = "\xa0\xaa", [0x245b] = "\xc5\xa3", [0x245c] = "\xa0\xab",
  [0x245d] = "\xea\xf2", [0x245e] = "\xa0\xac", [0x245f] = "\xc4\xb2",
  [0x2460] = "\xa0\xad", [0x2461] = "\xc4\xb5", [0x2462] = "\xc0\xce",
  [0x2463] = "\xa0\xae", [0x2464] = "\xa0\xaf", [0x2465] = "\xa0\xb0",
  [0x2466] = "\xea\xf3", [0x2467] = "\xc4\xc1", [0x2468] = "\xa0\xb1",
  [0x2469] = "\xce\xef", [0x246a] = "\xa0\xb2", [0x246b] = "\xa0\xb3",
  [0x246c] = "\xa0\xb4", [0x246d] = "\xa0\xb5", [0x246e] = "\xea\xf0",
  [0x246f] = "\xea\xf4", [0x2470] = "\xa0\xb6", [0x2471] = "\xa0\xb7",
  [0x2472] = "\xc9\xfc", [0x2473] = "\xa0\xb8", [0x2474] = "\xa0\xb9",
  [0x2475] = "\xc7\xa3", [0x2476] = "\xa0\xba", [0x2477] = "\xa0\xbb",
  [0x2478] = "\xa0\xbc", [0x2479] = "\xcc\xd8", [0x247a] = "\xce\xfe",
  [0x247b] = "\xa0\xbd", [0x247c] = "\xa0\xbe", [0x247d] = "\xa0\xbf",
  [0x247e] = "\xea\xf5", [0x247f] = "\xea\xf6", [0x2480] = "\xcf\xac",
  [0x2481] = "\xc0\xe7", [0x2482] = "\xa0\xc0", [0x2483] = "\xa0\xc1",
  [0x2484] = "\xea\xf7", [0x2485] = "\xa0\xc2", [0x2486] = "\xa0\xc3",
  [0x2487] = "\xa0\xc4", [0x2488] = "\xa0\xc5", [0x2489] = "\xa0\xc6",
  [0x248a] = "\xb6\xbf", [0x248b] = "\xea\xf8", [0x248c] = "\xa0\xc7",
  [0x248d] = "\xea\xf9", [0x248e] = "\xa0\xc8", [0x248f] = "\xea\xfa",
  [0x2490] = "\xa0\xc9", [0x2491] = "\xa0\xca", [0x2492] = "\xea\xfb",
  [0x2493] = "\xa0\xcb", [0x2494] = "\xa0\xcc", [0x2495] = "\xa0\xcd",
  [0x2496] = "\xa0\xce", [0x2497] = "\xa0\xcf", [0x2498] = "\xa0\xd0",
  [0x2499] = "\xa0\xd1", [0x249a] = "\xa0\xd2", [0x249b] = "\xa0\xd3",
  [0x249c] = "\xa0\xd4", [0x249d] = "\xa0\xd5", [0x249e] = "\xa0\xd6",
  [0x249f] = "\xea\xf1", [0x24a0] = "\xa0\xd7", [0x24a1] = "\xa0\xd8",
  [0x24a2] = "\xa0\xd9", [0x24a3] = "\xa0\xda", [0x24a4] = "\xa0\xdb",
  [0x24a5] = "\xa0\xdc", [0x24a6] = "\xa0\xdd", [0x24a7] = "\xa0\xde",
  [0x24a8] = "\xa0\xdf", [0x24a9] = "\xa0\xe0", [0x24aa] = "\xa0\xe1",
  [0x24ab] = "\xa0\xe2", [0x24ac] = "\xc8\xae", [0x24ad] = "\xe1\xeb",
  [0x24ae] = "\xa0\xe3", [0x24af] = "\xb7\xb8", [0x24b0] = "\xe1\xec",
  [0x24b1] = "\xa0\xe4", [0x24b2] = "\xa0\xe5", [0x24b3] = "\xa0\xe6",
  [0x24b4] = "\xe1\xed", [0x24b5] = "\xa0\xe7", [0x24b6] = "\xd7\xb4",
  [0x24b7] = "\xe1\xee", [0x24b8] = "\xe1\xef", [0x24b9] = "\xd3\xcc",
  [0x24ba] = "\xa0\xe8", [0x24bb] = "\xa0\xe9", [0x24bc] = "\xa0\xea",
  [0x24bd] = "\xa0\xeb", [0x24be] = "\xa0\xec", [0x24bf] = "\xa0\xed",
  [0x24c0] = "\xa0\xee", [0x24c1] = "\xe1\xf1", [0x24c2] = "\xbf\xf1",
  [0x24c3] = "\xe1\xf0", [0x24c4] = "\xb5\xd2", [0x24c5] = "\xa0\xef",
  [0x24c6] = "\xa0\xf0", [0x24c7] = "\xa0\xf1", [0x24c8] = "\xb1\xb7",
  [0x24c9] = "\xa0\xf2", [0x24ca] = "\xa0\xf3", [0x24cb] = "\xa0\xf4",
  [0x24cc] = "\xa0\xf5", [0x24cd] = "\xe1\xf3", [0x24ce] = "\xe1\xf2",
  [0x24cf] = "\xa0\xf6", [0x24d0] = "\xba\xfc", [0x24d1] = "\xa0\xf7",
  [0x24d2] = "\xe1\xf4", [0x24d3] = "\xa0\xf8", [0x24d4] = "\xa0\xf9",
  [0x24d5] = "\xa0\xfa", [0x24d6] = "\xa0\xfb", [0x24d7] = "\xb9\xb7",
  [0x24d8] = "\xa0\xfc", [0x24d9] = "\xbe\xd1", [0x24da] = "\xa0\xfd",
  [0x24db] = "\xa0\xfe", [0x24dc] = "\xaa\x40", [0x24dd] = "\xaa\x41",
  [0x24de] = "\xc4\xfc", [0x24df] = "\xaa\x42", [0x24e0] = "\xba\xdd",
  [0x24e1] = "\xbd\xc6", [0x24e2] = "\xaa\x43", [0x24e3] = "\xaa\x44",
  [0x24e4] = "\xaa\x45", [0x24e5] = "\xaa\x46", [0x24e6] = "\xaa\x47",
  [0x24e7] = "\xaa\x48", [0x24e8] = "\xe1\xf5", [0x24e9] = "\xe1\xf7",
  [0x24ea] = "\xaa\x49", [0x24eb] = "\xaa\x4a", [0x24ec] = "\xb6\xc0",
  [0x24ed] = "\xcf\xc1", [0x24ee] = "\xca\xa8", [0x24ef] = "\xe1\xf6",
  [0x24f0] = "\xd5\xf8", [0x24f1] = "\xd3\xfc", [0x24f2] = "\xe1\xf8",
  [0x24f3] = "\xe1\xfc", [0x24f4] = "\xe1\xf9", [0x24f5] = "\xaa\x4b",
  [0x24f6] = "\xaa\x4c", [0x24f7] = "\xe1\xfa", [0x24f8] = "\xc0\xea",
  [0x24f9] = "\xaa\x4d", [0x24fa] = "\xe1\xfe", [0x24fb] = "\xe2\xa1",
  [0x24fc] = "\xc0\xc7", [0x24fd] = "\xaa\x4e", [0x24fe] = "\xaa\x4f",
  [0x24ff] = "\xaa\x50", [0x2500] = "\xaa\x51", [0x2501] = "\xe1\xfb",
  [0x2502] = "\xaa\x52", [0x2503] = "\xe1\xfd", [0x2504] = "\xaa\x53",
  [0x2505] = "\xaa\x54", [0x2506] = "\xaa\x55", [0x2507] = "\xaa\x56",
  [0x2508] = "\xaa\x57", [0x2509] = "\xaa\x58", [0x250a] = "\xe2\xa5",
  [0x250b] = "\xaa\x59", [0x250c] = "\xaa\x5a", [0x250d] = "\xaa\x5b",
  [0x250e] = "\xc1\xd4", [0x250f] = "\xaa\x5c", [0x2510] = "\xaa\x5d",
  [0x2511] = "\xaa\x5e", [0x2512] = "\xaa\x5f", [0x2513] = "\xe2\xa3",
  [0x2514] = "\xaa\x60", [0x2515] = "\xe2\xa8", [0x2516] = "\xb2\xfe",
  [0x2517] = "\xe2\xa2", [0x2518] = "\xaa\x61", [0x2519] = "\xaa\x62",
  [0x251a] = "\xaa\x63", [0x251b] = "\xc3\xcd", [0x251c] = "\xb2\xc2",
  [0x251d] = "\xe2\xa7", [0x251e] = "\xe2\xa6", [0x251f] = "\xaa\x64",
  [0x2520] = "\xaa\x65", [0x2521] = "\xe2\xa4", [0x2522] = "\xe2\xa9",
  [0x2523] = "\xaa\x66", [0x2524] = "\xaa\x67", [0x2525] = "\xe2\xab",
  [0x2526] = "\xaa\x68", [0x2527] = "\xaa\x69", [0x2528] = "\xaa\x6a",
  [0x2529] = "\xd0\xc9", [0x252a] = "\xd6\xed", [0x252b] = "\xc3\xa8",
  [0x252c] = "\xe2\xac", [0x252d] = "\xaa\x6b", [0x252e] = "\xcf\xd7",
  [0x252f] = "\xaa\x6c", [0x2530] = "\xaa\x6d", [0x2531] = "\xe2\xae",
  [0x2532] = "\xaa\x6e", [0x2533] = "\xaa\x6f", [0x2534] = "\xba\xef",
  [0x2535] = "\xaa\x70", [0x2536] = "\xaa\x71", [0x2537] = "\xe9\xe0",
  [0x2538] = "\xe2\xad", [0x2539] = "\xe2\xaa", [0x253a] = "\xaa\x72",
  [0x253b] = "\xaa\x73", [0x253c] = "\xaa\x74", [0x253d] = "\xaa\x75",
  [0x253e] = "\xbb\xab", [0x253f] = "\xd4\xb3", [0x2540] = "\xaa\x76",
  [0x2541] = "\xaa\x77", [0x2542] = "\xaa\x78", [0x2543] = "\xaa\x79",
  [0x2544] = "\xaa\x7a", [0x2545] = "\xaa\x7b", [0x2546] = "\xaa\x7c",
  [0x2547] = "\xaa\x7d", [0x2548] = "\xaa\x7e", [0x2549] = "\xaa\x80",
  [0x254a] = "\xaa\x81", [0x254b] = "\xaa\x82", [0x254c] = "\xaa\x83",
  [0x254d] = "\xe2\xb0", [0x254e] = "\xaa\x84", [0x254f] = "\xaa\x85",
  [0x2550] = "\xe2\xaf", [0x2551] = "\xaa\x86", [0x2552] = "\xe9\xe1",
  [0x2553] = "\xaa\x87", [0x2554] = "\xaa\x88", [0x2555] = "\xaa\x89",
  [0x2556] = "\xaa\x8a", [0x2557] = "\xe2\xb1", [0x2558] = "\xaa\x8b",
  [0x2559] = "\xaa\x8c", [0x255a] = "\xaa\x8d", [0x255b] = "\xaa\x8e",
  [0x255c] = "\xaa\x8f", [0x255d] = "\xaa\x90", [0x255e] = "\xaa\x91",
  [0x255f] = "\xaa\x92", [0x2560] = "\xe2\xb2", [0x2561] = "\xaa\x93",
  [0x2562] = "\xaa\x94", [0x2563] = "\xaa\x95", [0x2564] = "\xaa\x96",
  [0x2565] = "\xaa\x97", [0x2566] = "\xaa\x98", [0x2567] = "\xaa\x99",
  [0x2568] = "\xaa\x9a", [0x2569] = "\xaa\x9b", [0x256a] = "\xaa\x9c",
  [0x256b] = "\xaa\x9d", [0x256c] = "\xe2\xb3", [0x256d] = "\xcc\xa1",
  [0x256e] = "\xaa\x9e", [0x256f] = "\xe2\xb4", [0x2570] = "\xaa\x9f",
  [0x2571] = "\xaa\xa0", [0x2572] = "\xab\x40", [0x2573] = "\xab\x41",
  [0x2574] = "\xab\x42", [0x2575] = "\xab\x43", [0x2576] = "\xab\x44",
  [0x2577] = "\xab\x45", [0x2578] = "\xab\x46", [0x2579] = "\xab\x47",
  [0x257a] = "\xab\x48", [0x257b] = "\xab\x49", [0x257c] = "\xab\x4a",
  [0x257d] = "\xab\x4b", [0x257e] = "\xe2\xb5", [0x257f] = "\xab\x4c",
  [0x2580] = "\xab\x4d", [0x2581] = "\xab\x4e", [0x2582] = "\xab\x4f",
  [0x2583] = "\xab\x50", [0x2584] = "\xd0\xfe", [0x2585] = "\xab\x51",
  [0x2586] = "\xab\x52", [0x2587] = "\xc2\xca", [0x2588] = "\xab\x53",
  [0x2589] = "\xd3\xf1", [0x258a] = "\xab\x54", [0x258b] = "\xcd\xf5",
  [0x258c] = "\xab\x55", [0x258d] = "\xab\x56", [0x258e] = "\xe7\xe0",
  [0x258f] = "\xab\x57", [0x2590] = "\xab\x58", [0x2591] = "\xe7\xe1",
  [0x2592] = "\xab\x59", [0x2593] = "\xab\x5a", [0x2594] = "\xab\x5b",
  [0x2595] = "\xab\x5c", [0x2596] = "\xbe\xc1", [0x2597] = "\xab\x5d",
  [0x2598] = "\xab\x5e", [0x2599] = "\xab\x5f", [0x259a] = "\xab\x60",
  [0x259b] = "\xc2\xea", [0x259c] = "\xab\x61", [0x259d] = "\xab\x62",
  [0x259e] = "\xab\x63", [0x259f] = "\xe7\xe4", [0x25a0] = "\xab\x64",
  [0x25a1] = "\xab\x65", [0x25a2] = "\xe7\xe3", [0x25a3] = "\xab\x66",
  [0x25a4] = "\xab\x67", [0x25a5] = "\xab\x68", [0x25a6] = "\xab\x69",
  [0x25a7] = "\xab\x6a", [0x25a8] = "\xab\x6b", [0x25a9] = "\xcd\xe6",
  [0x25aa] = "\xab\x6c", [0x25ab] = "\xc3\xb5", [0x25ac] = "\xab\x6d",
  [0x25ad] = "\xab\x6e", [0x25ae] = "\xe7\xe2", [0x25af] = "\xbb\xb7",
  [0x25b0] = "\xcf\xd6", [0x25b1] = "\xab\x6f", [0x25b2] = "\xc1\xe1",
  [0x25b3] = "\xe7\xe9", [0x25b4] = "\xab\x70", [0x25b5] = "\xab\x71",
  [0x25b6] = "\xab\x72", [0x25b7] = "\xe7\xe8", [0x25b8] = "\xab\x73",
  [0x25b9] = "\xab\x74", [0x25ba] = "\xe7\xf4", [0x25bb] = "\xb2\xa3",
  [0x25bc] = "\xab\x75", [0x25bd] = "\xab\x76", [0x25be] = "\xab\x77",
  [0x25bf] = "\xab\x78", [0x25c0] = "\xe7\xea", [0x25c1] = "\xab\x79",
  [0x25c2] = "\xe7\xe6", [0x25c3] = "\xab\x7a", [0x25c4] = "\xab\x7b",
  [0x25c5] = "\xab\x7c", [0x25c6] = "\xab\x7d", [0x25c7] = "\xab\x7e",
  [0x25c8] = "\xe7\xec", [0x25c9] = "\xe7\xeb", [0x25ca] = "\xc9\xba",
  [0x25cb] = "\xab\x80", [0x25cc] = "\xab\x81", [0x25cd] = "\xd5\xe4",
  [0x25ce] = "\xab\x82", [0x25cf] = "\xe7\xe5", [0x25d0] = "\xb7\xa9",
  [0x25d1] = "\xe7\xe7", [0x25d2] = "\xab\x83", [0x25d3] = "\xab\x84",
  [0x25d4] = "\xab\x85", [0x25d5] = "\xab\x86", [0x25d6] = "\xab\x87",
  [0x25d7] = "\xab\x88", [0x25d8] = "\xab\x89", [0x25d9] = "\xe7\xee",
  [0x25da] = "\xab\x8a", [0x25db] = "\xab\x8b", [0x25dc] = "\xab\x8c",
  [0x25dd] = "\xab\x8d", [0x25de] = "\xe7\xf3", [0x25df] = "\xab\x8e",
  [0x25e0] = "\xd6\xe9", [0x25e1] = "\xab\x8f", [0x25e2] = "\xab\x90",
  [0x25e3] = "\xab\x91", [0x25e4] = "\xab\x92", [0x25e5] = "\xe7\xed",
  [0x25e6] = "\xab\x93", [0x25e7] = "\xe7\xf2", [0x25e8] = "\xab\x94",
  [0x25e9] = "\xe7\xf1", [0x25ea] = "\xab\x95", [0x25eb] = "\xab\x96",
  [0x25ec] = "\xab\x97", [0x25ed] = "\xb0\xe0", [0x25ee] = "\xab\x98",
  [0x25ef] = "\xab\x99", [0x25f0] = "\xab\x9a", [0x25f1] = "\xab\x9b",
  [0x25f2] = "\xe7\xf5", [0x25f3] = "\xab\x9c", [0x25f4] = "\xab\x9d",
  [0x25f5] = "\xab\x9e", [0x25f6] = "\xab\x9f", [0x25f7] = "\xab\xa0",
  [0x25f8] = "\xac\x40", [0x25f9] = "\xac\x41", [0x25fa] = "\xac\x42",
  [0x25fb] = "\xac\x43", [0x25fc] = "\xac\x44", [0x25fd] = "\xac\x45",
  [0x25fe] = "\xac\x46", [0x25ff] = "\xac\x47", [0x2600] = "\xac\x48",
  [0x2601] = "\xac\x49", [0x2602] = "\xac\x4a", [0x2603] = "\xc7\xf2",
  [0x2604] = "\xac\x4b", [0x2605] = "\xc0\xc5", [0x2606] = "\xc0\xed",
  [0x2607] = "\xac\x4c", [0x2608] = "\xac\x4d", [0x2609] = "\xc1\xf0",
  [0x260a] = "\xe7\xf0", [0x260b] = "\xac\x4e", [0x260c] = "\xac\x4f",
  [0x260d] = "\xac\x50", [0x260e] = "\xac\x51", [0x260f] = "\xe7\xf6",
  [0x2610] = "\xcb\xf6", [0x2611] = "\xac\x52", [0x2612] = "\xac\x53",
  [0x2613] = "\xac\x54", [0x2614] = "\xac\x55", [0x2615] = "\xac\x56",
  [0x2616] = "\xac\x57", [0x2617] = "\xac\x58", [0x2618] = "\xac\x59",
  [0x2619] = "\xac\x5a", [0x261a] = "\xe8\xa2", [0x261b] = "\xe8\xa1",
  [0x261c] = "\xac\x5b", [0x261d] = "\xac\x5c", [0x261e] = "\xac\x5d",
  [0x261f] = "\xac\x5e", [0x2620] = "\xac\x5f", [0x2621] = "\xac\x60",
  [0x2622] = "\xd7\xc1", [0x2623] = "\xac\x61", [0x2624] = "\xac\x62",
  [0x2625] = "\xe7\xfa", [0x2626] = "\xe7\xf9", [0x2627] = "\xac\x63",
  [0x2628] = "\xe7\xfb", [0x2629] = "\xac\x64", [0x262a] = "\xe7\xf7",
  [0x262b] = "\xac\x65", [0x262c] = "\xe7\xfe", [0x262d] = "\xac\x66",
  [0x262e] = "\xe7\xfd", [0x262f] = "\xac\x67", [0x2630] = "\xe7\xfc",
  [0x2631] = "\xac\x68", [0x2632] = "\xac\x69", [0x2633] = "\xc1\xd5",
  [0x2634] = "\xc7\xd9", [0x2635] = "\xc5\xfd", [0x2636] = "\xc5\xc3",
  [0x2637] = "\xac\x6a", [0x2638] = "\xac\x6b", [0x2639] = "\xac\x6c",
  [0x263a] = "\xac\x6d", [0x263b] = "\xac\x6e", [0x263c] = "\xc7\xed",
  [0x263d] = "\xac\x6f", [0x263e] = "\xac\x70", [0x263f] = "\xac\x71",
  [0x2640] = "\xac\x72", [0x2641] = "\xe8\xa3", [0x2642] = "\xac\x73",
  [0x2643] = "\xac\x74", [0x2644] = "\xac\x75", [0x2645] = "\xac\x76",
  [0x2646] = "\xac\x77", [0x2647] = "\xac\x78", [0x2648] = "\xac\x79",
  [0x2649] = "\xac\x7a", [0x264a] = "\xac\x7b", [0x264b] = "\xac\x7c",
  [0x264c] = "\xac\x7d", [0x264d] = "\xac\x7e", [0x264e] = "\xac\x80",
  [0x264f] = "\xac\x81", [0x2650] = "\xac\x82", [0x2651] = "\xac\x83",
  [0x2652] = "\xac\x84", [0x2653] = "\xac\x85", [0x2654] = "\xac\x86",
  [0x2655] = "\xe8\xa6", [0x2656] = "\xac\x87", [0x2657] = "\xe8\xa5",
  [0x2658] = "\xac\x88", [0x2659] = "\xe8\xa7", [0x265a] = "\xba\xf7",
  [0x265b] = "\xe7\xf8", [0x265c] = "\xe8\xa4", [0x265d] = "\xac\x89",
  [0x265e] = "\xc8\xf0", [0x265f] = "\xc9\xaa", [0x2660] = "\xac\x8a",
  [0x2661] = "\xac\x8b", [0x2662] = "\xac\x8c", [0x2663] = "\xac\x8d",
  [0x2664] = "\xac\x8e", [0x2665] = "\xac\x8f", [0x2666] = "\xac\x90",
  [0x2667] = "\xac\x91", [0x2668] = "\xac\x92", [0x2669] = "\xac\x93",
  [0x266a] = "\xac\x94", [0x266b] = "\xac\x95", [0x266c] = "\xac\x96",
  [0x266d] = "\xe8\xa9", [0x266e] = "\xac\x97", [0x266f] = "\xac\x98",
  [0x2670] = "\xb9\xe5", [0x2671] = "\xac\x99", [0x2672] = "\xac\x9a",
  [0x2673] = "\xac\x9b", [0x2674] = "\xac\x9c", [0x2675] = "\xac\x9d",
  [0x2676] = "\xd1\xfe", [0x2677] = "\xe8\xa8", [0x2678] = "\xac\x9e",
  [0x2679] = "\xac\x9f", [0x267a] = "\xac\xa0", [0x267b] = "\xad\x40",
  [0x267c] = "\xad\x41", [0x267d] = "\xad\x42", [0x267e] = "\xe8\xaa",
  [0x267f] = "\xad\x43", [0x2680] = "\xe8\xad", [0x2681] = "\xe8\xae",
  [0x2682] = "\xad\x44", [0x2683] = "\xc1\xa7", [0x2684] = "\xad\x45",
  [0x2685] = "\xad\x46", [0x2686] = "\xad\x47", [0x2687] = "\xe8\xaf",
  [0x2688] = "\xad\x48", [0x2689] = "\xad\x49", [0x268a] = "\xad\x4a",
  [0x268b] = "\xe8\xb0", [0x268c] = "\xad\x4b", [0x268d] = "\xad\x4c",
  [0x268e] = "\xe8\xac", [0x268f] = "\xad\x4d", [0x2690] = "\xe8\xb4",
  [0x2691] = "\xad\x4e", [0x2692] = "\xad\x4f", [0x2693] = "\xad\x50",
  [0x2694] = "\xad\x51", [0x2695] = "\xad\x52", [0x2696] = "\xad\x53",
  [0x2697] = "\xad\x54", [0x2698] = "\xad\x55", [0x2699] = "\xad\x56",
  [0x269a] = "\xad\x57", [0x269b] = "\xad\x58", [0x269c] = "\xe8\xab",
  [0x269d] = "\xad\x59", [0x269e] = "\xe8\xb1", [0x269f] = "\xad\x5a",
  [0x26a0] = "\xad\x5b", [0x26a1] = "\xad\x5c", [0x26a2] = "\xad\x5d",
  [0x26a3] = "\xad\x5e", [0x26a4] = "\xad\x5f", [0x26a5] = "\xad\x60",
  [0x26a6] = "\xad\x61", [0x26a7] = "\xe8\xb5", [0x26a8] = "\xe8\xb2",
  [0x26a9] = "\xe8\xb3", [0x26aa] = "\xad\x62", [0x26ab] = "\xad\x63",
  [0x26ac] = "\xad\x64", [0x26ad] = "\xad\x65", [0x26ae] = "\xad\x66",
  [0x26af] = "\xad\x67", [0x26b0] = "\xad\x68", [0x26b1] = "\xad\x69",
  [0x26b2] = "\xad\x6a", [0x26b3] = "\xad\x6b", [0x26b4] = "\xad\x6c",
  [0x26b5] = "\xad\x6d", [0x26b6] = "\xad\x6e", [0x26b7] = "\xad\x6f",
  [0x26b8] = "\xad\x70", [0x26b9] = "\xad\x71", [0x26ba] = "\xe8\xb7",
  [0x26bb] = "\xad\x72", [0x26bc] = "\xad\x73", [0x26bd] = "\xad\x74",
  [0x26be] = "\xad\x75", [0x26bf] = "\xad\x76", [0x26c0] = "\xad\x77",
  [0x26c1] = "\xad\x78", [0x26c2] = "\xad\x79", [0x26c3] = "\xad\x7a",
  [0x26c4] = "\xad\x7b", [0x26c5] = "\xad\x7c", [0x26c6] = "\xad\x7d",
  [0x26c7] = "\xad\x7e", [0x26c8] = "\xad\x80", [0x26c9] = "\xad\x81",
  [0x26ca] = "\xad\x82", [0x26cb] = "\xad\x83", [0x26cc] = "\xad\x84",
  [0x26cd] = "\xad\x85", [0x26ce] = "\xad\x86", [0x26cf] = "\xad\x87",
  [0x26d0] = "\xad\x88", [0x26d1] = "\xad\x89", [0x26d2] = "\xe8\xb6",
  [0x26d3] = "\xad\x8a", [0x26d4] = "\xad\x8b", [0x26d5] = "\xad\x8c",
  [0x26d6] = "\xad\x8d", [0x26d7] = "\xad\x8e", [0x26d8] = "\xad\x8f",
  [0x26d9] = "\xad\x90", [0x26da] = "\xad\x91", [0x26db] = "\xad\x92",
  [0x26dc] = "\xb9\xcf", [0x26dd] = "\xad\x93", [0x26de] = "\xf0\xac",
  [0x26df] = "\xad\x94", [0x26e0] = "\xf0\xad", [0x26e1] = "\xad\x95",
  [0x26e2] = "\xc6\xb0", [0x26e3] = "\xb0\xea", [0x26e4] = "\xc8\xbf",
  [0x26e5] = "\xad\x96", [0x26e6] = "\xcd\xdf", [0x26e7] = "\xad\x97",
  [0x26e8] = "\xad\x98", [0x26e9] = "\xad\x99", [0x26ea] = "\xad\x9a",
  [0x26eb] = "\xad\x9b", [0x26ec] = "\xad\x9c", [0x26ed] = "\xad\x9d",
  [0x26ee] = "\xce\xcd", [0x26ef] = "\xea\xb1", [0x26f0] = "\xad\x9e",
  [0x26f1] = "\xad\x9f", [0x26f2] = "\xad\xa0", [0x26f3] = "\xae\x40",
  [0x26f4] = "\xea\xb2", [0x26f5] = "\xae\x41", [0x26f6] = "\xc6\xbf",
  [0x26f7] = "\xb4\xc9", [0x26f8] = "\xae\x42", [0x26f9] = "\xae\x43",
  [0x26fa] = "\xae\x44", [0x26fb] = "\xae\x45", [0x26fc] = "\xae\x46",
  [0x26fd] = "\xae\x47", [0x26fe] = "\xae\x48", [0x26ff] = "\xea\xb3",
  [0x2700] = "\xae\x49", [0x2701] = "\xae\x4a", [0x2702] = "\xae\x4b",
  [0x2703] = "\xae\x4c", [0x2704] = "\xd5\xe7", [0x2705] = "\xae\x4d",
  [0x2706] = "\xae\x4e", [0x2707] = "\xae\x4f", [0x2708] = "\xae\x50",
  [0x2709] = "\xae\x51", [0x270a] = "\xae\x52", [0x270b] = "\xae\x53",
  [0x270c] = "\xae\x54", [0x270d] = "\xdd\xf9", [0x270e] = "\xae\x55",
  [0x270f] = "\xea\xb4", [0x2710] = "\xae\x56", [0x2711] = "\xea\xb5",
  [0x2712] = "\xae\x57", [0x2713] = "\xea\xb6", [0x2714] = "\xae\x58",
  [0x2715] = "\xae\x59", [0x2716] = "\xae\x5a", [0x2717] = "\xae\x5b",
  [0x2718] = "\xb8\xca", [0x2719] = "\xdf\xb0", [0x271a] = "\xc9\xf5",
  [0x271b] = "\xae\x5c", [0x271c] = "\xcc\xf0", [0x271d] = "\xae\x5d",
  [0x271e] = "\xae\x5e", [0x271f] = "\xc9\xfa", [0x2720] = "\xae\x5f",
  [0x2721] = "\xae\x60", [0x2722] = "\xae\x61", [0x2723] = "\xae\x62",
  [0x2724] = "\xae\x63", [0x2725] = "\xc9\xfb", [0x2726] = "\xae\x64",
  [0x2727] = "\xae\x65", [0x2728] = "\xd3\xc3", [0x2729] = "\xcb\xa6",
  [0x272a] = "\xae\x66", [0x272b] = "\xb8\xa6", [0x272c] = "\xf0\xae",
  [0x272d] = "\xb1\xc2", [0x272e] = "\xae\x67", [0x272f] = "\xe5\xb8",
  [0x2730] = "\xcc\xef", [0x2731] = "\xd3\xc9", [0x2732] = "\xbc\xd7",
  [0x2733] = "\xc9\xea", [0x2734] = "\xae\x68", [0x2735] = "\xb5\xe7",
  [0x2736] = "\xae\x69", [0x2737] = "\xc4\xd0", [0x2738] = "\xb5\xe9",
  [0x2739] = "\xae\x6a", [0x273a] = "\xee\xae", [0x273b] = "\xbb\xad",
  [0x273c] = "\xae\x6b", [0x273d] = "\xae\x6c", [0x273e] = "\xe7\xde",
  [0x273f] = "\xae\x6d", [0x2740] = "\xee\xaf", [0x2741] = "\xae\x6e",
  [0x2742] = "\xae\x6f", [0x2743] = "\xae\x70", [0x2744] = "\xae\x71",
  [0x2745] = "\xb3\xa9", [0x2746] = "\xae\x72", [0x2747] = "\xae\x73",
  [0x2748] = "\xee\xb2", [0x2749] = "\xae\x74", [0x274a] = "\xae\x75",
  [0x274b] = "\xee\xb1", [0x274c] = "\xbd\xe7", [0x274d] = "\xae\x76",
  [0x274e] = "\xee\xb0", [0x274f] = "\xce\xb7", [0x2750] = "\xae\x77",
  [0x2751] = "\xae\x78", [0x2752] = "\xae\x79", [0x2753] = "\xae\x7a",
  [0x2754] = "\xc5\xcf", [0x2755] = "\xae\x7b", [0x2756] = "\xae\x7c",
  [0x2757] = "\xae\x7d", [0x2758] = "\xae\x7e", [0x2759] = "\xc1\xf4",
  [0x275a] = "\xdb\xce", [0x275b] = "\xee\xb3", [0x275c] = "\xd0\xf3",
  [0x275d] = "\xae\x80", [0x275e] = "\xae\x81", [0x275f] = "\xae\x82",
  [0x2760] = "\xae\x83", [0x2761] = "\xae\x84", [0x2762] = "\xae\x85",
  [0x2763] = "\xae\x86", [0x2764] = "\xae\x87", [0x2765] = "\xc2\xd4",
  [0x2766] = "\xc6\xe8", [0x2767] = "\xae\x88", [0x2768] = "\xae\x89",
  [0x2769] = "\xae\x8a", [0x276a] = "\xb7\xac", [0x276b] = "\xae\x8b",
  [0x276c] = "\xae\x8c", [0x276d] = "\xae\x8d", [0x276e] = "\xae\x8e",
  [0x276f] = "\xae\x8f", [0x2770] = "\xae\x90", [0x2771] = "\xae\x91",
  [0x2772] = "\xee\xb4", [0x2773] = "\xae\x92", [0x2774] = "\xb3\xeb",
  [0x2775] = "\xae\x93", [0x2776] = "\xae\x94", [0x2777] = "\xae\x95",
  [0x2778] = "\xbb\xfb", [0x2779] = "\xee\xb5", [0x277a] = "\xae\x96",
  [0x277b] = "\xae\x97", [0x277c] = "\xae\x98", [0x277d] = "\xae\x99",
  [0x277e] = "\xae\x9a", [0x277f] = "\xe7\xdc", [0x2780] = "\xae\x9b",
  [0x2781] = "\xae\x9c", [0x2782] = "\xae\x9d", [0x2783] = "\xee\xb6",
  [0x2784] = "\xae\x9e", [0x2785] = "\xae\x9f", [0x2786] = "\xbd\xae",
  [0x2787] = "\xae\xa0", [0x2788] = "\xaf\x40", [0x2789] = "\xaf\x41",
  [0x278a] = "\xaf\x42", [0x278b] = "\xf1\xe2", [0x278c] = "\xaf\x43",
  [0x278d] = "\xaf\x44", [0x278e] = "\xaf\x45", [0x278f] = "\xca\xe8",
  [0x2790] = "\xaf\x46", [0x2791] = "\xd2\xc9", [0x2792] = "\xf0\xda",
  [0x2793] = "\xaf\x47", [0x2794] = "\xf0\xdb", [0x2795] = "\xaf\x48",
  [0x2796] = "\xf0\xdc", [0x2797] = "\xc1\xc6", [0x2798] = "\xaf\x49",
  [0x2799] = "\xb8\xed", [0x279a] = "\xbe\xce", [0x279b] = "\xaf\x4a",
  [0x279c] = "\xaf\x4b", [0x279d] = "\xf0\xde", [0x279e] = "\xaf\x4c",
  [0x279f] = "\xc5\xb1", [0x27a0] = "\xf0\xdd", [0x27a1] = "\xd1\xf1",
  [0x27a2] = "\xaf\x4d", [0x27a3] = "\xf0\xe0", [0x27a4] = "\xb0\xcc",
  [0x27a5] = "\xbd\xea", [0x27a6] = "\xaf\x4e", [0x27a7] = "\xaf\x4f",
  [0x27a8] = "\xaf\x50", [0x27a9] = "\xaf\x51", [0x27aa] = "\xaf\x52",
  [0x27ab] = "\xd2\xdf", [0x27ac] = "\xf0\xdf", [0x27ad] = "\xaf\x53",
  [0x27ae] = "\xb4\xaf", [0x27af] = "\xb7\xe8", [0x27b0] = "\xf0\xe6",
  [0x27b1] = "\xf0\xe5", [0x27b2] = "\xc6\xa3", [0x27b3] = "\xf0\xe1",
  [0x27b4] = "\xf0\xe2", [0x27b5] = "\xb4\xc3", [0x27b6] = "\xaf\x54",
  [0x27b7] = "\xaf\x55", [0x27b8] = "\xf0\xe3", [0x27b9] = "\xd5\xee",
  [0x27ba] = "\xaf\x56", [0x27bb] = "\xaf\x57", [0x27bc] = "\xcc\xdb",
  [0x27bd] = "\xbe\xd2", [0x27be] = "\xbc\xb2", [0x27bf] = "\xaf\x58",
  [0x27c0] = "\xaf\x59", [0x27c1] = "\xaf\x5a", [0x27c2] = "\xf0\xe8",
  [0x27c3] = "\xf0\xe7", [0x27c4] = "\xf0\xe4", [0x27c5] = "\xb2\xa1",
  [0x27c6] = "\xaf\x5b", [0x27c7] = "\xd6\xa2", [0x27c8] = "\xd3\xb8",
  [0x27c9] = "\xbe\xb7", [0x27ca] = "\xc8\xac", [0x27cb] = "\xaf\x5c",
  [0x27cc] = "\xaf\x5d", [0x27cd] = "\xf0\xea", [0x27ce] = "\xaf\x5e",
  [0x27cf] = "\xaf\x5f", [0x27d0] = "\xaf\x60", [0x27d1] = "\xaf\x61",
  [0x27d2] = "\xd1\xf7", [0x27d3] = "\xaf\x62", [0x27d4] = "\xd6\xcc",
  [0x27d5] = "\xba\xdb", [0x27d6] = "\xf0\xe9", [0x27d7] = "\xaf\x63",
  [0x27d8] = "\xb6\xbb", [0x27d9] = "\xaf\x64", [0x27da] = "\xaf\x65",
  [0x27db] = "\xcd\xb4", [0x27dc] = "\xaf\x66", [0x27dd] = "\xaf\x67",
  [0x27de] = "\xc6\xa6", [0x27df] = "\xaf\x68", [0x27e0] = "\xaf\x69",
  [0x27e1] = "\xaf\x6a", [0x27e2] = "\xc1\xa1", [0x27e3] = "\xf0\xeb",
  [0x27e4] = "\xf0\xee", [0x27e5] = "\xaf\x6b", [0x27e6] = "\xf0\xed",
  [0x27e7] = "\xf0\xf0", [0x27e8] = "\xf0\xec", [0x27e9] = "\xaf\x6c",
  [0x27ea] = "\xbb\xbe", [0x27eb] = "\xf0\xef", [0x27ec] = "\xaf\x6d",
  [0x27ed] = "\xaf\x6e", [0x27ee] = "\xaf\x6f", [0x27ef] = "\xaf\x70",
  [0x27f0] = "\xcc\xb5", [0x27f1] = "\xf0\xf2", [0x27f2] = "\xaf\x71",
  [0x27f3] = "\xaf\x72", [0x27f4] = "\xb3\xd5", [0x27f5] = "\xaf\x73",
  [0x27f6] = "\xaf\x74", [0x27f7] = "\xaf\x75", [0x27f8] = "\xaf\x76",
  [0x27f9] = "\xb1\xd4", [0x27fa] = "\xaf\x77", [0x27fb] = "\xaf\x78",
  [0x27fc] = "\xf0\xf3", [0x27fd] = "\xaf\x79", [0x27fe] = "\xaf\x7a",
  [0x27ff] = "\xf0\xf4", [0x2800] = "\xf0\xf6", [0x2801] = "\xb4\xe1",
  [0x2802] = "\xaf\x7b", [0x2803] = "\xf0\xf1", [0x2804] = "\xaf\x7c",
  [0x2805] = "\xf0\xf7", [0x2806] = "\xaf\x7d", [0x2807] = "\xaf\x7e",
  [0x2808] = "\xaf\x80", [0x2809] = "\xaf\x81", [0x280a] = "\xf0\xfa",
  [0x280b] = "\xaf\x82", [0x280c] = "\xf0\xf8", [0x280d] = "\xaf\x83",
  [0x280e] = "\xaf\x84", [0x280f] = "\xaf\x85", [0x2810] = "\xf0\xf5",
  [0x2811] = "\xaf\x86", [0x2812] = "\xaf\x87", [0x2813] = "\xaf\x88",
  [0x2814] = "\xaf\x89", [0x2815] = "\xf0\xfd", [0x2816] = "\xaf\x8a",
  [0x2817] = "\xf0\xf9", [0x2818] = "\xf0\xfc", [0x2819] = "\xf0\xfe",
  [0x281a] = "\xaf\x8b", [0x281b] = "\xf1\xa1", [0x281c] = "\xaf\x8c",
  [0x281d] = "\xaf\x8d", [0x281e] = "\xaf\x8e", [0x281f] = "\xce\xc1",
  [0x2820] = "\xf1\xa4", [0x2821] = "\xaf\x8f", [0x2822] = "\xf1\xa3",
  [0x2823] = "\xaf\x90", [0x2824] = "\xc1\xf6", [0x2825] = "\xf0\xfb",
  [0x2826] = "\xca\xdd", [0x2827] = "\xaf\x91", [0x2828] = "\xaf\x92",
  [0x2829] = "\xb4\xf1", [0x282a] = "\xb1\xf1", [0x282b] = "\xcc\xb1",
  [0x282c] = "\xaf\x93", [0x282d] = "\xf1\xa6", [0x282e] = "\xaf\x94",
  [0x282f] = "\xaf\x95", [0x2830] = "\xf1\xa7", [0x2831] = "\xaf\x96",
  [0x2832] = "\xaf\x97", [0x2833] = "\xf1\xac", [0x2834] = "\xd5\xce",
  [0x2835] = "\xf1\xa9", [0x2836] = "\xaf\x98", [0x2837] = "\xaf\x99",
  [0x2838] = "\xc8\xb3", [0x2839] = "\xaf\x9a", [0x283a] = "\xaf\x9b",
  [0x283b] = "\xaf\x9c", [0x283c] = "\xf1\xa2", [0x283d] = "\xaf\x9d",
  [0x283e] = "\xf1\xab", [0x283f] = "\xf1\xa8", [0x2840] = "\xf1\xa5",
  [0x2841] = "\xaf\x9e", [0x2842] = "\xaf\x9f", [0x2843] = "\xf1\xaa",
  [0x2844] = "\xaf\xa0", [0x2845] = "\xb0\x40", [0x2846] = "\xb0\x41",
  [0x2847] = "\xb0\x42", [0x2848] = "\xb0\x43", [0x2849] = "\xb0\x44",
  [0x284a] = "\xb0\x45", [0x284b] = "\xb0\x46", [0x284c] = "\xb0\xa9",
  [0x284d] = "\xf1\xad", [0x284e] = "\xb0\x47", [0x284f] = "\xb0\x48",
  [0x2850] = "\xb0\x49", [0x2851] = "\xb0\x4a", [0x2852] = "\xb0\x4b",
  [0x2853] = "\xb0\x4c", [0x2854] = "\xf1\xaf", [0x2855] = "\xb0\x4d",
  [0x2856] = "\xf1\xb1", [0x2857] = "\xb0\x4e", [0x2858] = "\xb0\x4f",
  [0x2859] = "\xb0\x50", [0x285a] = "\xb0\x51", [0x285b] = "\xb0\x52",
  [0x285c] = "\xf1\xb0", [0x285d] = "\xb0\x53", [0x285e] = "\xf1\xae",
  [0x285f] = "\xb0\x54", [0x2860] = "\xb0\x55", [0x2861] = "\xb0\x56",
  [0x2862] = "\xb0\x57", [0x2863] = "\xd1\xa2", [0x2864] = "\xb0\x58",
  [0x2865] = "\xb0\x59", [0x2866] = "\xb0\x5a", [0x2867] = "\xb0\x5b",
  [0x2868] = "\xb0\x5c", [0x2869] = "\xb0\x5d", [0x286a] = "\xb0\x5e",
  [0x286b] = "\xf1\xb2", [0x286c] = "\xb0\x5f", [0x286d] = "\xb0\x60",
  [0x286e] = "\xb0\x61", [0x286f] = "\xf1\xb3", [0x2870] = "\xb0\x62",
  [0x2871] = "\xb0\x63", [0x2872] = "\xb0\x64", [0x2873] = "\xb0\x65",
  [0x2874] = "\xb0\x66", [0x2875] = "\xb0\x67", [0x2876] = "\xb0\x68",
  [0x2877] = "\xb0\x69", [0x2878] = "\xb9\xef", [0x2879] = "\xb0\x6a",
  [0x287a] = "\xb0\x6b", [0x287b] = "\xb5\xc7", [0x287c] = "\xb0\x6c",
  [0x287d] = "\xb0\xd7", [0x287e] = "\xb0\xd9", [0x287f] = "\xb0\x6d",
  [0x2880] = "\xb0\x6e", [0x2881] = "\xb0\x6f", [0x2882] = "\xd4\xed",
  [0x2883] = "\xb0\x70", [0x2884] = "\xb5\xc4", [0x2885] = "\xb0\x71",
  [0x2886] = "\xbd\xd4", [0x2887] = "\xbb\xca", [0x2888] = "\xf0\xa7",
  [0x2889] = "\xb0\x72", [0x288a] = "\xb0\x73", [0x288b] = "\xb8\xde",
  [0x288c] = "\xb0\x74", [0x288d] = "\xb0\x75", [0x288e] = "\xf0\xa8",
  [0x288f] = "\xb0\x76", [0x2890] = "\xb0\x77", [0x2891] = "\xb0\xa8",
  [0x2892] = "\xb0\x78", [0x2893] = "\xf0\xa9", [0x2894] = "\xb0\x79",
  [0x2895] = "\xb0\x7a", [0x2896] = "\xcd\xee", [0x2897] = "\xb0\x7b",
  [0x2898] = "\xb0\x7c", [0x2899] = "\xf0\xaa", [0x289a] = "\xb0\x7d",
  [0x289b] = "\xb0\x7e", [0x289c] = "\xb0\x80", [0x289d] = "\xb0\x81",
  [0x289e] = "\xb0\x82", [0x289f] = "\xb0\x83", [0x28a0] = "\xb0\x84",
  [0x28a1] = "\xb0\x85", [0x28a2] = "\xb0\x86", [0x28a3] = "\xb0\x87",
  [0x28a4] = "\xf0\xab", [0x28a5] = "\xb0\x88", [0x28a6] = "\xb0\x89",
  [0x28a7] = "\xb0\x8a", [0x28a8] = "\xb0\x8b", [0x28a9] = "\xb0\x8c",
  [0x28aa] = "\xb0\x8d", [0x28ab] = "\xb0\x8e", [0x28ac] = "\xb0\x8f",
  [0x28ad] = "\xb0\x90", [0x28ae] = "\xc6\xa4", [0x28af] = "\xb0\x91",
  [0x28b0] = "\xb0\x92", [0x28b1] = "\xd6\xe5", [0x28b2] = "\xf1\xe4",
  [0x28b3] = "\xb0\x93", [0x28b4] = "\xf1\xe5", [0x28b5] = "\xb0\x94",
  [0x28b6] = "\xb0\x95", [0x28b7] = "\xb0\x96", [0x28b8] = "\xb0\x97",
  [0x28b9] = "\xb0\x98", [0x28ba] = "\xb0\x99", [0x28bb] = "\xb0\x9a",
  [0x28bc] = "\xb0\x9b", [0x28bd] = "\xb0\x9c", [0x28be] = "\xb0\x9d",
  [0x28bf] = "\xc3\xf3", [0x28c0] = "\xb0\x9e", [0x28c1] = "\xb0\x9f",
  [0x28c2] = "\xd3\xdb", [0x28c3] = "\xb0\xa0", [0x28c4] = "\xb1\x40",
  [0x28c5] = "\xd6\xd1", [0x28c6] = "\xc5\xe8", [0x28c7] = "\xb1\x41",
  [0x28c8] = "\xd3\xaf", [0x28c9] = "\xb1\x42", [0x28ca] = "\xd2\xe6",
  [0x28cb] = "\xb1\x43", [0x28cc] = "\xb1\x44", [0x28cd] = "\xee\xc1",
  [0x28ce] = "\xb0\xbb", [0x28cf] = "\xd5\xb5", [0x28d0] = "\xd1\xce",
  [0x28d1] = "\xbc\xe0", [0x28d2] = "\xba\xd0", [0x28d3] = "\xb1\x45",
  [0x28d4] = "\xbf\xf8", [0x28d5] = "\xb1\x46", [0x28d6] = "\xb8\xc7",
  [0x28d7] = "\xb5\xc1", [0x28d8] = "\xc5\xcc", [0x28d9] = "\xb1\x47",
  [0x28da] = "\xb1\x48", [0x28db] = "\xca\xa2", [0x28dc] = "\xb1\x49",
  [0x28dd] = "\xb1\x4a", [0x28de] = "\xb1\x4b", [0x28df] = "\xc3\xcb",
  [0x28e0] = "\xb1\x4c", [0x28e1] = "\xb1\x4d", [0x28e2] = "\xb1\x4e",
  [0x28e3] = "\xb1\x4f", [0x28e4] = "\xb1\x50", [0x28e5] = "\xee\xc2",
  [0x28e6] = "\xb1\x51", [0x28e7] = "\xb1\x52", [0x28e8] = "\xb1\x53",
  [0x28e9] = "\xb1\x54", [0x28ea] = "\xb1\x55", [0x28eb] = "\xb1\x56",
  [0x28ec] = "\xb1\x57", [0x28ed] = "\xb1\x58", [0x28ee] = "\xc4\xbf",
  [0x28ef] = "\xb6\xa2", [0x28f0] = "\xb1\x59", [0x28f1] = "\xed\xec",
  [0x28f2] = "\xc3\xa4", [0x28f3] = "\xb1\x5a", [0x28f4] = "\xd6\xb1",
  [0x28f5] = "\xb1\x5b", [0x28f6] = "\xb1\x5c", [0x28f7] = "\xb1\x5d",
  [0x28f8] = "\xcf\xe0", [0x28f9] = "\xed\xef", [0x28fa] = "\xb1\x5e",
  [0x28fb] = "\xb1\x5f", [0x28fc] = "\xc5\xce", [0x28fd] = "\xb1\x60",
  [0x28fe] = "\xb6\xdc", [0x28ff] = "\xb1\x61", [0x2900] = "\xb1\x62",
  [0x2901] = "\xca\xa1", [0x2902] = "\xb1\x63", [0x2903] = "\xb1\x64",
  [0x2904] = "\xed\xed", [0x2905] = "\xb1\x65", [0x2906] = "\xb1\x66",
  [0x2907] = "\xed\xf0", [0x2908] = "\xed\xf1", [0x2909] = "\xc3\xbc",
  [0x290a] = "\xb1\x67", [0x290b] = "\xbf\xb4", [0x290c] = "\xb1\x68",
  [0x290d] = "\xed\xee", [0x290e] = "\xb1\x69", [0x290f] = "\xb1\x6a",
  [0x2910] = "\xb1\x6b", [0x2911] = "\xb1\x6c", [0x2912] = "\xb1\x6d",
  [0x2913] = "\xb1\x6e", [0x2914] = "\xb1\x6f", [0x2915] = "\xb1\x70",
  [0x2916] = "\xb1\x71", [0x2917] = "\xb1\x72", [0x2918] = "\xb1\x73",
  [0x2919] = "\xed\xf4", [0x291a] = "\xed\xf2", [0x291b] = "\xb1\x74",
  [0x291c] = "\xb1\x75", [0x291d] = "\xb1\x76", [0x291e] = "\xb1\x77",
  [0x291f] = "\xd5\xe6", [0x2920] = "\xc3\xdf", [0x2921] = "\xb1\x78",
  [0x2922] = "\xed\xf3", [0x2923] = "\xb1\x79", [0x2924] = "\xb1\x7a",
  [0x2925] = "\xb1\x7b", [0x2926] = "\xed\xf6", [0x2927] = "\xb1\x7c",
  [0x2928] = "\xd5\xa3", [0x2929] = "\xd1\xa3", [0x292a] = "\xb1\x7d",
  [0x292b] = "\xb1\x7e", [0x292c] = "\xb1\x80", [0x292d] = "\xed\xf5",
  [0x292e] = "\xb1\x81", [0x292f] = "\xc3\xd0", [0x2930] = "\xb1\x82",
  [0x2931] = "\xb1\x83", [0x2932] = "\xb1\x84", [0x2933] = "\xb1\x85",
  [0x2934] = "\xb1\x86", [0x2935] = "\xed\xf7", [0x2936] = "\xbf\xf4",
  [0x2937] = "\xbe\xec", [0x2938] = "\xed\xf8", [0x2939] = "\xb1\x87",
  [0x293a] = "\xcc\xf7", [0x293b] = "\xb1\x88", [0x293c] = "\xd1\xdb",
  [0x293d] = "\xb1\x89", [0x293e] = "\xb1\x8a", [0x293f] = "\xb1\x8b",
  [0x2940] = "\xd7\xc5", [0x2941] = "\xd5\xf6", [0x2942] = "\xb1\x8c",
  [0x2943] = "\xed\xfc", [0x2944] = "\xb1\x8d", [0x2945] = "\xb1\x8e",
  [0x2946] = "\xb1\x8f", [0x2947] = "\xed\xfb", [0x2948] = "\xb1\x90",
  [0x2949] = "\xb1\x91", [0x294a] = "\xb1\x92", [0x294b] = "\xb1\x93",
  [0x294c] = "\xb1\x94", [0x294d] = "\xb1\x95", [0x294e] = "\xb1\x96",
  [0x294f] = "\xb1\x97", [0x2950] = "\xed\xf9", [0x2951] = "\xed\xfa",
  [0x2952] = "\xb1\x98", [0x2953] = "\xb1\x99", [0x2954] = "\xb1\x9a",
  [0x2955] = "\xb1\x9b", [0x2956] = "\xb1\x9c", [0x2957] = "\xb1\x9d",
  [0x2958] = "\xb1\x9e", [0x2959] = "\xb1\x9f", [0x295a] = "\xed\xfd",
  [0x295b] = "\xbe\xa6", [0x295c] = "\xb1\xa0", [0x295d] = "\xb2\x40",
  [0x295e] = "\xb2\x41", [0x295f] = "\xb2\x42", [0x2960] = "\xb2\x43",
  [0x2961] = "\xcb\xaf", [0x2962] = "\xee\xa1", [0x2963] = "\xb6\xbd",
  [0x2964] = "\xb2\x44", [0x2965] = "\xee\xa2", [0x2966] = "\xc4\xc0",
  [0x2967] = "\xb2\x45", [0x2968] = "\xed\xfe", [0x2969] = "\xb2\x46",
  [0x296a] = "\xb2\x47", [0x296b] = "\xbd\xde", [0x296c] = "\xb2\xc7",
  [0x296d] = "\xb2\x48", [0x296e] = "\xb2\x49", [0x296f] = "\xb2\x4a",
  [0x2970] = "\xb2\x4b", [0x2971] = "\xb2\x4c", [0x2972] = "\xb2\x4d",
  [0x2973] = "\xb2\x4e", [0x2974] = "\xb2\x4f", [0x2975] = "\xb2\x50",
  [0x2976] = "\xb2\x51", [0x2977] = "\xb2\x52", [0x2978] = "\xb2\x53",
  [0x2979] = "\xb6\xc3", [0x297a] = "\xb2\x54", [0x297b] = "\xb2\x55",
  [0x297c] = "\xb2\x56", [0x297d] = "\xee\xa5", [0x297e] = "\xd8\xba",
  [0x297f] = "\xee\xa3", [0x2980] = "\xee\xa6", [0x2981] = "\xb2\x57",
  [0x2982] = "\xb2\x58", [0x2983] = "\xb2\x59", [0x2984] = "\xc3\xe9",
  [0x2985] = "\xb3\xf2", [0x2986] = "\xb2\x5a", [0x2987] = "\xb2\x5b",
  [0x2988] = "\xb2\x5c", [0x2989] = "\xb2\x5d", [0x298a] = "\xb2\x5e",
  [0x298b] = "\xb2\x5f", [0x298c] = "\xee\xa7", [0x298d] = "\xee\xa4",
  [0x298e] = "\xcf\xb9", [0x298f] = "\xb2\x60", [0x2990] = "\xb2\x61",
  [0x2991] = "\xee\xa8", [0x2992] = "\xc2\xf7", [0x2993] = "\xb2\x62",
  [0x2994] = "\xb2\x63", [0x2995] = "\xb2\x64", [0x2996] = "\xb2\x65",
  [0x2997] = "\xb2\x66", [0x2998] = "\xb2\x67", [0x2999] = "\xb2\x68",
  [0x299a] = "\xb2\x69", [0x299b] = "\xb2\x6a", [0x299c] = "\xb2\x6b",
  [0x299d] = "\xb2\x6c", [0x299e] = "\xb2\x6d", [0x299f] = "\xee\xa9",
  [0x29a0] = "\xee\xaa", [0x29a1] = "\xb2\x6e", [0x29a2] = "\xde\xab",
  [0x29a3] = "\xb2\x6f", [0x29a4] = "\xb2\x70", [0x29a5] = "\xc6\xb3",
  [0x29a6] = "\xb2\x71", [0x29a7] = "\xc7\xc6", [0x29a8] = "\xb2\x72",
  [0x29a9] = "\xd6\xf5", [0x29aa] = "\xb5\xc9", [0x29ab] = "\xb2\x73",
  [0x29ac] = "\xcb\xb2", [0x29ad] = "\xb2\x74", [0x29ae] = "\xb2\x75",
  [0x29af] = "\xb2\x76", [0x29b0] = "\xee\xab", [0x29b1] = "\xb2\x77",
  [0x29b2] = "\xb2\x78", [0x29b3] = "\xcd\xab", [0x29b4] = "\xb2\x79",
  [0x29b5] = "\xee\xac", [0x29b6] = "\xb2\x7a", [0x29b7] = "\xb2\x7b",
  [0x29b8] = "\xb2\x7c", [0x29b9] = "\xb2\x7d", [0x29ba] = "\xb2\x7e",
  [0x29bb] = "\xd5\xb0", [0x29bc] = "\xb2\x80", [0x29bd] = "\xee\xad",
  [0x29be] = "\xb2\x81", [0x29bf] = "\xf6\xc4", [0x29c0] = "\xb2\x82",
  [0x29c1] = "\xb2\x83", [0x29c2] = "\xb2\x84", [0x29c3] = "\xb2\x85",
  [0x29c4] = "\xb2\x86", [0x29c5] = "\xb2\x87", [0x29c6] = "\xb2\x88",
  [0x29c7] = "\xb2\x89", [0x29c8] = "\xb2\x8a", [0x29c9] = "\xb2\x8b",
  [0x29ca] = "\xb2\x8c", [0x29cb] = "\xb2\x8d", [0x29cc] = "\xb2\x8e",
  [0x29cd] = "\xdb\xc7", [0x29ce] = "\xb2\x8f", [0x29cf] = "\xb2\x90",
  [0x29d0] = "\xb2\x91", [0x29d1] = "\xb2\x92", [0x29d2] = "\xb2\x93",
  [0x29d3] = "\xb2\x94", [0x29d4] = "\xb2\x95", [0x29d5] = "\xb2\x96",
  [0x29d6] = "\xb2\x97", [0x29d7] = "\xb4\xa3", [0x29d8] = "\xb2\x98",
  [0x29d9] = "\xb2\x99", [0x29da] = "\xb2\x9a", [0x29db] = "\xc3\xac",
  [0x29dc] = "\xf1\xe6", [0x29dd] = "\xb2\x9b", [0x29de] = "\xb2\x9c",
  [0x29df] = "\xb2\x9d", [0x29e0] = "\xb2\x9e", [0x29e1] = "\xb2\x9f",
  [0x29e2] = "\xca\xb8", [0x29e3] = "\xd2\xd3", [0x29e4] = "\xb2\xa0",
  [0x29e5] = "\xd6\xaa", [0x29e6] = "\xb3\x40", [0x29e7] = "\xef\xf2",
  [0x29e8] = "\xb3\x41", [0x29e9] = "\xbe\xd8", [0x29ea] = "\xb3\x42",
  [0x29eb] = "\xbd\xc3", [0x29ec] = "\xef\xf3", [0x29ed] = "\xb6\xcc",
  [0x29ee] = "\xb0\xab", [0x29ef] = "\xb3\x43", [0x29f0] = "\xb3\x44",
  [0x29f1] = "\xb3\x45", [0x29f2] = "\xb3\x46", [0x29f3] = "\xca\xaf",
  [0x29f4] = "\xb3\x47", [0x29f5] = "\xb3\x48", [0x29f6] = "\xed\xb6",
  [0x29f7] = "\xb3\x49", [0x29f8] = "\xed\xb7", [0x29f9] = "\xb3\x4a",
  [0x29fa] = "\xb3\x4b", [0x29fb] = "\xb3\x4c", [0x29fc] = "\xb3\x4d",
  [0x29fd] = "\xce\xf9", [0x29fe] = "\xb7\xaf", [0x29ff] = "\xbf\xf3",
  [0x2a00] = "\xed\xb8", [0x2a01] = "\xc2\xeb", [0x2a02] = "\xc9\xb0",
  [0x2a03] = "\xb3\x4e", [0x2a04] = "\xb3\x4f", [0x2a05] = "\xb3\x50",
  [0x2a06] = "\xb3\x51", [0x2a07] = "\xb3\x52", [0x2a08] = "\xb3\x53",
  [0x2a09] = "\xed\xb9", [0x2a0a] = "\xb3\x54", [0x2a0b] = "\xb3\x55",
  [0x2a0c] = "\xc6\xf6", [0x2a0d] = "\xbf\xb3", [0x2a0e] = "\xb3\x56",
  [0x2a0f] = "\xb3\x57", [0x2a10] = "\xb3\x58", [0x2a11] = "\xed\xbc",
  [0x2a12] = "\xc5\xf8", [0x2a13] = "\xb3\x59", [0x2a14] = "\xd1\xd0",
  [0x2a15] = "\xb3\x5a", [0x2a16] = "\xd7\xa9", [0x2a17] = "\xed\xba",
  [0x2a18] = "\xed\xbb", [0x2a19] = "\xb3\x5b", [0x2a1a] = "\xd1\xe2",
  [0x2a1b] = "\xb3\x5c", [0x2a1c] = "\xed\xbf", [0x2a1d] = "\xed\xc0",
  [0x2a1e] = "\xb3\x5d", [0x2a1f] = "\xed\xc4", [0x2a20] = "\xb3\x5e",
  [0x2a21] = "\xb3\x5f", [0x2a22] = "\xb3\x60", [0x2a23] = "\xed\xc8",
  [0x2a24] = "\xb3\x61", [0x2a25] = "\xed\xc6", [0x2a26] = "\xed\xce",
  [0x2a27] = "\xd5\xe8", [0x2a28] = "\xb3\x62", [0x2a29] = "\xed\xc9",
  [0x2a2a] = "\xb3\x63", [0x2a2b] = "\xb3\x64", [0x2a2c] = "\xed\xc7",
  [0x2a2d] = "\xed\xbe", [0x2a2e] = "\xb3\x65", [0x2a2f] = "\xb3\x66",
  [0x2a30] = "\xc5\xe9", [0x2a31] = "\xb3\x67", [0x2a32] = "\xb3\x68",
  [0x2a33] = "\xb3\x69", [0x2a34] = "\xc6\xc6", [0x2a35] = "\xb3\x6a",
  [0x2a36] = "\xb3\x6b", [0x2a37] = "\xc9\xe9", [0x2a38] = "\xd4\xd2",
  [0x2a39] = "\xed\xc1", [0x2a3a] = "\xed\xc2", [0x2a3b] = "\xed\xc3",
  [0x2a3c] = "\xed\xc5", [0x2a3d] = "\xb3\x6c", [0x2a3e] = "\xc0\xf9",
  [0x2a3f] = "\xb3\x6d", [0x2a40] = "\xb4\xa1", [0x2a41] = "\xb3\x6e",
  [0x2a42] = "\xb3\x6f", [0x2a43] = "\xb3\x70", [0x2a44] = "\xb3\x71",
  [0x2a45] = "\xb9\xe8", [0x2a46] = "\xb3\x72", [0x2a47] = "\xed\xd0",
  [0x2a48] = "\xb3\x73", [0x2a49] = "\xb3\x74", [0x2a4a] = "\xb3\x75",
  [0x2a4b] = "\xb3\x76", [0x2a4c] = "\xed\xd1", [0x2a4d] = "\xb3\x77",
  [0x2a4e] = "\xed\xca", [0x2a4f] = "\xb3\x78", [0x2a50] = "\xed\xcf",
  [0x2a51] = "\xb3\x79", [0x2a52] = "\xce\xf8", [0x2a53] = "\xb3\x7a",
  [0x2a54] = "\xb3\x7b", [0x2a55] = "\xcb\xb6", [0x2a56] = "\xed\xcc",
  [0x2a57] = "\xed\xcd", [0x2a58] = "\xb3\x7c", [0x2a59] = "\xb3\x7d",
  [0x2a5a] = "\xb3\x7e", [0x2a5b] = "\xb3\x80", [0x2a5c] = "\xb3\x81",
  [0x2a5d] = "\xcf\xf5", [0x2a5e] = "\xb3\x82", [0x2a5f] = "\xb3\x83",
  [0x2a60] = "\xb3\x84", [0x2a61] = "\xb3\x85", [0x2a62] = "\xb3\x86",
  [0x2a63] = "\xb3\x87", [0x2a64] = "\xb3\x88", [0x2a65] = "\xb3\x89",
  [0x2a66] = "\xb3\x8a", [0x2a67] = "\xb3\x8b", [0x2a68] = "\xb3\x8c",
  [0x2a69] = "\xb3\x8d", [0x2a6a] = "\xed\xd2", [0x2a6b] = "\xc1\xf2",
  [0x2a6c] = "\xd3\xb2", [0x2a6d] = "\xed\xcb", [0x2a6e] = "\xc8\xb7",
  [0x2a6f] = "\xb3\x8e", [0x2a70] = "\xb3\x8f", [0x2a71] = "\xb3\x90",
  [0x2a72] = "\xb3\x91", [0x2a73] = "\xb3\x92", [0x2a74] = "\xb3\x93",
  [0x2a75] = "\xb3\x94", [0x2a76] = "\xb3\x95", [0x2a77] = "\xbc\xef",
  [0x2a78] = "\xb3\x96", [0x2a79] = "\xb3\x97", [0x2a7a] = "\xb3\x98",
  [0x2a7b] = "\xb3\x99", [0x2a7c] = "\xc5\xf0", [0x2a7d] = "\xb3\x9a",
  [0x2a7e] = "\xb3\x9b", [0x2a7f] = "\xb3\x9c", [0x2a80] = "\xb3\x9d",
  [0x2a81] = "\xb3\x9e", [0x2a82] = "\xb3\x9f", [0x2a83] = "\xb3\xa0",
  [0x2a84] = "\xb4\x40", [0x2a85] = "\xb4\x41", [0x2a86] = "\xb4\x42",
  [0x2a87] = "\xed\xd6", [0x2a88] = "\xb4\x43", [0x2a89] = "\xb5\xef",
  [0x2a8a] = "\xb4\x44", [0x2a8b] = "\xb4\x45", [0x2a8c] = "\xc2\xb5",
  [0x2a8d] = "\xb0\xad", [0x2a8e] = "\xcb\xe9", [0x2a8f] = "\xb4\x46",
  [0x2a90] = "\xb4\x47", [0x2a91] = "\xb1\xae", [0x2a92] = "\xb4\x48",
  [0x2a93] = "\xed\xd4", [0x2a94] = "\xb4\x49", [0x2a95] = "\xb4\x4a",
  [0x2a96] = "\xb4\x4b", [0x2a97] = "\xcd\xeb", [0x2a98] = "\xb5\xe2",
  [0x2a99] = "\xb4\x4c", [0x2a9a] = "\xed\xd5", [0x2a9b] = "\xed\xd3",
  [0x2a9c] = "\xed\xd7", [0x2a9d] = "\xb4\x4d", [0x2a9e] = "\xb4\x4e",
  [0x2a9f] = "\xb5\xfa", [0x2aa0] = "\xb4\x4f", [0x2aa1] = "\xed\xd8",
  [0x2aa2] = "\xb4\x50", [0x2aa3] = "\xed\xd9", [0x2aa4] = "\xb4\x51",
  [0x2aa5] = "\xed\xdc", [0x2aa6] = "\xb4\x52", [0x2aa7] = "\xb1\xcc",
  [0x2aa8] = "\xb4\x53", [0x2aa9] = "\xb4\x54", [0x2aaa] = "\xb4\x55",
  [0x2aab] = "\xb4\x56", [0x2aac] = "\xb4\x57", [0x2aad] = "\xb4\x58",
  [0x2aae] = "\xb4\x59", [0x2aaf] = "\xb4\x5a", [0x2ab0] = "\xc5\xf6",
  [0x2ab1] = "\xbc\xee", [0x2ab2] = "\xed\xda", [0x2ab3] = "\xcc\xbc",
  [0x2ab4] = "\xb2\xea", [0x2ab5] = "\xb4\x5b", [0x2ab6] = "\xb4\x5c",
  [0x2ab7] = "\xb4\x5d", [0x2ab8] = "\xb4\x5e", [0x2ab9] = "\xed\xdb",
  [0x2aba] = "\xb4\x5f", [0x2abb] = "\xb4\x60", [0x2abc] = "\xb4\x61",
  [0x2abd] = "\xb4\x62", [0x2abe] = "\xc4\xeb", [0x2abf] = "\xb4\x63",
  [0x2ac0] = "\xb4\x64", [0x2ac1] = "\xb4\xc5", [0x2ac2] = "\xb4\x65",
  [0x2ac3] = "\xb4\x66", [0x2ac4] = "\xb4\x67", [0x2ac5] = "\xb0\xf5",
  [0x2ac6] = "\xb4\x68", [0x2ac7] = "\xb4\x69", [0x2ac8] = "\xb4\x6a",
  [0x2ac9] = "\xed\xdf", [0x2aca] = "\xc0\xda", [0x2acb] = "\xb4\xe8",
  [0x2acc] = "\xb4\x6b", [0x2acd] = "\xb4\x6c", [0x2ace] = "\xb4\x6d",
  [0x2acf] = "\xb4\x6e", [0x2ad0] = "\xc5\xcd", [0x2ad1] = "\xb4\x6f",
  [0x2ad2] = "\xb4\x70", [0x2ad3] = "\xb4\x71", [0x2ad4] = "\xed\xdd",
  [0x2ad5] = "\xbf\xc4", [0x2ad6] = "\xb4\x72", [0x2ad7] = "\xb4\x73",
  [0x2ad8] = "\xb4\x74", [0x2ad9] = "\xed\xde", [0x2ada] = "\xb4\x75",
  [0x2adb] = "\xb4\x76", [0x2adc] = "\xb4\x77", [0x2add] = "\xb4\x78",
  [0x2ade] = "\xb4\x79", [0x2adf] = "\xb4\x7a", [0x2ae0] = "\xb4\x7b",
  [0x2ae1] = "\xb4\x7c", [0x2ae2] = "\xb4\x7d", [0x2ae3] = "\xb4\x7e",
  [0x2ae4] = "\xb4\x80", [0x2ae5] = "\xb4\x81", [0x2ae6] = "\xb4\x82",
  [0x2ae7] = "\xb4\x83", [0x2ae8] = "\xc4\xa5", [0x2ae9] = "\xb4\x84",
  [0x2aea] = "\xb4\x85", [0x2aeb] = "\xb4\x86", [0x2aec] = "\xed\xe0",
  [0x2aed] = "\xb4\x87", [0x2aee] = "\xb4\x88", [0x2aef] = "\xb4\x89",
  [0x2af0] = "\xb4\x8a", [0x2af1] = "\xb4\x8b", [0x2af2] = "\xed\xe1",
  [0x2af3] = "\xb4\x8c", [0x2af4] = "\xed\xe3", [0x2af5] = "\xb4\x8d",
  [0x2af6] = "\xb4\x8e", [0x2af7] = "\xc1\xd7", [0x2af8] = "\xb4\x8f",
  [0x2af9] = "\xb4\x90", [0x2afa] = "\xbb\xc7", [0x2afb] = "\xb4\x91",
  [0x2afc] = "\xb4\x92", [0x2afd] = "\xb4\x93", [0x2afe] = "\xb4\x94",
  [0x2aff] = "\xb4\x95", [0x2b00] = "\xb4\x96", [0x2b01] = "\xbd\xb8",
  [0x2b02] = "\xb4\x97", [0x2b03] = "\xb4\x98", [0x2b04] = "\xb4\x99",
  [0x2b05] = "\xed\xe2", [0x2b06] = "\xb4\x9a", [0x2b07] = "\xb4\x9b",
  [0x2b08] = "\xb4\x9c", [0x2b09] = "\xb4\x9d", [0x2b0a] = "\xb4\x9e",
  [0x2b0b] = "\xb4\x9f", [0x2b0c] = "\xb4\xa0", [0x2b0d] = "\xb5\x40",
  [0x2b0e] = "\xb5\x41", [0x2b0f] = "\xb5\x42", [0x2b10] = "\xb5\x43",
  [0x2b11] = "\xb5\x44", [0x2b12] = "\xb5\x45", [0x2b13] = "\xed\xe4",
  [0x2b14] = "\xb5\x46", [0x2b15] = "\xb5\x47", [0x2b16] = "\xb5\x48",
  [0x2b17] = "\xb5\x49", [0x2b18] = "\xb5\x4a", [0x2b19] = "\xb5\x4b",
  [0x2b1a] = "\xb5\x4c", [0x2b1b] = "\xb5\x4d", [0x2b1c] = "\xb5\x4e",
  [0x2b1d] = "\xb5\x4f", [0x2b1e] = "\xed\xe6", [0x2b1f] = "\xb5\x50",
  [0x2b20] = "\xb5\x51", [0x2b21] = "\xb5\x52", [0x2b22] = "\xb5\x53",
  [0x2b23] = "\xb5\x54", [0x2b24] = "\xed\xe5", [0x2b25] = "\xb5\x55",
  [0x2b26] = "\xb5\x56", [0x2b27] = "\xb5\x57", [0x2b28] = "\xb5\x58",
  [0x2b29] = "\xb5\x59", [0x2b2a] = "\xb5\x5a", [0x2b2b] = "\xb5\x5b",
  [0x2b2c] = "\xb5\x5c", [0x2b2d] = "\xb5\x5d", [0x2b2e] = "\xb5\x5e",
  [0x2b2f] = "\xb5\x5f", [0x2b30] = "\xb5\x60", [0x2b31] = "\xb5\x61",
  [0x2b32] = "\xb5\x62", [0x2b33] = "\xb5\x63", [0x2b34] = "\xed\xe7",
  [0x2b35] = "\xb5\x64", [0x2b36] = "\xb5\x65", [0x2b37] = "\xb5\x66",
  [0x2b38] = "\xb5\x67", [0x2b39] = "\xb5\x68", [0x2b3a] = "\xca\xbe",
  [0x2b3b] = "\xec\xea", [0x2b3c] = "\xc0\xf1", [0x2b3d] = "\xb5\x69",
  [0x2b3e] = "\xc9\xe7", [0x2b3f] = "\xb5\x6a", [0x2b40] = "\xec\xeb",
  [0x2b41] = "\xc6\xee", [0x2b42] = "\xb5\x6b", [0x2b43] = "\xb5\x6c",
  [0x2b44] = "\xb5\x6d", [0x2b45] = "\xb5\x6e", [0x2b46] = "\xec\xec",
  [0x2b47] = "\xb5\x6f", [0x2b48] = "\xc6\xed", [0x2b49] = "\xec\xed",
  [0x2b4a] = "\xb5\x70", [0x2b4b] = "\xb5\x71", [0x2b4c] = "\xb5\x72",
  [0x2b4d] = "\xb5\x73", [0x2b4e] = "\xb5\x74", [0x2b4f] = "\xb5\x75",
  [0x2b50] = "\xb5\x76", [0x2b51] = "\xb5\x77", [0x2b52] = "\xb5\x78",
  [0x2b53] = "\xec\xf0", [0x2b54] = "\xb5\x79", [0x2b55] = "\xb5\x7a",
  [0x2b56] = "\xd7\xe6", [0x2b57] = "\xec\xf3", [0x2b58] = "\xb5\x7b",
  [0x2b59] = "\xb5\x7c", [0x2b5a] = "\xec\xf1", [0x2b5b] = "\xec\xee",
  [0x2b5c] = "\xec\xef", [0x2b5d] = "\xd7\xa3", [0x2b5e] = "\xc9\xf1",
  [0x2b5f] = "\xcb\xee", [0x2b60] = "\xec\xf4", [0x2b61] = "\xb5\x7d",
  [0x2b62] = "\xec\xf2", [0x2b63] = "\xb5\x7e", [0x2b64] = "\xb5\x80",
  [0x2b65] = "\xcf\xe9", [0x2b66] = "\xb5\x81", [0x2b67] = "\xec\xf6",
  [0x2b68] = "\xc6\xb1", [0x2b69] = "\xb5\x82", [0x2b6a] = "\xb5\x83",
  [0x2b6b] = "\xb5\x84", [0x2b6c] = "\xb5\x85", [0x2b6d] = "\xbc\xc0",
  [0x2b6e] = "\xb5\x86", [0x2b6f] = "\xec\xf5", [0x2b70] = "\xb5\x87",
  [0x2b71] = "\xb5\x88", [0x2b72] = "\xb5\x89", [0x2b73] = "\xb5\x8a",
  [0x2b74] = "\xb5\x8b", [0x2b75] = "\xb5\x8c", [0x2b76] = "\xb5\x8d",
  [0x2b77] = "\xb5\xbb", [0x2b78] = "\xbb\xf6", [0x2b79] = "\xb5\x8e",
  [0x2b7a] = "\xec\xf7", [0x2b7b] = "\xb5\x8f", [0x2b7c] = "\xb5\x90",
  [0x2b7d] = "\xb5\x91", [0x2b7e] = "\xb5\x92", [0x2b7f] = "\xb5\x93",
  [0x2b80] = "\xd9\xf7", [0x2b81] = "\xbd\xfb", [0x2b82] = "\xb5\x94",
  [0x2b83] = "\xb5\x95", [0x2b84] = "\xc2\xbb", [0x2b85] = "\xec\xf8",
  [0x2b86] = "\xb5\x96", [0x2b87] = "\xb5\x97", [0x2b88] = "\xb5\x98",
  [0x2b89] = "\xb5\x99", [0x2b8a] = "\xec\xf9", [0x2b8b] = "\xb5\x9a",
  [0x2b8c] = "\xb5\x9b", [0x2b8d] = "\xb5\x9c", [0x2b8e] = "\xb5\x9d",
  [0x2b8f] = "\xb8\xa3", [0x2b90] = "\xb5\x9e", [0x2b91] = "\xb5\x9f",
  [0x2b92] = "\xb5\xa0", [0x2b93] = "\xb6\x40", [0x2b94] = "\xb6\x41",
  [0x2b95] = "\xb6\x42", [0x2b96] = "\xb6\x43", [0x2b97] = "\xb6\x44",
  [0x2b98] = "\xb6\x45", [0x2b99] = "\xb6\x46", [0x2b9a] = "\xec\xfa",
  [0x2b9b] = "\xb6\x47", [0x2b9c] = "\xb6\x48", [0x2b9d] = "\xb6\x49",
  [0x2b9e] = "\xb6\x4a", [0x2b9f] = "\xb6\x4b", [0x2ba0] = "\xb6\x4c",
  [0x2ba1] = "\xb6\x4d", [0x2ba2] = "\xb6\x4e", [0x2ba3] = "\xb6\x4f",
  [0x2ba4] = "\xb6\x50", [0x2ba5] = "\xb6\x51", [0x2ba6] = "\xb6\x52",
  [0x2ba7] = "\xec\xfb", [0x2ba8] = "\xb6\x53", [0x2ba9] = "\xb6\x54",
  [0x2baa] = "\xb6\x55", [0x2bab] = "\xb6\x56", [0x2bac] = "\xb6\x57",
  [0x2bad] = "\xb6\x58", [0x2bae] = "\xb6\x59", [0x2baf] = "\xb6\x5a",
  [0x2bb0] = "\xb6\x5b", [0x2bb1] = "\xb6\x5c", [0x2bb2] = "\xb6\x5d",
  [0x2bb3] = "\xec\xfc", [0x2bb4] = "\xb6\x5e", [0x2bb5] = "\xb6\x5f",
  [0x2bb6] = "\xb6\x60", [0x2bb7] = "\xb6\x61", [0x2bb8] = "\xb6\x62",
  [0x2bb9] = "\xd3\xed", [0x2bba] = "\xd8\xae", [0x2bbb] = "\xc0\xeb",
  [0x2bbc] = "\xb6\x63", [0x2bbd] = "\xc7\xdd", [0x2bbe] = "\xba\xcc",
  [0x2bbf] = "\xb6\x64", [0x2bc0] = "\xd0\xe3", [0x2bc1] = "\xcb\xbd",
  [0x2bc2] = "\xb6\x65", [0x2bc3] = "\xcd\xba", [0x2bc4] = "\xb6\x66",
  [0x2bc5] = "\xb6\x67", [0x2bc6] = "\xb8\xd1", [0x2bc7] = "\xb6\x68",
  [0x2bc8] = "\xb6\x69", [0x2bc9] = "\xb1\xfc", [0x2bca] = "\xb6\x6a",
  [0x2bcb] = "\xc7\xef", [0x2bcc] = "\xb6\x6b", [0x2bcd] = "\xd6\xd6",
  [0x2bce] = "\xb6\x6c", [0x2bcf] = "\xb6\x6d", [0x2bd0] = "\xb6\x6e",
  [0x2bd1] = "\xbf\xc6", [0x2bd2] = "\xc3\xeb", [0x2bd3] = "\xb6\x6f",
  [0x2bd4] = "\xb6\x70", [0x2bd5] = "\xef\xf5", [0x2bd6] = "\xb6\x71",
  [0x2bd7] = "\xb6\x72", [0x2bd8] = "\xc3\xd8", [0x2bd9] = "\xb6\x73",
  [0x2bda] = "\xb6\x74", [0x2bdb] = "\xb6\x75", [0x2bdc] = "\xb6\x76",
  [0x2bdd] = "\xb6\x77", [0x2bde] = "\xb6\x78", [0x2bdf] = "\xd7\xe2",
  [0x2be0] = "\xb6\x79", [0x2be1] = "\xb6\x7a", [0x2be2] = "\xb6\x7b",
  [0x2be3] = "\xef\xf7", [0x2be4] = "\xb3\xd3", [0x2be5] = "\xb6\x7c",
  [0x2be6] = "\xc7\xd8", [0x2be7] = "\xd1\xed", [0x2be8] = "\xb6\x7d",
  [0x2be9] = "\xd6\xc8", [0x2bea] = "\xb6\x7e", [0x2beb] = "\xef\xf8",
  [0x2bec] = "\xb6\x80", [0x2bed] = "\xef\xf6", [0x2bee] = "\xb6\x81",
  [0x2bef] = "\xbb\xfd", [0x2bf0] = "\xb3\xc6", [0x2bf1] = "\xb6\x82",
  [0x2bf2] = "\xb6\x83", [0x2bf3] = "\xb6\x84", [0x2bf4] = "\xb6\x85",
  [0x2bf5] = "\xb6\x86", [0x2bf6] = "\xb6\x87", [0x2bf7] = "\xb6\x88",
  [0x2bf8] = "\xbd\xd5", [0x2bf9] = "\xb6\x89", [0x2bfa] = "\xb6\x8a",
  [0x2bfb] = "\xd2\xc6", [0x2bfc] = "\xb6\x8b", [0x2bfd] = "\xbb\xe0",
  [0x2bfe] = "\xb6\x8c", [0x2bff] = "\xb6\x8d", [0x2c00] = "\xcf\xa1",
  [0x2c01] = "\xb6\x8e", [0x2c02] = "\xef\xfc", [0x2c03] = "\xef\xfb",
  [0x2c04] = "\xb6\x8f", [0x2c05] = "\xb6\x90", [0x2c06] = "\xef\xf9",
  [0x2c07] = "\xb6\x91", [0x2c08] = "\xb6\x92", [0x2c09] = "\xb6\x93",
  [0x2c0a] = "\xb6\x94", [0x2c0b] = "\xb3\xcc", [0x2c0c] = "\xb6\x95",
  [0x2c0d] = "\xc9\xd4", [0x2c0e] = "\xcb\xb0", [0x2c0f] = "\xb6\x96",
  [0x2c10] = "\xb6\x97", [0x2c11] = "\xb6\x98", [0x2c12] = "\xb6\x99",
  [0x2c13] = "\xb6\x9a", [0x2c14] = "\xef\xfe", [0x2c15] = "\xb6\x9b",
  [0x2c16] = "\xb6\x9c", [0x2c17] = "\xb0\xde", [0x2c18] = "\xb6\x9d",
  [0x2c19] = "\xb6\x9e", [0x2c1a] = "\xd6\xc9", [0x2c1b] = "\xb6\x9f",
  [0x2c1c] = "\xb6\xa0", [0x2c1d] = "\xb7\x40", [0x2c1e] = "\xef\xfd",
  [0x2c1f] = "\xb7\x41", [0x2c20] = "\xb3\xed", [0x2c21] = "\xb7\x42",
  [0x2c22] = "\xb7\x43", [0x2c23] = "\xf6\xd5", [0x2c24] = "\xb7\x44",
  [0x2c25] = "\xb7\x45", [0x2c26] = "\xb7\x46", [0x2c27] = "\xb7\x47",
  [0x2c28] = "\xb7\x48", [0x2c29] = "\xb7\x49", [0x2c2a] = "\xb7\x4a",
  [0x2c2b] = "\xb7\x4b", [0x2c2c] = "\xb7\x4c", [0x2c2d] = "\xb7\x4d",
  [0x2c2e] = "\xb7\x4e", [0x2c2f] = "\xb7\x4f", [0x2c30] = "\xb7\x50",
  [0x2c31] = "\xb7\x51", [0x2c32] = "\xb7\x52", [0x2c33] = "\xce\xc8",
  [0x2c34] = "\xb7\x53", [0x2c35] = "\xb7\x54", [0x2c36] = "\xb7\x55",
  [0x2c37] = "\xf0\xa2", [0x2c38] = "\xb7\x56", [0x2c39] = "\xf0\xa1",
  [0x2c3a] = "\xb7\x57", [0x2c3b] = "\xb5\xbe", [0x2c3c] = "\xbc\xda",
  [0x2c3d] = "\xbb\xfc", [0x2c3e] = "\xb7\x58", [0x2c3f] = "\xb8\xe5",
  [0x2c40] = "\xb7\x59", [0x2c41] = "\xb7\x5a", [0x2c42] = "\xb7\x5b",
  [0x2c43] = "\xb7\x5c", [0x2c44] = "\xb7\x5d", [0x2c45] = "\xb7\x5e",
  [0x2c46] = "\xc4\xc2", [0x2c47] = "\xb7\x5f", [0x2c48] = "\xb7\x60",
  [0x2c49] = "\xb7\x61", [0x2c4a] = "\xb7\x62", [0x2c4b] = "\xb7\x63",
  [0x2c4c] = "\xb7\x64", [0x2c4d] = "\xb7\x65", [0x2c4e] = "\xb7\x66",
  [0x2c4f] = "\xb7\x67", [0x2c50] = "\xb7\x68", [0x2c51] = "\xf0\xa3",
  [0x2c52] = "\xb7\x69", [0x2c53] = "\xb7\x6a", [0x2c54] = "\xb7\x6b",
  [0x2c55] = "\xb7\x6c", [0x2c56] = "\xb7\x6d", [0x2c57] = "\xcb\xeb",
  [0x2c58] = "\xb7\x6e", [0x2c59] = "\xb7\x6f", [0x2c5a] = "\xb7\x70",
  [0x2c5b] = "\xb7\x71", [0x2c5c] = "\xb7\x72", [0x2c5d] = "\xb7\x73",
  [0x2c5e] = "\xb7\x74", [0x2c5f] = "\xb7\x75", [0x2c60] = "\xb7\x76",
  [0x2c61] = "\xb7\x77", [0x2c62] = "\xb7\x78", [0x2c63] = "\xb7\x79",
  [0x2c64] = "\xb7\x7a", [0x2c65] = "\xb7\x7b", [0x2c66] = "\xb7\x7c",
  [0x2c67] = "\xb7\x7d", [0x2c68] = "\xb7\x7e", [0x2c69] = "\xb7\x80",
  [0x2c6a] = "\xb7\x81", [0x2c6b] = "\xb7\x82", [0x2c6c] = "\xb7\x83",
  [0x2c6d] = "\xb7\x84", [0x2c6e] = "\xb7\x85", [0x2c6f] = "\xb7\x86",
  [0x2c70] = "\xf0\xa6", [0x2c71] = "\xb7\x87", [0x2c72] = "\xb7\x88",
  [0x2c73] = "\xb7\x89", [0x2c74] = "\xd1\xa8", [0x2c75] = "\xb7\x8a",
  [0x2c76] = "\xbe\xbf", [0x2c77] = "\xc7\xee", [0x2c78] = "\xf1\xb6",
  [0x2c79] = "\xf1\xb7", [0x2c7a] = "\xbf\xd5", [0x2c7b] = "\xb7\x8b",
  [0x2c7c] = "\xb7\x8c", [0x2c7d] = "\xb7\x8d", [0x2c7e] = "\xb7\x8e",
  [0x2c7f] = "\xb4\xa9", [0x2c80] = "\xf1\xb8", [0x2c81] = "\xcd\xbb",
  [0x2c82] = "\xb7\x8f", [0x2c83] = "\xc7\xd4", [0x2c84] = "\xd5\xad",
  [0x2c85] = "\xb7\x90", [0x2c86] = "\xf1\xb9", [0x2c87] = "\xb7\x91",
  [0x2c88] = "\xf1\xba", [0x2c89] = "\xb7\x92", [0x2c8a] = "\xb7\x93",
  [0x2c8b] = "\xb7\x94", [0x2c8c] = "\xb7\x95", [0x2c8d] = "\xc7\xcf",
  [0x2c8e] = "\xb7\x96", [0x2c8f] = "\xb7\x97", [0x2c90] = "\xb7\x98",
  [0x2c91] = "\xd2\xa4", [0x2c92] = "\xd6\xcf", [0x2c93] = "\xb7\x99",
  [0x2c94] = "\xb7\x9a", [0x2c95] = "\xf1\xbb", [0x2c96] = "\xbd\xd1",
  [0x2c97] = "\xb4\xb0", [0x2c98] = "\xbe\xbd", [0x2c99] = "\xb7\x9b",
  [0x2c9a] = "\xb7\x9c", [0x2c9b] = "\xb7\x9d", [0x2c9c] = "\xb4\xdc",
  [0x2c9d] = "\xce\xd1", [0x2c9e] = "\xb7\x9e", [0x2c9f] = "\xbf\xdf",
  [0x2ca0] = "\xf1\xbd", [0x2ca1] = "\xb7\x9f", [0x2ca2] = "\xb7\xa0",
  [0x2ca3] = "\xb8\x40", [0x2ca4] = "\xb8\x41", [0x2ca5] = "\xbf\xfa",
  [0x2ca6] = "\xf1\xbc", [0x2ca7] = "\xb8\x42", [0x2ca8] = "\xf1\xbf",
  [0x2ca9] = "\xb8\x43", [0x2caa] = "\xb8\x44", [0x2cab] = "\xb8\x45",
  [0x2cac] = "\xf1\xbe", [0x2cad] = "\xf1\xc0", [0x2cae] = "\xb8\x46",
  [0x2caf] = "\xb8\x47", [0x2cb0] = "\xb8\x48", [0x2cb1] = "\xb8\x49",
  [0x2cb2] = "\xb8\x4a", [0x2cb3] = "\xf1\xc1", [0x2cb4] = "\xb8\x4b",
  [0x2cb5] = "\xb8\x4c", [0x2cb6] = "\xb8\x4d", [0x2cb7] = "\xb8\x4e",
  [0x2cb8] = "\xb8\x4f", [0x2cb9] = "\xb8\x50", [0x2cba] = "\xb8\x51",
  [0x2cbb] = "\xb8\x52", [0x2cbc] = "\xb8\x53", [0x2cbd] = "\xb8\x54",
  [0x2cbe] = "\xb8\x55", [0x2cbf] = "\xc1\xfe", [0x2cc0] = "\xb8\x56",
  [0x2cc1] = "\xb8\x57", [0x2cc2] = "\xb8\x58", [0x2cc3] = "\xb8\x59",
  [0x2cc4] = "\xb8\x5a", [0x2cc5] = "\xb8\x5b", [0x2cc6] = "\xb8\x5c",
  [0x2cc7] = "\xb8\x5d", [0x2cc8] = "\xb8\x5e", [0x2cc9] = "\xb8\x5f",
  [0x2cca] = "\xb8\x60", [0x2ccb] = "\xc1\xa2", [0x2ccc] = "\xb8\x61",
  [0x2ccd] = "\xb8\x62", [0x2cce] = "\xb8\x63", [0x2ccf] = "\xb8\x64",
  [0x2cd0] = "\xb8\x65", [0x2cd1] = "\xb8\x66", [0x2cd2] = "\xb8\x67",
  [0x2cd3] = "\xb8\x68", [0x2cd4] = "\xb8\x69", [0x2cd5] = "\xb8\x6a",
  [0x2cd6] = "\xca\xfa", [0x2cd7] = "\xb8\x6b", [0x2cd8] = "\xb8\x6c",
  [0x2cd9] = "\xd5\xbe", [0x2cda] = "\xb8\x6d", [0x2cdb] = "\xb8\x6e",
  [0x2cdc] = "\xb8\x6f", [0x2cdd] = "\xb8\x70", [0x2cde] = "\xbe\xba",
  [0x2cdf] = "\xbe\xb9", [0x2ce0] = "\xd5\xc2", [0x2ce1] = "\xb8\x71",
  [0x2ce2] = "\xb8\x72", [0x2ce3] = "\xbf\xa2", [0x2ce4] = "\xb8\x73",
  [0x2ce5] = "\xcd\xaf", [0x2ce6] = "\xf1\xb5", [0x2ce7] = "\xb8\x74",
  [0x2ce8] = "\xb8\x75", [0x2ce9] = "\xb8\x76", [0x2cea] = "\xb8\x77",
  [0x2ceb] = "\xb8\x78", [0x2cec] = "\xb8\x79", [0x2ced] = "\xbd\xdf",
  [0x2cee] = "\xb8\x7a", [0x2cef] = "\xb6\xcb", [0x2cf0] = "\xb8\x7b",
  [0x2cf1] = "\xb8\x7c", [0x2cf2] = "\xb8\x7d", [0x2cf3] = "\xb8\x7e",
  [0x2cf4] = "\xb8\x80", [0x2cf5] = "\xb8\x81", [0x2cf6] = "\xb8\x82",
  [0x2cf7] = "\xb8\x83", [0x2cf8] = "\xb8\x84", [0x2cf9] = "\xd6\xf1",
  [0x2cfa] = "\xf3\xc3", [0x2cfb] = "\xb8\x85", [0x2cfc] = "\xb8\x86",
  [0x2cfd] = "\xf3\xc4", [0x2cfe] = "\xb8\x87", [0x2cff] = "\xb8\xcd",
  [0x2d00] = "\xb8\x88", [0x2d01] = "\xb8\x89", [0x2d02] = "\xb8\x8a",
  [0x2d03] = "\xf3\xc6", [0x2d04] = "\xf3\xc7", [0x2d05] = "\xb8\x8b",
  [0x2d06] = "\xb0\xca", [0x2d07] = "\xb8\x8c", [0x2d08] = "\xf3\xc5",
  [0x2d09] = "\xb8\x8d", [0x2d0a] = "\xf3\xc9", [0x2d0b] = "\xcb\xf1",
  [0x2d0c] = "\xb8\x8e", [0x2d0d] = "\xb8\x8f", [0x2d0e] = "\xb8\x90",
  [0x2d0f] = "\xf3\xcb", [0x2d10] = "\xb8\x91", [0x2d11] = "\xd0\xa6",
  [0x2d12] = "\xb8\x92", [0x2d13] = "\xb8\x93", [0x2d14] = "\xb1\xca",
  [0x2d15] = "\xf3\xc8", [0x2d16] = "\xb8\x94", [0x2d17] = "\xb8\x95",
  [0x2d18] = "\xb8\x96", [0x2d19] = "\xf3\xcf", [0x2d1a] = "\xb8\x97",
  [0x2d1b] = "\xb5\xd1", [0x2d1c] = "\xb8\x98", [0x2d1d] = "\xb8\x99",
  [0x2d1e] = "\xf3\xd7", [0x2d1f] = "\xb8\x9a", [0x2d20] = "\xf3\xd2",
  [0x2d21] = "\xb8\x9b", [0x2d22] = "\xb8\x9c", [0x2d23] = "\xb8\x9d",
  [0x2d24] = "\xf3\xd4", [0x2d25] = "\xf3\xd3", [0x2d26] = "\xb7\xfb",
  [0x2d27] = "\xb8\x9e", [0x2d28] = "\xb1\xbf", [0x2d29] = "\xb8\x9f",
  [0x2d2a] = "\xf3\xce", [0x2d2b] = "\xf3\xca", [0x2d2c] = "\xb5\xda",
  [0x2d2d] = "\xb8\xa0", [0x2d2e] = "\xf3\xd0", [0x2d2f] = "\xb9\x40",
  [0x2d30] = "\xb9\x41", [0x2d31] = "\xf3\xd1", [0x2d32] = "\xb9\x42",
  [0x2d33] = "\xf3\xd5", [0x2d34] = "\xb9\x43", [0x2d35] = "\xb9\x44",
  [0x2d36] = "\xb9\x45", [0x2d37] = "\xb9\x46", [0x2d38] = "\xf3\xcd",
  [0x2d39] = "\xb9\x47", [0x2d3a] = "\xbc\xe3", [0x2d3b] = "\xb9\x48",
  [0x2d3c] = "\xc1\xfd", [0x2d3d] = "\xb9\x49", [0x2d3e] = "\xf3\xd6",
  [0x2d3f] = "\xb9\x4a", [0x2d40] = "\xb9\x4b", [0x2d41] = "\xb9\x4c",
  [0x2d42] = "\xb9\x4d", [0x2d43] = "\xb9\x4e", [0x2d44] = "\xb9\x4f",
  [0x2d45] = "\xf3\xda", [0x2d46] = "\xb9\x50", [0x2d47] = "\xf3\xcc",
  [0x2d48] = "\xb9\x51", [0x2d49] = "\xb5\xc8", [0x2d4a] = "\xb9\x52",
  [0x2d4b] = "\xbd\xee", [0x2d4c] = "\xf3\xdc", [0x2d4d] = "\xb9\x53",
  [0x2d4e] = "\xb9\x54", [0x2d4f] = "\xb7\xa4", [0x2d50] = "\xbf\xf0",
  [0x2d51] = "\xd6\xfe", [0x2d52] = "\xcd\xb2", [0x2d53] = "\xb9\x55",
  [0x2d54] = "\xb4\xf0", [0x2d55] = "\xb9\x56", [0x2d56] = "\xb2\xdf",
  [0x2d57] = "\xb9\x57", [0x2d58] = "\xf3\xd8", [0x2d59] = "\xb9\x58",
  [0x2d5a] = "\xf3\xd9", [0x2d5b] = "\xc9\xb8", [0x2d5c] = "\xb9\x59",
  [0x2d5d] = "\xf3\xdd", [0x2d5e] = "\xb9\x5a", [0x2d5f] = "\xb9\x5b",
  [0x2d60] = "\xf3\xde", [0x2d61] = "\xb9\x5c", [0x2d62] = "\xf3\xe1",
  [0x2d63] = "\xb9\x5d", [0x2d64] = "\xb9\x5e", [0x2d65] = "\xb9\x5f",
  [0x2d66] = "\xb9\x60", [0x2d67] = "\xb9\x61", [0x2d68] = "\xb9\x62",
  [0x2d69] = "\xb9\x63", [0x2d6a] = "\xb9\x64", [0x2d6b] = "\xb9\x65",
  [0x2d6c] = "\xb9\x66", [0x2d6d] = "\xb9\x67", [0x2d6e] = "\xf3\xdf",
  [0x2d6f] = "\xb9\x68", [0x2d70] = "\xb9\x69", [0x2d71] = "\xf3\xe3",
  [0x2d72] = "\xf3\xe2", [0x2d73] = "\xb9\x6a", [0x2d74] = "\xb9\x6b",
  [0x2d75] = "\xf3\xdb", [0x2d76] = "\xb9\x6c", [0x2d77] = "\xbf\xea",
  [0x2d78] = "\xb9\x6d", [0x2d79] = "\xb3\xef", [0x2d7a] = "\xb9\x6e",
  [0x2d7b] = "\xf3\xe0", [0x2d7c] = "\xb9\x6f", [0x2d7d] = "\xb9\x70",
  [0x2d7e] = "\xc7\xa9", [0x2d7f] = "\xb9\x71", [0x2d80] = "\xbc\xf2",
  [0x2d81] = "\xb9\x72", [0x2d82] = "\xb9\x73", [0x2d83] = "\xb9\x74",
  [0x2d84] = "\xb9\x75", [0x2d85] = "\xf3\xeb", [0x2d86] = "\xb9\x76",
  [0x2d87] = "\xb9\x77", [0x2d88] = "\xb9\x78", [0x2d89] = "\xb9\x79",
  [0x2d8a] = "\xb9\x7a", [0x2d8b] = "\xb9\x7b", [0x2d8c] = "\xb9\x7c",
  [0x2d8d] = "\xb9\xbf", [0x2d8e] = "\xb9\x7d", [0x2d8f] = "\xb9\x7e",
  [0x2d90] = "\xf3\xe4", [0x2d91] = "\xb9\x80", [0x2d92] = "\xb9\x81",
  [0x2d93] = "\xb9\x82", [0x2d94] = "\xb2\xad", [0x2d95] = "\xbb\xfe",
  [0x2d96] = "\xb9\x83", [0x2d97] = "\xcb\xe3", [0x2d98] = "\xb9\x84",
  [0x2d99] = "\xb9\x85", [0x2d9a] = "\xb9\x86", [0x2d9b] = "\xb9\x87",
  [0x2d9c] = "\xf3\xed", [0x2d9d] = "\xf3\xe9", [0x2d9e] = "\xb9\x88",
  [0x2d9f] = "\xb9\x89", [0x2da0] = "\xb9\x8a", [0x2da1] = "\xb9\xdc",
  [0x2da2] = "\xf3\xee", [0x2da3] = "\xb9\x8b", [0x2da4] = "\xb9\x8c",
  [0x2da5] = "\xb9\x8d", [0x2da6] = "\xf3\xe5", [0x2da7] = "\xf3\xe6",
  [0x2da8] = "\xf3\xea", [0x2da9] = "\xc2\xe1", [0x2daa] = "\xf3\xec",
  [0x2dab] = "\xf3\xef", [0x2dac] = "\xf3\xe8", [0x2dad] = "\xbc\xfd",
  [0x2dae] = "\xb9\x8e", [0x2daf] = "\xb9\x8f", [0x2db0] = "\xb9\x90",
  [0x2db1] = "\xcf\xe4", [0x2db2] = "\xb9\x91", [0x2db3] = "\xb9\x92",
  [0x2db4] = "\xf3\xf0", [0x2db5] = "\xb9\x93", [0x2db6] = "\xb9\x94",
  [0x2db7] = "\xb9\x95", [0x2db8] = "\xf3\xe7", [0x2db9] = "\xb9\x96",
  [0x2dba] = "\xb9\x97", [0x2dbb] = "\xb9\x98", [0x2dbc] = "\xb9\x99",
  [0x2dbd] = "\xb9\x9a", [0x2dbe] = "\xb9\x9b", [0x2dbf] = "\xb9\x9c",
  [0x2dc0] = "\xb9\x9d", [0x2dc1] = "\xf3\xf2", [0x2dc2] = "\xb9\x9e",
  [0x2dc3] = "\xb9\x9f", [0x2dc4] = "\xb9\xa0", [0x2dc5] = "\xba\x40",
  [0x2dc6] = "\xd7\xad", [0x2dc7] = "\xc6\xaa", [0x2dc8] = "\xba\x41",
  [0x2dc9] = "\xba\x42", [0x2dca] = "\xba\x43", [0x2dcb] = "\xba\x44",
  [0x2dcc] = "\xf3\xf3", [0x2dcd] = "\xba\x45", [0x2dce] = "\xba\x46",
  [0x2dcf] = "\xba\x47", [0x2dd0] = "\xba\x48", [0x2dd1] = "\xf3\xf1",
  [0x2dd2] = "\xba\x49", [0x2dd3] = "\xc2\xa8", [0x2dd4] = "\xba\x4a",
  [0x2dd5] = "\xba\x4b", [0x2dd6] = "\xba\x4c", [0x2dd7] = "\xba\x4d",
  [0x2dd8] = "\xba\x4e", [0x2dd9] = "\xb8\xdd", [0x2dda] = "\xf3\xf5",
  [0x2ddb] = "\xba\x4f", [0x2ddc] = "\xba\x50", [0x2ddd] = "\xf3\xf4",
  [0x2dde] = "\xba\x51", [0x2ddf] = "\xba\x52", [0x2de0] = "\xba\x53",
  [0x2de1] = "\xb4\xdb", [0x2de2] = "\xba\x54", [0x2de3] = "\xba\x55",
  [0x2de4] = "\xba\x56", [0x2de5] = "\xf3\xf6", [0x2de6] = "\xf3\xf7",
  [0x2de7] = "\xba\x57", [0x2de8] = "\xba\x58", [0x2de9] = "\xba\x59",
  [0x2dea] = "\xf3\xf8", [0x2deb] = "\xba\x5a", [0x2dec] = "\xba\x5b",
  [0x2ded] = "\xba\x5c", [0x2dee] = "\xc0\xba", [0x2def] = "\xba\x5d",
  [0x2df0] = "\xba\x5e", [0x2df1] = "\xc0\xe9", [0x2df2] = "\xba\x5f",
  [0x2df3] = "\xba\x60", [0x2df4] = "\xba\x61", [0x2df5] = "\xba\x62",
  [0x2df6] = "\xba\x63", [0x2df7] = "\xc5\xf1", [0x2df8] = "\xba\x64",
  [0x2df9] = "\xba\x65", [0x2dfa] = "\xba\x66", [0x2dfb] = "\xba\x67",
  [0x2dfc] = "\xf3\xfb", [0x2dfd] = "\xba\x68", [0x2dfe] = "\xf3\xfa",
  [0x2dff] = "\xba\x69", [0x2e00] = "\xba\x6a", [0x2e01] = "\xba\x6b",
  [0x2e02] = "\xba\x6c", [0x2e03] = "\xba\x6d", [0x2e04] = "\xba\x6e",
  [0x2e05] = "\xba\x6f", [0x2e06] = "\xba\x70", [0x2e07] = "\xb4\xd8",
  [0x2e08] = "\xba\x71", [0x2e09] = "\xba\x72", [0x2e0a] = "\xba\x73",
  [0x2e0b] = "\xf3\xfe", [0x2e0c] = "\xf3\xf9", [0x2e0d] = "\xba\x74",
  [0x2e0e] = "\xba\x75", [0x2e0f] = "\xf3\xfc", [0x2e10] = "\xba\x76",
  [0x2e11] = "\xba\x77", [0x2e12] = "\xba\x78", [0x2e13] = "\xba\x79",
  [0x2e14] = "\xba\x7a", [0x2e15] = "\xba\x7b", [0x2e16] = "\xf3\xfd",
  [0x2e17] = "\xba\x7c", [0x2e18] = "\xba\x7d", [0x2e19] = "\xba\x7e",
  [0x2e1a] = "\xba\x80", [0x2e1b] = "\xba\x81", [0x2e1c] = "\xba\x82",
  [0x2e1d] = "\xba\x83", [0x2e1e] = "\xba\x84", [0x2e1f] = "\xf4\xa1",
  [0x2e20] = "\xba\x85", [0x2e21] = "\xba\x86", [0x2e22] = "\xba\x87",
  [0x2e23] = "\xba\x88", [0x2e24] = "\xba\x89", [0x2e25] = "\xba\x8a",
  [0x2e26] = "\xf4\xa3", [0x2e27] = "\xbb\xc9", [0x2e28] = "\xba\x8b",
  [0x2e29] = "\xba\x8c", [0x2e2a] = "\xf4\xa2", [0x2e2b] = "\xba\x8d",
  [0x2e2c] = "\xba\x8e", [0x2e2d] = "\xba\x8f", [0x2e2e] = "\xba\x90",
  [0x2e2f] = "\xba\x91", [0x2e30] = "\xba\x92", [0x2e31] = "\xba\x93",
  [0x2e32] = "\xba\x94", [0x2e33] = "\xba\x95", [0x2e34] = "\xba\x96",
  [0x2e35] = "\xba\x97", [0x2e36] = "\xba\x98", [0x2e37] = "\xba\x99",
  [0x2e38] = "\xf4\xa4", [0x2e39] = "\xba\x9a", [0x2e3a] = "\xba\x9b",
  [0x2e3b] = "\xba\x9c", [0x2e3c] = "\xba\x9d", [0x2e3d] = "\xba\x9e",
  [0x2e3e] = "\xba\x9f", [0x2e3f] = "\xb2\xbe", [0x2e40] = "\xf4\xa6",
  [0x2e41] = "\xf4\xa5", [0x2e42] = "\xba\xa0", [0x2e43] = "\xbb\x40",
  [0x2e44] = "\xbb\x41", [0x2e45] = "\xbb\x42", [0x2e46] = "\xbb\x43",
  [0x2e47] = "\xbb\x44", [0x2e48] = "\xbb\x45", [0x2e49] = "\xbb\x46",
  [0x2e4a] = "\xbb\x47", [0x2e4b] = "\xbb\x48", [0x2e4c] = "\xbb\x49",
  [0x2e4d] = "\xbc\xae", [0x2e4e] = "\xbb\x4a", [0x2e4f] = "\xbb\x4b",
  [0x2e50] = "\xbb\x4c", [0x2e51] = "\xbb\x4d", [0x2e52] = "\xbb\x4e",
  [0x2e53] = "\xbb\x4f", [0x2e54] = "\xbb\x50", [0x2e55] = "\xbb\x51",
  [0x2e56] = "\xbb\x52", [0x2e57] = "\xbb\x53", [0x2e58] = "\xbb\x54",
  [0x2e59] = "\xbb\x55", [0x2e5a] = "\xbb\x56", [0x2e5b] = "\xbb\x57",
  [0x2e5c] = "\xbb\x58", [0x2e5d] = "\xbb\x59", [0x2e5e] = "\xbb\x5a",
  [0x2e5f] = "\xbb\x5b", [0x2e60] = "\xbb\x5c", [0x2e61] = "\xbb\x5d",
  [0x2e62] = "\xbb\x5e", [0x2e63] = "\xbb\x5f", [0x2e64] = "\xbb\x60",
  [0x2e65] = "\xbb\x61", [0x2e66] = "\xbb\x62", [0x2e67] = "\xbb\x63",
  [0x2e68] = "\xbb\x64", [0x2e69] = "\xbb\x65", [0x2e6a] = "\xbb\x66",
  [0x2e6b] = "\xbb\x67", [0x2e6c] = "\xbb\x68", [0x2e6d] = "\xbb\x69",
  [0x2e6e] = "\xbb\x6a", [0x2e6f] = "\xbb\x6b", [0x2e70] = "\xbb\x6c",
  [0x2e71] = "\xbb\x6d", [0x2e72] = "\xbb\x6e", [0x2e73] = "\xc3\xd7",
  [0x2e74] = "\xd9\xe1", [0x2e75] = "\xbb\x6f", [0x2e76] = "\xbb\x70",
  [0x2e77] = "\xbb\x71", [0x2e78] = "\xbb\x72", [0x2e79] = "\xbb\x73",
  [0x2e7a] = "\xbb\x74", [0x2e7b] = "\xc0\xe0", [0x2e7c] = "\xf4\xcc",
  [0x2e7d] = "\xd7\xd1", [0x2e7e] = "\xbb\x75", [0x2e7f] = "\xbb\x76",
  [0x2e80] = "\xbb\x77", [0x2e81] = "\xbb\x78", [0x2e82] = "\xbb\x79",
  [0x2e83] = "\xbb\x7a", [0x2e84] = "\xbb\x7b", [0x2e85] = "\xbb\x7c",
  [0x2e86] = "\xbb\x7d", [0x2e87] = "\xbb\x7e", [0x2e88] = "\xbb\x80",
  [0x2e89] = "\xb7\xdb", [0x2e8a] = "\xbb\x81", [0x2e8b] = "\xbb\x82",
  [0x2e8c] = "\xbb\x83", [0x2e8d] = "\xbb\x84", [0x2e8e] = "\xbb\x85",
  [0x2e8f] = "\xbb\x86", [0x2e90] = "\xbb\x87", [0x2e91] = "\xf4\xce",
  [0x2e92] = "\xc1\xa3", [0x2e93] = "\xbb\x88", [0x2e94] = "\xbb\x89",
  [0x2e95] = "\xc6\xc9", [0x2e96] = "\xbb\x8a", [0x2e97] = "\xb4\xd6",
  [0x2e98] = "\xd5\xb3", [0x2e99] = "\xbb\x8b", [0x2e9a] = "\xbb\x8c",
  [0x2e9b] = "\xbb\x8d", [0x2e9c] = "\xf4\xd0", [0x2e9d] = "\xf4\xcf",
  [0x2e9e] = "\xf4\xd1", [0x2e9f] = "\xcb\xda", [0x2ea0] = "\xbb\x8e",
  [0x2ea1] = "\xbb\x8f", [0x2ea2] = "\xf4\xd2", [0x2ea3] = "\xbb\x90",
  [0x2ea4] = "\xd4\xc1", [0x2ea5] = "\xd6\xe0", [0x2ea6] = "\xbb\x91",
  [0x2ea7] = "\xbb\x92", [0x2ea8] = "\xbb\x93", [0x2ea9] = "\xbb\x94",
  [0x2eaa] = "\xb7\xe0", [0x2eab] = "\xbb\x95", [0x2eac] = "\xbb\x96",
  [0x2ead] = "\xbb\x97", [0x2eae] = "\xc1\xb8", [0x2eaf] = "\xbb\x98",
  [0x2eb0] = "\xbb\x99", [0x2eb1] = "\xc1\xbb", [0x2eb2] = "\xf4\xd3",
  [0x2eb3] = "\xbe\xac", [0x2eb4] = "\xbb\x9a", [0x2eb5] = "\xbb\x9b",
  [0x2eb6] = "\xbb\x9c", [0x2eb7] = "\xbb\x9d", [0x2eb8] = "\xbb\x9e",
  [0x2eb9] = "\xb4\xe2", [0x2eba] = "\xbb\x9f", [0x2ebb] = "\xbb\xa0",
  [0x2ebc] = "\xf4\xd4", [0x2ebd] = "\xf4\xd5", [0x2ebe] = "\xbe\xab",
  [0x2ebf] = "\xbc\x40", [0x2ec0] = "\xbc\x41", [0x2ec1] = "\xf4\xd6",
  [0x2ec2] = "\xbc\x42", [0x2ec3] = "\xbc\x43", [0x2ec4] = "\xbc\x44",
  [0x2ec5] = "\xf4\xdb", [0x2ec6] = "\xbc\x45", [0x2ec7] = "\xf4\xd7",
  [0x2ec8] = "\xf4\xda", [0x2ec9] = "\xbc\x46", [0x2eca] = "\xba\xfd",
  [0x2ecb] = "\xbc\x47", [0x2ecc] = "\xf4\xd8", [0x2ecd] = "\xf4\xd9",
  [0x2ece] = "\xbc\x48", [0x2ecf] = "\xbc\x49", [0x2ed0] = "\xbc\x4a",
  [0x2ed1] = "\xbc\x4b", [0x2ed2] = "\xbc\x4c", [0x2ed3] = "\xbc\x4d",
  [0x2ed4] = "\xbc\x4e", [0x2ed5] = "\xb8\xe2", [0x2ed6] = "\xcc\xc7",
  [0x2ed7] = "\xf4\xdc", [0x2ed8] = "\xbc\x4f", [0x2ed9] = "\xb2\xda",
  [0x2eda] = "\xbc\x50", [0x2edb] = "\xbc\x51", [0x2edc] = "\xc3\xd3",
  [0x2edd] = "\xbc\x52", [0x2ede] = "\xbc\x53", [0x2edf] = "\xd4\xe3",
  [0x2ee0] = "\xbf\xb7", [0x2ee1] = "\xbc\x54", [0x2ee2] = "\xbc\x55",
  [0x2ee3] = "\xbc\x56", [0x2ee4] = "\xbc\x57", [0x2ee5] = "\xbc\x58",
  [0x2ee6] = "\xbc\x59", [0x2ee7] = "\xbc\x5a", [0x2ee8] = "\xf4\xdd",
  [0x2ee9] = "\xbc\x5b", [0x2eea] = "\xbc\x5c", [0x2eeb] = "\xbc\x5d",
  [0x2eec] = "\xbc\x5e", [0x2eed] = "\xbc\x5f", [0x2eee] = "\xbc\x60",
  [0x2eef] = "\xc5\xb4", [0x2ef0] = "\xbc\x61", [0x2ef1] = "\xbc\x62",
  [0x2ef2] = "\xbc\x63", [0x2ef3] = "\xbc\x64", [0x2ef4] = "\xbc\x65",
  [0x2ef5] = "\xbc\x66", [0x2ef6] = "\xbc\x67", [0x2ef7] = "\xbc\x68",
  [0x2ef8] = "\xf4\xe9", [0x2ef9] = "\xbc\x69", [0x2efa] = "\xbc\x6a",
  [0x2efb] = "\xcf\xb5", [0x2efc] = "\xbc\x6b", [0x2efd] = "\xbc\x6c",
  [0x2efe] = "\xbc\x6d", [0x2eff] = "\xbc\x6e", [0x2f00] = "\xbc\x6f",
  [0x2f01] = "\xbc\x70", [0x2f02] = "\xbc\x71", [0x2f03] = "\xbc\x72",
  [0x2f04] = "\xbc\x73", [0x2f05] = "\xbc\x74", [0x2f06] = "\xbc\x75",
  [0x2f07] = "\xbc\x76", [0x2f08] = "\xbc\x77", [0x2f09] = "\xbc\x78",
  [0x2f0a] = "\xce\xc9", [0x2f0b] = "\xbc\x79", [0x2f0c] = "\xbc\x7a",
  [0x2f0d] = "\xbc\x7b", [0x2f0e] = "\xbc\x7c", [0x2f0f] = "\xbc\x7d",
  [0x2f10] = "\xbc\x7e", [0x2f11] = "\xbc\x80", [0x2f12] = "\xbc\x81",
  [0x2f13] = "\xbc\x82", [0x2f14] = "\xbc\x83", [0x2f15] = "\xbc\x84",
  [0x2f16] = "\xbc\x85", [0x2f17] = "\xbc\x86", [0x2f18] = "\xbc\x87",
  [0x2f19] = "\xbc\x88", [0x2f1a] = "\xbc\x89", [0x2f1b] = "\xbc\x8a",
  [0x2f1c] = "\xbc\x8b", [0x2f1d] = "\xbc\x8c", [0x2f1e] = "\xbc\x8d",
  [0x2f1f] = "\xbc\x8e", [0x2f20] = "\xcb\xd8", [0x2f21] = "\xbc\x8f",
  [0x2f22] = "\xcb\xf7", [0x2f23] = "\xbc\x90", [0x2f24] = "\xbc\x91",
  [0x2f25] = "\xbc\x92", [0x2f26] = "\xbc\x93", [0x2f27] = "\xbd\xf4",
  [0x2f28] = "\xbc\x94", [0x2f29] = "\xbc\x95", [0x2f2a] = "\xbc\x96",
  [0x2f2b] = "\xd7\xcf", [0x2f2c] = "\xbc\x97", [0x2f2d] = "\xbc\x98",
  [0x2f2e] = "\xbc\x99", [0x2f2f] = "\xc0\xdb", [0x2f30] = "\xbc\x9a",
  [0x2f31] = "\xbc\x9b", [0x2f32] = "\xbc\x9c", [0x2f33] = "\xbc\x9d",
  [0x2f34] = "\xbc\x9e", [0x2f35] = "\xbc\x9f", [0x2f36] = "\xbc\xa0",
  [0x2f37] = "\xbd\x40", [0x2f38] = "\xbd\x41", [0x2f39] = "\xbd\x42",
  [0x2f3a] = "\xbd\x43", [0x2f3b] = "\xbd\x44", [0x2f3c] = "\xbd\x45",
  [0x2f3d] = "\xbd\x46", [0x2f3e] = "\xbd\x47", [0x2f3f] = "\xbd\x48",
  [0x2f40] = "\xbd\x49", [0x2f41] = "\xbd\x4a", [0x2f42] = "\xbd\x4b",
  [0x2f43] = "\xbd\x4c", [0x2f44] = "\xbd\x4d", [0x2f45] = "\xbd\x4e",
  [0x2f46] = "\xbd\x4f", [0x2f47] = "\xbd\x50", [0x2f48] = "\xbd\x51",
  [0x2f49] = "\xbd\x52", [0x2f4a] = "\xbd\x53", [0x2f4b] = "\xbd\x54",
  [0x2f4c] = "\xbd\x55", [0x2f4d] = "\xbd\x56", [0x2f4e] = "\xbd\x57",
  [0x2f4f] = "\xbd\x58", [0x2f50] = "\xbd\x59", [0x2f51] = "\xbd\x5a",
  [0x2f52] = "\xbd\x5b", [0x2f53] = "\xbd\x5c", [0x2f54] = "\xbd\x5d",
  [0x2f55] = "\xbd\x5e", [0x2f56] = "\xbd\x5f", [0x2f57] = "\xbd\x60",
  [0x2f58] = "\xbd\x61", [0x2f59] = "\xbd\x62", [0x2f5a] = "\xbd\x63",
  [0x2f5b] = "\xbd\x64", [0x2f5c] = "\xbd\x65", [0x2f5d] = "\xbd\x66",
  [0x2f5e] = "\xbd\x67", [0x2f5f] = "\xbd\x68", [0x2f60] = "\xbd\x69",
  [0x2f61] = "\xbd\x6a", [0x2f62] = "\xbd\x6b", [0x2f63] = "\xbd\x6c",
  [0x2f64] = "\xbd\x6d", [0x2f65] = "\xbd\x6e", [0x2f66] = "\xbd\x6f",
  [0x2f67] = "\xbd\x70", [0x2f68] = "\xbd\x71", [0x2f69] = "\xbd\x72",
  [0x2f6a] = "\xbd\x73", [0x2f6b] = "\xbd\x74", [0x2f6c] = "\xbd\x75",
  [0x2f6d] = "\xbd\x76", [0x2f6e] = "\xd0\xf5", [0x2f6f] = "\xbd\x77",
  [0x2f70] = "\xbd\x78", [0x2f71] = "\xbd\x79", [0x2f72] = "\xbd\x7a",
  [0x2f73] = "\xbd\x7b", [0x2f74] = "\xbd\x7c", [0x2f75] = "\xbd\x7d",
  [0x2f76] = "\xbd\x7e", [0x2f77] = "\xf4\xea", [0x2f78] = "\xbd\x80",
  [0x2f79] = "\xbd\x81", [0x2f7a] = "\xbd\x82", [0x2f7b] = "\xbd\x83",
  [0x2f7c] = "\xbd\x84", [0x2f7d] = "\xbd\x85", [0x2f7e] = "\xbd\x86",
  [0x2f7f] = "\xbd\x87", [0x2f80] = "\xbd\x88", [0x2f81] = "\xbd\x89",
  [0x2f82] = "\xbd\x8a", [0x2f83] = "\xbd\x8b", [0x2f84] = "\xbd\x8c",
  [0x2f85] = "\xbd\x8d", [0x2f86] = "\xbd\x8e", [0x2f87] = "\xbd\x8f",
  [0x2f88] = "\xbd\x90", [0x2f89] = "\xbd\x91", [0x2f8a] = "\xbd\x92",
  [0x2f8b] = "\xbd\x93", [0x2f8c] = "\xbd\x94", [0x2f8d] = "\xbd\x95",
  [0x2f8e] = "\xbd\x96", [0x2f8f] = "\xbd\x97", [0x2f90] = "\xbd\x98",
  [0x2f91] = "\xbd\x99", [0x2f92] = "\xbd\x9a", [0x2f93] = "\xbd\x9b",
  [0x2f94] = "\xbd\x9c", [0x2f95] = "\xbd\x9d", [0x2f96] = "\xbd\x9e",
  [0x2f97] = "\xbd\x9f", [0x2f98] = "\xbd\xa0", [0x2f99] = "\xbe\x40",
  [0x2f9a] = "\xbe\x41", [0x2f9b] = "\xbe\x42", [0x2f9c] = "\xbe\x43",
  [0x2f9d] = "\xbe\x44", [0x2f9e] = "\xbe\x45", [0x2f9f] = "\xbe\x46",
  [0x2fa0] = "\xbe\x47", [0x2fa1] = "\xbe\x48", [0x2fa2] = "\xbe\x49",
  [0x2fa3] = "\xbe\x4a", [0x2fa4] = "\xbe\x4b", [0x2fa5] = "\xbe\x4c",
  [0x2fa6] = "\xf4\xeb", [0x2fa7] = "\xbe\x4d", [0x2fa8] = "\xbe\x4e",
  [0x2fa9] = "\xbe\x4f", [0x2faa] = "\xbe\x50", [0x2fab] = "\xbe\x51",
  [0x2fac] = "\xbe\x52", [0x2fad] = "\xbe\x53", [0x2fae] = "\xf4\xec",
  [0x2faf] = "\xbe\x54", [0x2fb0] = "\xbe\x55", [0x2fb1] = "\xbe\x56",
  [0x2fb2] = "\xbe\x57", [0x2fb3] = "\xbe\x58", [0x2fb4] = "\xbe\x59",
  [0x2fb5] = "\xbe\x5a", [0x2fb6] = "\xbe\x5b", [0x2fb7] = "\xbe\x5c",
  [0x2fb8] = "\xbe\x5d", [0x2fb9] = "\xbe\x5e", [0x2fba] = "\xbe\x5f",
  [0x2fbb] = "\xbe\x60", [0x2fbc] = "\xbe\x61", [0x2fbd] = "\xbe\x62",
  [0x2fbe] = "\xbe\x63", [0x2fbf] = "\xbe\x64", [0x2fc0] = "\xbe\x65",
  [0x2fc1] = "\xbe\x66", [0x2fc2] = "\xbe\x67", [0x2fc3] = "\xbe\x68",
  [0x2fc4] = "\xbe\x69", [0x2fc5] = "\xbe\x6a", [0x2fc6] = "\xbe\x6b",
  [0x2fc7] = "\xbe\x6c", [0x2fc8] = "\xbe\x6d", [0x2fc9] = "\xbe\x6e",
  [0x2fca] = "\xbe\x6f", [0x2fcb] = "\xbe\x70", [0x2fcc] = "\xbe\x71",
  [0x2fcd] = "\xbe\x72", [0x2fce] = "\xbe\x73", [0x2fcf] = "\xbe\x74",
  [0x2fd0] = "\xbe\x75", [0x2fd1] = "\xbe\x76", [0x2fd2] = "\xbe\x77",
  [0x2fd3] = "\xbe\x78", [0x2fd4] = "\xbe\x79", [0x2fd5] = "\xbe\x7a",
  [0x2fd6] = "\xbe\x7b", [0x2fd7] = "\xbe\x7c", [0x2fd8] = "\xbe\x7d",
  [0x2fd9] = "\xbe\x7e", [0x2fda] = "\xbe\x80", [0x2fdb] = "\xbe\x81",
  [0x2fdc] = "\xbe\x82", [0x2fdd] = "\xbe\x83", [0x2fde] = "\xbe\x84",
  [0x2fdf] = "\xbe\x85", [0x2fe0] = "\xbe\x86", [0x2fe1] = "\xbe\x87",
  [0x2fe2] = "\xbe\x88", [0x2fe3] = "\xbe\x89", [0x2fe4] = "\xbe\x8a",
  [0x2fe5] = "\xbe\x8b", [0x2fe6] = "\xbe\x8c", [0x2fe7] = "\xbe\x8d",
  [0x2fe8] = "\xbe\x8e", [0x2fe9] = "\xbe\x8f", [0x2fea] = "\xbe\x90",
  [0x2feb] = "\xbe\x91", [0x2fec] = "\xbe\x92", [0x2fed] = "\xbe\x93",
  [0x2fee] = "\xbe\x94", [0x2fef] = "\xbe\x95", [0x2ff0] = "\xbe\x96",
  [0x2ff1] = "\xbe\x97", [0x2ff2] = "\xbe\x98", [0x2ff3] = "\xbe\x99",
  [0x2ff4] = "\xbe\x9a", [0x2ff5] = "\xbe\x9b", [0x2ff6] = "\xbe\x9c",
  [0x2ff7] = "\xbe\x9d", [0x2ff8] = "\xbe\x9e", [0x2ff9] = "\xbe\x9f",
  [0x2ffa] = "\xbe\xa0", [0x2ffb] = "\xbf\x40", [0x2ffc] = "\xbf\x41",
  [0x2ffd] = "\xbf\x42", [0x2ffe] = "\xbf\x43", [0x2fff] = "\xbf\x44",
  [0x3000] = "\xbf\x45", [0x3001] = "\xbf\x46", [0x3002] = "\xbf\x47",
  [0x3003] = "\xbf\x48", [0x3004] = "\xbf\x49", [0x3005] = "\xbf\x4a",
  [0x3006] = "\xbf\x4b", [0x3007] = "\xbf\x4c", [0x3008] = "\xbf\x4d",
  [0x3009] = "\xbf\x4e", [0x300a] = "\xbf\x4f", [0x300b] = "\xbf\x50",
  [0x300c] = "\xbf\x51", [0x300d] = "\xbf\x52", [0x300e] = "\xbf\x53",
  [0x300f] = "\xbf\x54", [0x3010] = "\xbf\x55", [0x3011] = "\xbf\x56",
  [0x3012] = "\xbf\x57", [0x3013] = "\xbf\x58", [0x3014] = "\xbf\x59",
  [0x3015] = "\xbf\x5a", [0x3016] = "\xbf\x5b", [0x3017] = "\xbf\x5c",
  [0x3018] = "\xbf\x5d", [0x3019] = "\xbf\x5e", [0x301a] = "\xbf\x5f",
  [0x301b] = "\xbf\x60", [0x301c] = "\xbf\x61", [0x301d] = "\xbf\x62",
  [0x301e] = "\xbf\x63", [0x301f] = "\xbf\x64", [0x3020] = "\xbf\x65",
  [0x3021] = "\xbf\x66", [0x3022] = "\xbf\x67", [0x3023] = "\xbf\x68",
  [0x3024] = "\xbf\x69", [0x3025] = "\xbf\x6a", [0x3026] = "\xbf\x6b",
  [0x3027] = "\xbf\x6c", [0x3028] = "\xbf\x6d", [0x3029] = "\xbf\x6e",
  [0x302a] = "\xbf\x6f", [0x302b] = "\xbf\x70", [0x302c] = "\xbf\x71",
  [0x302d] = "\xbf\x72", [0x302e] = "\xbf\x73", [0x302f] = "\xbf\x74",
  [0x3030] = "\xbf\x75", [0x3031] = "\xbf\x76", [0x3032] = "\xbf\x77",
  [0x3033] = "\xbf\x78", [0x3034] = "\xbf\x79", [0x3035] = "\xbf\x7a",
  [0x3036] = "\xbf\x7b", [0x3037] = "\xbf\x7c", [0x3038] = "\xbf\x7d",
  [0x3039] = "\xbf\x7e", [0x303a] = "\xbf\x80", [0x303b] = "\xf7\xe3",
  [0x303c] = "\xbf\x81", [0x303d] = "\xbf\x82", [0x303e] = "\xbf\x83",
  [0x303f] = "\xbf\x84", [0x3040] = "\xbf\x85", [0x3041] = "\xb7\xb1",
  [0x3042] = "\xbf\x86", [0x3043] = "\xbf\x87", [0x3044] = "\xbf\x88",
  [0x3045] = "\xbf\x89", [0x3046] = "\xbf\x8a", [0x3047] = "\xf4\xed",
  [0x3048] = "\xbf\x8b", [0x3049] = "\xbf\x8c", [0x304a] = "\xbf\x8d",
  [0x304b] = "\xbf\x8e", [0x304c] = "\xbf\x8f", [0x304d] = "\xbf\x90",
  [0x304e] = "\xbf\x91", [0x304f] = "\xbf\x92", [0x3050] = "\xbf\x93",
  [0x3051] = "\xbf\x94", [0x3052] = "\xbf\x95", [0x3053] = "\xbf\x96",
  [0x3054] = "\xbf\x97", [0x3055] = "\xbf\x98", [0x3056] = "\xbf\x99",
  [0x3057] = "\xbf\x9a", [0x3058] = "\xbf\x9b", [0x3059] = "\xbf\x9c",
  [0x305a] = "\xbf\x9d", [0x305b] = "\xbf\x9e", [0x305c] = "\xbf\x9f",
  [0x305d] = "\xbf\xa0", [0x305e] = "\xc0\x40", [0x305f] = "\xc0\x41",
  [0x3060] = "\xc0\x42", [0x3061] = "\xc0\x43", [0x3062] = "\xc0\x44",
  [0x3063] = "\xc0\x45", [0x3064] = "\xc0\x46", [0x3065] = "\xc0\x47",
  [0x3066] = "\xc0\x48", [0x3067] = "\xc0\x49", [0x3068] = "\xc0\x4a",
  [0x3069] = "\xc0\x4b", [0x306a] = "\xc0\x4c", [0x306b] = "\xc0\x4d",
  [0x306c] = "\xc0\x4e", [0x306d] = "\xc0\x4f", [0x306e] = "\xc0\x50",
  [0x306f] = "\xc0\x51", [0x3070] = "\xc0\x52", [0x3071] = "\xc0\x53",
  [0x3072] = "\xc0\x54", [0x3073] = "\xc0\x55", [0x3074] = "\xc0\x56",
  [0x3075] = "\xc0\x57", [0x3076] = "\xc0\x58", [0x3077] = "\xc0\x59",
  [0x3078] = "\xc0\x5a", [0x3079] = "\xc0\x5b", [0x307a] = "\xc0\x5c",
  [0x307b] = "\xc0\x5d", [0x307c] = "\xc0\x5e", [0x307d] = "\xc0\x5f",
  [0x307e] = "\xc0\x60", [0x307f] = "\xc0\x61", [0x3080] = "\xc0\x62",
  [0x3081] = "\xc0\x63", [0x3082] = "\xd7\xeb", [0x3083] = "\xc0\x64",
  [0x3084] = "\xc0\x65", [0x3085] = "\xc0\x66", [0x3086] = "\xc0\x67",
  [0x3087] = "\xc0\x68", [0x3088] = "\xc0\x69", [0x3089] = "\xc0\x6a",
  [0x308a] = "\xc0\x6b", [0x308b] = "\xc0\x6c", [0x308c] = "\xc0\x6d",
  [0x308d] = "\xc0\x6e", [0x308e] = "\xc0\x6f", [0x308f] = "\xc0\x70",
  [0x3090] = "\xc0\x71", [0x3091] = "\xc0\x72", [0x3092] = "\xc0\x73",
  [0x3093] = "\xc0\x74", [0x3094] = "\xc0\x75", [0x3095] = "\xc0\x76",
  [0x3096] = "\xc0\x77", [0x3097] = "\xc0\x78", [0x3098] = "\xc0\x79",
  [0x3099] = "\xc0\x7a", [0x309a] = "\xc0\x7b", [0x309b] = "\xf4\xee",
  [0x309c] = "\xc0\x7c", [0x309d] = "\xc0\x7d", [0x309e] = "\xc0\x7e",
  [0x309f] = "\xe6\xf9", [0x30a0] = "\xbe\xc0", [0x30a1] = "\xe6\xfa",
  [0x30a2] = "\xba\xec", [0x30a3] = "\xe6\xfb", [0x30a4] = "\xcf\xcb",
  [0x30a5] = "\xe6\xfc", [0x30a6] = "\xd4\xbc", [0x30a7] = "\xbc\xb6",
  [0x30a8] = "\xe6\xfd", [0x30a9] = "\xe6\xfe", [0x30aa] = "\xbc\xcd",
  [0x30ab] = "\xc8\xd2", [0x30ac] = "\xce\xb3", [0x30ad] = "\xe7\xa1",
  [0x30ae] = "\xc0\x80", [0x30af] = "\xb4\xbf", [0x30b0] = "\xe7\xa2",
  [0x30b1] = "\xc9\xb4", [0x30b2] = "\xb8\xd9", [0x30b3] = "\xc4\xc9",
  [0x30b4] = "\xc0\x81", [0x30b5] = "\xd7\xdd", [0x30b6] = "\xc2\xda",
  [0x30b7] = "\xb7\xd7", [0x30b8] = "\xd6\xbd", [0x30b9] = "\xce\xc6",
  [0x30ba] = "\xb7\xc4", [0x30bb] = "\xc0\x82", [0x30bc] = "\xc0\x83",
  [0x30bd] = "\xc5\xa6", [0x30be] = "\xe7\xa3", [0x30bf] = "\xcf\xdf",
  [0x30c0] = "\xe7\xa4", [0x30c1] = "\xe7\xa5", [0x30c2] = "\xe7\xa6",
  [0x30c3] = "\xc1\xb7", [0x30c4] = "\xd7\xe9", [0x30c5] = "\xc9\xf0",
  [0x30c6] = "\xcf\xb8", [0x30c7] = "\xd6\xaf", [0x30c8] = "\xd6\xd5",
  [0x30c9] = "\xe7\xa7", [0x30ca] = "\xb0\xed", [0x30cb] = "\xe7\xa8",
  [0x30cc] = "\xe7\xa9", [0x30cd] = "\xc9\xdc", [0x30ce] = "\xd2\xef",
  [0x30cf] = "\xbe\xad", [0x30d0] = "\xe7\xaa", [0x30d1] = "\xb0\xf3",
  [0x30d2] = "\xc8\xde", [0x30d3] = "\xbd\xe1", [0x30d4] = "\xe7\xab",
  [0x30d5] = "\xc8\xc6", [0x30d6] = "\xc0\x84", [0x30d7] = "\xe7\xac",
  [0x30d8] = "\xbb\xe6", [0x30d9] = "\xb8\xf8", [0x30da] = "\xd1\xa4",
  [0x30db] = "\xe7\xad", [0x30dc] = "\xc2\xe7", [0x30dd] = "\xbe\xf8",
  [0x30de] = "\xbd\xca", [0x30df] = "\xcd\xb3", [0x30e0] = "\xe7\xae",
  [0x30e1] = "\xe7\xaf", [0x30e2] = "\xbe\xee", [0x30e3] = "\xd0\xe5",
  [0x30e4] = "\xc0\x85", [0x30e5] = "\xcb\xe7", [0x30e6] = "\xcc\xd0",
  [0x30e7] = "\xbc\xcc", [0x30e8] = "\xe7\xb0", [0x30e9] = "\xbc\xa8",
  [0x30ea] = "\xd0\xf7", [0x30eb] = "\xe7\xb1", [0x30ec] = "\xc0\x86",
  [0x30ed] = "\xd0\xf8", [0x30ee] = "\xe7\xb2", [0x30ef] = "\xe7\xb3",
  [0x30f0] = "\xb4\xc2", [0x30f1] = "\xe7\xb4", [0x30f2] = "\xe7\xb5",
  [0x30f3] = "\xc9\xfe", [0x30f4] = "\xce\xac", [0x30f5] = "\xc3\xe0",
  [0x30f6] = "\xe7\xb7", [0x30f7] = "\xb1\xc1", [0x30f8] = "\xb3\xf1",
  [0x30f9] = "\xc0\x87", [0x30fa] = "\xe7\xb8", [0x30fb] = "\xe7\xb9",
  [0x30fc] = "\xd7\xdb", [0x30fd] = "\xd5\xc0", [0x30fe] = "\xe7\xba",
  [0x30ff] = "\xc2\xcc", [0x3100] = "\xd7\xba", [0x3101] = "\xe7\xbb",
  [0x3102] = "\xe7\xbc", [0x3103] = "\xe7\xbd", [0x3104] = "\xbc\xea",
  [0x3105] = "\xc3\xe5", [0x3106] = "\xc0\xc2", [0x3107] = "\xe7\xbe",
  [0x3108] = "\xe7\xbf", [0x3109] = "\xbc\xa9", [0x310a] = "\xc0\x88",
  [0x310b] = "\xe7\xc0", [0x310c] = "\xe7\xc1", [0x310d] = "\xe7\xb6",
  [0x310e] = "\xb6\xd0", [0x310f] = "\xe7\xc2", [0x3110] = "\xc0\x89",
  [0x3111] = "\xe7\xc3", [0x3112] = "\xe7\xc4", [0x3113] = "\xbb\xba",
  [0x3114] = "\xb5\xde", [0x3115] = "\xc2\xc6", [0x3116] = "\xb1\xe0",
  [0x3117] = "\xe7\xc5", [0x3118] = "\xd4\xb5", [0x3119] = "\xe7\xc6",
  [0x311a] = "\xb8\xbf", [0x311b] = "\xe7\xc8", [0x311c] = "\xe7\xc7",
  [0x311d] = "\xb7\xec", [0x311e] = "\xc0\x8a", [0x311f] = "\xe7\xc9",
  [0x3120] = "\xb2\xf8", [0x3121] = "\xe7\xca", [0x3122] = "\xe7\xcb",
  [0x3123] = "\xe7\xcc", [0x3124] = "\xe7\xcd", [0x3125] = "\xe7\xce",
  [0x3126] = "\xe7\xcf", [0x3127] = "\xe7\xd0", [0x3128] = "\xd3\xa7",
  [0x3129] = "\xcb\xf5", [0x312a] = "\xe7\xd1", [0x312b] = "\xe7\xd2",
  [0x312c] = "\xe7\xd3", [0x312d] = "\xe7\xd4", [0x312e] = "\xc9\xc9",
  [0x312f] = "\xe7\xd5", [0x3130] = "\xe7\xd6", [0x3131] = "\xe7\xd7",
  [0x3132] = "\xe7\xd8", [0x3133] = "\xe7\xd9", [0x3134] = "\xbd\xc9",
  [0x3135] = "\xe7\xda", [0x3136] = "\xf3\xbe", [0x3137] = "\xc0\x8b",
  [0x3138] = "\xb8\xd7", [0x3139] = "\xc0\x8c", [0x313a] = "\xc8\xb1",
  [0x313b] = "\xc0\x8d", [0x313c] = "\xc0\x8e", [0x313d] = "\xc0\x8f",
  [0x313e] = "\xc0\x90", [0x313f] = "\xc0\x91", [0x3140] = "\xc0\x92",
  [0x3141] = "\xc0\x93", [0x3142] = "\xf3\xbf", [0x3143] = "\xc0\x94",
  [0x3144] = "\xf3\xc0", [0x3145] = "\xf3\xc1", [0x3146] = "\xc0\x95",
  [0x3147] = "\xc0\x96", [0x3148] = "\xc0\x97", [0x3149] = "\xc0\x98",
  [0x314a] = "\xc0\x99", [0x314b] = "\xc0\x9a", [0x314c] = "\xc0\x9b",
  [0x314d] = "\xc0\x9c", [0x314e] = "\xc0\x9d", [0x314f] = "\xc0\x9e",
  [0x3150] = "\xb9\xde", [0x3151] = "\xcd\xf8", [0x3152] = "\xc0\x9f",
  [0x3153] = "\xc0\xa0", [0x3154] = "\xd8\xe8", [0x3155] = "\xba\xb1",
  [0x3156] = "\xc1\x40", [0x3157] = "\xc2\xde", [0x3158] = "\xee\xb7",
  [0x3159] = "\xc1\x41", [0x315a] = "\xb7\xa3", [0x315b] = "\xc1\x42",
  [0x315c] = "\xc1\x43", [0x315d] = "\xc1\x44", [0x315e] = "\xc1\x45",
  [0x315f] = "\xee\xb9", [0x3160] = "\xc1\x46", [0x3161] = "\xee\xb8",
  [0x3162] = "\xb0\xd5", [0x3163] = "\xc1\x47", [0x3164] = "\xc1\x48",
  [0x3165] = "\xc1\x49", [0x3166] = "\xc1\x4a", [0x3167] = "\xc1\x4b",
  [0x3168] = "\xee\xbb", [0x3169] = "\xd5\xd6", [0x316a] = "\xd7\xef",
  [0x316b] = "\xc1\x4c", [0x316c] = "\xc1\x4d", [0x316d] = "\xc1\x4e",
  [0x316e] = "\xd6\xc3", [0x316f] = "\xc1\x4f", [0x3170] = "\xc1\x50",
  [0x3171] = "\xee\xbd", [0x3172] = "\xca\xf0", [0x3173] = "\xc1\x51",
  [0x3174] = "\xee\xbc", [0x3175] = "\xc1\x52", [0x3176] = "\xc1\x53",
  [0x3177] = "\xc1\x54", [0x3178] = "\xc1\x55", [0x3179] = "\xee\xbe",
  [0x317a] = "\xc1\x56", [0x317b] = "\xc1\x57", [0x317c] = "\xc1\x58",
  [0x317d] = "\xc1\x59", [0x317e] = "\xee\xc0", [0x317f] = "\xc1\x5a",
  [0x3180] = "\xc1\x5b", [0x3181] = "\xee\xbf", [0x3182] = "\xc1\x5c",
  [0x3183] = "\xc1\x5d", [0x3184] = "\xc1\x5e", [0x3185] = "\xc1\x5f",
  [0x3186] = "\xc1\x60", [0x3187] = "\xc1\x61", [0x3188] = "\xc1\x62",
  [0x3189] = "\xc1\x63", [0x318a] = "\xd1\xf2", [0x318b] = "\xc1\x64",
  [0x318c] = "\xc7\xbc", [0x318d] = "\xc1\x65", [0x318e] = "\xc3\xc0",
  [0x318f] = "\xc1\x66", [0x3190] = "\xc1\x67", [0x3191] = "\xc1\x68",
  [0x3192] = "\xc1\x69", [0x3193] = "\xc1\x6a", [0x3194] = "\xb8\xe1",
  [0x3195] = "\xc1\x6b", [0x3196] = "\xc1\x6c", [0x3197] = "\xc1\x6d",
  [0x3198] = "\xc1\x6e", [0x3199] = "\xc1\x6f", [0x319a] = "\xc1\xe7",
  [0x319b] = "\xc1\x70", [0x319c] = "\xc1\x71", [0x319d] = "\xf4\xc6",
  [0x319e] = "\xd0\xdf", [0x319f] = "\xf4\xc7", [0x31a0] = "\xc1\x72",
  [0x31a1] = "\xcf\xdb", [0x31a2] = "\xc1\x73", [0x31a3] = "\xc1\x74",
  [0x31a4] = "\xc8\xba", [0x31a5] = "\xc1\x75", [0x31a6] = "\xc1\x76",
  [0x31a7] = "\xf4\xc8", [0x31a8] = "\xc1\x77", [0x31a9] = "\xc1\x78",
  [0x31aa] = "\xc1\x79", [0x31ab] = "\xc1\x7a", [0x31ac] = "\xc1\x7b",
  [0x31ad] = "\xc1\x7c", [0x31ae] = "\xc1\x7d", [0x31af] = "\xf4\xc9",
  [0x31b0] = "\xf4\xca", [0x31b1] = "\xc1\x7e", [0x31b2] = "\xf4\xcb",
  [0x31b3] = "\xc1\x80", [0x31b4] = "\xc1\x81", [0x31b5] = "\xc1\x82",
  [0x31b6] = "\xc1\x83", [0x31b7] = "\xc1\x84", [0x31b8] = "\xd9\xfa",
  [0x31b9] = "\xb8\xfe", [0x31ba] = "\xc1\x85", [0x31bb] = "\xc1\x86",
  [0x31bc] = "\xe5\xf1", [0x31bd] = "\xd3\xf0", [0x31be] = "\xc1\x87",
  [0x31bf] = "\xf4\xe0", [0x31c0] = "\xc1\x88", [0x31c1] = "\xce\xcc",
  [0x31c2] = "\xc1\x89", [0x31c3] = "\xc1\x8a", [0x31c4] = "\xc1\x8b",
  [0x31c5] = "\xb3\xe1", [0x31c6] = "\xc1\x8c", [0x31c7] = "\xc1\x8d",
  [0x31c8] = "\xc1\x8e", [0x31c9] = "\xc1\x8f", [0x31ca] = "\xf1\xb4",
  [0x31cb] = "\xc1\x90", [0x31cc] = "\xd2\xee", [0x31cd] = "\xc1\x91",
  [0x31ce] = "\xf4\xe1", [0x31cf] = "\xc1\x92", [0x31d0] = "\xc1\x93",
  [0x31d1] = "\xc1\x94", [0x31d2] = "\xc1\x95", [0x31d3] = "\xc1\x96",
  [0x31d4] = "\xcf\xe8", [0x31d5] = "\xf4\xe2", [0x31d6] = "\xc1\x97",
  [0x31d7] = "\xc1\x98", [0x31d8] = "\xc7\xcc", [0x31d9] = "\xc1\x99",
  [0x31da] = "\xc1\x9a", [0x31db] = "\xc1\x9b", [0x31dc] = "\xc1\x9c",
  [0x31dd] = "\xc1\x9d", [0x31de] = "\xc1\x9e", [0x31df] = "\xb5\xd4",
  [0x31e0] = "\xb4\xe4", [0x31e1] = "\xf4\xe4", [0x31e2] = "\xc1\x9f",
  [0x31e3] = "\xc1\xa0", [0x31e4] = "\xc2\x40", [0x31e5] = "\xf4\xe3",
  [0x31e6] = "\xf4\xe5", [0x31e7] = "\xc2\x41", [0x31e8] = "\xc2\x42",
  [0x31e9] = "\xf4\xe6", [0x31ea] = "\xc2\x43", [0x31eb] = "\xc2\x44",
  [0x31ec] = "\xc2\x45", [0x31ed] = "\xc2\x46", [0x31ee] = "\xf4\xe7",
  [0x31ef] = "\xc2\x47", [0x31f0] = "\xba\xb2", [0x31f1] = "\xb0\xbf",
  [0x31f2] = "\xc2\x48", [0x31f3] = "\xf4\xe8", [0x31f4] = "\xc2\x49",
  [0x31f5] = "\xc2\x4a", [0x31f6] = "\xc2\x4b", [0x31f7] = "\xc2\x4c",
  [0x31f8] = "\xc2\x4d", [0x31f9] = "\xc2\x4e", [0x31fa] = "\xc2\x4f",
  [0x31fb] = "\xb7\xad", [0x31fc] = "\xd2\xed", [0x31fd] = "\xc2\x50",
  [0x31fe] = "\xc2\x51", [0x31ff] = "\xc2\x52", [0x3200] = "\xd2\xab",
  [0x3201] = "\xc0\xcf", [0x3202] = "\xc2\x53", [0x3203] = "\xbf\xbc",
  [0x3204] = "\xeb\xa3", [0x3205] = "\xd5\xdf", [0x3206] = "\xea\xc8",
  [0x3207] = "\xc2\x54", [0x3208] = "\xc2\x55", [0x3209] = "\xc2\x56",
  [0x320a] = "\xc2\x57", [0x320b] = "\xf1\xf3", [0x320c] = "\xb6\xf8",
  [0x320d] = "\xcb\xa3", [0x320e] = "\xc2\x58", [0x320f] = "\xc2\x59",
  [0x3210] = "\xc4\xcd", [0x3211] = "\xc2\x5a", [0x3212] = "\xf1\xe7",
  [0x3213] = "\xc2\x5b", [0x3214] = "\xf1\xe8", [0x3215] = "\xb8\xfb",
  [0x3216] = "\xf1\xe9", [0x3217] = "\xba\xc4", [0x3218] = "\xd4\xc5",
  [0x3219] = "\xb0\xd2", [0x321a] = "\xc2\x5c", [0x321b] = "\xc2\x5d",
  [0x321c] = "\xf1\xea", [0x321d] = "\xc2\x5e", [0x321e] = "\xc2\x5f",
  [0x321f] = "\xc2\x60", [0x3220] = "\xf1\xeb", [0x3221] = "\xc2\x61",
  [0x3222] = "\xf1\xec", [0x3223] = "\xc2\x62", [0x3224] = "\xc2\x63",
  [0x3225] = "\xf1\xed", [0x3226] = "\xf1\xee", [0x3227] = "\xf1\xef",
  [0x3228] = "\xf1\xf1", [0x3229] = "\xf1\xf0", [0x322a] = "\xc5\xd5",
  [0x322b] = "\xc2\x64", [0x322c] = "\xc2\x65", [0x322d] = "\xc2\x66",
  [0x322e] = "\xc2\x67", [0x322f] = "\xc2\x68", [0x3230] = "\xc2\x69",
  [0x3231] = "\xf1\xf2", [0x3232] = "\xc2\x6a", [0x3233] = "\xb6\xfa",
  [0x3234] = "\xc2\x6b", [0x3235] = "\xf1\xf4", [0x3236] = "\xd2\xae",
  [0x3237] = "\xde\xc7", [0x3238] = "\xcb\xca", [0x3239] = "\xc2\x6c",
  [0x323a] = "\xc2\x6d", [0x323b] = "\xb3\xdc", [0x323c] = "\xc2\x6e",
  [0x323d] = "\xb5\xa2", [0x323e] = "\xc2\x6f", [0x323f] = "\xb9\xa2",
  [0x3240] = "\xc2\x70", [0x3241] = "\xc2\x71", [0x3242] = "\xc4\xf4",
  [0x3243] = "\xf1\xf5", [0x3244] = "\xc2\x72", [0x3245] = "\xc2\x73",
  [0x3246] = "\xf1\xf6", [0x3247] = "\xc2\x74", [0x3248] = "\xc2\x75",
  [0x3249] = "\xc2\x76", [0x324a] = "\xc1\xc4", [0x324b] = "\xc1\xfb",
  [0x324c] = "\xd6\xb0", [0x324d] = "\xf1\xf7", [0x324e] = "\xc2\x77",
  [0x324f] = "\xc2\x78", [0x3250] = "\xc2\x79", [0x3251] = "\xc2\x7a",
  [0x3252] = "\xf1\xf8", [0x3253] = "\xc2\x7b", [0x3254] = "\xc1\xaa",
  [0x3255] = "\xc2\x7c", [0x3256] = "\xc2\x7d", [0x3257] = "\xc2\x7e",
  [0x3258] = "\xc6\xb8", [0x3259] = "\xc2\x80", [0x325a] = "\xbe\xdb",
  [0x325b] = "\xc2\x81", [0x325c] = "\xc2\x82", [0x325d] = "\xc2\x83",
  [0x325e] = "\xc2\x84", [0x325f] = "\xc2\x85", [0x3260] = "\xc2\x86",
  [0x3261] = "\xc2\x87", [0x3262] = "\xc2\x88", [0x3263] = "\xc2\x89",
  [0x3264] = "\xc2\x8a", [0x3265] = "\xc2\x8b", [0x3266] = "\xc2\x8c",
  [0x3267] = "\xc2\x8d", [0x3268] = "\xc2\x8e", [0x3269] = "\xf1\xf9",
  [0x326a] = "\xb4\xcf", [0x326b] = "\xc2\x8f", [0x326c] = "\xc2\x90",
  [0x326d] = "\xc2\x91", [0x326e] = "\xc2\x92", [0x326f] = "\xc2\x93",
  [0x3270] = "\xc2\x94", [0x3271] = "\xf1\xfa", [0x3272] = "\xc2\x95",
  [0x3273] = "\xc2\x96", [0x3274] = "\xc2\x97", [0x3275] = "\xc2\x98",
  [0x3276] = "\xc2\x99", [0x3277] = "\xc2\x9a", [0x3278] = "\xc2\x9b",
  [0x3279] = "\xc2\x9c", [0x327a] = "\xc2\x9d", [0x327b] = "\xc2\x9e",
  [0x327c] = "\xc2\x9f", [0x327d] = "\xc2\xa0", [0x327e] = "\xc3\x40",
  [0x327f] = "\xed\xb2", [0x3280] = "\xed\xb1", [0x3281] = "\xc3\x41",
  [0x3282] = "\xc3\x42", [0x3283] = "\xcb\xe0", [0x3284] = "\xd2\xde",
  [0x3285] = "\xc3\x43", [0x3286] = "\xcb\xc1", [0x3287] = "\xd5\xd8",
  [0x3288] = "\xc3\x44", [0x3289] = "\xc8\xe2", [0x328a] = "\xc3\x45",
  [0x328b] = "\xc0\xdf", [0x328c] = "\xbc\xa1", [0x328d] = "\xc3\x46",
  [0x328e] = "\xc3\x47", [0x328f] = "\xc3\x48", [0x3290] = "\xc3\x49",
  [0x3291] = "\xc3\x4a", [0x3292] = "\xc3\x4b", [0x3293] = "\xeb\xc1",
  [0x3294] = "\xc3\x4c", [0x3295] = "\xc3\x4d", [0x3296] = "\xd0\xa4",
  [0x3297] = "\xc3\x4e", [0x3298] = "\xd6\xe2", [0x3299] = "\xc3\x4f",
  [0x329a] = "\xb6\xc7", [0x329b] = "\xb8\xd8", [0x329c] = "\xeb\xc0",
  [0x329d] = "\xb8\xce", [0x329e] = "\xc3\x50", [0x329f] = "\xeb\xbf",
  [0x32a0] = "\xb3\xa6", [0x32a1] = "\xb9\xc9", [0x32a2] = "\xd6\xab",
  [0x32a3] = "\xc3\x51", [0x32a4] = "\xb7\xf4", [0x32a5] = "\xb7\xca",
  [0x32a6] = "\xc3\x52", [0x32a7] = "\xc3\x53", [0x32a8] = "\xc3\x54",
  [0x32a9] = "\xbc\xe7", [0x32aa] = "\xb7\xbe", [0x32ab] = "\xeb\xc6",
  [0x32ac] = "\xc3\x55", [0x32ad] = "\xeb\xc7", [0x32ae] = "\xb0\xb9",
  [0x32af] = "\xbf\xcf", [0x32b0] = "\xc3\x56", [0x32b1] = "\xeb\xc5",
  [0x32b2] = "\xd3\xfd", [0x32b3] = "\xc3\x57", [0x32b4] = "\xeb\xc8",
  [0x32b5] = "\xc3\x58", [0x32b6] = "\xc3\x59", [0x32b7] = "\xeb\xc9",
  [0x32b8] = "\xc3\x5a", [0x32b9] = "\xc3\x5b", [0x32ba] = "\xb7\xce",
  [0x32bb] = "\xc3\x5c", [0x32bc] = "\xeb\xc2", [0x32bd] = "\xeb\xc4",
  [0x32be] = "\xc9\xf6", [0x32bf] = "\xd6\xd7", [0x32c0] = "\xd5\xcd",
  [0x32c1] = "\xd0\xb2", [0x32c2] = "\xeb\xcf", [0x32c3] = "\xce\xb8",
  [0x32c4] = "\xeb\xd0", [0x32c5] = "\xc3\x5d", [0x32c6] = "\xb5\xa8",
  [0x32c7] = "\xc3\x5e", [0x32c8] = "\xc3\x5f", [0x32c9] = "\xc3\x60",
  [0x32ca] = "\xc3\x61", [0x32cb] = "\xc3\x62", [0x32cc] = "\xb1\xb3",
  [0x32cd] = "\xeb\xd2", [0x32ce] = "\xcc\xa5", [0x32cf] = "\xc3\x63",
  [0x32d0] = "\xc3\x64", [0x32d1] = "\xc3\x65", [0x32d2] = "\xc3\x66",
  [0x32d3] = "\xc3\x67", [0x32d4] = "\xc3\x68", [0x32d5] = "\xc3\x69",
  [0x32d6] = "\xc5\xd6", [0x32d7] = "\xeb\xd3", [0x32d8] = "\xc3\x6a",
  [0x32d9] = "\xeb\xd1", [0x32da] = "\xc5\xdf", [0x32db] = "\xeb\xce",
  [0x32dc] = "\xca\xa4", [0x32dd] = "\xeb\xd5", [0x32de] = "\xb0\xfb",
  [0x32df] = "\xc3\x6b", [0x32e0] = "\xc3\x6c", [0x32e1] = "\xba\xfa",
  [0x32e2] = "\xc3\x6d", [0x32e3] = "\xc3\x6e", [0x32e4] = "\xd8\xb7",
  [0x32e5] = "\xf1\xe3", [0x32e6] = "\xc3\x6f", [0x32e7] = "\xeb\xca",
  [0x32e8] = "\xeb\xcb", [0x32e9] = "\xeb\xcc", [0x32ea] = "\xeb\xcd",
  [0x32eb] = "\xeb\xd6", [0x32ec] = "\xe6\xc0", [0x32ed] = "\xeb\xd9",
  [0x32ee] = "\xc3\x70", [0x32ef] = "\xbf\xe8", [0x32f0] = "\xd2\xc8",
  [0x32f1] = "\xeb\xd7", [0x32f2] = "\xeb\xdc", [0x32f3] = "\xb8\xec",
  [0x32f4] = "\xeb\xd8", [0x32f5] = "\xc3\x71", [0x32f6] = "\xbd\xba",
  [0x32f7] = "\xc3\x72", [0x32f8] = "\xd0\xd8", [0x32f9] = "\xc3\x73",
  [0x32fa] = "\xb0\xb7", [0x32fb] = "\xc3\x74", [0x32fc] = "\xeb\xdd",
  [0x32fd] = "\xc4\xdc", [0x32fe] = "\xc3\x75", [0x32ff] = "\xc3\x76",
  [0x3300] = "\xc3\x77", [0x3301] = "\xc3\x78", [0x3302] = "\xd6\xac",
  [0x3303] = "\xc3\x79", [0x3304] = "\xc3\x7a", [0x3305] = "\xc3\x7b",
  [0x3306] = "\xb4\xe0", [0x3307] = "\xc3\x7c", [0x3308] = "\xc3\x7d",
  [0x3309] = "\xc2\xf6", [0x330a] = "\xbc\xb9", [0x330b] = "\xc3\x7e",
  [0x330c] = "\xc3\x80", [0x330d] = "\xeb\xda", [0x330e] = "\xeb\xdb",
  [0x330f] = "\xd4\xe0", [0x3310] = "\xc6\xea", [0x3311] = "\xc4\xd4",
  [0x3312] = "\xeb\xdf", [0x3313] = "\xc5\xa7", [0x3314] = "\xd9\xf5",
  [0x3315] = "\xc3\x81", [0x3316] = "\xb2\xb1", [0x3317] = "\xc3\x82",
  [0x3318] = "\xeb\xe4", [0x3319] = "\xc3\x83", [0x331a] = "\xbd\xc5",
  [0x331b] = "\xc3\x84", [0x331c] = "\xc3\x85", [0x331d] = "\xc3\x86",
  [0x331e] = "\xeb\xe2", [0x331f] = "\xc3\x87", [0x3320] = "\xc3\x88",
  [0x3321] = "\xc3\x89", [0x3322] = "\xc3\x8a", [0x3323] = "\xc3\x8b",
  [0x3324] = "\xc3\x8c", [0x3325] = "\xc3\x8d", [0x3326] = "\xc3\x8e",
  [0x3327] = "\xc3\x8f", [0x3328] = "\xc3\x90", [0x3329] = "\xc3\x91",
  [0x332a] = "\xc3\x92", [0x332b] = "\xc3\x93", [0x332c] = "\xeb\xe3",
  [0x332d] = "\xc3\x94", [0x332e] = "\xc3\x95", [0x332f] = "\xb8\xac",
  [0x3330] = "\xc3\x96", [0x3331] = "\xcd\xd1", [0x3332] = "\xeb\xe5",
  [0x3333] = "\xc3\x97", [0x3334] = "\xc3\x98", [0x3335] = "\xc3\x99",
  [0x3336] = "\xeb\xe1", [0x3337] = "\xc3\x9a", [0x3338] = "\xc1\xb3",
  [0x3339] = "\xc3\x9b", [0x333a] = "\xc3\x9c", [0x333b] = "\xc3\x9d",
  [0x333c] = "\xc3\x9e", [0x333d] = "\xc3\x9f", [0x333e] = "\xc6\xa2",
  [0x333f] = "\xc3\xa0", [0x3340] = "\xc4\x40", [0x3341] = "\xc4\x41",
  [0x3342] = "\xc4\x42", [0x3343] = "\xc4\x43", [0x3344] = "\xc4\x44",
  [0x3345] = "\xc4\x45", [0x3346] = "\xcc\xf3", [0x3347] = "\xc4\x46",
  [0x3348] = "\xeb\xe6", [0x3349] = "\xc4\x47", [0x334a] = "\xc0\xb0",
  [0x334b] = "\xd2\xb8", [0x334c] = "\xeb\xe7", [0x334d] = "\xc4\x48",
  [0x334e] = "\xc4\x49", [0x334f] = "\xc4\x4a", [0x3350] = "\xb8\xaf",
  [0x3351] = "\xb8\xad", [0x3352] = "\xc4\x4b", [0x3353] = "\xeb\xe8",
  [0x3354] = "\xc7\xbb", [0x3355] = "\xcd\xf3", [0x3356] = "\xc4\x4c",
  [0x3357] = "\xc4\x4d", [0x3358] = "\xc4\x4e", [0x3359] = "\xeb\xea",
  [0x335a] = "\xeb\xeb", [0x335b] = "\xc4\x4f", [0x335c] = "\xc4\x50",
  [0x335d] = "\xc4\x51", [0x335e] = "\xc4\x52", [0x335f] = "\xc4\x53",
  [0x3360] = "\xeb\xed", [0x3361] = "\xc4\x54", [0x3362] = "\xc4\x55",
  [0x3363] = "\xc4\x56", [0x3364] = "\xc4\x57", [0x3365] = "\xd0\xc8",
  [0x3366] = "\xc4\x58", [0x3367] = "\xeb\xf2", [0x3368] = "\xc4\x59",
  [0x3369] = "\xeb\xee", [0x336a] = "\xc4\x5a", [0x336b] = "\xc4\x5b",
  [0x336c] = "\xc4\x5c", [0x336d] = "\xeb\xf1", [0x336e] = "\xc8\xf9",
  [0x336f] = "\xc4\x5d", [0x3370] = "\xd1\xfc", [0x3371] = "\xeb\xec",
  [0x3372] = "\xc4\x5e", [0x3373] = "\xc4\x5f", [0x3374] = "\xeb\xe9",
  [0x3375] = "\xc4\x60", [0x3376] = "\xc4\x61", [0x3377] = "\xc4\x62",
  [0x3378] = "\xc4\x63", [0x3379] = "\xb8\xb9", [0x337a] = "\xcf\xd9",
  [0x337b] = "\xc4\xe5", [0x337c] = "\xeb\xef", [0x337d] = "\xeb\xf0",
  [0x337e] = "\xcc\xda", [0x337f] = "\xcd\xc8", [0x3380] = "\xb0\xf2",
  [0x3381] = "\xc4\x64", [0x3382] = "\xeb\xf6", [0x3383] = "\xc4\x65",
  [0x3384] = "\xc4\x66", [0x3385] = "\xc4\x67", [0x3386] = "\xc4\x68",
  [0x3387] = "\xc4\x69", [0x3388] = "\xeb\xf5", [0x3389] = "\xc4\x6a",
  [0x338a] = "\xb2\xb2", [0x338b] = "\xc4\x6b", [0x338c] = "\xc4\x6c",
  [0x338d] = "\xc4\x6d", [0x338e] = "\xc4\x6e", [0x338f] = "\xb8\xe0",
  [0x3390] = "\xc4\x6f", [0x3391] = "\xeb\xf7", [0x3392] = "\xc4\x70",
  [0x3393] = "\xc4\x71", [0x3394] = "\xc4\x72", [0x3395] = "\xc4\x73",
  [0x3396] = "\xc4\x74", [0x3397] = "\xc4\x75", [0x3398] = "\xb1\xec",
  [0x3399] = "\xc4\x76", [0x339a] = "\xc4\x77", [0x339b] = "\xcc\xc5",
  [0x339c] = "\xc4\xa4", [0x339d] = "\xcf\xa5", [0x339e] = "\xc4\x78",
  [0x339f] = "\xc4\x79", [0x33a0] = "\xc4\x7a", [0x33a1] = "\xc4\x7b",
  [0x33a2] = "\xc4\x7c", [0x33a3] = "\xeb\xf9", [0x33a4] = "\xc4\x7d",
  [0x33a5] = "\xc4\x7e", [0x33a6] = "\xec\xa2", [0x33a7] = "\xc4\x80",
  [0x33a8] = "\xc5\xf2", [0x33a9] = "\xc4\x81", [0x33aa] = "\xeb\xfa",
  [0x33ab] = "\xc4\x82", [0x33ac] = "\xc4\x83", [0x33ad] = "\xc4\x84",
  [0x33ae] = "\xc4\x85", [0x33af] = "\xc4\x86", [0x33b0] = "\xc4\x87",
  [0x33b1] = "\xc4\x88", [0x33b2] = "\xc4\x89", [0x33b3] = "\xc9\xc5",
  [0x33b4] = "\xc4\x8a", [0x33b5] = "\xc4\x8b", [0x33b6] = "\xc4\x8c",
  [0x33b7] = "\xc4\x8d", [0x33b8] = "\xc4\x8e", [0x33b9] = "\xc4\x8f",
  [0x33ba] = "\xe2\xdf", [0x33bb] = "\xeb\xfe", [0x33bc] = "\xc4\x90",
  [0x33bd] = "\xc4\x91", [0x33be] = "\xc4\x92", [0x33bf] = "\xc4\x93",
  [0x33c0] = "\xcd\xce", [0x33c1] = "\xec\xa1", [0x33c2] = "\xb1\xdb",
  [0x33c3] = "\xd3\xb7", [0x33c4] = "\xc4\x94", [0x33c5] = "\xc4\x95",
  [0x33c6] = "\xd2\xdc", [0x33c7] = "\xc4\x96", [0x33c8] = "\xc4\x97",
  [0x33c9] = "\xc4\x98", [0x33ca] = "\xeb\xfd", [0x33cb] = "\xc4\x99",
  [0x33cc] = "\xeb\xfb", [0x33cd] = "\xc4\x9a", [0x33ce] = "\xc4\x9b",
  [0x33cf] = "\xc4\x9c", [0x33d0] = "\xc4\x9d", [0x33d1] = "\xc4\x9e",
  [0x33d2] = "\xc4\x9f", [0x33d3] = "\xc4\xa0", [0x33d4] = "\xc5\x40",
  [0x33d5] = "\xc5\x41", [0x33d6] = "\xc5\x42", [0x33d7] = "\xc5\x43",
  [0x33d8] = "\xc5\x44", [0x33d9] = "\xc5\x45", [0x33da] = "\xc5\x46",
  [0x33db] = "\xc5\x47", [0x33dc] = "\xc5\x48", [0x33dd] = "\xc5\x49",
  [0x33de] = "\xc5\x4a", [0x33df] = "\xc5\x4b", [0x33e0] = "\xc5\x4c",
  [0x33e1] = "\xc5\x4d", [0x33e2] = "\xc5\x4e", [0x33e3] = "\xb3\xbc",
  [0x33e4] = "\xc5\x4f", [0x33e5] = "\xc5\x50", [0x33e6] = "\xc5\x51",
  [0x33e7] = "\xea\xb0", [0x33e8] = "\xc5\x52", [0x33e9] = "\xc5\x53",
  [0x33ea] = "\xd7\xd4", [0x33eb] = "\xc5\x54", [0x33ec] = "\xf4\xab",
  [0x33ed] = "\xb3\xf4", [0x33ee] = "\xc5\x55", [0x33ef] = "\xc5\x56",
  [0x33f0] = "\xc5\x57", [0x33f1] = "\xc5\x58", [0x33f2] = "\xc5\x59",
  [0x33f3] = "\xd6\xc1", [0x33f4] = "\xd6\xc2", [0x33f5] = "\xc5\x5a",
  [0x33f6] = "\xc5\x5b", [0x33f7] = "\xc5\x5c", [0x33f8] = "\xc5\x5d",
  [0x33f9] = "\xc5\x5e", [0x33fa] = "\xc5\x5f", [0x33fb] = "\xd5\xe9",
  [0x33fc] = "\xbe\xca", [0x33fd] = "\xc5\x60", [0x33fe] = "\xf4\xa7",
  [0x33ff] = "\xc5\x61", [0x3400] = "\xd2\xa8", [0x3401] = "\xf4\xa8",
  [0x3402] = "\xf4\xa9", [0x3403] = "\xc5\x62", [0x3404] = "\xf4\xaa",
  [0x3405] = "\xbe\xcb", [0x3406] = "\xd3\xdf", [0x3407] = "\xc5\x63",
  [0x3408] = "\xc5\x64", [0x3409] = "\xc5\x65", [0x340a] = "\xc5\x66",
  [0x340b] = "\xc5\x67", [0x340c] = "\xc9\xe0", [0x340d] = "\xc9\xe1",
  [0x340e] = "\xc5\x68", [0x340f] = "\xc5\x69", [0x3410] = "\xf3\xc2",
  [0x3411] = "\xc5\x6a", [0x3412] = "\xca\xe6", [0x3413] = "\xc5\x6b",
  [0x3414] = "\xcc\xf2", [0x3415] = "\xc5\x6c", [0x3416] = "\xc5\x6d",
  [0x3417] = "\xc5\x6e", [0x3418] = "\xc5\x6f", [0x3419] = "\xc5\x70",
  [0x341a] = "\xc5\x71", [0x341b] = "\xe2\xb6", [0x341c] = "\xcb\xb4",
  [0x341d] = "\xc5\x72", [0x341e] = "\xce\xe8", [0x341f] = "\xd6\xdb",
  [0x3420] = "\xc5\x73", [0x3421] = "\xf4\xad", [0x3422] = "\xf4\xae",
  [0x3423] = "\xf4\xaf", [0x3424] = "\xc5\x74", [0x3425] = "\xc5\x75",
  [0x3426] = "\xc5\x76", [0x3427] = "\xc5\x77", [0x3428] = "\xf4\xb2",
  [0x3429] = "\xc5\x78", [0x342a] = "\xba\xbd", [0x342b] = "\xf4\xb3",
  [0x342c] = "\xb0\xe3", [0x342d] = "\xf4\xb0", [0x342e] = "\xc5\x79",
  [0x342f] = "\xf4\xb1", [0x3430] = "\xbd\xa2", [0x3431] = "\xb2\xd5",
  [0x3432] = "\xc5\x7a", [0x3433] = "\xf4\xb6", [0x3434] = "\xf4\xb7",
  [0x3435] = "\xb6\xe6", [0x3436] = "\xb2\xb0", [0x3437] = "\xcf\xcf",
  [0x3438] = "\xf4\xb4", [0x3439] = "\xb4\xac", [0x343a] = "\xc5\x7b",
  [0x343b] = "\xf4\xb5", [0x343c] = "\xc5\x7c", [0x343d] = "\xc5\x7d",
  [0x343e] = "\xf4\xb8", [0x343f] = "\xc5\x7e", [0x3440] = "\xc5\x80",
  [0x3441] = "\xc5\x81", [0x3442] = "\xc5\x82", [0x3443] = "\xc5\x83",
  [0x3444] = "\xf4\xb9", [0x3445] = "\xc5\x84", [0x3446] = "\xc5\x85",
  [0x3447] = "\xcd\xa7", [0x3448] = "\xc5\x86", [0x3449] = "\xf4\xba",
  [0x344a] = "\xc5\x87", [0x344b] = "\xf4\xbb", [0x344c] = "\xc5\x88",
  [0x344d] = "\xc5\x89", [0x344e] = "\xc5\x8a", [0x344f] = "\xf4\xbc",
  [0x3450] = "\xc5\x8b", [0x3451] = "\xc5\x8c", [0x3452] = "\xc5\x8d",
  [0x3453] = "\xc5\x8e", [0x3454] = "\xc5\x8f", [0x3455] = "\xc5\x90",
  [0x3456] = "\xc5\x91", [0x3457] = "\xc5\x92", [0x3458] = "\xcb\xd2",
  [0x3459] = "\xc5\x93", [0x345a] = "\xf4\xbd", [0x345b] = "\xc5\x94",
  [0x345c] = "\xc5\x95", [0x345d] = "\xc5\x96", [0x345e] = "\xc5\x97",
  [0x345f] = "\xf4\xbe", [0x3460] = "\xc5\x98", [0x3461] = "\xc5\x99",
  [0x3462] = "\xc5\x9a", [0x3463] = "\xc5\x9b", [0x3464] = "\xc5\x9c",
  [0x3465] = "\xc5\x9d", [0x3466] = "\xc5\x9e", [0x3467] = "\xc5\x9f",
  [0x3468] = "\xf4\xbf", [0x3469] = "\xc5\xa0", [0x346a] = "\xc6\x40",
  [0x346b] = "\xc6\x41", [0x346c] = "\xc6\x42", [0x346d] = "\xc6\x43",
  [0x346e] = "\xf4\xde", [0x346f] = "\xc1\xbc", [0x3470] = "\xbc\xe8",
  [0x3471] = "\xc6\x44", [0x3472] = "\xc9\xab", [0x3473] = "\xd1\xde",
  [0x3474] = "\xe5\xf5", [0x3475] = "\xc6\x45", [0x3476] = "\xc6\x46",
  [0x3477] = "\xc6\x47", [0x3478] = "\xc6\x48", [0x3479] = "\xdc\xb3",
  [0x347a] = "\xd2\xd5", [0x347b] = "\xc6\x49", [0x347c] = "\xc6\x4a",
  [0x347d] = "\xdc\xb4", [0x347e] = "\xb0\xac", [0x347f] = "\xdc\xb5",
  [0x3480] = "\xc6\x4b", [0x3481] = "\xc6\x4c", [0x3482] = "\xbd\xda",
  [0x3483] = "\xc6\x4d", [0x3484] = "\xdc\xb9", [0x3485] = "\xc6\x4e",
  [0x3486] = "\xc6\x4f", [0x3487] = "\xc6\x50", [0x3488] = "\xd8\xc2",
  [0x3489] = "\xc6\x51", [0x348a] = "\xdc\xb7", [0x348b] = "\xd3\xf3",
  [0x348c] = "\xc6\x52", [0x348d] = "\xc9\xd6", [0x348e] = "\xdc\xba",
  [0x348f] = "\xdc\xb6", [0x3490] = "\xc6\x53", [0x3491] = "\xdc\xbb",
  [0x3492] = "\xc3\xa2", [0x3493] = "\xc6\x54", [0x3494] = "\xc6\x55",
  [0x3495] = "\xc6\x56", [0x3496] = "\xc6\x57", [0x3497] = "\xdc\xbc",
  [0x3498] = "\xdc\xc5", [0x3499] = "\xdc\xbd", [0x349a] = "\xc6\x58",
  [0x349b] = "\xc6\x59", [0x349c] = "\xce\xdf", [0x349d] = "\xd6\xa5",
  [0x349e] = "\xc6\x5a", [0x349f] = "\xdc\xcf", [0x34a0] = "\xc6\x5b",
  [0x34a1] = "\xdc\xcd", [0x34a2] = "\xc6\x5c", [0x34a3] = "\xc6\x5d",
  [0x34a4] = "\xdc\xd2", [0x34a5] = "\xbd\xe6", [0x34a6] = "\xc2\xab",
  [0x34a7] = "\xc6\x5e", [0x34a8] = "\xdc\xb8", [0x34a9] = "\xdc\xcb",
  [0x34aa] = "\xdc\xce", [0x34ab] = "\xdc\xbe", [0x34ac] = "\xb7\xd2",
  [0x34ad] = "\xb0\xc5", [0x34ae] = "\xdc\xc7", [0x34af] = "\xd0\xbe",
  [0x34b0] = "\xdc\xc1", [0x34b1] = "\xbb\xa8", [0x34b2] = "\xc6\x5f",
  [0x34b3] = "\xb7\xbc", [0x34b4] = "\xdc\xcc", [0x34b5] = "\xc6\x60",
  [0x34b6] = "\xc6\x61", [0x34b7] = "\xdc\xc6", [0x34b8] = "\xdc\xbf",
  [0x34b9] = "\xc7\xdb", [0x34ba] = "\xc6\x62", [0x34bb] = "\xc6\x63",
  [0x34bc] = "\xc6\x64", [0x34bd] = "\xd1\xbf", [0x34be] = "\xdc\xc0",
  [0x34bf] = "\xc6\x65", [0x34c0] = "\xc6\x66", [0x34c1] = "\xdc\xca",
  [0x34c2] = "\xc6\x67", [0x34c3] = "\xc6\x68", [0x34c4] = "\xdc\xd0",
  [0x34c5] = "\xc6\x69", [0x34c6] = "\xc6\x6a", [0x34c7] = "\xce\xad",
  [0x34c8] = "\xdc\xc2", [0x34c9] = "\xc6\x6b", [0x34ca] = "\xdc\xc3",
  [0x34cb] = "\xdc\xc8", [0x34cc] = "\xdc\xc9", [0x34cd] = "\xb2\xd4",
  [0x34ce] = "\xdc\xd1", [0x34cf] = "\xcb\xd5", [0x34d0] = "\xc6\x6c",
  [0x34d1] = "\xd4\xb7", [0x34d2] = "\xdc\xdb", [0x34d3] = "\xdc\xdf",
  [0x34d4] = "\xcc\xa6", [0x34d5] = "\xdc\xe6", [0x34d6] = "\xc6\x6d",
  [0x34d7] = "\xc3\xe7", [0x34d8] = "\xdc\xdc", [0x34d9] = "\xc6\x6e",
  [0x34da] = "\xc6\x6f", [0x34db] = "\xbf\xc1", [0x34dc] = "\xdc\xd9",
  [0x34dd] = "\xc6\x70", [0x34de] = "\xb0\xfa", [0x34df] = "\xb9\xb6",
  [0x34e0] = "\xdc\xe5", [0x34e1] = "\xdc\xd3", [0x34e2] = "\xc6\x71",
  [0x34e3] = "\xdc\xc4", [0x34e4] = "\xdc\xd6", [0x34e5] = "\xc8\xf4",
  [0x34e6] = "\xbf\xe0", [0x34e7] = "\xc6\x72", [0x34e8] = "\xc6\x73",
  [0x34e9] = "\xc6\x74", [0x34ea] = "\xc6\x75", [0x34eb] = "\xc9\xbb",
  [0x34ec] = "\xc6\x76", [0x34ed] = "\xc6\x77", [0x34ee] = "\xc6\x78",
  [0x34ef] = "\xb1\xbd", [0x34f0] = "\xc6\x79", [0x34f1] = "\xd3\xa2",
  [0x34f2] = "\xc6\x7a", [0x34f3] = "\xc6\x7b", [0x34f4] = "\xdc\xda",
  [0x34f5] = "\xc6\x7c", [0x34f6] = "\xc6\x7d", [0x34f7] = "\xdc\xd5",
  [0x34f8] = "\xc6\x7e", [0x34f9] = "\xc6\xbb", [0x34fa] = "\xc6\x80",
  [0x34fb] = "\xdc\xde", [0x34fc] = "\xc6\x81", [0x34fd] = "\xc6\x82",
  [0x34fe] = "\xc6\x83", [0x34ff] = "\xc6\x84", [0x3500] = "\xc6\x85",
  [0x3501] = "\xd7\xc2", [0x3502] = "\xc3\xaf", [0x3503] = "\xb7\xb6",
  [0x3504] = "\xc7\xd1", [0x3505] = "\xc3\xa9", [0x3506] = "\xdc\xe2",
  [0x3507] = "\xdc\xd8", [0x3508] = "\xdc\xeb", [0x3509] = "\xdc\xd4",
  [0x350a] = "\xc6\x86", [0x350b] = "\xc6\x87", [0x350c] = "\xdc\xdd",
  [0x350d] = "\xc6\x88", [0x350e] = "\xbe\xa5", [0x350f] = "\xdc\xd7",
  [0x3510] = "\xc6\x89", [0x3511] = "\xdc\xe0", [0x3512] = "\xc6\x8a",
  [0x3513] = "\xc6\x8b", [0x3514] = "\xdc\xe3", [0x3515] = "\xdc\xe4",
  [0x3516] = "\xc6\x8c", [0x3517] = "\xdc\xf8", [0x3518] = "\xc6\x8d",
  [0x3519] = "\xc6\x8e", [0x351a] = "\xdc\xe1", [0x351b] = "\xdd\xa2",
  [0x351c] = "\xdc\xe7", [0x351d] = "\xc6\x8f", [0x351e] = "\xc6\x90",
  [0x351f] = "\xc6\x91", [0x3520] = "\xc6\x92", [0x3521] = "\xc6\x93",
  [0x3522] = "\xc6\x94", [0x3523] = "\xc6\x95", [0x3524] = "\xc6\x96",
  [0x3525] = "\xc6\x97", [0x3526] = "\xc6\x98", [0x3527] = "\xbc\xeb",
  [0x3528] = "\xb4\xc4", [0x3529] = "\xc6\x99", [0x352a] = "\xc6\x9a",
  [0x352b] = "\xc3\xa3", [0x352c] = "\xb2\xe7", [0x352d] = "\xdc\xfa",
  [0x352e] = "\xc6\x9b", [0x352f] = "\xdc\xf2", [0x3530] = "\xc6\x9c",
  [0x3531] = "\xdc\xef", [0x3532] = "\xc6\x9d", [0x3533] = "\xdc\xfc",
  [0x3534] = "\xdc\xee", [0x3535] = "\xd2\xf0", [0x3536] = "\xb2\xe8",
  [0x3537] = "\xc6\x9e", [0x3538] = "\xc8\xd7", [0x3539] = "\xc8\xe3",
  [0x353a] = "\xdc\xfb", [0x353b] = "\xc6\x9f", [0x353c] = "\xdc\xed",
  [0x353d] = "\xc6\xa0", [0x353e] = "\xc7\x40", [0x353f] = "\xc7\x41",
  [0x3540] = "\xdc\xf7", [0x3541] = "\xc7\x42", [0x3542] = "\xc7\x43",
  [0x3543] = "\xdc\xf5", [0x3544] = "\xc7\x44", [0x3545] = "\xc7\x45",
  [0x3546] = "\xbe\xa3", [0x3547] = "\xdc\xf4", [0x3548] = "\xc7\x46",
  [0x3549] = "\xb2\xdd", [0x354a] = "\xc7\x47", [0x354b] = "\xc7\x48",
  [0x354c] = "\xc7\x49", [0x354d] = "\xc7\x4a", [0x354e] = "\xc7\x4b",
  [0x354f] = "\xdc\xf3", [0x3550] = "\xbc\xf6", [0x3551] = "\xdc\xe8",
  [0x3552] = "\xbb\xc4", [0x3553] = "\xc7\x4c", [0x3554] = "\xc0\xf3",
  [0x3555] = "\xc7\x4d", [0x3556] = "\xc7\x4e", [0x3557] = "\xc7\x4f",
  [0x3558] = "\xc7\x50", [0x3559] = "\xc7\x51", [0x355a] = "\xbc\xd4",
  [0x355b] = "\xdc\xe9", [0x355c] = "\xdc\xea", [0x355d] = "\xc7\x52",
  [0x355e] = "\xdc\xf1", [0x355f] = "\xdc\xf6", [0x3560] = "\xdc\xf9",
  [0x3561] = "\xb5\xb4", [0x3562] = "\xc7\x53", [0x3563] = "\xc8\xd9",
  [0x3564] = "\xbb\xe7", [0x3565] = "\xdc\xfe", [0x3566] = "\xdc\xfd",
  [0x3567] = "\xd3\xab", [0x3568] = "\xdd\xa1", [0x3569] = "\xdd\xa3",
  [0x356a] = "\xdd\xa5", [0x356b] = "\xd2\xf1", [0x356c] = "\xdd\xa4",
  [0x356d] = "\xdd\xa6", [0x356e] = "\xdd\xa7", [0x356f] = "\xd2\xa9",
  [0x3570] = "\xc7\x54", [0x3571] = "\xc7\x55", [0x3572] = "\xc7\x56",
  [0x3573] = "\xc7\x57", [0x3574] = "\xc7\x58", [0x3575] = "\xc7\x59",
  [0x3576] = "\xc7\x5a", [0x3577] = "\xba\xc9", [0x3578] = "\xdd\xa9",
  [0x3579] = "\xc7\x5b", [0x357a] = "\xc7\x5c", [0x357b] = "\xdd\xb6",
  [0x357c] = "\xdd\xb1", [0x357d] = "\xdd\xb4", [0x357e] = "\xc7\x5d",
  [0x357f] = "\xc7\x5e", [0x3580] = "\xc7\x5f", [0x3581] = "\xc7\x60",
  [0x3582] = "\xc7\x61", [0x3583] = "\xc7\x62", [0x3584] = "\xc7\x63",
  [0x3585] = "\xdd\xb0", [0x3586] = "\xc6\xce", [0x3587] = "\xc7\x64",
  [0x3588] = "\xc7\x65", [0x3589] = "\xc0\xf2", [0x358a] = "\xc7\x66",
  [0x358b] = "\xc7\x67", [0x358c] = "\xc7\x68", [0x358d] = "\xc7\x69",
  [0x358e] = "\xc9\xaf", [0x358f] = "\xc7\x6a", [0x3590] = "\xc7\x6b",
  [0x3591] = "\xc7\x6c", [0x3592] = "\xdc\xec", [0x3593] = "\xdd\xae",
  [0x3594] = "\xc7\x6d", [0x3595] = "\xc7\x6e", [0x3596] = "\xc7\x6f",
  [0x3597] = "\xc7\x70", [0x3598] = "\xdd\xb7", [0x3599] = "\xc7\x71",
  [0x359a] = "\xc7\x72", [0x359b] = "\xdc\xf0", [0x359c] = "\xdd\xaf",
  [0x359d] = "\xc7\x73", [0x359e] = "\xdd\xb8", [0x359f] = "\xc7\x74",
  [0x35a0] = "\xdd\xac", [0x35a1] = "\xc7\x75", [0x35a2] = "\xc7\x76",
  [0x35a3] = "\xc7\x77", [0x35a4] = "\xc7\x78", [0x35a5] = "\xc7\x79",
  [0x35a6] = "\xc7\x7a", [0x35a7] = "\xc7\x7b", [0x35a8] = "\xdd\xb9",
  [0x35a9] = "\xdd\xb3", [0x35aa] = "\xdd\xad", [0x35ab] = "\xc4\xaa",
  [0x35ac] = "\xc7\x7c", [0x35ad] = "\xc7\x7d", [0x35ae] = "\xc7\x7e",
  [0x35af] = "\xc7\x80", [0x35b0] = "\xdd\xa8", [0x35b1] = "\xc0\xb3",
  [0x35b2] = "\xc1\xab", [0x35b3] = "\xdd\xaa", [0x35b4] = "\xdd\xab",
  [0x35b5] = "\xc7\x81", [0x35b6] = "\xdd\xb2", [0x35b7] = "\xbb\xf1",
  [0x35b8] = "\xdd\xb5", [0x35b9] = "\xd3\xa8", [0x35ba] = "\xdd\xba",
  [0x35bb] = "\xc7\x82", [0x35bc] = "\xdd\xbb", [0x35bd] = "\xc3\xa7",
  [0x35be] = "\xc7\x83", [0x35bf] = "\xc7\x84", [0x35c0] = "\xdd\xd2",
  [0x35c1] = "\xdd\xbc", [0x35c2] = "\xc7\x85", [0x35c3] = "\xc7\x86",
  [0x35c4] = "\xc7\x87", [0x35c5] = "\xdd\xd1", [0x35c6] = "\xc7\x88",
  [0x35c7] = "\xb9\xbd", [0x35c8] = "\xc7\x89", [0x35c9] = "\xc7\x8a",
  [0x35ca] = "\xbe\xd5", [0x35cb] = "\xc7\x8b", [0x35cc] = "\xbe\xfa",
  [0x35cd] = "\xc7\x8c", [0x35ce] = "\xc7\x8d", [0x35cf] = "\xba\xca",
  [0x35d0] = "\xc7\x8e", [0x35d1] = "\xc7\x8f", [0x35d2] = "\xc7\x90",
  [0x35d3] = "\xc7\x91", [0x35d4] = "\xdd\xca", [0x35d5] = "\xc7\x92",
  [0x35d6] = "\xdd\xc5", [0x35d7] = "\xc7\x93", [0x35d8] = "\xdd\xbf",
  [0x35d9] = "\xc7\x94", [0x35da] = "\xc7\x95", [0x35db] = "\xc7\x96",
  [0x35dc] = "\xb2\xcb", [0x35dd] = "\xdd\xc3", [0x35de] = "\xc7\x97",
  [0x35df] = "\xdd\xcb", [0x35e0] = "\xb2\xa4", [0x35e1] = "\xdd\xd5",
  [0x35e2] = "\xc7\x98", [0x35e3] = "\xc7\x99", [0x35e4] = "\xc7\x9a",
  [0x35e5] = "\xdd\xbe", [0x35e6] = "\xc7\x9b", [0x35e7] = "\xc7\x9c",
  [0x35e8] = "\xc7\x9d", [0x35e9] = "\xc6\xd0", [0x35ea] = "\xdd\xd0",
  [0x35eb] = "\xc7\x9e", [0x35ec] = "\xc7\x9f", [0x35ed] = "\xc7\xa0",
  [0x35ee] = "\xc8\x40", [0x35ef] = "\xc8\x41", [0x35f0] = "\xdd\xd4",
  [0x35f1] = "\xc1\xe2", [0x35f2] = "\xb7\xc6", [0x35f3] = "\xc8\x42",
  [0x35f4] = "\xc8\x43", [0x35f5] = "\xc8\x44", [0x35f6] = "\xc8\x45",
  [0x35f7] = "\xc8\x46", [0x35f8] = "\xdd\xce", [0x35f9] = "\xdd\xcf",
  [0x35fa] = "\xc8\x47", [0x35fb] = "\xc8\x48", [0x35fc] = "\xc8\x49",
  [0x35fd] = "\xdd\xc4", [0x35fe] = "\xc8\x4a", [0x35ff] = "\xc8\x4b",
  [0x3600] = "\xc8\x4c", [0x3601] = "\xdd\xbd", [0x3602] = "\xc8\x4d",
  [0x3603] = "\xdd\xcd", [0x3604] = "\xcc\xd1", [0x3605] = "\xc8\x4e",
  [0x3606] = "\xdd\xc9", [0x3607] = "\xc8\x4f", [0x3608] = "\xc8\x50",
  [0x3609] = "\xc8\x51", [0x360a] = "\xc8\x52", [0x360b] = "\xdd\xc2",
  [0x360c] = "\xc3\xc8", [0x360d] = "\xc6\xbc", [0x360e] = "\xce\xae",
  [0x360f] = "\xdd\xcc", [0x3610] = "\xc8\x53", [0x3611] = "\xdd\xc8",
  [0x3612] = "\xc8\x54", [0x3613] = "\xc8\x55", [0x3614] = "\xc8\x56",
  [0x3615] = "\xc8\x57", [0x3616] = "\xc8\x58", [0x3617] = "\xc8\x59",
  [0x3618] = "\xdd\xc1", [0x3619] = "\xc8\x5a", [0x361a] = "\xc8\x5b",
  [0x361b] = "\xc8\x5c", [0x361c] = "\xdd\xc6", [0x361d] = "\xc2\xdc",
  [0x361e] = "\xc8\x5d", [0x361f] = "\xc8\x5e", [0x3620] = "\xc8\x5f",
  [0x3621] = "\xc8\x60", [0x3622] = "\xc8\x61", [0x3623] = "\xc8\x62",
  [0x3624] = "\xd3\xa9", [0x3625] = "\xd3\xaa", [0x3626] = "\xdd\xd3",
  [0x3627] = "\xcf\xf4", [0x3628] = "\xc8\xf8", [0x3629] = "\xc8\x63",
  [0x362a] = "\xc8\x64", [0x362b] = "\xc8\x65", [0x362c] = "\xc8\x66",
  [0x362d] = "\xc8\x67", [0x362e] = "\xc8\x68", [0x362f] = "\xc8\x69",
  [0x3630] = "\xc8\x6a", [0x3631] = "\xdd\xe6", [0x3632] = "\xc8\x6b",
  [0x3633] = "\xc8\x6c", [0x3634] = "\xc8\x6d", [0x3635] = "\xc8\x6e",
  [0x3636] = "\xc8\x6f", [0x3637] = "\xc8\x70", [0x3638] = "\xdd\xc7",
  [0x3639] = "\xc8\x71", [0x363a] = "\xc8\x72", [0x363b] = "\xc8\x73",
  [0x363c] = "\xdd\xe0", [0x363d] = "\xc2\xe4", [0x363e] = "\xc8\x74",
  [0x363f] = "\xc8\x75", [0x3640] = "\xc8\x76", [0x3641] = "\xc8\x77",
  [0x3642] = "\xc8\x78", [0x3643] = "\xc8\x79", [0x3644] = "\xc8\x7a",
  [0x3645] = "\xc8\x7b", [0x3646] = "\xdd\xe1", [0x3647] = "\xc8\x7c",
  [0x3648] = "\xc8\x7d", [0x3649] = "\xc8\x7e", [0x364a] = "\xc8\x80",
  [0x364b] = "\xc8\x81", [0x364c] = "\xc8\x82", [0x364d] = "\xc8\x83",
  [0x364e] = "\xc8\x84", [0x364f] = "\xc8\x85", [0x3650] = "\xc8\x86",
  [0x3651] = "\xdd\xd7", [0x3652] = "\xc8\x87", [0x3653] = "\xc8\x88",
  [0x3654] = "\xc8\x89", [0x3655] = "\xc8\x8a", [0x3656] = "\xc8\x8b",
  [0x3657] = "\xd6\xf8", [0x3658] = "\xc8\x8c", [0x3659] = "\xdd\xd9",
  [0x365a] = "\xdd\xd8", [0x365b] = "\xb8\xf0", [0x365c] = "\xdd\xd6",
  [0x365d] = "\xc8\x8d", [0x365e] = "\xc8\x8e", [0x365f] = "\xc8\x8f",
  [0x3660] = "\xc8\x90", [0x3661] = "\xc6\xcf", [0x3662] = "\xc8\x91",
  [0x3663] = "\xb6\xad", [0x3664] = "\xc8\x92", [0x3665] = "\xc8\x93",
  [0x3666] = "\xc8\x94", [0x3667] = "\xc8\x95", [0x3668] = "\xc8\x96",
  [0x3669] = "\xdd\xe2", [0x366a] = "\xc8\x97", [0x366b] = "\xba\xf9",
  [0x366c] = "\xd4\xe1", [0x366d] = "\xdd\xe7", [0x366e] = "\xc8\x98",
  [0x366f] = "\xc8\x99", [0x3670] = "\xc8\x9a", [0x3671] = "\xb4\xd0",
  [0x3672] = "\xc8\x9b", [0x3673] = "\xdd\xda", [0x3674] = "\xc8\x9c",
  [0x3675] = "\xbf\xfb", [0x3676] = "\xdd\xe3", [0x3677] = "\xc8\x9d",
  [0x3678] = "\xdd\xdf", [0x3679] = "\xc8\x9e", [0x367a] = "\xdd\xdd",
  [0x367b] = "\xc8\x9f", [0x367c] = "\xc8\xa0", [0x367d] = "\xc9\x40",
  [0x367e] = "\xc9\x41", [0x367f] = "\xc9\x42", [0x3680] = "\xc9\x43",
  [0x3681] = "\xc9\x44", [0x3682] = "\xb5\xd9", [0x3683] = "\xc9\x45",
  [0x3684] = "\xc9\x46", [0x3685] = "\xc9\x47", [0x3686] = "\xc9\x48",
  [0x3687] = "\xdd\xdb", [0x3688] = "\xdd\xdc", [0x3689] = "\xdd\xde",
  [0x368a] = "\xc9\x49", [0x368b] = "\xbd\xaf", [0x368c] = "\xdd\xe4",
  [0x368d] = "\xc9\x4a", [0x368e] = "\xdd\xe5", [0x368f] = "\xc9\x4b",
  [0x3690] = "\xc9\x4c", [0x3691] = "\xc9\x4d", [0x3692] = "\xc9\x4e",
  [0x3693] = "\xc9\x4f", [0x3694] = "\xc9\x50", [0x3695] = "\xc9\x51",
  [0x3696] = "\xc9\x52", [0x3697] = "\xdd\xf5", [0x3698] = "\xc9\x53",
  [0x3699] = "\xc3\xc9", [0x369a] = "\xc9\x54", [0x369b] = "\xc9\x55",
  [0x369c] = "\xcb\xe2", [0x369d] = "\xc9\x56", [0x369e] = "\xc9\x57",
  [0x369f] = "\xc9\x58", [0x36a0] = "\xc9\x59", [0x36a1] = "\xdd\xf2",
  [0x36a2] = "\xc9\x5a", [0x36a3] = "\xc9\x5b", [0x36a4] = "\xc9\x5c",
  [0x36a5] = "\xc9\x5d", [0x36a6] = "\xc9\x5e", [0x36a7] = "\xc9\x5f",
  [0x36a8] = "\xc9\x60", [0x36a9] = "\xc9\x61", [0x36aa] = "\xc9\x62",
  [0x36ab] = "\xc9\x63", [0x36ac] = "\xc9\x64", [0x36ad] = "\xc9\x65",
  [0x36ae] = "\xc9\x66", [0x36af] = "\xd8\xe1", [0x36b0] = "\xc9\x67",
  [0x36b1] = "\xc9\x68", [0x36b2] = "\xc6\xd1", [0x36b3] = "\xc9\x69",
  [0x36b4] = "\xdd\xf4", [0x36b5] = "\xc9\x6a", [0x36b6] = "\xc9\x6b",
  [0x36b7] = "\xc9\x6c", [0x36b8] = "\xd5\xf4", [0x36b9] = "\xdd\xf3",
  [0x36ba] = "\xdd\xf0", [0x36bb] = "\xc9\x6d", [0x36bc] = "\xc9\x6e",
  [0x36bd] = "\xdd\xec", [0x36be] = "\xc9\x6f", [0x36bf] = "\xdd\xef",
  [0x36c0] = "\xc9\x70", [0x36c1] = "\xdd\xe8", [0x36c2] = "\xc9\x71",
  [0x36c3] = "\xc9\x72", [0x36c4] = "\xd0\xee", [0x36c5] = "\xc9\x73",
  [0x36c6] = "\xc9\x74", [0x36c7] = "\xc9\x75", [0x36c8] = "\xc9\x76",
  [0x36c9] = "\xc8\xd8", [0x36ca] = "\xdd\xee", [0x36cb] = "\xc9\x77",
  [0x36cc] = "\xc9\x78", [0x36cd] = "\xdd\xe9", [0x36ce] = "\xc9\x79",
  [0x36cf] = "\xc9\x7a", [0x36d0] = "\xdd\xea", [0x36d1] = "\xcb\xf2",
  [0x36d2] = "\xc9\x7b", [0x36d3] = "\xdd\xed", [0x36d4] = "\xc9\x7c",
  [0x36d5] = "\xc9\x7d", [0x36d6] = "\xb1\xcd", [0x36d7] = "\xc9\x7e",
  [0x36d8] = "\xc9\x80", [0x36d9] = "\xc9\x81", [0x36da] = "\xc9\x82",
  [0x36db] = "\xc9\x83", [0x36dc] = "\xc9\x84", [0x36dd] = "\xc0\xb6",
  [0x36de] = "\xc9\x85", [0x36df] = "\xbc\xbb", [0x36e0] = "\xdd\xf1",
  [0x36e1] = "\xc9\x86", [0x36e2] = "\xc9\x87", [0x36e3] = "\xdd\xf7",
  [0x36e4] = "\xc9\x88", [0x36e5] = "\xdd\xf6", [0x36e6] = "\xdd\xeb",
  [0x36e7] = "\xc9\x89", [0x36e8] = "\xc9\x8a", [0x36e9] = "\xc9\x8b",
  [0x36ea] = "\xc9\x8c", [0x36eb] = "\xc9\x8d", [0x36ec] = "\xc5\xee",
  [0x36ed] = "\xc9\x8e", [0x36ee] = "\xc9\x8f", [0x36ef] = "\xc9\x90",
  [0x36f0] = "\xdd\xfb", [0x36f1] = "\xc9\x91", [0x36f2] = "\xc9\x92",
  [0x36f3] = "\xc9\x93", [0x36f4] = "\xc9\x94", [0x36f5] = "\xc9\x95",
  [0x36f6] = "\xc9\x96", [0x36f7] = "\xc9\x97", [0x36f8] = "\xc9\x98",
  [0x36f9] = "\xc9\x99", [0x36fa] = "\xc9\x9a", [0x36fb] = "\xc9\x9b",
  [0x36fc] = "\xde\xa4", [0x36fd] = "\xc9\x9c", [0x36fe] = "\xc9\x9d",
  [0x36ff] = "\xde\xa3", [0x3700] = "\xc9\x9e", [0x3701] = "\xc9\x9f",
  [0x3702] = "\xc9\xa0", [0x3703] = "\xca\x40", [0x3704] = "\xca\x41",
  [0x3705] = "\xca\x42", [0x3706] = "\xca\x43", [0x3707] = "\xca\x44",
  [0x3708] = "\xca\x45", [0x3709] = "\xca\x46", [0x370a] = "\xca\x47",
  [0x370b] = "\xca\x48", [0x370c] = "\xdd\xf8", [0x370d] = "\xca\x49",
  [0x370e] = "\xca\x4a", [0x370f] = "\xca\x4b", [0x3710] = "\xca\x4c",
  [0x3711] = "\xc3\xef", [0x3712] = "\xca\x4d", [0x3713] = "\xc2\xfb",
  [0x3714] = "\xca\x4e", [0x3715] = "\xca\x4f", [0x3716] = "\xca\x50",
  [0x3717] = "\xd5\xe1", [0x3718] = "\xca\x51", [0x3719] = "\xca\x52",
  [0x371a] = "\xce\xb5", [0x371b] = "\xca\x53", [0x371c] = "\xca\x54",
  [0x371d] = "\xca\x55", [0x371e] = "\xca\x56", [0x371f] = "\xdd\xfd",
  [0x3720] = "\xca\x57", [0x3721] = "\xb2\xcc", [0x3722] = "\xca\x58",
  [0x3723] = "\xca\x59", [0x3724] = "\xca\x5a", [0x3725] = "\xca\x5b",
  [0x3726] = "\xca\x5c", [0x3727] = "\xca\x5d", [0x3728] = "\xca\x5e",
  [0x3729] = "\xca\x5f", [0x372a] = "\xca\x60", [0x372b] = "\xc4\xe8",
  [0x372c] = "\xca\xdf", [0x372d] = "\xca\x61", [0x372e] = "\xca\x62",
  [0x372f] = "\xca\x63", [0x3730] = "\xca\x64", [0x3731] = "\xca\x65",
  [0x3732] = "\xca\x66", [0x3733] = "\xca\x67", [0x3734] = "\xca\x68",
  [0x3735] = "\xca\x69", [0x3736] = "\xca\x6a", [0x3737] = "\xc7\xbe",
  [0x3738] = "\xdd\xfa", [0x3739] = "\xdd\xfc", [0x373a] = "\xdd\xfe",
  [0x373b] = "\xde\xa2", [0x373c] = "\xb0\xaa", [0x373d] = "\xb1\xce",
  [0x373e] = "\xca\x6b", [0x373f] = "\xca\x6c", [0x3740] = "\xca\x6d",
  [0x3741] = "\xca\x6e", [0x3742] = "\xca\x6f", [0x3743] = "\xde\xac",
  [0x3744] = "\xca\x70", [0x3745] = "\xca\x71", [0x3746] = "\xca\x72",
  [0x3747] = "\xca\x73", [0x3748] = "\xde\xa6", [0x3749] = "\xbd\xb6",
  [0x374a] = "\xc8\xef", [0x374b] = "\xca\x74", [0x374c] = "\xca\x75",
  [0x374d] = "\xca\x76", [0x374e] = "\xca\x77", [0x374f] = "\xca\x78",
  [0x3750] = "\xca\x79", [0x3751] = "\xca\x7a", [0x3752] = "\xca\x7b",
  [0x3753] = "\xca\x7c", [0x3754] = "\xca\x7d", [0x3755] = "\xca\x7e",
  [0x3756] = "\xde\xa1", [0x3757] = "\xca\x80", [0x3758] = "\xca\x81",
  [0x3759] = "\xde\xa5", [0x375a] = "\xca\x82", [0x375b] = "\xca\x83",
  [0x375c] = "\xca\x84", [0x375d] = "\xca\x85", [0x375e] = "\xde\xa9",
  [0x375f] = "\xca\x86", [0x3760] = "\xca\x87", [0x3761] = "\xca\x88",
  [0x3762] = "\xca\x89", [0x3763] = "\xca\x8a", [0x3764] = "\xde\xa8",
  [0x3765] = "\xca\x8b", [0x3766] = "\xca\x8c", [0x3767] = "\xca\x8d",
  [0x3768] = "\xde\xa7", [0x3769] = "\xca\x8e", [0x376a] = "\xca\x8f",
  [0x376b] = "\xca\x90", [0x376c] = "\xca\x91", [0x376d] = "\xca\x92",
  [0x376e] = "\xca\x93", [0x376f] = "\xca\x94", [0x3770] = "\xca\x95",
  [0x3771] = "\xca\x96", [0x3772] = "\xde\xad", [0x3773] = "\xca\x97",
  [0x3774] = "\xd4\xcc", [0x3775] = "\xca\x98", [0x3776] = "\xca\x99",
  [0x3777] = "\xca\x9a", [0x3778] = "\xca\x9b", [0x3779] = "\xde\xb3",
  [0x377a] = "\xde\xaa", [0x377b] = "\xde\xae", [0x377c] = "\xca\x9c",
  [0x377d] = "\xca\x9d", [0x377e] = "\xc0\xd9", [0x377f] = "\xca\x9e",
  [0x3780] = "\xca\x9f", [0x3781] = "\xca\xa0", [0x3782] = "\xcb\x40",
  [0x3783] = "\xcb\x41", [0x3784] = "\xb1\xa1", [0x3785] = "\xde\xb6",
  [0x3786] = "\xcb\x42", [0x3787] = "\xde\xb1", [0x3788] = "\xcb\x43",
  [0x3789] = "\xcb\x44", [0x378a] = "\xcb\x45", [0x378b] = "\xcb\x46",
  [0x378c] = "\xcb\x47", [0x378d] = "\xcb\x48", [0x378e] = "\xcb\x49",
  [0x378f] = "\xde\xb2", [0x3790] = "\xcb\x4a", [0x3791] = "\xcb\x4b",
  [0x3792] = "\xcb\x4c", [0x3793] = "\xcb\x4d", [0x3794] = "\xcb\x4e",
  [0x3795] = "\xcb\x4f", [0x3796] = "\xcb\x50", [0x3797] = "\xcb\x51",
  [0x3798] = "\xcb\x52", [0x3799] = "\xcb\x53", [0x379a] = "\xcb\x54",
  [0x379b] = "\xd1\xa6", [0x379c] = "\xde\xb5", [0x379d] = "\xcb\x55",
  [0x379e] = "\xcb\x56", [0x379f] = "\xcb\x57", [0x37a0] = "\xcb\x58",
  [0x37a1] = "\xcb\x59", [0x37a2] = "\xcb\x5a", [0x37a3] = "\xcb\x5b",
  [0x37a4] = "\xde\xaf", [0x37a5] = "\xcb\x5c", [0x37a6] = "\xcb\x5d",
  [0x37a7] = "\xcb\x5e", [0x37a8] = "\xde\xb0", [0x37a9] = "\xcb\x5f",
  [0x37aa] = "\xd0\xbd", [0x37ab] = "\xcb\x60", [0x37ac] = "\xcb\x61",
  [0x37ad] = "\xcb\x62", [0x37ae] = "\xde\xb4", [0x37af] = "\xca\xed",
  [0x37b0] = "\xde\xb9", [0x37b1] = "\xcb\x63", [0x37b2] = "\xcb\x64",
  [0x37b3] = "\xcb\x65", [0x37b4] = "\xcb\x66", [0x37b5] = "\xcb\x67",
  [0x37b6] = "\xcb\x68", [0x37b7] = "\xde\xb8", [0x37b8] = "\xcb\x69",
  [0x37b9] = "\xde\xb7", [0x37ba] = "\xcb\x6a", [0x37bb] = "\xcb\x6b",
  [0x37bc] = "\xcb\x6c", [0x37bd] = "\xcb\x6d", [0x37be] = "\xcb\x6e",
  [0x37bf] = "\xcb\x6f", [0x37c0] = "\xcb\x70", [0x37c1] = "\xde\xbb",
  [0x37c2] = "\xcb\x71", [0x37c3] = "\xcb\x72", [0x37c4] = "\xcb\x73",
  [0x37c5] = "\xcb\x74", [0x37c6] = "\xcb\x75", [0x37c7] = "\xcb\x76",
  [0x37c8] = "\xcb\x77", [0x37c9] = "\xbd\xe5", [0x37ca] = "\xcb\x78",
  [0x37cb] = "\xcb\x79", [0x37cc] = "\xcb\x7a", [0x37cd] = "\xcb\x7b",
  [0x37ce] = "\xcb\x7c", [0x37cf] = "\xb2\xd8", [0x37d0] = "\xc3\xea",
  [0x37d1] = "\xcb\x7d", [0x37d2] = "\xcb\x7e", [0x37d3] = "\xde\xba",
  [0x37d4] = "\xcb\x80", [0x37d5] = "\xc5\xba", [0x37d6] = "\xcb\x81",
  [0x37d7] = "\xcb\x82", [0x37d8] = "\xcb\x83", [0x37d9] = "\xcb\x84",
  [0x37da] = "\xcb\x85", [0x37db] = "\xcb\x86", [0x37dc] = "\xde\xbc",
  [0x37dd] = "\xcb\x87", [0x37de] = "\xcb\x88", [0x37df] = "\xcb\x89",
  [0x37e0] = "\xcb\x8a", [0x37e1] = "\xcb\x8b", [0x37e2] = "\xcb\x8c",
  [0x37e3] = "\xcb\x8d", [0x37e4] = "\xcc\xd9", [0x37e5] = "\xcb\x8e",
  [0x37e6] = "\xcb\x8f", [0x37e7] = "\xcb\x90", [0x37e8] = "\xcb\x91",
  [0x37e9] = "\xb7\xaa", [0x37ea] = "\xcb\x92", [0x37eb] = "\xcb\x93",
  [0x37ec] = "\xcb\x94", [0x37ed] = "\xcb\x95", [0x37ee] = "\xcb\x96",
  [0x37ef] = "\xcb\x97", [0x37f0] = "\xcb\x98", [0x37f1] = "\xcb\x99",
  [0x37f2] = "\xcb\x9a", [0x37f3] = "\xcb\x9b", [0x37f4] = "\xcb\x9c",
  [0x37f5] = "\xcb\x9d", [0x37f6] = "\xcb\x9e", [0x37f7] = "\xcb\x9f",
  [0x37f8] = "\xcb\xa0", [0x37f9] = "\xcc\x40", [0x37fa] = "\xcc\x41",
  [0x37fb] = "\xd4\xe5", [0x37fc] = "\xcc\x42", [0x37fd] = "\xcc\x43",
  [0x37fe] = "\xcc\x44", [0x37ff] = "\xde\xbd", [0x3800] = "\xcc\x45",
  [0x3801] = "\xcc\x46", [0x3802] = "\xcc\x47", [0x3803] = "\xcc\x48",
  [0x3804] = "\xcc\x49", [0x3805] = "\xde\xbf", [0x3806] = "\xcc\x4a",
  [0x3807] = "\xcc\x4b", [0x3808] = "\xcc\x4c", [0x3809] = "\xcc\x4d",
  [0x380a] = "\xcc\x4e", [0x380b] = "\xcc\x4f", [0x380c] = "\xcc\x50",
  [0x380d] = "\xcc\x51", [0x380e] = "\xcc\x52", [0x380f] = "\xcc\x53",
  [0x3810] = "\xcc\x54", [0x3811] = "\xc4\xa2", [0x3812] = "\xcc\x55",
  [0x3813] = "\xcc\x56", [0x3814] = "\xcc\x57", [0x3815] = "\xcc\x58",
  [0x3816] = "\xde\xc1", [0x3817] = "\xcc\x59", [0x3818] = "\xcc\x5a",
  [0x3819] = "\xcc\x5b", [0x381a] = "\xcc\x5c", [0x381b] = "\xcc\x5d",
  [0x381c] = "\xcc\x5e", [0x381d] = "\xcc\x5f", [0x381e] = "\xcc\x60",
  [0x381f] = "\xcc\x61", [0x3820] = "\xcc\x62", [0x3821] = "\xcc\x63",
  [0x3822] = "\xcc\x64", [0x3823] = "\xcc\x65", [0x3824] = "\xcc\x66",
  [0x3825] = "\xcc\x67", [0x3826] = "\xcc\x68", [0x3827] = "\xde\xbe",
  [0x3828] = "\xcc\x69", [0x3829] = "\xde\xc0", [0x382a] = "\xcc\x6a",
  [0x382b] = "\xcc\x6b", [0x382c] = "\xcc\x6c", [0x382d] = "\xcc\x6d",
  [0x382e] = "\xcc\x6e", [0x382f] = "\xcc\x6f", [0x3830] = "\xcc\x70",
  [0x3831] = "\xcc\x71", [0x3832] = "\xcc\x72", [0x3833] = "\xcc\x73",
  [0x3834] = "\xcc\x74", [0x3835] = "\xcc\x75", [0x3836] = "\xcc\x76",
  [0x3837] = "\xcc\x77", [0x3838] = "\xd5\xba", [0x3839] = "\xcc\x78",
  [0x383a] = "\xcc\x79", [0x383b] = "\xcc\x7a", [0x383c] = "\xde\xc2",
  [0x383d] = "\xcc\x7b", [0x383e] = "\xcc\x7c", [0x383f] = "\xcc\x7d",
  [0x3840] = "\xcc\x7e", [0x3841] = "\xcc\x80", [0x3842] = "\xcc\x81",
  [0x3843] = "\xcc\x82", [0x3844] = "\xcc\x83", [0x3845] = "\xcc\x84",
  [0x3846] = "\xcc\x85", [0x3847] = "\xcc\x86", [0x3848] = "\xcc\x87",
  [0x3849] = "\xcc\x88", [0x384a] = "\xcc\x89", [0x384b] = "\xcc\x8a",
  [0x384c] = "\xcc\x8b", [0x384d] = "\xf2\xae", [0x384e] = "\xbb\xa2",
  [0x384f] = "\xc2\xb2", [0x3850] = "\xc5\xb0", [0x3851] = "\xc2\xc7",
  [0x3852] = "\xcc\x8c", [0x3853] = "\xcc\x8d", [0x3854] = "\xf2\xaf",
  [0x3855] = "\xcc\x8e", [0x3856] = "\xcc\x8f", [0x3857] = "\xcc\x90",
  [0x3858] = "\xcc\x91", [0x3859] = "\xcc\x92", [0x385a] = "\xd0\xe9",
  [0x385b] = "\xcc\x93", [0x385c] = "\xcc\x94", [0x385d] = "\xcc\x95",
  [0x385e] = "\xd3\xdd", [0x385f] = "\xcc\x96", [0x3860] = "\xcc\x97",
  [0x3861] = "\xcc\x98", [0x3862] = "\xeb\xbd", [0x3863] = "\xcc\x99",
  [0x3864] = "\xcc\x9a", [0x3865] = "\xcc\x9b", [0x3866] = "\xcc\x9c",
  [0x3867] = "\xcc\x9d", [0x3868] = "\xcc\x9e", [0x3869] = "\xcc\x9f",
  [0x386a] = "\xcc\xa0", [0x386b] = "\xb3\xe6", [0x386c] = "\xf2\xb0",
  [0x386d] = "\xcd\x40", [0x386e] = "\xf2\xb1", [0x386f] = "\xcd\x41",
  [0x3870] = "\xcd\x42", [0x3871] = "\xca\xad", [0x3872] = "\xcd\x43",
  [0x3873] = "\xcd\x44", [0x3874] = "\xcd\x45", [0x3875] = "\xcd\x46",
  [0x3876] = "\xcd\x47", [0x3877] = "\xcd\x48", [0x3878] = "\xcd\x49",
  [0x3879] = "\xba\xe7", [0x387a] = "\xf2\xb3", [0x387b] = "\xf2\xb5",
  [0x387c] = "\xf2\xb4", [0x387d] = "\xcb\xe4", [0x387e] = "\xcf\xba",
  [0x387f] = "\xf2\xb2", [0x3880] = "\xca\xb4", [0x3881] = "\xd2\xcf",
  [0x3882] = "\xc2\xec", [0x3883] = "\xcd\x4a", [0x3884] = "\xcd\x4b",
  [0x3885] = "\xcd\x4c", [0x3886] = "\xcd\x4d", [0x3887] = "\xcd\x4e",
  [0x3888] = "\xcd\x4f", [0x3889] = "\xcd\x50", [0x388a] = "\xce\xc3",
  [0x388b] = "\xf2\xb8", [0x388c] = "\xb0\xf6", [0x388d] = "\xf2\xb7",
  [0x388e] = "\xcd\x51", [0x388f] = "\xcd\x52", [0x3890] = "\xcd\x53",
  [0x3891] = "\xcd\x54", [0x3892] = "\xcd\x55", [0x3893] = "\xf2\xbe",
  [0x3894] = "\xcd\x56", [0x3895] = "\xb2\xcf", [0x3896] = "\xcd\x57",
  [0x3897] = "\xcd\x58", [0x3898] = "\xcd\x59", [0x3899] = "\xcd\x5a",
  [0x389a] = "\xcd\x5b", [0x389b] = "\xcd\x5c", [0x389c] = "\xd1\xc1",
  [0x389d] = "\xf2\xba", [0x389e] = "\xcd\x5d", [0x389f] = "\xcd\x5e",
  [0x38a0] = "\xcd\x5f", [0x38a1] = "\xcd\x60", [0x38a2] = "\xcd\x61",
  [0x38a3] = "\xf2\xbc", [0x38a4] = "\xd4\xe9", [0x38a5] = "\xcd\x62",
  [0x38a6] = "\xcd\x63", [0x38a7] = "\xf2\xbb", [0x38a8] = "\xf2\xb6",
  [0x38a9] = "\xf2\xbf", [0x38aa] = "\xf2\xbd", [0x38ab] = "\xcd\x64",
  [0x38ac] = "\xf2\xb9", [0x38ad] = "\xcd\x65", [0x38ae] = "\xcd\x66",
  [0x38af] = "\xf2\xc7", [0x38b0] = "\xf2\xc4", [0x38b1] = "\xf2\xc6",
  [0x38b2] = "\xcd\x67", [0x38b3] = "\xcd\x68", [0x38b4] = "\xf2\xca",
  [0x38b5] = "\xf2\xc2", [0x38b6] = "\xf2\xc0", [0x38b7] = "\xcd\x69",
  [0x38b8] = "\xcd\x6a", [0x38b9] = "\xcd\x6b", [0x38ba] = "\xf2\xc5",
  [0x38bb] = "\xcd\x6c", [0x38bc] = "\xcd\x6d", [0x38bd] = "\xcd\x6e",
  [0x38be] = "\xcd\x6f", [0x38bf] = "\xcd\x70", [0x38c0] = "\xd6\xfb",
  [0x38c1] = "\xcd\x71", [0x38c2] = "\xcd\x72", [0x38c3] = "\xcd\x73",
  [0x38c4] = "\xf2\xc1", [0x38c5] = "\xcd\x74", [0x38c6] = "\xc7\xf9",
  [0x38c7] = "\xc9\xdf", [0x38c8] = "\xcd\x75", [0x38c9] = "\xf2\xc8",
  [0x38ca] = "\xb9\xc6", [0x38cb] = "\xb5\xb0", [0x38cc] = "\xcd\x76",
  [0x38cd] = "\xcd\x77", [0x38ce] = "\xf2\xc3", [0x38cf] = "\xf2\xc9",
  [0x38d0] = "\xf2\xd0", [0x38d1] = "\xf2\xd6", [0x38d2] = "\xcd\x78",
  [0x38d3] = "\xcd\x79", [0x38d4] = "\xbb\xd7", [0x38d5] = "\xcd\x7a",
  [0x38d6] = "\xcd\x7b", [0x38d7] = "\xcd\x7c", [0x38d8] = "\xf2\xd5",
  [0x38d9] = "\xcd\xdc", [0x38da] = "\xcd\x7d", [0x38db] = "\xd6\xeb",
  [0x38dc] = "\xcd\x7e", [0x38dd] = "\xcd\x80", [0x38de] = "\xf2\xd2",
  [0x38df] = "\xf2\xd4", [0x38e0] = "\xcd\x81", [0x38e1] = "\xcd\x82",
  [0x38e2] = "\xcd\x83", [0x38e3] = "\xcd\x84", [0x38e4] = "\xb8\xf2",
  [0x38e5] = "\xcd\x85", [0x38e6] = "\xcd\x86", [0x38e7] = "\xcd\x87",
  [0x38e8] = "\xcd\x88", [0x38e9] = "\xf2\xcb", [0x38ea] = "\xcd\x89",
  [0x38eb] = "\xcd\x8a", [0x38ec] = "\xcd\x8b", [0x38ed] = "\xf2\xce",
  [0x38ee] = "\xc2\xf9", [0x38ef] = "\xcd\x8c", [0x38f0] = "\xd5\xdd",
  [0x38f1] = "\xf2\xcc", [0x38f2] = "\xf2\xcd", [0x38f3] = "\xf2\xcf",
  [0x38f4] = "\xf2\xd3", [0x38f5] = "\xcd\x8d", [0x38f6] = "\xcd\x8e",
  [0x38f7] = "\xcd\x8f", [0x38f8] = "\xf2\xd9", [0x38f9] = "\xd3\xbc",
  [0x38fa] = "\xcd\x90", [0x38fb] = "\xcd\x91", [0x38fc] = "\xcd\x92",
  [0x38fd] = "\xcd\x93", [0x38fe] = "\xb6\xea", [0x38ff] = "\xcd\x94",
  [0x3900] = "\xca\xf1", [0x3901] = "\xcd\x95", [0x3902] = "\xb7\xe4",
  [0x3903] = "\xf2\xd7", [0x3904] = "\xcd\x96", [0x3905] = "\xcd\x97",
  [0x3906] = "\xcd\x98", [0x3907] = "\xf2\xd8", [0x3908] = "\xf2\xda",
  [0x3909] = "\xf2\xdd", [0x390a] = "\xf2\xdb", [0x390b] = "\xcd\x99",
  [0x390c] = "\xcd\x9a", [0x390d] = "\xf2\xdc", [0x390e] = "\xcd\x9b",
  [0x390f] = "\xcd\x9c", [0x3910] = "\xcd\x9d", [0x3911] = "\xcd\x9e",
  [0x3912] = "\xd1\xd1", [0x3913] = "\xf2\xd1", [0x3914] = "\xcd\x9f",
  [0x3915] = "\xcd\xc9", [0x3916] = "\xcd\xa0", [0x3917] = "\xce\xcf",
  [0x3918] = "\xd6\xa9", [0x3919] = "\xce\x40", [0x391a] = "\xf2\xe3",
  [0x391b] = "\xce\x41", [0x391c] = "\xc3\xdb", [0x391d] = "\xce\x42",
  [0x391e] = "\xf2\xe0", [0x391f] = "\xce\x43", [0x3920] = "\xce\x44",
  [0x3921] = "\xc0\xaf", [0x3922] = "\xf2\xec", [0x3923] = "\xf2\xde",
  [0x3924] = "\xce\x45", [0x3925] = "\xf2\xe1", [0x3926] = "\xce\x46",
  [0x3927] = "\xce\x47", [0x3928] = "\xce\x48", [0x3929] = "\xf2\xe8",
  [0x392a] = "\xce\x49", [0x392b] = "\xce\x4a", [0x392c] = "\xce\x4b",
  [0x392d] = "\xce\x4c", [0x392e] = "\xf2\xe2", [0x392f] = "\xce\x4d",
  [0x3930] = "\xce\x4e", [0x3931] = "\xf2\xe7", [0x3932] = "\xce\x4f",
  [0x3933] = "\xce\x50", [0x3934] = "\xf2\xe6", [0x3935] = "\xce\x51",
  [0x3936] = "\xce\x52", [0x3937] = "\xf2\xe9", [0x3938] = "\xce\x53",
  [0x3939] = "\xce\x54", [0x393a] = "\xce\x55", [0x393b] = "\xf2\xdf",
  [0x393c] = "\xce\x56", [0x393d] = "\xce\x57", [0x393e] = "\xf2\xe4",
  [0x393f] = "\xf2\xea", [0x3940] = "\xce\x58", [0x3941] = "\xce\x59",
  [0x3942] = "\xce\x5a", [0x3943] = "\xce\x5b", [0x3944] = "\xce\x5c",
  [0x3945] = "\xce\x5d", [0x3946] = "\xce\x5e", [0x3947] = "\xd3\xac",
  [0x3948] = "\xf2\xe5", [0x3949] = "\xb2\xf5", [0x394a] = "\xce\x5f",
  [0x394b] = "\xce\x60", [0x394c] = "\xf2\xf2", [0x394d] = "\xce\x61",
  [0x394e] = "\xd0\xab", [0x394f] = "\xce\x62", [0x3950] = "\xce\x63",
  [0x3951] = "\xce\x64", [0x3952] = "\xce\x65", [0x3953] = "\xf2\xf5",
  [0x3954] = "\xce\x66", [0x3955] = "\xce\x67", [0x3956] = "\xce\x68",
  [0x3957] = "\xbb\xc8", [0x3958] = "\xce\x69", [0x3959] = "\xf2\xf9",
  [0x395a] = "\xce\x6a", [0x395b] = "\xce\x6b", [0x395c] = "\xce\x6c",
  [0x395d] = "\xce\x6d", [0x395e] = "\xce\x6e", [0x395f] = "\xce\x6f",
  [0x3960] = "\xf2\xf0", [0x3961] = "\xce\x70", [0x3962] = "\xce\x71",
  [0x3963] = "\xf2\xf6", [0x3964] = "\xf2\xf8", [0x3965] = "\xf2\xfa",
  [0x3966] = "\xce\x72", [0x3967] = "\xce\x73", [0x3968] = "\xce\x74",
  [0x3969] = "\xce\x75", [0x396a] = "\xce\x76", [0x396b] = "\xce\x77",
  [0x396c] = "\xce\x78", [0x396d] = "\xce\x79", [0x396e] = "\xf2\xf3",
  [0x396f] = "\xce\x7a", [0x3970] = "\xf2\xf1", [0x3971] = "\xce\x7b",
  [0x3972] = "\xce\x7c", [0x3973] = "\xce\x7d", [0x3974] = "\xba\xfb",
  [0x3975] = "\xce\x7e", [0x3976] = "\xb5\xfb", [0x3977] = "\xce\x80",
  [0x3978] = "\xce\x81", [0x3979] = "\xce\x82", [0x397a] = "\xce\x83",
  [0x397b] = "\xf2\xef", [0x397c] = "\xf2\xf7", [0x397d] = "\xf2\xed",
  [0x397e] = "\xf2\xee", [0x397f] = "\xce\x84", [0x3980] = "\xce\x85",
  [0x3981] = "\xce\x86", [0x3982] = "\xf2\xeb", [0x3983] = "\xf3\xa6",
  [0x3984] = "\xce\x87", [0x3985] = "\xf3\xa3", [0x3986] = "\xce\x88",
  [0x3987] = "\xce\x89", [0x3988] = "\xf3\xa2", [0x3989] = "\xce\x8a",
  [0x398a] = "\xce\x8b", [0x398b] = "\xf2\xf4", [0x398c] = "\xce\x8c",
  [0x398d] = "\xc8\xda", [0x398e] = "\xce\x8d", [0x398f] = "\xce\x8e",
  [0x3990] = "\xce\x8f", [0x3991] = "\xce\x90", [0x3992] = "\xce\x91",
  [0x3993] = "\xf2\xfb", [0x3994] = "\xce\x92", [0x3995] = "\xce\x93",
  [0x3996] = "\xce\x94", [0x3997] = "\xf3\xa5", [0x3998] = "\xce\x95",
  [0x3999] = "\xce\x96", [0x399a] = "\xce\x97", [0x399b] = "\xce\x98",
  [0x399c] = "\xce\x99", [0x399d] = "\xce\x9a", [0x399e] = "\xce\x9b",
  [0x399f] = "\xc3\xf8", [0x39a0] = "\xce\x9c", [0x39a1] = "\xce\x9d",
  [0x39a2] = "\xce\x9e", [0x39a3] = "\xce\x9f", [0x39a4] = "\xce\xa0",
  [0x39a5] = "\xcf\x40", [0x39a6] = "\xcf\x41", [0x39a7] = "\xcf\x42",
  [0x39a8] = "\xf2\xfd", [0x39a9] = "\xcf\x43", [0x39aa] = "\xcf\x44",
  [0x39ab] = "\xf3\xa7", [0x39ac] = "\xf3\xa9", [0x39ad] = "\xf3\xa4",
  [0x39ae] = "\xcf\x45", [0x39af] = "\xf2\xfc", [0x39b0] = "\xcf\x46",
  [0x39b1] = "\xcf\x47", [0x39b2] = "\xcf\x48", [0x39b3] = "\xf3\xab",
  [0x39b4] = "\xcf\x49", [0x39b5] = "\xf3\xaa", [0x39b6] = "\xcf\x4a",
  [0x39b7] = "\xcf\x4b", [0x39b8] = "\xcf\x4c", [0x39b9] = "\xcf\x4d",
  [0x39ba] = "\xc2\xdd", [0x39bb] = "\xcf\x4e", [0x39bc] = "\xcf\x4f",
  [0x39bd] = "\xf3\xae", [0x39be] = "\xcf\x50", [0x39bf] = "\xcf\x51",
  [0x39c0] = "\xf3\xb0", [0x39c1] = "\xcf\x52", [0x39c2] = "\xcf\x53",
  [0x39c3] = "\xcf\x54", [0x39c4] = "\xcf\x55", [0x39c5] = "\xcf\x56",
  [0x39c6] = "\xf3\xa1", [0x39c7] = "\xcf\x57", [0x39c8] = "\xcf\x58",
  [0x39c9] = "\xcf\x59", [0x39ca] = "\xf3\xb1", [0x39cb] = "\xf3\xac",
  [0x39cc] = "\xcf\x5a", [0x39cd] = "\xcf\x5b", [0x39ce] = "\xcf\x5c",
  [0x39cf] = "\xcf\x5d", [0x39d0] = "\xcf\x5e", [0x39d1] = "\xf3\xaf",
  [0x39d2] = "\xf2\xfe", [0x39d3] = "\xf3\xad", [0x39d4] = "\xcf\x5f",
  [0x39d5] = "\xcf\x60", [0x39d6] = "\xcf\x61", [0x39d7] = "\xcf\x62",
  [0x39d8] = "\xcf\x63", [0x39d9] = "\xcf\x64", [0x39da] = "\xcf\x65",
  [0x39db] = "\xf3\xb2", [0x39dc] = "\xcf\x66", [0x39dd] = "\xcf\x67",
  [0x39de] = "\xcf\x68", [0x39df] = "\xcf\x69", [0x39e0] = "\xf3\xb4",
  [0x39e1] = "\xcf\x6a", [0x39e2] = "\xcf\x6b", [0x39e3] = "\xcf\x6c",
  [0x39e4] = "\xcf\x6d", [0x39e5] = "\xf3\xa8", [0x39e6] = "\xcf\x6e",
  [0x39e7] = "\xcf\x6f", [0x39e8] = "\xcf\x70", [0x39e9] = "\xcf\x71",
  [0x39ea] = "\xf3\xb3", [0x39eb] = "\xcf\x72", [0x39ec] = "\xcf\x73",
  [0x39ed] = "\xcf\x74", [0x39ee] = "\xf3\xb5", [0x39ef] = "\xcf\x75",
  [0x39f0] = "\xcf\x76", [0x39f1] = "\xcf\x77", [0x39f2] = "\xcf\x78",
  [0x39f3] = "\xcf\x79", [0x39f4] = "\xcf\x7a", [0x39f5] = "\xcf\x7b",
  [0x39f6] = "\xcf\x7c", [0x39f7] = "\xcf\x7d", [0x39f8] = "\xcf\x7e",
  [0x39f9] = "\xd0\xb7", [0x39fa] = "\xcf\x80", [0x39fb] = "\xcf\x81",
  [0x39fc] = "\xcf\x82", [0x39fd] = "\xcf\x83", [0x39fe] = "\xf3\xb8",
  [0x39ff] = "\xcf\x84", [0x3a00] = "\xcf\x85", [0x3a01] = "\xcf\x86",
  [0x3a02] = "\xcf\x87", [0x3a03] = "\xd9\xf9", [0x3a04] = "\xcf\x88",
  [0x3a05] = "\xcf\x89", [0x3a06] = "\xcf\x8a", [0x3a07] = "\xcf\x8b",
  [0x3a08] = "\xcf\x8c", [0x3a09] = "\xcf\x8d", [0x3a0a] = "\xf3\xb9",
  [0x3a0b] = "\xcf\x8e", [0x3a0c] = "\xcf\x8f", [0x3a0d] = "\xcf\x90",
  [0x3a0e] = "\xcf\x91", [0x3a0f] = "\xcf\x92", [0x3a10] = "\xcf\x93",
  [0x3a11] = "\xcf\x94", [0x3a12] = "\xcf\x95", [0x3a13] = "\xf3\xb7",
  [0x3a14] = "\xcf\x96", [0x3a15] = "\xc8\xe4", [0x3a16] = "\xf3\xb6",
  [0x3a17] = "\xcf\x97", [0x3a18] = "\xcf\x98", [0x3a19] = "\xcf\x99",
  [0x3a1a] = "\xcf\x9a", [0x3a1b] = "\xf3\xba", [0x3a1c] = "\xcf\x9b",
  [0x3a1d] = "\xcf\x9c", [0x3a1e] = "\xcf\x9d", [0x3a1f] = "\xcf\x9e",
  [0x3a20] = "\xcf\x9f", [0x3a21] = "\xf3\xbb", [0x3a22] = "\xb4\xc0",
  [0x3a23] = "\xcf\xa0", [0x3a24] = "\xd0\x40", [0x3a25] = "\xd0\x41",
  [0x3a26] = "\xd0\x42", [0x3a27] = "\xd0\x43", [0x3a28] = "\xd0\x44",
  [0x3a29] = "\xd0\x45", [0x3a2a] = "\xd0\x46", [0x3a2b] = "\xd0\x47",
  [0x3a2c] = "\xd0\x48", [0x3a2d] = "\xd0\x49", [0x3a2e] = "\xd0\x4a",
  [0x3a2f] = "\xd0\x4b", [0x3a30] = "\xd0\x4c", [0x3a31] = "\xd0\x4d",
  [0x3a32] = "\xee\xc3", [0x3a33] = "\xd0\x4e", [0x3a34] = "\xd0\x4f",
  [0x3a35] = "\xd0\x50", [0x3a36] = "\xd0\x51", [0x3a37] = "\xd0\x52",
  [0x3a38] = "\xd0\x53", [0x3a39] = "\xf3\xbc", [0x3a3a] = "\xd0\x54",
  [0x3a3b] = "\xd0\x55", [0x3a3c] = "\xf3\xbd", [0x3a3d] = "\xd0\x56",
  [0x3a3e] = "\xd0\x57", [0x3a3f] = "\xd0\x58", [0x3a40] = "\xd1\xaa",
  [0x3a41] = "\xd0\x59", [0x3a42] = "\xd0\x5a", [0x3a43] = "\xd0\x5b",
  [0x3a44] = "\xf4\xac", [0x3a45] = "\xd0\xc6", [0x3a46] = "\xd0\x5c",
  [0x3a47] = "\xd0\x5d", [0x3a48] = "\xd0\x5e", [0x3a49] = "\xd0\x5f",
  [0x3a4a] = "\xd0\x60", [0x3a4b] = "\xd0\x61", [0x3a4c] = "\xd0\xd0",
  [0x3a4d] = "\xd1\xdc", [0x3a4e] = "\xd0\x62", [0x3a4f] = "\xd0\x63",
  [0x3a50] = "\xd0\x64", [0x3a51] = "\xd0\x65", [0x3a52] = "\xd0\x66",
  [0x3a53] = "\xd0\x67", [0x3a54] = "\xcf\xce", [0x3a55] = "\xd0\x68",
  [0x3a56] = "\xd0\x69", [0x3a57] = "\xbd\xd6", [0x3a58] = "\xd0\x6a",
  [0x3a59] = "\xd1\xc3", [0x3a5a] = "\xd0\x6b", [0x3a5b] = "\xd0\x6c",
  [0x3a5c] = "\xd0\x6d", [0x3a5d] = "\xd0\x6e", [0x3a5e] = "\xd0\x6f",
  [0x3a5f] = "\xd0\x70", [0x3a60] = "\xd0\x71", [0x3a61] = "\xba\xe2",
  [0x3a62] = "\xe1\xe9", [0x3a63] = "\xd2\xc2", [0x3a64] = "\xf1\xc2",
  [0x3a65] = "\xb2\xb9", [0x3a66] = "\xd0\x72", [0x3a67] = "\xd0\x73",
  [0x3a68] = "\xb1\xed", [0x3a69] = "\xf1\xc3", [0x3a6a] = "\xd0\x74",
  [0x3a6b] = "\xc9\xc0", [0x3a6c] = "\xb3\xc4", [0x3a6d] = "\xd0\x75",
  [0x3a6e] = "\xd9\xf2", [0x3a6f] = "\xd0\x76", [0x3a70] = "\xcb\xa5",
  [0x3a71] = "\xd0\x77", [0x3a72] = "\xf1\xc4", [0x3a73] = "\xd0\x78",
  [0x3a74] = "\xd0\x79", [0x3a75] = "\xd0\x7a", [0x3a76] = "\xd0\x7b",
  [0x3a77] = "\xd6\xd4", [0x3a78] = "\xd0\x7c", [0x3a79] = "\xd0\x7d",
  [0x3a7a] = "\xd0\x7e", [0x3a7b] = "\xd0\x80", [0x3a7c] = "\xd0\x81",
  [0x3a7d] = "\xf1\xc5", [0x3a7e] = "\xf4\xc0", [0x3a7f] = "\xf1\xc6",
  [0x3a80] = "\xd0\x82", [0x3a81] = "\xd4\xac", [0x3a82] = "\xf1\xc7",
  [0x3a83] = "\xd0\x83", [0x3a84] = "\xb0\xc0", [0x3a85] = "\xf4\xc1",
  [0x3a86] = "\xd0\x84", [0x3a87] = "\xd0\x85", [0x3a88] = "\xf4\xc2",
  [0x3a89] = "\xd0\x86", [0x3a8a] = "\xd0\x87", [0x3a8b] = "\xb4\xfc",
  [0x3a8c] = "\xd0\x88", [0x3a8d] = "\xc5\xdb", [0x3a8e] = "\xd0\x89",
  [0x3a8f] = "\xd0\x8a", [0x3a90] = "\xd0\x8b", [0x3a91] = "\xd0\x8c",
  [0x3a92] = "\xcc\xbb", [0x3a93] = "\xd0\x8d", [0x3a94] = "\xd0\x8e",
  [0x3a95] = "\xd0\x8f", [0x3a96] = "\xd0\xe4", [0x3a97] = "\xd0\x90",
  [0x3a98] = "\xd0\x91", [0x3a99] = "\xd0\x92", [0x3a9a] = "\xd0\x93",
  [0x3a9b] = "\xd0\x94", [0x3a9c] = "\xcd\xe0", [0x3a9d] = "\xd0\x95",
  [0x3a9e] = "\xd0\x96", [0x3a9f] = "\xd0\x97", [0x3aa0] = "\xd0\x98",
  [0x3aa1] = "\xd0\x99", [0x3aa2] = "\xf1\xc8", [0x3aa3] = "\xd0\x9a",
  [0x3aa4] = "\xd9\xf3", [0x3aa5] = "\xd0\x9b", [0x3aa6] = "\xd0\x9c",
  [0x3aa7] = "\xd0\x9d", [0x3aa8] = "\xd0\x9e", [0x3aa9] = "\xd0\x9f",
  [0x3aaa] = "\xd0\xa0", [0x3aab] = "\xb1\xbb", [0x3aac] = "\xd1\x40",
  [0x3aad] = "\xcf\xae", [0x3aae] = "\xd1\x41", [0x3aaf] = "\xd1\x42",
  [0x3ab0] = "\xd1\x43", [0x3ab1] = "\xb8\xa4", [0x3ab2] = "\xd1\x44",
  [0x3ab3] = "\xd1\x45", [0x3ab4] = "\xd1\x46", [0x3ab5] = "\xd1\x47",
  [0x3ab6] = "\xd1\x48", [0x3ab7] = "\xf1\xca", [0x3ab8] = "\xd1\x49",
  [0x3ab9] = "\xd1\x4a", [0x3aba] = "\xd1\x4b", [0x3abb] = "\xd1\x4c",
  [0x3abc] = "\xf1\xcb", [0x3abd] = "\xd1\x4d", [0x3abe] = "\xd1\x4e",
  [0x3abf] = "\xd1\x4f", [0x3ac0] = "\xd1\x50", [0x3ac1] = "\xb2\xc3",
  [0x3ac2] = "\xc1\xd1", [0x3ac3] = "\xd1\x51", [0x3ac4] = "\xd1\x52",
  [0x3ac5] = "\xd7\xb0", [0x3ac6] = "\xf1\xc9", [0x3ac7] = "\xd1\x53",
  [0x3ac8] = "\xd1\x54", [0x3ac9] = "\xf1\xcc", [0x3aca] = "\xd1\x55",
  [0x3acb] = "\xd1\x56", [0x3acc] = "\xd1\x57", [0x3acd] = "\xd1\x58",
  [0x3ace] = "\xf1\xce", [0x3acf] = "\xd1\x59", [0x3ad0] = "\xd1\x5a",
  [0x3ad1] = "\xd1\x5b", [0x3ad2] = "\xd9\xf6", [0x3ad3] = "\xd1\x5c",
  [0x3ad4] = "\xd2\xe1", [0x3ad5] = "\xd4\xa3", [0x3ad6] = "\xd1\x5d",
  [0x3ad7] = "\xd1\x5e", [0x3ad8] = "\xf4\xc3", [0x3ad9] = "\xc8\xb9",
  [0x3ada] = "\xd1\x5f", [0x3adb] = "\xd1\x60", [0x3adc] = "\xd1\x61",
  [0x3add] = "\xd1\x62", [0x3ade] = "\xd1\x63", [0x3adf] = "\xf4\xc4",
  [0x3ae0] = "\xd1\x64", [0x3ae1] = "\xd1\x65", [0x3ae2] = "\xf1\xcd",
  [0x3ae3] = "\xf1\xcf", [0x3ae4] = "\xbf\xe3", [0x3ae5] = "\xf1\xd0",
  [0x3ae6] = "\xd1\x66", [0x3ae7] = "\xd1\x67", [0x3ae8] = "\xf1\xd4",
  [0x3ae9] = "\xd1\x68", [0x3aea] = "\xd1\x69", [0x3aeb] = "\xd1\x6a",
  [0x3aec] = "\xd1\x6b", [0x3aed] = "\xd1\x6c", [0x3aee] = "\xd1\x6d",
  [0x3aef] = "\xd1\x6e", [0x3af0] = "\xf1\xd6", [0x3af1] = "\xf1\xd1",
  [0x3af2] = "\xd1\x6f", [0x3af3] = "\xc9\xd1", [0x3af4] = "\xc5\xe1",
  [0x3af5] = "\xd1\x70", [0x3af6] = "\xd1\x71", [0x3af7] = "\xd1\x72",
  [0x3af8] = "\xc2\xe3", [0x3af9] = "\xb9\xfc", [0x3afa] = "\xd1\x73",
  [0x3afb] = "\xd1\x74", [0x3afc] = "\xf1\xd3", [0x3afd] = "\xd1\x75",
  [0x3afe] = "\xf1\xd5", [0x3aff] = "\xd1\x76", [0x3b00] = "\xd1\x77",
  [0x3b01] = "\xd1\x78", [0x3b02] = "\xb9\xd3", [0x3b03] = "\xd1\x79",
  [0x3b04] = "\xd1\x7a", [0x3b05] = "\xd1\x7b", [0x3b06] = "\xd1\x7c",
  [0x3b07] = "\xd1\x7d", [0x3b08] = "\xd1\x7e", [0x3b09] = "\xd1\x80",
  [0x3b0a] = "\xf1\xdb", [0x3b0b] = "\xd1\x81", [0x3b0c] = "\xd1\x82",
  [0x3b0d] = "\xd1\x83", [0x3b0e] = "\xd1\x84", [0x3b0f] = "\xd1\x85",
  [0x3b10] = "\xba\xd6", [0x3b11] = "\xd1\x86", [0x3b12] = "\xb0\xfd",
  [0x3b13] = "\xf1\xd9", [0x3b14] = "\xd1\x87", [0x3b15] = "\xd1\x88",
  [0x3b16] = "\xd1\x89", [0x3b17] = "\xd1\x8a", [0x3b18] = "\xd1\x8b",
  [0x3b19] = "\xf1\xd8", [0x3b1a] = "\xf1\xd2", [0x3b1b] = "\xf1\xda",
  [0x3b1c] = "\xd1\x8c", [0x3b1d] = "\xd1\x8d", [0x3b1e] = "\xd1\x8e",
  [0x3b1f] = "\xd1\x8f", [0x3b20] = "\xd1\x90", [0x3b21] = "\xf1\xd7",
  [0x3b22] = "\xd1\x91", [0x3b23] = "\xd1\x92", [0x3b24] = "\xd1\x93",
  [0x3b25] = "\xc8\xec", [0x3b26] = "\xd1\x94", [0x3b27] = "\xd1\x95",
  [0x3b28] = "\xd1\x96", [0x3b29] = "\xd1\x97", [0x3b2a] = "\xcd\xca",
  [0x3b2b] = "\xf1\xdd", [0x3b2c] = "\xd1\x98", [0x3b2d] = "\xd1\x99",
  [0x3b2e] = "\xd1\x9a", [0x3b2f] = "\xd1\x9b", [0x3b30] = "\xe5\xbd",
  [0x3b31] = "\xd1\x9c", [0x3b32] = "\xd1\x9d", [0x3b33] = "\xd1\x9e",
  [0x3b34] = "\xf1\xdc", [0x3b35] = "\xd1\x9f", [0x3b36] = "\xf1\xde",
  [0x3b37] = "\xd1\xa0", [0x3b38] = "\xd2\x40", [0x3b39] = "\xd2\x41",
  [0x3b3a] = "\xd2\x42", [0x3b3b] = "\xd2\x43", [0x3b3c] = "\xd2\x44",
  [0x3b3d] = "\xd2\x45", [0x3b3e] = "\xd2\x46", [0x3b3f] = "\xd2\x47",
  [0x3b40] = "\xd2\x48", [0x3b41] = "\xf1\xdf", [0x3b42] = "\xd2\x49",
  [0x3b43] = "\xd2\x4a", [0x3b44] = "\xcf\xe5", [0x3b45] = "\xd2\x4b",
  [0x3b46] = "\xd2\x4c", [0x3b47] = "\xd2\x4d", [0x3b48] = "\xd2\x4e",
  [0x3b49] = "\xd2\x4f", [0x3b4a] = "\xd2\x50", [0x3b4b] = "\xd2\x51",
  [0x3b4c] = "\xd2\x52", [0x3b4d] = "\xd2\x53", [0x3b4e] = "\xd2\x54",
  [0x3b4f] = "\xd2\x55", [0x3b50] = "\xd2\x56", [0x3b51] = "\xd2\x57",
  [0x3b52] = "\xd2\x58", [0x3b53] = "\xd2\x59", [0x3b54] = "\xd2\x5a",
  [0x3b55] = "\xd2\x5b", [0x3b56] = "\xd2\x5c", [0x3b57] = "\xd2\x5d",
  [0x3b58] = "\xd2\x5e", [0x3b59] = "\xd2\x5f", [0x3b5a] = "\xd2\x60",
  [0x3b5b] = "\xd2\x61", [0x3b5c] = "\xd2\x62", [0x3b5d] = "\xd2\x63",
  [0x3b5e] = "\xf4\xc5", [0x3b5f] = "\xbd\xf3", [0x3b60] = "\xd2\x64",
  [0x3b61] = "\xd2\x65", [0x3b62] = "\xd2\x66", [0x3b63] = "\xd2\x67",
  [0x3b64] = "\xd2\x68", [0x3b65] = "\xd2\x69", [0x3b66] = "\xf1\xe0",
  [0x3b67] = "\xd2\x6a", [0x3b68] = "\xd2\x6b", [0x3b69] = "\xd2\x6c",
  [0x3b6a] = "\xd2\x6d", [0x3b6b] = "\xd2\x6e", [0x3b6c] = "\xd2\x6f",
  [0x3b6d] = "\xd2\x70", [0x3b6e] = "\xd2\x71", [0x3b6f] = "\xd2\x72",
  [0x3b70] = "\xd2\x73", [0x3b71] = "\xd2\x74", [0x3b72] = "\xd2\x75",
  [0x3b73] = "\xd2\x76", [0x3b74] = "\xd2\x77", [0x3b75] = "\xd2\x78",
  [0x3b76] = "\xd2\x79", [0x3b77] = "\xd2\x7a", [0x3b78] = "\xd2\x7b",
  [0x3b79] = "\xd2\x7c", [0x3b7a] = "\xd2\x7d", [0x3b7b] = "\xf1\xe1",
  [0x3b7c] = "\xd2\x7e", [0x3b7d] = "\xd2\x80", [0x3b7e] = "\xd2\x81",
  [0x3b7f] = "\xce\xf7", [0x3b80] = "\xd2\x82", [0x3b81] = "\xd2\xaa",
  [0x3b82] = "\xd2\x83", [0x3b83] = "\xf1\xfb", [0x3b84] = "\xd2\x84",
  [0x3b85] = "\xd2\x85", [0x3b86] = "\xb8\xb2", [0x3b87] = "\xd2\x86",
  [0x3b88] = "\xd2\x87", [0x3b89] = "\xd2\x88", [0x3b8a] = "\xd2\x89",
  [0x3b8b] = "\xd2\x8a", [0x3b8c] = "\xd2\x8b", [0x3b8d] = "\xd2\x8c",
  [0x3b8e] = "\xd2\x8d", [0x3b8f] = "\xd2\x8e", [0x3b90] = "\xd2\x8f",
  [0x3b91] = "\xd2\x90", [0x3b92] = "\xd2\x91", [0x3b93] = "\xd2\x92",
  [0x3b94] = "\xd2\x93", [0x3b95] = "\xd2\x94", [0x3b96] = "\xd2\x95",
  [0x3b97] = "\xd2\x96", [0x3b98] = "\xd2\x97", [0x3b99] = "\xd2\x98",
  [0x3b9a] = "\xd2\x99", [0x3b9b] = "\xd2\x9a", [0x3b9c] = "\xd2\x9b",
  [0x3b9d] = "\xd2\x9c", [0x3b9e] = "\xd2\x9d", [0x3b9f] = "\xd2\x9e",
  [0x3ba0] = "\xd2\x9f", [0x3ba1] = "\xd2\xa0", [0x3ba2] = "\xd3\x40",
  [0x3ba3] = "\xd3\x41", [0x3ba4] = "\xd3\x42", [0x3ba5] = "\xd3\x43",
  [0x3ba6] = "\xd3\x44", [0x3ba7] = "\xd3\x45", [0x3ba8] = "\xd3\x46",
  [0x3ba9] = "\xd3\x47", [0x3baa] = "\xd3\x48", [0x3bab] = "\xd3\x49",
  [0x3bac] = "\xd3\x4a", [0x3bad] = "\xd3\x4b", [0x3bae] = "\xd3\x4c",
  [0x3baf] = "\xd3\x4d", [0x3bb0] = "\xd3\x4e", [0x3bb1] = "\xd3\x4f",
  [0x3bb2] = "\xd3\x50", [0x3bb3] = "\xd3\x51", [0x3bb4] = "\xd3\x52",
  [0x3bb5] = "\xd3\x53", [0x3bb6] = "\xd3\x54", [0x3bb7] = "\xd3\x55",
  [0x3bb8] = "\xd3\x56", [0x3bb9] = "\xd3\x57", [0x3bba] = "\xd3\x58",
  [0x3bbb] = "\xd3\x59", [0x3bbc] = "\xd3\x5a", [0x3bbd] = "\xd3\x5b",
  [0x3bbe] = "\xd3\x5c", [0x3bbf] = "\xd3\x5d", [0x3bc0] = "\xd3\x5e",
  [0x3bc1] = "\xbc\xfb", [0x3bc2] = "\xb9\xdb", [0x3bc3] = "\xd3\x5f",
  [0x3bc4] = "\xb9\xe6", [0x3bc5] = "\xc3\xd9", [0x3bc6] = "\xca\xd3",
  [0x3bc7] = "\xea\xe8", [0x3bc8] = "\xc0\xc0", [0x3bc9] = "\xbe\xf5",
  [0x3bca] = "\xea\xe9", [0x3bcb] = "\xea\xea", [0x3bcc] = "\xea\xeb",
  [0x3bcd] = "\xd3\x60", [0x3bce] = "\xea\xec", [0x3bcf] = "\xea\xed",
  [0x3bd0] = "\xea\xee", [0x3bd1] = "\xea\xef", [0x3bd2] = "\xbd\xc7",
  [0x3bd3] = "\xd3\x61", [0x3bd4] = "\xd3\x62", [0x3bd5] = "\xd3\x63",
  [0x3bd6] = "\xf5\xfb", [0x3bd7] = "\xd3\x64", [0x3bd8] = "\xd3\x65",
  [0x3bd9] = "\xd3\x66", [0x3bda] = "\xf5\xfd", [0x3bdb] = "\xd3\x67",
  [0x3bdc] = "\xf5\xfe", [0x3bdd] = "\xd3\x68", [0x3bde] = "\xf5\xfc",
  [0x3bdf] = "\xd3\x69", [0x3be0] = "\xd3\x6a", [0x3be1] = "\xd3\x6b",
  [0x3be2] = "\xd3\x6c", [0x3be3] = "\xbd\xe2", [0x3be4] = "\xd3\x6d",
  [0x3be5] = "\xf6\xa1", [0x3be6] = "\xb4\xa5", [0x3be7] = "\xd3\x6e",
  [0x3be8] = "\xd3\x6f", [0x3be9] = "\xd3\x70", [0x3bea] = "\xd3\x71",
  [0x3beb] = "\xf6\xa2", [0x3bec] = "\xd3\x72", [0x3bed] = "\xd3\x73",
  [0x3bee] = "\xd3\x74", [0x3bef] = "\xf6\xa3", [0x3bf0] = "\xd3\x75",
  [0x3bf1] = "\xd3\x76", [0x3bf2] = "\xd3\x77", [0x3bf3] = "\xec\xb2",
  [0x3bf4] = "\xd3\x78", [0x3bf5] = "\xd3\x79", [0x3bf6] = "\xd3\x7a",
  [0x3bf7] = "\xd3\x7b", [0x3bf8] = "\xd3\x7c", [0x3bf9] = "\xd3\x7d",
  [0x3bfa] = "\xd3\x7e", [0x3bfb] = "\xd3\x80", [0x3bfc] = "\xd3\x81",
  [0x3bfd] = "\xd3\x82", [0x3bfe] = "\xd3\x83", [0x3bff] = "\xd3\x84",
  [0x3c00] = "\xd1\xd4", [0x3c01] = "\xd3\x85", [0x3c02] = "\xd3\x86",
  [0x3c03] = "\xd3\x87", [0x3c04] = "\xd3\x88", [0x3c05] = "\xd3\x89",
  [0x3c06] = "\xd3\x8a", [0x3c07] = "\xd9\xea", [0x3c08] = "\xd3\x8b",
  [0x3c09] = "\xd3\x8c", [0x3c0a] = "\xd3\x8d", [0x3c0b] = "\xd3\x8e",
  [0x3c0c] = "\xd3\x8f", [0x3c0d] = "\xd3\x90", [0x3c0e] = "\xd3\x91",
  [0x3c0f] = "\xd3\x92", [0x3c10] = "\xd3\x93", [0x3c11] = "\xd3\x94",
  [0x3c12] = "\xd3\x95", [0x3c13] = "\xd3\x96", [0x3c14] = "\xd3\x97",
  [0x3c15] = "\xd3\x98", [0x3c16] = "\xd3\x99", [0x3c17] = "\xd3\x9a",
  [0x3c18] = "\xd3\x9b", [0x3c19] = "\xd3\x9c", [0x3c1a] = "\xd3\x9d",
  [0x3c1b] = "\xd3\x9e", [0x3c1c] = "\xd3\x9f", [0x3c1d] = "\xd3\xa0",
  [0x3c1e] = "\xd4\x40", [0x3c1f] = "\xd4\x41", [0x3c20] = "\xd4\x42",
  [0x3c21] = "\xd4\x43", [0x3c22] = "\xd4\x44", [0x3c23] = "\xd4\x45",
  [0x3c24] = "\xd4\x46", [0x3c25] = "\xd4\x47", [0x3c26] = "\xd4\x48",
  [0x3c27] = "\xd4\x49", [0x3c28] = "\xd4\x4a", [0x3c29] = "\xd4\x4b",
  [0x3c2a] = "\xd4\x4c", [0x3c2b] = "\xd4\x4d", [0x3c2c] = "\xd4\x4e",
  [0x3c2d] = "\xd4\x4f", [0x3c2e] = "\xd4\x50", [0x3c2f] = "\xd4\x51",
  [0x3c30] = "\xd4\x52", [0x3c31] = "\xd4\x53", [0x3c32] = "\xd4\x54",
  [0x3c33] = "\xd4\x55", [0x3c34] = "\xd4\x56", [0x3c35] = "\xd4\x57",
  [0x3c36] = "\xd4\x58", [0x3c37] = "\xd4\x59", [0x3c38] = "\xd4\x5a",
  [0x3c39] = "\xd4\x5b", [0x3c3a] = "\xd4\x5c", [0x3c3b] = "\xd4\x5d",
  [0x3c3c] = "\xd4\x5e", [0x3c3d] = "\xd4\x5f", [0x3c3e] = "\xf6\xa4",
  [0x3c3f] = "\xd4\x60", [0x3c40] = "\xd4\x61", [0x3c41] = "\xd4\x62",
  [0x3c42] = "\xd4\x63", [0x3c43] = "\xd4\x64", [0x3c44] = "\xd4\x65",
  [0x3c45] = "\xd4\x66", [0x3c46] = "\xd4\x67", [0x3c47] = "\xd4\x68",
  [0x3c48] = "\xee\xba", [0x3c49] = "\xd4\x69", [0x3c4a] = "\xd4\x6a",
  [0x3c4b] = "\xd4\x6b", [0x3c4c] = "\xd4\x6c", [0x3c4d] = "\xd4\x6d",
  [0x3c4e] = "\xd4\x6e", [0x3c4f] = "\xd4\x6f", [0x3c50] = "\xd4\x70",
  [0x3c51] = "\xd4\x71", [0x3c52] = "\xd4\x72", [0x3c53] = "\xd4\x73",
  [0x3c54] = "\xd4\x74", [0x3c55] = "\xd4\x75", [0x3c56] = "\xd4\x76",
  [0x3c57] = "\xd4\x77", [0x3c58] = "\xd4\x78", [0x3c59] = "\xd4\x79",
  [0x3c5a] = "\xd4\x7a", [0x3c5b] = "\xd4\x7b", [0x3c5c] = "\xd4\x7c",
  [0x3c5d] = "\xd4\x7d", [0x3c5e] = "\xd4\x7e", [0x3c5f] = "\xd4\x80",
  [0x3c60] = "\xd4\x81", [0x3c61] = "\xd4\x82", [0x3c62] = "\xd4\x83",
  [0x3c63] = "\xd4\x84", [0x3c64] = "\xd4\x85", [0x3c65] = "\xd4\x86",
  [0x3c66] = "\xd4\x87", [0x3c67] = "\xd4\x88", [0x3c68] = "\xd4\x89",
  [0x3c69] = "\xd4\x8a", [0x3c6a] = "\xd4\x8b", [0x3c6b] = "\xd4\x8c",
  [0x3c6c] = "\xd4\x8d", [0x3c6d] = "\xd4\x8e", [0x3c6e] = "\xd4\x8f",
  [0x3c6f] = "\xd4\x90", [0x3c70] = "\xd4\x91", [0x3c71] = "\xd4\x92",
  [0x3c72] = "\xd4\x93", [0x3c73] = "\xd4\x94", [0x3c74] = "\xd4\x95",
  [0x3c75] = "\xd4\x96", [0x3c76] = "\xd4\x97", [0x3c77] = "\xd4\x98",
  [0x3c78] = "\xd4\x99", [0x3c79] = "\xd5\xb2", [0x3c7a] = "\xd4\x9a",
  [0x3c7b] = "\xd4\x9b", [0x3c7c] = "\xd4\x9c", [0x3c7d] = "\xd4\x9d",
  [0x3c7e] = "\xd4\x9e", [0x3c7f] = "\xd4\x9f", [0x3c80] = "\xd4\xa0",
  [0x3c81] = "\xd5\x40", [0x3c82] = "\xd5\x41", [0x3c83] = "\xd5\x42",
  [0x3c84] = "\xd5\x43", [0x3c85] = "\xd5\x44", [0x3c86] = "\xd5\x45",
  [0x3c87] = "\xd5\x46", [0x3c88] = "\xd5\x47", [0x3c89] = "\xd3\xfe",
  [0x3c8a] = "\xcc\xdc", [0x3c8b] = "\xd5\x48", [0x3c8c] = "\xd5\x49",
  [0x3c8d] = "\xd5\x4a", [0x3c8e] = "\xd5\x4b", [0x3c8f] = "\xd5\x4c",
  [0x3c90] = "\xd5\x4d", [0x3c91] = "\xd5\x4e", [0x3c92] = "\xd5\x4f",
  [0x3c93] = "\xca\xc4", [0x3c94] = "\xd5\x50", [0x3c95] = "\xd5\x51",
  [0x3c96] = "\xd5\x52", [0x3c97] = "\xd5\x53", [0x3c98] = "\xd5\x54",
  [0x3c99] = "\xd5\x55", [0x3c9a] = "\xd5\x56", [0x3c9b] = "\xd5\x57",
  [0x3c9c] = "\xd5\x58", [0x3c9d] = "\xd5\x59", [0x3c9e] = "\xd5\x5a",
  [0x3c9f] = "\xd5\x5b", [0x3ca0] = "\xd5\x5c", [0x3ca1] = "\xd5\x5d",
  [0x3ca2] = "\xd5\x5e", [0x3ca3] = "\xd5\x5f", [0x3ca4] = "\xd5\x60",
  [0x3ca5] = "\xd5\x61", [0x3ca6] = "\xd5\x62", [0x3ca7] = "\xd5\x63",
  [0x3ca8] = "\xd5\x64", [0x3ca9] = "\xd5\x65", [0x3caa] = "\xd5\x66",
  [0x3cab] = "\xd5\x67", [0x3cac] = "\xd5\x68", [0x3cad] = "\xd5\x69",
  [0x3cae] = "\xd5\x6a", [0x3caf] = "\xd5\x6b", [0x3cb0] = "\xd5\x6c",
  [0x3cb1] = "\xd5\x6d", [0x3cb2] = "\xd5\x6e", [0x3cb3] = "\xd5\x6f",
  [0x3cb4] = "\xd5\x70", [0x3cb5] = "\xd5\x71", [0x3cb6] = "\xd5\x72",
  [0x3cb7] = "\xd5\x73", [0x3cb8] = "\xd5\x74", [0x3cb9] = "\xd5\x75",
  [0x3cba] = "\xd5\x76", [0x3cbb] = "\xd5\x77", [0x3cbc] = "\xd5\x78",
  [0x3cbd] = "\xd5\x79", [0x3cbe] = "\xd5\x7a", [0x3cbf] = "\xd5\x7b",
  [0x3cc0] = "\xd5\x7c", [0x3cc1] = "\xd5\x7d", [0x3cc2] = "\xd5\x7e",
  [0x3cc3] = "\xd5\x80", [0x3cc4] = "\xd5\x81", [0x3cc5] = "\xd5\x82",
  [0x3cc6] = "\xd5\x83", [0x3cc7] = "\xd5\x84", [0x3cc8] = "\xd5\x85",
  [0x3cc9] = "\xd5\x86", [0x3cca] = "\xd5\x87", [0x3ccb] = "\xd5\x88",
  [0x3ccc] = "\xd5\x89", [0x3ccd] = "\xd5\x8a", [0x3cce] = "\xd5\x8b",
  [0x3ccf] = "\xd5\x8c", [0x3cd0] = "\xd5\x8d", [0x3cd1] = "\xd5\x8e",
  [0x3cd2] = "\xd5\x8f", [0x3cd3] = "\xd5\x90", [0x3cd4] = "\xd5\x91",
  [0x3cd5] = "\xd5\x92", [0x3cd6] = "\xd5\x93", [0x3cd7] = "\xd5\x94",
  [0x3cd8] = "\xd5\x95", [0x3cd9] = "\xd5\x96", [0x3cda] = "\xd5\x97",
  [0x3cdb] = "\xd5\x98", [0x3cdc] = "\xd5\x99", [0x3cdd] = "\xd5\x9a",
  [0x3cde] = "\xd5\x9b", [0x3cdf] = "\xd5\x9c", [0x3ce0] = "\xd5\x9d",
  [0x3ce1] = "\xd5\x9e", [0x3ce2] = "\xd5\x9f", [0x3ce3] = "\xd5\xa0",
  [0x3ce4] = "\xd6\x40", [0x3ce5] = "\xd6\x41", [0x3ce6] = "\xd6\x42",
  [0x3ce7] = "\xd6\x43", [0x3ce8] = "\xd6\x44", [0x3ce9] = "\xd6\x45",
  [0x3cea] = "\xd6\x46", [0x3ceb] = "\xd6\x47", [0x3cec] = "\xd6\x48",
  [0x3ced] = "\xd6\x49", [0x3cee] = "\xd6\x4a", [0x3cef] = "\xd6\x4b",
  [0x3cf0] = "\xd6\x4c", [0x3cf1] = "\xd6\x4d", [0x3cf2] = "\xd6\x4e",
  [0x3cf3] = "\xd6\x4f", [0x3cf4] = "\xd6\x50", [0x3cf5] = "\xd6\x51",
  [0x3cf6] = "\xd6\x52", [0x3cf7] = "\xd6\x53", [0x3cf8] = "\xd6\x54",
  [0x3cf9] = "\xd6\x55", [0x3cfa] = "\xd6\x56", [0x3cfb] = "\xd6\x57",
  [0x3cfc] = "\xd6\x58", [0x3cfd] = "\xd6\x59", [0x3cfe] = "\xd6\x5a",
  [0x3cff] = "\xd6\x5b", [0x3d00] = "\xd6\x5c", [0x3d01] = "\xd6\x5d",
  [0x3d02] = "\xd6\x5e", [0x3d03] = "\xd6\x5f", [0x3d04] = "\xd6\x60",
  [0x3d05] = "\xd6\x61", [0x3d06] = "\xd6\x62", [0x3d07] = "\xe5\xc0",
  [0x3d08] = "\xd6\x63", [0x3d09] = "\xd6\x64", [0x3d0a] = "\xd6\x65",
  [0x3d0b] = "\xd6\x66", [0x3d0c] = "\xd6\x67", [0x3d0d] = "\xd6\x68",
  [0x3d0e] = "\xd6\x69", [0x3d0f] = "\xd6\x6a", [0x3d10] = "\xd6\x6b",
  [0x3d11] = "\xd6\x6c", [0x3d12] = "\xd6\x6d", [0x3d13] = "\xd6\x6e",
  [0x3d14] = "\xd6\x6f", [0x3d15] = "\xd6\x70", [0x3d16] = "\xd6\x71",
  [0x3d17] = "\xd6\x72", [0x3d18] = "\xd6\x73", [0x3d19] = "\xd6\x74",
  [0x3d1a] = "\xd6\x75", [0x3d1b] = "\xd6\x76", [0x3d1c] = "\xd6\x77",
  [0x3d1d] = "\xd6\x78", [0x3d1e] = "\xd6\x79", [0x3d1f] = "\xd6\x7a",
  [0x3d20] = "\xd6\x7b", [0x3d21] = "\xd6\x7c", [0x3d22] = "\xd6\x7d",
  [0x3d23] = "\xd6\x7e", [0x3d24] = "\xd6\x80", [0x3d25] = "\xd6\x81",
  [0x3d26] = "\xf6\xa5", [0x3d27] = "\xd6\x82", [0x3d28] = "\xd6\x83",
  [0x3d29] = "\xd6\x84", [0x3d2a] = "\xd6\x85", [0x3d2b] = "\xd6\x86",
  [0x3d2c] = "\xd6\x87", [0x3d2d] = "\xd6\x88", [0x3d2e] = "\xd6\x89",
  [0x3d2f] = "\xd6\x8a", [0x3d30] = "\xd6\x8b", [0x3d31] = "\xd6\x8c",
  [0x3d32] = "\xd6\x8d", [0x3d33] = "\xd6\x8e", [0x3d34] = "\xd6\x8f",
  [0x3d35] = "\xd6\x90", [0x3d36] = "\xd6\x91", [0x3d37] = "\xd6\x92",
  [0x3d38] = "\xd6\x93", [0x3d39] = "\xd6\x94", [0x3d3a] = "\xd6\x95",
  [0x3d3b] = "\xd6\x96", [0x3d3c] = "\xd6\x97", [0x3d3d] = "\xd6\x98",
  [0x3d3e] = "\xd6\x99", [0x3d3f] = "\xd6\x9a", [0x3d40] = "\xd6\x9b",
  [0x3d41] = "\xd6\x9c", [0x3d42] = "\xd6\x9d", [0x3d43] = "\xd6\x9e",
  [0x3d44] = "\xd6\x9f", [0x3d45] = "\xd6\xa0", [0x3d46] = "\xd7\x40",
  [0x3d47] = "\xd7\x41", [0x3d48] = "\xd7\x42", [0x3d49] = "\xd7\x43",
  [0x3d4a] = "\xd7\x44", [0x3d4b] = "\xd7\x45", [0x3d4c] = "\xd7\x46",
  [0x3d4d] = "\xd7\x47", [0x3d4e] = "\xd7\x48", [0x3d4f] = "\xd7\x49",
  [0x3d50] = "\xd7\x4a", [0x3d51] = "\xd7\x4b", [0x3d52] = "\xd7\x4c",
  [0x3d53] = "\xd7\x4d", [0x3d54] = "\xd7\x4e", [0x3d55] = "\xd7\x4f",
  [0x3d56] = "\xd7\x50", [0x3d57] = "\xd7\x51", [0x3d58] = "\xd7\x52",
  [0x3d59] = "\xd7\x53", [0x3d5a] = "\xd7\x54", [0x3d5b] = "\xd7\x55",
  [0x3d5c] = "\xd7\x56", [0x3d5d] = "\xd7\x57", [0x3d5e] = "\xd7\x58",
  [0x3d5f] = "\xd7\x59", [0x3d60] = "\xd7\x5a", [0x3d61] = "\xd7\x5b",
  [0x3d62] = "\xd7\x5c", [0x3d63] = "\xd7\x5d", [0x3d64] = "\xd7\x5e",
  [0x3d65] = "\xd7\x5f", [0x3d66] = "\xbe\xaf", [0x3d67] = "\xd7\x60",
  [0x3d68] = "\xd7\x61", [0x3d69] = "\xd7\x62", [0x3d6a] = "\xd7\x63",
  [0x3d6b] = "\xd7\x64", [0x3d6c] = "\xc6\xa9", [0x3d6d] = "\xd7\x65",
  [0x3d6e] = "\xd7\x66", [0x3d6f] = "\xd7\x67", [0x3d70] = "\xd7\x68",
  [0x3d71] = "\xd7\x69", [0x3d72] = "\xd7\x6a", [0x3d73] = "\xd7\x6b",
  [0x3d74] = "\xd7\x6c", [0x3d75] = "\xd7\x6d", [0x3d76] = "\xd7\x6e",
  [0x3d77] = "\xd7\x6f", [0x3d78] = "\xd7\x70", [0x3d79] = "\xd7\x71",
  [0x3d7a] = "\xd7\x72", [0x3d7b] = "\xd7\x73", [0x3d7c] = "\xd7\x74",
  [0x3d7d] = "\xd7\x75", [0x3d7e] = "\xd7\x76", [0x3d7f] = "\xd7\x77",
  [0x3d80] = "\xd7\x78", [0x3d81] = "\xd7\x79", [0x3d82] = "\xd7\x7a",
  [0x3d83] = "\xd7\x7b", [0x3d84] = "\xd7\x7c", [0x3d85] = "\xd7\x7d",
  [0x3d86] = "\xd7\x7e", [0x3d87] = "\xd7\x80", [0x3d88] = "\xd7\x81",
  [0x3d89] = "\xd7\x82", [0x3d8a] = "\xd7\x83", [0x3d8b] = "\xd7\x84",
  [0x3d8c] = "\xd7\x85", [0x3d8d] = "\xd7\x86", [0x3d8e] = "\xd7\x87",
  [0x3d8f] = "\xd7\x88", [0x3d90] = "\xd7\x89", [0x3d91] = "\xd7\x8a",
  [0x3d92] = "\xd7\x8b", [0x3d93] = "\xd7\x8c", [0x3d94] = "\xd7\x8d",
  [0x3d95] = "\xd7\x8e", [0x3d96] = "\xd7\x8f", [0x3d97] = "\xd7\x90",
  [0x3d98] = "\xd7\x91", [0x3d99] = "\xd7\x92", [0x3d9a] = "\xd7\x93",
  [0x3d9b] = "\xd7\x94", [0x3d9c] = "\xd7\x95", [0x3d9d] = "\xd7\x96",
  [0x3d9e] = "\xd7\x97", [0x3d9f] = "\xd7\x98", [0x3da0] = "\xda\xa5",
  [0x3da1] = "\xbc\xc6", [0x3da2] = "\xb6\xa9", [0x3da3] = "\xb8\xbc",
  [0x3da4] = "\xc8\xcf", [0x3da5] = "\xbc\xa5", [0x3da6] = "\xda\xa6",
  [0x3da7] = "\xda\xa7", [0x3da8] = "\xcc\xd6", [0x3da9] = "\xc8\xc3",
  [0x3daa] = "\xda\xa8", [0x3dab] = "\xc6\xfd", [0x3dac] = "\xd7\x99",
  [0x3dad] = "\xd1\xb5", [0x3dae] = "\xd2\xe9", [0x3daf] = "\xd1\xb6",
  [0x3db0] = "\xbc\xc7", [0x3db1] = "\xd7\x9a", [0x3db2] = "\xbd\xb2",
  [0x3db3] = "\xbb\xe4", [0x3db4] = "\xda\xa9", [0x3db5] = "\xda\xaa",
  [0x3db6] = "\xd1\xc8", [0x3db7] = "\xda\xab", [0x3db8] = "\xd0\xed",
  [0x3db9] = "\xb6\xef", [0x3dba] = "\xc2\xdb", [0x3dbb] = "\xd7\x9b",
  [0x3dbc] = "\xcb\xcf", [0x3dbd] = "\xb7\xed", [0x3dbe] = "\xc9\xe8",
  [0x3dbf] = "\xb7\xc3", [0x3dc0] = "\xbe\xf7", [0x3dc1] = "\xd6\xa4",
  [0x3dc2] = "\xda\xac", [0x3dc3] = "\xda\xad", [0x3dc4] = "\xc6\xc0",
  [0x3dc5] = "\xd7\xe7", [0x3dc6] = "\xca\xb6", [0x3dc7] = "\xd7\x9c",
  [0x3dc8] = "\xd5\xa9", [0x3dc9] = "\xcb\xdf", [0x3dca] = "\xd5\xef",
  [0x3dcb] = "\xda\xae", [0x3dcc] = "\xd6\xdf", [0x3dcd] = "\xb4\xca",
  [0x3dce] = "\xda\xb0", [0x3dcf] = "\xda\xaf", [0x3dd0] = "\xd7\x9d",
  [0x3dd1] = "\xd2\xeb", [0x3dd2] = "\xda\xb1", [0x3dd3] = "\xda\xb2",
  [0x3dd4] = "\xda\xb3", [0x3dd5] = "\xca\xd4", [0x3dd6] = "\xda\xb4",
  [0x3dd7] = "\xca\xab", [0x3dd8] = "\xda\xb5", [0x3dd9] = "\xda\xb6",
  [0x3dda] = "\xb3\xcf", [0x3ddb] = "\xd6\xef", [0x3ddc] = "\xda\xb7",
  [0x3ddd] = "\xbb\xb0", [0x3dde] = "\xb5\xae", [0x3ddf] = "\xda\xb8",
  [0x3de0] = "\xda\xb9", [0x3de1] = "\xb9\xee", [0x3de2] = "\xd1\xaf",
  [0x3de3] = "\xd2\xe8", [0x3de4] = "\xda\xba", [0x3de5] = "\xb8\xc3",
  [0x3de6] = "\xcf\xea", [0x3de7] = "\xb2\xef", [0x3de8] = "\xda\xbb",
  [0x3de9] = "\xda\xbc", [0x3dea] = "\xd7\x9e", [0x3deb] = "\xbd\xeb",
  [0x3dec] = "\xce\xdc", [0x3ded] = "\xd3\xef", [0x3dee] = "\xda\xbd",
  [0x3def] = "\xce\xf3", [0x3df0] = "\xda\xbe", [0x3df1] = "\xd3\xd5",
  [0x3df2] = "\xbb\xe5", [0x3df3] = "\xda\xbf", [0x3df4] = "\xcb\xb5",
  [0x3df5] = "\xcb\xd0", [0x3df6] = "\xda\xc0", [0x3df7] = "\xc7\xeb",
  [0x3df8] = "\xd6\xee", [0x3df9] = "\xda\xc1", [0x3dfa] = "\xc5\xb5",
  [0x3dfb] = "\xb6\xc1", [0x3dfc] = "\xda\xc2", [0x3dfd] = "\xb7\xcc",
  [0x3dfe] = "\xbf\xce", [0x3dff] = "\xda\xc3", [0x3e00] = "\xda\xc4",
  [0x3e01] = "\xcb\xad", [0x3e02] = "\xda\xc5", [0x3e03] = "\xb5\xf7",
  [0x3e04] = "\xda\xc6", [0x3e05] = "\xc1\xc2", [0x3e06] = "\xd7\xbb",
  [0x3e07] = "\xda\xc7", [0x3e08] = "\xcc\xb8", [0x3e09] = "\xd7\x9f",
  [0x3e0a] = "\xd2\xea", [0x3e0b] = "\xc4\xb1", [0x3e0c] = "\xda\xc8",
  [0x3e0d] = "\xb5\xfd", [0x3e0e] = "\xbb\xd1", [0x3e0f] = "\xda\xc9",
  [0x3e10] = "\xd0\xb3", [0x3e11] = "\xda\xca", [0x3e12] = "\xda\xcb",
  [0x3e13] = "\xce\xbd", [0x3e14] = "\xda\xcc", [0x3e15] = "\xda\xcd",
  [0x3e16] = "\xda\xce", [0x3e17] = "\xb2\xf7", [0x3e18] = "\xda\xd1",
  [0x3e19] = "\xda\xcf", [0x3e1a] = "\xd1\xe8", [0x3e1b] = "\xda\xd0",
  [0x3e1c] = "\xc3\xd5", [0x3e1d] = "\xda\xd2", [0x3e1e] = "\xd7\xa0",
  [0x3e1f] = "\xda\xd3", [0x3e20] = "\xda\xd4", [0x3e21] = "\xda\xd5",
  [0x3e22] = "\xd0\xbb", [0x3e23] = "\xd2\xa5", [0x3e24] = "\xb0\xf9",
  [0x3e25] = "\xda\xd6", [0x3e26] = "\xc7\xab", [0x3e27] = "\xda\xd7",
  [0x3e28] = "\xbd\xf7", [0x3e29] = "\xc3\xa1", [0x3e2a] = "\xda\xd8",
  [0x3e2b] = "\xda\xd9", [0x3e2c] = "\xc3\xfd", [0x3e2d] = "\xcc\xb7",
  [0x3e2e] = "\xda\xda", [0x3e2f] = "\xda\xdb", [0x3e30] = "\xc0\xbe",
  [0x3e31] = "\xc6\xd7", [0x3e32] = "\xda\xdc", [0x3e33] = "\xda\xdd",
  [0x3e34] = "\xc7\xb4", [0x3e35] = "\xda\xde", [0x3e36] = "\xda\xdf",
  [0x3e37] = "\xb9\xc8", [0x3e38] = "\xd8\x40", [0x3e39] = "\xd8\x41",
  [0x3e3a] = "\xd8\x42", [0x3e3b] = "\xd8\x43", [0x3e3c] = "\xd8\x44",
  [0x3e3d] = "\xd8\x45", [0x3e3e] = "\xd8\x46", [0x3e3f] = "\xd8\x47",
  [0x3e40] = "\xd8\x48", [0x3e41] = "\xbb\xed", [0x3e42] = "\xd8\x49",
  [0x3e43] = "\xd8\x4a", [0x3e44] = "\xd8\x4b", [0x3e45] = "\xd8\x4c",
  [0x3e46] = "\xb6\xb9", [0x3e47] = "\xf4\xf8", [0x3e48] = "\xd8\x4d",
  [0x3e49] = "\xf4\xf9", [0x3e4a] = "\xd8\x4e", [0x3e4b] = "\xd8\x4f",
  [0x3e4c] = "\xcd\xe3", [0x3e4d] = "\xd8\x50", [0x3e4e] = "\xd8\x51",
  [0x3e4f] = "\xd8\x52", [0x3e50] = "\xd8\x53", [0x3e51] = "\xd8\x54",
  [0x3e52] = "\xd8\x55", [0x3e53] = "\xd8\x56", [0x3e54] = "\xd8\x57",
  [0x3e55] = "\xf5\xb9", [0x3e56] = "\xd8\x58", [0x3e57] = "\xd8\x59",
  [0x3e58] = "\xd8\x5a", [0x3e59] = "\xd8\x5b", [0x3e5a] = "\xeb\xe0",
  [0x3e5b] = "\xd8\x5c", [0x3e5c] = "\xd8\x5d", [0x3e5d] = "\xd8\x5e",
  [0x3e5e] = "\xd8\x5f", [0x3e5f] = "\xd8\x60", [0x3e60] = "\xd8\x61",
  [0x3e61] = "\xcf\xf3", [0x3e62] = "\xbb\xbf", [0x3e63] = "\xd8\x62",
  [0x3e64] = "\xd8\x63", [0x3e65] = "\xd8\x64", [0x3e66] = "\xd8\x65",
  [0x3e67] = "\xd8\x66", [0x3e68] = "\xd8\x67", [0x3e69] = "\xd8\x68",
  [0x3e6a] = "\xba\xc0", [0x3e6b] = "\xd4\xa5", [0x3e6c] = "\xd8\x69",
  [0x3e6d] = "\xd8\x6a", [0x3e6e] = "\xd8\x6b", [0x3e6f] = "\xd8\x6c",
  [0x3e70] = "\xd8\x6d", [0x3e71] = "\xd8\x6e", [0x3e72] = "\xd8\x6f",
  [0x3e73] = "\xe1\xd9", [0x3e74] = "\xd8\x70", [0x3e75] = "\xd8\x71",
  [0x3e76] = "\xd8\x72", [0x3e77] = "\xd8\x73", [0x3e78] = "\xf5\xf4",
  [0x3e79] = "\xb1\xaa", [0x3e7a] = "\xb2\xf2", [0x3e7b] = "\xd8\x74",
  [0x3e7c] = "\xd8\x75", [0x3e7d] = "\xd8\x76", [0x3e7e] = "\xd8\x77",
  [0x3e7f] = "\xd8\x78", [0x3e80] = "\xd8\x79", [0x3e81] = "\xd8\x7a",
  [0x3e82] = "\xf5\xf5", [0x3e83] = "\xd8\x7b", [0x3e84] = "\xd8\x7c",
  [0x3e85] = "\xf5\xf7", [0x3e86] = "\xd8\x7d", [0x3e87] = "\xd8\x7e",
  [0x3e88] = "\xd8\x80", [0x3e89] = "\xba\xd1", [0x3e8a] = "\xf5\xf6",
  [0x3e8b] = "\xd8\x81", [0x3e8c] = "\xc3\xb2", [0x3e8d] = "\xd8\x82",
  [0x3e8e] = "\xd8\x83", [0x3e8f] = "\xd8\x84", [0x3e90] = "\xd8\x85",
  [0x3e91] = "\xd8\x86", [0x3e92] = "\xd8\x87", [0x3e93] = "\xd8\x88",
  [0x3e94] = "\xf5\xf9", [0x3e95] = "\xd8\x89", [0x3e96] = "\xd8\x8a",
  [0x3e97] = "\xd8\x8b", [0x3e98] = "\xf5\xf8", [0x3e99] = "\xd8\x8c",
  [0x3e9a] = "\xd8\x8d", [0x3e9b] = "\xd8\x8e", [0x3e9c] = "\xd8\x8f",
  [0x3e9d] = "\xd8\x90", [0x3e9e] = "\xd8\x91", [0x3e9f] = "\xd8\x92",
  [0x3ea0] = "\xd8\x93", [0x3ea1] = "\xd8\x94", [0x3ea2] = "\xd8\x95",
  [0x3ea3] = "\xd8\x96", [0x3ea4] = "\xd8\x97", [0x3ea5] = "\xd8\x98",
  [0x3ea6] = "\xd8\x99", [0x3ea7] = "\xd8\x9a", [0x3ea8] = "\xd8\x9b",
  [0x3ea9] = "\xd8\x9c", [0x3eaa] = "\xd8\x9d", [0x3eab] = "\xd8\x9e",
  [0x3eac] = "\xd8\x9f", [0x3ead] = "\xd8\xa0", [0x3eae] = "\xd9\x40",
  [0x3eaf] = "\xd9\x41", [0x3eb0] = "\xd9\x42", [0x3eb1] = "\xd9\x43",
  [0x3eb2] = "\xd9\x44", [0x3eb3] = "\xd9\x45", [0x3eb4] = "\xd9\x46",
  [0x3eb5] = "\xd9\x47", [0x3eb6] = "\xd9\x48", [0x3eb7] = "\xd9\x49",
  [0x3eb8] = "\xd9\x4a", [0x3eb9] = "\xd9\x4b", [0x3eba] = "\xd9\x4c",
  [0x3ebb] = "\xd9\x4d", [0x3ebc] = "\xd9\x4e", [0x3ebd] = "\xd9\x4f",
  [0x3ebe] = "\xd9\x50", [0x3ebf] = "\xd9\x51", [0x3ec0] = "\xd9\x52",
  [0x3ec1] = "\xd9\x53", [0x3ec2] = "\xd9\x54", [0x3ec3] = "\xd9\x55",
  [0x3ec4] = "\xd9\x56", [0x3ec5] = "\xd9\x57", [0x3ec6] = "\xd9\x58",
  [0x3ec7] = "\xd9\x59", [0x3ec8] = "\xd9\x5a", [0x3ec9] = "\xd9\x5b",
  [0x3eca] = "\xd9\x5c", [0x3ecb] = "\xd9\x5d", [0x3ecc] = "\xd9\x5e",
  [0x3ecd] = "\xd9\x5f", [0x3ece] = "\xd9\x60", [0x3ecf] = "\xd9\x61",
  [0x3ed0] = "\xd9\x62", [0x3ed1] = "\xd9\x63", [0x3ed2] = "\xd9\x64",
  [0x3ed3] = "\xd9\x65", [0x3ed4] = "\xd9\x66", [0x3ed5] = "\xd9\x67",
  [0x3ed6] = "\xd9\x68", [0x3ed7] = "\xd9\x69", [0x3ed8] = "\xd9\x6a",
  [0x3ed9] = "\xd9\x6b", [0x3eda] = "\xd9\x6c", [0x3edb] = "\xd9\x6d",
  [0x3edc] = "\xd9\x6e", [0x3edd] = "\xd9\x6f", [0x3ede] = "\xd9\x70",
  [0x3edf] = "\xd9\x71", [0x3ee0] = "\xd9\x72", [0x3ee1] = "\xd9\x73",
  [0x3ee2] = "\xd9\x74", [0x3ee3] = "\xd9\x75", [0x3ee4] = "\xd9\x76",
  [0x3ee5] = "\xd9\x77", [0x3ee6] = "\xd9\x78", [0x3ee7] = "\xd9\x79",
  [0x3ee8] = "\xd9\x7a", [0x3ee9] = "\xd9\x7b", [0x3eea] = "\xd9\x7c",
  [0x3eeb] = "\xd9\x7d", [0x3eec] = "\xd9\x7e", [0x3eed] = "\xd9\x80",
  [0x3eee] = "\xd9\x81", [0x3eef] = "\xd9\x82", [0x3ef0] = "\xd9\x83",
  [0x3ef1] = "\xd9\x84", [0x3ef2] = "\xd9\x85", [0x3ef3] = "\xd9\x86",
  [0x3ef4] = "\xd9\x87", [0x3ef5] = "\xd9\x88", [0x3ef6] = "\xd9\x89",
  [0x3ef7] = "\xd9\x8a", [0x3ef8] = "\xd9\x8b", [0x3ef9] = "\xd9\x8c",
  [0x3efa] = "\xd9\x8d", [0x3efb] = "\xd9\x8e", [0x3efc] = "\xd9\x8f",
  [0x3efd] = "\xd9\x90", [0x3efe] = "\xd9\x91", [0x3eff] = "\xd9\x92",
  [0x3f00] = "\xd9\x93", [0x3f01] = "\xd9\x94", [0x3f02] = "\xd9\x95",
  [0x3f03] = "\xd9\x96", [0x3f04] = "\xd9\x97", [0x3f05] = "\xd9\x98",
  [0x3f06] = "\xd9\x99", [0x3f07] = "\xd9\x9a", [0x3f08] = "\xd9\x9b",
  [0x3f09] = "\xd9\x9c", [0x3f0a] = "\xd9\x9d", [0x3f0b] = "\xd9\x9e",
  [0x3f0c] = "\xd9\x9f", [0x3f0d] = "\xd9\xa0", [0x3f0e] = "\xda\x40",
  [0x3f0f] = "\xda\x41", [0x3f10] = "\xda\x42", [0x3f11] = "\xda\x43",
  [0x3f12] = "\xda\x44", [0x3f13] = "\xda\x45", [0x3f14] = "\xda\x46",
  [0x3f15] = "\xda\x47", [0x3f16] = "\xda\x48", [0x3f17] = "\xda\x49",
  [0x3f18] = "\xda\x4a", [0x3f19] = "\xda\x4b", [0x3f1a] = "\xda\x4c",
  [0x3f1b] = "\xda\x4d", [0x3f1c] = "\xda\x4e", [0x3f1d] = "\xb1\xb4",
  [0x3f1e] = "\xd5\xea", [0x3f1f] = "\xb8\xba", [0x3f20] = "\xda\x4f",
  [0x3f21] = "\xb9\xb1", [0x3f22] = "\xb2\xc6", [0x3f23] = "\xd4\xf0",
  [0x3f24] = "\xcf\xcd", [0x3f25] = "\xb0\xdc", [0x3f26] = "\xd5\xcb",
  [0x3f27] = "\xbb\xf5", [0x3f28] = "\xd6\xca", [0x3f29] = "\xb7\xb7",
  [0x3f2a] = "\xcc\xb0", [0x3f2b] = "\xc6\xb6", [0x3f2c] = "\xb1\xe1",
  [0x3f2d] = "\xb9\xba", [0x3f2e] = "\xd6\xfc", [0x3f2f] = "\xb9\xe1",
  [0x3f30] = "\xb7\xa1", [0x3f31] = "\xbc\xfa", [0x3f32] = "\xea\xda",
  [0x3f33] = "\xea\xdb", [0x3f34] = "\xcc\xf9", [0x3f35] = "\xb9\xf3",
  [0x3f36] = "\xea\xdc", [0x3f37] = "\xb4\xfb", [0x3f38] = "\xc3\xb3",
  [0x3f39] = "\xb7\xd1", [0x3f3a] = "\xba\xd8", [0x3f3b] = "\xea\xdd",
  [0x3f3c] = "\xd4\xf4", [0x3f3d] = "\xea\xde", [0x3f3e] = "\xbc\xd6",
  [0x3f3f] = "\xbb\xdf", [0x3f40] = "\xea\xdf", [0x3f41] = "\xc1\xde",
  [0x3f42] = "\xc2\xb8", [0x3f43] = "\xd4\xdf", [0x3f44] = "\xd7\xca",
  [0x3f45] = "\xea\xe0", [0x3f46] = "\xea\xe1", [0x3f47] = "\xea\xe4",
  [0x3f48] = "\xea\xe2", [0x3f49] = "\xea\xe3", [0x3f4a] = "\xc9\xde",
  [0x3f4b] = "\xb8\xb3", [0x3f4c] = "\xb6\xc4", [0x3f4d] = "\xea\xe5",
  [0x3f4e] = "\xca\xea", [0x3f4f] = "\xc9\xcd", [0x3f50] = "\xb4\xcd",
  [0x3f51] = "\xda\x50", [0x3f52] = "\xda\x51", [0x3f53] = "\xe2\xd9",
  [0x3f54] = "\xc5\xe2", [0x3f55] = "\xea\xe6", [0x3f56] = "\xc0\xb5",
  [0x3f57] = "\xda\x52", [0x3f58] = "\xd7\xb8", [0x3f59] = "\xea\xe7",
  [0x3f5a] = "\xd7\xac", [0x3f5b] = "\xc8\xfc", [0x3f5c] = "\xd8\xd3",
  [0x3f5d] = "\xd8\xcd", [0x3f5e] = "\xd4\xde", [0x3f5f] = "\xda\x53",
  [0x3f60] = "\xd4\xf9", [0x3f61] = "\xc9\xc4", [0x3f62] = "\xd3\xae",
  [0x3f63] = "\xb8\xd3", [0x3f64] = "\xb3\xe0", [0x3f65] = "\xda\x54",
  [0x3f66] = "\xc9\xe2", [0x3f67] = "\xf4\xf6", [0x3f68] = "\xda\x55",
  [0x3f69] = "\xda\x56", [0x3f6a] = "\xda\x57", [0x3f6b] = "\xba\xd5",
  [0x3f6c] = "\xda\x58", [0x3f6d] = "\xf4\xf7", [0x3f6e] = "\xda\x59",
  [0x3f6f] = "\xda\x5a", [0x3f70] = "\xd7\xdf", [0x3f71] = "\xda\x5b",
  [0x3f72] = "\xda\x5c", [0x3f73] = "\xf4\xf1", [0x3f74] = "\xb8\xb0",
  [0x3f75] = "\xd5\xd4", [0x3f76] = "\xb8\xcf", [0x3f77] = "\xc6\xf0",
  [0x3f78] = "\xda\x5d", [0x3f79] = "\xda\x5e", [0x3f7a] = "\xda\x5f",
  [0x3f7b] = "\xda\x60", [0x3f7c] = "\xda\x61", [0x3f7d] = "\xda\x62",
  [0x3f7e] = "\xda\x63", [0x3f7f] = "\xda\x64", [0x3f80] = "\xda\x65",
  [0x3f81] = "\xb3\xc3", [0x3f82] = "\xda\x66", [0x3f83] = "\xda\x67",
  [0x3f84] = "\xf4\xf2", [0x3f85] = "\xb3\xac", [0x3f86] = "\xda\x68",
  [0x3f87] = "\xda\x69", [0x3f88] = "\xda\x6a", [0x3f89] = "\xda\x6b",
  [0x3f8a] = "\xd4\xbd", [0x3f8b] = "\xc7\xf7", [0x3f8c] = "\xda\x6c",
  [0x3f8d] = "\xda\x6d", [0x3f8e] = "\xda\x6e", [0x3f8f] = "\xda\x6f",
  [0x3f90] = "\xda\x70", [0x3f91] = "\xf4\xf4", [0x3f92] = "\xda\x71",
  [0x3f93] = "\xda\x72", [0x3f94] = "\xf4\xf3", [0x3f95] = "\xda\x73",
  [0x3f96] = "\xda\x74", [0x3f97] = "\xda\x75", [0x3f98] = "\xda\x76",
  [0x3f99] = "\xda\x77", [0x3f9a] = "\xda\x78", [0x3f9b] = "\xda\x79",
  [0x3f9c] = "\xda\x7a", [0x3f9d] = "\xda\x7b", [0x3f9e] = "\xda\x7c",
  [0x3f9f] = "\xcc\xcb", [0x3fa0] = "\xda\x7d", [0x3fa1] = "\xda\x7e",
  [0x3fa2] = "\xda\x80", [0x3fa3] = "\xc8\xa4", [0x3fa4] = "\xda\x81",
  [0x3fa5] = "\xda\x82", [0x3fa6] = "\xda\x83", [0x3fa7] = "\xda\x84",
  [0x3fa8] = "\xda\x85", [0x3fa9] = "\xda\x86", [0x3faa] = "\xda\x87",
  [0x3fab] = "\xda\x88", [0x3fac] = "\xda\x89", [0x3fad] = "\xda\x8a",
  [0x3fae] = "\xda\x8b", [0x3faf] = "\xda\x8c", [0x3fb0] = "\xda\x8d",
  [0x3fb1] = "\xf4\xf5", [0x3fb2] = "\xda\x8e", [0x3fb3] = "\xd7\xe3",
  [0x3fb4] = "\xc5\xbf", [0x3fb5] = "\xf5\xc0", [0x3fb6] = "\xda\x8f",
  [0x3fb7] = "\xda\x90", [0x3fb8] = "\xf5\xbb", [0x3fb9] = "\xda\x91",
  [0x3fba] = "\xf5\xc3", [0x3fbb] = "\xda\x92", [0x3fbc] = "\xf5\xc2",
  [0x3fbd] = "\xda\x93", [0x3fbe] = "\xd6\xba", [0x3fbf] = "\xf5\xc1",
  [0x3fc0] = "\xda\x94", [0x3fc1] = "\xda\x95", [0x3fc2] = "\xda\x96",
  [0x3fc3] = "\xd4\xbe", [0x3fc4] = "\xf5\xc4", [0x3fc5] = "\xda\x97",
  [0x3fc6] = "\xf5\xcc", [0x3fc7] = "\xda\x98", [0x3fc8] = "\xda\x99",
  [0x3fc9] = "\xda\x9a", [0x3fca] = "\xda\x9b", [0x3fcb] = "\xb0\xcf",
  [0x3fcc] = "\xb5\xf8", [0x3fcd] = "\xda\x9c", [0x3fce] = "\xf5\xc9",
  [0x3fcf] = "\xf5\xca", [0x3fd0] = "\xda\x9d", [0x3fd1] = "\xc5\xdc",
  [0x3fd2] = "\xda\x9e", [0x3fd3] = "\xda\x9f", [0x3fd4] = "\xda\xa0",
  [0x3fd5] = "\xdb\x40", [0x3fd6] = "\xf5\xc5", [0x3fd7] = "\xf5\xc6",
  [0x3fd8] = "\xdb\x41", [0x3fd9] = "\xdb\x42", [0x3fda] = "\xf5\xc7",
  [0x3fdb] = "\xf5\xcb", [0x3fdc] = "\xdb\x43", [0x3fdd] = "\xbe\xe0",
  [0x3fde] = "\xf5\xc8", [0x3fdf] = "\xb8\xfa", [0x3fe0] = "\xdb\x44",
  [0x3fe1] = "\xdb\x45", [0x3fe2] = "\xdb\x46", [0x3fe3] = "\xf5\xd0",
  [0x3fe4] = "\xf5\xd3", [0x3fe5] = "\xdb\x47", [0x3fe6] = "\xdb\x48",
  [0x3fe7] = "\xdb\x49", [0x3fe8] = "\xbf\xe7", [0x3fe9] = "\xdb\x4a",
  [0x3fea] = "\xb9\xf2", [0x3feb] = "\xf5\xbc", [0x3fec] = "\xf5\xcd",
  [0x3fed] = "\xdb\x4b", [0x3fee] = "\xdb\x4c", [0x3fef] = "\xc2\xb7",
  [0x3ff0] = "\xdb\x4d", [0x3ff1] = "\xdb\x4e", [0x3ff2] = "\xdb\x4f",
  [0x3ff3] = "\xcc\xf8", [0x3ff4] = "\xdb\x50", [0x3ff5] = "\xbc\xf9",
  [0x3ff6] = "\xdb\x51", [0x3ff7] = "\xf5\xce", [0x3ff8] = "\xf5\xcf",
  [0x3ff9] = "\xf5\xd1", [0x3ffa] = "\xb6\xe5", [0x3ffb] = "\xf5\xd2",
  [0x3ffc] = "\xdb\x52", [0x3ffd] = "\xf5\xd5", [0x3ffe] = "\xdb\x53",
  [0x3fff] = "\xdb\x54", [0x4000] = "\xdb\x55", [0x4001] = "\xdb\x56",
  [0x4002] = "\xdb\x57", [0x4003] = "\xdb\x58", [0x4004] = "\xdb\x59",
  [0x4005] = "\xf5\xbd", [0x4006] = "\xdb\x5a", [0x4007] = "\xdb\x5b",
  [0x4008] = "\xdb\x5c", [0x4009] = "\xf5\xd4", [0x400a] = "\xd3\xbb",
  [0x400b] = "\xdb\x5d", [0x400c] = "\xb3\xec", [0x400d] = "\xdb\x5e",
  [0x400e] = "\xdb\x5f", [0x400f] = "\xcc\xa4", [0x4010] = "\xdb\x60",
  [0x4011] = "\xdb\x61", [0x4012] = "\xdb\x62", [0x4013] = "\xdb\x63",
  [0x4014] = "\xf5\xd6", [0x4015] = "\xdb\x64", [0x4016] = "\xdb\x65",
  [0x4017] = "\xdb\x66", [0x4018] = "\xdb\x67", [0x4019] = "\xdb\x68",
  [0x401a] = "\xdb\x69", [0x401b] = "\xdb\x6a", [0x401c] = "\xdb\x6b",
  [0x401d] = "\xf5\xd7", [0x401e] = "\xbe\xe1", [0x401f] = "\xf5\xd8",
  [0x4020] = "\xdb\x6c", [0x4021] = "\xdb\x6d", [0x4022] = "\xcc\xdf",
  [0x4023] = "\xf5\xdb", [0x4024] = "\xdb\x6e", [0x4025] = "\xdb\x6f",
  [0x4026] = "\xdb\x70", [0x4027] = "\xdb\x71", [0x4028] = "\xdb\x72",
  [0x4029] = "\xb2\xc8", [0x402a] = "\xd7\xd9", [0x402b] = "\xdb\x73",
  [0x402c] = "\xf5\xd9", [0x402d] = "\xdb\x74", [0x402e] = "\xf5\xda",
  [0x402f] = "\xf5\xdc", [0x4030] = "\xdb\x75", [0x4031] = "\xf5\xe2",
  [0x4032] = "\xdb\x76", [0x4033] = "\xdb\x77", [0x4034] = "\xdb\x78",
  [0x4035] = "\xf5\xe0", [0x4036] = "\xdb\x79", [0x4037] = "\xdb\x7a",
  [0x4038] = "\xdb\x7b", [0x4039] = "\xf5\xdf", [0x403a] = "\xf5\xdd",
  [0x403b] = "\xdb\x7c", [0x403c] = "\xdb\x7d", [0x403d] = "\xf5\xe1",
  [0x403e] = "\xdb\x7e", [0x403f] = "\xdb\x80", [0x4040] = "\xf5\xde",
  [0x4041] = "\xf5\xe4", [0x4042] = "\xf5\xe5", [0x4043] = "\xdb\x81",
  [0x4044] = "\xcc\xe3", [0x4045] = "\xdb\x82", [0x4046] = "\xdb\x83",
  [0x4047] = "\xe5\xbf", [0x4048] = "\xb5\xb8", [0x4049] = "\xf5\xe3",
  [0x404a] = "\xf5\xe8", [0x404b] = "\xcc\xa3", [0x404c] = "\xdb\x84",
  [0x404d] = "\xdb\x85", [0x404e] = "\xdb\x86", [0x404f] = "\xdb\x87",
  [0x4050] = "\xdb\x88", [0x4051] = "\xf5\xe6", [0x4052] = "\xf5\xe7",
  [0x4053] = "\xdb\x89", [0x4054] = "\xdb\x8a", [0x4055] = "\xdb\x8b",
  [0x4056] = "\xdb\x8c", [0x4057] = "\xdb\x8d", [0x4058] = "\xdb\x8e",
  [0x4059] = "\xf5\xbe", [0x405a] = "\xdb\x8f", [0x405b] = "\xdb\x90",
  [0x405c] = "\xdb\x91", [0x405d] = "\xdb\x92", [0x405e] = "\xdb\x93",
  [0x405f] = "\xdb\x94", [0x4060] = "\xdb\x95", [0x4061] = "\xdb\x96",
  [0x4062] = "\xdb\x97", [0x4063] = "\xdb\x98", [0x4064] = "\xdb\x99",
  [0x4065] = "\xdb\x9a", [0x4066] = "\xb1\xc4", [0x4067] = "\xdb\x9b",
  [0x4068] = "\xdb\x9c", [0x4069] = "\xf5\xbf", [0x406a] = "\xdb\x9d",
  [0x406b] = "\xdb\x9e", [0x406c] = "\xb5\xc5", [0x406d] = "\xb2\xe4",
  [0x406e] = "\xdb\x9f", [0x406f] = "\xf5\xec", [0x4070] = "\xf5\xe9",
  [0x4071] = "\xdb\xa0", [0x4072] = "\xb6\xd7", [0x4073] = "\xdc\x40",
  [0x4074] = "\xf5\xed", [0x4075] = "\xdc\x41", [0x4076] = "\xf5\xea",
  [0x4077] = "\xdc\x42", [0x4078] = "\xdc\x43", [0x4079] = "\xdc\x44",
  [0x407a] = "\xdc\x45", [0x407b] = "\xdc\x46", [0x407c] = "\xf5\xeb",
  [0x407d] = "\xdc\x47", [0x407e] = "\xdc\x48", [0x407f] = "\xb4\xda",
  [0x4080] = "\xdc\x49", [0x4081] = "\xd4\xea", [0x4082] = "\xdc\x4a",
  [0x4083] = "\xdc\x4b", [0x4084] = "\xdc\x4c", [0x4085] = "\xf5\xee",
  [0x4086] = "\xdc\x4d", [0x4087] = "\xb3\xf9", [0x4088] = "\xdc\x4e",
  [0x4089] = "\xdc\x4f", [0x408a] = "\xdc\x50", [0x408b] = "\xdc\x51",
  [0x408c] = "\xdc\x52", [0x408d] = "\xdc\x53", [0x408e] = "\xdc\x54",
  [0x408f] = "\xf5\xef", [0x4090] = "\xf5\xf1", [0x4091] = "\xdc\x55",
  [0x4092] = "\xdc\x56", [0x4093] = "\xdc\x57", [0x4094] = "\xf5\xf0",
  [0x4095] = "\xdc\x58", [0x4096] = "\xdc\x59", [0x4097] = "\xdc\x5a",
  [0x4098] = "\xdc\x5b", [0x4099] = "\xdc\x5c", [0x409a] = "\xdc\x5d",
  [0x409b] = "\xdc\x5e", [0x409c] = "\xf5\xf2", [0x409d] = "\xdc\x5f",
  [0x409e] = "\xf5\xf3", [0x409f] = "\xdc\x60", [0x40a0] = "\xdc\x61",
  [0x40a1] = "\xdc\x62", [0x40a2] = "\xdc\x63", [0x40a3] = "\xdc\x64",
  [0x40a4] = "\xdc\x65", [0x40a5] = "\xdc\x66", [0x40a6] = "\xdc\x67",
  [0x40a7] = "\xdc\x68", [0x40a8] = "\xdc\x69", [0x40a9] = "\xdc\x6a",
  [0x40aa] = "\xdc\x6b", [0x40ab] = "\xc9\xed", [0x40ac] = "\xb9\xaa",
  [0x40ad] = "\xdc\x6c", [0x40ae] = "\xdc\x6d", [0x40af] = "\xc7\xfb",
  [0x40b0] = "\xdc\x6e", [0x40b1] = "\xdc\x6f", [0x40b2] = "\xb6\xe3",
  [0x40b3] = "\xdc\x70", [0x40b4] = "\xdc\x71", [0x40b5] = "\xdc\x72",
  [0x40b6] = "\xdc\x73", [0x40b7] = "\xdc\x74", [0x40b8] = "\xdc\x75",
  [0x40b9] = "\xdc\x76", [0x40ba] = "\xcc\xc9", [0x40bb] = "\xdc\x77",
  [0x40bc] = "\xdc\x78", [0x40bd] = "\xdc\x79", [0x40be] = "\xdc\x7a",
  [0x40bf] = "\xdc\x7b", [0x40c0] = "\xdc\x7c", [0x40c1] = "\xdc\x7d",
  [0x40c2] = "\xdc\x7e", [0x40c3] = "\xdc\x80", [0x40c4] = "\xdc\x81",
  [0x40c5] = "\xdc\x82", [0x40c6] = "\xdc\x83", [0x40c7] = "\xdc\x84",
  [0x40c8] = "\xdc\x85", [0x40c9] = "\xdc\x86", [0x40ca] = "\xdc\x87",
  [0x40cb] = "\xdc\x88", [0x40cc] = "\xdc\x89", [0x40cd] = "\xdc\x8a",
  [0x40ce] = "\xea\xa6", [0x40cf] = "\xdc\x8b", [0x40d0] = "\xdc\x8c",
  [0x40d1] = "\xdc\x8d", [0x40d2] = "\xdc\x8e", [0x40d3] = "\xdc\x8f",
  [0x40d4] = "\xdc\x90", [0x40d5] = "\xdc\x91", [0x40d6] = "\xdc\x92",
  [0x40d7] = "\xdc\x93", [0x40d8] = "\xdc\x94", [0x40d9] = "\xdc\x95",
  [0x40da] = "\xdc\x96", [0x40db] = "\xdc\x97", [0x40dc] = "\xdc\x98",
  [0x40dd] = "\xdc\x99", [0x40de] = "\xdc\x9a", [0x40df] = "\xdc\x9b",
  [0x40e0] = "\xdc\x9c", [0x40e1] = "\xdc\x9d", [0x40e2] = "\xdc\x9e",
  [0x40e3] = "\xdc\x9f", [0x40e4] = "\xdc\xa0", [0x40e5] = "\xdd\x40",
  [0x40e6] = "\xdd\x41", [0x40e7] = "\xdd\x42", [0x40e8] = "\xdd\x43",
  [0x40e9] = "\xdd\x44", [0x40ea] = "\xdd\x45", [0x40eb] = "\xdd\x46",
  [0x40ec] = "\xdd\x47", [0x40ed] = "\xdd\x48", [0x40ee] = "\xdd\x49",
  [0x40ef] = "\xdd\x4a", [0x40f0] = "\xdd\x4b", [0x40f1] = "\xdd\x4c",
  [0x40f2] = "\xdd\x4d", [0x40f3] = "\xdd\x4e", [0x40f4] = "\xdd\x4f",
  [0x40f5] = "\xdd\x50", [0x40f6] = "\xdd\x51", [0x40f7] = "\xdd\x52",
  [0x40f8] = "\xdd\x53", [0x40f9] = "\xdd\x54", [0x40fa] = "\xdd\x55",
  [0x40fb] = "\xdd\x56", [0x40fc] = "\xdd\x57", [0x40fd] = "\xdd\x58",
  [0x40fe] = "\xdd\x59", [0x40ff] = "\xdd\x5a", [0x4100] = "\xdd\x5b",
  [0x4101] = "\xdd\x5c", [0x4102] = "\xdd\x5d", [0x4103] = "\xdd\x5e",
  [0x4104] = "\xdd\x5f", [0x4105] = "\xdd\x60", [0x4106] = "\xdd\x61",
  [0x4107] = "\xdd\x62", [0x4108] = "\xdd\x63", [0x4109] = "\xdd\x64",
  [0x410a] = "\xdd\x65", [0x410b] = "\xdd\x66", [0x410c] = "\xdd\x67",
  [0x410d] = "\xdd\x68", [0x410e] = "\xdd\x69", [0x410f] = "\xdd\x6a",
  [0x4110] = "\xdd\x6b", [0x4111] = "\xdd\x6c", [0x4112] = "\xdd\x6d",
  [0x4113] = "\xdd\x6e", [0x4114] = "\xdd\x6f", [0x4115] = "\xdd\x70",
  [0x4116] = "\xdd\x71", [0x4117] = "\xdd\x72", [0x4118] = "\xdd\x73",
  [0x4119] = "\xdd\x74", [0x411a] = "\xdd\x75", [0x411b] = "\xdd\x76",
  [0x411c] = "\xdd\x77", [0x411d] = "\xdd\x78", [0x411e] = "\xdd\x79",
  [0x411f] = "\xdd\x7a", [0x4120] = "\xdd\x7b", [0x4121] = "\xdd\x7c",
  [0x4122] = "\xdd\x7d", [0x4123] = "\xdd\x7e", [0x4124] = "\xdd\x80",
  [0x4125] = "\xdd\x81", [0x4126] = "\xdd\x82", [0x4127] = "\xdd\x83",
  [0x4128] = "\xdd\x84", [0x4129] = "\xdd\x85", [0x412a] = "\xdd\x86",
  [0x412b] = "\xdd\x87", [0x412c] = "\xdd\x88", [0x412d] = "\xdd\x89",
  [0x412e] = "\xdd\x8a", [0x412f] = "\xdd\x8b", [0x4130] = "\xdd\x8c",
  [0x4131] = "\xdd\x8d", [0x4132] = "\xdd\x8e", [0x4133] = "\xdd\x8f",
  [0x4134] = "\xdd\x90", [0x4135] = "\xdd\x91", [0x4136] = "\xdd\x92",
  [0x4137] = "\xdd\x93", [0x4138] = "\xdd\x94", [0x4139] = "\xdd\x95",
  [0x413a] = "\xdd\x96", [0x413b] = "\xdd\x97", [0x413c] = "\xdd\x98",
  [0x413d] = "\xdd\x99", [0x413e] = "\xdd\x9a", [0x413f] = "\xdd\x9b",
  [0x4140] = "\xdd\x9c", [0x4141] = "\xdd\x9d", [0x4142] = "\xdd\x9e",
  [0x4143] = "\xdd\x9f", [0x4144] = "\xdd\xa0", [0x4145] = "\xde\x40",
  [0x4146] = "\xde\x41", [0x4147] = "\xde\x42", [0x4148] = "\xde\x43",
  [0x4149] = "\xde\x44", [0x414a] = "\xde\x45", [0x414b] = "\xde\x46",
  [0x414c] = "\xde\x47", [0x414d] = "\xde\x48", [0x414e] = "\xde\x49",
  [0x414f] = "\xde\x4a", [0x4150] = "\xde\x4b", [0x4151] = "\xde\x4c",
  [0x4152] = "\xde\x4d", [0x4153] = "\xde\x4e", [0x4154] = "\xde\x4f",
  [0x4155] = "\xde\x50", [0x4156] = "\xde\x51", [0x4157] = "\xde\x52",
  [0x4158] = "\xde\x53", [0x4159] = "\xde\x54", [0x415a] = "\xde\x55",
  [0x415b] = "\xde\x56", [0x415c] = "\xde\x57", [0x415d] = "\xde\x58",
  [0x415e] = "\xde\x59", [0x415f] = "\xde\x5a", [0x4160] = "\xde\x5b",
  [0x4161] = "\xde\x5c", [0x4162] = "\xde\x5d", [0x4163] = "\xde\x5e",
  [0x4164] = "\xde\x5f", [0x4165] = "\xde\x60", [0x4166] = "\xb3\xb5",
  [0x4167] = "\xd4\xfe", [0x4168] = "\xb9\xec", [0x4169] = "\xd0\xf9",
  [0x416a] = "\xde\x61", [0x416b] = "\xe9\xed", [0x416c] = "\xd7\xaa",
  [0x416d] = "\xe9\xee", [0x416e] = "\xc2\xd6", [0x416f] = "\xc8\xed",
  [0x4170] = "\xba\xe4", [0x4171] = "\xe9\xef", [0x4172] = "\xe9\xf0",
  [0x4173] = "\xe9\xf1", [0x4174] = "\xd6\xe1", [0x4175] = "\xe9\xf2",
  [0x4176] = "\xe9\xf3", [0x4177] = "\xe9\xf5", [0x4178] = "\xe9\xf4",
  [0x4179] = "\xe9\xf6", [0x417a] = "\xe9\xf7", [0x417b] = "\xc7\xe1",
  [0x417c] = "\xe9\xf8", [0x417d] = "\xd4\xd8", [0x417e] = "\xe9\xf9",
  [0x417f] = "\xbd\xce", [0x4180] = "\xde\x62", [0x4181] = "\xe9\xfa",
  [0x4182] = "\xe9\xfb", [0x4183] = "\xbd\xcf", [0x4184] = "\xe9\xfc",
  [0x4185] = "\xb8\xa8", [0x4186] = "\xc1\xbe", [0x4187] = "\xe9\xfd",
  [0x4188] = "\xb1\xb2", [0x4189] = "\xbb\xd4", [0x418a] = "\xb9\xf5",
  [0x418b] = "\xe9\xfe", [0x418c] = "\xde\x63", [0x418d] = "\xea\xa1",
  [0x418e] = "\xea\xa2", [0x418f] = "\xea\xa3", [0x4190] = "\xb7\xf8",
  [0x4191] = "\xbc\xad", [0x4192] = "\xde\x64", [0x4193] = "\xca\xe4",
  [0x4194] = "\xe0\xce", [0x4195] = "\xd4\xaf", [0x4196] = "\xcf\xbd",
  [0x4197] = "\xd5\xb7", [0x4198] = "\xea\xa4", [0x4199] = "\xd5\xde",
  [0x419a] = "\xea\xa5", [0x419b] = "\xd0\xc1", [0x419c] = "\xb9\xbc",
  [0x419d] = "\xde\x65", [0x419e] = "\xb4\xc7", [0x419f] = "\xb1\xd9",
  [0x41a0] = "\xde\x66", [0x41a1] = "\xde\x67", [0x41a2] = "\xde\x68",
  [0x41a3] = "\xc0\xb1", [0x41a4] = "\xde\x69", [0x41a5] = "\xde\x6a",
  [0x41a6] = "\xde\x6b", [0x41a7] = "\xde\x6c", [0x41a8] = "\xb1\xe6",
  [0x41a9] = "\xb1\xe7", [0x41aa] = "\xde\x6d", [0x41ab] = "\xb1\xe8",
  [0x41ac] = "\xde\x6e", [0x41ad] = "\xde\x6f", [0x41ae] = "\xde\x70",
  [0x41af] = "\xde\x71", [0x41b0] = "\xb3\xbd", [0x41b1] = "\xc8\xe8",
  [0x41b2] = "\xde\x72", [0x41b3] = "\xde\x73", [0x41b4] = "\xde\x74",
  [0x41b5] = "\xde\x75", [0x41b6] = "\xe5\xc1", [0x41b7] = "\xde\x76",
  [0x41b8] = "\xde\x77", [0x41b9] = "\xb1\xdf", [0x41ba] = "\xde\x78",
  [0x41bb] = "\xde\x79", [0x41bc] = "\xde\x7a", [0x41bd] = "\xc1\xc9",
  [0x41be] = "\xb4\xef", [0x41bf] = "\xde\x7b", [0x41c0] = "\xde\x7c",
  [0x41c1] = "\xc7\xa8", [0x41c2] = "\xd3\xd8", [0x41c3] = "\xde\x7d",
  [0x41c4] = "\xc6\xf9", [0x41c5] = "\xd1\xb8", [0x41c6] = "\xde\x7e",
  [0x41c7] = "\xb9\xfd", [0x41c8] = "\xc2\xf5", [0x41c9] = "\xde\x80",
  [0x41ca] = "\xde\x81", [0x41cb] = "\xde\x82", [0x41cc] = "\xde\x83",
  [0x41cd] = "\xde\x84", [0x41ce] = "\xd3\xad", [0x41cf] = "\xde\x85",
  [0x41d0] = "\xd4\xcb", [0x41d1] = "\xbd\xfc", [0x41d2] = "\xde\x86",
  [0x41d3] = "\xe5\xc2", [0x41d4] = "\xb7\xb5", [0x41d5] = "\xe5\xc3",
  [0x41d6] = "\xde\x87", [0x41d7] = "\xde\x88", [0x41d8] = "\xbb\xb9",
  [0x41d9] = "\xd5\xe2", [0x41da] = "\xde\x89", [0x41db] = "\xbd\xf8",
  [0x41dc] = "\xd4\xb6", [0x41dd] = "\xce\xa5", [0x41de] = "\xc1\xac",
  [0x41df] = "\xb3\xd9", [0x41e0] = "\xde\x8a", [0x41e1] = "\xde\x8b",
  [0x41e2] = "\xcc\xf6", [0x41e3] = "\xde\x8c", [0x41e4] = "\xe5\xc6",
  [0x41e5] = "\xe5\xc4", [0x41e6] = "\xe5\xc8", [0x41e7] = "\xde\x8d",
  [0x41e8] = "\xe5\xca", [0x41e9] = "\xe5\xc7", [0x41ea] = "\xb5\xcf",
  [0x41eb] = "\xc6\xc8", [0x41ec] = "\xde\x8e", [0x41ed] = "\xb5\xfc",
  [0x41ee] = "\xe5\xc5", [0x41ef] = "\xde\x8f", [0x41f0] = "\xca\xf6",
  [0x41f1] = "\xde\x90", [0x41f2] = "\xde\x91", [0x41f3] = "\xe5\xc9",
  [0x41f4] = "\xde\x92", [0x41f5] = "\xde\x93", [0x41f6] = "\xde\x94",
  [0x41f7] = "\xc3\xd4", [0x41f8] = "\xb1\xc5", [0x41f9] = "\xbc\xa3",
  [0x41fa] = "\xde\x95", [0x41fb] = "\xde\x96", [0x41fc] = "\xde\x97",
  [0x41fd] = "\xd7\xb7", [0x41fe] = "\xde\x98", [0x41ff] = "\xde\x99",
  [0x4200] = "\xcd\xcb", [0x4201] = "\xcb\xcd", [0x4202] = "\xca\xca",
  [0x4203] = "\xcc\xd3", [0x4204] = "\xe5\xcc", [0x4205] = "\xe5\xcb",
  [0x4206] = "\xc4\xe6", [0x4207] = "\xde\x9a", [0x4208] = "\xde\x9b",
  [0x4209] = "\xd1\xa1", [0x420a] = "\xd1\xb7", [0x420b] = "\xe5\xcd",
  [0x420c] = "\xde\x9c", [0x420d] = "\xe5\xd0", [0x420e] = "\xde\x9d",
  [0x420f] = "\xcd\xb8", [0x4210] = "\xd6\xf0", [0x4211] = "\xe5\xcf",
  [0x4212] = "\xb5\xdd", [0x4213] = "\xde\x9e", [0x4214] = "\xcd\xbe",
  [0x4215] = "\xde\x9f", [0x4216] = "\xe5\xd1", [0x4217] = "\xb6\xba",
  [0x4218] = "\xde\xa0", [0x4219] = "\xdf\x40", [0x421a] = "\xcd\xa8",
  [0x421b] = "\xb9\xe4", [0x421c] = "\xdf\x41", [0x421d] = "\xca\xc5",
  [0x421e] = "\xb3\xd1", [0x421f] = "\xcb\xd9", [0x4220] = "\xd4\xec",
  [0x4221] = "\xe5\xd2", [0x4222] = "\xb7\xea", [0x4223] = "\xdf\x42",
  [0x4224] = "\xdf\x43", [0x4225] = "\xdf\x44", [0x4226] = "\xe5\xce",
  [0x4227] = "\xdf\x45", [0x4228] = "\xdf\x46", [0x4229] = "\xdf\x47",
  [0x422a] = "\xdf\x48", [0x422b] = "\xdf\x49", [0x422c] = "\xdf\x4a",
  [0x422d] = "\xe5\xd5", [0x422e] = "\xb4\xfe", [0x422f] = "\xe5\xd6",
  [0x4230] = "\xdf\x4b", [0x4231] = "\xdf\x4c", [0x4232] = "\xdf\x4d",
  [0x4233] = "\xdf\x4e", [0x4234] = "\xdf\x4f", [0x4235] = "\xe5\xd3",
  [0x4236] = "\xe5\xd4", [0x4237] = "\xdf\x50", [0x4238] = "\xd2\xdd",
  [0x4239] = "\xdf\x51", [0x423a] = "\xdf\x52", [0x423b] = "\xc2\xdf",
  [0x423c] = "\xb1\xc6", [0x423d] = "\xdf\x53", [0x423e] = "\xd3\xe2",
  [0x423f] = "\xdf\x54", [0x4240] = "\xdf\x55", [0x4241] = "\xb6\xdd",
  [0x4242] = "\xcb\xec", [0x4243] = "\xdf\x56", [0x4244] = "\xe5\xd7",
  [0x4245] = "\xdf\x57", [0x4246] = "\xdf\x58", [0x4247] = "\xd3\xf6",
  [0x4248] = "\xdf\x59", [0x4249] = "\xdf\x5a", [0x424a] = "\xdf\x5b",
  [0x424b] = "\xdf\x5c", [0x424c] = "\xdf\x5d", [0x424d] = "\xb1\xe9",
  [0x424e] = "\xdf\x5e", [0x424f] = "\xb6\xf4", [0x4250] = "\xe5\xda",
  [0x4251] = "\xe5\xd8", [0x4252] = "\xe5\xd9", [0x4253] = "\xb5\xc0",
  [0x4254] = "\xdf\x5f", [0x4255] = "\xdf\x60", [0x4256] = "\xdf\x61",
  [0x4257] = "\xd2\xc5", [0x4258] = "\xe5\xdc", [0x4259] = "\xdf\x62",
  [0x425a] = "\xdf\x63", [0x425b] = "\xe5\xde", [0x425c] = "\xdf\x64",
  [0x425d] = "\xdf\x65", [0x425e] = "\xdf\x66", [0x425f] = "\xdf\x67",
  [0x4260] = "\xdf\x68", [0x4261] = "\xdf\x69", [0x4262] = "\xe5\xdd",
  [0x4263] = "\xc7\xb2", [0x4264] = "\xdf\x6a", [0x4265] = "\xd2\xa3",
  [0x4266] = "\xdf\x6b", [0x4267] = "\xdf\x6c", [0x4268] = "\xe5\xdb",
  [0x4269] = "\xdf\x6d", [0x426a] = "\xdf\x6e", [0x426b] = "\xdf\x6f",
  [0x426c] = "\xdf\x70", [0x426d] = "\xd4\xe2", [0x426e] = "\xd5\xda",
  [0x426f] = "\xdf\x71", [0x4270] = "\xdf\x72", [0x4271] = "\xdf\x73",
  [0x4272] = "\xdf\x74", [0x4273] = "\xdf\x75", [0x4274] = "\xe5\xe0",
  [0x4275] = "\xd7\xf1", [0x4276] = "\xdf\x76", [0x4277] = "\xdf\x77",
  [0x4278] = "\xdf\x78", [0x4279] = "\xdf\x79", [0x427a] = "\xdf\x7a",
  [0x427b] = "\xdf\x7b", [0x427c] = "\xdf\x7c", [0x427d] = "\xe5\xe1",
  [0x427e] = "\xdf\x7d", [0x427f] = "\xb1\xdc", [0x4280] = "\xd1\xfb",
  [0x4281] = "\xdf\x7e", [0x4282] = "\xe5\xe2", [0x4283] = "\xe5\xe4",
  [0x4284] = "\xdf\x80", [0x4285] = "\xdf\x81", [0x4286] = "\xdf\x82",
  [0x4287] = "\xdf\x83", [0x4288] = "\xe5\xe3", [0x4289] = "\xdf\x84",
  [0x428a] = "\xdf\x85", [0x428b] = "\xe5\xe5", [0x428c] = "\xdf\x86",
  [0x428d] = "\xdf\x87", [0x428e] = "\xdf\x88", [0x428f] = "\xdf\x89",
  [0x4290] = "\xdf\x8a", [0x4291] = "\xd2\xd8", [0x4292] = "\xdf\x8b",
  [0x4293] = "\xb5\xcb", [0x4294] = "\xdf\x8c", [0x4295] = "\xe7\xdf",
  [0x4296] = "\xdf\x8d", [0x4297] = "\xda\xf5", [0x4298] = "\xdf\x8e",
  [0x4299] = "\xda\xf8", [0x429a] = "\xdf\x8f", [0x429b] = "\xda\xf6",
  [0x429c] = "\xdf\x90", [0x429d] = "\xda\xf7", [0x429e] = "\xdf\x91",
  [0x429f] = "\xdf\x92", [0x42a0] = "\xdf\x93", [0x42a1] = "\xda\xfa",
  [0x42a2] = "\xd0\xcf", [0x42a3] = "\xc4\xc7", [0x42a4] = "\xdf\x94",
  [0x42a5] = "\xdf\x95", [0x42a6] = "\xb0\xee", [0x42a7] = "\xdf\x96",
  [0x42a8] = "\xdf\x97", [0x42a9] = "\xdf\x98", [0x42aa] = "\xd0\xb0",
  [0x42ab] = "\xdf\x99", [0x42ac] = "\xda\xf9", [0x42ad] = "\xdf\x9a",
  [0x42ae] = "\xd3\xca", [0x42af] = "\xba\xaa", [0x42b0] = "\xdb\xa2",
  [0x42b1] = "\xc7\xf1", [0x42b2] = "\xdf\x9b", [0x42b3] = "\xda\xfc",
  [0x42b4] = "\xda\xfb", [0x42b5] = "\xc9\xdb", [0x42b6] = "\xda\xfd",
  [0x42b7] = "\xdf\x9c", [0x42b8] = "\xdb\xa1", [0x42b9] = "\xd7\xde",
  [0x42ba] = "\xda\xfe", [0x42bb] = "\xc1\xda", [0x42bc] = "\xdf\x9d",
  [0x42bd] = "\xdf\x9e", [0x42be] = "\xdb\xa5", [0x42bf] = "\xdf\x9f",
  [0x42c0] = "\xdf\xa0", [0x42c1] = "\xd3\xf4", [0x42c2] = "\xe0\x40",
  [0x42c3] = "\xe0\x41", [0x42c4] = "\xdb\xa7", [0x42c5] = "\xdb\xa4",
  [0x42c6] = "\xe0\x42", [0x42c7] = "\xdb\xa8", [0x42c8] = "\xe0\x43",
  [0x42c9] = "\xe0\x44", [0x42ca] = "\xbd\xbc", [0x42cb] = "\xe0\x45",
  [0x42cc] = "\xe0\x46", [0x42cd] = "\xe0\x47", [0x42ce] = "\xc0\xc9",
  [0x42cf] = "\xdb\xa3", [0x42d0] = "\xdb\xa6", [0x42d1] = "\xd6\xa3",
  [0x42d2] = "\xe0\x48", [0x42d3] = "\xdb\xa9", [0x42d4] = "\xe0\x49",
  [0x42d5] = "\xe0\x4a", [0x42d6] = "\xe0\x4b", [0x42d7] = "\xdb\xad",
  [0x42d8] = "\xe0\x4c", [0x42d9] = "\xe0\x4d", [0x42da] = "\xe0\x4e",
  [0x42db] = "\xdb\xae", [0x42dc] = "\xdb\xac", [0x42dd] = "\xba\xc2",
  [0x42de] = "\xe0\x4f", [0x42df] = "\xe0\x50", [0x42e0] = "\xe0\x51",
  [0x42e1] = "\xbf\xa4", [0x42e2] = "\xdb\xab", [0x42e3] = "\xe0\x52",
  [0x42e4] = "\xe0\x53", [0x42e5] = "\xe0\x54", [0x42e6] = "\xdb\xaa",
  [0x42e7] = "\xd4\xc7", [0x42e8] = "\xb2\xbf", [0x42e9] = "\xe0\x55",
  [0x42ea] = "\xe0\x56", [0x42eb] = "\xdb\xaf", [0x42ec] = "\xe0\x57",
  [0x42ed] = "\xb9\xf9", [0x42ee] = "\xe0\x58", [0x42ef] = "\xdb\xb0",
  [0x42f0] = "\xe0\x59", [0x42f1] = "\xe0\x5a", [0x42f2] = "\xe0\x5b",
  [0x42f3] = "\xe0\x5c", [0x42f4] = "\xb3\xbb", [0x42f5] = "\xe0\x5d",
  [0x42f6] = "\xe0\x5e", [0x42f7] = "\xe0\x5f", [0x42f8] = "\xb5\xa6",
  [0x42f9] = "\xe0\x60", [0x42fa] = "\xe0\x61", [0x42fb] = "\xe0\x62",
  [0x42fc] = "\xe0\x63", [0x42fd] = "\xb6\xbc", [0x42fe] = "\xdb\xb1",
  [0x42ff] = "\xe0\x64", [0x4300] = "\xe0\x65", [0x4301] = "\xe0\x66",
  [0x4302] = "\xb6\xf5", [0x4303] = "\xe0\x67", [0x4304] = "\xdb\xb2",
  [0x4305] = "\xe0\x68", [0x4306] = "\xe0\x69", [0x4307] = "\xe0\x6a",
  [0x4308] = "\xe0\x6b", [0x4309] = "\xe0\x6c", [0x430a] = "\xe0\x6d",
  [0x430b] = "\xe0\x6e", [0x430c] = "\xe0\x6f", [0x430d] = "\xe0\x70",
  [0x430e] = "\xe0\x71", [0x430f] = "\xe0\x72", [0x4310] = "\xe0\x73",
  [0x4311] = "\xe0\x74", [0x4312] = "\xe0\x75", [0x4313] = "\xe0\x76",
  [0x4314] = "\xe0\x77", [0x4315] = "\xe0\x78", [0x4316] = "\xe0\x79",
  [0x4317] = "\xe0\x7a", [0x4318] = "\xe0\x7b", [0x4319] = "\xb1\xc9",
  [0x431a] = "\xe0\x7c", [0x431b] = "\xe0\x7d", [0x431c] = "\xe0\x7e",
  [0x431d] = "\xe0\x80", [0x431e] = "\xdb\xb4", [0x431f] = "\xe0\x81",
  [0x4320] = "\xe0\x82", [0x4321] = "\xe0\x83", [0x4322] = "\xdb\xb3",
  [0x4323] = "\xdb\xb5", [0x4324] = "\xe0\x84", [0x4325] = "\xe0\x85",
  [0x4326] = "\xe0\x86", [0x4327] = "\xe0\x87", [0x4328] = "\xe0\x88",
  [0x4329] = "\xe0\x89", [0x432a] = "\xe0\x8a", [0x432b] = "\xe0\x8b",
  [0x432c] = "\xe0\x8c", [0x432d] = "\xe0\x8d", [0x432e] = "\xe0\x8e",
  [0x432f] = "\xdb\xb7", [0x4330] = "\xe0\x8f", [0x4331] = "\xdb\xb6",
  [0x4332] = "\xe0\x90", [0x4333] = "\xe0\x91", [0x4334] = "\xe0\x92",
  [0x4335] = "\xe0\x93", [0x4336] = "\xe0\x94", [0x4337] = "\xe0\x95",
  [0x4338] = "\xe0\x96", [0x4339] = "\xdb\xb8", [0x433a] = "\xe0\x97",
  [0x433b] = "\xe0\x98", [0x433c] = "\xe0\x99", [0x433d] = "\xe0\x9a",
  [0x433e] = "\xe0\x9b", [0x433f] = "\xe0\x9c", [0x4340] = "\xe0\x9d",
  [0x4341] = "\xe0\x9e", [0x4342] = "\xe0\x9f", [0x4343] = "\xdb\xb9",
  [0x4344] = "\xe0\xa0", [0x4345] = "\xe1\x40", [0x4346] = "\xdb\xba",
  [0x4347] = "\xe1\x41", [0x4348] = "\xe1\x42", [0x4349] = "\xd3\xcf",
  [0x434a] = "\xf4\xfa", [0x434b] = "\xc7\xf5", [0x434c] = "\xd7\xc3",
  [0x434d] = "\xc5\xe4", [0x434e] = "\xf4\xfc", [0x434f] = "\xf4\xfd",
  [0x4350] = "\xf4\xfb", [0x4351] = "\xe1\x43", [0x4352] = "\xbe\xc6",
  [0x4353] = "\xe1\x44", [0x4354] = "\xe1\x45", [0x4355] = "\xe1\x46",
  [0x4356] = "\xe1\x47", [0x4357] = "\xd0\xef", [0x4358] = "\xe1\x48",
  [0x4359] = "\xe1\x49", [0x435a] = "\xb7\xd3", [0x435b] = "\xe1\x4a",
  [0x435c] = "\xe1\x4b", [0x435d] = "\xd4\xcd", [0x435e] = "\xcc\xaa",
  [0x435f] = "\xe1\x4c", [0x4360] = "\xe1\x4d", [0x4361] = "\xf5\xa2",
  [0x4362] = "\xf5\xa1", [0x4363] = "\xba\xa8", [0x4364] = "\xf4\xfe",
  [0x4365] = "\xcb\xd6", [0x4366] = "\xe1\x4e", [0x4367] = "\xe1\x4f",
  [0x4368] = "\xe1\x50", [0x4369] = "\xf5\xa4", [0x436a] = "\xc0\xd2",
  [0x436b] = "\xe1\x51", [0x436c] = "\xb3\xea", [0x436d] = "\xe1\x52",
  [0x436e] = "\xcd\xaa", [0x436f] = "\xf5\xa5", [0x4370] = "\xf5\xa3",
  [0x4371] = "\xbd\xb4", [0x4372] = "\xf5\xa8", [0x4373] = "\xe1\x53",
  [0x4374] = "\xf5\xa9", [0x4375] = "\xbd\xcd", [0x4376] = "\xc3\xb8",
  [0x4377] = "\xbf\xe1", [0x4378] = "\xcb\xe1", [0x4379] = "\xf5\xaa",
  [0x437a] = "\xe1\x54", [0x437b] = "\xe1\x55", [0x437c] = "\xe1\x56",
  [0x437d] = "\xf5\xa6", [0x437e] = "\xf5\xa7", [0x437f] = "\xc4\xf0",
  [0x4380] = "\xe1\x57", [0x4381] = "\xe1\x58", [0x4382] = "\xe1\x59",
  [0x4383] = "\xe1\x5a", [0x4384] = "\xe1\x5b", [0x4385] = "\xf5\xac",
  [0x4386] = "\xe1\x5c", [0x4387] = "\xb4\xbc", [0x4388] = "\xe1\x5d",
  [0x4389] = "\xd7\xed", [0x438a] = "\xe1\x5e", [0x438b] = "\xb4\xd7",
  [0x438c] = "\xf5\xab", [0x438d] = "\xf5\xae", [0x438e] = "\xe1\x5f",
  [0x438f] = "\xe1\x60", [0x4390] = "\xf5\xad", [0x4391] = "\xf5\xaf",
  [0x4392] = "\xd0\xd1", [0x4393] = "\xe1\x61", [0x4394] = "\xe1\x62",
  [0x4395] = "\xe1\x63", [0x4396] = "\xe1\x64", [0x4397] = "\xe1\x65",
  [0x4398] = "\xe1\x66", [0x4399] = "\xe1\x67", [0x439a] = "\xc3\xd1",
  [0x439b] = "\xc8\xa9", [0x439c] = "\xe1\x68", [0x439d] = "\xe1\x69",
  [0x439e] = "\xe1\x6a", [0x439f] = "\xe1\x6b", [0x43a0] = "\xe1\x6c",
  [0x43a1] = "\xe1\x6d", [0x43a2] = "\xf5\xb0", [0x43a3] = "\xf5\xb1",
  [0x43a4] = "\xe1\x6e", [0x43a5] = "\xe1\x6f", [0x43a6] = "\xe1\x70",
  [0x43a7] = "\xe1\x71", [0x43a8] = "\xe1\x72", [0x43a9] = "\xe1\x73",
  [0x43aa] = "\xf5\xb2", [0x43ab] = "\xe1\x74", [0x43ac] = "\xe1\x75",
  [0x43ad] = "\xf5\xb3", [0x43ae] = "\xf5\xb4", [0x43af] = "\xf5\xb5",
  [0x43b0] = "\xe1\x76", [0x43b1] = "\xe1\x77", [0x43b2] = "\xe1\x78",
  [0x43b3] = "\xe1\x79", [0x43b4] = "\xf5\xb7", [0x43b5] = "\xf5\xb6",
  [0x43b6] = "\xe1\x7a", [0x43b7] = "\xe1\x7b", [0x43b8] = "\xe1\x7c",
  [0x43b9] = "\xe1\x7d", [0x43ba] = "\xf5\xb8", [0x43bb] = "\xe1\x7e",
  [0x43bc] = "\xe1\x80", [0x43bd] = "\xe1\x81", [0x43be] = "\xe1\x82",
  [0x43bf] = "\xe1\x83", [0x43c0] = "\xe1\x84", [0x43c1] = "\xe1\x85",
  [0x43c2] = "\xe1\x86", [0x43c3] = "\xe1\x87", [0x43c4] = "\xe1\x88",
  [0x43c5] = "\xe1\x89", [0x43c6] = "\xe1\x8a", [0x43c7] = "\xb2\xc9",
  [0x43c8] = "\xe1\x8b", [0x43c9] = "\xd3\xd4", [0x43ca] = "\xca\xcd",
  [0x43cb] = "\xe1\x8c", [0x43cc] = "\xc0\xef", [0x43cd] = "\xd6\xd8",
  [0x43ce] = "\xd2\xb0", [0x43cf] = "\xc1\xbf", [0x43d0] = "\xe1\x8d",
  [0x43d1] = "\xbd\xf0", [0x43d2] = "\xe1\x8e", [0x43d3] = "\xe1\x8f",
  [0x43d4] = "\xe1\x90", [0x43d5] = "\xe1\x91", [0x43d6] = "\xe1\x92",
  [0x43d7] = "\xe1\x93", [0x43d8] = "\xe1\x94", [0x43d9] = "\xe1\x95",
  [0x43da] = "\xe1\x96", [0x43db] = "\xe1\x97", [0x43dc] = "\xb8\xaa",
  [0x43dd] = "\xe1\x98", [0x43de] = "\xe1\x99", [0x43df] = "\xe1\x9a",
  [0x43e0] = "\xe1\x9b", [0x43e1] = "\xe1\x9c", [0x43e2] = "\xe1\x9d",
  [0x43e3] = "\xe1\x9e", [0x43e4] = "\xe1\x9f", [0x43e5] = "\xe1\xa0",
  [0x43e6] = "\xe2\x40", [0x43e7] = "\xe2\x41", [0x43e8] = "\xe2\x42",
  [0x43e9] = "\xe2\x43", [0x43ea] = "\xe2\x44", [0x43eb] = "\xe2\x45",
  [0x43ec] = "\xe2\x46", [0x43ed] = "\xe2\x47", [0x43ee] = "\xe2\x48",
  [0x43ef] = "\xe2\x49", [0x43f0] = "\xe2\x4a", [0x43f1] = "\xe2\x4b",
  [0x43f2] = "\xe2\x4c", [0x43f3] = "\xe2\x4d", [0x43f4] = "\xe2\x4e",
  [0x43f5] = "\xe2\x4f", [0x43f6] = "\xe2\x50", [0x43f7] = "\xe2\x51",
  [0x43f8] = "\xe2\x52", [0x43f9] = "\xe2\x53", [0x43fa] = "\xe2\x54",
  [0x43fb] = "\xe2\x55", [0x43fc] = "\xe2\x56", [0x43fd] = "\xe2\x57",
  [0x43fe] = "\xe2\x58", [0x43ff] = "\xe2\x59", [0x4400] = "\xe2\x5a",
  [0x4401] = "\xe2\x5b", [0x4402] = "\xe2\x5c", [0x4403] = "\xe2\x5d",
  [0x4404] = "\xe2\x5e", [0x4405] = "\xe2\x5f", [0x4406] = "\xe2\x60",
  [0x4407] = "\xe2\x61", [0x4408] = "\xe2\x62", [0x4409] = "\xe2\x63",
  [0x440a] = "\xe2\x64", [0x440b] = "\xe2\x65", [0x440c] = "\xe2\x66",
  [0x440d] = "\xe2\x67", [0x440e] = "\xe2\x68", [0x440f] = "\xe2\x69",
  [0x4410] = "\xe2\x6a", [0x4411] = "\xe2\x6b", [0x4412] = "\xe2\x6c",
  [0x4413] = "\xe2\x6d", [0x4414] = "\xe2\x6e", [0x4415] = "\xe2\x6f",
  [0x4416] = "\xe2\x70", [0x4417] = "\xe2\x71", [0x4418] = "\xe2\x72",
  [0x4419] = "\xe2\x73", [0x441a] = "\xe2\x74", [0x441b] = "\xe2\x75",
  [0x441c] = "\xe2\x76", [0x441d] = "\xe2\x77", [0x441e] = "\xe2\x78",
  [0x441f] = "\xe2\x79", [0x4420] = "\xe2\x7a", [0x4421] = "\xe2\x7b",
  [0x4422] = "\xe2\x7c", [0x4423] = "\xe2\x7d", [0x4424] = "\xe2\x7e",
  [0x4425] = "\xe2\x80", [0x4426] = "\xe2\x81", [0x4427] = "\xe2\x82",
  [0x4428] = "\xe2\x83", [0x4429] = "\xe2\x84", [0x442a] = "\xe2\x85",
  [0x442b] = "\xe2\x86", [0x442c] = "\xe2\x87", [0x442d] = "\xe2\x88",
  [0x442e] = "\xe2\x89", [0x442f] = "\xe2\x8a", [0x4430] = "\xe2\x8b",
  [0x4431] = "\xe2\x8c", [0x4432] = "\xe2\x8d", [0x4433] = "\xe2\x8e",
  [0x4434] = "\xe2\x8f", [0x4435] = "\xe2\x90", [0x4436] = "\xe2\x91",
  [0x4437] = "\xe2\x92", [0x4438] = "\xe2\x93", [0x4439] = "\xe2\x94",
  [0x443a] = "\xe2\x95", [0x443b] = "\xe2\x96", [0x443c] = "\xe2\x97",
  [0x443d] = "\xe2\x98", [0x443e] = "\xe2\x99", [0x443f] = "\xe2\x9a",
  [0x4440] = "\xe2\x9b", [0x4441] = "\xe2\x9c", [0x4442] = "\xe2\x9d",
  [0x4443] = "\xe2\x9e", [0x4444] = "\xe2\x9f", [0x4445] = "\xe2\xa0",
  [0x4446] = "\xe3\x40", [0x4447] = "\xe3\x41", [0x4448] = "\xe3\x42",
  [0x4449] = "\xe3\x43", [0x444a] = "\xe3\x44", [0x444b] = "\xe3\x45",
  [0x444c] = "\xe3\x46", [0x444d] = "\xe3\x47", [0x444e] = "\xe3\x48",
  [0x444f] = "\xe3\x49", [0x4450] = "\xe3\x4a", [0x4451] = "\xe3\x4b",
  [0x4452] = "\xe3\x4c", [0x4453] = "\xe3\x4d", [0x4454] = "\xe3\x4e",
  [0x4455] = "\xe3\x4f", [0x4456] = "\xe3\x50", [0x4457] = "\xe3\x51",
  [0x4458] = "\xe3\x52", [0x4459] = "\xe3\x53", [0x445a] = "\xe3\x54",
  [0x445b] = "\xe3\x55", [0x445c] = "\xe3\x56", [0x445d] = "\xe3\x57",
  [0x445e] = "\xe3\x58", [0x445f] = "\xe3\x59", [0x4460] = "\xe3\x5a",
  [0x4461] = "\xe3\x5b", [0x4462] = "\xe3\x5c", [0x4463] = "\xe3\x5d",
  [0x4464] = "\xe3\x5e", [0x4465] = "\xe3\x5f", [0x4466] = "\xe3\x60",
  [0x4467] = "\xe3\x61", [0x4468] = "\xe3\x62", [0x4469] = "\xe3\x63",
  [0x446a] = "\xe3\x64", [0x446b] = "\xe3\x65", [0x446c] = "\xe3\x66",
  [0x446d] = "\xe3\x67", [0x446e] = "\xe3\x68", [0x446f] = "\xe3\x69",
  [0x4470] = "\xe3\x6a", [0x4471] = "\xe3\x6b", [0x4472] = "\xe3\x6c",
  [0x4473] = "\xe3\x6d", [0x4474] = "\xbc\xf8", [0x4475] = "\xe3\x6e",
  [0x4476] = "\xe3\x6f", [0x4477] = "\xe3\x70", [0x4478] = "\xe3\x71",
  [0x4479] = "\xe3\x72", [0x447a] = "\xe3\x73", [0x447b] = "\xe3\x74",
  [0x447c] = "\xe3\x75", [0x447d] = "\xe3\x76", [0x447e] = "\xe3\x77",
  [0x447f] = "\xe3\x78", [0x4480] = "\xe3\x79", [0x4481] = "\xe3\x7a",
  [0x4482] = "\xe3\x7b", [0x4483] = "\xe3\x7c", [0x4484] = "\xe3\x7d",
  [0x4485] = "\xe3\x7e", [0x4486] = "\xe3\x80", [0x4487] = "\xe3\x81",
  [0x4488] = "\xe3\x82", [0x4489] = "\xe3\x83", [0x448a] = "\xe3\x84",
  [0x448b] = "\xe3\x85", [0x448c] = "\xe3\x86", [0x448d] = "\xe3\x87",
  [0x448e] = "\xf6\xc6", [0x448f] = "\xe3\x88", [0x4490] = "\xe3\x89",
  [0x4491] = "\xe3\x8a", [0x4492] = "\xe3\x8b", [0x4493] = "\xe3\x8c",
  [0x4494] = "\xe3\x8d", [0x4495] = "\xe3\x8e", [0x4496] = "\xe3\x8f",
  [0x4497] = "\xe3\x90", [0x4498] = "\xe3\x91", [0x4499] = "\xe3\x92",
  [0x449a] = "\xe3\x93", [0x449b] = "\xe3\x94", [0x449c] = "\xe3\x95",
  [0x449d] = "\xe3\x96", [0x449e] = "\xe3\x97", [0x449f] = "\xe3\x98",
  [0x44a0] = "\xe3\x99", [0x44a1] = "\xe3\x9a", [0x44a2] = "\xe3\x9b",
  [0x44a3] = "\xe3\x9c", [0x44a4] = "\xe3\x9d", [0x44a5] = "\xe3\x9e",
  [0x44a6] = "\xe3\x9f", [0x44a7] = "\xe3\xa0", [0x44a8] = "\xe4\x40",
  [0x44a9] = "\xe4\x41", [0x44aa] = "\xe4\x42", [0x44ab] = "\xe4\x43",
  [0x44ac] = "\xe4\x44", [0x44ad] = "\xe4\x45", [0x44ae] = "\xf6\xc7",
  [0x44af] = "\xe4\x46", [0x44b0] = "\xe4\x47", [0x44b1] = "\xe4\x48",
  [0x44b2] = "\xe4\x49", [0x44b3] = "\xe4\x4a", [0x44b4] = "\xe4\x4b",
  [0x44b5] = "\xe4\x4c", [0x44b6] = "\xe4\x4d", [0x44b7] = "\xe4\x4e",
  [0x44b8] = "\xe4\x4f", [0x44b9] = "\xe4\x50", [0x44ba] = "\xe4\x51",
  [0x44bb] = "\xe4\x52", [0x44bc] = "\xe4\x53", [0x44bd] = "\xe4\x54",
  [0x44be] = "\xe4\x55", [0x44bf] = "\xe4\x56", [0x44c0] = "\xe4\x57",
  [0x44c1] = "\xe4\x58", [0x44c2] = "\xe4\x59", [0x44c3] = "\xe4\x5a",
  [0x44c4] = "\xe4\x5b", [0x44c5] = "\xe4\x5c", [0x44c6] = "\xe4\x5d",
  [0x44c7] = "\xe4\x5e", [0x44c8] = "\xf6\xc8", [0x44c9] = "\xe4\x5f",
  [0x44ca] = "\xe4\x60", [0x44cb] = "\xe4\x61", [0x44cc] = "\xe4\x62",
  [0x44cd] = "\xe4\x63", [0x44ce] = "\xe4\x64", [0x44cf] = "\xe4\x65",
  [0x44d0] = "\xe4\x66", [0x44d1] = "\xe4\x67", [0x44d2] = "\xe4\x68",
  [0x44d3] = "\xe4\x69", [0x44d4] = "\xe4\x6a", [0x44d5] = "\xe4\x6b",
  [0x44d6] = "\xe4\x6c", [0x44d7] = "\xe4\x6d", [0x44d8] = "\xe4\x6e",
  [0x44d9] = "\xe4\x6f", [0x44da] = "\xe4\x70", [0x44db] = "\xe4\x71",
  [0x44dc] = "\xe4\x72", [0x44dd] = "\xe4\x73", [0x44de] = "\xe4\x74",
  [0x44df] = "\xe4\x75", [0x44e0] = "\xe4\x76", [0x44e1] = "\xe4\x77",
  [0x44e2] = "\xe4\x78", [0x44e3] = "\xe4\x79", [0x44e4] = "\xe4\x7a",
  [0x44e5] = "\xe4\x7b", [0x44e6] = "\xe4\x7c", [0x44e7] = "\xe4\x7d",
  [0x44e8] = "\xe4\x7e", [0x44e9] = "\xe4\x80", [0x44ea] = "\xe4\x81",
  [0x44eb] = "\xe4\x82", [0x44ec] = "\xe4\x83", [0x44ed] = "\xe4\x84",
  [0x44ee] = "\xe4\x85", [0x44ef] = "\xe4\x86", [0x44f0] = "\xe4\x87",
  [0x44f1] = "\xe4\x88", [0x44f2] = "\xe4\x89", [0x44f3] = "\xe4\x8a",
  [0x44f4] = "\xe4\x8b", [0x44f5] = "\xe4\x8c", [0x44f6] = "\xe4\x8d",
  [0x44f7] = "\xe4\x8e", [0x44f8] = "\xe4\x8f", [0x44f9] = "\xe4\x90",
  [0x44fa] = "\xe4\x91", [0x44fb] = "\xe4\x92", [0x44fc] = "\xe4\x93",
  [0x44fd] = "\xe4\x94", [0x44fe] = "\xe4\x95", [0x44ff] = "\xe4\x96",
  [0x4500] = "\xe4\x97", [0x4501] = "\xe4\x98", [0x4502] = "\xe4\x99",
  [0x4503] = "\xe4\x9a", [0x4504] = "\xe4\x9b", [0x4505] = "\xe4\x9c",
  [0x4506] = "\xe4\x9d", [0x4507] = "\xe4\x9e", [0x4508] = "\xe4\x9f",
  [0x4509] = "\xe4\xa0", [0x450a] = "\xe5\x40", [0x450b] = "\xe5\x41",
  [0x450c] = "\xe5\x42", [0x450d] = "\xe5\x43", [0x450e] = "\xe5\x44",
  [0x450f] = "\xe5\x45", [0x4510] = "\xe5\x46", [0x4511] = "\xe5\x47",
  [0x4512] = "\xe5\x48", [0x4513] = "\xe5\x49", [0x4514] = "\xe5\x4a",
  [0x4515] = "\xe5\x4b", [0x4516] = "\xe5\x4c", [0x4517] = "\xe5\x4d",
  [0x4518] = "\xe5\x4e", [0x4519] = "\xe5\x4f", [0x451a] = "\xe5\x50",
  [0x451b] = "\xe5\x51", [0x451c] = "\xe5\x52", [0x451d] = "\xe5\x53",
  [0x451e] = "\xe5\x54", [0x451f] = "\xe5\x55", [0x4520] = "\xe5\x56",
  [0x4521] = "\xe5\x57", [0x4522] = "\xe5\x58", [0x4523] = "\xe5\x59",
  [0x4524] = "\xe5\x5a", [0x4525] = "\xe5\x5b", [0x4526] = "\xe5\x5c",
  [0x4527] = "\xe5\x5d", [0x4528] = "\xe5\x5e", [0x4529] = "\xe5\x5f",
  [0x452a] = "\xe5\x60", [0x452b] = "\xe5\x61", [0x452c] = "\xe5\x62",
  [0x452d] = "\xe5\x63", [0x452e] = "\xe5\x64", [0x452f] = "\xe5\x65",
  [0x4530] = "\xe5\x66", [0x4531] = "\xe5\x67", [0x4532] = "\xe5\x68",
  [0x4533] = "\xe5\x69", [0x4534] = "\xe5\x6a", [0x4535] = "\xe5\x6b",
  [0x4536] = "\xe5\x6c", [0x4537] = "\xe5\x6d", [0x4538] = "\xe5\x6e",
  [0x4539] = "\xe5\x6f", [0x453a] = "\xe5\x70", [0x453b] = "\xe5\x71",
  [0x453c] = "\xe5\x72", [0x453d] = "\xe5\x73", [0x453e] = "\xf6\xc9",
  [0x453f] = "\xe5\x74", [0x4540] = "\xe5\x75", [0x4541] = "\xe5\x76",
  [0x4542] = "\xe5\x77", [0x4543] = "\xe5\x78", [0x4544] = "\xe5\x79",
  [0x4545] = "\xe5\x7a", [0x4546] = "\xe5\x7b", [0x4547] = "\xe5\x7c",
  [0x4548] = "\xe5\x7d", [0x4549] = "\xe5\x7e", [0x454a] = "\xe5\x80",
  [0x454b] = "\xe5\x81", [0x454c] = "\xe5\x82", [0x454d] = "\xe5\x83",
  [0x454e] = "\xe5\x84", [0x454f] = "\xe5\x85", [0x4550] = "\xe5\x86",
  [0x4551] = "\xe5\x87", [0x4552] = "\xe5\x88", [0x4553] = "\xe5\x89",
  [0x4554] = "\xe5\x8a", [0x4555] = "\xe5\x8b", [0x4556] = "\xe5\x8c",
  [0x4557] = "\xe5\x8d", [0x4558] = "\xe5\x8e", [0x4559] = "\xe5\x8f",
  [0x455a] = "\xe5\x90", [0x455b] = "\xe5\x91", [0x455c] = "\xe5\x92",
  [0x455d] = "\xe5\x93", [0x455e] = "\xe5\x94", [0x455f] = "\xe5\x95",
  [0x4560] = "\xe5\x96", [0x4561] = "\xe5\x97", [0x4562] = "\xe5\x98",
  [0x4563] = "\xe5\x99", [0x4564] = "\xe5\x9a", [0x4565] = "\xe5\x9b",
  [0x4566] = "\xe5\x9c", [0x4567] = "\xe5\x9d", [0x4568] = "\xe5\x9e",
  [0x4569] = "\xe5\x9f", [0x456a] = "\xf6\xca", [0x456b] = "\xe5\xa0",
  [0x456c] = "\xe6\x40", [0x456d] = "\xe6\x41", [0x456e] = "\xe6\x42",
  [0x456f] = "\xe6\x43", [0x4570] = "\xe6\x44", [0x4571] = "\xe6\x45",
  [0x4572] = "\xe6\x46", [0x4573] = "\xe6\x47", [0x4574] = "\xe6\x48",
  [0x4575] = "\xe6\x49", [0x4576] = "\xe6\x4a", [0x4577] = "\xe6\x4b",
  [0x4578] = "\xe6\x4c", [0x4579] = "\xe6\x4d", [0x457a] = "\xe6\x4e",
  [0x457b] = "\xe6\x4f", [0x457c] = "\xe6\x50", [0x457d] = "\xe6\x51",
  [0x457e] = "\xe6\x52", [0x457f] = "\xe6\x53", [0x4580] = "\xe6\x54",
  [0x4581] = "\xe6\x55", [0x4582] = "\xe6\x56", [0x4583] = "\xe6\x57",
  [0x4584] = "\xe6\x58", [0x4585] = "\xe6\x59", [0x4586] = "\xe6\x5a",
  [0x4587] = "\xe6\x5b", [0x4588] = "\xe6\x5c", [0x4589] = "\xe6\x5d",
  [0x458a] = "\xe6\x5e", [0x458b] = "\xe6\x5f", [0x458c] = "\xe6\x60",
  [0x458d] = "\xe6\x61", [0x458e] = "\xe6\x62", [0x458f] = "\xf6\xcc",
  [0x4590] = "\xe6\x63", [0x4591] = "\xe6\x64", [0x4592] = "\xe6\x65",
  [0x4593] = "\xe6\x66", [0x4594] = "\xe6\x67", [0x4595] = "\xe6\x68",
  [0x4596] = "\xe6\x69", [0x4597] = "\xe6\x6a", [0x4598] = "\xe6\x6b",
  [0x4599] = "\xe6\x6c", [0x459a] = "\xe6\x6d", [0x459b] = "\xe6\x6e",
  [0x459c] = "\xe6\x6f", [0x459d] = "\xe6\x70", [0x459e] = "\xe6\x71",
  [0x459f] = "\xe6\x72", [0x45a0] = "\xe6\x73", [0x45a1] = "\xe6\x74",
  [0x45a2] = "\xe6\x75", [0x45a3] = "\xe6\x76", [0x45a4] = "\xe6\x77",
  [0x45a5] = "\xe6\x78", [0x45a6] = "\xe6\x79", [0x45a7] = "\xe6\x7a",
  [0x45a8] = "\xe6\x7b", [0x45a9] = "\xe6\x7c", [0x45aa] = "\xe6\x7d",
  [0x45ab] = "\xe6\x7e", [0x45ac] = "\xe6\x80", [0x45ad] = "\xe6\x81",
  [0x45ae] = "\xe6\x82", [0x45af] = "\xe6\x83", [0x45b0] = "\xe6\x84",
  [0x45b1] = "\xe6\x85", [0x45b2] = "\xe6\x86", [0x45b3] = "\xe6\x87",
  [0x45b4] = "\xe6\x88", [0x45b5] = "\xe6\x89", [0x45b6] = "\xe6\x8a",
  [0x45b7] = "\xe6\x8b", [0x45b8] = "\xe6\x8c", [0x45b9] = "\xe6\x8d",
  [0x45ba] = "\xe6\x8e", [0x45bb] = "\xe6\x8f", [0x45bc] = "\xe6\x90",
  [0x45bd] = "\xe6\x91", [0x45be] = "\xe6\x92", [0x45bf] = "\xe6\x93",
  [0x45c0] = "\xe6\x94", [0x45c1] = "\xe6\x95", [0x45c2] = "\xe6\x96",
  [0x45c3] = "\xe6\x97", [0x45c4] = "\xe6\x98", [0x45c5] = "\xe6\x99",
  [0x45c6] = "\xe6\x9a", [0x45c7] = "\xe6\x9b", [0x45c8] = "\xe6\x9c",
  [0x45c9] = "\xe6\x9d", [0x45ca] = "\xf6\xcb", [0x45cb] = "\xe6\x9e",
  [0x45cc] = "\xe6\x9f", [0x45cd] = "\xe6\xa0", [0x45ce] = "\xe7\x40",
  [0x45cf] = "\xe7\x41", [0x45d0] = "\xe7\x42", [0x45d1] = "\xe7\x43",
  [0x45d2] = "\xe7\x44", [0x45d3] = "\xe7\x45", [0x45d4] = "\xe7\x46",
  [0x45d5] = "\xe7\x47", [0x45d6] = "\xf7\xe9", [0x45d7] = "\xe7\x48",
  [0x45d8] = "\xe7\x49", [0x45d9] = "\xe7\x4a", [0x45da] = "\xe7\x4b",
  [0x45db] = "\xe7\x4c", [0x45dc] = "\xe7\x4d", [0x45dd] = "\xe7\x4e",
  [0x45de] = "\xe7\x4f", [0x45df] = "\xe7\x50", [0x45e0] = "\xe7\x51",
  [0x45e1] = "\xe7\x52", [0x45e2] = "\xe7\x53", [0x45e3] = "\xe7\x54",
  [0x45e4] = "\xe7\x55", [0x45e5] = "\xe7\x56", [0x45e6] = "\xe7\x57",
  [0x45e7] = "\xe7\x58", [0x45e8] = "\xe7\x59", [0x45e9] = "\xe7\x5a",
  [0x45ea] = "\xe7\x5b", [0x45eb] = "\xe7\x5c", [0x45ec] = "\xe7\x5d",
  [0x45ed] = "\xe7\x5e", [0x45ee] = "\xe7\x5f", [0x45ef] = "\xe7\x60",
  [0x45f0] = "\xe7\x61", [0x45f1] = "\xe7\x62", [0x45f2] = "\xe7\x63",
  [0x45f3] = "\xe7\x64", [0x45f4] = "\xe7\x65", [0x45f5] = "\xe7\x66",
  [0x45f6] = "\xe7\x67", [0x45f7] = "\xe7\x68", [0x45f8] = "\xe7\x69",
  [0x45f9] = "\xe7\x6a", [0x45fa] = "\xe7\x6b", [0x45fb] = "\xe7\x6c",
  [0x45fc] = "\xe7\x6d", [0x45fd] = "\xe7\x6e", [0x45fe] = "\xe7\x6f",
  [0x45ff] = "\xe7\x70", [0x4600] = "\xe7\x71", [0x4601] = "\xe7\x72",
  [0x4602] = "\xe7\x73", [0x4603] = "\xe7\x74", [0x4604] = "\xe7\x75",
  [0x4605] = "\xe7\x76", [0x4606] = "\xe7\x77", [0x4607] = "\xe7\x78",
  [0x4608] = "\xe7\x79", [0x4609] = "\xe7\x7a", [0x460a] = "\xe7\x7b",
  [0x460b] = "\xe7\x7c", [0x460c] = "\xe7\x7d", [0x460d] = "\xe7\x7e",
  [0x460e] = "\xe7\x80", [0x460f] = "\xe7\x81", [0x4610] = "\xe7\x82",
  [0x4611] = "\xe7\x83", [0x4612] = "\xe7\x84", [0x4613] = "\xe7\x85",
  [0x4614] = "\xe7\x86", [0x4615] = "\xe7\x87", [0x4616] = "\xe7\x88",
  [0x4617] = "\xe7\x89", [0x4618] = "\xe7\x8a", [0x4619] = "\xe7\x8b",
  [0x461a] = "\xe7\x8c", [0x461b] = "\xe7\x8d", [0x461c] = "\xe7\x8e",
  [0x461d] = "\xe7\x8f", [0x461e] = "\xe7\x90", [0x461f] = "\xe7\x91",
  [0x4620] = "\xe7\x92", [0x4621] = "\xe7\x93", [0x4622] = "\xe7\x94",
  [0x4623] = "\xe7\x95", [0x4624] = "\xe7\x96", [0x4625] = "\xe7\x97",
  [0x4626] = "\xe7\x98", [0x4627] = "\xe7\x99", [0x4628] = "\xe7\x9a",
  [0x4629] = "\xe7\x9b", [0x462a] = "\xe7\x9c", [0x462b] = "\xe7\x9d",
  [0x462c] = "\xe7\x9e", [0x462d] = "\xe7\x9f", [0x462e] = "\xe7\xa0",
  [0x462f] = "\xe8\x40", [0x4630] = "\xe8\x41", [0x4631] = "\xe8\x42",
  [0x4632] = "\xe8\x43", [0x4633] = "\xe8\x44", [0x4634] = "\xe8\x45",
  [0x4635] = "\xe8\x46", [0x4636] = "\xe8\x47", [0x4637] = "\xe8\x48",
  [0x4638] = "\xe8\x49", [0x4639] = "\xe8\x4a", [0x463a] = "\xe8\x4b",
  [0x463b] = "\xe8\x4c", [0x463c] = "\xe8\x4d", [0x463d] = "\xe8\x4e",
  [0x463e] = "\xf6\xcd", [0x463f] = "\xe8\x4f", [0x4640] = "\xe8\x50",
  [0x4641] = "\xe8\x51", [0x4642] = "\xe8\x52", [0x4643] = "\xe8\x53",
  [0x4644] = "\xe8\x54", [0x4645] = "\xe8\x55", [0x4646] = "\xe8\x56",
  [0x4647] = "\xe8\x57", [0x4648] = "\xe8\x58", [0x4649] = "\xe8\x59",
  [0x464a] = "\xe8\x5a", [0x464b] = "\xe8\x5b", [0x464c] = "\xe8\x5c",
  [0x464d] = "\xe8\x5d", [0x464e] = "\xe8\x5e", [0x464f] = "\xe8\x5f",
  [0x4650] = "\xe8\x60", [0x4651] = "\xe8\x61", [0x4652] = "\xe8\x62",
  [0x4653] = "\xe8\x63", [0x4654] = "\xe8\x64", [0x4655] = "\xe8\x65",
  [0x4656] = "\xe8\x66", [0x4657] = "\xe8\x67", [0x4658] = "\xe8\x68",
  [0x4659] = "\xe8\x69", [0x465a] = "\xe8\x6a", [0x465b] = "\xe8\x6b",
  [0x465c] = "\xe8\x6c", [0x465d] = "\xe8\x6d", [0x465e] = "\xe8\x6e",
  [0x465f] = "\xe8\x6f", [0x4660] = "\xe8\x70", [0x4661] = "\xe8\x71",
  [0x4662] = "\xe8\x72", [0x4663] = "\xe8\x73", [0x4664] = "\xe8\x74",
  [0x4665] = "\xe8\x75", [0x4666] = "\xe8\x76", [0x4667] = "\xe8\x77",
  [0x4668] = "\xe8\x78", [0x4669] = "\xe8\x79", [0x466a] = "\xe8\x7a",
  [0x466b] = "\xf6\xce", [0x466c] = "\xe8\x7b", [0x466d] = "\xe8\x7c",
  [0x466e] = "\xe8\x7d", [0x466f] = "\xe8\x7e", [0x4670] = "\xe8\x80",
  [0x4671] = "\xe8\x81", [0x4672] = "\xe8\x82", [0x4673] = "\xe8\x83",
  [0x4674] = "\xe8\x84", [0x4675] = "\xe8\x85", [0x4676] = "\xe8\x86",
  [0x4677] = "\xe8\x87", [0x4678] = "\xe8\x88", [0x4679] = "\xe8\x89",
  [0x467a] = "\xe8\x8a", [0x467b] = "\xe8\x8b", [0x467c] = "\xe8\x8c",
  [0x467d] = "\xe8\x8d", [0x467e] = "\xe8\x8e", [0x467f] = "\xe8\x8f",
  [0x4680] = "\xe8\x90", [0x4681] = "\xe8\x91", [0x4682] = "\xe8\x92",
  [0x4683] = "\xe8\x93", [0x4684] = "\xe8\x94", [0x4685] = "\xee\xc4",
  [0x4686] = "\xee\xc5", [0x4687] = "\xee\xc6", [0x4688] = "\xd5\xeb",
  [0x4689] = "\xb6\xa4", [0x468a] = "\xee\xc8", [0x468b] = "\xee\xc7",
  [0x468c] = "\xee\xc9", [0x468d] = "\xee\xca", [0x468e] = "\xc7\xa5",
  [0x468f] = "\xee\xcb", [0x4690] = "\xee\xcc", [0x4691] = "\xe8\x95",
  [0x4692] = "\xb7\xb0", [0x4693] = "\xb5\xf6", [0x4694] = "\xee\xcd",
  [0x4695] = "\xee\xcf", [0x4696] = "\xe8\x96", [0x4697] = "\xee\xce",
  [0x4698] = "\xe8\x97", [0x4699] = "\xb8\xc6", [0x469a] = "\xee\xd0",
  [0x469b] = "\xee\xd1", [0x469c] = "\xee\xd2", [0x469d] = "\xb6\xdb",
  [0x469e] = "\xb3\xae", [0x469f] = "\xd6\xd3", [0x46a0] = "\xc4\xc6",
  [0x46a1] = "\xb1\xb5", [0x46a2] = "\xb8\xd6", [0x46a3] = "\xee\xd3",
  [0x46a4] = "\xee\xd4", [0x46a5] = "\xd4\xbf", [0x46a6] = "\xc7\xd5",
  [0x46a7] = "\xbe\xfb", [0x46a8] = "\xce\xd9", [0x46a9] = "\xb9\xb3",
  [0x46aa] = "\xee\xd6", [0x46ab] = "\xee\xd5", [0x46ac] = "\xee\xd8",
  [0x46ad] = "\xee\xd7", [0x46ae] = "\xc5\xa5", [0x46af] = "\xee\xd9",
  [0x46b0] = "\xee\xda", [0x46b1] = "\xc7\xae", [0x46b2] = "\xee\xdb",
  [0x46b3] = "\xc7\xaf", [0x46b4] = "\xee\xdc", [0x46b5] = "\xb2\xa7",
  [0x46b6] = "\xee\xdd", [0x46b7] = "\xee\xde", [0x46b8] = "\xee\xdf",
  [0x46b9] = "\xee\xe0", [0x46ba] = "\xee\xe1", [0x46bb] = "\xd7\xea",
  [0x46bc] = "\xee\xe2", [0x46bd] = "\xee\xe3", [0x46be] = "\xbc\xd8",
  [0x46bf] = "\xee\xe4", [0x46c0] = "\xd3\xcb", [0x46c1] = "\xcc\xfa",
  [0x46c2] = "\xb2\xac", [0x46c3] = "\xc1\xe5", [0x46c4] = "\xee\xe5",
  [0x46c5] = "\xc7\xa6", [0x46c6] = "\xc3\xad", [0x46c7] = "\xe8\x98",
  [0x46c8] = "\xee\xe6", [0x46c9] = "\xee\xe7", [0x46ca] = "\xee\xe8",
  [0x46cb] = "\xee\xe9", [0x46cc] = "\xee\xea", [0x46cd] = "\xee\xeb",
  [0x46ce] = "\xee\xec", [0x46cf] = "\xe8\x99", [0x46d0] = "\xee\xed",
  [0x46d1] = "\xee\xee", [0x46d2] = "\xee\xef", [0x46d3] = "\xe8\x9a",
  [0x46d4] = "\xe8\x9b", [0x46d5] = "\xee\xf0", [0x46d6] = "\xee\xf1",
  [0x46d7] = "\xee\xf2", [0x46d8] = "\xee\xf4", [0x46d9] = "\xee\xf3",
  [0x46da] = "\xe8\x9c", [0x46db] = "\xee\xf5", [0x46dc] = "\xcd\xad",
  [0x46dd] = "\xc2\xc1", [0x46de] = "\xee\xf6", [0x46df] = "\xee\xf7",
  [0x46e0] = "\xee\xf8", [0x46e1] = "\xd5\xa1", [0x46e2] = "\xee\xf9",
  [0x46e3] = "\xcf\xb3", [0x46e4] = "\xee\xfa", [0x46e5] = "\xee\xfb",
  [0x46e6] = "\xe8\x9d", [0x46e7] = "\xee\xfc", [0x46e8] = "\xee\xfd",
  [0x46e9] = "\xef\xa1", [0x46ea] = "\xee\xfe", [0x46eb] = "\xef\xa2",
  [0x46ec] = "\xb8\xf5", [0x46ed] = "\xc3\xfa", [0x46ee] = "\xef\xa3",
  [0x46ef] = "\xef\xa4", [0x46f0] = "\xbd\xc2", [0x46f1] = "\xd2\xbf",
  [0x46f2] = "\xb2\xf9", [0x46f3] = "\xef\xa5", [0x46f4] = "\xef\xa6",
  [0x46f5] = "\xef\xa7", [0x46f6] = "\xd2\xf8", [0x46f7] = "\xef\xa8",
  [0x46f8] = "\xd6\xfd", [0x46f9] = "\xef\xa9", [0x46fa] = "\xc6\xcc",
  [0x46fb] = "\xe8\x9e", [0x46fc] = "\xef\xaa", [0x46fd] = "\xef\xab",
  [0x46fe] = "\xc1\xb4", [0x46ff] = "\xef\xac", [0x4700] = "\xcf\xfa",
  [0x4701] = "\xcb\xf8", [0x4702] = "\xef\xae", [0x4703] = "\xef\xad",
  [0x4704] = "\xb3\xfa", [0x4705] = "\xb9\xf8", [0x4706] = "\xef\xaf",
  [0x4707] = "\xef\xb0", [0x4708] = "\xd0\xe2", [0x4709] = "\xef\xb1",
  [0x470a] = "\xef\xb2", [0x470b] = "\xb7\xe6", [0x470c] = "\xd0\xbf",
  [0x470d] = "\xef\xb3", [0x470e] = "\xef\xb4", [0x470f] = "\xef\xb5",
  [0x4710] = "\xc8\xf1", [0x4711] = "\xcc\xe0", [0x4712] = "\xef\xb6",
  [0x4713] = "\xef\xb7", [0x4714] = "\xef\xb8", [0x4715] = "\xef\xb9",
  [0x4716] = "\xef\xba", [0x4717] = "\xd5\xe0", [0x4718] = "\xef\xbb",
  [0x4719] = "\xb4\xed", [0x471a] = "\xc3\xaa", [0x471b] = "\xef\xbc",
  [0x471c] = "\xe8\x9f", [0x471d] = "\xef\xbd", [0x471e] = "\xef\xbe",
  [0x471f] = "\xef\xbf", [0x4720] = "\xe8\xa0", [0x4721] = "\xce\xfd",
  [0x4722] = "\xef\xc0", [0x4723] = "\xc2\xe0", [0x4724] = "\xb4\xb8",
  [0x4725] = "\xd7\xb6", [0x4726] = "\xbd\xf5", [0x4727] = "\xe9\x40",
  [0x4728] = "\xcf\xc7", [0x4729] = "\xef\xc3", [0x472a] = "\xef\xc1",
  [0x472b] = "\xef\xc2", [0x472c] = "\xef\xc4", [0x472d] = "\xb6\xa7",
  [0x472e] = "\xbc\xfc", [0x472f] = "\xbe\xe2", [0x4730] = "\xc3\xcc",
  [0x4731] = "\xef\xc5", [0x4732] = "\xef\xc6", [0x4733] = "\xe9\x41",
  [0x4734] = "\xef\xc7", [0x4735] = "\xef\xcf", [0x4736] = "\xef\xc8",
  [0x4737] = "\xef\xc9", [0x4738] = "\xef\xca", [0x4739] = "\xc7\xc2",
  [0x473a] = "\xef\xf1", [0x473b] = "\xb6\xcd", [0x473c] = "\xef\xcb",
  [0x473d] = "\xe9\x42", [0x473e] = "\xef\xcc", [0x473f] = "\xef\xcd",
  [0x4740] = "\xb6\xc6", [0x4741] = "\xc3\xbe", [0x4742] = "\xef\xce",
  [0x4743] = "\xe9\x43", [0x4744] = "\xef\xd0", [0x4745] = "\xef\xd1",
  [0x4746] = "\xef\xd2", [0x4747] = "\xd5\xf2", [0x4748] = "\xe9\x44",
  [0x4749] = "\xef\xd3", [0x474a] = "\xc4\xf7", [0x474b] = "\xe9\x45",
  [0x474c] = "\xef\xd4", [0x474d] = "\xc4\xf8", [0x474e] = "\xef\xd5",
  [0x474f] = "\xef\xd6", [0x4750] = "\xb8\xe4", [0x4751] = "\xb0\xf7",
  [0x4752] = "\xef\xd7", [0x4753] = "\xef\xd8", [0x4754] = "\xef\xd9",
  [0x4755] = "\xe9\x46", [0x4756] = "\xef\xda", [0x4757] = "\xef\xdb",
  [0x4758] = "\xef\xdc", [0x4759] = "\xef\xdd", [0x475a] = "\xe9\x47",
  [0x475b] = "\xef\xde", [0x475c] = "\xbe\xb5", [0x475d] = "\xef\xe1",
  [0x475e] = "\xef\xdf", [0x475f] = "\xef\xe0", [0x4760] = "\xe9\x48",
  [0x4761] = "\xef\xe2", [0x4762] = "\xef\xe3", [0x4763] = "\xc1\xcd",
  [0x4764] = "\xef\xe4", [0x4765] = "\xef\xe5", [0x4766] = "\xef\xe6",
  [0x4767] = "\xef\xe7", [0x4768] = "\xef\xe8", [0x4769] = "\xef\xe9",
  [0x476a] = "\xef\xea", [0x476b] = "\xef\xeb", [0x476c] = "\xef\xec",
  [0x476d] = "\xc0\xd8", [0x476e] = "\xe9\x49", [0x476f] = "\xef\xed",
  [0x4770] = "\xc1\xad", [0x4771] = "\xef\xee", [0x4772] = "\xef\xef",
  [0x4773] = "\xef\xf0", [0x4774] = "\xe9\x4a", [0x4775] = "\xe9\x4b",
  [0x4776] = "\xcf\xe2", [0x4777] = "\xe9\x4c", [0x4778] = "\xe9\x4d",
  [0x4779] = "\xe9\x4e", [0x477a] = "\xe9\x4f", [0x477b] = "\xe9\x50",
  [0x477c] = "\xe9\x51", [0x477d] = "\xe9\x52", [0x477e] = "\xe9\x53",
  [0x477f] = "\xb3\xa4", [0x4780] = "\xe9\x54", [0x4781] = "\xe9\x55",
  [0x4782] = "\xe9\x56", [0x4783] = "\xe9\x57", [0x4784] = "\xe9\x58",
  [0x4785] = "\xe9\x59", [0x4786] = "\xe9\x5a", [0x4787] = "\xe9\x5b",
  [0x4788] = "\xe9\x5c", [0x4789] = "\xe9\x5d", [0x478a] = "\xe9\x5e",
  [0x478b] = "\xe9\x5f", [0x478c] = "\xe9\x60", [0x478d] = "\xe9\x61",
  [0x478e] = "\xe9\x62", [0x478f] = "\xe9\x63", [0x4790] = "\xe9\x64",
  [0x4791] = "\xe9\x65", [0x4792] = "\xe9\x66", [0x4793] = "\xe9\x67",
  [0x4794] = "\xe9\x68", [0x4795] = "\xe9\x69", [0x4796] = "\xe9\x6a",
  [0x4797] = "\xe9\x6b", [0x4798] = "\xe9\x6c", [0x4799] = "\xe9\x6d",
  [0x479a] = "\xe9\x6e", [0x479b] = "\xe9\x6f", [0x479c] = "\xe9\x70",
  [0x479d] = "\xe9\x71", [0x479e] = "\xe9\x72", [0x479f] = "\xe9\x73",
  [0x47a0] = "\xe9\x74", [0x47a1] = "\xe9\x75", [0x47a2] = "\xe9\x76",
  [0x47a3] = "\xe9\x77", [0x47a4] = "\xe9\x78", [0x47a5] = "\xe9\x79",
  [0x47a6] = "\xe9\x7a", [0x47a7] = "\xe9\x7b", [0x47a8] = "\xe9\x7c",
  [0x47a9] = "\xe9\x7d", [0x47aa] = "\xe9\x7e", [0x47ab] = "\xe9\x80",
  [0x47ac] = "\xe9\x81", [0x47ad] = "\xe9\x82", [0x47ae] = "\xe9\x83",
  [0x47af] = "\xe9\x84", [0x47b0] = "\xe9\x85", [0x47b1] = "\xe9\x86",
  [0x47b2] = "\xe9\x87", [0x47b3] = "\xe9\x88", [0x47b4] = "\xe9\x89",
  [0x47b5] = "\xe9\x8a", [0x47b6] = "\xe9\x8b", [0x47b7] = "\xe9\x8c",
  [0x47b8] = "\xe9\x8d", [0x47b9] = "\xe9\x8e", [0x47ba] = "\xe9\x8f",
  [0x47bb] = "\xe9\x90", [0x47bc] = "\xe9\x91", [0x47bd] = "\xe9\x92",
  [0x47be] = "\xe9\x93", [0x47bf] = "\xe9\x94", [0x47c0] = "\xe9\x95",
  [0x47c1] = "\xe9\x96", [0x47c2] = "\xe9\x97", [0x47c3] = "\xe9\x98",
  [0x47c4] = "\xe9\x99", [0x47c5] = "\xe9\x9a", [0x47c6] = "\xe9\x9b",
  [0x47c7] = "\xe9\x9c", [0x47c8] = "\xe9\x9d", [0x47c9] = "\xe9\x9e",
  [0x47ca] = "\xe9\x9f", [0x47cb] = "\xe9\xa0", [0x47cc] = "\xea\x40",
  [0x47cd] = "\xea\x41", [0x47ce] = "\xea\x42", [0x47cf] = "\xea\x43",
  [0x47d0] = "\xea\x44", [0x47d1] = "\xea\x45", [0x47d2] = "\xea\x46",
  [0x47d3] = "\xea\x47", [0x47d4] = "\xea\x48", [0x47d5] = "\xea\x49",
  [0x47d6] = "\xea\x4a", [0x47d7] = "\xea\x4b", [0x47d8] = "\xea\x4c",
  [0x47d9] = "\xea\x4d", [0x47da] = "\xea\x4e", [0x47db] = "\xea\x4f",
  [0x47dc] = "\xea\x50", [0x47dd] = "\xea\x51", [0x47de] = "\xea\x52",
  [0x47df] = "\xea\x53", [0x47e0] = "\xea\x54", [0x47e1] = "\xea\x55",
  [0x47e2] = "\xea\x56", [0x47e3] = "\xea\x57", [0x47e4] = "\xea\x58",
  [0x47e5] = "\xea\x59", [0x47e6] = "\xea\x5a", [0x47e7] = "\xea\x5b",
  [0x47e8] = "\xc3\xc5", [0x47e9] = "\xe3\xc5", [0x47ea] = "\xc9\xc1",
  [0x47eb] = "\xe3\xc6", [0x47ec] = "\xea\x5c", [0x47ed] = "\xb1\xd5",
  [0x47ee] = "\xce\xca", [0x47ef] = "\xb4\xb3", [0x47f0] = "\xc8\xf2",
  [0x47f1] = "\xe3\xc7", [0x47f2] = "\xcf\xd0", [0x47f3] = "\xe3\xc8",
  [0x47f4] = "\xbc\xe4", [0x47f5] = "\xe3\xc9", [0x47f6] = "\xe3\xca",
  [0x47f7] = "\xc3\xc6", [0x47f8] = "\xd5\xa2", [0x47f9] = "\xc4\xd6",
  [0x47fa] = "\xb9\xeb", [0x47fb] = "\xce\xc5", [0x47fc] = "\xe3\xcb",
  [0x47fd] = "\xc3\xf6", [0x47fe] = "\xe3\xcc", [0x47ff] = "\xea\x5d",
  [0x4800] = "\xb7\xa7", [0x4801] = "\xb8\xf3", [0x4802] = "\xba\xd2",
  [0x4803] = "\xe3\xcd", [0x4804] = "\xe3\xce", [0x4805] = "\xd4\xc4",
  [0x4806] = "\xe3\xcf", [0x4807] = "\xea\x5e", [0x4808] = "\xe3\xd0",
  [0x4809] = "\xd1\xcb", [0x480a] = "\xe3\xd1", [0x480b] = "\xe3\xd2",
  [0x480c] = "\xe3\xd3", [0x480d] = "\xe3\xd4", [0x480e] = "\xd1\xd6",
  [0x480f] = "\xe3\xd5", [0x4810] = "\xb2\xfb", [0x4811] = "\xc0\xbb",
  [0x4812] = "\xe3\xd6", [0x4813] = "\xea\x5f", [0x4814] = "\xc0\xab",
  [0x4815] = "\xe3\xd7", [0x4816] = "\xe3\xd8", [0x4817] = "\xe3\xd9",
  [0x4818] = "\xea\x60", [0x4819] = "\xe3\xda", [0x481a] = "\xe3\xdb",
  [0x481b] = "\xea\x61", [0x481c] = "\xb8\xb7", [0x481d] = "\xda\xe2",
  [0x481e] = "\xea\x62", [0x481f] = "\xb6\xd3", [0x4820] = "\xea\x63",
  [0x4821] = "\xda\xe4", [0x4822] = "\xda\xe3", [0x4823] = "\xea\x64",
  [0x4824] = "\xea\x65", [0x4825] = "\xea\x66", [0x4826] = "\xea\x67",
  [0x4827] = "\xea\x68", [0x4828] = "\xea\x69", [0x4829] = "\xea\x6a",
  [0x482a] = "\xda\xe6", [0x482b] = "\xea\x6b", [0x482c] = "\xea\x6c",
  [0x482d] = "\xea\x6d", [0x482e] = "\xc8\xee", [0x482f] = "\xea\x6e",
  [0x4830] = "\xea\x6f", [0x4831] = "\xda\xe5", [0x4832] = "\xb7\xc0",
  [0x4833] = "\xd1\xf4", [0x4834] = "\xd2\xf5", [0x4835] = "\xd5\xf3",
  [0x4836] = "\xbd\xd7", [0x4837] = "\xea\x70", [0x4838] = "\xea\x71",
  [0x4839] = "\xea\x72", [0x483a] = "\xea\x73", [0x483b] = "\xd7\xe8",
  [0x483c] = "\xda\xe8", [0x483d] = "\xda\xe7", [0x483e] = "\xea\x74",
  [0x483f] = "\xb0\xa2", [0x4840] = "\xcd\xd3", [0x4841] = "\xea\x75",
  [0x4842] = "\xda\xe9", [0x4843] = "\xea\x76", [0x4844] = "\xb8\xbd",
  [0x4845] = "\xbc\xca", [0x4846] = "\xc2\xbd", [0x4847] = "\xc2\xa4",
  [0x4848] = "\xb3\xc2", [0x4849] = "\xda\xea", [0x484a] = "\xea\x77",
  [0x484b] = "\xc2\xaa", [0x484c] = "\xc4\xb0", [0x484d] = "\xbd\xb5",
  [0x484e] = "\xea\x78", [0x484f] = "\xea\x79", [0x4850] = "\xcf\xde",
  [0x4851] = "\xea\x7a", [0x4852] = "\xea\x7b", [0x4853] = "\xea\x7c",
  [0x4854] = "\xda\xeb", [0x4855] = "\xc9\xc2", [0x4856] = "\xea\x7d",
  [0x4857] = "\xea\x7e", [0x4858] = "\xea\x80", [0x4859] = "\xea\x81",
  [0x485a] = "\xea\x82", [0x485b] = "\xb1\xdd", [0x485c] = "\xea\x83",
  [0x485d] = "\xea\x84", [0x485e] = "\xea\x85", [0x485f] = "\xda\xec",
  [0x4860] = "\xea\x86", [0x4861] = "\xb6\xb8", [0x4862] = "\xd4\xba",
  [0x4863] = "\xea\x87", [0x4864] = "\xb3\xfd", [0x4865] = "\xea\x88",
  [0x4866] = "\xea\x89", [0x4867] = "\xda\xed", [0x4868] = "\xd4\xc9",
  [0x4869] = "\xcf\xd5", [0x486a] = "\xc5\xe3", [0x486b] = "\xea\x8a",
  [0x486c] = "\xda\xee", [0x486d] = "\xea\x8b", [0x486e] = "\xea\x8c",
  [0x486f] = "\xea\x8d", [0x4870] = "\xea\x8e", [0x4871] = "\xea\x8f",
  [0x4872] = "\xda\xef", [0x4873] = "\xea\x90", [0x4874] = "\xda\xf0",
  [0x4875] = "\xc1\xea", [0x4876] = "\xcc\xd5", [0x4877] = "\xcf\xdd",
  [0x4878] = "\xea\x91", [0x4879] = "\xea\x92", [0x487a] = "\xea\x93",
  [0x487b] = "\xea\x94", [0x487c] = "\xea\x95", [0x487d] = "\xea\x96",
  [0x487e] = "\xea\x97", [0x487f] = "\xea\x98", [0x4880] = "\xea\x99",
  [0x4881] = "\xea\x9a", [0x4882] = "\xea\x9b", [0x4883] = "\xea\x9c",
  [0x4884] = "\xea\x9d", [0x4885] = "\xd3\xe7", [0x4886] = "\xc2\xa1",
  [0x4887] = "\xea\x9e", [0x4888] = "\xda\xf1", [0x4889] = "\xea\x9f",
  [0x488a] = "\xea\xa0", [0x488b] = "\xcb\xe5", [0x488c] = "\xeb\x40",
  [0x488d] = "\xda\xf2", [0x488e] = "\xeb\x41", [0x488f] = "\xcb\xe6",
  [0x4890] = "\xd2\xfe", [0x4891] = "\xeb\x42", [0x4892] = "\xeb\x43",
  [0x4893] = "\xeb\x44", [0x4894] = "\xb8\xf4", [0x4895] = "\xeb\x45",
  [0x4896] = "\xeb\x46", [0x4897] = "\xda\xf3", [0x4898] = "\xb0\xaf",
  [0x4899] = "\xcf\xb6", [0x489a] = "\xeb\x47", [0x489b] = "\xeb\x48",
  [0x489c] = "\xd5\xcf", [0x489d] = "\xeb\x49", [0x489e] = "\xeb\x4a",
  [0x489f] = "\xeb\x4b", [0x48a0] = "\xeb\x4c", [0x48a1] = "\xeb\x4d",
  [0x48a2] = "\xeb\x4e", [0x48a3] = "\xeb\x4f", [0x48a4] = "\xeb\x50",
  [0x48a5] = "\xeb\x51", [0x48a6] = "\xeb\x52", [0x48a7] = "\xcb\xed",
  [0x48a8] = "\xeb\x53", [0x48a9] = "\xeb\x54", [0x48aa] = "\xeb\x55",
  [0x48ab] = "\xeb\x56", [0x48ac] = "\xeb\x57", [0x48ad] = "\xeb\x58",
  [0x48ae] = "\xeb\x59", [0x48af] = "\xeb\x5a", [0x48b0] = "\xda\xf4",
  [0x48b1] = "\xeb\x5b", [0x48b2] = "\xeb\x5c", [0x48b3] = "\xe3\xc4",
  [0x48b4] = "\xeb\x5d", [0x48b5] = "\xeb\x5e", [0x48b6] = "\xc1\xa5",
  [0x48b7] = "\xeb\x5f", [0x48b8] = "\xeb\x60", [0x48b9] = "\xf6\xbf",
  [0x48ba] = "\xeb\x61", [0x48bb] = "\xeb\x62", [0x48bc] = "\xf6\xc0",
  [0x48bd] = "\xf6\xc1", [0x48be] = "\xc4\xd1", [0x48bf] = "\xeb\x63",
  [0x48c0] = "\xc8\xb8", [0x48c1] = "\xd1\xe3", [0x48c2] = "\xeb\x64",
  [0x48c3] = "\xeb\x65", [0x48c4] = "\xd0\xdb", [0x48c5] = "\xd1\xc5",
  [0x48c6] = "\xbc\xaf", [0x48c7] = "\xb9\xcd", [0x48c8] = "\xeb\x66",
  [0x48c9] = "\xef\xf4", [0x48ca] = "\xeb\x67", [0x48cb] = "\xeb\x68",
  [0x48cc] = "\xb4\xc6", [0x48cd] = "\xd3\xba", [0x48ce] = "\xf6\xc2",
  [0x48cf] = "\xb3\xfb", [0x48d0] = "\xeb\x69", [0x48d1] = "\xeb\x6a",
  [0x48d2] = "\xf6\xc3", [0x48d3] = "\xeb\x6b", [0x48d4] = "\xeb\x6c",
  [0x48d5] = "\xb5\xf1", [0x48d6] = "\xeb\x6d", [0x48d7] = "\xeb\x6e",
  [0x48d8] = "\xeb\x6f", [0x48d9] = "\xeb\x70", [0x48da] = "\xeb\x71",
  [0x48db] = "\xeb\x72", [0x48dc] = "\xeb\x73", [0x48dd] = "\xeb\x74",
  [0x48de] = "\xeb\x75", [0x48df] = "\xeb\x76", [0x48e0] = "\xf6\xc5",
  [0x48e1] = "\xeb\x77", [0x48e2] = "\xeb\x78", [0x48e3] = "\xeb\x79",
  [0x48e4] = "\xeb\x7a", [0x48e5] = "\xeb\x7b", [0x48e6] = "\xeb\x7c",
  [0x48e7] = "\xeb\x7d", [0x48e8] = "\xd3\xea", [0x48e9] = "\xf6\xa7",
  [0x48ea] = "\xd1\xa9", [0x48eb] = "\xeb\x7e", [0x48ec] = "\xeb\x80",
  [0x48ed] = "\xeb\x81", [0x48ee] = "\xeb\x82", [0x48ef] = "\xf6\xa9",
  [0x48f0] = "\xeb\x83", [0x48f1] = "\xeb\x84", [0x48f2] = "\xeb\x85",
  [0x48f3] = "\xf6\xa8", [0x48f4] = "\xeb\x86", [0x48f5] = "\xeb\x87",
  [0x48f6] = "\xc1\xe3", [0x48f7] = "\xc0\xd7", [0x48f8] = "\xeb\x88",
  [0x48f9] = "\xb1\xa2", [0x48fa] = "\xeb\x89", [0x48fb] = "\xeb\x8a",
  [0x48fc] = "\xeb\x8b", [0x48fd] = "\xeb\x8c", [0x48fe] = "\xce\xed",
  [0x48ff] = "\xeb\x8d", [0x4900] = "\xd0\xe8", [0x4901] = "\xf6\xab",
  [0x4902] = "\xeb\x8e", [0x4903] = "\xeb\x8f", [0x4904] = "\xcf\xf6",
  [0x4905] = "\xeb\x90", [0x4906] = "\xf6\xaa", [0x4907] = "\xd5\xf0",
  [0x4908] = "\xf6\xac", [0x4909] = "\xc3\xb9", [0x490a] = "\xeb\x91",
  [0x490b] = "\xeb\x92", [0x490c] = "\xeb\x93", [0x490d] = "\xbb\xf4",
  [0x490e] = "\xf6\xae", [0x490f] = "\xf6\xad", [0x4910] = "\xeb\x94",
  [0x4911] = "\xeb\x95", [0x4912] = "\xeb\x96", [0x4913] = "\xc4\xde",
  [0x4914] = "\xeb\x97", [0x4915] = "\xeb\x98", [0x4916] = "\xc1\xd8",
  [0x4917] = "\xeb\x99", [0x4918] = "\xeb\x9a", [0x4919] = "\xeb\x9b",
  [0x491a] = "\xeb\x9c", [0x491b] = "\xeb\x9d", [0x491c] = "\xcb\xaa",
  [0x491d] = "\xeb\x9e", [0x491e] = "\xcf\xbc", [0x491f] = "\xeb\x9f",
  [0x4920] = "\xeb\xa0", [0x4921] = "\xec\x40", [0x4922] = "\xec\x41",
  [0x4923] = "\xec\x42", [0x4924] = "\xec\x43", [0x4925] = "\xec\x44",
  [0x4926] = "\xec\x45", [0x4927] = "\xec\x46", [0x4928] = "\xec\x47",
  [0x4929] = "\xec\x48", [0x492a] = "\xf6\xaf", [0x492b] = "\xec\x49",
  [0x492c] = "\xec\x4a", [0x492d] = "\xf6\xb0", [0x492e] = "\xec\x4b",
  [0x492f] = "\xec\x4c", [0x4930] = "\xf6\xb1", [0x4931] = "\xec\x4d",
  [0x4932] = "\xc2\xb6", [0x4933] = "\xec\x4e", [0x4934] = "\xec\x4f",
  [0x4935] = "\xec\x50", [0x4936] = "\xec\x51", [0x4937] = "\xec\x52",
  [0x4938] = "\xb0\xd4", [0x4939] = "\xc5\xf9", [0x493a] = "\xec\x53",
  [0x493b] = "\xec\x54", [0x493c] = "\xec\x55", [0x493d] = "\xec\x56",
  [0x493e] = "\xf6\xb2", [0x493f] = "\xec\x57", [0x4940] = "\xec\x58",
  [0x4941] = "\xec\x59", [0x4942] = "\xec\x5a", [0x4943] = "\xec\x5b",
  [0x4944] = "\xec\x5c", [0x4945] = "\xec\x5d", [0x4946] = "\xec\x5e",
  [0x4947] = "\xec\x5f", [0x4948] = "\xec\x60", [0x4949] = "\xec\x61",
  [0x494a] = "\xec\x62", [0x494b] = "\xec\x63", [0x494c] = "\xec\x64",
  [0x494d] = "\xec\x65", [0x494e] = "\xec\x66", [0x494f] = "\xec\x67",
  [0x4950] = "\xec\x68", [0x4951] = "\xec\x69", [0x4952] = "\xc7\xe0",
  [0x4953] = "\xf6\xa6", [0x4954] = "\xec\x6a", [0x4955] = "\xec\x6b",
  [0x4956] = "\xbe\xb8", [0x4957] = "\xec\x6c", [0x4958] = "\xec\x6d",
  [0x4959] = "\xbe\xb2", [0x495a] = "\xec\x6e", [0x495b] = "\xb5\xe5",
  [0x495c] = "\xec\x6f", [0x495d] = "\xec\x70", [0x495e] = "\xb7\xc7",
  [0x495f] = "\xec\x71", [0x4960] = "\xbf\xbf", [0x4961] = "\xc3\xd2",
  [0x4962] = "\xc3\xe6", [0x4963] = "\xec\x72", [0x4964] = "\xec\x73",
  [0x4965] = "\xd8\xcc", [0x4966] = "\xec\x74", [0x4967] = "\xec\x75",
  [0x4968] = "\xec\x76", [0x4969] = "\xb8\xef", [0x496a] = "\xec\x77",
  [0x496b] = "\xec\x78", [0x496c] = "\xec\x79", [0x496d] = "\xec\x7a",
  [0x496e] = "\xec\x7b", [0x496f] = "\xec\x7c", [0x4970] = "\xec\x7d",
  [0x4971] = "\xec\x7e", [0x4972] = "\xec\x80", [0x4973] = "\xbd\xf9",
  [0x4974] = "\xd1\xa5", [0x4975] = "\xec\x81", [0x4976] = "\xb0\xd0",
  [0x4977] = "\xec\x82", [0x4978] = "\xec\x83", [0x4979] = "\xec\x84",
  [0x497a] = "\xec\x85", [0x497b] = "\xec\x86", [0x497c] = "\xf7\xb0",
  [0x497d] = "\xec\x87", [0x497e] = "\xec\x88", [0x497f] = "\xec\x89",
  [0x4980] = "\xec\x8a", [0x4981] = "\xec\x8b", [0x4982] = "\xec\x8c",
  [0x4983] = "\xec\x8d", [0x4984] = "\xec\x8e", [0x4985] = "\xf7\xb1",
  [0x4986] = "\xec\x8f", [0x4987] = "\xec\x90", [0x4988] = "\xec\x91",
  [0x4989] = "\xec\x92", [0x498a] = "\xec\x93", [0x498b] = "\xd0\xac",
  [0x498c] = "\xec\x94", [0x498d] = "\xb0\xb0", [0x498e] = "\xec\x95",
  [0x498f] = "\xec\x96", [0x4990] = "\xec\x97", [0x4991] = "\xf7\xb2",
  [0x4992] = "\xf7\xb3", [0x4993] = "\xec\x98", [0x4994] = "\xf7\xb4",
  [0x4995] = "\xec\x99", [0x4996] = "\xec\x9a", [0x4997] = "\xec\x9b",
  [0x4998] = "\xc7\xca", [0x4999] = "\xec\x9c", [0x499a] = "\xec\x9d",
  [0x499b] = "\xec\x9e", [0x499c] = "\xec\x9f", [0x499d] = "\xec\xa0",
  [0x499e] = "\xed\x40", [0x499f] = "\xed\x41", [0x49a0] = "\xbe\xcf",
  [0x49a1] = "\xed\x42", [0x49a2] = "\xed\x43", [0x49a3] = "\xf7\xb7",
  [0x49a4] = "\xed\x44", [0x49a5] = "\xed\x45", [0x49a6] = "\xed\x46",
  [0x49a7] = "\xed\x47", [0x49a8] = "\xed\x48", [0x49a9] = "\xed\x49",
  [0x49aa] = "\xed\x4a", [0x49ab] = "\xf7\xb6", [0x49ac] = "\xed\x4b",
  [0x49ad] = "\xb1\xde", [0x49ae] = "\xed\x4c", [0x49af] = "\xf7\xb5",
  [0x49b0] = "\xed\x4d", [0x49b1] = "\xed\x4e", [0x49b2] = "\xf7\xb8",
  [0x49b3] = "\xed\x4f", [0x49b4] = "\xf7\xb9", [0x49b5] = "\xed\x50",
  [0x49b6] = "\xed\x51", [0x49b7] = "\xed\x52", [0x49b8] = "\xed\x53",
  [0x49b9] = "\xed\x54", [0x49ba] = "\xed\x55", [0x49bb] = "\xed\x56",
  [0x49bc] = "\xed\x57", [0x49bd] = "\xed\x58", [0x49be] = "\xed\x59",
  [0x49bf] = "\xed\x5a", [0x49c0] = "\xed\x5b", [0x49c1] = "\xed\x5c",
  [0x49c2] = "\xed\x5d", [0x49c3] = "\xed\x5e", [0x49c4] = "\xed\x5f",
  [0x49c5] = "\xed\x60", [0x49c6] = "\xed\x61", [0x49c7] = "\xed\x62",
  [0x49c8] = "\xed\x63", [0x49c9] = "\xed\x64", [0x49ca] = "\xed\x65",
  [0x49cb] = "\xed\x66", [0x49cc] = "\xed\x67", [0x49cd] = "\xed\x68",
  [0x49ce] = "\xed\x69", [0x49cf] = "\xed\x6a", [0x49d0] = "\xed\x6b",
  [0x49d1] = "\xed\x6c", [0x49d2] = "\xed\x6d", [0x49d3] = "\xed\x6e",
  [0x49d4] = "\xed\x6f", [0x49d5] = "\xed\x70", [0x49d6] = "\xed\x71",
  [0x49d7] = "\xed\x72", [0x49d8] = "\xed\x73", [0x49d9] = "\xed\x74",
  [0x49da] = "\xed\x75", [0x49db] = "\xed\x76", [0x49dc] = "\xed\x77",
  [0x49dd] = "\xed\x78", [0x49de] = "\xed\x79", [0x49df] = "\xed\x7a",
  [0x49e0] = "\xed\x7b", [0x49e1] = "\xed\x7c", [0x49e2] = "\xed\x7d",
  [0x49e3] = "\xed\x7e", [0x49e4] = "\xed\x80", [0x49e5] = "\xed\x81",
  [0x49e6] = "\xce\xa4", [0x49e7] = "\xc8\xcd", [0x49e8] = "\xed\x82",
  [0x49e9] = "\xba\xab", [0x49ea] = "\xe8\xb8", [0x49eb] = "\xe8\xb9",
  [0x49ec] = "\xe8\xba", [0x49ed] = "\xbe\xc2", [0x49ee] = "\xed\x83",
  [0x49ef] = "\xed\x84", [0x49f0] = "\xed\x85", [0x49f1] = "\xed\x86",
  [0x49f2] = "\xed\x87", [0x49f3] = "\xd2\xf4", [0x49f4] = "\xed\x88",
  [0x49f5] = "\xd4\xcf", [0x49f6] = "\xc9\xd8", [0x49f7] = "\xed\x89",
  [0x49f8] = "\xed\x8a", [0x49f9] = "\xed\x8b", [0x49fa] = "\xed\x8c",
  [0x49fb] = "\xed\x8d", [0x49fc] = "\xed\x8e", [0x49fd] = "\xed\x8f",
  [0x49fe] = "\xed\x90", [0x49ff] = "\xed\x91", [0x4a00] = "\xed\x92",
  [0x4a01] = "\xed\x93", [0x4a02] = "\xed\x94", [0x4a03] = "\xed\x95",
  [0x4a04] = "\xed\x96", [0x4a05] = "\xed\x97", [0x4a06] = "\xed\x98",
  [0x4a07] = "\xed\x99", [0x4a08] = "\xed\x9a", [0x4a09] = "\xed\x9b",
  [0x4a0a] = "\xed\x9c", [0x4a0b] = "\xed\x9d", [0x4a0c] = "\xed\x9e",
  [0x4a0d] = "\xed\x9f", [0x4a0e] = "\xed\xa0", [0x4a0f] = "\xee\x40",
  [0x4a10] = "\xee\x41", [0x4a11] = "\xee\x42", [0x4a12] = "\xee\x43",
  [0x4a13] = "\xee\x44", [0x4a14] = "\xee\x45", [0x4a15] = "\xee\x46",
  [0x4a16] = "\xee\x47", [0x4a17] = "\xee\x48", [0x4a18] = "\xee\x49",
  [0x4a19] = "\xee\x4a", [0x4a1a] = "\xee\x4b", [0x4a1b] = "\xee\x4c",
  [0x4a1c] = "\xee\x4d", [0x4a1d] = "\xee\x4e", [0x4a1e] = "\xee\x4f",
  [0x4a1f] = "\xee\x50", [0x4a20] = "\xee\x51", [0x4a21] = "\xee\x52",
  [0x4a22] = "\xee\x53", [0x4a23] = "\xee\x54", [0x4a24] = "\xee\x55",
  [0x4a25] = "\xee\x56", [0x4a26] = "\xee\x57", [0x4a27] = "\xee\x58",
  [0x4a28] = "\xee\x59", [0x4a29] = "\xee\x5a", [0x4a2a] = "\xee\x5b",
  [0x4a2b] = "\xee\x5c", [0x4a2c] = "\xee\x5d", [0x4a2d] = "\xee\x5e",
  [0x4a2e] = "\xee\x5f", [0x4a2f] = "\xee\x60", [0x4a30] = "\xee\x61",
  [0x4a31] = "\xee\x62", [0x4a32] = "\xee\x63", [0x4a33] = "\xee\x64",
  [0x4a34] = "\xee\x65", [0x4a35] = "\xee\x66", [0x4a36] = "\xee\x67",
  [0x4a37] = "\xee\x68", [0x4a38] = "\xee\x69", [0x4a39] = "\xee\x6a",
  [0x4a3a] = "\xee\x6b", [0x4a3b] = "\xee\x6c", [0x4a3c] = "\xee\x6d",
  [0x4a3d] = "\xee\x6e", [0x4a3e] = "\xee\x6f", [0x4a3f] = "\xee\x70",
  [0x4a40] = "\xee\x71", [0x4a41] = "\xee\x72", [0x4a42] = "\xee\x73",
  [0x4a43] = "\xee\x74", [0x4a44] = "\xee\x75", [0x4a45] = "\xee\x76",
  [0x4a46] = "\xee\x77", [0x4a47] = "\xee\x78", [0x4a48] = "\xee\x79",
  [0x4a49] = "\xee\x7a", [0x4a4a] = "\xee\x7b", [0x4a4b] = "\xee\x7c",
  [0x4a4c] = "\xee\x7d", [0x4a4d] = "\xee\x7e", [0x4a4e] = "\xee\x80",
  [0x4a4f] = "\xee\x81", [0x4a50] = "\xee\x82", [0x4a51] = "\xee\x83",
  [0x4a52] = "\xee\x84", [0x4a53] = "\xee\x85", [0x4a54] = "\xee\x86",
  [0x4a55] = "\xee\x87", [0x4a56] = "\xee\x88", [0x4a57] = "\xee\x89",
  [0x4a58] = "\xee\x8a", [0x4a59] = "\xee\x8b", [0x4a5a] = "\xee\x8c",
  [0x4a5b] = "\xee\x8d", [0x4a5c] = "\xee\x8e", [0x4a5d] = "\xee\x8f",
  [0x4a5e] = "\xee\x90", [0x4a5f] = "\xee\x91", [0x4a60] = "\xee\x92",
  [0x4a61] = "\xee\x93", [0x4a62] = "\xee\x94", [0x4a63] = "\xee\x95",
  [0x4a64] = "\xee\x96", [0x4a65] = "\xee\x97", [0x4a66] = "\xee\x98",
  [0x4a67] = "\xee\x99", [0x4a68] = "\xee\x9a", [0x4a69] = "\xee\x9b",
  [0x4a6a] = "\xee\x9c", [0x4a6b] = "\xee\x9d", [0x4a6c] = "\xee\x9e",
  [0x4a6d] = "\xee\x9f", [0x4a6e] = "\xee\xa0", [0x4a6f] = "\xef\x40",
  [0x4a70] = "\xef\x41", [0x4a71] = "\xef\x42", [0x4a72] = "\xef\x43",
  [0x4a73] = "\xef\x44", [0x4a74] = "\xef\x45", [0x4a75] = "\xd2\xb3",
  [0x4a76] = "\xb6\xa5", [0x4a77] = "\xc7\xea", [0x4a78] = "\xf1\xfc",
  [0x4a79] = "\xcf\xee", [0x4a7a] = "\xcb\xb3", [0x4a7b] = "\xd0\xeb",
  [0x4a7c] = "\xe7\xef", [0x4a7d] = "\xcd\xe7", [0x4a7e] = "\xb9\xcb",
  [0x4a7f] = "\xb6\xd9", [0x4a80] = "\xf1\xfd", [0x4a81] = "\xb0\xe4",
  [0x4a82] = "\xcb\xcc", [0x4a83] = "\xf1\xfe", [0x4a84] = "\xd4\xa4",
  [0x4a85] = "\xc2\xad", [0x4a86] = "\xc1\xec", [0x4a87] = "\xc6\xc4",
  [0x4a88] = "\xbe\xb1", [0x4a89] = "\xf2\xa1", [0x4a8a] = "\xbc\xd5",
  [0x4a8b] = "\xef\x46", [0x4a8c] = "\xf2\xa2", [0x4a8d] = "\xf2\xa3",
  [0x4a8e] = "\xef\x47", [0x4a8f] = "\xf2\xa4", [0x4a90] = "\xd2\xc3",
  [0x4a91] = "\xc6\xb5", [0x4a92] = "\xef\x48", [0x4a93] = "\xcd\xc7",
  [0x4a94] = "\xf2\xa5", [0x4a95] = "\xef\x49", [0x4a96] = "\xd3\xb1",
  [0x4a97] = "\xbf\xc5", [0x4a98] = "\xcc\xe2", [0x4a99] = "\xef\x4a",
  [0x4a9a] = "\xf2\xa6", [0x4a9b] = "\xf2\xa7", [0x4a9c] = "\xd1\xd5",
  [0x4a9d] = "\xb6\xee", [0x4a9e] = "\xf2\xa8", [0x4a9f] = "\xf2\xa9",
  [0x4aa0] = "\xb5\xdf", [0x4aa1] = "\xf2\xaa", [0x4aa2] = "\xf2\xab",
  [0x4aa3] = "\xef\x4b", [0x4aa4] = "\xb2\xfc", [0x4aa5] = "\xf2\xac",
  [0x4aa6] = "\xf2\xad", [0x4aa7] = "\xc8\xa7", [0x4aa8] = "\xef\x4c",
  [0x4aa9] = "\xef\x4d", [0x4aaa] = "\xef\x4e", [0x4aab] = "\xef\x4f",
  [0x4aac] = "\xef\x50", [0x4aad] = "\xef\x51", [0x4aae] = "\xef\x52",
  [0x4aaf] = "\xef\x53", [0x4ab0] = "\xef\x54", [0x4ab1] = "\xef\x55",
  [0x4ab2] = "\xef\x56", [0x4ab3] = "\xef\x57", [0x4ab4] = "\xef\x58",
  [0x4ab5] = "\xef\x59", [0x4ab6] = "\xef\x5a", [0x4ab7] = "\xef\x5b",
  [0x4ab8] = "\xef\x5c", [0x4ab9] = "\xef\x5d", [0x4aba] = "\xef\x5e",
  [0x4abb] = "\xef\x5f", [0x4abc] = "\xef\x60", [0x4abd] = "\xef\x61",
  [0x4abe] = "\xef\x62", [0x4abf] = "\xef\x63", [0x4ac0] = "\xef\x64",
  [0x4ac1] = "\xef\x65", [0x4ac2] = "\xef\x66", [0x4ac3] = "\xef\x67",
  [0x4ac4] = "\xef\x68", [0x4ac5] = "\xef\x69", [0x4ac6] = "\xef\x6a",
  [0x4ac7] = "\xef\x6b", [0x4ac8] = "\xef\x6c", [0x4ac9] = "\xef\x6d",
  [0x4aca] = "\xef\x6e", [0x4acb] = "\xef\x6f", [0x4acc] = "\xef\x70",
  [0x4acd] = "\xef\x71", [0x4ace] = "\xb7\xe7", [0x4acf] = "\xef\x72",
  [0x4ad0] = "\xef\x73", [0x4ad1] = "\xec\xa9", [0x4ad2] = "\xec\xaa",
  [0x4ad3] = "\xec\xab", [0x4ad4] = "\xef\x74", [0x4ad5] = "\xec\xac",
  [0x4ad6] = "\xef\x75", [0x4ad7] = "\xef\x76", [0x4ad8] = "\xc6\xae",
  [0x4ad9] = "\xec\xad", [0x4ada] = "\xec\xae", [0x4adb] = "\xef\x77",
  [0x4adc] = "\xef\x78", [0x4add] = "\xef\x79", [0x4ade] = "\xb7\xc9",
  [0x4adf] = "\xca\xb3", [0x4ae0] = "\xef\x7a", [0x4ae1] = "\xef\x7b",
  [0x4ae2] = "\xef\x7c", [0x4ae3] = "\xef\x7d", [0x4ae4] = "\xef\x7e",
  [0x4ae5] = "\xef\x80", [0x4ae6] = "\xef\x81", [0x4ae7] = "\xe2\xb8",
  [0x4ae8] = "\xf7\xcf", [0x4ae9] = "\xef\x82", [0x4aea] = "\xef\x83",
  [0x4aeb] = "\xef\x84", [0x4aec] = "\xef\x85", [0x4aed] = "\xef\x86",
  [0x4aee] = "\xef\x87", [0x4aef] = "\xef\x88", [0x4af0] = "\xef\x89",
  [0x4af1] = "\xef\x8a", [0x4af2] = "\xef\x8b", [0x4af3] = "\xef\x8c",
  [0x4af4] = "\xef\x8d", [0x4af5] = "\xef\x8e", [0x4af6] = "\xef\x8f",
  [0x4af7] = "\xef\x90", [0x4af8] = "\xef\x91", [0x4af9] = "\xef\x92",
  [0x4afa] = "\xef\x93", [0x4afb] = "\xef\x94", [0x4afc] = "\xef\x95",
  [0x4afd] = "\xef\x96", [0x4afe] = "\xef\x97", [0x4aff] = "\xef\x98",
  [0x4b00] = "\xef\x99", [0x4b01] = "\xef\x9a", [0x4b02] = "\xef\x9b",
  [0x4b03] = "\xef\x9c", [0x4b04] = "\xef\x9d", [0x4b05] = "\xef\x9e",
  [0x4b06] = "\xef\x9f", [0x4b07] = "\xef\xa0", [0x4b08] = "\xf0\x40",
  [0x4b09] = "\xf0\x41", [0x4b0a] = "\xf0\x42", [0x4b0b] = "\xf0\x43",
  [0x4b0c] = "\xf0\x44", [0x4b0d] = "\xf7\xd0", [0x4b0e] = "\xf0\x45",
  [0x4b0f] = "\xf0\x46", [0x4b10] = "\xb2\xcd", [0x4b11] = "\xf0\x47",
  [0x4b12] = "\xf0\x48", [0x4b13] = "\xf0\x49", [0x4b14] = "\xf0\x4a",
  [0x4b15] = "\xf0\x4b", [0x4b16] = "\xf0\x4c", [0x4b17] = "\xf0\x4d",
  [0x4b18] = "\xf0\x4e", [0x4b19] = "\xf0\x4f", [0x4b1a] = "\xf0\x50",
  [0x4b1b] = "\xf0\x51", [0x4b1c] = "\xf0\x52", [0x4b1d] = "\xf0\x53",
  [0x4b1e] = "\xf0\x54", [0x4b1f] = "\xf0\x55", [0x4b20] = "\xf0\x56",
  [0x4b21] = "\xf0\x57", [0x4b22] = "\xf0\x58", [0x4b23] = "\xf0\x59",
  [0x4b24] = "\xf0\x5a", [0x4b25] = "\xf0\x5b", [0x4b26] = "\xf0\x5c",
  [0x4b27] = "\xf0\x5d", [0x4b28] = "\xf0\x5e", [0x4b29] = "\xf0\x5f",
  [0x4b2a] = "\xf0\x60", [0x4b2b] = "\xf0\x61", [0x4b2c] = "\xf0\x62",
  [0x4b2d] = "\xf0\x63", [0x4b2e] = "\xf7\xd1", [0x4b2f] = "\xf0\x64",
  [0x4b30] = "\xf0\x65", [0x4b31] = "\xf0\x66", [0x4b32] = "\xf0\x67",
  [0x4b33] = "\xf0\x68", [0x4b34] = "\xf0\x69", [0x4b35] = "\xf0\x6a",
  [0x4b36] = "\xf0\x6b", [0x4b37] = "\xf0\x6c", [0x4b38] = "\xf0\x6d",
  [0x4b39] = "\xf0\x6e", [0x4b3a] = "\xf0\x6f", [0x4b3b] = "\xf0\x70",
  [0x4b3c] = "\xf0\x71", [0x4b3d] = "\xf0\x72", [0x4b3e] = "\xf0\x73",
  [0x4b3f] = "\xf0\x74", [0x4b40] = "\xf0\x75", [0x4b41] = "\xf0\x76",
  [0x4b42] = "\xf0\x77", [0x4b43] = "\xf0\x78", [0x4b44] = "\xf0\x79",
  [0x4b45] = "\xf0\x7a", [0x4b46] = "\xf0\x7b", [0x4b47] = "\xf0\x7c",
  [0x4b48] = "\xf0\x7d", [0x4b49] = "\xf0\x7e", [0x4b4a] = "\xf0\x80",
  [0x4b4b] = "\xf0\x81", [0x4b4c] = "\xf0\x82", [0x4b4d] = "\xf0\x83",
  [0x4b4e] = "\xf0\x84", [0x4b4f] = "\xf0\x85", [0x4b50] = "\xf0\x86",
  [0x4b51] = "\xf0\x87", [0x4b52] = "\xf0\x88", [0x4b53] = "\xf0\x89",
  [0x4b54] = "\xf7\xd3", [0x4b55] = "\xf7\xd2", [0x4b56] = "\xf0\x8a",
  [0x4b57] = "\xf0\x8b", [0x4b58] = "\xf0\x8c", [0x4b59] = "\xf0\x8d",
  [0x4b5a] = "\xf0\x8e", [0x4b5b] = "\xf0\x8f", [0x4b5c] = "\xf0\x90",
  [0x4b5d] = "\xf0\x91", [0x4b5e] = "\xf0\x92", [0x4b5f] = "\xf0\x93",
  [0x4b60] = "\xf0\x94", [0x4b61] = "\xf0\x95", [0x4b62] = "\xf0\x96",
  [0x4b63] = "\xe2\xbb", [0x4b64] = "\xf0\x97", [0x4b65] = "\xbc\xa2",
  [0x4b66] = "\xf0\x98", [0x4b67] = "\xe2\xbc", [0x4b68] = "\xe2\xbd",
  [0x4b69] = "\xe2\xbe", [0x4b6a] = "\xe2\xbf", [0x4b6b] = "\xe2\xc0",
  [0x4b6c] = "\xe2\xc1", [0x4b6d] = "\xb7\xb9", [0x4b6e] = "\xd2\xfb",
  [0x4b6f] = "\xbd\xa4", [0x4b70] = "\xca\xce", [0x4b71] = "\xb1\xa5",
  [0x4b72] = "\xcb\xc7", [0x4b73] = "\xf0\x99", [0x4b74] = "\xe2\xc2",
  [0x4b75] = "\xb6\xfc", [0x4b76] = "\xc8\xc4", [0x4b77] = "\xe2\xc3",
  [0x4b78] = "\xf0\x9a", [0x4b79] = "\xf0\x9b", [0x4b7a] = "\xbd\xc8",
  [0x4b7b] = "\xf0\x9c", [0x4b7c] = "\xb1\xfd", [0x4b7d] = "\xe2\xc4",
  [0x4b7e] = "\xf0\x9d", [0x4b7f] = "\xb6\xf6", [0x4b80] = "\xe2\xc5",
  [0x4b81] = "\xc4\xd9", [0x4b82] = "\xf0\x9e", [0x4b83] = "\xf0\x9f",
  [0x4b84] = "\xe2\xc6", [0x4b85] = "\xcf\xda", [0x4b86] = "\xb9\xdd",
  [0x4b87] = "\xe2\xc7", [0x4b88] = "\xc0\xa1", [0x4b89] = "\xf0\xa0",
  [0x4b8a] = "\xe2\xc8", [0x4b8b] = "\xb2\xf6", [0x4b8c] = "\xf1\x40",
  [0x4b8d] = "\xe2\xc9", [0x4b8e] = "\xf1\x41", [0x4b8f] = "\xc1\xf3",
  [0x4b90] = "\xe2\xca", [0x4b91] = "\xe2\xcb", [0x4b92] = "\xc2\xf8",
  [0x4b93] = "\xe2\xcc", [0x4b94] = "\xe2\xcd", [0x4b95] = "\xe2\xce",
  [0x4b96] = "\xca\xd7", [0x4b97] = "\xd8\xb8", [0x4b98] = "\xd9\xe5",
  [0x4b99] = "\xcf\xe3", [0x4b9a] = "\xf1\x42", [0x4b9b] = "\xf1\x43",
  [0x4b9c] = "\xf1\x44", [0x4b9d] = "\xf1\x45", [0x4b9e] = "\xf1\x46",
  [0x4b9f] = "\xf1\x47", [0x4ba0] = "\xf1\x48", [0x4ba1] = "\xf1\x49",
  [0x4ba2] = "\xf1\x4a", [0x4ba3] = "\xf1\x4b", [0x4ba4] = "\xf1\x4c",
  [0x4ba5] = "\xf0\xa5", [0x4ba6] = "\xf1\x4d", [0x4ba7] = "\xf1\x4e",
  [0x4ba8] = "\xdc\xb0", [0x4ba9] = "\xf1\x4f", [0x4baa] = "\xf1\x50",
  [0x4bab] = "\xf1\x51", [0x4bac] = "\xf1\x52", [0x4bad] = "\xf1\x53",
  [0x4bae] = "\xf1\x54", [0x4baf] = "\xf1\x55", [0x4bb0] = "\xf1\x56",
  [0x4bb1] = "\xf1\x57", [0x4bb2] = "\xf1\x58", [0x4bb3] = "\xf1\x59",
  [0x4bb4] = "\xf1\x5a", [0x4bb5] = "\xf1\x5b", [0x4bb6] = "\xf1\x5c",
  [0x4bb7] = "\xf1\x5d", [0x4bb8] = "\xf1\x5e", [0x4bb9] = "\xf1\x5f",
  [0x4bba] = "\xf1\x60", [0x4bbb] = "\xf1\x61", [0x4bbc] = "\xf1\x62",
  [0x4bbd] = "\xf1\x63", [0x4bbe] = "\xf1\x64", [0x4bbf] = "\xf1\x65",
  [0x4bc0] = "\xf1\x66", [0x4bc1] = "\xf1\x67", [0x4bc2] = "\xf1\x68",
  [0x4bc3] = "\xf1\x69", [0x4bc4] = "\xf1\x6a", [0x4bc5] = "\xf1\x6b",
  [0x4bc6] = "\xf1\x6c", [0x4bc7] = "\xf1\x6d", [0x4bc8] = "\xf1\x6e",
  [0x4bc9] = "\xf1\x6f", [0x4bca] = "\xf1\x70", [0x4bcb] = "\xf1\x71",
  [0x4bcc] = "\xf1\x72", [0x4bcd] = "\xf1\x73", [0x4bce] = "\xf1\x74",
  [0x4bcf] = "\xf1\x75", [0x4bd0] = "\xf1\x76", [0x4bd1] = "\xf1\x77",
  [0x4bd2] = "\xf1\x78", [0x4bd3] = "\xf1\x79", [0x4bd4] = "\xf1\x7a",
  [0x4bd5] = "\xf1\x7b", [0x4bd6] = "\xf1\x7c", [0x4bd7] = "\xf1\x7d",
  [0x4bd8] = "\xf1\x7e", [0x4bd9] = "\xf1\x80", [0x4bda] = "\xf1\x81",
  [0x4bdb] = "\xf1\x82", [0x4bdc] = "\xf1\x83", [0x4bdd] = "\xf1\x84",
  [0x4bde] = "\xf1\x85", [0x4bdf] = "\xf1\x86", [0x4be0] = "\xf1\x87",
  [0x4be1] = "\xf1\x88", [0x4be2] = "\xf1\x89", [0x4be3] = "\xf1\x8a",
  [0x4be4] = "\xf1\x8b", [0x4be5] = "\xf1\x8c", [0x4be6] = "\xf1\x8d",
  [0x4be7] = "\xf1\x8e", [0x4be8] = "\xf1\x8f", [0x4be9] = "\xf1\x90",
  [0x4bea] = "\xf1\x91", [0x4beb] = "\xf1\x92", [0x4bec] = "\xf1\x93",
  [0x4bed] = "\xf1\x94", [0x4bee] = "\xf1\x95", [0x4bef] = "\xf1\x96",
  [0x4bf0] = "\xf1\x97", [0x4bf1] = "\xf1\x98", [0x4bf2] = "\xf1\x99",
  [0x4bf3] = "\xf1\x9a", [0x4bf4] = "\xf1\x9b", [0x4bf5] = "\xf1\x9c",
  [0x4bf6] = "\xf1\x9d", [0x4bf7] = "\xf1\x9e", [0x4bf8] = "\xf1\x9f",
  [0x4bf9] = "\xf1\xa0", [0x4bfa] = "\xf2\x40", [0x4bfb] = "\xf2\x41",
  [0x4bfc] = "\xf2\x42", [0x4bfd] = "\xf2\x43", [0x4bfe] = "\xf2\x44",
  [0x4bff] = "\xf2\x45", [0x4c00] = "\xf2\x46", [0x4c01] = "\xf2\x47",
  [0x4c02] = "\xf2\x48", [0x4c03] = "\xf2\x49", [0x4c04] = "\xf2\x4a",
  [0x4c05] = "\xf2\x4b", [0x4c06] = "\xf2\x4c", [0x4c07] = "\xf2\x4d",
  [0x4c08] = "\xf2\x4e", [0x4c09] = "\xf2\x4f", [0x4c0a] = "\xf2\x50",
  [0x4c0b] = "\xf2\x51", [0x4c0c] = "\xf2\x52", [0x4c0d] = "\xf2\x53",
  [0x4c0e] = "\xf2\x54", [0x4c0f] = "\xf2\x55", [0x4c10] = "\xf2\x56",
  [0x4c11] = "\xf2\x57", [0x4c12] = "\xf2\x58", [0x4c13] = "\xf2\x59",
  [0x4c14] = "\xf2\x5a", [0x4c15] = "\xf2\x5b", [0x4c16] = "\xf2\x5c",
  [0x4c17] = "\xf2\x5d", [0x4c18] = "\xf2\x5e", [0x4c19] = "\xf2\x5f",
  [0x4c1a] = "\xf2\x60", [0x4c1b] = "\xf2\x61", [0x4c1c] = "\xf2\x62",
  [0x4c1d] = "\xf2\x63", [0x4c1e] = "\xf2\x64", [0x4c1f] = "\xf2\x65",
  [0x4c20] = "\xf2\x66", [0x4c21] = "\xf2\x67", [0x4c22] = "\xf2\x68",
  [0x4c23] = "\xf2\x69", [0x4c24] = "\xf2\x6a", [0x4c25] = "\xf2\x6b",
  [0x4c26] = "\xf2\x6c", [0x4c27] = "\xf2\x6d", [0x4c28] = "\xf2\x6e",
  [0x4c29] = "\xf2\x6f", [0x4c2a] = "\xf2\x70", [0x4c2b] = "\xf2\x71",
  [0x4c2c] = "\xf2\x72", [0x4c2d] = "\xf2\x73", [0x4c2e] = "\xf2\x74",
  [0x4c2f] = "\xf2\x75", [0x4c30] = "\xf2\x76", [0x4c31] = "\xf2\x77",
  [0x4c32] = "\xf2\x78", [0x4c33] = "\xf2\x79", [0x4c34] = "\xf2\x7a",
  [0x4c35] = "\xf2\x7b", [0x4c36] = "\xf2\x7c", [0x4c37] = "\xf2\x7d",
  [0x4c38] = "\xf2\x7e", [0x4c39] = "\xf2\x80", [0x4c3a] = "\xf2\x81",
  [0x4c3b] = "\xf2\x82", [0x4c3c] = "\xf2\x83", [0x4c3d] = "\xf2\x84",
  [0x4c3e] = "\xf2\x85", [0x4c3f] = "\xf2\x86", [0x4c40] = "\xf2\x87",
  [0x4c41] = "\xf2\x88", [0x4c42] = "\xf2\x89", [0x4c43] = "\xf2\x8a",
  [0x4c44] = "\xf2\x8b", [0x4c45] = "\xf2\x8c", [0x4c46] = "\xf2\x8d",
  [0x4c47] = "\xf2\x8e", [0x4c48] = "\xf2\x8f", [0x4c49] = "\xf2\x90",
  [0x4c4a] = "\xf2\x91", [0x4c4b] = "\xf2\x92", [0x4c4c] = "\xf2\x93",
  [0x4c4d] = "\xf2\x94", [0x4c4e] = "\xf2\x95", [0x4c4f] = "\xf2\x96",
  [0x4c50] = "\xf2\x97", [0x4c51] = "\xf2\x98", [0x4c52] = "\xf2\x99",
  [0x4c53] = "\xf2\x9a", [0x4c54] = "\xf2\x9b", [0x4c55] = "\xf2\x9c",
  [0x4c56] = "\xf2\x9d", [0x4c57] = "\xf2\x9e", [0x4c58] = "\xf2\x9f",
  [0x4c59] = "\xf2\xa0", [0x4c5a] = "\xf3\x40", [0x4c5b] = "\xf3\x41",
  [0x4c5c] = "\xf3\x42", [0x4c5d] = "\xf3\x43", [0x4c5e] = "\xf3\x44",
  [0x4c5f] = "\xf3\x45", [0x4c60] = "\xf3\x46", [0x4c61] = "\xf3\x47",
  [0x4c62] = "\xf3\x48", [0x4c63] = "\xf3\x49", [0x4c64] = "\xf3\x4a",
  [0x4c65] = "\xf3\x4b", [0x4c66] = "\xf3\x4c", [0x4c67] = "\xf3\x4d",
  [0x4c68] = "\xf3\x4e", [0x4c69] = "\xf3\x4f", [0x4c6a] = "\xf3\x50",
  [0x4c6b] = "\xf3\x51", [0x4c6c] = "\xc2\xed", [0x4c6d] = "\xd4\xa6",
  [0x4c6e] = "\xcd\xd4", [0x4c6f] = "\xd1\xb1", [0x4c70] = "\xb3\xdb",
  [0x4c71] = "\xc7\xfd", [0x4c72] = "\xf3\x52", [0x4c73] = "\xb2\xb5",
  [0x4c74] = "\xc2\xbf", [0x4c75] = "\xe6\xe0", [0x4c76] = "\xca\xbb",
  [0x4c77] = "\xe6\xe1", [0x4c78] = "\xe6\xe2", [0x4c79] = "\xbe\xd4",
  [0x4c7a] = "\xe6\xe3", [0x4c7b] = "\xd7\xa4", [0x4c7c] = "\xcd\xd5",
  [0x4c7d] = "\xe6\xe5", [0x4c7e] = "\xbc\xdd", [0x4c7f] = "\xe6\xe4",
  [0x4c80] = "\xe6\xe6", [0x4c81] = "\xe6\xe7", [0x4c82] = "\xc2\xee",
  [0x4c83] = "\xf3\x53", [0x4c84] = "\xbd\xbe", [0x4c85] = "\xe6\xe8",
  [0x4c86] = "\xc2\xe6", [0x4c87] = "\xba\xa7", [0x4c88] = "\xe6\xe9",
  [0x4c89] = "\xf3\x54", [0x4c8a] = "\xe6\xea", [0x4c8b] = "\xb3\xd2",
  [0x4c8c] = "\xd1\xe9", [0x4c8d] = "\xf3\x55", [0x4c8e] = "\xf3\x56",
  [0x4c8f] = "\xbf\xa5", [0x4c90] = "\xe6\xeb", [0x4c91] = "\xc6\xef",
  [0x4c92] = "\xe6\xec", [0x4c93] = "\xe6\xed", [0x4c94] = "\xf3\x57",
  [0x4c95] = "\xf3\x58", [0x4c96] = "\xe6\xee", [0x4c97] = "\xc6\xad",
  [0x4c98] = "\xe6\xef", [0x4c99] = "\xf3\x59", [0x4c9a] = "\xc9\xa7",
  [0x4c9b] = "\xe6\xf0", [0x4c9c] = "\xe6\xf1", [0x4c9d] = "\xe6\xf2",
  [0x4c9e] = "\xe5\xb9", [0x4c9f] = "\xe6\xf3", [0x4ca0] = "\xe6\xf4",
  [0x4ca1] = "\xc2\xe2", [0x4ca2] = "\xe6\xf5", [0x4ca3] = "\xe6\xf6",
  [0x4ca4] = "\xd6\xe8", [0x4ca5] = "\xe6\xf7", [0x4ca6] = "\xf3\x5a",
  [0x4ca7] = "\xe6\xf8", [0x4ca8] = "\xb9\xc7", [0x4ca9] = "\xf3\x5b",
  [0x4caa] = "\xf3\x5c", [0x4cab] = "\xf3\x5d", [0x4cac] = "\xf3\x5e",
  [0x4cad] = "\xf3\x5f", [0x4cae] = "\xf3\x60", [0x4caf] = "\xf3\x61",
  [0x4cb0] = "\xf7\xbb", [0x4cb1] = "\xf7\xba", [0x4cb2] = "\xf3\x62",
  [0x4cb3] = "\xf3\x63", [0x4cb4] = "\xf3\x64", [0x4cb5] = "\xf3\x65",
  [0x4cb6] = "\xf7\xbe", [0x4cb7] = "\xf7\xbc", [0x4cb8] = "\xba\xa1",
  [0x4cb9] = "\xf3\x66", [0x4cba] = "\xf7\xbf", [0x4cbb] = "\xf3\x67",
  [0x4cbc] = "\xf7\xc0", [0x4cbd] = "\xf3\x68", [0x4cbe] = "\xf3\x69",
  [0x4cbf] = "\xf3\x6a", [0x4cc0] = "\xf7\xc2", [0x4cc1] = "\xf7\xc1",
  [0x4cc2] = "\xf7\xc4", [0x4cc3] = "\xf3\x6b", [0x4cc4] = "\xf3\x6c",
  [0x4cc5] = "\xf7\xc3", [0x4cc6] = "\xf3\x6d", [0x4cc7] = "\xf3\x6e",
  [0x4cc8] = "\xf3\x6f", [0x4cc9] = "\xf3\x70", [0x4cca] = "\xf3\x71",
  [0x4ccb] = "\xf7\xc5", [0x4ccc] = "\xf7\xc6", [0x4ccd] = "\xf3\x72",
  [0x4cce] = "\xf3\x73", [0x4ccf] = "\xf3\x74", [0x4cd0] = "\xf3\x75",
  [0x4cd1] = "\xf7\xc7", [0x4cd2] = "\xf3\x76", [0x4cd3] = "\xcb\xe8",
  [0x4cd4] = "\xf3\x77", [0x4cd5] = "\xf3\x78", [0x4cd6] = "\xf3\x79",
  [0x4cd7] = "\xf3\x7a", [0x4cd8] = "\xb8\xdf", [0x4cd9] = "\xf3\x7b",
  [0x4cda] = "\xf3\x7c", [0x4cdb] = "\xf3\x7d", [0x4cdc] = "\xf3\x7e",
  [0x4cdd] = "\xf3\x80", [0x4cde] = "\xf3\x81", [0x4cdf] = "\xf7\xd4",
  [0x4ce0] = "\xf3\x82", [0x4ce1] = "\xf7\xd5", [0x4ce2] = "\xf3\x83",
  [0x4ce3] = "\xf3\x84", [0x4ce4] = "\xf3\x85", [0x4ce5] = "\xf3\x86",
  [0x4ce6] = "\xf7\xd6", [0x4ce7] = "\xf3\x87", [0x4ce8] = "\xf3\x88",
  [0x4ce9] = "\xf3\x89", [0x4cea] = "\xf3\x8a", [0x4ceb] = "\xf7\xd8",
  [0x4cec] = "\xf3\x8b", [0x4ced] = "\xf7\xda", [0x4cee] = "\xf3\x8c",
  [0x4cef] = "\xf7\xd7", [0x4cf0] = "\xf3\x8d", [0x4cf1] = "\xf3\x8e",
  [0x4cf2] = "\xf3\x8f", [0x4cf3] = "\xf3\x90", [0x4cf4] = "\xf3\x91",
  [0x4cf5] = "\xf3\x92", [0x4cf6] = "\xf3\x93", [0x4cf7] = "\xf3\x94",
  [0x4cf8] = "\xf3\x95", [0x4cf9] = "\xf7\xdb", [0x4cfa] = "\xf3\x96",
  [0x4cfb] = "\xf7\xd9", [0x4cfc] = "\xf3\x97", [0x4cfd] = "\xf3\x98",
  [0x4cfe] = "\xf3\x99", [0x4cff] = "\xf3\x9a", [0x4d00] = "\xf3\x9b",
  [0x4d01] = "\xf3\x9c", [0x4d02] = "\xf3\x9d", [0x4d03] = "\xd7\xd7",
  [0x4d04] = "\xf3\x9e", [0x4d05] = "\xf3\x9f", [0x4d06] = "\xf3\xa0",
  [0x4d07] = "\xf4\x40", [0x4d08] = "\xf7\xdc", [0x4d09] = "\xf4\x41",
  [0x4d0a] = "\xf4\x42", [0x4d0b] = "\xf4\x43", [0x4d0c] = "\xf4\x44",
  [0x4d0d] = "\xf4\x45", [0x4d0e] = "\xf4\x46", [0x4d0f] = "\xf7\xdd",
  [0x4d10] = "\xf4\x47", [0x4d11] = "\xf4\x48", [0x4d12] = "\xf4\x49",
  [0x4d13] = "\xf7\xde", [0x4d14] = "\xf4\x4a", [0x4d15] = "\xf4\x4b",
  [0x4d16] = "\xf4\x4c", [0x4d17] = "\xf4\x4d", [0x4d18] = "\xf4\x4e",
  [0x4d19] = "\xf4\x4f", [0x4d1a] = "\xf4\x50", [0x4d1b] = "\xf4\x51",
  [0x4d1c] = "\xf4\x52", [0x4d1d] = "\xf4\x53", [0x4d1e] = "\xf4\x54",
  [0x4d1f] = "\xf7\xdf", [0x4d20] = "\xf4\x55", [0x4d21] = "\xf4\x56",
  [0x4d22] = "\xf4\x57", [0x4d23] = "\xf7\xe0", [0x4d24] = "\xf4\x58",
  [0x4d25] = "\xf4\x59", [0x4d26] = "\xf4\x5a", [0x4d27] = "\xf4\x5b",
  [0x4d28] = "\xf4\x5c", [0x4d29] = "\xf4\x5d", [0x4d2a] = "\xf4\x5e",
  [0x4d2b] = "\xf4\x5f", [0x4d2c] = "\xf4\x60", [0x4d2d] = "\xf4\x61",
  [0x4d2e] = "\xf4\x62", [0x4d2f] = "\xdb\xcb", [0x4d30] = "\xf4\x63",
  [0x4d31] = "\xf4\x64", [0x4d32] = "\xd8\xaa", [0x4d33] = "\xf4\x65",
  [0x4d34] = "\xf4\x66", [0x4d35] = "\xf4\x67", [0x4d36] = "\xf4\x68",
  [0x4d37] = "\xf4\x69", [0x4d38] = "\xf4\x6a", [0x4d39] = "\xf4\x6b",
  [0x4d3a] = "\xf4\x6c", [0x4d3b] = "\xe5\xf7", [0x4d3c] = "\xb9\xed",
  [0x4d3d] = "\xf4\x6d", [0x4d3e] = "\xf4\x6e", [0x4d3f] = "\xf4\x6f",
  [0x4d40] = "\xf4\x70", [0x4d41] = "\xbf\xfd", [0x4d42] = "\xbb\xea",
  [0x4d43] = "\xf7\xc9", [0x4d44] = "\xc6\xc7", [0x4d45] = "\xf7\xc8",
  [0x4d46] = "\xf4\x71", [0x4d47] = "\xf7\xca", [0x4d48] = "\xf7\xcc",
  [0x4d49] = "\xf7\xcb", [0x4d4a] = "\xf4\x72", [0x4d4b] = "\xf4\x73",
  [0x4d4c] = "\xf4\x74", [0x4d4d] = "\xf7\xcd", [0x4d4e] = "\xf4\x75",
  [0x4d4f] = "\xce\xba", [0x4d50] = "\xf4\x76", [0x4d51] = "\xf7\xce",
  [0x4d52] = "\xf4\x77", [0x4d53] = "\xf4\x78", [0x4d54] = "\xc4\xa7",
  [0x4d55] = "\xf4\x79", [0x4d56] = "\xf4\x7a", [0x4d57] = "\xf4\x7b",
  [0x4d58] = "\xf4\x7c", [0x4d59] = "\xf4\x7d", [0x4d5a] = "\xf4\x7e",
  [0x4d5b] = "\xf4\x80", [0x4d5c] = "\xf4\x81", [0x4d5d] = "\xf4\x82",
  [0x4d5e] = "\xf4\x83", [0x4d5f] = "\xf4\x84", [0x4d60] = "\xf4\x85",
  [0x4d61] = "\xf4\x86", [0x4d62] = "\xf4\x87", [0x4d63] = "\xf4\x88",
  [0x4d64] = "\xf4\x89", [0x4d65] = "\xf4\x8a", [0x4d66] = "\xf4\x8b",
  [0x4d67] = "\xf4\x8c", [0x4d68] = "\xf4\x8d", [0x4d69] = "\xf4\x8e",
  [0x4d6a] = "\xf4\x8f", [0x4d6b] = "\xf4\x90", [0x4d6c] = "\xf4\x91",
  [0x4d6d] = "\xf4\x92", [0x4d6e] = "\xf4\x93", [0x4d6f] = "\xf4\x94",
  [0x4d70] = "\xf4\x95", [0x4d71] = "\xf4\x96", [0x4d72] = "\xf4\x97",
  [0x4d73] = "\xf4\x98", [0x4d74] = "\xf4\x99", [0x4d75] = "\xf4\x9a",
  [0x4d76] = "\xf4\x9b", [0x4d77] = "\xf4\x9c", [0x4d78] = "\xf4\x9d",
  [0x4d79] = "\xf4\x9e", [0x4d7a] = "\xf4\x9f", [0x4d7b] = "\xf4\xa0",
  [0x4d7c] = "\xf5\x40", [0x4d7d] = "\xf5\x41", [0x4d7e] = "\xf5\x42",
  [0x4d7f] = "\xf5\x43", [0x4d80] = "\xf5\x44", [0x4d81] = "\xf5\x45",
  [0x4d82] = "\xf5\x46", [0x4d83] = "\xf5\x47", [0x4d84] = "\xf5\x48",
  [0x4d85] = "\xf5\x49", [0x4d86] = "\xf5\x4a", [0x4d87] = "\xf5\x4b",
  [0x4d88] = "\xf5\x4c", [0x4d89] = "\xf5\x4d", [0x4d8a] = "\xf5\x4e",
  [0x4d8b] = "\xf5\x4f", [0x4d8c] = "\xf5\x50", [0x4d8d] = "\xf5\x51",
  [0x4d8e] = "\xf5\x52", [0x4d8f] = "\xf5\x53", [0x4d90] = "\xf5\x54",
  [0x4d91] = "\xf5\x55", [0x4d92] = "\xf5\x56", [0x4d93] = "\xf5\x57",
  [0x4d94] = "\xf5\x58", [0x4d95] = "\xf5\x59", [0x4d96] = "\xf5\x5a",
  [0x4d97] = "\xf5\x5b", [0x4d98] = "\xf5\x5c", [0x4d99] = "\xf5\x5d",
  [0x4d9a] = "\xf5\x5e", [0x4d9b] = "\xf5\x5f", [0x4d9c] = "\xf5\x60",
  [0x4d9d] = "\xf5\x61", [0x4d9e] = "\xf5\x62", [0x4d9f] = "\xf5\x63",
  [0x4da0] = "\xf5\x64", [0x4da1] = "\xf5\x65", [0x4da2] = "\xf5\x66",
  [0x4da3] = "\xf5\x67", [0x4da4] = "\xf5\x68", [0x4da5] = "\xf5\x69",
  [0x4da6] = "\xf5\x6a", [0x4da7] = "\xf5\x6b", [0x4da8] = "\xf5\x6c",
  [0x4da9] = "\xf5\x6d", [0x4daa] = "\xf5\x6e", [0x4dab] = "\xf5\x6f",
  [0x4dac] = "\xf5\x70", [0x4dad] = "\xf5\x71", [0x4dae] = "\xf5\x72",
  [0x4daf] = "\xf5\x73", [0x4db0] = "\xf5\x74", [0x4db1] = "\xf5\x75",
  [0x4db2] = "\xf5\x76", [0x4db3] = "\xf5\x77", [0x4db4] = "\xf5\x78",
  [0x4db5] = "\xf5\x79", [0x4db6] = "\xf5\x7a", [0x4db7] = "\xf5\x7b",
  [0x4db8] = "\xf5\x7c", [0x4db9] = "\xf5\x7d", [0x4dba] = "\xf5\x7e",
  [0x4dbb] = "\xf5\x80", [0x4dbc] = "\xf5\x81", [0x4dbd] = "\xf5\x82",
  [0x4dbe] = "\xf5\x83", [0x4dbf] = "\xf5\x84", [0x4dc0] = "\xf5\x85",
  [0x4dc1] = "\xf5\x86", [0x4dc2] = "\xf5\x87", [0x4dc3] = "\xf5\x88",
  [0x4dc4] = "\xf5\x89", [0x4dc5] = "\xf5\x8a", [0x4dc6] = "\xf5\x8b",
  [0x4dc7] = "\xf5\x8c", [0x4dc8] = "\xf5\x8d", [0x4dc9] = "\xf5\x8e",
  [0x4dca] = "\xf5\x8f", [0x4dcb] = "\xf5\x90", [0x4dcc] = "\xf5\x91",
  [0x4dcd] = "\xf5\x92", [0x4dce] = "\xf5\x93", [0x4dcf] = "\xf5\x94",
  [0x4dd0] = "\xf5\x95", [0x4dd1] = "\xf5\x96", [0x4dd2] = "\xf5\x97",
  [0x4dd3] = "\xf5\x98", [0x4dd4] = "\xf5\x99", [0x4dd5] = "\xf5\x9a",
  [0x4dd6] = "\xf5\x9b", [0x4dd7] = "\xf5\x9c", [0x4dd8] = "\xf5\x9d",
  [0x4dd9] = "\xf5\x9e", [0x4dda] = "\xf5\x9f", [0x4ddb] = "\xf5\xa0",
  [0x4ddc] = "\xf6\x40", [0x4ddd] = "\xf6\x41", [0x4dde] = "\xf6\x42",
  [0x4ddf] = "\xf6\x43", [0x4de0] = "\xf6\x44", [0x4de1] = "\xf6\x45",
  [0x4de2] = "\xf6\x46", [0x4de3] = "\xf6\x47", [0x4de4] = "\xf6\x48",
  [0x4de5] = "\xf6\x49", [0x4de6] = "\xf6\x4a", [0x4de7] = "\xf6\x4b",
  [0x4de8] = "\xf6\x4c", [0x4de9] = "\xf6\x4d", [0x4dea] = "\xf6\x4e",
  [0x4deb] = "\xf6\x4f", [0x4dec] = "\xf6\x50", [0x4ded] = "\xf6\x51",
  [0x4dee] = "\xf6\x52", [0x4def] = "\xf6\x53", [0x4df0] = "\xf6\x54",
  [0x4df1] = "\xf6\x55", [0x4df2] = "\xf6\x56", [0x4df3] = "\xf6\x57",
  [0x4df4] = "\xf6\x58", [0x4df5] = "\xf6\x59", [0x4df6] = "\xf6\x5a",
  [0x4df7] = "\xf6\x5b", [0x4df8] = "\xf6\x5c", [0x4df9] = "\xf6\x5d",
  [0x4dfa] = "\xf6\x5e", [0x4dfb] = "\xf6\x5f", [0x4dfc] = "\xf6\x60",
  [0x4dfd] = "\xf6\x61", [0x4dfe] = "\xf6\x62", [0x4dff] = "\xf6\x63",
  [0x4e00] = "\xf6\x64", [0x4e01] = "\xf6\x65", [0x4e02] = "\xf6\x66",
  [0x4e03] = "\xf6\x67", [0x4e04] = "\xf6\x68", [0x4e05] = "\xf6\x69",
  [0x4e06] = "\xf6\x6a", [0x4e07] = "\xf6\x6b", [0x4e08] = "\xf6\x6c",
  [0x4e09] = "\xf6\x6d", [0x4e0a] = "\xf6\x6e", [0x4e0b] = "\xf6\x6f",
  [0x4e0c] = "\xf6\x70", [0x4e0d] = "\xf6\x71", [0x4e0e] = "\xf6\x72",
  [0x4e0f] = "\xf6\x73", [0x4e10] = "\xf6\x74", [0x4e11] = "\xf6\x75",
  [0x4e12] = "\xf6\x76", [0x4e13] = "\xf6\x77", [0x4e14] = "\xf6\x78",
  [0x4e15] = "\xf6\x79", [0x4e16] = "\xf6\x7a", [0x4e17] = "\xf6\x7b",
  [0x4e18] = "\xf6\x7c", [0x4e19] = "\xf6\x7d", [0x4e1a] = "\xf6\x7e",
  [0x4e1b] = "\xf6\x80", [0x4e1c] = "\xf6\x81", [0x4e1d] = "\xf6\x82",
  [0x4e1e] = "\xf6\x83", [0x4e1f] = "\xf6\x84", [0x4e20] = "\xf6\x85",
  [0x4e21] = "\xf6\x86", [0x4e22] = "\xf6\x87", [0x4e23] = "\xf6\x88",
  [0x4e24] = "\xf6\x89", [0x4e25] = "\xf6\x8a", [0x4e26] = "\xf6\x8b",
  [0x4e27] = "\xf6\x8c", [0x4e28] = "\xf6\x8d", [0x4e29] = "\xf6\x8e",
  [0x4e2a] = "\xf6\x8f", [0x4e2b] = "\xf6\x90", [0x4e2c] = "\xf6\x91",
  [0x4e2d] = "\xf6\x92", [0x4e2e] = "\xf6\x93", [0x4e2f] = "\xf6\x94",
  [0x4e30] = "\xf6\x95", [0x4e31] = "\xf6\x96", [0x4e32] = "\xf6\x97",
  [0x4e33] = "\xf6\x98", [0x4e34] = "\xf6\x99", [0x4e35] = "\xf6\x9a",
  [0x4e36] = "\xf6\x9b", [0x4e37] = "\xf6\x9c", [0x4e38] = "\xf6\x9d",
  [0x4e39] = "\xf6\x9e", [0x4e3a] = "\xf6\x9f", [0x4e3b] = "\xf6\xa0",
  [0x4e3c] = "\xf7\x40", [0x4e3d] = "\xf7\x41", [0x4e3e] = "\xf7\x42",
  [0x4e3f] = "\xf7\x43", [0x4e40] = "\xf7\x44", [0x4e41] = "\xf7\x45",
  [0x4e42] = "\xf7\x46", [0x4e43] = "\xf7\x47", [0x4e44] = "\xf7\x48",
  [0x4e45] = "\xf7\x49", [0x4e46] = "\xf7\x4a", [0x4e47] = "\xf7\x4b",
  [0x4e48] = "\xf7\x4c", [0x4e49] = "\xf7\x4d", [0x4e4a] = "\xf7\x4e",
  [0x4e4b] = "\xf7\x4f", [0x4e4c] = "\xf7\x50", [0x4e4d] = "\xf7\x51",
  [0x4e4e] = "\xf7\x52", [0x4e4f] = "\xf7\x53", [0x4e50] = "\xf7\x54",
  [0x4e51] = "\xf7\x55", [0x4e52] = "\xf7\x56", [0x4e53] = "\xf7\x57",
  [0x4e54] = "\xf7\x58", [0x4e55] = "\xf7\x59", [0x4e56] = "\xf7\x5a",
  [0x4e57] = "\xf7\x5b", [0x4e58] = "\xf7\x5c", [0x4e59] = "\xf7\x5d",
  [0x4e5a] = "\xf7\x5e", [0x4e5b] = "\xf7\x5f", [0x4e5c] = "\xf7\x60",
  [0x4e5d] = "\xf7\x61", [0x4e5e] = "\xf7\x62", [0x4e5f] = "\xf7\x63",
  [0x4e60] = "\xf7\x64", [0x4e61] = "\xf7\x65", [0x4e62] = "\xf7\x66",
  [0x4e63] = "\xf7\x67", [0x4e64] = "\xf7\x68", [0x4e65] = "\xf7\x69",
  [0x4e66] = "\xf7\x6a", [0x4e67] = "\xf7\x6b", [0x4e68] = "\xf7\x6c",
  [0x4e69] = "\xf7\x6d", [0x4e6a] = "\xf7\x6e", [0x4e6b] = "\xf7\x6f",
  [0x4e6c] = "\xf7\x70", [0x4e6d] = "\xf7\x71", [0x4e6e] = "\xf7\x72",
  [0x4e6f] = "\xf7\x73", [0x4e70] = "\xf7\x74", [0x4e71] = "\xf7\x75",
  [0x4e72] = "\xf7\x76", [0x4e73] = "\xf7\x77", [0x4e74] = "\xf7\x78",
  [0x4e75] = "\xf7\x79", [0x4e76] = "\xf7\x7a", [0x4e77] = "\xf7\x7b",
  [0x4e78] = "\xf7\x7c", [0x4e79] = "\xf7\x7d", [0x4e7a] = "\xf7\x7e",
  [0x4e7b] = "\xf7\x80", [0x4e7c] = "\xd3\xe3", [0x4e7d] = "\xf7\x81",
  [0x4e7e] = "\xf7\x82", [0x4e7f] = "\xf6\xcf", [0x4e80] = "\xf7\x83",
  [0x4e81] = "\xc2\xb3", [0x4e82] = "\xf6\xd0", [0x4e83] = "\xf7\x84",
  [0x4e84] = "\xf7\x85", [0x4e85] = "\xf6\xd1", [0x4e86] = "\xf6\xd2",
  [0x4e87] = "\xf6\xd3", [0x4e88] = "\xf6\xd4", [0x4e89] = "\xf7\x86",
  [0x4e8a] = "\xf7\x87", [0x4e8b] = "\xf6\xd6", [0x4e8c] = "\xf7\x88",
  [0x4e8d] = "\xb1\xab", [0x4e8e] = "\xf6\xd7", [0x4e8f] = "\xf7\x89",
  [0x4e90] = "\xf6\xd8", [0x4e91] = "\xf6\xd9", [0x4e92] = "\xf6\xda",
  [0x4e93] = "\xf7\x8a", [0x4e94] = "\xf6\xdb", [0x4e95] = "\xf6\xdc",
  [0x4e96] = "\xf7\x8b", [0x4e97] = "\xf7\x8c", [0x4e98] = "\xf7\x8d",
  [0x4e99] = "\xf7\x8e", [0x4e9a] = "\xf6\xdd", [0x4e9b] = "\xf6\xde",
  [0x4e9c] = "\xcf\xca", [0x4e9d] = "\xf7\x8f", [0x4e9e] = "\xf6\xdf",
  [0x4e9f] = "\xf6\xe0", [0x4ea0] = "\xf6\xe1", [0x4ea1] = "\xf6\xe2",
  [0x4ea2] = "\xf6\xe3", [0x4ea3] = "\xf6\xe4", [0x4ea4] = "\xc0\xf0",
  [0x4ea5] = "\xf6\xe5", [0x4ea6] = "\xf6\xe6", [0x4ea7] = "\xf6\xe7",
  [0x4ea8] = "\xf6\xe8", [0x4ea9] = "\xf6\xe9", [0x4eaa] = "\xf7\x90",
  [0x4eab] = "\xf6\xea", [0x4eac] = "\xf7\x91", [0x4ead] = "\xf6\xeb",
  [0x4eae] = "\xf6\xec", [0x4eaf] = "\xf7\x92", [0x4eb0] = "\xf6\xed",
  [0x4eb1] = "\xf6\xee", [0x4eb2] = "\xf6\xef", [0x4eb3] = "\xf6\xf0",
  [0x4eb4] = "\xf6\xf1", [0x4eb5] = "\xf6\xf2", [0x4eb6] = "\xf6\xf3",
  [0x4eb7] = "\xf6\xf4", [0x4eb8] = "\xbe\xa8", [0x4eb9] = "\xf7\x93",
  [0x4eba] = "\xf6\xf5", [0x4ebb] = "\xf6\xf6", [0x4ebc] = "\xf6\xf7",
  [0x4ebd] = "\xf6\xf8", [0x4ebe] = "\xf7\x94", [0x4ebf] = "\xf7\x95",
  [0x4ec0] = "\xf7\x96", [0x4ec1] = "\xf7\x97", [0x4ec2] = "\xf7\x98",
  [0x4ec3] = "\xc8\xfa", [0x4ec4] = "\xf6\xf9", [0x4ec5] = "\xf6\xfa",
  [0x4ec6] = "\xf6\xfb", [0x4ec7] = "\xf6\xfc", [0x4ec8] = "\xf7\x99",
  [0x4ec9] = "\xf7\x9a", [0x4eca] = "\xf6\xfd", [0x4ecb] = "\xf6\xfe",
  [0x4ecc] = "\xf7\xa1", [0x4ecd] = "\xf7\xa2", [0x4ece] = "\xf7\xa3",
  [0x4ecf] = "\xf7\xa4", [0x4ed0] = "\xf7\xa5", [0x4ed1] = "\xf7\x9b",
  [0x4ed2] = "\xf7\x9c", [0x4ed3] = "\xf7\xa6", [0x4ed4] = "\xf7\xa7",
  [0x4ed5] = "\xf7\xa8", [0x4ed6] = "\xb1\xee", [0x4ed7] = "\xf7\xa9",
  [0x4ed8] = "\xf7\xaa", [0x4ed9] = "\xf7\xab", [0x4eda] = "\xf7\x9d",
  [0x4edb] = "\xf7\x9e", [0x4edc] = "\xf7\xac", [0x4edd] = "\xf7\xad",
  [0x4ede] = "\xc1\xdb", [0x4edf] = "\xf7\xae", [0x4ee0] = "\xf7\x9f",
  [0x4ee1] = "\xf7\xa0", [0x4ee2] = "\xf7\xaf", [0x4ee3] = "\xf8\x40",
  [0x4ee4] = "\xf8\x41", [0x4ee5] = "\xf8\x42", [0x4ee6] = "\xf8\x43",
  [0x4ee7] = "\xf8\x44", [0x4ee8] = "\xf8\x45", [0x4ee9] = "\xf8\x46",
  [0x4eea] = "\xf8\x47", [0x4eeb] = "\xf8\x48", [0x4eec] = "\xf8\x49",
  [0x4eed] = "\xf8\x4a", [0x4eee] = "\xf8\x4b", [0x4eef] = "\xf8\x4c",
  [0x4ef0] = "\xf8\x4d", [0x4ef1] = "\xf8\x4e", [0x4ef2] = "\xf8\x4f",
  [0x4ef3] = "\xf8\x50", [0x4ef4] = "\xf8\x51", [0x4ef5] = "\xf8\x52",
  [0x4ef6] = "\xf8\x53", [0x4ef7] = "\xf8\x54", [0x4ef8] = "\xf8\x55",
  [0x4ef9] = "\xf8\x56", [0x4efa] = "\xf8\x57", [0x4efb] = "\xf8\x58",
  [0x4efc] = "\xf8\x59", [0x4efd] = "\xf8\x5a", [0x4efe] = "\xf8\x5b",
  [0x4eff] = "\xf8\x5c", [0x4f00] = "\xf8\x5d", [0x4f01] = "\xf8\x5e",
  [0x4f02] = "\xf8\x5f", [0x4f03] = "\xf8\x60", [0x4f04] = "\xf8\x61",
  [0x4f05] = "\xf8\x62", [0x4f06] = "\xf8\x63", [0x4f07] = "\xf8\x64",
  [0x4f08] = "\xf8\x65", [0x4f09] = "\xf8\x66", [0x4f0a] = "\xf8\x67",
  [0x4f0b] = "\xf8\x68", [0x4f0c] = "\xf8\x69", [0x4f0d] = "\xf8\x6a",
  [0x4f0e] = "\xf8\x6b", [0x4f0f] = "\xf8\x6c", [0x4f10] = "\xf8\x6d",
  [0x4f11] = "\xf8\x6e", [0x4f12] = "\xf8\x6f", [0x4f13] = "\xf8\x70",
  [0x4f14] = "\xf8\x71", [0x4f15] = "\xf8\x72", [0x4f16] = "\xf8\x73",
  [0x4f17] = "\xf8\x74", [0x4f18] = "\xf8\x75", [0x4f19] = "\xf8\x76",
  [0x4f1a] = "\xf8\x77", [0x4f1b] = "\xf8\x78", [0x4f1c] = "\xf8\x79",
  [0x4f1d] = "\xf8\x7a", [0x4f1e] = "\xf8\x7b", [0x4f1f] = "\xf8\x7c",
  [0x4f20] = "\xf8\x7d", [0x4f21] = "\xf8\x7e", [0x4f22] = "\xf8\x80",
  [0x4f23] = "\xf8\x81", [0x4f24] = "\xf8\x82", [0x4f25] = "\xf8\x83",
  [0x4f26] = "\xf8\x84", [0x4f27] = "\xf8\x85", [0x4f28] = "\xf8\x86",
  [0x4f29] = "\xf8\x87", [0x4f2a] = "\xf8\x88", [0x4f2b] = "\xf8\x89",
  [0x4f2c] = "\xf8\x8a", [0x4f2d] = "\xf8\x8b", [0x4f2e] = "\xf8\x8c",
  [0x4f2f] = "\xf8\x8d", [0x4f30] = "\xf8\x8e", [0x4f31] = "\xf8\x8f",
  [0x4f32] = "\xf8\x90", [0x4f33] = "\xf8\x91", [0x4f34] = "\xf8\x92",
  [0x4f35] = "\xf8\x93", [0x4f36] = "\xf8\x94", [0x4f37] = "\xf8\x95",
  [0x4f38] = "\xf8\x96", [0x4f39] = "\xf8\x97", [0x4f3a] = "\xf8\x98",
  [0x4f3b] = "\xf8\x99", [0x4f3c] = "\xf8\x9a", [0x4f3d] = "\xf8\x9b",
  [0x4f3e] = "\xf8\x9c", [0x4f3f] = "\xf8\x9d", [0x4f40] = "\xf8\x9e",
  [0x4f41] = "\xf8\x9f", [0x4f42] = "\xf8\xa0", [0x4f43] = "\xf9\x40",
  [0x4f44] = "\xf9\x41", [0x4f45] = "\xf9\x42", [0x4f46] = "\xf9\x43",
  [0x4f47] = "\xf9\x44", [0x4f48] = "\xf9\x45", [0x4f49] = "\xf9\x46",
  [0x4f4a] = "\xf9\x47", [0x4f4b] = "\xf9\x48", [0x4f4c] = "\xf9\x49",
  [0x4f4d] = "\xf9\x4a", [0x4f4e] = "\xf9\x4b", [0x4f4f] = "\xf9\x4c",
  [0x4f50] = "\xf9\x4d", [0x4f51] = "\xf9\x4e", [0x4f52] = "\xf9\x4f",
  [0x4f53] = "\xf9\x50", [0x4f54] = "\xf9\x51", [0x4f55] = "\xf9\x52",
  [0x4f56] = "\xf9\x53", [0x4f57] = "\xf9\x54", [0x4f58] = "\xf9\x55",
  [0x4f59] = "\xf9\x56", [0x4f5a] = "\xf9\x57", [0x4f5b] = "\xf9\x58",
  [0x4f5c] = "\xf9\x59", [0x4f5d] = "\xf9\x5a", [0x4f5e] = "\xf9\x5b",
  [0x4f5f] = "\xf9\x5c", [0x4f60] = "\xf9\x5d", [0x4f61] = "\xf9\x5e",
  [0x4f62] = "\xf9\x5f", [0x4f63] = "\xf9\x60", [0x4f64] = "\xf9\x61",
  [0x4f65] = "\xf9\x62", [0x4f66] = "\xf9\x63", [0x4f67] = "\xf9\x64",
  [0x4f68] = "\xf9\x65", [0x4f69] = "\xf9\x66", [0x4f6a] = "\xf9\x67",
  [0x4f6b] = "\xf9\x68", [0x4f6c] = "\xf9\x69", [0x4f6d] = "\xf9\x6a",
  [0x4f6e] = "\xf9\x6b", [0x4f6f] = "\xf9\x6c", [0x4f70] = "\xf9\x6d",
  [0x4f71] = "\xf9\x6e", [0x4f72] = "\xf9\x6f", [0x4f73] = "\xf9\x70",
  [0x4f74] = "\xf9\x71", [0x4f75] = "\xf9\x72", [0x4f76] = "\xf9\x73",
  [0x4f77] = "\xf9\x74", [0x4f78] = "\xf9\x75", [0x4f79] = "\xf9\x76",
  [0x4f7a] = "\xf9\x77", [0x4f7b] = "\xf9\x78", [0x4f7c] = "\xf9\x79",
  [0x4f7d] = "\xf9\x7a", [0x4f7e] = "\xf9\x7b", [0x4f7f] = "\xf9\x7c",
  [0x4f80] = "\xf9\x7d", [0x4f81] = "\xf9\x7e", [0x4f82] = "\xf9\x80",
  [0x4f83] = "\xf9\x81", [0x4f84] = "\xf9\x82", [0x4f85] = "\xf9\x83",
  [0x4f86] = "\xf9\x84", [0x4f87] = "\xf9\x85", [0x4f88] = "\xf9\x86",
  [0x4f89] = "\xf9\x87", [0x4f8a] = "\xf9\x88", [0x4f8b] = "\xf9\x89",
  [0x4f8c] = "\xf9\x8a", [0x4f8d] = "\xf9\x8b", [0x4f8e] = "\xf9\x8c",
  [0x4f8f] = "\xf9\x8d", [0x4f90] = "\xf9\x8e", [0x4f91] = "\xf9\x8f",
  [0x4f92] = "\xf9\x90", [0x4f93] = "\xf9\x91", [0x4f94] = "\xf9\x92",
  [0x4f95] = "\xf9\x93", [0x4f96] = "\xf9\x94", [0x4f97] = "\xf9\x95",
  [0x4f98] = "\xf9\x96", [0x4f99] = "\xf9\x97", [0x4f9a] = "\xf9\x98",
  [0x4f9b] = "\xf9\x99", [0x4f9c] = "\xf9\x9a", [0x4f9d] = "\xf9\x9b",
  [0x4f9e] = "\xf9\x9c", [0x4f9f] = "\xf9\x9d", [0x4fa0] = "\xf9\x9e",
  [0x4fa1] = "\xf9\x9f", [0x4fa2] = "\xf9\xa0", [0x4fa3] = "\xfa\x40",
  [0x4fa4] = "\xfa\x41", [0x4fa5] = "\xfa\x42", [0x4fa6] = "\xfa\x43",
  [0x4fa7] = "\xfa\x44", [0x4fa8] = "\xfa\x45", [0x4fa9] = "\xfa\x46",
  [0x4faa] = "\xfa\x47", [0x4fab] = "\xfa\x48", [0x4fac] = "\xfa\x49",
  [0x4fad] = "\xfa\x4a", [0x4fae] = "\xfa\x4b", [0x4faf] = "\xfa\x4c",
  [0x4fb0] = "\xfa\x4d", [0x4fb1] = "\xfa\x4e", [0x4fb2] = "\xfa\x4f",
  [0x4fb3] = "\xfa\x50", [0x4fb4] = "\xfa\x51", [0x4fb5] = "\xfa\x52",
  [0x4fb6] = "\xfa\x53", [0x4fb7] = "\xfa\x54", [0x4fb8] = "\xfa\x55",
  [0x4fb9] = "\xfa\x56", [0x4fba] = "\xfa\x57", [0x4fbb] = "\xfa\x58",
  [0x4fbc] = "\xfa\x59", [0x4fbd] = "\xfa\x5a", [0x4fbe] = "\xfa\x5b",
  [0x4fbf] = "\xfa\x5c", [0x4fc0] = "\xfa\x5d", [0x4fc1] = "\xfa\x5e",
  [0x4fc2] = "\xfa\x5f", [0x4fc3] = "\xfa\x60", [0x4fc4] = "\xfa\x61",
  [0x4fc5] = "\xfa\x62", [0x4fc6] = "\xfa\x63", [0x4fc7] = "\xfa\x64",
  [0x4fc8] = "\xfa\x65", [0x4fc9] = "\xfa\x66", [0x4fca] = "\xfa\x67",
  [0x4fcb] = "\xfa\x68", [0x4fcc] = "\xfa\x69", [0x4fcd] = "\xfa\x6a",
  [0x4fce] = "\xfa\x6b", [0x4fcf] = "\xfa\x6c", [0x4fd0] = "\xfa\x6d",
  [0x4fd1] = "\xfa\x6e", [0x4fd2] = "\xfa\x6f", [0x4fd3] = "\xfa\x70",
  [0x4fd4] = "\xfa\x71", [0x4fd5] = "\xfa\x72", [0x4fd6] = "\xfa\x73",
  [0x4fd7] = "\xfa\x74", [0x4fd8] = "\xfa\x75", [0x4fd9] = "\xfa\x76",
  [0x4fda] = "\xfa\x77", [0x4fdb] = "\xfa\x78", [0x4fdc] = "\xfa\x79",
  [0x4fdd] = "\xfa\x7a", [0x4fde] = "\xfa\x7b", [0x4fdf] = "\xfa\x7c",
  [0x4fe0] = "\xfa\x7d", [0x4fe1] = "\xfa\x7e", [0x4fe2] = "\xfa\x80",
  [0x4fe3] = "\xfa\x81", [0x4fe4] = "\xfa\x82", [0x4fe5] = "\xfa\x83",
  [0x4fe6] = "\xfa\x84", [0x4fe7] = "\xfa\x85", [0x4fe8] = "\xfa\x86",
  [0x4fe9] = "\xfa\x87", [0x4fea] = "\xfa\x88", [0x4feb] = "\xfa\x89",
  [0x4fec] = "\xfa\x8a", [0x4fed] = "\xfa\x8b", [0x4fee] = "\xfa\x8c",
  [0x4fef] = "\xfa\x8d", [0x4ff0] = "\xfa\x8e", [0x4ff1] = "\xfa\x8f",
  [0x4ff2] = "\xfa\x90", [0x4ff3] = "\xfa\x91", [0x4ff4] = "\xfa\x92",
  [0x4ff5] = "\xfa\x93", [0x4ff6] = "\xfa\x94", [0x4ff7] = "\xfa\x95",
  [0x4ff8] = "\xfa\x96", [0x4ff9] = "\xfa\x97", [0x4ffa] = "\xfa\x98",
  [0x4ffb] = "\xfa\x99", [0x4ffc] = "\xfa\x9a", [0x4ffd] = "\xfa\x9b",
  [0x4ffe] = "\xfa\x9c", [0x4fff] = "\xfa\x9d", [0x5000] = "\xfa\x9e",
  [0x5001] = "\xfa\x9f", [0x5002] = "\xfa\xa0", [0x5003] = "\xfb\x40",
  [0x5004] = "\xfb\x41", [0x5005] = "\xfb\x42", [0x5006] = "\xfb\x43",
  [0x5007] = "\xfb\x44", [0x5008] = "\xfb\x45", [0x5009] = "\xfb\x46",
  [0x500a] = "\xfb\x47", [0x500b] = "\xfb\x48", [0x500c] = "\xfb\x49",
  [0x500d] = "\xfb\x4a", [0x500e] = "\xfb\x4b", [0x500f] = "\xfb\x4c",
  [0x5010] = "\xfb\x4d", [0x5011] = "\xfb\x4e", [0x5012] = "\xfb\x4f",
  [0x5013] = "\xfb\x50", [0x5014] = "\xfb\x51", [0x5015] = "\xfb\x52",
  [0x5016] = "\xfb\x53", [0x5017] = "\xfb\x54", [0x5018] = "\xfb\x55",
  [0x5019] = "\xfb\x56", [0x501a] = "\xfb\x57", [0x501b] = "\xfb\x58",
  [0x501c] = "\xfb\x59", [0x501d] = "\xfb\x5a", [0x501e] = "\xfb\x5b",
  [0x501f] = "\xc4\xf1", [0x5020] = "\xf0\xaf", [0x5021] = "\xbc\xa6",
  [0x5022] = "\xf0\xb0", [0x5023] = "\xc3\xf9", [0x5024] = "\xfb\x5c",
  [0x5025] = "\xc5\xb8", [0x5026] = "\xd1\xbb", [0x5027] = "\xfb\x5d",
  [0x5028] = "\xf0\xb1", [0x5029] = "\xf0\xb2", [0x502a] = "\xf0\xb3",
  [0x502b] = "\xf0\xb4", [0x502c] = "\xf0\xb5", [0x502d] = "\xd1\xbc",
  [0x502e] = "\xfb\x5e", [0x502f] = "\xd1\xec", [0x5030] = "\xfb\x5f",
  [0x5031] = "\xf0\xb7", [0x5032] = "\xf0\xb6", [0x5033] = "\xd4\xa7",
  [0x5034] = "\xfb\x60", [0x5035] = "\xcd\xd2", [0x5036] = "\xf0\xb8",
  [0x5037] = "\xf0\xba", [0x5038] = "\xf0\xb9", [0x5039] = "\xf0\xbb",
  [0x503a] = "\xf0\xbc", [0x503b] = "\xfb\x61", [0x503c] = "\xfb\x62",
  [0x503d] = "\xb8\xeb", [0x503e] = "\xf0\xbd", [0x503f] = "\xba\xe8",
  [0x5040] = "\xfb\x63", [0x5041] = "\xf0\xbe", [0x5042] = "\xf0\xbf",
  [0x5043] = "\xbe\xe9", [0x5044] = "\xf0\xc0", [0x5045] = "\xb6\xec",
  [0x5046] = "\xf0\xc1", [0x5047] = "\xf0\xc2", [0x5048] = "\xf0\xc3",
  [0x5049] = "\xf0\xc4", [0x504a] = "\xc8\xb5", [0x504b] = "\xf0\xc5",
  [0x504c] = "\xf0\xc6", [0x504d] = "\xfb\x64", [0x504e] = "\xf0\xc7",
  [0x504f] = "\xc5\xf4", [0x5050] = "\xfb\x65", [0x5051] = "\xf0\xc8",
  [0x5052] = "\xfb\x66", [0x5053] = "\xfb\x67", [0x5054] = "\xfb\x68",
  [0x5055] = "\xf0\xc9", [0x5056] = "\xfb\x69", [0x5057] = "\xf0\xca",
  [0x5058] = "\xf7\xbd", [0x5059] = "\xfb\x6a", [0x505a] = "\xf0\xcb",
  [0x505b] = "\xf0\xcc", [0x505c] = "\xf0\xcd", [0x505d] = "\xfb\x6b",
  [0x505e] = "\xf0\xce", [0x505f] = "\xfb\x6c", [0x5060] = "\xfb\x6d",
  [0x5061] = "\xfb\x6e", [0x5062] = "\xfb\x6f", [0x5063] = "\xf0\xcf",
  [0x5064] = "\xba\xd7", [0x5065] = "\xfb\x70", [0x5066] = "\xf0\xd0",
  [0x5067] = "\xf0\xd1", [0x5068] = "\xf0\xd2", [0x5069] = "\xf0\xd3",
  [0x506a] = "\xf0\xd4", [0x506b] = "\xf0\xd5", [0x506c] = "\xf0\xd6",
  [0x506d] = "\xf0\xd8", [0x506e] = "\xfb\x71", [0x506f] = "\xfb\x72",
  [0x5070] = "\xd3\xa5", [0x5071] = "\xf0\xd7", [0x5072] = "\xfb\x73",
  [0x5073] = "\xf0\xd9", [0x5074] = "\xfb\x74", [0x5075] = "\xfb\x75",
  [0x5076] = "\xfb\x76", [0x5077] = "\xfb\x77", [0x5078] = "\xfb\x78",
  [0x5079] = "\xfb\x79", [0x507a] = "\xfb\x7a", [0x507b] = "\xfb\x7b",
  [0x507c] = "\xfb\x7c", [0x507d] = "\xfb\x7d", [0x507e] = "\xf5\xba",
  [0x507f] = "\xc2\xb9", [0x5080] = "\xfb\x7e", [0x5081] = "\xfb\x80",
  [0x5082] = "\xf7\xe4", [0x5083] = "\xfb\x81", [0x5084] = "\xfb\x82",
  [0x5085] = "\xfb\x83", [0x5086] = "\xfb\x84", [0x5087] = "\xf7\xe5",
  [0x5088] = "\xf7\xe6", [0x5089] = "\xfb\x85", [0x508a] = "\xfb\x86",
  [0x508b] = "\xf7\xe7", [0x508c] = "\xfb\x87", [0x508d] = "\xfb\x88",
  [0x508e] = "\xfb\x89", [0x508f] = "\xfb\x8a", [0x5090] = "\xfb\x8b",
  [0x5091] = "\xfb\x8c", [0x5092] = "\xf7\xe8", [0x5093] = "\xc2\xb4",
  [0x5094] = "\xfb\x8d", [0x5095] = "\xfb\x8e", [0x5096] = "\xfb\x8f",
  [0x5097] = "\xfb\x90", [0x5098] = "\xfb\x91", [0x5099] = "\xfb\x92",
  [0x509a] = "\xfb\x93", [0x509b] = "\xfb\x94", [0x509c] = "\xfb\x95",
  [0x509d] = "\xf7\xea", [0x509e] = "\xfb\x96", [0x509f] = "\xf7\xeb",
  [0x50a0] = "\xfb\x97", [0x50a1] = "\xfb\x98", [0x50a2] = "\xfb\x99",
  [0x50a3] = "\xfb\x9a", [0x50a4] = "\xfb\x9b", [0x50a5] = "\xfb\x9c",
  [0x50a6] = "\xc2\xf3", [0x50a7] = "\xfb\x9d", [0x50a8] = "\xfb\x9e",
  [0x50a9] = "\xfb\x9f", [0x50aa] = "\xfb\xa0", [0x50ab] = "\xfc\x40",
  [0x50ac] = "\xfc\x41", [0x50ad] = "\xfc\x42", [0x50ae] = "\xfc\x43",
  [0x50af] = "\xfc\x44", [0x50b0] = "\xfc\x45", [0x50b1] = "\xfc\x46",
  [0x50b2] = "\xfc\x47", [0x50b3] = "\xfc\x48", [0x50b4] = "\xf4\xf0",
  [0x50b5] = "\xfc\x49", [0x50b6] = "\xfc\x4a", [0x50b7] = "\xfc\x4b",
  [0x50b8] = "\xf4\xef", [0x50b9] = "\xfc\x4c", [0x50ba] = "\xfc\x4d",
  [0x50bb] = "\xc2\xe9", [0x50bc] = "\xfc\x4e", [0x50bd] = "\xf7\xe1",
  [0x50be] = "\xf7\xe2", [0x50bf] = "\xfc\x4f", [0x50c0] = "\xfc\x50",
  [0x50c1] = "\xfc\x51", [0x50c2] = "\xfc\x52", [0x50c3] = "\xfc\x53",
  [0x50c4] = "\xbb\xc6", [0x50c5] = "\xfc\x54", [0x50c6] = "\xfc\x55",
  [0x50c7] = "\xfc\x56", [0x50c8] = "\xfc\x57", [0x50c9] = "\xd9\xe4",
  [0x50ca] = "\xfc\x58", [0x50cb] = "\xfc\x59", [0x50cc] = "\xfc\x5a",
  [0x50cd] = "\xca\xf2", [0x50ce] = "\xc0\xe8", [0x50cf] = "\xf0\xa4",
  [0x50d0] = "\xfc\x5b", [0x50d1] = "\xba\xda", [0x50d2] = "\xfc\x5c",
  [0x50d3] = "\xfc\x5d", [0x50d4] = "\xc7\xad", [0x50d5] = "\xfc\x5e",
  [0x50d6] = "\xfc\x5f", [0x50d7] = "\xfc\x60", [0x50d8] = "\xc4\xac",
  [0x50d9] = "\xfc\x61", [0x50da] = "\xfc\x62", [0x50db] = "\xf7\xec",
  [0x50dc] = "\xf7\xed", [0x50dd] = "\xf7\xee", [0x50de] = "\xfc\x63",
  [0x50df] = "\xf7\xf0", [0x50e0] = "\xf7\xef", [0x50e1] = "\xfc\x64",
  [0x50e2] = "\xf7\xf1", [0x50e3] = "\xfc\x65", [0x50e4] = "\xfc\x66",
  [0x50e5] = "\xf7\xf4", [0x50e6] = "\xfc\x67", [0x50e7] = "\xf7\xf3",
  [0x50e8] = "\xfc\x68", [0x50e9] = "\xf7\xf2", [0x50ea] = "\xf7\xf5",
  [0x50eb] = "\xfc\x69", [0x50ec] = "\xfc\x6a", [0x50ed] = "\xfc\x6b",
  [0x50ee] = "\xfc\x6c", [0x50ef] = "\xf7\xf6", [0x50f0] = "\xfc\x6d",
  [0x50f1] = "\xfc\x6e", [0x50f2] = "\xfc\x6f", [0x50f3] = "\xfc\x70",
  [0x50f4] = "\xfc\x71", [0x50f5] = "\xfc\x72", [0x50f6] = "\xfc\x73",
  [0x50f7] = "\xfc\x74", [0x50f8] = "\xfc\x75", [0x50f9] = "\xed\xe9",
  [0x50fa] = "\xfc\x76", [0x50fb] = "\xed\xea", [0x50fc] = "\xed\xeb",
  [0x50fd] = "\xfc\x77", [0x50fe] = "\xf6\xbc", [0x50ff] = "\xfc\x78",
  [0x5100] = "\xfc\x79", [0x5101] = "\xfc\x7a", [0x5102] = "\xfc\x7b",
  [0x5103] = "\xfc\x7c", [0x5104] = "\xfc\x7d", [0x5105] = "\xfc\x7e",
  [0x5106] = "\xfc\x80", [0x5107] = "\xfc\x81", [0x5108] = "\xfc\x82",
  [0x5109] = "\xfc\x83", [0x510a] = "\xfc\x84", [0x510b] = "\xf6\xbd",
  [0x510c] = "\xfc\x85", [0x510d] = "\xf6\xbe", [0x510e] = "\xb6\xa6",
  [0x510f] = "\xfc\x86", [0x5110] = "\xd8\xbe", [0x5111] = "\xfc\x87",
  [0x5112] = "\xfc\x88", [0x5113] = "\xb9\xc4", [0x5114] = "\xfc\x89",
  [0x5115] = "\xfc\x8a", [0x5116] = "\xfc\x8b", [0x5117] = "\xd8\xbb",
  [0x5118] = "\xfc\x8c", [0x5119] = "\xdc\xb1", [0x511a] = "\xfc\x8d",
  [0x511b] = "\xfc\x8e", [0x511c] = "\xfc\x8f", [0x511d] = "\xfc\x90",
  [0x511e] = "\xfc\x91", [0x511f] = "\xfc\x92", [0x5120] = "\xca\xf3",
  [0x5121] = "\xfc\x93", [0x5122] = "\xf7\xf7", [0x5123] = "\xfc\x94",
  [0x5124] = "\xfc\x95", [0x5125] = "\xfc\x96", [0x5126] = "\xfc\x97",
  [0x5127] = "\xfc\x98", [0x5128] = "\xfc\x99", [0x5129] = "\xfc\x9a",
  [0x512a] = "\xfc\x9b", [0x512b] = "\xfc\x9c", [0x512c] = "\xf7\xf8",
  [0x512d] = "\xfc\x9d", [0x512e] = "\xfc\x9e", [0x512f] = "\xf7\xf9",
  [0x5130] = "\xfc\x9f", [0x5131] = "\xfc\xa0", [0x5132] = "\xfd\x40",
  [0x5133] = "\xfd\x41", [0x5134] = "\xfd\x42", [0x5135] = "\xfd\x43",
  [0x5136] = "\xfd\x44", [0x5137] = "\xf7\xfb", [0x5138] = "\xfd\x45",
  [0x5139] = "\xf7\xfa", [0x513a] = "\xfd\x46", [0x513b] = "\xb1\xc7",
  [0x513c] = "\xfd\x47", [0x513d] = "\xf7\xfc", [0x513e] = "\xf7\xfd",
  [0x513f] = "\xfd\x48", [0x5140] = "\xfd\x49", [0x5141] = "\xfd\x4a",
  [0x5142] = "\xfd\x4b", [0x5143] = "\xfd\x4c", [0x5144] = "\xf7\xfe",
  [0x5145] = "\xfd\x4d", [0x5146] = "\xfd\x4e", [0x5147] = "\xfd\x4f",
  [0x5148] = "\xfd\x50", [0x5149] = "\xfd\x51", [0x514a] = "\xfd\x52",
  [0x514b] = "\xfd\x53", [0x514c] = "\xfd\x54", [0x514d] = "\xfd\x55",
  [0x514e] = "\xfd\x56", [0x514f] = "\xfd\x57", [0x5150] = "\xc6\xeb",
  [0x5151] = "\xec\xb4", [0x5152] = "\xfd\x58", [0x5153] = "\xfd\x59",
  [0x5154] = "\xfd\x5a", [0x5155] = "\xfd\x5b", [0x5156] = "\xfd\x5c",
  [0x5157] = "\xfd\x5d", [0x5158] = "\xfd\x5e", [0x5159] = "\xfd\x5f",
  [0x515a] = "\xfd\x60", [0x515b] = "\xfd\x61", [0x515c] = "\xfd\x62",
  [0x515d] = "\xfd\x63", [0x515e] = "\xfd\x64", [0x515f] = "\xfd\x65",
  [0x5160] = "\xfd\x66", [0x5161] = "\xfd\x67", [0x5162] = "\xfd\x68",
  [0x5163] = "\xfd\x69", [0x5164] = "\xfd\x6a", [0x5165] = "\xfd\x6b",
  [0x5166] = "\xfd\x6c", [0x5167] = "\xfd\x6d", [0x5168] = "\xfd\x6e",
  [0x5169] = "\xfd\x6f", [0x516a] = "\xfd\x70", [0x516b] = "\xfd\x71",
  [0x516c] = "\xfd\x72", [0x516d] = "\xfd\x73", [0x516e] = "\xfd\x74",
  [0x516f] = "\xfd\x75", [0x5170] = "\xfd\x76", [0x5171] = "\xfd\x77",
  [0x5172] = "\xfd\x78", [0x5173] = "\xfd\x79", [0x5174] = "\xfd\x7a",
  [0x5175] = "\xfd\x7b", [0x5176] = "\xfd\x7c", [0x5177] = "\xfd\x7d",
  [0x5178] = "\xfd\x7e", [0x5179] = "\xfd\x80", [0x517a] = "\xfd\x81",
  [0x517b] = "\xfd\x82", [0x517c] = "\xfd\x83", [0x517d] = "\xfd\x84",
  [0x517e] = "\xfd\x85", [0x517f] = "\xb3\xdd", [0x5180] = "\xf6\xb3",
  [0x5181] = "\xfd\x86", [0x5182] = "\xfd\x87", [0x5183] = "\xf6\xb4",
  [0x5184] = "\xc1\xe4", [0x5185] = "\xf6\xb5", [0x5186] = "\xf6\xb6",
  [0x5187] = "\xf6\xb7", [0x5188] = "\xf6\xb8", [0x5189] = "\xf6\xb9",
  [0x518a] = "\xf6\xba", [0x518b] = "\xc8\xa3", [0x518c] = "\xf6\xbb",
  [0x518d] = "\xfd\x88", [0x518e] = "\xfd\x89", [0x518f] = "\xfd\x8a",
  [0x5190] = "\xfd\x8b", [0x5191] = "\xfd\x8c", [0x5192] = "\xfd\x8d",
  [0x5193] = "\xfd\x8e", [0x5194] = "\xfd\x8f", [0x5195] = "\xfd\x90",
  [0x5196] = "\xfd\x91", [0x5197] = "\xfd\x92", [0x5198] = "\xfd\x93",
  [0x5199] = "\xc1\xfa", [0x519a] = "\xb9\xa8", [0x519b] = "\xed\xe8",
  [0x519c] = "\xfd\x94", [0x519d] = "\xfd\x95", [0x519e] = "\xfd\x96",
  [0x519f] = "\xb9\xea", [0x51a0] = "\xd9\xdf", [0x51a1] = "\xfd\x97",
  [0x51a2] = "\xfd\x98", [0x51a3] = "\xfd\x99", [0x51a4] = "\xfd\x9a",
  [0x51a5] = "\xfd\x9b",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0xE[78]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0xe7c7,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab9[][2] =
{
#if USE_PRIVATE_AREA
  [0x0000] = "\xa8\xbc", [0x0001] = "\xa8\xbf", [0x0020] = "\xa9\x89",
  [0x0021] = "\xa9\x8a", [0x0022] = "\xa9\x8b", [0x0023] = "\xa9\x8c",
  [0x0024] = "\xa9\x8d", [0x0025] = "\xa9\x8e", [0x0026] = "\xa9\x8f",
  [0x0027] = "\xa9\x90", [0x0028] = "\xa9\x91", [0x0029] = "\xa9\x92",
  [0x002a] = "\xa9\x93", [0x002b] = "\xa9\x94", [0x002c] = "\xa9\x95",
  [0x004e] = "\xfe\x50", [0x004f] = "\xfe\x51", [0x0050] = "\xfe\x52",
  [0x0051] = "\xfe\x53", [0x0052] = "\xfe\x54", [0x0053] = "\xfe\x55",
  [0x0054] = "\xfe\x56", [0x0055] = "\xfe\x57", [0x0056] = "\xfe\x58",
  [0x0057] = "\xfe\x59", [0x0058] = "\xfe\x5a", [0x0059] = "\xfe\x5b",
  [0x005a] = "\xfe\x5c", [0x005b] = "\xfe\x5d", [0x005c] = "\xfe\x5e",
  [0x005d] = "\xfe\x5f", [0x005e] = "\xfe\x60", [0x005f] = "\xfe\x61",
  [0x0060] = "\xfe\x62", [0x0061] = "\xfe\x63", [0x0062] = "\xfe\x64",
  [0x0063] = "\xfe\x65", [0x0064] = "\xfe\x66", [0x0065] = "\xfe\x67",
  [0x0066] = "\xfe\x68", [0x0067] = "\xfe\x69", [0x0068] = "\xfe\x6a",
  [0x0069] = "\xfe\x6b", [0x006a] = "\xfe\x6c", [0x006b] = "\xfe\x6d",
  [0x006c] = "\xfe\x6e", [0x006d] = "\xfe\x6f", [0x006e] = "\xfe\x70",
  [0x006f] = "\xfe\x71", [0x0070] = "\xfe\x72", [0x0071] = "\xfe\x73",
  [0x0072] = "\xfe\x74", [0x0073] = "\xfe\x75", [0x0074] = "\xfe\x76",
  [0x0075] = "\xfe\x77", [0x0076] = "\xfe\x78", [0x0077] = "\xfe\x79",
  [0x0078] = "\xfe\x7a", [0x0079] = "\xfe\x7b", [0x007a] = "\xfe\x7c",
  [0x007b] = "\xfe\x7d", [0x007c] = "\xfe\x7e", [0x007d] = "\xfe\x80",
  [0x007e] = "\xfe\x81", [0x007f] = "\xfe\x82", [0x0080] = "\xfe\x83",
  [0x0081] = "\xfe\x84", [0x0082] = "\xfe\x85", [0x0083] = "\xfe\x86",
  [0x0084] = "\xfe\x87", [0x0085] = "\xfe\x88", [0x0086] = "\xfe\x89",
  [0x0087] = "\xfe\x8a", [0x0088] = "\xfe\x8b", [0x0089] = "\xfe\x8c",
  [0x008a] = "\xfe\x8d", [0x008b] = "\xfe\x8e", [0x008c] = "\xfe\x8f",
  [0x008d] = "\xfe\x90", [0x008e] = "\xfe\x91", [0x008f] = "\xfe\x92",
  [0x0090] = "\xfe\x93", [0x0091] = "\xfe\x94", [0x0092] = "\xfe\x95",
  [0x0093] = "\xfe\x96", [0x0094] = "\xfe\x97", [0x0095] = "\xfe\x98",
  [0x0096] = "\xfe\x99", [0x0097] = "\xfe\x9a", [0x0098] = "\xfe\x9b",
  [0x0099] = "\xfe\x9c", [0x009a] = "\xfe\x9d", [0x009b] = "\xfe\x9e",
  [0x009c] = "\xfe\x9f", [0x009d] = "\xfe\xa0",
#endif
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0xFA[0-2]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0xfa0c,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab10[][2] =
{
  [0x0000] = "\xfe\x40", [0x0001] = "\xfe\x41", [0x0002] = "\xfe\x42",
  [0x0003] = "\xfe\x43", [0x0005] = "\xfe\x44", [0x0007] = "\xfe\x45",
  [0x0008] = "\xfe\x46", [0x000c] = "\xfe\x47", [0x0013] = "\xfe\x48",
  [0x0014] = "\xfe\x49", [0x0015] = "\xfe\x4a", [0x0017] = "\xfe\x4b",
  [0x0018] = "\xfe\x4c", [0x001b] = "\xfe\x4d", [0x001c] = "\xfe\x4e",
  [0x001d] = "\xfe\x4f",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0xFE[3-6]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0xfe30,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab11[][2] =
{
  [0x0000] = "\xa9\x55", [0x0001] = "\xa6\xf2", [0x0003] = "\xa6\xf4",
  [0x0004] = "\xa6\xf5", [0x0005] = "\xa6\xe0", [0x0006] = "\xa6\xe1",
  [0x0007] = "\xa6\xf0", [0x0008] = "\xa6\xf1", [0x0009] = "\xa6\xe2",
  [0x000a] = "\xa6\xe3", [0x000b] = "\xa6\xee", [0x000c] = "\xa6\xef",
  [0x000d] = "\xa6\xe6", [0x000e] = "\xa6\xe7", [0x000f] = "\xa6\xe4",
  [0x0010] = "\xa6\xe5", [0x0011] = "\xa6\xe8", [0x0012] = "\xa6\xe9",
  [0x0013] = "\xa6\xea", [0x0014] = "\xa6\xeb", [0x0019] = "\xa9\x68",
  [0x001a] = "\xa9\x69", [0x001b] = "\xa9\x6a", [0x001c] = "\xa9\x6b",
  [0x001d] = "\xa9\x6c", [0x001e] = "\xa9\x6d", [0x001f] = "\xa9\x6e",
  [0x0020] = "\xa9\x6f", [0x0021] = "\xa9\x70", [0x0022] = "\xa9\x71",
  [0x0024] = "\xa9\x72", [0x0025] = "\xa9\x73", [0x0026] = "\xa9\x74",
  [0x0027] = "\xa9\x75", [0x0029] = "\xa9\x76", [0x002a] = "\xa9\x77",
  [0x002b] = "\xa9\x78", [0x002c] = "\xa9\x79", [0x002d] = "\xa9\x7a",
  [0x002e] = "\xa9\x7b", [0x002f] = "\xa9\x7c", [0x0030] = "\xa9\x7d",
  [0x0031] = "\xa9\x7e", [0x0032] = "\xa9\x80", [0x0033] = "\xa9\x81",
  [0x0034] = "\xa9\x82", [0x0035] = "\xa9\x83", [0x0036] = "\xa9\x84",
  [0x0038] = "\xa9\x85", [0x0039] = "\xa9\x86", [0x003a] = "\xa9\x87",
  [0x003b] = "\xa9\x88",
};

/* The table can be created using

   awk '{ print $2, $1 }' gbk.txt | sort | egrep '^0xFF[0-5]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $gbk, %rest) = split;
     local($u)=hex($ucs4);
     local($g)=hex($gbk);
     printf ("\n ") if ($n % 3 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\",", $u - 0xff01,
             $g < 0x100 ? $g : int($g/256), $g < 0x100 ? 0 : $g&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
static const char __gbk_from_ucs4_tab12[][2] =
{
  [0x0000] = "\xa3\xa1", [0x0001] = "\xa3\xa2", [0x0002] = "\xa3\xa3",
  [0x0003] = "\xa1\xe7", [0x0004] = "\xa3\xa5", [0x0005] = "\xa3\xa6",
  [0x0006] = "\xa3\xa7", [0x0007] = "\xa3\xa8", [0x0008] = "\xa3\xa9",
  [0x0009] = "\xa3\xaa", [0x000a] = "\xa3\xab", [0x000b] = "\xa3\xac",
  [0x000c] = "\xa3\xad", [0x000d] = "\xa3\xae", [0x000e] = "\xa3\xaf",
  [0x000f] = "\xa3\xb0", [0x0010] = "\xa3\xb1", [0x0011] = "\xa3\xb2",
  [0x0012] = "\xa3\xb3", [0x0013] = "\xa3\xb4", [0x0014] = "\xa3\xb5",
  [0x0015] = "\xa3\xb6", [0x0016] = "\xa3\xb7", [0x0017] = "\xa3\xb8",
  [0x0018] = "\xa3\xb9", [0x0019] = "\xa3\xba", [0x001a] = "\xa3\xbb",
  [0x001b] = "\xa3\xbc", [0x001c] = "\xa3\xbd", [0x001d] = "\xa3\xbe",
  [0x001e] = "\xa3\xbf", [0x001f] = "\xa3\xc0", [0x0020] = "\xa3\xc1",
  [0x0021] = "\xa3\xc2", [0x0022] = "\xa3\xc3", [0x0023] = "\xa3\xc4",
  [0x0024] = "\xa3\xc5", [0x0025] = "\xa3\xc6", [0x0026] = "\xa3\xc7",
  [0x0027] = "\xa3\xc8", [0x0028] = "\xa3\xc9", [0x0029] = "\xa3\xca",
  [0x002a] = "\xa3\xcb", [0x002b] = "\xa3\xcc", [0x002c] = "\xa3\xcd",
  [0x002d] = "\xa3\xce", [0x002e] = "\xa3\xcf", [0x002f] = "\xa3\xd0",
  [0x0030] = "\xa3\xd1", [0x0031] = "\xa3\xd2", [0x0032] = "\xa3\xd3",
  [0x0033] = "\xa3\xd4", [0x0034] = "\xa3\xd5", [0x0035] = "\xa3\xd6",
  [0x0036] = "\xa3\xd7", [0x0037] = "\xa3\xd8", [0x0038] = "\xa3\xd9",
  [0x0039] = "\xa3\xda", [0x003a] = "\xa3\xdb", [0x003b] = "\xa3\xdc",
  [0x003c] = "\xa3\xdd", [0x003d] = "\xa3\xde", [0x003e] = "\xa3\xdf",
  [0x003f] = "\xa3\xe0", [0x0040] = "\xa3\xe1", [0x0041] = "\xa3\xe2",
  [0x0042] = "\xa3\xe3", [0x0043] = "\xa3\xe4", [0x0044] = "\xa3\xe5",
  [0x0045] = "\xa3\xe6", [0x0046] = "\xa3\xe7", [0x0047] = "\xa3\xe8",
  [0x0048] = "\xa3\xe9", [0x0049] = "\xa3\xea", [0x004a] = "\xa3\xeb",
  [0x004b] = "\xa3\xec", [0x004c] = "\xa3\xed", [0x004d] = "\xa3\xee",
  [0x004e] = "\xa3\xef", [0x004f] = "\xa3\xf0", [0x0050] = "\xa3\xf1",
  [0x0051] = "\xa3\xf2", [0x0052] = "\xa3\xf3", [0x0053] = "\xa3\xf4",
  [0x0054] = "\xa3\xf5", [0x0055] = "\xa3\xf6", [0x0056] = "\xa3\xf7",
  [0x0057] = "\xa3\xf8", [0x0058] = "\xa3\xf9", [0x0059] = "\xa3\xfa",
  [0x005a] = "\xa3\xfb", [0x005b] = "\xa3\xfc", [0x005c] = "\xa3\xfd",
  [0x005d] = "\xa1\xab",
};


/* Definitions used in the body of the `gconv' function.  */
#define CHARSET_NAME		"GBK//"
#define FROM_LOOP		from_gbk
#define TO_LOOP			to_gbk
#define DEFINE_INIT		1
#define DEFINE_FINI		1
#define MIN_NEEDED_FROM		1
#define MAX_NEEDED_FROM		2
#define MIN_NEEDED_TO		4


/* First define the conversion function from GBK to UCS4.  */
#define MIN_NEEDED_INPUT	MIN_NEEDED_FROM
#define MAX_NEEDED_INPUT	MAX_NEEDED_FROM
#define MIN_NEEDED_OUTPUT	MIN_NEEDED_TO
#define LOOPFCT			FROM_LOOP
#define BODY \
  {									      \
    uint32_t ch = *inptr;						      \
									      \
    if (ch <= 0x7f)							      \
      ++inptr;								      \
    else								      \
      if (__builtin_expect (ch <= 0x80, 0)				      \
	  || __builtin_expect (ch > 0xfe, 0))				      \
	{								      \
	  /* This is illegal.  */					      \
	  if (! ignore_errors_p ())					      \
	    {								      \
	      result = __GCONV_ILLEGAL_INPUT;				      \
	      break;							      \
	    }								      \
									      \
	  ++inptr;							      \
	  ++*irreversible;						      \
	  continue;							      \
	}								      \
      else								      \
	{								      \
	  /* Two or more byte character.  First test whether the	      \
	     next character is also available.  */			      \
	  uint32_t ch2;							      \
	  int idx;							      \
									      \
	  if (__builtin_expect (inptr + 1 >= inend, 0))			      \
	    {								      \
	      /* The second character is not available.  Store		      \
		 the intermediate result.  */				      \
	      result = __GCONV_INCOMPLETE_INPUT;			      \
	      break;							      \
	    }								      \
									      \
	  ch2 = inptr[1];						      \
									      \
	  /* All second bytes of a multibyte character must be >= 0x40, and   \
	     the __gbk_to_ucs table only covers the range up to 0xfe 0xa0. */ \
	  if (__builtin_expect (ch2 < 0x40, 0)				      \
	      || (__builtin_expect (ch, 0x81) == 0xfe && ch2 > 0xa0))	      \
	    {								      \
	      /* This is an illegal character.  */			      \
	      if (! ignore_errors_p ())					      \
		{							      \
		  /* This is an illegal character.  */			      \
		  result = __GCONV_ILLEGAL_INPUT;			      \
		  break;						      \
		}							      \
									      \
	      ++inptr;							      \
	      ++*irreversible;						      \
	      continue;							      \
	    }								      \
									      \
	  /* This is code set 1: GBK.  */				      \
	  idx = (ch - 0x81)*192 + (ch2 - 0x40);				      \
									      \
	  ch = __gbk_to_ucs[idx];					      \
									      \
	  if (__builtin_expect (ch, 1) == 0 && *inptr != '\0')		      \
	    {								      \
	      /* This is an illegal character.  */			      \
	      if (! ignore_errors_p ())					      \
		{							      \
		  /* This is an illegal character.  */			      \
		  result = __GCONV_ILLEGAL_INPUT;			      \
		  break;						      \
		}							      \
									      \
	      inptr += 2;						      \
	      ++*irreversible;						      \
	      continue;							      \
	    }								      \
									      \
	  inptr += 2;							      \
	}								      \
									      \
    put32 (outptr, ch);							      \
    outptr += 4;							      \
  }
#define LOOP_NEED_FLAGS
#include <iconv/loop.c>


/* Next, define the other direction.  */
#define MIN_NEEDED_INPUT	MIN_NEEDED_TO
#define MIN_NEEDED_OUTPUT	MIN_NEEDED_FROM
#define MAX_NEEDED_OUTPUT	MAX_NEEDED_FROM
#define LOOPFCT			TO_LOOP
#define BODY \
  {									      \
    uint32_t ch = get32 (inptr);					      \
    unsigned char buf[2];						      \
    const unsigned char *cp = buf;					      \
									      \
    if (ch <= L'\x7f')							      \
      /* It's plain ASCII.  */						      \
      *outptr++ = (unsigned char) ch;					      \
    else								      \
      {									      \
      switch (ch)							      \
	{								      \
	case 0xa4 ... 0x101:						      \
	  cp = __gbk_from_ucs4_tab1[ch - 0xa4];				      \
	  break;							      \
	case 0x113:							      \
	  cp = "\xa8\xa5";						      \
	  break;							      \
	case 0x11b:							      \
	  cp = "\xa8\xa7";						      \
	  break;							      \
	case 0x12b:							      \
	  cp = "\xa8\xa9";						      \
	  break;							      \
	case 0x144:							      \
	  cp = "\xa8\xbd";						      \
	  break;							      \
	case 0x148:							      \
	  cp = "\xa8\xbe";						      \
	  break;							      \
	case 0x14d:							      \
	  cp = "\xa8\xad";						      \
	  break;							      \
	case 0x16b:							      \
	  cp = "\xa8\xb1";						      \
	  break;							      \
	case 0x1ce:							      \
	  cp = "\xa8\xa3";						      \
	  break;							      \
	case 0x1d0:							      \
	  cp = "\xa8\xab";						      \
	  break;							      \
	case 0x1d2:							      \
	  cp = "\xa8\xaf";						      \
	  break;							      \
	case 0x1d4:							      \
	  cp = "\xa8\xb3";						      \
	  break;							      \
	case 0x1d6:							      \
	  cp = "\xa8\xb5";						      \
	  break;							      \
	case 0x1d8:							      \
	  cp = "\xa8\xb6";						      \
	  break;							      \
	case 0x1da:							      \
	  cp = "\xa8\xb7";						      \
	  break;							      \
	case 0x1dc:							      \
	  cp = "\xa8\xb8";						      \
	  break;							      \
	case 0x251:							      \
	  cp = "\xa8\xbb";						      \
	  break;							      \
	case 0x261:							      \
	  cp = "\xa8\xc0";						      \
	  break;							      \
	case 0x2c7 ... 0x2cb:						      \
	  cp = "\xa1\xa6\0\0\0\0\0\0\xa1\xa5\0\0\xa8\x40\0\0\xa8\x41" + ((ch - 0x2c7) * 4); \
	  break;							      \
	case 0x2d9:							      \
	  cp = "\xa8\x42";						      \
	  break;							      \
	case 0x391 ... 0x3c9:						      \
	  cp = __gbk_from_ucs4_tab2[ch - 0x391];			      \
	  break;							      \
	case 0x401 ... 0x451:						      \
	  cp = __gbk_from_ucs4_tab3[ch - 0x401];			      \
	  break;							      \
	case 0x2010 ... 0x203b:						      \
	  cp = __gbk_from_ucs4_tab4[ch - 0x2010];			      \
	  break;							      \
	case 0x2103 ... 0x22bf:						      \
	  cp = __gbk_from_ucs4_tab5[ch - 0x2103];			      \
	  break;							      \
	case 0x2312:							      \
	  cp = "\xa1\xd0";						      \
	  break;							      \
	case 0x2460 ... 0x249b:						      \
	  cp = __gbk_from_ucs4_tab6[ch - 0x2460];			      \
	  break;							      \
	case 0x2500 ... 0x254b:						      \
	  buf[0] = '\xa9';						      \
	  buf[1] = '\xa4' + (ch - 0x2500);				      \
	  break;							      \
	case 0x2550 ... 0x2573:						      \
	  buf[0] = '\xa8';						      \
	  buf[1] = '\x54' + (ch - 0x2550);				      \
	  break;							      \
	case 0x2581 ... 0x2587:						      \
	  buf[0] = '\xa8';						      \
	  buf[1] = '\x78' + (ch - 0x2581);				      \
	  break;							      \
	case 0x2588 ... 0x258f:						      \
	  buf[0] = '\xa8';						      \
	  buf[1] = '\x80' + (ch - 0x2588);				      \
	  break;							      \
	case 0x2593 ... 0x2595:						      \
	  cp = "\xa8\x88\0\0\xa8\x89\0\0\xa8\x8a" + ((ch - 0x2593) * 4);      \
	  break;							      \
	case 0x25a0:							      \
	  cp = "\xa1\xf6";						      \
	  break;							      \
	case 0x25a1:							      \
	  cp = "\xa1\xf5";						      \
	  break;							      \
	case 0x25b2:							      \
	  cp = "\xa1\xf8";						      \
	  break;							      \
	case 0x25b3:							      \
	  cp = "\xa1\xf7";						      \
	  break;							      \
	case 0x25bc:							      \
	  cp = "\xa8\x8b";						      \
	  break;							      \
	case 0x25bd:							      \
	  cp = "\xa8\x8c";						      \
	  break;							      \
	case 0x25c6:							      \
	  cp = "\xa1\xf4";						      \
	  break;							      \
	case 0x25c7:							      \
	  cp = "\xa1\xf3";						      \
	  break;							      \
	case 0x25cb:							      \
	  cp = "\xa1\xf0";						      \
	  break;							      \
	case 0x25ce:							      \
	  cp = "\xa1\xf2";						      \
	  break;							      \
	case 0x25cf:							      \
	  cp = "\xa1\xf1";						      \
	  break;							      \
	case 0x25e2 ... 0x25e5:						      \
	  buf[0] = '\xa8';						      \
	  buf[1] = '\x8d' + (ch - 0x25e2);				      \
	  break;							      \
	case 0x2605:							      \
	  cp = "\xa1\xef";						      \
	  break;							      \
	case 0x2606:							      \
	  cp = "\xa1\xee";						      \
	  break;							      \
	case 0x2609:							      \
	  cp = "\xa8\x91";						      \
	  break;							      \
	case 0x2640:							      \
	  cp = "\xa1\xe2";						      \
	  break;							      \
	case 0x2642:							      \
	  cp = "\xa1\xe1";						      \
	  break;							      \
	case 0x3000 ... 0x3129:						      \
	  cp = __gbk_from_ucs4_tab7[ch - 0x3000];			      \
	  break;							      \
	case 0x3220 ... 0x3229:						      \
	  buf[0] = '\xa2';						      \
	  buf[1] = '\xe5' + (ch - 0x3220);				      \
	  break;							      \
	case 0x3231:							      \
	  cp = "\xa9\x5a";						      \
	  break;							      \
	case 0x32a3:							      \
	  cp = "\xa9\x49";						      \
	  break;							      \
	case 0x338e:							      \
	  cp = "\xa9\x4a";						      \
	  break;							      \
	case 0x338f:							      \
	  cp = "\xa9\x4b";						      \
	  break;							      \
	case 0x339c:							      \
	  cp = "\xa9\x4c";						      \
	  break;							      \
	case 0x339d:							      \
	  cp = "\xa9\x4d";						      \
	  break;							      \
	case 0x339e:							      \
	  cp = "\xa9\x4e";						      \
	  break;							      \
	case 0x33a1:							      \
	  cp = "\xa9\x4f";						      \
	  break;							      \
	case 0x33c4:							      \
	  cp = "\xa9\x50";						      \
	  break;							      \
	case 0x33ce:							      \
	  cp = "\xa9\x51";						      \
	  break;							      \
	case 0x33d1:							      \
	  cp = "\xa9\x52";						      \
	  break;							      \
	case 0x33d2:							      \
	  cp = "\xa9\x53";						      \
	  break;							      \
	case 0x33d5:							      \
	  cp = "\xa9\x54";						      \
	  break;							      \
	case 0x4e00 ... 0x9fa5:						      \
	  cp = __gbk_from_ucs4_tab8[ch - 0x4e00];			      \
	  break;							      \
	case 0xe7c7 ... 0xe864:						      \
	  cp = USE_PRIVATE_AREA ? __gbk_from_ucs4_tab9[ch - 0xe7c7] : "\0\0"; \
	  break;							      \
	case 0xf92c:							      \
	  cp = "\xfd\x9c";						      \
	  break;							      \
	case 0xf979:							      \
	  cp = "\xfd\x9d";						      \
	  break;							      \
	case 0xf995:							      \
	  cp = "\xfd\x9e";						      \
	  break;							      \
	case 0xf9e7:							      \
	  cp = "\xfd\x9f";						      \
	  break;							      \
	case 0xf9f1:							      \
	  cp = "\xfd\xa0";						      \
	  break;							      \
	case 0xfa0c ... 0xfa29:						      \
	  cp = __gbk_from_ucs4_tab10[ch - 0xfa0c];			      \
	  break;							      \
	case 0xfe30 ... 0xfe6b:						      \
	  cp = __gbk_from_ucs4_tab11[ch - 0xfe30];			      \
	  break;							      \
	case 0xff01 ... 0xff5e:						      \
	  cp = __gbk_from_ucs4_tab12[ch - 0xff01];			      \
	  break;							      \
	case 0xffe0 ... 0xffe5:						      \
	  cp = "\xa1\xe9\0\0\xa1\xea\0\0\xa9\x56\0\0\xa3\xfe\0\0\xa9\x57\0\0\xa3\xa4" + ((ch - 0xffe0) * 4); \
	  break;							      \
	default:							      \
	  UNICODE_TAG_HANDLER (ch, 4);					      \
	  cp = "";							      \
	  break; 							      \
	}								      \
      if (__builtin_expect (cp[0], '\1') == '\0' && ch != 0)		      \
	{								      \
	  /* Illegal character.  */					      \
	  STANDARD_ERR_HANDLER (4);					      \
	}								      \
      /* See whether there is enough room for the second byte we write.  */   \
      else if (cp[1] != '\0' && __builtin_expect (outptr + 1 >= outend, 0))   \
	{								      \
	  /* We have not enough room.  */				      \
	  result = __GCONV_FULL_OUTPUT;					      \
	  break;							      \
	}								      \
      else								      \
	{								      \
	  *outptr++ = cp[0];						      \
	  if (cp[1] != '\0')						      \
	    *outptr++ = cp[1];						      \
	}								      \
    }									      \
									      \
    inptr += 4;                                                               \
  }
#define LOOP_NEED_FLAGS
#include <iconv/loop.c>


/* Now define the toplevel functions.  */
#include <iconv/skeleton.c>
