/* misc.c -- miscellaneous functions for dap */

/*  Copyright (C) 2001 Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "dap_make.h"

extern FILE *dap_err;

static int mdays[] = { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

int dap_numdate(char date[])        /* number of days since 12311969 */
{
int mon;
int day;
int yr;
int m;
int y;
int ndays;

mon = 10 * (date[0] - '0') + date[1] - '0';
day = 10 * (date[2] - '0') + date[3] - '0';
yr = 10 * (10 * (10 * (date[4] - '0') + date[5] - '0') + date[6] - '0') + date[7] - '0';
for (m = 1, ndays = day; m < mon; m++)
        ndays += mdays[m];
if ((mon > 2) && !(yr % 4) && (yr % 100))
        ndays++;
if (yr < 1752)
	return -1;
for (y = 1752; y < yr; y++)
        {
        ndays += 365;
        if (!(y % 4) && ((y % 100) || !(y % 400)))
                ndays++;
        }
return ndays;
}

void dap_datenum(int n, char *d)
{
int mon;
int day;
int yr;
int ndays;

if (n <= 0)
	{
	strcpy(d, "?");
	return;
	}
yr = 1752;
sprintf(d, "0101%4d", yr);
ndays = dap_numdate(d);
while (ndays <= n)
	{
	if (yr < 10000)
		{
		sprintf(d, "0101%4d", ++yr);
		ndays = dap_numdate(d);
		}
	else
		{
		strcpy(d, "?");
		return;
		}
	}
--yr;
mon = 1;
sprintf(d, "%02d01%4d", mon, yr);
ndays = dap_numdate(d);
while (ndays <= n)
	{
	sprintf(d, "%02d01%4d", ++mon, yr);
	if (mon <= 12)
		ndays = dap_numdate(d);
	else
		break;
	}
--mon;
day = 1;
sprintf(d, "%02d%02d%4d", mon, day,  yr);
ndays = dap_numdate(d);
while (ndays < n)
	{
	sprintf(d, "%02d%02d%4d", mon, ++day, yr);
	if (day <= mdays[mon])
		ndays = dap_numdate(d);
	else
		exit(1);
	}
}

double dap_bincoeff(double n, double r)
{
double b;

for (b = 1.0; r > 0.0; r -= 1.0, n -= 1.0)
        b *= n / r;
return rint(b);
}

double dap_maximize(double (*f)(double x[]), int nx, double x[],
					double step, double tol, char *trace)
{
int tr;
double htol;
double *x1;
double f0;
double f1;
double step1;
double savex;
int maxdir;
double dirstep;
double maxf;
int first;
int n;
int d;
double s[2];

maxdir = 0;
dirstep = 0.0;
maxf = 0.0;
x1 = (double *) dap_malloc(nx * sizeof(double), "");
if (trace)
	{
	if (!trace[0])
		tr = 0;
	else if (!strcmp(trace, "TRACE"))
		tr = 1;
	else if (!strcmp(trace, "PAUSE"))
		tr = 2;
	else
		{
		fprintf(dap_err, "(maximize) Bad tracing option: %s\n", trace);
		exit(1);
		}
	}
else
	tr = 0;
htol = tol / 2;
for (f0 = (*f)(x), s[0] = -step, s[1] = step; ; )
	{
	step1 = step * htol;
	for (n = 0; n < nx; n++)
		x1[n] = x[n] + varunif() * step1;
	for (n = 0, first = 1; n < nx; n++)
		{
		savex = x1[n];
		for (d = 0; d < 2; d++)
			{
			x1[n] = savex + s[d];
			f1 = (*f)(x1);
			if (first || f1 > maxf)
				{
				first = 0;
				maxdir = n;
				dirstep = s[d];
				maxf = f1;
				}
			}
		x1[n] = savex;
		}
	if (tr == 1)
		{
		fprintf(dap_err, "(maximize) f = %g, step = %g\n", f0, step);
		for (n = 0; n < nx; n++)
			fprintf(dap_err, "%g ", x[n]);
		putc('\n', dap_err);
		}
	else if (tr == 2)
		{
		fprintf(stderr, "(maximize) f = %g, step = %g\n", f0, step);
		for (n = 0; n < nx; n++)
			fprintf(stderr, "%g ", x[n]);
		putc('\n', stderr);
		}
	if (maxf <= f0)
		{
		step /= 2.0;
		s[0] = -step;
		s[1] = step;
		if (tr)
			{
			fprintf(dap_err, "(maximize) Setting step to %g\n", step);
			if (tr == 2)
				getchar();
			}
		if (step < tol)
			break;
		}
	else
		{
		f0 = maxf;
		for (n = 0; n < nx; n++)
			x[n] = x1[n];
		x[maxdir] += dirstep;
		if (tr)
			{
			fputs("(maximize)", dap_err);
			for (n = 0; n < nx; n++)
				fprintf(dap_err, " %g", x[n]);
			putc('\n', dap_err);
			if (tr == 2)
				getchar();
			}
		}
	}
free(x1);
return f0;
}
