package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ProcessConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableProcessConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableProcessConfig implements ProcessConfig {
  private final Boolean privileged;
  private final @Nullable String user;
  private final Boolean tty;
  private final String entrypoint;
  private final List<String> arguments;

  private ImmutableProcessConfig(
      Boolean privileged,
      @Nullable String user,
      Boolean tty,
      String entrypoint,
      List<String> arguments) {
    this.privileged = privileged;
    this.user = user;
    this.tty = tty;
    this.entrypoint = entrypoint;
    this.arguments = arguments;
  }

  /**
   * @return The value of the {@code privileged} attribute
   */
  @JsonProperty("privileged")
  @Override
  public Boolean privileged() {
    return privileged;
  }

  /**
   * @return The value of the {@code user} attribute
   */
  @JsonProperty("user")
  @Override
  public @Nullable String user() {
    return user;
  }

  /**
   * @return The value of the {@code tty} attribute
   */
  @JsonProperty("tty")
  @Override
  public Boolean tty() {
    return tty;
  }

  /**
   * @return The value of the {@code entrypoint} attribute
   */
  @JsonProperty("entrypoint")
  @Override
  public String entrypoint() {
    return entrypoint;
  }

  /**
   * @return The value of the {@code arguments} attribute
   */
  @JsonProperty("arguments")
  @Override
  public List<String> arguments() {
    return arguments;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessConfig#privileged() privileged} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for privileged
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessConfig withPrivileged(Boolean value) {
    Boolean newValue = Objects.requireNonNull(value, "privileged");
    if (this.privileged.equals(newValue)) return this;
    return new ImmutableProcessConfig(newValue, this.user, this.tty, this.entrypoint, this.arguments);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessConfig#user() user} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for user (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessConfig withUser(@Nullable String value) {
    if (Objects.equals(this.user, value)) return this;
    return new ImmutableProcessConfig(this.privileged, value, this.tty, this.entrypoint, this.arguments);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessConfig#tty() tty} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for tty
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessConfig withTty(Boolean value) {
    Boolean newValue = Objects.requireNonNull(value, "tty");
    if (this.tty.equals(newValue)) return this;
    return new ImmutableProcessConfig(this.privileged, this.user, newValue, this.entrypoint, this.arguments);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProcessConfig#entrypoint() entrypoint} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for entrypoint
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableProcessConfig withEntrypoint(String value) {
    String newValue = Objects.requireNonNull(value, "entrypoint");
    if (this.entrypoint.equals(newValue)) return this;
    return new ImmutableProcessConfig(this.privileged, this.user, this.tty, newValue, this.arguments);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ProcessConfig#arguments() arguments}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableProcessConfig withArguments(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableProcessConfig(this.privileged, this.user, this.tty, this.entrypoint, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ProcessConfig#arguments() arguments}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of arguments elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableProcessConfig withArguments(Iterable<String> elements) {
    if (this.arguments == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableProcessConfig(this.privileged, this.user, this.tty, this.entrypoint, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableProcessConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableProcessConfig
        && equalTo(0, (ImmutableProcessConfig) another);
  }

  private boolean equalTo(int synthetic, ImmutableProcessConfig another) {
    return privileged.equals(another.privileged)
        && Objects.equals(user, another.user)
        && tty.equals(another.tty)
        && entrypoint.equals(another.entrypoint)
        && arguments.equals(another.arguments);
  }

  /**
   * Computes a hash code from attributes: {@code privileged}, {@code user}, {@code tty}, {@code entrypoint}, {@code arguments}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + privileged.hashCode();
    h += (h << 5) + Objects.hashCode(user);
    h += (h << 5) + tty.hashCode();
    h += (h << 5) + entrypoint.hashCode();
    h += (h << 5) + arguments.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code ProcessConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ProcessConfig{"
        + "privileged=" + privileged
        + ", user=" + user
        + ", tty=" + tty
        + ", entrypoint=" + entrypoint
        + ", arguments=" + arguments
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ProcessConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ProcessConfig instance
   */
  public static ImmutableProcessConfig copyOf(ProcessConfig instance) {
    if (instance instanceof ImmutableProcessConfig) {
      return (ImmutableProcessConfig) instance;
    }
    return ImmutableProcessConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableProcessConfig ImmutableProcessConfig}.
   * <pre>
   * ImmutableProcessConfig.builder()
   *    .privileged(Boolean) // required {@link ProcessConfig#privileged() privileged}
   *    .user(String | null) // nullable {@link ProcessConfig#user() user}
   *    .tty(Boolean) // required {@link ProcessConfig#tty() tty}
   *    .entrypoint(String) // required {@link ProcessConfig#entrypoint() entrypoint}
   *    .argument|addAllArguments(String) // {@link ProcessConfig#arguments() arguments} elements
   *    .build();
   * </pre>
   * @return A new ImmutableProcessConfig builder
   */
  public static ImmutableProcessConfig.Builder builder() {
    return new ImmutableProcessConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableProcessConfig ImmutableProcessConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_PRIVILEGED = 0x1L;
    private static final long INIT_BIT_TTY = 0x2L;
    private static final long INIT_BIT_ENTRYPOINT = 0x4L;
    private long initBits = 0x7L;

    private Boolean privileged;
    private String user;
    private Boolean tty;
    private String entrypoint;
    private List<String> arguments = new ArrayList<String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ProcessConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ProcessConfig instance) {
      Objects.requireNonNull(instance, "instance");
      this.privileged(instance.privileged());
      @Nullable String userValue = instance.user();
      if (userValue != null) {
        user(userValue);
      }
      this.tty(instance.tty());
      this.entrypoint(instance.entrypoint());
      addAllArguments(instance.arguments());
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessConfig#privileged() privileged} attribute.
     * @param privileged The value for privileged 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("privileged")
    public final Builder privileged(Boolean privileged) {
      this.privileged = Objects.requireNonNull(privileged, "privileged");
      initBits &= ~INIT_BIT_PRIVILEGED;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessConfig#user() user} attribute.
     * @param user The value for user (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("user")
    public final Builder user(@Nullable String user) {
      this.user = user;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessConfig#tty() tty} attribute.
     * @param tty The value for tty 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("tty")
    public final Builder tty(Boolean tty) {
      this.tty = Objects.requireNonNull(tty, "tty");
      initBits &= ~INIT_BIT_TTY;
      return this;
    }

    /**
     * Initializes the value for the {@link ProcessConfig#entrypoint() entrypoint} attribute.
     * @param entrypoint The value for entrypoint 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("entrypoint")
    public final Builder entrypoint(String entrypoint) {
      this.entrypoint = Objects.requireNonNull(entrypoint, "entrypoint");
      initBits &= ~INIT_BIT_ENTRYPOINT;
      return this;
    }

    /**
     * Adds one element to {@link ProcessConfig#arguments() arguments} list.
     * @param element A arguments element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder argument(String element) {
      this.arguments.add(Objects.requireNonNull(element, "arguments element"));
      return this;
    }

    /**
     * Adds elements to {@link ProcessConfig#arguments() arguments} list.
     * @param elements An array of arguments elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder arguments(String... elements) {
      for (String element : elements) {
        this.arguments.add(Objects.requireNonNull(element, "arguments element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ProcessConfig#arguments() arguments} list.
     * @param elements An iterable of arguments elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("arguments")
    public final Builder arguments(Iterable<String> elements) {
      this.arguments.clear();
      return addAllArguments(elements);
    }

    /**
     * Adds elements to {@link ProcessConfig#arguments() arguments} list.
     * @param elements An iterable of arguments elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllArguments(Iterable<String> elements) {
      for (String element : elements) {
        this.arguments.add(Objects.requireNonNull(element, "arguments element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableProcessConfig ImmutableProcessConfig}.
     * @return An immutable instance of ProcessConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableProcessConfig build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableProcessConfig(privileged, user, tty, entrypoint, createUnmodifiableList(true, arguments));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_PRIVILEGED) != 0) attributes.add("privileged");
      if ((initBits & INIT_BIT_TTY) != 0) attributes.add("tty");
      if ((initBits & INIT_BIT_ENTRYPOINT) != 0) attributes.add("entrypoint");
      return "Cannot build ProcessConfig, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
