/**
 * File:    ImportJWTTransformationDesc.java
 * Created: 12.12.2008
 *
 *
/*******************************************************************************
 * Copyright (c) 2008  Open Wide (www.openwide.fr)
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Mickael Istria, Open Wide, Lyon, France
 *      - Creation and implementation
 *      - Refactoring to fit with jwt-we extensions
 *		- Refactoring to share this composite in wizards and dialogs
 *    Guillaume Decarnin, Open Wide, Lyon, France
 *      - Small improves on UI
 *******************************************************************************/
package org.eclipse.jwt.transformations.importTransformations;

import java.io.File;
import java.util.Set;

import org.eclipse.jwt.transformations.base.JWTTransformationRequestDesc;
import org.eclipse.jwt.transformations.base.TransformationsMessages;
import org.eclipse.jwt.transformations.base.init.ProcessServiceMembers;
import org.eclipse.jwt.transformations.base.init.TransformationRegistry;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;

/**
 * Definition an import transformation request description and a generic
 * UI composite that can be used any time we want user to input a transformation
 * request
 * 
 * TODO give it the ability to open imported model in a JWT Workflow Editor
 * instance
 * @author mistria
 *
 */
public class ImportJWTTransformationRequestDesc extends JWTTransformationRequestDesc {

	public ImportJWTTransformationRequestDesc(Composite parent, Listener listener) {
		super(parent, listener);
	}
	
	public ImportJWTTransformationRequestDesc(Composite parent) {
		super(parent);
	}

	public Composite createControl() {
		Composite composite = new Composite(parentComposite, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// ////////////////
		// In File Group
		Group inFileGroup = new Group(composite, SWT.None);
		inFileGroup.setText(TransformationsMessages.importInputFileGroup);
		inFileGroup.setLayout(new GridLayout(2, false));
		inFileGroup.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Text inFileText = new Text(inFileGroup, SWT.SINGLE | SWT.BORDER);
		GridData filegridData = new GridData(GridData.FILL_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL);
		inFileText.setLayoutData(filegridData);

		inFileText.setText(""); //$NON-NLS-1$
		
		final Button inFileBrowse = new Button(inFileGroup, SWT.NULL);
		inFileBrowse.setText(TransformationsMessages.browse);
		// End of In File Group
		// ///////////////////////
		
		// //////////////////////
		// Transformation Group
		final Group transformationGroup = new Group(composite, SWT.NULL);
		transformationGroup.setText(TransformationsMessages.transformationGroup);
		transformationGroup.setLayout(new GridLayout(1, false));
		transformationGroup.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Combo transformationsCombo = new Combo(transformationGroup, SWT.READ_ONLY);
		transformationsCombo.setText(TransformationsMessages.chooseImportTransformation);
		transformationsCombo.setLayoutData(new GridData(GridData.FILL_BOTH));
		Set<String> transformationsName = TransformationRegistry.getInstance()
				.getTransformationsName();
		for (String trans : transformationsName) {
			if (ProcessServiceMembers.Type.IMPORT.equals(
					TransformationRegistry.getInstance().getTransformation(trans).getTye()))
			transformationsCombo.add(trans);
		}

		final Label descriptionLabel = new Label(transformationGroup, SWT.LEFT | SWT.WRAP | SWT.H_SCROLL);
		// End of Transformation Group
		///////////////////////////////
		
		// ////////////////////
		// Out File Group
		Group outFileGroup = new Group(composite, SWT.NULL);
		outFileGroup.setText(TransformationsMessages.outputFileGroup);
		outFileGroup.setLayout(new GridLayout(2, false));
		outFileGroup.setLayoutData(new GridData(GridData.FILL_BOTH));

		final Text outFileText = new Text(outFileGroup, SWT.SINGLE | SWT.BORDER);
		outFileText.setLayoutData(filegridData);

		Button outFileBrowse = new Button(outFileGroup, SWT.NULL);
		outFileBrowse.setText(TransformationsMessages.browse);
		
		final Button checkbox = new Button(outFileGroup, SWT.CHECK);
		checkbox.setSelection(false);
		checkbox.setText(TransformationsMessages.openCreatedJWTModel);
		checkbox.setLayoutData(new GridData(SWT.NULL, SWT.NULL, false, false,
				2, 1));
		checkbox.setEnabled(true);
		// End of Out File Group
		// ///////////////////////

		
		///////////////
		// Listeners //
		inFileBrowse.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				// opens a dialog to select the file
				FileDialog dialog = new FileDialog(getShell(), SWT.OPEN);
				dialog.setText(TransformationsMessages.selectImportInputFile);
				dialog.setFilterExtensions(new String[] { "*.*" }); //$NON-NLS-1$
				inFilePath = dialog.open();
				if ((inFilePath != null) && (inFilePath.length() != 0)) {
					inFileText.setText(inFilePath);
					listener.handleEvent(e);
				}
			}
		});
		outFileBrowse.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				// opens a dialog to select the file
				FileDialog dialog = new FileDialog(getShell(), SWT.SAVE);
				dialog.setText(TransformationsMessages.selectImportOutputFile);
				dialog.setFilterExtensions(new String[] { "*.*" }); //$NON-NLS-1$
				outFileText.setText(dialog.open());
				if ((outFilePath != null) && (outFilePath.length() != 0)) {
					outFileText.setText(outFilePath);
					listener.handleEvent(e);
				}
			}
		});

		inFileText.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				inFilePath = inFileText.getText();
				listener.handleEvent(e);
			}
		});
		outFileText.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				outFilePath = outFileText.getText();
				listener.handleEvent(e);
			}
		});
		checkbox.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				openCreatedFile = checkbox.getSelection();
			}
		});
		transformationsCombo.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event e) {
				transformationName = transformationsCombo.getText();
				descriptionLabel.setText(TransformationRegistry.getInstance()
								.getTransformation(
										transformationsCombo.getText())
								.getDescription());
				transformationGroup.layout(true);
				parentComposite.update();
				listener.handleEvent(e);
			}
		});
		
		return composite;
	}
	
	public boolean hasAllRequiredInfo() {
		return 
			inFilePath != null && new File(inFilePath).exists() &&
			transformationName != null && TransformationRegistry.getInstance().getTransformationsName().contains(transformationName) &&
			outFilePath != null && outFilePath.length() != 0;
	}
}
