/**
 * Copyright (c) 2010-2015, Marton Bur, Zoltan Ujhelyi, Akos Horvath, Istvan Rath and Danil Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Marton Bur - initial API and implementation
 */
package org.eclipse.incquery.runtime.localsearch.planner;

import com.google.common.base.Objects;
import com.google.common.collect.Sets;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.incquery.runtime.localsearch.planner.PConstraintCategory;
import org.eclipse.incquery.runtime.matchers.context.IInputKey;
import org.eclipse.incquery.runtime.matchers.context.IQueryMetaContext;
import org.eclipse.incquery.runtime.matchers.context.IQueryRuntimeContext;
import org.eclipse.incquery.runtime.matchers.context.InputKeyImplication;
import org.eclipse.incquery.runtime.matchers.psystem.PBody;
import org.eclipse.incquery.runtime.matchers.psystem.PConstraint;
import org.eclipse.incquery.runtime.matchers.psystem.PVariable;
import org.eclipse.incquery.runtime.matchers.psystem.basicdeferred.ExportedParameter;
import org.eclipse.incquery.runtime.matchers.psystem.basicenumerables.ConstantValue;
import org.eclipse.incquery.runtime.matchers.psystem.basicenumerables.TypeConstraint;
import org.eclipse.incquery.runtime.matchers.tuple.Tuple;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * Wraps a PConstraint together with information required for the planner. Currently contains information about the expected binding state of
 * the affected variables also called application condition, and the cost of the enforcement, based on the meta and/or the runtime context.
 * 
 * @author Marton Bur
 */
@SuppressWarnings("all")
public class PConstraintInfo {
  private PConstraint constraint;
  
  private Set<PVariable> boundMaskVariables;
  
  private Set<PVariable> freeMaskVariables;
  
  private Set<PConstraintInfo> sameWithDifferentBindings;
  
  private IQueryRuntimeContext runtimeContext;
  
  private static float MAX_COST = 250.0f;
  
  private static float DEFAULT_COST = (PConstraintInfo.MAX_COST - 100.0f);
  
  private float cost;
  
  /**
   * Instantiates the wrapper
   * @param constraintfor which the information is added and stored
   * @param boundMaskVariablesthe bound variables in the operation mask
   * @param freeMaskVariablesthe free variables in the operation mask
   * @param sameWithDifferentBindingsduring the planning process, multiple operation adornments are considered for a constraint, so that it
   * is represented by multiple plan infos. This parameter contains all plan infos that are for the same
   * constraint, but with different adornment
   * @param runtimeContextthe runtime query context
   */
  public PConstraintInfo(final PConstraint constraint, final Set<PVariable> boundMaskVariables, final Set<PVariable> freeMaskVariables, final Set<PConstraintInfo> sameWithDifferentBindings, final IQueryRuntimeContext runtimeContext) {
    this.constraint = constraint;
    this.boundMaskVariables = boundMaskVariables;
    this.freeMaskVariables = freeMaskVariables;
    this.sameWithDifferentBindings = sameWithDifferentBindings;
    this.runtimeContext = runtimeContext;
    this.calculateCost(constraint);
  }
  
  protected void _calculateCost(final ConstantValue constant) {
    this.cost = 1.0f;
    return;
  }
  
  protected void _calculateCost(final TypeConstraint typeConstraint) {
    IInputKey supplierKey = ((TypeConstraint) this.constraint).getSupplierKey();
    long arity = supplierKey.getArity();
    if ((arity == 1)) {
      this.calculateUnaryConstraintCost(supplierKey);
    } else {
      if ((arity == 2)) {
        long edgeCount = this.runtimeContext.countTuples(supplierKey, null);
        Tuple _variablesTuple = ((TypeConstraint) this.constraint).getVariablesTuple();
        Object _get = _variablesTuple.get(0);
        PVariable srcVariable = ((PVariable) _get);
        Tuple _variablesTuple_1 = ((TypeConstraint) this.constraint).getVariablesTuple();
        Object _get_1 = _variablesTuple_1.get(1);
        PVariable dstVariable = ((PVariable) _get_1);
        boolean isInverse = false;
        boolean _and = false;
        boolean _contains = this.freeMaskVariables.contains(srcVariable);
        if (!_contains) {
          _and = false;
        } else {
          boolean _contains_1 = this.boundMaskVariables.contains(dstVariable);
          _and = _contains_1;
        }
        if (_and) {
          isInverse = true;
        }
        boolean _or = false;
        boolean _contains_2 = this.freeMaskVariables.contains(srcVariable);
        if (_contains_2) {
          _or = true;
        } else {
          boolean _contains_3 = this.freeMaskVariables.contains(dstVariable);
          _or = _contains_3;
        }
        if (_or) {
          this.calculateBinaryExtendCost(supplierKey, srcVariable, dstVariable, isInverse, edgeCount);
        } else {
          this.cost = 1.0f;
        }
      } else {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Cost calculation for arity ");
        _builder.append(arity, "");
        _builder.append(" is not implemented yet");
        throw new RuntimeException(_builder.toString());
      }
    }
  }
  
  protected void calculateBinaryExtendCost(final IInputKey supplierKey, final PVariable srcVariable, final PVariable dstVariable, final boolean isInverse, final long edgeCount) {
    IQueryMetaContext metaContext = this.runtimeContext.getMetaContext();
    Collection<InputKeyImplication> implications = metaContext.getImplications(supplierKey);
    long srcCount = (-1);
    long dstCount = (-1);
    for (final InputKeyImplication implication : implications) {
      {
        List<Integer> impliedIndices = implication.getImpliedIndices();
        boolean _and = false;
        int _size = impliedIndices.size();
        boolean _equals = (_size == 1);
        if (!_equals) {
          _and = false;
        } else {
          boolean _contains = impliedIndices.contains(Integer.valueOf(0));
          _and = _contains;
        }
        if (_and) {
          IInputKey _impliedKey = implication.getImpliedKey();
          int _countTuples = this.runtimeContext.countTuples(_impliedKey, null);
          srcCount = _countTuples;
        } else {
          boolean _and_1 = false;
          int _size_1 = impliedIndices.size();
          boolean _equals_1 = (_size_1 == 1);
          if (!_equals_1) {
            _and_1 = false;
          } else {
            boolean _contains_1 = impliedIndices.contains(Integer.valueOf(1));
            _and_1 = _contains_1;
          }
          if (_and_1) {
            IInputKey _impliedKey_1 = implication.getImpliedKey();
            int _countTuples_1 = this.runtimeContext.countTuples(_impliedKey_1, null);
            dstCount = _countTuples_1;
          }
        }
      }
    }
    boolean _and = false;
    boolean _contains = this.freeMaskVariables.contains(srcVariable);
    if (!_contains) {
      _and = false;
    } else {
      boolean _contains_1 = this.freeMaskVariables.contains(dstVariable);
      _and = _contains_1;
    }
    if (_and) {
      this.cost = (dstCount * srcCount);
    } else {
      long srcNodeCount = (-1);
      long dstNodeCount = (-1);
      if (isInverse) {
        srcNodeCount = dstCount;
        dstNodeCount = srcCount;
      } else {
        srcNodeCount = srcCount;
        dstNodeCount = dstCount;
      }
      if (((srcNodeCount > (-1)) && (edgeCount > (-1)))) {
        if ((srcNodeCount == 0)) {
          this.cost = 0;
        } else {
          this.cost = (((float) edgeCount) / srcNodeCount);
        }
      } else {
        if (((srcCount > (-1)) && (dstCount > (-1)))) {
          if ((srcCount != 0)) {
            this.cost = (dstNodeCount / srcNodeCount);
          } else {
            this.cost = 1.0f;
          }
        } else {
          Map<Set<PVariable>, Set<PVariable>> functionalDependencies = this.constraint.getFunctionalDependencies(metaContext);
          Set<PVariable> impliedVariables = functionalDependencies.get(this.boundMaskVariables);
          boolean _and_1 = false;
          boolean _notEquals = (!Objects.equal(impliedVariables, null));
          if (!_notEquals) {
            _and_1 = false;
          } else {
            boolean _containsAll = impliedVariables.containsAll(this.freeMaskVariables);
            _and_1 = _containsAll;
          }
          if (_and_1) {
            this.cost = 1.0f;
          } else {
            this.cost = PConstraintInfo.DEFAULT_COST;
          }
        }
      }
    }
    return;
  }
  
  protected float calculateUnaryConstraintCost(final IInputKey supplierKey) {
    float _xblockexpression = (float) 0;
    {
      Tuple _variablesTuple = ((TypeConstraint) this.constraint).getVariablesTuple();
      Object _get = _variablesTuple.get(0);
      PVariable variable = ((PVariable) _get);
      float _xifexpression = (float) 0;
      boolean _contains = this.boundMaskVariables.contains(variable);
      if (_contains) {
        _xifexpression = this.cost = 0.9f;
      } else {
        int _countTuples = this.runtimeContext.countTuples(supplierKey, null);
        _xifexpression = this.cost = _countTuples;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  protected void _calculateCost(final ExportedParameter exportedParam) {
    this.cost = PConstraintInfo.MAX_COST;
  }
  
  /**
   * Default cost calculation strategy
   */
  protected void _calculateCost(final PConstraint constraint) {
    boolean _isEmpty = this.freeMaskVariables.isEmpty();
    if (_isEmpty) {
      this.cost = 1.0f;
    } else {
      this.cost = PConstraintInfo.DEFAULT_COST;
    }
  }
  
  public PConstraint getConstraint() {
    return this.constraint;
  }
  
  public Set<PVariable> getFreeVariables() {
    return this.freeMaskVariables;
  }
  
  public Set<PVariable> getBoundVariables() {
    return this.boundMaskVariables;
  }
  
  public Set<PConstraintInfo> getSameWithDifferentBindings() {
    return this.sameWithDifferentBindings;
  }
  
  public float getCost() {
    return this.cost;
  }
  
  public PConstraintCategory getCategory(final PBody pBody, final Set<PVariable> boundVariables) {
    Sets.SetView<PVariable> _intersection = Sets.<PVariable>intersection(this.freeMaskVariables, boundVariables);
    int _size = _intersection.size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      return PConstraintCategory.PAST;
    } else {
      Set<PVariable> _allVariables = pBody.getAllVariables();
      Sets.SetView<PVariable> _difference = Sets.<PVariable>difference(_allVariables, boundVariables);
      Sets.SetView<PVariable> _intersection_1 = Sets.<PVariable>intersection(this.boundMaskVariables, _difference);
      int _size_1 = _intersection_1.size();
      boolean _greaterThan_1 = (_size_1 > 0);
      if (_greaterThan_1) {
        return PConstraintCategory.FUTURE;
      } else {
        return PConstraintCategory.PRESENT;
      }
    }
  }
  
  @Override
  public String toString() {
    StringConcatenation _builder = new StringConcatenation();
    String _format = String.format("\n");
    _builder.append(_format, "");
    String _string = this.constraint.toString();
    _builder.append(_string, "");
    _builder.append(", bound variables: ");
    _builder.append(this.boundMaskVariables, "");
    _builder.append(", cost: ");
    String _format_1 = String.format("%.2f", Float.valueOf(this.cost));
    _builder.append(_format_1, "");
    return _builder.toString();
  }
  
  protected void calculateCost(final PConstraint exportedParam) {
    if (exportedParam instanceof ExportedParameter) {
      _calculateCost((ExportedParameter)exportedParam);
      return;
    } else if (exportedParam instanceof ConstantValue) {
      _calculateCost((ConstantValue)exportedParam);
      return;
    } else if (exportedParam instanceof TypeConstraint) {
      _calculateCost((TypeConstraint)exportedParam);
      return;
    } else if (exportedParam != null) {
      _calculateCost(exportedParam);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(exportedParam).toString());
    }
  }
}
