/*******************************************************************************
 * Copyright © 2000, 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *
 *******************************************************************************/
package org.eclipse.edt.ide.core.internal.model;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceProxy;
import org.eclipse.core.resources.IResourceProxyVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.edt.ide.core.model.EGLModelException;
import org.eclipse.edt.ide.core.model.IEGLModel;
import org.eclipse.edt.ide.core.model.IEGLPathEntry;
import org.eclipse.edt.ide.core.model.IEGLProject;
import org.eclipse.edt.ide.core.model.IPackageFragmentRoot;


/**
 * @author mattclem
 *
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
public class MovePackageFragmentRootOperation extends CopyPackageFragmentRootOperation {
	/*
	 * Renames the classpath entries equal to the given path in the given project.
	 * If an entry with the destination path already existed, remove it.
	 */
	protected void renameEntryInClasspath(IPath rootPath, IEGLProject project) throws EGLModelException {
			
		IEGLPathEntry[] classpath = project.getRawEGLPath();
		IEGLPathEntry[] newClasspath = null;
		int cpLength = classpath.length;
		int newCPIndex = -1;

		for (int i = 0; i < cpLength; i++) {
			IEGLPathEntry entry = classpath[i];
			IPath entryPath = entry.getPath();
			if (rootPath.equals(entryPath)) {
				// rename entry
				if (newClasspath == null) {
					newClasspath = new IEGLPathEntry[cpLength];
					System.arraycopy(classpath, 0, newClasspath, 0, i);
					newCPIndex = i;
				}
				newClasspath[newCPIndex++] = copy(entry);
			} else if (this.destination.equals(entryPath)) {
				// remove entry equals to destination
				if (newClasspath == null) {
					newClasspath = new IEGLPathEntry[cpLength];
					System.arraycopy(classpath, 0, newClasspath, 0, i);
					newCPIndex = i;
				}
			} else if (newClasspath != null) {
				newClasspath[newCPIndex++] = entry;
			}
		}
		
		if (newClasspath != null) {
			if (newCPIndex < newClasspath.length) {
				System.arraycopy(newClasspath, 0, newClasspath = new IEGLPathEntry[newCPIndex], 0, newCPIndex);
			}
			project.setRawEGLPath(newClasspath, fMonitor);
		}
	}

	public MovePackageFragmentRootOperation(
		IPackageFragmentRoot root,
		IPath destination,
		int updateResourceFlags,
		int updateModelFlags,
		IEGLPathEntry sibling) {
			
		super(
			root,
			destination,
			updateResourceFlags,
			updateModelFlags,
			sibling);
	}
	protected void executeOperation() throws EGLModelException {
		
		IPackageFragmentRoot root = (IPackageFragmentRoot)this.getElementToProcess();
		IEGLPathEntry rootEntry = root.getRawEGLPathEntry();
		IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
		
		// move resource
		if (!root.isExternal() && (this.updateModelFlags & IPackageFragmentRoot.NO_RESOURCE_MODIFICATION) == 0) {
			moveResource(root, rootEntry, workspaceRoot);
		}
		
		// update refering projects classpath excluding orignating project
		IEGLProject originatingProject = root.getEGLProject();
		if ((this.updateModelFlags & IPackageFragmentRoot.OTHER_REFERRING_PROJECTS_EGLPATH) != 0) {
			updateReferringProjectClasspaths(rootEntry.getPath(), originatingProject);
		}

		boolean isRename = this.destination.segment(0).equals(originatingProject.getElementName());
		boolean updateOriginating = (this.updateModelFlags & IPackageFragmentRoot.ORIGINATING_PROJECT_EGLPATH) != 0;
		boolean updateDestination = (this.updateModelFlags & IPackageFragmentRoot.DESTINATION_PROJECT_EGLPATH) != 0;

		// update originating classpath
		if (updateOriginating) {
			if (isRename && updateDestination) {
				renameEntryInClasspath(rootEntry.getPath(), originatingProject);
			} else {
				removeEntryFromClasspath(rootEntry.getPath(), originatingProject);
			}
		}
		
		// update destination classpath
		if (updateDestination) {
			if (!isRename || !updateOriginating) {
				addEntryToEGLPath(rootEntry, workspaceRoot);
			}  // else reference has been updated when updating originating project classpath
		}
	}
	protected void moveResource(
		IPackageFragmentRoot root,
		IEGLPathEntry rootEntry,
		final IWorkspaceRoot workspaceRoot)
		throws EGLModelException {
			
		final char[][] exclusionPatterns = ((EGLPathEntry)rootEntry).fullExclusionPatternChars();
		IResource rootResource = root.getResource();
		if (rootEntry.getEntryKind() != IEGLPathEntry.CPE_SOURCE || exclusionPatterns == null) {
			try {
				IResource destRes;
				if ((this.updateModelFlags & IPackageFragmentRoot.REPLACE) != 0
						&& (destRes = workspaceRoot.findMember(this.destination)) != null) {
					destRes.delete(this.updateResourceFlags, fMonitor);
				}
				rootResource.move(this.destination, this.updateResourceFlags, fMonitor);
			} catch (CoreException e) {
				throw new EGLModelException(e);
			}
		} else {
			final int sourceSegmentCount = rootEntry.getPath().segmentCount();
			final IFolder destFolder = workspaceRoot.getFolder(this.destination);
			final IPath[] nestedFolders = getNestedFolders(root);
			IResourceProxyVisitor visitor = new IResourceProxyVisitor() {
				public boolean visit(IResourceProxy proxy) throws CoreException {
					if (proxy.getType() == IResource.FOLDER) {
						IPath path = proxy.requestFullPath();
						if (prefixesOneOf(path, nestedFolders)) {
							if (equalsOneOf(path, nestedFolders)) {
								// nested source folder
								return false;
							} else {
								// folder containing nested source folder
								IFolder folder = destFolder.getFolder(path.removeFirstSegments(sourceSegmentCount));
								if ((updateModelFlags & IPackageFragmentRoot.REPLACE) != 0
										&& folder.exists()) {
									return true;
								}
								folder.create(updateResourceFlags, true, fMonitor);
								return true;
							}
						} else {
							// subtree doesn't contain any nested source folders
							IPath destPath = destination.append(path.removeFirstSegments(sourceSegmentCount));
							IResource destRes;
							if ((updateModelFlags & IPackageFragmentRoot.REPLACE) != 0
									&& (destRes = workspaceRoot.findMember(destPath)) != null) {
								destRes.delete(updateResourceFlags, fMonitor);
							}
							proxy.requestResource().move(destPath, updateResourceFlags, fMonitor);
							return false;
						}
					} else {
						IPath path = proxy.requestFullPath();
						IPath destPath = destination.append(path.removeFirstSegments(sourceSegmentCount));
						IResource destRes;
						if ((updateModelFlags & IPackageFragmentRoot.REPLACE) != 0
								&& (destRes = workspaceRoot.findMember(destPath)) != null) {
							destRes.delete(updateResourceFlags, fMonitor);
						}
						proxy.requestResource().move(destPath, updateResourceFlags, fMonitor);
						return false;
					}
				}
			};
			try {
				rootResource.accept(visitor, IResource.NONE);
			} catch (CoreException e) {
				throw new EGLModelException(e);
			}
		}
		this.setAttribute(HAS_MODIFIED_RESOURCE_ATTR, TRUE); 
	}
	/*
	 * Renames the classpath entries equal to the given path in all Java projects.
	 */
	protected void updateReferringProjectClasspaths(IPath rootPath, IEGLProject projectOfRoot) throws EGLModelException {
		IEGLModel model = this.getEGLModel();
		IEGLProject[] projects = model.getEGLProjects();
		for (int i = 0, length = projects.length; i < length; i++) {
			IEGLProject project = projects[i];
			if (project.equals(projectOfRoot)) continue;
			renameEntryInClasspath(rootPath, project);
		}
	}
	/*
	 * Removes the classpath entry equal to the given path from the given project's classpath.
	 */
	protected void removeEntryFromClasspath(IPath rootPath, IEGLProject project) throws EGLModelException {
		
		IEGLPathEntry[] classpath = project.getRawEGLPath();
		IEGLPathEntry[] newClasspath = null;
		int cpLength = classpath.length;
		int newCPIndex = -1;
		
		for (int i = 0; i < cpLength; i++) {
			IEGLPathEntry entry = classpath[i];
			if (rootPath.equals(entry.getPath())) {
				if (newClasspath == null) {
					newClasspath = new IEGLPathEntry[cpLength];
					System.arraycopy(classpath, 0, newClasspath, 0, i);
					newCPIndex = i;
				}
			} else if (newClasspath != null) {
				newClasspath[newCPIndex++] = entry;
			}
		}
		
		if (newClasspath != null) {
			if (newCPIndex < newClasspath.length) {
				System.arraycopy(newClasspath, 0, newClasspath = new IEGLPathEntry[newCPIndex], 0, newCPIndex);
			}
			project.setRawEGLPath(newClasspath, fMonitor);
		}
	}
}
