/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.driver;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.tinkerpop.gremlin.driver.Client;
import org.apache.tinkerpop.gremlin.driver.Cluster;
import org.apache.tinkerpop.gremlin.driver.Connection;
import org.apache.tinkerpop.gremlin.driver.ConnectionFactory;
import org.apache.tinkerpop.gremlin.driver.Host;
import org.apache.tinkerpop.gremlin.driver.ResultSet;
import org.apache.tinkerpop.gremlin.driver.Settings;
import org.apache.tinkerpop.gremlin.driver.exception.ConnectionException;
import org.apache.tinkerpop.gremlin.util.ExceptionHelper;
import org.apache.tinkerpop.gremlin.util.TimeUtil;
import org.apache.tinkerpop.gremlin.util.message.RequestMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class ConnectionPool {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionPool.class);
    public static final int MAX_POOL_SIZE = 128;
    private static final int CONNECTION_SETUP_TIME_DELTA = 25;
    public final Host host;
    private final Cluster cluster;
    private final Client client;
    private final List<Connection> connections;
    private final AtomicInteger open;
    private final Queue<Connection> availableConnections = new ConcurrentLinkedQueue<Connection>();
    private final Set<Connection> bin = new CopyOnWriteArraySet<Connection>();
    private final int maxPoolSize;
    private final String poolLabel;
    private final AtomicInteger scheduledForCreation = new AtomicInteger();
    private final AtomicReference<ConnectionResult> latestConnectionResult = new AtomicReference<ConnectionResult>(new ConnectionResult());
    private final AtomicReference<CompletableFuture<Void>> closeFuture = new AtomicReference();
    private final AtomicInteger waiter = new AtomicInteger();
    private final Lock waitLock = new ReentrantLock(true);
    private final Condition hasAvailableConnection = this.waitLock.newCondition();
    ConnectionFactory connectionFactory;

    public ConnectionPool(Host host, Client client) {
        this(host, client, Optional.empty());
    }

    public ConnectionPool(Host host, Client client, Optional<Integer> overrideMaxPoolSize) {
        this(host, client, overrideMaxPoolSize, new ConnectionFactory.DefaultConnectionFactory());
    }

    ConnectionPool(Host host, Client client, Optional<Integer> overrideMaxPoolSize, ConnectionFactory connectionFactory) {
        this.host = host;
        this.client = client;
        this.cluster = client.cluster;
        this.connectionFactory = connectionFactory;
        this.poolLabel = "Connection Pool {host=" + host + "}";
        Settings.ConnectionPoolSettings settings = this.settings();
        this.maxPoolSize = overrideMaxPoolSize.orElse(settings.maxSize);
        this.connections = new CopyOnWriteArrayList<Connection>();
        this.open = new AtomicInteger();
        try {
            CompletableFuture.runAsync(() -> {
                ConnectionResult result = new ConnectionResult();
                try {
                    Connection conn = connectionFactory.create(this);
                    this.connections.add(conn);
                    this.availableConnections.add(conn);
                    this.open.incrementAndGet();
                }
                catch (ConnectionException e) {
                    result.setFailureCause(e);
                    throw new CompletionException(e);
                }
                finally {
                    result.setTimeNow();
                    if (this.latestConnectionResult.get() == null || this.latestConnectionResult.get().getTime() < result.getTime()) {
                        this.latestConnectionResult.set(result);
                    }
                }
            }, this.cluster.connectionScheduler()).join();
        }
        catch (CancellationException ce) {
            logger.warn("Initialization of connection pool cancelled for {}", (Object)this.getPoolInfo(), (Object)ce);
            throw ce;
        }
        catch (CompletionException ce) {
            this.closeAsync();
            Throwable result = ce.getCause() == null ? ce : ce.getCause();
            throw new CompletionException("Could not initialize connection in pool. Closing the connection pool.", result);
        }
        logger.info("Opening connection pool on {}", (Object)host);
    }

    public Settings.ConnectionPoolSettings settings() {
        return this.cluster.connectionPoolSettings();
    }

    public Connection borrowConnection(long timeout, TimeUnit unit) throws TimeoutException, ConnectionException {
        logger.debug("Borrowing connection from pool on {} - timeout in {} {} with {}", new Object[]{this.host, timeout, unit, Thread.currentThread()});
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
        }
        Connection availableConnection = this.getAvailableConnection();
        if (availableConnection == null) {
            return this.waitForConnection(timeout, unit);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Borrowed {} on {} with {}", new Object[]{availableConnection.getConnectionInfo(), this.host, Thread.currentThread()});
        }
        return availableConnection;
    }

    public void returnConnection(Connection connection) throws ConnectionException {
        logger.debug("Attempting to return {} on {}", (Object)connection, (Object)this.host);
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
        }
        connection.isBorrowed().set(false);
        if (connection.isDead()) {
            logger.debug("Marking {} as dead", (Object)this.host);
            this.replaceConnection(connection);
        } else {
            if (this.bin.contains(connection)) {
                logger.debug("{} is already in the bin and it has no inflight requests so it is safe to close", (Object)connection);
                if (this.bin.remove(connection)) {
                    connection.closeAsync();
                }
                return;
            }
            int poolSize = this.connections.size();
            if (poolSize > this.maxPoolSize) {
                if (logger.isDebugEnabled()) {
                    logger.debug("destroy {}", (Object)connection.getConnectionInfo());
                }
                this.destroyConnection(connection);
            } else {
                logger.debug("Pool size is {} - returning connection to pool: {}", (Object)poolSize, (Object)connection);
                this.availableConnections.add(connection);
                this.announceAvailableConnection();
            }
        }
    }

    Client getClient() {
        return this.client;
    }

    Cluster getCluster() {
        return this.cluster;
    }

    public boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    public synchronized CompletableFuture<Void> closeAsync() {
        if (this.closeFuture.get() != null) {
            return this.closeFuture.get();
        }
        logger.info("Signalled closing of connection pool on {} with current connection size of {}", (Object)this.host, (Object)this.connections.size());
        this.announceAllAvailableConnection();
        CompletableFuture<Void> future = this.killAvailableConnections();
        this.closeFuture.set(future);
        return future;
    }

    int numConnectionsWaitingToCleanup() {
        return this.bin.size();
    }

    private CompletableFuture<Void> killAvailableConnections() {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>(this.connections.size() + this.bin.size());
        for (Connection connection : this.connections) {
            CompletableFuture<Void> future = connection.closeAsync();
            future.thenRun(this.open::decrementAndGet);
            futures.add(future);
        }
        for (Connection connection : this.bin) {
            futures.add(connection.closeAsync());
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    void replaceConnection(Connection connection) {
        logger.info("Replace {}", (Object)connection);
        if (connection.isBeingReplaced.getAndSet(true) || this.isClosed()) {
            return;
        }
        this.considerNewConnection();
        this.destroyConnection(connection);
    }

    private void considerNewConnection() {
        int inCreation;
        logger.debug("Considering new connection on {} where pool size is {}", (Object)this.host, (Object)this.connections.size());
        do {
            inCreation = this.scheduledForCreation.get();
            logger.debug("There are {} connections scheduled for creation on {}", (Object)inCreation, (Object)this.host);
            if (inCreation < 1) continue;
            return;
        } while (!this.scheduledForCreation.compareAndSet(inCreation, inCreation + 1));
        this.newConnection();
    }

    private void newConnection() {
        this.cluster.connectionScheduler().submit(() -> {
            this.scheduledForCreation.decrementAndGet();
            this.addConnectionIfUnderMaximum();
            return null;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean addConnectionIfUnderMaximum() {
        int opened;
        do {
            if ((opened = this.open.get()) < this.maxPoolSize) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened + 1));
        int openCountToActOn = opened;
        if (this.isClosed()) {
            this.open.decrementAndGet();
            return false;
        }
        ConnectionResult result = new ConnectionResult();
        try {
            Connection conn = this.connectionFactory.create(this);
            logger.debug("Created connection {}", (Object)conn.getConnectionInfo());
            this.connections.add(conn);
            this.availableConnections.add(conn);
        }
        catch (Exception ex) {
            this.open.decrementAndGet();
            logger.error(String.format("Connections[%s] were under maximum allowed[%s], but there was an error creating a new connection", openCountToActOn, this.maxPoolSize), (Throwable)ex);
            this.considerHostUnavailable();
            result.setFailureCause(ex);
            boolean bl = false;
            return bl;
        }
        finally {
            result.setTimeNow();
            if (this.latestConnectionResult.get().getTime() < result.getTime()) {
                this.latestConnectionResult.set(result);
            }
        }
        this.announceAllAvailableConnection();
        return true;
    }

    public void destroyConnection(Connection connection) {
        if (!this.bin.contains(connection) && !connection.isClosing()) {
            this.bin.add(connection);
            this.connections.remove(connection);
            this.availableConnections.remove(connection);
            this.open.decrementAndGet();
        }
        if ((connection.isDead() || !connection.isBorrowed().get()) && this.bin.remove(connection)) {
            CompletableFuture<Void> closeFuture = connection.closeAsync();
            closeFuture.whenComplete((v, t) -> logger.debug("Destroyed {}{}{}", new Object[]{connection.getConnectionInfo(), System.lineSeparator(), this.getPoolInfo()}));
        }
    }

    private Connection waitForConnection(long timeout, TimeUnit unit) throws TimeoutException, ConnectionException {
        long start = System.nanoTime();
        long remaining = timeout;
        long to = timeout;
        do {
            try {
                this.awaitAvailableConnection(remaining, unit);
            }
            catch (InterruptedException e) {
                logger.warn("Wait was interrupted", (Throwable)e);
                Thread.currentThread().interrupt();
                to = 0L;
            }
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
            }
            Connection conn = this.getAvailableConnection();
            if (conn != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Return {} on {} after waiting on {}", new Object[]{conn.getConnectionInfo(), this.host, Thread.currentThread()});
                }
                return conn;
            }
            remaining = to - TimeUtil.timeSince((long)start, (TimeUnit)unit);
            logger.debug("Continue to wait for connection on {} if {} > 0 on {}", new Object[]{this.host, remaining, Thread.currentThread()});
        } while (remaining > 0L);
        StringBuilder cause = new StringBuilder("Potential Cause: ");
        ConnectionResult res = this.latestConnectionResult.get();
        if (System.currentTimeMillis() - res.getTime() < (long)(this.cluster.getMaxWaitForConnection() + 25) && res.getFailureCause() != null) {
            cause.append(ExceptionHelper.getRootCause((Throwable)res.getFailureCause()).getMessage());
        } else if (this.open.get() >= this.maxPoolSize) {
            cause.append(String.format("Number of active requests (%s) exceeds pool size (%s). Consider increasing the value for maxConnectionPoolSize.", this.open.get(), this.maxPoolSize));
        } else {
            cause.setLength(0);
        }
        String timeoutErrorMessage = String.format("Timed-out (%s %s) waiting for connection on %s. %s%s%s", new Object[]{timeout, unit, this.host, cause, System.lineSeparator(), this.getPoolInfo()});
        logger.error(timeoutErrorMessage);
        TimeoutException timeoutException = new TimeoutException(timeoutErrorMessage);
        this.considerHostUnavailable();
        throw timeoutException;
    }

    public void considerHostUnavailable() {
        boolean maybeUnhealthy = this.connections.stream().allMatch(Connection::isDead);
        if (maybeUnhealthy) {
            this.host.tryReconnectingImmediately(this::tryReconnect);
            if (!this.host.isAvailable()) {
                this.connections.forEach(this::destroyConnection);
                this.cluster.loadBalancingStrategy().onUnavailable(this.host);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean tryReconnect(Host h) {
        logger.debug("Trying to re-establish connection on {}", (Object)h);
        Connection connection = null;
        try {
            connection = this.connectionFactory.create(this);
            RequestMessage ping = this.client.buildMessage(this.cluster.validationRequest()).create();
            CompletableFuture<ResultSet> f = new CompletableFuture<ResultSet>();
            connection.write(ping, f);
            f.get().all().get();
            this.cluster.loadBalancingStrategy().onAvailable(h);
            boolean bl = true;
            return bl;
        }
        catch (Exception ex) {
            logger.error(String.format("Failed reconnect attempt on %s%s%s", h, System.lineSeparator(), this.getPoolInfo()), (Throwable)ex);
            boolean bl = false;
            return bl;
        }
        finally {
            if (connection != null) {
                connection.closeAsync();
            }
        }
    }

    private void announceAvailableConnection() {
        logger.debug("Announce connection available on {}", (Object)this.host);
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signal();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    private Connection getAvailableConnection() {
        Connection available = null;
        Connection head = this.availableConnections.poll();
        while (head != null) {
            if (!head.isDead() && !head.isBorrowed().get() && head.isBorrowed().compareAndSet(false, true)) {
                available = head;
                break;
            }
            head = this.availableConnections.poll();
        }
        if (available == null && this.connections.size() < this.maxPoolSize) {
            this.considerNewConnection();
        }
        return available;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void awaitAvailableConnection(long timeout, TimeUnit unit) throws InterruptedException {
        logger.debug("Wait {} {} for an available connection on {} with {}", new Object[]{timeout, unit, this.host, Thread.currentThread()});
        this.waitLock.lock();
        this.waiter.incrementAndGet();
        try {
            this.hasAvailableConnection.await(timeout, unit);
        }
        finally {
            this.waiter.decrementAndGet();
            this.waitLock.unlock();
        }
    }

    private void announceAllAvailableConnection() {
        logger.debug("Announce to all connection available on {}", (Object)this.host);
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signalAll();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    public String getPoolInfo() {
        return this.getPoolInfo(null);
    }

    public String getPoolInfo(Connection connectionToCallout) {
        StringBuilder sb = new StringBuilder("ConnectionPool (");
        sb.append(this.host.toString());
        sb.append(")");
        if (this.connections.isEmpty()) {
            sb.append("- no connections in pool");
        } else {
            int connectionCount = this.connections.size();
            sb.append(System.lineSeparator());
            sb.append(String.format("Connection Pool Status (size=%s available=%s max=%s toCreate=%s bin=%s waiter=%s)", connectionCount, this.availableConnections.size(), this.maxPoolSize, this.scheduledForCreation.get(), this.bin.size(), this.waiter.get()));
            sb.append(System.lineSeparator());
            this.appendConnections(sb, connectionToCallout, (CopyOnWriteArrayList)this.connections);
            sb.append(System.lineSeparator());
            sb.append("-- bin --");
            sb.append(System.lineSeparator());
            this.appendConnections(sb, connectionToCallout, new CopyOnWriteArrayList<Connection>(this.bin));
        }
        return sb.toString().trim();
    }

    private void appendConnections(StringBuilder sb, Connection connectionToCallout, CopyOnWriteArrayList<Connection> connections) {
        Iterator<Connection> it = connections.iterator();
        while (it.hasNext()) {
            Connection c = it.next();
            if (c.equals(connectionToCallout)) {
                sb.append("==> ");
            } else {
                sb.append("> ");
            }
            sb.append(c.getConnectionInfo(false));
            if (!it.hasNext()) continue;
            sb.append(System.lineSeparator());
        }
    }

    public String toString() {
        return this.poolLabel;
    }

    public static class ConnectionResult {
        private long timeOfConnectionAttempt;
        private Throwable failureCause;

        public Throwable getFailureCause() {
            return this.failureCause;
        }

        public long getTime() {
            return this.timeOfConnectionAttempt;
        }

        public void setFailureCause(Throwable cause) {
            this.failureCause = cause;
        }

        public void setTimeNow() {
            this.timeOfConnectionAttempt = System.currentTimeMillis();
        }
    }
}

