/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.tinkerpop.gremlin.process.traversal.strategy.optimization;

import org.apache.tinkerpop.gremlin.process.computer.traversal.step.map.TraversalVertexProgramStep;
import org.apache.tinkerpop.gremlin.process.traversal.Translator;
import org.apache.tinkerpop.gremlin.process.traversal.Traversal;
import org.apache.tinkerpop.gremlin.process.traversal.TraversalStrategy;
import org.apache.tinkerpop.gremlin.process.traversal.dsl.graph.__;
import org.apache.tinkerpop.gremlin.process.traversal.step.GValue;
import org.apache.tinkerpop.gremlin.process.traversal.step.map.OrderGlobalStep;
import org.apache.tinkerpop.gremlin.process.traversal.strategy.GValueManagerVerifier;
import org.apache.tinkerpop.gremlin.process.traversal.translator.GroovyTranslator;
import org.apache.tinkerpop.gremlin.process.traversal.util.EmptyTraversal;
import org.apache.tinkerpop.gremlin.process.traversal.util.TraversalHelper;
import org.junit.Test;
import org.junit.experimental.runners.Enclosed;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;

import static org.junit.Assert.assertEquals;

/**
 * @author Marko A. Rodriguez (http://markorodriguez.com)
 */
@RunWith(Enclosed.class)
public class OrderLimitStrategyTest {
    private static final Translator.ScriptTranslator translator = GroovyTranslator.of("__");

    @RunWith(Parameterized.class)
    public static class StandardTest {
        @Parameterized.Parameter(value = 0)
        public Traversal.Admin traversal;

        @Parameterized.Parameter(value = 1)
        public long limit;


        @Test
        public void doTest() {
            final String repr = translator.translate(traversal.getBytecode()).getScript();
            traversal.asAdmin().setParent(new TraversalVertexProgramStep(EmptyTraversal.instance(), EmptyTraversal.instance())); // trick it
            GValueManagerVerifier.verify(traversal, OrderLimitStrategy.instance()).
                    afterApplying().variablesArePreserved();
            assertEquals(repr, limit, TraversalHelper.getFirstStepOfAssignableClass(OrderGlobalStep.class, traversal.asAdmin()).get().getLimit());
        }

        @Parameterized.Parameters(name = "{0}")
        public static Iterable<Object[]> generateTestParameters() {
            return Arrays.asList(new Object[][]{
                    {__.order().limit(1), 1l},
                    {__.out().order().range(7, 15), 15l},
                    {__.order().select("a").limit(7), 7l},
                    {__.order().out().limit(10), Long.MAX_VALUE}});
        }
    }

    /**
     * Tests that GValueManager correctly preserves GValue parameters when OrderLimitStrategy is applied.
     */
    @RunWith(Parameterized.class)
    public static class GValueTest {
        @Parameterized.Parameter(value = 0)
        public Traversal.Admin<?, ?> traversal;

        @Parameterized.Parameter(value = 1)
        public String[] expectedNames;

        @Parameterized.Parameter(value = 2)
        public String[] expectedUnpinnedNames;

        @Parameterized.Parameter(value = 3)
        public Collection<TraversalStrategy> additionalStrategies;

        @Parameterized.Parameters(name = "{0}")
        public static Iterable<Object[]> generateTestParameters() {
            return Arrays.asList(new Object[][]{
                    {
                        __.order().limit(GValue.ofLong("x", 5L)).asAdmin(),
                        new String[]{"x"},
                        new String[]{},
                        Collections.emptyList(),
                    },
                    {
                        __.out().order().range(GValue.ofLong("low", 3L), GValue.ofLong("high", 10L)).asAdmin(),
                        new String[]{"low", "high"},
                        new String[]{"low"},
                        Collections.emptyList(),
                    },
                    {
                        __.order().select("a").limit(GValue.ofLong("x", 7L)).asAdmin(),
                        new String[]{"x"},
                        new String[]{},
                        Collections.emptyList(),
                    }
            });
        }

        @Test
        public void shouldPreserveGValueParameters() {
            traversal.asAdmin().setParent(new TraversalVertexProgramStep(EmptyTraversal.instance(), EmptyTraversal.instance())); // trick it
            GValueManagerVerifier.verify(traversal, OrderLimitStrategy.instance()).
                    beforeApplying().
                        hasVariables(expectedNames).
                    afterApplying().
                        hasUnpinnedVariables(expectedUnpinnedNames);
        }
    }
}
