// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions;

import org.apache.doris.nereids.annotation.Developing;
import org.apache.doris.nereids.exceptions.UnboundException;
import org.apache.doris.nereids.trees.TreeNode;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.types.DataType;

import com.google.common.collect.ImmutableList;

import java.util.List;

/**
 * ExpressionTrait.
 */
public interface ExpressionTrait extends TreeNode<Expression> {

    boolean nullable();

    default boolean notNullable() {
        return !nullable();
    }

    // check legality before do type coercion.
    // maybe we should merge checkInputDataTypes and checkLegality later.
    @Developing
    default void checkLegalityBeforeTypeCoercion() {}

    @Developing
    default void checkLegalityAfterRewrite() {}

    default List<Expression> getArguments() {
        return children();
    }

    default Expression getArgument(int index) {
        return child(index);
    }

    default List<DataType> getArgumentsTypes() {
        return getArguments()
                .stream()
                .map(Expression::getDataType)
                .collect(ImmutableList.toImmutableList());
    }

    default DataType getArgumentType(int index) {
        return child(index).getDataType();
    }

    default DataType getDataType() throws UnboundException {
        throw new UnboundException(toSql() + ".getDataType()");
    }

    default String toSql() throws UnboundException {
        throw new UnboundException("sql");
    }

    /**
     * foldable() mainly use in fold expression. Udf and UniqueFunction are not foldable.
     * But if want to check an expression contains non-idempotent, such as `rand()`, `uuid()`, etc.,
     * you should use Expression::containsUniqueFunction instead.
     */
    default boolean foldable() {
        return true;
    }

    /**
     * Identify the expression itself is deterministic or not, default true
     */
    default boolean isDeterministic() {
        return true;
    }

    /**
     * Identify the expression is containing non-deterministic expr or not
     */
    default boolean containsNondeterministic() {
        return anyMatch(expr -> !((ExpressionTrait) expr).isDeterministic());
    }
}
