/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.apache.causeway.testdomain.interact;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import jakarta.inject.Named;
import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlType;

import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.TestPropertySource;

import org.apache.causeway.applib.annotation.Action;
import org.apache.causeway.applib.annotation.DomainObject;
import org.apache.causeway.applib.annotation.Nature;
import org.apache.causeway.applib.annotation.Where;
import org.apache.causeway.applib.services.wrapper.control.AsyncControl;
import org.apache.causeway.core.config.presets.CausewayPresets;
import org.apache.causeway.core.metamodel.object.ManagedObject;
import org.apache.causeway.testdomain.conf.Configuration_headless;
import org.apache.causeway.testdomain.util.interaction.InteractionTestAbstract;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

@SpringBootTest(
        classes = {
                Configuration_headless.class,
                CommandArgumentTest.CommandArgDemo.class,
        },
        properties = {
        })
@TestPropertySource({
    CausewayPresets.SilenceMetaModel,
    CausewayPresets.SilenceProgrammingModel
})
class CommandArgumentTest extends InteractionTestAbstract {

    @XmlRootElement(name = "root")
    @XmlType
    @XmlAccessorType(XmlAccessType.FIELD)
    @Named("regressiontests.CommandResult")
    @DomainObject(nature=Nature.VIEW_MODEL)
    @NoArgsConstructor
    @AllArgsConstructor(staticName = "of")
    public static class CommandResult {

        @Getter @Setter
        private String resultAsString;

    }

    @XmlRootElement(name = "root")
    @XmlType
    @XmlAccessorType(XmlAccessType.FIELD)
    @Named("regressiontests.CommandArgDemo")
    @DomainObject(nature=Nature.VIEW_MODEL)
    public static class CommandArgDemo {

        @Action
        public CommandResult list(final List<Long> someIds){
            var stringified = ""+someIds;
            assertEquals("[1, 2, 3]", stringified);
            return CommandResult.of(stringified);
        }

    }

    @Test
    void listParam_shouldAllowInvocation() {

        var actionInteraction = startActionInteractionOn(CommandArgDemo.class, "list", Where.OBJECT_FORMS)
        .checkVisibility()
        .checkUsability();

        var pendingArgs = actionInteraction.startParameterNegotiation().get();

        pendingArgs.setParamValue(0, objectManager.adapt(List.of(1L, 2L, 3L)));

        var resultOrVeto = actionInteraction.invokeWith(pendingArgs);
        assertTrue(resultOrVeto.isSuccess());

        var stringified = resultOrVeto.getSuccess()
                .map(ManagedObject::getPojo)
                .map(CommandResult.class::cast)
                .map(CommandResult::getResultAsString)
                .orElse(null);
        assertEquals("[1, 2, 3]", stringified);
    }

    @Test
    void listParam_shouldAllowAsyncInvocation() throws InterruptedException, ExecutionException, TimeoutException {

        var commandArgDemo = new CommandArgDemo();

        var control = AsyncControl.defaults();

        var stringified = wrapperFactory.asyncWrap(commandArgDemo, control)
            .applyAsync(commandResult->commandResult.list(List.of(1L, 2L, 3L)))
            .tryGet(3L, TimeUnit.DAYS) // wait till done
            .valueAsNonNullElseFail()
            .getResultAsString();

        assertEquals("[1, 2, 3]", stringified);
    }

}
