/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.test.infra.kafka.services;

import org.apache.camel.spi.annotations.InfraService;
import org.apache.camel.test.infra.common.TestUtils;
import org.apache.camel.test.infra.common.services.ContainerEnvironmentUtil;
import org.apache.camel.test.infra.common.services.ContainerService;
import org.apache.camel.test.infra.kafka.common.KafkaProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testcontainers.containers.Network;
import org.testcontainers.redpanda.RedpandaContainer;

@InfraService(service = KafkaInfraService.class,
              description = "Apache Kafka, Distributed event streaming platform",
              serviceAlias = "kafka", serviceImplementationAlias = "redpanda")
public class RedpandaInfraService implements KafkaInfraService, ContainerService<RedpandaContainer> {
    private static final Logger LOG = LoggerFactory.getLogger(RedpandaInfraService.class);

    private final RedpandaContainer redpandaContainer;

    public RedpandaInfraService() {
        this("redpanda-" + TestUtils.randomWithRange(1, 100));
    }

    public RedpandaInfraService(String redpandaInstanceName) {
        Network network = Network.newNetwork();

        redpandaContainer = initRedpandaContainer(network, redpandaInstanceName);
        String name = ContainerEnvironmentUtil.containerName(this.getClass());
        if (name != null) {
            redpandaContainer.withCreateContainerCmdModifier(cmd -> cmd.withName(name));
        }
    }

    public RedpandaInfraService(RedpandaContainer redpandaContainer) {
        this.redpandaContainer = redpandaContainer;
    }

    protected RedpandaContainer initRedpandaContainer(Network network, String instanceName) {
        class TestInfraRedpandaContainer extends RedpandaTransactionsEnabledContainer {
            public TestInfraRedpandaContainer(boolean fixedPort) {
                super(RedpandaTransactionsEnabledContainer.REDPANDA_CONTAINER);

                ContainerEnvironmentUtil.configurePort(this, fixedPort, 9092);
            }
        }

        return new TestInfraRedpandaContainer(ContainerEnvironmentUtil.isFixedPort(this.getClass()));
    }

    protected Integer getKafkaPort() {
        return redpandaContainer.getMappedPort(RedpandaTransactionsEnabledContainer.REDPANDA_PORT);
    }

    @Override
    public String getBootstrapServers() {
        return redpandaContainer.getHost() + ":" + getKafkaPort();
    }

    @Override
    public String brokers() {
        return getBootstrapServers();
    }

    @Override
    public void registerProperties() {
        System.setProperty(KafkaProperties.KAFKA_BOOTSTRAP_SERVERS, getBootstrapServers());
    }

    @Override
    public void initialize() {
        redpandaContainer.start();

        registerProperties();
        LOG.info("Redpanda bootstrap server running at address {}", getBootstrapServers());
    }

    private boolean stopped() {
        return !redpandaContainer.isRunning();
    }

    @Override
    public void shutdown() {
        LOG.info("Stopping Redpanda container");
        redpandaContainer.stop();
    }

    @Override
    public RedpandaContainer getContainer() {
        return redpandaContainer;
    }
}
