#' @title Adjust P-values in a `dana` object
#'
#' @description
#' Applies multiple testing correction to P-values from differential analysis results
#' returned by the `dana()` function. Supports multiple adjustment methods and both
#' coefficient and likelihood ratio test (LRT) P-values.
#'
#' @details
#' Available adjustment methods include: `"BH"`, `"bonferroni"`, `"BY"`, `"fdr"`,
#' `"hochberg"`, `"holm"`, `"hommel"`, `"IHW"`, and `"storey"`.
#'
#' @param dana_obj A `dana` class object returned by the `dana()` function.
#' @param padj_by Character string. Whether P-value adjustment should be done globally
#'  across all coefficients (`"all"`) or separately for each coefficient term (`"terms"`).
#' @param padj_method Character vector of one or more methods for adjusting P-values from
#'  coefficient tests. Defaults to `"BH"`.
#' @param padj_method_LRT Character vector of one or more methods for adjusting P-values from
#'  LRT tests. Defaults to `"BH"`. P-values from LRT tests will always be adjusted
#'  independently for each LRT term.
#' @param verbose Logical. Whether to print informative messages. Defaults to `TRUE`.
#' @param ... Additional arguments passed to `IHW::ihw()` or `qvalue::qvalue()`.
#'
#' @return A modified `dana` object with new columns in the `$fit` and `$lrt` data frames
#'  for each adjusted P-value method applied (e.g. `padj_BH`, `padj_storey_group`).
#'
#' @seealso
#' * [dana()] for fitting differential analysis models on omics datasets.
#' * [IHW::ihw()] for inverted hypothesis weighting method details.
#' * [qvalue::qvalue()] for qvalue method details.
#'
#' @examples
#' set.seed(123)
#' mock_X <- matrix(rnorm(20 * 5), nrow = 20)
#' colnames(mock_X) <- paste0("feat_", seq_len(5))
#' rownames(mock_X) <- paste0("sample_", seq_len(20))
#'
#' sample_data <- data.frame(
#'   sample_id = rownames(mock_X),
#'   group = factor(rep(c("A", "B"), each = 10)),
#'   time = factor(rep(c("T1", "T2"), times = 10)),
#'   subject_id = factor(rep(seq_len(10), each = 2)),
#'   stringsAsFactors = FALSE
#' )
#' rownames(sample_data) <- sample_data$sample_id
#'
#' fit_df <- data.frame(
#'   feat_id = rep(colnames(mock_X), each = 2),
#'   Coefficient = rep(c("(Intercept)", "groupB"), 5),
#'   Estimate = rnorm(10),
#'   `Pr(>|t|)` = runif(10),
#'   stringsAsFactors = FALSE
#' )
#'
#' # Mock dana object
#' dana_obj <- list(
#'   X = mock_X,
#'   sdata = sample_data,
#'   formula_rhs = ~ group,
#'   fit = fit_df,
#'   lrt = data.frame(),
#'   ranef = data.frame()
#' )
#' class(dana_obj) <- "dana"
#'
#' # Add adjusted P-values
#' dana_obj <- dana_obj |>
#'   adjust_pval(dana_obj,
#'               padj_method = c("BH", "bonferroni"),
#'               padj_method_LRT = NULL,
#'               padj_by = "terms",
#'               verbose = FALSE)
#'
#' @export
adjust_pval <- function(dana_obj, padj_by = c("all", "terms"),
                        padj_method = c("BH", "bonferroni", "BY",
                                        "fdr", "hochberg", "holm",
                                        "hommel", "IHW", "storey"),
                        padj_method_LRT = c("BH", "bonferroni", "BY",
                                            "fdr", "hochberg", "holm",
                                            "hommel", "IHW", "storey"),
                        verbose = TRUE, ...) {
  # Match or set arguments
  padj_by <- match.arg(padj_by)
  padj_method <- match.arg(padj_method, several.ok = TRUE)
  padj_method_LRT <- match.arg(padj_method_LRT, several.ok = TRUE)

  # Check parameters
  if (!inherits(dana_obj, "dana")) {
    stop("'dana_obj' must be a 'dana' class object generated by 'readyomics::dana()' function.")
  }

  # Set adjustment method
  adj_method <- function(i, p_val, ...) {
    if (i == "IHW") {
      ihw_obj <- IHW::ihw(p_val, ...)
      adj_vals <- IHW::adj_pvalues(ihw_obj)
    } else if (i == "storey") {
      adj_vals <- try(qvalue::qvalue(p_val, ...)$qvalues)
      if (inherits(adj_vals, "try-error")) {
        warning("qvalue failed; falling back to BH adjustment.\n")
        adj_vals <- stats::p.adjust(p_val, method = "BH")
      }
    } else {
      adj_vals <- stats::p.adjust(p_val, method = i)
    }
    return(adj_vals)
  }

  # Adjust P-values in fit result
  p_nominal <- grep("Pr", colnames(dana_obj$fit), value = TRUE)
  if (padj_by == "all") {
    p_include <- !(dana_obj$fit[["Coefficient"]] %in% c("(Intercept)")) &
                 !is.na(dana_obj$fit[[p_nominal]])
    p_nominal_include <- dana_obj$fit[[p_nominal]][p_include]
    for (i in padj_method) {
      padj_label <- paste("padj", i, sep = "_")
      dana_obj$fit[[padj_label]] <- NA
      dana_obj$fit[[padj_label]][p_include] <- adj_method(i, p_nominal_include, ...)
    }
  } else {
    fit_terms <- unique(dana_obj$fit[["Coefficient"]])
    fit_terms <- fit_terms[fit_terms != "(Intercept)"]
    for (t in fit_terms) {
      p_include <- (dana_obj$fit[["Coefficient"]] == t) &
                   !is.na(dana_obj$fit[[p_nominal]])
      p_nominal_include <- dana_obj$fit[[p_nominal]][p_include]
      for (i in padj_method) {
        padj_label <- paste("padj", i, t, sep = "_")
        dana_obj$fit[[padj_label]] <- NA
        dana_obj$fit[[padj_label]][p_include] <- adj_method(i, p_nominal_include, ...)
      }
    }
  }

  # Adjust P values in lrt result
  if (nrow(dana_obj$lrt) == 0) {
    if (verbose) message("No LRT tests detected, 'padj_method_LRT' will be ignored.\n")
  } else {
    p_nominal_lrt <- grep("Pr", colnames(dana_obj$lrt), value = TRUE)
    lrt_terms <- unique(dana_obj$lrt[["term"]])
    for (l in lrt_terms) {
      p_include <- (dana_obj$lrt[["term"]] == l) &
                   !is.na(dana_obj$lrt[[p_nominal_lrt]])
      p_nominal_lrt_include <- dana_obj$lrt[[p_nominal_lrt]][p_include]
      for (i in padj_method_LRT) {
        padj_label <- paste("padj", i, l, sep = "_")
        dana_obj$lrt[[padj_label]] <- NA
        dana_obj$lrt[[padj_label]][p_include] <- adj_method(i, p_nominal_lrt_include, ...)
      }
    }
    # Add LRT P-values to dana_obj$fit table (only for fixed effects)
    lrt_fixed_terms <- lrt_terms[!grepl("\\|", lrt_terms)]
    if (length(lrt_fixed_terms) != 0) {
      for (t in lrt_fixed_terms) {
        term_match <- stringr::str_split_1(t, pattern = ":")
        term_n <- length(term_match)
        for (m in padj_method_LRT) {
          padj_LRT <- paste("padj", m, t, sep = "_")
          padj_LRT_name <- paste(padj_LRT, "LRT", sep = "_")
          dana_obj$fit[[padj_LRT_name]] <- sapply(seq_len(nrow(dana_obj$fit)),
                                                  function(x) {
                                                    feat_id <- dana_obj$fit$feat_id[x]
                                                    coeff_id <- dana_obj$fit$Coefficient[x]
                                                    if (term_n == 1) {
                                                      # Main effect LRT P value
                                                      if (!grepl(":", coeff_id) && grepl(term_match[1], coeff_id)) {
                                                        dana_obj$lrt[[padj_LRT]][dana_obj$lrt$feat_id == feat_id &
                                                                                   dana_obj$lrt$term == term_match[1] &
                                                                                   !is.na(dana_obj$lrt[[padj_LRT]])]
                                                      } else if (grepl(":", coeff_id) && grepl(term_match[1], coeff_id)) {
                                                        # Add LRT P values from associated interaction terms if any
                                                        coeff_match <- stringr::str_split_1(coeff_id, pattern = ":")
                                                        term_candidates <- unique(dana_obj$lrt$term[grepl(term_match[1], dana_obj$lrt$term) &
                                                                                                      grepl(":", dana_obj$lrt$term)])
                                                        term_final <- character()
                                                        for (i in term_candidates) {
                                                          term_second <- stringr::str_split_1(i, pattern = ":")[2]
                                                          if (grepl(term_second, coeff_match[2])) {
                                                            term_final <- append(term_final, i)
                                                          }
                                                        }
                                                        if (length(term_final) == 1) {
                                                          dana_obj$lrt[[padj_LRT]][dana_obj$lrt$feat_id == feat_id &
                                                                                     dana_obj$lrt$term == term_final]
                                                        } else {
                                                          NA
                                                        }
                                                      } else {
                                                        NA
                                                      }
                                                    } else if (term_n == 2) {
                                                      # Interaction term LRT P value
                                                      if (grepl(":", coeff_id) && grepl(paste0(term_match[1], ".*", term_match[2]), coeff_id)) {
                                                        dana_obj$lrt[[padj_LRT]][dana_obj$lrt$feat_id == feat_id &
                                                                                   dana_obj$lrt$term == t &
                                                                                   !is.na(dana_obj$lrt[[padj_LRT]])]
                                                      } else {
                                                        NA
                                                      }
                                                    } else {
                                                      NA
                                                    }
                                                  })
        }
      }
    }
  }

  return(dana_obj)
}

#' @title Add taxonomic information to `dana` object
#'
#' @description
#' Appends features taxonomy to the `dana` object tables.
#'
#' @details
#' - If `taxa_rank = "asv"`, a `taxon_name` is constructed by pasting the ASV ID
#'   to the `species` (if available) or `genus` name.
#' - For other ranks, `taxon_name` is taken directly from the corresponding column
#'   in `taxa_table`.
#' - All higher-level taxonomy ranks available in `taxa_table` are also appended.
#'
#' @param dana_obj A `dana` object returned by `dana()`.
#' @param taxa_table A taxonomy table `data.frame` with taxonomy ranks in columns
#'  and row names corresponding to `feat_id`s in `dana` object.
#' @param taxa_rank A character string specifying the taxonomy level of input features.
#'  Accepts one of: `"asv"`, `"substrain"`, `"strain"`, `"species"`, `"genus"`,
#'   `"family"`, `"order"`, `"class"`, `"phylum"`, or `"domain"`.
#'
#' @return
#' A modified version of `dana_obj`, with taxonomy information added to relevant tables.
#'
#' @seealso [dana()] for fitting differential analysis models on omics datasets.
#'
#' @examples
#' set.seed(123)
#' mock_X <- matrix(rnorm(20 * 5), nrow = 20)
#' colnames(mock_X) <- paste0("feat_", seq_len(5))
#' rownames(mock_X) <- paste0("sample_", seq_len(20))
#'
#' mock_taxa <- data.frame(
#'   Domain = rep("Bacteria", 5),
#'   Phylum = c("Firmicutes", "Bacteroidota", "Proteobacteria",
#'              "Actinobacteriota", "Firmicutes"),
#'   Class = c("Bacilli", "Bacteroidia", "Gammaproteobacteria",
#'             "Actinobacteria", "Clostridia"),
#'   Order = c("Lactobacillales", "Bacteroidales", "Enterobacterales",
#'             "Bifidobacteriales", "Clostridiales"),
#'   Family = c("Lactobacillaceae", "Bacteroidaceae", "Enterobacteriaceae",
#'              "Bifidobacteriaceae", "Clostridiaceae"),
#'   Genus = c("Lactobacillus", "Bacteroides", "Escherichia",
#'             "Bifidobacterium", "Clostridium"),
#'   Species = c("acidophilus", "fragilis", "coli", "longum", "butyricum"),
#'   row.names = paste0("feat_", seq_len(5)),
#'   stringsAsFactors = FALSE
#' )
#'
#' sample_data <- data.frame(
#'   sample_id = rownames(mock_X),
#'   group = factor(rep(c("A", "B"), each = 10)),
#'   time = factor(rep(c("T1", "T2"), times = 10)),
#'   subject_id = factor(rep(seq_len(10), each = 2)),
#'   stringsAsFactors = FALSE
#' )
#' rownames(sample_data) <- sample_data$sample_id
#'
#' fit_df <- data.frame(
#'   feat_id = rep(colnames(mock_X), each = 2),
#'   Coefficient = rep(c("(Intercept)", "groupB"), 5),
#'   Estimate = rnorm(10),
#'   `Pr(>|t|)` = runif(10),
#'   padj = runif(10),
#'   stringsAsFactors = FALSE
#' )
#'
#' # Mock dana object
#' dana_obj <- list(
#'   X = mock_X,
#'   sdata = sample_data,
#'   formula_rhs = ~ group,
#'   fit = fit_df,
#'   lrt = data.frame(),
#'   ranef = data.frame()
#' )
#' class(dana_obj) <- "dana"
#'
#' # Add taxonomy
#' dana_obj <- dana_obj |>
#'   add_taxa(mock_taxa, taxa_rank = "genus")
#'
#' @export
add_taxa <- function(dana_obj, taxa_table,
                     taxa_rank = c("asv", "substrain", "strain", "species",
                                   "genus", "family", "order", "class",
                                   "phylum", "domain")) {
  # Match or set arguments
  taxa_rank <- match.arg(taxa_rank)
  taxa_df <- taxa_table
  colnames(taxa_df) <- tolower(colnames(taxa_df))
  rank_order <- c("substrain", "strain", "species", "genus",
                  "family", "order", "class", "phylum", "domain")

  # Check paramaters
  if (taxa_rank != "asv" && !taxa_rank %in% colnames(taxa_df)) {
    stop("Cannot find provided 'taxa_rank': ", taxa_rank, " in 'taxa_table'.")
  }

  rank_match <- rank_order[which(rank_order %in% colnames(taxa_df))]
  if (length(rank_match) == 0) {
    stop("Taxon ranks in 'taxa_df' are not any of: ", rank_order)
  }

  update_df <- function(df) {
    idx <- match(df$feat_id, rownames(taxa_df))
    if (anyNA(idx)) {
      stop("Some 'feat_id's in dana object were not found in 'taxa_table': ",
           paste(utils::head(df$feat_id[is.na(idx)]), collapse = ", "))
    }

    df$taxon_rank <- taxa_rank

    if (taxa_rank == "asv") {
      if ("species" %in% colnames(taxa_df)) {
        df$taxon_name <- paste(taxa_df$species[idx], df$feat_id, sep = "_")
        start_rank <- which(rank_match == "species")
      } else {
        df$taxon_name <- paste(taxa_df$genus[idx], df$feat_id, sep = "_")
        start_rank <- which(rank_match == "genus")
      }
    } else {
      df$taxon_name <- taxa_df[[taxa_rank]][idx]
      start_rank <- which(rank_match == taxa_rank) + 1
    }

    # Add higher rank labels
    if (start_rank < length(rank_match)) {
      for (a in start_rank:length(rank_match)) {
        df[[rank_match[a]]] <- taxa_df[[rank_match[a]]][idx]
      }
    }
    df
  }

  # Update dana elements if they are non-empty
  dana_obj[c("fit", "lrt", "ranef")] <-
    purrr::modify_if(dana_obj[c("fit", "lrt", "ranef")], ~ nrow(.) > 0, update_df)

  return(dana_obj)
}

#' @title Append feature names to a `dana` object
#'
#' @description
#' Adds a `feat_name` column to the `dana` object to map `feat_id` to original labels.
#'
#' @param dana_obj A `dana` object returned by `dana()`.
#' @param feat_names A data frame mapping `feat_id` to `feat_name`.
#'  Must contain columns `"feat_id"` and `"feat_name"`.
#'
#' @return
#' A modified version of `dana_obj`, with a `feat_name` column added to applicable components.
#'
#' @seealso [dana()] for fitting differential analysis models on omics datasets.
#'
#' @examples
#' set.seed(123)
#' mock_X <- matrix(rnorm(20 * 5), nrow = 20)
#' colnames(mock_X) <- paste0("feat_", seq_len(5))
#' rownames(mock_X) <- paste0("sample_", seq_len(20))
#'
#' mock_names <- data.frame(
#'   feat_id = paste0("feat_", seq_len(5)),
#'   feat_name = c(
#'     "Glucose",
#'     "Lactic acid",
#'     "Citric acid",
#'     "Palmitic acid",
#'     "Cholesterol"
#'   ),
#'   stringsAsFactors = FALSE
#' )
#'
#' sample_data <- data.frame(
#'   sample_id = rownames(mock_X),
#'   group = factor(rep(c("A", "B"), each = 10)),
#'   time = factor(rep(c("T1", "T2"), times = 10)),
#'   subject_id = factor(rep(seq_len(10), each = 2)),
#'   stringsAsFactors = FALSE
#' )
#' rownames(sample_data) <- sample_data$sample_id
#'
#' fit_df <- data.frame(
#'   feat_id = rep(colnames(mock_X), each = 2),
#'   Coefficient = rep(c("(Intercept)", "groupB"), 5),
#'   Estimate = rnorm(10),
#'   `Pr(>|t|)` = runif(10),
#'   padj = runif(10),
#'   stringsAsFactors = FALSE
#' )
#'
#' # Mock dana object
#' dana_obj <- list(
#'   X = mock_X,
#'   sdata = sample_data,
#'   formula_rhs = ~ group,
#'   fit = fit_df,
#'   lrt = data.frame(),
#'   ranef = data.frame()
#' )
#' class(dana_obj) <- "dana"
#'
#' # Add fearure labels
#' dana_obj <- dana_obj |>
#'   add_feat_name(mock_names)
#'
#' @export
add_feat_name <- function(dana_obj, feat_names) {
  # Check parameters
  if (!all(c("feat_id", "feat_name") %in% colnames(feat_names))) {
    stop("Columns 'feat_id' and 'feat_name' were not found in 'feat_names' table.")
  }
  dana_obj[c("fit", "lrt", "ranef")] <-
    purrr::modify_if(dana_obj[c("fit", "lrt", "ranef")], ~ nrow(.) > 0, \(df) {
      idx <- match(df$feat_id, feat_names$feat_id)

      if (anyNA(idx)) {
        stop("Some 'feat_id' values in dana object not found in 'feat_names':\n",
             paste(utils::head(df$feat_id[is.na(idx)]), collapse = ", "))
      }

      df$feat_name <- feat_names$feat_name[idx]
      df
    })
  return(dana_obj)
}

#' @name ready_plot_helpers
#'
#' @title Internal plotting helpers
#'
#' @description
#' Internal helper functions used by `ready_plots()` for plotting various visualizations,
#' including volcano plots, heatmaps, dot plots of model coefficients, and feature plots.
#'
#' These functions are not exported and are meant for internal use only.
#' They all share a consistent visual style using `ready_theme()`.
#'
#' @importFrom rlang .data
#'
#' @keywords internal
#'
#' @noRd

#' @description Volcano plot of feature-level effect estimates vs adjusted P-values.
#' @param p_table A data frame with `Estimate`, `Coefficient`, and `feat_id`.
#' @param padj_colname Name of the column containing adjusted P-values.
#' @param alpha Significance threshold.
#' @param feat_label Column containing desired feature labels.
#' @param ... Additional arguments passed to `ready_theme()`.
plot_volcano <- function(p_table, padj_colname, alpha, feat_label, ...) {
  p_table <- p_table |>
    dplyr::mutate(signif = dplyr::case_when(
                           .data[["Estimate"]] < 0 & (.data[[padj_colname]] < alpha) ~ "Down",
                           .data[["Estimate"]] > 0 & (.data[[padj_colname]] < alpha) ~ "Up",
                           .default = "No"
                           ),
                  signif_label = ifelse(signif == "No", NA, .data[[feat_label]]),
                  padj_log = -log10(.data[[padj_colname]])
    )

  ggplot2::ggplot(data = p_table,
                  ggplot2::aes(x = .data[["Estimate"]],
                               y = .data[["padj_log"]],
                               colour = .data[["signif"]],
                               label = .data[["signif_label"]])) +
    ggplot2::geom_point() +
    ggrepel::geom_text_repel(max.overlaps = 10, show.legend = FALSE) +
    ggplot2::scale_color_manual(values = c(Down = "midnightblue",
                                           No = "darkgrey",
                                           Up = "darkorange2")) +
    ggplot2::geom_vline(xintercept = c(0),
                        linetype = 2,
                        colour = "black") +
    ggplot2::geom_hline(yintercept = -log10(alpha),
                        linetype = 2,
                        colour = "black") +
    ggplot2::labs(x = "Coefficient",
                  y = expression("-log"[10]*"(Padj)"),
                  colour = "Significant") +
    ready_theme(...)
}

#' @description Internal heatmap of feature-level estimates across coefficients.
#' @param p_table A data frame with `Estimate`, `Coefficient`, and `feat_id`.
#' @param feat_label Column containing desired feature labels.
#' @param ... Additional arguments passed to `ready_theme()`.
plot_heatmap <- function(p_table, feat_label, ...) {
  min_est <- min(p_table$Estimate, na.rm = TRUE)
  max_est <- max(p_table$Estimate, na.rm = TRUE)

  ggplot2::ggplot(data = p_table,
                  ggplot2::aes(x = .data[["Coefficient"]],
                               y = .data[[feat_label]],
                               fill = .data[["Estimate"]])) +
    ggplot2::geom_tile(colour = "darkgrey") +
    ggplot2::scale_fill_gradientn(colours = c("#313695","white","#a50026"),
                                  values = scales::rescale(c(min_est, 0, max_est)),
                                  limits = c(min_est, max_est)) +
    ggplot2::scale_x_discrete(expand = c(0, 0)) +
    ggplot2::scale_y_discrete(expand = c(0, 0)) +
    ggplot2::labs(x = NULL,
                  y = NULL,
                  fill = "Coefficient") +
    ready_theme(...)
}

#' @description Dot plot where size reflects significance (-log10(padj)).
#' @param p_table A data frame with `Estimate`, `Coefficient`, `feat_id`, and padj column.
#' @param padj_colname Name of the column with adjusted P-values.
#' @param feat_label Column containing desired feature labels.
#' @param ... Additional arguments passed to `ready_theme()`.
#' @importFrom rlang :=
plot_point <- function(p_table, padj_colname, feat_label, ...) {
  # Find coefficient with the max Estimate
  max_coef <- p_table |>
    dplyr::group_by(.data[["Coefficient"]]) |>
    dplyr::summarise(max_est = max(.data[["Estimate"]], na.rm = TRUE), .groups = "drop") |>
    dplyr::arrange(dplyr::desc(.data[["max_est"]])) |>
    dplyr::slice(1) |>
    dplyr::pull(.data[["Coefficient"]])

  # Get the ordering of feat_label based on Estimate from that coefficient only
  ordering <- p_table |>
    dplyr::filter(.data[["Coefficient"]] == max_coef) |>
    dplyr::arrange(.data[["Estimate"]]) |>
    dplyr::pull(!!rlang::sym(feat_label)) |>
    unique()

  # Apply ordering
  p_table <- p_table |>
    dplyr::mutate(
      padj_log = -log10(.data[[padj_colname]]),
      !!feat_label := factor(.data[[feat_label]], levels = ordering)
    )

  ggplot2::ggplot(data = p_table,
                  ggplot2::aes(x = .data[["Estimate"]],
                               y = .data[[feat_label]],
                               colour = .data[["Coefficient"]],
                               size = .data[["padj_log"]])) +
    ggplot2::geom_point(alpha = 0.75,
                        show.legend = c(colour = TRUE,
                                        size = TRUE)) +
    ggplot2::scale_size_continuous(range = c(1, 3)) +
    ggplot2::scale_x_continuous(expand = c(0.1, 0)) +
    ggplot2::geom_vline(xintercept = 0,
                        linetype = 2,
                        linewidth = 0.1) +
    ggplot2::scale_colour_brewer(palette = "Dark2") +
    ggplot2::guides(colour = ggplot2::guide_legend(order = 1),
                    size = ggplot2::guide_legend(order = 2)) +
    ggplot2::labs(x = "Coefficient",
                  y = NULL,
                  colour = "Group",
                  size = expression("-log"[10]*"(Padj)")) +
    ggplot2::theme(panel.grid.major.y = ggplot2::element_line(linetype = 2,
                                                              linewidth = 0.1,
                                                              colour = "black")) +
    ready_theme(...)
}

#' @description Internal plot of feature values using scatter, boxplot, or violin geoms.
#' @param p_table A long-format data frame with `value`, `feat_id`, and grouping variable.
#' @param plot_type One of `"scatter"`, `"boxplot"`, or `"violin"`.
#' @param sdata_var Name of the grouping variable.
#' @param paired_id Optional column name indicating paired observations.
#' @param feat_label Column containing desired feature labels.
#' @param group_colours Optional names vector of custom fill colours.
#' @param ... Additional arguments passed to `ready_theme()`.
plot_feat <- function(p_table, plot_type = c("scatter", "boxplot", "violin"),
                      sdata_var, paired_id = NULL, feat_label, group_colours, ...) {

  plot_type <- match.arg(plot_type)

  # Basic plot object
  p <- ggplot2::ggplot(p_table,
                       ggplot2::aes(x = .data[[sdata_var]],
                                    y = .data[["value"]])) +
       ggplot2::facet_wrap(ggplot2::vars(.data[[feat_label]]),
                                         scales = "free_y") +
       ggplot2::labs(x = NULL,
                     y = NULL) +
       ready_theme(...)

  if (!is.null(paired_id)) {
    p <- p +
      ggplot2::geom_line(ggplot2::aes(group = .data[[paired_id]]),
                                      colour = "darkgrey",
                                      linewidth = 0.1)
  }

  if (!is.null(group_colours) && plot_type != "scatter") {
    p <- p +
      ggplot2::scale_fill_manual(values = group_colours)
  }

  if (plot_type == "scatter") {
    p <- p + ggplot2::geom_point(alpha = 0.75)

  } else if (plot_type == "boxplot") {
    p <- p +
      ggplot2::geom_boxplot(ggplot2::aes(fill = .data[[sdata_var]]),
                            alpha = 0.75,
                            outlier.shape = NA) +
      ggplot2::geom_jitter(alpha = 0.75,
                           width = 0.25)

  } else if (plot_type == "violin") {
    p <- p +
      ggplot2::geom_violin(ggplot2::aes(fill = .data[[sdata_var]]),
                           linewidth = 0.1,
                           scale = "width",
                           show.legend = FALSE) +
      ggplot2::geom_boxplot(width = 0.1,
                            outlier.size = 0.25,
                            linewidth = 0.1)

  }
  return(p)
}

#' @description Ridge plot of feature value distributions across groups.
#' @param p_table A data frame with `value`, `feat_id`, and grouping variable.
#' @param sdata_var Name of the grouping variable.
#' @param feat_label Column containing desired feature labels.
#' @param group_colours Optional names vector of custom fill colours.
#' @param ... Additional arguments passed to `ready_theme()`.
plot_feat_ridge <- function(p_table, sdata_var, feat_label, group_colours, ...) {
  p <- ggplot2::ggplot(p_table,
                ggplot2::aes(x = .data[["value"]],
                             y = .data[[feat_label]],
                             fill = .data[[sdata_var]])) +
       ggridges::geom_density_ridges(alpha = 0.6,
                                     linewidth = 0.05) +
       ggplot2::scale_x_continuous(expand = c(0, 0)) +
       ggplot2::scale_y_discrete(expand = c(0, 0)) +
       ggplot2::coord_cartesian(clip = "off") +
       ggplot2::labs(fill = "Group") +
       ggridges::theme_ridges(center_axis_labels = TRUE,
                              font_size = 8,
                              grid = FALSE,
                              line_size = 1) +
      ready_theme(...)

  if (!is.null(group_colours)) {
    p <- p +
      ggplot2::scale_fill_manual(values = group_colours)
  }

  return(p)
}

#' @description Minimal and consistent ggplot2 theme for use across all ready plots.
#' @param ... Additional `ggplot2::theme()` parameters to override defaults.
ready_theme <- function(...) {
  ggplot2::theme(legend.background = ggplot2::element_blank(),
                 legend.key = ggplot2::element_blank(),
                 legend.key.size = grid::unit(0.25, "cm"),
                 legend.margin = ggplot2::margin(0, 0, 0, -5),
                 legend.spacing = grid::unit(-0.01, "cm"),
                 legend.title = ggplot2::element_text(margin = ggplot2::margin(0, 0, 1, 0)),
                 line = ggplot2::element_line(linewidth = 0.1),
                 panel.background = ggplot2::element_blank(),
                 panel.border = ggplot2::element_rect(linetype = 1,
                                                      fill = NA,
                                                      linewidth = 0.1),
                 plot.title = ggplot2::element_text(margin = ggplot2::margin(0, 0, 0, 0)),
                 strip.background = ggplot2::element_rect(linewidth = 0.1,
                                                          colour = "black"),
                 text = ggplot2::element_text(size = 7),
                 ...)
}
