% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Piecemeal.R
\name{Piecemeal}
\alias{Piecemeal}
\title{The \code{Piecemeal} \code{\link[R6:R6Class]{R6}} Class}
\description{
This class exports methods for configuring a simulation, running it, debugging failed configurations, and resuming the simulation. See \href{../doc/piecemeal.html}{the vignette \code{vignette("piecemeal")}} for long worked example.
}
\details{
A chain of \code{R6} method calls is used to specify the setup and the worker functions, the treatment configurations to be passed to the worker, and parallelism and other simulation settings. Then, when \verb{$run()} is called, the cluster is started, worker nodes are initialised, and every combination of random seed and treatment configuration is passed to \code{\link[=clusterApplyLB]{clusterApplyLB()}} (if parallel processing is enabled).

On the worker nodes, the worker function is not called directly; rather, care is taken to make sure that the specified configuration and seed is not already being worked on. This makes it safe to, e.g., queue multiple jobs for the same simulation. If the configuration is available, \code{set.seed()} is called with the seed and then the worker function is run.

Errors in the worker function are caught and error messages saved and returned.
}
\note{
If no treatment is specified, the function is called with no arguments (or just \code{.seed}).

If \code{Piecemeal$options(error = "auto")} (the default) is set, changing some configuration settings, including the worker function, the setup code, and the exported variables, will automatically set a flag to run \code{clean()} before the next run.

The estimation method is a simple ratio, so it may be biased under some circumstances. Also, it does not check if the runs have been completed successfully.
}
\examples{
# Initialise, with the output directory.
sim <- piecemeal::init(file.path(tempdir(), "piecemeal_demo"))
# Clear the previous simulation, if present.
sim$reset()

# Set up a simulation:
sim$
  # for every combination of x = 1, 2 and y = 1, 3, 9, 27,
  factorial(x = 2^(0:1), y = 3^(0:3))$
  # each replicated 3 times,
  nrep(3)$
  # first load library 'rlang',
  setup({library(rlang)})$
  # then for each x, y, and seed, evaluate
  worker(function(x, y) {
    p <- x*y
    u <- runif(1)
    dbl(p = p, u = u)
  })$
  # on a cluster with two nodes.
  cluster(2)

# Summarise
sim

# Go!
sim$run()

# Get a table with the results.
sim$result_df()

# For a more involved version of this example, see vignette("piecemeal").

}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Piecemeal-new}{\code{Piecemeal$new()}}
\item \href{#method-Piecemeal-cluster}{\code{Piecemeal$cluster()}}
\item \href{#method-Piecemeal-export_vars}{\code{Piecemeal$export_vars()}}
\item \href{#method-Piecemeal-setup}{\code{Piecemeal$setup()}}
\item \href{#method-Piecemeal-worker}{\code{Piecemeal$worker()}}
\item \href{#method-Piecemeal-treatments}{\code{Piecemeal$treatments()}}
\item \href{#method-Piecemeal-factorial}{\code{Piecemeal$factorial()}}
\item \href{#method-Piecemeal-nrep}{\code{Piecemeal$nrep()}}
\item \href{#method-Piecemeal-seeds}{\code{Piecemeal$seeds()}}
\item \href{#method-Piecemeal-run}{\code{Piecemeal$run()}}
\item \href{#method-Piecemeal-todo}{\code{Piecemeal$todo()}}
\item \href{#method-Piecemeal-result_list}{\code{Piecemeal$result_list()}}
\item \href{#method-Piecemeal-result_df}{\code{Piecemeal$result_df()}}
\item \href{#method-Piecemeal-reset}{\code{Piecemeal$reset()}}
\item \href{#method-Piecemeal-clean}{\code{Piecemeal$clean()}}
\item \href{#method-Piecemeal-erred}{\code{Piecemeal$erred()}}
\item \href{#method-Piecemeal-options}{\code{Piecemeal$options()}}
\item \href{#method-Piecemeal-print}{\code{Piecemeal$print()}}
\item \href{#method-Piecemeal-status}{\code{Piecemeal$status()}}
\item \href{#method-Piecemeal-eta}{\code{Piecemeal$eta()}}
\item \href{#method-Piecemeal-clone}{\code{Piecemeal$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-new"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{Piecemeal} instance.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$new(outdir)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{outdir}}{the directory to hold the partial simulation results.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-cluster"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-cluster}{}}}
\subsection{Method \code{cluster()}}{
Cluster settings for the piecemeal run.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$cluster(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{either arguments to \code{\link[=makeCluster]{makeCluster()}} or a single argument containing either an existing cluster or \code{NULL} to disable clustering.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-export_vars"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-export_vars}{}}}
\subsection{Method \code{export_vars()}}{
Specify variables to be copied from the manager node to the worker nodes' global environment. (See \code{\link[parallel:clusterApply]{parallel::clusterExport()}}.)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$export_vars(varlist, envir = parent.frame(), .add = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{varlist}}{a character vector with variable names.}

\item{\code{envir}}{the environment on the manager node from which to take the variables; defaults to the current environment.}

\item{\code{.add}}{whether the new variables should be added to the current list (if \code{TRUE}, the default) or replace it (if \code{FALSE}).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-setup"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-setup}{}}}
\subsection{Method \code{setup()}}{
Specify code to be run on each worker node at the start of the simulation; if running locally, it will be evaluated in the global environment.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$setup(
  expr = {
 }
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{expr}}{an expression; if passed, replaces the previous expression; if empty, resets it to nothing.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-worker"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-worker}{}}}
\subsection{Method \code{worker()}}{
Specify the function to be run for each treatment configuration; it will be run in the global environment.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$worker(fun)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{fun}}{a function whose arguments are specified by \verb{$treatments()} and \verb{$factorial()}; if it has \code{.seed} as a named argument, the seed will be passed as well.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-treatments"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-treatments}{}}}
\subsection{Method \code{treatments()}}{
Specify a list of treatment configurations to be run.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$treatments(l, .add = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{l}}{a list, typically of lists of arguments to be passed to the function specified by \code{worker}; it is recommended that these be as compact as possible, since they are \code{\link{serialize}}d and sent to the worker node for every combination of treatment configuration and random seed.}

\item{\code{.add}}{whether the new treatment configurations should be added to the current list (if \code{TRUE}, the default) or replace it (if \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-factorial"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-factorial}{}}}
\subsection{Method \code{factorial()}}{
Specify a list of treatment configurations to be run in a factorial design.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$factorial(..., .filter = function(...) TRUE, .add = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{vectors or lists whose Cartesian product will added to the treatment list; it is recommended that these be as compact as possible, since they are \code{\link{serialize}}d and sent to the worker node for every combination of treatment configuration and random seed.}

\item{\code{.filter}}{a function that takes the same arguments as worker and returns \code{FALSE} if the treatment configuration should be skipped; defaults to accepting all configurations.}

\item{\code{.add}}{whether the new treatment configurations should be added to the current list (if \code{TRUE}, the default) or replace it (if \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-nrep"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-nrep}{}}}
\subsection{Method \code{nrep()}}{
Specify a number of replications for each treatment configuration (starts out at 1).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$nrep(nrep)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{nrep}}{a positive integer giving the number of replications; the seeds will be set to \code{1:nrep}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-seeds"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-seeds}{}}}
\subsection{Method \code{seeds()}}{
Specify the seeds to be used for each replication of each treatment configuration.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$seeds(seeds)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{seeds}}{an integer vector of seeds; its length will be used to infer the number of replications.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-run"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-run}{}}}
\subsection{Method \code{run()}}{
Run the simulation.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$run(shuffle = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{shuffle}}{Should the treatment configurations be run in a random order (\code{TRUE}, the default) or in the order in which they were added (\code{FALSE})?}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisibly, a character vector with an element for each seed and treatment configuration combination attempted, indicating its file name and status, including errors.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-todo"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-todo}{}}}
\subsection{Method \code{todo()}}{
List the configurations still to be run.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$todo()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A list of lists with arguments to the worker functions and worker-specific configuration settings; also an attribute \code{"done"} giving the number of configurations skipped because they are already done.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-result_list"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-result_list}{}}}
\subsection{Method \code{result_list()}}{
Scan through the results files and collate them into a list.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$result_list(n = Inf, trt_tf = identity, out_tf = identity)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{n}}{maximum number of files to load; if less than the number of results, a systematic sample is taken.}

\item{\code{trt_tf, out_tf}}{functions that take the treatment configuration list and the output (if not an error) respectively, and transform them; this is helpful when, for example, the output is big and so loading all the files will run out of memory.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list of lists containing the contents of the result files.
\describe{
\item{\code{treatment}}{arguments passed to the worker}
\item{\code{seed}}{the seed set just before calling the worker}
\item{\code{output}}{value returned by the worker, or a \code{try-error} returned by \code{\link[=try]{try()}}}
\item{\code{OK}}{whether the worker succeeded or produced an error}
\item{\code{config}}{miscellaneous configuration settings such as the file name}
}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-result_df"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-result_df}{}}}
\subsection{Method \code{result_df()}}{
Scan through the results files and collate them into a data frame.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$result_df(trt_tf = identity, out_tf = identity, rds = FALSE, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{trt_tf, out_tf}}{functions that take the treatment configuration list and the output respectively, and return named lists that become data frame columns; a special value \code{I} instead creates columns \code{treatment} and/or \code{output} with the respective lists copied as is.}

\item{\code{rds}}{whether to include an \code{.rds} column described below.}

\item{\code{...}}{additional arguments, passed to \code{Piecemeal$result_list()}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A data frame with columns corresponding to the values returned by \code{trt_tf} and \code{out_tf}, with the following additional columns:
\describe{
\item{\code{.seed}}{the random seed used.}
\item{\code{.rds}}{the path to the RDS file (if requested).}
}
Runs that erred are filtered out.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-reset"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-reset}{}}}
\subsection{Method \code{reset()}}{
Clear the simulation results so far.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$reset(confirm = interactive())}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{confirm}}{whether the user should be prompted to confirm deletion.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-clean"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-clean}{}}}
\subsection{Method \code{clean()}}{
Delete the result files for which the worker function produced an error and/or which were somehow corrupted, or based on some other predicate.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$clean(which = function(res) !res$OK)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{which}}{a function of a result list (see \code{Piecemeal$result_list()}) returning \code{TRUE} if the result file is to be deleted and \code{FALSE} otherwise.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-erred"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-erred}{}}}
\subsection{Method \code{erred()}}{
List the configurations for which the worker function failed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$erred()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-options"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-options}{}}}
\subsection{Method \code{options()}}{
Set miscellaneous options.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$options(split = c(1L, 1L), error = c("auto", "save", "skip", "stop"))}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{split}}{a two-element vector indicating whether the output files should be split up into subdirectories and how deeply, the first for splitting configurations and the second for splitting seeds; this can improve performance on some file systems.}

\item{\code{error}}{how to handle worker errors:\describe{
\item{\code{"save"}}{save the seed, the configuration, and the status, preventing future runs until the file is removed using \code{Piecemeal$clean()}.}
\item{\code{"skip"}}{return the error message as a part of \code{run()}'s return value, but do not save the RDS file; the next \code{run()} will attempt to run the worker for that configuration and seed again.}
\item{\code{"stop"}}{allow the error to propagate; can be used in conjunction with \code{Piecemeal$cluster(NULL)} and (global) \code{options(error = recover)} to debug the worker.}
\item{\code{"auto"}}{(default) as \code{"save"}, but if any of the methods that change how each configuration is run (i.e., \verb{$worker()}, \verb{$setup()}, and \verb{$export_vars()}) is called, \verb{$clean()} will be called automatically before the next \verb{$run()}.}
}}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-print"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-print}{}}}
\subsection{Method \code{print()}}{
Print the current simulation settings, including whether there is enough information to run it.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{additional arguments, currently unused.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-status"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-status}{}}}
\subsection{Method \code{status()}}{
Summarise the current status of the simulation, including the number of runs succeeded, the number of runs still to be done, the number of runs currently running, the errors encountered, and, if started, the estimated time to completion at the current rate.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$status(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{additional arguments, currently passed to \code{Piecemeal$eta()}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-eta"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-eta}{}}}
\subsection{Method \code{eta()}}{
Estimate the rate at which runs are being completed and how much more time is needed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$eta(window = 3600)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{window}}{initial time window to use, either a \code{\link{difftime}} object or the number in seconds; defaults to 1 hour.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The window used is actually between the last completed run and the earliest run in the \code{window} before that. This allows to take an interrupted simulation and estimate how much more time (at the most recent rate) is needed.
}

\subsection{Returns}{
A list with elements \code{window}, \code{recent}, \code{cost}, \code{left}, \code{rate}, and \code{eta}, containing, respectively, the time window, the number of runs completed in this time, the average time per completion, the estimated time left (all in seconds), the corresponding rate (in Hertz), and the expected time of completion.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Piecemeal-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Piecemeal-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Piecemeal$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
