% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimateR.R
\name{estimateR}
\alias{estimateR}
\alias{estimateR_mixed}
\title{Estimate latent correlation matrix}
\usage{
estimateR(
  X,
  type = "trunc",
  method = "original",
  use.nearPD = TRUE,
  nu = 0.01,
  tol = 0.001,
  verbose = FALSE
)

estimateR_mixed(
  X1,
  X2,
  type1 = "trunc",
  type2 = "continuous",
  method = "original",
  use.nearPD = TRUE,
  nu = 0.01,
  tol = 0.001,
  verbose = FALSE
)
}
\arguments{
\item{X}{A numeric data matrix (n by p), n is the sample size and p is the number of variables.}

\item{type}{A type of variables in \code{X}, must be one of "continuous", "binary" or "trunc".}

\item{method}{The calculation method of latent correlation. Either "original" method or "approx". If \code{method = "approx"}, multilinear approximation method is used, which is much faster than the original method (requires \code{chebpol} R package). If \code{method = "original"}, optimization of the bridge inverse function is used. The default is "original".}

\item{use.nearPD}{A logical value indicating whether to use \link[Matrix]{nearPD} or not when the resulting correlation estimator is not positive definite (have at least one negative eigenvalue).}

\item{nu}{Shrinkage parameter for correlation matrix, must be between 0 and 1, the default value is 0.01.}

\item{tol}{Desired accuracy when calculating the solution of bridge function.}

\item{verbose}{If \code{verbose = FALSE}, printing information whether nearPD is used or not is disabled. The default value is FALSE.}

\item{X1}{A numeric data matrix (n by p1).}

\item{X2}{A numeric data matrix (n by p2).}

\item{type1}{A type of variables in \code{X1}, must be one of "continuous", "binary" or "trunc".}

\item{type2}{A type of variables in \code{X2}, must be one of "continuous", "binary" or "trunc".}
}
\value{
\code{estimateR} returns
\itemize{
      \item type: Type of the data matrix \code{X}
      \item R: Estimated p by p latent correlation matrix of \code{X}
}

\code{estimateR_mixed} returns
\itemize{
      \item type1: Type of the data matrix \code{X1}
      \item type2: Type of the data matrix \code{X2}
      \item R: Estimated latent correlation matrix of whole \code{X} = (\code{X1}, \code{X2}) (p1+p2 by p1+p2)
      \item R1: Estimated latent correlation matrix of \code{X1} (p1 by p1)
      \item R2: Estimated latent correlation matrix of \code{X2} (p2 by p2)
      \item R12: Estimated latent correlation matrix between \code{X1} and \code{X2} (p1 by p2)
}
}
\description{
Estimation of latent correlation matrix from observed data of (possibly) mixed types (continuous/binary/truncated continuous) based on the latent Gaussian copula model.
}
\examples{
### Data setting
n <- 100; p1 <- 15; p2 <- 10 # sample size and dimensions for two datasets.
maxcancor <- 0.9 # true canonical correlation

### Correlation structure within each data set
set.seed(0)
perm1 <- sample(1:p1, size = p1);
Sigma1 <- autocor(p1, 0.7)[perm1, perm1]
blockind <- sample(1:3, size = p2, replace = TRUE);
Sigma2 <- blockcor(blockind, 0.7)
mu <- rbinom(p1+p2, 1, 0.5)

### true variable indices for each dataset
trueidx1 <- c(rep(1, 3), rep(0, p1-3))
trueidx2 <- c(rep(1, 2), rep(0, p2-2))

### Data generation
simdata <- GenerateData(n=n, trueidx1 = trueidx1, trueidx2 = trueidx2, maxcancor = maxcancor,
                        Sigma1 = Sigma1, Sigma2 = Sigma2,
                        copula1 = "exp", copula2 = "cube",
                        muZ = mu,
                        type1 = "trunc", type2 = "continuous",
                        c1 = rep(1, p1), c2 =  rep(0, p2)
)
X1 <- simdata$X1
X2 <- simdata$X2

### Check the range of truncation levels of variables
range(colMeans(X1 == 0))
range(colMeans(X2 == 0))

### Estimate latent correlation matrix
# with original method
R1_org <- estimateR(X1, type = "trunc", method = "original")$R
# with faster approximation method
R1_approx <- estimateR(X1, type = "trunc", method = "approx")$R
R12_approx <- estimateR_mixed(X1, X2, type1 = "trunc", type2 = "continuous", method = "approx")$R12


}
\references{
Fan J., Liu H., Ning Y. and Zou H. (2017) "High dimensional semiparametric latent graphicalmodel for mixed data" <doi:10.1111/rssb.12168>.

Yoon G., Carroll R.J. and Gaynanova I. (2020) "Sparse semiparametric canonical correlation analysis for data of mixed types" <doi:10.1093/biomet/asaa007>.

Yoon G., Mueller C.L., Gaynanova I. (2020) "Fast computation of latent correlations" <arXiv:2006.13875>.
}
